/**
 * @copyright    (C) 2013 - 2016 Robert Bosch GmbH.
 *               The reproduction, distribution and utilization of this file as well as the
 *               communication of its contents to others without express authorization is prohibited.
 *               Offenders will be held liable for the payment of damages.
 *               All rights reserved in the event of the grant of a patent, utility model or design.
 * @brief        This file contains the access to the eMMC, which saves transaction safe data into eMMC 
 * @addtogroup   PDD access 
 * @{
 */ 

/*
 * global function:
 * -- PDD_eMMCUserAccessInit:
 *       init routine for eMMC access 
 * -- PDD_eMMCUserAccessDeInit:
 *       deinit routine for eMMC flash access
 * -- PDD_eMMCUserAccessGetDataStreamSize:
 *       get size of data, without header 
 * -- PDD_eMMCUserAccessErase:
 *       erase all sector
 * -- PDD_eMMCUserAccessReadDataStream:
 *       function for read from eMMC 
 * -- PDD_eMMCUserAccessWriteDataStream:
 *       function for write to eMMC
 * -- PDD_eMMCUserAccessGetEraseCounter
 *       get erase counter for trace
 *
 * local function:
 * -- PDD_bWriteToClusterEmmc:
 *        write "data" to the next free cluster
 * -- PDD_bReadFromClusterEmmc:
 *        read the data from the last valid cluster to "data"
 * -- PDD_bReadFromLastClusterValidData:
 *        read data from the last cluster with valid data
 * -- PDD_s32FindValidClusterEmmc:
 *        find valid cluster
 * -- PDD_bIsClusterValidEmmc:
 *        check if cluster valid
 * -- PDD_bIsClusterValidDataWritten
 *        check if cluster has valid written data
 * -- PDD_bReadeMMC:
 *        read from eMMC
 * -- PDD_bWriteeMMC:
 *        write to eMMC
 * -- PDD_u8GetCheckDataStream:
 *        check the datastream name and get the table number
 */

/******************************************************************************/
/* include the system interface                                               */
/******************************************************************************/
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <limits.h>   
#include "system_types.h"
#include "pdd_variant.h"
#include "errno.h"
#ifdef PDD_TESTMANGER_ACTIVE  //define is set in pdd_variant.h
#include "pdd.h"
#include "pdd_testmanager.h"
#else
#include <sys/mman.h>
#include <fcntl.h>
#include <time.h>
#include "system_definition.h"
#include "pdd.h"
#endif
#include "pdd_config_nor_user.h"
#include "pdd_config_emmc_user.h"
#include "pdd_private.h"
#include "pdd_trace.h"

#ifdef __cplusplus
extern "C" {
#endif

/******************************************************************************* 
|defines and macros 
|------------------------------------------------------------------------------*/
/*magic number */
#define PDD_RAW_EMMC_USER_MAGIC                                     0xDEAD
#define PDD_RAW_EMMC_USER_MIN_CHUNK_SIZE                            0x20
#define PDD_RAW_EMMC_USER_NO_OF_CLUSTER                             2

/*defines for state of cluster => save in header  */
#define PDD_RAW_EMMC_USER_STATE_FORMAT_BEGIN                        0x01
#define PDD_RAW_EMMC_USER_STATE_FORMAT_END                          0x02
#define PDD_RAW_EMMC_USER_STATE_WRITE_BEGIN                         0x04
#define PDD_RAW_EMMC_USER_STATE_WRITE_END                           0x08
#define PDD_RAW_EMMC_USER_STATE_INVALID                             0x10
#define PDD_RAW_EMMC_USER_STATE_OLD_VERSION                         0x20
#define PDD_RAW_EMMC_USER_STATE_VALID                               0xFF

/*define for alignement*/
#define CHIP_ALIGNMENT (vtspPddRaweMMCInfo ->u16ChunkSize)
#define CHIP_ALIGNMENT_MASK (~(CHIP_ALIGNMENT-1))
#define ALIGN_TO_CHIP(x) (((tS32)x)+((CHIP_ALIGNMENT-1) & CHIP_ALIGNMENT_MASK))

/*position in the header of a cluster*/
#define PDD_RAW_EMMC_USER_MAGIC_POSITION                            (0x00*CHIP_ALIGNMENT)
#define PDD_RAW_EMMC_USER_STATUS_POSITION_FORMAT_BEGIN              (0x01*CHIP_ALIGNMENT)
#define PDD_RAW_EMMC_USER_STATUS_POSITION_FORMAT_END                (0x02*CHIP_ALIGNMENT)
#define PDD_RAW_EMMC_USER_STATUS_POSITION_WRITE_BEGIN               (0x03*CHIP_ALIGNMENT)
#define PDD_RAW_EMMC_USER_STATUS_POSITION_WRITE_END                 (0x04*CHIP_ALIGNMENT)
#define PDD_RAW_EMMC_USER_STATUS_POSITION_INVALID                   (0x05*CHIP_ALIGNMENT)
#define PDD_RAW_EMMC_USER_STATUS_POSITION_OLD_VERSION               (0x06*CHIP_ALIGNMENT)
#define PDD_RAW_EMMC_USER_CHECKSUM_POSITION                         (0x07*CHIP_ALIGNMENT)

/*define for header size*/
#define PDD_RAW_EMMC_USER_HEADER_SIZE                               (PDD_RAW_EMMC_USER_MAX_WRITE_POSITION*CHIP_ALIGNMENT)

/*define for return header update*/
#define PDD_RAW_EMMC_USER_UPDATE_HEADER                             1

#ifdef PDD_TESTMANGER_ACTIVE
#define PDD_RAW_EMMC_USER_NUMBER_DATASTREAM_TM                      vu8NumberDataStreamTm
#define PDD_RAW_EMMC_USER_HEADER_SIZE_DATA_STREAM                   (vu8NumberDataStreamTm*sizeof(tsPddRaweMMCUserHeaderDataStream))
#endif

/*32k cluster (if change the size, older persistent data not valid=> erase flash)*/
#define PDD_RAW_EMMC_USER_CLUSTER_SIZE                              0x8000

#ifdef PDD_TESTMANGER_ACTIVE
#define M_PDD_RAW_EMMC_ACCESS_OPEN(name)                            PddTm_tOpen(name)
#define M_PDD_RAW_EMMC_ACCESS_CLOSE(handle)                         PddTm_vClose(handle)
#define M_PDD_RAW_EMMC_ACCESS_READ(handle,buffer,size,offset)       PddTm_s32Read(handle,buffer,size,offset)
#define M_PDD_RAW_EMMC_ACCESS_WRITE(handle,buffer,size,offset)      PddTm_s32Write(handle,buffer,size,offset)
#define M_PDD_RAW_EMMC_ACCESS_ERASE(handle,size,offset)             PddTm_s32Erase(handle,size,offset)
#else
#define M_PDD_RAW_EMMC_ACCESS_OPEN(name)                            eMMC_open(name)
#define M_PDD_RAW_EMMC_ACCESS_CLOSE(handle)                         eMMC_close(handle)
#define M_PDD_RAW_EMMC_ACCESS_READ(handle,buffer,size,offset)       eMMC_read(handle,buffer,size,offset)
#define M_PDD_RAW_EMMC_ACCESS_WRITE(handle,buffer,size,offset)      eMMC_write(handle,buffer,size,offset)
#endif


/******************************************************************************* 
|typedef 
|------------------------------------------------------------------------------*/
/* structure for magic*/
typedef struct
{
  tU16  u16MagicLow;
  tU16   u16ChunkSize;
  tU32  u32ClusterUpdatedCount;
}tsPddRAWeMMCUserHeaderMagic;

/* structure for checksum */
typedef struct
{
  tU32  u32CheckSum;
  tU32  u32CheckSumXor;
}tsPddRAWeMMCUserHeaderChecksum;

/******************************************************************************/
/* static  variable                                                           */
/******************************************************************************/
static tsPddRaweMMCUserInfo*            vtspPddRaweMMCInfo      = NULL;

#ifdef PDD_TESTMANGER_ACTIVE
static int                              RaweMMCAccessHdl        = NULL;
static tU8                              vu8NumberDataStreamTm   = 0;
#else
static unsigned int                     RaweMMCAccessHdl        = 0;
#endif
/************************************************************************
| variable definition (scope: module-global)
|-----------------------------------------------------------------------*/

/******************************************************************************/
/* declaration local function                                                 */
/******************************************************************************/
static tS32   PDD_s32FindValidClusterEmmc(tsPddRaweMMCUserInfo*);
static tBool  PDD_bIsClusterValidEmmc(tsPddRaweMMCUserInfo*, tU32, tBool*);
static tBool  PDD_bReadEmmc(tsPddRaweMMCUserInfo* PtsDevInfo, tU32 Pu32Cluster, void* PvpDestAdress, tU32 Pu32Offset, tU32 Pu32Len);
static tBool  PDD_bWriteEmmc(tsPddRaweMMCUserInfo*, tS32, tU8*, tU32, tU32);
static tBool  PDD_bEraseEmmc(void);
static tU8    PDD_u8GetCheckDataStreamDefinedHeaderEmmc(const char*, tsPddRaweMMCUserInfo*);
#if PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM
static tBool  PDD_bWriteToClusterEmmc(tsPddRaweMMCUserInfo*, tU8*);
static tBool  PDD_bReadFromClusterEmmc(tsPddRaweMMCUserInfo*, void*);
static tU8    PDD_u8GetCheckDataStreamReadHeaderEmmc(char*, tsPddRaweMMCUserHeaderDataStream*);
static tS32   PDD_s32UpdateHeaderDataStreamEmmc(tsPddRaweMMCUserInfo*, tS32, tU8);
static void   PDD_vUpdateDataEmmc(tsPddRaweMMCUserInfo*, void*);
static tBool  PDD_bUpdateMagicAndWriteCountofCluster(tsPddRaweMMCUserInfo* PtsDevInfo, tS32  Vs32ClusterNo);
#endif
#ifdef PDD_TESTMANGER_ACTIVE
static tBool  PDD_bGetNumberDataStreamTmAndNameDataStream(tsPddRaweMMCUserHeaderDataStream*);
#endif

/******************************************************************************/
/* declaration global function                                                 */
/******************************************************************************/

/******************************************************************************
* FUNCTION: PDD_eMMCUserAccessInit()
*
* DESCRIPTION: init routine for eMMC raw access
*
* PARAMETERS:
*
* RETURNS:
*      positive value: PDD_OK
*      negative value: error code
*
* HISTORY:Created  2013 04 23
*****************************************************************************/
tS32 PDD_eMMCUserAccessInit(void)
{
    tS32                                  Vs32ErrorCode = PDD_OK;
#if PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM
    tU8                                   Vu8Inc;
    tsPddRaweMMCUserHeaderDataStream*     VspHeaderDataStream;
    const tsPddRaweMMCUserConfig*         VspPddRaweMMCUserConfig;
    if (!RaweMMCAccessHdl)
    {
        Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_INIT_FAIL;
        PDD_TRACE(PDD_ERROR, &Vs32ErrorCode, sizeof(tS32));
    }
    else
    {
        vtspPddRaweMMCInfo->u32ClusterSize = PDD_RAW_EMMC_USER_CLUSTER_SIZE;
        /*get num blocks*/
        vtspPddRaweMMCInfo->u32NumCluster = PDD_RAW_EMMC_USER_NO_OF_CLUSTER;

        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_NUMBER_OF_CLUSTER, &vtspPddRaweMMCInfo->u32NumCluster, sizeof(tU32));

        /*init other variable*/
        vtspPddRaweMMCInfo->s32LastClusterUsed = -1;
        /* get num cluster*/
            /* get chunk size */
        vtspPddRaweMMCInfo->u16ChunkSize = PDD_RAW_EMMC_USER_MIN_CHUNK_SIZE;
        /* trace chunk size */
        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_SIZE_CHUNK, &vtspPddRaweMMCInfo->u16ChunkSize, sizeof(vtspPddRaweMMCInfo->u16ChunkSize));

        /* init variable, which depends on vtspPddRaweMMCInfo->u16ChunkSize*/
        vtspPddRaweMMCInfo->u32ClusterSizeData = vtspPddRaweMMCInfo->u32ClusterSize - (tU32)PDD_RAW_EMMC_USER_HEADER_SIZE;

        /* init buffer and clear header*/
        memset(vtspPddRaweMMCInfo->u8pBufferDataOld, 0xff, vtspPddRaweMMCInfo->u32ClusterSizeData);
        /* get possible filename*/
#ifndef PDD_TESTMANGER_ACTIVE
        memset(vtspPddRaweMMCInfo->sHeaderDataStream, 0, PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM*sizeof(tsPddRaweMMCUserHeaderDataStream));
        for (Vu8Inc = 0; Vu8Inc < PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM; Vu8Inc++)
#else
        memset(vtspPddRaweMMCInfo->sHeaderDataStream, 0, PDD_RAW_EMMC_USER_CONFIG_MAX_NUMBER_DATASTREAM*sizeof(tsPddRaweMMCUserHeaderDataStream));
        for (Vu8Inc = 0; Vu8Inc < PDD_RAW_EMMC_USER_CONFIG_MAX_NUMBER_DATASTREAM; Vu8Inc++)
#endif
        {/* get possible filename*/
            VspHeaderDataStream = &vtspPddRaweMMCInfo->sHeaderDataStream[Vu8Inc];
            VspPddRaweMMCUserConfig = &vrPddRaweMMCUserConfig[Vu8Inc];
#ifdef PDD_TESTMANGER_ACTIVE
            /*only name "kds_data" is copy to name*/
            if (Vu8Inc < PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM_TM)
#endif
                memmove(VspHeaderDataStream->u8DataStreamName, VspPddRaweMMCUserConfig->u8DataStreamName, PDD_RAW_EMMC_MAX_SIZE_NAME);
            /*offset,lenght,count is set by first application access*/
            VspHeaderDataStream->u32Offset = 0;
            VspHeaderDataStream->u32Lenght = 0;
            VspHeaderDataStream->u32WriteCount = 0;
        }
        /*trace out header */
        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_DATA_STREAM_HEADER, &vtspPddRaweMMCInfo->sHeaderDataStream[0], PDD_RAW_EMMC_USER_HEADER_SIZE_DATA_STREAM);
    }/* end else*/
#endif
    return(Vs32ErrorCode);
}
/******************************************************************************
* FUNCTION: PDD_eMMCUserAccessDeInit()
*
* DESCRIPTION: deinit routine for eMMC access
*
* PARAMETERS:
*
* RETURNS:
*      positive value: PDD_OK
*      negative value: error code
*
* HISTORY:Created  2013 04 23
*****************************************************************************/
tS32 PDD_eMMCUserAccessDeInit(void)
{
    tS32                              Vs32ErrorCode = PDD_OK;
#if PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM
    tU8                               Vu8Inc;
    tsPddRaweMMCUserHeaderDataStream* VspHeaderDataStream;

    if (!RaweMMCAccessHdl)
    {
        Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_INIT_FAIL;
        PDD_TRACE(PDD_ERROR, &Vs32ErrorCode, sizeof(tS32));
    }
    else
    {/* cluster size*/
        vtspPddRaweMMCInfo->u32ClusterSize = 0;
        /* num blocks*/
        vtspPddRaweMMCInfo->u32NumCluster = 0;

        /*init other variable*/
        vtspPddRaweMMCInfo->s32LastClusterUsed = -1;
        /* chunk size */
        vtspPddRaweMMCInfo->u16ChunkSize = 1;
        /*u32ClusterSizeData*/
        vtspPddRaweMMCInfo->u32ClusterSizeData = 0;
        /* clear datastream*/
#ifndef PDD_TESTMANGER_ACTIVE
        for (Vu8Inc = 0; Vu8Inc < PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM; Vu8Inc++)
#else
        for (Vu8Inc = 0; Vu8Inc < PDD_RAW_EMMC_USER_CONFIG_MAX_NUMBER_DATASTREAM; Vu8Inc++)
#endif
        {/* get possible filename*/
            VspHeaderDataStream = &vtspPddRaweMMCInfo->sHeaderDataStream[Vu8Inc];
            memset(VspHeaderDataStream->u8DataStreamName, 0x00, PDD_RAW_EMMC_MAX_SIZE_NAME);
            VspHeaderDataStream->u32Offset = 0;
            VspHeaderDataStream->u32Lenght = 0;
            VspHeaderDataStream->u32WriteCount = 0;
        }
    }
#endif
    return(Vs32ErrorCode);
}
/******************************************************************************
* FUNCTION: PDD_eMMCUserAccessInitProcess()
*
* DESCRIPTION:
*
* PARAMETERS:
*
* RETURNS:
*
* HISTORY:Created  2013 04 23
*****************************************************************************/
tS32 PDD_eMMCUserAccessInitProcess(void)
{
    tS32  Vs32ErrorCode = PDD_OK;
    vtspPddRaweMMCInfo = PDD_GetRaweMMCUserInfoShm();
    if (!RaweMMCAccessHdl)
    { /* PDD*/
        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_BLOCK_DEVICE_OPEN, 0, 0);
        RaweMMCAccessHdl = M_PDD_RAW_EMMC_ACCESS_OPEN("PDD_eMMCUserEarly");
        if (!RaweMMCAccessHdl)
        {
            Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_DEVICE_WRONG_HANDLE;
            PDD_TRACE(PDD_ERROR, &Vs32ErrorCode, sizeof(tS32));
        }
    }
    return(Vs32ErrorCode);
}
/******************************************************************************
* FUNCTION: PDD_eMMCUserAccessDeInitProcess()
*
* DESCRIPTION:
*
* PARAMETERS:
*
*
* RETURNS:
*
* HISTORY:Created  2013 04 23
*****************************************************************************/
tS32 PDD_eMMCUserAccessDeInitProcess(void)
{
    tS32  Vs32ErrorCode = PDD_OK;
    M_PDD_RAW_EMMC_ACCESS_CLOSE(RaweMMCAccessHdl);
    RaweMMCAccessHdl = 0;
    vtspPddRaweMMCInfo = NULL;
    return(Vs32ErrorCode);
}
/******************************************************************************
* FUNCTION: PDD_eMMCUserAccessGetDataStreamSize()
*
* DESCRIPTION: get size of data, without header
*
* PARAMETERS:
*      PtsDataStreamName: name of the data stream
*
* RETURNS:
*      positive value: size of datastream with header
*      negative value: error code
*
* HISTORY:Created  2013 04 23
*****************************************************************************/
tS32 PDD_eMMCUserAccessGetDataStreamSize(const char* PtsDataStreamName)
{
    tS32               Vs32Size = PDD_OK;
    tU8                VubDataStream;

    if (PtsDataStreamName != NULL)
    {
        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_DATA_STREAM_NAME, PtsDataStreamName, (tU32)strlen(PtsDataStreamName) + 1);
        /*check stream*/
        VubDataStream = PDD_u8GetCheckDataStreamDefinedHeaderEmmc(PtsDataStreamName, vtspPddRaweMMCInfo);
        /* check if name in configuration*/
        if (VubDataStream == 0xff)
        { /*error in generate header for PDD?: if defined location for NOR (Datapool or PDD),
            header of PDD gets the name for this stream */
            tChar  Vs8Buffer[200];
            Vs32Size = PDD_ERROR_RAW_EMMC_USER_DATA_STREAM_NOT_DEFINED;
            snprintf(Vs8Buffer, sizeof(Vs8Buffer) - 1, "0x%04x: %d datastream:'%s' not defined", Vs32Size, Vs32Size, PtsDataStreamName);
            PDD_TRACE(PDD_ERROR_STR, Vs8Buffer, (tU32)strlen(Vs8Buffer) + 1);
            PDD_SET_ERROR_ENTRY(&Vs8Buffer[0]);
        }
        else
        { /* read data stream configuration */
            if (PDD_bReadDataStreamConfigurationEmmc(vtspPddRaweMMCInfo) == TRUE)
            { /* set size */
#if PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM
                Vs32Size = (tS32)vtspPddRaweMMCInfo->sHeaderDataStream[VubDataStream].u32Lenght;
#else
                Vs32Size = PDD_ERROR_RAW_EMMC_USER_DATA_STREAM_NOT_DEFINED;
#endif
            }
            else
            { /*configuration wrong number > PDD_RAW_EMMC_USER_CONFIG_MAX_NUMBER_DATASTREAM*/
                Vs32Size = PDD_ERROR_RAW_EMMC_USER_CONFIGURATION_HEADER;
                PDD_TRACE(PDD_ERROR, &Vs32Size, (tU32)sizeof(tS32));
                PDD_SET_ERROR_ENTRY("RAW_EMMC_USER: error configuration header");
            }
        }/*end else*/
    }
    else
        Vs32Size = PDD_ERROR_RAW_EMMC_USER_INVALID_INPUT;
    return(Vs32Size);
}

/******************************************************************************
* FUNCTION: PDD_eMMCUserAccessErase()
*
* DESCRIPTION: erase all the cluster
*
* PARAMETERS:
*
*
* RETURNS:
*      positive value: PDD_OK
*      negative value: error code
*
* HISTORY:Created  2013 04 23
*****************************************************************************/
tS32 PDD_eMMCUserAccessErase(void)
{
    tS32               Vs32ErrorCode = PDD_OK;
#if PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM 
    if (FALSE == PDD_bEraseEmmc())
    {
        Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_WRITE_CLUSTER;
        PDD_TRACE(PDD_ERROR, &Vs32ErrorCode, sizeof(tS32));
    }
    if (Vs32ErrorCode == PDD_OK)
    {/*clear old buffer*/
        if (vtspPddRaweMMCInfo != NULL)
        {
            memset(&vtspPddRaweMMCInfo->u8pBufferDataOld[0], 0x00, vtspPddRaweMMCInfo->u32ClusterSizeData);
            /*set last cluster used to unvalid*/
            vtspPddRaweMMCInfo->s32LastClusterUsed = -1;
            /*no valid header*/
            vtspPddRaweMMCInfo->sHeaderDataStream[0].u32Offset = 0;
            /*set erase counter to zero*/
            vtspPddRaweMMCInfo->u32ClusterUpdatedCount = 0;
        }
    }
#endif
    return(Vs32ErrorCode);
}

/******************************************************************************
* FUNCTION: PDD_eMMCUserAccessReadPartition()
*
* DESCRIPTION: read the complete partition
*
* PARAMETERS:
*
*
* RETURNS:
*      positive value: PDD_OK
*      negative value: error code
*
* HISTORY:Created  2014 04 23
*****************************************************************************/
tS32 PDD_eMMCUserAccessReadPartition(void *PvpReadBuffer)
{
    tS32               Vs32ErrorCode = PDD_OK;
    if (PvpReadBuffer != NULL)
    {
        if (PDD_bReadEmmc(vtspPddRaweMMCInfo, 0, PvpReadBuffer, 0, PDD_RAW_EMMC_USER_MIN_NUMBER_OF_CLUSTER*PDD_RAW_EMMC_USER_MIN_CLUSTER_SIZE) == FALSE)
        {
            Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_READ_DUMP;
        }
    }
    else
        Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_INVALID_INPUT;
    return(Vs32ErrorCode);
}
/******************************************************************************
* FUNCTION: PDD_eMMCUserAccessReadActualCluster()
*
* DESCRIPTION: read actual cluster
*
* PARAMETERS:
*
*
* RETURNS:
*      positive value: PDD_OK
*      negative value: error code
*
* HISTORY:Created  2014 05 07
*****************************************************************************/
tS32 PDD_eMMCUserAccessReadActualCluster(void *PvpReadBuffer)
{
    tS32               Vs32ErrorCode = PDD_OK;
    tS32               Vs32FoundCluster = -1;
    if (PvpReadBuffer != NULL)
    {
        Vs32FoundCluster = PDD_s32FindValidClusterEmmc(vtspPddRaweMMCInfo);
        if (Vs32FoundCluster >= 0)
        {
            if (PDD_bReadEmmc(vtspPddRaweMMCInfo, (tU32)Vs32FoundCluster, PvpReadBuffer, 0, PDD_RAW_EMMC_USER_CLUSTER_SIZE) == FALSE)
            {
                Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_READ_DUMP;
            }
        }
        else
        {
            Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_READ_DUMP;
        }
    }
    else
        Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_INVALID_INPUT;
    return(Vs32ErrorCode);
}
/******************************************************************************
* FUNCTION: PDD_eMMCUserAccessWritePartition()
*
* DESCRIPTION: write data to parttion PDD_eMMCUserEarly
*
* PARAMETERS:
*      Ps32Cluster: Cluster
*
*
* RETURNS:
*      positive value: PDD_OK
*      negative value: error code
*
* HISTORY:Created  2014 05 07
*****************************************************************************/
tS32 PDD_eMMCUserAccessWritePartition(void *Ppu8WriteBuffer)
{
    tS32  Vs32ErrorCode = PDD_OK;
    if (Ppu8WriteBuffer != NULL)
    {
        if (PDD_bWriteEmmc(vtspPddRaweMMCInfo, 0, Ppu8WriteBuffer, 0, PDD_RAW_EMMC_USER_MIN_NUMBER_OF_CLUSTER*PDD_RAW_EMMC_USER_MIN_CLUSTER_SIZE) == FALSE)
        {
            Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_WRITE_CLUSTER;
        }
    }
    else
        Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_INVALID_INPUT;
    return(Vs32ErrorCode);
}
/******************************************************************************
* FUNCTION: PDD_eMMCUserAccessGetActualCluster()
*
* DESCRIPTION: get actual cluster
*
* PARAMETERS:
*      Ps32Cluster: Cluster
*
*
* RETURNS:
*      positive value: PDD_OK
*      negative value: error code
*
* HISTORY:Created  2014 05 08
*****************************************************************************/
tS32 PDD_eMMCUserAccessGetActualCluster(void)
{
    return(PDD_s32FindValidClusterEmmc(vtspPddRaweMMCInfo));
}
/******************************************************************************
* FUNCTION: PDD_eMMCUserAccessReadDataStream()
*
* DESCRIPTION: function for read from eMMC
*
* PARAMETERS:
*      PtsDataStreamName: name of the data stream
* RETURNS:
*      positive value: size of read data
*      negative value: error code
*
* HISTORY:Created  2013 04 23
*****************************************************************************/
tS32 PDD_eMMCUserAccessReadDataStream(const char* PtsDataStreamName, void *PpvReadBuffer, tS32 Ps32SizeReadBuffer)
{
    tS32               Vs32ErrorCode = PDD_OK;
#if PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM
    void*              VvpReadBuffer = NULL;
    tU8                VubDataStream;
    tU32               Vu32Size;
    tU32               Vu32Offset;

    if ((PtsDataStreamName != NULL) && (PpvReadBuffer != NULL))
    {
        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_DATA_STREAM_NAME, PtsDataStreamName, (tU32)strlen(PtsDataStreamName) + 1);
        /*check stream*/
        VubDataStream = PDD_u8GetCheckDataStreamDefinedHeaderEmmc(PtsDataStreamName, vtspPddRaweMMCInfo);
        /* check if name in configuration*/
        if (VubDataStream == 0xff)
        {/*error in generate header for PDD?: if defined location for NOR_NOR (Datapool or PDD),
             header of PDD gets the name for this stream */
            tChar  Vs8Buffer[200];
            Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_DATA_STREAM_NOT_DEFINED;
            snprintf(Vs8Buffer, sizeof(Vs8Buffer) - 1, "0x%04x: %d stream:'%s'", Vs32ErrorCode, Vs32ErrorCode, PtsDataStreamName);
            PDD_TRACE(PDD_ERROR_STR, Vs8Buffer, (tU32)strlen(Vs8Buffer) + 1);
        }
        else
        { /* read data stream configuration */
            if (PDD_bReadDataStreamConfigurationEmmc(vtspPddRaweMMCInfo) == FALSE)
            {/*configuration wrong number > PDD_RAW_EMMC_USER_CONFIG_MAX_NUMBER_DATASTREAM*/
                Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_CONFIGURATION_HEADER;
                PDD_TRACE(PDD_ERROR, &Vs32ErrorCode, sizeof(tS32));
                PDD_SET_ERROR_ENTRY("RAW_EMMC_USER: error configuration header");
            }
            else
            {/*get size and offset*/
                Vu32Size = vtspPddRaweMMCInfo->sHeaderDataStream[VubDataStream].u32Lenght;
                Vu32Offset = vtspPddRaweMMCInfo->sHeaderDataStream[VubDataStream].u32Offset;
                /*check length*/
                if (Ps32SizeReadBuffer < (tS32)Vu32Size)
                {
                    Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_WRONG_SIZE;
                }
                else
                { /* allocate buffer*/
                    VvpReadBuffer = malloc(vtspPddRaweMMCInfo->u32ClusterSizeData);
                    /* check buffer*/
                    if (VvpReadBuffer == NULL)
                    { /*system: no space available*/
                        Vs32ErrorCode = PDD_ERROR_NO_BUFFER;
                        PDD_TRACE(PDD_ERROR, &Vs32ErrorCode, sizeof(tS32));
                        PDD_FATAL_M_ASSERT_ALWAYS();
                    }
                    else
                    { /* read complete cluster data*/
                        if (PDD_bReadFromClusterEmmc(vtspPddRaweMMCInfo, VvpReadBuffer) == FALSE)
                        {
                            tChar  Vs8Buffer[200];
                            Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_READ_FROM_CLUSTER;
                            snprintf(Vs8Buffer, sizeof(Vs8Buffer) - 1, "0x%04x: %d stream:'%s'", Vs32ErrorCode, Vs32ErrorCode, PtsDataStreamName);
                            PDD_TRACE(PDD_ERROR_STR, Vs8Buffer, (tU32)strlen(Vs8Buffer) + 1);
                        }
                        else
                        { /* copy data to buffer*/
                            tU8* Vu8pSource = ((tU8*)VvpReadBuffer) + Vu32Offset;
                            memmove(PpvReadBuffer, Vu8pSource, Vu32Size);
                            /* set size*/
                            Vs32ErrorCode = (tS32)Vu32Size;
                        }
                        /* set buffer to free*/
                        free(VvpReadBuffer);
                    }/*end else chechk pointer*/
                }/*end else*/
            }
        }
    }
    else
        Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_INVALID_INPUT;
#else 
    Vs32ErrorCode = PDD_ERROR_NOT_SUPPORTED;
    PDD_PARAMETER_INTENTIONALLY_UNUSED(PtsDataStreamName);
    PDD_PARAMETER_INTENTIONALLY_UNUSED(PpvReadBuffer);
    PDD_PARAMETER_INTENTIONALLY_UNUSED(Ps32SizeReadBuffer);
#endif
    return(Vs32ErrorCode);
}
/******************************************************************************
* FUNCTION: PDD_eMMCUserAccessWriteDataStream()
*
* DESCRIPTION: function for write to nor nor
*
* PARAMETERS:
*      PtsDataStreamName: name of the data stream
*
* RETURNS:
*      positive value: PDD_OK
*      negative value: error code
*
* HISTORY:Created  2013 04 23
*****************************************************************************/
tS32 PDD_eMMCUserAccessWriteDataStream(const char* PtsDataStreamName, void *Ppu8WriteBuffer, tS32 Ps32SizeWriteBuffer)
{
    tS32               Vs32ErrorCode = PDD_OK;
#if PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM
    tU8                VubDataStream;

    if ((PtsDataStreamName != NULL) && (Ppu8WriteBuffer != NULL))
    {
        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_DATA_STREAM_NAME, PtsDataStreamName, (tU32)strlen(PtsDataStreamName) + 1);
        /*check stream*/
        VubDataStream = PDD_u8GetCheckDataStreamDefinedHeaderEmmc(PtsDataStreamName, vtspPddRaweMMCInfo);
        /* check if name in configuration*/
        if (VubDataStream == 0xff)
        {/*error in generate header for PDD?: if defined location for NOR_NOR (Datapool or PDD),
           header of PDD gets the name for this stream */
            tChar  Vs8Buffer[200];
            Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_DATA_STREAM_NOT_DEFINED;
            snprintf(Vs8Buffer, sizeof(Vs8Buffer) - 1, "0x%04x: %d stream:'%s'", Vs32ErrorCode, Vs32ErrorCode, PtsDataStreamName);
            PDD_TRACE(PDD_ERROR_STR, Vs8Buffer, (tU32)strlen(Vs8Buffer) + 1);
        }
        else
        { /* allocate buffer*/
            void* VpWriteBuffer = malloc(vtspPddRaweMMCInfo->u32ClusterSizeData);
            /* check buffer*/
            if (VpWriteBuffer == NULL)
            { /*system: no space available*/
                Vs32ErrorCode = PDD_ERROR_NO_BUFFER;
                PDD_TRACE(PDD_ERROR, &Vs32ErrorCode, sizeof(tS32));
                PDD_FATAL_M_ASSERT_ALWAYS();
            }
            else
            {
                tU32 Vu32Offset;
                /* read data stream configuration */
                if (PDD_bReadDataStreamConfigurationEmmc(vtspPddRaweMMCInfo) == FALSE)
                {/*configuration wrong number > PDD_NOR_USER_CONFIG_MAX_NUMBER_DATASTREAM*/
                    Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_CONFIGURATION_HEADER;
                    PDD_TRACE(PDD_ERROR, &Vs32ErrorCode, (tU32)sizeof(tS32));
                    PDD_SET_ERROR_ENTRY("RAW_EMMC_USER: error configuration header");
                }
                else
                {/* copy old buffer to bufferwrite oldbuffer must be set if read cluster!! */
                    memmove(VpWriteBuffer, &vtspPddRaweMMCInfo->u8pBufferDataOld[0], vtspPddRaweMMCInfo->u32ClusterSizeData);
                    /*update header if need */
                    Vs32ErrorCode = PDD_s32UpdateHeaderDataStreamEmmc(vtspPddRaweMMCInfo, Ps32SizeWriteBuffer, VubDataStream);
                    if (Vs32ErrorCode == PDD_RAW_EMMC_USER_UPDATE_HEADER)
                    {/* update data*/
                        PDD_vUpdateDataEmmc(vtspPddRaweMMCInfo, VpWriteBuffer);
                        /* memmove new header into write buffer*/
                        memmove(VpWriteBuffer, &vtspPddRaweMMCInfo->sHeaderDataStream[0], PDD_RAW_EMMC_USER_HEADER_SIZE_DATA_STREAM);
                        /*set error code to OK*/
                        Vs32ErrorCode = PDD_OK;
                    }
                    /*check error*/
                    if (Vs32ErrorCode == PDD_OK)
                    {/* write data into buffer */
                        tU8* Vu8pWriteBuffer = (tU8*)VpWriteBuffer;
                        Vu32Offset = vtspPddRaweMMCInfo->sHeaderDataStream[VubDataStream].u32Offset;
                        memmove(Vu8pWriteBuffer + Vu32Offset, Ppu8WriteBuffer, (size_t)Ps32SizeWriteBuffer);
                        /* compare old with new data */
                        if (memcmp(VpWriteBuffer, &vtspPddRaweMMCInfo->u8pBufferDataOld[0], vtspPddRaweMMCInfo->u32ClusterSizeData) != 0)
                        { /* increment counter: write cycle for the datastream*/
                            tsPddRaweMMCUserHeaderDataStream* VtsHeaderStream = (tsPddRaweMMCUserHeaderDataStream*)VpWriteBuffer;
                            VtsHeaderStream[VubDataStream].u32WriteCount++;
                            vtspPddRaweMMCInfo->sHeaderDataStream[VubDataStream].u32WriteCount++;
                            /* save  buffer*/
                            if ((PDD_bWriteToClusterEmmc(vtspPddRaweMMCInfo, VpWriteBuffer)) == FALSE)
                            {/* error */
                                tChar  Vs8Buffer[200];
                                Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_WRITE_TO_CLUSTER;
                                snprintf(Vs8Buffer, sizeof(Vs8Buffer) - 1, "0x%04x: %d stream:'%s'", Vs32ErrorCode, Vs32ErrorCode, PtsDataStreamName);
                                PDD_TRACE(PDD_ERROR_STR, Vs8Buffer, (tU32)strlen(Vs8Buffer) + 1);
                            }
                            else
                            { /* save data into old buffer*/
                                memmove(&vtspPddRaweMMCInfo->u8pBufferDataOld[0], VpWriteBuffer, vtspPddRaweMMCInfo->u32ClusterSizeData);
                                /*get size write */
                                Vs32ErrorCode = Ps32SizeWriteBuffer;
                            }
                        }
                    }
                }/*end else check configuration*/
                free(VpWriteBuffer);
            }/*end else check pointer*/
        }/*end else check datastream*/
    }
    else
        Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_INVALID_INPUT;
#else
    Vs32ErrorCode = PDD_ERROR_NOT_SUPPORTED;
    PDD_PARAMETER_INTENTIONALLY_UNUSED(PtsDataStreamName);
    PDD_PARAMETER_INTENTIONALLY_UNUSED(Ppu8WriteBuffer);
    PDD_PARAMETER_INTENTIONALLY_UNUSED(Ps32SizeWriteBuffer);
#endif        
    return(Vs32ErrorCode);
}

/******************************************************************************
* FUNCTION: PDD_NorUserAccessReadDataStreamEarly()
*
* DESCRIPTION: function for read a data stream early with fast access
*              (without check of whole cluster)
*
* PARAMETERS:
*      PtsDataStreamName:  name of the data stream
*      Ppu8ReadBuffer:     read buffer
*      Ps32SizeReadBuffer: size of read buffer
*
* RETURNS:
*      positive value: PDD_OK
*      negative value: error code
*
* HISTORY:Created  2015 10 12
*****************************************************************************/
tS32  PDD_RaweMMCAccessReadDataStreamEarly(const char* PtsDataStreamName, void *PvpReadBuffer, tS32 Ps32SizeReadBuffer)
{
    tS32               Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_READ_FROM_CLUSTER;
#if PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM
    tU8                VubDataStream;
    tS32               Vs32FoundCluster = -1;

    if ((PtsDataStreamName != NULL) && (PvpReadBuffer != NULL))
    {
        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_DATA_STREAM_NAME, PtsDataStreamName, (tU32)strlen((tChar*)PtsDataStreamName) + 1);
        /*check stream defined*/
        VubDataStream = PDD_u8GetCheckDataStreamDefinedHeaderEmmc(PtsDataStreamName, vtspPddRaweMMCInfo);

        if (VubDataStream == 0xff)
        { /*error in generate header for PDD?: if defined location for NOR_NOR (Datapool or PDD),
            header of PDD gets the name for this stream */
            tChar Vs8Buffer[200];
            Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_DATA_STREAM_NOT_DEFINED;
            snprintf(Vs8Buffer, sizeof(Vs8Buffer) - 1, "0x%04x: %d stream:'%s'", Vs32ErrorCode, Vs32ErrorCode, PtsDataStreamName);
            PDD_TRACE(PDD_ERROR_STR, Vs8Buffer, (tU32)strlen(Vs8Buffer) + 1);
        }
        else
        { /*find valid cluster*/
            Vs32FoundCluster = PDD_s32FindValidClusterEmmc(vtspPddRaweMMCInfo);

            if (Vs32FoundCluster >= 0)
            {
                /*read checksum*/
                tsPddRAWeMMCUserHeaderChecksum     VtsHeaderChecksum;
                memset(&VtsHeaderChecksum, 0, sizeof(tsPddRAWeMMCUserHeaderChecksum));
                tBool VbResult = PDD_bReadEmmc(vtspPddRaweMMCInfo, (tU32)Vs32FoundCluster, &VtsHeaderChecksum, (tU32)PDD_RAW_EMMC_USER_CHECKSUM_POSITION, (tU32)sizeof(VtsHeaderChecksum));
                if (VbResult && (VtsHeaderChecksum.u32CheckSum == (~VtsHeaderChecksum.u32CheckSumXor)))
                { //read first datastream header entry to get the size of the header
                    tsPddRaweMMCUserHeaderDataStream VsHeaderDataStreamFirst;
                    if (PDD_bReadEmmc(vtspPddRaweMMCInfo, (tU32)Vs32FoundCluster, &VsHeaderDataStreamFirst, (tU32)PDD_RAW_EMMC_USER_HEADER_SIZE, (tU32)sizeof(tsPddRaweMMCUserHeaderDataStream)) == TRUE)
                    { /*get number of read datastreams (=first offset/size for one header)*/
                        tU8 Vu8NumberDataStreamRead = (tU8)(VsHeaderDataStreamFirst.u32Offset / (sizeof(tsPddRaweMMCUserHeaderDataStream)));
                        //determine size of header data stream
                        tS32 Vs32Size = ((tS32)sizeof(tsPddRaweMMCUserHeaderDataStream)*Vu8NumberDataStreamRead); //size greater as in defined configuration
                          /* allocate buffer*/
                        void* VvpBufRead = (void *)malloc((size_t)Vs32Size);
                        /* check buffer*/
                        if (VvpBufRead == NULL)
                        { /*system: no space available*/
                            Vs32ErrorCode = PDD_ERROR_NO_BUFFER;
                            PDD_TRACE(PDD_ERROR, &Vs32ErrorCode, sizeof(tS32));
                        }
                        else
                        { /* read header datastream*/
                            if (PDD_bReadEmmc(vtspPddRaweMMCInfo, (tU32)Vs32FoundCluster, VvpBufRead, (tU32)PDD_RAW_EMMC_USER_HEADER_SIZE, (tU32)Vs32Size) == TRUE)
                            { //check stream available
                                tsPddRaweMMCUserHeaderDataStream* VspHeaderDataStreamRead;
                                tU8                               Vu8StreamReadPos;

                                VspHeaderDataStreamRead = (tsPddRaweMMCUserHeaderDataStream*)VvpBufRead;
                                Vu8StreamReadPos = PDD_u8GetCheckDataStreamReadHeaderEmmc((char*)PtsDataStreamName, VspHeaderDataStreamRead);
                                /*check if datastream exist*/
                                if (Vu8StreamReadPos != 0xff)
                                { //get offset 
                                    tU32 Vu32OffsetRead = VspHeaderDataStreamRead[Vu8StreamReadPos].u32Offset;
                                    tU32 Vu32LenghtRead = VspHeaderDataStreamRead[Vu8StreamReadPos].u32Lenght;
                                    //check lenght
                                    if (Vu32LenghtRead > (tU32)Ps32SizeReadBuffer)
                                        Vu32LenghtRead = (tU32)Ps32SizeReadBuffer;
                                    //read offset
                                    if (PDD_bReadEmmc(vtspPddRaweMMCInfo, (tU32)Vs32FoundCluster, PvpReadBuffer, (tU32)PDD_RAW_EMMC_USER_HEADER_SIZE + Vu32OffsetRead, Vu32LenghtRead) == TRUE)
                                    {
                                        Vs32ErrorCode = (tS32)Vu32LenghtRead;
                                    }
                                }
                            }
                        }
                        free(VvpBufRead);
                    }
                }
            }
        }
        if (Vs32ErrorCode < PDD_OK)
        {
            tChar  Vs8Buffer[200];
            snprintf(Vs8Buffer, sizeof(Vs8Buffer) - 1, "0x%04x: %d stream:'%s'", Vs32ErrorCode, Vs32ErrorCode, PtsDataStreamName);
            PDD_TRACE(PDD_ERROR_STR, Vs8Buffer, (tU32)strlen(Vs8Buffer) + 1); ;
        }
    }
    else
        Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_INVALID_INPUT;
#else
    Vs32ErrorCode = PDD_ERROR_NOT_SUPPORTED;
    PDD_PARAMETER_INTENTIONALLY_UNUSED(PtsDataStreamName);
    PDD_PARAMETER_INTENTIONALLY_UNUSED(PvpReadBuffer);
    PDD_PARAMETER_INTENTIONALLY_UNUSED(Ps32SizeReadBuffer);
#endif
    return(Vs32ErrorCode);
}
/******************************************************************************
* FUNCTION: PDD_eMMCUserAccessGetEraseCounter()
*
* DESCRIPTION: get erase sector count
*
* PARAMETERS:
*
* RETURNS:
*      count
*
* HISTORY:Created  2013 08 23
*****************************************************************************/
tU32 PDD_eMMCUserAccessGetWriteCounter(void)
{
    return(vtspPddRaweMMCInfo->u32ClusterUpdatedCount);
}

/******************************************************************************
* FUNCTION: PDD_s32FindValidClusterEmmc()
*
* DESCRIPTION: find valid cluster
*
* PARAMETERS:
*        PtsDevInfo : pointer of the device info
*
* RETURNS:
*      the number of the vaild cluster, -1 means no cluster is valid
*
* HISTORY:Created  2013 04 23
*****************************************************************************/
static tS32   PDD_s32FindValidClusterEmmc(tsPddRaweMMCUserInfo* PtsDevInfo)
{
    tS32   Vs32FoundCluster = -1;
    tU8    Vu8FoundClusterIsNew = FALSE;
    tBool  VbIsNew = FALSE;
    tU8    Vu8Inc;
    if (PtsDevInfo != NULL)
    {
        tU32   Vu32NumCluster = PtsDevInfo->u32NumCluster;
        /* for all cluster, find a valid cluster*/
        for (Vu8Inc = 0; Vu8Inc < Vu32NumCluster; ++Vu8Inc)
        {
            if (PDD_bIsClusterValidEmmc(PtsDevInfo, Vu8Inc, &VbIsNew))
            {
                if (Vs32FoundCluster >= 0)
                {/* take sector only if it is new than the found one */
                    if (!Vu8FoundClusterIsNew && VbIsNew)
                    {
                        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_CLUSTER_REPLACE_OLD, &Vs32FoundCluster, (tU32)sizeof(Vs32FoundCluster));
                        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_CLUSTER_REPLACE_NEW, &Vu8Inc, (tU32)sizeof(Vu8Inc));
                        Vu8FoundClusterIsNew = VbIsNew;
                        Vs32FoundCluster = Vu8Inc;
                    }
                }
                else
                {/* first valid sector found */
                    Vu8FoundClusterIsNew = VbIsNew;
                    Vs32FoundCluster = Vu8Inc;
                }
                if (Vu8FoundClusterIsNew)
                {/* more than one new cluster cannot be found */
                    break;
                }
            }
        }
        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_CLUSTER_TO_READ, &Vs32FoundCluster, (tU32)sizeof(Vs32FoundCluster));
        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_CLUSTER_TO_READ_STATUS, &Vu8FoundClusterIsNew, (tU32)sizeof(Vu8FoundClusterIsNew));
    }
    return(Vs32FoundCluster);
}
/******************************************************************************
* FUNCTION: PDD_bIsClusterValidEmmc()
*
* DESCRIPTION: check if cluster valid
*
* PARAMETERS:
*        PtsDevInfo   : pointer of the device info
*        PtClusterNum : the number of the cluster, 0 <= cluster < u32NumCluster
*        PpbIsNew     : pointer to bool whether block has flags as "new"
*
* RETURNS:
*      TRUE => cluster is valid
*
* HISTORY:Created  2013 04 23
*****************************************************************************/
static tBool  PDD_bIsClusterValidEmmc(tsPddRaweMMCUserInfo* PtsDevInfo, tU32 u32ClusterNum, tBool* PpbIsNew)
{
    tBool  VbReturn = FALSE;
    tU8    VubHeader[PDD_RAW_EMMC_USER_HEADER_SIZE];

    if ((PtsDevInfo != NULL) && (PpbIsNew != NULL))
    {/*read header*/
        memset(&VubHeader[0], 0, (size_t)PDD_RAW_EMMC_USER_HEADER_SIZE);

        if (!PDD_bReadEmmc(PtsDevInfo, u32ClusterNum, &VubHeader[0], (tU32)PDD_RAW_EMMC_USER_MAGIC_POSITION, (tU32)PDD_RAW_EMMC_USER_HEADER_SIZE))
        {/* error read */
            VbReturn = FALSE;
        }
        else
        {
            tsPddRAWeMMCUserHeaderMagic*  VtspHeaderMagic = (tsPddRAWeMMCUserHeaderMagic*)(void*)&VubHeader[PDD_RAW_EMMC_USER_MAGIC_POSITION];
            /* get state new */
            *PpbIsNew = ((VubHeader[PDD_RAW_EMMC_USER_STATUS_POSITION_OLD_VERSION] & (tU8)(PDD_RAW_EMMC_USER_STATE_OLD_VERSION)) != 0);
            /* if magic correct*/
            if ((VtspHeaderMagic->u16ChunkSize == CHIP_ALIGNMENT) && (VtspHeaderMagic->u16MagicLow == PDD_RAW_EMMC_USER_MAGIC))
            { /* get erase sector count*/
                PtsDevInfo->u32ClusterUpdatedCount = VtspHeaderMagic->u32ClusterUpdatedCount;
                PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_WRITE_COUNTER, &PtsDevInfo->u32ClusterUpdatedCount, sizeof(tU32));
                /*check state*/
                if ((VubHeader[PDD_RAW_EMMC_USER_STATUS_POSITION_WRITE_BEGIN] == (tU8)(~PDD_RAW_EMMC_USER_STATE_WRITE_BEGIN))
                    && (VubHeader[PDD_RAW_EMMC_USER_STATUS_POSITION_WRITE_END] == (tU8)(~PDD_RAW_EMMC_USER_STATE_WRITE_END))
                    && (VubHeader[PDD_RAW_EMMC_USER_STATUS_POSITION_INVALID] != (tU8)(~PDD_RAW_EMMC_USER_STATE_INVALID)))
                { /*cluster valid*/
                    VbReturn = TRUE;
                }
            }
            else
            {
                VbReturn = FALSE;
            }
        }
    }
    return(VbReturn);
}

/******************************************************************************
* FUNCTION: PDD_bReadEmmc()
*
* DESCRIPTION: read from flash
*
* PARAMETERS:
*     PtsDevInfo    : pointer of the device info
*     Ps32Cluster   : the number of the cluster, 0 <= cluster < u32NumCluster
*     PvpDestAdress : address to read the byte
*     Pu32Offset    : offset in the cluster
*     Pu32Len       : byte to read
*
* RETURNS:
*      TRUE => all OK
*
* HISTORY:Created  2013 04 23
*****************************************************************************/
static tBool  PDD_bReadEmmc(tsPddRaweMMCUserInfo* PtsDevInfo, tU32 Pu32Cluster, void* PvpDestAdress, tU32 Pu32Offset, tU32 Pu32Len)
{
    tBool VbReturn = FALSE;
    if (PtsDevInfo != NULL)
    {
        tU32  Vu32NumCluster = PtsDevInfo->u32NumCluster;
        tU32  Vu32ClusterSize = PtsDevInfo->u32ClusterSize;

        /*check cluster range*/
        if (Pu32Cluster < Vu32NumCluster)
        {
            if (M_PDD_RAW_EMMC_ACCESS_READ(RaweMMCAccessHdl, PvpDestAdress, Pu32Len, ((Pu32Cluster * Vu32ClusterSize) + Pu32Offset)) == TRUE)
            { /*success*/
                VbReturn = TRUE;
            }
            if (!VbReturn)
            {
                PDD_SET_ERROR_ENTRY("RAW_EMMC_USER: read from eMMC fails");
            }
        }
    }
    return(VbReturn);
}
/******************************************************************************
* FUNCTION: PDD_bWriteEmmc()
*
* DESCRIPTION: write to flash
*
* PARAMETERS:
*   PtsDevInfo      : pointer of the driver info
*   Ps32Cluster     : the number of the cluster, 0 <= cluster < u32NumCluster
*   Pu8SourceAdress : address to write from
*   Pu32Offset      : offset in the cluster
*   Pu32Len         : byte to write
*
* RETURNS:
*      TRUE => all OK
*
* HISTORY:Created  2013 04 23
*****************************************************************************/
static tBool PDD_bWriteEmmc(tsPddRaweMMCUserInfo* PtsDevInfo, tS32 Ps32Cluster, tU8 *Pu8SourceAdress, tU32 Pu32Offset, tU32 Pu32Len)
{
    tBool VbReturn = FALSE;
    tChar    Vs8Buffer[128];

    if (PtsDevInfo != NULL)
    {
        if (Ps32Cluster < (tS32)PtsDevInfo->u32NumCluster)
        {
            tU8* VubpWriteBuffer = (tU8*)malloc((size_t)(ALIGN_TO_CHIP(Pu32Len)));
            if (VubpWriteBuffer == NULL)
            { /*system: no space available*/
                tS32 Vs32ErrorCode = PDD_ERROR_NO_BUFFER;
                PDD_TRACE(PDD_ERROR, &Vs32ErrorCode, (tU32)sizeof(tS32));
                PDD_FATAL_M_ASSERT_ALWAYS();
            }
            else
            {
                memset(VubpWriteBuffer, 0x00, (size_t)ALIGN_TO_CHIP(Pu32Len));
                memmove(VubpWriteBuffer, Pu8SourceAdress, (size_t)Pu32Len);
                /*write to flash*/
                if (M_PDD_RAW_EMMC_ACCESS_WRITE(RaweMMCAccessHdl, VubpWriteBuffer, (tU32)ALIGN_TO_CHIP(Pu32Len), (((tU32)Ps32Cluster * PtsDevInfo->u32ClusterSize) + (Pu32Offset))) == TRUE)
                {
                    VbReturn = TRUE;
                }
                free(VubpWriteBuffer);
                /*check error */
                if (VbReturn == FALSE)
                { /* LFX_write failed */
                    snprintf(Vs8Buffer, sizeof(Vs8Buffer), "RAW_EMMC_USER: write to eMMC fails errno %d", errno);
                    PDD_SET_ERROR_ENTRY(&Vs8Buffer[0]);
                }
            }
        }
    }
    return(VbReturn);
}

/******************************************************************************
* FUNCTION: PDD_bEraseEmmc()
*
* DESCRIPTION: write to Emmc
*
* PARAMETERS:
*
* RETURNS:
*       TRUE => cluster is filled with 0x00
*
* HISTORY:Created by selvan krishnan 2018 04 23
*****************************************************************************/
static tBool  PDD_bEraseEmmc(void)
{
    tBool   VbReturn = FALSE;
    tChar   Vs8Buffer[128];

    tU8* VubpWriteBuffer = (tU8*)malloc(PDD_RAW_EMMC_USER_CLUSTER_SIZE * PDD_RAW_EMMC_USER_NO_OF_CLUSTER);
    if (VubpWriteBuffer != NULL)
    {
        memset(VubpWriteBuffer, 0x00, (PDD_RAW_EMMC_USER_CLUSTER_SIZE * PDD_RAW_EMMC_USER_NO_OF_CLUSTER));
        /*write to flash*/
        if (M_PDD_RAW_EMMC_ACCESS_WRITE(RaweMMCAccessHdl, VubpWriteBuffer, (PDD_RAW_EMMC_USER_CLUSTER_SIZE * PDD_RAW_EMMC_USER_NO_OF_CLUSTER), 0) == TRUE)
        {
            VbReturn = TRUE;
        }
        free(VubpWriteBuffer);
        /*check error */
        if (VbReturn == FALSE)
        { /* eMMC_write failed */
            snprintf(Vs8Buffer, sizeof(Vs8Buffer), "RAW_EMMC_USER: Erase to eMMC fails errno %d", errno);
            PDD_SET_ERROR_ENTRY(&Vs8Buffer[0]);
        }
    }
    return(VbReturn);
}

/******************************************************************************
* FUNCTION: PDD_u8GetCheckDataStreamDefinedHeaderEmmc()
*
* DESCRIPTION: check the datastream name and get the table number in the defined header
*
* PARAMETERS:
*   PtsDevInfo: pointer data stream
*
* RETURNS:
*       table number
*
* HISTORY:Created  2013 04 25
*****************************************************************************/
static tU8    PDD_u8GetCheckDataStreamDefinedHeaderEmmc(const char* PtsDataStreamName, tsPddRaweMMCUserInfo* PtsDevInfo)
{
    tU8                               Vu8DataStream = 0xff;
#if PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM 
    tU8                               Vu8Inc;
    char*                             VstrPosFolder;
    tsPddRaweMMCUserHeaderDataStream* VspHeaderDataStream;

    if ((PtsDataStreamName != NULL) && (PtsDevInfo != NULL))
    {
        VstrPosFolder = strrchr((char*)PtsDataStreamName, '/'); //compare without folder
        /*check datastream with folder */
        if (VstrPosFolder != NULL)
        {
            PtsDataStreamName = VstrPosFolder + 1;
        }
        /*trace out header */
        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_DATA_STREAM_HEADER, &PtsDevInfo->sHeaderDataStream[0], PDD_RAW_EMMC_USER_HEADER_SIZE_DATA_STREAM);
        /* for each datastream*/
#ifndef PDD_TESTMANGER_ACTIVE
        for (Vu8Inc = 0; Vu8Inc < PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM; Vu8Inc++)
#else
        for (Vu8Inc = 0; Vu8Inc < PDD_RAW_EMMC_USER_CONFIG_MAX_NUMBER_DATASTREAM; Vu8Inc++)
#endif
        {
            VspHeaderDataStream = &PtsDevInfo->sHeaderDataStream[Vu8Inc];
            if (strncmp((tChar *)VspHeaderDataStream->u8DataStreamName, PtsDataStreamName, PDD_RAW_EMMC_MAX_SIZE_NAME) == 0)
            {/*filename are identical*/
                Vu8DataStream = Vu8Inc;
                break;
            }
        }
    }
#else
    PDD_PARAMETER_INTENTIONALLY_UNUSED(PtsDataStreamName);
    PDD_PARAMETER_INTENTIONALLY_UNUSED(PtsDevInfo);
#endif
    return(Vu8DataStream);
}

/******************************************************************************
* FUNCTION: PDD_bReadDataStreamConfigurationEmmc()
*
* DESCRIPTION: read data configuration from eMMC
*
* PARAMETERS:
*   PtsDevInfo: pointer of the driver info
*
* RETURNS:
*       TRUE => configuration read success
*
* HISTORY:Created  2013 04 25
*****************************************************************************/
tBool PDD_bReadDataStreamConfigurationEmmc(tsPddRaweMMCUserInfo* PtsDevInfo)
{
    tBool                              VbReturn = TRUE;
#if PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM
    tU8                                Vu8Inc;
    tU8                                Vu8StreamReadPos;
    if (PtsDevInfo != NULL)
    {
        void*                              VvpBufRead = NULL;
        tsPddRaweMMCUserHeaderDataStream*  VspHeaderDataStreamDefined = &PtsDevInfo->sHeaderDataStream[0];
        tsPddRaweMMCUserHeaderDataStream*  VspHeaderDataStreamRead;

        if (VspHeaderDataStreamDefined->u32Offset == 0)
        { /* offset and lenght not set =>read from flash*/
          /* allocate buffer*/
            if (vtspPddRaweMMCInfo != NULL)
            {
                VvpBufRead = (void *)malloc(vtspPddRaweMMCInfo->u32ClusterSizeData);
            }
            /* check buffer*/
            if (VvpBufRead == NULL)
            { /*system: no space available*/
                tS32 Vs32ErrorCode = PDD_ERROR_NO_BUFFER;
                PDD_TRACE(PDD_ERROR, &Vs32ErrorCode, sizeof(tS32));
                PDD_FATAL_M_ASSERT_ALWAYS();
            }
            else
            { /* read complete cluster data*/
                memset(VvpBufRead, 0, vtspPddRaweMMCInfo->u32ClusterSizeData);
                if (PDD_bReadFromClusterEmmc(PtsDevInfo, VvpBufRead) == TRUE)
                {
                    tU32  Vu32Offset = 0;
                    tU32  Vu32OffsetRead = 0;
                    /*get the header of the read datastream */
                    VspHeaderDataStreamRead = (tsPddRaweMMCUserHeaderDataStream*)VvpBufRead;
                    /*for each defined datastream */
#ifdef PDD_TESTMANGER_ACTIVE
                    if (PDD_bGetNumberDataStreamTmAndNameDataStream((tsPddRaweMMCUserHeaderDataStream*)VvpBufRead) <= 0)
                    {
                        VbReturn = FALSE;
                    }
                    else
                    {
                        for (Vu8Inc = 0; Vu8Inc < PDD_RAW_EMMC_USER_NUMBER_DATASTREAM_TM; Vu8Inc++)
#else        
                    for (Vu8Inc = 0; Vu8Inc < PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM; Vu8Inc++)
#endif
                    {/*check if exist in read datastream and get the position*/
                        VspHeaderDataStreamDefined = &PtsDevInfo->sHeaderDataStream[Vu8Inc];
                        Vu8StreamReadPos = PDD_u8GetCheckDataStreamReadHeaderEmmc((char*)VspHeaderDataStreamDefined->u8DataStreamName, (tsPddRaweMMCUserHeaderDataStream*)VvpBufRead);
                        /*check if datastream exist*/
                        if (Vu8StreamReadPos != 0xff)
                        { /*datastream exists*/
                            tU8* Vu8pBufRead = (tU8*)VvpBufRead;
                            /*get lenght, count and offset*/
                            Vu32OffsetRead = VspHeaderDataStreamRead[Vu8StreamReadPos].u32Offset;
                            VspHeaderDataStreamDefined->u32Lenght = VspHeaderDataStreamRead[Vu8StreamReadPos].u32Lenght;
                            VspHeaderDataStreamDefined->u32Offset = (tU32)PDD_RAW_EMMC_USER_HEADER_SIZE_DATA_STREAM + Vu32Offset;
                            VspHeaderDataStreamDefined->u32WriteCount = VspHeaderDataStreamRead[Vu8StreamReadPos].u32WriteCount;
                            /*copy read data into old buffer*/
                            memmove(&PtsDevInfo->u8pBufferDataOld[VspHeaderDataStreamDefined->u32Offset], (Vu8pBufRead + Vu32OffsetRead), VspHeaderDataStreamDefined->u32Lenght);
                            /*get new offset*/
                            Vu32Offset += VspHeaderDataStreamDefined->u32Lenght;
                        }
                        else
                        {/*datastream doesn't exist*/
                            VspHeaderDataStreamDefined->u32Lenght = 0;
                            VspHeaderDataStreamDefined->u32Offset = (tU32)PDD_RAW_EMMC_USER_HEADER_SIZE_DATA_STREAM + Vu32Offset;
                            VspHeaderDataStreamDefined->u32WriteCount = 0;
                        }
                    }/*end for*/
#ifdef PDD_TESTMANGER_ACTIVE
                    }/*end else*/
#endif
                }/*end if*/
            else
            { /* no data saved => set to default*/
            /* for each datastream*/
#ifndef PDD_TESTMANGER_ACTIVE
                for (Vu8Inc = 0; Vu8Inc < PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM; Vu8Inc++)
#else
                vu8NumberDataStreamTm = PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM_TM;
                for (Vu8Inc = 0; Vu8Inc < PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM_TM; Vu8Inc++)
#endif
                {
                    VspHeaderDataStreamDefined = &PtsDevInfo->sHeaderDataStream[Vu8Inc];
                    VspHeaderDataStreamDefined->u32Lenght = 0;
                    VspHeaderDataStreamDefined->u32Offset = PDD_RAW_EMMC_USER_HEADER_SIZE_DATA_STREAM;
                    VspHeaderDataStreamDefined->u32WriteCount = 0;
                }
            }
            /* memmove new header into old buffer*/
            memmove(&PtsDevInfo->u8pBufferDataOld[0], &PtsDevInfo->sHeaderDataStream[0], PDD_RAW_EMMC_USER_HEADER_SIZE_DATA_STREAM);
            /* check if data new => header changed*/
            if (memcmp(&PtsDevInfo->u8pBufferDataOld[0], VvpBufRead, vtspPddRaweMMCInfo->u32ClusterSizeData) != 0)
            {/*save data*/
                if ((PDD_bWriteToClusterEmmc(vtspPddRaweMMCInfo, PtsDevInfo->u8pBufferDataOld)) == FALSE)
                {/* error */
                    tS32 Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_WRITE_TO_CLUSTER;
                    PDD_TRACE(PDD_ERROR, &Vs32ErrorCode, sizeof(tS32));
                }
            }
            free(VvpBufRead);
            }
        }/*end if offset unknown*/
    PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_DATA_STREAM_HEADER, &PtsDevInfo->sHeaderDataStream[0], PDD_RAW_EMMC_USER_HEADER_SIZE_DATA_STREAM);
    }
#else
    VbReturn = FALSE;
    PDD_PARAMETER_INTENTIONALLY_UNUSED(PtsDevInfo);
#endif
    return(VbReturn);
}

#if PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM
/******************************************************************************
* FUNCTION: PDD_bWriteToClusterEmmc()
*
* DESCRIPTION: write "data" to the next free cluster
*
* PARAMETERS:
*       PtsDevInfo   : device information
*       Ppu8Data     : pointer to the data (the size must be u32ClusterSizeData)
*
* RETURNS:
*      TRUE:  success
*      FALSE: failed
*
* HISTORY:Created  2013 04 23
*****************************************************************************/
static tBool  PDD_bWriteToClusterEmmc(tsPddRaweMMCUserInfo* PtsDevInfo, tU8* Ppu8Data)
{
    tU8       Vu8StateValue;
    tU32      Vu32CrcCheckSum;
    tU32      u32WriteCount = 0;
    tS32      Vs32ClusterNum = -1;
    tBool     VbResult = TRUE;

    if (PtsDevInfo != NULL)
    {
        /*find next valid cluster*/
        Vs32ClusterNum = PDD_s32FindValidClusterEmmc(PtsDevInfo);

        /* check if cluster found*/
        if (Vs32ClusterNum >= 0)
        { /*  valid cluster found*/
            if (Vs32ClusterNum == 0)
            {
                Vs32ClusterNum = 1;
            }
            else if (Vs32ClusterNum == 1)
            {
                Vs32ClusterNum = 0;
            }
        }
        else
        {/*  valid cluster not found*/
           /* take the first cluster */
            Vs32ClusterNum = 0;
        }
        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_CLUSTER_OLD, &PtsDevInfo->s32LastClusterUsed, sizeof(PtsDevInfo->s32LastClusterUsed));
        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_CLUSTER_NEW, &Vs32ClusterNum, sizeof(Vs32ClusterNum));
        /* if no error*/
        if (VbResult == TRUE)
        {/* set the old cluster to the old state */
            if (PtsDevInfo->s32LastClusterUsed >= 0 && PtsDevInfo->s32LastClusterUsed < (tS32)PtsDevInfo->u32NumCluster)
            {
                Vu8StateValue = (tU8)~PDD_RAW_EMMC_USER_STATE_OLD_VERSION;
                VbResult = PDD_bWriteEmmc(PtsDevInfo, PtsDevInfo->s32LastClusterUsed, &Vu8StateValue, (tU32)PDD_RAW_EMMC_USER_STATUS_POSITION_OLD_VERSION, 1);
                if (VbResult)
                {
                    VbResult = PDD_bReadEmmc(PtsDevInfo, (tU32)PtsDevInfo->s32LastClusterUsed, &u32WriteCount, (tU32)4, (tU32)sizeof(u32WriteCount));
                    if (VbResult)
                    {
                        if ((PtsDevInfo->s32LastClusterUsed == 0) && (u32WriteCount == 1))
                        {
                            PtsDevInfo->u32ClusterUpdatedCount = 1;
                        }
                        else
                        {
                            VbResult = PDD_bReadEmmc(PtsDevInfo, (tU32)Vs32ClusterNum, &u32WriteCount, (tU32)4, (tU32)sizeof(u32WriteCount));
                            if (VbResult)
                            {
                                PtsDevInfo->u32ClusterUpdatedCount = u32WriteCount + 1;
                            }
                        }
                    }
                }
            }
            else
            {
                PtsDevInfo->u32ClusterUpdatedCount = 1;
            }
            PDD_bUpdateMagicAndWriteCountofCluster(PtsDevInfo, Vs32ClusterNum);
            /* calc crc */
            Vu32CrcCheckSum = PDD_ValidationCalcCrc32(Ppu8Data, PtsDevInfo->u32ClusterSizeData);
            /*write checksum*/
            if (VbResult)
            {
                tsPddRAWeMMCUserHeaderChecksum VtsHeaderChecksum;
                /* put checksum and CRC to header */
                VtsHeaderChecksum.u32CheckSum = Vu32CrcCheckSum;
                VtsHeaderChecksum.u32CheckSumXor = ~Vu32CrcCheckSum;
                VbResult = PDD_bWriteEmmc(PtsDevInfo, Vs32ClusterNum, (tU8*)&VtsHeaderChecksum, (tU32)PDD_RAW_EMMC_USER_CHECKSUM_POSITION, (tU32)sizeof(VtsHeaderChecksum));
            }
            /*write state PDD_NOR_USER_STATE_WRITE_BEGIN */
            if (VbResult)
            {
                Vu8StateValue = (tU8)~PDD_RAW_EMMC_USER_STATE_WRITE_BEGIN;
                VbResult = PDD_bWriteEmmc(PtsDevInfo, Vs32ClusterNum, &Vu8StateValue, (tU32)PDD_RAW_EMMC_USER_STATUS_POSITION_WRITE_BEGIN, 1);
            }
            /* write the data stream */
            if (VbResult)
            {
                VbResult = PDD_bWriteEmmc(PtsDevInfo, Vs32ClusterNum, Ppu8Data, (tU32)PDD_RAW_EMMC_USER_HEADER_SIZE, PtsDevInfo->u32ClusterSizeData);
            }
            /* set the new cluster to "write end" state */
            if (VbResult)
            {
                Vu8StateValue = (tU8)~PDD_RAW_EMMC_USER_STATE_WRITE_END;
                VbResult = PDD_bWriteEmmc(PtsDevInfo, Vs32ClusterNum, &Vu8StateValue, (tU32)PDD_RAW_EMMC_USER_STATUS_POSITION_WRITE_END, 1);
            }
            /* set the new cluster to "valid" state */
            if (VbResult)
            {
                Vu8StateValue = (tU8)PDD_RAW_EMMC_USER_STATE_VALID;
                VbResult = PDD_bWriteEmmc(PtsDevInfo, Vs32ClusterNum, &Vu8StateValue, (tU32)PDD_RAW_EMMC_USER_STATUS_POSITION_OLD_VERSION, 1);
                VbResult = PDD_bWriteEmmc(PtsDevInfo, Vs32ClusterNum, &Vu8StateValue, (tU32)PDD_RAW_EMMC_USER_STATUS_POSITION_INVALID, 1);
            }

            /* if last cluster exist*/
            if (VbResult && (PtsDevInfo->s32LastClusterUsed >= 0) && (PtsDevInfo->s32LastClusterUsed < (tS32)PtsDevInfo->u32NumCluster))
            {/* set it to "invalid" */
                Vu8StateValue = (tU8)~PDD_RAW_EMMC_USER_STATE_INVALID;
                VbResult = PDD_bWriteEmmc(PtsDevInfo, PtsDevInfo->s32LastClusterUsed, &Vu8StateValue, (tU32)PDD_RAW_EMMC_USER_STATUS_POSITION_INVALID, 1);
            }
            if (VbResult)
            {/* set the new cluster as "last used" */
                PtsDevInfo->s32LastClusterUsed = Vs32ClusterNum;
            }
        }
        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_CLUSTER_WRITE_TO, &Vs32ClusterNum, (tU32)sizeof(Vs32ClusterNum));
        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_RETURN, &VbResult, (tU32)sizeof(VbResult));
    }
    else
        VbResult = FALSE;
    return(VbResult);
}
/******************************************************************************
* FUNCTION: PDD_bReadFromClusterEmmc()
*
* DESCRIPTION: read the data from the last valid cluster to "Ppu8Data"
*
* PARAMETERS:
*       PtsDevInfo   : device information
*       Ppu8Data     : pointer to the data (the size must be u32ClusterSizeData)
*
* RETURNS:
*      TRUE:  success
*      FALSE: failed
*
* HISTORY:Created  2013 04 23
*****************************************************************************/
static tBool  PDD_bReadFromClusterEmmc(tsPddRaweMMCUserInfo* PtsDevInfo, void* PvpData)
{
    tBool                          VbResult = TRUE;
    tU32                           Vu32CrcCheckSum;
    tS32                           Vs32FoundCluster = -1;
    if (PtsDevInfo != NULL)
    {
        tU32                           Vu32ClusterSizeData = PtsDevInfo->u32ClusterSizeData;
        tsPddRAWeMMCUserHeaderChecksum VtsHeaderChecksum;

        /*find valid cluster*/
        Vs32FoundCluster = PDD_s32FindValidClusterEmmc(PtsDevInfo);

        if (Vs32FoundCluster >= 0)
        { /*read checksum*/
            memset(&VtsHeaderChecksum, 0, sizeof(tsPddRAWeMMCUserHeaderChecksum));
            VbResult = PDD_bReadEmmc(PtsDevInfo, (tU32)Vs32FoundCluster, &VtsHeaderChecksum, (tU32)PDD_RAW_EMMC_USER_CHECKSUM_POSITION, (tU32)sizeof(VtsHeaderChecksum));
            if (VbResult && (VtsHeaderChecksum.u32CheckSum == (~VtsHeaderChecksum.u32CheckSumXor)))
            { /*read data from flash*/
                VbResult = PDD_bReadEmmc(PtsDevInfo, (tU32)Vs32FoundCluster, PvpData, (tU32)PDD_RAW_EMMC_USER_HEADER_SIZE, Vu32ClusterSizeData);
                if (VbResult)
                {/* calc crc */
                    Vu32CrcCheckSum = PDD_ValidationCalcCrc32(PvpData, Vu32ClusterSizeData);
                    /*compare checksum*/
                    VbResult = (Vu32CrcCheckSum == VtsHeaderChecksum.u32CheckSum);
                    /*checksum OK?*/
                    if (VbResult == FALSE)
                    { //checksum error
                        tS32 Vs32ErrorCode = PDD_ERROR_RAW_EMMC_USER_CHECKSUM;
                        PDD_TRACE(PDD_ERROR, &Vs32ErrorCode, (tU32)sizeof(Vs32ErrorCode));
                        PDD_SET_ERROR_ENTRY("\n RAW_EMMC_USER: checksum error");
                    }
                    /* the s32LastClusterUsed must be set always (It is also necessary, during checksum failure */
                    /* Vs32FoundCluster must set to s32LastClusterUsed, because during next write cycle this s32LastClusterUsed should set to invalid*/
                    /* see test u32PDDNorUserLoadDumpChecksumWrong */
                    PtsDevInfo->s32LastClusterUsed = Vs32FoundCluster;
                    PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_CLUSTER_LAST_USED, &PtsDevInfo->s32LastClusterUsed, (tU32)sizeof(PtsDevInfo->s32LastClusterUsed));
                }
            }
            else
            {
                VbResult = FALSE;
            }
        }
        else
        {
            VbResult = FALSE;
        }
        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_RETURN, &VbResult, (tU32)sizeof(VbResult));
    }
    else
        VbResult = FALSE;
    return(VbResult);
}
/******************************************************************************
* FUNCTION: PDD_u8GetCheckDataStreamReadHeaderEmmc()
*
* DESCRIPTION: check the datastream name and get the table number in the read header
*
* PARAMETERS:
*   PtsDevInfo: pointer data stream
*
* RETURNS:
*       table number
*
* HISTORY:Created  2013 04 25
*****************************************************************************/
static tU8    PDD_u8GetCheckDataStreamReadHeaderEmmc(char* PtsDataStreamNameDefined, tsPddRaweMMCUserHeaderDataStream* PspHeaderDataStreamRead)
{
    tU8  Vu8Inc;
    tU8  Vu8NumberDataStreamRead;
    tU8  Vu8DataStream = 0xff;

    if ((PtsDataStreamNameDefined != NULL) && (PspHeaderDataStreamRead != NULL))
    {
        /*get number of read datastreams (=first offset/size for one header)*/
        Vu8NumberDataStreamRead = (tU8)(PspHeaderDataStreamRead->u32Offset / (sizeof(tsPddRaweMMCUserHeaderDataStream)));
        /*trace out header */
        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_NUMBER_READ_DATA_STREAM, &Vu8NumberDataStreamRead, sizeof(tU8));
        /*check if size defined name !=0 */
        if (strlen(PtsDataStreamNameDefined) != 0)
        { /* for each datastream*/
            for (Vu8Inc = 0; Vu8Inc < Vu8NumberDataStreamRead; Vu8Inc++)
            {
                if (memcmp(PspHeaderDataStreamRead->u8DataStreamName, PtsDataStreamNameDefined, strlen(PtsDataStreamNameDefined)) == 0)
                {/*filename are identical*/
                    Vu8DataStream = Vu8Inc;
                    break;
                }
                PspHeaderDataStreamRead++;
            }/*end for*/
        }
    }
    return(Vu8DataStream);
}
/******************************************************************************
* FUNCTION: PDD_s32UpdateHeaderDataStreamEmmc()
*
* DESCRIPTION: check the datastream name and get the tabel number
*
* PARAMETERS:
*   PtsDevInfo      : pointer data stream
*
* RETURNS:
*       PDD_NOR_USER_UPDATE_HEADER                 => update header
*       PDD_OK                                     => no update header
*       PDD_ERROR_NOR_USER_NO_SPACE_CLUSTER_FULL   => size to great
*
*
* HISTORY:Created  2013 04 25
*****************************************************************************/
static tS32   PDD_s32UpdateHeaderDataStreamEmmc(tsPddRaweMMCUserInfo* PtsDevInfo, tS32 Ps32SizeWrite, tU8 Pu8DataStream)
{
    tS32                              Vs32ReturnCode = PDD_OK;
    tU8                               Vu8Inc;
    tU32                              Vu32Size;
    tU32                              Vu32Offset;
    tU32                              Vu32MaxSize;
    if (PtsDevInfo != NULL)
    {
        tsPddRaweMMCUserHeaderDataStream* VspHeaderDataStream = &PtsDevInfo->sHeaderDataStream[Pu8DataStream];

        /*get size and offset from actual header */
        Vu32Size = VspHeaderDataStream->u32Lenght;
        Vu32Offset = VspHeaderDataStream->u32Offset;
        /* new size? => Header should be updatete */
        if (Vu32Size != (tU32)Ps32SizeWrite)
        { /*check if new lenght fit into cluster size; maxsize= (lastoffset + lastlength) + Diff(newsize-oldsize)*/
            Vu32MaxSize = (tU32)Ps32SizeWrite - Vu32Size;
#ifndef PDD_TESTMANGER_ACTIVE
            VspHeaderDataStream = &PtsDevInfo->sHeaderDataStream[PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM - 1];
#else
            VspHeaderDataStream = &PtsDevInfo->sHeaderDataStream[PDD_RAW_EMMC_USER_NUMBER_DATASTREAM_TM - 1];
#endif
            Vu32MaxSize = Vu32MaxSize + VspHeaderDataStream->u32Offset + VspHeaderDataStream->u32Lenght;
            if (Vu32MaxSize >= PtsDevInfo->u32ClusterSizeData)
            { /* error no space cluster full*/
                Vs32ReturnCode = PDD_ERROR_RAW_EMMC_USER_NO_SPACE_CLUSTER_FULL;
                PDD_TRACE(PDD_ERROR, &Vs32ReturnCode, sizeof(tS32));
                //NORMAL_M_ASSERT_ALWAYS();
            }
            else
            { /* change header datastream offset and lenght */
              /* for each datastream*/
#ifndef PDD_TESTMANGER_ACTIVE
                for (Vu8Inc = Pu8DataStream; Vu8Inc < PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM; Vu8Inc++)
#else
                for (Vu8Inc = Pu8DataStream; Vu8Inc < PDD_RAW_EMMC_USER_NUMBER_DATASTREAM_TM; Vu8Inc++)
#endif
                {
                    VspHeaderDataStream = &PtsDevInfo->sHeaderDataStream[Vu8Inc];
                    /* save new size*/
                    if (Vu8Inc == Pu8DataStream)
                    {
                        VspHeaderDataStream->u32Lenght = (tU32)Ps32SizeWrite;
                    }
                    /*set new offset*/
                    VspHeaderDataStream->u32Offset = Vu32Offset;
                    /*determine new offset*/
                    Vu32Offset = VspHeaderDataStream->u32Offset + VspHeaderDataStream->u32Lenght;
                }
                Vs32ReturnCode = PDD_RAW_EMMC_USER_UPDATE_HEADER;
                /*trace out header */
                PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_DATA_STREAM_HEADER, &PtsDevInfo->sHeaderDataStream[0], PDD_RAW_EMMC_USER_HEADER_SIZE_DATA_STREAM);
            }
        }
    }
    else
        Vs32ReturnCode = PDD_ERROR_RAW_EMMC_USER_INVALID_INPUT;
    return(Vs32ReturnCode);
}

/******************************************************************************
* FUNCTION: PDD_vUpdateDataEmmc()
*
* DESCRIPTION: check the datastream name and get the tabel number
*
* PARAMETERS:
*   PtsDevInfo      : pointer data stream
*
* RETURNS:
*       TRUE => configuration read success
*
* HISTORY:Created  2013 04 26
*****************************************************************************/
static void  PDD_vUpdateDataEmmc(tsPddRaweMMCUserInfo* PtsDevInfo, void* PvpWriteBuffer)
{
    tsPddRaweMMCUserHeaderDataStream* VspHeaderDataStreamNew = NULL;
    if ((PtsDevInfo != NULL) && (PvpWriteBuffer != NULL))
    {
        tsPddRaweMMCUserHeaderDataStream* VspHeaderDataStreamOld = (tsPddRaweMMCUserHeaderDataStream*)PvpWriteBuffer;
        tU8                               Vu8Inc;
        tU8*                              Vu8pWriteBuffer = (tU8*)PvpWriteBuffer;
        tU32                              Vu32SizeCopy = 0;  //Fixed Coverity error:REVERSE_NEGATIVE error category
        /* for each datastream*/
#ifndef PDD_TESTMANGER_ACTIVE
        for (Vu8Inc = 0; Vu8Inc < PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM; Vu8Inc++)
#else
        for (Vu8Inc = 0; Vu8Inc < PDD_RAW_EMMC_USER_NUMBER_DATASTREAM_TM; Vu8Inc++)
#endif
        {
            VspHeaderDataStreamNew = &PtsDevInfo->sHeaderDataStream[Vu8Inc];
            /*if offset different*/
            if (VspHeaderDataStreamNew->u32Offset != VspHeaderDataStreamOld->u32Offset)
            {/*copy data from old buffer to new buffer*/
                Vu32SizeCopy = VspHeaderDataStreamNew->u32Lenght;
                if (VspHeaderDataStreamNew->u32Lenght > VspHeaderDataStreamOld->u32Lenght)
                {
                    Vu32SizeCopy = VspHeaderDataStreamOld->u32Lenght;
                }
                if (VspHeaderDataStreamNew->u32Offset > PtsDevInfo->u32ClusterSize)
                {  /* error no space cluster full*/
                    PDD_SET_ERROR_ENTRY("RAW_EMMC_USER: no space cluster full");
                    PDD_FATAL_M_ASSERT_ALWAYS();
                }
                if (VspHeaderDataStreamOld->u32Offset > PtsDevInfo->u32ClusterSize)
                { /* error no space cluster full*/
                    PDD_SET_ERROR_ENTRY("RAW_EMMC_USER: no space cluster full");
                    PDD_FATAL_M_ASSERT_ALWAYS();
                }
                memmove(Vu8pWriteBuffer + VspHeaderDataStreamNew->u32Offset,
                    PtsDevInfo->u8pBufferDataOld + VspHeaderDataStreamOld->u32Offset,
                    Vu32SizeCopy);
            }
            /*increment pointer*/
            VspHeaderDataStreamOld++;
        }/*end for*/
        if (Vu32SizeCopy != 0)
        {
            VspHeaderDataStreamNew = &PtsDevInfo->sHeaderDataStream[Vu8Inc - 1];
            tU32 Vu32Offset = VspHeaderDataStreamNew->u32Offset + VspHeaderDataStreamNew->u32Lenght;
            tS32 Vs32Size = (tS32)(vtspPddRaweMMCInfo->u32ClusterSizeData - Vu32Offset - 1);
            memset(Vu8pWriteBuffer + Vu32Offset, 0x00, (size_t)Vs32Size);
        }
    }
}

/******************************************************************************
* FUNCTION: PDD_bUpdateMagicAndWriteCountofCluster()
*
* DESCRIPTION: 
*
* PARAMETERS:
*   PtsDevInfo      : pointer data stream
*   Vs32ClusterNo   : number of the cluster
*
* RETURNS:
*       TRUE => configuration read success
*
* HISTORY:Created  2013 04 26
*****************************************************************************/
static tBool PDD_bUpdateMagicAndWriteCountofCluster(tsPddRaweMMCUserInfo* PtsDevInfo, tS32  Vs32ClusterNo)
{
    tBool                        VbResult = FALSE;
    tsPddRAWeMMCUserHeaderMagic  VtsHeaderMagic;

    if (PtsDevInfo != NULL)
    {
        /*set magic*/
        VtsHeaderMagic.u16MagicLow = PDD_RAW_EMMC_USER_MAGIC;
        VtsHeaderMagic.u16ChunkSize = PDD_RAW_EMMC_USER_MIN_CHUNK_SIZE;
        /*save counter erase and trace level*/
        VtsHeaderMagic.u32ClusterUpdatedCount = PtsDevInfo->u32ClusterUpdatedCount;
        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_WRITE_COUNTER, &PtsDevInfo->u32ClusterUpdatedCount, sizeof(tU32));
        /*write magic to flash*/
        VbResult = PDD_bWriteEmmc(PtsDevInfo, Vs32ClusterNo, (tU8*)&VtsHeaderMagic, PDD_RAW_EMMC_USER_MAGIC_POSITION, sizeof(VtsHeaderMagic));
    }
    return VbResult;
}
#endif

#ifdef PDD_TESTMANGER_ACTIVE
/******************************************************************************
* FUNCTION: PDD_bGetNumberDataStreamTmAndNameDataStream()
*
* DESCRIPTION: determine number of datastream for testmanager and get datastream names
*
* PARAMETERS:
*   PspHeaderDataStreamRead      : pointerbeginn header stream
*
*
* HISTORY:Created  2014 04 28
*****************************************************************************/
static tBool   PDD_bGetNumberDataStreamTmAndNameDataStream(tsPddRaweMMCUserHeaderDataStream* PspHeaderDataStreamRead)
{
    tU8     Vu8Inc;
    tU8     Vu8DataStreamKDS = 0xff;
    tBool   VbReturn = FALSE;
#if PDD_RAW_EMMC_USER_CONFIG_NUMBER_DATASTREAM
    if (PspHeaderDataStreamRead != NULL)
    {
        /*get number of read datastreams (=first offset/size for one header)*/
        vu8NumberDataStreamTm = (tU8)(PspHeaderDataStreamRead->u32Offset / (sizeof(tsPddRaweMMCUserHeaderDataStream)));
        /*trace out header */
        PDD_TRACE(PDD_RAW_EMMC_USER_DEBUG_NUMBER_READ_DATA_STREAM, &vu8NumberDataStreamTm, sizeof(tU8));
        /*get pointer defined */
        tsPddRaweMMCUserHeaderDataStream *VspHeaderDataStreamDefined;
        /* for each datastream*/
        for (Vu8Inc = 0; Vu8Inc < vu8NumberDataStreamTm; Vu8Inc++)
        { /*get pointer defined */
            VspHeaderDataStreamDefined = &vtspPddRaweMMCInfo->sHeaderDataStream[Vu8Inc];
            /*copy read name to defined header*/
            memmove(VspHeaderDataStreamDefined->u8DataStreamName, PspHeaderDataStreamRead->u8DataStreamName, strlen((char*)PspHeaderDataStreamRead->u8DataStreamName));
            if (memcmp(PspHeaderDataStreamRead->u8DataStreamName, PDD_RAW_EMMC_USER_DATASTREAM_NAME_KDS, strlen(PDD_RAW_EMMC_USER_DATASTREAM_NAME_KDS)) == 0)
            {/*filename are identical kds found*/
                Vu8DataStreamKDS = Vu8Inc;
                VbReturn = TRUE;
            }
            PspHeaderDataStreamRead++;
        }
        if (Vu8DataStreamKDS == 0xff)
        {/*KDS not found increment kds datastream*/
            vu8NumberDataStreamTm++;
        }
        if (vu8NumberDataStreamTm > PDD_RAW_EMMC_USER_CONFIG_MAX_NUMBER_DATASTREAM)
        {
            VbReturn = FALSE;
        }
    }
#endif
    return(VbReturn);
}
#endif

#ifdef __cplusplus
}
#endif
/******************************************************************************/
/* End of File pdd_access_raw_emmc_user.c                                     */
/******************************************************************************/
