/************************************************************************
 
 *FILE:           alltypes.h
 
 *SW-COMPONENT:   FI, framework for Functional Interface
 
 *DESCRIPTION:    Interface for all types defined in the different FIs
                  
 *AUTHOR:         Jochen Bruns (CM-DI/ENS41)
 
 *COPYRIGHT:      (c) 2001 Blaupunkt GmbH
 
 *HISTORY:
           $ Log $
 * 
 *    Rev 1.0   Oct 7 2001 17:16:00   bnj2hi
 * Initial revision
 *  
 ************************************************************************/

#ifndef MOST_ALLFITYPES_HEADER
#define MOST_ALLFITYPES_HEADER

#include <vector>

class most_fi_tclVisitorBase;

const tU16 MOST_FI_C_U16_ERROR_INTERNALFAILURE = 1;
const tU16 MOST_FI_C_U16_ERROR_INVALIDBINCONTAINERWRONGVERSION = 2;
const tU16 MOST_FI_C_U16_ERROR_INVALIDBINCONTAINERUNEXPECTEDDATA = 3;

class most_fi_tclTypeBase : public fi_tclTypeBase
{
public:
  most_fi_tclTypeBase() {}
  virtual ~most_fi_tclTypeBase() {}
   
  virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& /*oVisitor*/) {}
};
///////////////////////////////////////////////////////////////////////////////
//
// special version including project specific MOST definitions
//
///////////////////////////////////////////////////////////////////////////////

#include "most_defines_p.h"

///////////////////////////////////////////////////////////////////////////////
//
// special version including MOST compatible types
//
// classes omitted intentionally for 
// U24, S24 and multilanguage string
//
///////////////////////////////////////////////////////////////////////////////

class most_fi_tclVisitableTypeBase :public most_fi_tclTypeBase
{
public:
   /* inline */ virtual void vTakeVisitor(most_fi_tclVisitorBase& /*oVisitor*/)
      {  }

   // globally define BIG Endian
   /* inline */ virtual fi_tclIOContextBase::tenByteOrder enGetByteOrder() const
      {  return fi_tclIOContextBase::EN_MOTOROLA;  }

};

// -------------------------------------------------------------------------------------

class most_fi_tclU8 : public most_fi_tclVisitableTypeBase
{
public:
   most_fi_tclU8():u8Value(0) {}
   tU8 u8Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const most_fi_tclU8& roRef) const
      { return (u8Value == roRef.u8Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return most_typesfi_tclToken::EN_U8;}

   inline operator tU8&(tVoid) { return u8Value; }; 
};

// -------------------------------------------------------------------------------------

class most_fi_tclS8 : public most_fi_tclVisitableTypeBase
{
public:
   most_fi_tclS8():s8Value(0) {}
   tS8 s8Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const most_fi_tclS8& roRef) const
      { return (s8Value == roRef.s8Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return most_typesfi_tclToken::EN_S8;}

   inline operator tS8&(tVoid) { return s8Value; }; 
};

// -------------------------------------------------------------------------------------

class most_fi_tclU16 : public most_fi_tclVisitableTypeBase
{
public:
   most_fi_tclU16():u16Value(0) {}
   tU16 u16Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const most_fi_tclU16& roRef) const
      { return (u16Value == roRef.u16Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return most_typesfi_tclToken::EN_U16;}

   inline operator tU16&(tVoid) { return u16Value; }; 
};

// -------------------------------------------------------------------------------------

class most_fi_tclS16 : public most_fi_tclVisitableTypeBase
{
public:
   most_fi_tclS16():s16Value(0) {}
   tS16 s16Value;
   
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
  inline tBool operator==(const most_fi_tclS16& roRef) const
      { return (s16Value == roRef.s16Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return most_typesfi_tclToken::EN_S16;}

   inline operator tS16&(tVoid) { return s16Value; }; 
};

// -------------------------------------------------------------------------------------

class most_fi_tclU32 : public most_fi_tclVisitableTypeBase
{
public:
   most_fi_tclU32():u32Value(0) {}
   tU32 u32Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const most_fi_tclU32& roRef) const
      { return (u32Value == roRef.u32Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return most_typesfi_tclToken::EN_U32;}

   inline operator tU32&(tVoid) { return u32Value; }; 
};

// -------------------------------------------------------------------------------------

class most_fi_tclS32 : public most_fi_tclVisitableTypeBase
{
public:
   most_fi_tclS32():s32Value(0) {}
   tS32 s32Value;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const most_fi_tclS32& roRef) const
      { return (s32Value == roRef.s32Value); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return most_typesfi_tclToken::EN_S32;}

   inline operator tS32&(tVoid) { return s32Value; }; 
};

// -------------------------------------------------------------------------------------

class most_fi_tclBool : public most_fi_tclVisitableTypeBase
{
public:
   most_fi_tclBool():bValue(0) {}
   tBool bValue;

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   inline tBool operator==(const most_fi_tclBool& roRef) const
      { return (bValue == roRef.bValue); }
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId(tVoid) const { return most_typesfi_tclToken::EN_BOOL;}
};

// -------------------------------------------------------------------------------------

class most_fi_tcl_String : public most_fi_tclVisitableTypeBase
{
public:
   most_fi_tcl_String():szValue(0) {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_String(const most_fi_tcl_String& coRef);
   most_fi_tcl_String& operator=(const most_fi_tcl_String& coRef);
   virtual ~most_fi_tcl_String();
#endif

   tString szValue;

   virtual void vDestroy();
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   tBool operator==(const most_fi_tcl_String& roRef) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);

   enum tenCharSet {
      FI_EN_UTF16 = 0
     , FI_EN_ISO8859_15M = 1
     , FI_EN_UTF8 = 2
#ifdef FI_MOST_GB18030_CHAR_SET_ID
     , FI_EN_GB18030 = FI_MOST_GB18030_CHAR_SET_ID
#endif
   /*  , FI_EN_RDS
     , FI_EN_DAB_01
     , FI_EN_DAB_10
     , FI_EN_DAB_11
     , FI_EN_SHIFT_JIS */
   /* , 0x08 - 0xBF Reserved
      , 0xC00xEF System Integrator (e.g. Car Maker)
      , 0xF00xFF Supplier */
   };
   virtual tS32 s32GetTypeId() const { return most_typesfi_tclToken::EN_STRING;}

   /* The following functions are used to import string values into FI-objects
    * and to export strings from FI-Objects.
    * The character encoding (ISO-Modified or UTF8) is provided as parameter
    * Attention: string values are copied by these functions, thus incoming and/or
    * outgoing strings have to be deleted externally.
    */
   // tBool bSet(const figroup_fi_tcl_String& corfoInit, tenCharSet enCharSet);
   tBool bSet(tCString coszInit, tenCharSet enCharSet = FI_EN_UTF8);
   tString szGet(tenCharSet enCharSet = FI_EN_UTF8) const;
};

// -------------------------------------------------------------------------------------

class most_fi_tcl_TrailingStream : public most_fi_tclVisitableTypeBase
{
private:
   tU32 _u32DataSize; 
   tU8* au8Data;

public:
   tVoid  vSetData(tU8 const * pau8DataSource, tU32 u32DataSize);
   tVoid  vGetData(tU8* pau8DestinationBuffer, tU32 u32BufferSize) const;
   tU8&  operator[] (tU32 u32Index);
   tU32   u32GetDataSize() { return _u32DataSize; }
   
   most_fi_tcl_TrailingStream() : _u32DataSize(0), au8Data(OSAL_NULL) { }
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_TrailingStream(const most_fi_tcl_TrailingStream& coRef);
   most_fi_tcl_TrailingStream& operator=(const most_fi_tcl_TrailingStream& coRef);
   virtual ~most_fi_tcl_TrailingStream();
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY
   virtual tVoid vDestroy(tVoid);
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   tBool operator==(const most_fi_tcl_TrailingStream& roRef) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return most_typesfi_tclToken::EN_TRAILINGSTREAM;}
};

// -------------------------------------------------------------------------------------

class most_fi_tcl_ShortStream : public most_fi_tclVisitableTypeBase
{
private:
   tU8 _u8DataSize;
   tU8* au8Data;

public:   
   // data access functions
   tVoid vSetData(tU8 const * pau8DataSource, tU8 u8DataSize);
   tVoid vGetData(tU8* pau8DestinationBuffer, tU32 u32BufferSize) const;
   tU8&  operator[] (tU8 u8Index);
   tU8   u8GetDataSize() { return _u8DataSize; }
   
   most_fi_tcl_ShortStream() : _u8DataSize(0), au8Data(OSAL_NULL) { }
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_ShortStream(const most_fi_tcl_ShortStream& coRef);
   most_fi_tcl_ShortStream& operator=(const most_fi_tcl_ShortStream& coRef);
   virtual ~most_fi_tcl_ShortStream();
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY
   virtual tVoid vDestroy(tVoid);
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   tBool operator==(const most_fi_tcl_ShortStream& oRef) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return most_typesfi_tclToken::EN_SHORTSTREAM;}
};

// -------------------------------------------------------------------------------------

class most_fi_tcl_ClassifiedStream : public most_fi_tclVisitableTypeBase
{
private:
   tU16 _u16DataSize;
   tU8* au8Data;

public:   
   tString sClassifier;
   
   most_fi_tcl_ClassifiedStream() : _u16DataSize(0),au8Data(OSAL_NULL),sClassifier(OSAL_NULL) { }
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_ClassifiedStream(const most_fi_tcl_ClassifiedStream& coRef);
   most_fi_tcl_ClassifiedStream& operator=(const most_fi_tcl_ClassifiedStream& coRef);
   virtual ~most_fi_tcl_ClassifiedStream();
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY
   virtual tVoid vDestroy(tVoid);
   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   tBool operator==(const most_fi_tcl_ClassifiedStream& roRef) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& oOutContext) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& oOutContext);
   virtual void vTakeVisitor(most_fi_tclVisitorBase& oVisitor);
   virtual tS32 s32GetTypeId() const { return most_typesfi_tclToken::EN_CLASSIFIEDSTREAM;}

   // data access functions
   tVoid vSetData(tU8 const * pau8DataSource, tU16 u16DataSize);
   tVoid vGetData(tU8* pau8DestinationBuffer, tU32 u32BufferSize) const;
   tU8&  operator[] (tU16 u16Index);
   tU16  u16GetDataSize() { return _u16DataSize; }
   
   // Import / export string values.
   // Attention: string values are copied by these functions, thus incoming and/or
   // outgoing strings have to be deleted externally.
   tString szGetDataAsString();
   tVoid   vSetDataFromString(tCString szText);
   tString szGetClassifier() const;
   tVoid   vSetClassifier(tCString szClassifier);
};

// *************************************************************************************

class most_fi_tcl_e8_ErrorCode;
class most_fi_tcl_e8_AdPrAsObjectDetectionEventEnum;
class most_fi_tcl_e8_AdPrAsParkInstructionValue;
class most_fi_tcl_e8_AdPrAsParkInstructionAPA2Value;
class most_fi_tcl_e8_AdPrAsHeaderIconLeftAPA2Value;
class most_fi_tcl_e8_AdPrAsHeaderIconRightAPA2Value;
class most_fi_tcl_e8_AdPrAsFooterTextAPA2Value;
class most_fi_tcl_e8_AdPrAsInfotainmentSoftButtonFocusValue;
class most_fi_tcl_e8_AtManDisplayType;
class most_fi_tcl_e8_AtManAckType;
class most_fi_tcl_e8_AtManPriority;
class most_fi_tcl_e8_AtManButtonAction;
class most_fi_tcl_AtManBluetooth_Call_Active_with_Known_Contact;
class most_fi_tcl_AtManEmail_sent_successfully_alert;
class most_fi_tcl_AtManNew_Email_received_alert;
class most_fi_tcl_AtManNew_Text_Received_Alert;
class most_fi_tcl_AtManCall_Ended_With_Known_Contact_Alert;
class most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_on_Route_alert;
class most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_alert;
class most_fi_tcl_AtManID_of_weather_event__in_Weather_Advisory_alert;
class most_fi_tcl_AtManText_Message_Sent_Successfully_Alert;
class most_fi_tcl_AtManText_Message_Delivery_Failure_Alert;
class most_fi_tcl_AtManEmail_Delivery_Failure_Alert;
class most_fi_tcl_AtManAlertParameterDataStream;
class most_fi_tcl_AtManButtonActionStream;
class most_fi_tcl_AtManCurrentAlertStreamItem;
class most_fi_tcl_AtManCurrentAlertStream;
class most_fi_tcl_AtManAlertStreamItem;
class most_fi_tcl_AtManAlertStream;
class most_fi_tcl_e8_AtManAlertDisplayedEnum;
class most_fi_tcl_AtManDisplayTypeStream;
class most_fi_tcl_AtManAlertIDStreamItem;
class most_fi_tcl_AtManAlertIDStream;
class most_fi_tcl_e8_AmpATCConfiguration;
class most_fi_tcl_AmpDSPModeEnabledStream;
class most_fi_tcl_e8_AudCuCueTypeEnum;
class most_fi_tcl_e8_AudCuCueCalibrationEnum;
class most_fi_tcl_AudCuCueCalibrationStreamItem;
class most_fi_tcl_AudCuCueCalibrationStream;
class most_fi_tcl_e8_AudCuCueEnableEnum;
class most_fi_tcl_AudCuCueEnableStreamItem;
class most_fi_tcl_AudCuCueEnableStream;
class most_fi_tcl_e8_AudCuPlayStatusEnum;
class most_fi_tcl_e8_AudCuAudibleFeedbackType;
class most_fi_tcl_AVManConnectionListItem;
class most_fi_tcl_AVManConnectionList;
class most_fi_tcl_e8_AVManLogicalAVChannel;
class most_fi_tcl_e8_AVManRequestResult;
class most_fi_tcl_e8_AVManChannelStatus;
class most_fi_tcl_AVManBaseStatusItem;
class most_fi_tcl_AVManBaseStatus;
class most_fi_tcl_AVManRearStatusItem;
class most_fi_tcl_AVManRearStatus;
class most_fi_tcl_AVManHeadphoneStatusItem;
class most_fi_tcl_AVManHeadphoneStatus;
class most_fi_tcl_AVManVideoStatusItem;
class most_fi_tcl_AVManVideoStatus;
class most_fi_tcl_e8_AVManBaseVoiceMode;
class most_fi_tcl_e8_AVManRearVoiceMode;
class most_fi_tcl_e8_AVManVolumeGroup;
class most_fi_tcl_e8_AVManAVCType;
class most_fi_tcl_e8_AVManATCSetting;
class most_fi_tcl_AVManVolumeGroupExtentStreamItem;
class most_fi_tcl_AVManVolumeGroupExtentStream;
class most_fi_tcl_e8_AVManRearVolumeGroup;
class most_fi_tcl_AVManRearVolumeGroupExtentStreamItem;
class most_fi_tcl_AVManRearVolumeGroupExtentStream;
class most_fi_tcl_e8_AuxInInsertionStatus;
class most_fi_tcl_e8_AuxTchProximityState;
class most_fi_tcl_AuxTchInteractionInfoListItem;
class most_fi_tcl_AuxTchInteractionInfoList;
class most_fi_tcl_e8_CntStkSwipeApplication;
class most_fi_tcl_e8_CntStkDayNightModeOverrideSetting;
class most_fi_tcl_CntStkAppsItem;
class most_fi_tcl_CntStkApps;
class most_fi_tcl_e8_CntStkSearchResult;
class most_fi_tcl_e8_CntStkAckType;
class most_fi_tcl_CntStkButtonText;
class most_fi_tcl_e8_CntStkApplicationID;
class most_fi_tcl_e8_BCMSetApproachLightingValue;
class most_fi_tcl_e8_BCMSetAutoReverseGearRearWiperSettingValue;
class most_fi_tcl_e8_BCMSetDriverPersonalizationSettingValue;
class most_fi_tcl_e8_BCMSetExitLightTimeSettingValue;
class most_fi_tcl_e8_BCMSetRemoteStartSettingValue;
class most_fi_tcl_e8_BCMSetRemoteStartCooledSeatSettingValue;
class most_fi_tcl_e8_BCMSetRemoteStartHeatedSeatSettingValue;
class most_fi_tcl_e8_BCMSetPowertrainPerformanceSettingValue;
class most_fi_tcl_e8_BCMSetDaytimeTaillightsSettingValue;
class most_fi_tcl_e8_BCMSetRemoteStartCooledDrPasSeatSettingValue;
class most_fi_tcl_e8_BCMSetRemoteStartHeatedDrPasSeatSettingValue;
class most_fi_tcl_e8_BCMSetRainSenseWipersSettingValue;
class most_fi_tcl_e8_BCMSetRemoteWindowOperationSettingValue;
class most_fi_tcl_e8_BCMApproachLightingSettingValue;
class most_fi_tcl_e8_BCMAutoReverseGearRearWiperSettingValue;
class most_fi_tcl_e8_BCMDriverPersonalizationSettingValue;
class most_fi_tcl_e8_BCMExitLightTimeSettingValue;
class most_fi_tcl_e8_BCMRemoteStartSettingValue;
class most_fi_tcl_e8_BCMRemoteStartCooledSeatSettingValue;
class most_fi_tcl_e8_BCMRemoteStartHeatedSeatSettingValue;
class most_fi_tcl_e8_BCMPowertrainPerformanceSettingValue;
class most_fi_tcl_e8_BCMDaytimeTaillightsSettingValue;
class most_fi_tcl_e8_BCMRemoteStartCooledDrPasSeatSettingValue;
class most_fi_tcl_e8_BCMRemoteStartHeatedDrPasSeatSettingValue;
class most_fi_tcl_e8_BCMRainSenseWipersSettingValue;
class most_fi_tcl_e8_BCMRemoteWindowOperationSettingValue;
class most_fi_tcl_e8_DorLckSetAutoLockingSettingValue;
class most_fi_tcl_e8_DorLckSetAutoUnlockingSettingValue;
class most_fi_tcl_e8_DorLckSetLastDoorClosedLockingSettingValue;
class most_fi_tcl_e8_DorLckSetManualTransUnlockingSettingValue;
class most_fi_tcl_e8_DorLckSetPassiveLockingSettingValue;
class most_fi_tcl_e8_DorLckSetPassiveUnlockingSettingValue;
class most_fi_tcl_e8_DorLckSetRelockRemoteDoorSettingValue;
class most_fi_tcl_e8_DorLckSetRelockRemoteUnlockedDoorSettingValue;
class most_fi_tcl_e8_DorLckSetRemoteInVehicleReminderSettingValue;
class most_fi_tcl_e8_DorLckSetRemoteLockingFeedbackSettingValue;
class most_fi_tcl_e8_DorLckSetRemoteSlidingDoorSettingValue;
class most_fi_tcl_e8_DorLckSetRemoteUnlocklLightingFeedbackSettingValue;
class most_fi_tcl_e8_DorLckSetSelectiveUnlockingSettingValue;
class most_fi_tcl_e8_DorLckSetOpenDoorAntiLockOutSettingValue;
class most_fi_tcl_e8_DorLckAutoLockingSettingValue;
class most_fi_tcl_e8_DorLckAutoUnlockingSettingValue;
class most_fi_tcl_e8_DorLckLastDoorClosedLockingSettingValue;
class most_fi_tcl_e8_DorLckManualTransAutoUnlockingSettingValue;
class most_fi_tcl_e8_DorLckPassiveLockingSettingValue;
class most_fi_tcl_e8_DorLckPassiveUnlockingSettingValue;
class most_fi_tcl_e8_DorLckRelockRemoteDoorSettingValue;
class most_fi_tcl_e8_DorLckRelockRemoteUnlockedDoorSettingValue;
class most_fi_tcl_e8_DorLckRemoteInVehicleReminderSettingValue;
class most_fi_tcl_e8_DorLckRemoteLockingFeedbackSettingValue;
class most_fi_tcl_e8_DorLckRemoteSlidingDoorSettingValue;
class most_fi_tcl_e8_DorLckRemoteUnlockLightingFeedbackSettingValue;
class most_fi_tcl_e8_DorLckSelectiveUnlockingSettingValue;
class most_fi_tcl_e8_DorLckOpenDoorAntiLockOutSettingValue;
class most_fi_tcl_e8_ECCSetAirQualitySensorSettingValue;
class most_fi_tcl_e8_ECCSetAutoCoolSeatsSettingValue;
class most_fi_tcl_e8_ECCSetAutoFanSettingValue;
class most_fi_tcl_e8_ECCSetAutoHeatedSeatsSettingValue;
class most_fi_tcl_e8_ECCSetAutoZoneTempSettingValue;
class most_fi_tcl_e8_ECCSetHVACRemoteStartSettingSettingValue;
class most_fi_tcl_e8_ECCSetACModeSettingValue;
class most_fi_tcl_e8_ECCSetAutoDefogSettingValue;
class most_fi_tcl_e8_ECCSetRearZoneStartupSettingsValue;
class most_fi_tcl_e8_ECCSetRearDefogStartupSettingsValue;
class most_fi_tcl_e8_ECCSetPollutionControlSettingsValue;
class most_fi_tcl_e8_ECCSetIonizerSettingsValue;
class most_fi_tcl_e8_ECCSetAutoAirDistrSettingsValue;
class most_fi_tcl_e8_ECCAirQualitySensorSettingValue;
class most_fi_tcl_e8_ECCAutoCoolSeatsSettingValue;
class most_fi_tcl_e8_ECCAutoFanSettingValue;
class most_fi_tcl_e8_ECCAutoHeatedSeatsSettingValue;
class most_fi_tcl_e8_ECCAutoZoneTempSettingValue;
class most_fi_tcl_e8_ECCHVACRemoteStartSettingValue;
class most_fi_tcl_e8_ECCACModeSettingValue;
class most_fi_tcl_e8_ECCAutoDefogSettingValue;
class most_fi_tcl_e8_ECCRearZoneStartupSettingValue;
class most_fi_tcl_e8_ECCRearDefogStartupSettingValue;
class most_fi_tcl_e8_ECCPollutionControlSettingValue;
class most_fi_tcl_e8_ECCIonizerSettingValue;
class most_fi_tcl_e8_ECCAutoAirDistrSettingValue;
class most_fi_tcl_e8_IPCSetSpeedScaleIlluminationSettingValue;
class most_fi_tcl_e8_IPCSetSportModeBackLightingSettingValue;
class most_fi_tcl_e8_IPCSetEcoModeTelltaleEnableSettingValue;
class most_fi_tcl_e8_IPCSpeedScaleIlluminationSettingValue;
class most_fi_tcl_e8_IPCSportModeBacklightingSettingValue;
class most_fi_tcl_e8_IPCEcoModeTelltaleEnableSettingValue;
class most_fi_tcl_e8_MemSetSetDriverSeatEasyExitSettingValue;
class most_fi_tcl_e8_MemSetSetFoldingMirrorsSettingValue;
class most_fi_tcl_e8_MemSetSetMemoryColumnSettingValue;
class most_fi_tcl_e8_MemSetSetParkingTiltMirrorsSettingValue;
class most_fi_tcl_e8_MemSetSetRemoteMemoryUnlockRecallSettingValue;
class most_fi_tcl_e8_MemSetSetMemoryTiltColumnSettingValue;
class most_fi_tcl_e8_MemSetSetCrashAvoidanceAlertTypeSettingValue;
class most_fi_tcl_e8_MemSetSetReverseTiltMirrorsSettingValue;
class most_fi_tcl_e8_MemSetSetEasyExitSettingValue;
class most_fi_tcl_e8_MemSetSetMemoryRemoteRecallSettingValue;
class most_fi_tcl_e8_MemSetDriverSeatEasyExitSettingValue;
class most_fi_tcl_e8_MemSetFoldingMirrorsSettingValue;
class most_fi_tcl_e8_MemSetMemoryColumnSettingValue;
class most_fi_tcl_e8_MemSetParkingTiltMirrorsSettingValue;
class most_fi_tcl_e8_MemSetRemoteMemoryUnlockRecallSettingValue;
class most_fi_tcl_e8_MemSetMemoryTiltColumnSettingValue;
class most_fi_tcl_e8_MemSetCrashAvoidanceAlertTypeSettingValue;
class most_fi_tcl_e8_MemSetReverseTiltMirrorsSettingValue;
class most_fi_tcl_e8_MemSetEasyExitSettingValue;
class most_fi_tcl_e8_MemSetMemoryRemoteRecallSettingValue;
class most_fi_tcl_e8_ObjCmpSet24GHzRadarsSettingValue;
class most_fi_tcl_e8_ObjCmp24GHzRadarsSettingValue;
class most_fi_tcl_e8_PASetParkAssistSettingValue;
class most_fi_tcl_e8_PASetParkAssistWithTowbarSettingValue;
class most_fi_tcl_e8_PAParkAssistSettingValue;
class most_fi_tcl_e8_PAParkAssistWithTowbarSettingValue;
class most_fi_tcl_e8_SBZASetSideBlindZoneAlertSettingValue;
class most_fi_tcl_e8_SBZASetRearCrossTrafficAlertSettingValue;
class most_fi_tcl_e8_SBZASetLaneChangeAlertSettingValue;
class most_fi_tcl_e8_SBZASideBlindZoneAlertSettingValue;
class most_fi_tcl_e8_SBZARearCrossTrafficAlertSettingValue;
class most_fi_tcl_e8_SBZALaneChangeAlertSettingValue;
class most_fi_tcl_DABTunDABStationListItem;
class most_fi_tcl_DABTunDABStationList;
class most_fi_tcl_e8_DABTunServiceTypeFlag;
class most_fi_tcl_DABTunDABStationListResultItem;
class most_fi_tcl_DABTunDABStationListResult;
class most_fi_tcl_DABTunDABObjectIDList;
class most_fi_tcl_e8_DABTunBroadcastType;
class most_fi_tcl_DABTunDynamicLabelPlusCmd;
class most_fi_tcl_e8_DABTunDABTuningStatus;
class most_fi_tcl_e8_DABTunDABWaveBandSwitch;
class most_fi_tcl_e8_DABTunStationListUpdateStatus;
class most_fi_tcl_b16_DABTunMask;
class most_fi_tcl_DABTunFilterDataItem;
class most_fi_tcl_DABTunFilterData;
class most_fi_tcl_e8_DABTunGroupID;
class most_fi_tcl_e8_DABTunTunerSelection;
class most_fi_tcl_DABTunTMCMessage_;
class most_fi_tcl_DABTunTimeData;
class most_fi_tcl_DABTunDAB_RDSGroupMessage;
class most_fi_tcl_e8_DABTunBackgroundTMCStatusInfo;
class most_fi_tcl_e8_DABTunDABTMCStatusInfo;
class most_fi_tcl_e8_DABTunForegroundTMCStatusInfo;
class most_fi_tcl_e8_DABTunTMCSenderNameInfo;
class most_fi_tcl_DABTunTMCStationDataItem;
class most_fi_tcl_DABTunTMCStationData;
class most_fi_tcl_DevCltInfoListItem;
class most_fi_tcl_DevCltInfoList;
class most_fi_tcl_e8_DevPrjDeviceType;
class most_fi_tcl_e8_DevPrjDeviceInterfaceType;
class most_fi_tcl_DevPrjDeviceListItem;
class most_fi_tcl_DevPrjDeviceList;
class most_fi_tcl_e8_DevPrjRequestedDisplayContext;
class most_fi_tcl_e8_DevPrjDigitalIPodOutApplication;
class most_fi_tcl_e8_DevPrjEchoCancellationNoiseReductionSetting;
class most_fi_tcl_e8_DevPrjLaunchResult;
class most_fi_tcl_e8_DevPrjApplicationCertificateType;
class most_fi_tcl_e8_DevPrjAppActivityState;
class most_fi_tcl_DevPrjApplicationListItem;
class most_fi_tcl_DevPrjApplicationList;
class most_fi_tcl_e8_DevPrjSwitchEnumeration;
class most_fi_tcl_e8_DevPrjSwitchEventTypeEnumeration;
class most_fi_tcl_e8_DevPrjButtonAction;
class most_fi_tcl_e8_DevPrjSessionState;
class most_fi_tcl_DevPrjConnectivityMappingItem;
class most_fi_tcl_DevPrjConnectivityMapping;
class most_fi_tcl_DevPrjDeviceUserParametersItem;
class most_fi_tcl_DevPrjDeviceUserParameters;
class most_fi_tcl_e8_DevPrjBTChangeInfo;
class most_fi_tcl_e8_DevPrjNavigationAppState;
class most_fi_tcl_e8_DevPrjPhoneAppState;
class most_fi_tcl_e8_DevPrjSpeechAppState;
class most_fi_tcl_e8_DiPOSwitchReqResponse;
class most_fi_tcl_e8_PlaybackStopReason;
class most_fi_tcl_e8_FavManAudioChannel;
class most_fi_tcl_e8_FavManVideoChannel;
class most_fi_tcl_e8_FavManFavoriteType;
class most_fi_tcl_FavManFavoriteTypesList;
class most_fi_tcl_FavManFavoriteHMIDataItem;
class most_fi_tcl_FavManFavoriteHMIData;
class most_fi_tcl_e8_FavManListChangeType;
class most_fi_tcl_FavManChangeItems;
class most_fi_tcl_FMTunImageData;
class most_fi_tcl_e8_GMLnGWAPAIconDisplayRequest;
class most_fi_tcl_e8_GMLnGWDrivelineCustomizationSettings;
class most_fi_tcl_e8_GMLnGWSteeringCustomizationSettings;
class most_fi_tcl_e8_GMLnGWSuspensionCustomizationSettings;
class most_fi_tcl_e8_GMLnGWForwardCollisionAlertCustomizationSettings;
class most_fi_tcl_e8_GMLnGWGoNotifierCustomizationSettings;
class most_fi_tcl_e8_GMLnGWCollisionPreparationSystemCustomizationSettings;
class most_fi_tcl_e8_GMLnGWLeftRightHandTrafficCustomizationSettings;
class most_fi_tcl_e8_GMLnGWSeatBeltTighteningCustomizationSettings;
class most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingWithGPSCustomizationSetting;
class most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingCustomizationSetting;
class most_fi_tcl_e8_GMLnGWLeftRightHandTrafficWithGPSCustomizationSettings;
class most_fi_tcl_e8_GMLnGWAutoHighBeamAssistCustomizationSettings;
class most_fi_tcl_e8_GMLnGWAutoHighBeamAssistWithSensitivityCustomizationSettings;
class most_fi_tcl_e8_GMLnGWAdaptiveHighBeamAssistCustomizationSettings;
class most_fi_tcl_e8_GMLnGWAdaptiveHighBeamAssistWithSensitivityCustomizationSettings;
class most_fi_tcl_e8_GMLnGWChargePowerLossAlertCustomizationSettings;
class most_fi_tcl_e8_GMLnGWChargeCordTheftAlertCustomizationSettings;
class most_fi_tcl_e8_GMLnGWChargeSystemAudibleIndicationCustomizationSettings;
class most_fi_tcl_e8_GMLnGWDriverDrowsinessDetectionCustomizationSetting;
class most_fi_tcl_e8_GMLnGWPTOSetOneSpeedCustomizationSetting;
class most_fi_tcl_e8_GMLnGWPTOSetTwoSpeedCustomizationSetting;
class most_fi_tcl_e8_GMLnGWPTOEngineRunTimerCustomizationSetting;
class most_fi_tcl_e8_GMLnGWPTOTapSetSpeedCustomizationSetting;
class most_fi_tcl_e8_GMLnGWPTOStandbySpeedCustomizationSetting;
class most_fi_tcl_e8_GMLnGWPerfModeDisplayCustomizationSettings;
class most_fi_tcl_e8_GMLnGWPerfModeSoundCustomizationSettings;
class most_fi_tcl_e8_GMLnGWPedestrianFriendlyAlertCustomizationSetting;
class most_fi_tcl_e8_GMLnGWEngineAssistedHeatingCustomizationSetting;
class most_fi_tcl_e8_GMLnGWEngineAssistedHeatingPluggedInCustomizationSetting;
class most_fi_tcl_e8_GMLnGWElevatedIdleCustomizationSetting;
class most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingWithoutIntelligentLightDistCustomizationSetting;
class most_fi_tcl_e8_GMLnGWPerfModeMainMenuType;
class most_fi_tcl_e8_GMLnGWPerfModeAdapCruiseControlCustomizationSettings;
class most_fi_tcl_e8_GMLnGWPerfModeDriverSeatControlCustomizationSettings;
class most_fi_tcl_e8_GMLnGWPerfModePassengerSeatControlCustomizationSettings;
class most_fi_tcl_e8_GMLnGWPerfModeDrivingStyleCustomizationSettings;
class most_fi_tcl_e8_GMLnGWPerfModeVisualization_AdapCruiseControl;
class most_fi_tcl_e8_GMLnGWPerfModeVisualization_Displays;
class most_fi_tcl_e8_GMLnGWPerfModeVisualization_Driveline;
class most_fi_tcl_e8_GMLnGWPerfModeVisualization_DriverSeat;
class most_fi_tcl_e8_GMLnGWPerfModeVisualization_Engine;
class most_fi_tcl_e8_GMLnGWPerfModeVisualization_Exhaust;
class most_fi_tcl_e8_GMLnGWPerfModeVisualization_PassengerSeat;
class most_fi_tcl_e8_GMLnGWPerfModeVisualization_Sound;
class most_fi_tcl_e8_GMLnGWPerfModeVisualization_Steering;
class most_fi_tcl_e8_GMLnGWPerfModeVisualization_Suspension;
class most_fi_tcl_e8_GMLnGWPerfModeVisualization_Trans;
class most_fi_tcl_e8_GMLnGWExtendedHillStartAssistCustomizationSetting;
class most_fi_tcl_e8_GMLnGWRearPedestrianDetectionCustomizationSetting;
class most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomizationSetting;
class most_fi_tcl_e8_GMLnGWFrontPedestrianDetectionCustomizationSetting;
class most_fi_tcl_e8_GMLnGWHandsfreeLiftgateOpenCloseCustomizationSettings;
class most_fi_tcl_e8_GMLnGWRearSeatReminderCustomizationSetting;
class most_fi_tcl_e8_GMLnGWRunningBoardsCustomizationSetting;
class most_fi_tcl_e8_GMLnGWDriverSelectedMode1Status;
class most_fi_tcl_e8_GMLnGWDriverSelectedMode2Status;
class most_fi_tcl_e8_GMLnGWDriverSelectedMode3Status;
class most_fi_tcl_e8_GMLnGWDriverSelectedMode4Status;
class most_fi_tcl_e8_GMLnGWDriverSelectedMode5Status;
class most_fi_tcl_e8_GMLnGWDriverSelectedMode6Status;
class most_fi_tcl_e8_GMLnGWDriverSelectedMode7Status;
class most_fi_tcl_e8_GMLnGWDriverSelectedMode8Status;
class most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemAudibleWarningRequest;
class most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemDisplayMessageRequest;
class most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemStatusIndicationRequest;
class most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemWarningIndicationRequest;
class most_fi_tcl_e8_GMLnGWForwardObjectAlertVehicleAheadIndicationRequest;
class most_fi_tcl_e8_GMLnGWForwardObjectAlertWarningChimeInhibitRequest;
class most_fi_tcl_e8_GMLnGWForwardObjectAlertWarningIndicationRequest;
class most_fi_tcl_GMLnGWForwardObjectAlertStream;
class most_fi_tcl_e8_GMLnGWLaneDepartureWarningAudibleWarningLocation;
class most_fi_tcl_e8_GMLnGWLaneDepartureWarningIndicationControlIndicationRequest;
class most_fi_tcl_e8_GMLnGWLeftLaneDepartureWarningStatus;
class most_fi_tcl_e8_GMLnGWRightLaneDepartureWarningStatus;
class most_fi_tcl_e8_GMLnGWLaneKeepingAssistAudibleWarningLocation;
class most_fi_tcl_e8_GMLnGWLaneKeepingAssistIndicationControlIndicationRequest;
class most_fi_tcl_e8_GMLnGWLeftTurnLampActive;
class most_fi_tcl_e8_GMLnGWRightTurnLampActive;
class most_fi_tcl_GMLnGWExtendedLightingStatusStream;
class most_fi_tcl_e8_GMLnGWHighVoltageBatteryLimitationDueToTemperatureIndication;
class most_fi_tcl_e8_GMLnGWHighVoltageChargerSystemStatus;
class most_fi_tcl_GMLnGWExtendedHMIHybridVehicleStatus;
class most_fi_tcl_e8_GMLnGWHighVoltageChargerPowerLevel;
class most_fi_tcl_GMLnGWPropulsionDisplayInfoStream;
class most_fi_tcl_GMLnGWRedundantHighVoltageBatteryPackStream;
class most_fi_tcl_GMLnGWAirbagImpactDataStream;
class most_fi_tcl_e8_GMLnGWHillDescentControlSystemStatus;
class most_fi_tcl_e8_GMLnGWVehicleAheadDistanceIndicationRequestIndicationLevel;
class most_fi_tcl_e8_GMLnGWPedestrianWarningIndicationRequest;
class most_fi_tcl_e8_GMLnGWSpeedLimiterSettingTypeActive;
class most_fi_tcl_e8_GMLnGWSpeedLimitVisionFusedSpeedMask;
class most_fi_tcl_e8_GMLnGWSpeedLimitVisionFusedSpeedUnit;
class most_fi_tcl_GMLnGWSpeedLimitVisionFusedSpeedStream;
class most_fi_tcl_e8_GMLnGWPhoneSpeedRecAudioPromptStatus;
class most_fi_tcl_e8_GMLnGWTransmissionEngagedState;
class most_fi_tcl_e8_GMLnGWVehicleStabilityEnhancementMode;
class most_fi_tcl_e8_GMLnGWVehicleStabilityEnhancementStatus;
class most_fi_tcl_GMLnGWVehicleStabilityInfoStream;
class most_fi_tcl_e8_GMLnGWVehicleDisplayUnits;
class most_fi_tcl_GMLnGWVINDigitStream;
class most_fi_tcl_e8_GMLnGWTransmissionShiftLeverPosition;
class most_fi_tcl_e8_GMLnGWTransmissionEstimatedGearPosition;
class most_fi_tcl_e8_GMLnGWSWCSwitchState;
class most_fi_tcl_e8_GMLnGWRearCrossTrafficAlertLeftIndicationControl;
class most_fi_tcl_e8_GMLnGWRearCrossTrafficAlertRightIndicationControl;
class most_fi_tcl_e8_GMLnGWParkAssistFrontRegion1ObjectStatus;
class most_fi_tcl_e8_GMLnGWParkAssistFrontRegion2ObjectStatus;
class most_fi_tcl_e8_GMLnGWParkAssistFrontRegion3ObjectStatus;
class most_fi_tcl_e8_GMLnGWParkAssistFrontRegion4ObjectStatus;
class most_fi_tcl_e8_GMLnGWParkAssistFrontSystemStatus;
class most_fi_tcl_e8_GMLnGWParkAssistRearRegion1ObjectStatus;
class most_fi_tcl_e8_GMLnGWParkAssistRearRegion2ObjectStatus;
class most_fi_tcl_e8_GMLnGWParkAssistRearRegion3ObjectStatus;
class most_fi_tcl_e8_GMLnGWParkAssistRearRegion4ObjectStatus;
class most_fi_tcl_e8_GMLnGWParkAssistRearSystemStatus;
class most_fi_tcl_e8_GMLnGWXMCurrentStationService;
class most_fi_tcl_e8_GMLnGWXMRadioServiceProvider;
class most_fi_tcl_e8_GMLnGWSpeedLimitUnits;
class most_fi_tcl_e8_GMLnGWLanguageSelection;
class most_fi_tcl_e8_GMLnGWFuelEconomyMetricDisplayUnits;
class most_fi_tcl_e8_GMLnGWPowerConvertibleTopStatus;
class most_fi_tcl_e8_GMLnGWEngineCylinderDeactivationMode;
class most_fi_tcl_e8_GMLnGWVehicleMovementState;
class most_fi_tcl_e8_GMLnGWFrontPassengerSeatbeltReminderOccupancyStatus;
class most_fi_tcl_e8_GMLnGWFrontPassengerSeatOccupancyFaultStatus;
class most_fi_tcl_e8_GMLnGWTargaTopState;
class most_fi_tcl_e8_GMLnGWTargaTopStateValidity;
class most_fi_tcl_e8_GMLnGWCompassOctantHeading;
class most_fi_tcl_e8_GMLnGWFunctionalRoadClass;
class most_fi_tcl_e8_GMLnGWInternationalStdAlphaCountryCodeChar1;
class most_fi_tcl_e8_GMLnGWInternationalStdAlphaCountryCodeChar2;
class most_fi_tcl_e8_GMLnGWLaneCategory;
class most_fi_tcl_e8_GMLnGWRoadSpeedCategoryType;
class most_fi_tcl_e8_GMLnGWDTCFaultType;
class most_fi_tcl_e8_GMLnGWAutomaticBeamSelectStatus;
class most_fi_tcl_e8_GMLnGWTractionControlSystemActive;
class most_fi_tcl_e8_GMLnGWTractionControlSystemOperatingMode;
class most_fi_tcl_e8_GMLnGWTractionControlSystemOperatingStatus;
class most_fi_tcl_e8_GMLnGWTransmissionShiftModeStatus;
class most_fi_tcl_e8_GMLnGWCruiseAndSpeedLimiterSwitchStatus;
class most_fi_tcl_e8_GMLnGWCruiseControlActive;
class most_fi_tcl_e8_GMLnGWCruiseControlEnabled;
class most_fi_tcl_e8_GMLnGWAutomaticTransmissionCommandedGear;
class most_fi_tcl_e8_GMLnGWVehicleTopSpeedLimitModeActive;
class most_fi_tcl_e8_GMLnGWRemoteVehicleStartEngineRunning;
class most_fi_tcl_e8_GMLnGWAirConditioningCompressorCommand;
class most_fi_tcl_e8_GMLnGWFuelModeStatus;
class most_fi_tcl_e8_GMLnGWAdaptiveCruiseControlActive;
class most_fi_tcl_e8_GMLnGWCruiseControlDriverSelectedSpeedActive;
class most_fi_tcl_e8_GMLnGWAntilockBrakeSystemActive;
class most_fi_tcl_e8_GMLnGWDriverWindowPositionStatus;
class most_fi_tcl_e8_GMLnGWLeftRearWindowPositionStatus;
class most_fi_tcl_e8_GMLnGWPassengerWindowPositionStatus;
class most_fi_tcl_e8_GMLnGWRightRearWindowPositionStatus;
class most_fi_tcl_e8_GMLnGWBrakeLightsActive;
class most_fi_tcl_e8_GMLnGWDaytimeRunningLampsActive;
class most_fi_tcl_e8_GMLnGWHazardSwitchActive;
class most_fi_tcl_e8_GMLnGWHeadlampBeamSelectStatus;
class most_fi_tcl_e8_GMLnGWMainLightSwitch;
class most_fi_tcl_e8_GMLnGWTurnSwitchActivation;
class most_fi_tcl_e8_GMLnGWPowerLiftgateMotionStatus;
class most_fi_tcl_e8_GMLnGWRearClosureObstacleDetected;
class most_fi_tcl_e8_GMLnGWLeftSecondaryTurnIndicatorFailed;
class most_fi_tcl_e8_GMLnGWRearLeftParkLampFailed;
class most_fi_tcl_e8_GMLnGWRearRightParkLampFailed;
class most_fi_tcl_e8_GMLnGWRightSecondaryTurnIndicatorFailed;
class most_fi_tcl_e8_GMLnGWTrailerBrakeLightFailed;
class most_fi_tcl_e8_GMLnGWTrailerFogLightFailed;
class most_fi_tcl_e8_GMLnGWTrailerFogLightPresent;
class most_fi_tcl_e8_GMLnGWTrailerHitchSwitchActive;
class most_fi_tcl_e8_GMLnGWTrailerTailLightFailed;
class most_fi_tcl_e8_GMLnGWSpareTireStatus;
class most_fi_tcl_e8_GMLnGWTireInflationMonitoringSystemResetPerformed;
class most_fi_tcl_e8_GMLnGWTirePressureMonitoringSystemFailed;
class most_fi_tcl_e8_GMLnGWParkingAssistanceFunctionFailed;
class most_fi_tcl_e8_GMLnGWWasherFluidLow;
class most_fi_tcl_e8_GMLnGWWindshieldWiperActive;
class most_fi_tcl_e8_GMLnGWHoodStatus;
class most_fi_tcl_e8_GMLnGWWirelessChargingSystemChargingStatus;
class most_fi_tcl_e8_GMLnGWDigitalMapConditionalSpeedLimit;
class most_fi_tcl_e8_GMLnGWDigitalMapConditionalSpeedLimitType;
class most_fi_tcl_e8_GMLnGWDigitalMapSpeedCategory;
class most_fi_tcl_e8_GMLnGWDigitalMapVersionQuarter;
class most_fi_tcl_e8_GMLnGWDigitalMapDrivingSide;
class most_fi_tcl_e8_GMLnGWDigitalMapPassingRestriction;
class most_fi_tcl_GMLnGWADASISLiteExtendedInfoStream;
class most_fi_tcl_e8_GMLnGWElectricParkBrakeStatus;
class most_fi_tcl_e8_GMLnGWGMLANGPSConfiguration;
class most_fi_tcl_e8_GMLnGWCalendarMonth;
class most_fi_tcl_e8_GMLnGWInteriorDimmingAnimationType;
class most_fi_tcl_e8_GMLnGWInteriorDimmingColorType;
class most_fi_tcl_e8_GMLnGWBatteryChargeCordAlert;
class most_fi_tcl_e8_GMLnGWChargeDisplayStatus;
class most_fi_tcl_e8_GMLnGWChargingPortDoorState;
class most_fi_tcl_e8_GMLnGWLowPowerDayOfWeek;
class most_fi_tcl_e8_GMLnGWHighPowerDayOfWeek;
class most_fi_tcl_e8_GMLnGWStartTimeHighPowerDayOfWeek;
class most_fi_tcl_e8_GMLnGWStartTimeLowPowerDayOfWeek;
class most_fi_tcl_e8_GMLnGWMaintenanceModeDisplayRequest;
class most_fi_tcl_e8_GMLnGWNextPlannedDepartureDayOfWeek;
class most_fi_tcl_e8_GMLnGWHybridModeDisplayRequest;
class most_fi_tcl_e8_GMLnGWTimeBasedChargingModeRequest;
class most_fi_tcl_e8_GMLnGWTimeBasedChargingModeResponse;
class most_fi_tcl_e8_GMLnGWChargeRateDaySetRequest;
class most_fi_tcl_e8_GMLnGWChargeRateEnableSetRequest;
class most_fi_tcl_e8_GMLnGWChargeRateSelectedSetRequest;
class most_fi_tcl_e8_GMLnGWChargeRateDayStatus;
class most_fi_tcl_e8_GMLnGWChargeRateEnableStatus;
class most_fi_tcl_e8_GMLnGWChargeRateSelectedStatus;
class most_fi_tcl_e8_GMLnGWDelayedChargingDay;
class most_fi_tcl_e8_GMLnGWDelayedChargingSeason;
class most_fi_tcl_e8_GMLnGWDelayedChargingSelected;
class most_fi_tcl_e8_GMLnGWDelayedChargingDayStatus;
class most_fi_tcl_e8_GMLnGWDelayedChargingSeasonStatus;
class most_fi_tcl_e8_GMLnGWDelayedChargingSelectedStatus;
class most_fi_tcl_e8_GMLnGWRateChargingDay;
class most_fi_tcl_e8_GMLnGWRateChargingSeason;
class most_fi_tcl_e8_GMLnGWRateChargingSelected;
class most_fi_tcl_e8_GMLnGWRateChargingSelectedRate;
class most_fi_tcl_e8_GMLnGWRateChargingDayStatus;
class most_fi_tcl_e8_GMLnGWRateChargingSeasonStatus;
class most_fi_tcl_e8_GMLnGWRateChargingSelectedRateStatus;
class most_fi_tcl_e8_GMLnGWRateChargingSelectedStatus;
class most_fi_tcl_e8_GMLnGWSeason;
class most_fi_tcl_e8_GMLnGWSeasonMonth;
class most_fi_tcl_e8_GMLnGWSeasonSelected;
class most_fi_tcl_e8_GMLnGWSeasonMonthStatus;
class most_fi_tcl_e8_GMLnGWSeasonSelectedStatus;
class most_fi_tcl_e8_GMLnGWSeasonStatus;
class most_fi_tcl_e8_GMLnGWSetTempOverrideNextPlannedDepartureAdtive;
class most_fi_tcl_e8_GMLnGWTempOverrideNextPlannedDepartureDayOfWeek;
class most_fi_tcl_e8_GMLnGWChargeLevelPreferenceStatus;
class most_fi_tcl_e8_GMLnGWUserInterfaceType;
class most_fi_tcl_e8_GMLnGWSetChargingSystemChargeLevelPreference;
class most_fi_tcl_e8_GMLnGWTimeOfDayChargingDisplayTemporaryOverrideStatus;
class most_fi_tcl_e8_GMLnGWJumpStartStatus;
class most_fi_tcl_e8_GMLnGWHybridChargeModeState;
class most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphMeasurement;
class most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphXAxisTickMarkInterval;
class most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphYAxisMaximumValue;
class most_fi_tcl_GMLnGWEnergyEconomyDataStream;
class most_fi_tcl_e8_GMLnGWRateBasedChargeCompletionTimePreferenceRequest;
class most_fi_tcl_e8_GMLnGWRateBasedChargeCompletionTimePreferenceResponse;
class most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerClearStoredPositionRequest;
class most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerDataResetRequest_PositionNumber;
class most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerClearStoredPositionResponse;
class most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerDataResetResponse_PositionNumber;
class most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStoreRequest;
class most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStorage_PositionUpdatedStatus;
class most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStorage_PositionUpdateLocation;
class most_fi_tcl_e8_GMLnGWLocationBasedChargingVehicleLocationStatus_GPSLocation;
class most_fi_tcl_GMLnGWEnergyUsageScoreDataStream;
class most_fi_tcl_GMLnGWEnergyUsageScoreScalingDataStream;
class most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphConfigurationDataMeasurementUnit;
class most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphConfigurationDataXAxisTickMarkInterval;
class most_fi_tcl_e8_GMLnGWEnergyUsageScreenMeasurementUnits;
class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_BackAndCushionBolsterInOut;
class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_BackBolsterInOut;
class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_CushionBolsterInOut;
class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_CushionLengthAdjust;
class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_DisplaySize;
class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestFwdRwd;
class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestTiltForwardRearward;
class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestUpDown;
class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestUpDwnFwdRwd;
class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestWingsInOut;
class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarForwardRearward;
class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarUpDown;
class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarUpDownForwardRearward;
class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_Massage;
class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_MassageIntensity;
class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_MassageType;
class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_UpperShoulderForwardRearward;
class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_BackAndCushionBolsterInOut;
class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_BackBolsterInOut;
class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_CushionBolsterInOut;
class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_CushionLengthAdjust;
class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_DisplaySize;
class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestFwdRwd;
class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestTiltForwardRearward;
class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestUpDown;
class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestUpDwnFwdRwd;
class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestWingsInOut;
class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarForwardRearward;
class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarUpDown;
class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarUpDownForwardRearward;
class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_Massage;
class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_MassageIntensity;
class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_MassageType;
class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_UpperShoulderForwardRearward;
class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_BackAndCushionBolsterInOut;
class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_BackBolsterInOut;
class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_CushionBolsterInOut;
class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_CushionLengthAdjust;
class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_DisplaySize;
class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestFwdRwd;
class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestTiltForwardRearward;
class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestUpDown;
class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestUpDwnFwdRwd;
class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestWingsInOut;
class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarForwardRearward;
class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarUpDown;
class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarUpDownForwardRearward;
class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_Massage;
class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_MassageIntensity;
class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_MassageType;
class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_UpperShoulderForwardRearward;
class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_BackAndCushionBolsterInOut;
class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_BackBolsterInOut;
class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_CushionBolsterInOut;
class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_CushionLengthAdjust;
class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_DisplaySize;
class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestFwdRwd;
class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestTiltForwardRearward;
class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestUpDown;
class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestUpDwnFwdRwd;
class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestWingsInOut;
class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarForwardRearward;
class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarUpDown;
class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarUpDownForwardRearward;
class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_Massage;
class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_MassageIntensity;
class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_MassageType;
class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_UpperShoulderForwardRearward;
class most_fi_tcl_e8_GMLnGWRearSeatKey;
class most_fi_tcl_e8_GMLnGWTeenDriverRequest;
class most_fi_tcl_e8_GMLnGWTeenDriverResponse;
class most_fi_tcl_e8_GMLnGWSpeedLimitStatus;
class most_fi_tcl_e8_GMLnGWOverspeedWarningCurrentStatus;
class most_fi_tcl_GMLnGWVideoProcessingFeatureConfigurationStream;
class most_fi_tcl_e8_GMLnGWTouchProximity;
class most_fi_tcl_GMLnGWWifiSSIDDigits1_8;
class most_fi_tcl_GMLnGWWifiSSIDDigits9_16;
class most_fi_tcl_GMLnGWWifiSSIDDigits17_24;
class most_fi_tcl_GMLnGWWifiPassPhraseDigits1_8;
class most_fi_tcl_GMLnGWWifiPassPhraseDigits9_16;
class most_fi_tcl_GMLnGWWifiPassPhraseDigits17_24;
class most_fi_tcl_e8_GMLnGWInfotainmentWifiEnabled;
class most_fi_tcl_e8_GMLnGWWifiEncryptionType;
class most_fi_tcl_e8_GMLnGWWifiSecurityType;
class most_fi_tcl_e8_GMLnGWWifiAssociationRequest;
class most_fi_tcl_GMLnGWWifiStationMediaAccessControlAddress;
class most_fi_tcl_e8_HVACHVACSettingsEventEnum;
class most_fi_tcl_e8_HVACFanSpeedReductionValue;
class most_fi_tcl_e8_HVACFront_BlowerModeValue;
class most_fi_tcl_e8_HVACFront_FanSpeedValue;
class most_fi_tcl_e8_HVACFront_LeftSetTempValue;
class most_fi_tcl_e8_HVACFront_RightSetTempValue;
class most_fi_tcl_e8_HVACRear_BlowerModeValue;
class most_fi_tcl_e8_HVACRear_FanSpeedValue;
class most_fi_tcl_e8_HVACRear_SetPointSetTempValue;
class most_fi_tcl_e8_HVACZoneValue;
class most_fi_tcl_e8_HVACAirQualitySensorValue;
class most_fi_tcl_e8_HVACAutoDefogValue;
class most_fi_tcl_e8_HVACAirConditionValue;
class most_fi_tcl_e8_HVACAirInletValue;
class most_fi_tcl_e8_HVACMaximum_Front_FanSpeedValue;
class most_fi_tcl_e8_HVACFront_LeftThermalSeatValue;
class most_fi_tcl_e8_HVACFront_RightThermalSeatValue;
class most_fi_tcl_e8_HVACRear_DefogValue;
class most_fi_tcl_e8_HVACHybridEconModeValue;
class most_fi_tcl_e8_HVACHighVoltageHeaterValue;
class most_fi_tcl_e8_HVACRear_PanelLockValue;
class most_fi_tcl_e8_HVACRear_PanelSyncValue;
class most_fi_tcl_e8_HVACFTouchButton_HybridFanOnlyValue;
class most_fi_tcl_e8_HVACFTouchButton_HybridEconValue;
class most_fi_tcl_e8_HVACFTouchButton_HybridComfortValue;
class most_fi_tcl_e8_HVACFTouchButton_LFAutoHeatedSeatValue;
class most_fi_tcl_e8_HVACFTouchButton_RFAutoHeatedSeatValue;
class most_fi_tcl_e8_HVACFTouchButton_HybridHVHeaterValue;
class most_fi_tcl_e8_HVACFTouchButton_ACValue;
class most_fi_tcl_e8_HVACFTouchButton_RecircValue;
class most_fi_tcl_e8_HVACFTouchButton_AutoRecircValue;
class most_fi_tcl_e8_HVACFTouchButton_Mode_ACValue;
class most_fi_tcl_e8_HVACFTouchButton_Mode_BiLevelValue;
class most_fi_tcl_e8_HVACFTouchButton_Mode_HeaterDefrostValue;
class most_fi_tcl_e8_HVACFTouchButton_Mode_HeaterValue;
class most_fi_tcl_e8_HVACFTouchButton_FrontFanIncValue;
class most_fi_tcl_e8_HVACFTouchButton_FrontFanDecValue;
class most_fi_tcl_e8_HVACFTouchButton_AutoValue;
class most_fi_tcl_e8_HVACFTouchButton_SyncValue;
class most_fi_tcl_e8_HVACFTouchButton_ECOACValue;
class most_fi_tcl_e8_HVACFTouchButton_FreshAirValue;
class most_fi_tcl_e8_HVACFTouchButton_Mode_WindshieldValue;
class most_fi_tcl_e8_HVACFTouchButton_Mode_DefrostValue;
class most_fi_tcl_e8_HVACACHeaterStateValue;
class most_fi_tcl_e8_HVACFTouchButton_LeftSetTempIncValue;
class most_fi_tcl_e8_HVACFTouchButton_LeftSetTempDecValue;
class most_fi_tcl_e8_HVACFTouchButton_RightSetTempIncValue;
class most_fi_tcl_e8_HVACFTouchButton_RightSetTempDecValue;
class most_fi_tcl_e8_HVACFTouchButton_RearDefogValue;
class most_fi_tcl_e8_HVACFTouchButton_RearMode_HeaterValue;
class most_fi_tcl_e8_HVACFTouchButton_RearMode_BiLevelValue;
class most_fi_tcl_e8_HVACFTouchButton_RearMode_ACValue;
class most_fi_tcl_e8_HVACFTouchButton_RearFanIncValue;
class most_fi_tcl_e8_HVACFTouchButton_RearFanDecValue;
class most_fi_tcl_e8_HVACFTouchButton_RearSetTempIncValue;
class most_fi_tcl_e8_HVACFTouchButton_RearSetTempDecValue;
class most_fi_tcl_e8_HVACFTouchButton_RearPanelLockIncValue;
class most_fi_tcl_e8_HVACFTouchButton_RearPanelSyncValue;
class most_fi_tcl_e8_HVACFTouchButton_RearPanelPowerValue;
class most_fi_tcl_e8_HVACFTouchButton_RearPanelAutoValue;
class most_fi_tcl_e8_HVACFTouchButton_ACIncValue;
class most_fi_tcl_e8_HVACFTouchButton_RecircIncValue;
class most_fi_tcl_e8_HVACMaximum_Rear_FanSpeedValue;
class most_fi_tcl_e8_HVACMaximum_Front_SetTempsValue;
class most_fi_tcl_e8_HVACMaximum_Rear_SetTempsValue;
class most_fi_tcl_e8_HVACFTouchButton_ClimateModeIncValue;
class most_fi_tcl_e8_HVACFront_Right_BlowerModeValue;
class most_fi_tcl_e8_HVACFTouchButton_Right_Mode_ACValue;
class most_fi_tcl_e8_HVACFTouchButton_Right_Mode_BiLevelValue;
class most_fi_tcl_e8_HVACFTouchButton_Right_Mode_HeaterValue;
class most_fi_tcl_e8_HVACFTouchButton_Right_RearSetTempIncValue;
class most_fi_tcl_e8_HVACFTouchButton_Right_RearSetTempDecValue;
class most_fi_tcl_e8_HVACFTouchButton_RearModeIncValue;
class most_fi_tcl_e8_HVACFTouchButton_RearModeDecValue;
class most_fi_tcl_e8_HVACRight_Rear_SetPointSetTempValue;
class most_fi_tcl_e8_HVACIonizerStateValue;
class most_fi_tcl_e8_HVACHVACSettingsValidEnum;
class most_fi_tcl_e8_ICSCtlSwitchEnumeration;
class most_fi_tcl_e8_ICSCtlSwitchEventTypeEnumeration;
class most_fi_tcl_e8_ICSCtlSliderProximity;
class most_fi_tcl_e8_iAppsIBErrorCode;
class most_fi_tcl_e8_iAppsListType;
class most_fi_tcl_e8_iAppsActiveState;
class most_fi_tcl_e8_iAppsSourceLocation;
class most_fi_tcl_e8_iAppsAppConnectionRequired;
class most_fi_tcl_e8_iAppsAppVehicleStateRequired;
class most_fi_tcl_iAppsAppObjectsItem;
class most_fi_tcl_iAppsAppObjects;
class most_fi_tcl_e8_iAppsChange;
class most_fi_tcl_iAppsItems;
class most_fi_tcl_iAppsRendererPosition;
class most_fi_tcl_iAppsActiveAppsItem;
class most_fi_tcl_iAppsActiveApps;
class most_fi_tcl_iAppsImageData;
class most_fi_tcl_e8_iAppsConnectionType;
class most_fi_tcl_e8_iAppsConnectionStatus;
class most_fi_tcl_e8_iAppsConnectionBandwidth;
class most_fi_tcl_iAppsConnectionAvailabilityItem;
class most_fi_tcl_iAppsConnectionAvailability;
class most_fi_tcl_iAppsUpdateHTMLItem;
class most_fi_tcl_iAppsUpdateHTML;
class most_fi_tcl_e8_iAppsMethod;
class most_fi_tcl_e8_iAppsSyncrhonizationState;
class most_fi_tcl_iAppsSelectorButtonStreamItem;
class most_fi_tcl_iAppsSelectorButtonStream;
class most_fi_tcl_e8_iAppsSelectorButtonState;
class most_fi_tcl_e8_iAppsInternetApplicationSystemAction;
class most_fi_tcl_iAppsMessageHandle;
class most_fi_tcl_e8_iAppsButtonAction;
class most_fi_tcl_e8_iAppsSortOrder;
class most_fi_tcl_iAppsAutocompleteResultList;
class most_fi_tcl_e8_iAppsPOIDistanceType;
class most_fi_tcl_e8_iAppsDistanceUnits;
class most_fi_tcl_iAppsSearchResultListItem;
class most_fi_tcl_iAppsSearchResultList;
class most_fi_tcl_e8_iAppsImageFormat;
class most_fi_tcl_e8_iRadioListType;
class most_fi_tcl_e8_iRadioFileFormat;
class most_fi_tcl_iRadioStationsItem;
class most_fi_tcl_iRadioStations;
class most_fi_tcl_e8_iRadioChange;
class most_fi_tcl_iRadioItems;
class most_fi_tcl_e8_iRadioStationType;
class most_fi_tcl_iRadioStation;
class most_fi_tcl_e8_iRadioRating;
class most_fi_tcl_iRadioSong;
class most_fi_tcl_e8_iRadioAction;
class most_fi_tcl_iRadioImageData;
class most_fi_tcl_e8_iRadioInternetRadioSource;
class most_fi_tcl_e8_iRadioPlaybackState;
class most_fi_tcl_e8_iRadioConnectionStatus;
class most_fi_tcl_e8_iRadioSynchronizationState;
class most_fi_tcl_e8_iRadioPandoraAccountStatus;
class most_fi_tcl_iRadioStationListItem;
class most_fi_tcl_iRadioStationList;
class most_fi_tcl_e8_LVMRmtEnable;
class most_fi_tcl_e8_MPlayListType;
class most_fi_tcl_e8_MPlayCategoryType;
class most_fi_tcl_e8_MPlayDeviceType;
class most_fi_tcl_e8_MPlayFileFormat;
class most_fi_tcl_e8_MPlayMediaType;
class most_fi_tcl_MPlayMediaObjectsItem;
class most_fi_tcl_MPlayMediaObjects;
class most_fi_tcl_e8_MPlayChange;
class most_fi_tcl_MPlayItems;
class most_fi_tcl_e8_MPlayNowPlayingState;
class most_fi_tcl_MPlayMediaObject;
class most_fi_tcl_e8_MPlayState;
class most_fi_tcl_e8_MPlayMode;
class most_fi_tcl_e8_MPlayRepeat;
class most_fi_tcl_e8_MPlayAction;
class most_fi_tcl_e8_MPlayPlaypointFormat;
class most_fi_tcl_e8_MPlayDeviceIndexedState;
class most_fi_tcl_MPlayIndexingStateResultItem;
class most_fi_tcl_MPlayIndexingStateResult;
class most_fi_tcl_MPlayImageData;
class most_fi_tcl_MPlaySearchKeyboardListItem;
class most_fi_tcl_MPlaySearchKeyboardList;
class most_fi_tcl_MPlayDeviceInfoItem;
class most_fi_tcl_MPlayDeviceInfo;
class most_fi_tcl_e8_MPlayFileType;
class most_fi_tcl_MPlayFileListItem;
class most_fi_tcl_MPlayFileList;
class most_fi_tcl_e8_MPlayTagTransferStatus;
class most_fi_tcl_MPlayTagTransferItem_1;
class most_fi_tcl_MPlayTagTransfer_1;
class most_fi_tcl_MPlayUntransferredTagStreamItem;
class most_fi_tcl_MPlayUntransferredTagStream;
class most_fi_tcl_MPlayTagTransferItem_2;
class most_fi_tcl_MPlayTagTransfer_2;
class most_fi_tcl_e8_NavDistanceUnits;
class most_fi_tcl_e8_NavManeuverEnumeration;
class most_fi_tcl_e8_NavGuidanceStatus;
class most_fi_tcl_NavManeuverListStreamItem;
class most_fi_tcl_NavManeuverListStream;
class most_fi_tcl_e8_NavPOIListTypeEnum;
class most_fi_tcl_NavPOI_ListItem;
class most_fi_tcl_NavPOI_List;
class most_fi_tcl_e8_NavDestinationListTypeEnum;
class most_fi_tcl_NavDestinationInfoItem;
class most_fi_tcl_NavDestinationInfo;
class most_fi_tcl_e8_NavListChangeType;
class most_fi_tcl_NavChangedItemIndexes;
class most_fi_tcl_NavCurrentVehiclePositionStreamItem;
class most_fi_tcl_NavCurrentVehiclePositionStream;
class most_fi_tcl_NavManeuverExtendedListStreamItem;
class most_fi_tcl_NavManeuverExtendedListStream;
class most_fi_tcl_e8_NavAddressDisplayFormat;
class most_fi_tcl_e8_OnsAudButtonEnum;
class most_fi_tcl_e8_OnsAudButtonEvent;
class most_fi_tcl_e8_OnsDatOnStarDataSettingsValidEnum;
class most_fi_tcl_e8_OnsDatOnStarDataSettingsEventEnum;
class most_fi_tcl_e8_OnsDatPlaceOnStarCallValue;
class most_fi_tcl_e8_OnsDatButton1SelectedValue;
class most_fi_tcl_e8_OnsDatButton2SelectedValue;
class most_fi_tcl_e8_OnsDatBluetooth_InCallStatusValue;
class most_fi_tcl_e8_OnsDatVoiceRecognitionStatusValue;
class most_fi_tcl_e8_OnsDatOnStarCallTypeValue;
class most_fi_tcl_e8_OnsDatPacketConnectionTypeValue;
class most_fi_tcl_e8_OnsDatInVehicleAlertTypeValue;
class most_fi_tcl_e8_OnsDatOnStarLanguageSettingValue;
class most_fi_tcl_e8_OnsDatSubmenuIndicator;
class most_fi_tcl_OnsDatMenuListValueItem;
class most_fi_tcl_OnsDatMenuListValue;
class most_fi_tcl_e8_OnsDatTTYFunctionRestrictionTypeValue;
class most_fi_tcl_e8_OnsDatEmbeddedCellularRssiTypeValue;
class most_fi_tcl_e8_OnsDatEmbeddedCellularTechnologyTypeValue;
class most_fi_tcl_e8_OnsCalHandsFreePhoneEventEnum;
class most_fi_tcl_e8_OnsCalRingTone_SelectValue;
class most_fi_tcl_e8_OnsCalC1_CallStateValue;
class most_fi_tcl_e8_OnsCalC1_NumberTypeValue;
class most_fi_tcl_e8_OnsCalC2_CallStateValue;
class most_fi_tcl_e8_OnsCalC2_NumberTypeValue;
class most_fi_tcl_e8_OnsCalnumberType;
class most_fi_tcl_OnsCalCLT_IncomingValueItem;
class most_fi_tcl_OnsCalCLT_IncomingValue;
class most_fi_tcl_OnsCalCLT_MissedValueItem;
class most_fi_tcl_OnsCalCLT_MissedValue;
class most_fi_tcl_OnsCalCLT_OutgoingValueItem;
class most_fi_tcl_OnsCalCLT_OutgoingValue;
class most_fi_tcl_OnsCalConf_MembersValueItem;
class most_fi_tcl_OnsCalConf_MembersValue;
class most_fi_tcl_e8_OnsCalEmergencyCall_CallStateValue;
class most_fi_tcl_e8_OnsCalIn_NumberTypeValue;
class most_fi_tcl_e8_OnsCalNetwork_SignalStrengthValue;
class most_fi_tcl_e8_OnsCalNetwork_StatusValue;
class most_fi_tcl_e8_OnsCalPhone_FailureCodeValue;
class most_fi_tcl_e8_OnsCalRingToneValue;
class most_fi_tcl_e8_OnsCalSecurity_FailureCodeValue;
class most_fi_tcl_e8_OnsCalHandsFreePhoneValidEnum;
class most_fi_tcl_e8_OnsNavTBTNavigationDisplayEventEnum;
class most_fi_tcl_e8_OnsNavCurrentManeuverValue;
class most_fi_tcl_e8_OnsNavTBTNavigationDisplayValidEnum;
class most_fi_tcl_e8_PrgMasFileStatus;
class most_fi_tcl_PrgMasProgManFile;
class most_fi_tcl_e8_PrgMasMode;
class most_fi_tcl_PrgMasFile;
class most_fi_tcl_e8_PrgMasUpdateType;
class most_fi_tcl_PrgMasLicFile;
class most_fi_tcl_PrgMasMDFile;
class most_fi_tcl_e8_PrgMasProgModeStatus;
class most_fi_tcl_PrgMasFileAndDeviceNameStreamItem;
class most_fi_tcl_PrgMasFileAndDeviceNameStream;
class most_fi_tcl_e8_PrgMasClientErrorMode;
class most_fi_tcl_PrgMasInvalidFileString;
class most_fi_tcl_e8_PrgMasOTAMsgType;
class most_fi_tcl_PrgMasManFileAndStatusStreamItem;
class most_fi_tcl_PrgMasManFileAndStatusStream;
class most_fi_tcl_e8_PrgMasAsisResultScreenButton;
class most_fi_tcl_e8_PrgMasAsIsSavingStatus;
class most_fi_tcl_e8_PrgMasImmobilizationStatusIcon;
class most_fi_tcl_e8_PrgMasTermsConditionsResponseStatus;
class most_fi_tcl_e8_PrgMasInstallFailureResponseStatus;
class most_fi_tcl_e8_PrgMasFinalConfirmationResponseStatus;
class most_fi_tcl_e8_RSEFrontVideoState;
class most_fi_tcl_e8_RSEFrontRSESources;
class most_fi_tcl_e8_RSEDiskPlayState;
class most_fi_tcl_e8_RSENextPreviousFlag;
class most_fi_tcl_e8_RSEPlayPauseAction;
class most_fi_tcl_e8_RSERSEAccessibilityModeSetting;
class most_fi_tcl_SDARSCategoryListStreamItem;
class most_fi_tcl_SDARSCategoryListStream;
class most_fi_tcl_e8_SDARSChannelAdvisory;
class most_fi_tcl_e8_SDARSSeekType;
class most_fi_tcl_e8_SDARSSubscriptionStatus;
class most_fi_tcl_e8_SDARSTSListType;
class most_fi_tcl_SDARSTuneSelectListStreamItem;
class most_fi_tcl_SDARSTuneSelectListStream;
class most_fi_tcl_SDARSCategoryChannelListStreamItem;
class most_fi_tcl_SDARSCategoryChannelListStream;
class most_fi_tcl_e8_SDARSSXMAdvisory;
class most_fi_tcl_SDARSChannelListDetailsItem;
class most_fi_tcl_SDARSChannelListDetails;
class most_fi_tcl_SDARSChannelListStreamItem;
class most_fi_tcl_SDARSChannelListStream;
class most_fi_tcl_e8_SDARSSDARSModuleType;
class most_fi_tcl_SDARSExtendedChannelListStreamItem;
class most_fi_tcl_SDARSExtendedChannelListStream;
class most_fi_tcl_e8_SDARSDataAuthorization;
class most_fi_tcl_SDARSSIDOnOffStreamItem;
class most_fi_tcl_SDARSSIDOnOffStream;
class most_fi_tcl_e8_TunHDAvailable;
class most_fi_tcl_e8_TunRDSAvailable;
class most_fi_tcl_e8_TunSeekType;
class most_fi_tcl_e8_TunStationListUpdateStatus;
class most_fi_tcl_TunObjectIDList;
class most_fi_tcl_TunStationListResultItem;
class most_fi_tcl_TunStationListResult;
class most_fi_tcl_TunStationListItem;
class most_fi_tcl_TunStationList;
class most_fi_tcl_e8_AudDskFileType;
class most_fi_tcl_e8_AudDskFileFormat;
class most_fi_tcl_AudDskFileListItem;
class most_fi_tcl_AudDskFileList;
class most_fi_tcl_e8_AudDskChange;
class most_fi_tcl_AudDskItems;
class most_fi_tcl_e8_AudDskDeckStatus;
class most_fi_tcl_e8_AudDskDeckEvent;
class most_fi_tcl_e8_AudDskDiskType;
class most_fi_tcl_e8_AudDskPlaybackState;
class most_fi_tcl_AudDskNowPlaying;
class most_fi_tcl_e8_AudDskMediaEvent;
class most_fi_tcl_e8_AudDskPlaybackAction;
class most_fi_tcl_e8_AudDskRepeatState;
class most_fi_tcl_e8_AudDskSeek;
class most_fi_tcl_e8_FavPrvAudioChannel;
class most_fi_tcl_e8_FavPrvVideoChannel;
class most_fi_tcl_e8_FavPrvFavoriteType;
class most_fi_tcl_FavPrvFavoriteInfoItem;
class most_fi_tcl_FavPrvFavoriteInfo;
class most_fi_tcl_e8_FavPrvListChangeType;
class most_fi_tcl_FavPrvChangedFavorites;
class most_fi_tcl_e8_HDRadHDTuningStatus;
class most_fi_tcl_HDRadPSD;
class most_fi_tcl_e8_HDRadHDBandSelect;
class most_fi_tcl_e8_HDRadHDRcvrState;
class most_fi_tcl_e8_TrfARDSTuningState;
class most_fi_tcl_b16_RDSMask;
class most_fi_tcl_RDSFilterDataItem;
class most_fi_tcl_RDSFilterData;
class most_fi_tcl_e8_RDSGroupID;
class most_fi_tcl_e8_RDSTunerSelection;
class most_fi_tcl_RDSTMCMessage_;
class most_fi_tcl_RDSTimeData;
class most_fi_tcl_RDSRDSGroupMessage;
class most_fi_tcl_e8_RDSBackgroundTMCStatusInfo;
class most_fi_tcl_e8_RDSForegroundTMCStatusInfo;
class most_fi_tcl_e8_RDSTMCSenderNameInfo;
class most_fi_tcl_RDSTMCStationDataItem;
class most_fi_tcl_RDSTMCStationData;
class most_fi_tcl_e8_SnkAudioChannelName;
class most_fi_tcl_e8_SnkContentProtection;
class most_fi_tcl_e8_SnkContentType;
class most_fi_tcl_e8_SnkTransmissionClass;
class most_fi_tcl_SnkAudioChannels_Resolution;
class most_fi_tcl_SnkAudioChannelListItem;
class most_fi_tcl_SnkAudioChannelList;
class most_fi_tcl_SnkAudioChannels_AudioChannelList;
class most_fi_tcl_SnkContentDescription;
class most_fi_tcl_VolVolumeListItem;
class most_fi_tcl_VolVolumeList;
class most_fi_tcl_e8_SrcAudioChannelName;
class most_fi_tcl_e8_SrcContentProtection;
class most_fi_tcl_e8_SrcContentType;
class most_fi_tcl_e8_SrcTransmissionClass;
class most_fi_tcl_SrcAudioChannels_Resolution;
class most_fi_tcl_SrcAudioChannelListItem;
class most_fi_tcl_SrcAudioChannelList;
class most_fi_tcl_SrcAudioChannels_AudioChannelList;
class most_fi_tcl_SrcContentDescription;
class most_fi_tcl_e8_SrcActivity;
class most_fi_tcl_e8_TShTimeShiftControlEnum;
class most_fi_tcl_e8_TShTimeShiftStateEnum;
class most_fi_tcl_TchPadTouchDataListItem;
class most_fi_tcl_TchPadTouchDataList;
class most_fi_tcl_e8_TchPadHapticFeedbackProfile;
class most_fi_tcl_e8_TchPadProximityState;
class most_fi_tcl_e8_TchPadButtonTypeEnum;
class most_fi_tcl_TchPadButtonStatesItem;
class most_fi_tcl_TchPadButtonStates;
class most_fi_tcl_e8_SpcHMISRButton;
class most_fi_tcl_e8_SpcHMISpeechWindow;
class most_fi_tcl_e8_SpcHMISRStatus;
class most_fi_tcl_e8_SpcHMIAction;
class most_fi_tcl_SpcHMINBestListStreamItem;
class most_fi_tcl_SpcHMINBestListStream;
class most_fi_tcl_SpcHMIExampleListStream;
class most_fi_tcl_e8_SpcHMISRLanguage;
class most_fi_tcl_SpcHMIDynamicControlTextStreamItem;
class most_fi_tcl_SpcHMIDynamicControlTextStream;
class most_fi_tcl_e8_SpcHMIControlEnableState;
class most_fi_tcl_SpcHMIControlEnableStreamItem;
class most_fi_tcl_SpcHMIControlEnableStream;
class most_fi_tcl_e8_SpcHMIDisplayEnum;
class most_fi_tcl_e8_SpcHMIPromptVerboseness;
class most_fi_tcl_e8_SpcHMIConfirmationLevel;
class most_fi_tcl_SpcHMISRLanguageStream;
class most_fi_tcl_e8_SpcHMISRDomain;
class most_fi_tcl_e8_SpcHMISRDomainSpecificStatus;
class most_fi_tcl_e8_SpcHMIDeviceTranscriptionState;
class most_fi_tcl_SpcHMISRDomainDynamicDataStatusItem;
class most_fi_tcl_SpcHMISRDomainDynamicDataStatus;
class most_fi_tcl_SpcHMISRDomainStatusListStreamItem;
class most_fi_tcl_SpcHMISRDomainStatusListStream;
class most_fi_tcl_e8_SRecTParameterLoadStatus;
class most_fi_tcl_e8_SRecTInterfaceEnable;
class most_fi_tcl_e8_SRecTRecognitionError;
class most_fi_tcl_SRecTRecognitionResultsItem;
class most_fi_tcl_SRecTRecognitionResults;
class most_fi_tcl_e8_SRecTNFSStatus;
class most_fi_tcl_e8_SRecTRecordingsEnable;
class most_fi_tcl_SRecTConfidenceThresholdsItem;
class most_fi_tcl_SRecTConfidenceThresholds;
class most_fi_tcl_e8_SyStaAnimationState;
class most_fi_tcl_e8_SyStaSystemStateEnumeration;
class most_fi_tcl_e8_SyStaModuleLockReportingEnum;
class most_fi_tcl_e8_SyStaNoVINStates;
class most_fi_tcl_e8_SyStaNoCalibrationStates;
class most_fi_tcl_e8_SyStaTheftLockStates;
class most_fi_tcl_e8_SyStaDateFormatEnum;
class most_fi_tcl_e8_SyStaCalendarMonth;
class most_fi_tcl_SyStaTimeOfDayStream;
class most_fi_tcl_e8_SyStaTimeDisplayFormatEnum;
class most_fi_tcl_e8_SyStaAutoTimeUpdateSettingEnum;
class most_fi_tcl_e8_TestFBTestMode;
class most_fi_tcl_TestFBTestFIlenameOrBuffer;
class most_fi_tcl_e8_TestFBErrorInfo;
class most_fi_tcl_e8_TestFBMostChannel;
class most_fi_tcl_e8_TTSReadMode;
class most_fi_tcl_e8_TTSLogicalAVChannel;
class most_fi_tcl_e8_TTSMarkerMode;
class most_fi_tcl_e8_TTSMarkerType;
class most_fi_tcl_e8_TTSPriority;
class most_fi_tcl_e8_TTSRequestStatus;
class most_fi_tcl_e8_TTSTextType;
class most_fi_tcl_e8_TTSTTSLanguage;
class most_fi_tcl_e8_TTSPlaymode;
class most_fi_tcl_TTSAvailableTTSLanguageStream;
class most_fi_tcl_e8_TTSRequestContinuousSessionStatus;
class most_fi_tcl_e8_TTSContinuousSessionPlaymode;
class most_fi_tcl_e8_ErrorCode : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ErrorCode();
   enum tenType {
      FI_EN_FBLOCKIDNOTAVAILABLE = 1UL,
      FI_EN_INSTIDNOTAVAILABLE = 2UL,
      FI_EN_FKTIDNOTAVAILABLE = 3UL,
      FI_EN_OPTYPENOTAVAILABLE = 4UL,
      FI_EN_INVALIDLENGTH = 5UL,
      FI_EN_PARAMETERWRONG_OUTOFRANGE = 6UL,
      FI_EN_PARAMETERNOTAVAILABLE = 7UL,
      FI_EN_PARAMETERMISSING = 8UL,
      FI_EN_TOOMANYPARAMETERS = 9UL,
      FI_EN_FUNCTIONSPECIFIC = 32UL,
      FI_EN_BUSY = 64UL,
      FI_EN_NOTAVAILABLE = 65UL
   };
   tenType enType;

   most_fi_tcl_e8_ErrorCode(most_fi_tcl_e8_ErrorCode::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ErrorCode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ErrorCode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AdPrAsObjectDetectionEventEnum : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AdPrAsObjectDetectionEventEnum();
   enum tenType {
      FI_EN_E8OBJECTDETECTIONFUCLASSRESET = 0UL,
      FI_EN_E8ACTIVATEADVANCEDPARKASSISTANCE = 1UL,
      FI_EN_E8CANCELADVANCEDPARKASSISTANCE = 2UL,
      FI_EN_E8ACTIVATEAPAONINFOTAINMENTDISPLAY = 3UL,
      FI_EN_E8CANCELAPAONINFOTAINMENTDISPLAY = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_AdPrAsObjectDetectionEventEnum(most_fi_tcl_e8_AdPrAsObjectDetectionEventEnum::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AdPrAsObjectDetectionEventEnum();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AdPrAsObjectDetectionEventEnum& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AdPrAsParkInstructionValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AdPrAsParkInstructionValue();
   enum tenType {
      FI_EN_E8APANOTAVAILABLE = 0UL,
      FI_EN_E8SEARCHINGFORPARKINGSLOTRIGHT = 1UL,
      FI_EN_E8PREPARETOSTOPBACKWARDS = 2UL,
      FI_EN_E8STOP = 3UL,
      FI_EN_E8DRIVEBACKWARDS = 4UL,
      FI_EN_E8TURNSTEERINGWHEELRIGHT = 5UL,
      FI_EN_E8TURNSTEERINGWHEELLEFT = 6UL,
      FI_EN_E8DRIVEFORWARD = 7UL,
      FI_EN_E8ENDPOSITIONACHIEVED = 8UL,
      FI_EN_E8LIMITDRIVESPEED = 9UL,
      FI_EN_E8TURNSTEERINGWHEELFULLYRIGHT = 10UL,
      FI_EN_E8TURNSTEERINGWHEELFULLYLEFT = 11UL,
      FI_EN_E8CANCEL = 12UL,
      FI_EN_E8HOLDSTEERINGPOSITIONRIGHT = 13UL,
      FI_EN_E8HOLDSTERINGPOSITIONLEFT = 14UL,
      FI_EN_E8PARKINGFAILEDDOOVER = 15UL,
      FI_EN_E8SEARCHINGFORPARKINGSLOTLEFT = 16UL,
      FI_EN_E8PREPARETOSTOPFORWARD = 17UL,
      FI_EN_E8TURNSTEERINGWHEELPARTIALRIGHTCORRECTLEFT = 18UL,
      FI_EN_E8TURNSTEERINGWHEELPARTIALLEFTCORRECTRIGHT = 19UL,
      FI_EN_E8STOPTIGHTPARKING = 20UL
   };
   tenType enType;

   most_fi_tcl_e8_AdPrAsParkInstructionValue(most_fi_tcl_e8_AdPrAsParkInstructionValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AdPrAsParkInstructionValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AdPrAsParkInstructionValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AdPrAsParkInstructionAPA2Value : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AdPrAsParkInstructionAPA2Value();
   enum tenType {
      FI_EN_E8APANOTAVAILABLESTEERINGFAILURE = 0UL,
      FI_EN_E8SEARCHINGFORPARKINGSLOTRIGHTPARALLEL = 1UL,
      FI_EN_E8SEARCHINGFORPARKINGSLOTLEFTPARALLEL = 2UL,
      FI_EN_E8SEARCHINGFORPARKINGSLOTRIGHTPARALLELPERP1 = 3UL,
      FI_EN_E8SEARCHINGFORPARKINGSLOTLEFTPARALLELPERP1 = 4UL,
      FI_EN_E8LIMITDRIVESPEED = 5UL,
      FI_EN_E8AUTOSTEERINGACTIVATED = 6UL,
      FI_EN_E8AUTOSTEERINGACTIVATEDREABOFF = 7UL,
      FI_EN_E8PARKINGCOMPLETED = 8UL,
      FI_EN_E8PARALLELPARKINGRIGHT = 9UL,
      FI_EN_E8PARALLELPARKINGRIGHTPROGRESSFORWARD = 10UL,
      FI_EN_E8PARALLELPARKINGRIGHTPROGRESSBACKWARD = 11UL,
      FI_EN_E8PARALLELPARKINGRIGHTSTOP = 12UL,
      FI_EN_E8PARALLELPARKINGLEFT = 13UL,
      FI_EN_E8PARALLELPARKINGLEFTPROGRESSFORWARD = 14UL,
      FI_EN_E8PARALLELPARKINGLEFTPROGRESSBACKWARD = 15UL,
      FI_EN_E8PARALLELPARKINGLEFTSTOP = 16UL,
      FI_EN_E8SEARCHINGFORPARKINGSLOTRIGHTPREP1 = 17UL,
      FI_EN_E8SEARCHINGFORPARKINGSLOTLEFTPREP1 = 18UL,
      FI_EN_E8PERPENDICULARPARKINGRIGHT = 19UL,
      FI_EN_E8PERPENDICULARPARKINGRIGHTPROGRESS = 20UL,
      FI_EN_E8PERPENDICULARPARKINGRIGHTSTOP = 21UL,
      FI_EN_E8PERPENDICULARPARKINGLEFT = 22UL,
      FI_EN_E8PERPENDICULARPARKINGLEFTPROGRESS = 23UL,
      FI_EN_E8PERPENDICULARPARKINGLEFTSTOP = 24UL,
      FI_EN_E8PARKINGFAILED = 25UL,
      FI_EN_E8CANCEL = 26UL,
      FI_EN_E8UNPARKFROMRIGHT = 27UL,
      FI_EN_E8UNPARKFROMLEFT = 28UL,
      FI_EN_E8PARKEXITFAIL = 29UL,
      FI_EN_E8PARKEXITCOMPLETED = 30UL,
      FI_EN_E8PARKMOVESEXCEEDED = 31UL,
      FI_EN_E8OVERSPEED = 32UL,
      FI_EN_E8DRIVERINTERFERENCE = 33UL,
      FI_EN_E8CHECKOWNERSMANUAL = 34UL,
      FI_EN_E8PARKINGSPACEFOUNDRIGHTPARALLEL = 35UL,
      FI_EN_E8PARKINGSPACEFOUNDLEFTPARALLEL = 36UL,
      FI_EN_E8PARKINGSPACEFOUNDRIGHTPERP = 37UL,
      FI_EN_E8PARKINGSPACEFOUNDLEFTPERP = 38UL,
      FI_EN_E8MOVEAHEADRIGHTPARALLEL = 39UL,
      FI_EN_E8MOVEAHEADLEFTPARALLEL = 40UL,
      FI_EN_E8MOVEAHEADRIGHTPERP = 41UL,
      FI_EN_E8MOVEAHEADLEFTPERP = 42UL,
      FI_EN_E8PARKASSISTSELECT = 43UL
   };
   tenType enType;

   most_fi_tcl_e8_AdPrAsParkInstructionAPA2Value(most_fi_tcl_e8_AdPrAsParkInstructionAPA2Value::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AdPrAsParkInstructionAPA2Value();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AdPrAsParkInstructionAPA2Value& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AdPrAsHeaderIconLeftAPA2Value : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AdPrAsHeaderIconLeftAPA2Value();
   enum tenType {
      FI_EN_E8LEFTICONSTOP = 0UL,
      FI_EN_E8LEFTICONREARGEAR = 1UL,
      FI_EN_E8LEFTICONDOWNARROW = 2UL,
      FI_EN_E8LEFTICONUPARROW = 3UL,
      FI_EN_E8LEFTICONDRIVEGEAR = 4UL,
      FI_EN_E8BLANK = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_AdPrAsHeaderIconLeftAPA2Value(most_fi_tcl_e8_AdPrAsHeaderIconLeftAPA2Value::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AdPrAsHeaderIconLeftAPA2Value();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AdPrAsHeaderIconLeftAPA2Value& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AdPrAsHeaderIconRightAPA2Value : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AdPrAsHeaderIconRightAPA2Value();
   enum tenType {
      FI_EN_E8RIGHTICONSTEERINGENABLED = 0UL,
      FI_EN_E8RIGHTICONSTEERINGDISABLED = 1UL,
      FI_EN_E8BLANK = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_AdPrAsHeaderIconRightAPA2Value(most_fi_tcl_e8_AdPrAsHeaderIconRightAPA2Value::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AdPrAsHeaderIconRightAPA2Value();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AdPrAsHeaderIconRightAPA2Value& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AdPrAsFooterTextAPA2Value : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AdPrAsFooterTextAPA2Value();
   enum tenType {
      FI_EN_E8PRESSSETCLRFORCROSSPARKING = 0UL,
      FI_EN_E8PRESSSETCLRFORPARALLELPARKING = 1UL,
      FI_EN_E8PRESSSETCLRFORUNPARK = 2UL,
      FI_EN_E8AUTOSTEERINGDEACTIVATED = 3UL,
      FI_EN_E8PARKINGFAILED = 4UL,
      FI_EN_E8PARKINGASSISTANCENOTAVAILABLE = 5UL,
      FI_EN_E8BLANK = 6UL,
      FI_EN_E8PRESSAPAFORCROSSPARKING = 7UL,
      FI_EN_E8PRESSAPAFORPARALLELPARKING = 8UL,
      FI_EN_E8PRESSAPAFORUNPARK = 9UL,
      FI_EN_E8EMERGENCYAUTOBRAKING = 10UL,
      FI_EN_E8MAXSPEED = 11UL,
      FI_EN_E8DEACTIVATEDSTEERFAILURE = 12UL
   };
   tenType enType;

   most_fi_tcl_e8_AdPrAsFooterTextAPA2Value(most_fi_tcl_e8_AdPrAsFooterTextAPA2Value::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AdPrAsFooterTextAPA2Value();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AdPrAsFooterTextAPA2Value& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AdPrAsInfotainmentSoftButtonFocusValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AdPrAsInfotainmentSoftButtonFocusValue();
   enum tenType {
      FI_EN_E8SPARE = 0UL,
      FI_EN_E8INFOTAINMENTSOFTBUTTON1 = 1UL,
      FI_EN_E8INFOTAINMENTSOFTBUTTON2 = 2UL,
      FI_EN_E8INFOTAINMENTSOFTBUTTON3 = 3UL,
      FI_EN_E8INFOTAINMENTSOFTBUTTON4 = 4UL,
      FI_EN_E8INFOTAINMENTSOFTBUTTON5 = 5UL,
      FI_EN_E8INFOTAINMENTSOFTBUTTON6 = 6UL,
      FI_EN_E8INFOTAINMENTSOFTBUTTON7 = 7UL,
      FI_EN_E8INFOTAINMENTSOFTBUTTON8 = 8UL,
      FI_EN_E8INFOTAINMENTSOFTBUTTON9 = 9UL
   };
   tenType enType;

   most_fi_tcl_e8_AdPrAsInfotainmentSoftButtonFocusValue(most_fi_tcl_e8_AdPrAsInfotainmentSoftButtonFocusValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AdPrAsInfotainmentSoftButtonFocusValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AdPrAsInfotainmentSoftButtonFocusValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AtManDisplayType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AtManDisplayType();
   enum tenType {
      FI_EN_E8DISPLAY_TYPE_HUD = 0UL,
      FI_EN_E8DISPLAY_TYPE_CLUSTER = 1UL,
      FI_EN_E8DISPLAY_TYPE_CENTERSTACK = 2UL,
      FI_EN_E8DISPLAY_TYPE_RSE = 3UL,
      FI_EN_E8DISPLAY_TYPE_OTHER = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_AtManDisplayType(most_fi_tcl_e8_AtManDisplayType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AtManDisplayType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AtManDisplayType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AtManAckType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AtManAckType();
   enum tenType {
      FI_EN_E8ACK_TYPE_NON_ACKNOWLEDGEABLE = 0UL,
      FI_EN_E8ACK_TYPE_MANUAL_ONLY = 1UL,
      FI_EN_E8ACK_TYPE_AUTO_MANUAL = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_AtManAckType(most_fi_tcl_e8_AtManAckType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AtManAckType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AtManAckType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AtManPriority : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AtManPriority();
   enum tenType {
      FI_EN_E8IMMEDIATE = 1UL,
      FI_EN_E8HIGH = 2UL,
      FI_EN_E8MEDIUM = 3UL,
      FI_EN_E8LOW = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_AtManPriority(most_fi_tcl_e8_AtManPriority::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AtManPriority();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AtManPriority& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AtManButtonAction : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AtManButtonAction();
   enum tenType {
      FI_EN_E8DISMISS_ACTION = 0UL,
      FI_EN_E8MORE_INFO_ACTION = 1UL,
      FI_EN_E8VIEW_ACTION = 2UL,
      FI_EN_E8HANDSFREE_ACTION = 3UL,
      FI_EN_E8ROUTE_ACTION = 4UL,
      FI_EN_E8REDIAL_ACTION = 5UL,
      FI_EN_E8CANCEL_ACTION = 6UL,
      FI_EN_E8TUNE_ACTION = 7UL,
      FI_EN_E8START_ACTION = 8UL,
      FI_EN_E8OVERRIDE_ACTION = 9UL,
      FI_EN_E8CHANGE_ACTION = 10UL,
      FI_EN_E8SNOOZE_ACTION = 11UL,
      FI_EN_E8LISTEN_ACTION = 12UL,
      FI_EN_E8DOWNLOAD_ACTION = 13UL,
      FI_EN_E8SWITCH_TO_HANDSET_ACTION = 14UL,
      FI_EN_E8INTERNET_APPLICATION_ACTION_1 = 15UL,
      FI_EN_E8INTERNET_APPLICATION_ACTION_2 = 16UL,
      FI_EN_E8INTERNET_APPLICATION_ACTION_3 = 17UL,
      FI_EN_E8DEVICE_PROJECTION_APPLICATION_ACTION_1 = 18UL,
      FI_EN_E8DEVICE_PROJECTION_APPLICATION_ACTION_2 = 19UL,
      FI_EN_E8DEVICE_PROJECTION_APPLICATION_ACTION_3 = 20UL,
      FI_EN_E8OTHER_ACTION = 255UL
   };
   tenType enType;

   most_fi_tcl_e8_AtManButtonAction(most_fi_tcl_e8_AtManButtonAction::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AtManButtonAction();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AtManButtonAction& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AtManBluetooth_Call_Active_with_Known_Contact : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AtManBluetooth_Call_Active_with_Known_Contact();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AtManBluetooth_Call_Active_with_Known_Contact& operator=(const most_fi_tcl_AtManBluetooth_Call_Active_with_Known_Contact& coRef);
   most_fi_tcl_AtManBluetooth_Call_Active_with_Known_Contact(const most_fi_tcl_AtManBluetooth_Call_Active_with_Known_Contact& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   tU32 u32ContactHandle;
   virtual ~most_fi_tcl_AtManBluetooth_Call_Active_with_Known_Contact();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AtManBluetooth_Call_Active_with_Known_Contact& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AtManEmail_sent_successfully_alert : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AtManEmail_sent_successfully_alert();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AtManEmail_sent_successfully_alert& operator=(const most_fi_tcl_AtManEmail_sent_successfully_alert& coRef);
   most_fi_tcl_AtManEmail_sent_successfully_alert(const most_fi_tcl_AtManEmail_sent_successfully_alert& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   tU32 u32MsgHandleLower;
   tU32 u32MsgHandleUpper;
   virtual ~most_fi_tcl_AtManEmail_sent_successfully_alert();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AtManEmail_sent_successfully_alert& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AtManNew_Email_received_alert : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AtManNew_Email_received_alert();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AtManNew_Email_received_alert& operator=(const most_fi_tcl_AtManNew_Email_received_alert& coRef);
   most_fi_tcl_AtManNew_Email_received_alert(const most_fi_tcl_AtManNew_Email_received_alert& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   tU32 u32MsgHandleLower;
   tU32 u32MsgHandleUpper;
   virtual ~most_fi_tcl_AtManNew_Email_received_alert();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AtManNew_Email_received_alert& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AtManNew_Text_Received_Alert : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AtManNew_Text_Received_Alert();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AtManNew_Text_Received_Alert& operator=(const most_fi_tcl_AtManNew_Text_Received_Alert& coRef);
   most_fi_tcl_AtManNew_Text_Received_Alert(const most_fi_tcl_AtManNew_Text_Received_Alert& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   tU32 u32MsgHandleLower;
   tU32 u32MsgHandleUpper;
   virtual ~most_fi_tcl_AtManNew_Text_Received_Alert();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AtManNew_Text_Received_Alert& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AtManCall_Ended_With_Known_Contact_Alert : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AtManCall_Ended_With_Known_Contact_Alert();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AtManCall_Ended_With_Known_Contact_Alert& operator=(const most_fi_tcl_AtManCall_Ended_With_Known_Contact_Alert& coRef);
   most_fi_tcl_AtManCall_Ended_With_Known_Contact_Alert(const most_fi_tcl_AtManCall_Ended_With_Known_Contact_Alert& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 u16CallInstance;
   tU32 u32ContactHandle;
   virtual ~most_fi_tcl_AtManCall_Ended_With_Known_Contact_Alert();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AtManCall_Ended_With_Known_Contact_Alert& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_on_Route_alert : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_on_Route_alert();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_on_Route_alert& operator=(const most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_on_Route_alert& coRef);
   most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_on_Route_alert(const most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_on_Route_alert& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32TrafficEventID;
   tU32 u32TrafficEventIconID;
   virtual ~most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_on_Route_alert();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_on_Route_alert& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_alert : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_alert();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_alert& operator=(const most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_alert& coRef);
   most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_alert(const most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_alert& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32TrafficEventID;
   tU32 u32TrafficEventIconID;
   virtual ~most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_alert();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_alert& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AtManID_of_weather_event__in_Weather_Advisory_alert : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AtManID_of_weather_event__in_Weather_Advisory_alert();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AtManID_of_weather_event__in_Weather_Advisory_alert& operator=(const most_fi_tcl_AtManID_of_weather_event__in_Weather_Advisory_alert& coRef);
   most_fi_tcl_AtManID_of_weather_event__in_Weather_Advisory_alert(const most_fi_tcl_AtManID_of_weather_event__in_Weather_Advisory_alert& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32WeatherEventID;
   tU32 u32WeatherEventIconID;
   virtual ~most_fi_tcl_AtManID_of_weather_event__in_Weather_Advisory_alert();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AtManID_of_weather_event__in_Weather_Advisory_alert& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AtManText_Message_Sent_Successfully_Alert : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AtManText_Message_Sent_Successfully_Alert();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AtManText_Message_Sent_Successfully_Alert& operator=(const most_fi_tcl_AtManText_Message_Sent_Successfully_Alert& coRef);
   most_fi_tcl_AtManText_Message_Sent_Successfully_Alert(const most_fi_tcl_AtManText_Message_Sent_Successfully_Alert& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   tU32 u32MsgHandleLower;
   tU32 u32MsgHandleUpper;
   virtual ~most_fi_tcl_AtManText_Message_Sent_Successfully_Alert();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AtManText_Message_Sent_Successfully_Alert& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AtManText_Message_Delivery_Failure_Alert : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AtManText_Message_Delivery_Failure_Alert();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AtManText_Message_Delivery_Failure_Alert& operator=(const most_fi_tcl_AtManText_Message_Delivery_Failure_Alert& coRef);
   most_fi_tcl_AtManText_Message_Delivery_Failure_Alert(const most_fi_tcl_AtManText_Message_Delivery_Failure_Alert& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   tU32 u32MsgHandleLower;
   tU32 u32MsgHandleUpper;
   virtual ~most_fi_tcl_AtManText_Message_Delivery_Failure_Alert();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AtManText_Message_Delivery_Failure_Alert& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AtManEmail_Delivery_Failure_Alert : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AtManEmail_Delivery_Failure_Alert();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AtManEmail_Delivery_Failure_Alert& operator=(const most_fi_tcl_AtManEmail_Delivery_Failure_Alert& coRef);
   most_fi_tcl_AtManEmail_Delivery_Failure_Alert(const most_fi_tcl_AtManEmail_Delivery_Failure_Alert& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   tU32 u32MsgHandleLower;
   tU32 u32MsgHandleUpper;
   virtual ~most_fi_tcl_AtManEmail_Delivery_Failure_Alert();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AtManEmail_Delivery_Failure_Alert& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AtManAlertParameterDataStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AtManAlertParameterDataStream():u32DynamicSize(0), _ActiveMemberOffset(0u), pau8Data(OSAL_NULL), _oBluetooth_Call_Active_with_Known_Contact(), _u8DeviceHandle(), _oEmail_sent_successfully_alert(), _oNew_Email_received_alert(), _oNew_Text_Received_Alert(), _u16ManeuverID(), _oCall_Ended_With_Known_Contact_Alert(), _u16CallInstance(), _oID_of_TrafficEvent_in_Traffic_Ahead_on_Route_alert(), _oID_of_TrafficEvent_in_Traffic_Ahead_alert(), _u32ChannelObjectID(), _oID_of_weather_event__in_Weather_Advisory_alert(), _oText_Message_Sent_Successfully_Alert(), _oText_Message_Delivery_Failure_Alert(), _oEmail_Delivery_Failure_Alert() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AtManAlertParameterDataStream& operator=(const most_fi_tcl_AtManAlertParameterDataStream& coRef);
   most_fi_tcl_AtManAlertParameterDataStream(const most_fi_tcl_AtManAlertParameterDataStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   tU32 u32DynamicSize;
   size_t _ActiveMemberOffset;
   tU8* pau8Data;

   most_fi_tcl_AtManBluetooth_Call_Active_with_Known_Contact _oBluetooth_Call_Active_with_Known_Contact;
   most_fi_tclU8 _u8DeviceHandle;
   most_fi_tcl_AtManEmail_sent_successfully_alert _oEmail_sent_successfully_alert;
   most_fi_tcl_AtManNew_Email_received_alert _oNew_Email_received_alert;
   most_fi_tcl_AtManNew_Text_Received_Alert _oNew_Text_Received_Alert;
   most_fi_tclU16 _u16ManeuverID;
   most_fi_tcl_AtManCall_Ended_With_Known_Contact_Alert _oCall_Ended_With_Known_Contact_Alert;
   most_fi_tclU16 _u16CallInstance;
   most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_on_Route_alert _oID_of_TrafficEvent_in_Traffic_Ahead_on_Route_alert;
   most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_alert _oID_of_TrafficEvent_in_Traffic_Ahead_alert;
   most_fi_tclU32 _u32ChannelObjectID;
   most_fi_tcl_AtManID_of_weather_event__in_Weather_Advisory_alert _oID_of_weather_event__in_Weather_Advisory_alert;
   most_fi_tcl_AtManText_Message_Sent_Successfully_Alert _oText_Message_Sent_Successfully_Alert;
   most_fi_tcl_AtManText_Message_Delivery_Failure_Alert _oText_Message_Delivery_Failure_Alert;
   most_fi_tcl_AtManEmail_Delivery_Failure_Alert _oEmail_Delivery_Failure_Alert;

   // helper functions
   inline size_t Offset(const most_fi_tclVisitableTypeBase& roMember) const
      { return (size_t)(&roMember) - (size_t)this; }
   const most_fi_tclVisitableTypeBase* pcActiveMember() const;
   most_fi_tclVisitableTypeBase& roVariant(most_fi_tclVisitableTypeBase& roSelectedVariant);

public:
   // variant access functions
   most_fi_tcl_AtManBluetooth_Call_Active_with_Known_Contact& oBluetooth_Call_Active_with_Known_Contact();
   tU8& u8DeviceHandle();
   most_fi_tcl_AtManEmail_sent_successfully_alert& oEmail_sent_successfully_alert();
   most_fi_tcl_AtManNew_Email_received_alert& oNew_Email_received_alert();
   most_fi_tcl_AtManNew_Text_Received_Alert& oNew_Text_Received_Alert();
   tU16& u16ManeuverID();
   most_fi_tcl_AtManCall_Ended_With_Known_Contact_Alert& oCall_Ended_With_Known_Contact_Alert();
   tU16& u16CallInstance();
   most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_on_Route_alert& oID_of_TrafficEvent_in_Traffic_Ahead_on_Route_alert();
   most_fi_tcl_AtManID_of_TrafficEvent_in_Traffic_Ahead_alert& oID_of_TrafficEvent_in_Traffic_Ahead_alert();
   tU32& u32ChannelObjectID();
   most_fi_tcl_AtManID_of_weather_event__in_Weather_Advisory_alert& oID_of_weather_event__in_Weather_Advisory_alert();
   most_fi_tcl_AtManText_Message_Sent_Successfully_Alert& oText_Message_Sent_Successfully_Alert();
   most_fi_tcl_AtManText_Message_Delivery_Failure_Alert& oText_Message_Delivery_Failure_Alert();
   most_fi_tcl_AtManEmail_Delivery_Failure_Alert& oEmail_Delivery_Failure_Alert();
   virtual ~most_fi_tcl_AtManAlertParameterDataStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AtManAlertParameterDataStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AtManButtonActionStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AtManButtonActionStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AtManButtonActionStream& operator=(const most_fi_tcl_AtManButtonActionStream& coRef);
   most_fi_tcl_AtManButtonActionStream(const most_fi_tcl_AtManButtonActionStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_e8_AtManButtonAction, std::allocator<most_fi_tcl_e8_AtManButtonAction> >e8Items;

   most_fi_tcl_e8_AtManButtonAction& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_AtManButtonActionStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AtManButtonActionStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AtManCurrentAlertStreamItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AtManCurrentAlertStreamItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AtManCurrentAlertStreamItem& operator=(const most_fi_tcl_AtManCurrentAlertStreamItem& coRef);
   most_fi_tcl_AtManCurrentAlertStreamItem(const most_fi_tcl_AtManCurrentAlertStreamItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_AtManDisplayType e8DisplayType;
   most_fi_tcl_String sAlertText;
   tU16 u16AutoTimeout;
   most_fi_tcl_e8_AtManAckType e8AckType;
   most_fi_tcl_String sAlertTitle;
   tU32 u32AlertID;
   most_fi_tcl_e8_AtManPriority e8Priority;
   most_fi_tcl_AtManAlertParameterDataStream oAlertParameterDataStream;
   tU16 u16InstanceNum;
   most_fi_tcl_String sButton1Text;
   most_fi_tcl_String sButton2Text;
   most_fi_tcl_String sButton3Text;
   most_fi_tcl_AtManButtonActionStream oButtonActionStream;
   virtual ~most_fi_tcl_AtManCurrentAlertStreamItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AtManCurrentAlertStreamItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AtManCurrentAlertStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AtManCurrentAlertStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AtManCurrentAlertStream& operator=(const most_fi_tcl_AtManCurrentAlertStream& coRef);
   most_fi_tcl_AtManCurrentAlertStream(const most_fi_tcl_AtManCurrentAlertStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_AtManCurrentAlertStreamItem*, std::allocator<most_fi_tcl_AtManCurrentAlertStreamItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_AtManCurrentAlertStreamItem, std::allocator<most_fi_tcl_AtManCurrentAlertStreamItem> >oItems;

   most_fi_tcl_AtManCurrentAlertStreamItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_AtManCurrentAlertStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AtManCurrentAlertStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AtManAlertStreamItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AtManAlertStreamItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AtManAlertStreamItem& operator=(const most_fi_tcl_AtManAlertStreamItem& coRef);
   most_fi_tcl_AtManAlertStreamItem(const most_fi_tcl_AtManAlertStreamItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32AlertID;
   most_fi_tcl_String sAlertText;
   tU16 u16InstanceNum;
   virtual ~most_fi_tcl_AtManAlertStreamItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AtManAlertStreamItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AtManAlertStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AtManAlertStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AtManAlertStream& operator=(const most_fi_tcl_AtManAlertStream& coRef);
   most_fi_tcl_AtManAlertStream(const most_fi_tcl_AtManAlertStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_AtManAlertStreamItem*, std::allocator<most_fi_tcl_AtManAlertStreamItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_AtManAlertStreamItem, std::allocator<most_fi_tcl_AtManAlertStreamItem> >oItems;

   most_fi_tcl_AtManAlertStreamItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_AtManAlertStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AtManAlertStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AtManAlertDisplayedEnum : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AtManAlertDisplayedEnum();
   enum tenType {
      FI_EN_E8ALERT_NOT_DISPLAYED = 0UL,
      FI_EN_E8ALERT_INITIALLY_DISPLAYED = 1UL,
      FI_EN_E8ALERT_COMPLETELY_DISPLAYED = 3UL,
      FI_EN_E8ALERT_REJECTED = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_AtManAlertDisplayedEnum(most_fi_tcl_e8_AtManAlertDisplayedEnum::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AtManAlertDisplayedEnum();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AtManAlertDisplayedEnum& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AtManDisplayTypeStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AtManDisplayTypeStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AtManDisplayTypeStream& operator=(const most_fi_tcl_AtManDisplayTypeStream& coRef);
   most_fi_tcl_AtManDisplayTypeStream(const most_fi_tcl_AtManDisplayTypeStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_e8_AtManDisplayType, std::allocator<most_fi_tcl_e8_AtManDisplayType> >e8Items;

   most_fi_tcl_e8_AtManDisplayType& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_AtManDisplayTypeStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AtManDisplayTypeStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AtManAlertIDStreamItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AtManAlertIDStreamItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AtManAlertIDStreamItem& operator=(const most_fi_tcl_AtManAlertIDStreamItem& coRef);
   most_fi_tcl_AtManAlertIDStreamItem(const most_fi_tcl_AtManAlertIDStreamItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32AlertID;
   tU16 u16InstanceNum;
   virtual ~most_fi_tcl_AtManAlertIDStreamItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AtManAlertIDStreamItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AtManAlertIDStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AtManAlertIDStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AtManAlertIDStream& operator=(const most_fi_tcl_AtManAlertIDStream& coRef);
   most_fi_tcl_AtManAlertIDStream(const most_fi_tcl_AtManAlertIDStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_AtManAlertIDStreamItem, std::allocator<most_fi_tcl_AtManAlertIDStreamItem> >oItems;

   most_fi_tcl_AtManAlertIDStreamItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_AtManAlertIDStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AtManAlertIDStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AmpATCConfiguration : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AmpATCConfiguration();
   enum tenType {
      FI_EN_E8ATC_ALL = 0UL,
      FI_EN_E8ATC_CUSTOM_TALK_ONLY = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_AmpATCConfiguration(most_fi_tcl_e8_AmpATCConfiguration::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AmpATCConfiguration();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AmpATCConfiguration& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AmpDSPModeEnabledStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AmpDSPModeEnabledStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AmpDSPModeEnabledStream& operator=(const most_fi_tcl_AmpDSPModeEnabledStream& coRef);
   most_fi_tcl_AmpDSPModeEnabledStream(const most_fi_tcl_AmpDSPModeEnabledStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU8, std::allocator<tU8> >u8Items;

   tU8& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_AmpDSPModeEnabledStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AmpDSPModeEnabledStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AudCuCueTypeEnum : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AudCuCueTypeEnum();
   enum tenType {
      FI_EN_E8ACUE_WELCOME_CUE = 1UL,
      FI_EN_E8ACUE_SHUTDOWN_CUE = 2UL,
      FI_EN_E8ACUE_STARTUP_CUE = 3UL,
      FI_EN_E8ACUE_BEEP_ALERT_CUE = 4UL,
      FI_EN_E8ACUE_RUN_CUE_ = 5UL,
      FI_EN_E8ACUE_STOP_CUE_ = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_AudCuCueTypeEnum(most_fi_tcl_e8_AudCuCueTypeEnum::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AudCuCueTypeEnum();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AudCuCueTypeEnum& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AudCuCueCalibrationEnum : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AudCuCueCalibrationEnum();
   enum tenType {
      FI_EN_E8ACUE_CALIBRATION_DISABLED = 0UL,
      FI_EN_E8ACUE_CALIBRATION_ENABLED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_AudCuCueCalibrationEnum(most_fi_tcl_e8_AudCuCueCalibrationEnum::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AudCuCueCalibrationEnum();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AudCuCueCalibrationEnum& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AudCuCueCalibrationStreamItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AudCuCueCalibrationStreamItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AudCuCueCalibrationStreamItem& operator=(const most_fi_tcl_AudCuCueCalibrationStreamItem& coRef);
   most_fi_tcl_AudCuCueCalibrationStreamItem(const most_fi_tcl_AudCuCueCalibrationStreamItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_AudCuCueTypeEnum e8CueTypeEnum;
   most_fi_tcl_e8_AudCuCueCalibrationEnum e8CueCalibrationEnum;
   virtual ~most_fi_tcl_AudCuCueCalibrationStreamItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AudCuCueCalibrationStreamItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AudCuCueCalibrationStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AudCuCueCalibrationStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AudCuCueCalibrationStream& operator=(const most_fi_tcl_AudCuCueCalibrationStream& coRef);
   most_fi_tcl_AudCuCueCalibrationStream(const most_fi_tcl_AudCuCueCalibrationStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_AudCuCueCalibrationStreamItem, std::allocator<most_fi_tcl_AudCuCueCalibrationStreamItem> >oItems;

   most_fi_tcl_AudCuCueCalibrationStreamItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_AudCuCueCalibrationStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AudCuCueCalibrationStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AudCuCueEnableEnum : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AudCuCueEnableEnum();
   enum tenType {
      FI_EN_E8ACUE_DISABLED = 0UL,
      FI_EN_E8ACUE_ENABLED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_AudCuCueEnableEnum(most_fi_tcl_e8_AudCuCueEnableEnum::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AudCuCueEnableEnum();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AudCuCueEnableEnum& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AudCuCueEnableStreamItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AudCuCueEnableStreamItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AudCuCueEnableStreamItem& operator=(const most_fi_tcl_AudCuCueEnableStreamItem& coRef);
   most_fi_tcl_AudCuCueEnableStreamItem(const most_fi_tcl_AudCuCueEnableStreamItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_AudCuCueTypeEnum e8CueTypeEnum;
   most_fi_tcl_e8_AudCuCueEnableEnum e8CueEnableEnum;
   virtual ~most_fi_tcl_AudCuCueEnableStreamItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AudCuCueEnableStreamItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AudCuCueEnableStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AudCuCueEnableStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AudCuCueEnableStream& operator=(const most_fi_tcl_AudCuCueEnableStream& coRef);
   most_fi_tcl_AudCuCueEnableStream(const most_fi_tcl_AudCuCueEnableStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_AudCuCueEnableStreamItem, std::allocator<most_fi_tcl_AudCuCueEnableStreamItem> >oItems;

   most_fi_tcl_AudCuCueEnableStreamItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_AudCuCueEnableStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AudCuCueEnableStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AudCuPlayStatusEnum : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AudCuPlayStatusEnum();
   enum tenType {
      FI_EN_E8ACUE_PLAY_STATUS_IDLE = 0UL,
      FI_EN_E8ACUE_PLAY_STATUS_PLAY = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_AudCuPlayStatusEnum(most_fi_tcl_e8_AudCuPlayStatusEnum::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AudCuPlayStatusEnum();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AudCuPlayStatusEnum& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AudCuAudibleFeedbackType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AudCuAudibleFeedbackType();
   enum tenType {
      FI_EN_E8AUDIBLE_FBACK_BUTTON_PRESS = 1UL,
      FI_EN_E8AUDIBLE_FBACK_TOUCH_SCREEN_PRESS = 2UL,
      FI_EN_E8AUDIBLE_FBACK_TOUCH_SCREEN_SWIPE = 3UL,
      FI_EN_E8AUDIBLE_FBACK_ROW_SHOW_CHANGE = 4UL,
      FI_EN_E8AUDIBLE_FBACK_PAGE_CHANGE = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_AudCuAudibleFeedbackType(most_fi_tcl_e8_AudCuAudibleFeedbackType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AudCuAudibleFeedbackType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AudCuAudibleFeedbackType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AVManConnectionListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AVManConnectionListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AVManConnectionListItem& operator=(const most_fi_tcl_AVManConnectionListItem& coRef);
   most_fi_tcl_AVManConnectionListItem(const most_fi_tcl_AVManConnectionListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 u16BlockWidth;
   tU16 u16ConnectionLabel;
   tU8 u8SinkFBlock;
   tU8 u8SinkInstID;
   tU8 u8SinkNr;
   tU8 u8SourceFBlock;
   tU8 u8SourceInstID;
   tU8 u8SourceNr;
   virtual ~most_fi_tcl_AVManConnectionListItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AVManConnectionListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AVManConnectionList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AVManConnectionList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AVManConnectionList& operator=(const most_fi_tcl_AVManConnectionList& coRef);
   most_fi_tcl_AVManConnectionList(const most_fi_tcl_AVManConnectionList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_AVManConnectionListItem, std::allocator<most_fi_tcl_AVManConnectionListItem> >oItems;

   most_fi_tcl_AVManConnectionListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_AVManConnectionList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AVManConnectionList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AVManLogicalAVChannel : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AVManLogicalAVChannel();
   enum tenType {
      FI_EN_E8LC_NONE = 0UL,
      FI_EN_E8LC_MAIN_AUDIO = 1UL,
      FI_EN_E8LC_LVM = 2UL,
      FI_EN_E8LC_ASYNCH_MSG = 3UL,
      FI_EN_E8LC_INCOMING_RING_TONE = 4UL,
      FI_EN_E8LC_PHONE = 5UL,
      FI_EN_E8LC_SYNCH_MSG = 6UL,
      FI_EN_E8LC_SPEECH_REC = 7UL,
      FI_EN_E8LC_EMER_PHONE = 8UL,
      FI_EN_E8LC_SHORT_MIX_ALERT = 9UL,
      FI_EN_E8LC_MIX_ALERT_MSG = 10UL,
      FI_EN_E8LC_ALERT_TONE = 11UL,
      FI_EN_E8LC_ADVISOR_PHONE = 12UL,
      FI_EN_E8LC_AUDIO_CUE = 13UL,
      FI_EN_E8LC_TRAFFIC = 14UL,
      FI_EN_E8LC_EMER_MSG = 15UL,
      FI_EN_E8LC_SOFT_ALERT_TONE = 16UL,
      FI_EN_E8LC_REAR_AUDIO = 17UL,
      FI_EN_E8LC_REAR_ASYNCH_MSG = 18UL,
      FI_EN_E8LC_REAR_INCOMING_RING_TONE = 19UL,
      FI_EN_E8LC_REAR_PHONE = 20UL,
      FI_EN_E8LC_REAR_SYNCH_MSG = 21UL,
      FI_EN_E8LC_REAR_SPEECH_REC = 22UL,
      FI_EN_E8LC_REAR_SHORT_MIX_ALERT = 23UL,
      FI_EN_E8LC_REAR_MIX_ALERT_MSG = 24UL,
      FI_EN_E8LC_REAR_ALERT_TONE = 25UL,
      FI_EN_E8LC_INTERNET_APP_AUDIO = 26UL,
      FI_EN_E8LC_REAR_SOFT_ALERT_TONE = 27UL,
      FI_EN_E8LC_REAR_INTERNET_APP_AUDIO = 28UL,
      FI_EN_E8LC_HEADPHONE_1 = 33UL,
      FI_EN_E8LC_HEADPHONE_2 = 34UL,
      FI_EN_E8LC_HEADPHONE_3 = 35UL,
      FI_EN_E8LC_HEADPHONE_4 = 36UL,
      FI_EN_E8LC_RSE_VIDEO_1 = 49UL,
      FI_EN_E8LC_RSE_VIDEO_2 = 50UL,
      FI_EN_E8LC_RSE_VIDEO_3 = 51UL,
      FI_EN_E8LC_RSE_VIDEO_4 = 52UL,
      FI_EN_E8LC_CLUSTER_VIDEO = 53UL,
      FI_EN_E8LC_CENTER_STACK_VIDEO = 54UL
   };
   tenType enType;

   most_fi_tcl_e8_AVManLogicalAVChannel(most_fi_tcl_e8_AVManLogicalAVChannel::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AVManLogicalAVChannel();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AVManLogicalAVChannel& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AVManRequestResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AVManRequestResult();
   enum tenType {
      FI_EN_E8RS_DENIED = 0UL,
      FI_EN_E8RS_GRANTED = 1UL,
      FI_EN_E8RS_SUSPENDED = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_AVManRequestResult(most_fi_tcl_e8_AVManRequestResult::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AVManRequestResult();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AVManRequestResult& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AVManChannelStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AVManChannelStatus();
   enum tenType {
      FI_EN_E8CS_UNAVAILABLE = 0UL,
      FI_EN_E8CS_INACTIVE = 1UL,
      FI_EN_E8CS_ACTIVE = 2UL,
      FI_EN_E8CS_SUSPENDED = 5UL,
      FI_EN_E8CS_STARTUP_WAIT = 6UL,
      FI_EN_E8CS_FAULT_RECOVER = 7UL,
      FI_EN_E8CS_UNKNOWN = 255UL
   };
   tenType enType;

   most_fi_tcl_e8_AVManChannelStatus(most_fi_tcl_e8_AVManChannelStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AVManChannelStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AVManChannelStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AVManBaseStatusItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AVManBaseStatusItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AVManBaseStatusItem& operator=(const most_fi_tcl_AVManBaseStatusItem& coRef);
   most_fi_tcl_AVManBaseStatusItem(const most_fi_tcl_AVManBaseStatusItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_AVManLogicalAVChannel e8LogicalAVChannel;
   most_fi_tcl_e8_AVManChannelStatus e8ChannelStatus;
   tU8 u8SourceFBlock;
   tU8 u8SourceInstID;
   tU8 u8SourceNr;
   virtual ~most_fi_tcl_AVManBaseStatusItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AVManBaseStatusItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AVManBaseStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AVManBaseStatus() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AVManBaseStatus& operator=(const most_fi_tcl_AVManBaseStatus& coRef);
   most_fi_tcl_AVManBaseStatus(const most_fi_tcl_AVManBaseStatus& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_AVManBaseStatusItem, std::allocator<most_fi_tcl_AVManBaseStatusItem> >oItems;

   most_fi_tcl_AVManBaseStatusItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_AVManBaseStatus();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AVManBaseStatus& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AVManRearStatusItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AVManRearStatusItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AVManRearStatusItem& operator=(const most_fi_tcl_AVManRearStatusItem& coRef);
   most_fi_tcl_AVManRearStatusItem(const most_fi_tcl_AVManRearStatusItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_AVManLogicalAVChannel e8LogicalAVChannel;
   most_fi_tcl_e8_AVManChannelStatus e8ChannelStatus;
   tU8 u8SourceFBlock;
   tU8 u8SourceInstID;
   tU8 u8SourceNr;
   virtual ~most_fi_tcl_AVManRearStatusItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AVManRearStatusItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AVManRearStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AVManRearStatus() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AVManRearStatus& operator=(const most_fi_tcl_AVManRearStatus& coRef);
   most_fi_tcl_AVManRearStatus(const most_fi_tcl_AVManRearStatus& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_AVManRearStatusItem, std::allocator<most_fi_tcl_AVManRearStatusItem> >oItems;

   most_fi_tcl_AVManRearStatusItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_AVManRearStatus();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AVManRearStatus& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AVManHeadphoneStatusItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AVManHeadphoneStatusItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AVManHeadphoneStatusItem& operator=(const most_fi_tcl_AVManHeadphoneStatusItem& coRef);
   most_fi_tcl_AVManHeadphoneStatusItem(const most_fi_tcl_AVManHeadphoneStatusItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_AVManLogicalAVChannel e8LogicalAVChannel;
   most_fi_tcl_e8_AVManChannelStatus e8ChannelStatus;
   tU8 u8SourceFBlock;
   tU8 u8SourceInstID;
   tU8 u8SourceNr;
   virtual ~most_fi_tcl_AVManHeadphoneStatusItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AVManHeadphoneStatusItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AVManHeadphoneStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AVManHeadphoneStatus() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AVManHeadphoneStatus& operator=(const most_fi_tcl_AVManHeadphoneStatus& coRef);
   most_fi_tcl_AVManHeadphoneStatus(const most_fi_tcl_AVManHeadphoneStatus& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_AVManHeadphoneStatusItem, std::allocator<most_fi_tcl_AVManHeadphoneStatusItem> >oItems;

   most_fi_tcl_AVManHeadphoneStatusItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_AVManHeadphoneStatus();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AVManHeadphoneStatus& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AVManVideoStatusItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AVManVideoStatusItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AVManVideoStatusItem& operator=(const most_fi_tcl_AVManVideoStatusItem& coRef);
   most_fi_tcl_AVManVideoStatusItem(const most_fi_tcl_AVManVideoStatusItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_AVManLogicalAVChannel e8LogicalAVChannel;
   most_fi_tcl_e8_AVManChannelStatus e8ChannelStatus;
   tU8 u8SourceFBlock;
   tU8 u8SourceInstID;
   tU8 u8SourceNr;
   virtual ~most_fi_tcl_AVManVideoStatusItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AVManVideoStatusItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AVManVideoStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AVManVideoStatus() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AVManVideoStatus& operator=(const most_fi_tcl_AVManVideoStatus& coRef);
   most_fi_tcl_AVManVideoStatus(const most_fi_tcl_AVManVideoStatus& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_AVManVideoStatusItem, std::allocator<most_fi_tcl_AVManVideoStatusItem> >oItems;

   most_fi_tcl_AVManVideoStatusItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_AVManVideoStatus();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AVManVideoStatus& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AVManBaseVoiceMode : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AVManBaseVoiceMode();
   enum tenType {
      FI_EN_E8VOICE_MODE_OFF = 0UL,
      FI_EN_E8VOICE_MODE_ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_AVManBaseVoiceMode(most_fi_tcl_e8_AVManBaseVoiceMode::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AVManBaseVoiceMode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AVManBaseVoiceMode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AVManRearVoiceMode : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AVManRearVoiceMode();
   enum tenType {
      FI_EN_E8VOICE_MODE_OFF = 0UL,
      FI_EN_E8VOICE_MODE_ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_AVManRearVoiceMode(most_fi_tcl_e8_AVManRearVoiceMode::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AVManRearVoiceMode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AVManRearVoiceMode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AVManVolumeGroup : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AVManVolumeGroup();
   enum tenType {
      FI_EN_E8VG_NONE = 0UL,
      FI_EN_E8VG_MAIN = 1UL,
      FI_EN_E8VG_PHONE = 2UL,
      FI_EN_E8VG_EMERGENCY_PHONE = 3UL,
      FI_EN_E8VG_PROMPT = 4UL,
      FI_EN_E8VG_RING_TONE = 5UL,
      FI_EN_E8VG_TRAFFIC = 6UL,
      FI_EN_E8VG_AUDIO_CUE = 7UL,
      FI_EN_E8VG_CHIME = 9UL
   };
   tenType enType;

   most_fi_tcl_e8_AVManVolumeGroup(most_fi_tcl_e8_AVManVolumeGroup::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AVManVolumeGroup();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AVManVolumeGroup& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AVManAVCType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AVManAVCType();
   enum tenType {
      FI_EN_E8AVC_UNKNOWN = 0UL,
      FI_EN_E8AVC_SCV = 1UL,
      FI_EN_E8AVC_VNC = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_AVManAVCType(most_fi_tcl_e8_AVManAVCType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AVManAVCType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AVManAVCType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AVManATCSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AVManATCSetting();
   enum tenType {
      FI_EN_E8ATC_CUSTOM = 0UL,
      FI_EN_E8ATC_POP = 1UL,
      FI_EN_E8ATC_ROCK = 2UL,
      FI_EN_E8ATC_COUNTRY = 3UL,
      FI_EN_E8ATC_JAZZ = 4UL,
      FI_EN_E8ATC_TALK = 5UL,
      FI_EN_E8ATC_CLASSICAL = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_AVManATCSetting(most_fi_tcl_e8_AVManATCSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AVManATCSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AVManATCSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AVManVolumeGroupExtentStreamItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AVManVolumeGroupExtentStreamItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AVManVolumeGroupExtentStreamItem& operator=(const most_fi_tcl_AVManVolumeGroupExtentStreamItem& coRef);
   most_fi_tcl_AVManVolumeGroupExtentStreamItem(const most_fi_tcl_AVManVolumeGroupExtentStreamItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_AVManVolumeGroup e8VolumeGroup;
   tU8 u8MinimumVolume;
   tU8 u8MaximumVolume;
   virtual ~most_fi_tcl_AVManVolumeGroupExtentStreamItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AVManVolumeGroupExtentStreamItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AVManVolumeGroupExtentStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AVManVolumeGroupExtentStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AVManVolumeGroupExtentStream& operator=(const most_fi_tcl_AVManVolumeGroupExtentStream& coRef);
   most_fi_tcl_AVManVolumeGroupExtentStream(const most_fi_tcl_AVManVolumeGroupExtentStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_AVManVolumeGroupExtentStreamItem, std::allocator<most_fi_tcl_AVManVolumeGroupExtentStreamItem> >oItems;

   most_fi_tcl_AVManVolumeGroupExtentStreamItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_AVManVolumeGroupExtentStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AVManVolumeGroupExtentStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AVManRearVolumeGroup : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AVManRearVolumeGroup();
   enum tenType {
      FI_EN_E8VG_NONE = 0UL,
      FI_EN_E8VG_REAR_MAIN = 1UL,
      FI_EN_E8VG_REAR_PHONE = 2UL,
      FI_EN_E8VG_REAR_RING_TONE = 3UL,
      FI_EN_E8VG_REAR_PROMPT = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_AVManRearVolumeGroup(most_fi_tcl_e8_AVManRearVolumeGroup::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AVManRearVolumeGroup();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AVManRearVolumeGroup& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AVManRearVolumeGroupExtentStreamItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AVManRearVolumeGroupExtentStreamItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AVManRearVolumeGroupExtentStreamItem& operator=(const most_fi_tcl_AVManRearVolumeGroupExtentStreamItem& coRef);
   most_fi_tcl_AVManRearVolumeGroupExtentStreamItem(const most_fi_tcl_AVManRearVolumeGroupExtentStreamItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_AVManRearVolumeGroup e8RearVolumeGroup;
   tU8 u8MinimumVolume;
   tU8 u8MaximumVolume;
   virtual ~most_fi_tcl_AVManRearVolumeGroupExtentStreamItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AVManRearVolumeGroupExtentStreamItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AVManRearVolumeGroupExtentStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AVManRearVolumeGroupExtentStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AVManRearVolumeGroupExtentStream& operator=(const most_fi_tcl_AVManRearVolumeGroupExtentStream& coRef);
   most_fi_tcl_AVManRearVolumeGroupExtentStream(const most_fi_tcl_AVManRearVolumeGroupExtentStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_AVManRearVolumeGroupExtentStreamItem, std::allocator<most_fi_tcl_AVManRearVolumeGroupExtentStreamItem> >oItems;

   most_fi_tcl_AVManRearVolumeGroupExtentStreamItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_AVManRearVolumeGroupExtentStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AVManRearVolumeGroupExtentStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AuxInInsertionStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AuxInInsertionStatus();
   enum tenType {
      FI_EN_E8AUXJACK_REMOVED = 0UL,
      FI_EN_E8AUXJACK_INSERTED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_AuxInInsertionStatus(most_fi_tcl_e8_AuxInInsertionStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AuxInInsertionStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AuxInInsertionStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_AuxTchProximityState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_AuxTchProximityState();
   enum tenType {
      FI_EN_E8TOUCH_PROXIMITY_FAR = 0UL,
      FI_EN_E8TOUCH_PROXIMITY_NEAR = 1UL,
      FI_EN_E8TOUCH_PROXIMITY_TOUCH = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_AuxTchProximityState(most_fi_tcl_e8_AuxTchProximityState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_AuxTchProximityState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_AuxTchProximityState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AuxTchInteractionInfoListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AuxTchInteractionInfoListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AuxTchInteractionInfoListItem& operator=(const most_fi_tcl_AuxTchInteractionInfoListItem& coRef);
   most_fi_tcl_AuxTchInteractionInfoListItem(const most_fi_tcl_AuxTchInteractionInfoListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 u16X1;
   tU16 u16Y1;
   tU16 u16X2;
   tU16 u16Y2;
   tS16 s16VX1;
   tS16 s16VY1;
   tS16 s16VX2;
   tS16 s16VY2;
   tU32 u32TimeStamp;
   most_fi_tcl_e8_AuxTchProximityState e8ProximityState;
   virtual ~most_fi_tcl_AuxTchInteractionInfoListItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AuxTchInteractionInfoListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_AuxTchInteractionInfoList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_AuxTchInteractionInfoList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_AuxTchInteractionInfoList& operator=(const most_fi_tcl_AuxTchInteractionInfoList& coRef);
   most_fi_tcl_AuxTchInteractionInfoList(const most_fi_tcl_AuxTchInteractionInfoList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_AuxTchInteractionInfoListItem, std::allocator<most_fi_tcl_AuxTchInteractionInfoListItem> >oItems;

   most_fi_tcl_AuxTchInteractionInfoListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_AuxTchInteractionInfoList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_AuxTchInteractionInfoList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_CntStkSwipeApplication : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_CntStkSwipeApplication();
   enum tenType {
      FI_EN_E8SWIPE_APPLICATION_INACTIVE = 0UL,
      FI_EN_E8SWIPE_APPLICATION_NAVIGATION_MAP = 1UL,
      FI_EN_E8SWIPE_APPLICATION_AUDIO_NOW_PLAYING = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_CntStkSwipeApplication(most_fi_tcl_e8_CntStkSwipeApplication::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_CntStkSwipeApplication();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_CntStkSwipeApplication& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_CntStkDayNightModeOverrideSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_CntStkDayNightModeOverrideSetting();
   enum tenType {
      FI_EN_E8DAY_NIGHT_OVERRIDE_NOT_SUPPORTED = 0UL,
      FI_EN_E8DAY_NIGHT_OVERRIDE_AUTOMATIC = 1UL,
      FI_EN_E8DAY_NIGHT_OVERRIDE_DAY_MODE = 2UL,
      FI_EN_E8DAY_NIGHT_OVERRIDE_NIGHT_MODE = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_CntStkDayNightModeOverrideSetting(most_fi_tcl_e8_CntStkDayNightModeOverrideSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_CntStkDayNightModeOverrideSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_CntStkDayNightModeOverrideSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_CntStkAppsItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_CntStkAppsItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_CntStkAppsItem& operator=(const most_fi_tcl_CntStkAppsItem& coRef);
   most_fi_tcl_CntStkAppsItem(const most_fi_tcl_CntStkAppsItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32AppPositions;
   tU32 u32AppID;
   virtual ~most_fi_tcl_CntStkAppsItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_CntStkAppsItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_CntStkApps : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_CntStkApps() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_CntStkApps& operator=(const most_fi_tcl_CntStkApps& coRef);
   most_fi_tcl_CntStkApps(const most_fi_tcl_CntStkApps& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_CntStkAppsItem, std::allocator<most_fi_tcl_CntStkAppsItem> >oItems;

   most_fi_tcl_CntStkAppsItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_CntStkApps();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_CntStkApps& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_CntStkSearchResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_CntStkSearchResult();
   enum tenType {
      FI_EN_E8SUCCESS = 0UL,
      FI_EN_E8NO_NAV = 1UL,
      FI_EN_E8NO_RESULTS = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_CntStkSearchResult(most_fi_tcl_e8_CntStkSearchResult::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_CntStkSearchResult();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_CntStkSearchResult& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_CntStkAckType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_CntStkAckType();
   enum tenType {
      FI_EN_E8ACK_TYPE_NON_ACKNOWLEDGEABLE = 0UL,
      FI_EN_E8ACK_TYPE_MANUAL_ONLY = 1UL,
      FI_EN_E8ACK_TYPE_AUTO_MANUAL = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_CntStkAckType(most_fi_tcl_e8_CntStkAckType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_CntStkAckType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_CntStkAckType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_CntStkButtonText : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_CntStkButtonText() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_CntStkButtonText& operator=(const most_fi_tcl_CntStkButtonText& coRef);
   most_fi_tcl_CntStkButtonText(const most_fi_tcl_CntStkButtonText& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_String*, std::allocator<most_fi_tcl_String*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_String, std::allocator<most_fi_tcl_String> >sItems;

   most_fi_tcl_String& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_CntStkButtonText();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_CntStkButtonText& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_CntStkApplicationID : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_CntStkApplicationID();
   enum tenType {
      FI_EN_E8INTERNET_APP = 0UL,
      FI_EN_E8NOW_PLAYING = 21UL,
      FI_EN_E8CD = 22UL,
      FI_EN_E8USB_IPOD = 23UL,
      FI_EN_E8BT_AUDIO = 24UL,
      FI_EN_E8AM_AUDIO = 25UL,
      FI_EN_E8FM_AUDIO = 26UL,
      FI_EN_E8XM_AUDIO = 27UL,
      FI_EN_E8DAB_AUDIO = 28UL,
      FI_EN_E8AUX_AUDIO = 29UL,
      FI_EN_E8MY_MEDIA = 30UL,
      FI_EN_E8DVD_PLAYBACK = 34UL,
      FI_EN_E8RSE = 35UL,
      FI_EN_E8REAR_CAMERA = 36UL,
      FI_EN_E8PANDORA = 40UL,
      FI_EN_E8PHOTO = 41UL,
      FI_EN_E8EMAIL = 49UL,
      FI_EN_E8WEATHER = 50UL,
      FI_EN_E8SPORTS = 51UL,
      FI_EN_E8MOVIES = 52UL,
      FI_EN_E8FUEL = 53UL,
      FI_EN_E8STOCKS = 54UL,
      FI_EN_E8TEXT_SENT_LIST = 61UL,
      FI_EN_E8TEXT_TO_LIST = 62UL,
      FI_EN_E8BT_SETTINGS = 63UL,
      FI_EN_E8CONTACTS_ADD = 64UL,
      FI_EN_E8RECENT_CALLS = 65UL,
      FI_EN_E8PHONEBOOK = 66UL,
      FI_EN_E8TEXT_MESSAGE = 67UL,
      FI_EN_E8PHONE_CALL = 68UL,
      FI_EN_E8BT_SETUP = 69UL,
      FI_EN_E8HOME_SCREEN = 70UL,
      FI_EN_E8SETTINGS = 71UL,
      FI_EN_E8TONE_SETTINGS = 72UL,
      FI_EN_E8HVAC = 73UL,
      FI_EN_E8REAR_HVAC = 74UL,
      FI_EN_E8POWERFLOW = 75UL,
      FI_EN_E8ONSTAR = 76UL,
      FI_EN_E8LVM1 = 77UL,
      FI_EN_E8LVM2 = 78UL,
      FI_EN_E8LVM3 = 79UL,
      FI_EN_E8VOICE_REC = 80UL,
      FI_EN_E8DRIVE_MODE = 81UL,
      FI_EN_E8CLOCK = 82UL,
      FI_EN_E8RADIO_OFF = 83UL,
      FI_EN_E8OFF = 84UL,
      FI_EN_E8COMPASS = 85UL,
      FI_EN_E8TBT = 86UL,
      FI_EN_E8ADDRESS_BOOK = 87UL,
      FI_EN_E8POI = 88UL,
      FI_EN_E8DESTINATION_ENTRY = 89UL,
      FI_EN_E8MAP_NAVIGATION = 90UL,
      FI_EN_E8NAVI_MODIFYROUTE_LIST = 93UL,
      FI_EN_E8NAVI_WAYPOINT_LIST = 94UL,
      FI_EN_E8NAV = 95UL,
      FI_EN_E8NAVI_MAP_DESTINATION_LANDING_PAGE = 96UL,
      FI_EN_E8NAVI_LOCATION_SETTINGS = 97UL,
      FI_EN_E8NAVI_MAP_MENU = 98UL,
      FI_EN_E8NAVI_TRAFFIC_MENU = 99UL,
      FI_EN_E8NAVI_ONSTAR_MENU = 100UL,
      FI_EN_E8NAVI_MAP_DESTINATION_DETAILS = 101UL,
      FI_EN_E8NAVI_MAP_TURNLIST = 102UL,
      FI_EN_E8NAVI_MAP_MYROUTES_EDIT = 103UL,
      FI_EN_E8NAVI_MAP_SAVEDTRIPS_LIST = 104UL,
      FI_EN_E8NAVI_MAP_SAVEDTRIPS_LIST_FAVORITE = 105UL,
      FI_EN_E8NAVI_QUICKSTATUSPANE = 106UL,
      FI_EN_E8NAVI_CONTACT_EDIT_MODE = 107UL,
      FI_EN_E8NAVI_MAP_LIST_FUELTYPES = 108UL,
      FI_EN_E8NAVI_MAP_FUELSTATION_ICON_ON_OFF = 109UL,
      FI_EN_E8NAVI_MAP_MENU_DAYNIGHTMODE = 110UL,
      FI_EN_E8NAVI_MAP_RECENTDESTINATION_EDIT = 111UL,
      FI_EN_E8NAVI_TRAFFIC_LIST = 112UL,
      FI_EN_E8EV_CHARGELIMIT_SETTING = 118UL,
      FI_EN_E8EV_CHARGE_MODE = 119UL,
      FI_EN_E8EV_CHARGE_RATE_INFO = 120UL,
      FI_EN_E8EV_CHARGE_RATE = 121UL,
      FI_EN_E8EV_CHARGING = 122UL,
      FI_EN_E8EV_DEFAULT_CHARGE_MODE = 123UL,
      FI_EN_E8EV_DEPARTURETIME_EDIT = 124UL,
      FI_EN_E8EV_ELECTRIC_RATE = 125UL,
      FI_EN_E8EV_ELECTRIC_DATE_EDIT = 126UL,
      FI_EN_E8EV_ELCETRIC_SEASON_EDIT = 127UL,
      FI_EN_E8HYBRID_ELECTRICSCHEDULE_YEARLY = 128UL,
      FI_EN_E8EV_DETAILS = 129UL,
      FI_EN_E8EV_EFFICIENCY = 130UL,
      FI_EN_E8EV_HISTOGRAM = 131UL,
      FI_EN_E8EV_USAGE = 132UL,
      FI_EN_E8EV_RATE_EDIT = 133UL,
      FI_EN_E8EV_CHARGE_MODE_FAULT = 134UL,
      FI_EN_E8EV_TEMP_DEPARTURETIME_EDIT = 135UL,
      FI_EN_E8EV_TEMP_CHARGE = 136UL,
      FI_EN_E8EV_RATEPERIOD_ADD = 137UL,
      FI_EN_E8EV_RATEPERIOD_SUBTRACT = 138UL,
      FI_EN_E8SEAT_TEMP_STATUS = 143UL,
      FI_EN_E8SETTINGS_AUTOVOLUME = 144UL,
      FI_EN_E8SETTINGS_DISPLAY_MAIN = 145UL,
      FI_EN_E8SETTINGS_DISPLAY_MODE = 146UL,
      FI_EN_E8SETTINGS_DISPLAY_THEMES = 147UL,
      FI_EN_E8SETTINGS_DISPLAY_TOUCHCALIBRATION = 148UL,
      FI_EN_E8SETTINGS_LANGUAGE_MAIN = 149UL,
      FI_EN_E8SETTINGS_PROXIMITY_MODE = 150UL,
      FI_EN_E8SETTINGS_SPEECH_MAIN = 151UL,
      FI_EN_E8SETTINGS_TOUCHPAD_MAIN = 152UL,
      FI_EN_E8SETTINGS_VALET_LOCK = 153UL,
      FI_EN_E8SETTINGS_VEHICLE_MAIN = 154UL,
      FI_EN_E8SETTINGS_SET_DATETIME_MAIN = 155UL,
      FI_EN_E8SETTINGS_TIMEDATE_AUTOSETTIME = 156UL,
      FI_EN_E8SETTINGS_TIMEDATE_SETDATE = 157UL,
      FI_EN_E8SETTINGS_TIMEDATE_SETTIME = 158UL,
      FI_EN_E8SETTINGS_RVC_MAIN = 159UL,
      FI_EN_E8SETTINGS_RVC_REARPARKASSISTSYMBOLS = 160UL,
      FI_EN_E8SETTINGS_VEHICLE_CLIMATEAIRQUAL = 161UL,
      FI_EN_E8SETTINGS_VEHICLE_COLLISIONDETECT = 162UL,
      FI_EN_E8SETTINGS_VEHICLE_COMFORTCONV = 163UL,
      FI_EN_E8SETTINGS_SPORTMODECUSTOM = 164UL,
      FI_EN_E8SETTINGS_SPI_MAIN = 165UL,
      FI_EN_E8SETTINGS_DRIVINGMODE_MAIN = 166UL,
      FI_EN_E8SETTINGS_VEHICLE_HYBRID = 167UL,
      FI_EN_E8SETTINGS_VEHICLE_LIGHTING = 168UL,
      FI_EN_E8SETTINGS_VEHICLE_PTO = 169UL,
      FI_EN_E8SETTINGS_VEHICLE_PWRDOORLCK = 170UL,
      FI_EN_E8SETTINGS_AUDIO_BUTTONBEEP = 171UL,
      FI_EN_E8SETTINGS_RADIO_ALERTTONE = 172UL,
      FI_EN_E8SETTINGS_RADIO_BOSEAUDIOPILOT = 173UL,
      FI_EN_E8SETTINGS_RADIO_CUEVOLUME = 174UL,
      FI_EN_E8SETTINGS_RADIO_MAIN = 175UL,
      FI_EN_E8SETTINGS_RADIO_MANAGE_FAV = 176UL,
      FI_EN_E8SETTINGS_RADIO_MAXSTARTUPVOL = 177UL,
      FI_EN_E8SETTINGS_RADIO_NUMBERFAVS = 178UL,
      FI_EN_E8SETTINGS_VEHICLE_REMLOCK = 179UL,
      FI_EN_E8SETTINGS_FACTRESTORE_MAIN = 180UL,
      FI_EN_E8SETTINGS_OPENSOURCE = 181UL,
      FI_EN_E8SETTINGS_SPORTMODE_MAIN = 182UL,
      FI_EN_E8SETTINGS_TEENDRIVER_MAIN = 183UL,
      FI_EN_E8SETTINGS_TOUCHPAD_CURSOR_TRACKING_SPEED = 184UL,
      FI_EN_E8SETTINGS_TOUCHPAD_INVERSION = 185UL,
      FI_EN_E8SETTINGS_TOUCHPAD_INVERSION_INFO = 186UL,
      FI_EN_E8SETTINGS_TOUCHPAD_SURFACE_INFO = 187UL,
      FI_EN_E8SETTINGS_TOUCHPAD_TOUCHPADSURFACE = 188UL,
      FI_EN_E8SETTINGS_CLIMATE_STATE = 189UL,
      FI_EN_E8SETTINGS_LIGHTING = 190UL,
      FI_EN_E8SETTINGS_WIFI = 191UL,
      FI_EN_E8SETTINGS_WIFI_NETWORKS = 192UL,
      FI_EN_E8DIAG_DEVSCREEN_HMIINFO = 198UL,
      FI_EN_E8DIAG_DEVSCREEN_MAIN_LIST = 199UL,
      FI_EN_E8DIAG_TESTSCREEN = 200UL,
      FI_EN_E8UNKNOWN = 255UL
   };
   tenType enType;

   most_fi_tcl_e8_CntStkApplicationID(most_fi_tcl_e8_CntStkApplicationID::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_CntStkApplicationID();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_CntStkApplicationID& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMSetApproachLightingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMSetApproachLightingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8FLASHLIGHTSON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMSetApproachLightingValue(most_fi_tcl_e8_BCMSetApproachLightingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMSetApproachLightingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMSetApproachLightingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMSetAutoReverseGearRearWiperSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMSetAutoReverseGearRearWiperSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMSetAutoReverseGearRearWiperSettingValue(most_fi_tcl_e8_BCMSetAutoReverseGearRearWiperSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMSetAutoReverseGearRearWiperSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMSetAutoReverseGearRearWiperSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMSetDriverPersonalizationSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMSetDriverPersonalizationSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMSetDriverPersonalizationSettingValue(most_fi_tcl_e8_BCMSetDriverPersonalizationSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMSetDriverPersonalizationSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMSetDriverPersonalizationSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMSetExitLightTimeSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMSetExitLightTimeSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8THIRTYSECONDS = 1UL,
      FI_EN_E8ONEMINUTES = 2UL,
      FI_EN_E8TWOMINUTES = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMSetExitLightTimeSettingValue(most_fi_tcl_e8_BCMSetExitLightTimeSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMSetExitLightTimeSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMSetExitLightTimeSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMSetRemoteStartSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMSetRemoteStartSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMSetRemoteStartSettingValue(most_fi_tcl_e8_BCMSetRemoteStartSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMSetRemoteStartSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMSetRemoteStartSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMSetRemoteStartCooledSeatSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMSetRemoteStartCooledSeatSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMSetRemoteStartCooledSeatSettingValue(most_fi_tcl_e8_BCMSetRemoteStartCooledSeatSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMSetRemoteStartCooledSeatSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMSetRemoteStartCooledSeatSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMSetRemoteStartHeatedSeatSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMSetRemoteStartHeatedSeatSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMSetRemoteStartHeatedSeatSettingValue(most_fi_tcl_e8_BCMSetRemoteStartHeatedSeatSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMSetRemoteStartHeatedSeatSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMSetRemoteStartHeatedSeatSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMSetPowertrainPerformanceSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMSetPowertrainPerformanceSettingValue();
   enum tenType {
      FI_EN_E8NORMALOFF = 0UL,
      FI_EN_E8SPORT = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMSetPowertrainPerformanceSettingValue(most_fi_tcl_e8_BCMSetPowertrainPerformanceSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMSetPowertrainPerformanceSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMSetPowertrainPerformanceSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMSetDaytimeTaillightsSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMSetDaytimeTaillightsSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMSetDaytimeTaillightsSettingValue(most_fi_tcl_e8_BCMSetDaytimeTaillightsSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMSetDaytimeTaillightsSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMSetDaytimeTaillightsSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMSetRemoteStartCooledDrPasSeatSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMSetRemoteStartCooledDrPasSeatSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8DRIVER_AND_PASSENGER = 1UL,
      FI_EN_E8DRIVER = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMSetRemoteStartCooledDrPasSeatSettingValue(most_fi_tcl_e8_BCMSetRemoteStartCooledDrPasSeatSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMSetRemoteStartCooledDrPasSeatSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMSetRemoteStartCooledDrPasSeatSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMSetRemoteStartHeatedDrPasSeatSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMSetRemoteStartHeatedDrPasSeatSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8DRIVER_AND_PASSENGER = 1UL,
      FI_EN_E8DRIVER = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMSetRemoteStartHeatedDrPasSeatSettingValue(most_fi_tcl_e8_BCMSetRemoteStartHeatedDrPasSeatSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMSetRemoteStartHeatedDrPasSeatSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMSetRemoteStartHeatedDrPasSeatSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMSetRainSenseWipersSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMSetRainSenseWipersSettingValue();
   enum tenType {
      FI_EN_E8OFF_ = 0UL,
      FI_EN_E8ON_ = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMSetRainSenseWipersSettingValue(most_fi_tcl_e8_BCMSetRainSenseWipersSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMSetRainSenseWipersSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMSetRainSenseWipersSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMSetRemoteWindowOperationSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMSetRemoteWindowOperationSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMSetRemoteWindowOperationSettingValue(most_fi_tcl_e8_BCMSetRemoteWindowOperationSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMSetRemoteWindowOperationSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMSetRemoteWindowOperationSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMApproachLightingSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMApproachLightingSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8FLASHLIGHTSON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMApproachLightingSettingValue(most_fi_tcl_e8_BCMApproachLightingSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMApproachLightingSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMApproachLightingSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMAutoReverseGearRearWiperSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMAutoReverseGearRearWiperSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMAutoReverseGearRearWiperSettingValue(most_fi_tcl_e8_BCMAutoReverseGearRearWiperSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMAutoReverseGearRearWiperSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMAutoReverseGearRearWiperSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMDriverPersonalizationSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMDriverPersonalizationSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMDriverPersonalizationSettingValue(most_fi_tcl_e8_BCMDriverPersonalizationSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMDriverPersonalizationSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMDriverPersonalizationSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMExitLightTimeSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMExitLightTimeSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8THIRTYSECONDS = 1UL,
      FI_EN_E8ONEMINUTES = 2UL,
      FI_EN_E8TWOMINUTES = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMExitLightTimeSettingValue(most_fi_tcl_e8_BCMExitLightTimeSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMExitLightTimeSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMExitLightTimeSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMRemoteStartSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMRemoteStartSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMRemoteStartSettingValue(most_fi_tcl_e8_BCMRemoteStartSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMRemoteStartSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMRemoteStartSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMRemoteStartCooledSeatSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMRemoteStartCooledSeatSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMRemoteStartCooledSeatSettingValue(most_fi_tcl_e8_BCMRemoteStartCooledSeatSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMRemoteStartCooledSeatSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMRemoteStartCooledSeatSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMRemoteStartHeatedSeatSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMRemoteStartHeatedSeatSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMRemoteStartHeatedSeatSettingValue(most_fi_tcl_e8_BCMRemoteStartHeatedSeatSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMRemoteStartHeatedSeatSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMRemoteStartHeatedSeatSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMPowertrainPerformanceSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMPowertrainPerformanceSettingValue();
   enum tenType {
      FI_EN_E8NORMALOFF = 0UL,
      FI_EN_E8SPORT = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMPowertrainPerformanceSettingValue(most_fi_tcl_e8_BCMPowertrainPerformanceSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMPowertrainPerformanceSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMPowertrainPerformanceSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMDaytimeTaillightsSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMDaytimeTaillightsSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMDaytimeTaillightsSettingValue(most_fi_tcl_e8_BCMDaytimeTaillightsSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMDaytimeTaillightsSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMDaytimeTaillightsSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMRemoteStartCooledDrPasSeatSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMRemoteStartCooledDrPasSeatSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8DRIVER_AND_PASSENGER = 1UL,
      FI_EN_E8DRIVER = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMRemoteStartCooledDrPasSeatSettingValue(most_fi_tcl_e8_BCMRemoteStartCooledDrPasSeatSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMRemoteStartCooledDrPasSeatSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMRemoteStartCooledDrPasSeatSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMRemoteStartHeatedDrPasSeatSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMRemoteStartHeatedDrPasSeatSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8DRIVER_AND_PASSENGER = 1UL,
      FI_EN_E8DRIVER = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMRemoteStartHeatedDrPasSeatSettingValue(most_fi_tcl_e8_BCMRemoteStartHeatedDrPasSeatSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMRemoteStartHeatedDrPasSeatSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMRemoteStartHeatedDrPasSeatSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMRainSenseWipersSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMRainSenseWipersSettingValue();
   enum tenType {
      FI_EN_E8OFF_ = 0UL,
      FI_EN_E8ON_ = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMRainSenseWipersSettingValue(most_fi_tcl_e8_BCMRainSenseWipersSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMRainSenseWipersSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMRainSenseWipersSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_BCMRemoteWindowOperationSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_BCMRemoteWindowOperationSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_BCMRemoteWindowOperationSettingValue(most_fi_tcl_e8_BCMRemoteWindowOperationSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_BCMRemoteWindowOperationSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_BCMRemoteWindowOperationSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckSetAutoLockingSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckSetAutoLockingSettingValue();
   enum tenType {
      FI_EN_E8AUTOLOCKINGOFF = 0UL,
      FI_EN_E8AUTOLOCKINGON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckSetAutoLockingSettingValue(most_fi_tcl_e8_DorLckSetAutoLockingSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckSetAutoLockingSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckSetAutoLockingSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckSetAutoUnlockingSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckSetAutoUnlockingSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8DRIVERDOOR = 1UL,
      FI_EN_E8ALLDOORS = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckSetAutoUnlockingSettingValue(most_fi_tcl_e8_DorLckSetAutoUnlockingSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckSetAutoUnlockingSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckSetAutoUnlockingSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckSetLastDoorClosedLockingSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckSetLastDoorClosedLockingSettingValue();
   enum tenType {
      FI_EN_E8LASTDOORCLOSEDLOCKINGOFF = 0UL,
      FI_EN_E8LASTDOORCLOSEDLOCKINGON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckSetLastDoorClosedLockingSettingValue(most_fi_tcl_e8_DorLckSetLastDoorClosedLockingSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckSetLastDoorClosedLockingSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckSetLastDoorClosedLockingSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckSetManualTransUnlockingSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckSetManualTransUnlockingSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8DRIVERDOOR = 1UL,
      FI_EN_E8ALLDOORS = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckSetManualTransUnlockingSettingValue(most_fi_tcl_e8_DorLckSetManualTransUnlockingSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckSetManualTransUnlockingSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckSetManualTransUnlockingSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckSetPassiveLockingSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckSetPassiveLockingSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL,
      FI_EN_E8ONWITHCHIRP = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckSetPassiveLockingSettingValue(most_fi_tcl_e8_DorLckSetPassiveLockingSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckSetPassiveLockingSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckSetPassiveLockingSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckSetPassiveUnlockingSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckSetPassiveUnlockingSettingValue();
   enum tenType {
      FI_EN_E8DRIVERDOOR = 0UL,
      FI_EN_E8ALLDOORS = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckSetPassiveUnlockingSettingValue(most_fi_tcl_e8_DorLckSetPassiveUnlockingSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckSetPassiveUnlockingSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckSetPassiveUnlockingSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckSetRelockRemoteDoorSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckSetRelockRemoteDoorSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckSetRelockRemoteDoorSettingValue(most_fi_tcl_e8_DorLckSetRelockRemoteDoorSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckSetRelockRemoteDoorSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckSetRelockRemoteDoorSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckSetRelockRemoteUnlockedDoorSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckSetRelockRemoteUnlockedDoorSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckSetRelockRemoteUnlockedDoorSettingValue(most_fi_tcl_e8_DorLckSetRelockRemoteUnlockedDoorSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckSetRelockRemoteUnlockedDoorSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckSetRelockRemoteUnlockedDoorSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckSetRemoteInVehicleReminderSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckSetRemoteInVehicleReminderSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckSetRemoteInVehicleReminderSettingValue(most_fi_tcl_e8_DorLckSetRemoteInVehicleReminderSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckSetRemoteInVehicleReminderSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckSetRemoteInVehicleReminderSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckSetRemoteLockingFeedbackSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckSetRemoteLockingFeedbackSettingValue();
   enum tenType {
      FI_EN_E8HORNCHIRPONLY = 2UL,
      FI_EN_E8HORNANDLIGHTSOFF = 3UL,
      FI_EN_E8HORNANDLIGHTSON = 1UL,
      FI_EN_E8FLASHLIGHTSONLY = 0UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckSetRemoteLockingFeedbackSettingValue(most_fi_tcl_e8_DorLckSetRemoteLockingFeedbackSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckSetRemoteLockingFeedbackSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckSetRemoteLockingFeedbackSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckSetRemoteSlidingDoorSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckSetRemoteSlidingDoorSettingValue();
   enum tenType {
      FI_EN_E8COURTESY = 0UL,
      FI_EN_E8SECURITY = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckSetRemoteSlidingDoorSettingValue(most_fi_tcl_e8_DorLckSetRemoteSlidingDoorSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckSetRemoteSlidingDoorSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckSetRemoteSlidingDoorSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckSetRemoteUnlocklLightingFeedbackSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckSetRemoteUnlocklLightingFeedbackSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8FLASH = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckSetRemoteUnlocklLightingFeedbackSettingValue(most_fi_tcl_e8_DorLckSetRemoteUnlocklLightingFeedbackSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckSetRemoteUnlocklLightingFeedbackSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckSetRemoteUnlocklLightingFeedbackSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckSetSelectiveUnlockingSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckSetSelectiveUnlockingSettingValue();
   enum tenType {
      FI_EN_E8DRIVERDOORONLY = 0UL,
      FI_EN_E8ALLDOORS = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckSetSelectiveUnlockingSettingValue(most_fi_tcl_e8_DorLckSetSelectiveUnlockingSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckSetSelectiveUnlockingSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckSetSelectiveUnlockingSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckSetOpenDoorAntiLockOutSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckSetOpenDoorAntiLockOutSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckSetOpenDoorAntiLockOutSettingValue(most_fi_tcl_e8_DorLckSetOpenDoorAntiLockOutSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckSetOpenDoorAntiLockOutSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckSetOpenDoorAntiLockOutSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckAutoLockingSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckAutoLockingSettingValue();
   enum tenType {
      FI_EN_E8AUTOLOCKINGOFF = 0UL,
      FI_EN_E8AUTOLOCKINGON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckAutoLockingSettingValue(most_fi_tcl_e8_DorLckAutoLockingSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckAutoLockingSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckAutoLockingSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckAutoUnlockingSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckAutoUnlockingSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8DRIVERDOOR = 1UL,
      FI_EN_E8ALLDOORS = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckAutoUnlockingSettingValue(most_fi_tcl_e8_DorLckAutoUnlockingSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckAutoUnlockingSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckAutoUnlockingSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckLastDoorClosedLockingSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckLastDoorClosedLockingSettingValue();
   enum tenType {
      FI_EN_E8LASTDOORCLOSEDLOCKINGOFF = 0UL,
      FI_EN_E8LASTDOORCLOSEDLOCKINGON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckLastDoorClosedLockingSettingValue(most_fi_tcl_e8_DorLckLastDoorClosedLockingSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckLastDoorClosedLockingSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckLastDoorClosedLockingSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckManualTransAutoUnlockingSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckManualTransAutoUnlockingSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8DRIVERDOOR = 1UL,
      FI_EN_E8ALLDOORS = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckManualTransAutoUnlockingSettingValue(most_fi_tcl_e8_DorLckManualTransAutoUnlockingSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckManualTransAutoUnlockingSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckManualTransAutoUnlockingSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckPassiveLockingSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckPassiveLockingSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL,
      FI_EN_E8ONWITHCHIRP = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckPassiveLockingSettingValue(most_fi_tcl_e8_DorLckPassiveLockingSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckPassiveLockingSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckPassiveLockingSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckPassiveUnlockingSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckPassiveUnlockingSettingValue();
   enum tenType {
      FI_EN_E8DRIVERDOOR = 0UL,
      FI_EN_E8ALLDOORS = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckPassiveUnlockingSettingValue(most_fi_tcl_e8_DorLckPassiveUnlockingSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckPassiveUnlockingSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckPassiveUnlockingSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckRelockRemoteDoorSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckRelockRemoteDoorSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckRelockRemoteDoorSettingValue(most_fi_tcl_e8_DorLckRelockRemoteDoorSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckRelockRemoteDoorSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckRelockRemoteDoorSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckRelockRemoteUnlockedDoorSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckRelockRemoteUnlockedDoorSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckRelockRemoteUnlockedDoorSettingValue(most_fi_tcl_e8_DorLckRelockRemoteUnlockedDoorSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckRelockRemoteUnlockedDoorSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckRelockRemoteUnlockedDoorSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckRemoteInVehicleReminderSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckRemoteInVehicleReminderSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckRemoteInVehicleReminderSettingValue(most_fi_tcl_e8_DorLckRemoteInVehicleReminderSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckRemoteInVehicleReminderSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckRemoteInVehicleReminderSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckRemoteLockingFeedbackSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckRemoteLockingFeedbackSettingValue();
   enum tenType {
      FI_EN_E8HORNCHIRPONLY = 2UL,
      FI_EN_E8HORNANDLIGHTSOFF = 3UL,
      FI_EN_E8HORNANDLIGHTSON = 1UL,
      FI_EN_E8FLASHLIGHTSONLY = 0UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckRemoteLockingFeedbackSettingValue(most_fi_tcl_e8_DorLckRemoteLockingFeedbackSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckRemoteLockingFeedbackSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckRemoteLockingFeedbackSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckRemoteSlidingDoorSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckRemoteSlidingDoorSettingValue();
   enum tenType {
      FI_EN_E8COURTESY = 0UL,
      FI_EN_E8SECURITY = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckRemoteSlidingDoorSettingValue(most_fi_tcl_e8_DorLckRemoteSlidingDoorSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckRemoteSlidingDoorSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckRemoteSlidingDoorSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckRemoteUnlockLightingFeedbackSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckRemoteUnlockLightingFeedbackSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8FLASH = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckRemoteUnlockLightingFeedbackSettingValue(most_fi_tcl_e8_DorLckRemoteUnlockLightingFeedbackSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckRemoteUnlockLightingFeedbackSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckRemoteUnlockLightingFeedbackSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckSelectiveUnlockingSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckSelectiveUnlockingSettingValue();
   enum tenType {
      FI_EN_E8DRIVERDOORONLY = 0UL,
      FI_EN_E8ALLDOORS = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckSelectiveUnlockingSettingValue(most_fi_tcl_e8_DorLckSelectiveUnlockingSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckSelectiveUnlockingSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckSelectiveUnlockingSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DorLckOpenDoorAntiLockOutSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DorLckOpenDoorAntiLockOutSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DorLckOpenDoorAntiLockOutSettingValue(most_fi_tcl_e8_DorLckOpenDoorAntiLockOutSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DorLckOpenDoorAntiLockOutSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DorLckOpenDoorAntiLockOutSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCSetAirQualitySensorSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCSetAirQualitySensorSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8LOWSENSITIVITY = 1UL,
      FI_EN_E8HISENSITIVITY = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCSetAirQualitySensorSettingValue(most_fi_tcl_e8_ECCSetAirQualitySensorSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCSetAirQualitySensorSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCSetAirQualitySensorSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCSetAutoCoolSeatsSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCSetAutoCoolSeatsSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCSetAutoCoolSeatsSettingValue(most_fi_tcl_e8_ECCSetAutoCoolSeatsSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCSetAutoCoolSeatsSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCSetAutoCoolSeatsSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCSetAutoFanSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCSetAutoFanSettingValue();
   enum tenType {
      FI_EN_E8LOW = 0UL,
      FI_EN_E8NORMAL = 1UL,
      FI_EN_E8HIGH = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCSetAutoFanSettingValue(most_fi_tcl_e8_ECCSetAutoFanSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCSetAutoFanSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCSetAutoFanSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCSetAutoHeatedSeatsSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCSetAutoHeatedSeatsSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCSetAutoHeatedSeatsSettingValue(most_fi_tcl_e8_ECCSetAutoHeatedSeatsSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCSetAutoHeatedSeatsSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCSetAutoHeatedSeatsSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCSetAutoZoneTempSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCSetAutoZoneTempSettingValue();
   enum tenType {
      FI_EN_E8SINGLEZONE = 0UL,
      FI_EN_E8DUALZONE = 1UL,
      FI_EN_E8LAST_SETTING = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCSetAutoZoneTempSettingValue(most_fi_tcl_e8_ECCSetAutoZoneTempSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCSetAutoZoneTempSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCSetAutoZoneTempSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCSetHVACRemoteStartSettingSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCSetHVACRemoteStartSettingSettingValue();
   enum tenType {
      FI_EN_E8AUTOMATIC = 0UL,
      FI_EN_E8LAST_SETTING = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCSetHVACRemoteStartSettingSettingValue(most_fi_tcl_e8_ECCSetHVACRemoteStartSettingSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCSetHVACRemoteStartSettingSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCSetHVACRemoteStartSettingSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCSetACModeSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCSetACModeSettingValue();
   enum tenType {
      FI_EN_E8AC_OFF = 0UL,
      FI_EN_E8AC_ON = 1UL,
      FI_EN_E8LAST_SETTING = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCSetACModeSettingValue(most_fi_tcl_e8_ECCSetACModeSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCSetACModeSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCSetACModeSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCSetAutoDefogSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCSetAutoDefogSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCSetAutoDefogSettingValue(most_fi_tcl_e8_ECCSetAutoDefogSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCSetAutoDefogSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCSetAutoDefogSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCSetRearZoneStartupSettingsValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCSetRearZoneStartupSettingsValue();
   enum tenType {
      FI_EN_E8REAR_OFF = 0UL,
      FI_EN_E8REAR_MIMIC_FRONT = 1UL,
      FI_EN_E8REAR_LAST_KNOWN = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCSetRearZoneStartupSettingsValue(most_fi_tcl_e8_ECCSetRearZoneStartupSettingsValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCSetRearZoneStartupSettingsValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCSetRearZoneStartupSettingsValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCSetRearDefogStartupSettingsValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCSetRearDefogStartupSettingsValue();
   enum tenType {
      FI_EN_E8MANUAL = 0UL,
      FI_EN_E8AUTOMATIC = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCSetRearDefogStartupSettingsValue(most_fi_tcl_e8_ECCSetRearDefogStartupSettingsValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCSetRearDefogStartupSettingsValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCSetRearDefogStartupSettingsValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCSetPollutionControlSettingsValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCSetPollutionControlSettingsValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCSetPollutionControlSettingsValue(most_fi_tcl_e8_ECCSetPollutionControlSettingsValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCSetPollutionControlSettingsValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCSetPollutionControlSettingsValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCSetIonizerSettingsValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCSetIonizerSettingsValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCSetIonizerSettingsValue(most_fi_tcl_e8_ECCSetIonizerSettingsValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCSetIonizerSettingsValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCSetIonizerSettingsValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCSetAutoAirDistrSettingsValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCSetAutoAirDistrSettingsValue();
   enum tenType {
      FI_EN_E8DIRECT_AIRFLOW = 0UL,
      FI_EN_E8DIFFUSE_AIRFLOW = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCSetAutoAirDistrSettingsValue(most_fi_tcl_e8_ECCSetAutoAirDistrSettingsValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCSetAutoAirDistrSettingsValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCSetAutoAirDistrSettingsValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCAirQualitySensorSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCAirQualitySensorSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8LOWSENSITIVITY = 1UL,
      FI_EN_E8HISENSITIVITY = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCAirQualitySensorSettingValue(most_fi_tcl_e8_ECCAirQualitySensorSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCAirQualitySensorSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCAirQualitySensorSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCAutoCoolSeatsSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCAutoCoolSeatsSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCAutoCoolSeatsSettingValue(most_fi_tcl_e8_ECCAutoCoolSeatsSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCAutoCoolSeatsSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCAutoCoolSeatsSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCAutoFanSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCAutoFanSettingValue();
   enum tenType {
      FI_EN_E8LOW = 0UL,
      FI_EN_E8NORMAL = 1UL,
      FI_EN_E8HIGH = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCAutoFanSettingValue(most_fi_tcl_e8_ECCAutoFanSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCAutoFanSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCAutoFanSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCAutoHeatedSeatsSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCAutoHeatedSeatsSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCAutoHeatedSeatsSettingValue(most_fi_tcl_e8_ECCAutoHeatedSeatsSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCAutoHeatedSeatsSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCAutoHeatedSeatsSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCAutoZoneTempSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCAutoZoneTempSettingValue();
   enum tenType {
      FI_EN_E8SINGLEZONE = 0UL,
      FI_EN_E8DUALZONE = 1UL,
      FI_EN_E8LAST_SETTING = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCAutoZoneTempSettingValue(most_fi_tcl_e8_ECCAutoZoneTempSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCAutoZoneTempSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCAutoZoneTempSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCHVACRemoteStartSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCHVACRemoteStartSettingValue();
   enum tenType {
      FI_EN_E8AUTOMATIC = 0UL,
      FI_EN_E8LAST_SETTING = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCHVACRemoteStartSettingValue(most_fi_tcl_e8_ECCHVACRemoteStartSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCHVACRemoteStartSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCHVACRemoteStartSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCACModeSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCACModeSettingValue();
   enum tenType {
      FI_EN_E8AC_OFF = 0UL,
      FI_EN_E8AC_ON = 1UL,
      FI_EN_E8LAST_SETTING = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCACModeSettingValue(most_fi_tcl_e8_ECCACModeSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCACModeSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCACModeSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCAutoDefogSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCAutoDefogSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCAutoDefogSettingValue(most_fi_tcl_e8_ECCAutoDefogSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCAutoDefogSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCAutoDefogSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCRearZoneStartupSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCRearZoneStartupSettingValue();
   enum tenType {
      FI_EN_E8REAR_OFF = 0UL,
      FI_EN_E8REAR_MIMIC_FRONT = 1UL,
      FI_EN_E8REAR_LAST_KNOWN = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCRearZoneStartupSettingValue(most_fi_tcl_e8_ECCRearZoneStartupSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCRearZoneStartupSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCRearZoneStartupSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCRearDefogStartupSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCRearDefogStartupSettingValue();
   enum tenType {
      FI_EN_E8MANUAL = 0UL,
      FI_EN_E8AUTOMATIC = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCRearDefogStartupSettingValue(most_fi_tcl_e8_ECCRearDefogStartupSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCRearDefogStartupSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCRearDefogStartupSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCPollutionControlSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCPollutionControlSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCPollutionControlSettingValue(most_fi_tcl_e8_ECCPollutionControlSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCPollutionControlSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCPollutionControlSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCIonizerSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCIonizerSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCIonizerSettingValue(most_fi_tcl_e8_ECCIonizerSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCIonizerSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCIonizerSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ECCAutoAirDistrSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ECCAutoAirDistrSettingValue();
   enum tenType {
      FI_EN_E8DIRECT_AIRFLOW = 0UL,
      FI_EN_E8DIFFUSE_AIRFLOW = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_ECCAutoAirDistrSettingValue(most_fi_tcl_e8_ECCAutoAirDistrSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ECCAutoAirDistrSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ECCAutoAirDistrSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_IPCSetSpeedScaleIlluminationSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_IPCSetSpeedScaleIlluminationSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_IPCSetSpeedScaleIlluminationSettingValue(most_fi_tcl_e8_IPCSetSpeedScaleIlluminationSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_IPCSetSpeedScaleIlluminationSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_IPCSetSpeedScaleIlluminationSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_IPCSetSportModeBackLightingSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_IPCSetSportModeBackLightingSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_IPCSetSportModeBackLightingSettingValue(most_fi_tcl_e8_IPCSetSportModeBackLightingSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_IPCSetSportModeBackLightingSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_IPCSetSportModeBackLightingSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_IPCSetEcoModeTelltaleEnableSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_IPCSetEcoModeTelltaleEnableSettingValue();
   enum tenType {
      FI_EN_E8DISABLED = 0UL,
      FI_EN_E8ENABLED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_IPCSetEcoModeTelltaleEnableSettingValue(most_fi_tcl_e8_IPCSetEcoModeTelltaleEnableSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_IPCSetEcoModeTelltaleEnableSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_IPCSetEcoModeTelltaleEnableSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_IPCSpeedScaleIlluminationSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_IPCSpeedScaleIlluminationSettingValue();
   enum tenType {
      FI_EN_E8FULL = 0UL,
      FI_EN_E8PARTIAL = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_IPCSpeedScaleIlluminationSettingValue(most_fi_tcl_e8_IPCSpeedScaleIlluminationSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_IPCSpeedScaleIlluminationSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_IPCSpeedScaleIlluminationSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_IPCSportModeBacklightingSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_IPCSportModeBacklightingSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_IPCSportModeBacklightingSettingValue(most_fi_tcl_e8_IPCSportModeBacklightingSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_IPCSportModeBacklightingSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_IPCSportModeBacklightingSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_IPCEcoModeTelltaleEnableSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_IPCEcoModeTelltaleEnableSettingValue();
   enum tenType {
      FI_EN_E8DISABLED = 0UL,
      FI_EN_E8ENABLED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_IPCEcoModeTelltaleEnableSettingValue(most_fi_tcl_e8_IPCEcoModeTelltaleEnableSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_IPCEcoModeTelltaleEnableSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_IPCEcoModeTelltaleEnableSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MemSetSetDriverSeatEasyExitSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MemSetSetDriverSeatEasyExitSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_MemSetSetDriverSeatEasyExitSettingValue(most_fi_tcl_e8_MemSetSetDriverSeatEasyExitSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MemSetSetDriverSeatEasyExitSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MemSetSetDriverSeatEasyExitSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MemSetSetFoldingMirrorsSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MemSetSetFoldingMirrorsSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_MemSetSetFoldingMirrorsSettingValue(most_fi_tcl_e8_MemSetSetFoldingMirrorsSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MemSetSetFoldingMirrorsSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MemSetSetFoldingMirrorsSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MemSetSetMemoryColumnSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MemSetSetMemoryColumnSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8COLUMNIN = 1UL,
      FI_EN_E8COLUMNUP = 2UL,
      FI_EN_E8COLUMNINANDUP = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_MemSetSetMemoryColumnSettingValue(most_fi_tcl_e8_MemSetSetMemoryColumnSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MemSetSetMemoryColumnSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MemSetSetMemoryColumnSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MemSetSetParkingTiltMirrorsSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MemSetSetParkingTiltMirrorsSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8DRIVERPASSENGER = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_MemSetSetParkingTiltMirrorsSettingValue(most_fi_tcl_e8_MemSetSetParkingTiltMirrorsSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MemSetSetParkingTiltMirrorsSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MemSetSetParkingTiltMirrorsSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MemSetSetRemoteMemoryUnlockRecallSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MemSetSetRemoteMemoryUnlockRecallSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_MemSetSetRemoteMemoryUnlockRecallSettingValue(most_fi_tcl_e8_MemSetSetRemoteMemoryUnlockRecallSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MemSetSetRemoteMemoryUnlockRecallSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MemSetSetRemoteMemoryUnlockRecallSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MemSetSetMemoryTiltColumnSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MemSetSetMemoryTiltColumnSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8COLUMNUP = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_MemSetSetMemoryTiltColumnSettingValue(most_fi_tcl_e8_MemSetSetMemoryTiltColumnSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MemSetSetMemoryTiltColumnSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MemSetSetMemoryTiltColumnSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MemSetSetCrashAvoidanceAlertTypeSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MemSetSetCrashAvoidanceAlertTypeSettingValue();
   enum tenType {
      FI_EN_E8CHIME = 0UL,
      FI_EN_E8SEAT = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_MemSetSetCrashAvoidanceAlertTypeSettingValue(most_fi_tcl_e8_MemSetSetCrashAvoidanceAlertTypeSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MemSetSetCrashAvoidanceAlertTypeSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MemSetSetCrashAvoidanceAlertTypeSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MemSetSetReverseTiltMirrorsSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MemSetSetReverseTiltMirrorsSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8DRIVER_AND_PASSENGER = 1UL,
      FI_EN_E8DRIVER = 2UL,
      FI_EN_E8PASSENGER = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_MemSetSetReverseTiltMirrorsSettingValue(most_fi_tcl_e8_MemSetSetReverseTiltMirrorsSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MemSetSetReverseTiltMirrorsSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MemSetSetReverseTiltMirrorsSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MemSetSetEasyExitSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MemSetSetEasyExitSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_MemSetSetEasyExitSettingValue(most_fi_tcl_e8_MemSetSetEasyExitSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MemSetSetEasyExitSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MemSetSetEasyExitSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MemSetSetMemoryRemoteRecallSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MemSetSetMemoryRemoteRecallSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8AT_UNLOCK = 1UL,
      FI_EN_E8AT_START = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_MemSetSetMemoryRemoteRecallSettingValue(most_fi_tcl_e8_MemSetSetMemoryRemoteRecallSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MemSetSetMemoryRemoteRecallSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MemSetSetMemoryRemoteRecallSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MemSetDriverSeatEasyExitSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MemSetDriverSeatEasyExitSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_MemSetDriverSeatEasyExitSettingValue(most_fi_tcl_e8_MemSetDriverSeatEasyExitSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MemSetDriverSeatEasyExitSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MemSetDriverSeatEasyExitSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MemSetFoldingMirrorsSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MemSetFoldingMirrorsSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_MemSetFoldingMirrorsSettingValue(most_fi_tcl_e8_MemSetFoldingMirrorsSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MemSetFoldingMirrorsSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MemSetFoldingMirrorsSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MemSetMemoryColumnSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MemSetMemoryColumnSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8COLUMNIN = 1UL,
      FI_EN_E8COLUMNUP = 2UL,
      FI_EN_E8COLUMNINANDUP = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_MemSetMemoryColumnSettingValue(most_fi_tcl_e8_MemSetMemoryColumnSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MemSetMemoryColumnSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MemSetMemoryColumnSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MemSetParkingTiltMirrorsSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MemSetParkingTiltMirrorsSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8DRIVERPASSENGER = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_MemSetParkingTiltMirrorsSettingValue(most_fi_tcl_e8_MemSetParkingTiltMirrorsSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MemSetParkingTiltMirrorsSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MemSetParkingTiltMirrorsSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MemSetRemoteMemoryUnlockRecallSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MemSetRemoteMemoryUnlockRecallSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_MemSetRemoteMemoryUnlockRecallSettingValue(most_fi_tcl_e8_MemSetRemoteMemoryUnlockRecallSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MemSetRemoteMemoryUnlockRecallSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MemSetRemoteMemoryUnlockRecallSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MemSetMemoryTiltColumnSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MemSetMemoryTiltColumnSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8COLUMNUP = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_MemSetMemoryTiltColumnSettingValue(most_fi_tcl_e8_MemSetMemoryTiltColumnSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MemSetMemoryTiltColumnSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MemSetMemoryTiltColumnSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MemSetCrashAvoidanceAlertTypeSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MemSetCrashAvoidanceAlertTypeSettingValue();
   enum tenType {
      FI_EN_E8CHIME = 0UL,
      FI_EN_E8SEAT = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_MemSetCrashAvoidanceAlertTypeSettingValue(most_fi_tcl_e8_MemSetCrashAvoidanceAlertTypeSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MemSetCrashAvoidanceAlertTypeSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MemSetCrashAvoidanceAlertTypeSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MemSetReverseTiltMirrorsSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MemSetReverseTiltMirrorsSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8DRIVER_AND_PASSENGER = 1UL,
      FI_EN_E8DRIVER = 2UL,
      FI_EN_E8PASSENGER = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_MemSetReverseTiltMirrorsSettingValue(most_fi_tcl_e8_MemSetReverseTiltMirrorsSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MemSetReverseTiltMirrorsSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MemSetReverseTiltMirrorsSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MemSetEasyExitSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MemSetEasyExitSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_MemSetEasyExitSettingValue(most_fi_tcl_e8_MemSetEasyExitSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MemSetEasyExitSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MemSetEasyExitSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MemSetMemoryRemoteRecallSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MemSetMemoryRemoteRecallSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8AT_UNLOCK = 1UL,
      FI_EN_E8AT_START = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_MemSetMemoryRemoteRecallSettingValue(most_fi_tcl_e8_MemSetMemoryRemoteRecallSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MemSetMemoryRemoteRecallSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MemSetMemoryRemoteRecallSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ObjCmpSet24GHzRadarsSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ObjCmpSet24GHzRadarsSettingValue();
   enum tenType {
      FI_EN_E8_OFF_DISABLES_FEATURES = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_ObjCmpSet24GHzRadarsSettingValue(most_fi_tcl_e8_ObjCmpSet24GHzRadarsSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ObjCmpSet24GHzRadarsSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ObjCmpSet24GHzRadarsSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ObjCmp24GHzRadarsSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ObjCmp24GHzRadarsSettingValue();
   enum tenType {
      FI_EN_E8_OFF_DISABLES_FEATURES = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_ObjCmp24GHzRadarsSettingValue(most_fi_tcl_e8_ObjCmp24GHzRadarsSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ObjCmp24GHzRadarsSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ObjCmp24GHzRadarsSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PASetParkAssistSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PASetParkAssistSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_PASetParkAssistSettingValue(most_fi_tcl_e8_PASetParkAssistSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PASetParkAssistSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PASetParkAssistSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PASetParkAssistWithTowbarSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PASetParkAssistWithTowbarSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL,
      FI_EN_E8TOWBAR = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_PASetParkAssistWithTowbarSettingValue(most_fi_tcl_e8_PASetParkAssistWithTowbarSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PASetParkAssistWithTowbarSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PASetParkAssistWithTowbarSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PAParkAssistSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PAParkAssistSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_PAParkAssistSettingValue(most_fi_tcl_e8_PAParkAssistSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PAParkAssistSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PAParkAssistSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PAParkAssistWithTowbarSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PAParkAssistWithTowbarSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL,
      FI_EN_E8TOWBAR = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_PAParkAssistWithTowbarSettingValue(most_fi_tcl_e8_PAParkAssistWithTowbarSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PAParkAssistWithTowbarSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PAParkAssistWithTowbarSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_SBZASetSideBlindZoneAlertSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_SBZASetSideBlindZoneAlertSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_SBZASetSideBlindZoneAlertSettingValue(most_fi_tcl_e8_SBZASetSideBlindZoneAlertSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_SBZASetSideBlindZoneAlertSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_SBZASetSideBlindZoneAlertSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_SBZASetRearCrossTrafficAlertSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_SBZASetRearCrossTrafficAlertSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_SBZASetRearCrossTrafficAlertSettingValue(most_fi_tcl_e8_SBZASetRearCrossTrafficAlertSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_SBZASetRearCrossTrafficAlertSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_SBZASetRearCrossTrafficAlertSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_SBZASetLaneChangeAlertSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_SBZASetLaneChangeAlertSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_SBZASetLaneChangeAlertSettingValue(most_fi_tcl_e8_SBZASetLaneChangeAlertSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_SBZASetLaneChangeAlertSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_SBZASetLaneChangeAlertSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_SBZASideBlindZoneAlertSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_SBZASideBlindZoneAlertSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_SBZASideBlindZoneAlertSettingValue(most_fi_tcl_e8_SBZASideBlindZoneAlertSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_SBZASideBlindZoneAlertSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_SBZASideBlindZoneAlertSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_SBZARearCrossTrafficAlertSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_SBZARearCrossTrafficAlertSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_SBZARearCrossTrafficAlertSettingValue(most_fi_tcl_e8_SBZARearCrossTrafficAlertSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_SBZARearCrossTrafficAlertSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_SBZARearCrossTrafficAlertSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_SBZALaneChangeAlertSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_SBZALaneChangeAlertSettingValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_SBZALaneChangeAlertSettingValue(most_fi_tcl_e8_SBZALaneChangeAlertSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_SBZALaneChangeAlertSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_SBZALaneChangeAlertSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DABTunDABStationListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DABTunDABStationListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DABTunDABStationListItem& operator=(const most_fi_tcl_DABTunDABStationListItem& coRef);
   most_fi_tcl_DABTunDABStationListItem(const most_fi_tcl_DABTunDABStationListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32ObjectID;
   tU32 u32Frequency;
   most_fi_tcl_String sChannelLabel;
   virtual ~most_fi_tcl_DABTunDABStationListItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DABTunDABStationListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DABTunDABStationList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DABTunDABStationList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DABTunDABStationList& operator=(const most_fi_tcl_DABTunDABStationList& coRef);
   most_fi_tcl_DABTunDABStationList(const most_fi_tcl_DABTunDABStationList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_DABTunDABStationListItem*, std::allocator<most_fi_tcl_DABTunDABStationListItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_DABTunDABStationListItem, std::allocator<most_fi_tcl_DABTunDABStationListItem> >oItems;

   most_fi_tcl_DABTunDABStationListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_DABTunDABStationList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DABTunDABStationList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DABTunServiceTypeFlag : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DABTunServiceTypeFlag();
   enum tenType {
      FI_EN_E8SERVICE_LABEL = 0UL,
      FI_EN_E8SERVICE_COMPONENT_LABEL = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DABTunServiceTypeFlag(most_fi_tcl_e8_DABTunServiceTypeFlag::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DABTunServiceTypeFlag();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DABTunServiceTypeFlag& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DABTunDABStationListResultItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DABTunDABStationListResultItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DABTunDABStationListResultItem& operator=(const most_fi_tcl_DABTunDABStationListResultItem& coRef);
   most_fi_tcl_DABTunDABStationListResultItem(const most_fi_tcl_DABTunDABStationListResultItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32ObjectID;
   tU32 u32Frequency;
   most_fi_tcl_String sChannelLabel;
   most_fi_tcl_String sEnsembleLabel;
   tU16 u16EnsembleCharFlag;
   most_fi_tcl_String sServiceLabel;
   tU16 u16ServiceLabelCharFlag;
   most_fi_tcl_String sServiceComponentLabel;
   tU16 u16ServiceCompLabelCharFlag;
   tU8 u8PTYCategory;
   most_fi_tcl_e8_DABTunServiceTypeFlag e8ServiceTypeFlag;
   virtual ~most_fi_tcl_DABTunDABStationListResultItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DABTunDABStationListResultItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DABTunDABStationListResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DABTunDABStationListResult() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DABTunDABStationListResult& operator=(const most_fi_tcl_DABTunDABStationListResult& coRef);
   most_fi_tcl_DABTunDABStationListResult(const most_fi_tcl_DABTunDABStationListResult& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_DABTunDABStationListResultItem*, std::allocator<most_fi_tcl_DABTunDABStationListResultItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_DABTunDABStationListResultItem, std::allocator<most_fi_tcl_DABTunDABStationListResultItem> >oItems;

   most_fi_tcl_DABTunDABStationListResultItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_DABTunDABStationListResult();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DABTunDABStationListResult& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DABTunDABObjectIDList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DABTunDABObjectIDList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DABTunDABObjectIDList& operator=(const most_fi_tcl_DABTunDABObjectIDList& coRef);
   most_fi_tcl_DABTunDABObjectIDList(const most_fi_tcl_DABTunDABObjectIDList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU32, std::allocator<tU32> >u32Items;

   tU32& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_DABTunDABObjectIDList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DABTunDABObjectIDList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DABTunBroadcastType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DABTunBroadcastType();
   enum tenType {
      FI_EN_E8DAB = 0UL,
      FI_EN_E8DAB_PLUS = 1UL,
      FI_EN_E8DMB = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_DABTunBroadcastType(most_fi_tcl_e8_DABTunBroadcastType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DABTunBroadcastType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DABTunBroadcastType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DABTunDynamicLabelPlusCmd : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DABTunDynamicLabelPlusCmd() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DABTunDynamicLabelPlusCmd& operator=(const most_fi_tcl_DABTunDynamicLabelPlusCmd& coRef);
   most_fi_tcl_DABTunDynamicLabelPlusCmd(const most_fi_tcl_DABTunDynamicLabelPlusCmd& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU8, std::allocator<tU8> >u8Items;

   tU8& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_DABTunDynamicLabelPlusCmd();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DABTunDynamicLabelPlusCmd& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DABTunDABTuningStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DABTunDABTuningStatus();
   enum tenType {
      FI_EN_E8NO_DAB_STATIONS_AVAILABLE = 0UL,
      FI_EN_E8TUNING = 1UL,
      FI_EN_E8REMOVE_TUNING_MESSAGE = 2UL,
      FI_EN_E8DAB_STATION_NOT_AVAILABLE = 3UL,
      FI_EN_E8DAB_DAB_LINKING_IN_PROGRESS = 4UL,
      FI_EN_E8DAB_FM_LINKING_ACTIVE = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_DABTunDABTuningStatus(most_fi_tcl_e8_DABTunDABTuningStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DABTunDABTuningStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DABTunDABTuningStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DABTunDABWaveBandSwitch : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DABTunDABWaveBandSwitch();
   enum tenType {
      FI_EN_E8BAND_III_PLUS_L_BAND = 0UL,
      FI_EN_E8BAND_III_ONLY = 1UL,
      FI_EN_E8L_BAND_ONLY = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_DABTunDABWaveBandSwitch(most_fi_tcl_e8_DABTunDABWaveBandSwitch::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DABTunDABWaveBandSwitch();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DABTunDABWaveBandSwitch& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DABTunStationListUpdateStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DABTunStationListUpdateStatus();
   enum tenType {
      FI_EN_E8DAB_STATION_LIST_UPDATE_INACTIVE = 0UL,
      FI_EN_E8DAB_STATION_LIST_UPDATE_STARTED = 1UL,
      FI_EN_E8DAB_STATION_LIST_UPDATE_IN_PROGRESS = 2UL,
      FI_EN_E8DAB_STATION_LIST_UPDATE_INTERRUPTED = 3UL,
      FI_EN_E8DAB_STATION_LIST_UPDATE_CANCELLED = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_DABTunStationListUpdateStatus(most_fi_tcl_e8_DABTunStationListUpdateStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DABTunStationListUpdateStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DABTunStationListUpdateStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_b16_DABTunMask : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_b16_DABTunMask();

   enum {
      FI_C_U16_BIT_BSID_FILTER_ACTIVE = 0x1UL,
      FI_C_U16_BIT_BMGS_FILTER_ACTIVE = 0x2UL,
      FI_C_U16_BIT_BAFI_FILTER_ACTIVE = 0x4UL,
      FI_C_U16_BIT_BCC_FILTER_ACTIVE = 0x8UL,
      FI_C_U16_BIT_BLTN_FILTER_ACTIVE = 0x10UL,
      FI_C_U16_BIT_BST_FILTER_ACTIVE = 0x20UL,
      FI_C_U16_BIT_BSPN_FILTER_ACTIVE = 0x40UL,
      FI_C_U16_BIT_BXX_FILTER_ACTIVE = 0x80UL
   };
   tBool bbSID_FILTER_ACTIVE() const {return (u16Value & FI_C_U16_BIT_BSID_FILTER_ACTIVE) != 0; }
   tVoid vSetbSID_FILTER_ACTIVE(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_BSID_FILTER_ACTIVE):(u16Value & ~(tU16)FI_C_U16_BIT_BSID_FILTER_ACTIVE)); }
   tBool bbMGS_FILTER_ACTIVE() const {return (u16Value & FI_C_U16_BIT_BMGS_FILTER_ACTIVE) != 0; }
   tVoid vSetbMGS_FILTER_ACTIVE(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_BMGS_FILTER_ACTIVE):(u16Value & ~(tU16)FI_C_U16_BIT_BMGS_FILTER_ACTIVE)); }
   tBool bbAFI_FILTER_ACTIVE() const {return (u16Value & FI_C_U16_BIT_BAFI_FILTER_ACTIVE) != 0; }
   tVoid vSetbAFI_FILTER_ACTIVE(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_BAFI_FILTER_ACTIVE):(u16Value & ~(tU16)FI_C_U16_BIT_BAFI_FILTER_ACTIVE)); }
   tBool bbCC_FILTER_ACTIVE() const {return (u16Value & FI_C_U16_BIT_BCC_FILTER_ACTIVE) != 0; }
   tVoid vSetbCC_FILTER_ACTIVE(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_BCC_FILTER_ACTIVE):(u16Value & ~(tU16)FI_C_U16_BIT_BCC_FILTER_ACTIVE)); }
   tBool bbLTN_FILTER_ACTIVE() const {return (u16Value & FI_C_U16_BIT_BLTN_FILTER_ACTIVE) != 0; }
   tVoid vSetbLTN_FILTER_ACTIVE(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_BLTN_FILTER_ACTIVE):(u16Value & ~(tU16)FI_C_U16_BIT_BLTN_FILTER_ACTIVE)); }
   tBool bbST_FILTER_ACTIVE() const {return (u16Value & FI_C_U16_BIT_BST_FILTER_ACTIVE) != 0; }
   tVoid vSetbST_FILTER_ACTIVE(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_BST_FILTER_ACTIVE):(u16Value & ~(tU16)FI_C_U16_BIT_BST_FILTER_ACTIVE)); }
   tBool bbSPN_FILTER_ACTIVE() const {return (u16Value & FI_C_U16_BIT_BSPN_FILTER_ACTIVE) != 0; }
   tVoid vSetbSPN_FILTER_ACTIVE(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_BSPN_FILTER_ACTIVE):(u16Value & ~(tU16)FI_C_U16_BIT_BSPN_FILTER_ACTIVE)); }
   tBool bbXX_FILTER_ACTIVE() const {return (u16Value & FI_C_U16_BIT_BXX_FILTER_ACTIVE) != 0; }
   tVoid vSetbXX_FILTER_ACTIVE(tBool bSetBit) { u16Value = (tU16)((bSetBit)?(u16Value | FI_C_U16_BIT_BXX_FILTER_ACTIVE):(u16Value & ~(tU16)FI_C_U16_BIT_BXX_FILTER_ACTIVE)); }
   struct rBitValues {
      tBool bSID_FILTER_ACTIVE:1;
      tBool bMGS_FILTER_ACTIVE:1;
      tBool bAFI_FILTER_ACTIVE:1;
      tBool bCC_FILTER_ACTIVE:1;
      tBool bLTN_FILTER_ACTIVE:1;
      tBool bST_FILTER_ACTIVE:1;
      tBool bSPN_FILTER_ACTIVE:1;
      tBool bXX_FILTER_ACTIVE:1;
      tBool dummy8:1;
      tBool dummy9:1;
      tBool dummy10:1;
      tBool dummy11:1;
      tBool dummy12:1;
      tBool dummy13:1;
      tBool dummy14:1;
      tBool dummy15:1;
   };
   union {
      tU16 u16Value;
      rBitValues bits;
   };
   virtual ~most_fi_tcl_b16_DABTunMask();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_b16_DABTunMask& roRef) const
      {  return (u16Value == roRef.u16Value);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DABTunFilterDataItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DABTunFilterDataItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DABTunFilterDataItem& operator=(const most_fi_tcl_DABTunFilterDataItem& coRef);
   most_fi_tcl_DABTunFilterDataItem(const most_fi_tcl_DABTunFilterDataItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_b16_DABTunMask bMask;
   tU8 u8QualityFilter;
   tU16 u16PI;
   tU8 u8AFI;
   tU8 u8CC;
   tU8 u8LTN;
   tU8 u8MGS;
   tU8 u8SID;
   most_fi_tcl_String sSPN;
   tU8 u8ST;
   virtual ~most_fi_tcl_DABTunFilterDataItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DABTunFilterDataItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DABTunFilterData : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DABTunFilterData() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DABTunFilterData& operator=(const most_fi_tcl_DABTunFilterData& coRef);
   most_fi_tcl_DABTunFilterData(const most_fi_tcl_DABTunFilterData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_DABTunFilterDataItem*, std::allocator<most_fi_tcl_DABTunFilterDataItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_DABTunFilterDataItem, std::allocator<most_fi_tcl_DABTunFilterDataItem> >oItems;

   most_fi_tcl_DABTunFilterDataItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_DABTunFilterData();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DABTunFilterData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DABTunGroupID : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DABTunGroupID();
   enum tenType {
      FI_EN_E8ID_TMC_MESSAGE_GROUP = 0UL,
      FI_EN_E8ID_TIME_DATA_GROUP = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DABTunGroupID(most_fi_tcl_e8_DABTunGroupID::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DABTunGroupID();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DABTunGroupID& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DABTunTunerSelection : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DABTunTunerSelection();
   enum tenType {
      FI_EN_E8FOREGROUND_FM_TUNER = 1UL,
      FI_EN_E8BACKGROUND_FM_TUNER = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_DABTunTunerSelection(most_fi_tcl_e8_DABTunTunerSelection::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DABTunTunerSelection();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DABTunTunerSelection& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DABTunTMCMessage_ : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DABTunTMCMessage_() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DABTunTMCMessage_& operator=(const most_fi_tcl_DABTunTMCMessage_& coRef);
   most_fi_tcl_DABTunTMCMessage_(const most_fi_tcl_DABTunTMCMessage_& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU8, std::allocator<tU8> >u8Items;

   tU8& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_DABTunTMCMessage_();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DABTunTMCMessage_& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DABTunTimeData : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DABTunTimeData();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DABTunTimeData& operator=(const most_fi_tcl_DABTunTimeData& coRef);
   most_fi_tcl_DABTunTimeData(const most_fi_tcl_DABTunTimeData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8Hour;
   tS8 s8LocalTimeOffset;
   tU8 u8Minutes;
   virtual ~most_fi_tcl_DABTunTimeData();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DABTunTimeData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DABTunDAB_RDSGroupMessage : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DABTunDAB_RDSGroupMessage():u32DynamicSize(0), _ActiveMemberOffset(0u), pau8Data(OSAL_NULL), _u8TMCDataByte(), _oTimeData() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DABTunDAB_RDSGroupMessage& operator=(const most_fi_tcl_DABTunDAB_RDSGroupMessage& coRef);
   most_fi_tcl_DABTunDAB_RDSGroupMessage(const most_fi_tcl_DABTunDAB_RDSGroupMessage& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   tU32 u32DynamicSize;
   size_t _ActiveMemberOffset;
   tU8* pau8Data;

   most_fi_tcl_DABTunTMCMessage_ _u8TMCDataByte;
   most_fi_tcl_DABTunTimeData _oTimeData;

   // helper functions
   inline size_t Offset(const most_fi_tclVisitableTypeBase& roMember) const
      { return (size_t)(&roMember) - (size_t)this; }
   const most_fi_tclVisitableTypeBase* pcActiveMember() const;
   most_fi_tclVisitableTypeBase& roVariant(most_fi_tclVisitableTypeBase& roSelectedVariant);

public:
   // variant access functions
   most_fi_tcl_DABTunTMCMessage_& u8TMCDataByte();
   most_fi_tcl_DABTunTimeData& oTimeData();
   virtual ~most_fi_tcl_DABTunDAB_RDSGroupMessage();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DABTunDAB_RDSGroupMessage& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DABTunBackgroundTMCStatusInfo : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DABTunBackgroundTMCStatusInfo();
   enum tenType {
      FI_EN_E8TMC_NOT_AVAILABLE = 0UL,
      FI_EN_E8TMC_FILTERED_NOT_AVAILABLE = 1UL,
      FI_EN_E8TMC_TUNED = 2UL,
      FI_EN_E8TMC_RECEPTION_LOSS = 3UL,
      FI_EN_E8TMC_NO_STATIONS_FOUND = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_DABTunBackgroundTMCStatusInfo(most_fi_tcl_e8_DABTunBackgroundTMCStatusInfo::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DABTunBackgroundTMCStatusInfo();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DABTunBackgroundTMCStatusInfo& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DABTunDABTMCStatusInfo : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DABTunDABTMCStatusInfo();
   enum tenType {
      FI_EN_E8TMC_NOT_AVAILABLE = 0UL,
      FI_EN_E8TMC_FILTERED_NOT_AVAILABLE = 1UL,
      FI_EN_E8TMC_TUNED = 2UL,
      FI_EN_E8TMC_RECEPTION_LOSS = 3UL,
      FI_EN_E8TMC_NO_STATIONS_FOUND = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_DABTunDABTMCStatusInfo(most_fi_tcl_e8_DABTunDABTMCStatusInfo::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DABTunDABTMCStatusInfo();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DABTunDABTMCStatusInfo& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DABTunForegroundTMCStatusInfo : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DABTunForegroundTMCStatusInfo();
   enum tenType {
      FI_EN_E8TMC_NOT_AVAILABLE = 0UL,
      FI_EN_E8TMC_FILTERED_NOT_AVAILABLE = 1UL,
      FI_EN_E8TMC_TUNED = 2UL,
      FI_EN_E8TMC_RECEPTION_LOSS = 3UL,
      FI_EN_E8TMC_NO_STATIONS_FOUND = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_DABTunForegroundTMCStatusInfo(most_fi_tcl_e8_DABTunForegroundTMCStatusInfo::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DABTunForegroundTMCStatusInfo();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DABTunForegroundTMCStatusInfo& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DABTunTMCSenderNameInfo : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DABTunTMCSenderNameInfo();
   enum tenType {
      FI_EN_E8TMC_STATION_NO_RDS_NAME = 0UL,
      FI_EN_E8TMC_STATION_RDS_NAME = 1UL,
      FI_EN_E8TMC_STATION_PS_NAME = 2UL,
      FI_EN_E8TMC_STATION_PS_COMMON_NAME = 3UL,
      FI_EN_E8TMC_STATION_CUST_DEFINED = 4UL,
      FI_EN_E8TMC_STATION_NEW_NAME = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_DABTunTMCSenderNameInfo(most_fi_tcl_e8_DABTunTMCSenderNameInfo::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DABTunTMCSenderNameInfo();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DABTunTMCSenderNameInfo& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DABTunTMCStationDataItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DABTunTMCStationDataItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DABTunTMCStationDataItem& operator=(const most_fi_tcl_DABTunTMCStationDataItem& coRef);
   most_fi_tcl_DABTunTMCStationDataItem(const most_fi_tcl_DABTunTMCStationDataItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 u16PI;
   tU8 u8AFI;
   tU8 u8CC;
   most_fi_tcl_String sTMCSenderName;
   most_fi_tcl_e8_DABTunTMCSenderNameInfo e8TMCSenderNameInfo;
   tU8 u8LTN;
   tU8 u8MGS;
   tU8 u8SID;
   most_fi_tcl_String sSPN;
   tU8 u8ST;
   tU8 u8QualityFilter;
   most_fi_tcl_e8_DABTunTunerSelection e8TunerSelection;
   virtual ~most_fi_tcl_DABTunTMCStationDataItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DABTunTMCStationDataItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DABTunTMCStationData : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DABTunTMCStationData() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DABTunTMCStationData& operator=(const most_fi_tcl_DABTunTMCStationData& coRef);
   most_fi_tcl_DABTunTMCStationData(const most_fi_tcl_DABTunTMCStationData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_DABTunTMCStationDataItem*, std::allocator<most_fi_tcl_DABTunTMCStationDataItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_DABTunTMCStationDataItem, std::allocator<most_fi_tcl_DABTunTMCStationDataItem> >oItems;

   most_fi_tcl_DABTunTMCStationDataItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_DABTunTMCStationData();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DABTunTMCStationData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DevCltInfoListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DevCltInfoListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DevCltInfoListItem& operator=(const most_fi_tcl_DevCltInfoListItem& coRef);
   most_fi_tcl_DevCltInfoListItem(const most_fi_tcl_DevCltInfoListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sLeftColumn;
   most_fi_tcl_String sRightColumn;
   virtual ~most_fi_tcl_DevCltInfoListItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DevCltInfoListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DevCltInfoList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DevCltInfoList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DevCltInfoList& operator=(const most_fi_tcl_DevCltInfoList& coRef);
   most_fi_tcl_DevCltInfoList(const most_fi_tcl_DevCltInfoList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_DevCltInfoListItem*, std::allocator<most_fi_tcl_DevCltInfoListItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_DevCltInfoListItem, std::allocator<most_fi_tcl_DevCltInfoListItem> >oItems;

   most_fi_tcl_DevCltInfoListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_DevCltInfoList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DevCltInfoList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DevPrjDeviceType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DevPrjDeviceType();
   enum tenType {
      FI_EN_E8DEV_TYPE_UNKNOWN = 0UL,
      FI_EN_E8DEV_TYPE_DIGITAL_IPOD_OUT = 1UL,
      FI_EN_E8DEV_TYPE_MIRRORLINK = 2UL,
      FI_EN_E8DEV_TYPE_GOOGLE_AUTO_LINK = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_DevPrjDeviceType(most_fi_tcl_e8_DevPrjDeviceType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DevPrjDeviceType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DevPrjDeviceType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DevPrjDeviceInterfaceType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DevPrjDeviceInterfaceType();
   enum tenType {
      FI_EN_E8DEV_IFACE_WIRED_USB = 0UL,
      FI_EN_E8DEV_IFACE_WIFI = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DevPrjDeviceInterfaceType(most_fi_tcl_e8_DevPrjDeviceInterfaceType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DevPrjDeviceInterfaceType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DevPrjDeviceInterfaceType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DevPrjDeviceListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DevPrjDeviceListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DevPrjDeviceListItem& operator=(const most_fi_tcl_DevPrjDeviceListItem& coRef);
   most_fi_tcl_DevPrjDeviceListItem(const most_fi_tcl_DevPrjDeviceListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32DeviceHandle;
   most_fi_tcl_e8_DevPrjDeviceType e8DeviceType;
   most_fi_tcl_e8_DevPrjDeviceInterfaceType e8DeviceInterfaceType;
   most_fi_tcl_String sDeviceName;
   most_fi_tcl_String sDeviceVersion;
   tBool bActiveDevice;
   tBool bDeviceProjectionEnabled;
   virtual ~most_fi_tcl_DevPrjDeviceListItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DevPrjDeviceListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DevPrjDeviceList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DevPrjDeviceList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DevPrjDeviceList& operator=(const most_fi_tcl_DevPrjDeviceList& coRef);
   most_fi_tcl_DevPrjDeviceList(const most_fi_tcl_DevPrjDeviceList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_DevPrjDeviceListItem*, std::allocator<most_fi_tcl_DevPrjDeviceListItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_DevPrjDeviceListItem, std::allocator<most_fi_tcl_DevPrjDeviceListItem> >oItems;

   most_fi_tcl_DevPrjDeviceListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_DevPrjDeviceList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DevPrjDeviceList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DevPrjRequestedDisplayContext : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DevPrjRequestedDisplayContext();
   enum tenType {
      FI_EN_E8DISPLAY_CONTEXT_UNKNOWN = 0UL,
      FI_EN_E8DISPLAY_CONTEXT_SPEECH_REC = 1UL,
      FI_EN_E8DISPLAY_CONTEXT_PHONE = 2UL,
      FI_EN_E8DISPLAY_CONTEXT_NAVIGATION = 3UL,
      FI_EN_E8DISPLAY_CONTEXT_MEDIA = 4UL,
      FI_EN_E8DISPLAY_CONTEXT_INFORMATIONAL = 5UL,
      FI_EN_E8DISPLAY_CONTEXT_EMERGENCY = 6UL,
      FI_EN_E8DISPLAY_CONTEXT_SPEAKING = 7UL,
      FI_EN_E8DISPLAY_CONTEXT_TBT_NAVIGATION = 8UL
   };
   tenType enType;

   most_fi_tcl_e8_DevPrjRequestedDisplayContext(most_fi_tcl_e8_DevPrjRequestedDisplayContext::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DevPrjRequestedDisplayContext();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DevPrjRequestedDisplayContext& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DevPrjDigitalIPodOutApplication : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DevPrjDigitalIPodOutApplication();
   enum tenType {
      FI_EN_E8NOT_USED = 0UL,
      FI_EN_E8NO_URL = 1UL,
      FI_EN_E8MAPS = 2UL,
      FI_EN_E8MOBILEPHONE = 3UL,
      FI_EN_E8TEL_NUMBER = 4UL,
      FI_EN_E8SIRI_PREWARN = 5UL,
      FI_EN_E8SIRI_BUTTONDOWN = 6UL,
      FI_EN_E8SIRI_BUTTONUP = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_DevPrjDigitalIPodOutApplication(most_fi_tcl_e8_DevPrjDigitalIPodOutApplication::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DevPrjDigitalIPodOutApplication();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DevPrjDigitalIPodOutApplication& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DevPrjEchoCancellationNoiseReductionSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DevPrjEchoCancellationNoiseReductionSetting();
   enum tenType {
      FI_EN_E8ECNR_NOCHANGE = 0UL,
      FI_EN_E8ECNR_VOICE = 1UL,
      FI_EN_E8ECNR_SERVER = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_DevPrjEchoCancellationNoiseReductionSetting(most_fi_tcl_e8_DevPrjEchoCancellationNoiseReductionSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DevPrjEchoCancellationNoiseReductionSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DevPrjEchoCancellationNoiseReductionSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DevPrjLaunchResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DevPrjLaunchResult();
   enum tenType {
      FI_EN_E8LAUNCH_SUCCESSFUL = 0UL,
      FI_EN_E8LAUNCH_FAILED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DevPrjLaunchResult(most_fi_tcl_e8_DevPrjLaunchResult::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DevPrjLaunchResult();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DevPrjLaunchResult& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DevPrjApplicationCertificateType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DevPrjApplicationCertificateType();
   enum tenType {
      FI_EN_E8NO_RESTRICTION = 0UL,
      FI_EN_E8PARK_ONLY = 1UL,
      FI_EN_E8BELOW_SPEED = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_DevPrjApplicationCertificateType(most_fi_tcl_e8_DevPrjApplicationCertificateType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DevPrjApplicationCertificateType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DevPrjApplicationCertificateType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DevPrjAppActivityState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DevPrjAppActivityState();
   enum tenType {
      FI_EN_E8DP_APP_STATE_FOREGROUND = 0UL,
      FI_EN_E8DP_APP_STATE_BACKGROUND = 1UL,
      FI_EN_E8DP_APP_STATE_NOTRUNNING = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_DevPrjAppActivityState(most_fi_tcl_e8_DevPrjAppActivityState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DevPrjAppActivityState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DevPrjAppActivityState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DevPrjApplicationListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DevPrjApplicationListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DevPrjApplicationListItem& operator=(const most_fi_tcl_DevPrjApplicationListItem& coRef);
   most_fi_tcl_DevPrjApplicationListItem(const most_fi_tcl_DevPrjApplicationListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32ApplicationHandle;
   most_fi_tcl_String sApplicationName;
   most_fi_tcl_String sApplicationIconURL;
   most_fi_tcl_e8_DevPrjApplicationCertificateType e8ApplicationCertificateType;
   most_fi_tcl_e8_DevPrjAppActivityState e8AppActivityState;
   virtual ~most_fi_tcl_DevPrjApplicationListItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DevPrjApplicationListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DevPrjApplicationList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DevPrjApplicationList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DevPrjApplicationList& operator=(const most_fi_tcl_DevPrjApplicationList& coRef);
   most_fi_tcl_DevPrjApplicationList(const most_fi_tcl_DevPrjApplicationList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_DevPrjApplicationListItem*, std::allocator<most_fi_tcl_DevPrjApplicationListItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_DevPrjApplicationListItem, std::allocator<most_fi_tcl_DevPrjApplicationListItem> >oItems;

   most_fi_tcl_DevPrjApplicationListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_DevPrjApplicationList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DevPrjApplicationList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DevPrjSwitchEnumeration : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DevPrjSwitchEnumeration();
   enum tenType {
      FI_EN_E8DS_ALL_RELEASE = 0UL,
      FI_EN_E8DS_FAVORITES_UP_SWITCH = 1UL,
      FI_EN_E8DS_FAVORITES_DOWN_SWITCH = 2UL,
      FI_EN_E8DS_POWER_SWITCH = 3UL,
      FI_EN_E8DS_AUDIO_SWITCH = 6UL,
      FI_EN_E8DS_PHONE_SWITCH = 7UL,
      FI_EN_E8DS_MENU_SWITCH = 8UL,
      FI_EN_E8DS_BACK_SWITCH = 9UL,
      FI_EN_E8DS_HOME_SWITCH = 10UL,
      FI_EN_E8DS_BAND_SWITCH = 11UL,
      FI_EN_E8DS_TAG1_SWITCH = 12UL,
      FI_EN_E8DS_TAG2_SWITCH = 13UL,
      FI_EN_E8DS_TAG3_SWITCH = 14UL,
      FI_EN_E8DS_TAG4_SWITCH = 15UL,
      FI_EN_E8DS_TAG5_SWITCH = 16UL,
      FI_EN_E8DS_NEXT_SWITCH = 17UL,
      FI_EN_E8DS_PREV_SWITCH = 18UL,
      FI_EN_E8DS_DISC_AUX_SWITCH = 34UL,
      FI_EN_E8DS_TONE_SWITCH = 35UL,
      FI_EN_E8DS_SWC_UP_SWITCH = 36UL,
      FI_EN_E8DS_SWC_DOWN_SWITCH = 37UL,
      FI_EN_E8DS_SWC_ENTER_SWITCH = 38UL,
      FI_EN_E8DS_SWC_LEFT_SWITCH = 39UL,
      FI_EN_E8DS_SWC_RIGHT_SWITCH = 40UL,
      FI_EN_E8DS_SWC_PTT_SWITCH = 41UL,
      FI_EN_E8DS_SWC_END_SWITCH = 42UL
   };
   tenType enType;

   most_fi_tcl_e8_DevPrjSwitchEnumeration(most_fi_tcl_e8_DevPrjSwitchEnumeration::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DevPrjSwitchEnumeration();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DevPrjSwitchEnumeration& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DevPrjSwitchEventTypeEnumeration : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DevPrjSwitchEventTypeEnumeration();
   enum tenType {
      FI_EN_E8ICS_BUTTON_STATUS_RELEASED = 0UL,
      FI_EN_E8ICS_BUTTON_STATUS_PRESSED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DevPrjSwitchEventTypeEnumeration(most_fi_tcl_e8_DevPrjSwitchEventTypeEnumeration::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DevPrjSwitchEventTypeEnumeration();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DevPrjSwitchEventTypeEnumeration& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DevPrjButtonAction : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DevPrjButtonAction();
   enum tenType {
      FI_EN_E8DEVICE_PROJECTION_APPLICATION_ACTION_1 = 18UL,
      FI_EN_E8DEVICE_PROJECTION_APPLICATION_ACTION_2 = 19UL,
      FI_EN_E8DEVICE_PROJECTION_APPLICATION_ACTION_3 = 20UL
   };
   tenType enType;

   most_fi_tcl_e8_DevPrjButtonAction(most_fi_tcl_e8_DevPrjButtonAction::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DevPrjButtonAction();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DevPrjButtonAction& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DevPrjSessionState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DevPrjSessionState();
   enum tenType {
      FI_EN_E8SESSION_UNKNOWN = 0UL,
      FI_EN_E8SESSION_INACTIVE = 1UL,
      FI_EN_E8SESSION_ACTIVE = 2UL,
      FI_EN_E8SESSION_SUSPENDED = 3UL,
      FI_EN_E8SESSION_SUSPENDED_NON_ML = 4UL,
      FI_EN_E8SESSION_SUSPENDED_NON_DRIVE = 5UL,
      FI_EN_E8SESSION_INIT_IN_PROGRESS = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_DevPrjSessionState(most_fi_tcl_e8_DevPrjSessionState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DevPrjSessionState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DevPrjSessionState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DevPrjConnectivityMappingItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DevPrjConnectivityMappingItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DevPrjConnectivityMappingItem& operator=(const most_fi_tcl_DevPrjConnectivityMappingItem& coRef);
   most_fi_tcl_DevPrjConnectivityMappingItem(const most_fi_tcl_DevPrjConnectivityMappingItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32DeviceHandle;
   tU8 u8BTDeviceHandle;
   most_fi_tcl_String sBluetoothDeviceAddress;
   tU8 u8USBDeviceTag;
   most_fi_tcl_String sUSBSerialNumber;
   most_fi_tcl_String sWiFiDirectMAC;
   virtual ~most_fi_tcl_DevPrjConnectivityMappingItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DevPrjConnectivityMappingItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DevPrjConnectivityMapping : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DevPrjConnectivityMapping() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DevPrjConnectivityMapping& operator=(const most_fi_tcl_DevPrjConnectivityMapping& coRef);
   most_fi_tcl_DevPrjConnectivityMapping(const most_fi_tcl_DevPrjConnectivityMapping& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_DevPrjConnectivityMappingItem*, std::allocator<most_fi_tcl_DevPrjConnectivityMappingItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_DevPrjConnectivityMappingItem, std::allocator<most_fi_tcl_DevPrjConnectivityMappingItem> >oItems;

   most_fi_tcl_DevPrjConnectivityMappingItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_DevPrjConnectivityMapping();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DevPrjConnectivityMapping& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DevPrjDeviceUserParametersItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DevPrjDeviceUserParametersItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DevPrjDeviceUserParametersItem& operator=(const most_fi_tcl_DevPrjDeviceUserParametersItem& coRef);
   most_fi_tcl_DevPrjDeviceUserParametersItem(const most_fi_tcl_DevPrjDeviceUserParametersItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32DeviceHandle;
   tBool bUserAuthorizationStatus;
   tBool bHandsetInteractionRequired;
   virtual ~most_fi_tcl_DevPrjDeviceUserParametersItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DevPrjDeviceUserParametersItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_DevPrjDeviceUserParameters : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_DevPrjDeviceUserParameters() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_DevPrjDeviceUserParameters& operator=(const most_fi_tcl_DevPrjDeviceUserParameters& coRef);
   most_fi_tcl_DevPrjDeviceUserParameters(const most_fi_tcl_DevPrjDeviceUserParameters& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_DevPrjDeviceUserParametersItem, std::allocator<most_fi_tcl_DevPrjDeviceUserParametersItem> >oItems;

   most_fi_tcl_DevPrjDeviceUserParametersItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_DevPrjDeviceUserParameters();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_DevPrjDeviceUserParameters& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DevPrjBTChangeInfo : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DevPrjBTChangeInfo();
   enum tenType {
      FI_EN_E8NO_CHANGE = 0UL,
      FI_EN_E8SWITCH_BT_TO_ML = 1UL,
      FI_EN_E8SWITCH_BT_TO_CARPLAY = 2UL,
      FI_EN_E8SWITCH_ML_TO_BT = 3UL,
      FI_EN_E8SWITCH_CARPLAY_TO_BT = 4UL,
      FI_EN_E8SWITCH_BT_TO_ANDROID = 5UL,
      FI_EN_E8SWITCH_ANDRIOD_TO_BT = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_DevPrjBTChangeInfo(most_fi_tcl_e8_DevPrjBTChangeInfo::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DevPrjBTChangeInfo();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DevPrjBTChangeInfo& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DevPrjNavigationAppState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DevPrjNavigationAppState();
   enum tenType {
      FI_EN_E8SPI_APP_STATE_NAV_UNKNOWN = 0UL,
      FI_EN_E8SPI_APP_STATE_NAV_ACTIVE = 1UL,
      FI_EN_E8SPI_APP_STATE_NAV_NOTACTIVE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_DevPrjNavigationAppState(most_fi_tcl_e8_DevPrjNavigationAppState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DevPrjNavigationAppState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DevPrjNavigationAppState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DevPrjPhoneAppState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DevPrjPhoneAppState();
   enum tenType {
      FI_EN_E8SPI_APP_STATE_PHONE_UNKNOWN = 0UL,
      FI_EN_E8SPI_APP_STATE_PHONE_ACTIVE = 1UL,
      FI_EN_E8SPI_APP_STATE_PHONE_NOTACTIVE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_DevPrjPhoneAppState(most_fi_tcl_e8_DevPrjPhoneAppState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DevPrjPhoneAppState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DevPrjPhoneAppState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DevPrjSpeechAppState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DevPrjSpeechAppState();
   enum tenType {
      FI_EN_E8SPI_APP_STATE_SPEECH_UNKNOWN = 0UL,
      FI_EN_E8SPI_APP_STATE_SPEECH_END = 1UL,
      FI_EN_E8SPI_APP_STATE_SPEECH_SPEAKING = 2UL,
      FI_EN_E8SPI_APP_STATE_SPEECH_RECOGNISING = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_DevPrjSpeechAppState(most_fi_tcl_e8_DevPrjSpeechAppState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DevPrjSpeechAppState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DevPrjSpeechAppState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_DiPOSwitchReqResponse : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_DiPOSwitchReqResponse();
   enum tenType {
      FI_EN_E8DIPO_ROLE_SWITCH_NOT_REQUIRED = 0UL,
      FI_EN_E8DIPO_ROLE_SWITCH_REQUIRED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_DiPOSwitchReqResponse(most_fi_tcl_e8_DiPOSwitchReqResponse::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_DiPOSwitchReqResponse();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_DiPOSwitchReqResponse& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_PlaybackStopReason : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_PlaybackStopReason();
   enum tenType {
      FI_EN_E8REASON_SOURCESWITCH = 0UL,
      FI_EN_E8REASON_MEDIADEVICESWITCH = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_PlaybackStopReason(most_fi_tcl_e8_PlaybackStopReason::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_PlaybackStopReason();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_PlaybackStopReason& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_FavManAudioChannel : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_FavManAudioChannel();
   enum tenType {
      FI_EN_E8LC_NONE = 0UL,
      FI_EN_E8LC_MAIN_AUDIO = 1UL,
      FI_EN_E8LC_PHONE = 5UL,
      FI_EN_E8LC_REAR_AUDIO = 17UL,
      FI_EN_E8LC_HEADPHONE_1 = 33UL,
      FI_EN_E8LC_HEADPHONE_2 = 34UL,
      FI_EN_E8LC_HEADPHONE_3 = 35UL,
      FI_EN_E8LC_HEADPHONE_4 = 36UL
   };
   tenType enType;

   most_fi_tcl_e8_FavManAudioChannel(most_fi_tcl_e8_FavManAudioChannel::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_FavManAudioChannel();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_FavManAudioChannel& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_FavManVideoChannel : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_FavManVideoChannel();
   enum tenType {
      FI_EN_E8LC_NONE = 0UL
   };
   tenType enType;

   most_fi_tcl_e8_FavManVideoChannel(most_fi_tcl_e8_FavManVideoChannel::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_FavManVideoChannel();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_FavManVideoChannel& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_FavManFavoriteType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_FavManFavoriteType();
   enum tenType {
      FI_EN_E8FT_ALL = 0UL,
      FI_EN_E8FT_BLANK = 1UL,
      FI_EN_E8FT_AM_FREQUENCY = 2UL,
      FI_EN_E8FT_FM_FREQUENCY = 3UL,
      FI_EN_E8FT_SDARS_CHANNEL = 4UL,
      FI_EN_E8FT_PANDORA_STATION = 5UL,
      FI_EN_E8FT_INTERNET_STATION = 6UL,
      FI_EN_E8FT_PLAYLIST = 7UL,
      FI_EN_E8FT_MEDIA = 8UL,
      FI_EN_E8FT_PODCAST = 9UL,
      FI_EN_E8FT_VIDEO = 10UL,
      FI_EN_E8FT_AUDIO_BOOK = 11UL,
      FI_EN_E8FT_POI_CATEGORY = 12UL,
      FI_EN_E8FT_DESTINATION = 13UL,
      FI_EN_E8FT_FUNCTION = 14UL,
      FI_EN_E8FT_APPLICATION = 15UL,
      FI_EN_E8FT_CONTACT_NAME = 16UL,
      FI_EN_E8FT_PHONE_NUMBER = 17UL,
      FI_EN_E8FT_TONE_CONTROLS = 18UL,
      FI_EN_E8FT_SEARCH_TERM_POI = 19UL,
      FI_EN_E8FT_SEARCH_TERM_ADDRESS = 20UL,
      FI_EN_E8FT_POI_CHAIN = 21UL,
      FI_EN_E8FT_MYROUTE = 22UL,
      FI_EN_E8FT_SAVED_TRIP = 23UL,
      FI_EN_E8FT_DAB_STATION = 24UL
   };
   tenType enType;

   most_fi_tcl_e8_FavManFavoriteType(most_fi_tcl_e8_FavManFavoriteType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_FavManFavoriteType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_FavManFavoriteType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_FavManFavoriteTypesList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_FavManFavoriteTypesList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_FavManFavoriteTypesList& operator=(const most_fi_tcl_FavManFavoriteTypesList& coRef);
   most_fi_tcl_FavManFavoriteTypesList(const most_fi_tcl_FavManFavoriteTypesList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_e8_FavManFavoriteType, std::allocator<most_fi_tcl_e8_FavManFavoriteType> >e8Items;

   most_fi_tcl_e8_FavManFavoriteType& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_FavManFavoriteTypesList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_FavManFavoriteTypesList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_FavManFavoriteHMIDataItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_FavManFavoriteHMIDataItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_FavManFavoriteHMIDataItem& operator=(const most_fi_tcl_FavManFavoriteHMIDataItem& coRef);
   most_fi_tcl_FavManFavoriteHMIDataItem(const most_fi_tcl_FavManFavoriteHMIDataItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32UID;
   tBool bActive;
   tBool bAvailable;
   most_fi_tcl_String sLabelText;
   most_fi_tcl_e8_FavManFavoriteType e8FavoriteType;
   most_fi_tcl_String sDescriptorText;
   tU8 u8FBlockID;
   tU8 u8InstanceID;
   virtual ~most_fi_tcl_FavManFavoriteHMIDataItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_FavManFavoriteHMIDataItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_FavManFavoriteHMIData : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_FavManFavoriteHMIData() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_FavManFavoriteHMIData& operator=(const most_fi_tcl_FavManFavoriteHMIData& coRef);
   most_fi_tcl_FavManFavoriteHMIData(const most_fi_tcl_FavManFavoriteHMIData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_FavManFavoriteHMIDataItem*, std::allocator<most_fi_tcl_FavManFavoriteHMIDataItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_FavManFavoriteHMIDataItem, std::allocator<most_fi_tcl_FavManFavoriteHMIDataItem> >oItems;

   most_fi_tcl_FavManFavoriteHMIDataItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_FavManFavoriteHMIData();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_FavManFavoriteHMIData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_FavManListChangeType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_FavManListChangeType();
   enum tenType {
      FI_EN_E8LCH_CONTENT_CHANGED = 0UL,
      FI_EN_E8LCH_ITEMS_ADDED = 1UL,
      FI_EN_E8LCH_ITEMS_REMOVED = 2UL,
      FI_EN_E8LCH_ITEMS_CHANGED = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_FavManListChangeType(most_fi_tcl_e8_FavManListChangeType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_FavManListChangeType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_FavManListChangeType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_FavManChangeItems : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_FavManChangeItems() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_FavManChangeItems& operator=(const most_fi_tcl_FavManChangeItems& coRef);
   most_fi_tcl_FavManChangeItems(const most_fi_tcl_FavManChangeItems& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU32, std::allocator<tU32> >u32Items;

   tU32& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_FavManChangeItems();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_FavManChangeItems& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_FMTunImageData : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_FMTunImageData() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_FMTunImageData& operator=(const most_fi_tcl_FMTunImageData& coRef);
   most_fi_tcl_FMTunImageData(const most_fi_tcl_FMTunImageData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU8, std::allocator<tU8> >u8Items;

   tU8& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_FMTunImageData();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_FMTunImageData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWAPAIconDisplayRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWAPAIconDisplayRequest();
   enum tenType {
      FI_EN_E8APA_NO_ICON = 0UL,
      FI_EN_E8APA_FORWARD = 1UL,
      FI_EN_E8APA_REVERSE = 2UL,
      FI_EN_E8APA_STOP = 3UL,
      FI_EN_E8APA_COMPLETE = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWAPAIconDisplayRequest(most_fi_tcl_e8_GMLnGWAPAIconDisplayRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWAPAIconDisplayRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWAPAIconDisplayRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDrivelineCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDrivelineCustomizationSettings();
   enum tenType {
      FI_EN_E8DRIVELINE_CUSTOMIZATION_DEFAULT = 0UL,
      FI_EN_E8DRIVELINE_CUSTOMIZATION_NORMAL = 1UL,
      FI_EN_E8DRIVELINE_CUSTOMIZATION_SPORT = 2UL,
      FI_EN_E8DRIVELINE_CUSTOMIZATION_PERF_MODE_3 = 3UL,
      FI_EN_E8DRIVELINE_CUSTOMIZATION_PERF_MODE_4 = 4UL,
      FI_EN_E8DRIVELINE_CUSTOMIZATION_PERF_MODE_5 = 5UL,
      FI_EN_E8DRIVELINE_CUSTOMIZATION_PERF_MODE_6 = 6UL,
      FI_EN_E8DRIVELINE_CUSTOMIZATION_PERF_MODE_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDrivelineCustomizationSettings(most_fi_tcl_e8_GMLnGWDrivelineCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDrivelineCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDrivelineCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWSteeringCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWSteeringCustomizationSettings();
   enum tenType {
      FI_EN_E8STEERING_CUSTOMIZATION_DEFAULT = 0UL,
      FI_EN_E8STEERING_CUSTOMIZATION_NORMAL = 1UL,
      FI_EN_E8STEERING_CUSTOMIZATION_SPORT = 2UL,
      FI_EN_E8STEERING_CUSTOMIZATION_PERF_MODE_3 = 3UL,
      FI_EN_E8STEERING_CUSTOMIZATION_PERF_MODE_4 = 4UL,
      FI_EN_E8STEERING_CUSTOMIZATION_PERF_MODE_5 = 5UL,
      FI_EN_E8STEERING_CUSTOMIZATION_PERF_MODE_6 = 6UL,
      FI_EN_E8STEERING_CUSTOMIZATION_PERF_MODE_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWSteeringCustomizationSettings(most_fi_tcl_e8_GMLnGWSteeringCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWSteeringCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWSteeringCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWSuspensionCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWSuspensionCustomizationSettings();
   enum tenType {
      FI_EN_E8SUSPENSION_CUSTOMIZATION_DEFAULT = 0UL,
      FI_EN_E8SUSPENSION_CUSTOMIZATION_NORMAL = 1UL,
      FI_EN_E8SUSPENSION_CUSTOMIZATION_SPORT = 2UL,
      FI_EN_E8SUSPENSION_CUSTOMIZATION_PERF_MODE_3 = 3UL,
      FI_EN_E8SUSPENSION_CUSTOMIZATION_PERF_MODE_4 = 4UL,
      FI_EN_E8SUSPENSION_CUSTOMIZATION_PERF_MODE_5 = 5UL,
      FI_EN_E8SUSPENSION_CUSTOMIZATION_PERF_MODE_6 = 6UL,
      FI_EN_E8SUSPENSION_CUSTOMIZATION_PERF_MODE_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWSuspensionCustomizationSettings(most_fi_tcl_e8_GMLnGWSuspensionCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWSuspensionCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWSuspensionCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWForwardCollisionAlertCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWForwardCollisionAlertCustomizationSettings();
   enum tenType {
      FI_EN_E8FORWARD_COLLISION_ALERT_CUSTOMIZATION_UNKNOWN = 0UL,
      FI_EN_E8FORWARD_COLLISION_ALERT_CUSTOMIZATION_OFF = 1UL,
      FI_EN_E8FORWARD_COLLISION_ALERT_CUSTOMIZATION_ON = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWForwardCollisionAlertCustomizationSettings(most_fi_tcl_e8_GMLnGWForwardCollisionAlertCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWForwardCollisionAlertCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWForwardCollisionAlertCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWGoNotifierCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWGoNotifierCustomizationSettings();
   enum tenType {
      FI_EN_E8GO_NOTIFIER_CUSTOMIZATION_UNKNOWN = 0UL,
      FI_EN_E8GO_NOTIFIER_CUSTOMIZATION_OFF = 1UL,
      FI_EN_E8GO_NOTIFIER_CUSTOMIZATION_ON = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWGoNotifierCustomizationSettings(most_fi_tcl_e8_GMLnGWGoNotifierCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWGoNotifierCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWGoNotifierCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWCollisionPreparationSystemCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWCollisionPreparationSystemCustomizationSettings();
   enum tenType {
      FI_EN_E8COLLISION_PREP_SYS_CUSTOMIZATION_NOVALUE = 0UL,
      FI_EN_E8COLLISION_PREP_SYS_CUSTOMIZATION_OFF = 1UL,
      FI_EN_E8COLLISION_PREP_SYS_CUSTOMIZATION_ALERT = 2UL,
      FI_EN_E8COLLISION_PREP_SYS_CUSTOMIZATION_ALERT_AND_BRAKE = 3UL,
      FI_EN_E8COLLISION_PREP_SYS_CUSTOMIZATION_ALERT_AND_BRAKE_STEER = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWCollisionPreparationSystemCustomizationSettings(most_fi_tcl_e8_GMLnGWCollisionPreparationSystemCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWCollisionPreparationSystemCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWCollisionPreparationSystemCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftRightHandTrafficCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftRightHandTrafficCustomizationSettings();
   enum tenType {
      FI_EN_E8LEFT_RIGHT_HAND_TRAFFIC_CUSTOMIZATION_NOVALUE = 0UL,
      FI_EN_E8LEFT_RIGHT_HAND_TRAFFIC_CUSTOMIZATION_LEFTHAND_DRIVE = 1UL,
      FI_EN_E8LEFT_RIGHT_HAND_TRAFFIC_CUSTOMIZATION_RIGHTHAND_DRIVE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftRightHandTrafficCustomizationSettings(most_fi_tcl_e8_GMLnGWLeftRightHandTrafficCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftRightHandTrafficCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftRightHandTrafficCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWSeatBeltTighteningCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWSeatBeltTighteningCustomizationSettings();
   enum tenType {
      FI_EN_E8SEAT_BELT_TIGHTENING_CUSTOMIZATION_UNKNOWN = 0UL,
      FI_EN_E8SEAT_BELT_TIGHTENING_CUSTOMIZATION_LOW = 1UL,
      FI_EN_E8SEAT_BELT_TIGHTENING_CUSTOMIZATION_NORMAL = 2UL,
      FI_EN_E8SEAT_BELT_TIGHTENING_CUSTOMIZATION_PERFORMANCE = 3UL,
      FI_EN_E8SEAT_BELT_TIGHTENING_CUSTOMIZATION_LOCKED = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWSeatBeltTighteningCustomizationSettings(most_fi_tcl_e8_GMLnGWSeatBeltTighteningCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWSeatBeltTighteningCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWSeatBeltTighteningCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingWithGPSCustomizationSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingWithGPSCustomizationSetting();
   enum tenType {
      FI_EN_E8ADAPTIVE_FWD_LIGHTING_WITH_GPS_UNKNOWN = 0UL,
      FI_EN_E8ADAPTIVE_FWD_LIGHTING_WITH_GPS_CORNER_AND_CURVE_LIGHTING_ONLY = 1UL,
      FI_EN_E8ADAPTIVE_FWD_LIGHTING_WITH_GPS_INTELLIGENT_LIGHT_DISTRIBUTION = 2UL,
      FI_EN_E8ADAPTIVE_FWD_LIGHTING_WITH_GPS_ASSISTANCE = 3UL,
      FI_EN_E8ADAPTIVE_FWD_LIGHTING_WITH_GPS_OFF = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingWithGPSCustomizationSetting(most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingWithGPSCustomizationSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingWithGPSCustomizationSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingWithGPSCustomizationSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingCustomizationSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingCustomizationSetting();
   enum tenType {
      FI_EN_E8ADAPTIVE_FWD_LIGHTING_UNKNOWN = 0UL,
      FI_EN_E8ADAPTIVE_FWD_LIGHTING_CORNER_AND_CURVE_LIGHTING_ONLY = 1UL,
      FI_EN_E8ADAPTIVE_FWD_LIGHTING_INTELLIGENT_LIGHT_DISTRIBUTION = 2UL,
      FI_EN_E8ADAPTIVE_FWD_LIGHTING_OFF = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingCustomizationSetting(most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingCustomizationSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingCustomizationSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingCustomizationSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftRightHandTrafficWithGPSCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftRightHandTrafficWithGPSCustomizationSettings();
   enum tenType {
      FI_EN_E8LEFT_RIGHT_HAND_TRAFFIC_CUSTOMIZATION_NOVALUE = 0UL,
      FI_EN_E8LEFT_RIGHT_HAND_TRAFFIC_CUSTOMIZATION_LEFTHAND_DRIVE = 1UL,
      FI_EN_E8LEFT_RIGHT_HAND_TRAFFIC_CUSTOMIZATION_RIGHTHAND_DRIVE = 2UL,
      FI_EN_E8LEFT_RIGHT_HAND_TRAFFIC_CUSTOMIZATION_AUTOMATIC = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftRightHandTrafficWithGPSCustomizationSettings(most_fi_tcl_e8_GMLnGWLeftRightHandTrafficWithGPSCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftRightHandTrafficWithGPSCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftRightHandTrafficWithGPSCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWAutoHighBeamAssistCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWAutoHighBeamAssistCustomizationSettings();
   enum tenType {
      FI_EN_E8AUTO_HIGH_BEAM_ASSIST_CUSTOMIZATION_NOVALUE = 0UL,
      FI_EN_E8AUTO_HIGH_BEAM_ASSIST_CUSTOMIZATION_OFF = 1UL,
      FI_EN_E8AUTO_HIGH_BEAM_ASSIST_CUSTOMIZATION_ON = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWAutoHighBeamAssistCustomizationSettings(most_fi_tcl_e8_GMLnGWAutoHighBeamAssistCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWAutoHighBeamAssistCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWAutoHighBeamAssistCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWAutoHighBeamAssistWithSensitivityCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWAutoHighBeamAssistWithSensitivityCustomizationSettings();
   enum tenType {
      FI_EN_E8AUTO_HIGH_BEAM_ASSIST_CUSTOMIZATION_NOVALUE = 0UL,
      FI_EN_E8AUTO_HIGH_BEAM_ASSIST_CUSTOMIZATION_OFF = 1UL,
      FI_EN_E8AUTO_HIGH_BEAM_ASSIST_CUSTOMIZATION_ON_NORMAL_SENSITIVITY = 2UL,
      FI_EN_E8AUTO_HIGH_BEAM_ASSIST_CUSTOMIZATION_ON_REDUCED_SENSITIVITY = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWAutoHighBeamAssistWithSensitivityCustomizationSettings(most_fi_tcl_e8_GMLnGWAutoHighBeamAssistWithSensitivityCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWAutoHighBeamAssistWithSensitivityCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWAutoHighBeamAssistWithSensitivityCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWAdaptiveHighBeamAssistCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWAdaptiveHighBeamAssistCustomizationSettings();
   enum tenType {
      FI_EN_E8ADAPTIVE_HIGH_BEAM_ASSIST_CUSTOMIZATION_NOVALUE = 0UL,
      FI_EN_E8ADAPTIVE_HIGH_BEAM_ASSIST_CUSTOMIZATION_OFF = 1UL,
      FI_EN_E8ADAPTIVE_HIGH_BEAM_ASSIST_CUSTOMIZATION_ON = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWAdaptiveHighBeamAssistCustomizationSettings(most_fi_tcl_e8_GMLnGWAdaptiveHighBeamAssistCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWAdaptiveHighBeamAssistCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWAdaptiveHighBeamAssistCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWAdaptiveHighBeamAssistWithSensitivityCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWAdaptiveHighBeamAssistWithSensitivityCustomizationSettings();
   enum tenType {
      FI_EN_E8ADAPTIVE_HIGH_BEAM_ASSIST_WITH_SENSITIVITY_CUSTOMIZATION_NOVALUE = 0UL,
      FI_EN_E8ADAPTIVE_HIGH_BEAM_ASSIST_WITH_SENSITIVITY_CUSTOMIZATION_OFF = 1UL,
      FI_EN_E8ADAPTIVE_HIGH_BEAM_ASSIST_WITH_SENSITIVITY_CUSTOMIZATION_ON_NORMAL_SENSITIVITY = 2UL,
      FI_EN_E8ADAPTIVE_HIGH_BEAM_ASSIST_WITH_SENSITIVITY_CUSTOMIZATION_ON_REDUCED_SENSITIVITY = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWAdaptiveHighBeamAssistWithSensitivityCustomizationSettings(most_fi_tcl_e8_GMLnGWAdaptiveHighBeamAssistWithSensitivityCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWAdaptiveHighBeamAssistWithSensitivityCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWAdaptiveHighBeamAssistWithSensitivityCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWChargePowerLossAlertCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWChargePowerLossAlertCustomizationSettings();
   enum tenType {
      FI_EN_E8CHARGE_POWER_LOSS_ALERT_CUSTOMIZATION_NOVALUE = 0UL,
      FI_EN_E8CHARGE_POWER_LOSS_ALERT_CUSTOMIZATION_OFF = 1UL,
      FI_EN_E8CHARGE_POWER_LOSS_ALERT_CUSTOMIZATION_ON = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWChargePowerLossAlertCustomizationSettings(most_fi_tcl_e8_GMLnGWChargePowerLossAlertCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWChargePowerLossAlertCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWChargePowerLossAlertCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWChargeCordTheftAlertCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWChargeCordTheftAlertCustomizationSettings();
   enum tenType {
      FI_EN_E8CHARGE_CORD_THEFT_ALERT_CUSTOMIZATION_NOVALUE = 0UL,
      FI_EN_E8CHARGE_CORD_THEFT_ALERT_CUSTOMIZATION_OFF = 1UL,
      FI_EN_E8CHARGE_CORD_THEFT_ALERT_CUSTOMIZATION_ON = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWChargeCordTheftAlertCustomizationSettings(most_fi_tcl_e8_GMLnGWChargeCordTheftAlertCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWChargeCordTheftAlertCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWChargeCordTheftAlertCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWChargeSystemAudibleIndicationCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWChargeSystemAudibleIndicationCustomizationSettings();
   enum tenType {
      FI_EN_E8CHARGE_SYSTEM_AUDIBLE_INDICATIONS_CUSTOMIZATION_NOVALUE = 0UL,
      FI_EN_E8CHARGE_SYSTEM_AUDIBLE_INDICATIONS_CUSTOMIZATION_OFF = 1UL,
      FI_EN_E8CHARGE_SYSTEM_AUDIBLE_INDICATIONS_CUSTOMIZATION_ON = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWChargeSystemAudibleIndicationCustomizationSettings(most_fi_tcl_e8_GMLnGWChargeSystemAudibleIndicationCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWChargeSystemAudibleIndicationCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWChargeSystemAudibleIndicationCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverDrowsinessDetectionCustomizationSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverDrowsinessDetectionCustomizationSetting();
   enum tenType {
      FI_EN_E8DRIVER_DROWSINESS_DETECTION_CUSTOMIZATION_NOVALUE = 0UL,
      FI_EN_E8DRIVER_DROWSINESS_DETECTION_CUSTOMIZATION_OFF = 1UL,
      FI_EN_E8DRIVER_DROWSINESS_DETECTION_CUSTOMIZATION_ON = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverDrowsinessDetectionCustomizationSetting(most_fi_tcl_e8_GMLnGWDriverDrowsinessDetectionCustomizationSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverDrowsinessDetectionCustomizationSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverDrowsinessDetectionCustomizationSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPTOSetOneSpeedCustomizationSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPTOSetOneSpeedCustomizationSetting();
   enum tenType {
      FI_EN_E8PTO_SET_ONE_SPEED_UNKNOWN_NO_ACTION = 0UL,
      FI_EN_E8PTO_SET_ONE_SPEED_1 = 1UL,
      FI_EN_E8PTO_SET_ONE_SPEED_2 = 2UL,
      FI_EN_E8PTO_SET_ONE_SPEED_3 = 3UL,
      FI_EN_E8PTO_SET_ONE_SPEED_4 = 4UL,
      FI_EN_E8PTO_SET_ONE_SPEED_5 = 5UL,
      FI_EN_E8PTO_SET_ONE_SPEED_6 = 6UL,
      FI_EN_E8PTO_SET_ONE_SPEED_7 = 7UL,
      FI_EN_E8PTO_SET_ONE_SPEED_8 = 8UL,
      FI_EN_E8PTO_SET_ONE_SPEED_9 = 9UL,
      FI_EN_E8PTO_SET_ONE_SPEED_10 = 10UL,
      FI_EN_E8PTO_SET_ONE_SPEED_11 = 11UL,
      FI_EN_E8PTO_SET_ONE_SPEED_12 = 12UL,
      FI_EN_E8PTO_SET_ONE_SPEED_13 = 13UL,
      FI_EN_E8PTO_SET_ONE_SPEED_14 = 14UL,
      FI_EN_E8PTO_SET_ONE_SPEED_15 = 15UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPTOSetOneSpeedCustomizationSetting(most_fi_tcl_e8_GMLnGWPTOSetOneSpeedCustomizationSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPTOSetOneSpeedCustomizationSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPTOSetOneSpeedCustomizationSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPTOSetTwoSpeedCustomizationSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPTOSetTwoSpeedCustomizationSetting();
   enum tenType {
      FI_EN_E8PTO_SET_TWO_SPEED_UNKNOWN_NO_ACTION = 0UL,
      FI_EN_E8PTO_SET_TWO_SPEED_1 = 1UL,
      FI_EN_E8PTO_SET_TWO_SPEED_2 = 2UL,
      FI_EN_E8PTO_SET_TWO_SPEED_3 = 3UL,
      FI_EN_E8PTO_SET_TWO_SPEED_4 = 4UL,
      FI_EN_E8PTO_SET_TWO_SPEED_5 = 5UL,
      FI_EN_E8PTO_SET_TWO_SPEED_6 = 6UL,
      FI_EN_E8PTO_SET_TWO_SPEED_7 = 7UL,
      FI_EN_E8PTO_SET_TWO_SPEED_8 = 8UL,
      FI_EN_E8PTO_SET_TWO_SPEED_9 = 9UL,
      FI_EN_E8PTO_SET_TWO_SPEED_10 = 10UL,
      FI_EN_E8PTO_SET_TWO_SPEED_11 = 11UL,
      FI_EN_E8PTO_SET_TWO_SPEED_12 = 12UL,
      FI_EN_E8PTO_SET_TWO_SPEED_13 = 13UL,
      FI_EN_E8PTO_SET_TWO_SPEED_14 = 14UL,
      FI_EN_E8PTO_SET_TWO_SPEED_15 = 15UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPTOSetTwoSpeedCustomizationSetting(most_fi_tcl_e8_GMLnGWPTOSetTwoSpeedCustomizationSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPTOSetTwoSpeedCustomizationSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPTOSetTwoSpeedCustomizationSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPTOEngineRunTimerCustomizationSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPTOEngineRunTimerCustomizationSetting();
   enum tenType {
      FI_EN_E8PTO_ENGINE_RUN_TIMER_UNKNOWN_NO_ACTION = 0UL,
      FI_EN_E8PTO_ENGINE_RUN_TIMER_SETTING_1 = 1UL,
      FI_EN_E8PTO_ENGINE_RUN_TIMER_SETTING_2 = 2UL,
      FI_EN_E8PTO_ENGINE_RUN_TIMER_SETTING_3 = 3UL,
      FI_EN_E8PTO_ENGINE_RUN_TIMER_SETTING_4 = 4UL,
      FI_EN_E8PTO_ENGINE_RUN_TIMER_SETTING_5 = 5UL,
      FI_EN_E8PTO_ENGINE_RUN_TIMER_SETTING_6 = 6UL,
      FI_EN_E8PTO_ENGINE_RUN_TIMER_SETTING_7 = 7UL,
      FI_EN_E8PTO_ENGINE_RUN_TIMER_SETTING_8 = 8UL,
      FI_EN_E8PTO_ENGINE_RUN_TIMER_SETTING_9 = 9UL,
      FI_EN_E8PTO_ENGINE_RUN_TIMER_SETTING_10 = 10UL,
      FI_EN_E8PTO_ENGINE_RUN_TIMER_SETTING_11 = 11UL,
      FI_EN_E8PTO_ENGINE_RUN_TIMER_SETTING_12 = 12UL,
      FI_EN_E8PTO_ENGINE_RUN_TIMER_SETTING_13 = 13UL,
      FI_EN_E8PTO_ENGINE_RUN_TIMER_SETTING_14 = 14UL,
      FI_EN_E8PTO_ENGINE_RUN_TIMER_SETTING_15 = 15UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPTOEngineRunTimerCustomizationSetting(most_fi_tcl_e8_GMLnGWPTOEngineRunTimerCustomizationSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPTOEngineRunTimerCustomizationSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPTOEngineRunTimerCustomizationSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPTOTapSetSpeedCustomizationSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPTOTapSetSpeedCustomizationSetting();
   enum tenType {
      FI_EN_E8PTO_TAP_SET_SPEED_UNKNOWN_NO_ACTION = 0UL,
      FI_EN_E8PTO_TAP_SET_SPEED_1 = 1UL,
      FI_EN_E8PTO_TAP_SET_SPEED_2 = 2UL,
      FI_EN_E8PTO_TAP_SET_SPEED_3 = 3UL,
      FI_EN_E8PTO_TAP_SET_SPEED_4 = 4UL,
      FI_EN_E8PTO_TAP_SET_SPEED_5 = 5UL,
      FI_EN_E8PTO_TAP_SET_SPEED_6 = 6UL,
      FI_EN_E8PTO_TAP_SET_SPEED_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPTOTapSetSpeedCustomizationSetting(most_fi_tcl_e8_GMLnGWPTOTapSetSpeedCustomizationSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPTOTapSetSpeedCustomizationSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPTOTapSetSpeedCustomizationSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPTOStandbySpeedCustomizationSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPTOStandbySpeedCustomizationSetting();
   enum tenType {
      FI_EN_E8PTO_STANDBY_SPEED_UNKNOWN_NO_ACTION = 0UL,
      FI_EN_E8PTO_STANDBY_SPEED_1 = 1UL,
      FI_EN_E8PTO_STANDBY_SPEED_2 = 2UL,
      FI_EN_E8PTO_STANDBY_SPEED_3 = 3UL,
      FI_EN_E8PTO_STANDBY_SPEED_4 = 4UL,
      FI_EN_E8PTO_STANDBY_SPEED_5 = 5UL,
      FI_EN_E8PTO_STANDBY_SPEED_6 = 6UL,
      FI_EN_E8PTO_STANDBY_SPEED_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPTOStandbySpeedCustomizationSetting(most_fi_tcl_e8_GMLnGWPTOStandbySpeedCustomizationSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPTOStandbySpeedCustomizationSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPTOStandbySpeedCustomizationSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPerfModeDisplayCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPerfModeDisplayCustomizationSettings();
   enum tenType {
      FI_EN_E8PERF_MODE_DISPLAY_CUSTOMIZATION_DEFAULT = 0UL,
      FI_EN_E8PERF_MODE_DISPLAY_CUSTOMIZATION_PERF_MODE_1 = 1UL,
      FI_EN_E8PERF_MODE_DISPLAY_CUSTOMIZATION_PERF_MODE_2 = 2UL,
      FI_EN_E8PERF_MODE_DISPLAY_CUSTOMIZATION_PERF_MODE_3 = 3UL,
      FI_EN_E8PERF_MODE_DISPLAY_CUSTOMIZATION_PERF_MODE_4 = 4UL,
      FI_EN_E8PERF_MODE_DISPLAY_CUSTOMIZATION_PERF_MODE_5 = 5UL,
      FI_EN_E8PERF_MODE_DISPLAY_CUSTOMIZATION_PERF_MODE_6 = 6UL,
      FI_EN_E8PERF_MODE_DISPLAY_CUSTOMIZATION_AUTO = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPerfModeDisplayCustomizationSettings(most_fi_tcl_e8_GMLnGWPerfModeDisplayCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPerfModeDisplayCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPerfModeDisplayCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPerfModeSoundCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPerfModeSoundCustomizationSettings();
   enum tenType {
      FI_EN_E8PERF_MODE_ADAP_CRUISE_CONTROL_CUSTOMIZATION_DEFAULT = 0UL,
      FI_EN_E8PERF_MODE_ADAP_CRUISE_CONTROL_CUSTOMIZATION_PERF_MODE_1 = 1UL,
      FI_EN_E8PERF_MODE_ADAP_CRUISE_CONTROL_CUSTOMIZATION_PERF_MODE_2 = 2UL,
      FI_EN_E8PERF_MODE_ADAP_CRUISE_CONTROL_CUSTOMIZATION_PERF_MODE_3 = 3UL,
      FI_EN_E8PERF_MODE_ADAP_CRUISE_CONTROL_CUSTOMIZATION_PERF_MODE_4 = 4UL,
      FI_EN_E8PERF_MODE_ADAP_CRUISE_CONTROL_CUSTOMIZATION_PERF_MODE_5 = 5UL,
      FI_EN_E8PERF_MODE_ADAP_CRUISE_CONTROL_CUSTOMIZATION_PERF_MODE_6 = 6UL,
      FI_EN_E8PERF_MODE_ADAP_CRUISE_CONTROL_CUSTOMIZATION_AUTO = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPerfModeSoundCustomizationSettings(most_fi_tcl_e8_GMLnGWPerfModeSoundCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPerfModeSoundCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPerfModeSoundCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPedestrianFriendlyAlertCustomizationSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPedestrianFriendlyAlertCustomizationSetting();
   enum tenType {
      FI_EN_E8PED_FRIENDLY_ALERT_SETTING_UNKNOWN = 0UL,
      FI_EN_E8PED_FRIENDLY_ALERT_OFF = 1UL,
      FI_EN_E8PED_FRIENDLY_ALERT_ON = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPedestrianFriendlyAlertCustomizationSetting(most_fi_tcl_e8_GMLnGWPedestrianFriendlyAlertCustomizationSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPedestrianFriendlyAlertCustomizationSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPedestrianFriendlyAlertCustomizationSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWEngineAssistedHeatingCustomizationSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWEngineAssistedHeatingCustomizationSetting();
   enum tenType {
      FI_EN_E8ENGINE_ASSISTED_HEATING_CUSTOMIZATION_SETTING_UNKNOWN = 0UL,
      FI_EN_E8ENGINE_ASSISTED_HEATING_NORMAL_COLD_OAT = 1UL,
      FI_EN_E8ENGINE_ASSISTED_HEATING_LOWER_VERY_COLD_OAT = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWEngineAssistedHeatingCustomizationSetting(most_fi_tcl_e8_GMLnGWEngineAssistedHeatingCustomizationSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWEngineAssistedHeatingCustomizationSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWEngineAssistedHeatingCustomizationSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWEngineAssistedHeatingPluggedInCustomizationSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWEngineAssistedHeatingPluggedInCustomizationSetting();
   enum tenType {
      FI_EN_E8ENGINE_ASSISTED_HEATING_PLUGGED_IN_CUSTOMIZATION_SETTING_UNKNOWN = 0UL,
      FI_EN_E8ENGINE_ASSISTED_HEATING_PLUGGED_IN_OFF = 1UL,
      FI_EN_E8ENGINE_ASSISTED_HEATING_PLUGGED_IN_ON = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWEngineAssistedHeatingPluggedInCustomizationSetting(most_fi_tcl_e8_GMLnGWEngineAssistedHeatingPluggedInCustomizationSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWEngineAssistedHeatingPluggedInCustomizationSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWEngineAssistedHeatingPluggedInCustomizationSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWElevatedIdleCustomizationSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWElevatedIdleCustomizationSetting();
   enum tenType {
      FI_EN_E8ELEVATED_IDLE_UNKNOWN = 0UL,
      FI_EN_E8ELEVATED_IDLE_DISABLED = 1UL,
      FI_EN_E8ELEVATED_IDLE_ENABLED = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWElevatedIdleCustomizationSetting(most_fi_tcl_e8_GMLnGWElevatedIdleCustomizationSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWElevatedIdleCustomizationSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWElevatedIdleCustomizationSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingWithoutIntelligentLightDistCustomizationSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingWithoutIntelligentLightDistCustomizationSetting();
   enum tenType {
      FI_EN_E8ADAPTIVE_FWD_LIGHTING_WITH_GPS_UNKNOWN = 0UL,
      FI_EN_E8ADAPTIVE_FWD_LIGHTING_WITH_GPS_CORNER_AND_CURVE_LIGHTING_ONLY = 1UL,
      FI_EN_E8ADAPTIVE_FWD_LIGHTING_WITH_GPS_OFF = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingWithoutIntelligentLightDistCustomizationSetting(most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingWithoutIntelligentLightDistCustomizationSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingWithoutIntelligentLightDistCustomizationSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWAdaptiveForwardLightingWithoutIntelligentLightDistCustomizationSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPerfModeMainMenuType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPerfModeMainMenuType();
   enum tenType {
      FI_EN_E8PERF_MODE_MAIN_MENU_NO_ACTION = 0UL,
      FI_EN_E8PERF_MODE_MAIN_MENU_TYPE_1 = 1UL,
      FI_EN_E8PERF_MODE_MAIN_MENU_TYPE_2 = 2UL,
      FI_EN_E8PERF_MODE_MAIN_MENU_TYPE_3 = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPerfModeMainMenuType(most_fi_tcl_e8_GMLnGWPerfModeMainMenuType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPerfModeMainMenuType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPerfModeMainMenuType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPerfModeAdapCruiseControlCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPerfModeAdapCruiseControlCustomizationSettings();
   enum tenType {
      FI_EN_E8PERF_MODE_ADAP_CRUISE_CONTROL_CUSTOMIZATION_DEFAULT = 0UL,
      FI_EN_E8PERF_MODE_ADAP_CRUISE_CONTROL_CUSTOMIZATION_PERF_MODE_1 = 1UL,
      FI_EN_E8PERF_MODE_ADAP_CRUISE_CONTROL_CUSTOMIZATION_PERF_MODE_2 = 2UL,
      FI_EN_E8PERF_MODE_ADAP_CRUISE_CONTROL_CUSTOMIZATION_PERF_MODE_3 = 3UL,
      FI_EN_E8PERF_MODE_ADAP_CRUISE_CONTROL_CUSTOMIZATION_PERF_MODE_4 = 4UL,
      FI_EN_E8PERF_MODE_ADAP_CRUISE_CONTROL_CUSTOMIZATION_PERF_MODE_5 = 5UL,
      FI_EN_E8PERF_MODE_ADAP_CRUISE_CONTROL_CUSTOMIZATION_PERF_MODE_6 = 6UL,
      FI_EN_E8PERF_MODE_ADAP_CRUISE_CONTROL_CUSTOMIZATION_AUTO = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPerfModeAdapCruiseControlCustomizationSettings(most_fi_tcl_e8_GMLnGWPerfModeAdapCruiseControlCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPerfModeAdapCruiseControlCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPerfModeAdapCruiseControlCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPerfModeDriverSeatControlCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPerfModeDriverSeatControlCustomizationSettings();
   enum tenType {
      FI_EN_E8PERF_MODE_DRIVER_SEAT_CUSTOMIZATION_DEFAULT = 0UL,
      FI_EN_E8PERF_MODE_DRIVER_SEAT_CONTROL_CUSTOMIZATION_PERF_MODE_1 = 1UL,
      FI_EN_E8PERF_MODE_DRIVER_SEAT_CONTROL_CUSTOMIZATION_PERF_MODE_2 = 2UL,
      FI_EN_E8PERF_MODE_DRIVER_SEAT_CONTROL_CUSTOMIZATION_PERF_MODE_3 = 3UL,
      FI_EN_E8PERF_MODE_DRIVER_SEAT_CONTROL_CUSTOMIZATION_PERF_MODE_4 = 4UL,
      FI_EN_E8PERF_MODE_DRIVER_SEAT_CONTROL_CUSTOMIZATION_PERF_MODE_5 = 5UL,
      FI_EN_E8PERF_MODE_DRIVER_SEAT_CONTROL_CUSTOMIZATION_PERF_MODE_6 = 6UL,
      FI_EN_E8PERF_MODE_DRIVER_SEAT_CONTROL_CUSTOMIZATION_AUTO = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPerfModeDriverSeatControlCustomizationSettings(most_fi_tcl_e8_GMLnGWPerfModeDriverSeatControlCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPerfModeDriverSeatControlCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPerfModeDriverSeatControlCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPerfModePassengerSeatControlCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPerfModePassengerSeatControlCustomizationSettings();
   enum tenType {
      FI_EN_E8PERF_MODE_PASSENGER_SEAT_CUSTOMIZATION_DEFAULT = 0UL,
      FI_EN_E8PERF_MODE_PASSENGER_SEAT_CONTROL_CUSTOMIZATION_PERF_MODE_1 = 1UL,
      FI_EN_E8PERF_MODE_PASSENGER_SEAT_CONTROL_CUSTOMIZATION_PERF_MODE_2 = 2UL,
      FI_EN_E8PERF_MODE_PASSENGER_SEAT_CONTROL_CUSTOMIZATION_PERF_MODE_3 = 3UL,
      FI_EN_E8PERF_MODE_PASSENGER_SEAT_CONTROL_CUSTOMIZATION_PERF_MODE_4 = 4UL,
      FI_EN_E8PERF_MODE_PASSENGER_SEAT_CONTROL_CUSTOMIZATION_PERF_MODE_5 = 5UL,
      FI_EN_E8PERF_MODE_PASSENGER_SEAT_CONTROL_CUSTOMIZATION_PERF_MODE_6 = 6UL,
      FI_EN_E8PERF_MODE_PASSENGER_SEAT_CONTROL_CUSTOMIZATION_AUTO = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPerfModePassengerSeatControlCustomizationSettings(most_fi_tcl_e8_GMLnGWPerfModePassengerSeatControlCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPerfModePassengerSeatControlCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPerfModePassengerSeatControlCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPerfModeDrivingStyleCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPerfModeDrivingStyleCustomizationSettings();
   enum tenType {
      FI_EN_E8PERF_MODE_DRIVING_STYLE_CUSTOMIZATION_DEFAULT = 0UL,
      FI_EN_E8PERF_MODE_DRIVING_STYLE_CUSTOMIZATION_PERF_MODE_1 = 1UL,
      FI_EN_E8PERF_MODE_DRIVING_STYLE_CUSTOMIZATION_PERF_MODE_2 = 2UL,
      FI_EN_E8PERF_MODE_DRIVING_STYLE_CUSTOMIZATION_PERF_MODE_3 = 3UL,
      FI_EN_E8PERF_MODE_DRIVING_STYLE_CUSTOMIZATION_PERF_MODE_4 = 4UL,
      FI_EN_E8PERF_MODE_DRIVING_STYLE_CUSTOMIZATION_PERF_MODE_5 = 5UL,
      FI_EN_E8PERF_MODE_DRIVING_STYLE_CUSTOMIZATION_PERF_MODE_6 = 6UL,
      FI_EN_E8PERF_MODE_DRIVING_STYLE_CUSTOMIZATION_PERF_MODE_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPerfModeDrivingStyleCustomizationSettings(most_fi_tcl_e8_GMLnGWPerfModeDrivingStyleCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPerfModeDrivingStyleCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPerfModeDrivingStyleCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPerfModeVisualization_AdapCruiseControl : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_AdapCruiseControl();
   enum tenType {
      FI_EN_E8PERF_MODE_VISUALIZATION_ADAP_CRUISE_CONTROL_TYPE_DEFAULT = 0UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_ADAP_CRUISE_CONTROL_TYPE_1 = 1UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_ADAP_CRUISE_CONTROL_TYPE_2 = 2UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_ADAP_CRUISE_CONTROL_TYPE_3 = 3UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_ADAP_CRUISE_CONTROL_TYPE_4 = 4UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_ADAP_CRUISE_CONTROL_TYPE_5 = 5UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_ADAP_CRUISE_CONTROL_TYPE_6 = 6UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_ADAP_CRUISE_CONTROL_TYPE_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_AdapCruiseControl(most_fi_tcl_e8_GMLnGWPerfModeVisualization_AdapCruiseControl::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPerfModeVisualization_AdapCruiseControl();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPerfModeVisualization_AdapCruiseControl& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPerfModeVisualization_Displays : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_Displays();
   enum tenType {
      FI_EN_E8PERF_MODE_VISUALIZATION_DISPLAYS_TYPE_DEFAULT = 0UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DISPLAYS_TYPE_1 = 1UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DISPLAYS_TYPE_2 = 2UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DISPLAYS_TYPE_3 = 3UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DISPLAYS_TYPE_4 = 4UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DISPLAYS_TYPE_5 = 5UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DISPLAYS_TYPE_6 = 6UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DISPLAYS_TYPE_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_Displays(most_fi_tcl_e8_GMLnGWPerfModeVisualization_Displays::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPerfModeVisualization_Displays();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPerfModeVisualization_Displays& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPerfModeVisualization_Driveline : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_Driveline();
   enum tenType {
      FI_EN_E8PERF_MODE_VISUALIZATION_DRIVELINE_TYPE_DEFAULT = 0UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DRIVELINE_TYPE_1 = 1UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DRIVELINE_TYPE_2 = 2UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DRIVELINE_TYPE_3 = 3UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DRIVELINE_TYPE_4 = 4UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DRIVELINE_TYPE_5 = 5UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DRIVELINE_TYPE_6 = 6UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DRIVELINE_TYPE_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_Driveline(most_fi_tcl_e8_GMLnGWPerfModeVisualization_Driveline::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPerfModeVisualization_Driveline();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPerfModeVisualization_Driveline& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPerfModeVisualization_DriverSeat : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_DriverSeat();
   enum tenType {
      FI_EN_E8PERF_MODE_VISUALIZATION_DRIVER_SEAT_TYPE_DEFAULT = 0UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DRIVER_SEAT_TYPE_1 = 1UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DRIVER_SEAT_TYPE_2 = 2UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DRIVER_SEAT_TYPE_3 = 3UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DRIVER_SEAT_TYPE_4 = 4UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DRIVER_SEAT_TYPE_5 = 5UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DRIVER_SEAT_TYPE_6 = 6UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_DRIVER_SEAT_TYPE_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_DriverSeat(most_fi_tcl_e8_GMLnGWPerfModeVisualization_DriverSeat::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPerfModeVisualization_DriverSeat();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPerfModeVisualization_DriverSeat& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPerfModeVisualization_Engine : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_Engine();
   enum tenType {
      FI_EN_E8PERF_MODE_VISUALIZATION_ENGINE_TYPE_DEFAULT = 0UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_ENGINE_TYPE_1 = 1UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_ENGINE_TYPE_2 = 2UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_ENGINE_TYPE_3 = 3UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_ENGINE_TYPE_4 = 4UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_ENGINE_TYPE_5 = 5UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_ENGINE_TYPE_6 = 6UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_ENGINE_TYPE_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_Engine(most_fi_tcl_e8_GMLnGWPerfModeVisualization_Engine::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPerfModeVisualization_Engine();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPerfModeVisualization_Engine& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPerfModeVisualization_Exhaust : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_Exhaust();
   enum tenType {
      FI_EN_E8PERF_MODE_VISUALIZATION_EXHAUST_TYPE_DEFAULT = 0UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_EXHAUST_TYPE_1 = 1UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_EXHAUST_TYPE_2 = 2UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_EXHAUST_TYPE_3 = 3UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_EXHAUST_TYPE_4 = 4UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_EXHAUST_TYPE_5 = 5UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_EXHAUST_TYPE_6 = 6UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_EXHAUST_TYPE_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_Exhaust(most_fi_tcl_e8_GMLnGWPerfModeVisualization_Exhaust::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPerfModeVisualization_Exhaust();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPerfModeVisualization_Exhaust& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPerfModeVisualization_PassengerSeat : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_PassengerSeat();
   enum tenType {
      FI_EN_E8PERF_MODE_VISUALIZATION_PASSENGER_SEAT_TYPE_DEFAULT = 0UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_PASSENGER_SEAT_TYPE_1 = 1UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_PASSENGER_SEAT_TYPE_2 = 2UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_PASSENGER_SEAT_TYPE_3 = 3UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_PASSENGER_SEAT_TYPE_4 = 4UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_PASSENGER_SEAT_TYPE_5 = 5UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_PASSENGER_SEAT_TYPE_6 = 6UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_PASSENGER_SEAT_TYPE_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_PassengerSeat(most_fi_tcl_e8_GMLnGWPerfModeVisualization_PassengerSeat::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPerfModeVisualization_PassengerSeat();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPerfModeVisualization_PassengerSeat& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPerfModeVisualization_Sound : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_Sound();
   enum tenType {
      FI_EN_E8PERF_MODE_VISUALIZATION_SOUND_TYPE_DEFAULT = 0UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_SOUND_TYPE_1 = 1UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_SOUND_TYPE_2 = 2UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_SOUND_TYPE_3 = 3UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_SOUND_TYPE_4 = 4UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_SOUND_TYPE_5 = 5UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_SOUND_TYPE_6 = 6UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_SOUND_TYPE_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_Sound(most_fi_tcl_e8_GMLnGWPerfModeVisualization_Sound::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPerfModeVisualization_Sound();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPerfModeVisualization_Sound& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPerfModeVisualization_Steering : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_Steering();
   enum tenType {
      FI_EN_E8PERF_MODE_VISUALIZATION_STEERING_TYPE_DEFAULT = 0UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_STEERING_TYPE_1 = 1UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_STEERING_TYPE_2 = 2UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_STEERING_TYPE_3 = 3UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_STEERING_TYPE_4 = 4UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_STEERING_TYPE_5 = 5UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_STEERING_TYPE_6 = 6UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_STEERING_TYPE_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_Steering(most_fi_tcl_e8_GMLnGWPerfModeVisualization_Steering::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPerfModeVisualization_Steering();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPerfModeVisualization_Steering& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPerfModeVisualization_Suspension : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_Suspension();
   enum tenType {
      FI_EN_E8PERF_MODE_VISUALIZATION_SUSPENSION_TYPE_DEFAULT = 0UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_SUSPENSION_TYPE_1 = 1UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_SUSPENSION_TYPE_2 = 2UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_SUSPENSION_TYPE_3 = 3UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_SUSPENSION_TYPE_4 = 4UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_SUSPENSION_TYPE_5 = 5UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_SUSPENSION_TYPE_6 = 6UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_SUSPENSION_TYPE_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_Suspension(most_fi_tcl_e8_GMLnGWPerfModeVisualization_Suspension::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPerfModeVisualization_Suspension();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPerfModeVisualization_Suspension& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPerfModeVisualization_Trans : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_Trans();
   enum tenType {
      FI_EN_E8PERF_MODE_VISUALIZATION_TRANS_TYPE_DEFAULT = 0UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_TRANS_TYPE_1 = 1UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_TRANS_TYPE_2 = 2UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_TRANS_TYPE_3 = 3UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_TRANS_TYPE_4 = 4UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_TRANS_TYPE_5 = 5UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_TRANS_TYPE_6 = 6UL,
      FI_EN_E8PERF_MODE_VISUALIZATION_TRANS_TYPE_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPerfModeVisualization_Trans(most_fi_tcl_e8_GMLnGWPerfModeVisualization_Trans::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPerfModeVisualization_Trans();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPerfModeVisualization_Trans& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWExtendedHillStartAssistCustomizationSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWExtendedHillStartAssistCustomizationSetting();
   enum tenType {
      FI_EN_E8EXT_HILL_START_ASST_SETTING_UNKNOWN = 0UL,
      FI_EN_E8EXT_HILL_START_ASST_EXTENDED_HOLD = 1UL,
      FI_EN_E8EXT_HILL_START_ASST_STANDARD_HOLD = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWExtendedHillStartAssistCustomizationSetting(most_fi_tcl_e8_GMLnGWExtendedHillStartAssistCustomizationSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWExtendedHillStartAssistCustomizationSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWExtendedHillStartAssistCustomizationSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRearPedestrianDetectionCustomizationSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRearPedestrianDetectionCustomizationSetting();
   enum tenType {
      FI_EN_E8REAR_PED_DETECT_SETTING_UNKNOWN = 0UL,
      FI_EN_E8REAR_PED_DETECT_OFF = 1UL,
      FI_EN_E8REAR_PED_DETECT_ON = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRearPedestrianDetectionCustomizationSetting(most_fi_tcl_e8_GMLnGWRearPedestrianDetectionCustomizationSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRearPedestrianDetectionCustomizationSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRearPedestrianDetectionCustomizationSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomizationSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomizationSetting();
   enum tenType {
      FI_EN_E8LBC_CUST_SETTING_NO_ACTION = 0UL,
      FI_EN_E8LBC_CUST_SETTING_OFF = 1UL,
      FI_EN_E8LBC_CUST_SETTING_ON = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomizationSetting(most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomizationSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomizationSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomizationSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWFrontPedestrianDetectionCustomizationSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWFrontPedestrianDetectionCustomizationSetting();
   enum tenType {
      FI_EN_E8FRONT_PED_DETECT_SETTING_NO_ACTION = 0UL,
      FI_EN_E8FRONT_PED_DETECT_OFF = 1UL,
      FI_EN_E8FRONT_PED_DETECT_ALERT = 2UL,
      FI_EN_E8FRONT_PED_DETECT_ALERT_AND_BRAKE = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWFrontPedestrianDetectionCustomizationSetting(most_fi_tcl_e8_GMLnGWFrontPedestrianDetectionCustomizationSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWFrontPedestrianDetectionCustomizationSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWFrontPedestrianDetectionCustomizationSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWHandsfreeLiftgateOpenCloseCustomizationSettings : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWHandsfreeLiftgateOpenCloseCustomizationSettings();
   enum tenType {
      FI_EN_E8HANDSFREE_LIFTGATE_CUSTOMIZATION_NOVALUE = 0UL,
      FI_EN_E8HANDSFREE_LIFTGATE_CUSTOMIZATION_OFF = 1UL,
      FI_EN_E8HANDSFREE_LIFTGATE_CUSTOMIZATION_ON = 2UL,
      FI_EN_E8HANDSFREE_LIFTGATE_CUSTOMIZATION_ON_OPEN_ONLY = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWHandsfreeLiftgateOpenCloseCustomizationSettings(most_fi_tcl_e8_GMLnGWHandsfreeLiftgateOpenCloseCustomizationSettings::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWHandsfreeLiftgateOpenCloseCustomizationSettings();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWHandsfreeLiftgateOpenCloseCustomizationSettings& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRearSeatReminderCustomizationSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRearSeatReminderCustomizationSetting();
   enum tenType {
      FI_EN_E8REAR_SEAT_REMINDER_NO_ACTION = 0UL,
      FI_EN_E8REAR_SEAT_REMINDER_OFF = 1UL,
      FI_EN_E8REAR_SEAT_REMINDER_ON = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRearSeatReminderCustomizationSetting(most_fi_tcl_e8_GMLnGWRearSeatReminderCustomizationSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRearSeatReminderCustomizationSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRearSeatReminderCustomizationSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRunningBoardsCustomizationSetting : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRunningBoardsCustomizationSetting();
   enum tenType {
      FI_EN_E8RUNNING_BOARDS_NO_ACTION = 0UL,
      FI_EN_E8RUNNING_BOARDS_OFF = 1UL,
      FI_EN_E8RUNNING_BOARDS_ON = 2UL,
      FI_EN_E8RUNNING_BOARDS_EXTENDED = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRunningBoardsCustomizationSetting(most_fi_tcl_e8_GMLnGWRunningBoardsCustomizationSetting::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRunningBoardsCustomizationSetting();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRunningBoardsCustomizationSetting& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSelectedMode1Status : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSelectedMode1Status();
   enum tenType {
      FI_EN_E8MODE_NOT_ACTIVE = 0UL,
      FI_EN_E8MODE_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSelectedMode1Status(most_fi_tcl_e8_GMLnGWDriverSelectedMode1Status::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSelectedMode1Status();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSelectedMode1Status& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSelectedMode2Status : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSelectedMode2Status();
   enum tenType {
      FI_EN_E8MODE_NOT_ACTIVE = 0UL,
      FI_EN_E8MODE_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSelectedMode2Status(most_fi_tcl_e8_GMLnGWDriverSelectedMode2Status::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSelectedMode2Status();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSelectedMode2Status& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSelectedMode3Status : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSelectedMode3Status();
   enum tenType {
      FI_EN_E8MODE_NOT_ACTIVE = 0UL,
      FI_EN_E8MODE_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSelectedMode3Status(most_fi_tcl_e8_GMLnGWDriverSelectedMode3Status::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSelectedMode3Status();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSelectedMode3Status& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSelectedMode4Status : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSelectedMode4Status();
   enum tenType {
      FI_EN_E8MODE_NOT_ACTIVE = 0UL,
      FI_EN_E8MODE_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSelectedMode4Status(most_fi_tcl_e8_GMLnGWDriverSelectedMode4Status::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSelectedMode4Status();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSelectedMode4Status& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSelectedMode5Status : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSelectedMode5Status();
   enum tenType {
      FI_EN_E8MODE_NOT_ACTIVE = 0UL,
      FI_EN_E8MODE_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSelectedMode5Status(most_fi_tcl_e8_GMLnGWDriverSelectedMode5Status::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSelectedMode5Status();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSelectedMode5Status& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSelectedMode6Status : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSelectedMode6Status();
   enum tenType {
      FI_EN_E8MODE_NOT_ACTIVE = 0UL,
      FI_EN_E8MODE_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSelectedMode6Status(most_fi_tcl_e8_GMLnGWDriverSelectedMode6Status::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSelectedMode6Status();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSelectedMode6Status& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSelectedMode7Status : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSelectedMode7Status();
   enum tenType {
      FI_EN_E8MODE_NOT_ACTIVE = 0UL,
      FI_EN_E8MODE_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSelectedMode7Status(most_fi_tcl_e8_GMLnGWDriverSelectedMode7Status::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSelectedMode7Status();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSelectedMode7Status& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSelectedMode8Status : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSelectedMode8Status();
   enum tenType {
      FI_EN_E8MODE_NOT_ACTIVE = 0UL,
      FI_EN_E8MODE_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSelectedMode8Status(most_fi_tcl_e8_GMLnGWDriverSelectedMode8Status::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSelectedMode8Status();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSelectedMode8Status& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemAudibleWarningRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemAudibleWarningRequest();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8WARNING_1_ACTIVE = 1UL,
      FI_EN_E8WARNING_2_ACTIVE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemAudibleWarningRequest(most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemAudibleWarningRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemAudibleWarningRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemAudibleWarningRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemDisplayMessageRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemDisplayMessageRequest();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8MESSAGE_1_ACTIVE = 1UL,
      FI_EN_E8MESSAGE_2_ACTIVE = 2UL,
      FI_EN_E8MESSAGE_3_ACTIVE = 3UL,
      FI_EN_E8MESSAGE_4_ACTIVE = 4UL,
      FI_EN_E8MESSAGE_5_ACTIVE = 5UL,
      FI_EN_E8MESSAGE_6_ACTIVE = 6UL,
      FI_EN_E8MESSAGE_7_ACTIVE = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemDisplayMessageRequest(most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemDisplayMessageRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemDisplayMessageRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemDisplayMessageRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemStatusIndicationRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemStatusIndicationRequest();
   enum tenType {
      FI_EN_E8NO_INDICATION = 0UL,
      FI_EN_E8CONTINUOUS_INDICATION = 1UL,
      FI_EN_E8FLASH_RATE_1_INDICATION = 2UL,
      FI_EN_E8FLASH_RATE_2_INDICATION = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemStatusIndicationRequest(most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemStatusIndicationRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemStatusIndicationRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemStatusIndicationRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemWarningIndicationRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemWarningIndicationRequest();
   enum tenType {
      FI_EN_E8NO_INDICATION = 0UL,
      FI_EN_E8CONTINUOUS_INDICATION = 1UL,
      FI_EN_E8FLASH_RATE_1_INDICATION = 2UL,
      FI_EN_E8FLASH_RATE_2_INDICATION = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemWarningIndicationRequest(most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemWarningIndicationRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemWarningIndicationRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWElectricParkBrakeSystemWarningIndicationRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWForwardObjectAlertVehicleAheadIndicationRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWForwardObjectAlertVehicleAheadIndicationRequest();
   enum tenType {
      FI_EN_E8NO_ALERT = 0UL,
      FI_EN_E8ALERT_LEVEL_1 = 1UL,
      FI_EN_E8ALERT_LEVEL_2 = 2UL,
      FI_EN_E8ALERT_LEVEL_3 = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWForwardObjectAlertVehicleAheadIndicationRequest(most_fi_tcl_e8_GMLnGWForwardObjectAlertVehicleAheadIndicationRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWForwardObjectAlertVehicleAheadIndicationRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWForwardObjectAlertVehicleAheadIndicationRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWForwardObjectAlertWarningChimeInhibitRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWForwardObjectAlertWarningChimeInhibitRequest();
   enum tenType {
      FI_EN_E8ALLOW_CHIME = 0UL,
      FI_EN_E8INHIBIT_CHIME = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWForwardObjectAlertWarningChimeInhibitRequest(most_fi_tcl_e8_GMLnGWForwardObjectAlertWarningChimeInhibitRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWForwardObjectAlertWarningChimeInhibitRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWForwardObjectAlertWarningChimeInhibitRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWForwardObjectAlertWarningIndicationRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWForwardObjectAlertWarningIndicationRequest();
   enum tenType {
      FI_EN_E8NO_ALERT = 0UL,
      FI_EN_E8ALERT_LEVEL_1 = 1UL,
      FI_EN_E8ALERT_LEVEL_2 = 2UL,
      FI_EN_E8ALERT_LEVEL_3 = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWForwardObjectAlertWarningIndicationRequest(most_fi_tcl_e8_GMLnGWForwardObjectAlertWarningIndicationRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWForwardObjectAlertWarningIndicationRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWForwardObjectAlertWarningIndicationRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWForwardObjectAlertStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWForwardObjectAlertStream();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWForwardObjectAlertStream& operator=(const most_fi_tcl_GMLnGWForwardObjectAlertStream& coRef);
   most_fi_tcl_GMLnGWForwardObjectAlertStream(const most_fi_tcl_GMLnGWForwardObjectAlertStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_GMLnGWForwardObjectAlertVehicleAheadIndicationRequest e8ForwardObjectAlertVehicleAheadIndicationRequest;
   most_fi_tcl_e8_GMLnGWForwardObjectAlertWarningChimeInhibitRequest e8ForwardObjectAlertWarningChimeInhibitRequest;
   most_fi_tcl_e8_GMLnGWForwardObjectAlertWarningIndicationRequest e8ForwardObjectAlertWarningIndicationRequest;
   tBool bForwardCollisionAlertOffIndicationOn;
   virtual ~most_fi_tcl_GMLnGWForwardObjectAlertStream();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWForwardObjectAlertStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLaneDepartureWarningAudibleWarningLocation : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLaneDepartureWarningAudibleWarningLocation();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8LEFT_FRONT_SPEAKER = 1UL,
      FI_EN_E8RIGHT_FRONT_SPEAKER = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLaneDepartureWarningAudibleWarningLocation(most_fi_tcl_e8_GMLnGWLaneDepartureWarningAudibleWarningLocation::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLaneDepartureWarningAudibleWarningLocation();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLaneDepartureWarningAudibleWarningLocation& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLaneDepartureWarningIndicationControlIndicationRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLaneDepartureWarningIndicationControlIndicationRequest();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8CONTINUOUS_INDICATION = 1UL,
      FI_EN_E8FLASH_RATE_1 = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLaneDepartureWarningIndicationControlIndicationRequest(most_fi_tcl_e8_GMLnGWLaneDepartureWarningIndicationControlIndicationRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLaneDepartureWarningIndicationControlIndicationRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLaneDepartureWarningIndicationControlIndicationRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftLaneDepartureWarningStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftLaneDepartureWarningStatus();
   enum tenType {
      FI_EN_E8LEFT_LANE_NOT_READY_TO_ASSIST = 0UL,
      FI_EN_E8LEFT_LANE_READY_TO_ASSIST = 1UL,
      FI_EN_E8LEFT_LANE_ALERT = 2UL,
      FI_EN_E8SERVICE_ALIGNMENT = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftLaneDepartureWarningStatus(most_fi_tcl_e8_GMLnGWLeftLaneDepartureWarningStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftLaneDepartureWarningStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftLaneDepartureWarningStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightLaneDepartureWarningStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightLaneDepartureWarningStatus();
   enum tenType {
      FI_EN_E8RIGHT_LANE_NOT_READY_TO_ASSIST = 0UL,
      FI_EN_E8RIGHT_LANE_READY_TO_ASSIST = 1UL,
      FI_EN_E8RIGHT_LANE_ALERT = 2UL,
      FI_EN_E8SERVICE_ALIGNMENT = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightLaneDepartureWarningStatus(most_fi_tcl_e8_GMLnGWRightLaneDepartureWarningStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightLaneDepartureWarningStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightLaneDepartureWarningStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLaneKeepingAssistAudibleWarningLocation : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLaneKeepingAssistAudibleWarningLocation();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8LEFT_FRONT_SPEAKER = 1UL,
      FI_EN_E8RIGHT_FRONT_SPEAKER = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLaneKeepingAssistAudibleWarningLocation(most_fi_tcl_e8_GMLnGWLaneKeepingAssistAudibleWarningLocation::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLaneKeepingAssistAudibleWarningLocation();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLaneKeepingAssistAudibleWarningLocation& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLaneKeepingAssistIndicationControlIndicationRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLaneKeepingAssistIndicationControlIndicationRequest();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8CONTINUOUS_INDICATION = 1UL,
      FI_EN_E8FLASH_RATE_1 = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLaneKeepingAssistIndicationControlIndicationRequest(most_fi_tcl_e8_GMLnGWLaneKeepingAssistIndicationControlIndicationRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLaneKeepingAssistIndicationControlIndicationRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLaneKeepingAssistIndicationControlIndicationRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftTurnLampActive : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftTurnLampActive();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON_WITHOUT_TELLTALE = 1UL,
      FI_EN_E8ON_WITH_TELLTALE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftTurnLampActive(most_fi_tcl_e8_GMLnGWLeftTurnLampActive::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftTurnLampActive();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftTurnLampActive& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightTurnLampActive : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightTurnLampActive();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8ON_WITHOUT_TELLTALE = 1UL,
      FI_EN_E8ON_WITH_TELLTALE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightTurnLampActive(most_fi_tcl_e8_GMLnGWRightTurnLampActive::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightTurnLampActive();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightTurnLampActive& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWExtendedLightingStatusStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWExtendedLightingStatusStream();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWExtendedLightingStatusStream& operator=(const most_fi_tcl_GMLnGWExtendedLightingStatusStream& coRef);
   most_fi_tcl_GMLnGWExtendedLightingStatusStream(const most_fi_tcl_GMLnGWExtendedLightingStatusStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tBool bHighBeamIndicationOn;
   tBool bFrontFogLightIndicationOn;
   most_fi_tcl_e8_GMLnGWLeftTurnLampActive e8LeftTurnLampActive;
   most_fi_tcl_e8_GMLnGWRightTurnLampActive e8RightTurnLampActive;
   virtual ~most_fi_tcl_GMLnGWExtendedLightingStatusStream();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWExtendedLightingStatusStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWHighVoltageBatteryLimitationDueToTemperatureIndication : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWHighVoltageBatteryLimitationDueToTemperatureIndication();
   enum tenType {
      FI_EN_E8NORMAL = 0UL,
      FI_EN_E8COLD = 1UL,
      FI_EN_E8HOT = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWHighVoltageBatteryLimitationDueToTemperatureIndication(most_fi_tcl_e8_GMLnGWHighVoltageBatteryLimitationDueToTemperatureIndication::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWHighVoltageBatteryLimitationDueToTemperatureIndication();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWHighVoltageBatteryLimitationDueToTemperatureIndication& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWHighVoltageChargerSystemStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWHighVoltageChargerSystemStatus();
   enum tenType {
      FI_EN_E8DEFAULT = 0UL,
      FI_EN_E8IDLE = 1UL,
      FI_EN_E8INITIALIZING = 2UL,
      FI_EN_E8ACTIVE = 3UL,
      FI_EN_E8COMPLETE = 4UL,
      FI_EN_E8ABORTED = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWHighVoltageChargerSystemStatus(most_fi_tcl_e8_GMLnGWHighVoltageChargerSystemStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWHighVoltageChargerSystemStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWHighVoltageChargerSystemStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWExtendedHMIHybridVehicleStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWExtendedHMIHybridVehicleStatus();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWExtendedHMIHybridVehicleStatus& operator=(const most_fi_tcl_GMLnGWExtendedHMIHybridVehicleStatus& coRef);
   most_fi_tcl_GMLnGWExtendedHMIHybridVehicleStatus(const most_fi_tcl_GMLnGWExtendedHMIHybridVehicleStatus& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8OffBoardHighVoltageBulkChargingCompleteTimeDayOfWeek;
   tU8 u8OffBoardHighVoltageBulkChargingCompleteTimeHourOfDay;
   tU8 u8OffBoardHighVoltageBulkChargingCompleteTimeMinuteOfHour;
   tBool bHighVoltageBatteryChargeCordConnectedIndicationOn;
   most_fi_tcl_e8_GMLnGWHighVoltageBatteryLimitationDueToTemperatureIndication e8HighVoltageBatteryLimitationDueToTemperatureIndication;
   most_fi_tcl_e8_GMLnGWHighVoltageChargerSystemStatus e8HighVoltageChargerSystemStatus;
   virtual ~most_fi_tcl_GMLnGWExtendedHMIHybridVehicleStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWExtendedHMIHybridVehicleStatus& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWHighVoltageChargerPowerLevel : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWHighVoltageChargerPowerLevel();
   enum tenType {
      FI_EN_E8UNAVAILABLE = 0UL,
      FI_EN_E8AC_LEVEL1 = 1UL,
      FI_EN_E8AC_LEVEL2 = 2UL,
      FI_EN_E8DC = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWHighVoltageChargerPowerLevel(most_fi_tcl_e8_GMLnGWHighVoltageChargerPowerLevel::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWHighVoltageChargerPowerLevel();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWHighVoltageChargerPowerLevel& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWPropulsionDisplayInfoStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWPropulsionDisplayInfoStream();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWPropulsionDisplayInfoStream& operator=(const most_fi_tcl_GMLnGWPropulsionDisplayInfoStream& coRef);
   most_fi_tcl_GMLnGWPropulsionDisplayInfoStream(const most_fi_tcl_GMLnGWPropulsionDisplayInfoStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8PropulsionCapabilityDisplayOperatingPosition;
   tU16 u16PropulsionDisplayTotalPowerLevel;
   tS16 s16PropulsionDisplayTotalPowerLevelPercent;
   virtual ~most_fi_tcl_GMLnGWPropulsionDisplayInfoStream();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWPropulsionDisplayInfoStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWRedundantHighVoltageBatteryPackStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWRedundantHighVoltageBatteryPackStream();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWRedundantHighVoltageBatteryPackStream& operator=(const most_fi_tcl_GMLnGWRedundantHighVoltageBatteryPackStream& coRef);
   most_fi_tcl_GMLnGWRedundantHighVoltageBatteryPackStream(const most_fi_tcl_GMLnGWRedundantHighVoltageBatteryPackStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tS16 s16RedundantHighVoltageBatteryPackCurrent;
   tBool bRedundantHighVoltageBatteryPackCurrentValidity;
   tU16 u16RedundantHighVoltageBatteryPackVoltage;
   tBool bRedundantHighVoltageBatteryPackVoltageValidity;
   virtual ~most_fi_tcl_GMLnGWRedundantHighVoltageBatteryPackStream();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWRedundantHighVoltageBatteryPackStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWAirbagImpactDataStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWAirbagImpactDataStream();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWAirbagImpactDataStream& operator=(const most_fi_tcl_GMLnGWAirbagImpactDataStream& coRef);
   most_fi_tcl_GMLnGWAirbagImpactDataStream(const most_fi_tcl_GMLnGWAirbagImpactDataStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tBool bNotificationEventDeployedLoopStatusLeftSideAirbagDeployed;
   tBool bNotificationEventDeployedLoopStatusRightSideAirbagDeployed;
   tBool bNotificationEventDeployedLoopStatusSideCurtainAirbagDeployed;
   virtual ~most_fi_tcl_GMLnGWAirbagImpactDataStream();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWAirbagImpactDataStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWHillDescentControlSystemStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWHillDescentControlSystemStatus();
   enum tenType {
      FI_EN_E8NORMAL = 0UL,
      FI_EN_E8ENABLED = 1UL,
      FI_EN_E8ACTIVE = 2UL,
      FI_EN_E8FAILED = 3UL,
      FI_EN_E8TEMPORARILY_INHIBITED = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWHillDescentControlSystemStatus(most_fi_tcl_e8_GMLnGWHillDescentControlSystemStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWHillDescentControlSystemStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWHillDescentControlSystemStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWVehicleAheadDistanceIndicationRequestIndicationLevel : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWVehicleAheadDistanceIndicationRequestIndicationLevel();
   enum tenType {
      FI_EN_E8NO_INDICATION = 0UL,
      FI_EN_E8INDICATION_LEVEL_1 = 1UL,
      FI_EN_E8INDICATION_LEVEL_2 = 2UL,
      FI_EN_E8INDICATION_LEVEL_3 = 3UL,
      FI_EN_E8INDICATION_LEVEL_4 = 4UL,
      FI_EN_E8INDICATION_LEVEL_5 = 5UL,
      FI_EN_E8INDICATION_LEVEL_6 = 6UL,
      FI_EN_E8INDICATION_LEVEL_7 = 7UL,
      FI_EN_E8INDICATION_LEVEL_8 = 8UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWVehicleAheadDistanceIndicationRequestIndicationLevel(most_fi_tcl_e8_GMLnGWVehicleAheadDistanceIndicationRequestIndicationLevel::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWVehicleAheadDistanceIndicationRequestIndicationLevel();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWVehicleAheadDistanceIndicationRequestIndicationLevel& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPedestrianWarningIndicationRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPedestrianWarningIndicationRequest();
   enum tenType {
      FI_EN_E8NONE = 0UL,
      FI_EN_E8PEDESTRIAN_DETECTED = 1UL,
      FI_EN_E8PEDESTRIAN_ALERT = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPedestrianWarningIndicationRequest(most_fi_tcl_e8_GMLnGWPedestrianWarningIndicationRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPedestrianWarningIndicationRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPedestrianWarningIndicationRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWSpeedLimiterSettingTypeActive : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWSpeedLimiterSettingTypeActive();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8MANUAL = 1UL,
      FI_EN_E8AUTO = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWSpeedLimiterSettingTypeActive(most_fi_tcl_e8_GMLnGWSpeedLimiterSettingTypeActive::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWSpeedLimiterSettingTypeActive();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWSpeedLimiterSettingTypeActive& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWSpeedLimitVisionFusedSpeedMask : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWSpeedLimitVisionFusedSpeedMask();
   enum tenType {
      FI_EN_E8DONT_USE_DATA = 0UL,
      FI_EN_E8USE_DATA = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWSpeedLimitVisionFusedSpeedMask(most_fi_tcl_e8_GMLnGWSpeedLimitVisionFusedSpeedMask::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWSpeedLimitVisionFusedSpeedMask();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWSpeedLimitVisionFusedSpeedMask& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWSpeedLimitVisionFusedSpeedUnit : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWSpeedLimitVisionFusedSpeedUnit();
   enum tenType {
      FI_EN_E8KM_H = 0UL,
      FI_EN_E8MPH = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWSpeedLimitVisionFusedSpeedUnit(most_fi_tcl_e8_GMLnGWSpeedLimitVisionFusedSpeedUnit::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWSpeedLimitVisionFusedSpeedUnit();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWSpeedLimitVisionFusedSpeedUnit& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWSpeedLimitVisionFusedSpeedStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWSpeedLimitVisionFusedSpeedStream();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWSpeedLimitVisionFusedSpeedStream& operator=(const most_fi_tcl_GMLnGWSpeedLimitVisionFusedSpeedStream& coRef);
   most_fi_tcl_GMLnGWSpeedLimitVisionFusedSpeedStream(const most_fi_tcl_GMLnGWSpeedLimitVisionFusedSpeedStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8SpeedLimitVisionFusedSpeed;
   most_fi_tcl_e8_GMLnGWSpeedLimitVisionFusedSpeedMask e8SpeedLimitVisionFusedSpeedMask;
   most_fi_tcl_e8_GMLnGWSpeedLimitVisionFusedSpeedUnit e8SpeedLimitVisionFusedSpeedUnit;
   virtual ~most_fi_tcl_GMLnGWSpeedLimitVisionFusedSpeedStream();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWSpeedLimitVisionFusedSpeedStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPhoneSpeedRecAudioPromptStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPhoneSpeedRecAudioPromptStatus();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8SPEECH_REC_ACTIVE = 1UL,
      FI_EN_E8AUDIO_PROMPT_ACTIVE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPhoneSpeedRecAudioPromptStatus(most_fi_tcl_e8_GMLnGWPhoneSpeedRecAudioPromptStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPhoneSpeedRecAudioPromptStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPhoneSpeedRecAudioPromptStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTransmissionEngagedState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTransmissionEngagedState();
   enum tenType {
      FI_EN_E8TRANSMISSION_NOT_ENGAGED = 0UL,
      FI_EN_E8TRANSMISSION_ENGAGED_IN_FORWARD = 1UL,
      FI_EN_E8TRANSMISSION_ENGAGED_IN_REVERSE = 2UL,
      FI_EN_E8OPERATING_STATE_NOT_REACHED = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTransmissionEngagedState(most_fi_tcl_e8_GMLnGWTransmissionEngagedState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTransmissionEngagedState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTransmissionEngagedState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWVehicleStabilityEnhancementMode : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWVehicleStabilityEnhancementMode();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8NORMAL = 1UL,
      FI_EN_E8COMPETITIVE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWVehicleStabilityEnhancementMode(most_fi_tcl_e8_GMLnGWVehicleStabilityEnhancementMode::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWVehicleStabilityEnhancementMode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWVehicleStabilityEnhancementMode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWVehicleStabilityEnhancementStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWVehicleStabilityEnhancementStatus();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8FAULT = 2UL,
      FI_EN_E8WARMING_UP = 3UL,
      FI_EN_E8NOT_READY = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWVehicleStabilityEnhancementStatus(most_fi_tcl_e8_GMLnGWVehicleStabilityEnhancementStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWVehicleStabilityEnhancementStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWVehicleStabilityEnhancementStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWVehicleStabilityInfoStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWVehicleStabilityInfoStream();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWVehicleStabilityInfoStream& operator=(const most_fi_tcl_GMLnGWVehicleStabilityInfoStream& coRef);
   most_fi_tcl_GMLnGWVehicleStabilityInfoStream(const most_fi_tcl_GMLnGWVehicleStabilityInfoStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_GMLnGWVehicleStabilityEnhancementMode e8VehicleStabilityEnhancementMode;
   most_fi_tcl_e8_GMLnGWVehicleStabilityEnhancementStatus e8VehicleStabilityEnhancementStatus;
   virtual ~most_fi_tcl_GMLnGWVehicleStabilityInfoStream();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWVehicleStabilityInfoStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWVehicleDisplayUnits : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWVehicleDisplayUnits();
   enum tenType {
      FI_EN_E8VEHICLE_DISPLAY_UNITS_METRIC = 0UL,
      FI_EN_E8VEHICLE_DISPLAY_UNITS_US = 1UL,
      FI_EN_E8VEHICLE_DISPLAY_UNITS_IMPERIAL = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWVehicleDisplayUnits(most_fi_tcl_e8_GMLnGWVehicleDisplayUnits::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWVehicleDisplayUnits();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWVehicleDisplayUnits& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWVINDigitStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWVINDigitStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWVINDigitStream& operator=(const most_fi_tcl_GMLnGWVINDigitStream& coRef);
   most_fi_tcl_GMLnGWVINDigitStream(const most_fi_tcl_GMLnGWVINDigitStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU8, std::allocator<tU8> >u8Items;

   tU8& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_GMLnGWVINDigitStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWVINDigitStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTransmissionShiftLeverPosition : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTransmissionShiftLeverPosition();
   enum tenType {
      FI_EN_E8TRANS_SHIFT_LEVER_POS_BETWEEN_RANGES = 0UL,
      FI_EN_E8TRANS_SHIFT_LEVER_POS_PARK_RANGE = 1UL,
      FI_EN_E8TRANS_SHIFT_LEVER_POS_REVERSE_RANGE = 2UL,
      FI_EN_E8TRANS_SHIFT_LEVER_POS_NEUTRAL_RANGE = 3UL,
      FI_EN_E8TRANS_SHIFT_LEVER_POS_FORWARD_RANGE_A = 4UL,
      FI_EN_E8TRANS_SHIFT_LEVER_POS_FORWARD_RANGE_B = 5UL,
      FI_EN_E8TRANS_SHIFT_LEVER_POS_FORWARD_RANGE_C = 6UL,
      FI_EN_E8TRANS_SHIFT_LEVER_POS_FORWARD_RANGE_D = 7UL,
      FI_EN_E8TRANS_SHIFT_LEVER_POS_FORWARD_RANGE_E = 8UL,
      FI_EN_E8TRANS_SHIFT_LEVER_POS_FORWARD_RANGE_F = 9UL,
      FI_EN_E8TRANS_SHIFT_LEVER_POS_LEVER_POSITION_UNKNOWN = 15UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTransmissionShiftLeverPosition(most_fi_tcl_e8_GMLnGWTransmissionShiftLeverPosition::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTransmissionShiftLeverPosition();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTransmissionShiftLeverPosition& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTransmissionEstimatedGearPosition : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTransmissionEstimatedGearPosition();
   enum tenType {
      FI_EN_E8TRANS_EST_GEAR_POS_NOT_SUPPORTED = 0UL,
      FI_EN_E8TRANS_EST_GEAR_POS_FIRST_GEAR = 1UL,
      FI_EN_E8TRANS_EST_GEAR_POS_SECOND_GEAR = 2UL,
      FI_EN_E8TRANS_EST_GEAR_POS_THIRD_GEAR = 3UL,
      FI_EN_E8TRANS_EST_GEAR_POS_FOURTH_GEAR = 4UL,
      FI_EN_E8TRANS_EST_GEAR_POS_FIFTH_GEAR = 5UL,
      FI_EN_E8TRANS_EST_GEAR_POS_SIXTH_GEAR = 6UL,
      FI_EN_E8TRANS_EST_GEAR_POS_SEVENTH_GEAR = 7UL,
      FI_EN_E8TRANS_EST_GEAR_POS_EIGHTH_GEAR = 8UL,
      FI_EN_E8TRANS_EST_GEAR_POS_EVT_MODE1 = 10UL,
      FI_EN_E8TRANS_EST_GEAR_POS_EVT_MODE2 = 11UL,
      FI_EN_E8TRANS_EST_GEAR_POS_CVT_FORWARD_GEAR = 12UL,
      FI_EN_E8TRANS_EST_GEAR_POS_NEUTRAL_GEAR = 13UL,
      FI_EN_E8TRANS_EST_GEAR_POS_REVERSE_GEAR = 14UL,
      FI_EN_E8TRANS_EST_GEAR_POS_PARK_GEAR = 15UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTransmissionEstimatedGearPosition(most_fi_tcl_e8_GMLnGWTransmissionEstimatedGearPosition::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTransmissionEstimatedGearPosition();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTransmissionEstimatedGearPosition& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWSWCSwitchState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWSWCSwitchState();
   enum tenType {
      FI_EN_E8SWC_ALL_RELEASED = 0UL,
      FI_EN_E8SWC_BAND_PRESSED = 1UL,
      FI_EN_E8SWC_VOLUME_UP_PRESSED = 2UL,
      FI_EN_E8SWC_VOLUME_DOWN_PRESSED = 3UL,
      FI_EN_E8SWC_NEXT_PRESSED = 4UL,
      FI_EN_E8SWC_PREVIOUS_PRESSED = 5UL,
      FI_EN_E8SWC_PTT_PRESSED = 6UL,
      FI_EN_E8SWC_END_PRESSED = 7UL,
      FI_EN_E8SWC_SRC_PRESSED = 8UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWSWCSwitchState(most_fi_tcl_e8_GMLnGWSWCSwitchState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWSWCSwitchState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWSWCSwitchState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRearCrossTrafficAlertLeftIndicationControl : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRearCrossTrafficAlertLeftIndicationControl();
   enum tenType {
      FI_EN_E8OUTSIDE_AMBIENT_LIGHT_LEVEL_UNKNOWN = 0UL,
      FI_EN_E8REAR_CROSS_TRAFFIC_ALERT_INDICATION_CONTROL_CONTINUOUS_INDICATION = 1UL,
      FI_EN_E8REAR_CROSS_TRAFFIC_ALERT_INDICATION_CONTROL_FLASHING_INDICATION = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRearCrossTrafficAlertLeftIndicationControl(most_fi_tcl_e8_GMLnGWRearCrossTrafficAlertLeftIndicationControl::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRearCrossTrafficAlertLeftIndicationControl();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRearCrossTrafficAlertLeftIndicationControl& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRearCrossTrafficAlertRightIndicationControl : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRearCrossTrafficAlertRightIndicationControl();
   enum tenType {
      FI_EN_E8OUTSIDE_AMBIENT_LIGHT_LEVEL_UNKNOWN = 0UL,
      FI_EN_E8REAR_CROSS_TRAFFIC_ALERT_INDICATION_CONTROL_CONTINUOUS_INDICATION = 1UL,
      FI_EN_E8REAR_CROSS_TRAFFIC_ALERT_INDICATION_CONTROL_FLASHING_INDICATION = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRearCrossTrafficAlertRightIndicationControl(most_fi_tcl_e8_GMLnGWRearCrossTrafficAlertRightIndicationControl::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRearCrossTrafficAlertRightIndicationControl();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRearCrossTrafficAlertRightIndicationControl& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWParkAssistFrontRegion1ObjectStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWParkAssistFrontRegion1ObjectStatus();
   enum tenType {
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_UNKNOWN = 0UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_1 = 1UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_2 = 2UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_3 = 3UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_4 = 4UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_5 = 5UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_6 = 6UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_7 = 7UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_8 = 8UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_9 = 9UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_10 = 10UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_11 = 11UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_12 = 12UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_13 = 13UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_14 = 14UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_15 = 15UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWParkAssistFrontRegion1ObjectStatus(most_fi_tcl_e8_GMLnGWParkAssistFrontRegion1ObjectStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWParkAssistFrontRegion1ObjectStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWParkAssistFrontRegion1ObjectStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWParkAssistFrontRegion2ObjectStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWParkAssistFrontRegion2ObjectStatus();
   enum tenType {
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_UNKNOWN = 0UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_1 = 1UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_2 = 2UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_3 = 3UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_4 = 4UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_5 = 5UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_6 = 6UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_7 = 7UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_8 = 8UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_9 = 9UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_10 = 10UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_11 = 11UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_12 = 12UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_13 = 13UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_14 = 14UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_15 = 15UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWParkAssistFrontRegion2ObjectStatus(most_fi_tcl_e8_GMLnGWParkAssistFrontRegion2ObjectStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWParkAssistFrontRegion2ObjectStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWParkAssistFrontRegion2ObjectStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWParkAssistFrontRegion3ObjectStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWParkAssistFrontRegion3ObjectStatus();
   enum tenType {
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_UNKNOWN = 0UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_1 = 1UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_2 = 2UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_3 = 3UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_4 = 4UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_5 = 5UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_6 = 6UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_7 = 7UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_8 = 8UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_9 = 9UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_10 = 10UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_11 = 11UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_12 = 12UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_13 = 13UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_14 = 14UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_15 = 15UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWParkAssistFrontRegion3ObjectStatus(most_fi_tcl_e8_GMLnGWParkAssistFrontRegion3ObjectStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWParkAssistFrontRegion3ObjectStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWParkAssistFrontRegion3ObjectStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWParkAssistFrontRegion4ObjectStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWParkAssistFrontRegion4ObjectStatus();
   enum tenType {
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_UNKNOWN = 0UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_1 = 1UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_2 = 2UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_3 = 3UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_4 = 4UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_5 = 5UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_6 = 6UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_7 = 7UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_8 = 8UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_9 = 9UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_10 = 10UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_11 = 11UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_12 = 12UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_13 = 13UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_14 = 14UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_15 = 15UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWParkAssistFrontRegion4ObjectStatus(most_fi_tcl_e8_GMLnGWParkAssistFrontRegion4ObjectStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWParkAssistFrontRegion4ObjectStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWParkAssistFrontRegion4ObjectStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWParkAssistFrontSystemStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWParkAssistFrontSystemStatus();
   enum tenType {
      FI_EN_E8PARK_ASSIST_SYSTEM_STATUS_DISABLED = 0UL,
      FI_EN_E8PARK_ASSIST_SYSTEM_STATUS_ENABLED = 1UL,
      FI_EN_E8PARK_ASSIST_SYSTEM_STATUS_INHIBITED = 2UL,
      FI_EN_E8PARK_ASSIST_SYSTEM_STATUS_FAILED = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWParkAssistFrontSystemStatus(most_fi_tcl_e8_GMLnGWParkAssistFrontSystemStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWParkAssistFrontSystemStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWParkAssistFrontSystemStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWParkAssistRearRegion1ObjectStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWParkAssistRearRegion1ObjectStatus();
   enum tenType {
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_UNKNOWN = 0UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_1 = 1UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_2 = 2UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_3 = 3UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_4 = 4UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_5 = 5UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_6 = 6UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_7 = 7UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_8 = 8UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_9 = 9UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_10 = 10UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_11 = 11UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_12 = 12UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_13 = 13UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_14 = 14UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_15 = 15UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWParkAssistRearRegion1ObjectStatus(most_fi_tcl_e8_GMLnGWParkAssistRearRegion1ObjectStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWParkAssistRearRegion1ObjectStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWParkAssistRearRegion1ObjectStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWParkAssistRearRegion2ObjectStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWParkAssistRearRegion2ObjectStatus();
   enum tenType {
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_UNKNOWN = 0UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_1 = 1UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_2 = 2UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_3 = 3UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_4 = 4UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_5 = 5UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_6 = 6UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_7 = 7UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_8 = 8UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_9 = 9UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_10 = 10UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_11 = 11UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_12 = 12UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_13 = 13UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_14 = 14UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_15 = 15UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWParkAssistRearRegion2ObjectStatus(most_fi_tcl_e8_GMLnGWParkAssistRearRegion2ObjectStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWParkAssistRearRegion2ObjectStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWParkAssistRearRegion2ObjectStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWParkAssistRearRegion3ObjectStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWParkAssistRearRegion3ObjectStatus();
   enum tenType {
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_UNKNOWN = 0UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_1 = 1UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_2 = 2UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_3 = 3UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_4 = 4UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_5 = 5UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_6 = 6UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_7 = 7UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_8 = 8UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_9 = 9UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_10 = 10UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_11 = 11UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_12 = 12UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_13 = 13UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_14 = 14UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_15 = 15UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWParkAssistRearRegion3ObjectStatus(most_fi_tcl_e8_GMLnGWParkAssistRearRegion3ObjectStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWParkAssistRearRegion3ObjectStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWParkAssistRearRegion3ObjectStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWParkAssistRearRegion4ObjectStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWParkAssistRearRegion4ObjectStatus();
   enum tenType {
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_UNKNOWN = 0UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_1 = 1UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_2 = 2UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_3 = 3UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_4 = 4UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_5 = 5UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_6 = 6UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_7 = 7UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_8 = 8UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_9 = 9UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_10 = 10UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_11 = 11UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_12 = 12UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_13 = 13UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_14 = 14UL,
      FI_EN_E8PARK_ASSIST_OBJECT_STATUS_ZONE_15 = 15UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWParkAssistRearRegion4ObjectStatus(most_fi_tcl_e8_GMLnGWParkAssistRearRegion4ObjectStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWParkAssistRearRegion4ObjectStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWParkAssistRearRegion4ObjectStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWParkAssistRearSystemStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWParkAssistRearSystemStatus();
   enum tenType {
      FI_EN_E8PARK_ASSIST_SYSTEM_STATUS_DISABLED = 0UL,
      FI_EN_E8PARK_ASSIST_SYSTEM_STATUS_ENABLED = 1UL,
      FI_EN_E8PARK_ASSIST_SYSTEM_STATUS_INHIBITED = 2UL,
      FI_EN_E8PARK_ASSIST_SYSTEM_STATUS_FAILED = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWParkAssistRearSystemStatus(most_fi_tcl_e8_GMLnGWParkAssistRearSystemStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWParkAssistRearSystemStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWParkAssistRearSystemStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWXMCurrentStationService : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWXMCurrentStationService();
   enum tenType {
      FI_EN_E8XM_STATION_SERVICE_NODATA = 0UL,
      FI_EN_E8XM_STATION_SERVICE_GENERAL_SUBSCRIPTION = 1UL,
      FI_EN_E8XM_STATION_SERVICE_PAY_PER_USE = 2UL,
      FI_EN_E8XM_STATION_SERVICE_PROMOTION = 3UL,
      FI_EN_E8XM_STATION_SERVICE_UNSUBSCRIBED = 4UL,
      FI_EN_E8XM_STATION_SERVICE_OTHER = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWXMCurrentStationService(most_fi_tcl_e8_GMLnGWXMCurrentStationService::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWXMCurrentStationService();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWXMCurrentStationService& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWXMRadioServiceProvider : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWXMRadioServiceProvider();
   enum tenType {
      FI_EN_E8XM_SERVICE_PROVIDER_UNAVAILABLE = 0UL,
      FI_EN_E8XM_SERVICE_PROVIDER_XM = 1UL,
      FI_EN_E8XM_SERVICE_PROVIDER_SIRIUX = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWXMRadioServiceProvider(most_fi_tcl_e8_GMLnGWXMRadioServiceProvider::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWXMRadioServiceProvider();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWXMRadioServiceProvider& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWSpeedLimitUnits : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWSpeedLimitUnits();
   enum tenType {
      FI_EN_E8SPEED_LIMIT_UNITS_MPH = 0UL,
      FI_EN_E8SPEED_LIMIT_UNITS_KMH = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWSpeedLimitUnits(most_fi_tcl_e8_GMLnGWSpeedLimitUnits::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWSpeedLimitUnits();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWSpeedLimitUnits& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLanguageSelection : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLanguageSelection();
   enum tenType {
      FI_EN_E8LANGUAGE_SELECTION_NA_ENGLISH = 0UL,
      FI_EN_E8LANGUAGE_SELECTION_GERMAN = 1UL,
      FI_EN_E8LANGUAGE_SELECTION_ITALIAN = 2UL,
      FI_EN_E8LANGUAGE_SELECTION_SWEDISH = 3UL,
      FI_EN_E8LANGUAGE_SELECTION_FRENCH = 4UL,
      FI_EN_E8LANGUAGE_SELECTION_SPANISH = 5UL,
      FI_EN_E8LANGUAGE_SELECTION_DUTCH = 6UL,
      FI_EN_E8LANGUAGE_SELECTION_PORTUGUESE = 7UL,
      FI_EN_E8LANGUAGE_SELECTION_NORWEGIAN = 8UL,
      FI_EN_E8LANGUAGE_SELECTION_FINNISH = 9UL,
      FI_EN_E8LANGUAGE_SELECTION_DANISH = 10UL,
      FI_EN_E8LANGUAGE_SELECTION_GREEK = 11UL,
      FI_EN_E8LANGUAGE_SELECTION_JAPANESE = 12UL,
      FI_EN_E8LANGUAGE_SELECTION_ARABIC = 13UL,
      FI_EN_E8LANGUAGE_SELECTION_STANDARD_CHINESE = 14UL,
      FI_EN_E8LANGUAGE_SELECTION_POLISH = 15UL,
      FI_EN_E8LANGUAGE_SELECTION_TURKISH = 16UL,
      FI_EN_E8LANGUAGE_SELECTION_KOREAN = 17UL,
      FI_EN_E8LANGUAGE_SELECTION_TRADITIONAL_CHINESE = 18UL,
      FI_EN_E8LANGUAGE_SELECTION_UK_ENGLISH = 19UL,
      FI_EN_E8LANGUAGE_SELECTION_HUNGARIAN = 20UL,
      FI_EN_E8LANGUAGE_SELECTION_CZECH = 21UL,
      FI_EN_E8LANGUAGE_SELECTION_SLOVAK = 22UL,
      FI_EN_E8LANGUAGE_SELECTION_RUSSIAN = 23UL,
      FI_EN_E8LANGUAGE_SELECTION_BRAZILIAN_PORTUGUESE = 24UL,
      FI_EN_E8LANGUAGE_SELECTION_THAI = 25UL,
      FI_EN_E8LANGUAGE_SELECTION_BULGARIAN = 26UL,
      FI_EN_E8LANGUAGE_SELECTION_ROMANIAN = 27UL,
      FI_EN_E8LANGUAGE_SELECTION_SLOVENIAN = 28UL,
      FI_EN_E8LANGUAGE_SELECTION_CROATIAN = 29UL,
      FI_EN_E8LANGUAGE_SELECTION_UKRAINIAN = 30UL,
      FI_EN_E8LANGUAGE_SELECTION_NA_FRENCH = 31UL,
      FI_EN_E8LANGUAGE_SELECTION_NA_SPANISH = 32UL,
      FI_EN_E8LANGUAGE_SELECTION_CANTONESE = 33UL,
      FI_EN_E8LANGUAGE_SELECTION_UZBEK = 34UL,
      FI_EN_E8LANGUAGE_SELECTION_SERBIAN = 35UL,
      FI_EN_E8LANGUAGE_SELECTION_LITHUANIAN = 36UL,
      FI_EN_E8LANGUAGE_SELECTION_ESTONIAN = 37UL,
      FI_EN_E8LANGUAGE_SELECTION_LATVIAN = 38UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLanguageSelection(most_fi_tcl_e8_GMLnGWLanguageSelection::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLanguageSelection();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLanguageSelection& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWFuelEconomyMetricDisplayUnits : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWFuelEconomyMetricDisplayUnits();
   enum tenType {
      FI_EN_E8FUEL_ECONOMY_METRIC_UNITS_KILOMETERS_PER_LITER = 0UL,
      FI_EN_E8FUEL_ECONOMY_METRIC_UNITS_LITERS_PER_100KM = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWFuelEconomyMetricDisplayUnits(most_fi_tcl_e8_GMLnGWFuelEconomyMetricDisplayUnits::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWFuelEconomyMetricDisplayUnits();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWFuelEconomyMetricDisplayUnits& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPowerConvertibleTopStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPowerConvertibleTopStatus();
   enum tenType {
      FI_EN_E8POWER_CONV_TOP_FULLY_OPENED = 0UL,
      FI_EN_E8POWER_CONV_TOP_FULLY_CLOSED = 1UL,
      FI_EN_E8POWER_CONV_TOP_OPENING = 2UL,
      FI_EN_E8POWER_CONV_TOP_CLOSING = 3UL,
      FI_EN_E8POWER_CONV_TOP_NOT_SECURE_NOT_MOVING = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPowerConvertibleTopStatus(most_fi_tcl_e8_GMLnGWPowerConvertibleTopStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPowerConvertibleTopStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPowerConvertibleTopStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWEngineCylinderDeactivationMode : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWEngineCylinderDeactivationMode();
   enum tenType {
      FI_EN_E8ENG_CYL_DEACT_MODE_ALL_CYLINDERS_ACTIVE = 0UL,
      FI_EN_E8ENG_CYL_DEACT_MODE_DEACIVATION_IN_PROGRESS = 1UL,
      FI_EN_E8ENG_CYL_DEACT_MODE_HALF_TOTAL_CYLINDERS_ACTIVE = 2UL,
      FI_EN_E8ENG_CYL_DEACT_MODE_REACTIVATION_IN_PROGRESS = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWEngineCylinderDeactivationMode(most_fi_tcl_e8_GMLnGWEngineCylinderDeactivationMode::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWEngineCylinderDeactivationMode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWEngineCylinderDeactivationMode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWVehicleMovementState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWVehicleMovementState();
   enum tenType {
      FI_EN_E8VEHICLE_MOVEMENT_STATE_PARKED = 0UL,
      FI_EN_E8VEHICLE_MOVEMENT_STATE_NEUTRAL = 1UL,
      FI_EN_E8VEHICLE_MOVEMENT_STATE_FORWARD = 2UL,
      FI_EN_E8VEHICLE_MOVEMENT_STATE_REVERSE = 3UL,
      FI_EN_E8VEHICLE_MOVEMENT_STATE_INVALID = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWVehicleMovementState(most_fi_tcl_e8_GMLnGWVehicleMovementState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWVehicleMovementState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWVehicleMovementState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWFrontPassengerSeatbeltReminderOccupancyStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWFrontPassengerSeatbeltReminderOccupancyStatus();
   enum tenType {
      FI_EN_E8FRNT_PSGR_OCCUPANCY_EMPTY = 0UL,
      FI_EN_E8FRNT_PSGR_OCCUPANCY_OCCUPIED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWFrontPassengerSeatbeltReminderOccupancyStatus(most_fi_tcl_e8_GMLnGWFrontPassengerSeatbeltReminderOccupancyStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWFrontPassengerSeatbeltReminderOccupancyStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWFrontPassengerSeatbeltReminderOccupancyStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWFrontPassengerSeatOccupancyFaultStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWFrontPassengerSeatOccupancyFaultStatus();
   enum tenType {
      FI_EN_E8FRNT_PSGR_OCCUPANCY_NO_FAULTS_PRESENT = 0UL,
      FI_EN_E8FRNT_PSGR_OCCUPANCY_FAULT_PRESENT = 1UL,
      FI_EN_E8FRNT_PSGR_OCCUPANCY_FAULT_PRESENT_VEHICLE_MISMATCH = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWFrontPassengerSeatOccupancyFaultStatus(most_fi_tcl_e8_GMLnGWFrontPassengerSeatOccupancyFaultStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWFrontPassengerSeatOccupancyFaultStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWFrontPassengerSeatOccupancyFaultStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTargaTopState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTargaTopState();
   enum tenType {
      FI_EN_E8TARGA_TOP_STATE_OPEN = 0UL,
      FI_EN_E8TARGA_TOP_STATE_CLOSED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTargaTopState(most_fi_tcl_e8_GMLnGWTargaTopState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTargaTopState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTargaTopState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTargaTopStateValidity : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTargaTopStateValidity();
   enum tenType {
      FI_EN_E8VALID = 0UL,
      FI_EN_E8INVALID = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTargaTopStateValidity(most_fi_tcl_e8_GMLnGWTargaTopStateValidity::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTargaTopStateValidity();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTargaTopStateValidity& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWCompassOctantHeading : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWCompassOctantHeading();
   enum tenType {
      FI_EN_E8COMPASS_OCTANT_HEADING_NORTH = 0UL,
      FI_EN_E8COMPASS_OCTANT_HEADING_NORTHEAST = 1UL,
      FI_EN_E8COMPASS_OCTANT_HEADING_EAST = 2UL,
      FI_EN_E8COMPASS_OCTANT_HEADING_SOUTHEAST = 3UL,
      FI_EN_E8COMPASS_OCTANT_HEADING_SOUTH = 4UL,
      FI_EN_E8COMPASS_OCTANT_HEADING_SOUTHWEST = 5UL,
      FI_EN_E8COMPASS_OCTANT_HEADING_WEST = 6UL,
      FI_EN_E8COMPASS_OCTANT_HEADING_NORTHWEST = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWCompassOctantHeading(most_fi_tcl_e8_GMLnGWCompassOctantHeading::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWCompassOctantHeading();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWCompassOctantHeading& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWFunctionalRoadClass : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWFunctionalRoadClass();
   enum tenType {
      FI_EN_E8FUNCTIONAL_ROAD_CLASS_LOCAL = 0UL,
      FI_EN_E8FUNCTIONAL_ROAD_CLASS_SFREEWAY = 1UL,
      FI_EN_E8FUNCTIONAL_ROAD_CLASS_FREEWAY = 2UL,
      FI_EN_E8FUNCTIONAL_ROAD_CLASS_PED_MALL = 3UL,
      FI_EN_E8FUNCTIONAL_ROAD_CLASS_HIGHWAY = 4UL,
      FI_EN_E8FUNCTIONAL_ROAD_CLASS_ARTERIAL = 5UL,
      FI_EN_E8FUNCTIONAL_ROAD_CLASS_ALLEY = 6UL,
      FI_EN_E8FUNCTIONAL_ROAD_CLASS_COLLECTOR = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWFunctionalRoadClass(most_fi_tcl_e8_GMLnGWFunctionalRoadClass::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWFunctionalRoadClass();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWFunctionalRoadClass& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWInternationalStdAlphaCountryCodeChar1 : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWInternationalStdAlphaCountryCodeChar1();
   enum tenType {
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_UNKNOWN = 0UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_ = 1UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_B = 2UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_C = 3UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_D = 4UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_E = 5UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_F = 6UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_G = 7UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_H = 8UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_I = 9UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_J = 10UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_K = 11UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_L = 12UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_M = 13UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_N = 14UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_O = 15UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_P = 16UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_Q = 17UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_R = 18UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_S = 19UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_T = 20UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_U = 21UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_V = 22UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_W = 23UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_X = 24UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_Y = 25UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_Z = 26UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWInternationalStdAlphaCountryCodeChar1(most_fi_tcl_e8_GMLnGWInternationalStdAlphaCountryCodeChar1::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWInternationalStdAlphaCountryCodeChar1();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWInternationalStdAlphaCountryCodeChar1& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWInternationalStdAlphaCountryCodeChar2 : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWInternationalStdAlphaCountryCodeChar2();
   enum tenType {
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_UNKNOWN = 0UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_ = 1UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_B = 2UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_C = 3UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_D = 4UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_E = 5UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_F = 6UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_G = 7UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_H = 8UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_I = 9UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_J = 10UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_K = 11UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_L = 12UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_M = 13UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_N = 14UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_O = 15UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_P = 16UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_Q = 17UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_R = 18UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_S = 19UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_T = 20UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_U = 21UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_V = 22UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_W = 23UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_X = 24UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_Y = 25UL,
      FI_EN_E8INT_STD_COUNTRY_CODE_CHAR_Z = 26UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWInternationalStdAlphaCountryCodeChar2(most_fi_tcl_e8_GMLnGWInternationalStdAlphaCountryCodeChar2::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWInternationalStdAlphaCountryCodeChar2();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWInternationalStdAlphaCountryCodeChar2& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLaneCategory : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLaneCategory();
   enum tenType {
      FI_EN_E8LANE_CATEGORY_NOINFO = 0UL,
      FI_EN_E8LANE_CATEGORY_ONE_LANE = 1UL,
      FI_EN_E8LANE_CATEGORY_TWO_OR_THREE_LANES = 2UL,
      FI_EN_E8LANE_CATEGORY_FOUR_OR_MORE_LANES = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLaneCategory(most_fi_tcl_e8_GMLnGWLaneCategory::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLaneCategory();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLaneCategory& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRoadSpeedCategoryType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRoadSpeedCategoryType();
   enum tenType {
      FI_EN_E8ROAD_SPEED_CATEGORY_GT_130_KMH = 0UL,
      FI_EN_E8ROAD_SPEED_CATEGORY_81_TO_130_KMH = 1UL,
      FI_EN_E8ROAD_SPEED_CATEGORY_51_TO_80_KMH = 2UL,
      FI_EN_E8ROAD_SPEED_CATEGORY_0_TO_50_KMH = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRoadSpeedCategoryType(most_fi_tcl_e8_GMLnGWRoadSpeedCategoryType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRoadSpeedCategoryType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRoadSpeedCategoryType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDTCFaultType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDTCFaultType();
   enum tenType {
      FI_EN_E8NOT_SUPPORTED = 0UL,
      FI_EN_E8TYPE_A = 1UL,
      FI_EN_E8TYPE_B = 2UL,
      FI_EN_E8TYPE_C = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDTCFaultType(most_fi_tcl_e8_GMLnGWDTCFaultType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDTCFaultType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDTCFaultType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWAutomaticBeamSelectStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWAutomaticBeamSelectStatus();
   enum tenType {
      FI_EN_E8DISABLED = 0UL,
      FI_EN_E8LOW_BEAM = 1UL,
      FI_EN_E8HIGH_BEAM = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWAutomaticBeamSelectStatus(most_fi_tcl_e8_GMLnGWAutomaticBeamSelectStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWAutomaticBeamSelectStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWAutomaticBeamSelectStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTractionControlSystemActive : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTractionControlSystemActive();
   enum tenType {
      FI_EN_E8TRACTION_CONTROL_SYSTEM_NOT_ACTIVE = 0UL,
      FI_EN_E8TRACTION_CONTROL_SYSTEM_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTractionControlSystemActive(most_fi_tcl_e8_GMLnGWTractionControlSystemActive::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTractionControlSystemActive();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTractionControlSystemActive& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTractionControlSystemOperatingMode : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTractionControlSystemOperatingMode();
   enum tenType {
      FI_EN_E8TRACTION_CONTROL_SYSTEM_OPERATING_MODE_OFF = 0UL,
      FI_EN_E8TRACTION_CONTROL_SYSTEM_OPERATING_MODE_NORMAL = 1UL,
      FI_EN_E8TRACTION_CONTROL_SYSTEM_OPERATING_MODE_OFFROAD = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTractionControlSystemOperatingMode(most_fi_tcl_e8_GMLnGWTractionControlSystemOperatingMode::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTractionControlSystemOperatingMode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTractionControlSystemOperatingMode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTractionControlSystemOperatingStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTractionControlSystemOperatingStatus();
   enum tenType {
      FI_EN_E8TRACTION_CONTROL_SYSTEM_OPERATING_STATUS_INACTIVE = 0UL,
      FI_EN_E8TRACTION_CONTROL_SYSTEM_OPERATING_STATUS_ACTIVE = 1UL,
      FI_EN_E8TRACTION_CONTROL_SYSTEM_OPERATING_STATUS_FAULT = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTractionControlSystemOperatingStatus(most_fi_tcl_e8_GMLnGWTractionControlSystemOperatingStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTractionControlSystemOperatingStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTractionControlSystemOperatingStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTransmissionShiftModeStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTransmissionShiftModeStatus();
   enum tenType {
      FI_EN_E8TRANSMISSION_SHIFT_MODE_STATUS_NO_OVERRIDE_MODE = 0UL,
      FI_EN_E8TRANSMISSION_SHIFT_MODE_STATUS_PERFORMANCE_MODE = 1UL,
      FI_EN_E8TRANSMISSION_SHIFT_MODE_STATUS_LIFT_FOOT_CORNERING_MODE = 2UL,
      FI_EN_E8TRANSMISSION_SHIFT_MODE_STATUS_AUTO_GRADE_BRAKING_MODE = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTransmissionShiftModeStatus(most_fi_tcl_e8_GMLnGWTransmissionShiftModeStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTransmissionShiftModeStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTransmissionShiftModeStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWCruiseAndSpeedLimiterSwitchStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWCruiseAndSpeedLimiterSwitchStatus();
   enum tenType {
      FI_EN_E8CRUISE_SPEED_LIMITER_SWITCH_STATUS_INDETERMINATE = 0UL,
      FI_EN_E8CRUISE_SPEED_LIMITER_SWITCH_STATUS_NO_ACTIVATION = 1UL,
      FI_EN_E8CRUISE_SPEED_LIMITER_SWITCH_STATUS_RESUME = 2UL,
      FI_EN_E8CRUISE_SPEED_LIMITER_SWITCH_STATUS_SET = 3UL,
      FI_EN_E8CRUISE_SPEED_LIMITER_SWITCH_STATUS_SLSW_ONOFF_GAP = 4UL,
      FI_EN_E8CRUISE_SPEED_LIMITER_SWITCH_STATUS_CRUISE_ONOFF = 5UL,
      FI_EN_E8CRUISE_SPEED_LIMITER_SWITCH_STATUS_CANCEL = 6UL,
      FI_EN_E8CRUISE_SPEED_LIMITER_SWITCH_STATUS_OPEN_SHORT_GND = 7UL,
      FI_EN_E8CRUISE_SPEED_LIMITER_SWITCH_STATUS_SHORT_POWER = 8UL,
      FI_EN_E8CRUISE_SPEED_LIMITER_SWITCH_STATUS_BETWEEN_RANGES = 9UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWCruiseAndSpeedLimiterSwitchStatus(most_fi_tcl_e8_GMLnGWCruiseAndSpeedLimiterSwitchStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWCruiseAndSpeedLimiterSwitchStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWCruiseAndSpeedLimiterSwitchStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWCruiseControlActive : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWCruiseControlActive();
   enum tenType {
      FI_EN_E8CRUISE_CONTROL_INACTIVE = 0UL,
      FI_EN_E8CRUISE_CONTROL_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWCruiseControlActive(most_fi_tcl_e8_GMLnGWCruiseControlActive::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWCruiseControlActive();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWCruiseControlActive& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWCruiseControlEnabled : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWCruiseControlEnabled();
   enum tenType {
      FI_EN_E8CRUISE_CONTROL_DISABLED = 0UL,
      FI_EN_E8CRUISE_CONTROL_ENABLED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWCruiseControlEnabled(most_fi_tcl_e8_GMLnGWCruiseControlEnabled::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWCruiseControlEnabled();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWCruiseControlEnabled& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWAutomaticTransmissionCommandedGear : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWAutomaticTransmissionCommandedGear();
   enum tenType {
      FI_EN_E8AUTO_TRANS_CMD_GEAR_NOT_SUPPORTED = 0UL,
      FI_EN_E8AUTO_TRANS_CMD_GEAR_FIRST_GEAR = 1UL,
      FI_EN_E8AUTO_TRANS_CMD_GEAR_SECOND_GEAR = 2UL,
      FI_EN_E8AUTO_TRANS_CMD_GEAR_THIRD_GEAR = 3UL,
      FI_EN_E8AUTO_TRANS_CMD_GEAR_FOURTH_GEAR = 4UL,
      FI_EN_E8AUTO_TRANS_CMD_GEAR_FIFTH_GEAR = 5UL,
      FI_EN_E8AUTO_TRANS_CMD_GEAR_SIXTH_GEAR = 6UL,
      FI_EN_E8AUTO_TRANS_CMD_GEAR_SEVENTH_GEAR = 7UL,
      FI_EN_E8AUTO_TRANS_CMD_GEAR_EIGHTH_GEAR = 8UL,
      FI_EN_E8AUTO_TRANS_CMD_GEAR_EVTMODE1 = 10UL,
      FI_EN_E8AUTO_TRANS_CMD_GEAR_EVTMODE2 = 11UL,
      FI_EN_E8AUTO_TRANS_CMD_GEAR_CVT_FORWARD_GEAR = 12UL,
      FI_EN_E8AUTO_TRANS_CMD_GEAR_NEUTRAL_GEAR = 13UL,
      FI_EN_E8AUTO_TRANS_CMD_GEAR_REVERSE_GEAR = 14UL,
      FI_EN_E8AUTO_TRANS_CMD_GEAR_PARK_GEAR = 15UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWAutomaticTransmissionCommandedGear(most_fi_tcl_e8_GMLnGWAutomaticTransmissionCommandedGear::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWAutomaticTransmissionCommandedGear();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWAutomaticTransmissionCommandedGear& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWVehicleTopSpeedLimitModeActive : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWVehicleTopSpeedLimitModeActive();
   enum tenType {
      FI_EN_E8VEHICLE_TOP_SPEED_LIMIT_MODE_INACTIVE = 0UL,
      FI_EN_E8VEHICLE_TOP_SPEED_LIMIT_MODE_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWVehicleTopSpeedLimitModeActive(most_fi_tcl_e8_GMLnGWVehicleTopSpeedLimitModeActive::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWVehicleTopSpeedLimitModeActive();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWVehicleTopSpeedLimitModeActive& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRemoteVehicleStartEngineRunning : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRemoteVehicleStartEngineRunning();
   enum tenType {
      FI_EN_E8REMOTE_VEHICLE_START_ENGINE_RUNNING_INACTIVE = 0UL,
      FI_EN_E8REMOTE_VEHICLE_START_ENGINE_RUNNING_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRemoteVehicleStartEngineRunning(most_fi_tcl_e8_GMLnGWRemoteVehicleStartEngineRunning::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRemoteVehicleStartEngineRunning();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRemoteVehicleStartEngineRunning& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWAirConditioningCompressorCommand : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWAirConditioningCompressorCommand();
   enum tenType {
      FI_EN_E8AIR_COND_COMPRESSOR_CMD_OFF = 0UL,
      FI_EN_E8AIR_COND_COMPRESSOR_CMD_ON = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWAirConditioningCompressorCommand(most_fi_tcl_e8_GMLnGWAirConditioningCompressorCommand::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWAirConditioningCompressorCommand();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWAirConditioningCompressorCommand& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWFuelModeStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWFuelModeStatus();
   enum tenType {
      FI_EN_E8FUEL_MODE_STATUS_GASOLINE_MODE = 0UL,
      FI_EN_E8FUEL_MODE_STATUS_NATURAL_GAS_MODE = 1UL,
      FI_EN_E8FUEL_MODE_STATUS_LIQUIFIED_PETROLEUM_GAS_MODE = 2UL,
      FI_EN_E8FUEL_MODE_STATUS_ALTERNATIVE_FUEL_CUTOFF_ACTIVE = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWFuelModeStatus(most_fi_tcl_e8_GMLnGWFuelModeStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWFuelModeStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWFuelModeStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWAdaptiveCruiseControlActive : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWAdaptiveCruiseControlActive();
   enum tenType {
      FI_EN_E8ADAPTIVE_CRUISE_CONTROL_NOT_ACTIVE = 0UL,
      FI_EN_E8ADAPTIVE_CRUISE_CONTROL_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWAdaptiveCruiseControlActive(most_fi_tcl_e8_GMLnGWAdaptiveCruiseControlActive::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWAdaptiveCruiseControlActive();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWAdaptiveCruiseControlActive& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWCruiseControlDriverSelectedSpeedActive : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWCruiseControlDriverSelectedSpeedActive();
   enum tenType {
      FI_EN_E8CRUISE_CONTROL_DRIVER_SELECTED_SPEED_NOT_ACTIVE = 0UL,
      FI_EN_E8CRUISE_CONTROL_DRIVER_SELECTED_SPEED_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWCruiseControlDriverSelectedSpeedActive(most_fi_tcl_e8_GMLnGWCruiseControlDriverSelectedSpeedActive::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWCruiseControlDriverSelectedSpeedActive();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWCruiseControlDriverSelectedSpeedActive& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWAntilockBrakeSystemActive : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWAntilockBrakeSystemActive();
   enum tenType {
      FI_EN_E8ANTILOCK_BRAKE_SYSTEM_NOT_ACTIVE = 0UL,
      FI_EN_E8ANTILOCK_BRAKE_SYSTEM_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWAntilockBrakeSystemActive(most_fi_tcl_e8_GMLnGWAntilockBrakeSystemActive::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWAntilockBrakeSystemActive();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWAntilockBrakeSystemActive& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverWindowPositionStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverWindowPositionStatus();
   enum tenType {
      FI_EN_E8WINDOW_POSITION_STATUS_FULLY_CLOSED = 0UL,
      FI_EN_E8WINDOW_POSITION_STATUS_OPEN_LESS_THAN_A = 1UL,
      FI_EN_E8WINDOW_POSITION_STATUS_POSITION_A = 2UL,
      FI_EN_E8WINDOW_POSITION_STATUS_POSITION_B = 3UL,
      FI_EN_E8WINDOW_POSITION_STATUS_POSITION_C = 4UL,
      FI_EN_E8WINDOW_POSITION_STATUS_OPEN_MORE_THAN_C = 5UL,
      FI_EN_E8WINDOW_POSITION_STATUS_FULLY_OPENED = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverWindowPositionStatus(most_fi_tcl_e8_GMLnGWDriverWindowPositionStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverWindowPositionStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverWindowPositionStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftRearWindowPositionStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftRearWindowPositionStatus();
   enum tenType {
      FI_EN_E8WINDOW_POSITION_STATUS_FULLY_CLOSED = 0UL,
      FI_EN_E8WINDOW_POSITION_STATUS_OPEN_LESS_THAN_A = 1UL,
      FI_EN_E8WINDOW_POSITION_STATUS_POSITION_A = 2UL,
      FI_EN_E8WINDOW_POSITION_STATUS_POSITION_B = 3UL,
      FI_EN_E8WINDOW_POSITION_STATUS_POSITION_C = 4UL,
      FI_EN_E8WINDOW_POSITION_STATUS_OPEN_MORE_THAN_C = 5UL,
      FI_EN_E8WINDOW_POSITION_STATUS_FULLY_OPENED = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftRearWindowPositionStatus(most_fi_tcl_e8_GMLnGWLeftRearWindowPositionStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftRearWindowPositionStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftRearWindowPositionStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPassengerWindowPositionStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPassengerWindowPositionStatus();
   enum tenType {
      FI_EN_E8WINDOW_POSITION_STATUS_FULLY_CLOSED = 0UL,
      FI_EN_E8WINDOW_POSITION_STATUS_OPEN_LESS_THAN_A = 1UL,
      FI_EN_E8WINDOW_POSITION_STATUS_POSITION_A = 2UL,
      FI_EN_E8WINDOW_POSITION_STATUS_POSITION_B = 3UL,
      FI_EN_E8WINDOW_POSITION_STATUS_POSITION_C = 4UL,
      FI_EN_E8WINDOW_POSITION_STATUS_OPEN_MORE_THAN_C = 5UL,
      FI_EN_E8WINDOW_POSITION_STATUS_FULLY_OPENED = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPassengerWindowPositionStatus(most_fi_tcl_e8_GMLnGWPassengerWindowPositionStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPassengerWindowPositionStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPassengerWindowPositionStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightRearWindowPositionStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightRearWindowPositionStatus();
   enum tenType {
      FI_EN_E8WINDOW_POSITION_STATUS_FULLY_CLOSED = 0UL,
      FI_EN_E8WINDOW_POSITION_STATUS_OPEN_LESS_THAN_A = 1UL,
      FI_EN_E8WINDOW_POSITION_STATUS_POSITION_A = 2UL,
      FI_EN_E8WINDOW_POSITION_STATUS_POSITION_B = 3UL,
      FI_EN_E8WINDOW_POSITION_STATUS_POSITION_C = 4UL,
      FI_EN_E8WINDOW_POSITION_STATUS_OPEN_MORE_THAN_C = 5UL,
      FI_EN_E8WINDOW_POSITION_STATUS_FULLY_OPENED = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightRearWindowPositionStatus(most_fi_tcl_e8_GMLnGWRightRearWindowPositionStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightRearWindowPositionStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightRearWindowPositionStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWBrakeLightsActive : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWBrakeLightsActive();
   enum tenType {
      FI_EN_E8BRAKE_LIGHTS_INACTIVE = 0UL,
      FI_EN_E8BRAKE_LIGHTS_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWBrakeLightsActive(most_fi_tcl_e8_GMLnGWBrakeLightsActive::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWBrakeLightsActive();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWBrakeLightsActive& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDaytimeRunningLampsActive : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDaytimeRunningLampsActive();
   enum tenType {
      FI_EN_E8DRL_INACTIVE = 0UL,
      FI_EN_E8DRL_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDaytimeRunningLampsActive(most_fi_tcl_e8_GMLnGWDaytimeRunningLampsActive::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDaytimeRunningLampsActive();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDaytimeRunningLampsActive& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWHazardSwitchActive : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWHazardSwitchActive();
   enum tenType {
      FI_EN_E8HAZARD_SWITCH_INACTIVE = 0UL,
      FI_EN_E8HAZARD_SWITCH_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWHazardSwitchActive(most_fi_tcl_e8_GMLnGWHazardSwitchActive::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWHazardSwitchActive();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWHazardSwitchActive& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWHeadlampBeamSelectStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWHeadlampBeamSelectStatus();
   enum tenType {
      FI_EN_E8HEADLAMP_BEAM_SELECT_UNKNOWN = 0UL,
      FI_EN_E8HEADLAMP_BEAM_SELECT_LOW_BEAMS = 1UL,
      FI_EN_E8HEADLAMP_BEAM_SELECT_HIGH_BEAMS = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWHeadlampBeamSelectStatus(most_fi_tcl_e8_GMLnGWHeadlampBeamSelectStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWHeadlampBeamSelectStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWHeadlampBeamSelectStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWMainLightSwitch : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWMainLightSwitch();
   enum tenType {
      FI_EN_E8MAIN_LIGHT_SWITCH_AUTO = 0UL,
      FI_EN_E8MAIN_LIGHT_SWITCH_OFF = 1UL,
      FI_EN_E8MAIN_LIGHT_SWITCH_PARKLAMP = 2UL,
      FI_EN_E8MAIN_LIGHT_SWITCH_HEADLAMP = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWMainLightSwitch(most_fi_tcl_e8_GMLnGWMainLightSwitch::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWMainLightSwitch();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWMainLightSwitch& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTurnSwitchActivation : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTurnSwitchActivation();
   enum tenType {
      FI_EN_E8TURN_SWITCH_ACTIVATION_NO_ACTIVATION = 0UL,
      FI_EN_E8TURN_SWITCH_ACTIVATION_LEFT = 1UL,
      FI_EN_E8TURN_SWITCH_ACTIVATION_RIGHT = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTurnSwitchActivation(most_fi_tcl_e8_GMLnGWTurnSwitchActivation::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTurnSwitchActivation();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTurnSwitchActivation& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPowerLiftgateMotionStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPowerLiftgateMotionStatus();
   enum tenType {
      FI_EN_E8POWER_LIFTGATE_MOTION_STATUS_UNKNOWN = 0UL,
      FI_EN_E8POWER_LIFTGATE_MOTION_STATUS_FULL_OPEN = 1UL,
      FI_EN_E8POWER_LIFTGATE_MOTION_STATUS_CLOSED = 2UL,
      FI_EN_E8POWER_LIFTGATE_MOTION_STATUS_OPENING = 3UL,
      FI_EN_E8POWER_LIFTGATE_MOTION_STATUS_CLOSING = 4UL,
      FI_EN_E8POWER_LIFTGATE_MOTION_STATUS_STOPPED = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPowerLiftgateMotionStatus(most_fi_tcl_e8_GMLnGWPowerLiftgateMotionStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPowerLiftgateMotionStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPowerLiftgateMotionStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRearClosureObstacleDetected : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRearClosureObstacleDetected();
   enum tenType {
      FI_EN_E8REAR_CLOSURE_OBSTABLE_NOT_DETECTED = 0UL,
      FI_EN_E8REAR_CLOSURE_OBSTABLE_DETECTED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRearClosureObstacleDetected(most_fi_tcl_e8_GMLnGWRearClosureObstacleDetected::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRearClosureObstacleDetected();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRearClosureObstacleDetected& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftSecondaryTurnIndicatorFailed : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftSecondaryTurnIndicatorFailed();
   enum tenType {
      FI_EN_E8LEFT_SECONDARY_TURN_INDICATOR_NOT_FAILED = 0UL,
      FI_EN_E8LEFT_SECONDARY_TURN_INDICATOR_FAILED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftSecondaryTurnIndicatorFailed(most_fi_tcl_e8_GMLnGWLeftSecondaryTurnIndicatorFailed::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftSecondaryTurnIndicatorFailed();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftSecondaryTurnIndicatorFailed& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRearLeftParkLampFailed : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRearLeftParkLampFailed();
   enum tenType {
      FI_EN_E8REAR_LEFT_PARK_LAMP_NOT_FAILED = 0UL,
      FI_EN_E8REAR_LEFT_PARK_LAMP_FAILED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRearLeftParkLampFailed(most_fi_tcl_e8_GMLnGWRearLeftParkLampFailed::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRearLeftParkLampFailed();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRearLeftParkLampFailed& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRearRightParkLampFailed : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRearRightParkLampFailed();
   enum tenType {
      FI_EN_E8REAR_RIGHT_PARK_LAMP_NOT_FAILED = 0UL,
      FI_EN_E8REAR_RIGHT_PARK_LAMP_FAILED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRearRightParkLampFailed(most_fi_tcl_e8_GMLnGWRearRightParkLampFailed::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRearRightParkLampFailed();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRearRightParkLampFailed& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightSecondaryTurnIndicatorFailed : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightSecondaryTurnIndicatorFailed();
   enum tenType {
      FI_EN_E8RIGHT_SECONDARY_TURN_INDICATOR_NOT_FAILED = 0UL,
      FI_EN_E8RIGHT_SECONDARY_TURN_INDICATOR_FAILED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightSecondaryTurnIndicatorFailed(most_fi_tcl_e8_GMLnGWRightSecondaryTurnIndicatorFailed::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightSecondaryTurnIndicatorFailed();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightSecondaryTurnIndicatorFailed& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTrailerBrakeLightFailed : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTrailerBrakeLightFailed();
   enum tenType {
      FI_EN_E8TRAILER_BRAKE_LIGHT_NOT_FAILED = 0UL,
      FI_EN_E8TRAILER_BRAKE_LIGHT_FAILED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTrailerBrakeLightFailed(most_fi_tcl_e8_GMLnGWTrailerBrakeLightFailed::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTrailerBrakeLightFailed();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTrailerBrakeLightFailed& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTrailerFogLightFailed : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTrailerFogLightFailed();
   enum tenType {
      FI_EN_E8TRAILER_FOG_LIGHT_NOT_FAILED = 0UL,
      FI_EN_E8TRAILER_FOG_LIGHT_FAILED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTrailerFogLightFailed(most_fi_tcl_e8_GMLnGWTrailerFogLightFailed::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTrailerFogLightFailed();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTrailerFogLightFailed& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTrailerFogLightPresent : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTrailerFogLightPresent();
   enum tenType {
      FI_EN_E8TRAILER_FOG_LIGHT_NOT_PRESENT = 0UL,
      FI_EN_E8TRAILER_FOG_LIGHT_PRESENT = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTrailerFogLightPresent(most_fi_tcl_e8_GMLnGWTrailerFogLightPresent::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTrailerFogLightPresent();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTrailerFogLightPresent& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTrailerHitchSwitchActive : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTrailerHitchSwitchActive();
   enum tenType {
      FI_EN_E8TRAILER_HITCH_SWITCH_NOT_ACTIVE = 0UL,
      FI_EN_E8TRAILER_HITCH_SWITCH_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTrailerHitchSwitchActive(most_fi_tcl_e8_GMLnGWTrailerHitchSwitchActive::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTrailerHitchSwitchActive();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTrailerHitchSwitchActive& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTrailerTailLightFailed : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTrailerTailLightFailed();
   enum tenType {
      FI_EN_E8TRAILER_TAIL_LIGHT_NOT_FAILED = 0UL,
      FI_EN_E8TRAILER_TAIL_LIGHT_FAILED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTrailerTailLightFailed(most_fi_tcl_e8_GMLnGWTrailerTailLightFailed::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTrailerTailLightFailed();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTrailerTailLightFailed& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWSpareTireStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWSpareTireStatus();
   enum tenType {
      FI_EN_E8SPARE_TIRE_STATUS_UNDETERMINED = 0UL,
      FI_EN_E8SPARE_TIRE_STATUS_NO_SPARE_DETECTED = 1UL,
      FI_EN_E8SPARE_TIRE_STATUS_SPARE_DETECTED = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWSpareTireStatus(most_fi_tcl_e8_GMLnGWSpareTireStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWSpareTireStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWSpareTireStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTireInflationMonitoringSystemResetPerformed : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTireInflationMonitoringSystemResetPerformed();
   enum tenType {
      FI_EN_E8TIRE_INFLATION_MONITORING_NO_RESET_PERFORMED = 0UL,
      FI_EN_E8TIRE_INFLATION_MONITORING_RESET_PERFORMED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTireInflationMonitoringSystemResetPerformed(most_fi_tcl_e8_GMLnGWTireInflationMonitoringSystemResetPerformed::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTireInflationMonitoringSystemResetPerformed();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTireInflationMonitoringSystemResetPerformed& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTirePressureMonitoringSystemFailed : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTirePressureMonitoringSystemFailed();
   enum tenType {
      FI_EN_E8TIRE_PRESSURE_SYSTEM_MONITORING_NOT_FAILED = 0UL,
      FI_EN_E8TIRE_PRESSURE_SYSTEM_MONITORING_FAILED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTirePressureMonitoringSystemFailed(most_fi_tcl_e8_GMLnGWTirePressureMonitoringSystemFailed::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTirePressureMonitoringSystemFailed();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTirePressureMonitoringSystemFailed& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWParkingAssistanceFunctionFailed : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWParkingAssistanceFunctionFailed();
   enum tenType {
      FI_EN_E8PARKING_ASSISTANCE_NOT_FAILED = 0UL,
      FI_EN_E8PARKING_ASSISTANCE_FAILED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWParkingAssistanceFunctionFailed(most_fi_tcl_e8_GMLnGWParkingAssistanceFunctionFailed::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWParkingAssistanceFunctionFailed();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWParkingAssistanceFunctionFailed& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWWasherFluidLow : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWWasherFluidLow();
   enum tenType {
      FI_EN_E8WASHER_FLUID_NOT_LOW = 0UL,
      FI_EN_E8WASHER_FLUID_LOW = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWWasherFluidLow(most_fi_tcl_e8_GMLnGWWasherFluidLow::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWWasherFluidLow();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWWasherFluidLow& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWWindshieldWiperActive : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWWindshieldWiperActive();
   enum tenType {
      FI_EN_E8WINDSHIELD_WIPER_NOT_ACTIVE = 0UL,
      FI_EN_E8WINDSHIELD_WIPER_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWWindshieldWiperActive(most_fi_tcl_e8_GMLnGWWindshieldWiperActive::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWWindshieldWiperActive();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWWindshieldWiperActive& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWHoodStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWHoodStatus();
   enum tenType {
      FI_EN_E8HOOD_STATUS_CLOSED = 0UL,
      FI_EN_E8HOOD_STATUS_AJAR = 1UL,
      FI_EN_E8HOOD_STATUS_OPEN = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWHoodStatus(most_fi_tcl_e8_GMLnGWHoodStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWHoodStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWHoodStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWWirelessChargingSystemChargingStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWWirelessChargingSystemChargingStatus();
   enum tenType {
      FI_EN_E8WIRELESS_CHARGING_SYSTEM_STATUS_NOT_PRESENT = 0UL,
      FI_EN_E8WIRELESS_CHARGING_SYSTEM_STATUS_CHARGING_INACTIVE = 1UL,
      FI_EN_E8WIRELESS_CHARGING_SYSTEM_STATUS_CHARGING_ACTIVE = 2UL,
      FI_EN_E8WIRELESS_CHARGING_SYSTEM_STATUS_CHARGING_COMPLETE = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWWirelessChargingSystemChargingStatus(most_fi_tcl_e8_GMLnGWWirelessChargingSystemChargingStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWWirelessChargingSystemChargingStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWWirelessChargingSystemChargingStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDigitalMapConditionalSpeedLimit : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDigitalMapConditionalSpeedLimit();
   enum tenType {
      FI_EN_E8DIG_MAP_COND_SPEED_UNKNOWN = 0UL,
      FI_EN_E8DIG_MAP_COND_SPEED_1 = 1UL,
      FI_EN_E8DIG_MAP_COND_SPEED_2 = 2UL,
      FI_EN_E8DIG_MAP_COND_SPEED_3 = 3UL,
      FI_EN_E8DIG_MAP_COND_SPEED_4 = 4UL,
      FI_EN_E8DIG_MAP_COND_SPEED_5 = 5UL,
      FI_EN_E8DIG_MAP_COND_SPEED_6 = 6UL,
      FI_EN_E8DIG_MAP_COND_SPEED_7 = 7UL,
      FI_EN_E8DIG_MAP_COND_SPEED_8 = 8UL,
      FI_EN_E8DIG_MAP_COND_SPEED_9 = 9UL,
      FI_EN_E8DIG_MAP_COND_SPEED_A = 10UL,
      FI_EN_E8DIG_MAP_COND_SPEED_B = 11UL,
      FI_EN_E8DIG_MAP_COND_SPEED_C = 12UL,
      FI_EN_E8DIG_MAP_COND_SPEED_D = 13UL,
      FI_EN_E8DIG_MAP_COND_SPEED_E = 14UL,
      FI_EN_E8DIG_MAP_COND_SPEED_F = 15UL,
      FI_EN_E8DIG_MAP_COND_SPEED_10 = 16UL,
      FI_EN_E8DIG_MAP_COND_SPEED_11 = 17UL,
      FI_EN_E8DIG_MAP_COND_SPEED_12 = 18UL,
      FI_EN_E8DIG_MAP_COND_SPEED_13 = 19UL,
      FI_EN_E8DIG_MAP_COND_SPEED_14 = 20UL,
      FI_EN_E8DIG_MAP_COND_SPEED_15 = 21UL,
      FI_EN_E8DIG_MAP_COND_SPEED_16 = 22UL,
      FI_EN_E8DIG_MAP_COND_SPEED_17 = 23UL,
      FI_EN_E8DIG_MAP_COND_SPEED_18 = 24UL,
      FI_EN_E8DIG_MAP_COND_SPEED_19 = 25UL,
      FI_EN_E8DIG_MAP_COND_SPEED_1A = 26UL,
      FI_EN_E8DIG_MAP_COND_SPEED_1B = 27UL,
      FI_EN_E8DIG_MAP_COND_SPEED_1C = 28UL,
      FI_EN_E8DIG_MAP_COND_SPEED_1D = 29UL,
      FI_EN_E8DIG_MAP_COND_SPEED_1E = 30UL,
      FI_EN_E8DIG_MAP_COND_SPEED_NOT_AVAILABLE = 31UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDigitalMapConditionalSpeedLimit(most_fi_tcl_e8_GMLnGWDigitalMapConditionalSpeedLimit::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDigitalMapConditionalSpeedLimit();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDigitalMapConditionalSpeedLimit& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDigitalMapConditionalSpeedLimitType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDigitalMapConditionalSpeedLimitType();
   enum tenType {
      FI_EN_E8DIG_MAP_COND_SPD_LIM_LANE_DEPENDENT = 0UL,
      FI_EN_E8DIG_MAP_SPD_LIM_ONLY_BY_FOG = 1UL,
      FI_EN_E8DIG_MAP_SPD_LIM_ONLY_BY_NIGHT = 2UL,
      FI_EN_E8DIG_MAP_SPD_LIM_ONLY_BY_DAY = 3UL,
      FI_EN_E8DIG_MAP_SPD_LIM_BY_TIME_OF_DAY = 4UL,
      FI_EN_E8DIG_MAP_SPD_LIM_ONLY_BY_RAIN = 5UL,
      FI_EN_E8DIG_MAP_SPD_LIM_ONLY_BY_SNOW = 6UL,
      FI_EN_E8DIG_MAP_SPD_LIM_UNKNOWN = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDigitalMapConditionalSpeedLimitType(most_fi_tcl_e8_GMLnGWDigitalMapConditionalSpeedLimitType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDigitalMapConditionalSpeedLimitType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDigitalMapConditionalSpeedLimitType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDigitalMapSpeedCategory : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDigitalMapSpeedCategory();
   enum tenType {
      FI_EN_E8DIG_MAP_SPD_CAT_0 = 0UL,
      FI_EN_E8DIG_MAP_SPD_CAT_1 = 1UL,
      FI_EN_E8DIG_MAP_SPD_CAT_2 = 2UL,
      FI_EN_E8DIG_MAP_SPD_CAT_3 = 3UL,
      FI_EN_E8DIG_MAP_SPD_CAT_4 = 4UL,
      FI_EN_E8DIG_MAP_SPD_CAT_5 = 5UL,
      FI_EN_E8DIG_MAP_SPD_CAT_6 = 6UL,
      FI_EN_E8DIG_MAP_SPD_CAT_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDigitalMapSpeedCategory(most_fi_tcl_e8_GMLnGWDigitalMapSpeedCategory::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDigitalMapSpeedCategory();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDigitalMapSpeedCategory& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDigitalMapVersionQuarter : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDigitalMapVersionQuarter();
   enum tenType {
      FI_EN_E8DIG_MAP_VER_QUARTER_1 = 0UL,
      FI_EN_E8DIG_MAP_VER_QUARTER_2 = 1UL,
      FI_EN_E8DIG_MAP_VER_QUARTER_3 = 2UL,
      FI_EN_E8DIG_MAP_VER_QUARTER_4 = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDigitalMapVersionQuarter(most_fi_tcl_e8_GMLnGWDigitalMapVersionQuarter::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDigitalMapVersionQuarter();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDigitalMapVersionQuarter& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDigitalMapDrivingSide : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDigitalMapDrivingSide();
   enum tenType {
      FI_EN_E8DIG_MAP_DRIV_SIDE_LEFT = 0UL,
      FI_EN_E8DIG_MAP_DRIV_SIDE_RIGHT = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDigitalMapDrivingSide(most_fi_tcl_e8_GMLnGWDigitalMapDrivingSide::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDigitalMapDrivingSide();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDigitalMapDrivingSide& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDigitalMapPassingRestriction : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDigitalMapPassingRestriction();
   enum tenType {
      FI_EN_E8NO_PASSING_RESTRICTION = 0UL,
      FI_EN_E8GENERAL_NO_OVERTAKE = 1UL,
      FI_EN_E8LED_SIGNS = 2UL,
      FI_EN_E8CONDTNL_PASSENGER_CAR_WITH_TRAILER = 3UL,
      FI_EN_E8CONDTNL_TRACTORS_ALLOWED_OVERTAKE = 4UL,
      FI_EN_E8CONDITIONAL_WHEN_WET = 5UL,
      FI_EN_E8CONDITIONAL_WHEN_SNOW = 6UL,
      FI_EN_E8UNKNOWN_CONDITION = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDigitalMapPassingRestriction(most_fi_tcl_e8_GMLnGWDigitalMapPassingRestriction::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDigitalMapPassingRestriction();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDigitalMapPassingRestriction& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWADASISLiteExtendedInfoStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWADASISLiteExtendedInfoStream();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWADASISLiteExtendedInfoStream& operator=(const most_fi_tcl_GMLnGWADASISLiteExtendedInfoStream& coRef);
   most_fi_tcl_GMLnGWADASISLiteExtendedInfoStream(const most_fi_tcl_GMLnGWADASISLiteExtendedInfoStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_GMLnGWDigitalMapConditionalSpeedLimit e8DigitalMapConditionalSpeedLimit;
   most_fi_tcl_e8_GMLnGWDigitalMapConditionalSpeedLimitType e8DigitalMapConditionalSpeedLimitType;
   most_fi_tcl_e8_GMLnGWDigitalMapSpeedCategory e8DigitalMapSpeedCategory;
   tU8 u8DigitalMapVersionYear;
   most_fi_tcl_e8_GMLnGWDigitalMapVersionQuarter e8DigitalMapVersionQuarter;
   most_fi_tcl_e8_GMLnGWDigitalMapDrivingSide e8DigitalMapDrivingSide;
   tBool bDigitalMapPostedSpeedLimitAssured;
   most_fi_tcl_e8_GMLnGWDigitalMapPassingRestriction e8DigitalMapPassingRestriction;
   virtual ~most_fi_tcl_GMLnGWADASISLiteExtendedInfoStream();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWADASISLiteExtendedInfoStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWElectricParkBrakeStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWElectricParkBrakeStatus();
   enum tenType {
      FI_EN_E8RELEASED = 0UL,
      FI_EN_E8APPLIED = 1UL,
      FI_EN_E8FAULT = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWElectricParkBrakeStatus(most_fi_tcl_e8_GMLnGWElectricParkBrakeStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWElectricParkBrakeStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWElectricParkBrakeStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWGMLANGPSConfiguration : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWGMLANGPSConfiguration();
   enum tenType {
      FI_EN_E8GMLAN_GPS_NONE = 0UL,
      FI_EN_E8GMLAN_GPS_ONSTAR = 1UL,
      FI_EN_E8GMLAN_GPS_NAVIGATION = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWGMLANGPSConfiguration(most_fi_tcl_e8_GMLnGWGMLANGPSConfiguration::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWGMLANGPSConfiguration();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWGMLANGPSConfiguration& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWCalendarMonth : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWCalendarMonth();
   enum tenType {
      FI_EN_E8MONTH_UNKNOWN = 0UL,
      FI_EN_E8MONTH_JANUARY = 1UL,
      FI_EN_E8MONTH_FEBRUARY = 2UL,
      FI_EN_E8MONTH_MARCH = 3UL,
      FI_EN_E8MONTH_APRIL = 4UL,
      FI_EN_E8MONTH_MAY = 5UL,
      FI_EN_E8MONTH_JUNE = 6UL,
      FI_EN_E8MONTH_JULY = 7UL,
      FI_EN_E8MONTH_AUGUST = 8UL,
      FI_EN_E8MONTH_SEPTEMBER = 9UL,
      FI_EN_E8MONTH_OCTOBER = 10UL,
      FI_EN_E8MONTH_NOVEMBER = 11UL,
      FI_EN_E8MONTH_DECEMBER = 12UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWCalendarMonth(most_fi_tcl_e8_GMLnGWCalendarMonth::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWCalendarMonth();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWCalendarMonth& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWInteriorDimmingAnimationType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWInteriorDimmingAnimationType();
   enum tenType {
      FI_EN_E8INTERIOR_DIMMING_ANIMATION_TYPE_UNKNOWN = 0UL,
      FI_EN_E8INTERIOR_DIMMING_ANIMATION_OFF = 1UL,
      FI_EN_E8INTERIOR_DIMMING_ANIMATION_TYPE_1 = 2UL,
      FI_EN_E8INTERIOR_DIMMING_ANIMATION_TYPE_2 = 3UL,
      FI_EN_E8INTERIOR_DIMMING_ANIMATION_TYPE_3 = 4UL,
      FI_EN_E8INTERIOR_DIMMING_ANIMATION_TYPE_4 = 5UL,
      FI_EN_E8INTERIOR_DIMMING_ANIMATION_TYPE_5 = 6UL,
      FI_EN_E8INTERIOR_DIMMING_ANIMATION_TYPE_6 = 7UL,
      FI_EN_E8INTERIOR_DIMMING_ANIMATION_TYPE_7 = 8UL,
      FI_EN_E8INTERIOR_DIMMING_ANIMATION_TYPE_8 = 9UL,
      FI_EN_E8INTERIOR_DIMMING_ANIMATION_TYPE_9 = 10UL,
      FI_EN_E8INTERIOR_DIMMING_ANIMATION_TYPE_10 = 11UL,
      FI_EN_E8INTERIOR_DIMMING_ANIMATION_TYPE_11 = 12UL,
      FI_EN_E8INTERIOR_DIMMING_ANIMATION_TYPE_12 = 13UL,
      FI_EN_E8INTERIOR_DIMMING_ANIMATION_TYPE_13 = 14UL,
      FI_EN_E8INTERIOR_DIMMING_ANIMATION_TYPE_14 = 15UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWInteriorDimmingAnimationType(most_fi_tcl_e8_GMLnGWInteriorDimmingAnimationType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWInteriorDimmingAnimationType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWInteriorDimmingAnimationType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWInteriorDimmingColorType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWInteriorDimmingColorType();
   enum tenType {
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_UNKNOWN = 0UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_OFF = 1UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_1 = 2UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_2 = 3UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_3 = 4UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_4 = 5UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_5 = 6UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_6 = 7UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_7 = 8UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_8 = 9UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_9 = 10UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_10 = 11UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_11 = 12UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_12 = 13UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_13 = 14UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_14 = 15UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_15 = 16UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_16 = 17UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_17 = 18UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_18 = 19UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_19 = 20UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_20 = 21UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_21 = 22UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_22 = 23UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_23 = 24UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_24 = 25UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_25 = 26UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_26 = 27UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_27 = 28UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_28 = 29UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_29 = 30UL,
      FI_EN_E8INTERIOR_DIMMING_COLOR_TYPE_30 = 31UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWInteriorDimmingColorType(most_fi_tcl_e8_GMLnGWInteriorDimmingColorType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWInteriorDimmingColorType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWInteriorDimmingColorType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWBatteryChargeCordAlert : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWBatteryChargeCordAlert();
   enum tenType {
      FI_EN_E8CHARGE_CORD_ALERT_NONE = 0UL,
      FI_EN_E8CHARGE_CORD_ALERT_PEDESTRIAN = 1UL,
      FI_EN_E8CHARGE_CORD_ALERT_THEFT = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWBatteryChargeCordAlert(most_fi_tcl_e8_GMLnGWBatteryChargeCordAlert::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWBatteryChargeCordAlert();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWBatteryChargeCordAlert& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWChargeDisplayStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWChargeDisplayStatus();
   enum tenType {
      FI_EN_E8CHARGE_DISPLAY_NOACTION = 0UL,
      FI_EN_E8CHARGE_DISPLAY_STATUS_PENDING = 1UL,
      FI_EN_E8CHARGE_DISPLAY_STATUS_ACTIVE_CHARGING = 2UL,
      FI_EN_E8CHARGE_DISPLAY_STATUS_COMPLETE = 3UL,
      FI_EN_E8CHARGE_DISPLAY_STATUS_FAULTED = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWChargeDisplayStatus(most_fi_tcl_e8_GMLnGWChargeDisplayStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWChargeDisplayStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWChargeDisplayStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWChargingPortDoorState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWChargingPortDoorState();
   enum tenType {
      FI_EN_E8CHARGE_PORT_DOOR_INACTIVE = 0UL,
      FI_EN_E8CHARGE_PORT_DOOR_OPEN_PARK = 1UL,
      FI_EN_E8CHARGE_PORT_DOOR_OPEN_MOVING = 2UL,
      FI_EN_E8CHARGE_PORT_DOOR_FAULT = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWChargingPortDoorState(most_fi_tcl_e8_GMLnGWChargingPortDoorState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWChargingPortDoorState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWChargingPortDoorState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLowPowerDayOfWeek : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLowPowerDayOfWeek();
   enum tenType {
      FI_EN_E8DAY_OF_WEEK_UNKNOWN = 0UL,
      FI_EN_E8DAY_OF_WEEK_SUNDAY = 1UL,
      FI_EN_E8DAY_OF_WEEK_MONDAY = 2UL,
      FI_EN_E8DAY_OF_WEEK_TUESDAY = 3UL,
      FI_EN_E8DAY_OF_WEEK_WEDNESDAY = 4UL,
      FI_EN_E8DAY_OF_WEEK_THURSDAY = 5UL,
      FI_EN_E8DAY_OF_WEEK_FRIDAY = 6UL,
      FI_EN_E8DAY_OF_WEEK_SATURDAY = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLowPowerDayOfWeek(most_fi_tcl_e8_GMLnGWLowPowerDayOfWeek::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLowPowerDayOfWeek();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLowPowerDayOfWeek& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWHighPowerDayOfWeek : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWHighPowerDayOfWeek();
   enum tenType {
      FI_EN_E8DAY_OF_WEEK_UNKNOWN = 0UL,
      FI_EN_E8DAY_OF_WEEK_SUNDAY = 1UL,
      FI_EN_E8DAY_OF_WEEK_MONDAY = 2UL,
      FI_EN_E8DAY_OF_WEEK_TUESDAY = 3UL,
      FI_EN_E8DAY_OF_WEEK_WEDNESDAY = 4UL,
      FI_EN_E8DAY_OF_WEEK_THURSDAY = 5UL,
      FI_EN_E8DAY_OF_WEEK_FRIDAY = 6UL,
      FI_EN_E8DAY_OF_WEEK_SATURDAY = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWHighPowerDayOfWeek(most_fi_tcl_e8_GMLnGWHighPowerDayOfWeek::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWHighPowerDayOfWeek();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWHighPowerDayOfWeek& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWStartTimeHighPowerDayOfWeek : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWStartTimeHighPowerDayOfWeek();
   enum tenType {
      FI_EN_E8DAY_OF_WEEK_UNKNOWN = 0UL,
      FI_EN_E8DAY_OF_WEEK_SUNDAY = 1UL,
      FI_EN_E8DAY_OF_WEEK_MONDAY = 2UL,
      FI_EN_E8DAY_OF_WEEK_TUESDAY = 3UL,
      FI_EN_E8DAY_OF_WEEK_WEDNESDAY = 4UL,
      FI_EN_E8DAY_OF_WEEK_THURSDAY = 5UL,
      FI_EN_E8DAY_OF_WEEK_FRIDAY = 6UL,
      FI_EN_E8DAY_OF_WEEK_SATURDAY = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWStartTimeHighPowerDayOfWeek(most_fi_tcl_e8_GMLnGWStartTimeHighPowerDayOfWeek::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWStartTimeHighPowerDayOfWeek();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWStartTimeHighPowerDayOfWeek& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWStartTimeLowPowerDayOfWeek : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWStartTimeLowPowerDayOfWeek();
   enum tenType {
      FI_EN_E8DAY_OF_WEEK_UNKNOWN = 0UL,
      FI_EN_E8DAY_OF_WEEK_SUNDAY = 1UL,
      FI_EN_E8DAY_OF_WEEK_MONDAY = 2UL,
      FI_EN_E8DAY_OF_WEEK_TUESDAY = 3UL,
      FI_EN_E8DAY_OF_WEEK_WEDNESDAY = 4UL,
      FI_EN_E8DAY_OF_WEEK_THURSDAY = 5UL,
      FI_EN_E8DAY_OF_WEEK_FRIDAY = 6UL,
      FI_EN_E8DAY_OF_WEEK_SATURDAY = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWStartTimeLowPowerDayOfWeek(most_fi_tcl_e8_GMLnGWStartTimeLowPowerDayOfWeek::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWStartTimeLowPowerDayOfWeek();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWStartTimeLowPowerDayOfWeek& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWMaintenanceModeDisplayRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWMaintenanceModeDisplayRequest();
   enum tenType {
      FI_EN_E8MAINTENANCE_MODE_NO_REQUEST = 0UL,
      FI_EN_E8ENG_MAINTENANCE_MODE_DAY5_WARNING = 1UL,
      FI_EN_E8ENG_MAINTENANCE_MODE_DAY4_WARNING = 2UL,
      FI_EN_E8ENG_MAINTENANCE_MODE_DAY3_WARNING = 3UL,
      FI_EN_E8ENG_MAINTENANCE_MODE_DAY2_WARNING = 4UL,
      FI_EN_E8ENG_MAINTENANCE_MODE_DAY1_WARNING = 5UL,
      FI_EN_E8FORCE_ENGINE_MAINTENANCE = 6UL,
      FI_EN_E8CONTINUE_ENGINE_MAINTENANCE = 7UL,
      FI_EN_E8FUEL_MAINTENANCE_MODE_DAY5_WARNING = 8UL,
      FI_EN_E8FUEL_MAINTENANCE_MODE_DAY4_WARNING = 9UL,
      FI_EN_E8FUEL_MAINTENANCE_MODE_DAY3_WARNING = 10UL,
      FI_EN_E8FUEL_MAINTENANCE_MODE_DAY2_WARNING = 11UL,
      FI_EN_E8FUEL_MAINTENANCE_MODE_DAY1_WARNING = 12UL,
      FI_EN_E8FORCE_FUEL_MAINTENANCE = 13UL,
      FI_EN_E8CONTINUE_FUEL_MAINTENANCE = 14UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWMaintenanceModeDisplayRequest(most_fi_tcl_e8_GMLnGWMaintenanceModeDisplayRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWMaintenanceModeDisplayRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWMaintenanceModeDisplayRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWNextPlannedDepartureDayOfWeek : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWNextPlannedDepartureDayOfWeek();
   enum tenType {
      FI_EN_E8DAY_OF_WEEK_UNKNOWN = 0UL,
      FI_EN_E8DAY_OF_WEEK_SUNDAY = 1UL,
      FI_EN_E8DAY_OF_WEEK_MONDAY = 2UL,
      FI_EN_E8DAY_OF_WEEK_TUESDAY = 3UL,
      FI_EN_E8DAY_OF_WEEK_WEDNESDAY = 4UL,
      FI_EN_E8DAY_OF_WEEK_THURSDAY = 5UL,
      FI_EN_E8DAY_OF_WEEK_FRIDAY = 6UL,
      FI_EN_E8DAY_OF_WEEK_SATURDAY = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWNextPlannedDepartureDayOfWeek(most_fi_tcl_e8_GMLnGWNextPlannedDepartureDayOfWeek::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWNextPlannedDepartureDayOfWeek();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWNextPlannedDepartureDayOfWeek& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWHybridModeDisplayRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWHybridModeDisplayRequest();
   enum tenType {
      FI_EN_E8HYBRID_MODE_DISPLAY_DEFAULT_ACCESSORY = 0UL,
      FI_EN_E8HYBRID_MODE_DISPLAY_ENGINE_IDLE = 1UL,
      FI_EN_E8HYBRID_MODE_DISPLAY_AUTO_STOP = 2UL,
      FI_EN_E8HYBRID_MODE_DISPLAY_BATTERY_POWER_2WD = 3UL,
      FI_EN_E8HYBRID_MODE_DISPLAY_BATTERY_POWER_4WD = 4UL,
      FI_EN_E8HYBRID_MODE_DISPLAY_ENGINE_POWER_2WD = 5UL,
      FI_EN_E8HYBRID_MODE_DISPLAY_ENGINE_POWER_4WD = 6UL,
      FI_EN_E8HYBRID_MODE_DISPLAY_HYBRID_POWER_2WD = 7UL,
      FI_EN_E8HYBRID_MODE_DISPLAY_HYBRID_POWER_4WD = 8UL,
      FI_EN_E8HYBRID_MODE_DISPLAY_CHARGING_ENGINE_OFF = 9UL,
      FI_EN_E8HYBRID_MODE_DISPLAY_CHARGING_ENGINE_ON = 10UL,
      FI_EN_E8HYBRID_MODE_DISPLAY_UNDETERMINED = 11UL,
      FI_EN_E8HYBRID_MODE_DISPLAY_BATTERY_POWER_LOAD_SHARING = 12UL,
      FI_EN_E8HYBRID_MODE_DISPLAY_ENGINE_POWER_LOAD_SHARING = 13UL,
      FI_EN_E8HYBRID_MODE_DISPLAY_ENGINE_CHARGING_BATTERY_MOVING = 14UL,
      FI_EN_E8HYBRID_MODE_DISPLAY_ENGINE_CHARGING_BATTERY_WHILE_IDLE = 15UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWHybridModeDisplayRequest(most_fi_tcl_e8_GMLnGWHybridModeDisplayRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWHybridModeDisplayRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWHybridModeDisplayRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTimeBasedChargingModeRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTimeBasedChargingModeRequest();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_CHARGE_NOW = 1UL,
      FI_EN_E8TODC_DELAYED_CHARGE = 2UL,
      FI_EN_E8TODC_RATE_BASED_CHARGE = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTimeBasedChargingModeRequest(most_fi_tcl_e8_GMLnGWTimeBasedChargingModeRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTimeBasedChargingModeRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTimeBasedChargingModeRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTimeBasedChargingModeResponse : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTimeBasedChargingModeResponse();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_CHARGE_NOW = 1UL,
      FI_EN_E8TODC_DELAYED_CHARGE = 2UL,
      FI_EN_E8TODC_RATE_BASED_CHARGE = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTimeBasedChargingModeResponse(most_fi_tcl_e8_GMLnGWTimeBasedChargingModeResponse::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTimeBasedChargingModeResponse();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTimeBasedChargingModeResponse& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWChargeRateDaySetRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWChargeRateDaySetRequest();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_SUNDAY = 1UL,
      FI_EN_E8TODC_MONDAY = 2UL,
      FI_EN_E8TODC_TUESDAY = 3UL,
      FI_EN_E8TODC_WEDNESDAY = 4UL,
      FI_EN_E8TODC_THURSDAY = 5UL,
      FI_EN_E8TODC_FRIDAY = 6UL,
      FI_EN_E8TODC_SATURDAY = 7UL,
      FI_EN_E8TODC_ALLDAYS_SAME = 8UL,
      FI_EN_E8TODC_MON_TO_FRI_SAME = 9UL,
      FI_EN_E8TODC_SAT_SUN_SAME = 10UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWChargeRateDaySetRequest(most_fi_tcl_e8_GMLnGWChargeRateDaySetRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWChargeRateDaySetRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWChargeRateDaySetRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWChargeRateEnableSetRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWChargeRateEnableSetRequest();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_ENABLED = 1UL,
      FI_EN_E8TODC_DISABLED = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWChargeRateEnableSetRequest(most_fi_tcl_e8_GMLnGWChargeRateEnableSetRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWChargeRateEnableSetRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWChargeRateEnableSetRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWChargeRateSelectedSetRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWChargeRateSelectedSetRequest();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_LOW_RATE = 1UL,
      FI_EN_E8TODC_MEDIUM_RATE = 2UL,
      FI_EN_E8TODC_HIGH_RATE = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWChargeRateSelectedSetRequest(most_fi_tcl_e8_GMLnGWChargeRateSelectedSetRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWChargeRateSelectedSetRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWChargeRateSelectedSetRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWChargeRateDayStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWChargeRateDayStatus();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_SUNDAY = 1UL,
      FI_EN_E8TODC_MONDAY = 2UL,
      FI_EN_E8TODC_TUESDAY = 3UL,
      FI_EN_E8TODC_WEDNESDAY = 4UL,
      FI_EN_E8TODC_THURSDAY = 5UL,
      FI_EN_E8TODC_FRIDAY = 6UL,
      FI_EN_E8TODC_SATURDAY = 7UL,
      FI_EN_E8TODC_ALLDAYS_SAME = 8UL,
      FI_EN_E8TODC_MON_TO_FRI_SAME = 9UL,
      FI_EN_E8TODC_SAT_SUN_SAME = 10UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWChargeRateDayStatus(most_fi_tcl_e8_GMLnGWChargeRateDayStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWChargeRateDayStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWChargeRateDayStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWChargeRateEnableStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWChargeRateEnableStatus();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_ENABLED = 1UL,
      FI_EN_E8TODC_DISABLED = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWChargeRateEnableStatus(most_fi_tcl_e8_GMLnGWChargeRateEnableStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWChargeRateEnableStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWChargeRateEnableStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWChargeRateSelectedStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWChargeRateSelectedStatus();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_LOW_RATE = 1UL,
      FI_EN_E8TODC_MEDIUM_RATE = 2UL,
      FI_EN_E8TODC_HIGH_RATE = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWChargeRateSelectedStatus(most_fi_tcl_e8_GMLnGWChargeRateSelectedStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWChargeRateSelectedStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWChargeRateSelectedStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDelayedChargingDay : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDelayedChargingDay();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_SUNDAY = 1UL,
      FI_EN_E8TODC_MONDAY = 2UL,
      FI_EN_E8TODC_TUESDAY = 3UL,
      FI_EN_E8TODC_WEDNESDAY = 4UL,
      FI_EN_E8TODC_THURSDAY = 5UL,
      FI_EN_E8TODC_FRIDAY = 6UL,
      FI_EN_E8TODC_SATURDAY = 7UL,
      FI_EN_E8TODC_ALLDAYS_SAME = 8UL,
      FI_EN_E8TODC_MON_TO_FRI_SAME = 9UL,
      FI_EN_E8TODC_SAT_SUN_SAME = 10UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDelayedChargingDay(most_fi_tcl_e8_GMLnGWDelayedChargingDay::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDelayedChargingDay();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDelayedChargingDay& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDelayedChargingSeason : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDelayedChargingSeason();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_SUMMER = 1UL,
      FI_EN_E8TODC_WINTER = 2UL,
      FI_EN_E8TODC_YEARLY = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDelayedChargingSeason(most_fi_tcl_e8_GMLnGWDelayedChargingSeason::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDelayedChargingSeason();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDelayedChargingSeason& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDelayedChargingSelected : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDelayedChargingSelected();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_DEPARTURE_TIME_1 = 1UL,
      FI_EN_E8TODC_DEPARTURE_TIME_1_INVALID = 3UL,
      FI_EN_E8TODC_DEPARTURE_TIME_2 = 5UL,
      FI_EN_E8TODC_DEPARTURE_TIME_2_INVALID = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDelayedChargingSelected(most_fi_tcl_e8_GMLnGWDelayedChargingSelected::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDelayedChargingSelected();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDelayedChargingSelected& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDelayedChargingDayStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDelayedChargingDayStatus();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_SUNDAY = 1UL,
      FI_EN_E8TODC_MONDAY = 2UL,
      FI_EN_E8TODC_TUESDAY = 3UL,
      FI_EN_E8TODC_WEDNESDAY = 4UL,
      FI_EN_E8TODC_THURSDAY = 5UL,
      FI_EN_E8TODC_FRIDAY = 6UL,
      FI_EN_E8TODC_SATURDAY = 7UL,
      FI_EN_E8TODC_ALLDAYS_SAME = 8UL,
      FI_EN_E8TODC_MON_TO_FRI_SAME = 9UL,
      FI_EN_E8TODC_SAT_SUN_SAME = 10UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDelayedChargingDayStatus(most_fi_tcl_e8_GMLnGWDelayedChargingDayStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDelayedChargingDayStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDelayedChargingDayStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDelayedChargingSeasonStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDelayedChargingSeasonStatus();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_SUMMER = 1UL,
      FI_EN_E8TODC_WINTER = 2UL,
      FI_EN_E8TODC_YEARLY = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDelayedChargingSeasonStatus(most_fi_tcl_e8_GMLnGWDelayedChargingSeasonStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDelayedChargingSeasonStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDelayedChargingSeasonStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDelayedChargingSelectedStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDelayedChargingSelectedStatus();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_DEPARTURE_TIME_1 = 1UL,
      FI_EN_E8TODC_DEPARTURE_TIME_1_INVALID = 3UL,
      FI_EN_E8TODC_DEPARTURE_TIME_2 = 5UL,
      FI_EN_E8TODC_DEPARTURE_TIME_2_INVALID = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDelayedChargingSelectedStatus(most_fi_tcl_e8_GMLnGWDelayedChargingSelectedStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDelayedChargingSelectedStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDelayedChargingSelectedStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRateChargingDay : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRateChargingDay();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_SUNDAY = 1UL,
      FI_EN_E8TODC_MONDAY = 2UL,
      FI_EN_E8TODC_TUESDAY = 3UL,
      FI_EN_E8TODC_WEDNESDAY = 4UL,
      FI_EN_E8TODC_THURSDAY = 5UL,
      FI_EN_E8TODC_FRIDAY = 6UL,
      FI_EN_E8TODC_SATURDAY = 7UL,
      FI_EN_E8TODC_ALLDAYS_SAME = 8UL,
      FI_EN_E8TODC_MON_TO_FRI_SAME = 9UL,
      FI_EN_E8TODC_SAT_SUN_SAME = 10UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRateChargingDay(most_fi_tcl_e8_GMLnGWRateChargingDay::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRateChargingDay();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRateChargingDay& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRateChargingSeason : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRateChargingSeason();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_SUMMER = 1UL,
      FI_EN_E8TODC_WINTER = 2UL,
      FI_EN_E8TODC_YEARLY = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRateChargingSeason(most_fi_tcl_e8_GMLnGWRateChargingSeason::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRateChargingSeason();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRateChargingSeason& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRateChargingSelected : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRateChargingSelected();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_CHARGE_START = 1UL,
      FI_EN_E8TODC_CHARGE_END = 2UL,
      FI_EN_E8TODC_CHARGE_START_INVALID = 3UL,
      FI_EN_E8TODC_CHARGE_END_INVALID = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRateChargingSelected(most_fi_tcl_e8_GMLnGWRateChargingSelected::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRateChargingSelected();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRateChargingSelected& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRateChargingSelectedRate : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRateChargingSelectedRate();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_LOW_RATE = 1UL,
      FI_EN_E8TODC_MEDIUM_RATE = 2UL,
      FI_EN_E8TODC_HIGH_RATE = 3UL,
      FI_EN_E8TODC_INVALID_RATE = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRateChargingSelectedRate(most_fi_tcl_e8_GMLnGWRateChargingSelectedRate::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRateChargingSelectedRate();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRateChargingSelectedRate& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRateChargingDayStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRateChargingDayStatus();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_SUNDAY = 1UL,
      FI_EN_E8TODC_MONDAY = 2UL,
      FI_EN_E8TODC_TUESDAY = 3UL,
      FI_EN_E8TODC_WEDNESDAY = 4UL,
      FI_EN_E8TODC_THURSDAY = 5UL,
      FI_EN_E8TODC_FRIDAY = 6UL,
      FI_EN_E8TODC_SATURDAY = 7UL,
      FI_EN_E8TODC_ALLDAYS_SAME = 8UL,
      FI_EN_E8TODC_MON_TO_FRI_SAME = 9UL,
      FI_EN_E8TODC_SAT_SUN_SAME = 10UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRateChargingDayStatus(most_fi_tcl_e8_GMLnGWRateChargingDayStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRateChargingDayStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRateChargingDayStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRateChargingSeasonStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRateChargingSeasonStatus();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_SUMMER = 1UL,
      FI_EN_E8TODC_WINTER = 2UL,
      FI_EN_E8TODC_YEARLY = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRateChargingSeasonStatus(most_fi_tcl_e8_GMLnGWRateChargingSeasonStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRateChargingSeasonStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRateChargingSeasonStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRateChargingSelectedRateStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRateChargingSelectedRateStatus();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_LOW_RATE = 1UL,
      FI_EN_E8TODC_MEDIUM_RATE = 2UL,
      FI_EN_E8TODC_HIGH_RATE = 3UL,
      FI_EN_E8TODC_INVALID_RATE = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRateChargingSelectedRateStatus(most_fi_tcl_e8_GMLnGWRateChargingSelectedRateStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRateChargingSelectedRateStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRateChargingSelectedRateStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRateChargingSelectedStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRateChargingSelectedStatus();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_CHARGE_START = 1UL,
      FI_EN_E8TODC_CHARGE_END = 2UL,
      FI_EN_E8TODC_CHARGE_START_INVALID = 3UL,
      FI_EN_E8TODC_CHARGE_END_INVALID = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRateChargingSelectedStatus(most_fi_tcl_e8_GMLnGWRateChargingSelectedStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRateChargingSelectedStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRateChargingSelectedStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWSeason : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWSeason();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_SUMMER = 1UL,
      FI_EN_E8TODC_WINTER = 2UL,
      FI_EN_E8TODC_YEARLY = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWSeason(most_fi_tcl_e8_GMLnGWSeason::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWSeason();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWSeason& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWSeasonMonth : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWSeasonMonth();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_JANUARY = 1UL,
      FI_EN_E8TODC_FEBRUARY = 2UL,
      FI_EN_E8TODC_MARCH = 3UL,
      FI_EN_E8TODC_APRIL = 4UL,
      FI_EN_E8TODC_MAY = 5UL,
      FI_EN_E8TODC_JUNE = 6UL,
      FI_EN_E8TODC_JULY = 7UL,
      FI_EN_E8TODC_AUGUST = 8UL,
      FI_EN_E8TODC_SEPTEMBER = 9UL,
      FI_EN_E8TODC_OCTOBER = 10UL,
      FI_EN_E8TODC_NOVEMBER = 11UL,
      FI_EN_E8TODC_DECEMBER = 12UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWSeasonMonth(most_fi_tcl_e8_GMLnGWSeasonMonth::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWSeasonMonth();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWSeasonMonth& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWSeasonSelected : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWSeasonSelected();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_SELECTED_SEASON = 1UL,
      FI_EN_E8TODC_START_SEASON = 2UL,
      FI_EN_E8TODC_END_SEASON = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWSeasonSelected(most_fi_tcl_e8_GMLnGWSeasonSelected::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWSeasonSelected();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWSeasonSelected& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWSeasonMonthStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWSeasonMonthStatus();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_JANUARY = 1UL,
      FI_EN_E8TODC_FEBRUARY = 2UL,
      FI_EN_E8TODC_MARCH = 3UL,
      FI_EN_E8TODC_APRIL = 4UL,
      FI_EN_E8TODC_MAY = 5UL,
      FI_EN_E8TODC_JUNE = 6UL,
      FI_EN_E8TODC_JULY = 7UL,
      FI_EN_E8TODC_AUGUST = 8UL,
      FI_EN_E8TODC_SEPTEMBER = 9UL,
      FI_EN_E8TODC_OCTOBER = 10UL,
      FI_EN_E8TODC_NOVEMBER = 11UL,
      FI_EN_E8TODC_DECEMBER = 12UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWSeasonMonthStatus(most_fi_tcl_e8_GMLnGWSeasonMonthStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWSeasonMonthStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWSeasonMonthStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWSeasonSelectedStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWSeasonSelectedStatus();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_SELECTED_SEASON = 1UL,
      FI_EN_E8TODC_START_SEASON = 2UL,
      FI_EN_E8TODC_END_SEASON = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWSeasonSelectedStatus(most_fi_tcl_e8_GMLnGWSeasonSelectedStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWSeasonSelectedStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWSeasonSelectedStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWSeasonStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWSeasonStatus();
   enum tenType {
      FI_EN_E8TODC_NOACTION = 0UL,
      FI_EN_E8TODC_SUMMER = 1UL,
      FI_EN_E8TODC_WINTER = 2UL,
      FI_EN_E8TODC_YEARLY = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWSeasonStatus(most_fi_tcl_e8_GMLnGWSeasonStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWSeasonStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWSeasonStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWSetTempOverrideNextPlannedDepartureAdtive : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWSetTempOverrideNextPlannedDepartureAdtive();
   enum tenType {
      FI_EN_E8NO_ACTION = 0UL,
      FI_EN_E8FALSE_INACTIVE = 1UL,
      FI_EN_E8TRUE_ACTIVE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWSetTempOverrideNextPlannedDepartureAdtive(most_fi_tcl_e8_GMLnGWSetTempOverrideNextPlannedDepartureAdtive::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWSetTempOverrideNextPlannedDepartureAdtive();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWSetTempOverrideNextPlannedDepartureAdtive& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTempOverrideNextPlannedDepartureDayOfWeek : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTempOverrideNextPlannedDepartureDayOfWeek();
   enum tenType {
      FI_EN_E8DAY_OF_WEEK_UNKNOWN = 1UL,
      FI_EN_E8DAY_OF_WEEK_SUNDAY = 2UL,
      FI_EN_E8DAY_OF_WEEK_MONDAY = 3UL,
      FI_EN_E8DAY_OF_WEEK_TUESDAY = 4UL,
      FI_EN_E8DAY_OF_WEEK_WEDNESDAY = 5UL,
      FI_EN_E8DAY_OF_WEEK_THURSDAY = 6UL,
      FI_EN_E8DAY_OF_WEEK_FRIDAY = 7UL,
      FI_EN_E8DAY_OF_WEEK_SATURDAY = 8UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTempOverrideNextPlannedDepartureDayOfWeek(most_fi_tcl_e8_GMLnGWTempOverrideNextPlannedDepartureDayOfWeek::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTempOverrideNextPlannedDepartureDayOfWeek();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTempOverrideNextPlannedDepartureDayOfWeek& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWChargeLevelPreferenceStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWChargeLevelPreferenceStatus();
   enum tenType {
      FI_EN_E8CHARGE_LEVEL_PREF_NO_REDUCTION = 0UL,
      FI_EN_E8CHARGE_LEVEL_PREF_REDUCED_LEVEL_1 = 1UL,
      FI_EN_E8CHARGE_LEVEL_PREF_REDUCED_LEVEL_2 = 2UL,
      FI_EN_E8CHARGE_LEVEL_PREF_REDUCED_LEVEL_3 = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWChargeLevelPreferenceStatus(most_fi_tcl_e8_GMLnGWChargeLevelPreferenceStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWChargeLevelPreferenceStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWChargeLevelPreferenceStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWUserInterfaceType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWUserInterfaceType();
   enum tenType {
      FI_EN_E8USER_INTERFACE_1 = 0UL,
      FI_EN_E8USER_INTERFACE_2 = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWUserInterfaceType(most_fi_tcl_e8_GMLnGWUserInterfaceType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWUserInterfaceType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWUserInterfaceType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWSetChargingSystemChargeLevelPreference : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWSetChargingSystemChargeLevelPreference();
   enum tenType {
      FI_EN_E8SET_CHARGE_LEVEL_PREFERENCE_NO_ACTION = 0UL,
      FI_EN_E8SET_CHARGE_LEVEL_PREFERENCE_NO_REDUCTION = 1UL,
      FI_EN_E8SET_CHARGE_LEVEL_PREFERENCE_REDUCED_CURRENT_LEVEL_1 = 2UL,
      FI_EN_E8SET_CHARGE_LEVEL_PREFERENCE_REDUCED_CURRENT_LEVEL_2 = 3UL,
      FI_EN_E8SET_CHARGE_LEVEL_PREFERENCE_REDUCED_CURRENT_LEVEL_3 = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWSetChargingSystemChargeLevelPreference(most_fi_tcl_e8_GMLnGWSetChargingSystemChargeLevelPreference::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWSetChargingSystemChargeLevelPreference();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWSetChargingSystemChargeLevelPreference& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTimeOfDayChargingDisplayTemporaryOverrideStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTimeOfDayChargingDisplayTemporaryOverrideStatus();
   enum tenType {
      FI_EN_E8TODC_DISPLAY_TEMP_OVERRIDE_NO_OVERRIDES_ACTIVE = 0UL,
      FI_EN_E8TODC_DISPLAY_TEMP_OVERRIDE_TIME_OF_DAY_CHARGING = 1UL,
      FI_EN_E8TODC_DISPLAY_TEMP_OVERRIDE_NEXT_PLANNED_DEPARTURE = 2UL,
      FI_EN_E8TODC_DISPLAY_TEMP_OVERRIDE_TIME_OF_DAY_CHARGING_DISABLED = 3UL,
      FI_EN_E8TODC_DISPLAY_TEMP_OVERRIDE_ONBOARD_CHARGING_FAULT = 4UL,
      FI_EN_E8TODC_DISPLAY_TEMP_OVERRIDE_OFFBOARD_CHARGING_FAULT = 5UL,
      FI_EN_E8TODC_DISPLAY_TEMP_OVERRIDE_ONBOARD_AND_OFFBOARD_CHARGING_FAULT = 6UL,
      FI_EN_E8TODC_DISPLAY_TEMP_OVERRIDE_OFFBOARD_ACTIVE = 7UL,
      FI_EN_E8TODC_DISPLAY_TEMP_OVERRIDE_OFFBOARD_NOT_ACTIVE = 8UL,
      FI_EN_E8TODC_DISPLAY_TEMP_OVERRIDE_OFFBOARD_ACTIVE_BULK_DONE = 9UL,
      FI_EN_E8TODC_DISPLAY_TEMP_OVERRIDE_OFFBOARD_NOT_ACTIVE_BULK_DONE = 10UL,
      FI_EN_E8TODC_DISPLAY_TEMP_OVERRIDE_OFFBOARD_THERMAL_CONDITIONING_ONLY = 11UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTimeOfDayChargingDisplayTemporaryOverrideStatus(most_fi_tcl_e8_GMLnGWTimeOfDayChargingDisplayTemporaryOverrideStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTimeOfDayChargingDisplayTemporaryOverrideStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTimeOfDayChargingDisplayTemporaryOverrideStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWJumpStartStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWJumpStartStatus();
   enum tenType {
      FI_EN_E8JUMP_START_STATUS_INACTIVE = 0UL,
      FI_EN_E8JUMP_START_STATUS_IN_PROGRESS = 1UL,
      FI_EN_E8JUMP_START_STATUS_COMPLETE = 2UL,
      FI_EN_E8JUMP_START_STATUS_POWER_MODE = 3UL,
      FI_EN_E8JUMP_START_STATUS_DISABLED = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWJumpStartStatus(most_fi_tcl_e8_GMLnGWJumpStartStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWJumpStartStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWJumpStartStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWHybridChargeModeState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWHybridChargeModeState();
   enum tenType {
      FI_EN_E8HYBRID_CHARGE_MODE_STATE_CHARGE_DEPLETING = 0UL,
      FI_EN_E8HYBRID_CHARGE_MODE_STATE_CHARGE_SUSTAINING = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWHybridChargeModeState(most_fi_tcl_e8_GMLnGWHybridChargeModeState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWHybridChargeModeState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWHybridChargeModeState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphMeasurement : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphMeasurement();
   enum tenType {
      FI_EN_E8ECHG_MEAS_UNIT_KM_PER_KWH = 0UL,
      FI_EN_E8ECHG_MEAS_UNIT_MI_PER_KWH = 1UL,
      FI_EN_E8ECHG_MEAS_UNIT_KWH_PER_100KM = 2UL,
      FI_EN_E8ECHG_MEAS_UNIT_KM_PER_L = 3UL,
      FI_EN_E8ECHG_MEAS_UNIT_MI_PER_GAL = 4UL,
      FI_EN_E8ECHG_MEAS_UNIT_L_PER_100KM = 5UL,
      FI_EN_E8ECHG_MEAS_UNIT_MI_PER_KG = 6UL,
      FI_EN_E8ECHG_MEAS_UNIT_KG_PER_100KM = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphMeasurement(most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphMeasurement::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphMeasurement();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphMeasurement& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphXAxisTickMarkInterval : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphXAxisTickMarkInterval();
   enum tenType {
      FI_EN_E8ECHG_XAXIS_TICK_INTERVAL_0 = 0UL,
      FI_EN_E8ECHG_XAXIS_TICK_INTERVAL_1 = 1UL,
      FI_EN_E8ECHG_XAXIS_TICK_INTERVAL_2 = 2UL,
      FI_EN_E8ECHG_XAXIS_TICK_INTERVAL_3 = 3UL,
      FI_EN_E8ECHG_XAXIS_TICK_INTERVAL_4 = 4UL,
      FI_EN_E8ECHG_XAXIS_TICK_INTERVAL_5 = 5UL,
      FI_EN_E8ECHG_XAXIS_TICK_INTERVAL_6 = 6UL,
      FI_EN_E8ECHG_XAXIS_TICK_INTERVAL_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphXAxisTickMarkInterval(most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphXAxisTickMarkInterval::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphXAxisTickMarkInterval();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphXAxisTickMarkInterval& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphYAxisMaximumValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphYAxisMaximumValue();
   enum tenType {
      FI_EN_E8ECHG_YAXIS_MAX_VALUE__0 = 0UL,
      FI_EN_E8ECHG_YAXIS_MAX_VALUE__1 = 1UL,
      FI_EN_E8ECHG_YAXIS_MAX_VALUE__2 = 2UL,
      FI_EN_E8ECHG_YAXIS_MAX_VALUE__3 = 3UL,
      FI_EN_E8ECHG_YAXIS_MAX_VALUE__4 = 4UL,
      FI_EN_E8ECHG_YAXIS_MAX_VALUE__5 = 5UL,
      FI_EN_E8ECHG_YAXIS_MAX_VALUE__6 = 6UL,
      FI_EN_E8ECHG_YAXIS_MAX_VALUE__7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphYAxisMaximumValue(most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphYAxisMaximumValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphYAxisMaximumValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphYAxisMaximumValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWEnergyEconomyDataStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWEnergyEconomyDataStream();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWEnergyEconomyDataStream& operator=(const most_fi_tcl_GMLnGWEnergyEconomyDataStream& coRef);
   most_fi_tcl_GMLnGWEnergyEconomyDataStream(const most_fi_tcl_GMLnGWEnergyEconomyDataStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 u16ChargeCycleElectricalEnergyEconomyEquivalent;
   tU16 u16ChargeCycleFuelEconomy;
   tU16 u16ChargeCycleOverallEnergyEconomyEquivalent;
   tU16 u16LifetimeFuelEconomyEquivalent;
   virtual ~most_fi_tcl_GMLnGWEnergyEconomyDataStream();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWEnergyEconomyDataStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRateBasedChargeCompletionTimePreferenceRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRateBasedChargeCompletionTimePreferenceRequest();
   enum tenType {
      FI_EN_E8RBCCT_NOACTION = 0UL,
      FI_EN_E8RBCCT_EARLIEST_POSSIBLE = 1UL,
      FI_EN_E8RBCCT_LATEST_POSSIBLE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRateBasedChargeCompletionTimePreferenceRequest(most_fi_tcl_e8_GMLnGWRateBasedChargeCompletionTimePreferenceRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRateBasedChargeCompletionTimePreferenceRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRateBasedChargeCompletionTimePreferenceRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRateBasedChargeCompletionTimePreferenceResponse : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRateBasedChargeCompletionTimePreferenceResponse();
   enum tenType {
      FI_EN_E8RBCCT_NOACTION = 0UL,
      FI_EN_E8RBCCT_EARLIEST_POSSIBLE = 1UL,
      FI_EN_E8RBCCT_LATEST_POSSIBLE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRateBasedChargeCompletionTimePreferenceResponse(most_fi_tcl_e8_GMLnGWRateBasedChargeCompletionTimePreferenceResponse::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRateBasedChargeCompletionTimePreferenceResponse();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRateBasedChargeCompletionTimePreferenceResponse& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerClearStoredPositionRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerClearStoredPositionRequest();
   enum tenType {
      FI_EN_E8LBC_NOACTION = 0UL,
      FI_EN_E8LBC_REQUEST_CLEAR_POSITION = 1UL,
      FI_EN_E8LBC_CONFIRMATION_TO_CLEAR_POSITION = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerClearStoredPositionRequest(most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerClearStoredPositionRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerClearStoredPositionRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerClearStoredPositionRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerDataResetRequest_PositionNumber : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerDataResetRequest_PositionNumber();
   enum tenType {
      FI_EN_E8LBCRR_POS_NUM_NO_ACTION = 0UL,
      FI_EN_E8LBCRR_POS_NUM_POS_NUM_1 = 1UL,
      FI_EN_E8LBCRR_POS_NUM_POS_NUM_2 = 2UL,
      FI_EN_E8LBCRR_POS_NUM_POS_NUM_3 = 3UL,
      FI_EN_E8LBCRR_POS_NUM_POS_NUM_4 = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerDataResetRequest_PositionNumber(most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerDataResetRequest_PositionNumber::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerDataResetRequest_PositionNumber();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerDataResetRequest_PositionNumber& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerClearStoredPositionResponse : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerClearStoredPositionResponse();
   enum tenType {
      FI_EN_E8LBC_NOACTION = 0UL,
      FI_EN_E8LBC_REQUEST_CLEAR_POSITION = 1UL,
      FI_EN_E8LBC_CLEAR_POSITION_PERFORMED = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerClearStoredPositionResponse(most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerClearStoredPositionResponse::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerClearStoredPositionResponse();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerClearStoredPositionResponse& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerDataResetResponse_PositionNumber : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerDataResetResponse_PositionNumber();
   enum tenType {
      FI_EN_E8LBCRR_POS_NUM_NO_ACTION = 0UL,
      FI_EN_E8LBCRR_POS_NUM_POS_NUM_1 = 1UL,
      FI_EN_E8LBCRR_POS_NUM_POS_NUM_2 = 2UL,
      FI_EN_E8LBCRR_POS_NUM_POS_NUM_3 = 3UL,
      FI_EN_E8LBCRR_POS_NUM_POS_NUM_4 = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerDataResetResponse_PositionNumber(most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerDataResetResponse_PositionNumber::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerDataResetResponse_PositionNumber();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLocationBasedChargingCustomerDataResetResponse_PositionNumber& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStoreRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStoreRequest();
   enum tenType {
      FI_EN_E8LBC_POS_STORE_NO_ACTION = 0UL,
      FI_EN_E8LBC_POS_STORE_POS_1 = 1UL,
      FI_EN_E8LBC_POS_STORE_POS_2 = 2UL,
      FI_EN_E8LBC_POS_STORE_POS_3 = 3UL,
      FI_EN_E8LBC_POS_STORE_POS_4 = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStoreRequest(most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStoreRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStoreRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStoreRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStorage_PositionUpdatedStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStorage_PositionUpdatedStatus();
   enum tenType {
      FI_EN_E8LBCPSS_NOACTION = 0UL,
      FI_EN_E8LBCPSS_POSITION_UPDATED = 1UL,
      FI_EN_E8LBCPSS_POSITION_UPDATE_FAILED = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStorage_PositionUpdatedStatus(most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStorage_PositionUpdatedStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStorage_PositionUpdatedStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStorage_PositionUpdatedStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStorage_PositionUpdateLocation : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStorage_PositionUpdateLocation();
   enum tenType {
      FI_EN_E8LBC_POS_STORE_NO_ACTION = 0UL,
      FI_EN_E8LBC_POS_STORE_POS_1 = 1UL,
      FI_EN_E8LBC_POS_STORE_POS_2 = 2UL,
      FI_EN_E8LBC_POS_STORE_POS_3 = 3UL,
      FI_EN_E8LBC_POS_STORE_POS_4 = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStorage_PositionUpdateLocation(most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStorage_PositionUpdateLocation::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStorage_PositionUpdateLocation();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLocationBasedChargingPositionStorage_PositionUpdateLocation& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLocationBasedChargingVehicleLocationStatus_GPSLocation : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLocationBasedChargingVehicleLocationStatus_GPSLocation();
   enum tenType {
      FI_EN_E8LBCVLS_NOT_IN_STORED_GPS_REGION_NOACTION = 0UL,
      FI_EN_E8LBCVLS_VEHICLE_WITHIN_STORED_GPS_REGION_1 = 1UL,
      FI_EN_E8LBCVLS_VEHICLE_WITHIN_STORED_GPS_REGION_2 = 2UL,
      FI_EN_E8LBCVLS_VEHICLE_WITHIN_STORED_GPS_REGION_3 = 3UL,
      FI_EN_E8LBCVLS_VEHICLE_WITHIN_STORED_GPS_REGION_4 = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLocationBasedChargingVehicleLocationStatus_GPSLocation(most_fi_tcl_e8_GMLnGWLocationBasedChargingVehicleLocationStatus_GPSLocation::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLocationBasedChargingVehicleLocationStatus_GPSLocation();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLocationBasedChargingVehicleLocationStatus_GPSLocation& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWEnergyUsageScoreDataStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWEnergyUsageScoreDataStream();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWEnergyUsageScoreDataStream& operator=(const most_fi_tcl_GMLnGWEnergyUsageScoreDataStream& coRef);
   most_fi_tcl_GMLnGWEnergyUsageScoreDataStream(const most_fi_tcl_GMLnGWEnergyUsageScoreDataStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tS8 s8TechniqueEnergyUsageScoreAverageValue;
   tS8 s8TerrainEnergyUsageScoreAverageValue;
   tS8 s8OutsideTemperatureEnergyUsageScoreAverageValue;
   tS8 s8InsideTemperatureEnergyUsageScoreAverageValue;
   tS16 s16TotalEnergyUsageScoreAverageValue;
   virtual ~most_fi_tcl_GMLnGWEnergyUsageScoreDataStream();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWEnergyUsageScoreDataStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWEnergyUsageScoreScalingDataStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWEnergyUsageScoreScalingDataStream();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWEnergyUsageScoreScalingDataStream& operator=(const most_fi_tcl_GMLnGWEnergyUsageScoreScalingDataStream& coRef);
   most_fi_tcl_GMLnGWEnergyUsageScoreScalingDataStream(const most_fi_tcl_GMLnGWEnergyUsageScoreScalingDataStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tS8 s8OutsideTemperatureEnergyUsageScoreMinScaleValue;
   tS8 s8OutsideTemperatureEnergyUsageScoreMaxScaleValue;
   tS8 s8InsideTemperatureEnergyUsageScoreMinScaleValue;
   tS8 s8InsideTemperatureEnergyUsageScoreMaxScaleValue;
   tS8 s8TechniqueEnergyUsageScoreMinScaleValue;
   tS8 s8TechniqueEnergyUsageScoreMaxScaleValue;
   tS8 s8TerrainEnergyUsageScoreMinScaleValue;
   tS8 s8TerrainEnergyUsageScoreMaxScaleValue;
   virtual ~most_fi_tcl_GMLnGWEnergyUsageScoreScalingDataStream();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWEnergyUsageScoreScalingDataStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphConfigurationDataMeasurementUnit : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphConfigurationDataMeasurementUnit();
   enum tenType {
      FI_EN_E8ECHGCD_MEAS_UNIT_KM_PER_KWH = 0UL,
      FI_EN_E8ECHGCD_MEAS_UNIT_MI_PER_KWH = 1UL,
      FI_EN_E8ECHGCD_MEAS_UNIT_KWH_PER_100KM = 2UL,
      FI_EN_E8ECHGCD_MEAS_UNIT_KM_PER_L = 3UL,
      FI_EN_E8ECHGCD_MEAS_UNIT_MI_PER_GAL = 4UL,
      FI_EN_E8ECHGCD_MEAS_UNIT_L_PER_100KM = 5UL,
      FI_EN_E8ECHGCD_MEAS_UNIT_MPGE_COMBINED = 6UL,
      FI_EN_E8ECHGCD_MEAS_UNIT_L_PER_100KM_COMBINED = 7UL,
      FI_EN_E8ECHGCD_MEAS_UNIT_KM_PER_L_COMBINED = 8UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphConfigurationDataMeasurementUnit(most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphConfigurationDataMeasurementUnit::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphConfigurationDataMeasurementUnit();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphConfigurationDataMeasurementUnit& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphConfigurationDataXAxisTickMarkInterval : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphConfigurationDataXAxisTickMarkInterval();
   enum tenType {
      FI_EN_E8X_AXIS_TICK_MARK_INTERVAL_0 = 0UL,
      FI_EN_E8X_AXIS_TICK_MARK_INTERVAL_1 = 1UL,
      FI_EN_E8X_AXIS_TICK_MARK_INTERVAL_2 = 2UL,
      FI_EN_E8X_AXIS_TICK_MARK_INTERVAL_3 = 3UL,
      FI_EN_E8X_AXIS_TICK_MARK_INTERVAL_4 = 4UL,
      FI_EN_E8X_AXIS_TICK_MARK_INTERVAL_5 = 5UL,
      FI_EN_E8X_AXIS_TICK_MARK_INTERVAL_6 = 6UL,
      FI_EN_E8X_AXIS_TICK_MARK_INTERVAL_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphConfigurationDataXAxisTickMarkInterval(most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphConfigurationDataXAxisTickMarkInterval::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphConfigurationDataXAxisTickMarkInterval();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWEnergyConsumptionHistoryGraphConfigurationDataXAxisTickMarkInterval& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWEnergyUsageScreenMeasurementUnits : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWEnergyUsageScreenMeasurementUnits();
   enum tenType {
      FI_EN_E8EUSAGE_MEAS_UNIT_NO_ACTION = 0UL,
      FI_EN_E8EUSAGE_MEAS_UNIT_MPG = 1UL,
      FI_EN_E8EUSAGE_MEAS_UNIT_MPGE_COMBINED = 2UL,
      FI_EN_E8EUSAGE_MEAS_UNIT_L_100KM = 3UL,
      FI_EN_E8EUSAGE_MEAS_UNIT_L_100KM_COMBINED = 4UL,
      FI_EN_E8EUSAGE_MEAS_UNIT_KM_L = 5UL,
      FI_EN_E8EUSAGE_MEAS_UNIT_KM_L_COMBINED = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWEnergyUsageScreenMeasurementUnits(most_fi_tcl_e8_GMLnGWEnergyUsageScreenMeasurementUnits::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWEnergyUsageScreenMeasurementUnits();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWEnergyUsageScreenMeasurementUnits& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_BackAndCushionBolsterInOut : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_BackAndCushionBolsterInOut();
   enum tenType {
      FI_EN_E8BBCB_NOT_SELECTED = 0UL,
      FI_EN_E8BBCB_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8BBCB_IN = 2UL,
      FI_EN_E8BBCB_OUT = 3UL,
      FI_EN_E8BBCB_CUSHION_BOLSTER_IN = 4UL,
      FI_EN_E8BBCB_CUSHION_BOLSTER_OUT = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_BackAndCushionBolsterInOut(most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_BackAndCushionBolsterInOut::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_BackAndCushionBolsterInOut();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_BackAndCushionBolsterInOut& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_BackBolsterInOut : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_BackBolsterInOut();
   enum tenType {
      FI_EN_E8BB_NOT_SELECTED = 0UL,
      FI_EN_E8BB_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8BB_IN = 2UL,
      FI_EN_E8BB_OUT = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_BackBolsterInOut(most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_BackBolsterInOut::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_BackBolsterInOut();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_BackBolsterInOut& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_CushionBolsterInOut : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_CushionBolsterInOut();
   enum tenType {
      FI_EN_E8CB_NOT_SELECTED = 0UL,
      FI_EN_E8CB_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8CB_IN = 2UL,
      FI_EN_E8CB_OUT = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_CushionBolsterInOut(most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_CushionBolsterInOut::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_CushionBolsterInOut();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_CushionBolsterInOut& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_CushionLengthAdjust : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_CushionLengthAdjust();
   enum tenType {
      FI_EN_E8CLA_NOT_SELECTED = 0UL,
      FI_EN_E8CLA_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8CLA_FORWARD = 2UL,
      FI_EN_E8CLA_REARWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_CushionLengthAdjust(most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_CushionLengthAdjust::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_CushionLengthAdjust();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_CushionLengthAdjust& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_DisplaySize : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_DisplaySize();
   enum tenType {
      FI_EN_E8DS_FULL_SIZE = 0UL,
      FI_EN_E8DS_MEDIUM_SIZE = 1UL,
      FI_EN_E8DS_SMALL_SIZE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_DisplaySize(most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_DisplaySize::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_DisplaySize();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_DisplaySize& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestFwdRwd : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestFwdRwd();
   enum tenType {
      FI_EN_E8HRFR_NOT_SELECTED = 0UL,
      FI_EN_E8HRFR_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRFR_FORWARD = 2UL,
      FI_EN_E8HRFR_REARWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestFwdRwd(most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestFwdRwd::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestFwdRwd();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestFwdRwd& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestTiltForwardRearward : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestTiltForwardRearward();
   enum tenType {
      FI_EN_E8HRTILT_NOT_SELECTED = 0UL,
      FI_EN_E8HRTILT_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRTILT_FORWARD = 2UL,
      FI_EN_E8HRTILT_REARWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestTiltForwardRearward(most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestTiltForwardRearward::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestTiltForwardRearward();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestTiltForwardRearward& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestUpDown : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestUpDown();
   enum tenType {
      FI_EN_E8HRUD_NOT_SELECTED = 0UL,
      FI_EN_E8HRUD_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRUD_DOWN = 2UL,
      FI_EN_E8HRUD_UP = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestUpDown(most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestUpDown::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestUpDown();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestUpDown& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestUpDwnFwdRwd : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestUpDwnFwdRwd();
   enum tenType {
      FI_EN_E8HRUDFR_NOT_SELECTED = 0UL,
      FI_EN_E8HRUDFR_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRUDFR_UP = 2UL,
      FI_EN_E8HRUDFR_DOWN = 3UL,
      FI_EN_E8HRUDFR_FORWARD = 4UL,
      FI_EN_E8HRUDFR_REARWARD = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestUpDwnFwdRwd(most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestUpDwnFwdRwd::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestUpDwnFwdRwd();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestUpDwnFwdRwd& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestWingsInOut : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestWingsInOut();
   enum tenType {
      FI_EN_E8HRWING_NOT_SELECTED = 0UL,
      FI_EN_E8HRWING_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRWING_IN = 2UL,
      FI_EN_E8HRWING_OUT = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestWingsInOut(most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestWingsInOut::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestWingsInOut();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_HeadrestWingsInOut& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarForwardRearward : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarForwardRearward();
   enum tenType {
      FI_EN_E8LUM_FR_NOT_SELECTED = 0UL,
      FI_EN_E8LUM_FR_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8LUM_FR_FORWARD = 2UL,
      FI_EN_E8LUM_FR_REARWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarForwardRearward(most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarForwardRearward::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarForwardRearward();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarForwardRearward& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarUpDown : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarUpDown();
   enum tenType {
      FI_EN_E8LUM_UD_NOT_SELECTED = 0UL,
      FI_EN_E8LUM_UD_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8LUM_UD_UP = 2UL,
      FI_EN_E8LUM_UD_DOWN = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarUpDown(most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarUpDown::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarUpDown();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarUpDown& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarUpDownForwardRearward : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarUpDownForwardRearward();
   enum tenType {
      FI_EN_E8LUM_UDFR_NOT_SELECTED = 0UL,
      FI_EN_E8LUM_UDFR_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8LUM_UDFR_UP = 2UL,
      FI_EN_E8LUM_UDFR_DOWN = 3UL,
      FI_EN_E8LUM_UDFR_FORWARD = 4UL,
      FI_EN_E8LUM_UDFR_REARWARD = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarUpDownForwardRearward(most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarUpDownForwardRearward::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarUpDownForwardRearward();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_LumbarUpDownForwardRearward& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_Massage : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_Massage();
   enum tenType {
      FI_EN_E8MASSAGE_NOT_SELECTED = 0UL,
      FI_EN_E8SELECTED_MASSAGE_OFF = 1UL,
      FI_EN_E8SELECTED_MASSAGE_ON = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_Massage(most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_Massage::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_Massage();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_Massage& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_MassageIntensity : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_MassageIntensity();
   enum tenType {
      FI_EN_E8MASSAGE_OFF = 0UL,
      FI_EN_E8MASSAGE_INTENSITY_1 = 1UL,
      FI_EN_E8MASSAGE_INTENSITY_2 = 2UL,
      FI_EN_E8MASSAGE_INTENSITY_3 = 3UL,
      FI_EN_E8MASSAGE_INTENSITY_4 = 4UL,
      FI_EN_E8MASSAGE_INTENSITY_5 = 5UL,
      FI_EN_E8MASSAGE_INTENSITY_6 = 6UL,
      FI_EN_E8MASSAGE_INTENSITY_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_MassageIntensity(most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_MassageIntensity::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_MassageIntensity();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_MassageIntensity& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_MassageType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_MassageType();
   enum tenType {
      FI_EN_E8MASSAGE_TYPE_OFF = 0UL,
      FI_EN_E8MASSAGE_TYPE_1 = 1UL,
      FI_EN_E8MASSAGE_TYPE_2 = 2UL,
      FI_EN_E8MASSAGE_TYPE_3 = 3UL,
      FI_EN_E8MASSAGE_TYPE_4 = 4UL,
      FI_EN_E8MASSAGE_TYPE_5 = 5UL,
      FI_EN_E8MASSAGE_TYPE_6 = 6UL,
      FI_EN_E8MASSAGE_TYPE_7 = 7UL,
      FI_EN_E8MASSAGE_TYPE_8 = 8UL,
      FI_EN_E8MASSAGE_TYPE_9 = 9UL,
      FI_EN_E8MASSAGE_TYPE_10 = 10UL,
      FI_EN_E8MASSAGE_TYPE_11 = 11UL,
      FI_EN_E8MASSAGE_TYPE_12 = 12UL,
      FI_EN_E8MASSAGE_TYPE_13 = 13UL,
      FI_EN_E8MASSAGE_TYPE_14 = 14UL,
      FI_EN_E8MASSAGE_TYPE_15 = 15UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_MassageType(most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_MassageType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_MassageType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_MassageType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_UpperShoulderForwardRearward : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_UpperShoulderForwardRearward();
   enum tenType {
      FI_EN_E8HRUSFR_NOT_SELECTED = 0UL,
      FI_EN_E8HRUSFR_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRUSFR_FORWARD = 2UL,
      FI_EN_E8HRUSFR_REARWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_UpperShoulderForwardRearward(most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_UpperShoulderForwardRearward::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_UpperShoulderForwardRearward();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWDriverSeatControlStatus_UpperShoulderForwardRearward& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_BackAndCushionBolsterInOut : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_BackAndCushionBolsterInOut();
   enum tenType {
      FI_EN_E8BBCB_NOT_SELECTED = 0UL,
      FI_EN_E8BBCB_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8BBCB_IN = 2UL,
      FI_EN_E8BBCB_OUT = 3UL,
      FI_EN_E8BBCB_CUSHION_BOLSTER_IN = 4UL,
      FI_EN_E8BBCB_CUSHION_BOLSTER_OUT = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_BackAndCushionBolsterInOut(most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_BackAndCushionBolsterInOut::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_BackAndCushionBolsterInOut();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_BackAndCushionBolsterInOut& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_BackBolsterInOut : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_BackBolsterInOut();
   enum tenType {
      FI_EN_E8BB_NOT_SELECTED = 0UL,
      FI_EN_E8BB_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8BB_IN = 2UL,
      FI_EN_E8BB_OUT = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_BackBolsterInOut(most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_BackBolsterInOut::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_BackBolsterInOut();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_BackBolsterInOut& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_CushionBolsterInOut : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_CushionBolsterInOut();
   enum tenType {
      FI_EN_E8CB_NOT_SELECTED = 0UL,
      FI_EN_E8CB_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8CB_IN = 2UL,
      FI_EN_E8CB_OUT = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_CushionBolsterInOut(most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_CushionBolsterInOut::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_CushionBolsterInOut();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_CushionBolsterInOut& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_CushionLengthAdjust : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_CushionLengthAdjust();
   enum tenType {
      FI_EN_E8CLA_NOT_SELECTED = 0UL,
      FI_EN_E8CLA_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8CLA_FORWARD = 2UL,
      FI_EN_E8CLA_REARWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_CushionLengthAdjust(most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_CushionLengthAdjust::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_CushionLengthAdjust();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_CushionLengthAdjust& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_DisplaySize : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_DisplaySize();
   enum tenType {
      FI_EN_E8DS_FULL_SIZE = 0UL,
      FI_EN_E8DS_MEDIUM_SIZE = 1UL,
      FI_EN_E8DS_SMALL_SIZE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_DisplaySize(most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_DisplaySize::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_DisplaySize();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_DisplaySize& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestFwdRwd : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestFwdRwd();
   enum tenType {
      FI_EN_E8HRFR_NOT_SELECTED = 0UL,
      FI_EN_E8HRFR_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRFR_FORWARD = 2UL,
      FI_EN_E8HRFR_REARWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestFwdRwd(most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestFwdRwd::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestFwdRwd();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestFwdRwd& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestTiltForwardRearward : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestTiltForwardRearward();
   enum tenType {
      FI_EN_E8HRTILT_NOT_SELECTED = 0UL,
      FI_EN_E8HRTILT_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRTILT_FORWARD = 2UL,
      FI_EN_E8HRTILT_REARWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestTiltForwardRearward(most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestTiltForwardRearward::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestTiltForwardRearward();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestTiltForwardRearward& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestUpDown : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestUpDown();
   enum tenType {
      FI_EN_E8HRUD_NOT_SELECTED = 0UL,
      FI_EN_E8HRUD_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRUD_DOWN = 2UL,
      FI_EN_E8HRUD_UP = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestUpDown(most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestUpDown::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestUpDown();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestUpDown& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestUpDwnFwdRwd : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestUpDwnFwdRwd();
   enum tenType {
      FI_EN_E8HRUDFR_NOT_SELECTED = 0UL,
      FI_EN_E8HRUDFR_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRUDFR_UP = 2UL,
      FI_EN_E8HRUDFR_DOWN = 3UL,
      FI_EN_E8HRUDFR_FORWARD = 4UL,
      FI_EN_E8HRUDFR_REARWARD = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestUpDwnFwdRwd(most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestUpDwnFwdRwd::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestUpDwnFwdRwd();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestUpDwnFwdRwd& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestWingsInOut : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestWingsInOut();
   enum tenType {
      FI_EN_E8HRWING_NOT_SELECTED = 0UL,
      FI_EN_E8HRWING_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRWING_IN = 2UL,
      FI_EN_E8HRWING_OUT = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestWingsInOut(most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestWingsInOut::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestWingsInOut();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_HeadrestWingsInOut& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarForwardRearward : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarForwardRearward();
   enum tenType {
      FI_EN_E8LUM_FR_NOT_SELECTED = 0UL,
      FI_EN_E8LUM_FR_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8LUM_FR_FORWARD = 2UL,
      FI_EN_E8LUM_FR_REARWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarForwardRearward(most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarForwardRearward::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarForwardRearward();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarForwardRearward& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarUpDown : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarUpDown();
   enum tenType {
      FI_EN_E8LUM_UD_NOT_SELECTED = 0UL,
      FI_EN_E8LUM_UD_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8LUM_UD_UP = 2UL,
      FI_EN_E8LUM_UD_DOWN = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarUpDown(most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarUpDown::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarUpDown();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarUpDown& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarUpDownForwardRearward : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarUpDownForwardRearward();
   enum tenType {
      FI_EN_E8LUM_UDFR_NOT_SELECTED = 0UL,
      FI_EN_E8LUM_UDFR_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8LUM_UDFR_UP = 2UL,
      FI_EN_E8LUM_UDFR_DOWN = 3UL,
      FI_EN_E8LUM_UDFR_FORWARD = 4UL,
      FI_EN_E8LUM_UDFR_REARWARD = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarUpDownForwardRearward(most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarUpDownForwardRearward::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarUpDownForwardRearward();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_LumbarUpDownForwardRearward& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_Massage : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_Massage();
   enum tenType {
      FI_EN_E8MASSAGE_NOT_SELECTED = 0UL,
      FI_EN_E8SELECTED_MASSAGE_OFF = 1UL,
      FI_EN_E8SELECTED_MASSAGE_ON = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_Massage(most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_Massage::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_Massage();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_Massage& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_MassageIntensity : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_MassageIntensity();
   enum tenType {
      FI_EN_E8MASSAGE_OFF = 0UL,
      FI_EN_E8MASSAGE_INTENSITY_1 = 1UL,
      FI_EN_E8MASSAGE_INTENSITY_2 = 2UL,
      FI_EN_E8MASSAGE_INTENSITY_3 = 3UL,
      FI_EN_E8MASSAGE_INTENSITY_4 = 4UL,
      FI_EN_E8MASSAGE_INTENSITY_5 = 5UL,
      FI_EN_E8MASSAGE_INTENSITY_6 = 6UL,
      FI_EN_E8MASSAGE_INTENSITY_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_MassageIntensity(most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_MassageIntensity::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_MassageIntensity();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_MassageIntensity& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_MassageType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_MassageType();
   enum tenType {
      FI_EN_E8MASSAGE_TYPE_OFF = 0UL,
      FI_EN_E8MASSAGE_TYPE_1 = 1UL,
      FI_EN_E8MASSAGE_TYPE_2 = 2UL,
      FI_EN_E8MASSAGE_TYPE_3 = 3UL,
      FI_EN_E8MASSAGE_TYPE_4 = 4UL,
      FI_EN_E8MASSAGE_TYPE_5 = 5UL,
      FI_EN_E8MASSAGE_TYPE_6 = 6UL,
      FI_EN_E8MASSAGE_TYPE_7 = 7UL,
      FI_EN_E8MASSAGE_TYPE_8 = 8UL,
      FI_EN_E8MASSAGE_TYPE_9 = 9UL,
      FI_EN_E8MASSAGE_TYPE_10 = 10UL,
      FI_EN_E8MASSAGE_TYPE_11 = 11UL,
      FI_EN_E8MASSAGE_TYPE_12 = 12UL,
      FI_EN_E8MASSAGE_TYPE_13 = 13UL,
      FI_EN_E8MASSAGE_TYPE_14 = 14UL,
      FI_EN_E8MASSAGE_TYPE_15 = 15UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_MassageType(most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_MassageType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_MassageType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_MassageType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_UpperShoulderForwardRearward : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_UpperShoulderForwardRearward();
   enum tenType {
      FI_EN_E8HRUSFR_NOT_SELECTED = 0UL,
      FI_EN_E8HRUSFR_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRUSFR_FORWARD = 2UL,
      FI_EN_E8HRUSFR_REARWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_UpperShoulderForwardRearward(most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_UpperShoulderForwardRearward::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_UpperShoulderForwardRearward();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWPassengerSeatControlStatus_UpperShoulderForwardRearward& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_BackAndCushionBolsterInOut : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_BackAndCushionBolsterInOut();
   enum tenType {
      FI_EN_E8BBCB_NOT_SELECTED = 0UL,
      FI_EN_E8BBCB_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8BBCB_IN = 2UL,
      FI_EN_E8BBCB_OUT = 3UL,
      FI_EN_E8BBCB_CUSHION_BOLSTER_IN = 4UL,
      FI_EN_E8BBCB_CUSHION_BOLSTER_OUT = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_BackAndCushionBolsterInOut(most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_BackAndCushionBolsterInOut::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_BackAndCushionBolsterInOut();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_BackAndCushionBolsterInOut& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_BackBolsterInOut : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_BackBolsterInOut();
   enum tenType {
      FI_EN_E8BB_NOT_SELECTED = 0UL,
      FI_EN_E8BB_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8BB_IN = 2UL,
      FI_EN_E8BB_OUT = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_BackBolsterInOut(most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_BackBolsterInOut::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_BackBolsterInOut();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_BackBolsterInOut& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_CushionBolsterInOut : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_CushionBolsterInOut();
   enum tenType {
      FI_EN_E8CB_NOT_SELECTED = 0UL,
      FI_EN_E8CB_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8CB_IN = 2UL,
      FI_EN_E8CB_OUT = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_CushionBolsterInOut(most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_CushionBolsterInOut::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_CushionBolsterInOut();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_CushionBolsterInOut& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_CushionLengthAdjust : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_CushionLengthAdjust();
   enum tenType {
      FI_EN_E8CLA_NOT_SELECTED = 0UL,
      FI_EN_E8CLA_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8CLA_FORWARD = 2UL,
      FI_EN_E8CLA_REARWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_CushionLengthAdjust(most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_CushionLengthAdjust::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_CushionLengthAdjust();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_CushionLengthAdjust& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_DisplaySize : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_DisplaySize();
   enum tenType {
      FI_EN_E8DS_FULL_SIZE = 0UL,
      FI_EN_E8DS_MEDIUM_SIZE = 1UL,
      FI_EN_E8DS_SMALL_SIZE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_DisplaySize(most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_DisplaySize::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_DisplaySize();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_DisplaySize& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestFwdRwd : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestFwdRwd();
   enum tenType {
      FI_EN_E8HRFR_NOT_SELECTED = 0UL,
      FI_EN_E8HRFR_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRFR_FORWARD = 2UL,
      FI_EN_E8HRFR_REARWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestFwdRwd(most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestFwdRwd::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestFwdRwd();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestFwdRwd& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestTiltForwardRearward : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestTiltForwardRearward();
   enum tenType {
      FI_EN_E8HRTILT_NOT_SELECTED = 0UL,
      FI_EN_E8HRTILT_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRTILT_FORWARD = 2UL,
      FI_EN_E8HRTILT_REARWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestTiltForwardRearward(most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestTiltForwardRearward::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestTiltForwardRearward();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestTiltForwardRearward& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestUpDown : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestUpDown();
   enum tenType {
      FI_EN_E8HRUD_NOT_SELECTED = 0UL,
      FI_EN_E8HRUD_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRUD_UP = 2UL,
      FI_EN_E8HRUD_DOWN = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestUpDown(most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestUpDown::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestUpDown();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestUpDown& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestUpDwnFwdRwd : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestUpDwnFwdRwd();
   enum tenType {
      FI_EN_E8HRUDFR_NOT_SELECTED = 0UL,
      FI_EN_E8HRUDFR_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRUDFR_UP = 2UL,
      FI_EN_E8HRUDFR_DOWN = 3UL,
      FI_EN_E8HRUDFR_FORWARD = 4UL,
      FI_EN_E8HRUDFR_REARWARD = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestUpDwnFwdRwd(most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestUpDwnFwdRwd::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestUpDwnFwdRwd();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestUpDwnFwdRwd& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestWingsInOut : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestWingsInOut();
   enum tenType {
      FI_EN_E8HRWING_NOT_SELECTED = 0UL,
      FI_EN_E8HRWING_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRWING_IN = 2UL,
      FI_EN_E8HRWING_OUT = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestWingsInOut(most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestWingsInOut::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestWingsInOut();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_HeadrestWingsInOut& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarForwardRearward : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarForwardRearward();
   enum tenType {
      FI_EN_E8LUM_FR_NOT_SELECTED = 0UL,
      FI_EN_E8LUM_FR_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8LUM_FR_FORWARD = 2UL,
      FI_EN_E8LUM_FR_REARWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarForwardRearward(most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarForwardRearward::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarForwardRearward();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarForwardRearward& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarUpDown : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarUpDown();
   enum tenType {
      FI_EN_E8LUM_UD_NOT_SELECTED = 0UL,
      FI_EN_E8LUM_UD_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8LUM_UD_UP = 2UL,
      FI_EN_E8LUM_UD_DOWN = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarUpDown(most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarUpDown::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarUpDown();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarUpDown& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarUpDownForwardRearward : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarUpDownForwardRearward();
   enum tenType {
      FI_EN_E8LUM_UDFR_NOT_SELECTED = 0UL,
      FI_EN_E8LUM_UDFR_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8LUM_UDFR_UP = 2UL,
      FI_EN_E8LUM_UDFR_DOWN = 3UL,
      FI_EN_E8LUM_UDFR_FORWARD = 4UL,
      FI_EN_E8LUM_UDFR_REARWARD = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarUpDownForwardRearward(most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarUpDownForwardRearward::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarUpDownForwardRearward();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_LumbarUpDownForwardRearward& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_Massage : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_Massage();
   enum tenType {
      FI_EN_E8MASSAGE_NOT_SELECTED = 0UL,
      FI_EN_E8SELECTED_MASSAGE_OFF = 1UL,
      FI_EN_E8SELECTED_MASSAGE_ON = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_Massage(most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_Massage::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_Massage();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_Massage& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_MassageIntensity : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_MassageIntensity();
   enum tenType {
      FI_EN_E8MASSAGE_OFF = 0UL,
      FI_EN_E8MASSAGE_INTENSITY_1 = 1UL,
      FI_EN_E8MASSAGE_INTENSITY_2 = 2UL,
      FI_EN_E8MASSAGE_INTENSITY_3 = 3UL,
      FI_EN_E8MASSAGE_INTENSITY_4 = 4UL,
      FI_EN_E8MASSAGE_INTENSITY_5 = 5UL,
      FI_EN_E8MASSAGE_INTENSITY_6 = 6UL,
      FI_EN_E8MASSAGE_INTENSITY_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_MassageIntensity(most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_MassageIntensity::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_MassageIntensity();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_MassageIntensity& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_MassageType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_MassageType();
   enum tenType {
      FI_EN_E8MASSAGE_TYPE_OFF = 0UL,
      FI_EN_E8MASSAGE_TYPE_1 = 1UL,
      FI_EN_E8MASSAGE_TYPE_2 = 2UL,
      FI_EN_E8MASSAGE_TYPE_3 = 3UL,
      FI_EN_E8MASSAGE_TYPE_4 = 4UL,
      FI_EN_E8MASSAGE_TYPE_5 = 5UL,
      FI_EN_E8MASSAGE_TYPE_6 = 6UL,
      FI_EN_E8MASSAGE_TYPE_7 = 7UL,
      FI_EN_E8MASSAGE_TYPE_8 = 8UL,
      FI_EN_E8MASSAGE_TYPE_9 = 9UL,
      FI_EN_E8MASSAGE_TYPE_10 = 10UL,
      FI_EN_E8MASSAGE_TYPE_11 = 11UL,
      FI_EN_E8MASSAGE_TYPE_12 = 12UL,
      FI_EN_E8MASSAGE_TYPE_13 = 13UL,
      FI_EN_E8MASSAGE_TYPE_14 = 14UL,
      FI_EN_E8MASSAGE_TYPE_15 = 15UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_MassageType(most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_MassageType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_MassageType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_MassageType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_UpperShoulderForwardRearward : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_UpperShoulderForwardRearward();
   enum tenType {
      FI_EN_E8HRUSFR_NOT_SELECTED = 0UL,
      FI_EN_E8HRUSFR_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRUSFR_FORWARD = 2UL,
      FI_EN_E8HRUSFR_REARWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_UpperShoulderForwardRearward(most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_UpperShoulderForwardRearward::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_UpperShoulderForwardRearward();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWLeftRearSeatControlStatus_UpperShoulderForwardRearward& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_BackAndCushionBolsterInOut : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_BackAndCushionBolsterInOut();
   enum tenType {
      FI_EN_E8BBCB_NOT_SELECTED = 0UL,
      FI_EN_E8BBCB_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8BBCB_IN = 2UL,
      FI_EN_E8BBCB_OUT = 3UL,
      FI_EN_E8BBCB_CUSHION_BOLSTER_IN = 4UL,
      FI_EN_E8BBCB_CUSHION_BOLSTER_OUT = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_BackAndCushionBolsterInOut(most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_BackAndCushionBolsterInOut::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_BackAndCushionBolsterInOut();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_BackAndCushionBolsterInOut& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_BackBolsterInOut : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_BackBolsterInOut();
   enum tenType {
      FI_EN_E8BB_NOT_SELECTED = 0UL,
      FI_EN_E8BB_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8BB_IN = 2UL,
      FI_EN_E8BB_OUT = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_BackBolsterInOut(most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_BackBolsterInOut::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_BackBolsterInOut();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_BackBolsterInOut& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_CushionBolsterInOut : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_CushionBolsterInOut();
   enum tenType {
      FI_EN_E8CB_NOT_SELECTED = 0UL,
      FI_EN_E8CB_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8CB_IN = 2UL,
      FI_EN_E8CB_OUT = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_CushionBolsterInOut(most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_CushionBolsterInOut::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_CushionBolsterInOut();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_CushionBolsterInOut& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_CushionLengthAdjust : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_CushionLengthAdjust();
   enum tenType {
      FI_EN_E8CLA_NOT_SELECTED = 0UL,
      FI_EN_E8CLA_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8CLA_FORWARD = 2UL,
      FI_EN_E8CLA_REARWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_CushionLengthAdjust(most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_CushionLengthAdjust::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_CushionLengthAdjust();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_CushionLengthAdjust& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_DisplaySize : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_DisplaySize();
   enum tenType {
      FI_EN_E8DS_FULL_SIZE = 0UL,
      FI_EN_E8DS_MEDIUM_SIZE = 1UL,
      FI_EN_E8DS_SMALL_SIZE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_DisplaySize(most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_DisplaySize::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_DisplaySize();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_DisplaySize& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestFwdRwd : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestFwdRwd();
   enum tenType {
      FI_EN_E8HRFR_NOT_SELECTED = 0UL,
      FI_EN_E8HRFR_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRFR_FORWARD = 2UL,
      FI_EN_E8HRFR_REARWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestFwdRwd(most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestFwdRwd::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestFwdRwd();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestFwdRwd& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestTiltForwardRearward : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestTiltForwardRearward();
   enum tenType {
      FI_EN_E8HRTILT_NOT_SELECTED = 0UL,
      FI_EN_E8HRTILT_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRTILT_FORWARD = 2UL,
      FI_EN_E8HRTILT_REARWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestTiltForwardRearward(most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestTiltForwardRearward::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestTiltForwardRearward();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestTiltForwardRearward& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestUpDown : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestUpDown();
   enum tenType {
      FI_EN_E8HRUD_NOT_SELECTED = 0UL,
      FI_EN_E8HRUD_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRUD_UP = 2UL,
      FI_EN_E8HRUD_DOWN = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestUpDown(most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestUpDown::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestUpDown();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestUpDown& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestUpDwnFwdRwd : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestUpDwnFwdRwd();
   enum tenType {
      FI_EN_E8HRUDFR_NOT_SELECTED = 0UL,
      FI_EN_E8HRUDFR_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRUDFR_UP = 2UL,
      FI_EN_E8HRUDFR_DOWN = 3UL,
      FI_EN_E8HRUDFR_FORWARD = 4UL,
      FI_EN_E8HRUDFR_REARWARD = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestUpDwnFwdRwd(most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestUpDwnFwdRwd::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestUpDwnFwdRwd();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestUpDwnFwdRwd& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestWingsInOut : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestWingsInOut();
   enum tenType {
      FI_EN_E8HRWING_NOT_SELECTED = 0UL,
      FI_EN_E8HRWING_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRWING_IN = 2UL,
      FI_EN_E8HRWING_OUT = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestWingsInOut(most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestWingsInOut::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestWingsInOut();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_HeadrestWingsInOut& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarForwardRearward : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarForwardRearward();
   enum tenType {
      FI_EN_E8LUM_FR_NOT_SELECTED = 0UL,
      FI_EN_E8LUM_FR_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8LUM_FR_FORWARD = 2UL,
      FI_EN_E8LUM_FR_REARWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarForwardRearward(most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarForwardRearward::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarForwardRearward();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarForwardRearward& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarUpDown : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarUpDown();
   enum tenType {
      FI_EN_E8LUM_UD_NOT_SELECTED = 0UL,
      FI_EN_E8LUM_UD_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8LUM_UD_UP = 2UL,
      FI_EN_E8LUM_UD_DOWN = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarUpDown(most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarUpDown::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarUpDown();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarUpDown& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarUpDownForwardRearward : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarUpDownForwardRearward();
   enum tenType {
      FI_EN_E8LUM_UDFR_NOT_SELECTED = 0UL,
      FI_EN_E8LUM_UDFR_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8LUM_UDFR_UP = 2UL,
      FI_EN_E8LUM_UDFR_DOWN = 3UL,
      FI_EN_E8LUM_UDFR_FORWARD = 4UL,
      FI_EN_E8LUM_UDFR_REARWARD = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarUpDownForwardRearward(most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarUpDownForwardRearward::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarUpDownForwardRearward();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_LumbarUpDownForwardRearward& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_Massage : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_Massage();
   enum tenType {
      FI_EN_E8MASSAGE_NOT_SELECTED = 0UL,
      FI_EN_E8SELECTED_MASSAGE_OFF = 1UL,
      FI_EN_E8SELECTED_MASSAGE_ON = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_Massage(most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_Massage::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_Massage();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_Massage& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_MassageIntensity : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_MassageIntensity();
   enum tenType {
      FI_EN_E8MASSAGE_OFF = 0UL,
      FI_EN_E8MASSAGE_INTENSITY_1 = 1UL,
      FI_EN_E8MASSAGE_INTENSITY_2 = 2UL,
      FI_EN_E8MASSAGE_INTENSITY_3 = 3UL,
      FI_EN_E8MASSAGE_INTENSITY_4 = 4UL,
      FI_EN_E8MASSAGE_INTENSITY_5 = 5UL,
      FI_EN_E8MASSAGE_INTENSITY_6 = 6UL,
      FI_EN_E8MASSAGE_INTENSITY_7 = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_MassageIntensity(most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_MassageIntensity::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_MassageIntensity();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_MassageIntensity& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_MassageType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_MassageType();
   enum tenType {
      FI_EN_E8MASSAGE_TYPE_OFF = 0UL,
      FI_EN_E8MASSAGE_TYPE_1 = 1UL,
      FI_EN_E8MASSAGE_TYPE_2 = 2UL,
      FI_EN_E8MASSAGE_TYPE_3 = 3UL,
      FI_EN_E8MASSAGE_TYPE_4 = 4UL,
      FI_EN_E8MASSAGE_TYPE_5 = 5UL,
      FI_EN_E8MASSAGE_TYPE_6 = 6UL,
      FI_EN_E8MASSAGE_TYPE_7 = 7UL,
      FI_EN_E8MASSAGE_TYPE_8 = 8UL,
      FI_EN_E8MASSAGE_TYPE_9 = 9UL,
      FI_EN_E8MASSAGE_TYPE_10 = 10UL,
      FI_EN_E8MASSAGE_TYPE_11 = 11UL,
      FI_EN_E8MASSAGE_TYPE_12 = 12UL,
      FI_EN_E8MASSAGE_TYPE_13 = 13UL,
      FI_EN_E8MASSAGE_TYPE_14 = 14UL,
      FI_EN_E8MASSAGE_TYPE_15 = 15UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_MassageType(most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_MassageType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_MassageType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_MassageType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_UpperShoulderForwardRearward : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_UpperShoulderForwardRearward();
   enum tenType {
      FI_EN_E8HRUSFR_NOT_SELECTED = 0UL,
      FI_EN_E8HRUSFR_SELECTED_NO_MOVEMENT = 1UL,
      FI_EN_E8HRUSFR_FORWARD = 2UL,
      FI_EN_E8HRUSFR_REARWARD = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_UpperShoulderForwardRearward(most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_UpperShoulderForwardRearward::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_UpperShoulderForwardRearward();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRightRearSeatControlStatus_UpperShoulderForwardRearward& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWRearSeatKey : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWRearSeatKey();
   enum tenType {
      FI_EN_E8RSC_VOLUME_UP = 0UL,
      FI_EN_E8RSC_VOLUME_DOWN = 1UL,
      FI_EN_E8RSC_POWER = 2UL,
      FI_EN_E8RSC_SOURCE_SELECT = 3UL,
      FI_EN_E8RSC_PRESET = 4UL,
      FI_EN_E8RSC_SEEK_UP = 5UL,
      FI_EN_E8RSC_SEEK_DOWN = 6UL,
      FI_EN_E8RSC_OK = 7UL,
      FI_EN_E8RSC_NO_KEY_ACTIVE = 255UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWRearSeatKey(most_fi_tcl_e8_GMLnGWRearSeatKey::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWRearSeatKey();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWRearSeatKey& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTeenDriverRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTeenDriverRequest();
   enum tenType {
      FI_EN_E8TD_REQ_NO_ACTION = 0UL,
      FI_EN_E8TD_REQ_VERIFY_TEEN_PIN = 1UL,
      FI_EN_E8TD_REQ_UPDATE_TEEN_PIN = 2UL,
      FI_EN_E8TD_REQ_SET_TEEN_KEY = 3UL,
      FI_EN_E8TD_REQ_CLEAR_TEEN_KEY = 4UL,
      FI_EN_E8TD_REQ_CHECK_FOR_TEEN_KEY = 5UL,
      FI_EN_E8TD_REQ_CLEAR_TEEN_KEYS_AND_PIN = 6UL,
      FI_EN_E8TD_REQ_RESET_REPORT_CARD = 7UL,
      FI_EN_E8TD_REQ_REQUEST_REPORT_CARD_DATA = 8UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTeenDriverRequest(most_fi_tcl_e8_GMLnGWTeenDriverRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTeenDriverRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTeenDriverRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTeenDriverResponse : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTeenDriverResponse();
   enum tenType {
      FI_EN_E8TD_RESP_NO_ACTION = 0UL,
      FI_EN_E8TD_RESP_TEEN_PIN_VERIFIED = 1UL,
      FI_EN_E8TD_RESP_TEEN_PIN_UPDATED = 2UL,
      FI_EN_E8TD_RESP_TEEN_PIN_MISMATCH = 3UL,
      FI_EN_E8TD_RESP_TEEN_KEY_SET_COMPLETE = 4UL,
      FI_EN_E8TD_RESP_TEEN_KEY_CLEAR_COMPLETE = 5UL,
      FI_EN_E8TD_RESP_KEY_NOT_PRESENT = 6UL,
      FI_EN_E8TD_RESP_KEY_DETECTED_AS_TEEN_KEY = 7UL,
      FI_EN_E8TD_RESP_KEY_DETECTED_AS_NOT_TEEN_KEY = 8UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTeenDriverResponse(most_fi_tcl_e8_GMLnGWTeenDriverResponse::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTeenDriverResponse();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTeenDriverResponse& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWSpeedLimitStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWSpeedLimitStatus();
   enum tenType {
      FI_EN_E8SPD_LIMIT_STATUS_NO_ACTION = 0UL,
      FI_EN_E8SPD_LIMIT_STATUS_OFF = 1UL,
      FI_EN_E8SPD_LIMIT_STATUS_ON = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWSpeedLimitStatus(most_fi_tcl_e8_GMLnGWSpeedLimitStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWSpeedLimitStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWSpeedLimitStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWOverspeedWarningCurrentStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWOverspeedWarningCurrentStatus();
   enum tenType {
      FI_EN_E8OVRSPD_WARNING_STATUS_NO_ACTION = 0UL,
      FI_EN_E8OVRSPD_WARNING_STATUS_OFF = 1UL,
      FI_EN_E8OVRSPD_WARNING_STATUS_ON = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWOverspeedWarningCurrentStatus(most_fi_tcl_e8_GMLnGWOverspeedWarningCurrentStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWOverspeedWarningCurrentStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWOverspeedWarningCurrentStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWVideoProcessingFeatureConfigurationStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWVideoProcessingFeatureConfigurationStream();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWVideoProcessingFeatureConfigurationStream& operator=(const most_fi_tcl_GMLnGWVideoProcessingFeatureConfigurationStream& coRef);
   most_fi_tcl_GMLnGWVideoProcessingFeatureConfigurationStream(const most_fi_tcl_GMLnGWVideoProcessingFeatureConfigurationStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tBool bTriggeredVideoRecorderPresent;
   virtual ~most_fi_tcl_GMLnGWVideoProcessingFeatureConfigurationStream();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWVideoProcessingFeatureConfigurationStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWTouchProximity : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWTouchProximity();
   enum tenType {
      FI_EN_E8TP_FAR = 0UL,
      FI_EN_E8TP_NEAR = 1UL,
      FI_EN_E8TP_TOUCH = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWTouchProximity(most_fi_tcl_e8_GMLnGWTouchProximity::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWTouchProximity();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWTouchProximity& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWWifiSSIDDigits1_8 : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWWifiSSIDDigits1_8() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWWifiSSIDDigits1_8& operator=(const most_fi_tcl_GMLnGWWifiSSIDDigits1_8& coRef);
   most_fi_tcl_GMLnGWWifiSSIDDigits1_8(const most_fi_tcl_GMLnGWWifiSSIDDigits1_8& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU8, std::allocator<tU8> >u8Items;

   tU8& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_GMLnGWWifiSSIDDigits1_8();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWWifiSSIDDigits1_8& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWWifiSSIDDigits9_16 : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWWifiSSIDDigits9_16() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWWifiSSIDDigits9_16& operator=(const most_fi_tcl_GMLnGWWifiSSIDDigits9_16& coRef);
   most_fi_tcl_GMLnGWWifiSSIDDigits9_16(const most_fi_tcl_GMLnGWWifiSSIDDigits9_16& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU8, std::allocator<tU8> >u8Items;

   tU8& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_GMLnGWWifiSSIDDigits9_16();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWWifiSSIDDigits9_16& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWWifiSSIDDigits17_24 : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWWifiSSIDDigits17_24() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWWifiSSIDDigits17_24& operator=(const most_fi_tcl_GMLnGWWifiSSIDDigits17_24& coRef);
   most_fi_tcl_GMLnGWWifiSSIDDigits17_24(const most_fi_tcl_GMLnGWWifiSSIDDigits17_24& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU8, std::allocator<tU8> >u8Items;

   tU8& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_GMLnGWWifiSSIDDigits17_24();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWWifiSSIDDigits17_24& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWWifiPassPhraseDigits1_8 : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWWifiPassPhraseDigits1_8() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWWifiPassPhraseDigits1_8& operator=(const most_fi_tcl_GMLnGWWifiPassPhraseDigits1_8& coRef);
   most_fi_tcl_GMLnGWWifiPassPhraseDigits1_8(const most_fi_tcl_GMLnGWWifiPassPhraseDigits1_8& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU8, std::allocator<tU8> >u8Items;

   tU8& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_GMLnGWWifiPassPhraseDigits1_8();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWWifiPassPhraseDigits1_8& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWWifiPassPhraseDigits9_16 : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWWifiPassPhraseDigits9_16() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWWifiPassPhraseDigits9_16& operator=(const most_fi_tcl_GMLnGWWifiPassPhraseDigits9_16& coRef);
   most_fi_tcl_GMLnGWWifiPassPhraseDigits9_16(const most_fi_tcl_GMLnGWWifiPassPhraseDigits9_16& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU8, std::allocator<tU8> >u8Items;

   tU8& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_GMLnGWWifiPassPhraseDigits9_16();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWWifiPassPhraseDigits9_16& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWWifiPassPhraseDigits17_24 : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWWifiPassPhraseDigits17_24() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWWifiPassPhraseDigits17_24& operator=(const most_fi_tcl_GMLnGWWifiPassPhraseDigits17_24& coRef);
   most_fi_tcl_GMLnGWWifiPassPhraseDigits17_24(const most_fi_tcl_GMLnGWWifiPassPhraseDigits17_24& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU8, std::allocator<tU8> >u8Items;

   tU8& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_GMLnGWWifiPassPhraseDigits17_24();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWWifiPassPhraseDigits17_24& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWInfotainmentWifiEnabled : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWInfotainmentWifiEnabled();
   enum tenType {
      FI_EN_E8INFO_WIFI_DISABLED = 0UL,
      FI_EN_E8INFO_WIFI_ENABLED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWInfotainmentWifiEnabled(most_fi_tcl_e8_GMLnGWInfotainmentWifiEnabled::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWInfotainmentWifiEnabled();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWInfotainmentWifiEnabled& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWWifiEncryptionType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWWifiEncryptionType();
   enum tenType {
      FI_EN_E8WIFI_ENC_TYPE_OPEN = 0UL,
      FI_EN_E8WIFI_ENC_TYPE_AES = 1UL,
      FI_EN_E8WIFI_ENC_TYPE_TKIP = 2UL,
      FI_EN_E8WIFI_ENC_TYPE_AES_OR_TKIP = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWWifiEncryptionType(most_fi_tcl_e8_GMLnGWWifiEncryptionType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWWifiEncryptionType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWWifiEncryptionType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWWifiSecurityType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWWifiSecurityType();
   enum tenType {
      FI_EN_E8WIFI_SEC_TYPE_OPEN = 0UL,
      FI_EN_E8WIFI_SEC_TYPE_WEP = 1UL,
      FI_EN_E8WIFI_SEC_TYPE_WPA = 2UL,
      FI_EN_E8WIFI_SEC_TYPE_WPA2 = 3UL,
      FI_EN_E8WIFI_SEC_TYPE_IEEE802_1X = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWWifiSecurityType(most_fi_tcl_e8_GMLnGWWifiSecurityType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWWifiSecurityType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWWifiSecurityType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_GMLnGWWifiAssociationRequest : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_GMLnGWWifiAssociationRequest();
   enum tenType {
      FI_EN_E8WSR_NO_ACTION = 0UL,
      FI_EN_E8WSR_AVAILABLE = 1UL,
      FI_EN_E8WSR_SERVICE_SET_IDENTIFIER_REQUEST = 2UL,
      FI_EN_E8WSR_PASSPHRASE_REQUEST = 3UL,
      FI_EN_E8WSR_SECURITY_TYPE_REQUEST = 4UL,
      FI_EN_E8WSR_ENCRYPTION_REQUEST = 5UL,
      FI_EN_E8WSR_ALL_REQUEST = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_GMLnGWWifiAssociationRequest(most_fi_tcl_e8_GMLnGWWifiAssociationRequest::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_GMLnGWWifiAssociationRequest();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_GMLnGWWifiAssociationRequest& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_GMLnGWWifiStationMediaAccessControlAddress : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_GMLnGWWifiStationMediaAccessControlAddress() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_GMLnGWWifiStationMediaAccessControlAddress& operator=(const most_fi_tcl_GMLnGWWifiStationMediaAccessControlAddress& coRef);
   most_fi_tcl_GMLnGWWifiStationMediaAccessControlAddress(const most_fi_tcl_GMLnGWWifiStationMediaAccessControlAddress& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU8, std::allocator<tU8> >u8Items;

   tU8& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_GMLnGWWifiStationMediaAccessControlAddress();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_GMLnGWWifiStationMediaAccessControlAddress& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACHVACSettingsEventEnum : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACHVACSettingsEventEnum();
   enum tenType {
      FI_EN_E8HVACSETTINGSFUCLASSRESET = 0UL,
      FI_EN_E8FRONT_BLOWERMODEACTIVATION = 1UL,
      FI_EN_E8FRONT_FANSPEEDACTIVATION = 2UL,
      FI_EN_E8FRONT_LEFTSETTEMPACTIVATION = 3UL,
      FI_EN_E8FRONT_RIGHTSETTEMPACTIVATION = 4UL,
      FI_EN_E8REAR_BLOWERMODEACTIVATION = 5UL,
      FI_EN_E8REAR_FANSPEEDACTIVATION = 6UL,
      FI_EN_E8REAR_PANELLOCKEDOUT = 7UL,
      FI_EN_E8REAR_PANELOFF = 8UL,
      FI_EN_E8REAR_SETTEMPACTIVATION = 9UL,
      FI_EN_E8ZONEACTIVATION = 10UL,
      FI_EN_E8AQSACTIVATION = 11UL,
      FI_EN_E8AUTODEFOGACTIVATION = 12UL,
      FI_EN_E8AIRCONDITIONACTIVATION = 13UL,
      FI_EN_E8AIRINLETMODEACTIVATION = 14UL,
      FI_EN_E8FRONT_CLIMATEKEYACTIVATION = 15UL,
      FI_EN_E8FRONT_LEFTTHERMALSEATACTIVATION = 16UL,
      FI_EN_E8FRONT_RIGHTTHERMALSEATACTIVATION = 17UL,
      FI_EN_E8REARDEFOGACTIVATION = 18UL,
      FI_EN_E8HIGHVOLTAGEHEATERACTIVATION = 19UL,
      FI_EN_E8HYBRIDCLIMEFFBARACTIVATION = 20UL,
      FI_EN_E8HYBRIDECONACTIVATION = 21UL,
      FI_EN_E8FRONT2REAR_REARFANSPEEDACTIVATION = 22UL,
      FI_EN_E8FRONT2REAR_REARBLOWERMODEACTIVATION = 23UL,
      FI_EN_E8ACHEATERSTATEACTIVATION = 24UL,
      FI_EN_E8FRONT2REAR_REARSETTEMPACTIVATION = 25UL,
      FI_EN_E8FRONT2REAR_REARPANELLOCKACTIVATION = 26UL,
      FI_EN_E8FRONT2REAR_REARPANELSYNCACTIVATION = 27UL,
      FI_EN_E8FRONT_REARCLIMATEKEYACTIVATION = 28UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACHVACSettingsEventEnum(most_fi_tcl_e8_HVACHVACSettingsEventEnum::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACHVACSettingsEventEnum();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACHVACSettingsEventEnum& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFanSpeedReductionValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFanSpeedReductionValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFanSpeedReductionValue(most_fi_tcl_e8_HVACFanSpeedReductionValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFanSpeedReductionValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFanSpeedReductionValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFront_BlowerModeValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFront_BlowerModeValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8AUTO = 1UL,
      FI_EN_E8DEFROST = 2UL,
      FI_EN_E8HEATER = 3UL,
      FI_EN_E8HEATERDEFROST = 4UL,
      FI_EN_E8BILEVEL = 5UL,
      FI_EN_E8AC = 6UL,
      FI_EN_E8PURGE = 7UL,
      FI_EN_E8ECO = 8UL,
      FI_EN_E8DEFOG = 9UL,
      FI_EN_E8TRILEVEL = 10UL,
      FI_EN_E8EDEFROST = 11UL,
      FI_EN_E8EDEFOG = 12UL,
      FI_EN_E8EPANEL = 13UL,
      FI_EN_E8ETRILEVEL = 14UL,
      FI_EN_E8FLOORDIRECTIONCOMBI = 15UL,
      FI_EN_E8PANELDIRECTIONCOMBI = 16UL,
      FI_EN_E8WINDSCREENDIRECTIONCOMBI = 17UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFront_BlowerModeValue(most_fi_tcl_e8_HVACFront_BlowerModeValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFront_BlowerModeValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFront_BlowerModeValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFront_FanSpeedValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFront_FanSpeedValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8MAN_1 = 1UL,
      FI_EN_E8MAN_2 = 2UL,
      FI_EN_E8MAN_3 = 3UL,
      FI_EN_E8MAN_4 = 4UL,
      FI_EN_E8MAN_5 = 5UL,
      FI_EN_E8MAN_6 = 6UL,
      FI_EN_E8MAN_7 = 7UL,
      FI_EN_E8MAN_8 = 8UL,
      FI_EN_E8MAN_9 = 9UL,
      FI_EN_E8MAN_10 = 10UL,
      FI_EN_E8MAN_11 = 11UL,
      FI_EN_E8MAN_12 = 12UL,
      FI_EN_E8MAN_13 = 13UL,
      FI_EN_E8MAN_14 = 14UL,
      FI_EN_E8MAN_15 = 15UL,
      FI_EN_E8MAN_16 = 16UL,
      FI_EN_E8MAN_17 = 17UL,
      FI_EN_E8MAN_18 = 18UL,
      FI_EN_E8AUTO = 19UL,
      FI_EN_E8AUTO_LO = 20UL,
      FI_EN_E8AUTO_HI = 21UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFront_FanSpeedValue(most_fi_tcl_e8_HVACFront_FanSpeedValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFront_FanSpeedValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFront_FanSpeedValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFront_LeftSetTempValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFront_LeftSetTempValue();
   enum tenType {
      FI_EN_E8165_CELCIUS = 0UL,
      FI_EN_E817_CELCIUS = 1UL,
      FI_EN_E8175_CELCIUS = 2UL,
      FI_EN_E818_CELCIUS = 3UL,
      FI_EN_E8185_CELCIUS = 4UL,
      FI_EN_E819_CELCIUS = 5UL,
      FI_EN_E8195_CELCIUS = 6UL,
      FI_EN_E820_CELCIUS = 7UL,
      FI_EN_E8205_CELCIUS = 8UL,
      FI_EN_E821_CELCIUS = 9UL,
      FI_EN_E8215_CELCIUS = 10UL,
      FI_EN_E822_CELCIUS = 11UL,
      FI_EN_E8225_CELCIUS = 12UL,
      FI_EN_E823_CELCIUS = 13UL,
      FI_EN_E8235_CELCIUS = 14UL,
      FI_EN_E824_CELCIUS = 15UL,
      FI_EN_E8245_CELCIUS = 16UL,
      FI_EN_E825_CELCIUS = 17UL,
      FI_EN_E8255_CELCIUS = 18UL,
      FI_EN_E826_CELCIUS = 19UL,
      FI_EN_E8265_CELCIUS = 20UL,
      FI_EN_E827_CELCIUS = 21UL,
      FI_EN_E8275_CELCIUS = 22UL,
      FI_EN_E828_CELCIUS = 23UL,
      FI_EN_E8285_CELCIUS = 24UL,
      FI_EN_E829_CELCIUS = 25UL,
      FI_EN_E8295_CELCIUS = 26UL,
      FI_EN_E830_CELCIUS = 27UL,
      FI_EN_E8305_CELCIUS = 28UL,
      FI_EN_E831_CELCIUS = 29UL,
      FI_EN_E8315_CELCIUS = 30UL,
      FI_EN_E8LOW = 31UL,
      FI_EN_E8HI = 32UL,
      FI_EN_E816_CELCIUS = 33UL,
      FI_EN_E8155_CELCIUS = 34UL,
      FI_EN_E815_CELCIUS = 35UL,
      FI_EN_E8145_CELCIUS = 36UL,
      FI_EN_E814_CELCIUS = 37UL,
      FI_EN_E8OFF = 38UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFront_LeftSetTempValue(most_fi_tcl_e8_HVACFront_LeftSetTempValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFront_LeftSetTempValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFront_LeftSetTempValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFront_RightSetTempValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFront_RightSetTempValue();
   enum tenType {
      FI_EN_E8165_CELCIUS = 0UL,
      FI_EN_E817_CELCIUS = 1UL,
      FI_EN_E8175_CELCIUS = 2UL,
      FI_EN_E818_CELCIUS = 3UL,
      FI_EN_E8185_CELCIUS = 4UL,
      FI_EN_E819_CELCIUS = 5UL,
      FI_EN_E8195_CELCIUS = 6UL,
      FI_EN_E820_CELCIUS = 7UL,
      FI_EN_E8205_CELCIUS = 8UL,
      FI_EN_E821_CELCIUS = 9UL,
      FI_EN_E8215_CELCIUS = 10UL,
      FI_EN_E822_CELCIUS = 11UL,
      FI_EN_E8225_CELCIUS = 12UL,
      FI_EN_E823_CELCIUS = 13UL,
      FI_EN_E8235_CELCIUS = 14UL,
      FI_EN_E824_CELCIUS = 15UL,
      FI_EN_E8245_CELCIUS = 16UL,
      FI_EN_E825_CELCIUS = 17UL,
      FI_EN_E8255_CELCIUS = 18UL,
      FI_EN_E826_CELCIUS = 19UL,
      FI_EN_E8265_CELCIUS = 20UL,
      FI_EN_E827_CELCIUS = 21UL,
      FI_EN_E8275_CELCIUS = 22UL,
      FI_EN_E828_CELCIUS = 23UL,
      FI_EN_E8285_CELCIUS = 24UL,
      FI_EN_E829_CELCIUS = 25UL,
      FI_EN_E8295_CELCIUS = 26UL,
      FI_EN_E830_CELCIUS = 27UL,
      FI_EN_E8305_CELCIUS = 28UL,
      FI_EN_E831_CELCIUS = 29UL,
      FI_EN_E8315_CELCIUS = 30UL,
      FI_EN_E8LOW = 31UL,
      FI_EN_E8HI = 32UL,
      FI_EN_E816_CELCIUS = 33UL,
      FI_EN_E8155_CELCIUS = 34UL,
      FI_EN_E815_CELCIUS = 35UL,
      FI_EN_E8145_CELCIUS = 36UL,
      FI_EN_E814_CELCIUS = 37UL,
      FI_EN_E8OFF = 38UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFront_RightSetTempValue(most_fi_tcl_e8_HVACFront_RightSetTempValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFront_RightSetTempValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFront_RightSetTempValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACRear_BlowerModeValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACRear_BlowerModeValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8AUTO = 1UL,
      FI_EN_E8LOWER = 2UL,
      FI_EN_E8BILEVEL = 3UL,
      FI_EN_E8UPPER = 4UL,
      FI_EN_E8LOWERROOF = 5UL,
      FI_EN_E8TRILEVEL = 6UL,
      FI_EN_E8UPPERROOF = 7UL,
      FI_EN_E8ROOF = 8UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACRear_BlowerModeValue(most_fi_tcl_e8_HVACRear_BlowerModeValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACRear_BlowerModeValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACRear_BlowerModeValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACRear_FanSpeedValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACRear_FanSpeedValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8MAN_1 = 1UL,
      FI_EN_E8MAN_2 = 2UL,
      FI_EN_E8MAN_3 = 3UL,
      FI_EN_E8MAN_4 = 4UL,
      FI_EN_E8AUTO = 5UL,
      FI_EN_E8MAN_5 = 6UL,
      FI_EN_E8MAN_6 = 7UL,
      FI_EN_E8AUTO_LO = 8UL,
      FI_EN_E8AUTO_HI = 9UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACRear_FanSpeedValue(most_fi_tcl_e8_HVACRear_FanSpeedValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACRear_FanSpeedValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACRear_FanSpeedValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACRear_SetPointSetTempValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACRear_SetPointSetTempValue();
   enum tenType {
      FI_EN_E8165_CELCIUS = 0UL,
      FI_EN_E817_CELCIUS = 1UL,
      FI_EN_E8175_CELCIUS = 2UL,
      FI_EN_E818_CELCIUS = 3UL,
      FI_EN_E8185_CELCIUS = 4UL,
      FI_EN_E819_CELCIUS = 5UL,
      FI_EN_E8195_CELCIUS = 6UL,
      FI_EN_E820_CELCIUS = 7UL,
      FI_EN_E8205_CELCIUS = 8UL,
      FI_EN_E821_CELCIUS = 9UL,
      FI_EN_E8215_CELCIUS = 10UL,
      FI_EN_E822_CELCIUS = 11UL,
      FI_EN_E8225_CELCIUS = 12UL,
      FI_EN_E823_CELCIUS = 13UL,
      FI_EN_E8235_CELCIUS = 14UL,
      FI_EN_E824_CELCIUS = 15UL,
      FI_EN_E8245_CELCIUS = 16UL,
      FI_EN_E825_CELCIUS = 17UL,
      FI_EN_E8255_CELCIUS = 18UL,
      FI_EN_E826_CELCIUS = 19UL,
      FI_EN_E8265_CELCIUS = 20UL,
      FI_EN_E827_CELCIUS = 21UL,
      FI_EN_E8275_CELCIUS = 22UL,
      FI_EN_E828_CELCIUS = 23UL,
      FI_EN_E8285_CELCIUS = 24UL,
      FI_EN_E829_CELCIUS = 25UL,
      FI_EN_E8295_CELCIUS = 26UL,
      FI_EN_E830_CELCIUS = 27UL,
      FI_EN_E8305_CELCIUS = 28UL,
      FI_EN_E831_CELCIUS = 29UL,
      FI_EN_E8315_CELCIUS = 30UL,
      FI_EN_E8LOW = 31UL,
      FI_EN_E8HI = 32UL,
      FI_EN_E816_CELCIUS = 33UL,
      FI_EN_E8155_CELCIUS = 34UL,
      FI_EN_E815_CELCIUS = 35UL,
      FI_EN_E8145_CELCIUS = 36UL,
      FI_EN_E814_CELCIUS = 37UL,
      FI_EN_E8OFF = 38UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACRear_SetPointSetTempValue(most_fi_tcl_e8_HVACRear_SetPointSetTempValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACRear_SetPointSetTempValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACRear_SetPointSetTempValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACZoneValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACZoneValue();
   enum tenType {
      FI_EN_E8SINGLE_ZONE_CLIMATE_CONTROL = 0UL,
      FI_EN_E8DUAL_ZONE_CLIMATE_CONTROL = 1UL,
      FI_EN_E8TRI_ZONE_CLIMATE_CONTROL = 2UL,
      FI_EN_E8CLIMATE_ZONES_LINKED = 3UL,
      FI_EN_E8PASSENGER_CLIMATE_ZONE_ACTIVE = 4UL,
      FI_EN_E8REAR_CLIMATE_ZONE_ACTIVE = 5UL,
      FI_EN_E8REAR_CLIMATE_ZONE_LINKED = 6UL,
      FI_EN_E8REAR_CLIMATE_ZONE_OFF = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACZoneValue(most_fi_tcl_e8_HVACZoneValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACZoneValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACZoneValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACAirQualitySensorValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACAirQualitySensorValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACAirQualitySensorValue(most_fi_tcl_e8_HVACAirQualitySensorValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACAirQualitySensorValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACAirQualitySensorValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACAutoDefogValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACAutoDefogValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACAutoDefogValue(most_fi_tcl_e8_HVACAutoDefogValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACAutoDefogValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACAutoDefogValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACAirConditionValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACAirConditionValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8AUTO_AC_ON = 2UL,
      FI_EN_E8COMFORT_AC_ON = 3UL,
      FI_EN_E8ECO_AC_ON = 4UL,
      FI_EN_E8BATTERY_AC_ON = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACAirConditionValue(most_fi_tcl_e8_HVACAirConditionValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACAirConditionValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACAirConditionValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACAirInletValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACAirInletValue();
   enum tenType {
      FI_EN_E8FRESH_AIR_ = 0UL,
      FI_EN_E8RECIRCULATION_ = 1UL,
      FI_EN_E8AUTOAQS_RECIRCULATION_ = 2UL,
      FI_EN_E8INACTIVE = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACAirInletValue(most_fi_tcl_e8_HVACAirInletValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACAirInletValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACAirInletValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACMaximum_Front_FanSpeedValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACMaximum_Front_FanSpeedValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8MAN_1 = 1UL,
      FI_EN_E8MAN_2 = 2UL,
      FI_EN_E8MAN_3 = 3UL,
      FI_EN_E8MAN_4 = 4UL,
      FI_EN_E8MAN_5 = 5UL,
      FI_EN_E8MAN_6 = 6UL,
      FI_EN_E8MAN_7 = 7UL,
      FI_EN_E8MAN_8 = 8UL,
      FI_EN_E8MAN_9 = 9UL,
      FI_EN_E8MAN_10 = 10UL,
      FI_EN_E8MAN_11 = 11UL,
      FI_EN_E8MAN_12 = 12UL,
      FI_EN_E8MAN_13 = 13UL,
      FI_EN_E8MAN_14 = 14UL,
      FI_EN_E8MAN_15 = 15UL,
      FI_EN_E8MAN_16 = 16UL,
      FI_EN_E8MAN_17 = 17UL,
      FI_EN_E8MAN_18 = 18UL,
      FI_EN_E8AUTO = 19UL,
      FI_EN_E8AUTO_LO = 20UL,
      FI_EN_E8AUTO_HI = 21UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACMaximum_Front_FanSpeedValue(most_fi_tcl_e8_HVACMaximum_Front_FanSpeedValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACMaximum_Front_FanSpeedValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACMaximum_Front_FanSpeedValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFront_LeftThermalSeatValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFront_LeftThermalSeatValue();
   enum tenType {
      FI_EN_E8HEATEDSEATOFF = 0UL,
      FI_EN_E8HEATEDSEATLEVEL1 = 1UL,
      FI_EN_E8HEATEDSEATLEVEL2 = 2UL,
      FI_EN_E8HEATEDSEATLEVEL3 = 3UL,
      FI_EN_E8HEATEDSEATAUTO = 4UL,
      FI_EN_E8VENTILATEDSEATOFF = 5UL,
      FI_EN_E8VENTILATEDSEATLEVEL1 = 6UL,
      FI_EN_E8VENTILATEDSEATLEVEL2 = 7UL,
      FI_EN_E8VENTILATEDSEATLEVEL3 = 8UL,
      FI_EN_E8VENTILATEDSEATAUTO = 9UL,
      FI_EN_E8COOLEDSEATOFF = 10UL,
      FI_EN_E8COOLEDSEATLEVEL1 = 11UL,
      FI_EN_E8COOLEDSEATLEVEL2 = 12UL,
      FI_EN_E8COOLEDSEATLEVEL3 = 13UL,
      FI_EN_E8COOLEDSEATAUTO = 14UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFront_LeftThermalSeatValue(most_fi_tcl_e8_HVACFront_LeftThermalSeatValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFront_LeftThermalSeatValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFront_LeftThermalSeatValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFront_RightThermalSeatValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFront_RightThermalSeatValue();
   enum tenType {
      FI_EN_E8HEATEDSEATOFF = 0UL,
      FI_EN_E8HEATEDSEATLEVEL1 = 1UL,
      FI_EN_E8HEATEDSEATLEVEL2 = 2UL,
      FI_EN_E8HEATEDSEATLEVEL3 = 3UL,
      FI_EN_E8HEATEDSEATAUTO = 4UL,
      FI_EN_E8VENTILATEDSEATOFF = 5UL,
      FI_EN_E8VENTILATEDSEATLEVEL1 = 6UL,
      FI_EN_E8VENTILATEDSEATLEVEL2 = 7UL,
      FI_EN_E8VENTILATEDSEATLEVEL3 = 8UL,
      FI_EN_E8VENTILATEDSEATAUTO = 9UL,
      FI_EN_E8COOLEDSEATOFF = 10UL,
      FI_EN_E8COOLEDSEATLEVEL1 = 11UL,
      FI_EN_E8COOLEDSEATLEVEL2 = 12UL,
      FI_EN_E8COOLEDSEATLEVEL3 = 13UL,
      FI_EN_E8COOLEDSEATAUTO = 14UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFront_RightThermalSeatValue(most_fi_tcl_e8_HVACFront_RightThermalSeatValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFront_RightThermalSeatValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFront_RightThermalSeatValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACRear_DefogValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACRear_DefogValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACRear_DefogValue(most_fi_tcl_e8_HVACRear_DefogValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACRear_DefogValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACRear_DefogValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACHybridEconModeValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACHybridEconModeValue();
   enum tenType {
      FI_EN_E8COMFORT = 0UL,
      FI_EN_E8ECONOMY = 1UL,
      FI_EN_E8FAN_ONLY = 2UL,
      FI_EN_E8ACTIVE = 3UL,
      FI_EN_E8AUTO = 4UL,
      FI_EN_E8BATTERY_AC_ON = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACHybridEconModeValue(most_fi_tcl_e8_HVACHybridEconModeValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACHybridEconModeValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACHybridEconModeValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACHighVoltageHeaterValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACHighVoltageHeaterValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACHighVoltageHeaterValue(most_fi_tcl_e8_HVACHighVoltageHeaterValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACHighVoltageHeaterValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACHighVoltageHeaterValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACRear_PanelLockValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACRear_PanelLockValue();
   enum tenType {
      FI_EN_E8UNLOCKED = 0UL,
      FI_EN_E8LOCKED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACRear_PanelLockValue(most_fi_tcl_e8_HVACRear_PanelLockValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACRear_PanelLockValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACRear_PanelLockValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACRear_PanelSyncValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACRear_PanelSyncValue();
   enum tenType {
      FI_EN_E8UNSYNCED = 0UL,
      FI_EN_E8SYNCED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACRear_PanelSyncValue(most_fi_tcl_e8_HVACRear_PanelSyncValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACRear_PanelSyncValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACRear_PanelSyncValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_HybridFanOnlyValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_HybridFanOnlyValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_HybridFanOnlyValue(most_fi_tcl_e8_HVACFTouchButton_HybridFanOnlyValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_HybridFanOnlyValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_HybridFanOnlyValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_HybridEconValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_HybridEconValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_HybridEconValue(most_fi_tcl_e8_HVACFTouchButton_HybridEconValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_HybridEconValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_HybridEconValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_HybridComfortValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_HybridComfortValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_HybridComfortValue(most_fi_tcl_e8_HVACFTouchButton_HybridComfortValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_HybridComfortValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_HybridComfortValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_LFAutoHeatedSeatValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_LFAutoHeatedSeatValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_LFAutoHeatedSeatValue(most_fi_tcl_e8_HVACFTouchButton_LFAutoHeatedSeatValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_LFAutoHeatedSeatValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_LFAutoHeatedSeatValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_RFAutoHeatedSeatValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_RFAutoHeatedSeatValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_RFAutoHeatedSeatValue(most_fi_tcl_e8_HVACFTouchButton_RFAutoHeatedSeatValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_RFAutoHeatedSeatValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_RFAutoHeatedSeatValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_HybridHVHeaterValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_HybridHVHeaterValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_HybridHVHeaterValue(most_fi_tcl_e8_HVACFTouchButton_HybridHVHeaterValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_HybridHVHeaterValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_HybridHVHeaterValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_ACValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_ACValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_ACValue(most_fi_tcl_e8_HVACFTouchButton_ACValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_ACValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_ACValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_RecircValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_RecircValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_RecircValue(most_fi_tcl_e8_HVACFTouchButton_RecircValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_RecircValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_RecircValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_AutoRecircValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_AutoRecircValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_AutoRecircValue(most_fi_tcl_e8_HVACFTouchButton_AutoRecircValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_AutoRecircValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_AutoRecircValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_Mode_ACValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_Mode_ACValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_Mode_ACValue(most_fi_tcl_e8_HVACFTouchButton_Mode_ACValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_Mode_ACValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_Mode_ACValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_Mode_BiLevelValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_Mode_BiLevelValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_Mode_BiLevelValue(most_fi_tcl_e8_HVACFTouchButton_Mode_BiLevelValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_Mode_BiLevelValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_Mode_BiLevelValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_Mode_HeaterDefrostValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_Mode_HeaterDefrostValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_Mode_HeaterDefrostValue(most_fi_tcl_e8_HVACFTouchButton_Mode_HeaterDefrostValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_Mode_HeaterDefrostValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_Mode_HeaterDefrostValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_Mode_HeaterValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_Mode_HeaterValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_Mode_HeaterValue(most_fi_tcl_e8_HVACFTouchButton_Mode_HeaterValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_Mode_HeaterValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_Mode_HeaterValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_FrontFanIncValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_FrontFanIncValue();
   enum tenType {
      FI_EN_E8UNAVAILABLE = 0UL,
      FI_EN_E8AVAILABLE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_FrontFanIncValue(most_fi_tcl_e8_HVACFTouchButton_FrontFanIncValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_FrontFanIncValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_FrontFanIncValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_FrontFanDecValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_FrontFanDecValue();
   enum tenType {
      FI_EN_E8UNAVAILABLE = 0UL,
      FI_EN_E8AVAILABLE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_FrontFanDecValue(most_fi_tcl_e8_HVACFTouchButton_FrontFanDecValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_FrontFanDecValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_FrontFanDecValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_AutoValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_AutoValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_AutoValue(most_fi_tcl_e8_HVACFTouchButton_AutoValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_AutoValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_AutoValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_SyncValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_SyncValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_SyncValue(most_fi_tcl_e8_HVACFTouchButton_SyncValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_SyncValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_SyncValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_ECOACValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_ECOACValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_ECOACValue(most_fi_tcl_e8_HVACFTouchButton_ECOACValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_ECOACValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_ECOACValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_FreshAirValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_FreshAirValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_FreshAirValue(most_fi_tcl_e8_HVACFTouchButton_FreshAirValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_FreshAirValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_FreshAirValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_Mode_WindshieldValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_Mode_WindshieldValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_Mode_WindshieldValue(most_fi_tcl_e8_HVACFTouchButton_Mode_WindshieldValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_Mode_WindshieldValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_Mode_WindshieldValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_Mode_DefrostValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_Mode_DefrostValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_Mode_DefrostValue(most_fi_tcl_e8_HVACFTouchButton_Mode_DefrostValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_Mode_DefrostValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_Mode_DefrostValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACACHeaterStateValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACACHeaterStateValue();
   enum tenType {
      FI_EN_E8ACOFF_HEATEROFF = 0UL,
      FI_EN_E8ACOFF_HEATERON = 1UL,
      FI_EN_E8ACON_HEATEROFF = 2UL,
      FI_EN_E8ACON_HEATERON = 3UL,
      FI_EN_E8AC_ON = 4UL,
      FI_EN_E8AC_OFF = 5UL,
      FI_EN_E8HEATER_ON = 6UL,
      FI_EN_E8HEATER_OFF = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACACHeaterStateValue(most_fi_tcl_e8_HVACACHeaterStateValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACACHeaterStateValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACACHeaterStateValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_LeftSetTempIncValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_LeftSetTempIncValue();
   enum tenType {
      FI_EN_E8UNAVAILABLE = 0UL,
      FI_EN_E8AVAILABLE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_LeftSetTempIncValue(most_fi_tcl_e8_HVACFTouchButton_LeftSetTempIncValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_LeftSetTempIncValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_LeftSetTempIncValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_LeftSetTempDecValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_LeftSetTempDecValue();
   enum tenType {
      FI_EN_E8UNAVAILABLE = 0UL,
      FI_EN_E8AVAILABLE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_LeftSetTempDecValue(most_fi_tcl_e8_HVACFTouchButton_LeftSetTempDecValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_LeftSetTempDecValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_LeftSetTempDecValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_RightSetTempIncValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_RightSetTempIncValue();
   enum tenType {
      FI_EN_E8UNAVAILABLE = 0UL,
      FI_EN_E8AVAILABLE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_RightSetTempIncValue(most_fi_tcl_e8_HVACFTouchButton_RightSetTempIncValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_RightSetTempIncValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_RightSetTempIncValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_RightSetTempDecValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_RightSetTempDecValue();
   enum tenType {
      FI_EN_E8UNAVAILABLE = 0UL,
      FI_EN_E8AVAILABLE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_RightSetTempDecValue(most_fi_tcl_e8_HVACFTouchButton_RightSetTempDecValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_RightSetTempDecValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_RightSetTempDecValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_RearDefogValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_RearDefogValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_RearDefogValue(most_fi_tcl_e8_HVACFTouchButton_RearDefogValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_RearDefogValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_RearDefogValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_RearMode_HeaterValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_RearMode_HeaterValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_RearMode_HeaterValue(most_fi_tcl_e8_HVACFTouchButton_RearMode_HeaterValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_RearMode_HeaterValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_RearMode_HeaterValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_RearMode_BiLevelValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_RearMode_BiLevelValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_RearMode_BiLevelValue(most_fi_tcl_e8_HVACFTouchButton_RearMode_BiLevelValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_RearMode_BiLevelValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_RearMode_BiLevelValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_RearMode_ACValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_RearMode_ACValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_RearMode_ACValue(most_fi_tcl_e8_HVACFTouchButton_RearMode_ACValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_RearMode_ACValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_RearMode_ACValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_RearFanIncValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_RearFanIncValue();
   enum tenType {
      FI_EN_E8UNAVAILABLE = 0UL,
      FI_EN_E8AVAILABLE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_RearFanIncValue(most_fi_tcl_e8_HVACFTouchButton_RearFanIncValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_RearFanIncValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_RearFanIncValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_RearFanDecValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_RearFanDecValue();
   enum tenType {
      FI_EN_E8UNAVAILABLE = 0UL,
      FI_EN_E8AVAILABLE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_RearFanDecValue(most_fi_tcl_e8_HVACFTouchButton_RearFanDecValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_RearFanDecValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_RearFanDecValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_RearSetTempIncValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_RearSetTempIncValue();
   enum tenType {
      FI_EN_E8UNAVAILABLE = 0UL,
      FI_EN_E8AVAILABLE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_RearSetTempIncValue(most_fi_tcl_e8_HVACFTouchButton_RearSetTempIncValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_RearSetTempIncValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_RearSetTempIncValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_RearSetTempDecValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_RearSetTempDecValue();
   enum tenType {
      FI_EN_E8UNAVAILABLE = 0UL,
      FI_EN_E8AVAILABLE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_RearSetTempDecValue(most_fi_tcl_e8_HVACFTouchButton_RearSetTempDecValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_RearSetTempDecValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_RearSetTempDecValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_RearPanelLockIncValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_RearPanelLockIncValue();
   enum tenType {
      FI_EN_E8UNAVAILABLE = 0UL,
      FI_EN_E8AVAILABLE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_RearPanelLockIncValue(most_fi_tcl_e8_HVACFTouchButton_RearPanelLockIncValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_RearPanelLockIncValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_RearPanelLockIncValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_RearPanelSyncValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_RearPanelSyncValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_RearPanelSyncValue(most_fi_tcl_e8_HVACFTouchButton_RearPanelSyncValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_RearPanelSyncValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_RearPanelSyncValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_RearPanelPowerValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_RearPanelPowerValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_RearPanelPowerValue(most_fi_tcl_e8_HVACFTouchButton_RearPanelPowerValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_RearPanelPowerValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_RearPanelPowerValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_RearPanelAutoValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_RearPanelAutoValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_RearPanelAutoValue(most_fi_tcl_e8_HVACFTouchButton_RearPanelAutoValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_RearPanelAutoValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_RearPanelAutoValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_ACIncValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_ACIncValue();
   enum tenType {
      FI_EN_E8UNAVAILABLE = 0UL,
      FI_EN_E8AVAILABLE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_ACIncValue(most_fi_tcl_e8_HVACFTouchButton_ACIncValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_ACIncValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_ACIncValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_RecircIncValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_RecircIncValue();
   enum tenType {
      FI_EN_E8UNAVAILABLE = 0UL,
      FI_EN_E8AVAILABLE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_RecircIncValue(most_fi_tcl_e8_HVACFTouchButton_RecircIncValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_RecircIncValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_RecircIncValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACMaximum_Rear_FanSpeedValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACMaximum_Rear_FanSpeedValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8MAN_1 = 1UL,
      FI_EN_E8MAN_2 = 2UL,
      FI_EN_E8MAN_3 = 3UL,
      FI_EN_E8MAN_4 = 4UL,
      FI_EN_E8AUTO = 5UL,
      FI_EN_E8MAN_5 = 6UL,
      FI_EN_E8MAN_6 = 7UL,
      FI_EN_E8AUTO_LO = 8UL,
      FI_EN_E8AUTO_HI = 9UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACMaximum_Rear_FanSpeedValue(most_fi_tcl_e8_HVACMaximum_Rear_FanSpeedValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACMaximum_Rear_FanSpeedValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACMaximum_Rear_FanSpeedValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACMaximum_Front_SetTempsValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACMaximum_Front_SetTempsValue();
   enum tenType {
      FI_EN_E8165_CELCIUS = 0UL,
      FI_EN_E817_CELCIUS = 1UL,
      FI_EN_E8175_CELCIUS = 2UL,
      FI_EN_E818_CELCIUS = 3UL,
      FI_EN_E8185_CELCIUS = 4UL,
      FI_EN_E819_CELCIUS = 5UL,
      FI_EN_E8195_CELCIUS = 6UL,
      FI_EN_E820_CELCIUS = 7UL,
      FI_EN_E8205_CELCIUS = 8UL,
      FI_EN_E821_CELCIUS = 9UL,
      FI_EN_E8215_CELCIUS = 10UL,
      FI_EN_E822_CELCIUS = 11UL,
      FI_EN_E8225_CELCIUS = 12UL,
      FI_EN_E823_CELCIUS = 13UL,
      FI_EN_E8235_CELCIUS = 14UL,
      FI_EN_E824_CELCIUS = 15UL,
      FI_EN_E8245_CELCIUS = 16UL,
      FI_EN_E825_CELCIUS = 17UL,
      FI_EN_E8255_CELCIUS = 18UL,
      FI_EN_E826_CELCIUS = 19UL,
      FI_EN_E8265_CELCIUS = 20UL,
      FI_EN_E827_CELCIUS = 21UL,
      FI_EN_E8275_CELCIUS = 22UL,
      FI_EN_E828_CELCIUS = 23UL,
      FI_EN_E8285_CELCIUS = 24UL,
      FI_EN_E829_CELCIUS = 25UL,
      FI_EN_E8295_CELCIUS = 26UL,
      FI_EN_E830_CELCIUS = 27UL,
      FI_EN_E8305_CELCIUS = 28UL,
      FI_EN_E831_CELCIUS = 29UL,
      FI_EN_E8315_CELCIUS = 30UL,
      FI_EN_E8LOW = 31UL,
      FI_EN_E8HI = 32UL,
      FI_EN_E816_CELCIUS = 33UL,
      FI_EN_E8155_CELCIUS = 34UL,
      FI_EN_E815_CELCIUS = 35UL,
      FI_EN_E8145_CELCIUS = 36UL,
      FI_EN_E814_CELCIUS = 37UL,
      FI_EN_E8OFF = 38UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACMaximum_Front_SetTempsValue(most_fi_tcl_e8_HVACMaximum_Front_SetTempsValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACMaximum_Front_SetTempsValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACMaximum_Front_SetTempsValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACMaximum_Rear_SetTempsValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACMaximum_Rear_SetTempsValue();
   enum tenType {
      FI_EN_E8165_CELCIUS = 0UL,
      FI_EN_E817_CELCIUS = 1UL,
      FI_EN_E8175_CELCIUS = 2UL,
      FI_EN_E818_CELCIUS = 3UL,
      FI_EN_E8185_CELCIUS = 4UL,
      FI_EN_E819_CELCIUS = 5UL,
      FI_EN_E8195_CELCIUS = 6UL,
      FI_EN_E820_CELCIUS = 7UL,
      FI_EN_E8205_CELCIUS = 8UL,
      FI_EN_E821_CELCIUS = 9UL,
      FI_EN_E8215_CELCIUS = 10UL,
      FI_EN_E822_CELCIUS = 11UL,
      FI_EN_E8225_CELCIUS = 12UL,
      FI_EN_E823_CELCIUS = 13UL,
      FI_EN_E8235_CELCIUS = 14UL,
      FI_EN_E824_CELCIUS = 15UL,
      FI_EN_E8245_CELCIUS = 16UL,
      FI_EN_E825_CELCIUS = 17UL,
      FI_EN_E8255_CELCIUS = 18UL,
      FI_EN_E826_CELCIUS = 19UL,
      FI_EN_E8265_CELCIUS = 20UL,
      FI_EN_E827_CELCIUS = 21UL,
      FI_EN_E8275_CELCIUS = 22UL,
      FI_EN_E828_CELCIUS = 23UL,
      FI_EN_E8285_CELCIUS = 24UL,
      FI_EN_E829_CELCIUS = 25UL,
      FI_EN_E8295_CELCIUS = 26UL,
      FI_EN_E830_CELCIUS = 27UL,
      FI_EN_E8305_CELCIUS = 28UL,
      FI_EN_E831_CELCIUS = 29UL,
      FI_EN_E8315_CELCIUS = 30UL,
      FI_EN_E8LOW = 31UL,
      FI_EN_E8HI = 32UL,
      FI_EN_E816_CELCIUS = 33UL,
      FI_EN_E8155_CELCIUS = 34UL,
      FI_EN_E815_CELCIUS = 35UL,
      FI_EN_E8145_CELCIUS = 36UL,
      FI_EN_E814_CELCIUS = 37UL,
      FI_EN_E8OFF = 38UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACMaximum_Rear_SetTempsValue(most_fi_tcl_e8_HVACMaximum_Rear_SetTempsValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACMaximum_Rear_SetTempsValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACMaximum_Rear_SetTempsValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_ClimateModeIncValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_ClimateModeIncValue();
   enum tenType {
      FI_EN_E8UNAVAILABLE = 0UL,
      FI_EN_E8AVAILABLE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_ClimateModeIncValue(most_fi_tcl_e8_HVACFTouchButton_ClimateModeIncValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_ClimateModeIncValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_ClimateModeIncValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFront_Right_BlowerModeValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFront_Right_BlowerModeValue();
   enum tenType {
      FI_EN_E8OFF = 0UL,
      FI_EN_E8AUTO = 1UL,
      FI_EN_E8DEFROST = 2UL,
      FI_EN_E8HEATER = 3UL,
      FI_EN_E8HEATERDEFROST = 4UL,
      FI_EN_E8BILEVEL = 5UL,
      FI_EN_E8AC = 6UL,
      FI_EN_E8PURGE = 7UL,
      FI_EN_E8ECO = 8UL,
      FI_EN_E8DEFOG = 9UL,
      FI_EN_E8TRILEVEL = 10UL,
      FI_EN_E8EDEFROST = 11UL,
      FI_EN_E8EDEFOG = 12UL,
      FI_EN_E8EPANEL = 13UL,
      FI_EN_E8ETRILEVEL = 14UL,
      FI_EN_E8FLOORDIRECTIONCOMBI = 15UL,
      FI_EN_E8PANELDIRECTIONCOMBI = 16UL,
      FI_EN_E8WINDSCREENDIRECTIONCOMBI = 17UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFront_Right_BlowerModeValue(most_fi_tcl_e8_HVACFront_Right_BlowerModeValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFront_Right_BlowerModeValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFront_Right_BlowerModeValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_Right_Mode_ACValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_Right_Mode_ACValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_Right_Mode_ACValue(most_fi_tcl_e8_HVACFTouchButton_Right_Mode_ACValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_Right_Mode_ACValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_Right_Mode_ACValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_Right_Mode_BiLevelValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_Right_Mode_BiLevelValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_Right_Mode_BiLevelValue(most_fi_tcl_e8_HVACFTouchButton_Right_Mode_BiLevelValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_Right_Mode_BiLevelValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_Right_Mode_BiLevelValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_Right_Mode_HeaterValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_Right_Mode_HeaterValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8UNAVAILABLE = 2UL,
      FI_EN_E8INVALID = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_Right_Mode_HeaterValue(most_fi_tcl_e8_HVACFTouchButton_Right_Mode_HeaterValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_Right_Mode_HeaterValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_Right_Mode_HeaterValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_Right_RearSetTempIncValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_Right_RearSetTempIncValue();
   enum tenType {
      FI_EN_E8UNAVAILABLE = 0UL,
      FI_EN_E8AVAILABLE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_Right_RearSetTempIncValue(most_fi_tcl_e8_HVACFTouchButton_Right_RearSetTempIncValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_Right_RearSetTempIncValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_Right_RearSetTempIncValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_Right_RearSetTempDecValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_Right_RearSetTempDecValue();
   enum tenType {
      FI_EN_E8UNAVAILABLE = 0UL,
      FI_EN_E8AVAILABLE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_Right_RearSetTempDecValue(most_fi_tcl_e8_HVACFTouchButton_Right_RearSetTempDecValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_Right_RearSetTempDecValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_Right_RearSetTempDecValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_RearModeIncValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_RearModeIncValue();
   enum tenType {
      FI_EN_E8UNAVAILABLE = 0UL,
      FI_EN_E8AVAILABLE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_RearModeIncValue(most_fi_tcl_e8_HVACFTouchButton_RearModeIncValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_RearModeIncValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_RearModeIncValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACFTouchButton_RearModeDecValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACFTouchButton_RearModeDecValue();
   enum tenType {
      FI_EN_E8UNAVAILABLE = 0UL,
      FI_EN_E8AVAILABLE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACFTouchButton_RearModeDecValue(most_fi_tcl_e8_HVACFTouchButton_RearModeDecValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACFTouchButton_RearModeDecValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACFTouchButton_RearModeDecValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACRight_Rear_SetPointSetTempValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACRight_Rear_SetPointSetTempValue();
   enum tenType {
      FI_EN_E8165_CELCIUS = 0UL,
      FI_EN_E817_CELCIUS = 1UL,
      FI_EN_E8175_CELCIUS = 2UL,
      FI_EN_E818_CELCIUS = 3UL,
      FI_EN_E8185_CELCIUS = 4UL,
      FI_EN_E819_CELCIUS = 5UL,
      FI_EN_E8195_CELCIUS = 6UL,
      FI_EN_E820_CELCIUS = 7UL,
      FI_EN_E8205_CELCIUS = 8UL,
      FI_EN_E821_CELCIUS = 9UL,
      FI_EN_E8215_CELCIUS = 10UL,
      FI_EN_E822_CELCIUS = 11UL,
      FI_EN_E8225_CELCIUS = 12UL,
      FI_EN_E823_CELCIUS = 13UL,
      FI_EN_E8235_CELCIUS = 14UL,
      FI_EN_E824_CELCIUS = 15UL,
      FI_EN_E8245_CELCIUS = 16UL,
      FI_EN_E825_CELCIUS = 17UL,
      FI_EN_E8255_CELCIUS = 18UL,
      FI_EN_E826_CELCIUS = 19UL,
      FI_EN_E8265_CELCIUS = 20UL,
      FI_EN_E827_CELCIUS = 21UL,
      FI_EN_E8275_CELCIUS = 22UL,
      FI_EN_E828_CELCIUS = 23UL,
      FI_EN_E8285_CELCIUS = 24UL,
      FI_EN_E829_CELCIUS = 25UL,
      FI_EN_E8295_CELCIUS = 26UL,
      FI_EN_E830_CELCIUS = 27UL,
      FI_EN_E8305_CELCIUS = 28UL,
      FI_EN_E831_CELCIUS = 29UL,
      FI_EN_E8315_CELCIUS = 30UL,
      FI_EN_E8LOW = 31UL,
      FI_EN_E8HI = 32UL,
      FI_EN_E816_CELCIUS = 33UL,
      FI_EN_E8155_CELCIUS = 34UL,
      FI_EN_E815_CELCIUS = 35UL,
      FI_EN_E8145_CELCIUS = 36UL,
      FI_EN_E814_CELCIUS = 37UL,
      FI_EN_E8OFF = 38UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACRight_Rear_SetPointSetTempValue(most_fi_tcl_e8_HVACRight_Rear_SetPointSetTempValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACRight_Rear_SetPointSetTempValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACRight_Rear_SetPointSetTempValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACIonizerStateValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACIonizerStateValue();
   enum tenType {
      FI_EN_E8INACTIVE = 0UL,
      FI_EN_E8LO = 1UL,
      FI_EN_E8HI = 2UL,
      FI_EN_E8ON = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACIonizerStateValue(most_fi_tcl_e8_HVACIonizerStateValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACIonizerStateValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACIonizerStateValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_HVACHVACSettingsValidEnum : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_HVACHVACSettingsValidEnum();
   enum tenType {
      FI_EN_E8FU_INVALID = 0UL,
      FI_EN_E8FU_VALID = 1UL,
      FI_EN_E8FU_OFF = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_HVACHVACSettingsValidEnum(most_fi_tcl_e8_HVACHVACSettingsValidEnum::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_HVACHVACSettingsValidEnum();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_HVACHVACSettingsValidEnum& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ICSCtlSwitchEnumeration : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ICSCtlSwitchEnumeration();
   enum tenType {
      FI_EN_E8VI_ALL_RELEASE = 0UL,
      FI_EN_E8VI_FAVORITES_UP_SWITCH = 1UL,
      FI_EN_E8VI_FAVORITES_DOWN_SWITCH = 2UL,
      FI_EN_E8VI_POWER_SWITCH = 3UL,
      FI_EN_E8VI_VOLUME_UP_SWITCH = 4UL,
      FI_EN_E8VI_VOLUME_DOWN_SWITCH = 5UL,
      FI_EN_E8VI_AUDIO_SWITCH = 6UL,
      FI_EN_E8VI_PHONE_SWITCH = 7UL,
      FI_EN_E8VI_MENU_SWITCH = 8UL,
      FI_EN_E8VI_BACK_SWITCH = 9UL,
      FI_EN_E8VI_HOME_SWITCH = 10UL,
      FI_EN_E8VI_BAND_SWITCH = 11UL,
      FI_EN_E8VI_TAG1_SWITCH = 12UL,
      FI_EN_E8VI_TAG2_SWITCH = 13UL,
      FI_EN_E8VI_TAG3_SWITCH = 14UL,
      FI_EN_E8VI_TAG4_SWITCH = 15UL,
      FI_EN_E8VI_TAG5_SWITCH = 16UL,
      FI_EN_E8VI_NEXT_SWITCH = 17UL,
      FI_EN_E8VI_PREV_SWITCH = 18UL,
      FI_EN_E8VI_EJECT_SWITCH = 19UL,
      FI_EN_E8VI_STORAGE_UNLOCK_SWITCH = 20UL,
      FI_EN_E8VI_BENCH_MODE_SWITCH = 32UL,
      FI_EN_E8VI_DEV_SERVER_MODE_SWITCH = 33UL,
      FI_EN_E8VI_DISC_AUX_SWITCH = 34UL,
      FI_EN_E8VI_TONE_SWITCH = 35UL
   };
   tenType enType;

   most_fi_tcl_e8_ICSCtlSwitchEnumeration(most_fi_tcl_e8_ICSCtlSwitchEnumeration::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ICSCtlSwitchEnumeration();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ICSCtlSwitchEnumeration& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ICSCtlSwitchEventTypeEnumeration : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ICSCtlSwitchEventTypeEnumeration();
   enum tenType {
      FI_EN_E8ICS_BUTTON_STATUS_RELEASED = 0UL,
      FI_EN_E8ICS_BUTTON_STATUS_PRESSED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_ICSCtlSwitchEventTypeEnumeration(most_fi_tcl_e8_ICSCtlSwitchEventTypeEnumeration::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ICSCtlSwitchEventTypeEnumeration();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ICSCtlSwitchEventTypeEnumeration& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_ICSCtlSliderProximity : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_ICSCtlSliderProximity();
   enum tenType {
      FI_EN_E8ICS_NOPROXIMITY_NOTOUCH = 0UL,
      FI_EN_E8ICS_PROXIMITY_DETECTED = 1UL,
      FI_EN_E8ICS_TOUCH_DETECTED = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_ICSCtlSliderProximity(most_fi_tcl_e8_ICSCtlSliderProximity::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_ICSCtlSliderProximity();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_ICSCtlSliderProximity& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iAppsIBErrorCode : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iAppsIBErrorCode();
   enum tenType {
      FI_EN_E8ERR_IB_INVALID_LIST_TYPE = 192UL,
      FI_EN_E8ERR_IB_MISSING_FILTER_STRING = 193UL,
      FI_EN_E8ERR_IB_INVALID_INDEX = 194UL,
      FI_EN_E8ERR_IB_INVALID_AUDIO_CHANNEL = 195UL,
      FI_EN_E8ERR_IB_INVALID_URL = 196UL,
      FI_EN_E8ERR_IB_FILE_FORMAT = 197UL,
      FI_EN_E8ERR_IB_INVALID_HANDLE = 198UL,
      FI_EN_E8ERR_IB_INVALID_IMAGE = 199UL
   };
   tenType enType;

   most_fi_tcl_e8_iAppsIBErrorCode(most_fi_tcl_e8_iAppsIBErrorCode::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iAppsIBErrorCode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iAppsIBErrorCode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iAppsListType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iAppsListType();
   enum tenType {
      FI_EN_E8LTY_APPS = 0UL,
      FI_EN_E8LTY_DEVICE_APPS = 1UL,
      FI_EN_E8LTY_APP_LAUNCHABLE = 2UL,
      FI_EN_E8LTY_UPDATE_AVAILABLE = 3UL,
      FI_EN_E8LTY_BY_USERNAME = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_iAppsListType(most_fi_tcl_e8_iAppsListType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iAppsListType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iAppsListType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iAppsActiveState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iAppsActiveState();
   enum tenType {
      FI_EN_E8AS_NOT_ACTIVE = 0UL,
      FI_EN_E8AS_FOREGROUND = 1UL,
      FI_EN_E8AS_FOREGROUND_LOCKED = 2UL,
      FI_EN_E8AS_BACKGROUND = 3UL,
      FI_EN_E8AS_BACKGROUND_LOCKED = 4UL,
      FI_EN_E8AS_PAUSED = 5UL,
      FI_EN_E8AS_DISABLED = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_iAppsActiveState(most_fi_tcl_e8_iAppsActiveState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iAppsActiveState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iAppsActiveState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iAppsSourceLocation : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iAppsSourceLocation();
   enum tenType {
      FI_EN_E8LOCAL_SOURCE = 0UL,
      FI_EN_E8SERVER_SOURCE = 1UL,
      FI_EN_E8REMOTE_SOURCE = 2UL,
      FI_EN_E8TELEMATICS_SOURCE = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_iAppsSourceLocation(most_fi_tcl_e8_iAppsSourceLocation::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iAppsSourceLocation();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iAppsSourceLocation& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iAppsAppConnectionRequired : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iAppsAppConnectionRequired();
   enum tenType {
      FI_EN_E8ACR_NO_CONNECTION = 0UL,
      FI_EN_E8ACR_LOW_BANDWIDTH = 1UL,
      FI_EN_E8ACR_HIGH_BANDWIDTH = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_iAppsAppConnectionRequired(most_fi_tcl_e8_iAppsAppConnectionRequired::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iAppsAppConnectionRequired();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iAppsAppConnectionRequired& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iAppsAppVehicleStateRequired : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iAppsAppVehicleStateRequired();
   enum tenType {
      FI_EN_E8AVSR_NO_RESTRICTION = 0UL,
      FI_EN_E8AVSR_PARK_ONLY = 1UL,
      FI_EN_E8AVSR_BELOW_SPEED = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_iAppsAppVehicleStateRequired(most_fi_tcl_e8_iAppsAppVehicleStateRequired::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iAppsAppVehicleStateRequired();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iAppsAppVehicleStateRequired& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iAppsAppObjectsItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iAppsAppObjectsItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iAppsAppObjectsItem& operator=(const most_fi_tcl_iAppsAppObjectsItem& coRef);
   most_fi_tcl_iAppsAppObjectsItem(const most_fi_tcl_iAppsAppObjectsItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32AppID;
   most_fi_tcl_String sAppVersion;
   most_fi_tcl_String sAppTitle;
   most_fi_tcl_String sIconURL;
   most_fi_tcl_String sURL;
   most_fi_tcl_e8_iAppsActiveState e8ActiveState;
   most_fi_tcl_e8_iAppsSourceLocation e8SourceLocation;
   most_fi_tcl_e8_iAppsAppConnectionRequired e8AppConnectionRequired;
   most_fi_tcl_e8_iAppsAppVehicleStateRequired e8AppVehicleStateRequired;
   tU8 u8Notifications;
   tU8 u8Ordinal;
   virtual ~most_fi_tcl_iAppsAppObjectsItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iAppsAppObjectsItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iAppsAppObjects : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iAppsAppObjects() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iAppsAppObjects& operator=(const most_fi_tcl_iAppsAppObjects& coRef);
   most_fi_tcl_iAppsAppObjects(const most_fi_tcl_iAppsAppObjects& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_iAppsAppObjectsItem*, std::allocator<most_fi_tcl_iAppsAppObjectsItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_iAppsAppObjectsItem, std::allocator<most_fi_tcl_iAppsAppObjectsItem> >oItems;

   most_fi_tcl_iAppsAppObjectsItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_iAppsAppObjects();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iAppsAppObjects& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iAppsChange : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iAppsChange();
   enum tenType {
      FI_EN_E8LCH_CONTENT_CHANGED = 0UL,
      FI_EN_E8LCH_ITEMS_ADDED = 1UL,
      FI_EN_E8LCH_ITEMS_REMOVED = 2UL,
      FI_EN_E8LCH_ITEMS_CHANGED = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_iAppsChange(most_fi_tcl_e8_iAppsChange::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iAppsChange();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iAppsChange& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iAppsItems : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iAppsItems() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iAppsItems& operator=(const most_fi_tcl_iAppsItems& coRef);
   most_fi_tcl_iAppsItems(const most_fi_tcl_iAppsItems& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU32, std::allocator<tU32> >u32Items;

   tU32& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_iAppsItems();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iAppsItems& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iAppsRendererPosition : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iAppsRendererPosition();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iAppsRendererPosition& operator=(const most_fi_tcl_iAppsRendererPosition& coRef);
   most_fi_tcl_iAppsRendererPosition(const most_fi_tcl_iAppsRendererPosition& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 u16RendererHeight;
   tU16 u16RendererWidth;
   tU16 u16RendererXCoordinate;
   tU16 u16RendererYCoordinate;
   virtual ~most_fi_tcl_iAppsRendererPosition();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iAppsRendererPosition& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iAppsActiveAppsItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iAppsActiveAppsItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iAppsActiveAppsItem& operator=(const most_fi_tcl_iAppsActiveAppsItem& coRef);
   most_fi_tcl_iAppsActiveAppsItem(const most_fi_tcl_iAppsActiveAppsItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32RendererHandle;
   tU32 u32AppID;
   most_fi_tcl_e8_iAppsActiveState e8ActiveState;
   most_fi_tcl_e8_iAppsAppVehicleStateRequired e8AppVehicleStateRequired;
   virtual ~most_fi_tcl_iAppsActiveAppsItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iAppsActiveAppsItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iAppsActiveApps : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iAppsActiveApps() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iAppsActiveApps& operator=(const most_fi_tcl_iAppsActiveApps& coRef);
   most_fi_tcl_iAppsActiveApps(const most_fi_tcl_iAppsActiveApps& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_iAppsActiveAppsItem, std::allocator<most_fi_tcl_iAppsActiveAppsItem> >oItems;

   most_fi_tcl_iAppsActiveAppsItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_iAppsActiveApps();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iAppsActiveApps& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iAppsImageData : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iAppsImageData() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iAppsImageData& operator=(const most_fi_tcl_iAppsImageData& coRef);
   most_fi_tcl_iAppsImageData(const most_fi_tcl_iAppsImageData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU8, std::allocator<tU8> >u8Items;

   tU8& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_iAppsImageData();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iAppsImageData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iAppsConnectionType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iAppsConnectionType();
   enum tenType {
      FI_EN_E8NO_CONNECTION = 0UL,
      FI_EN_E8DUN_CONNECTION = 1UL,
      FI_EN_E8BT_SPP_CONNECTION = 2UL,
      FI_EN_E8TELEMATICS_CONNECTION = 3UL,
      FI_EN_E8USB_SERIAL_CONNECTION = 4UL,
      FI_EN_E8WIFI_CONNECTION = 5UL,
      FI_EN_E8PAN_CONNECTION = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_iAppsConnectionType(most_fi_tcl_e8_iAppsConnectionType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iAppsConnectionType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iAppsConnectionType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iAppsConnectionStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iAppsConnectionStatus();
   enum tenType {
      FI_EN_E8CONNECTION_NOT_AVAILABLE = 0UL,
      FI_EN_E8CONNECTION_DISABLED = 1UL,
      FI_EN_E8CONNECTION_ENABLED_INACTIVE = 2UL,
      FI_EN_E8CONNECTION_ENABLED_ACTIVE = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_iAppsConnectionStatus(most_fi_tcl_e8_iAppsConnectionStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iAppsConnectionStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iAppsConnectionStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iAppsConnectionBandwidth : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iAppsConnectionBandwidth();
   enum tenType {
      FI_EN_E8HIGH_BANDWIDTH = 0UL,
      FI_EN_E8LOW_BANDWIDTH = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_iAppsConnectionBandwidth(most_fi_tcl_e8_iAppsConnectionBandwidth::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iAppsConnectionBandwidth();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iAppsConnectionBandwidth& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iAppsConnectionAvailabilityItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iAppsConnectionAvailabilityItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iAppsConnectionAvailabilityItem& operator=(const most_fi_tcl_iAppsConnectionAvailabilityItem& coRef);
   most_fi_tcl_iAppsConnectionAvailabilityItem(const most_fi_tcl_iAppsConnectionAvailabilityItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_iAppsConnectionType e8ConnectionType;
   most_fi_tcl_e8_iAppsConnectionStatus e8ConnectionStatus;
   most_fi_tcl_e8_iAppsConnectionBandwidth e8ConnectionBandwidth;
   virtual ~most_fi_tcl_iAppsConnectionAvailabilityItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iAppsConnectionAvailabilityItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iAppsConnectionAvailability : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iAppsConnectionAvailability() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iAppsConnectionAvailability& operator=(const most_fi_tcl_iAppsConnectionAvailability& coRef);
   most_fi_tcl_iAppsConnectionAvailability(const most_fi_tcl_iAppsConnectionAvailability& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_iAppsConnectionAvailabilityItem, std::allocator<most_fi_tcl_iAppsConnectionAvailabilityItem> >oItems;

   most_fi_tcl_iAppsConnectionAvailabilityItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_iAppsConnectionAvailability();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iAppsConnectionAvailability& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iAppsUpdateHTMLItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iAppsUpdateHTMLItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iAppsUpdateHTMLItem& operator=(const most_fi_tcl_iAppsUpdateHTMLItem& coRef);
   most_fi_tcl_iAppsUpdateHTMLItem(const most_fi_tcl_iAppsUpdateHTMLItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sHTMLElementID;
   most_fi_tcl_ClassifiedStream oHTMLCode;
   virtual ~most_fi_tcl_iAppsUpdateHTMLItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iAppsUpdateHTMLItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iAppsUpdateHTML : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iAppsUpdateHTML() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iAppsUpdateHTML& operator=(const most_fi_tcl_iAppsUpdateHTML& coRef);
   most_fi_tcl_iAppsUpdateHTML(const most_fi_tcl_iAppsUpdateHTML& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_iAppsUpdateHTMLItem*, std::allocator<most_fi_tcl_iAppsUpdateHTMLItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_iAppsUpdateHTMLItem, std::allocator<most_fi_tcl_iAppsUpdateHTMLItem> >oItems;

   most_fi_tcl_iAppsUpdateHTMLItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_iAppsUpdateHTML();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iAppsUpdateHTML& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iAppsMethod : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iAppsMethod();
   enum tenType {
      FI_EN_E8GET = 0UL,
      FI_EN_E8POST = 1UL,
      FI_EN_E8PUT = 2UL,
      FI_EN_E8DELETE = 3UL,
      FI_EN_E8HEAD = 4UL,
      FI_EN_E8TRACE = 5UL,
      FI_EN_E8OPTIONS = 6UL,
      FI_EN_E8CONNECT = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_iAppsMethod(most_fi_tcl_e8_iAppsMethod::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iAppsMethod();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iAppsMethod& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iAppsSyncrhonizationState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iAppsSyncrhonizationState();
   enum tenType {
      FI_EN_E8SYNCHRONIZATION_IN_PROGRESS = 0UL,
      FI_EN_E8SYNCHRONIZATION_COMPLETE = 1UL,
      FI_EN_E8SYNCHRONIZATION_FAILED = 2UL,
      FI_EN_E8SYNCHRONIZATION_QUEUED = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_iAppsSyncrhonizationState(most_fi_tcl_e8_iAppsSyncrhonizationState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iAppsSyncrhonizationState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iAppsSyncrhonizationState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iAppsSelectorButtonStreamItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iAppsSelectorButtonStreamItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iAppsSelectorButtonStreamItem& operator=(const most_fi_tcl_iAppsSelectorButtonStreamItem& coRef);
   most_fi_tcl_iAppsSelectorButtonStreamItem(const most_fi_tcl_iAppsSelectorButtonStreamItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tBool bSelectorButtonEnable;
   tU8 u8SelectorButtonID;
   most_fi_tcl_String sSelectorButtonText;
   most_fi_tcl_String sSelectorButtonIconURL;
   virtual ~most_fi_tcl_iAppsSelectorButtonStreamItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iAppsSelectorButtonStreamItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iAppsSelectorButtonStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iAppsSelectorButtonStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iAppsSelectorButtonStream& operator=(const most_fi_tcl_iAppsSelectorButtonStream& coRef);
   most_fi_tcl_iAppsSelectorButtonStream(const most_fi_tcl_iAppsSelectorButtonStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_iAppsSelectorButtonStreamItem*, std::allocator<most_fi_tcl_iAppsSelectorButtonStreamItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_iAppsSelectorButtonStreamItem, std::allocator<most_fi_tcl_iAppsSelectorButtonStreamItem> >oItems;

   most_fi_tcl_iAppsSelectorButtonStreamItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_iAppsSelectorButtonStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iAppsSelectorButtonStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iAppsSelectorButtonState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iAppsSelectorButtonState();
   enum tenType {
      FI_EN_E8SELECTOR_BUTTON_RELEASED = 0UL,
      FI_EN_E8SELECTOR_BUTTON_PRESSED = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_iAppsSelectorButtonState(most_fi_tcl_e8_iAppsSelectorButtonState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iAppsSelectorButtonState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iAppsSelectorButtonState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iAppsInternetApplicationSystemAction : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iAppsInternetApplicationSystemAction();
   enum tenType {
      FI_EN_E8ACTION_PHONE_DIALED = 0UL,
      FI_EN_E8ACTION_TEXT_MESSAGE_SENT = 1UL,
      FI_EN_E8ACTION_NAVIGATION_SET_DESTINATION = 2UL,
      FI_EN_E8ACTION_NONE = 255UL
   };
   tenType enType;

   most_fi_tcl_e8_iAppsInternetApplicationSystemAction(most_fi_tcl_e8_iAppsInternetApplicationSystemAction::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iAppsInternetApplicationSystemAction();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iAppsInternetApplicationSystemAction& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iAppsMessageHandle : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iAppsMessageHandle();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iAppsMessageHandle& operator=(const most_fi_tcl_iAppsMessageHandle& coRef);
   most_fi_tcl_iAppsMessageHandle(const most_fi_tcl_iAppsMessageHandle& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceHandle;
   tU32 u32MsgHandleUpper;
   tU32 u32MsgHandleLower;
   virtual ~most_fi_tcl_iAppsMessageHandle();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iAppsMessageHandle& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iAppsButtonAction : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iAppsButtonAction();
   enum tenType {
      FI_EN_E8INTERNET_APPLICATION_ACTION_1 = 15UL,
      FI_EN_E8INTERNET_APPLICATION_ACTION_2 = 16UL,
      FI_EN_E8INTERNET_APPLICATION_ACTION_3 = 17UL
   };
   tenType enType;

   most_fi_tcl_e8_iAppsButtonAction(most_fi_tcl_e8_iAppsButtonAction::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iAppsButtonAction();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iAppsButtonAction& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iAppsSortOrder : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iAppsSortOrder();
   enum tenType {
      FI_EN_E8SORT_ORDER_ALPHABETICAL = 0UL,
      FI_EN_E8SORT_ORDER_DISTANCE = 1UL,
      FI_EN_E8SORT_ORDER_DEFAULT = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_iAppsSortOrder(most_fi_tcl_e8_iAppsSortOrder::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iAppsSortOrder();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iAppsSortOrder& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iAppsAutocompleteResultList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iAppsAutocompleteResultList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iAppsAutocompleteResultList& operator=(const most_fi_tcl_iAppsAutocompleteResultList& coRef);
   most_fi_tcl_iAppsAutocompleteResultList(const most_fi_tcl_iAppsAutocompleteResultList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_String*, std::allocator<most_fi_tcl_String*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_String, std::allocator<most_fi_tcl_String> >sItems;

   most_fi_tcl_String& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_iAppsAutocompleteResultList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iAppsAutocompleteResultList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iAppsPOIDistanceType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iAppsPOIDistanceType();
   enum tenType {
      FI_EN_E8POIDISTANCETYPE_USER_SPECIFIED_LOCATION = 0UL,
      FI_EN_E8POIDISTANCETYPE_FINAL_DESTINATION = 1UL,
      FI_EN_E8POIDISTANCETYPE_CURRENT_VEHICLE_LOCATION = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_iAppsPOIDistanceType(most_fi_tcl_e8_iAppsPOIDistanceType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iAppsPOIDistanceType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iAppsPOIDistanceType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iAppsDistanceUnits : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iAppsDistanceUnits();
   enum tenType {
      FI_EN_E8UNITS_METERS = 1UL,
      FI_EN_E8UNITS_KILOMETERS = 2UL,
      FI_EN_E8UNIT_YARDS = 3UL,
      FI_EN_E8UNIT_FEET = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_iAppsDistanceUnits(most_fi_tcl_e8_iAppsDistanceUnits::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iAppsDistanceUnits();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iAppsDistanceUnits& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iAppsSearchResultListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iAppsSearchResultListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iAppsSearchResultListItem& operator=(const most_fi_tcl_iAppsSearchResultListItem& coRef);
   most_fi_tcl_iAppsSearchResultListItem(const most_fi_tcl_iAppsSearchResultListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 u16POIIndex;
   most_fi_tcl_String sPOIName;
   most_fi_tcl_String sHouseNumber;
   most_fi_tcl_String sStreetName;
   most_fi_tcl_String sStreetType;
   most_fi_tcl_String sCity;
   most_fi_tcl_String sState;
   most_fi_tcl_String sBorough;
   most_fi_tcl_String sPostalCode;
   most_fi_tcl_String sCountry;
   most_fi_tcl_String sPhoneNo;
   tBool bLatLongValidity;
   tS32 s32Latitude;
   tS32 s32Longitude;
   most_fi_tcl_String sPOICategory;
   most_fi_tcl_String sPOIIconURL;
   most_fi_tcl_String sPOIImageURL;
   tU32 u32POIDistance;
   most_fi_tcl_e8_iAppsPOIDistanceType e8POIDistanceType;
   most_fi_tcl_e8_iAppsDistanceUnits e8DistanceUnits;
   tU16 u16Rating;
   most_fi_tcl_String sUserComments;
   most_fi_tcl_String sAdditionalInfo;
   virtual ~most_fi_tcl_iAppsSearchResultListItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iAppsSearchResultListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iAppsSearchResultList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iAppsSearchResultList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iAppsSearchResultList& operator=(const most_fi_tcl_iAppsSearchResultList& coRef);
   most_fi_tcl_iAppsSearchResultList(const most_fi_tcl_iAppsSearchResultList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_iAppsSearchResultListItem*, std::allocator<most_fi_tcl_iAppsSearchResultListItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_iAppsSearchResultListItem, std::allocator<most_fi_tcl_iAppsSearchResultListItem> >oItems;

   most_fi_tcl_iAppsSearchResultListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_iAppsSearchResultList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iAppsSearchResultList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iAppsImageFormat : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iAppsImageFormat();
   enum tenType {
      FI_EN_E8IMAGE_FORMAT_PNG = 0UL,
      FI_EN_E8IMAGE_FORMAT_JPEG = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_iAppsImageFormat(most_fi_tcl_e8_iAppsImageFormat::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iAppsImageFormat();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iAppsImageFormat& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iRadioListType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iRadioListType();
   enum tenType {
      FI_EN_E8LTY_STATION = 0UL,
      FI_EN_E8LTY_GENRE = 1UL,
      FI_EN_E8LTY_GENRE_STATION = 2UL,
      FI_EN_E8LTY_RATE = 3UL,
      FI_EN_E8LTY_RATE_STATION = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_iRadioListType(most_fi_tcl_e8_iRadioListType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iRadioListType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iRadioListType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iRadioFileFormat : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iRadioFileFormat();
   enum tenType {
      FI_EN_E8FFT_MP3 = 0UL,
      FI_EN_E8FFT_AAC = 1UL,
      FI_EN_E8FFT_AAC_PLUS = 2UL,
      FI_EN_E8FFT_WMA = 3UL,
      FI_EN_E8FFT_OGG_VORBIS = 4UL,
      FI_EN_E8FFT_PLS = 5UL,
      FI_EN_E8FFT_M3U = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_iRadioFileFormat(most_fi_tcl_e8_iRadioFileFormat::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iRadioFileFormat();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iRadioFileFormat& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iRadioStationsItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iRadioStationsItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iRadioStationsItem& operator=(const most_fi_tcl_iRadioStationsItem& coRef);
   most_fi_tcl_iRadioStationsItem(const most_fi_tcl_iRadioStationsItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32StationTag;
   most_fi_tcl_String sStationTitle;
   most_fi_tcl_String sURL;
   tU16 u16BitRate;
   most_fi_tcl_e8_iRadioFileFormat e8FileFormat;
   most_fi_tcl_String sGenre;
   most_fi_tcl_String sImageURL;
   tBool bIsPlaying;
   tU8 u8Ordinal;
   virtual ~most_fi_tcl_iRadioStationsItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iRadioStationsItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iRadioStations : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iRadioStations() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iRadioStations& operator=(const most_fi_tcl_iRadioStations& coRef);
   most_fi_tcl_iRadioStations(const most_fi_tcl_iRadioStations& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_iRadioStationsItem*, std::allocator<most_fi_tcl_iRadioStationsItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_iRadioStationsItem, std::allocator<most_fi_tcl_iRadioStationsItem> >oItems;

   most_fi_tcl_iRadioStationsItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_iRadioStations();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iRadioStations& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iRadioChange : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iRadioChange();
   enum tenType {
      FI_EN_E8LCH_CONTENT_CHANGED = 0UL,
      FI_EN_E8LCH_ITEMS_ADDED = 1UL,
      FI_EN_E8LCH_ITEMS_REMOVED = 2UL,
      FI_EN_E8LCH_ITEMS_CHANGED = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_iRadioChange(most_fi_tcl_e8_iRadioChange::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iRadioChange();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iRadioChange& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iRadioItems : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iRadioItems() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iRadioItems& operator=(const most_fi_tcl_iRadioItems& coRef);
   most_fi_tcl_iRadioItems(const most_fi_tcl_iRadioItems& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU32, std::allocator<tU32> >u32Items;

   tU32& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_iRadioItems();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iRadioItems& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iRadioStationType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iRadioStationType();
   enum tenType {
      FI_EN_E8STATION_TYPE_NORMAL = 0UL,
      FI_EN_E8STATION_TYPE_QUICK_MIX = 1UL,
      FI_EN_E8STATION_TYPE_SHARED = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_iRadioStationType(most_fi_tcl_e8_iRadioStationType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iRadioStationType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iRadioStationType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iRadioStation : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iRadioStation();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iRadioStation& operator=(const most_fi_tcl_iRadioStation& coRef);
   most_fi_tcl_iRadioStation(const most_fi_tcl_iRadioStation& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tBool bStationAvailable;
   tU32 u32StationTag;
   most_fi_tcl_String sStationTitle;
   most_fi_tcl_String sURL;
   tU16 u16BitRate;
   most_fi_tcl_e8_iRadioFileFormat e8FileFormat;
   most_fi_tcl_String sGenre;
   most_fi_tcl_String sImageURL;
   tBool bIsPlaying;
   tU8 u8Ordinal;
   most_fi_tcl_e8_iRadioStationType e8StationType;
   virtual ~most_fi_tcl_iRadioStation();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iRadioStation& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iRadioRating : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iRadioRating();
   enum tenType {
      FI_EN_E8RATE_POSITIVE = 0UL,
      FI_EN_E8RATE_NEGATIVE = 1UL,
      FI_EN_E8RATE_NEUTRAL = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_iRadioRating(most_fi_tcl_e8_iRadioRating::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iRadioRating();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iRadioRating& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iRadioSong : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iRadioSong();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iRadioSong& operator=(const most_fi_tcl_iRadioSong& coRef);
   most_fi_tcl_iRadioSong(const most_fi_tcl_iRadioSong& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sArtist;
   most_fi_tcl_String sAlbum;
   most_fi_tcl_String sSongTitle;
   most_fi_tcl_String sImageURL;
   tU16 u16BitRate;
   tU32 u32StationTag;
   most_fi_tcl_String sURL;
   most_fi_tcl_e8_iRadioRating e8Rating;
   tBool bBookmarkable;
   tBool bRateable;
   virtual ~most_fi_tcl_iRadioSong();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iRadioSong& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iRadioAction : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iRadioAction();
   enum tenType {
      FI_EN_E8PBA_PLAY = 0UL,
      FI_EN_E8PBA_PAUSE = 1UL,
      FI_EN_E8PBA_STOP = 2UL,
      FI_EN_E8PBA_PREV = 3UL,
      FI_EN_E8PBA_NEXT = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_iRadioAction(most_fi_tcl_e8_iRadioAction::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iRadioAction();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iRadioAction& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iRadioImageData : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iRadioImageData() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iRadioImageData& operator=(const most_fi_tcl_iRadioImageData& coRef);
   most_fi_tcl_iRadioImageData(const most_fi_tcl_iRadioImageData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU8, std::allocator<tU8> >u8Items;

   tU8& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_iRadioImageData();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iRadioImageData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iRadioInternetRadioSource : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iRadioInternetRadioSource();
   enum tenType {
      FI_EN_E8SRC_NONE = 0UL,
      FI_EN_E8SRC_GENERIC = 1UL,
      FI_EN_E8SRC_PANDORA = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_iRadioInternetRadioSource(most_fi_tcl_e8_iRadioInternetRadioSource::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iRadioInternetRadioSource();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iRadioInternetRadioSource& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iRadioPlaybackState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iRadioPlaybackState();
   enum tenType {
      FI_EN_E8PBS_PAUSED = 0UL,
      FI_EN_E8PBS_PLAYING = 1UL,
      FI_EN_E8PBS_STOPPED = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_iRadioPlaybackState(most_fi_tcl_e8_iRadioPlaybackState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iRadioPlaybackState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iRadioPlaybackState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iRadioConnectionStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iRadioConnectionStatus();
   enum tenType {
      FI_EN_E8CS_NO_INTERNET_CONNECTION = 0UL,
      FI_EN_E8CS_AQUIRING_STATION = 1UL,
      FI_EN_E8CS_CANNOT_CONNECT = 2UL,
      FI_EN_E8CS_CONNECTED = 3UL,
      FI_EN_E8CS_DISCONNECTED = 4UL,
      FI_EN_E8CS_BT_CONNECTED = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_iRadioConnectionStatus(most_fi_tcl_e8_iRadioConnectionStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iRadioConnectionStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iRadioConnectionStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iRadioSynchronizationState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iRadioSynchronizationState();
   enum tenType {
      FI_EN_E8SYNCHRONIZATION_STATE_IDLE = 0UL,
      FI_EN_E8SYNCHRONIZATION_STATE_RUNNING = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_iRadioSynchronizationState(most_fi_tcl_e8_iRadioSynchronizationState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iRadioSynchronizationState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iRadioSynchronizationState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_iRadioPandoraAccountStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_iRadioPandoraAccountStatus();
   enum tenType {
      FI_EN_E8PANDORA_ACCOUNT_NOT_SETUP = 0UL,
      FI_EN_E8PANDORA_ACCOUNT_NOT_SIGNEDIN = 1UL,
      FI_EN_E8PANDORA_ACCOUNT_SIGNED_IN = 2UL,
      FI_EN_E8PANDORA_ACCOUNT_NO_STATIONS = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_iRadioPandoraAccountStatus(most_fi_tcl_e8_iRadioPandoraAccountStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_iRadioPandoraAccountStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_iRadioPandoraAccountStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iRadioStationListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iRadioStationListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iRadioStationListItem& operator=(const most_fi_tcl_iRadioStationListItem& coRef);
   most_fi_tcl_iRadioStationListItem(const most_fi_tcl_iRadioStationListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32StationTag;
   most_fi_tcl_String sStationTitle;
   most_fi_tcl_String sURL;
   tU16 u16BitRate;
   most_fi_tcl_e8_iRadioFileFormat e8FileFormat;
   most_fi_tcl_String sGenre;
   most_fi_tcl_String sImageURL;
   tBool bIsPlaying;
   tU8 u8Ordinal;
   most_fi_tcl_e8_iRadioStationType e8StationType;
   virtual ~most_fi_tcl_iRadioStationListItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iRadioStationListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_iRadioStationList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_iRadioStationList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_iRadioStationList& operator=(const most_fi_tcl_iRadioStationList& coRef);
   most_fi_tcl_iRadioStationList(const most_fi_tcl_iRadioStationList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_iRadioStationListItem*, std::allocator<most_fi_tcl_iRadioStationListItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_iRadioStationListItem, std::allocator<most_fi_tcl_iRadioStationListItem> >oItems;

   most_fi_tcl_iRadioStationListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_iRadioStationList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_iRadioStationList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_LVMRmtEnable : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_LVMRmtEnable();
   enum tenType {
      FI_EN_E8SYS_RMT_ENABLE_OFF = 0UL,
      FI_EN_E8SYS_RMT_ENABLE_ON = 1UL,
      FI_EN_E8SYS_RMT_ENABLE_LVM_ACTIVE = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_LVMRmtEnable(most_fi_tcl_e8_LVMRmtEnable::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_LVMRmtEnable();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_LVMRmtEnable& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MPlayListType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MPlayListType();
   enum tenType {
      FI_EN_E8LTY_GENRE = 0UL,
      FI_EN_E8LTY_ARTIST = 1UL,
      FI_EN_E8LTY_ALBUM = 2UL,
      FI_EN_E8LTY_SONG = 3UL,
      FI_EN_E8LTY_GENRE_ARTIST = 4UL,
      FI_EN_E8LTY_GENRE_ARTIST_ALBUM = 5UL,
      FI_EN_E8LTY_GENRE_ARTIST_ALBUM_SONG = 6UL,
      FI_EN_E8LTY_GENRE_ARTIST_SONG = 7UL,
      FI_EN_E8LTY_GENRE_ALBUM = 8UL,
      FI_EN_E8LTY_GENRE_ALBUM_SONG = 9UL,
      FI_EN_E8LTY_GENRE_SONG = 10UL,
      FI_EN_E8LTY_ARTIST_ALBUM = 11UL,
      FI_EN_E8LTY_ARTIST_ALBUM_SONG = 12UL,
      FI_EN_E8LTY_ARTIST_SONG = 13UL,
      FI_EN_E8LTY_ALBUM_SONG = 14UL,
      FI_EN_E8LTY_PODCAST = 15UL,
      FI_EN_E8LTY_PODCAST_EPISODE = 16UL,
      FI_EN_E8LTY_AUDIOBOOK = 17UL,
      FI_EN_E8LTY_BOOKTITLE_CHAPTER = 18UL,
      FI_EN_E8LTY_AUTHOR = 19UL,
      FI_EN_E8LTY_AUTHOR_BOOKTITLE = 20UL,
      FI_EN_E8LTY_AUTHOR_BOOKTITLE_CHAPTER = 21UL,
      FI_EN_E8LTY_COMPOSER = 22UL,
      FI_EN_E8LTY_COMPOSER_ALBUM = 23UL,
      FI_EN_E8LTY_COMPOSER_ALBUM_SONG = 24UL,
      FI_EN_E8LTY_COMPOSER_SONG = 25UL,
      FI_EN_E8LTY_VIDEO = 26UL,
      FI_EN_E8LTY_VIDEO_EPISODE = 27UL,
      FI_EN_E8LTY_PLAYLIST = 28UL,
      FI_EN_E8LTY_PLAYLIST_SONG = 29UL,
      FI_EN_E8LTY_CURRENT_SELECTION = 30UL
   };
   tenType enType;

   most_fi_tcl_e8_MPlayListType(most_fi_tcl_e8_MPlayListType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MPlayListType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MPlayListType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MPlayCategoryType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MPlayCategoryType();
   enum tenType {
      FI_EN_E8CTY_NONE = 0UL,
      FI_EN_E8CTY_GENRE = 1UL,
      FI_EN_E8CTY_ARTIST = 2UL,
      FI_EN_E8CTY_ALBUM = 3UL,
      FI_EN_E8CTY_SONG = 4UL,
      FI_EN_E8CTY_COMPOSER = 5UL,
      FI_EN_E8CTY_AUTHOR = 6UL,
      FI_EN_E8CTY_TITLE = 7UL,
      FI_EN_E8CTY_CHAPTER = 8UL,
      FI_EN_E8CTY_NAME = 9UL,
      FI_EN_E8CTY_EPISODE = 10UL,
      FI_EN_E8CTY_PLAYLIST = 11UL
   };
   tenType enType;

   most_fi_tcl_e8_MPlayCategoryType(most_fi_tcl_e8_MPlayCategoryType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MPlayCategoryType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MPlayCategoryType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MPlayDeviceType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MPlayDeviceType();
   enum tenType {
      FI_EN_E8DTY_UNKNOWN = 0UL,
      FI_EN_E8DTY_USB = 1UL,
      FI_EN_E8DTY_IPOD = 2UL,
      FI_EN_E8DTY_SD = 3UL,
      FI_EN_E8DTY_BLUETOOTH = 4UL,
      FI_EN_E8DTY_IPHONE = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_MPlayDeviceType(most_fi_tcl_e8_MPlayDeviceType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MPlayDeviceType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MPlayDeviceType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MPlayFileFormat : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MPlayFileFormat();
   enum tenType {
      FI_EN_E8FFT_MP3 = 0UL,
      FI_EN_E8FFT_WMA = 1UL,
      FI_EN_E8FFT_AAC = 2UL,
      FI_EN_E8FFT_AAC_PLUS = 3UL,
      FI_EN_E8FFT_M4A = 4UL,
      FI_EN_E8FFT_MP4 = 5UL,
      FI_EN_E8FFT_WAV = 6UL,
      FI_EN_E8FFT_OGG = 7UL,
      FI_EN_E8FFT_3GP = 8UL,
      FI_EN_E8FFT_3G2 = 9UL,
      FI_EN_E8FFT_AIFF = 10UL,
      FI_EN_E8FFT_AA = 11UL,
      FI_EN_E8FFT_SBC = 12UL,
      FI_EN_E8FFT_MPEG = 13UL,
      FI_EN_E8FFT_AVI = 14UL,
      FI_EN_E8FFT_MOV = 15UL,
      FI_EN_E8FFT_DIVX = 16UL,
      FI_EN_E8FFT_WMV = 17UL,
      FI_EN_E8FFT_M4V = 18UL,
      FI_EN_E8FFT_FLV = 19UL,
      FI_EN_E8FFT_M3U = 20UL,
      FI_EN_E8FFT_PLS = 21UL,
      FI_EN_E8FFT_XSPF = 22UL,
      FI_EN_E8FFT_WPL = 23UL,
      FI_EN_E8FFT_B4S = 24UL,
      FI_EN_E8FFT_ASX = 25UL
   };
   tenType enType;

   most_fi_tcl_e8_MPlayFileFormat(most_fi_tcl_e8_MPlayFileFormat::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MPlayFileFormat();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MPlayFileFormat& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MPlayMediaType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MPlayMediaType();
   enum tenType {
      FI_EN_E8MTY_UNKNOWN = 0UL,
      FI_EN_E8MTY_MUSIC_FILE = 1UL,
      FI_EN_E8MTY_PODCAST = 2UL,
      FI_EN_E8MTY_AUDIOBOOK = 3UL,
      FI_EN_E8MTY_VIDEO = 4UL,
      FI_EN_E8MTY_PLAYLIST = 5UL,
      FI_EN_E8MRY_AUDIO_STREAM = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_MPlayMediaType(most_fi_tcl_e8_MPlayMediaType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MPlayMediaType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MPlayMediaType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MPlayMediaObjectsItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MPlayMediaObjectsItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MPlayMediaObjectsItem& operator=(const most_fi_tcl_MPlayMediaObjectsItem& coRef);
   most_fi_tcl_MPlayMediaObjectsItem(const most_fi_tcl_MPlayMediaObjectsItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sAlbumArt;
   most_fi_tcl_e8_MPlayCategoryType e8CategoryType;
   tU8 u8DeviceTag;
   most_fi_tcl_e8_MPlayDeviceType e8DeviceType;
   most_fi_tcl_String sDeviceVersion;
   most_fi_tcl_e8_MPlayFileFormat e8FileFormat;
   most_fi_tcl_String sFilename;
   tBool bIsPlaying;
   most_fi_tcl_e8_MPlayMediaType e8MediaType;
   most_fi_tcl_String sMetaDataField1;
   most_fi_tcl_String sMetaDataField2;
   most_fi_tcl_String sMetaDataField3;
   most_fi_tcl_String sMetaDataField4;
   tU32 u32MetaDataTag1;
   tU32 u32MetaDataTag2;
   tU32 u32MetaDataTag3;
   tU32 u32MetaDataTag4;
   tU32 u32Tag;
   tU32 u32TotalPlaytime;
   tU8 u8TrackNumber;
   virtual ~most_fi_tcl_MPlayMediaObjectsItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MPlayMediaObjectsItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MPlayMediaObjects : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MPlayMediaObjects() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MPlayMediaObjects& operator=(const most_fi_tcl_MPlayMediaObjects& coRef);
   most_fi_tcl_MPlayMediaObjects(const most_fi_tcl_MPlayMediaObjects& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_MPlayMediaObjectsItem*, std::allocator<most_fi_tcl_MPlayMediaObjectsItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_MPlayMediaObjectsItem, std::allocator<most_fi_tcl_MPlayMediaObjectsItem> >oItems;

   most_fi_tcl_MPlayMediaObjectsItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MPlayMediaObjects();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MPlayMediaObjects& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MPlayChange : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MPlayChange();
   enum tenType {
      FI_EN_E8LCH_CONTENT_CHANGED = 0UL,
      FI_EN_E8LCH_ITEMS_ADDED = 1UL,
      FI_EN_E8LCH_ITEMS_REMOVED = 2UL,
      FI_EN_E8LCH_ITEMS_CHANGED = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_MPlayChange(most_fi_tcl_e8_MPlayChange::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MPlayChange();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MPlayChange& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MPlayItems : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MPlayItems() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MPlayItems& operator=(const most_fi_tcl_MPlayItems& coRef);
   most_fi_tcl_MPlayItems(const most_fi_tcl_MPlayItems& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU32, std::allocator<tU32> >u32Items;

   tU32& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MPlayItems();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MPlayItems& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MPlayNowPlayingState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MPlayNowPlayingState();
   enum tenType {
      FI_EN_E8NP_NEW_TRACK = 0UL,
      FI_EN_E8NP_TRACK_START = 1UL,
      FI_EN_E8NP_TRACK_COMPLETE = 2UL,
      FI_EN_E8NP_LIST_START = 3UL,
      FI_EN_E8NP_LIST_COMPLETE = 4UL,
      FI_EN_E8NP_NO_DEVICE = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_MPlayNowPlayingState(most_fi_tcl_e8_MPlayNowPlayingState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MPlayNowPlayingState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MPlayNowPlayingState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MPlayMediaObject : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MPlayMediaObject();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MPlayMediaObject& operator=(const most_fi_tcl_MPlayMediaObject& coRef);
   most_fi_tcl_MPlayMediaObject(const most_fi_tcl_MPlayMediaObject& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sAlbumArt;
   most_fi_tcl_e8_MPlayCategoryType e8CategoryType;
   tU8 u8DeviceTag;
   most_fi_tcl_e8_MPlayDeviceType e8DeviceType;
   most_fi_tcl_String sDeviceVersion;
   most_fi_tcl_e8_MPlayFileFormat e8FileFormat;
   most_fi_tcl_String sFilename;
   tBool bIsPlaying;
   most_fi_tcl_e8_MPlayMediaType e8MediaType;
   most_fi_tcl_String sMetaDataField1;
   most_fi_tcl_String sMetaDataField2;
   most_fi_tcl_String sMetaDataField3;
   most_fi_tcl_String sMetaDataField4;
   tU32 u32MetaDataTag1;
   tU32 u32MetaDataTag2;
   tU32 u32MetaDataTag3;
   tU32 u32MetaDataTag4;
   tU32 u32Tag;
   tU32 u32TotalPlaytime;
   tU8 u8TrackNumber;
   tU16 u16ExtendedTrackNumber;
   virtual ~most_fi_tcl_MPlayMediaObject();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MPlayMediaObject& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MPlayState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MPlayState();
   enum tenType {
      FI_EN_E8PBS_PLAYING = 0UL,
      FI_EN_E8PBS_PAUSED = 1UL,
      FI_EN_E8PBS_STOPPED = 2UL,
      FI_EN_E8PBS_FFWD = 3UL,
      FI_EN_E8PBS_FREV = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_MPlayState(most_fi_tcl_e8_MPlayState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MPlayState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MPlayState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MPlayMode : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MPlayMode();
   enum tenType {
      FI_EN_E8PBM_NORMAL = 0UL,
      FI_EN_E8PBM_RANDOM = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_MPlayMode(most_fi_tcl_e8_MPlayMode::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MPlayMode();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MPlayMode& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MPlayRepeat : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MPlayRepeat();
   enum tenType {
      FI_EN_E8RPT_NONE = 0UL,
      FI_EN_E8RPT_ONE = 1UL,
      FI_EN_E8RPT_ALL = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_MPlayRepeat(most_fi_tcl_e8_MPlayRepeat::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MPlayRepeat();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MPlayRepeat& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MPlayAction : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MPlayAction();
   enum tenType {
      FI_EN_E8PBA_PLAY = 0UL,
      FI_EN_E8PBA_PAUSE = 1UL,
      FI_EN_E8PBA_STOP = 2UL,
      FI_EN_E8PBA_PREV = 3UL,
      FI_EN_E8PBA_NEXT = 4UL,
      FI_EN_E8PBA_FREV_START = 5UL,
      FI_EN_E8PBA_FREV_STOP = 6UL,
      FI_EN_E8PBA_FFWD_START = 7UL,
      FI_EN_E8PBA_FFWD_STOP = 8UL
   };
   tenType enType;

   most_fi_tcl_e8_MPlayAction(most_fi_tcl_e8_MPlayAction::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MPlayAction();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MPlayAction& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MPlayPlaypointFormat : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MPlayPlaypointFormat();
   enum tenType {
      FI_EN_E8PPF_PERCENTAGE = 0UL,
      FI_EN_E8PPF_ABSOLUTE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_MPlayPlaypointFormat(most_fi_tcl_e8_MPlayPlaypointFormat::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MPlayPlaypointFormat();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MPlayPlaypointFormat& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MPlayDeviceIndexedState : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MPlayDeviceIndexedState();
   enum tenType {
      FI_EN_E8IDS_NOT_SUPPORTED = 0UL,
      FI_EN_E8IDS_NOT_STARTED = 1UL,
      FI_EN_E8IDS_PARTIAL = 2UL,
      FI_EN_E8IDS_COMPLETE = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_MPlayDeviceIndexedState(most_fi_tcl_e8_MPlayDeviceIndexedState::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MPlayDeviceIndexedState();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MPlayDeviceIndexedState& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MPlayIndexingStateResultItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MPlayIndexingStateResultItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MPlayIndexingStateResultItem& operator=(const most_fi_tcl_MPlayIndexingStateResultItem& coRef);
   most_fi_tcl_MPlayIndexingStateResultItem(const most_fi_tcl_MPlayIndexingStateResultItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceTag;
   most_fi_tcl_e8_MPlayDeviceIndexedState e8DeviceIndexedState;
   tU8 u8IndexingPercentComplete;
   virtual ~most_fi_tcl_MPlayIndexingStateResultItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MPlayIndexingStateResultItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MPlayIndexingStateResult : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MPlayIndexingStateResult() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MPlayIndexingStateResult& operator=(const most_fi_tcl_MPlayIndexingStateResult& coRef);
   most_fi_tcl_MPlayIndexingStateResult(const most_fi_tcl_MPlayIndexingStateResult& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_MPlayIndexingStateResultItem, std::allocator<most_fi_tcl_MPlayIndexingStateResultItem> >oItems;

   most_fi_tcl_MPlayIndexingStateResultItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MPlayIndexingStateResult();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MPlayIndexingStateResult& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MPlayImageData : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MPlayImageData() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MPlayImageData& operator=(const most_fi_tcl_MPlayImageData& coRef);
   most_fi_tcl_MPlayImageData(const most_fi_tcl_MPlayImageData& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU8, std::allocator<tU8> >u8Items;

   tU8& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MPlayImageData();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MPlayImageData& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MPlaySearchKeyboardListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MPlaySearchKeyboardListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MPlaySearchKeyboardListItem& operator=(const most_fi_tcl_MPlaySearchKeyboardListItem& coRef);
   most_fi_tcl_MPlaySearchKeyboardListItem(const most_fi_tcl_MPlaySearchKeyboardListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sSearchKeyboardLetter;
   tBool bLetterAvailable;
   tU32 u32LetterStartIndex;
   tU32 u32LetterEndIndex;
   virtual ~most_fi_tcl_MPlaySearchKeyboardListItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MPlaySearchKeyboardListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MPlaySearchKeyboardList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MPlaySearchKeyboardList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MPlaySearchKeyboardList& operator=(const most_fi_tcl_MPlaySearchKeyboardList& coRef);
   most_fi_tcl_MPlaySearchKeyboardList(const most_fi_tcl_MPlaySearchKeyboardList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_MPlaySearchKeyboardListItem*, std::allocator<most_fi_tcl_MPlaySearchKeyboardListItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_MPlaySearchKeyboardListItem, std::allocator<most_fi_tcl_MPlaySearchKeyboardListItem> >oItems;

   most_fi_tcl_MPlaySearchKeyboardListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MPlaySearchKeyboardList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MPlaySearchKeyboardList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MPlayDeviceInfoItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MPlayDeviceInfoItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MPlayDeviceInfoItem& operator=(const most_fi_tcl_MPlayDeviceInfoItem& coRef);
   most_fi_tcl_MPlayDeviceInfoItem(const most_fi_tcl_MPlayDeviceInfoItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8DeviceTag;
   most_fi_tcl_String sDeviceName;
   most_fi_tcl_e8_MPlayDeviceType e8DeviceType;
   most_fi_tcl_String sDeviceSerialNumber;
   tBool bDeviceConnected;
   most_fi_tcl_e8_MPlayDeviceIndexedState e8DeviceIndexedState;
   tBool bDeviceActiveSource;
   virtual ~most_fi_tcl_MPlayDeviceInfoItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MPlayDeviceInfoItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MPlayDeviceInfo : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MPlayDeviceInfo() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MPlayDeviceInfo& operator=(const most_fi_tcl_MPlayDeviceInfo& coRef);
   most_fi_tcl_MPlayDeviceInfo(const most_fi_tcl_MPlayDeviceInfo& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_MPlayDeviceInfoItem*, std::allocator<most_fi_tcl_MPlayDeviceInfoItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_MPlayDeviceInfoItem, std::allocator<most_fi_tcl_MPlayDeviceInfoItem> >oItems;

   most_fi_tcl_MPlayDeviceInfoItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MPlayDeviceInfo();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MPlayDeviceInfo& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MPlayFileType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MPlayFileType();
   enum tenType {
      FI_EN_E8FT_FOLDER = 0UL,
      FI_EN_E8FT_AUDIO = 1UL,
      FI_EN_E8FT_VIDEO = 2UL,
      FI_EN_E8FT_PLAYLIST = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_MPlayFileType(most_fi_tcl_e8_MPlayFileType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MPlayFileType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MPlayFileType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MPlayFileListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MPlayFileListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MPlayFileListItem& operator=(const most_fi_tcl_MPlayFileListItem& coRef);
   most_fi_tcl_MPlayFileListItem(const most_fi_tcl_MPlayFileListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_MPlayFileType e8FileType;
   tU8 u8TrackNumber;
   most_fi_tcl_String sFilename;
   most_fi_tcl_e8_MPlayFileFormat e8FileFormat;
   tBool bIsPlaying;
   virtual ~most_fi_tcl_MPlayFileListItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MPlayFileListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MPlayFileList : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MPlayFileList() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MPlayFileList& operator=(const most_fi_tcl_MPlayFileList& coRef);
   most_fi_tcl_MPlayFileList(const most_fi_tcl_MPlayFileList& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_MPlayFileListItem*, std::allocator<most_fi_tcl_MPlayFileListItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_MPlayFileListItem, std::allocator<most_fi_tcl_MPlayFileListItem> >oItems;

   most_fi_tcl_MPlayFileListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MPlayFileList();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MPlayFileList& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_MPlayTagTransferStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_MPlayTagTransferStatus();
   enum tenType {
      FI_EN_E8TAG_TRANSFER_SUCCESS = 0UL,
      FI_EN_E8TAG_TRANSFER_COMM_ERROR = 1UL,
      FI_EN_E8TAG_TRANSFER_DEVICE_FULL = 2UL,
      FI_EN_E8TAG_TRANSFER_NOT_SUPPORTED = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_MPlayTagTransferStatus(most_fi_tcl_e8_MPlayTagTransferStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_MPlayTagTransferStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_MPlayTagTransferStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MPlayTagTransferItem_1 : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MPlayTagTransferItem_1();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MPlayTagTransferItem_1& operator=(const most_fi_tcl_MPlayTagTransferItem_1& coRef);
   most_fi_tcl_MPlayTagTransferItem_1(const most_fi_tcl_MPlayTagTransferItem_1& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8AmbiguousTag;
   tU8 u8ButtonPressed;
   most_fi_tcl_String sTagTrackName;
   most_fi_tcl_String sTagArtistName;
   most_fi_tcl_String sTagAlbum;
   most_fi_tcl_String sTagGenre;
   tU32 u32TagSongID;
   tU8 u8TagStorefrontID;
   most_fi_tcl_String sTagChannelNumber;
   most_fi_tcl_String sTagChannelName;
   most_fi_tcl_String sTagTimeStamp;
   most_fi_tcl_String sPodCastFeedURL;
   most_fi_tcl_String sStationURL;
   tU8 u8TagProgramNumber;
   most_fi_tcl_String sTagAffiliateID;
   most_fi_tcl_ShortStream oUnknownData;
   most_fi_tcl_String sStationURLFrequency;
   virtual ~most_fi_tcl_MPlayTagTransferItem_1();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MPlayTagTransferItem_1& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MPlayTagTransfer_1 : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MPlayTagTransfer_1() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MPlayTagTransfer_1& operator=(const most_fi_tcl_MPlayTagTransfer_1& coRef);
   most_fi_tcl_MPlayTagTransfer_1(const most_fi_tcl_MPlayTagTransfer_1& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_MPlayTagTransferItem_1*, std::allocator<most_fi_tcl_MPlayTagTransferItem_1*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_MPlayTagTransferItem_1, std::allocator<most_fi_tcl_MPlayTagTransferItem_1> >oItems;

   most_fi_tcl_MPlayTagTransferItem_1& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MPlayTagTransfer_1();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MPlayTagTransfer_1& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MPlayUntransferredTagStreamItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MPlayUntransferredTagStreamItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MPlayUntransferredTagStreamItem& operator=(const most_fi_tcl_MPlayUntransferredTagStreamItem& coRef);
   most_fi_tcl_MPlayUntransferredTagStreamItem(const most_fi_tcl_MPlayUntransferredTagStreamItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32TagSongID;
   most_fi_tcl_String sTagArtistName;
   most_fi_tcl_String sTagTrackName;
   virtual ~most_fi_tcl_MPlayUntransferredTagStreamItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MPlayUntransferredTagStreamItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MPlayUntransferredTagStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MPlayUntransferredTagStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MPlayUntransferredTagStream& operator=(const most_fi_tcl_MPlayUntransferredTagStream& coRef);
   most_fi_tcl_MPlayUntransferredTagStream(const most_fi_tcl_MPlayUntransferredTagStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_MPlayUntransferredTagStreamItem*, std::allocator<most_fi_tcl_MPlayUntransferredTagStreamItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_MPlayUntransferredTagStreamItem, std::allocator<most_fi_tcl_MPlayUntransferredTagStreamItem> >oItems;

   most_fi_tcl_MPlayUntransferredTagStreamItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MPlayUntransferredTagStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MPlayUntransferredTagStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MPlayTagTransferItem_2 : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MPlayTagTransferItem_2();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MPlayTagTransferItem_2& operator=(const most_fi_tcl_MPlayTagTransferItem_2& coRef);
   most_fi_tcl_MPlayTagTransferItem_2(const most_fi_tcl_MPlayTagTransferItem_2& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8AmbiguousTag;
   tU8 u8ButtonPressed;
   most_fi_tcl_String sTagTrackName;
   most_fi_tcl_String sTagArtistName;
   most_fi_tcl_String sTagAlbum;
   most_fi_tcl_String sTagGenre;
   tU32 u32TagSongID;
   tU32 u32TagStorefrontID;
   most_fi_tcl_String sTagChannelNumber;
   most_fi_tcl_String sTagChannelName;
   most_fi_tcl_String sTagTimeStamp;
   most_fi_tcl_String sPodCastFeedURL;
   most_fi_tcl_String sStationURL;
   tU8 u8TagProgramNumber;
   most_fi_tcl_String sTagAffiliateID;
   most_fi_tcl_ShortStream oUnknownData;
   most_fi_tcl_String sStationURLFrequency;
   virtual ~most_fi_tcl_MPlayTagTransferItem_2();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MPlayTagTransferItem_2& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_MPlayTagTransfer_2 : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_MPlayTagTransfer_2() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_MPlayTagTransfer_2& operator=(const most_fi_tcl_MPlayTagTransfer_2& coRef);
   most_fi_tcl_MPlayTagTransfer_2(const most_fi_tcl_MPlayTagTransfer_2& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_MPlayTagTransferItem_2*, std::allocator<most_fi_tcl_MPlayTagTransferItem_2*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_MPlayTagTransferItem_2, std::allocator<most_fi_tcl_MPlayTagTransferItem_2> >oItems;

   most_fi_tcl_MPlayTagTransferItem_2& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_MPlayTagTransfer_2();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_MPlayTagTransfer_2& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_NavDistanceUnits : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_NavDistanceUnits();
   enum tenType {
      FI_EN_E8METERS = 1UL,
      FI_EN_E8KILOMETERS = 2UL,
      FI_EN_E8YARDS = 3UL,
      FI_EN_E8FEET = 4UL,
      FI_EN_E8MILES = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_NavDistanceUnits(most_fi_tcl_e8_NavDistanceUnits::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_NavDistanceUnits();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_NavDistanceUnits& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_NavManeuverEnumeration : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_NavManeuverEnumeration();
   enum tenType {
      FI_EN_E8ME_NO_MANEUVER = 1UL,
      FI_EN_E8ME_CONTINUE_STRAIGHT = 2UL,
      FI_EN_E8ME_LEFT_TURN = 3UL,
      FI_EN_E8ME_RIGHT_TURN = 4UL,
      FI_EN_E8ME_SHARP_LEFT_TURN = 5UL,
      FI_EN_E8ME_SHARP__RIGHT_TURN = 6UL,
      FI_EN_E8ME_BEAR_LEFT = 7UL,
      FI_EN_E8ME_BEAR_RIGHT = 8UL,
      FI_EN_E8ME_MERGE_LEFT = 9UL,
      FI_EN_E8ME_MERGE_RIGHT = 10UL,
      FI_EN_E8ME_EXIT_LEFT = 11UL,
      FI_EN_E8ME_EXIT_RIGHT = 12UL,
      FI_EN_E8ME_U_TURN_LEFT = 13UL,
      FI_EN_E8ME_U_TURN_RIGHT = 14UL,
      FI_EN_E8ME_DESTINATION_AHEAD = 15UL,
      FI_EN_E8ME_AT_DESTINATION = 16UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_UNKNOWN_EXIT = 17UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_1ST_EXIT = 18UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_2ND_EXIT = 19UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_3RD_EXIT = 20UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_4TH_EXIT = 21UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_5TH_EXIT = 22UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_6TH_EXIT = 23UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_7TH_EXIT = 24UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_8TH_EXIT = 25UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_UNKNOWN_EXIT = 26UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_1ST_EXIT = 27UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_2ND_EXIT = 28UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_3RD_EXIT = 29UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_4TH_EXIT = 30UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_5TH_EXIT = 31UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_6TH_EXIT = 32UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_7TH_EXIT = 33UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_8TH_EXIT = 34UL,
      FI_EN_E8ME_0_ZERO_DEGREE_ARROW = 35UL,
      FI_EN_E8ME_45_ZERO_DEGREE_ARROW = 36UL,
      FI_EN_E8ME_90_ZERO_DEGREE_ARROW = 37UL,
      FI_EN_E8ME_135_ZERO_DEGREE_ARROW = 38UL,
      FI_EN_E8ME_180_ZERO_DEGREE_ARROW = 39UL,
      FI_EN_E8ME_225_ZERO_DEGREE_ARROW = 40UL,
      FI_EN_E8ME_270_ZERO_DEGREE_ARROW = 41UL,
      FI_EN_E8ME_315_ZERO_DEGREE_ARROW = 42UL,
      FI_EN_E8ME_REVERSE_DIRECTION = 43UL,
      FI_EN_E8ME_2ND_FORK_RIGHT = 44UL,
      FI_EN_E8ME_1ST_FORK_RIGHT = 45UL,
      FI_EN_E8ME_2ND_FORK_LEFT = 46UL,
      FI_EN_E8ME_1ST_FORK_LEFT = 47UL,
      FI_EN_E8ME_1ST_FORK_LEFT_4_OPTIONS = 48UL,
      FI_EN_E8ME_2ND_FORK_LEFT_4_OPTIONS = 49UL,
      FI_EN_E8ME_1ST_FORK_RIGHT_4_OPTIONS = 50UL,
      FI_EN_E8ME_2ND_FORK_RIGHT_4_OPTIONS = 51UL,
      FI_EN_E8ME_WAYPOINT_1 = 52UL,
      FI_EN_E8ME_WAYPOINT_2 = 53UL,
      FI_EN_E8ME_WAYPOINT_3 = 54UL,
      FI_EN_E8ME_WAYPOINT_4 = 55UL,
      FI_EN_E8ME_WAYPOINT_5 = 56UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_1ST_EXIT_30_DEGREE = 57UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_2ND_EXIT_30_DEGREE = 58UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_3RD_EXIT_30_DEGREE = 59UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_4TH_EXIT_30_DEGREE = 60UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_5TH_EXIT_30_DEGREE = 61UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_6TH_EXIT_30_DEGREE = 62UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_7TH_EXIT_30_DEGREE = 63UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_8TH_EXIT_30_DEGREE = 64UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_9TH_EXIT_30_DEGREE = 65UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_10TH_EXIT_30_DEGREE = 66UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_11TH_EXIT_30_DEGREE = 67UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_12TH_EXIT_30_DEGREE = 68UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_1ST_EXIT_30_DEGREE = 69UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_2ND_EXIT_30_DEGREE = 70UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_3RD_EXIT_30_DEGREE = 71UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_4TH_EXIT_30_DEGREE = 72UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_5TH_EXIT_30_DEGREE = 73UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_6TH_EXIT_30_DEGREE = 74UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_7TH_EXIT_30_DEGREE = 75UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_8TH_EXIT_30_DEGREE = 76UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_9TH_EXIT_30_DEGREE = 77UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_10TH_EXIT_30_DEGREE = 78UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_11TH_EXIT_30_DEGREE = 79UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_12TH_EXIT_30_DEGREE = 80UL,
      FI_EN_E8ME_WAYPOINT_6 = 81UL,
      FI_EN_E8ME_WAYPOINT_7 = 82UL,
      FI_EN_E8ME_WAYPOINT_8 = 83UL,
      FI_EN_E8ME_WAYPOINT_9 = 84UL,
      FI_EN_E8ME_WAYPOINT_10 = 85UL,
      FI_EN_E8ME_NOINFO = 86UL,
      FI_EN_E8ME_DESTINATION_AHEAD_LEFT = 87UL,
      FI_EN_E8ME_DESTINATION_AHEAD_RIGHT = 88UL,
      FI_EN_E8ME_AT_DESTINATION_LEFT = 89UL,
      FI_EN_E8ME_AT_DESTINATION_RIGHT = 90UL,
      FI_EN_E8ME_22_ZERO_DEGREE_ARROW = 91UL,
      FI_EN_E8ME_67_ZERO_DEGREE_ARROW = 92UL,
      FI_EN_E8ME_112_ZERO_DEGREE_ARROW = 93UL,
      FI_EN_E8ME_157_ZERO_DEGREE_ARROW = 94UL,
      FI_EN_E8ME_202_ZERO_DEGREE_ARROW = 95UL,
      FI_EN_E8ME_247_ZERO_DEGREE_ARROW = 96UL,
      FI_EN_E8ME_292_ZERO_DEGREE_ARROW = 97UL,
      FI_EN_E8ME_337_ZERO_DEGREE_ARROW = 98UL,
      FI_EN_E8ME_CALCULATE = 99UL,
      FI_EN_E8ME_RECALCULATE = 100UL,
      FI_EN_E8ME_TWO_FORK_LEFT = 101UL,
      FI_EN_E8ME_TWO_FORK_RIGHT = 102UL,
      FI_EN_E8ME_THREE_FORK_LEFT = 103UL,
      FI_EN_E8ME_THREE_FORK_RIGHT = 104UL,
      FI_EN_E8ME_THREE_FORK_MIDDLE = 105UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_22_DEGREE = 106UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_67_DEGREE = 107UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_112_DEGREE = 108UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_135_DEGREE = 109UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_157_DEGREE = 110UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_202_DEGREE = 111UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_247_DEGREE = 112UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_292_DEGREE = 113UL,
      FI_EN_E8ME_COUNTER_CLOCKWISE_TRAFFIC_CIRCLE_337_DEGREE = 114UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_22_DEGREE = 115UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_67_DEGREE = 116UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_112_DEGREE = 117UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_135_DEGREE = 118UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_157_DEGREE = 119UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_202_DEGREE = 120UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_247_DEGREE = 121UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_292_DEGREE = 122UL,
      FI_EN_E8ME_CLOCKWISE_TRAFFIC_CIRCLE_337_DEGREE = 123UL,
      FI_EN_E8ME_PREPARE_LEFT = 124UL,
      FI_EN_E8ME_PREPARE_RIGHT = 125UL
   };
   tenType enType;

   most_fi_tcl_e8_NavManeuverEnumeration(most_fi_tcl_e8_NavManeuverEnumeration::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_NavManeuverEnumeration();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_NavManeuverEnumeration& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_NavGuidanceStatus : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_NavGuidanceStatus();
   enum tenType {
      FI_EN_E8GA_GUIDANCE_INACTIVE = 0UL,
      FI_EN_E8GA_GUIDANCE_ACTIVE = 1UL,
      FI_EN_E8GA_GUIDANCE_SUSPENDED = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_NavGuidanceStatus(most_fi_tcl_e8_NavGuidanceStatus::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_NavGuidanceStatus();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_NavGuidanceStatus& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_NavManeuverListStreamItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_NavManeuverListStreamItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_NavManeuverListStreamItem& operator=(const most_fi_tcl_NavManeuverListStreamItem& coRef);
   most_fi_tcl_NavManeuverListStreamItem(const most_fi_tcl_NavManeuverListStreamItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 u16ManeuverID;
   tU16 u16ManeuverNumber;
   most_fi_tcl_e8_NavManeuverEnumeration e8ManeuverEnumeration;
   most_fi_tcl_String sManeuverStreetName;
   most_fi_tcl_String sManueverDistance;
   tU32 u32ManeuverTime;
   tBool bLatLongValidity;
   tS32 s32Latitude;
   tS32 s32Longitude;
   most_fi_tcl_e8_NavDistanceUnits e8DistanceUnits;
   virtual ~most_fi_tcl_NavManeuverListStreamItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_NavManeuverListStreamItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_NavManeuverListStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_NavManeuverListStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_NavManeuverListStream& operator=(const most_fi_tcl_NavManeuverListStream& coRef);
   most_fi_tcl_NavManeuverListStream(const most_fi_tcl_NavManeuverListStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_NavManeuverListStreamItem*, std::allocator<most_fi_tcl_NavManeuverListStreamItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_NavManeuverListStreamItem, std::allocator<most_fi_tcl_NavManeuverListStreamItem> >oItems;

   most_fi_tcl_NavManeuverListStreamItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_NavManeuverListStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_NavManeuverListStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_NavPOIListTypeEnum : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_NavPOIListTypeEnum();
   enum tenType {
      FI_EN_E8POI_LIST_ONBOARD = 1UL,
      FI_EN_E8POI_LIST_OFFBOARD = 3UL,
      FI_EN_E8POI_LIST_ADDRESS_BOOK = 4UL,
      FI_EN_E8POI_LIST_ALL = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_NavPOIListTypeEnum(most_fi_tcl_e8_NavPOIListTypeEnum::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_NavPOIListTypeEnum();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_NavPOIListTypeEnum& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_NavPOI_ListItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_NavPOI_ListItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_NavPOI_ListItem& operator=(const most_fi_tcl_NavPOI_ListItem& coRef);
   most_fi_tcl_NavPOI_ListItem(const most_fi_tcl_NavPOI_ListItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32DestinationId;
   most_fi_tcl_String sPOIName;
   most_fi_tcl_String sPOICategoryName;
   most_fi_tcl_String sDestinationIconURL;
   most_fi_tcl_String sPOICategoryIconURL;
   most_fi_tcl_String sPOIBrandIconURL;
   most_fi_tcl_String sDestinationTag;
   most_fi_tcl_String sDestHouseNumber;
   most_fi_tcl_String sDestCityName;
   most_fi_tcl_String sDestCountry;
   most_fi_tcl_String sDestPhoneNo;
   most_fi_tcl_String sDestPostalCode;
   most_fi_tcl_String sDestState;
   most_fi_tcl_String sDestStreetName;
   virtual ~most_fi_tcl_NavPOI_ListItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_NavPOI_ListItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_NavPOI_List : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_NavPOI_List() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_NavPOI_List& operator=(const most_fi_tcl_NavPOI_List& coRef);
   most_fi_tcl_NavPOI_List(const most_fi_tcl_NavPOI_List& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_NavPOI_ListItem*, std::allocator<most_fi_tcl_NavPOI_ListItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_NavPOI_ListItem, std::allocator<most_fi_tcl_NavPOI_ListItem> >oItems;

   most_fi_tcl_NavPOI_ListItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_NavPOI_List();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_NavPOI_List& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_NavDestinationListTypeEnum : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_NavDestinationListTypeEnum();
   enum tenType {
      FI_EN_E8DL_RECENT_DESTINATION_LIST = 1UL,
      FI_EN_E8DL_NAVIGATION_CONTACTS_LIST = 2UL,
      FI_EN_E8DL_FAVORITE_DESTINATIONS_LIST = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_NavDestinationListTypeEnum(most_fi_tcl_e8_NavDestinationListTypeEnum::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_NavDestinationListTypeEnum();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_NavDestinationListTypeEnum& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_NavDestinationInfoItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_NavDestinationInfoItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_NavDestinationInfoItem& operator=(const most_fi_tcl_NavDestinationInfoItem& coRef);
   most_fi_tcl_NavDestinationInfoItem(const most_fi_tcl_NavDestinationInfoItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU32 u32DestinationId;
   most_fi_tcl_String sDestinationName;
   virtual ~most_fi_tcl_NavDestinationInfoItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_NavDestinationInfoItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_NavDestinationInfo : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_NavDestinationInfo() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_NavDestinationInfo& operator=(const most_fi_tcl_NavDestinationInfo& coRef);
   most_fi_tcl_NavDestinationInfo(const most_fi_tcl_NavDestinationInfo& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_NavDestinationInfoItem*, std::allocator<most_fi_tcl_NavDestinationInfoItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_NavDestinationInfoItem, std::allocator<most_fi_tcl_NavDestinationInfoItem> >oItems;

   most_fi_tcl_NavDestinationInfoItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_NavDestinationInfo();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_NavDestinationInfo& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_NavListChangeType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_NavListChangeType();
   enum tenType {
      FI_EN_E8LCH_CONTENT_CHANGED = 0UL,
      FI_EN_E8LCH_ITEMS_ADDED = 1UL,
      FI_EN_E8LCH_ITEMS_REMOVED = 2UL,
      FI_EN_E8LCH_ITEMS_CHANGED = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_NavListChangeType(most_fi_tcl_e8_NavListChangeType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_NavListChangeType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_NavListChangeType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_NavChangedItemIndexes : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_NavChangedItemIndexes() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_NavChangedItemIndexes& operator=(const most_fi_tcl_NavChangedItemIndexes& coRef);
   most_fi_tcl_NavChangedItemIndexes(const most_fi_tcl_NavChangedItemIndexes& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<tU32, std::allocator<tU32> >u32Items;

   tU32& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_NavChangedItemIndexes();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_NavChangedItemIndexes& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_NavCurrentVehiclePositionStreamItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_NavCurrentVehiclePositionStreamItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_NavCurrentVehiclePositionStreamItem& operator=(const most_fi_tcl_NavCurrentVehiclePositionStreamItem& coRef);
   most_fi_tcl_NavCurrentVehiclePositionStreamItem(const most_fi_tcl_NavCurrentVehiclePositionStreamItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU8 u8CVPSequenceCounter;
   tS16 s16CVPXOffset;
   tS16 s16CVPYOffset;
   tS16 s16CVPTheta;
   virtual ~most_fi_tcl_NavCurrentVehiclePositionStreamItem();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_NavCurrentVehiclePositionStreamItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_NavCurrentVehiclePositionStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_NavCurrentVehiclePositionStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_NavCurrentVehiclePositionStream& operator=(const most_fi_tcl_NavCurrentVehiclePositionStream& coRef);
   most_fi_tcl_NavCurrentVehiclePositionStream(const most_fi_tcl_NavCurrentVehiclePositionStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
   std::vector<most_fi_tcl_NavCurrentVehiclePositionStreamItem, std::allocator<most_fi_tcl_NavCurrentVehiclePositionStreamItem> >oItems;

   most_fi_tcl_NavCurrentVehiclePositionStreamItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_NavCurrentVehiclePositionStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_NavCurrentVehiclePositionStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_NavManeuverExtendedListStreamItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_NavManeuverExtendedListStreamItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_NavManeuverExtendedListStreamItem& operator=(const most_fi_tcl_NavManeuverExtendedListStreamItem& coRef);
   most_fi_tcl_NavManeuverExtendedListStreamItem(const most_fi_tcl_NavManeuverExtendedListStreamItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   tU16 u16ManeuverID;
   tU16 u16ManeuverNumber;
   most_fi_tcl_e8_NavManeuverEnumeration e8ManeuverEnumeration;
   most_fi_tcl_String sManeuverStreetName;
   most_fi_tcl_String sManueverDistance;
   tU32 u32ManeuverTime;
   tBool bLatLongValidity;
   tS32 s32Latitude;
   tS32 s32Longitude;
   most_fi_tcl_e8_NavDistanceUnits e8DistanceUnits;
   tBool bTollBoothPresent;
   tBool bIsAbstractManeuver;
   tBool bHasManeuverStreetNameIcon;
   virtual ~most_fi_tcl_NavManeuverExtendedListStreamItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_NavManeuverExtendedListStreamItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_NavManeuverExtendedListStream : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_NavManeuverExtendedListStream() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_NavManeuverExtendedListStream& operator=(const most_fi_tcl_NavManeuverExtendedListStream& coRef);
   most_fi_tcl_NavManeuverExtendedListStream(const most_fi_tcl_NavManeuverExtendedListStream& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_NavManeuverExtendedListStreamItem*, std::allocator<most_fi_tcl_NavManeuverExtendedListStreamItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_NavManeuverExtendedListStreamItem, std::allocator<most_fi_tcl_NavManeuverExtendedListStreamItem> >oItems;

   most_fi_tcl_NavManeuverExtendedListStreamItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_NavManeuverExtendedListStream();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_NavManeuverExtendedListStream& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_NavAddressDisplayFormat : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_NavAddressDisplayFormat();
   enum tenType {
      FI_EN_E8HOUSENUMBER_STREETNAME = 0UL,
      FI_EN_E8STREETNAME_HOUSENUMBER = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_NavAddressDisplayFormat(most_fi_tcl_e8_NavAddressDisplayFormat::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_NavAddressDisplayFormat();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_NavAddressDisplayFormat& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsAudButtonEnum : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsAudButtonEnum();
   enum tenType {
      FI_EN_E8PTT = 0UL,
      FI_EN_E8END = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsAudButtonEnum(most_fi_tcl_e8_OnsAudButtonEnum::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsAudButtonEnum();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsAudButtonEnum& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsAudButtonEvent : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsAudButtonEvent();
   enum tenType {
      FI_EN_E8BUTTON_EVENT_QUICK = 0UL,
      FI_EN_E8BUTTON_EVENT_SHORT = 1UL,
      FI_EN_E8BUTTON_EVENT_MEDIUM = 2UL,
      FI_EN_E8BUTTON_EVENT_LONG = 3UL,
      FI_EN_E8BUTTON_EVENT_EXTRA_LONG = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsAudButtonEvent(most_fi_tcl_e8_OnsAudButtonEvent::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsAudButtonEvent();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsAudButtonEvent& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsDatOnStarDataSettingsValidEnum : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsDatOnStarDataSettingsValidEnum();
   enum tenType {
      FI_EN_E8FU_INVALID = 0UL,
      FI_EN_E8FU_VALID = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsDatOnStarDataSettingsValidEnum(most_fi_tcl_e8_OnsDatOnStarDataSettingsValidEnum::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsDatOnStarDataSettingsValidEnum();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsDatOnStarDataSettingsValidEnum& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsDatOnStarDataSettingsEventEnum : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsDatOnStarDataSettingsEventEnum();
   enum tenType {
      FI_EN_E8ONSTARDATASETTINGSFUCLASSRESET = 0UL,
      FI_EN_E8ONSTARCALLCONNECTING = 1UL,
      FI_EN_E8ONSTARCALLCONNECTED = 2UL,
      FI_EN_E8ONSTARCALLENDED = 3UL,
      FI_EN_E8ERRORCALLCONNECTIONFAILED = 4UL,
      FI_EN_E8SHOWINVEHICLEALERTDISPLAY = 5UL,
      FI_EN_E8ENDINVEHICLEALERTDISPLAY = 6UL,
      FI_EN_E8DESTINATIONENTRYASSIST = 7UL,
      FI_EN_E8ONSTARLANGUAGECHANGE = 9UL,
      FI_EN_E8SHOWMINDIGITS = 10UL,
      FI_EN_E8SHOWACQSETTING = 11UL,
      FI_EN_E8ENDUACDISPLAY = 12UL,
      FI_EN_E8INCOMINGONSTARCALL = 13UL,
      FI_EN_E8SHOWAUDIOMESSAGETEXT = 14UL,
      FI_EN_E8SHOWONSTARWELCOMETEXT = 15UL,
      FI_EN_E8SHOWXMDEACTIVATEDTEXT = 17UL,
      FI_EN_E8SHOWONSTARMINUTESTEXT = 18UL,
      FI_EN_E8ENDONSTARMESSAGETEXT = 20UL,
      FI_EN_E8ACTIVATEMESSAGEDIALOG = 27UL,
      FI_EN_E8ACTIVATE1BUTTONMESSAGEDIALOG = 28UL,
      FI_EN_E8ACTIVATE2BUTTONMESSAGEDIALOG = 29UL,
      FI_EN_E8ACTIVATETITLEDMENUTEMPLATE = 30UL,
      FI_EN_E8DEACTIVATEMENU = 31UL,
      FI_EN_E8TTYFUNCTIONRESTRICTION = 32UL,
      FI_EN_E8ONSTARPHONEBUTTONPRESSED = 33UL,
      FI_EN_E8CANCELNAVIGATIONROUTE = 34UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsDatOnStarDataSettingsEventEnum(most_fi_tcl_e8_OnsDatOnStarDataSettingsEventEnum::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsDatOnStarDataSettingsEventEnum();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsDatOnStarDataSettingsEventEnum& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsDatPlaceOnStarCallValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsDatPlaceOnStarCallValue();
   enum tenType {
      FI_EN_E8EMERGENCY = 0UL,
      FI_EN_E8ROADSIDEASSISTANCE = 1UL,
      FI_EN_E8VIRTUALADVISOR = 2UL,
      FI_EN_E8UNITADD = 3UL,
      FI_EN_E8ENROLLMENT = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsDatPlaceOnStarCallValue(most_fi_tcl_e8_OnsDatPlaceOnStarCallValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsDatPlaceOnStarCallValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsDatPlaceOnStarCallValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsDatButton1SelectedValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsDatButton1SelectedValue();
   enum tenType {
      FI_EN_E8PRESSED = 0UL,
      FI_EN_E8PRESSHOLD = 1UL,
      FI_EN_E8RELEASED = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsDatButton1SelectedValue(most_fi_tcl_e8_OnsDatButton1SelectedValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsDatButton1SelectedValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsDatButton1SelectedValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsDatButton2SelectedValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsDatButton2SelectedValue();
   enum tenType {
      FI_EN_E8PRESSED = 0UL,
      FI_EN_E8PRESSHOLD = 1UL,
      FI_EN_E8RELEASED = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsDatButton2SelectedValue(most_fi_tcl_e8_OnsDatButton2SelectedValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsDatButton2SelectedValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsDatButton2SelectedValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsDatBluetooth_InCallStatusValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsDatBluetooth_InCallStatusValue();
   enum tenType {
      FI_EN_E8HANDSFREE_BT_CALL_NOT_ACTIVE = 0UL,
      FI_EN_E8HANDSFREE_BT_CALL_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsDatBluetooth_InCallStatusValue(most_fi_tcl_e8_OnsDatBluetooth_InCallStatusValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsDatBluetooth_InCallStatusValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsDatBluetooth_InCallStatusValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsDatVoiceRecognitionStatusValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsDatVoiceRecognitionStatusValue();
   enum tenType {
      FI_EN_E8VOICE_RECOGNITION_NOT_ACTIVE = 0UL,
      FI_EN_E8VOICE_RECOGNITION_ACTIVE = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsDatVoiceRecognitionStatusValue(most_fi_tcl_e8_OnsDatVoiceRecognitionStatusValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsDatVoiceRecognitionStatusValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsDatVoiceRecognitionStatusValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsDatOnStarCallTypeValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsDatOnStarCallTypeValue();
   enum tenType {
      FI_EN_E8COLLISION = 0UL,
      FI_EN_E8EMERGENCY = 1UL,
      FI_EN_E8ROADSIDEASSISTANCE = 2UL,
      FI_EN_E8VIRTUALADVISOR = 3UL,
      FI_EN_E8UNITADD = 4UL,
      FI_EN_E8VEHICLEDATAUPLOAD = 5UL,
      FI_EN_E8ENROLLMENT = 6UL,
      FI_EN_E8OCCINITIATED = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsDatOnStarCallTypeValue(most_fi_tcl_e8_OnsDatOnStarCallTypeValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsDatOnStarCallTypeValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsDatOnStarCallTypeValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsDatPacketConnectionTypeValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsDatPacketConnectionTypeValue();
   enum tenType {
      FI_EN_E8USERROUTEDOWNLOAD = 0UL,
      FI_EN_E8INVEHICLEALERTDOWNLOAD = 1UL,
      FI_EN_E8VEHICLEDATAUPLOAD = 2UL,
      FI_EN_E8CDMETADATAREQUEST = 3UL,
      FI_EN_E8GENERALPACKETREQUEST = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsDatPacketConnectionTypeValue(most_fi_tcl_e8_OnsDatPacketConnectionTypeValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsDatPacketConnectionTypeValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsDatPacketConnectionTypeValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsDatInVehicleAlertTypeValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsDatInVehicleAlertTypeValue();
   enum tenType {
      FI_EN_E8DISASTER = 0UL,
      FI_EN_E8AMBER = 1UL,
      FI_EN_E8TRAFFIC = 2UL,
      FI_EN_E8WEATHER = 3UL,
      FI_EN_E8GENERIC = 4UL,
      FI_EN_E8CAMPAIGN = 5UL,
      FI_EN_E8REMINDER = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsDatInVehicleAlertTypeValue(most_fi_tcl_e8_OnsDatInVehicleAlertTypeValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsDatInVehicleAlertTypeValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsDatInVehicleAlertTypeValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsDatOnStarLanguageSettingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsDatOnStarLanguageSettingValue();
   enum tenType {
      FI_EN_E8NORTHAMERICANENGLISH = 0UL,
      FI_EN_E8NORTHAMERICANFRENCH = 1UL,
      FI_EN_E8NORTHAMERICANSPANISH = 2UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsDatOnStarLanguageSettingValue(most_fi_tcl_e8_OnsDatOnStarLanguageSettingValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsDatOnStarLanguageSettingValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsDatOnStarLanguageSettingValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsDatSubmenuIndicator : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsDatSubmenuIndicator();
   enum tenType {
      FI_EN_E8NOSUBMENU = 0UL,
      FI_EN_E8SUBMENU = 1UL,
      FI_EN_E8UNSELECTEDITEM = 2UL,
      FI_EN_E8SELECTEDITEM = 3UL,
      FI_EN_E8UNCHECKEDBOX = 4UL,
      FI_EN_E8CHECKEDBOX = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsDatSubmenuIndicator(most_fi_tcl_e8_OnsDatSubmenuIndicator::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsDatSubmenuIndicator();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsDatSubmenuIndicator& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_OnsDatMenuListValueItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_OnsDatMenuListValueItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_OnsDatMenuListValueItem& operator=(const most_fi_tcl_OnsDatMenuListValueItem& coRef);
   most_fi_tcl_OnsDatMenuListValueItem(const most_fi_tcl_OnsDatMenuListValueItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_String sListRecordString;
   most_fi_tcl_String sSettingString;
   most_fi_tcl_e8_OnsDatSubmenuIndicator e8SubmenuIndicator;
   virtual ~most_fi_tcl_OnsDatMenuListValueItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_OnsDatMenuListValueItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_OnsDatMenuListValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_OnsDatMenuListValue() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_OnsDatMenuListValue& operator=(const most_fi_tcl_OnsDatMenuListValue& coRef);
   most_fi_tcl_OnsDatMenuListValue(const most_fi_tcl_OnsDatMenuListValue& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_OnsDatMenuListValueItem*, std::allocator<most_fi_tcl_OnsDatMenuListValueItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_OnsDatMenuListValueItem, std::allocator<most_fi_tcl_OnsDatMenuListValueItem> >oItems;

   most_fi_tcl_OnsDatMenuListValueItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_OnsDatMenuListValue();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_OnsDatMenuListValue& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsDatTTYFunctionRestrictionTypeValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsDatTTYFunctionRestrictionTypeValue();
   enum tenType {
      FI_EN_E8PHONEBUTTON3BA = 0UL,
      FI_EN_E8ROADSIDEASSISTANCEVEHICLEMOVING = 1UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsDatTTYFunctionRestrictionTypeValue(most_fi_tcl_e8_OnsDatTTYFunctionRestrictionTypeValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsDatTTYFunctionRestrictionTypeValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsDatTTYFunctionRestrictionTypeValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsDatEmbeddedCellularRssiTypeValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsDatEmbeddedCellularRssiTypeValue();
   enum tenType {
      FI_EN_E80_BARS = 0UL,
      FI_EN_E81_BAR = 1UL,
      FI_EN_E82_BARS = 2UL,
      FI_EN_E83_BARS = 3UL,
      FI_EN_E84_BARS = 4UL,
      FI_EN_E85_BARS = 5UL,
      FI_EN_E80_BARS_ROAMING = 6UL,
      FI_EN_E81_BAR_ROAMING = 7UL,
      FI_EN_E82_BARS_ROAMING = 8UL,
      FI_EN_E83_BARS_ROAMING = 9UL,
      FI_EN_E84_BARS_ROAMING = 10UL,
      FI_EN_E85_BARS_ROAMING = 11UL,
      FI_EN_E8NO_INDICATION = 12UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsDatEmbeddedCellularRssiTypeValue(most_fi_tcl_e8_OnsDatEmbeddedCellularRssiTypeValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsDatEmbeddedCellularRssiTypeValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsDatEmbeddedCellularRssiTypeValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsDatEmbeddedCellularTechnologyTypeValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsDatEmbeddedCellularTechnologyTypeValue();
   enum tenType {
      FI_EN_E8NO_SERVICE = 0UL,
      FI_EN_E82G = 1UL,
      FI_EN_E83G_WIFI = 2UL,
      FI_EN_E84G_WIFI = 3UL,
      FI_EN_E84G_LTE_WIFI = 4UL,
      FI_EN_E82G_WIFI = 5UL,
      FI_EN_E8NO_INDICATION = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsDatEmbeddedCellularTechnologyTypeValue(most_fi_tcl_e8_OnsDatEmbeddedCellularTechnologyTypeValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsDatEmbeddedCellularTechnologyTypeValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsDatEmbeddedCellularTechnologyTypeValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsCalHandsFreePhoneEventEnum : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsCalHandsFreePhoneEventEnum();
   enum tenType {
      FI_EN_E8HANDSFREEPHONEFUCLASSRESET = 0UL,
      FI_EN_E8C1_CONNECTED = 1UL,
      FI_EN_E8C1_CONNECTEDCONFERENCE = 2UL,
      FI_EN_E8C1_DIALING = 3UL,
      FI_EN_E8C2_CONNECTED = 4UL,
      FI_EN_E8C2_CONNECTEDCONFERENCE = 5UL,
      FI_EN_E8C2_DIALING = 6UL,
      FI_EN_E8CALL_DIALINGFAILED = 7UL,
      FI_EN_E8CONF_DETACHFAILED = 8UL,
      FI_EN_E8CONF_MERGEDFAILED = 9UL,
      FI_EN_E8EMERGENCYCALL_CONNECTED = 10UL,
      FI_EN_E8EMERGENCYCALL_DIALING = 11UL,
      FI_EN_E8PHONE_EXTERNALPHONEINCRADLEREMAINDER = 12UL,
      FI_EN_E8PHONE_FACTORYSETTINGSRESTORED = 13UL,
      FI_EN_E8PHONE_FAILURE = 14UL,
      FI_EN_E8PHONE_IDLE = 15UL,
      FI_EN_E8PHONE_INITIATION = 16UL,
      FI_EN_E8PHONE_STANDBY = 17UL,
      FI_EN_E8SECURITY_FAILURE = 18UL,
      FI_EN_E8SECURITY_MISSMATCH = 19UL,
      FI_EN_E8SECURITY_REPLACE_MISSMATCH = 20UL,
      FI_EN_E8SECURITY_REPLACE_SUCCESFUL = 21UL,
      FI_EN_E8SECURITY_REPLACE_WRONGSEC1 = 22UL,
      FI_EN_E8SECURITY_SEC1_REQUESTED = 23UL,
      FI_EN_E8SECURITY_SEC2_REQUESTED = 24UL,
      FI_EN_E8SECURITY_TOGGLEENABLE_SUCCESFUL = 25UL,
      FI_EN_E8SECURITY_TOGGLEENABLE_WRONGSEC1 = 26UL,
      FI_EN_E8SECURITY_WRONGSEC1 = 27UL,
      FI_EN_E8SECURITY_WRONGSEC2 = 28UL,
      FI_EN_E8TRANSFER_CALLFAILURE = 29UL,
      FI_EN_E8TRANSFER_CALLSUCCESS = 30UL,
      FI_EN_E8C2_DIALINGCONFERENCE = 31UL,
      FI_EN_E8CALL_ACTIVESWAPPED = 32UL,
      FI_EN_E8CLT_DELETING = 33UL,
      FI_EN_E8CLT_ALLLISTSDELETED = 34UL,
      FI_EN_E8CLT_CALLDELETED = 35UL,
      FI_EN_E8FUNCTIONNOTAVAILABLE = 36UL,
      FI_EN_E8CALL_BUSY = 37UL,
      FI_EN_E8CLT_DELETEFAILED = 38UL,
      FI_EN_E8CONF_DETACHSUCCESSFUL = 39UL,
      FI_EN_E8CONF_MERGESUCCESSFUL = 40UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsCalHandsFreePhoneEventEnum(most_fi_tcl_e8_OnsCalHandsFreePhoneEventEnum::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsCalHandsFreePhoneEventEnum();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsCalHandsFreePhoneEventEnum& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsCalRingTone_SelectValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsCalRingTone_SelectValue();
   enum tenType {
      FI_EN_E8RINGTONE_1 = 0UL,
      FI_EN_E8RINGTONE_2 = 1UL,
      FI_EN_E8RINGTONE_3 = 2UL,
      FI_EN_E8DEPRECATED_RINGTONE_4 = 3UL,
      FI_EN_E8DEPRECATED_RINGTONE_5 = 4UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsCalRingTone_SelectValue(most_fi_tcl_e8_OnsCalRingTone_SelectValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsCalRingTone_SelectValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsCalRingTone_SelectValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsCalC1_CallStateValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsCalC1_CallStateValue();
   enum tenType {
      FI_EN_E8IDLE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8ONHOLD = 2UL,
      FI_EN_E8CONFERENCE = 3UL,
      FI_EN_E8DIALING = 4UL,
      FI_EN_E8RINGING = 5UL,
      FI_EN_E8BUSY = 6UL,
      FI_EN_E8CONFERENCE_ONHOLD = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsCalC1_CallStateValue(most_fi_tcl_e8_OnsCalC1_CallStateValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsCalC1_CallStateValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsCalC1_CallStateValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsCalC1_NumberTypeValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsCalC1_NumberTypeValue();
   enum tenType {
      FI_EN_E8HOME = 0UL,
      FI_EN_E8WORK = 1UL,
      FI_EN_E8CELL = 2UL,
      FI_EN_E8OTHER = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsCalC1_NumberTypeValue(most_fi_tcl_e8_OnsCalC1_NumberTypeValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsCalC1_NumberTypeValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsCalC1_NumberTypeValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsCalC2_CallStateValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsCalC2_CallStateValue();
   enum tenType {
      FI_EN_E8IDLE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8ONHOLD = 2UL,
      FI_EN_E8CONFERENCE = 3UL,
      FI_EN_E8DIALING = 4UL,
      FI_EN_E8RINGING = 5UL,
      FI_EN_E8BUSY = 6UL,
      FI_EN_E8CONFERENCE_ONHOLD = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsCalC2_CallStateValue(most_fi_tcl_e8_OnsCalC2_CallStateValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsCalC2_CallStateValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsCalC2_CallStateValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsCalC2_NumberTypeValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsCalC2_NumberTypeValue();
   enum tenType {
      FI_EN_E8HOME = 0UL,
      FI_EN_E8WORK = 1UL,
      FI_EN_E8CELL = 2UL,
      FI_EN_E8OTHER = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsCalC2_NumberTypeValue(most_fi_tcl_e8_OnsCalC2_NumberTypeValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsCalC2_NumberTypeValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsCalC2_NumberTypeValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsCalnumberType : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsCalnumberType();
   enum tenType {
      FI_EN_E8HOME = 0UL,
      FI_EN_E8WORK = 1UL,
      FI_EN_E8CELL = 2UL,
      FI_EN_E8OTHER = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsCalnumberType(most_fi_tcl_e8_OnsCalnumberType::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsCalnumberType();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsCalnumberType& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_OnsCalCLT_IncomingValueItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_OnsCalCLT_IncomingValueItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_OnsCalCLT_IncomingValueItem& operator=(const most_fi_tcl_OnsCalCLT_IncomingValueItem& coRef);
   most_fi_tcl_OnsCalCLT_IncomingValueItem(const most_fi_tcl_OnsCalCLT_IncomingValueItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_OnsCalnumberType e8numberType;
   most_fi_tcl_String sfirstName;
   most_fi_tcl_String slastName;
   most_fi_tcl_String sphoneNumber;
   virtual ~most_fi_tcl_OnsCalCLT_IncomingValueItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_OnsCalCLT_IncomingValueItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_OnsCalCLT_IncomingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_OnsCalCLT_IncomingValue() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_OnsCalCLT_IncomingValue& operator=(const most_fi_tcl_OnsCalCLT_IncomingValue& coRef);
   most_fi_tcl_OnsCalCLT_IncomingValue(const most_fi_tcl_OnsCalCLT_IncomingValue& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_OnsCalCLT_IncomingValueItem*, std::allocator<most_fi_tcl_OnsCalCLT_IncomingValueItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_OnsCalCLT_IncomingValueItem, std::allocator<most_fi_tcl_OnsCalCLT_IncomingValueItem> >oItems;

   most_fi_tcl_OnsCalCLT_IncomingValueItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_OnsCalCLT_IncomingValue();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_OnsCalCLT_IncomingValue& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_OnsCalCLT_MissedValueItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_OnsCalCLT_MissedValueItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_OnsCalCLT_MissedValueItem& operator=(const most_fi_tcl_OnsCalCLT_MissedValueItem& coRef);
   most_fi_tcl_OnsCalCLT_MissedValueItem(const most_fi_tcl_OnsCalCLT_MissedValueItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_OnsCalnumberType e8numberType;
   most_fi_tcl_String sfirstName;
   most_fi_tcl_String slastName;
   most_fi_tcl_String sphoneNumber;
   virtual ~most_fi_tcl_OnsCalCLT_MissedValueItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_OnsCalCLT_MissedValueItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_OnsCalCLT_MissedValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_OnsCalCLT_MissedValue() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_OnsCalCLT_MissedValue& operator=(const most_fi_tcl_OnsCalCLT_MissedValue& coRef);
   most_fi_tcl_OnsCalCLT_MissedValue(const most_fi_tcl_OnsCalCLT_MissedValue& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_OnsCalCLT_MissedValueItem*, std::allocator<most_fi_tcl_OnsCalCLT_MissedValueItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_OnsCalCLT_MissedValueItem, std::allocator<most_fi_tcl_OnsCalCLT_MissedValueItem> >oItems;

   most_fi_tcl_OnsCalCLT_MissedValueItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_OnsCalCLT_MissedValue();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_OnsCalCLT_MissedValue& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_OnsCalCLT_OutgoingValueItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_OnsCalCLT_OutgoingValueItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_OnsCalCLT_OutgoingValueItem& operator=(const most_fi_tcl_OnsCalCLT_OutgoingValueItem& coRef);
   most_fi_tcl_OnsCalCLT_OutgoingValueItem(const most_fi_tcl_OnsCalCLT_OutgoingValueItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_OnsCalnumberType e8numberType;
   most_fi_tcl_String sfirstName;
   most_fi_tcl_String slastName;
   most_fi_tcl_String sphoneNumber;
   virtual ~most_fi_tcl_OnsCalCLT_OutgoingValueItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_OnsCalCLT_OutgoingValueItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_OnsCalCLT_OutgoingValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_OnsCalCLT_OutgoingValue() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_OnsCalCLT_OutgoingValue& operator=(const most_fi_tcl_OnsCalCLT_OutgoingValue& coRef);
   most_fi_tcl_OnsCalCLT_OutgoingValue(const most_fi_tcl_OnsCalCLT_OutgoingValue& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_OnsCalCLT_OutgoingValueItem*, std::allocator<most_fi_tcl_OnsCalCLT_OutgoingValueItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_OnsCalCLT_OutgoingValueItem, std::allocator<most_fi_tcl_OnsCalCLT_OutgoingValueItem> >oItems;

   most_fi_tcl_OnsCalCLT_OutgoingValueItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_OnsCalCLT_OutgoingValue();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_OnsCalCLT_OutgoingValue& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_OnsCalConf_MembersValueItem : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_OnsCalConf_MembersValueItem();

#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_OnsCalConf_MembersValueItem& operator=(const most_fi_tcl_OnsCalConf_MembersValueItem& coRef);
   most_fi_tcl_OnsCalConf_MembersValueItem(const most_fi_tcl_OnsCalConf_MembersValueItem& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   most_fi_tcl_e8_OnsCalnumberType e8numberType;
   most_fi_tcl_String sfirstName;
   most_fi_tcl_String slastName;
   most_fi_tcl_String sphoneNumber;
   virtual ~most_fi_tcl_OnsCalConf_MembersValueItem();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_OnsCalConf_MembersValueItem& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_OnsCalConf_MembersValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_OnsCalConf_MembersValue() {}
#ifdef VARIANT_S_FTR_ENABLE_DEEP_COPY
   most_fi_tcl_OnsCalConf_MembersValue& operator=(const most_fi_tcl_OnsCalConf_MembersValue& coRef);
   most_fi_tcl_OnsCalConf_MembersValue(const most_fi_tcl_OnsCalConf_MembersValue& coRef);
#endif // VARIANT_S_FTR_ENABLE_DEEP_COPY

   // member variables
private:
   std::vector<most_fi_tcl_OnsCalConf_MembersValueItem*, std::allocator<most_fi_tcl_OnsCalConf_MembersValueItem*> > oLocalReaders;

public:
   std::vector<most_fi_tcl_OnsCalConf_MembersValueItem, std::allocator<most_fi_tcl_OnsCalConf_MembersValueItem> >oItems;

   most_fi_tcl_OnsCalConf_MembersValueItem& operator[](tU32 u32Index);
   virtual ~most_fi_tcl_OnsCalConf_MembersValue();
   virtual tVoid vDestroy();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   tBool operator==(const most_fi_tcl_OnsCalConf_MembersValue& roRef) const;
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsCalEmergencyCall_CallStateValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsCalEmergencyCall_CallStateValue();
   enum tenType {
      FI_EN_E8IDLE = 0UL,
      FI_EN_E8ACTIVE = 1UL,
      FI_EN_E8ONHOLD = 2UL,
      FI_EN_E8CONFERENCE = 3UL,
      FI_EN_E8DIALING = 4UL,
      FI_EN_E8RINGING = 5UL,
      FI_EN_E8BUSY = 6UL,
      FI_EN_E8CONFERENCE_ONHOLD = 7UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsCalEmergencyCall_CallStateValue(most_fi_tcl_e8_OnsCalEmergencyCall_CallStateValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsCalEmergencyCall_CallStateValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsCalEmergencyCall_CallStateValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsCalIn_NumberTypeValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsCalIn_NumberTypeValue();
   enum tenType {
      FI_EN_E8HOME = 0UL,
      FI_EN_E8WORK = 1UL,
      FI_EN_E8CELL = 2UL,
      FI_EN_E8OTHER = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsCalIn_NumberTypeValue(most_fi_tcl_e8_OnsCalIn_NumberTypeValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsCalIn_NumberTypeValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsCalIn_NumberTypeValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsCalNetwork_SignalStrengthValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsCalNetwork_SignalStrengthValue();
   enum tenType {
      FI_EN_E8LEVEL_0 = 0UL,
      FI_EN_E8LEVEL_1 = 1UL,
      FI_EN_E8LEVEL_2 = 2UL,
      FI_EN_E8LEVEL_3 = 3UL,
      FI_EN_E8LEVEL_4 = 4UL,
      FI_EN_E8LEVEL_5 = 5UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsCalNetwork_SignalStrengthValue(most_fi_tcl_e8_OnsCalNetwork_SignalStrengthValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsCalNetwork_SignalStrengthValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsCalNetwork_SignalStrengthValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsCalNetwork_StatusValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsCalNetwork_StatusValue();
   enum tenType {
      FI_EN_E8REGISTRED = 0UL,
      FI_EN_E8NO_NET = 1UL,
      FI_EN_E8AUTO_SEARCHING = 2UL,
      FI_EN_E8EMERGENCY_CALL_ONLY = 3UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsCalNetwork_StatusValue(most_fi_tcl_e8_OnsCalNetwork_StatusValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsCalNetwork_StatusValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsCalNetwork_StatusValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsCalPhone_FailureCodeValue : public most_fi_tclVisitableTypeBase
{
public:

   most_fi_tcl_e8_OnsCalPhone_FailureCodeValue();
   enum tenType {
      FI_EN_E8PHONE_UNSPECIFIED_FAILURE = 0UL,
      FI_EN_E8SUBSCRIPTION_MISSING = 1UL,
      FI_EN_E8SEC1_MISSING = 2UL,
      FI_EN_E8SEC2_MISSING = 3UL,
      FI_EN_E8PERMANENTLY_LOCKED = 4UL,
      FI_EN_E8PHONE_NOT_FOUND = 5UL,
      FI_EN_E8NOT_CORRECT_IN_CRADLE = 6UL
   };
   tenType enType;

   most_fi_tcl_e8_OnsCalPhone_FailureCodeValue(most_fi_tcl_e8_OnsCalPhone_FailureCodeValue::tenType newValue)
      :enType(newValue)
   {}

   virtual ~most_fi_tcl_e8_OnsCalPhone_FailureCodeValue();

   virtual tU32 u32GetSize(tU16 u16MajorVersion FI_DEFAULT_VERSION) const;
   virtual fi_tclOutContext& oWrite(fi_tclOutContext& rfoOut) const;
   virtual fi_tclInContext& oRead(fi_tclInContext& rfoIn);
   inline tBool operator==(const most_fi_tcl_e8_OnsCalPhone_FailureCodeValue& roRef) const
      {  return (enType == roRef.enType);  }
   virtual tS32 s32GetTypeId() const;
   virtual tVoid vTakeVisitor(most_fi_tclVisitorBase& rfoVisitor);
};

class most_fi_tcl_e8_OnsCalRingToneValue : public most_fi_tclVisitabl