/**
 * @copyright    (C) 2012 - 2016 Robert Bosch GmbH.
 *               The reproduction, distribution and utilization of this file as well as the
 *               communication of its contents to others without express authorization is prohibited.
 *               Offenders will be held liable for the payment of damages.
 *               All rights reserved in the event of the grant of a patent, utility model or design.
 * @brief        TTFIS command channel for datapool
 * @addtogroup   Datapool trace
 * @{
 */
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC		   
#include "osal_if.h"

#define DP_S_IMPORT_INTERFACE_DPTRACE
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_if.h"

#include "../dp/dpstore/dpPersPDDAcc.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
    #define ETG_DEFAULT_TRACE_CLASS DP_FI_TRACE_CLASS
    #include "trcGenProj/Header/dp_tclTraceIf.cpp.trc.h"
#endif

/*******************************************************************************
* FUNCTION   : dp_tclTraceIf::vTracePlugIn
* DESCRIPTION: Activate the TTFIS command channel for datapool
* PARAMETER  : None
* RETURNVALUE: None
*******************************************************************************/
tS32 dp_tclTraceIf::vTracePlugIn(tVoid)
{
   OSAL_trIOCtrlLaunchChannel  VtsMyTraceChannel;
   tS32                        Vs32Err=OSAL_ERROR;

   OSAL_tIODescriptor VtsHandle = OSAL_IOOpen(OSAL_C_STRING_DEVICE_TRACE, OSAL_EN_READWRITE);
   if (VtsHandle != OSAL_ERROR) 
   { 
      VtsMyTraceChannel.enTraceChannel = TR_TTFIS_DATAPOOL;
      VtsMyTraceChannel.pCallback      = (OSAL_tpfCallback)dp_tclTraceIf::vTraceCallbackFunction;
      Vs32Err = OSAL_s32IOControl (VtsHandle, OSAL_C_S32_IOCTRL_CALLBACK_REG, (intptr_t) &VtsMyTraceChannel);
      if (Vs32Err != OSAL_OK) 
	    {//bad
        ETG_TRACE_USR4(("MAIN :: Error TracePlugin"));
      }
      // close trace device and clear channel
      OSAL_s32IOClose(VtsHandle);	
   }
   //fprintf(stderr, "DP: vTracePlugIn return code %d\n",Vs32Err);
   return(Vs32Err);
}
/*******************************************************************************
* FUNCTION   : dp_tclTraceIf::vTraceUnplug
* DESCRIPTION: Deactivate the TTFIS command channel for datapool
* PARAMETER  : None
* RETURNVALUE: None
*******************************************************************************/
tVoid dp_tclTraceIf::vTraceUnplug(tVoid)
{
   OSAL_trIOCtrlLaunchChannel  VtsMyTraceChannel;
   tS32                        Vs32Err;

   OSAL_tIODescriptor VtsHandle = OSAL_IOOpen(OSAL_C_STRING_DEVICE_TRACE, OSAL_EN_READWRITE);
   if (VtsHandle != OSAL_ERROR) 
   { 
      VtsMyTraceChannel.enTraceChannel = TR_TTFIS_DATAPOOL;
      VtsMyTraceChannel.pCallback      = (OSAL_tpfCallback)dp_tclTraceIf::vTraceCallbackFunction;
      Vs32Err = OSAL_s32IOControl(VtsHandle, OSAL_C_S32_IOCTRL_CALLBACK_UNREG, (intptr_t) &VtsMyTraceChannel);
      if (Vs32Err == OSAL_OK) 
	    {
         //good
      } 
      else 
	    {   //bad
         ETG_TRACE_USR4(("MAIN :: Error TraceUnplug"));
      }
      // close trace device and clear channel
      OSAL_s32IOClose(VtsHandle);	
   }
}
/*******************************************************************************
* FUNCTION   : dp_tclTraceIf::vTraceCallbackFunction
* DESCRIPTION: Callback function for TTFIS command channel of datapool
* PARAMETER  : [IN] puchData = buffer with trace input data
* RETURNVALUE: None
*******************************************************************************/
tVoid dp_tclTraceIf::vTraceCallbackFunction(const tUChar* puchData) 
{
    tU32 u32MsgCode = (puchData[1]<<8) | puchData[2];
    switch (u32MsgCode) 
	  {
      case DP_CMD_DATAPOOLTRACE:
      {// Complete datapool will be traced..
        // DatapoolTrace: user specific code start
        dp_tclSrvIf dp; 
        dp.vTracePoolElement();
        // DatapoolTrace: user specific code end
      }break;
      case DP_CMD_DATAPOOLTRACEPOOL:
      {// Specific datapool will be traced. 'DatapoolTracePool CCA_APP_ID SubpoolId'.
        tU32 param1 =(puchData[3]<<8) | puchData[4];
        tU32 param2 =(puchData[5]<<8) | puchData[6];
        // DatapoolTracePool: user specific code start
        dp_tclSrvIf dp;
        tU32 u32PoolId = (param1<<16)|(param2);
        dp.vTracePoolElement(u32PoolId);
        // DatapoolTracePool: user specific code end
      }break;
      case DP_CMD_DATAPOOLTRACEELEMENT:
      { // Datapool element with given name will be traced.
        const tChar* pString1 = (const tChar*)&puchData[3];
        // DatapoolTraceElement: user specific code start
        dp_tclSrvIf dp;
        dp.vTracePoolElementByName(pString1);
        // DatapoolTraceElement: user specific code end
      }break;
      case DP_CMD_DATAPOOLTRACEHISTORY:
      { // Datapool history will be traced..
        // DatapoolTraceHistory: user specific code start
        dp_tclSrvIf dp;
        dp.vTracePoolHistory();
        // DatapoolTraceHistory: user specific code end
      }break;
      case DP_CMD_DATAPOOLSTORE:
      { // Force storing of the complete datapool.
        // DatapoolStore: user specific code start
        dp_tclSrvIf dp;
        ETG_TRACE_FATAL(("Datapool: Start 'StorePool' (time: %dms).", OSAL_ClockGetElapsedTime()));
        dp.s32StoreNow();
        ETG_TRACE_FATAL(("Datapool: pool stored (%dms).", OSAL_ClockGetElapsedTime()));
        // DatapoolStore: user specific code end
      }break;
      case DP_CMD_DATAPOOLSETDEFAULT:
      { // Datapool: set to default.
        tU32 param1 = (puchData[3]<<24) | (puchData[4]<<16) | (puchData[5] << 8) | puchData[6];
        tU32 param2 = (puchData[7]<<24) | (puchData[8]<<16) | (puchData[9] << 8) | puchData[10];
        // DatapoolSetDefault: user specific code start
        dp_tclSrvIf dp;
        ETG_TRACE_FATAL(("Datapool: Start SetDefault with %d (time: %dms).", param1,  OSAL_ClockGetElapsedTime()));
        dp.s32SetDefault(param1, param2);
        ETG_TRACE_FATAL(("Datapool: SetDefault finished (%dms).", OSAL_ClockGetElapsedTime()));
        // DatapoolSetDefault: user specific code end
      }break;
      case DP_CMD_DATAPOOLLOCK:
      { // Datapool: set to lock state.
        tU32 param1 = (puchData[3]<<24) | (puchData[4]<<16) | (puchData[5] << 8) | puchData[6];
        tU32 param2 = (puchData[7]<<24) | (puchData[8]<<16) | (puchData[9] << 8) | puchData[10];
        // DatapoolLock: user specific code start
        dp_tclSrvIf dp;
        ETG_TRACE_FATAL(("Datapool: Start SetDefault with %d (time: %dms).", param1,  OSAL_ClockGetElapsedTime()));
        (tVoid)dp.s32Lock(param1, param2);
        ETG_TRACE_FATAL(("Datapool: SetDefault finished (%dms).", OSAL_ClockGetElapsedTime()));
        // DatapoolLock: user specific code end
      }break;
      case DP_CMD_DATAPOOLUNLOCK:
      {  // Datapool: set to lock state.
         tU32 param1 = (puchData[3]<<24) | (puchData[4]<<16) | (puchData[5] << 8) | puchData[6];
         // DatapoolUnlock: user specific code start
         dp_tclSrvIf dp;
         ETG_TRACE_FATAL(("Datapool: Start SetDefault with %d (time: %dms).", param1,  OSAL_ClockGetElapsedTime()));
         (tVoid)dp.s32Unlock(param1);
         ETG_TRACE_FATAL(("Datapool: SetDefault finished (%dms).", OSAL_ClockGetElapsedTime()));
         // DatapoolUnlock: user specific code end
      }break;
      case DP_CMD_DATAPOOLCALCHASH:
      { // Datapool: trigger datapool to calculate a hash over all pool files (SHA1).
        // DatapoolCalcHash: user specific code start
        dp_tclSrvIf dp;
        tU32 u32Data[5];
        ETG_TRACE_FATAL(("Datapool: Start CalcHash (time: %dms).", OSAL_ClockGetElapsedTime()));
        if (0 < dp.s32CalcHash(u32Data, 5)) {
           ETG_TRACE_FATAL(("Datapool:: DP HASH: %08x%08x%08x%08x%08x.",
              u32Data[0],
              u32Data[1],
              u32Data[2],
              u32Data[3],
              u32Data[4]));
        }
        ETG_TRACE_FATAL(("Datapool: CalcHash finished (%dms).", OSAL_ClockGetElapsedTime()));
        // DatapoolCalcHash: user specific code end
      }break;
      case DP_CMD_DATAPOOLSETDEFAULTRAW:
      { // Datapool: set to default.
        tU32 param1 =(puchData[3]<<8) | puchData[4];
        tU32 param2 = (puchData[5]<<24) | (puchData[6]<<16) | (puchData[7] << 8) | puchData[8];
        tU32 param3 = (puchData[9]<<24) | (puchData[10]<<16) | (puchData[11] << 8) | puchData[12];
        // DatapoolSetDefaultRaw: user specific code start
        tU32 u32Starttime = OSAL_ClockGetElapsedTime();
        tS32 s32Ret = DP_s32SetDefault(param1, param2, param3);
        ETG_TRACE_FATAL(("Datapool: DefSet returned with '%u' after %dms", ETG_ENUM(EDpErrorCodes, s32Ret), OSAL_ClockGetElapsedTime()-u32Starttime));
        // DatapoolSetDefaultRaw: user specific code end
      }break;
      case DP_CMD_DATAPOOLLOCKRAW:
      {  // Datapool: set to lock state.
        tU32 param1 =(puchData[3]<<8) | puchData[4];
        tU32 param2 = (puchData[5]<<24) | (puchData[6]<<16) | (puchData[7] << 8) | puchData[8];
        tU32 param3 = (puchData[9]<<24) | (puchData[10]<<16) | (puchData[11] << 8) | puchData[12];

        // DatapoolLockRaw: user specific code start
        tU32 u32Starttime = OSAL_ClockGetElapsedTime();
        tS32 s32Ret = DP_s32Lock(param1, param2, param3);
        ETG_TRACE_FATAL(("Datapool: Lock returned with '%u' after %dms", ETG_ENUM(EDpErrorCodes, s32Ret), OSAL_ClockGetElapsedTime()-u32Starttime));
        // DatapoolLockRaw: user specific code end
      }break;
      case DP_CMD_DATAPOOLUNLOCKRAW:
      {// Datapool: set to lock state.
        tU32 param1 =(puchData[3]<<8) | puchData[4];
        tU32 param2 = (puchData[5]<<24) | (puchData[6]<<16) | (puchData[7] << 8) | puchData[8];

        // DatapoolUnlockRaw: user specific code start
        tU32 u32Starttime = OSAL_ClockGetElapsedTime();
        tS32 s32Ret = DP_s32Unlock(param1, param2);
        ETG_TRACE_FATAL(("Datapool: Unlock returned with '%u' after %dms", ETG_ENUM(EDpErrorCodes, s32Ret), OSAL_ClockGetElapsedTime()-u32Starttime));
        // DatapoolUnlockRaw: user specific code end
      }break;
      case DP_CMD_DATAPOOLWRITEELEMENT:
      {
        // Datapool: Test interface to write one element.
        tU32 param1 =(puchData[3]<<8) | puchData[4];
        tU32 param2 =(puchData[5]<<8) | puchData[6];
        tU16 param3 =(tU16)((((tU16)puchData[7])<<8) | puchData[8]);
        const tChar* pString4 = (const tChar*)&puchData[9];
        const tUChar* pStream5 = (const tUChar*)&puchData[39];

        // DatapoolWriteElement: user specific code start
        tU32 u32EntryLen =((tU32)(puchData[0]-(pStream5-&puchData[0]))) + 1;
        dp_tclBaseElement myDpElem(pString4);
        myDpElem.bFillData((tVoid*)pStream5, u32EntryLen);//lint !e1773 Attempt to cast away const (or volatile)
        tU32 s32Status = DP_s32SetElement((param1<<16|param2),&myDpElem, param3);
        ETG_TRACE_FATAL(("Datapool: write element status: '%u' for '%s'", ETG_ENUM(EDpErrorCodes, s32Status), pString4));
        // DatapoolWriteElement: user specific code end
      }break;
      case DP_CMD_DATAPOOLREADELEMENT:
      {
        // Datapool: Test interface to read one element.
        tU32 param1 =(puchData[3]<<8) | puchData[4];
        tU32 param2 =(puchData[5]<<8) | puchData[6];
        tU16 param3 =(tU16)((((tU16)puchData[7])<<8) | puchData[8]);
        const tChar* pString4 = (const tChar*)&puchData[9];

        // DatapoolReadElement: user specific code start
        tChar* strData = (tChar*) pString4;//lint !e1773 

        dp_tclBaseElement myDpElem(pString4);
        tU32 s32Status = DP_s32GetElement((param1<<16|param2), &myDpElem, param3);
        ETG_TRACE_FATAL(("Datapool: read element status: '%u' for '%s'", ETG_ENUM(EDpErrorCodes, s32Status), strData));
        ETG_TRACE_FATAL(("Datapool: data: %*x", ETG_LIST_LEN(myDpElem.u32GetDataLength()), ETG_LIST_PTR_T8(myDpElem.pvGetData())));
        // DatapoolReadElement: user specific code end
      }break;
      case DP_CMD_DATAPOOLENABLEELEMENTFORTRACE:
      {
        // Datapool element will be traced if Datapool trace class is enabled.
        const tChar* pString1 = (const tChar*)&puchData[3];

        // DatapoolEnableElementForTrace: user specific code start
        DP_vEnableElementForTrace(pString1);
        ETG_TRACE_FATAL(("Datapool: Element: '%s' is enabled for Trace now.", pString1));
        // DatapoolEnableElementForTrace: user specific code end
      }break;
      case DP_CMD_DATAPOOLENABLEALLELEMENTSFORTRACE:
      {
        // All datapool elements will be traced if Datapool trace class is enabled.
        // DatapoolEnableAllElementsForTrace: user specific code start
        DP_vEnableElementForTrace(NULL);
        // DatapoolEnableAllElementsForTrace: user specific code end
      }break;
	    case DP_CMD_PDD_NOR_USER_ERASE:
	    case DP_CMD_PDD_NOR_USER_PRINT_ACTUAL_DATA:
	    case DP_CMD_PDD_NOR_USER_GET_ERASE_COUNTER:
	    case DP_CMD_PDD_NOR_KERNEL_ERASE:
	    case DP_CMD_PDD_NOR_KERNEL_ERASE_SECTOR:
	    case DP_CMD_PDD_NOR_KERNEL_PRINT_ACTUAL_DATA:
	    case DP_CMD_PDD_NOR_KERNEL_GET_ERASE_COUNTER:
	    case DP_CMD_PDD_SCC_SET_ALL_POOLS_TO_INVALID:
	    case DP_CMD_PDD_NOR_USER_SAVE_DUMP_TO_FILE: 
	    case DP_CMD_PDD_NOR_USER_SAVE_ACTUAL_CLUSTER:
	    //case DP_CMD_PDD_NOR_USER_SAVE_DUMP_TO_NOR: command not activate! Risk!!
	    case DP_CMD_PDD_NOR_USER_GET_ACTUAL_CLUSTER:
	    case DP_CMD_PDD_SET_TRACE_LEVEL:
	    case DP_CMD_PDD_CLEAR_TRACE_LEVEL:
        case DP_CMD_PDD_CLEAR_POOL_FROM_LOCATION:
        { /*increment offset 0x40*/
            char VcBuffer[100];
            memset(VcBuffer,0,sizeof(VcBuffer));
            memmove(&VcBuffer[1],puchData,sizeof(VcBuffer)-1);
            VcBuffer[3]= (char)(VcBuffer[3]-(char)0x40);
        /*call PDD */
        dp_tclDpPersPddAccess VObjDpPdd;
        VObjDpPdd.vPddTraceCommand(&VcBuffer[0]);
            if(u32MsgCode!=DP_CMD_PDD_NOR_USER_GET_ACTUAL_CLUSTER)
            {
              ETG_TRACE_FATAL(("Datapool: pdd command done (putty console)"));
            }
            else
            {
              ETG_TRACE_FATAL(("Datapool: actual cluster is %d",VcBuffer[4]));
            }
          }
          break;
      default:
        //nothing to do
        break;
    }  
    return;
}
