
#include <adit-components/trace_interface.h>

#include <iostream>
#include <string.h>
#include <string>
#include <set>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <getopt.h>
#include <unistd.h>
#include <stdio.h>


/**
 * The component ID 0x97C0 was assigned for the stdout_to_ttfis "component".
 * The range for possible trace classes is 0x97C0 .. 0x97FF.
 * The trace class to be used can be passed to stdout_to_ttfis on the command line
 * with option -cxxxx where xxxx is the trace class in hex.
 * See .trc file for mapping between trace classes and HMI applications
 */
#define TR_CLASS_STDOUT_TO_TTFIS_DEFAULT           0x97C0

static unsigned short g_traceClass = TR_CLASS_STDOUT_TO_TTFIS_DEFAULT;
static bool g_printHelp = false;

static unsigned short validateTraceClass(const char* hexString)
{
   if (strlen(hexString) != 4)
   {
      printf("error: unexpected trace class string '%s'\n", hexString);
      return TR_CLASS_STDOUT_TO_TTFIS_DEFAULT;
   }

   unsigned long value;
   sscanf(hexString, "%x", &value);

   if ((value < 0x97C0) || (value > 0x97FF))
   {
      printf("error: invalid trace class value '%s' - must be in range 97C0..97FF\n", hexString);
      return TR_CLASS_STDOUT_TO_TTFIS_DEFAULT;
   }

   return (unsigned short) value;
}


static void parseOptions(int argc, char* argv[])
{
   int opt;

   opterr = 0;

   while ((opt = getopt(argc, argv, "hc:")) != -1)
   {
      switch (opt)
      {
         case 'c':
            g_traceClass = validateTraceClass(optarg);
            break;
         case 'h':
            g_printHelp = true;
            break;
         default:
            printf("error parsing options\n");
            return;
       }
   }
}


/**
 * Remove a specific kind of ESC sequences that are produced by ASF.
 * The sequences always start at the beginning of a line and end
 * with the letter 'm'. There can be multiple such sequences concatenated.
 */
static inline void stripEscSequence(std::string& line)
{
   while ((line.size() > 0) && (line[0] == 0x1b))
   {
      size_t pos = line.find('m');
      if (pos != std::string::npos)
      {
         line.erase(0, pos + 1);
      }
      else
      {
         break;
      }
   }
}


/**
 * Determine the appropriate trace level from line content.
 * Ensure that fatal traces always get through.
 * No further distinction done since there are trace lines without identifier and we cannot determine
 * whether they are on Info or Debug level.
 */
static inline TR_tenTraceLevel getLevel(const std::string& line)
{
   if (line.size() > 20)
   {
      std::string asfLevel = line.substr(13, 7);
      if (asfLevel == "|Fatal|")
      {
         return TR_LEVEL_FATAL;
      }
   }
   return TR_LEVEL_COMPONENT;
}


/**
 * Continuously read stdin and forward the output line by line to TTFIS.
 */
static void streamStdinToTtfis()
{
   while (!std::cin.eof())
   {
      std::string line;
      std::getline(std::cin, line);
      stripEscSequence(line);
      if (line.size())
      {
         TR_tenTraceLevel level = getLevel(line);
         BOOL selected = TR_core_bIsClassSelected(g_traceClass, level);
         if (selected != TR_ENABLED_CLASS_NONE)
         {
            TR_core_uwTraceOut(line.size(), g_traceClass, level, (U8*)line.c_str()); //lint !e1773
         }
         else
         {
            static bool printed = false;
            if (!printed)
            {
               printf("--------------\n", g_traceClass);
               printf("class = 0x%04x\n", g_traceClass);
               printf("level = %d\n", level);
               printf("selected = %d\n", selected);
               printf("--------------\n", g_traceClass);
               printed = true;
            }
               /* printf("%s\n", line.c_str()); */
         }
      }
   }
}


int main(int argc, char* argv[])
{

   parseOptions(argc, argv);
   if (!g_printHelp)
   {
      std::string text = "Entering to stdout to ttfis";
      TR_core_uwTraceOut(text.size(), g_traceClass, TR_LEVEL_FATAL, (U8*)text.c_str()); //lint !e1773
      streamStdinToTtfis();
   }
   else
   {
       std::string text = "stdout_to_ttfis: This program prints the stdout to a ttfis channel. \n\
You should provide -c <hexTraceClass> (range 97C0..97FF) to set class id explicite. \n\
Otherwise default class 97C0 is being used. \n\
This text is copied to console and ttfis for demo.";
       TR_core_uwTraceOut(text.size(), g_traceClass, TR_LEVEL_FATAL, (U8*)text.c_str()); //lint !e1773
       printf(text.c_str());
   }
   std::string line = "***** stdin was closed *****";
   TR_core_uwTraceOut(line.size(), g_traceClass, TR_LEVEL_FATAL, (U8*)line.c_str()); //lint !e1773

   _exit(0);
}
