/*******************************************************************************
*Copyright (C) 2016 Robert Bosch Car Multimedia GmbH
* This software is property of Robert Bosch.
* Unauthorized duplication and disclosure to third parties is prohibited.
*******************************************************************************/
/*******************************************************************************
* FILE         : tddrd_read_store.c.c
*             
* DESCRIPTION  :This file contains the operatios of reading from the frontend 
*				sub device and storing into the backend file
*                            
* AUTHOR(s)    :  Ravindra Prabha (Ravindra.Prabha@in.bosch.com)
*
* HISTORY      :
*------------------------------------------------------------------------
* Date      |       Version       | Author
*-----------|---------------------|--------------------------------------
*-----------|---------------------|--------------------------------------
*02.May.2017| Initial version 1.0|Ravindra Prabha(Ravindra.Prabha@in.bosch.com)
* -----------------------------------------------------------------------
* -----------------------------------------------------------------------
*******************************************************************************/

#include <tddrd.h>
#define MINSIZE_PER_COMPLETE_TDDR_DATA (( 5 * 1024*1024)/2)

int is_space_available_in_file(const char *filename,
											long int max_file_size)
{
	struct stat filestat;
	int avail = 0;	
	tddr_dev_info_t *ptddr_dev_info ;
	
	ptddr_dev_info = tddrd_get_devinfo();	

	if(filename && ptddr_dev_info)
	{
		memset(&filestat,0,sizeof(struct stat));
	
		if(stat(filename, &filestat) == 0)
		{
					/* Ensure whether enough space is there in the file to 
					 * write complete tddr data from frontend sub device. 
					 * Which means that after completely storing the available 
					 * tddr data, still the file should not exceed the maximum 
					 * permitted size
					 */
					if((filestat.st_size + MINSIZE_PER_COMPLETE_TDDR_DATA)
							< max_file_size)
					{
						avail = 1;			
					}
		}
	}
	return avail;
}

/******************************************************************************
* FUNCTION       : tddrd_worker_thread
* PARAMETER      : void                                       
* RETURNVALUE    : 0 : SUCCESS
*		  -1 : FAILURE
* DESCRIPTION    : This is worker thread entry function 
*		 	1). Allocate the object for the frontend
*		 	2). Allocate the object for the backend
*		 	3). Connect the frontend to the daemon
*		 	4). Connect the backend to the daemon
*		 	5.  a).Allocate the object for the socket
*			    b).Connect the socket to the daemon
*------------------------------------------------------------------------
* Date      |       Version       | Author & comments
*-----------|---------------------|--------------------------------------
*14.May.2017| Initialversion 1.0  | Ravindra Prabha (RBEI/ECF5)
* -----------------------------------------------------------------------
******************************************************************************/
void* tddrd_worker_thread(void* data)
{
	int bytes_written;
	int bytes_read;
	unsigned int num_messages = 0;
	tddr_dev_info_t *ptddr_dev_info ;
	tddrrb_data packet;
	int fd_out;
	/* Let s consist the memory little more than 128 bytes as it also holds 
	 * backend device name which is 128 bytes
	 */
	char s[140];
	int ii;
	tddrd_frontend_t *frontend;

	/*The argument to the thread consists the sub device id */
	if(data)
	{
		ii = * (int *)data;
	
		ptddr_dev_info = tddrd_get_devinfo();	
		if(ptddr_dev_info)
		{			
			/* create backend out file if it doesn't exist.Create the file with 
			 * Read/Write privileges
			 */
			fd_out = open(ptddr_dev_info->backend[ii].dev,
							O_WRONLY | O_APPEND | O_CREAT,
							S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH);
			if (-1 == fd_out) 
			{
				snprintf(s,sizeof(s),"open backend failed, devid=%d",ii);
				tddrd_log_error(__LINE__,2,s);
			}
			else 
			{
				/* If the file does not have enough space then dont go to 
				 * store the data
				 */
				if(!is_space_available_in_file(ptddr_dev_info->backend[ii].dev,
											ptddr_dev_info->max_file_length))
				{
					snprintf(s,sizeof(s),"no-space in out file,devid=%d",ii);
					tddrd_log_error(__LINE__,2,s);

					/* If the output file doen ot have enough space ,just overwrite 
					 * oldest entries stored in the file
					 */
					if(-1 == lseek(fd_out,0,SEEK_SET))
					{			
						/* can not be done anything else when SEEK_SET fails */
						snprintf(s,sizeof(s),
								"lseek error on outfile, devid=%d",ii);
						tddrd_log_error(__LINE__,2,s);
					}				
				}
				else
				{
					frontend = tddrd_get_frontend_node(ii);
				
					if(NULL != frontend) 
					{
						strcpy(s, "Start writing data from device: ");
						strcat(s, frontend->dev);
						strcat(s, "\n");

						bytes_written = write(fd_out, s, strlen(s));
						if (-1 == bytes_written) 
						{					
							snprintf(s,sizeof(s),
								"write error on out file, devid=%d",ii);
							tddrd_log_error(__LINE__,2,s);
						}

						do {
						int max_length = 256;
						char wbuf[max_length];
						int loc = 0;

							/* Read all the packets in the frontend sub device .
							 * Reads each packet at a time
							 */
							bytes_read = read(frontend->fd, &packet,
												sizeof(struct data));
							if (bytes_read > 0) 
							{
								num_messages++;
								loc = snprintf(wbuf, (unsigned int)max_length, 
									" Timestamp: %llu - event: %3u - context: %3u - id: %5u - pc: 0x%08x - lr: 0x%08x ",
									 	packet.d.time, packet.d.ev, packet.d.ctx
									 	,packet.d.id,packet.d.pc,packet.d.lr);
								if (loc < 0)
								{
									continue;
								}

								if (loc >= max_length)
								{
									loc = max_length -1;
								}
												
								wbuf[loc] = '\n';
								bytes_written = write(fd_out, wbuf, (unsigned int)(loc + 1));
								if (-1 == bytes_written) 
								{
									snprintf(s,sizeof(s),
									"write error on out file, devid=%d",ii);
									tddrd_log_error(__LINE__,2,s);
								}
								/* perform fsync operation once for every 1000 packets */
								if((0 == (num_messages % 1000)))
								{
									fsync(fd_out);
								}
								
							} 
							else if (bytes_read < 0)
							{	
								snprintf(s,sizeof(s),
										"read error on front devid=%d",ii);
								tddrd_log_error(__LINE__,2,s);
							} 
							else 
							{
								strcpy(s, "Finished writing data to device: ");
								strcat(s, ptddr_dev_info->backend[ii].dev);
								strcat(s, "\n");
								bytes_written = write(fd_out, s, strlen(s));
								if (-1 == bytes_written) 
								{
									snprintf(s,sizeof(s),
										"write error on out file, devid=%d",ii);
									tddrd_log_error(__LINE__,2,s);
								}
							}
								/* read as long as data is available with frontend 
								 * sub device and daemon is running
								 */
						} while ((bytes_read > 0)&&(is_tddrd_running())); 

						snprintf(s, 64, "Number of logs written: %u - Last bytes_read = % d\n",
								num_messages, bytes_read);
						write(fd_out, s, strlen(s));
						fsync(fd_out);
					}
					else 
					{
						snprintf(s,sizeof(s),
								"could not get frontend node, subdevice=%d",ii);
						tddrd_log_error(__LINE__,2,s);
					}
				}
				close(fd_out);
			}
		}
		else
		{
			snprintf(s,sizeof(s),"couldn't get ptddr_dev_info,subdevice=%d",ii);
			tddrd_log_error(__LINE__,2,s);
		}
	}
	else
	{
		snprintf(s,sizeof(s),"null argument to the thread");
		tddrd_log_error(__LINE__,2,s);
	}

	return (void*)0;
}


