/*****************************************************************************
| FILE:         resethandler.cpp
| PROJECT:      Platform
| SW-COMPONENT: OSAL
|-----------------------------------------------------------------------------
| DESCRIPTION:  This is the implementation file for the OSAL
|               (Operating System Abstraction Layer) 
|
|-----------------------------------------------------------------------------
| COPYRIGHT:    (c) 2017 Bosch GmbH
| HISTORY:      
| Date      | Modification               | Author
| 15.03.17  | Initial revision           | MRK2HI
| --.--.--  | ----------------           | -------, -----
|
|*****************************************************************************/
#ifndef ANDROID
/************************************************************************
| includes of component-internal interfaces
| (scope: component-local)
|-----------------------------------------------------------------------*/
#include "OsalConf.h"
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "Linux_osal.h"
#include "resethdr.h"

#if !defined _LINUXX86MAKE_ && !defined _LINUXX86_64_ && !defined ANDROID
#define SYSTEMD_SUPPORT
#endif

#ifdef SYSTEMD_SUPPORT
#include <systemd/sd-daemon.h>
#endif


#ifdef __cplusplus
extern "C" {
#endif

/************************************************************************
|defines and macros (scope: module-local)
|-----------------------------------------------------------------------*/
//#define TEST_RESETHANDLER
//#define SET_RESET_MARKER_AT_START

/************************************************************************
|typedefs (scope: module-local)
|-----------------------------------------------------------------------*/
typedef struct 
{
   char AppName[64];
}trNoResetApp;

/************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------*/
static trNoResetApp rNoResetApp[100];

/************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------*/

/************************************************************************
|function prototype (scope: module-local)
|-----------------------------------------------------------------------*/

/************************************************************************
|function prototype (scope: global)
|-----------------------------------------------------------------------*/

/************************************************************************
|function implementation (scope: module-local)
|-----------------------------------------------------------------------*/
static void sigaction_term_func(int n_signal, siginfo_t *siginfo, void *ptr)
{
	((void)siginfo);
	((void)ptr);
	vWritePrintfErrmem("Got SIGTERM %d from systemd -> force system reboot!!!",n_signal);
	eh_reboot();
}


static void child_handler(int signum)
{
    switch(signum) {
    case SIGALRM: exit(EXIT_FAILURE); break;
    case SIGUSR1: exit(EXIT_SUCCESS); break;
    case SIGCHLD: exit(EXIT_FAILURE); break;
    }
}

static void daemonize(void)
{
    pid_t pid, sid;

    /* already a daemon */
    if ( getppid() == 1 ) return;

    /* Trap signals that we expect to recieve */
    signal(SIGCHLD,child_handler);
    signal(SIGUSR1,child_handler);
    signal(SIGALRM,child_handler);
 
    /* Fork off the parent process */
    pid = fork();
    if (pid < 0) {
        exit(1);
    }
 
    /* Cancel certain signals */
    signal(SIGCHLD,SIG_DFL); /* A child process dies */
    signal(SIGTSTP,SIG_IGN); /* Various TTY signals */
    signal(SIGTTOU,SIG_IGN);
    signal(SIGTTIN,SIG_IGN);
    signal(SIGHUP, SIG_IGN); /* Ignore hangup signal */
    signal(SIGTERM,SIG_DFL); /* Die on SIGTERM */

    /* If we got a good PID, then we can exit the parent process. */
    if (pid > 0) {
        exit(0);
    }

    /* At this point we are executing as the child process */

    /* Change the file mode mask */
    umask(0);

    /* Create a new SID for the child process */
    sid = setsid();
    if (sid < 0) {
        exit(EXIT_FAILURE);
    }

    /* Change the current working directory.  This prevents the current
       directory from being locked; hence not being able to remove it. */
    if ((chdir("/")) < 0) {
        exit(1);
    }

    /* Redirect standard files to /dev/null */
    freopen( "/dev/null", "r", stdin);
    freopen( "/dev/null", "w", stdout);
    freopen( "/dev/null", "w", stderr);
}


long lGetOwnPidNameSpace(void)
{
  char Path[64];
  struct stat rbuf;
  snprintf(Path,64,"/proc/%d/ns/pid",getpid());  
  if(stat(Path,&rbuf) == 0)
  {
	 return rbuf.st_ino;
  }
  else
  {
     return OSAL_ERROR;
  }
  
#ifdef OLD_IMPL  
  long Ret=OSAL_ERROR;
  OSAL_tIODescriptor s32DevDesc;
  char ReadBuffer[1024];
  
  snprintf(Path,64,"ls -la /proc/%d/ns > /run/namespace.txt",getpid());  
  /* get own namespace ID */
  if(system(Path) == -1)
  {
     TraceString("RSETHDR cannot find namespace pid info in proc file system");
  }
  
  s32DevDesc = OSAL_IOOpen("/dev/root/run/namespace.txt",OSAL_EN_READWRITE);
  if(s32DevDesc == OSAL_ERROR)
  {
     TraceString("RSETHDR OSAL_IOOpen dev/ramdisk failed with 0x%x",OSAL_u32ErrorCode());
  }
  else
  {
     Ret = OSAL_s32IORead(s32DevDesc,(tPS8)&ReadBuffer[0],1024);
     if(Ret == OSAL_ERROR)
     {
        TraceString("RSETHDR OSAL_s32IORead failed with 0x%x",OSAL_u32ErrorCode());
     }
     OSAL_s32IOClose(s32DevDesc);
  }
  OSAL_s32IORemove("/dev/root/run/namespace.txt");
  if(Ret != OSAL_ERROR)
  {
     char *tmp1,*tmp2;
     if((tmp1 = strstr(ReadBuffer,"pid -> pid:[")) != 0)
     {
       tmp1 += strlen("pid -> pid:[");
       if((tmp2 = strstr(tmp1,"]")) != 0)
       {
          *tmp2 = '\0';
          errno = 0;
          Ret = strtoul(tmp1,NULL,10);
          if((errno == ERANGE && (Ret == LONG_MAX || Ret == LONG_MIN))
           ||(errno != 0 && Ret == 0)) 
           {
              Ret = 0;
           }
       }
     }
  }  
  return Ret;
#endif
}

tBool bTraceLastResetReason(void)
{
   OSAL_tIODescriptor s32DevDesc = 0;
   char   Buffer[256];
   s32DevDesc = OSAL_IOOpen(OSAL_C_STRING_DEVICE_FFS"/ResetReason.txt",OSAL_EN_READONLY);
   if(s32DevDesc != OSAL_ERROR)
   {
      if(OSAL_s32IORead(s32DevDesc,(tPS8)Buffer,256) > 0)
      {
            TraceString("RSETHDR Restart after Exception");
            TraceString(Buffer);
      }
      else
      {
            TraceString("RSETHDR Cannot read %s",OSAL_C_STRING_DEVICE_FFS"/ResetReason.txt");
      }
      (void)OSAL_s32IOClose(s32DevDesc);
      (void)OSAL_s32IORemove(OSAL_C_STRING_DEVICE_FFS"/ResetReason.txt");
      return TRUE;
   }
   if(OSAL_ClockGetElapsedTime() > 60000)return TRUE;
   else return FALSE;
}

int main(int argc,char** argv)
{
   OSAL_tIODescriptor s32DevDesc = 0;
   tS32 s32Ret,i;
   int sched_policy;
   struct sched_param param;
   OSAL_trIOCtrlRegistry hReg ;
   char   PidNsBuf[64] = {0};
   char   Buffer[256] = {0};
   char   Name[64] = {0};
   unsigned int prio = 0;
   unsigned int MemDump = 0;
   mqd_t mqdes;
   struct status_msg msg;
   struct mq_attr attr;
   struct sigaction act_term;
   char* pRes = NULL;
   long pidns = 0;
   char reboot = FALSE;
   char* pcReadBuf = NULL;
   ((void)argc);
   ((void)argv);
   
   /* overwrite SIGTERM handling to shutdown the daemon/reset the system */
   act_term.sa_sigaction = sigaction_term_func;
   act_term.sa_flags = SA_ONSTACK | SA_SIGINFO | SA_RESTART;
   sigemptyset(&act_term.sa_mask);
   if(sigaction(SIGTERM,  &act_term, NULL) == -1)
   {
      TraceString("RSETHDR Overwriting SIGTERM handler failed errno:%d",errno);
   }

   TraceString("RSETHDR Make daemon from resethdr");
   /* make a daemon from this process */
   daemonize();   
   
   /* get namespace for PID */
   pidns = lGetOwnPidNameSpace();
   if(pidns == 0)
   {
      snprintf(PidNsBuf,64,"0x%x",(unsigned int)0);
      TraceString("RSETHDR Cannot find namespace pid");
   }
   else
   {
      snprintf(PidNsBuf,64,"%p",(unsigned int)pidns);
      TraceString("RSETHDR with namespace pid:%s %ul",PidNsBuf,pidns);
   }
   /* read reset exclude list from registry if available*/
   s32DevDesc = OSAL_IOOpen("/dev/registry/"REGISTRY_BASE_PATH"/VERSIONS/RESET_CONF",OSAL_EN_READWRITE);
   if(s32DevDesc != OSAL_ERROR)
   {
      hReg.s32Type   =  OSAL_C_S32_VALUE_STRING;
      hReg.u32Size   =  64;
      for(i=0;i<100;i++)
      {
        hReg.ps8Value  =  (tPU8)rNoResetApp[i].AppName;
        snprintf(Name,64,"NO_RST_APP%d",i+1);	  
        hReg.pcos8Name =  (tPCS8)Name;    
        if((s32Ret = OSAL_s32IOControl( s32DevDesc,OSAL_C_S32_IOCTRL_REGGETVALUE,(intptr_t)&hReg)) == OSAL_ERROR)
        {
           break;
        }
      }
      prio = 98;
      hReg.s32Type   =  OSAL_C_S32_VALUE_S32;
      hReg.u32Size   =  (tU32)sizeof(tS32);
      hReg.ps8Value  =  (tPU8)&prio;
      hReg.pcos8Name =  (tPCS8)"APP_PRIO";    
      (void)OSAL_s32IOControl( s32DevDesc,OSAL_C_S32_IOCTRL_REGGETVALUE,(intptr_t)&hReg);
      hReg.ps8Value  =  (tPU8)&MemDump;
      hReg.pcos8Name =  (tPCS8)"MEM_DUMP";    
      (void)OSAL_s32IOControl( s32DevDesc,OSAL_C_S32_IOCTRL_REGGETVALUE,(intptr_t)&hReg);
      OSAL_s32IOClose(s32DevDesc);
      if(MemDump)
      {
         pcReadBuf = (char*)malloc(4096);
      }
   }
  
   /* set reset handler on realtime one below ADIT exception handler */
   s32Ret = pthread_getschedparam( pthread_self(), &sched_policy, &param);
   param.sched_priority = prio;
   sched_policy = SCHED_FIFO;  // SCHED_RR, SCHED_OTHER
   s32Ret = pthread_setschedparam( pthread_self(), sched_policy, &param);
   if(s32Ret == 0)
   {
      TraceString("RSETHDR is running with priority %d",prio);
   }
   
   /* prepare message queue for incoming reset information */   
   memset((void*)&attr,0,sizeof(mq_attr));
   attr.mq_maxmsg = MQ_SIZE_MAX;
   attr.mq_msgsize = sizeof(struct status_msg);
   prio = 0;
   if((mqdes = mq_open(MQ_NAME,
                       O_RDONLY | O_CREAT,
                       S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH,
                       &attr)) < 0)
   {
        TraceString("RSETHDR Unable to create message queue (%s)\n", strerror(errno));
        FATAL_M_ASSERT_ALWAYS();
   }

#ifdef SYSTEMD_SUPPORT
   /*ready to work */
   sd_notify(0, "READY=1");
#endif
   //   TraceString("ResetHandler Process will wait for incoming messages");
#ifdef SET_RESET_MARKER_AT_START
      sched_policy = bTraceLastResetReason();
#else
      sched_policy = 1;
#endif
      pOsalData->bDisableExceptionRst = 1;
      while (1)
      {
         i = (int)mq_receive(mqdes, (char *)&msg, sizeof(msg), &prio);
         if(i > 0)
         {
            TraceString("Termination message received:\n");
            TraceString("    app: %s\n", msg.app_name);
            TraceString("    service result: %s\n", msg.service_result);
            TraceString("    exit status: %s\n", msg.exit_status);
            TraceString("    exit code: %s\n", msg.exit_code);
            TraceString("    pidnamespace: %s\n", msg.pidnamespace);
            
            if(!strncmp(msg.pidnamespace,"0xffffffff",strlen("0xffffffff")))
            {
                memcpy(&msg.pidnamespace[2],&msg.pidnamespace[10],8);
                memset(&msg.pidnamespace[10],0,8);
            }
#ifdef TEST_RESETHANDLER
            TraceString("%s | %s",msg.pidnamespace,PidNsBuf);
#endif
            /* check if application is from container context */
            if(strncmp(msg.pidnamespace,PidNsBuf,strlen(PidNsBuf)))
            {
               vWritePrintfErrmem("No Reboot for App:%s service result:%s exit status: %s exit code:%s pidns:%s terminated -> root pidns:%s \n", 
                                  msg.app_name, msg.service_result, msg.exit_status, msg.exit_code,msg.pidnamespace,PidNsBuf);
            }
            else if(!strncmp(msg.exit_status,"USR2",4))/* SIGUSR2 is used for callstack generation and should not cause a reset. so filter out */
            {
               /* ignore and do nothing */ 
               TraceString("RSETHDR SIGUSR2 for %s received",msg.app_name);
            }
            else
            {
               reboot = 1;
            }	 	 
            /* check is reset is configured out via registry for this process */
            if(reboot == 1)
            {
               for(i=0;i<100;i++)
               {
                  if(rNoResetApp[i].AppName[0] == 0)break;
                  pRes = strstr(msg.app_name,rNoResetApp[i].AppName);
                  if(pRes)
                  {
                     TraceString("RSETHDR %s configured for exclude from system reboot",msg.app_name);
                     reboot = 0;
                     break;
                  }
               }
            }
            if(reboot == 1)
            {
               s32Ret = snprintf(Buffer,256,"Reboot for app:%s service result:%s exit status: %s exit code:%s pidns:%s \n", 
                                 msg.app_name, msg.service_result, msg.exit_status, msg.exit_code,msg.pidnamespace);
#ifdef SET_RESET_MARKER_AT_START
               s32DevDesc = OSAL_IOCreate(OSAL_C_STRING_DEVICE_FFS"/ResetReason.txt",OSAL_EN_READWRITE);
               if(s32DevDesc != OSAL_ERROR)
               {
                  if(OSAL_s32IOWrite(s32DevDesc,(tCS8*)Buffer,s32Ret) < s32Ret)
                  {
                     TraceString("RSETHDR Write File error:%x Reboot follows",OSAL_u32ErrorCode());
                     TraceString("RSETHDR %s",Buffer);
                     (void)OSAL_s32IOClose(s32DevDesc);
                  }
                  else
                  {
                     s32DevDesc = OSAL_IOOpen(OSAL_C_STRING_DEVICE_FFS"/ResetReason.txt",OSAL_EN_READWRITE);
                     if((s32DevDesc == OSAL_ERROR)||(OSAL_s32IOControl(s32DevDesc,OSAL_C_S32_IOCTRL_FIOFLUSH, (intptr_t)pidns) ==OSAL_ERROR))
                     {
                        TraceString("RSETHDR Flush File error:%x Reboot follows",OSAL_u32ErrorCode());
                        TraceString("RSETHDR %s",Buffer);
                     }
                  }
                  (void)OSAL_s32IOClose(s32DevDesc);
               }
               else
#endif
               {
                  TraceString("RSETHDR Reboot follows");
                  TraceString("RSETHDR %s",Buffer);
               }
               vWritePrintfErrmem(Buffer);
               vWriteMemStatToErrmem(pcReadBuf,4096,TRUE);
               eh_reboot();
            }
         }
        // if(sched_policy == 0)sched_policy = bTraceLastResetReason();
      }//end while
      if (errno)
      {
         TraceString("RSETHDR An error occurred: %s.\n", strerror(errno));
      }
    
   /* cleaunp resources before exiting the process*/
   mq_close(mqdes);
   mq_unlink(MQ_NAME);
   OSAL_vProcessExit();
   return s32Ret;
}


#ifdef __cplusplus
}
#endif

#endif
/************************************************************************
|end of resethandler.cpp
|-----------------------------------------------------------------------*/
