// OSALSocketInterface.cpp: implementation of the OSALSocketInterface class.
//
//////////////////////////////////////////////////////////////////////
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

//#pragma warning( disable : 4786 )

#ifdef VASCO_OS_WINNT
   #include "string.h"
   #include "stdio.h"
   #include "memory.h"
   #include "assert.h"
   #include "socket.h"
   #define NOMINMAX 
   #include "windows.h"
#elif defined VASCO_OS_LINUX
   #include <netinet/in.h>
   #include <netdb.h>
   #include <sys/socket.h>
   #include <arpa/inet.h>

   #include <stdlib.h>
   #include <stdio.h>
   #include <string.h>
   #include <memory.h>
   #include <cstdio>

   #define INVALID_SOCKET -1
//   #define FALSE 0
//   #define TRUE 1
   #define MAX_PATH 256
#define sprintf_s snprintf

#include <time.h>
#include <sys/time.h>

static void Sleep (unsigned int msec)
{
  struct timespec r;

  unsigned int sec = (msec) / 1000;
  unsigned int nsec = (msec - sec*1000) * 1000 * 1000;

  r.tv_sec  = sec;
  r.tv_nsec = nsec;

  nanosleep (&r, NULL);
}

#endif


#include "OSALInterface.h"
#include "OSALSocketInterface.h"


#include "ByteArrayWriter.h"
#include "ByteArrayReader.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

OSALSocketInterface::OSALSocketInterface() : m_cu32Times(4), m_cu32Factor(2)
{
    s32SocketNumber = -1;
    clientSocket = 0;
    m_bTimeOutFlag = FALSE;
    szHostName = NULL;
    log = 0 ;

    m_poQueueNameMap = new std::map<OSAL_tMQueueHandle, std::string>();
    m_poSharedMemoryMap = new std::map<void *, intptr_t>();
#ifdef VASCO_OS_WINNT
	m_pPostCS = (void*)new CRITICAL_SECTION;
    m_pReceiveCS = (void*)new CRITICAL_SECTION;
    InitializeCriticalSection((CRITICAL_SECTION*)m_pPostCS);
    InitializeCriticalSection((CRITICAL_SECTION*)m_pReceiveCS);
#endif
#ifdef VASCO_OS_LINUX
    pthread_mutex_init(&mPostMutex,NULL);
    pthread_mutex_init(&mRecieveMutex,NULL);
#endif
}

OSALSocketInterface::OSALSocketInterface(int s32SocketNo,const char* coszHostName) : m_cu32Times(4), m_cu32Factor(2)
{
    s32SocketNumber = s32SocketNo;
    szHostName = new char[strlen(coszHostName)+1];
    if (szHostName != NULL)
    {
        strcpy(szHostName,coszHostName);
    }
    clientSocket = 0;
    m_bTimeOutFlag = FALSE;
    log = 0 ;

    m_poQueueNameMap = new std::map<OSAL_tMQueueHandle, std::string>();
    m_poSharedMemoryMap = new std::map<void *, intptr_t>();
#ifdef VASCO_OS_WINNT
    m_pPostCS = (void*)new CRITICAL_SECTION;
    m_pReceiveCS = (void*)new CRITICAL_SECTION;
    InitializeCriticalSection((CRITICAL_SECTION*)m_pPostCS);
    InitializeCriticalSection((CRITICAL_SECTION*)m_pReceiveCS);
#endif
#ifdef VASCO_OS_LINUX
    pthread_mutex_init(&mPostMutex,NULL);
    pthread_mutex_init(&mRecieveMutex,NULL);
#endif
}

OSALSocketInterface::~OSALSocketInterface()
{
    if (szHostName != NULL)
    {
        delete[] szHostName;
        szHostName = NULL;
    }
    log = 0 ;
    m_poQueueNameMap->clear();
    m_poSharedMemoryMap->clear();

    delete m_poQueueNameMap;
    m_poQueueNameMap = NULL;
    delete m_poSharedMemoryMap;
    m_poSharedMemoryMap = NULL;
}


int OSALSocketInterface::s32SetTimeoutEventHandler( Socket_tpfTimeoutCallback pCallback, void *pvArg, tU32 u32TmOutValue )
{
    int s32RetVal = 0;
    if( pCallback == NULL ) // NULL means unregister the callback
    {
        m_bTimeOutFlag = FALSE;
        clientSocket->vClearSocketTimeout();
    }
    else
    {
        if ( u32TmOutValue != 0 && clientSocket->s32TimerInit(pCallback,pvArg,u32TmOutValue))
        {
            m_bTimeOutFlag = TRUE;
        }
        else 
            s32RetVal = -1; // Callback Register won't happen if Timeout value is zero
    }
    return s32RetVal;
}

int OSALSocketInterface::s32MsgQueueCreate(const char* szQueueName,tU32 u32MaxMsgs, 
                                           tU32 u32MaxLength, tU32 u32Access, 
                                           OSAL_tMQueueHandle& uiHandle)
{
    int s32MsgQueueCreateCmd = 39;
    int s32ReturnValue;

    if (clientSocket != 0) {

    	tU32 u32OutputSize = (tU32)(16 + strlen(szQueueName) + 1);
    	ahl_tclStreamer oOutputStream(u32OutputSize);

    	oOutputStream << s32MsgQueueCreateCmd;
    	oOutputStream.vWriteString(szQueueName, (tU16)strlen(szQueueName));
    	oOutputStream << (char) 0;
    	oOutputStream
    		<< u32MaxMsgs
			<< u32MaxLength
			<< u32Access;

        clientSocket->RawSend(oOutputStream.pGetStream(),u32OutputSize);


        tU32 u32InputSize = 8 + sizeof(uiHandle);
        ahl_tclStreamer oInputStreamer(u32InputSize);

        int s32MsgLength = (int)clientSocket->TimedReceive(oInputStreamer.pGetStream(), u32InputSize );
        if (s32MsgLength == (tS32)u32InputSize) {
        	oInputStreamer >> u32LastErrorCode;
        	oInputStreamer >> s32ReturnValue;
        	oInputStreamer >> uiHandle;
        } else {
            s32ReturnValue = -1;
            u32LastErrorCode = -1;
        }
    } else {
        s32ReturnValue = -1;
        u32LastErrorCode = -1;
    }
    return s32ReturnValue;
}

int OSALSocketInterface::s32MsgQueueDelete(const char* szQueueName)
{
    int s32MsgQueueDeleteCmd = 42;
    int s32ReturnValue;

    if (clientSocket != 0) {
		tU32 u32OutputSize = (tU32)(4 + strlen(szQueueName));
		ahl_tclStreamer oOutputStream(u32OutputSize);

		oOutputStream << s32MsgQueueDeleteCmd;
		oOutputStream.vWriteString(szQueueName,(tU16)strlen(szQueueName));

		clientSocket->RawSend(oOutputStream.pGetStream(), (int)(u32OutputSize));


		tU32 u32InputSize = 8;
		ahl_tclStreamer oInputStream(u32InputSize);
        int s32MsgLength = clientSocket->TimedReceive(oInputStream.pGetStream(), u32InputSize);
        if (s32MsgLength == (tS32)u32InputSize) {
        	oInputStream >> u32LastErrorCode;
        	oInputStream >> s32ReturnValue;
        } else {
            s32ReturnValue = -1;
            u32LastErrorCode = -1;
        }
    } else {
        s32ReturnValue = -1;
        u32LastErrorCode = -1;
    }

    return s32ReturnValue;
}

int OSALSocketInterface::s32MsgQueueOpen(const char* szQueueName,
                                         tU32 u32Access,
                                         OSAL_tMQueueHandle& uiHandle)
{
    int s32MsgQueueOpenCmd = 40;
    int s32ReturnValue;
    if (clientSocket != 0) {

    	tU32 u32OutputSize = (tU32)(8 + strlen(szQueueName) + 1);
    	ahl_tclStreamer oOutputStream(u32OutputSize);
    	oOutputStream << s32MsgQueueOpenCmd;
    	oOutputStream.vWriteString(szQueueName, (tU16)strlen(szQueueName));
    	oOutputStream << (char) 0;
    	oOutputStream << u32Access;

        clientSocket->RawSend(oOutputStream.pGetStream(),(int)(u32OutputSize));


        tU32 u32InputSize = 8 + sizeof(uiHandle);
        ahl_tclStreamer oInputStream(u32InputSize);

        int s32MsgLength = clientSocket->TimedReceive(oInputStream.pGetStream(),u32InputSize);

        if (s32MsgLength == (tS32)u32InputSize) {
        	oInputStream
				>> u32LastErrorCode
        		>> s32ReturnValue;

            if (s32ReturnValue == 0)
            {
            	oInputStream >> uiHandle;
                (*m_poQueueNameMap)[uiHandle] = std::string(szQueueName);
            }
        } else {
            s32ReturnValue = -1;
            u32LastErrorCode = -1;
        }
    } else {
        s32ReturnValue = -1;
        u32LastErrorCode = -1;
    }
    return s32ReturnValue;
}

int OSALSocketInterface::s32MsgQueueClose(OSAL_tMQueueHandle uiHandle)
{
    int s32MsgQueueCloseCmd = 41;
    int s32ReturnValue;

    if (clientSocket != 0) {
        tU32 u32OutputSize = (tU32)(4 + sizeof(uiHandle));
        ahl_tclStreamer oOutputStream(u32OutputSize);
        oOutputStream
			<< s32MsgQueueCloseCmd
        	<< uiHandle;
        clientSocket->RawSend(oOutputStream.pGetStream(),u32OutputSize);


        tU32 u32InputSize = 8;
        ahl_tclStreamer oInputStream(u32InputSize);
        int s32MsgLength = clientSocket->TimedReceive(oInputStream.pGetStream(),u32InputSize);
        if (s32MsgLength == (tS32)u32InputSize) {
        	oInputStream
				>> u32LastErrorCode
				>> s32ReturnValue;
        } else {
            s32ReturnValue = -1;
            u32LastErrorCode = -1;
        }

    } else {
        s32ReturnValue = -1;
        u32LastErrorCode = -1;
    }
    return s32ReturnValue;
}

int OSALSocketInterface::s32MsgPoolOpen()
{
    int s32MsgPoolOpenCmd = 1001;
    int s32ReturnValue;
    if (clientSocket != 0) {
        clientSocket->RawSend((char*)&s32MsgPoolOpenCmd,4);
        char recvBuf[8];
        int s32MsgLength = clientSocket->TimedReceive(recvBuf,8);
        if (s32MsgLength == 8) {
            memcpy(&u32LastErrorCode,recvBuf,4);
            memcpy(&s32ReturnValue,recvBuf+4,4);
        } else {
            s32ReturnValue = -1;
            u32LastErrorCode = -1;
        }
    }
    else {
        s32ReturnValue = -1;
        u32LastErrorCode = -1;
    }
    return s32ReturnValue;
}

int OSALSocketInterface::s32MsgPoolClose()
{
    int s32MsgPoolCloseCmd = 1002;
    int s32ReturnValue;
    if (clientSocket != 0) {
        clientSocket->RawSend((char*)&s32MsgPoolCloseCmd,4);

    	tU32 u32InputSize = 8;
    	ahl_tclStreamer oInputStream(u32InputSize);
        int s32MsgLength = clientSocket->TimedReceive(oInputStream.pGetStream(),u32InputSize);
        if (s32MsgLength == (tS32)u32InputSize) {
        	oInputStream
    			>> u32LastErrorCode
    			>> s32ReturnValue;
        } else {
            s32ReturnValue = -1;
            u32LastErrorCode = -1;
        }
    }
    else {
        s32ReturnValue = -1;
        u32LastErrorCode = -1;
    }
    return s32ReturnValue;
}

int OSALSocketInterface::s32MsgQueuePost(OSAL_tMQueueHandle uiHandle,
                                         tU32 u32MsgSize, 
                                         unsigned char *pu8Msg, 
                                         tU32 u32Prio)
{
   int s32MsgQueuePostCmd = 1010;
   int s32ReturnValue;


   if (clientSocket != 0) {
      tU32 u32OutputSize = (tU32)(12 + sizeof(uiHandle) + u32MsgSize);
	  ahl_tclStreamer oOutputStream(u32OutputSize);

	  oOutputStream
	  	  << s32MsgQueuePostCmd
		  << uiHandle
		  << u32MsgSize;
	  oOutputStream.vWriteString((const char*)pu8Msg,  (tU16)u32MsgSize);
	  oOutputStream << u32Prio;

      // print for debugging
      if (log)
      {
         fprintf(log,"<MSG_SEND channel=\"%s\"> <!-- outgoing message length = %d -->\n",(*m_poQueueNameMap)[uiHandle].c_str(),u32MsgSize);
         for (unsigned int i = 0; i < u32MsgSize; i++) {
            fprintf(log,"%02X%c",pu8Msg[i] & 0xFF,(((i%16)==15)?'\n':' '));
         }
         fprintf(log,"\n</MSG_SEND>\n");
         fflush(log);
      }
#ifdef VASCO_OS_WINNT
      EnterCriticalSection((CRITICAL_SECTION*)m_pPostCS);
#endif
#ifdef VASCO_OS_LINUX
      pthread_mutex_lock(&mPostMutex);
#endif
      clientSocket->RawSend(oOutputStream.pGetStream(),u32OutputSize);
      Sleep(10);

      tU32 u32InputStream = 8;
      ahl_tclStreamer oInputStream(u32InputStream);
      int s32MsgLength = clientSocket->TimedReceive(oInputStream.pGetStream(),u32InputStream);
      if (s32MsgLength == (tS32)u32InputStream) {
    	  oInputStream
			  >> u32LastErrorCode
			  >> s32ReturnValue;
      } else {
         s32ReturnValue = -1;
         u32LastErrorCode = -1;
      }
#ifdef VASCO_OS_WINNT
      LeaveCriticalSection((CRITICAL_SECTION*)m_pPostCS);
#endif
#ifdef VASCO_OS_LINUX
      pthread_mutex_unlock(&mPostMutex);
#endif
   } else {
      s32ReturnValue = -1;
      u32LastErrorCode = -1;
   }
   return s32ReturnValue;
}

int OSALSocketInterface::s32MsgQueueWait(OSAL_tMQueueHandle uiHandle, tU32 u32BufSize, 
                                         tU32 u32TimeOut, unsigned char *pu8MsgBuf,
                                         tU32 &u32Prio)
{
    int s32ReturnValue = -1;

    // Finite timeout for MQ wait
    // Socket timeout handler registered
    if ( m_bTimeOutFlag )
    {
        tU32 u32SplitTimeout = u32TimeOut / m_cu32Times;         // milliseconds
        tU32 u32RecvTimeout  = u32SplitTimeout * m_cu32Factor;   // milliseconds
        tU32 u32TotalTimeout = u32SplitTimeout;                  // milliseconds

        bool bSendAgain = TRUE;

        // Send sequence of requests to server with split timeout
        while ( (u32TotalTimeout <= u32TimeOut ) && bSendAgain )
        {
            s32ReturnValue = s32MsgQueueTimedWait( uiHandle, u32BufSize, u32SplitTimeout, pu8MsgBuf, u32Prio, u32RecvTimeout );
            if (( s32ReturnValue == 0 )&&(u32LastErrorCode == OSALRPC_E_OSALAPI_TIMEOUT)) 
            {
                u32TotalTimeout += u32SplitTimeout;
            }
            else // s32MsgQueueTimedWait returned a message
            {
                bSendAgain = FALSE;
            }
        }
    }
    else  // Socket timeout handler not registered; Should freeze!
    {
        s32ReturnValue = s32MsgQueueTimedWait( uiHandle, u32BufSize, u32TimeOut, pu8MsgBuf, u32Prio );
    }

    return s32ReturnValue;
}

int OSALSocketInterface::s32MsgQueueTimedWait(OSAL_tMQueueHandle uiHandle, tU32 u32BufSize, 
                                              tU32 u32TimeOut, unsigned char *pu8MsgBuf,
                                              tU32 &u32Prio, tU32 u32RecvTimeout)
{
   int s32MsgQueueWaitCmd = 1011;
   int s32ReturnValue;



   if (clientSocket != 0) {
	  tU32 u32OutputSize = 12 + sizeof(uiHandle);
	  ahl_tclStreamer oOutputStream(u32OutputSize);
	  oOutputStream
	  	  << s32MsgQueueWaitCmd
		  << uiHandle
		  << u32BufSize
		  << u32TimeOut;

#ifdef VASCO_OS_WINNT
      EnterCriticalSection((CRITICAL_SECTION*)m_pReceiveCS);
#endif
#ifdef VASCO_OS_LINUX
      pthread_mutex_lock(&mPostMutex);
#endif
      clientSocket->RawSend(oOutputStream.pGetStream(),u32OutputSize);

      Sleep(10);

      tU32 u32InputSize = 8;
      ahl_tclStreamer oInputStream(u32InputSize);
      int s32MsgLength = clientSocket->TimedReceive( oInputStream.pGetStream(), u32InputSize,u32RecvTimeout );
      if ( s32MsgLength == (tS32)u32InputSize )
      {
    	  oInputStream
	    >> u32LastErrorCode
	    >> s32ReturnValue;

	  if (s32ReturnValue > (int) u32BufSize)
	  {
	     //SSA8KOR : MMS31773
	     s32ReturnValue = -3;
	     u32LastErrorCode = -1;
	  }
	  else if (s32ReturnValue > 0) // Contains non-zero size message
            {
	      tU32 u32InputSize = s32ReturnValue + 12 - s32MsgLength;
	      ahl_tclStreamer oInputStream(u32InputSize);

                while ( s32MsgLength < ( 12 + s32ReturnValue ) )
                {
                    int s32TmpMsgLength = clientSocket->TimedReceive( oInputStream.pGetStream(), u32InputSize, u32RecvTimeout);
                    if (s32TmpMsgLength > 0)
                    {
                        s32MsgLength += s32TmpMsgLength;
                    } 
                    else 
                    {
                        if (s32TmpMsgLength < 0)
                        {
                            u32LastErrorCode = -5;
#ifdef VASCO_OS_WINNT
                            LeaveCriticalSection((CRITICAL_SECTION*)m_pReceiveCS);
#endif
#ifdef VASCO_OS_LINUX
                            pthread_mutex_unlock(&mPostMutex);
#endif
                            return -5;
                        }
                    }
                }

                if ( s32MsgLength == ( 12 + s32ReturnValue ) ) // Got the Full Message 
                {
                    oInputStream.vWriteString((const char*)pu8MsgBuf,  (tU16)(s32ReturnValue + 1));
                    memcpy( &u32Prio , oInputStream.pGetStream() + s32ReturnValue, 4);
                    // print for debugging
                    if (log)
                    {
                        fprintf(log,"<MSG_RECV> <!-- incoming message length = %d -->\n",s32ReturnValue);
                        for (int i = 0; i < s32ReturnValue; i++) {
                            fprintf(log,"%02X%c",pu8MsgBuf[i] & 0xFF,(((i%16)==15)?'\n':' '));
                        }
                        fprintf(log,"\n</MSG_RECV>\n");
                        fflush(log);
                    }
                }
                else
                {
                    s32ReturnValue = -2;
                    u32LastErrorCode = -1;
                }
            }
            else // zero size message. consume priority also
            {
                tU32 u32InputSize = 4;
                ahl_tclStreamer oInputStream(u32InputSize);
                int s32MsgLength = clientSocket->TimedReceive( oInputStream.pGetStream(), u32InputSize,u32RecvTimeout);
                if ( s32MsgLength == (tS32)u32InputSize )
                {
                    oInputStream >> u32Prio;
                }
                else
                {
                    s32ReturnValue = -3;
                    u32LastErrorCode = -1;
                }
            }
        }
        else
        {
            s32ReturnValue = -3;
            u32LastErrorCode = -1;
        }
#ifdef VASCO_OS_WINNT
     LeaveCriticalSection((CRITICAL_SECTION*)m_pReceiveCS);
#endif
#ifdef VASCO_OS_LINUX
     pthread_mutex_unlock(&mPostMutex);
#endif
    }
    else
    {
        s32ReturnValue = -4;
        u32LastErrorCode = -1;
    }

    return s32ReturnValue;

}

bool OSALSocketInterface::bInit(bool bEnableLog)
{
   static int iIFCounter = 0;
   iIFCounter++;
   bool bResult = false;
   clientSocket = new SimpleClient();
   if ( clientSocket != NULL )
   {
       log = 0;
       if (bEnableLog)
       {
           char szLogFileName[MAX_PATH] = { '\0' };
           sprintf_s( szLogFileName, MAX_PATH, "SocketTrace%u.log", iIFCounter );
           log = fopen( szLogFileName,"w" );
           if (log == 0) 
           {
               delete clientSocket;
               clientSocket = 0;
               return false;
           }
       }   

       if (s32SocketNumber > 0) 
       {
           if (clientSocket->Create(s32SocketNumber,szHostName) == SUCCESS) 
           {
               bResult = true;
           }
           else
           {
               if (log)
               {
                   fclose(log);
                   log = 0;
               }               
               delete clientSocket;
               clientSocket = 0;
           }
       }
   }
   return bResult;   

}

bool OSALSocketInterface::bEnd()
{
    if (log)
    {
        fclose(log);
        log = 0;
    }
    if (clientSocket != 0) {
        delete clientSocket;
        clientSocket = 0;
    }
    return true;
}

OSAL_tShMemHandle OSALSocketInterface::hSharedMemoryCreate(const char* szName,
		int access, unsigned int u32Size)
{
	OSAL_tShMemHandle retVal = -1;
	//   int systemErrorCode = 0;

	if (clientSocket != NULL)
	{
		tU32 u32OutputSize = (tU32)(12 + strlen(szName) + 1);
		ahl_tclStreamer oOutputStream(u32OutputSize);
		// functionID:
		oOutputStream << (unsigned int) 0x2d;
		// string:
		oOutputStream.vWriteString(szName, (tU16)strlen(szName));
		oOutputStream << (char) 0;
		// access rights:
		oOutputStream << access;
		// mem block size
		oOutputStream << u32Size;
		clientSocket->RawSend(oOutputStream.pGetStream(), u32OutputSize);

		tU32 u32InputSize = (tU32)(8 + sizeof(retVal));
		ahl_tclStreamer oInputStream(u32InputSize);
		int s32MsgLength = clientSocket->TimedReceive(oInputStream.pGetStream(), u32InputSize);
		if (s32MsgLength >= (tS32)u32InputSize)
		{
			unsigned int head = 0;
			oInputStream >> head;
			if (head == 0x802d)
			{
				oInputStream >> retVal;
				//  systemErrorCode = *((unsigned int*) (recvBuf + 8));
			}
		} else
		{
			return 0;
		}
	}

	return retVal;
}


OSAL_tShMemHandle OSALSocketInterface::hSharedMemoryOpen(const char* szName,
		int access)
{
	OSAL_tShMemHandle retVal = -1;
	//   int systemErrorCode = 0;

	if (clientSocket != NULL)
	{
		tU32 u32OutputSize = (tU32)(8 + strlen(szName) + 1);
		ahl_tclStreamer oOutputStream(u32OutputSize);
		// functionID:
		oOutputStream << (unsigned int) 0x2e;
		// string:
		oOutputStream.vWriteString(szName, (tU16)strlen(szName));
		oOutputStream << (char) 0;
		// access rights:
		oOutputStream << access;
		clientSocket->RawSend(oOutputStream.pGetStream(), u32OutputSize);

		//      Sleep(10);

		tU32 u32InputSize = (tU32)(8 + sizeof(retVal));
		ahl_tclStreamer oInputStream(u32InputSize);

		int s32MsgLength = clientSocket->TimedReceive(oInputStream.pGetStream(),
				u32InputSize);
		if (s32MsgLength >= (tS32)u32InputSize)
		{
			unsigned int head = 0;
			oInputStream >> head;
			if (head == 0x802e) {
				oInputStream >> retVal;
				//   systemErrorCode = *((unsigned int*) (recvBuf + 8));
			}
		} else {
			return 0;
		}
	}

	return retVal;
}

int OSALSocketInterface::s32SharedMemoryClose(OSAL_tShMemHandle handle)
{
	int retVal = -1;
	//   int systemErrorCode = 0;

	if (clientSocket != NULL)
	{
		tU32 u32OutputSize = (tU32)(4 + sizeof(handle));
		ahl_tclStreamer oOutputStream(u32OutputSize);
		oOutputStream << (unsigned int) 0x2f;	// functionID:
		oOutputStream << handle;				// handle:
		clientSocket->RawSend(oOutputStream.pGetStream(), u32OutputSize);

		tU32 u32InputSize = 12;
		ahl_tclStreamer oInputStream(u32InputSize);
		int s32MsgLength = clientSocket->TimedReceive(oInputStream.pGetStream(), u32InputSize);
		if (s32MsgLength >= (tS32)u32InputSize)
		{
			unsigned int head = 0;
			oInputStream >> head;
			if (head == 0x802f)
			{
				oInputStream >> retVal;
				//  systemErrorCode = *((unsigned int*) (recvBuf + 8));
			}
		} else
		{
			return 0;
		}
	}
	return retVal;
}

int OSALSocketInterface::s32SharedMemoryDelete(const char *szName)
{
	int retVal = -1;
	//   int systemErrorCode = 0;

	if (clientSocket != NULL)
	{
		tU32 u32OutputSize = (tU32)(4 + strlen(szName) + 1);
		ahl_tclStreamer oOutputStream(u32OutputSize);
		oOutputStream << (unsigned int) 0x30;	// functionID
		oOutputStream.vWriteString(szName, (tU16)strlen(szName));
		oOutputStream << (char) 0;
		clientSocket->RawSend(oOutputStream.pGetStream(), u32OutputSize);

		tU32 u32InputSize = 12;
		ahl_tclStreamer oInputStream(u32InputSize);
		int s32MsgLength = clientSocket->TimedReceive(oInputStream.pGetStream(), u32InputSize);
		if (s32MsgLength >= (tS32)u32InputSize)
		{
			unsigned int head = 0;
			oInputStream >> head;
			if (head == 0x8030)
			{
				oInputStream >> retVal;
				//   systemErrorCode = *((unsigned int*) (recvBuf + 8));
			}
		}
		else
		{
			return 0;
		}
	}
	return retVal;
}

void *OSALSocketInterface::pvSharedMemoryMap(OSAL_tShMemHandle handle, int access, unsigned int length, unsigned int offset)
{
	void *sharedMemoryBlock = NULL;
//    int systemErrorCode = 0;

	if (clientSocket != NULL)
	{
		tU32 u32OutputSize = (tU32)(16 + sizeof(handle));
		ahl_tclStreamer oOutputStream(u32OutputSize);
		oOutputStream
			<< (unsigned int) 0x31	// functionID
			<< handle
			<< access
			<< length
			<< offset;
		clientSocket->RawSend(oOutputStream.pGetStream(), u32OutputSize);

		tU32 u32InputSize = (tU32)(12 + sizeof(intptr_t) + length);
		ahl_tclStreamer oInputStream(u32InputSize);

		int s32MsgLength = clientSocket->TimedReceive(oInputStream.pGetStream(), u32InputSize);
		if (s32MsgLength >= (tS32)(8 + sizeof(intptr_t)))
		{
			unsigned int head = 0;
			oInputStream >> head;
			if (head == 0x8031)
			{
				intptr_t memBlockId = 0;
				unsigned int returnedSize = 0;
				oInputStream
					>> memBlockId
					>> returnedSize;
				//     systemErrorCode            = *((unsigned int *) (recvBuf + 12 + returnedSize));

				if (memBlockId != 0 && returnedSize == length)
				{
					sharedMemoryBlock = (void *) new char[length];
					memcpy(sharedMemoryBlock, oInputStream.pGetStream() + 8 + sizeof(intptr_t), length);
					(*m_poSharedMemoryMap)[sharedMemoryBlock] = memBlockId;
				}
				else
				{
					// Assert gibt es hier leider nicht...
					*((int*) 0x00000000) = 5;
				}
			}
		}
		else
		{
			return 0;
		}
	}
	return sharedMemoryBlock;
}

int OSALSocketInterface::s32SharedMemoryUnmap(void * sharedMemory, unsigned int size)
{
	int retVal = -1;
	// int systemErrorCode = 0;

	if (clientSocket != NULL)
	{
		tU32 u32OutputSize = (tU32)(8 + sizeof(intptr_t));
		ahl_tclStreamer oOutputStream(u32OutputSize);
		oOutputStream << (unsigned int) 0x32;	// functionID

		// lookup mapped pointer to shared memory area
		std::map<void *, intptr_t>::const_iterator shmIter;
		shmIter = m_poSharedMemoryMap->find(sharedMemory);

		if (shmIter != m_poSharedMemoryMap->end())
			oOutputStream << shmIter->second;  // shm block found
		else
			return retVal;   // shm block not found; unmapped already?

		oOutputStream << size;	// handle
		clientSocket->RawSend(oOutputStream.pGetStream(), u32OutputSize);

		//      Sleep(10);
		tU32 u32InputSize = 12;
		ahl_tclStreamer oInputStream(u32InputSize);
		int s32MsgLength = clientSocket->TimedReceive(oInputStream.pGetStream(), u32InputSize);
		if (s32MsgLength >= (tS32)u32InputSize)
		{
			unsigned int head = 0;
			oInputStream >> head;
			if (head == 0x8032)
			{
				oInputStream >> retVal;
				//    systemErrorCode = *((unsigned int*) (recvBuf + 8));
			}
		}
		else
		{
			return 0;
		}
	}

	if (retVal == 0 /* OSAL_OK */) {
		m_poSharedMemoryMap->erase(sharedMemory);
		delete[] (char*) sharedMemory;
	}

	return retVal;
}


int OSALSocketInterface::s32MsgPoolCreate(tU32 u32PoolSize)
{
    int s32MsgQueueCloseCmd = 1000;
    int s32ReturnValue;

    tU32 u32OutputSize = 8;
    ahl_tclStreamer oOutputStream(u32OutputSize);
    oOutputStream
		>> s32MsgQueueCloseCmd
		>> u32PoolSize;
	clientSocket->RawSend(oOutputStream.pGetStream(), u32OutputSize);

    s32ReturnValue = s32ReceiveErrorCode(u32LastErrorCode);

    return s32ReturnValue;
}

tS32 OSALSocketInterface::s32ReceiveErrorCode(tU32& u32ErrorCode)
{
	tS32 s32ReturnValue = 0;
	tU32 u32InputSize = 8;
	ahl_tclStreamer oInputStream(u32InputSize);
	int s32MsgLength = clientSocket->TimedReceive(oInputStream.pGetStream(), u32InputSize);
	if (s32MsgLength == (tS32)u32InputSize)
	{
		oInputStream
			>> u32ErrorCode
			>> s32ReturnValue;
	}
	else
	{
		s32ReturnValue = -1;
		u32ErrorCode = -1;
	}
	return s32ReturnValue;
}

int OSALSocketInterface::s32MsgPoolDelete()
{
    int s32MsgPoolCloseCmd = 1003;
    int s32ReturnValue;
    clientSocket->RawSend((char*)&s32MsgPoolCloseCmd,4);

    s32ReturnValue = s32ReceiveErrorCode(u32LastErrorCode);

    return s32ReturnValue;
}

OSAL_tIODescriptor OSALSocketInterface::s32IOOpen(tU32 u32Access, const char* szFileName)
{
    int s32Cmd = 6;
    OSAL_tIODescriptor uiReturnValue;
    if (clientSocket != 0)
    {
    	tU32 u32OutputSize = (tU32)(8 + strlen(szFileName) + 1);
    	ahl_tclStreamer oOutputStream(u32OutputSize);
    	oOutputStream
			<< s32Cmd
			<< u32Access;
    	oOutputStream.vWriteString(szFileName, (tU16)strlen(szFileName));
    	oOutputStream << (char) 0;
    	clientSocket->RawSend(oOutputStream.pGetStream(), u32OutputSize);

    	tU32 u32InputSize = (tU32)(4 + sizeof(uiReturnValue));
    	ahl_tclStreamer oInputStream(u32InputSize);
    	int s32MsgLength = clientSocket->TimedReceive(oInputStream.pGetStream(), u32InputSize);
        if (s32MsgLength == (tS32)u32InputSize)
        {
        	oInputStream
				>> u32LastErrorCode
				>> uiReturnValue;
        }
        else
        {
            uiReturnValue = -1;
            u32LastErrorCode = -1;
        }
    }
    else
    {
        uiReturnValue = -1;
        u32LastErrorCode = -1;
    }
    return uiReturnValue;
}

OSAL_tIODescriptor OSALSocketInterface::s32IOCreate(tU32 u32Access, const char* szFileName)
{
    int s32Cmd = 9;
    OSAL_tIODescriptor uiReturnValue;

    if (clientSocket != 0)
    {
    	tU32 u32OutputSize = (tU32)(8 + strlen(szFileName) + 1);
    	ahl_tclStreamer oOutputStream(u32OutputSize);
    	oOutputStream
			<< s32Cmd
			<< u32Access;
    	oOutputStream.vWriteString(szFileName, (tU16)strlen(szFileName));
    	oOutputStream << (char) 0;
        clientSocket->RawSend(oOutputStream.pGetStream(), u32OutputSize);

        tU32 u32InputSize = (tU32)(4 + sizeof(uiReturnValue));
        ahl_tclStreamer oInputStream(u32InputSize);
        int s32MsgLength = clientSocket->TimedReceive(oInputStream.pGetStream(), u32InputSize);
		if (s32MsgLength == (tS32)u32InputSize)
		{
			oInputStream
				>> u32LastErrorCode
				>> uiReturnValue;
		}
		else
		{
			uiReturnValue = -1;
			u32LastErrorCode = -1;
		}
    }
    else
    {
        uiReturnValue = -1;
        u32LastErrorCode = -1;
    }
    return uiReturnValue;
}

int OSALSocketInterface::s32IOClose(OSAL_tIODescriptor fd)
{
    int s32Cmd = 8;
    int s32ReturnValue;

    if (clientSocket != 0)
    {
    	tU32 u32OutputSize = (tU32)(4 + sizeof(fd));
    	ahl_tclStreamer oOutputStream(u32OutputSize);
    	oOutputStream
			<< s32Cmd
			<< fd;
        clientSocket->RawSend(oOutputStream.pGetStream(),u32OutputSize);

        s32ReturnValue = s32ReceiveErrorCode(u32LastErrorCode);
    } 
    else 
    {
        s32ReturnValue = -1;
        u32LastErrorCode = -1;
    }
    return s32ReturnValue;
}

int OSALSocketInterface::s32IOCtrl(OSAL_tIODescriptor fd, tS32 s32Fun, tS32 s32Size, unsigned char *ps8Arg)
{
    int s32Cmd = 1110;
    int s32ReturnValue = -1;
    if (clientSocket != 0) 
    {


    	tU32 u32OutputSize = (tU32)(4+4+4+sizeof(fd)+s32Size);
    	ahl_tclStreamer oOutputStream( u32OutputSize );
    	oOutputStream
			<< s32Cmd
			<< fd
			<< s32Fun
			<< s32Size
			<< ps8Arg;
    	if(ps8Arg)
        {
           oOutputStream.vWriteString((const char*) ps8Arg, (tU16)s32Size);
    	   clientSocket->RawSend(oOutputStream.pGetStream(), u32OutputSize);

           tU32 u32InputSize = (tU32)(12+s32Size);
           ahl_tclStreamer oInputStream( u32InputSize );
           clientSocket->TimedReceive(oInputStream.pGetStream(), u32InputSize);

           oInputStream >> u32LastErrorCode;
           oInputStream >> s32ReturnValue;
	       oInputStream.u32IncrementReadIndex(4);
	       oInputStream.vReadStringFromStream((char*)ps8Arg, s32Size, s32Size);
        }
    } 
    else 
    {
        s32ReturnValue = -1;
        u32LastErrorCode = -1;
    }
    return s32ReturnValue;
}

int OSALSocketInterface::s32IOCtrl(OSAL_tIODescriptor fd, tS32 s32Fun, intptr_t iArg)
{
    int s32Cmd = 11;
    int s32ReturnValue = -1;
    if (clientSocket != 0) 
    {
        tU32 u32OutputSize = (tU32)(4 + sizeof(fd) + 4 + sizeof(iArg));
        ahl_tclStreamer oOutputStream( u32OutputSize );
        oOutputStream
			<< s32Cmd
			<< fd
			<< s32Fun
        	<< iArg;
        clientSocket->RawSend(oOutputStream.pGetStream(), u32OutputSize);

        tU32 u32InputSize = 8;
        ahl_tclStreamer oInputStream(u32InputSize);
        clientSocket->TimedReceive(oInputStream.pGetStream(), u32InputSize);
        oInputStream
			>> u32LastErrorCode
			>> s32ReturnValue;
    } 
    else 
    {
        s32ReturnValue = -1;
        u32LastErrorCode = -1;
    }
    return s32ReturnValue;
}

int OSALSocketInterface::s32IORemove(const char* szFileName)
{
    int s32Cmd = 10;
    int s32ReturnValue;
    if (clientSocket != 0)
    {
    	tU32 u32OutputSize = (tU32)(4 + strlen(szFileName) + 1);
    	ahl_tclStreamer oOutputStream(u32OutputSize);

    	oOutputStream << s32Cmd;
    	oOutputStream.vWriteString(szFileName, (tU16)strlen(szFileName));
    	oOutputStream << (char) 0;
    	clientSocket->RawSend(oOutputStream.pGetStream(), u32OutputSize);

        s32ReturnValue = s32ReceiveErrorCode(u32LastErrorCode);
    }
    else
    {
        s32ReturnValue = -1;
        u32LastErrorCode = -1;
    }
    return s32ReturnValue;
}


int OSALSocketInterface::s32IORead(OSAL_tIODescriptor fd, tU32 u32BufSize, unsigned char* pu8Buf)
{
   int s32Cmd = 12;
   int s32ReturnValue;
   if (clientSocket != 0) 
   {
	   tU32 u32OutputSize = (tU32)(8 + sizeof(fd));
	   ahl_tclStreamer oOutputStream(u32OutputSize);
	   oOutputStream
	   	   << s32Cmd
		   << fd
		   << u32BufSize;
	   clientSocket->RawSend(oOutputStream.pGetStream(), u32OutputSize);

      ahl_tclStreamer oInputStream(4+u32BufSize);
      int s32MsgLength = 0;
      int s32TempRead = 0;
      // read error code
      if(clientSocket->RawReceive((char*)&u32LastErrorCode, 4) == ERROR_SOCKET_RECV)
      {}
      // read size
      if(clientSocket->RawReceive((char*)&s32TempRead,4) == ERROR_SOCKET_RECV)
      {}

      s32ReturnValue = 0;
      if (s32TempRead > 0)
      {
    	   do
    	   {
    		   s32TempRead -= s32MsgLength;
    		   s32MsgLength = clientSocket->RawReceive(oInputStream.pGetStream()+s32ReturnValue, s32TempRead);
    		   s32ReturnValue += s32MsgLength;
    	   }
    	   while (s32MsgLength < s32TempRead);
      }

      if (s32ReturnValue != 0)
    	   memcpy(pu8Buf, oInputStream.pGetStream(), s32ReturnValue);
      else
      {
         s32ReturnValue = -1;
         u32LastErrorCode = -1;
      }
   } 
   else 
   {
      s32ReturnValue = -1;
      u32LastErrorCode = -1;
   }
   return s32ReturnValue;

}

int OSALSocketInterface::s32IOWrite(OSAL_tIODescriptor fd, tU32 u32Size, unsigned char* pu8Buf)
{
    int s32Cmd = 7;
    int s32ReturnValue;
    if (clientSocket != 0) 
    {
    	tU32 u32OutputSize = (tU32)(8 + sizeof(fd) + u32Size);
    	ahl_tclStreamer oOutputStream(u32OutputSize);
    	oOutputStream
			<< s32Cmd
			<< fd
			<< u32Size;
    	oOutputStream.vWriteString((const char*) pu8Buf, (tU16)u32Size);
        clientSocket->RawSend(oOutputStream.pGetStream(), u32OutputSize);

        s32ReturnValue = s32ReceiveErrorCode(u32LastErrorCode);
    } 
    else 
    {
        s32ReturnValue = -1;
        u32LastErrorCode = -1;
    }
    return s32ReturnValue;
}


int OSALSocketInterface::s32IOEngFct ( const char *device, 
                                      tS32 s32Fun, 
                                      unsigned char *ps8Arg, 
                                      tS32 s32ArgSize )
{
    unsigned int u32Cmd = 0x35;     // IO-Engineering-Fct
    int s32ReturnValue = -1;

    if (clientSocket != 0) 
    {
    	tU32 u32OutputSize = (tU32)(8 + (strlen(device) + 1) + s32ArgSize + sizeof(s32Fun));
    	ahl_tclStreamer oOutputStream(u32OutputSize);
    	oOutputStream << u32Cmd;
    	oOutputStream.vWriteString(device,(tU16) strlen(device));
    	oOutputStream << (char) 0;
    	oOutputStream
			<< s32Fun
			<< s32ArgSize;
    	oOutputStream.vWriteString((const char*) ps8Arg,(tU16)s32ArgSize);
    	clientSocket->RawSend(oOutputStream.pGetStream(), u32OutputSize);

    	tU32 u32InputSize = (tU32)(16+s32ArgSize);
    	ahl_tclStreamer oInputStream(u32InputSize);
    	tU32 u32RetArgSize;
    	int s32MsgLength = (int)clientSocket->TimedReceive(oInputStream.pGetStream(), u32InputSize);
        if(s32MsgLength <= 0)
        {}
    	oInputStream
			>> u32Cmd
			>> u32LastErrorCode
			>> s32ReturnValue
			>> u32RetArgSize;
    	oInputStream.vReadStringFromStream((char*) ps8Arg, s32ArgSize, u32RetArgSize);
    } 
    else 
    {
        s32ReturnValue = -1;
        u32LastErrorCode = -1;
    }

    return s32ReturnValue;
}


int OSALSocketInterface::s32IOFakeExclAccess (const char *device)
{
    unsigned int u32Cmd = 0x36;     // IO-Engineering-Fct
    int s32ReturnValue = -1;

    if (clientSocket != 0) 
    {
        int reqSize = (int)(4 + (strlen(device) + 1));

        unsigned char *buf = new unsigned char[reqSize];
        if (buf != 0)
        {
            ByteArrayWriter writer (buf, reqSize);
            writer.bWriteU32(u32Cmd);
            writer.bWriteString(device);

            clientSocket->RawSend((char*) buf, writer.u32GetActualDataSize());
            int s32MsgLength = clientSocket->TimedReceive((char*) buf, 12);
            ByteArrayReader reader (buf, s32MsgLength);
            reader.bReadU32 (&u32Cmd);
            reader.bReadU32 (&u32LastErrorCode);
            reader.bReadS32 (&s32ReturnValue);

            delete [] buf;
        }
        else
        {
            s32ReturnValue = -1;
            u32LastErrorCode = -1;
        }
    } 
    else 
    {
        s32ReturnValue = -1;
        u32LastErrorCode = -1;
    }

    return s32ReturnValue;
}

int OSALSocketInterface::s32IOReleaseFakedExclAccess (const char *device)
{
    unsigned int u32Cmd = 0x37;     // IO-Engineering-Fct
    int s32ReturnValue = -1;

    if (clientSocket != 0) 
    {
        int reqSize = (int)(4 + (strlen(device) + 1));

        unsigned char *buf = new unsigned char[reqSize];
        if (buf != 0)
        {
            ByteArrayWriter writer (buf, reqSize);
            writer.bWriteU32(u32Cmd);
            writer.bWriteString(device);

            clientSocket->RawSend((char*) buf, writer.u32GetActualDataSize());
            int s32MsgLength = clientSocket->TimedReceive((char*) buf, reqSize);
            ByteArrayReader reader (buf, s32MsgLength);
            reader.bReadU32 (&u32Cmd);
            reader.bReadU32 (&u32LastErrorCode);
            reader.bReadS32 (&s32ReturnValue);

            delete [] buf;
        }
        else
        {
            s32ReturnValue = -1;
            u32LastErrorCode = -1;
        }
    } 
    else 
    {
        s32ReturnValue = -1;
        u32LastErrorCode = -1;
    }

    return s32ReturnValue;
}

int OSALSocketInterface::s32MiniSPMInitiateShutdown()
{
    unsigned int u32Cmd = 0x38;     // MiniSPM-Shutdown Request
    int s32ReturnValue = -1;

    if (clientSocket != 0) 
    {
        int reqSize = 4;

        unsigned char *buf = new unsigned char[reqSize];
        if (buf != 0)
        {
            ByteArrayWriter writer (buf, reqSize);
            writer.bWriteU32(u32Cmd);

            clientSocket->RawSend((char*) buf, writer.u32GetActualDataSize());
            int s32MsgLength = clientSocket->TimedReceive((char*) buf, reqSize);
            ByteArrayReader reader (buf, s32MsgLength);
            reader.bReadU32 (&u32Cmd);
            reader.bReadS32 (&s32ReturnValue);
            u32LastErrorCode = -1;

            delete [] buf;
        }
        else
        {
            s32ReturnValue = -1;
            u32LastErrorCode = -1;
        }
    } 
    else 
    {
        s32ReturnValue = -1;
        u32LastErrorCode = -1;
    }

    return s32ReturnValue;
}

int OSALSocketInterface::s32FSeek(OSAL_tIODescriptor fd,int offset, int origin)
{
    int curPos;
    int endPos;
    if (origin == 0)
    {
        return s32IOCtrl(fd, ((tS32)1002), (long)offset);
    }
    else if (origin == 1)
    {
        if (-1 == s32IOCtrl(fd, ((tS32)1001), sizeof(int), (unsigned char *) &curPos))
            return -1;
        offset += curPos;
        return s32IOCtrl(fd, ((tS32)1002), offset);
    }
    else if (origin == 2)
    {
        if (-1 == s32IOCtrl(fd, ((tS32)1004),sizeof(int), (unsigned char *) &endPos))
            return -1;
        if (-1 == s32IOCtrl(fd, ((tS32)1001), sizeof(int), (unsigned char *) &curPos))
            return -1;
        offset = curPos + endPos + offset;
        return s32IOCtrl(fd, ((tS32)1002), offset);
    }
    else
        return -1;
}

int OSALSocketInterface::s32FTell(OSAL_tIODescriptor fd)
{
    int curPos;
    if (-1 == s32IOCtrl(fd, ((tS32)1001),sizeof(int), (unsigned char *) &curPos))
        return -1;
    else
        return curPos;
}

int OSALSocketInterface::s32FGetpos(OSAL_tIODescriptor fd,intptr_t *ptr)
{
    return s32IOCtrl(fd, ((tS32)1001),sizeof(intptr_t), (unsigned char *) ptr);
}

int OSALSocketInterface::s32FSetpos(OSAL_tIODescriptor fd,const intptr_t *ptr)
{
    return s32IOCtrl(fd, ((tS32)1002), *ptr);
}

int OSALSocketInterface::s32FGetSize(OSAL_tIODescriptor fd)
{
    int front;
    int back;
    if (-1 == s32IOCtrl(fd, ((tS32)1001),sizeof(int), (unsigned char *) &front))
        return -1;
    if (-1 == s32IOCtrl(fd, ((tS32)1004),sizeof(int), (unsigned char *) &back))
        return -1;
    return front+back;
}

int OSALSocketInterface::s32CreateDir(OSAL_tIODescriptor fd,const char* szDirectory)
{
    return s32IOCtrl(fd, ((tS32)1005),256, (unsigned char *) szDirectory);
}

int OSALSocketInterface::s32RemoveDir(OSAL_tIODescriptor fd,const char* szDirectory)
{
    return s32IOCtrl(fd, ((tS32)1006), 256, (unsigned char *) szDirectory);
}

intptr_t OSALSocketInterface::prOpenDir(const char* szDirectory)
{
    intptr_t s32Cookie=0;
    OSAL_tIODescriptor iReturn=-1;

    char * Buffer = new char[sizeof(s32Cookie) + sizeof(iReturn) + 256];// Allocate memory for OSAL_trIOCtrlDir
    if (Buffer == NULL)
        return 0;

    iReturn = s32IOOpen(0x0004,szDirectory); // Open with READWRITE access
    if (iReturn == -1)
    {
        delete [] Buffer;
        return 0;
    }
    memcpy(Buffer,&iReturn,sizeof(iReturn)); // Copy the file descriptor
    memcpy(Buffer+sizeof(iReturn),&s32Cookie,sizeof(s32Cookie));// Copy the s32Cookie
    return (intptr_t)Buffer;
}

intptr_t OSALSocketInterface::prReadDir (intptr_t pDir)
{
    char * Buffer =(char*)pDir;
    intptr_t fd =-1;
    intptr_t pDirent=-1;

    memcpy(&fd,Buffer,sizeof(intptr_t));
    if (s32IOCtrl(fd, ((tS32)1003),sizeof(tU32) + sizeof(tS32) + 256,(unsigned char*)pDir) == -1)
    {
        return 0;
    }
    else
    {
        pDirent = pDir + sizeof(tU32) + sizeof(tS32); // Pointer to the Dirent structure
        return pDirent;
    }
}

int OSALSocketInterface::s32CloseDir (intptr_t pDir)
{
    char * Buffer = (char*) pDir;
    OSAL_tIODescriptor fd=-1;

    memcpy(&fd,Buffer,sizeof(fd));
    s32IOClose(fd);
    delete Buffer;

    return 0;
}

