/*****************************************************************************
| FILE:         osalmqueue.cpp
| PROJECT:      platform
| SW-COMPONENT: OSAL
|-----------------------------------------------------------------------------
| DESCRIPTION:  This is the implementation file for the OSAL
|               (Operating System Abstraction Layer) Message Queue-Functions.
|
|-----------------------------------------------------------------------------
| COPYRIGHT:    (c) 2010 Robert Bosch GmbH
| HISTORY:      
| Date      | Modification               | Author
| 03.10.05  | Initial revision           | MRK2HI
| --.--.--  | ----------------           | -------, -----
| 13.12.12  | Fix for OSAL MQ priority   | sja3kor
|           |  issue.(CFAGII-497)        | 
| 17.12.12  | Fix for MMS CFAGII-501.    |sja3kor 
|           | Boundary check condition   |
|           | changed.                   |          
| --.--.--  | ----------------           | -------, -----
| 16.01.13  |Fix for Wrong timeout for   |sja3kor 
|           |Linux messagequeue post/wait|
|           |CFAGII-506                  |          
|21.01.13   |Fix for MMS GMNGA48007      |SWM2KOR        
|*****************************************************************************/

/************************************************************************
| includes of component-internal interfaces
| (scope: component-local)
|-----------------------------------------------------------------------*/
#include "OsalConf.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "Linux_osal.h"

#include "ostrace.h"


#ifdef __cplusplus
extern "C" {
#endif



/************************************************************************
|defines and macros (scope: module-local)
|-----------------------------------------------------------------------*/

#define  LINUX_C_U32_MQUEUE_ID                   ((tU32)0x51554555) 
#define  USER_HANDLE_MAGIC                         ((tU32)0x75686D71) 

#define  NOTATHREAD                                  ((tS32)-1)

#ifdef OSAL_SHM_MQ
#define  LINUX_C_MQ_MAX_NAMELENGHT (OSAL_C_U32_MAX_NAMELENGTH + 2)
#else
#define  LINUX_C_MQ_MAX_NAMELENGHT (OSAL_C_U32_MAX_NAMELENGTH)
#endif


//#define NOTEMPTY_PATTERN  0x01
//#define BLOCKING_PATTERN  0x10
//#define LINUX_MQ

/************************************************************************
|typedefs (scope: module-local)
|-----------------------------------------------------------------------*/
typedef struct{
#ifdef QNX
     name_attach_t *attach;
#endif
     int             u32MqIdRef;  /* process specific Linux MQ handle referenz */
     int             u32MqId;     /* process specific Linux MQ handle */
     OSAL_tProcessID PrcId;       /* process ID of current process */
     tU32            u32PrcCnt;   /* open counter for a linux MQ of current process */
}trResToPidMap2;


/************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------*/
OSAL_tProcessID MqPid = 0;

static trResToPidMap2* prLiMqMap=NULL;

#ifndef OSAL_SHM_MQ_FOR_LINUX_MQ
static tBool bTraceHandle = TRUE;
#endif
/************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------*/
extern uintptr_t* pu32GetSharedBaseAdress(void);
extern tS32 s32GetPrcIdx(OSAL_tProcessID PrcId);
uintptr_t GetMemPoolOffset(trHandleMpf* pHandle, uintptr_t* pU32Mem);

#ifdef DBUS_MQ_SUPPORT
extern ConnectDBus      pConnectDBus;
extern DisConnectDBus   pDisConnectDBus;
extern MsgSendDBus      pMsgSendDBus;
extern MsgRecvDBus      pMsgRecvDBus;
extern RelIfDBus        pRelIfDBus;
#endif
/************************************************************************
|function prototype (scope: module-local)
|-----------------------------------------------------------------------*/
tS32 s32MqueueTableCreate(tVoid);
tS32 s32MqueueTableDeleteEntries(void);
#ifdef DBUS_MQ_SUPPORT
tS32 OSAL_s32DBusMessageQueueWait( OSAL_tMQueueHandle hMQ,tPU8 pu8Buffer,tU32 u32Length,tPU32 pu32Prio,OSAL_tMSecond msec);
tS32 OSAL_s32DBusMessageQueuePost( OSAL_tMQueueHandle hMQ,tPCU8 pcou8Msg,tU32  u32Length,tU32  u32Prio);
tS32 OSAL_s32DBusMessageQueueClose (OSAL_tMQueueHandle phMQueue);
tS32 OSAL_s32DBusMessageQueueOpen (tCString coszName,OSAL_tenAccess enAccess,OSAL_tMQueueHandle* phMQ);
tS32 OSAL_s32DBusMessageQueueCreate(tCString coszName,tU32 u32MaxMessages,tU32 u32MaxLength,OSAL_tenAccess enAccess,OSAL_tMQueueHandle* phMQ);
tS32 OSAL_s32DBusMessageQueueDelete(tCString coszName);
#endif

trMqueueElement* tMqueueTableSearchEntryByName(tCString coszName);

struct msgblock
{
    /*lint -esym(754,msgblock::u16MagicField) msgblock::u16MagicField is not referenced PQM_authorized_435*/       
     tU16 u16MagicField;   /*identify status of the following block OSAL_MSG_MAGIC     -> valid message
                                                                   OSAL_DEL_MSG_MAGIC -> deleted message
                                                                   OSAL_INV_MSG_MAGIC -> marker for begin of guard intervall */
     tU16 u16MsgOffset;    /* offset after OSAL_MSG_uHeader of the message for begin of message 1-11 bytes */
     /*lint -esym(754,msgblock::s32Offset) msgblock::s32Offset is not referenced PQM_authorized_435*/       
     tS32 s32Offset;       /* offset in the pool */
     /*lint -esym(754,msgblock::nsize) msgblock::nsize is not referenced PQM_authorized_435*/       
     tU32 nsize;           /* size in blocks need for the message*/
} ;/*lint !e612 */



static OSAL_tTimerHandle hMsgCntTimer; 
static int TraceCount = 0; 
static int MaxTraceCount = 0;
void vStopCountMessages(void* id)
{
   (tVoid)id; /* satisfy lint*/
   TraceCount++;
   if(MaxTraceCount == TraceCount)
   {
      uint32_t Val = pOsalData->MessageCountingActive;
      pOsalData->MessageCountingActive = 0;
      OSAL_s32TimerSetTime(hMsgCntTimer,0,0);
      OSAL_s32TimerDelete(hMsgCntTimer);
      TraceCount = 0; 
      MaxTraceCount = 0;
      TraceString("------------------------------------------------------------");
      TraceString("MSGCNT Messages Count Interval %d msec",Val);
      TraceString("------------------------------------------------------------");
   }
   TraceString("------------------------------------------------------------");
   TraceString("MSGCNT Number of OSAL Messages %d",pOsalData->MessageCount);
   TraceString("MSGCNT Number of OSAL CCA Messages %d",pOsalData->CCAMessageCount);
   TraceString("------------------------------------------------------------");
 //  vWritePrintfErrmem("MSGCNT Messages Count Interval %d msec",Val);
 //  vWritePrintfErrmem("MSGCNT Number of OSAL Messages %d",pOsalData->MessageCount);
 //  vWritePrintfErrmem("MSGCNT Number of OSAL CCA Messages %d",pOsalData->CCAMessageCount);
   return;
}

void StartMessageCountTimer(tU32 msec, tU32 interval)
{
   tU32 u32Tmo = msec;
   TraceString("MSGCNT Start %d msec Timer for message counting",msec);
   pOsalData->MessageCountingActive = msec;
   pOsalData->MessageCount = 0;
   pOsalData->CCAMessageCount = 0;
   if(interval)
   {
      MaxTraceCount = msec/interval;
      u32Tmo = MaxTraceCount;
   }
   else
   {
      MaxTraceCount = 1;
   }
   if(OSAL_s32TimerCreate(vStopCountMessages,NULL,&hMsgCntTimer) == OSAL_OK)
   {
      if(OSAL_s32TimerSetTime(hMsgCntTimer,u32Tmo,interval) == OSAL_ERROR)
      {
          NORMAL_M_ASSERT_ALWAYS();
      }
   }
   else
   {
       NORMAL_M_ASSERT_ALWAYS();
   }
}



void CountMessage(tU32 Size)
{
  if(pOsalData->MessageCountingActive)
  {
     LockOsal(&pOsalData->MsgCntLock);
     pOsalData->MessageCount++;
     if(Size == 8)
     {
        pOsalData->CCAMessageCount++;
     }
     UnLockOsal(&pOsalData->MsgCntLock);
  }
}

void vCheckMqTimeout(void)
{
#ifndef OSAL_SHM_MQ_FOR_LINUX_MQ
    tU32 tiIndex;
    tU32 u32Val = 0xA3A3A3A3;
    tBool bToBeClosed = FALSE;
    char buf[LINUX_C_MQ_MAX_NAMELENGHT+1];
    struct timespec rData  = {0,0};
    (void)u32GemTmoStruct(&rData  ,1000);
    if(!prLiMqMap)
    {
       /* create dummy queue */
       OSAL_tMQueueHandle hMQ;
       (void)OSAL_s32MessageQueueCreate("SET_TIME_DUMMY",1,8,OSAL_EN_READWRITE,&hMQ);
       (void)OSAL_s32MessageQueueClose(hMQ);
       (void)OSAL_s32MessageQueueDelete("SET_TIME_DUMMY");
    }
    if((pMsgQEl)&&(prLiMqMap))
    {
       for( tiIndex=0;  tiIndex <  pOsalData->MqueueTable.u32MaxEntries ; tiIndex++ )
       {
          if((pMsgQEl[tiIndex].bIsUsed == TRUE)&&(pMsgQEl[tiIndex].u32WaitTime))
          {
             pMsgQEl[tiIndex].bCheckTmo = TRUE;
             /* trigger event to force recalculation of timeout*/   
             if(prLiMqMap[pMsgQEl[tiIndex].u32EntryIdx].u32MqId == -1)
             {
                buf[0] = '/';
                strncpy(&buf[1],pMsgQEl[tiIndex].szName ,LINUX_C_MQ_MAX_NAMELENGHT);
//                TraceString("OSALTIM Trigger for MQ:%s",&buf[1]);
                prLiMqMap[pMsgQEl[tiIndex].u32EntryIdx].u32MqId = mq_open(buf, O_RDWR , OSAL_ACCESS_RIGTHS, NULL);
                bToBeClosed = TRUE;
             }
             if(mq_timedsend(prLiMqMap[pMsgQEl[tiIndex].u32EntryIdx].u32MqId,(const char*)&u32Val,sizeof(tU32),0,&rData) == -1)
             {
                pMsgQEl[tiIndex].bCheckTmo = FALSE;
                if(pOsalData->u32ClkMonotonic & 0x10000000)TraceString("OSALMQ Send SV Trigger to %s failed errno %d",pMsgQEl[tiIndex].szName,errno);
             }
             if(bToBeClosed)
             {
                mq_close(prLiMqMap[pMsgQEl[tiIndex].u32EntryIdx].u32MqId);
                prLiMqMap[pMsgQEl[tiIndex].u32EntryIdx].u32MqId = -1;
                bToBeClosed = FALSE;
             }
          }
       }
   }
   else
   {
      TraceString("OSALMQ Cannot check MQ timeouts -> no own MQ ");
   }
 #endif
}

/************************************************************************
|function implementation (scope: module-local)
|-----------------------------------------------------------------------*/
/*
20 00 00 01   27 00 00 00   02 00 00 45   00 00 00 00   00 00 00 00   70 00 ff ff   01 00 02 00   00 00 00 00   01 00 00 00
00 01 02 03   04       07   08       11   12       15   16       19   20       23   24    26 27   28       31   32       35
src   dest    size          versi co ty   s-sub d-sub   Time__Stamp   srvid regid   fncid Op act  cmdctr s sctr data
|     |       |                      |                                |             |     |
|     |       0x27 = 39 byte         type = 0x45 = ServiceData        |             |     Opcode 0x2= MethodStart
|     0x100= CCA_C_U16_APP_DIAGLOG                                    |             Function-ID 0x1= SaveTestResult
0x20= CCA_C_U16_APP_KBD                                               0x70= CCA_C_U16_SRV_DIAGLOG (ccaservice.h)
*/
tBool bFilterMsg(OSAL_trMessage* handle)
{
    tU16 u16Src,u16Dst,u16serv,u16Func;
    uintptr_t u32Offset;

    if((char)(handle->enLocation) == OSAL_EN_MEMORY_LOCAL)
    {
        u32Offset = (uintptr_t)(handle->u32Offset);
    }
    else
    {
       msgblock* ptr = (msgblock*)((uintptr_t)pu32GetSharedBaseAdress()+ handle->u32Offset);
       u32Offset = (uintptr_t)(ptr) + sizeof(msgblock)+ (tU32)ptr->u16MsgOffset ;
    }

    if(pOsalData->bTraceAllCca ==TRUE)
    {
       return TRUE;
    }

    if(pOsalData->u32CheckSrc != 0xffffffff)
    {
      memcpy(&u16Src,(void*)u32Offset,sizeof(tU16));
      if(pOsalData->u32CheckSrc == u16Src)
      {
        return TRUE;
      }
    }
    if(pOsalData->u32CheckDst != 0xffffffff)
    {
      memcpy(&u16Dst,(void*)(u32Offset + sizeof(tU16)),sizeof(tU16));
      if(pOsalData->u32CheckDst == u16Dst)
      {
        return TRUE;
      }
    }
    if(pOsalData->u32ServId != 0xffffffff)
    {
      memcpy(&u16serv,(void*)(u32Offset + 20),sizeof(tU16));
      if(pOsalData->u32ServId == u16serv)
      {
        return TRUE;
      }
    }
    if(pOsalData->u32FuncId != 0xffffffff)
    {
      memcpy(&u16Func,(void*)(u32Offset + 24),sizeof(tU16));
      if(pOsalData->u32FuncId == u16Func)
      {
        return TRUE;
      }
    }
    return FALSE;
}

void vTraceCcaMsg(OSAL_trMessage* handle, tU32 MsgSize,trMqueueElement *pCurrentEntry,tCString coszName)
{
   if(pCurrentEntry == NULL)// called by OSAL message API 
   {
      if(((char)(handle->enLocation) == OSAL_EN_MEMORY_LOCAL)
       ||((char)(handle->enLocation) == OSAL_EN_MEMORY_SHARED))
      {
         if(bFilterMsg(handle))
         {
            /* valid handle */
            TraceString("OSALMQ %s Task:%d Handle:%p u32Offset:%d Location:%d",
                                coszName,
                                OSAL_ThreadWhoAmI(),
                                handle,
                                handle->u32Offset,
                                handle->enLocation);
         }
      }
      else
      {
            /* invalid handle */
            TraceString("OSALMQ %s Task:%d Defect Handle:%p",
                               coszName,
                               OSAL_ThreadWhoAmI(),
                               handle);
      }
   }
   else
   {
     if((MsgSize == 8) /* size of CCA message */
      &&(pCurrentEntry->u32Marker != 0xffffffff)) //magic for exclude of supervision
     {
        /* CCA message queue */
        if(((char)(handle->enLocation) == OSAL_EN_MEMORY_LOCAL)
         ||((char)(handle->enLocation) == OSAL_EN_MEMORY_SHARED))
        {
           if(bFilterMsg(handle))
           {
              /* valid handle */
               TraceString("OSALMQ %s Task:%d Mbx:%s Handle:%p u32Offset:%d Location:%d",
                                  coszName,
                                  OSAL_ThreadWhoAmI(),
                                  pCurrentEntry->szName,
                                  handle,
                                  handle->u32Offset,
                                  handle->enLocation);
           }
        }
        else
        {
              /* invalid handle */
              TraceString("OSALMQ %s Task:%d Mbx:%s Defect Handle:%p",
                                  coszName,
                                  OSAL_ThreadWhoAmI(),
                                  pCurrentEntry->szName,
                                  handle);
        }
     }
   }
}

tVoid vMqueueLiMqMapInit(tVoid)
{
   tU32    i;
   if(prLiMqMap == NULL)
   {
       MqPid = OSAL_ProcessWhoAmI();
       prLiMqMap    = (trResToPidMap2*)malloc(pOsalData->u32MaxNrMqElements * sizeof(trResToPidMap2));
       if(prLiMqMap == NULL)
       {
          FATAL_M_ASSERT_ALWAYS();
       }
       else
       {
          for(i = 0; i < pOsalData->u32MaxNrMqElements; i++)
          {
             prLiMqMap[i].PrcId   = 0;
             prLiMqMap[i].u32MqId = -1;
             prLiMqMap[i].u32MqIdRef = -1;
          }
       }
    }
}

void vResetEntry(tU32 u32Index)
{
    memset(&pMsgQEl[u32Index],0,sizeof(trMqueueElement));
    pMsgQEl[u32Index].u32MqueueID   = LINUX_C_U32_MQUEUE_ID;
    pMsgQEl[u32Index].u32EntryIdx   = u32Index;
}
 
/*****************************************************************************
 *
 * FUNCTION:    s32MqueueTableCreate
 *
 * DESCRIPTION: This function creates the Mqueue Table List. If
 *              there isn't space it returns a error code.
 *
 * PARAMETER:   none
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.10.05  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
 tS32 s32MqueueTableCreate(tVoid)
{
   tU32 tiIndex;

   pOsalData->MqueueTable.u32UsedEntries      = 0;
   pOsalData->MqueueTable.u32MaxEntries       = pOsalData->u32MaxNrMqElements;

   for( tiIndex=0;  tiIndex <  pOsalData->MqueueTable.u32MaxEntries ; tiIndex++ )
   {
      pMsgQEl[tiIndex].u32MqueueID   = LINUX_C_U32_MQUEUE_ID;
      pMsgQEl[tiIndex].bIsUsed       = FALSE;
      pMsgQEl[tiIndex].bTraceCannel  = FALSE;
      pMsgQEl[tiIndex].u16Type       = 0;
      pMsgQEl[tiIndex].u32EntryIdx   = tiIndex;
   }
   return OSAL_OK;
}

/*void CheckandCorrectMqStaticValues(void)
{
   tU32 tiIndex;
   TraceString("OSALMQ Task %d detects Overwritten MQ data\n",OSAL_ThreadWhoAmI());
   for( tiIndex=0;  tiIndex <  pOsalData->u32MaxNrMqElements; tiIndex++ )
   {
      if(pMsgQEl[tiIndex].u32MqueueID != LINUX_C_U32_MQUEUE_ID)
      {
          vWritePrintfErrmem("MQ Magic for Index %d overwritten with 0x%x \n",tiIndex,pMsgQEl[tiIndex].u32MqueueID);
          TraceString("OSALMQ MQ Magic for Index %d overwritten with 0x%x \n",tiIndex,pMsgQEl[tiIndex].u32MqueueID);
          pMsgQEl[tiIndex].u32MqueueID   = LINUX_C_U32_MQUEUE_ID;
      }
      if(pMsgQEl[tiIndex].u32EntryIdx != tiIndex)
      {
          vWritePrintfErrmem("MQ Index for Index %d overwritten with 0x%x \n",tiIndex,pMsgQEl[tiIndex].u32EntryIdx);
          TraceString("OSALMQ MQ Index for Index %d overwritten with 0x%x \n",tiIndex,pMsgQEl[tiIndex].u32EntryIdx);
          pMsgQEl[tiIndex].u32EntryIdx = tiIndex;
      }
   }
}*/

/*****************************************************************************
 *
 * FUNCTION:    tMqueueTableGetFreeEntry
 *
 * DESCRIPTION: this function goes through the Mqueue List and returns the
 *              first MqueueElement.In case no entries are available a new
 *              defined number of entries is attempted to be added.
 *              In case of success the first new element is returned, otherwise
 *              a NULL pointer is returned.
 *
 *
 * PARAMETER:   tVoid
 *
 * RETURNVALUE: trMqueueElement*
 *                 free entry pointer or OSAL_NULL
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.10.05  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
trMqueueElement* tMqueueTableGetFreeEntry(tVoid)
{
   trMqueueElement *pCurrent  = &pMsgQEl[0];
   tU32 u32Ret;
   for(u32Ret=0;u32Ret<pOsalData->u32MaxNrMqElements;u32Ret++)
   {
         if(pCurrent->bIsUsed == FALSE)break;
         else pCurrent++;
   }
   if(u32Ret == pOsalData->u32MaxNrMqElements)
   {
      return NULL;
   }
   else 
   {
      pOsalData->MqueueTable.u32UsedEntries++;
      return pCurrent;
   }
}


void vCleanUpMqofContext(void)
{
   OSAL_tProcessID OwnPid = OSAL_ProcessWhoAmI();
   uintptr_t s32MqHandle = 0;
   tU32 i;
   trMqueueElement *pCurrentEntry = NULL;
   for(i=0;i<pOsalData->u32MqHandles;i++)
   {
      s32MqHandle = (uintptr_t)pvGetFirstElementForPid(&MqMemPoolHandle,OwnPid);
      if(s32MqHandle)
      {
         if(((tMQUserHandle*)s32MqHandle)->pOrigin)
         {
             /* cleanup queue */
             pCurrentEntry = ((tMQUserHandle*)s32MqHandle)->pOrigin; /*lint !e613 *//* pointer already checked*/ 
             if((pCurrentEntry->PID == OwnPid)&&(pCurrentEntry->bToDelete == FALSE))
             {
                TraceString("OSALMQ OSAL_s32MessageQueueDelete for %s",pCurrentEntry->szName);
                OSAL_s32MessageQueueDelete(pCurrentEntry->szName);
             }		 
             if(pCurrentEntry->u16Type == MQ_STD)
             {
                TraceString("OSALMQ OSAL_s32MessageQueueClose for %s",pCurrentEntry->szName);
                if(OSAL_s32MessageQueueClose(s32MqHandle) == OSAL_ERROR)
                {
                   break;
                }
             }
         }
         else
         {
             /* defect handle in pool */
             TraceString("OSALMQ  MQ pvGetFirstElementForPid gives handle without USER_HANDLE_MAGIC");
             break;
         }
      }
      else
      {
          /* last handle for MqPid was found */
          break;
      }
   }
}



/*****************************************************************************
 *
 * FUNCTION:     tMqueueTableSearchEntryByName
 *
 * DESCRIPTION:  return the pointer to a NU_MESSAGE_QUEUE with a given name or NULL
 *               in the case a NU_MESSAGE_QUEUE with this name doen't exist
 *
 * PARAMETER:    pName
 *
 * RETURNVALUE:  pointer to the searched NU_MESSAGE_QUEUE or NULL
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.10.05  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
trMqueueElement *tMqueueTableSearchEntryByName(tCString coszName)
{
   trMqueueElement *pCurrent = &pMsgQEl[0];
   tU32 u32Loop;
   for(u32Loop = 0; u32Loop < pOsalData->u32MaxNrMqElements; u32Loop++)
   {
      if((pCurrent->bIsUsed != 0)
        &&(OSAL_s32StringCompare(coszName, (tString)pCurrent->szName) == 0))break;
        pCurrent++;
    }
    if(u32Loop >= pOsalData->u32MaxNrMqElements)
    {
      pCurrent = NULL;
    }
    return pCurrent;
}
/*
static tU32 tMqueueTableSearchIndexByName(tCString coszName)
{
   trMqueueElement *pCurrent = &pMsgQEl[0];
   tU32 u32Loop;
   for(u32Loop = 0; u32Loop < pOsalData->u32MaxNrMqElements; u32Loop++)
   {
      if((pCurrent->bIsUsed != 0)
        &&(OSAL_s32StringCompare(coszName, (tString)pCurrent->szName) == 0))break;
        pCurrent++;
    }
    if(u32Loop >= pOsalData->u32MaxNrMqElements)
    {
      u32Loop = 0;
    }
    return u32Loop;
}*/

/*****************************************************************************
 *
 * FUNCTION:     tMqueueTableSearchEntryByHandle
 *
 * DESCRIPTION:  return the pointer to a NU_MESSAGE_QUEUE with a given handle or NULL
 *               in the case a NU_MESSAGE_QUEUE with this handle doen't exist
 *
 * PARAMETER:    pBox
 *
 * RETURNVALUE:  pointer to the searched NU_MESSAGE_QUEUE or NULL
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.10.05  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
 /*
static trMqueueElement* tMqueueTableSearchEntryByHandle(OSAL_tMQueueHandle handle)
{
   trMqueueElement *pCurrent = pMsgQEl;
   pCurrent = pCurrent + handle;

   if(pCurrent->bIsUsed == FALSE )pCurrent = NULL;
   return pCurrent;

   return(pCurrent);
}

*/

/*****************************************************************************
*
* FUNCTION:    bSharedMemoryTableDeleteEntryByName
 *
 * DESCRIPTION: this function goes throught the Semaphor List deletes the
 *              Semaphor Element with the given name 
 *
 * PARAMETER:   coszName (I)
 *                 event name wanted.
 *
 * RETURNVALUE: tBool  
 *                TRUE = success FALSE=failed
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.10.05  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
*****************************************************************************
static tBool bMqueueTableDeleteEntryByName(tCString coszName)
{
   trMqueueElement *pCurrent = MqueueTable.ptrHeader;

   while( (pCurrent!= OSAL_NULL)
       && (pCurrent->pBasicInfo != NULL) && (OSAL_s32StringCompare(coszName,(tString)pCurrent->pBasicInfo->szName) != 0) )
   {
      pCurrent = pCurrent->pNext;
   }

   if(pCurrent!= OSAL_NULL)
   {
      if(pCurrent->bIsUsed == FALSE )
     {
         pCurrent->u32MqueueID    = 0;
         pCurrent->nuNotEmptyMQEvent.exinf  = 0;
         pCurrent->nuNotEmptyMQEvent.flgatr = 0;
         pCurrent->nuNotEmptyMQEvent.iflgptn = 0;
         pCurrent->idNotEmptyFlg  = 0;
         pCurrent->bIsUsed        = 0;
         pCurrent->bToDelete      = 0;
         pCurrent->u16ContextID   = 0;
         pCurrent->u16OpenCounter = 0;
         pCurrent->pNext          = 0;
         memset((void*)&pCurrent->nuMqueue,0,sizeof(TE_MESSAGE_QUEUE));
         memset(pCurrent->pBasicInfo->szName,0,LINUX_C_MQ_MAX_NAMELENGHT);
         return TRUE;
     }
   }

   return FALSE;
}*/

/************************************************************************
|function implementation (scope: global)
|-----------------------------------------------------------------------*/

/*****************************************************************************
 *
 * FUNCTION:    s32MqueueTableDeleteEntries
 *
 * DESCRIPTION: This function deletes all elements from OSAL table,
 *
 * PARAMETER:
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 *
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.10.05  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
tS32 s32MqueueTableDeleteEntries()
{
   tS32 s32ReturnValue = OSAL_OK;
   tS32 s32DeleteCounter = 0;
   trMqueueElement *pCurrentEntry = pMsgQEl;

   if(LockOsal(&pOsalData->MqueueTable.rLock) == OSAL_OK)
   {
      /* search the used table */
      while( pCurrentEntry < &pMsgQEl[pOsalData->MqueueTable.u32UsedEntries] )
      {
         if( (pCurrentEntry->bIsUsed == TRUE) )
         {
            s32DeleteCounter++;
               pCurrentEntry->bIsUsed = FALSE;
               s32ReturnValue = s32DeleteCounter;
         }
         pCurrentEntry++;
      }
      UnLockOsal(&pOsalData->MqueueTable.rLock);
   }
   return(s32ReturnValue);
}


void vTraceMqInfo(tMQUserHandle* pCurrentEntry,tU8 Type,tBool bErrMem,tU32 u32Error,tCString coszName)
{
#ifdef SHORT_TRACE_OUTPUT
   char au8Buf[23 + LINUX_C_MQ_MAX_NAMELENGHT+1]={0,}; 
   tU32 u32Temp = (tU32)OSAL_ThreadWhoAmI();
   OSAL_M_INSERT_T8(&au8Buf[0],Type);
   bGetThreadNameForTID(&au8Buf[1],8,(tS32)u32Temp);
   OSAL_M_INSERT_T32(&au8Buf[9],u32Temp);
   if(pCurrentEntry)
   {  
       OSAL_M_INSERT_T32(&au8Buf[13],(tU32)pCurrentEntry->pOrigin);
       OSAL_M_INSERT_T32(&au8Buf[17],(tU32)pCurrentEntry);   
       OSAL_M_INSERT_T16(&au8Buf[21],(tU16)pCurrentEntry->enAccess);
       strncpy (&au8Buf[23],((trMqueueElement*)(pCurrentEntry->pOrigin))->szName,LINUX_C_MQ_MAX_NAMELENGHT);
   }
   else
   {  
      u32Temp = 0;
      OSAL_M_INSERT_T32(&au8Buf[13],u32Error);
      OSAL_M_INSERT_T32(&au8Buf[17],u32Error);
      OSAL_M_INSERT_T16(&au8Buf[21],(tU16)u32Temp);
      if(coszName)strncpy (&au8Buf[23],coszName,LINUX_C_MQ_MAX_NAMELENGHT);
   }
   u32Temp=strlen(&au8Buf[23]);
   LLD_vTrace(OSAL_C_TR_CLASS_SYS_MQ,TR_LEVEL_FATAL,au8Buf,u32Temp+23);
  
   if((pOsalData->bLogError)&&(bErrMem)/*(u32Error != OSAL_E_NOERROR)*/)
   {
      if(!pCurrentEntry)
      {
         vWriteToErrMem(OSAL_C_TR_CLASS_SYS_MQ,au8Buf,u32Temp+23,0);
      }
      else
      {
         if(!((trMqueueElement*)pCurrentEntry->pOrigin)->bTraceCannel)vWriteToErrMem(OSAL_C_TR_CLASS_SYS_MQ,au8Buf,u32Temp+23,0);
      }
   }
#else
   char name[TRACE_NAME_SIZE];
   tU32 u32Temp = (tU32)OSAL_ThreadWhoAmI();
   bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,(tS32)u32Temp);
    
   if(coszName == NULL)coszName ="Unknown";
   switch(Type)
   {
     case OSAL_MQ_CREATE:
          if(pCurrentEntry)
          {
             TraceString("OSALMQ Task:%s(ID:%d) Creates MQ Handle %p -> User Handle %p Access:%d MQ Name:%s",
                      name,(unsigned int)u32Temp,pCurrentEntry->pOrigin,pCurrentEntry,pCurrentEntry->enAccess,((trMqueueElement*)(pCurrentEntry->pOrigin))->szName);
             if((pOsalData->bLogError)&&(bErrMem))vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) Creates MQ Handle %p -> User Handle %p Access:%d MQ Name:%s \n",
                      name,(unsigned int)u32Temp,pCurrentEntry->pOrigin,pCurrentEntry,pCurrentEntry->enAccess,((trMqueueElement*)(pCurrentEntry->pOrigin))->szName);
          }
          else
          {
             TraceString("OSALMQ Task:%s(ID:%d) Creates MQ Handle 0x%x -> User Handle 0x%x Access:%d MQ Name:%s",
                      name,(unsigned int)u32Temp,(unsigned int)u32Error,(unsigned int)u32Error,(unsigned int)u32Error,coszName);
             if((pOsalData->bLogError)&&(bErrMem))vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) Creates MQ Handle 0x%x -> User Handle 0x%x Access:%d MQ Name:%s \n",
                      name,(unsigned int)u32Temp,(unsigned int)u32Error,(unsigned int)u32Error,(unsigned int)u32Error,coszName);
          }
         break;
     case OSAL_TSK_DELETE_MQ:
          if(pCurrentEntry)
          {
             TraceString("OSALMQ Task:%s(ID:%d) deleted MQ Name:%s",name,(unsigned int)u32Temp,((trMqueueElement*)(pCurrentEntry->pOrigin))->szName);
             if((pOsalData->bLogError)&&(bErrMem))vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) deleted MQ Name:%s \n",name,(unsigned int)u32Temp,((trMqueueElement*)(pCurrentEntry->pOrigin))->szName);
          }
          else
          {
             TraceString("OSALMQ Task:%s(ID:%d) deleted MQ Name:%s", name,(unsigned int)u32Temp,coszName);
             if((pOsalData->bLogError)&&(bErrMem))vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) deleted MQ Name:%s \n", name,(unsigned int)u32Temp,coszName);
          }
         break;
     case OSAL_TSK_CONNECT_TO_MQ:
          if(pCurrentEntry)
          {
             TraceString("OSALMQ Task:%s(ID:%d) connected to MQ Handle %p -> User Handle %p Access:%d MQ Name:%s",
                      name,(unsigned int)u32Temp,pCurrentEntry->pOrigin,(void*)pCurrentEntry,pCurrentEntry->enAccess,((trMqueueElement*)(pCurrentEntry->pOrigin))->szName);
             if((pOsalData->bLogError)&&(bErrMem))vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) connected to MQ Handle %p -> User Handle %p Access:%d MQ Name:%s \n",
                      name,(unsigned int)u32Temp,pCurrentEntry->pOrigin,(void*)pCurrentEntry,pCurrentEntry->enAccess,((trMqueueElement*)(pCurrentEntry->pOrigin))->szName);
          }
          else
          {
             TraceString("OSALMQ Task:%s(ID:%d) connected to MQ Handle 0x%x -> User Handle 0x%x Access:%d MQ Name:%s",
                      name,(unsigned int)u32Temp,(unsigned int)u32Error,(unsigned int)u32Error,(unsigned int)u32Error,coszName);
             if((pOsalData->bLogError)&&(bErrMem))vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) connected to MQ Handle 0x%x -> User Handle 0x%x Access:%d MQ Name:%s \n",
                      name,(unsigned int)u32Temp,(unsigned int)u32Error,(unsigned int)u32Error,(unsigned int)u32Error,coszName);
          }
         break;
     case OSAL_TSK_DISCONNECT_MQ:
          if(pCurrentEntry)
          {
             TraceString("OSALMQ Task:%s(ID:%d) disconnected from MQ Handle %px -> User Handle %p Access:%d MQ Name:%s",
                      name,(unsigned int)u32Temp,pCurrentEntry->pOrigin,(void*)pCurrentEntry,pCurrentEntry->enAccess,((trMqueueElement*)(pCurrentEntry->pOrigin))->szName);
             if((pOsalData->bLogError)&&(bErrMem))vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) disconnected from MQ Handle %px -> User Handle %p Access:%d MQ Name:%s \n",
                      name,(unsigned int)u32Temp,pCurrentEntry->pOrigin,(void*)pCurrentEntry,pCurrentEntry->enAccess,((trMqueueElement*)(pCurrentEntry->pOrigin))->szName);
          }
          else
          {
             TraceString("OSALMQ Task:%s(ID:%d) disconnected from MQ Handle 0x%x -> User Handle 0x%x Access:%d MQ Name:%s",
                      name,(unsigned int)u32Temp,(unsigned int)u32Error,(unsigned int)u32Error,(unsigned int)u32Error,coszName);
             if((pOsalData->bLogError)&&(bErrMem))vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) disconnected from MQ Handle 0x%x -> User Handle 0x%x Access:%d MQ Name:%s \n",
                      name,(unsigned int)u32Temp,(unsigned int)u32Error,(unsigned int)u32Error,(unsigned int)u32Error,coszName);
          }
         break;
     case OSAL_MQ_STATUS:
          if(pCurrentEntry)
          {
             TraceString("OSALMQ Task:%s(ID:%d) Message Queue Status Handle %p -> User Handle %p Access:%d MQ Name:%s",
                      name,(unsigned int)u32Temp,pCurrentEntry->pOrigin,(void*)pCurrentEntry,pCurrentEntry->enAccess,((trMqueueElement*)(pCurrentEntry->pOrigin))->szName);
             if((pOsalData->bLogError)&&(bErrMem))vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) Message Queue Status Handle %p -> User Handle %p Access:%d MQ Name:%s \n",
                      name,(unsigned int)u32Temp,pCurrentEntry->pOrigin,(void*)pCurrentEntry,pCurrentEntry->enAccess,((trMqueueElement*)(pCurrentEntry->pOrigin))->szName);
          }
          else
          {
             TraceString("OSALMQ Task:%s(ID:%d) Message Queue Status Handle 0x%x -> User Handle 0x%x Access:%d MQ Name:%s \n",
                      name,(unsigned int)u32Temp,(unsigned int)u32Error,(unsigned int)u32Error,(unsigned int)u32Error,coszName);
             if((pOsalData->bLogError)&&(bErrMem))vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) Message Queue Status Handle 0x%x -> User Handle 0x%x Access:%d MQ Name:%s \n",
                      name,(unsigned int)u32Temp,(unsigned int)u32Error,(unsigned int)u32Error,(unsigned int)u32Error,coszName);
          }
         break;
     default:
         TraceString("OSALMQ Unhandled Trace ID:%d",Type);
         if((pOsalData->bLogError)&&(bErrMem))vWritePrintfErrmem("OSALMQ Unhandled Trace ID:%d \n",Type);
         break;
   }
#endif
}


/*****************************************************************************
 *
 * FUNCTION: OSAL_s32MessageQueueCreate
 *
 * DESCRIPTION: Create a MQ with a given max number of messages, each of a
 *              given MaxLenght, with given access right
 *
 * PARAMETER:
 *
 *   u16ContextID                index to distinguish caller context
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.10.05  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
tU32 u32GenerateHandle(OSAL_tenAccess enAccess,trMqueueElement *pCurrentEntry,OSAL_tMQueueHandle* phMQ)
{
  tU32 u32ErrorCode = OSAL_E_NOERROR;
  
  /* create user handle */
  tMQUserHandle* pHandle = (tMQUserHandle *)OSAL_pvMemPoolFixSizeGetBlockOfPool(&MqMemPoolHandle);
  if(pHandle)
  {
     pHandle->enAccess = enAccess;
     pHandle->pOrigin  = pCurrentEntry;
     pHandle->PrcId = OSAL_ProcessWhoAmI();
     *phMQ = (OSAL_tMQueueHandle)pHandle;
  }
  else
  {
      u32ErrorCode = OSAL_E_NOSPACE;
  }
  return u32ErrorCode;
}


tS32 s32GetDevMqueueValue(tCString name)
{
  char cBuffer[256];
  int Ret = -1;
  
  snprintf(cBuffer,256,"/proc/sys/fs/mqueue/%s",name);
  int fd = open(cBuffer, O_RDONLY);
  if(fd != -1)
  {
    int Len = (int)read(fd,cBuffer,32);
    if(Len>0)
    {
       Ret = atoi(cBuffer);
    }
    close(fd);
  }
  return Ret;
}

tBool bIsDBusQueue(tCString coszName)
{
  tBool bRet = FALSE;
#ifdef DBUS_MQ_SUPPORT
  /* check if DBUS connection is configured */
  if((pOsalData->u32OsalMqDBus)&&(coszName))
  {
     if(strncmp(coszName,"dbus_",5) == 0)
     {
        if(pConnectDBus == NULL)
        {
            if(u32LoadSharedObject((void*)pConnectDBus,pOsalData->rLibrary[EN_SHARED_OSAL2].cLibraryNames) == OSAL_E_NOERROR)/*lint !e611 otherwise linker warning */
            {
               bRet = TRUE;
            }
        }
        else
        {
           bRet = TRUE;
        }
     }
  }
#else
  ((void)coszName);
#endif
  return bRet;
}

tS32 OSAL_s32MessageQueueCreate(tCString coszName,
                                tU32 u32MaxMessages,
                                tU32 u32MaxLength,
                                OSAL_tenAccess enAccess,
                                OSAL_tMQueueHandle* phMQ)
{

   vMqueueLiMqMapInit();
   
#ifdef DBUS_MQ_SUPPORT
   if(pOsalData->u32OsalMqDBus)
   {
      if(bIsDBusQueue(coszName))
      {
         return OSAL_s32DBusMessageQueueCreate(coszName,u32MaxMessages,u32MaxLength,enAccess,phMQ);
      }
   }
#endif
#ifdef OSAL_SHM_MQ_FOR_LINUX_MQ
   return OSAL_s32ShMemMessageQueueCreate(coszName, u32MaxMessages, u32MaxLength,enAccess,phMQ);
#else
   tS32 s32ReturnValue= OSAL_ERROR;
   tU32 u32ErrorCode=OSAL_E_NOERROR;
   trMqueueElement *pCurrentEntry = NULL;
   tU32 u32Loop = 0;
   struct rlimit limit;
   
   /* The name is not NULL */
   /* Parameter check */

   if( coszName && phMQ && (enAccess <= OSAL_EN_READWRITE) )
   {
      /* The Name is not too long */
      if( OSAL_u32StringLength(coszName) < LINUX_C_MQ_MAX_NAMELENGHT )
      {
         if(LockOsal(&pOsalData->MqueueTable.rLock) == OSAL_OK)
         {
            pCurrentEntry = tMqueueTableSearchEntryByName(coszName);
            if( pCurrentEntry == OSAL_NULL )
            {
               if((pCurrentEntry = tMqueueTableGetFreeEntry())!= OSAL_NULL )
               {
  /*                  if((pCurrentEntry->u32EntryIdx >= pOsalData->MqueueTable.u32UsedEntries)||(pCurrentEntry->u32MqueueID != LINUX_C_U32_MQUEUE_ID))
                  {
                     CheckandCorrectMqStaticValues();
                  }*/
                  /* find Index */
                  if(prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId != -1)
                  {
                     TraceString("OSALMQ  MQ(%s): Create->rLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId != -1", coszName);
                     FATAL_M_ASSERT_ALWAYS();
                  }
                  tU32 u32Retry = 0;
                  /* check if linux resource is available*/
                  char buf[LINUX_C_MQ_MAX_NAMELENGHT+1];
                  mq_attr osalmq_attr;
                  osalmq_attr.mq_maxmsg  = u32MaxMessages;
                  osalmq_attr.mq_msgsize = u32MaxLength;
                  osalmq_attr.mq_flags   = 0;
                  osalmq_attr.mq_curmsgs = 0;
                  buf[0] = '/';
                  strncpy(&buf[1],coszName,LINUX_C_MQ_MAX_NAMELENGHT);
                  //TraceString("OSALMQ MQ(%s)-> Limit msgsize_max:%d msg_max:%d",coszName,s32GetDevMqueueValue("msgsize_max"), s32GetDevMqueueValue("msg_max"));
                  do
                  {
                     s32ReturnValue = mq_open(buf, O_RDWR | O_CREAT | O_EXCL, OSAL_ACCESS_RIGTHS, &osalmq_attr);
                     if(s32ReturnValue == -1 )
                     {
                        u32ErrorCode = u32ConvertErrorCore(errno);
                        switch(errno)
                        {
                           case EEXIST:
                                if((s32ReturnValue = mq_open(buf, O_RDWR , OSAL_ACCESS_RIGTHS, NULL)) == -1)
                                {
                                   bGetThreadNameForTID(&buf[0],8, OSAL_ThreadWhoAmI());
                                   TraceString("OSALMQ MQ(%s): Create->Open by TID:%d(%s) failed Error:%d", coszName, OSAL_ThreadWhoAmI(),buf, errno);
                                   u32ErrorCode = u32ConvertErrorCore(errno);
                                   FATAL_M_ASSERT_ALWAYS();
                                }
                                else
                                {
                                   u32ErrorCode = OSAL_E_NOERROR;
//                                   vWritePrintfErrmem("MQ(%s): Create->Open by TID:%d(%s) failed Error:%d  \n", coszName, OSAL_ThreadWhoAmI(),buf, errno);
                                   TraceString("OSALMQ MQ(%s): Create->Open by TID:%d(%s) failed Error:%d", coszName, OSAL_ThreadWhoAmI(),buf, errno);
                                }
                                u32Retry += 2;
                               break;
                           case ENFILE:
#ifdef QNX
                                u32Retry += 2;
#else
                                if(getrlimit(RLIMIT_MSGQUEUE,&limit) == 0)
                                {
                                   TraceString("OSALMQ MQ(%s)-> ENFILE Limit Cur:%d Max:%d", coszName, limit.rlim_cur, limit.rlim_max);
                                   vWritePrintfErrmem("OSALMQ MQ(%s)-> ENFILE Limit Cur:%d Max:%d \n", coszName, limit.rlim_cur, limit.rlim_max);
                                }
                                else
                                {
                                   TraceString("OSALMQ MQ(%s): Create failed get RLIMIT_MSGQUEUE failed Error:%d", coszName, errno);
                                }
                                if(u32Retry == 0)
                                {
                                   u32ErrorCode = OSAL_E_NOERROR;
                                   limit.rlim_cur = pOsalData->u32MqResource;
                                   limit.rlim_max = pOsalData->u32MqResource;
                                   if(setrlimit(RLIMIT_MSGQUEUE,&limit) == -1)
                                   {
                                      TraceString("OSALMQ ENFILE Set Limit of MQ for Prc failed");
                                      vWritePrintfErrmem("OSALMQ ENFILE Set Limit of MQ for Prc failed \n");
                                   }
                                }
                                u32Retry++;
#endif								
                               break;
                           case EMFILE:
#ifdef QNX
                                u32Retry += 2;
#else
                                if(getrlimit(RLIMIT_MSGQUEUE,&limit) == 0)
                                {
                                   TraceString("OSALMQ MQ(%s)-> EMFILE Limit Cur:%d Max:%d",coszName, limit.rlim_cur, limit.rlim_max);
                                   vWritePrintfErrmem("OSALMQ MQ(%s)-> EMFILE Limit Cur:%d Max:%d \n", coszName, limit.rlim_cur, limit.rlim_max);
                               }
                                else
                                {
                                   TraceString("OSALMQ MQ(%s): Create failed get RLIMIT_MSGQUEUE failed Error:%d", coszName, errno);
                                }
                                if(u32Retry == 0)
                                {
                                   u32ErrorCode = OSAL_E_NOERROR;
                                   limit.rlim_cur = pOsalData->u32MqResource;
                                   limit.rlim_max = pOsalData->u32MqResource;
                                   if(setrlimit(RLIMIT_MSGQUEUE,&limit) == -1)
                                   {
                                      TraceString("OSALMQ EMFILE Set Limit of MQ for Prc failed");
                                      vWritePrintfErrmem("OSALMQ EMFILE Set Limit of MQ for Prc failed \n");
                                   }
                                }
                                u32Retry++;
#endif								
                               break;
                           case ENOSPC:
                                TraceString("OSALMQ MQ(%s): Create->Open failed  ENOSPC check /proc/sys/fs/mqueue/queues_max", coszName);
                                vWritePrintfErrmem("OSALMQ MQ(%s): Create->Open failed  ENOSPC check /proc/sys/fs/mqueue/queues_max\n", coszName);
                                u32Retry += 2;
                               break;
                          case EINVAL:
                                TraceString("OSALMQ MQ(%s): Create->Open failed  EINVAL mq_maxmsg:%d mq_msgsize:%d check /proc/sys/fs/mqueue", coszName,osalmq_attr.mq_maxmsg,osalmq_attr.mq_msgsize);
                                vWritePrintfErrmem("OSALMQ MQ(%s): Create->Open failed  EINVAL mq_maxmsg:%d mq_msgsize:%d check /proc/sys/fs/mqueue", coszName,osalmq_attr.mq_maxmsg,osalmq_attr.mq_msgsize);
                                TraceString("OSALMQ MQ(%s)-> EINVAL Limit msgsize_max:%d msg_max:%d",coszName,s32GetDevMqueueValue("msgsize_max"), s32GetDevMqueueValue("msg_max"));
                                vWritePrintfErrmem("OSALMQ MQ(%s)-> EINVAL Limit Cur:%d Max:%d \n", coszName,s32GetDevMqueueValue("msgsize_max"), s32GetDevMqueueValue("msg_max"));
                                u32Retry += 2;
                               break;
                          default:
                                TraceString("OSALMQ MQ(%s): Create->Open failed  Unexpected Error:%d", coszName, errno);
                                vWritePrintfErrmem("OSALMQ MQ(%s): Create->Open failed  Unexpected Error:%d \n", coszName, errno);
                                u32Retry += 2;
                                FATAL_M_ASSERT_ALWAYS();
                               break;
                        }//switch
                     }//if(s32ReturnValue == -1 )
                  }while(u32Retry == 1);
                  if(u32ErrorCode == OSAL_E_NOERROR)
                  {
                     if(s32OsalGroupId)
                     {
                        if(fchown(s32ReturnValue,(uid_t)-1,s32OsalGroupId) == -1)
                        {
                           vWritePrintfErrmem("OSALMQ MQ(%s): OSAL_s32MessageQueueCreate -> fchown error %d \n",coszName,errno);
                        }
                        if(fchmod(s32ReturnValue, OSAL_ACCESS_RIGTHS) == -1)
                        {
                           vWritePrintfErrmem("OSALMQ MQ(%s): OSAL_s32MessageQueueCreate -> fchmod error %d \n",coszName,errno);
                        }
                     }
                     pCurrentEntry->bIsUsed = TRUE;
                     pCurrentEntry->bTraceCannel = pOsalData->bMqTaceAll;
                     if(pOsalData->rMqName[u32Loop].szObjName[0] != 0)
                     {
                        for( u32Loop=0; u32Loop < MAX_NR_SUPERVISION_OBJ; u32Loop++ )
                        {		
                           if(pOsalData->rMqName[u32Loop].szObjName[0] == 0)break;
                           if(!strncmp(coszName,&pOsalData->rMqName[u32Loop].szObjName[0],strlen(&pOsalData->rMqName[u32Loop].szObjName[0])))
//                           if(!strncmp(coszName,&pOsalData->rMqName[u32Loop].szObjName[0],strlen(coszName)+1))
                           {
                              TraceString("OSALMQ Enable Trace for %s %d via %s",coszName,strlen(&pOsalData->rMqName[u32Loop].szObjName[0]),&pOsalData->rMqName[u32Loop].szObjName[0]);
                              pCurrentEntry->bTraceCannel = TRUE;
                              break;
						   }
                        }
                     }
                     pOsalData->u32MqResCount++;
                     if(pOsalData->u32MaxMqResCount < pOsalData->u32MqResCount)
                     {
                         pOsalData->u32MaxMqResCount = pOsalData->u32MqResCount;
                         if(pOsalData->u32MaxMqResCount > (pOsalData->u32MaxNrMqElements*9/10))
                         {
                            pOsalData->u32NrOfChanges++;
                         }
                     }
                     pCurrentEntry->pcallback = NULL;
                     pCurrentEntry->pcallbackarg = NULL;
                     pCurrentEntry->callbacktid = NOTATHREAD;
                     pCurrentEntry->callbackpid = NOTATHREAD;
                     pCurrentEntry->callbackpididx = NOTATHREAD;
                     pCurrentEntry->bNotify = FALSE;
#ifdef USE_EVENT_AS_CB
                     pCurrentEntry->mask = (OSAL_tEventMask) 0;
                     pCurrentEntry->enFlags = (OSAL_tenEventMaskFlag) 0;
#endif
                     pCurrentEntry->u16Type = MQ_STD;
                     pCurrentEntry->bOverflow = FALSE;
                     pCurrentEntry->PID = MqPid;
                     /* export the handle */
                     if((u32ErrorCode = u32GenerateHandle(enAccess, pCurrentEntry, phMQ)) != OSAL_E_NOERROR)
                     {
                        TraceString("OSALMQ MQ(%s): u32GenerateHandle Error:%d", coszName, u32ErrorCode);
                        FATAL_M_ASSERT_ALWAYS();
                     }

                     prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId = (tU32)s32ReturnValue;
                     prLiMqMap[pCurrentEntry->u32EntryIdx].PrcId = MqPid;
                     prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqIdRef = (tU32)(MqPid + s32ReturnValue);
                     prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt = 1;
                     (void)OSAL_szStringNCopy((tString)pCurrentEntry->szName,
                                               coszName,
                                               strlen(coszName));
                     pCurrentEntry->MaxMessages   = u32MaxMessages;
                     pCurrentEntry->MaxLength     = u32MaxLength;
                     pCurrentEntry->ActualMessage = 0;
                     /* init the other MQ fields */
                     pCurrentEntry->bToDelete = FALSE;
                     pCurrentEntry->u16OpenCounter = 1;
                     pCurrentEntry->RecTsk = 0;
                     pCurrentEntry->RecPrc = 0;
                     pCurrentEntry->s32LiHandle = s32ReturnValue;
                     pCurrentEntry->u32IdxMPT    = 0;
                     /* prepare message processing mesurements 
                        filter out CSM queues because their usage is trgggered by CAN callbacks */
                     if(!strncmp("CSMALU_MQ",coszName,strlen("CSMALU_MQ")))
                     {
                        pCurrentEntry->u32MessagePrcessing = MEASUREMENT_SWITCHOFF;
                     }
                     else
                     {
                        pCurrentEntry->u32MessagePrcessing = MSG_PROCESSING_INACTIVE;
                     }
                     for( u32Loop=0; u32Loop < OSAL_C_MSG_TIME_ARR_SIZE; u32Loop++ )
                     {
                        pCurrentEntry->u32MsgPrcTim[u32Loop]   = 0;
                        pCurrentEntry->u32MsgTimStamp[u32Loop] = 0;
                     }
                     pCurrentEntry->u32IdxLMPT    = 0;
                     for( u32Loop=0; u32Loop < OSAL_C_LONG_MSG_TIME_ARR_SIZE; u32Loop++ )
                     {
                        pCurrentEntry->u32LongMsgPrcTim[u32Loop]   = 0;
                        pCurrentEntry->u32LongMsgTimStamp[u32Loop] = 0;
                     }
                     pCurrentEntry->u16WaitCounter = 0;
                     pCurrentEntry->bBlocked = FALSE;
                     pCurrentEntry->u32Marker = 0;
                     if(pOsalData->u32ClkMonotonic & 0x00000001)
                     {
                        pCurrentEntry->u32WaitTime = 0;
                     }
                     else
                     { 
                        pCurrentEntry->u32WaitTime = 0xffffffff;
                     }
                  }//if(u32ErrorCode == OSAL_E_NOERROR)
               }
               else
               {
                  u32ErrorCode = OSAL_E_NOSPACE;
                  vWritePrintfErrmem("OSALMQ From %d OSAL Message Queues %d currently in use (Max used:%d)",(int)pOsalData->u32MaxNrMqElements,(int)pOsalData->u32MqResCount,(int)pOsalData->u32MaxMqResCount);
                  NORMAL_M_ASSERT_ALWAYS();
               }//if((pCurrentEntry = tMqueueTableGetFreeEntry())!= OSAL_NULL )
            }
            else
            {
               u32ErrorCode = OSAL_E_ALREADYEXISTS;
            }//if( pCurrentEntry == OSAL_NULL )
            /* UnLock the table */
            UnLockOsal(&pOsalData->MqueueTable.rLock);
         }
      }
      else
      {
         u32ErrorCode = OSAL_E_NAMETOOLONG;
      }
   }
   else
   {
      u32ErrorCode = OSAL_E_INVALIDVALUE;
   }

   if((u32ErrorCode != OSAL_E_NOERROR )||(phMQ == NULL))
   {
      s32ReturnValue = OSAL_ERROR;
      if(u32ErrorCode == OSAL_E_ALREADYEXISTS)
      {
        if((pCurrentEntry&&(pCurrentEntry->bTraceCannel == TRUE))||(u32OsalSTrace & 0x00000020))
        {
           vTraceMqInfo(NULL,OSAL_MQ_CREATE,FALSE,u32ErrorCode,coszName);
        }
      }
      else
      {
           vTraceMqInfo(NULL,OSAL_MQ_CREATE,TRUE,u32ErrorCode,coszName);
      }
      vSetErrorCode(OSAL_C_THREAD_ID_SELF, u32ErrorCode);
   }
   else
   {
      if((pCurrentEntry->bTraceCannel == TRUE)||(u32OsalSTrace & 0x00000020)) /*lint !e613 pCurrentEntry already checked */
      {
         vTraceMqInfo((tMQUserHandle*)*phMQ,OSAL_MQ_CREATE,FALSE,u32ErrorCode,coszName);
         if(bTraceHandle)
         {
             TraceString("OSALMQ Cre MQ:%s OSAL Handle:0x%x Linux Handle:%d Offset:0x%x Pid:%d Count:%d",coszName,phMQ,
                      prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId,/*lint !e613 pointer already checked */
                      GetMemPoolOffset(&MqMemPoolHandle, (uintptr_t*)phMQ),
                      prLiMqMap[pCurrentEntry->u32EntryIdx].PrcId,/*lint !e613 pointer already checked */
                      prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt);/*lint !e613 pointer already checked */
         }
      }
      s32ReturnValue= OSAL_OK;
   }
   return( s32ReturnValue );
#endif
}

#ifndef OSAL_SHM_MQ_FOR_LINUX_MQ
tU32 u32GetValidMqHandle(trMqueueElement *pCurrentEntry)
 {
   tU32 u32ErrorCode = OSAL_E_NOERROR;
   tS32 s32Val = 0;
   
   if((prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId >= 0)
   &&(prLiMqMap[pCurrentEntry->u32EntryIdx].PrcId == MqPid))
   {
     prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt++;
   }
   else
   {
       char buf[LINUX_C_MQ_MAX_NAMELENGHT+1];
       OSAL_szStringCopy(&buf[0], "/");
       OSAL_szStringCopy(&buf[1], pCurrentEntry->szName);
       s32Val = mq_open(buf, O_RDWR , OSAL_ACCESS_RIGTHS, NULL);
       if (s32Val == -1)
       {
         u32ErrorCode = u32ConvertErrorCore(errno);
         TraceString("OSALMQ mq_open %s failed with %d",pCurrentEntry->szName,errno);
       }
       else
       {
           prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId  = (tU32)s32Val;
           prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqIdRef  = (tU32)(s32Val + MqPid);
           prLiMqMap[pCurrentEntry->u32EntryIdx].PrcId = MqPid;
           prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt = 1;
       }
   }
   return u32ErrorCode;
}
#endif

/*****************************************************************************
 *
 * FUNCTION: OSAL_s32MessageQueueOpen
 *
 * DESCRIPTION: this function opens an existing OSAL event
 *
 * PARAMETER:   coszName (I)
 *                 event name to create.
 *              phEvent (->O)
 *                 pointer to the event handle.
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.10.05  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
tS32 OSAL_s32MessageQueueOpen (tCString coszName,
                               OSAL_tenAccess enAccess,
                               OSAL_tMQueueHandle* phMQ)
{
   /* Make handle invalid */
   if(phMQ)
   {
        *phMQ=OSAL_C_INVALID_HANDLE;
   }

   vMqueueLiMqMapInit();
#ifdef DBUS_MQ_SUPPORT
   if(pOsalData->u32OsalMqDBus)
   {
      if(bIsDBusQueue(coszName))
      {
          return OSAL_s32DBusMessageQueueOpen(coszName,enAccess, phMQ);
      }
   }   
#endif
#ifdef OSAL_SHM_MQ_FOR_LINUX_MQ
   return OSAL_s32ShMemMessageQueueOpen(coszName,enAccess,phMQ);
#else
   tS32 s32ReturnValue=OSAL_ERROR;
   tU32 u32ErrorCode=OSAL_E_NOERROR;
   trMqueueElement *pCurrentEntry = NULL;
   if( coszName && phMQ && (enAccess <= OSAL_EN_READWRITE) )
   {
      if(LockOsal(&pOsalData->MqueueTable.rLock) == OSAL_OK)
      {
         pCurrentEntry = tMqueueTableSearchEntryByName(coszName);
         if(pCurrentEntry != NULL)
         {
/*            if((pCurrentEntry->u32EntryIdx >= pOsalData->MqueueTable.u32UsedEntries)||(pCurrentEntry->u32MqueueID != LINUX_C_U32_MQUEUE_ID))
            {
               CheckandCorrectMqStaticValues();
            }*/
            if(pCurrentEntry->bToDelete)
            {  
//               u32ErrorCode = OSAL_E_NOPERMISSION;
                vWritePrintfErrmem("OSALMQ TID:%d MQ Open for %s Delete Marked Queue OpenCnt:%d \n",
                                   OSAL_ThreadWhoAmI(),pCurrentEntry->szName,pCurrentEntry->u16OpenCounter);
            }//if( pCurrentEntry!= OSAL_NULL )
          //  else
            {
               u32ErrorCode = u32GetValidMqHandle(pCurrentEntry);
               if(u32ErrorCode == OSAL_E_NOERROR)
               {
                  pCurrentEntry->u16OpenCounter++;
               }
               else
               {
                   NORMAL_M_ASSERT_ALWAYS();
               }
               u32ErrorCode = u32GenerateHandle(enAccess,pCurrentEntry,phMQ);
            }
         }// if(pCurrentEntry != NULL)
         else
         {
            u32ErrorCode = OSAL_E_DOESNOTEXIST;
         }//if( pCurrentEntry!= OSAL_NULL )
      }
      UnLockOsal(&pOsalData->MqueueTable.rLock);
   }
   else
   {
      u32ErrorCode = OSAL_E_INVALIDVALUE;
   }

   if((u32ErrorCode != OSAL_E_NOERROR )||(phMQ == NULL))
   {
      if(u32ErrorCode == OSAL_E_DOESNOTEXIST)
      {
        if(u32OsalSTrace & 0x00000020)
        {
           vTraceMqInfo(NULL,OSAL_TSK_CONNECT_TO_MQ,FALSE,u32ErrorCode,coszName);
        }
      }
      else
      {
         vTraceMqInfo(NULL,OSAL_TSK_CONNECT_TO_MQ,TRUE,u32ErrorCode,coszName);
      }
      vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
   }
   else
   {
      if((pCurrentEntry->bTraceCannel == TRUE)||(u32OsalSTrace & 0x00000020))/*lint !e613 pointer already checked */
      {
         vTraceMqInfo((tMQUserHandle*)*phMQ,OSAL_TSK_CONNECT_TO_MQ,FALSE,u32ErrorCode,coszName);
         if(bTraceHandle)
         {
             TraceString("OSALMQ Opn MQ:%s OSAL Handle:0x%x Linux Handle:%d Offset:0x%x Pid:%d Count:%d",coszName,phMQ,
                      prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId,/*lint !e613 pointer already checked */
                      GetMemPoolOffset(&MqMemPoolHandle, (uintptr_t*)phMQ),
                      prLiMqMap[pCurrentEntry->u32EntryIdx].PrcId,/*lint !e613 pointer already checked */
                      prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt);/*lint !e613 pointer already checked */
         }
      }
     s32ReturnValue = OSAL_OK;
   }
   return( s32ReturnValue );
#endif
}




/*****************************************************************************
 *
 * FUNCTION: OSAL_s32MessageQueueClose
 *
 * DESCRIPTION: this function creates an OSAL event.
 *
 * PARAMETER:   coszName (I)
 *                 event name to create. 
 *              phEvent (->O)
 *                 pointer to the event handle.
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.10.05  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
#ifndef OSAL_SHM_MQ_FOR_LINUX_MQ
 tU32 u32CheckForDelete(trMqueueElement *pCurrentEntry)
{
   tU32 u32ErrorCode = OSAL_E_NOERROR;
   if((prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt == 0)
    &&(!pCurrentEntry->u16OpenCounter)
    &&(pCurrentEntry->bToDelete ))
      {
          pOsalData->u32MqResCount--;
          char buf[LINUX_C_MQ_MAX_NAMELENGHT+1];
          buf[0] = '/';
          strncpy(&buf[1],pCurrentEntry->szName,LINUX_C_MQ_MAX_NAMELENGHT);
          if(pCurrentEntry->u16Type == MQ_STD)
          {
             if(mq_unlink(buf) == -1)
             {
                // chgrp("eco_osal","/dev/mqueue");
                TraceString("OSALMQ mq_unlink %d failed with errno %d",buf,errno);
                u32ErrorCode = u32ConvertErrorCore(errno);
             }
          }
      }
      else
      {
         u32ErrorCode = OSAL_E_BUSY;
      }//if(pCurrentEntry->uTypeMqInfo.pMqueue->u32LiPrcOpnCnt[u32count] == 0)
    return u32ErrorCode;
}
#endif

static void vPrintMqHandleIssue(OSAL_tMQueueHandle hMQueue)
{
   if(pOsalData->bLogError)
   {
      vWritePrintfErrmem("OSALMQ memstart:%p memend:%p Handle:%p \n", MqMemPoolHandle.u32memstart,MqMemPoolHandle.u32memend,hMQueue);
      NORMAL_M_ASSERT_ALWAYS();
   }
   TraceString("OSALMQ memstart:%p memend:%p Handle:%p", MqMemPoolHandle.u32memstart,MqMemPoolHandle.u32memend,hMQueue);
}

tU32 u32CheckMqHandle(OSAL_tMQueueHandle hMQueue)
{
   if((hMQueue <= MqMemPoolHandle.u32memstart)||(hMQueue >= MqMemPoolHandle.u32memend))
   {
      if(MqMemPoolHandle.pMem)
      {
         trOSAL_MPF* pPtr = (trOSAL_MPF*)MqMemPoolHandle.pMem;
         if(pPtr->u32ErrCnt == 0)
         {
           vPrintMqHandleIssue(hMQueue);
           return OSAL_E_INVALIDVALUE;
         }
         else
         {
            /* possible that Handle Memory was allocated via malloc */
            if(( hMQueue == 0)||((tU32)hMQueue == OSAL_C_INVALID_HANDLE))
            {
               vPrintMqHandleIssue(hMQueue);
               return OSAL_E_INVALIDVALUE;
            }
         }
      }
      else
      {
         vPrintMqHandleIssue(hMQueue);
         return OSAL_E_INVALIDVALUE;
      }
   }

   trMqueueElement* pCurrent = ((tMQUserHandle*)hMQueue)->pOrigin;
   if((pCurrent)&&(pCurrent->u32EntryIdx < pOsalData->MqueueTable.u32UsedEntries))
   {
      if((pCurrent != &pMsgQEl[pCurrent->u32EntryIdx])||(pCurrent->u32MqueueID != LINUX_C_U32_MQUEUE_ID))
      { 
         TraceString("OSALMQ Possible memory writer in MQ control block");
         return OSAL_E_BADFILEDESCRIPTOR;
      }
   }
   else
   {
      TraceString("OSALMQ Possible memory writer in MQ handle pool");
      return OSAL_E_BADFILEDESCRIPTOR;
   }
   return OSAL_E_NOERROR;
}


tS32 OSAL_s32MessageQueueClose (OSAL_tMQueueHandle phMQueue)
{
#if !defined OSAL_SHM_MQ_FOR_LINUX_MQ || defined DBUS_MQ_SUPPORT
  tU32 u32ErrorCode   = u32CheckMqHandle(phMQueue);;
#endif
#ifdef DBUS_MQ_SUPPORT
  if(pOsalData->u32OsalMqDBus)
  {
     if(u32ErrorCode == OSAL_E_NOERROR)
     {
        if(((trMqueueElement*)((tMQUserHandle*)phMQueue)->pOrigin)->u16Type == MQ_DBUS )
        {
           return OSAL_s32DBusMessageQueueClose(phMQueue);
        }
     }
  }
#endif

#ifdef OSAL_SHM_MQ_FOR_LINUX_MQ
  return OSAL_s32ShMemMessageQueueClose(phMQueue);
#else
  tS32 s32ReturnValue = OSAL_ERROR;
  trMqueueElement *pCurrentEntry = NULL;
    
  if(u32ErrorCode == OSAL_E_NOERROR)
  {
     if(((tMQUserHandle*)phMQueue)->PrcId != OSAL_ProcessWhoAmI())
     {
        TraceString("Process %d try to close handle from process %d",OSAL_ProcessWhoAmI(),((tMQUserHandle*)phMQueue)->PrcId);
        vWritePrintfErrmem("Process %d try to close handle from process %d",OSAL_ProcessWhoAmI(),((tMQUserHandle*)phMQueue)->PrcId);
		u32ErrorCode = OSAL_E_BADFILEDESCRIPTOR;
     }
     else
     {
        pCurrentEntry = ((tMQUserHandle*)phMQueue)->pOrigin;/*lint !e613 *//* pointer already checked*/ 
     }
  }
  if(pCurrentEntry)
  {
     if(LockOsal(&pOsalData->MqueueTable.rLock) == OSAL_OK)
     {
/*         if((pCurrentEntry->u32EntryIdx >= pOsalData->MqueueTable.u32UsedEntries)||(pCurrentEntry->u32MqueueID != LINUX_C_U32_MQUEUE_ID))
           {
               CheckandCorrectMqStaticValues();
           }*/
           if(pCurrentEntry->u16OpenCounter > 0 )
           {
               pCurrentEntry->u16OpenCounter--;
               if(MqPid != (tS32)prLiMqMap[pCurrentEntry->u32EntryIdx].PrcId)
               {
                  u32ErrorCode = OSAL_E_DOESNOTEXIST;
               }
               if(prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt > 0)
               {
                     prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt--;
               }
               if(prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt == 0)
               {
                  if(mq_close(prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId) == -1)
                  {
                     u32ErrorCode = u32ConvertErrorCore(errno);
                  }
                  prLiMqMap[pCurrentEntry->u32EntryIdx].PrcId = 0;
                  prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt = 0;
                  prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId  = -1;
                  prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqIdRef  = -1;
               }
               if((pCurrentEntry->bTraceCannel == TRUE)||(u32OsalSTrace & 0x00000020))
               {
                  vTraceMqInfo((tMQUserHandle*)phMQueue,OSAL_TSK_DISCONNECT_MQ,FALSE,u32ErrorCode,pCurrentEntry->szName);
                  if(bTraceHandle)
                  { 
                     TraceString("OSALMQ Del MQ:%s OSAL Handle:0x%x Linux Handle:%d Offset:0x%x Pid:%d Count:%d",pCurrentEntry->szName,&phMQueue,
                     prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId,
                     GetMemPoolOffset(&MqMemPoolHandle, (uintptr_t*)&phMQueue),
                     prLiMqMap[pCurrentEntry->u32EntryIdx].PrcId,
                     prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt);
                  }
               }
               if( !pCurrentEntry->u16OpenCounter && pCurrentEntry->bToDelete )
               {
                  u32ErrorCode = u32CheckForDelete(pCurrentEntry);
                  if(u32ErrorCode == OSAL_E_NOERROR)
                  {
                     vResetEntry(pCurrentEntry->u32EntryIdx);
                  }//if(u32ErrorCode == OSAL_E_NOERROR)
               }//if( !pCurrentEntry->u16OpenCounter && pCurrentEntry->bToDelete )
#ifdef USE_EVENT_AS_CB
               if(((tMQUserHandle*)phMQueue)->hNotifyEvent != 0)
               {
                  if(OSAL_s32EventClose(((tMQUserHandle*)phMQueue)->hNotifyEvent) != OSAL_OK)
                  {
                     NORMAL_M_ASSERT_ALWAYS();
                  }
                  ((tMQUserHandle*)phMQueue)->hNotifyEvent = 0;
               }
#endif
               /*destroy magic*/
               ((tMQUserHandle*)phMQueue)->pOrigin = NULL;/*lint !e613 *//* pointer already checked*/ 
               ((tMQUserHandle*)phMQueue)->PrcId = 0;
               if(OSAL_s32MemPoolFixSizeRelBlockOfPool(&MqMemPoolHandle,(void*)phMQueue) == OSAL_ERROR)
               {    NORMAL_M_ASSERT_ALWAYS();  }
               s32ReturnValue = OSAL_OK;
            }
            else
            {
               TraceString("OSALMQ Unexpected Open counter 0 for MQ %s",pCurrentEntry->szName);
               u32ErrorCode = OSAL_E_UNKNOWN;
            }
        UnLockOsal(&pOsalData->MqueueTable.rLock);
     }
  }
  else
  {
     if(u32ErrorCode == OSAL_E_NOERROR)u32ErrorCode = OSAL_E_DOESNOTEXIST; 
  }

  if( u32ErrorCode != OSAL_E_NOERROR )
  {
     tCString szName = "";
     if(pCurrentEntry)
     {
        szName = pCurrentEntry->szName;
     }
      vTraceMqInfo(NULL,OSAL_TSK_DISCONNECT_MQ,TRUE,u32ErrorCode,szName);
      vSetErrorCode( OSAL_C_THREAD_ID_SELF,u32ErrorCode);
  }
  return( s32ReturnValue );
#endif
}

/*****************************************************************************
 *
 * FUNCTION: OSAL_s32MessageQueueDelete
 *
 * DESCRIPTION: this function destroys an OSAL event.
 *
 * PARAMETER:   coszName (I)
 *                 event name to create.
 *              phEvent (->O)
 *                 pointer to the event handle.
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.10.05  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
tS32 OSAL_s32MessageQueueDelete(tCString coszName)
{
#ifdef DBUS_MQ_SUPPORT
    if(pOsalData->u32OsalMqDBus)
    {
       if(bIsDBusQueue(coszName))
       {
          return OSAL_s32DBusMessageQueueDelete(coszName);
       }
    }
#endif
#ifdef OSAL_SHM_MQ_FOR_LINUX_MQ
   return OSAL_s32ShMemMessageQueueDelete(coszName);
#else
  tS32 s32ReturnValue=OSAL_ERROR;
  tU32 u32ErrorCode=OSAL_E_NOERROR;
  trMqueueElement *pCurrentEntry = NULL;
  if( coszName )
  {
    pCurrentEntry = tMqueueTableSearchEntryByName(coszName);
    if(pCurrentEntry)
    {
/*       if((pCurrentEntry->u32EntryIdx >= pOsalData->MqueueTable.u32UsedEntries)||(pCurrentEntry->u32MqueueID != LINUX_C_U32_MQUEUE_ID))
       {
          CheckandCorrectMqStaticValues();
       }*/
       if(LockOsal(&pOsalData->MqueueTable.rLock) == OSAL_OK)
       {
           pCurrentEntry->bToDelete = TRUE;
           if(pCurrentEntry->u16OpenCounter == 0)
           {
              u32ErrorCode = u32CheckForDelete(pCurrentEntry);
              if((pCurrentEntry->bTraceCannel == TRUE)||(u32OsalSTrace & 0x00000020))/*lint !e613 pointer already checked */
              {
                 vTraceMqInfo(NULL,OSAL_TSK_DELETE_MQ,FALSE,u32ErrorCode,coszName);
                 if(bTraceHandle)
                 {
                    TraceString("OSALMQ Del MQ:%s OSAL Handle:0x%x Linux Handle:%d Offset:0x%x Pid:%d Count:%d",coszName,0,
                    prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId,/*lint !e613 pointer already checked */
                    GetMemPoolOffset(&MqMemPoolHandle, 0),
                    prLiMqMap[pCurrentEntry->u32EntryIdx].PrcId,/*lint !e613 pointer already checked */
                    prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt);/*lint !e613 pointer already checked */
                 }
              }
              vResetEntry(pCurrentEntry->u32EntryIdx);
           }
           UnLockOsal(&pOsalData->MqueueTable.rLock);
           s32ReturnValue = OSAL_OK;
       }
    }
    else
    {
       u32ErrorCode = OSAL_E_DOESNOTEXIST;
    }
  }
  else
  {
       u32ErrorCode = OSAL_E_INVALIDVALUE;
  }

  if( u32ErrorCode != OSAL_E_NOERROR )
  {
      vTraceMqInfo(NULL,OSAL_TSK_DELETE_MQ,TRUE,u32ErrorCode,coszName);
      vSetErrorCode(OSAL_C_THREAD_ID_SELF, u32ErrorCode);
  }
  return( s32ReturnValue );
#endif
}

/*****************************************************************************
 *
 * FUNCTION: vExecuteCallback
 *
 * DESCRIPTION: Exeecute callba.
 *
 * PARAMETER:   trMqueueElement* pointer to message queue element
 *
 * ERRORS:      OSAL_E_NOPERMISSION  access denied for callback code
 *
 * RETURNVALUE: u32Ret
 *                 it is the function return value:
 *                 - OSAL_E_NOERROR if everything goes right;
 *                 - OSAL_E_NOPERMISSION otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.12.07  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
void vExecuteCallback(void* pvArg)
{
  trMqueueElement* pCurrentEntry = (trMqueueElement*)pvArg;
  tU32 u32ErrorCode = OSAL_E_NOERROR;
  if((pCurrentEntry)
   &&(pCurrentEntry->u32MqueueID == LINUX_C_U32_MQUEUE_ID)
   &&(pCurrentEntry->bNotify == TRUE))
  {
#ifdef MQ_NOTIFY_VIA_OSAL
#else
     struct sigevent s = { 0 };
     s.sigev_notify = SIGEV_THREAD;
     s.sigev_notify_function = (void (*)(sigval_t))vExecuteCallback;
     s.sigev_notify_attributes = NULL;
     s.sigev_value.sival_ptr = pCurrentEntry;
     if(mq_notify((mqd_t)rLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId, &s) == -1)
     {
       //  FATAL_M_ASSERT_ALWAYS();
         u32ErrorCode = u32ConvertErrorCore(errno);
     }
#endif

     pCurrentEntry->pcallback(pCurrentEntry->pcallbackarg);

     if((pCurrentEntry->bTraceCannel == TRUE)
#ifndef MQ_NOTIFY_VIA_OSAL
         ||(u32ErrorCode != OSAL_E_NOERROR)
#endif
         )
     {
#ifdef SHORT_TRACE_OUTPUT
        char au8Buf[22 + LINUX_C_MQ_MAX_NAMELENGHT];
        tU32 Temp = (tU32)OSAL_ThreadWhoAmI();
        OSAL_M_INSERT_T8(&au8Buf[0],(tU8)OSAL_MQ_CALLBACK);
#ifdef MQ_NOTIFY_VIA_OSAL
        bGetThreadNameForTID(&au8Buf[1],8,OSAL_ThreadWhoAmI());
#else
        memcpy(&au8Buf[1],"LISYSTEM",8);
#endif
        OSAL_M_INSERT_T32(&au8Buf[9], Temp);
        OSAL_M_INSERT_T32(&au8Buf[13],(tU32)vExecuteCallback);
        OSAL_M_INSERT_T32(&au8Buf[17], u32ErrorCode);
        Temp = strlen(pCurrentEntry->szName);
        memcpy (&au8Buf[21],pCurrentEntry->szName,Temp);
        LLD_vTrace(OSAL_C_TR_CLASS_SYS_MQ, TR_LEVEL_DATA,au8Buf,21+Temp);
#else
        tU32 Temp = (tU32)OSAL_ThreadWhoAmI();
        char name[TRACE_NAME_SIZE];
        bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,OSAL_ThreadWhoAmI());
        TraceString("OSALMQ  Task:%s(ID:%d) Executes callback %p Error:0x%x for MQ:%s",
                  name,(unsigned int)Temp,(void*)vExecuteCallback,(unsigned int)u32ErrorCode,pCurrentEntry->szName);
#endif
     }
  }
}

tS32 u32GetSigRtMinId(OSAL_tProcessID PrcId)
{
   char Buffer[64];
   int fd;
   char* pTmp;
   tS32 s32Ret = -1;
   snprintf(Buffer,64,"%s/OSAL/Processes/%d",VOLATILE_DIR,(int)PrcId);
   if((fd = open(Buffer, O_RDONLY|O_CLOEXEC,OSAL_ACCESS_RIGTHS)) != -1)
   {
      int Len = (int)read(fd,Buffer,64);
      if(Len > 0)
      {
         pTmp = strstr(Buffer,":");
         if(pTmp)
         {
            pTmp++;
            s32Ret = atoi(pTmp);
         }
      }
      close(fd);
   }
   return s32Ret;
}


void vTriggerSigRtMin(OSAL_tProcessID ReqPid, char* Name)
{
   int i;
   
   if((i = u32GetSigRtMinId(ReqPid)) == -1)
   {
      i = SIG_BACKTRACE;
   }
   /* get callstack of receiver */
   if(pOsalData->u32DebugSignal != 0xffffffff)
   {
      if(kill((int)ReqPid,i) == -1)
      {   
         if(errno == EPERM)
         {
            vWritePrintfErrmem("%s Do Kill via OSAL Term Task due permissions \n",Name);
            vSendCallStackGenFromTerm(ReqPid,MBX_GEN_CS_USR2);
         }
         else
         {
            vWritePrintfErrmem("%s Kill SIG_BACKTRACE failed Error:%d \n",Name,errno);
         }
      }
   }
}

static void vReactOnMqOverflow(trMqueueElement* pCurrentEntry)
{
  tU32 u32Val1[OSAL_C_MSG_TIME_ARR_SIZE];
  tU32 u32Val2[OSAL_C_MSG_TIME_ARR_SIZE];
//  tU32 u32Val3[OSAL_C_MSG_TIME_ARR_SIZE];
  int i;
  char NameS[16];
  char NameR[16];
  tU32 u32CurrentIdx;

  if(pCurrentEntry->u32Marker == 0)
  {
     tU32 u32OverflowTime = 0;
     if(pCurrentEntry->u32MessagePrcessing != MEASUREMENT_SWITCHOFF)
     {
        u32OverflowTime = pOsalData->u32Seconds - pCurrentEntry->u32MsgTimStamp[pCurrentEntry->u32IdxMPT];
     }
     u32CurrentIdx = pCurrentEntry->u32IdxMPT;
     pCurrentEntry->u32Marker++;
     NameS[15] = 0;
     NameR[15] = 0;
     bGetThreadNameForTID(&NameS[0], 16,OSAL_ThreadWhoAmI());
     bGetThreadNameForTID(&NameR[0], 16,pCurrentEntry->RecTsk);
     /* format string and write to errmem */
     vWritePrintfErrmem("Message Queue %s overflow after %d Sec Send Task:%s Receiver Task:%s blocked for %d Sec \n",
                        pCurrentEntry->szName,
                        pOsalData->u32Seconds,
                        NameS,
                        NameR,
                        u32OverflowTime);
     if((pCurrentEntry->bOverflow == FALSE)&& (pCurrentEntry->u32MessagePrcessing != MEASUREMENT_SWITCHOFF))
     {
        /* prepare right sequence */
        for(i= 0;i<OSAL_C_MSG_TIME_ARR_SIZE;i++)
        {
           u32CurrentIdx++;
           if(u32CurrentIdx == OSAL_C_MSG_TIME_ARR_SIZE)
           {
              u32CurrentIdx = 0;
           } 
           u32Val1[i] = pCurrentEntry->u32MsgTimStamp[u32CurrentIdx];
           u32Val2[i] = pCurrentEntry->u32MsgPrcTim[u32CurrentIdx];
//           u32Val3[i] = pCurrentEntry->u32MsgCount[u32CurrentIdx];
        }
        vWritePrintfErrmem("Last Msg Processing times  Start:%d -> %d sec ,Start:%d -> %d sec,Start:%d -> %d sec,Start:%d -> %d sec,Start:%d -> %d sec \n",
                           u32Val1[0],u32Val2[0],u32Val1[1],u32Val2[1],u32Val1[2],u32Val2[2],u32Val1[3],u32Val2[3],u32Val1[4],u32Val2[4]);
        vWritePrintfErrmem("Last Msg Processing times  Start:%d -> %d sec ,Start:%d -> %d sec,Start:%d -> %d sec,Start:%d -> %d sec,Start:%d -> %d sec \n",
                           u32Val1[5],u32Val2[5],u32Val1[6],u32Val2[6],u32Val1[7],u32Val2[7],u32Val1[8],u32Val2[8],u32Val1[9],u32Val2[9]);

        for(i= 0;i<OSAL_C_LONG_MSG_TIME_ARR_SIZE;i++)
        {
           u32CurrentIdx = pCurrentEntry->u32IdxLMPT;
           u32CurrentIdx++;
           if(u32CurrentIdx == OSAL_C_LONG_MSG_TIME_ARR_SIZE)
           {
              u32CurrentIdx = 0;
           }
           u32Val1[i] = pCurrentEntry->u32LongMsgTimStamp[u32CurrentIdx];
           u32Val2[i] = pCurrentEntry->u32LongMsgPrcTim[u32CurrentIdx];
        }
        vWritePrintfErrmem("Long Msg Processing times  Start:%d -> %d sec ,Start:%d -> %d sec,Start:%d -> %d sec ,Start:%d -> %d sec,Start:%d -> %d sec \n",
                           u32Val1[5],u32Val2[5],u32Val1[6],u32Val2[6],u32Val1[7],u32Val2[7],u32Val1[8],u32Val2[8],u32Val1[9],u32Val2[9]);

        if(pOsalData->OverflowPid != 0)
        {
           vWritePrintfErrmem("MQ Handle MQ overflow simultanously");
        }
        else
        {
           pOsalData->OverflowPid = pCurrentEntry->RecPrc;
           pOsalData->OverflowTid = pCurrentEntry->RecTsk;
        }

        if(pCurrentEntry->RecPrc != 0)
        {
           vTriggerSigRtMin(pCurrentEntry->RecPrc,pCurrentEntry->szName);
        }
        pCurrentEntry->bOverflow = TRUE;
     }
  }
  /* to ensure callstack generation completed */
  OSAL_s32ThreadWait(1000);
  pCurrentEntry->u32Marker = 0;
}

void vGenerateFdListFromTerm(OSAL_tProcessID PID)
{
   OSAL_tMQueueHandle  hTermMq = OSAL_C_INVALID_HANDLE;
   tOsalMbxMsg rMsg;
   rMsg.rOsalMsg.Cmd  = MBX_TERM_FDL;
   rMsg.rOsalMsg.ID   = PID;
   if(OSAL_s32MessageQueueOpen(OSAL_TERM_LI_MQ,OSAL_EN_WRITEONLY,&hTermMq) == OSAL_OK)
   {
     // TraceString("OSAL_s32MessageQueueOpen OSAL_TERM_LI_MQ");
      if(OSAL_s32MessageQueuePost(hTermMq,(tPCU8)&rMsg,sizeof(tOsalMbxMsg),0) == OSAL_OK)
      {
         OSAL_s32ThreadWait(0);
      }
      OSAL_s32MessageQueueClose(hTermMq);
   }
}


void vRestoreMq(trMqueueElement* pCurrentEntry)
{
   if(pCurrentEntry->u32BFD_Count == 3)
   {
      NORMAL_M_ASSERT_ALWAYS();
      OSAL_vProcessExit();
   }
   if(pCurrentEntry->u32BFD_Count < 3)
   {
      if(prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId != (prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqIdRef-MqPid))
      {
          TraceString("OSALMQ %s(%d) TID:%d Restore Linux MQ %s -> Descriptor %d was invalid new %d Index:%d Overwritten Structure",
                      commandline,
                      MqPid,
                      OSAL_ThreadWhoAmI(),
                      pCurrentEntry->szName,
                      prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId, 
                      prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqIdRef-MqPid,
                      pCurrentEntry->u32EntryIdx);
         vWritePrintfErrmem("OSALMQ %s(%d) TID:%d Restore Linux MQ %s -> Descriptor %d was invalid new %d Index:%d Overwritten Structure \n",
                      commandline,
                      MqPid,
                      OSAL_ThreadWhoAmI(),
                      pCurrentEntry->szName,
                      prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId, 
                      prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqIdRef-MqPid,
                      pCurrentEntry->u32EntryIdx);   
         prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId = prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqIdRef-(tU32)MqPid;
      }
      else
      {
         TraceString("OSALMQ TID:%d Cannot Restore Linux MQ %s -> Descriptor %d was invalid new %d Index:%d",
                      OSAL_ThreadWhoAmI(),
                      pCurrentEntry->szName,
                      prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId, 
                      prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqIdRef-(tU32)MqPid,
                      pCurrentEntry->u32EntryIdx);
         vWritePrintfErrmem("OSALMQ TID:%d Cannot Restore Linux MQ %s -> Descriptor %d was invalid new %d Index:%d \n",
                      OSAL_ThreadWhoAmI(),
                      pCurrentEntry->szName,
                      prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId, 
                      prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqIdRef-(tU32)MqPid,
                      pCurrentEntry->u32EntryIdx);   
         char Command[128];
         snprintf(Command,128,"ls -la /proc/%d/fd",(int)MqPid);
         if(CreateProcess(Command,"/dev/errmem") == -1)
         {
            // for non root send to TERM task
            vGenerateFdListFromTerm(OSAL_ProcessWhoAmI());
         }
      }
   }
   pCurrentEntry->u32BFD_Count++;
}


void vPostToCallbackQueue(trMqueueElement* pCurrentEntry)
{
   tU32 u32Error = OSAL_E_NOERROR;
   tOsalMbxMsg rMsgRecBuffer;
   rMsgRecBuffer.rOsalMsg.Cmd = MBX_MQ_NOTIFY;
   rMsgRecBuffer.rOsalMsg.ID  = pCurrentEntry->u32EntryIdx;

   /* ensure phMQCbPrc is allocated */
   if(phMQCbPrc)
   {
      if(phMQCbPrc[pCurrentEntry->callbackpididx] == 0)
      {
         phMQCbPrc[pCurrentEntry->callbackpididx] = GetPrcLocalMsgQHandle(pCurrentEntry->callbackpid);
        /*   TraceString("OSALMQ TID:%d %s vPostToCallbackQueue install CB MQ Handle:%p",
                   OSAL_ThreadWhoAmI(), pCurrentEntry->szName,hMQCbPrc[pCurrentEntry->callbackpididx]);*/
      }
      if(OSAL_s32MessageQueuePost(phMQCbPrc[pCurrentEntry->callbackpididx],
                                  (tPCU8)&rMsgRecBuffer, 
                                  sizeof(tOsalMbxMsg),
                                  0) == OSAL_ERROR)
      {
         u32Error = OSAL_u32ErrorCode();
         TraceString("OSALMQ TID:%d %s vPostToCallbackQueue Error:0x%x Handle:%p",
                     OSAL_ThreadWhoAmI(), pCurrentEntry->szName,u32Error,phMQCbPrc[pCurrentEntry->callbackpididx]);
         /* check for block callback handler task */
         if((u32Error == OSAL_E_TIMEOUT)||(u32Error == OSAL_E_QUEUEFULL))
         {
            tU32 u32MaxMessages = 0;
            tU32 u32MaxLength = 0;
            tU32 u32Messages = 0;
            (void)OSAL_s32MessageQueueStatus(phMQCbPrc[pCurrentEntry->callbackpididx],&u32MaxMessages,&u32MaxLength,&u32Messages);
            vWritePrintfErrmem("OSALMQ %s seems to be blocked -> MaxMsg:%d ActMsg:%d \n",
                               prProcDat[pCurrentEntry->callbackpididx].szMqName,u32MaxMessages,u32Messages);
            int sig = 0;
            if((sig = u32GetSigRtMinId(pCurrentEntry->RecPrc)) == -1)
            {
               sig = SIG_BACKTRACE;
            }
            if(pOsalData->u32DebugSignal != 0xffffffff)
            {
               /* get callstack of receiver */
               if(kill((int)pCurrentEntry->RecPrc,sig) == -1)
               {   
                  if(errno == EPERM)
                  {
                     vWritePrintfErrmem("OSALMQ MQ:%s Do Kill via OSAL Term Task due permissions  \n",pCurrentEntry->szName);
                     vSendCallStackGenFromTerm(pCurrentEntry->RecPrc,MBX_GEN_CS_USR2);
                  }
                  else
                  {
                     vWritePrintfErrmem("OSALMQ MQ:%s Kill SIG_BACKTRACE failed Error:%d  \n",pCurrentEntry->szName,errno);
                  } 
               }
            }
         }
      }
   }
   else
   {
    //  TraceString("OSALMQ OSALMQ Callback HAndler MQ Structure not initialized");
      vInitCbMqStructure();
      vPostToCallbackQueue(pCurrentEntry);
   }
   /* do not set an error code when using callback system doesn't work, because message is 
      completely delivered to receiver and will be detected by timeout */
   if(u32Error != OSAL_E_NOERROR)
   {
      if((u32Error != OSAL_E_TIMEOUT)&&(u32Error != OSAL_E_TIMEOUT))
      {
         vWritePrintfErrmem("OSALMQ %s Reset Error Code:%d from CB HDR system \n",
                            prProcDat[pCurrentEntry->callbackpididx].szMqName,u32Error);
      }
      vSetErrorCode(OSAL_C_THREAD_ID_SELF, OSAL_E_NOERROR);

#ifdef USE_OSAL_CB_HDR_SIGNAL
      if(u32Error == OSAL_E_QUEUEFULL)
      {
         //tS32 s32Tid = pStdMQEl[((tMQUserHandle*phMQCbPrc[pCurrentEntry->callbackpididx])->pOrigin->u32EntryIdx].RecTsk;
         tS32 s32Pid = pStdMQEl[((tMQUserHandle*phMQCbPrc[pCurrentEntry->callbackpididx])->pOrigin->u32EntryIdx].RecPrc;
         /* check for callback handler task */
         if(s32Pid)
         {
               kill(s32Pid, OSAL_CB_HDR_SIGNAL);
         }
      }
#endif
   }
}


tU32 u32GemTmoStruct(struct timespec *tim,OSAL_tMSecond msec)
{
   tU32 u32ErrorCode = OSAL_E_NOERROR;

   if(!clock_gettime(CLOCK_REALTIME, tim))
   {
        /* If the Timers option is supported, the timeout shall be based on the CLOCK_REALTIME clock; 
          if the Timers option is not supported, the timeout shall be based on the system clock as returned by the time() function.*/
       tim->tv_sec += msec / 1000 ;
       tim->tv_nsec += (msec%1000)*1000000;
       if(tim->tv_nsec >= 1000000000)
       {
          tim->tv_sec += 1;
          tim->tv_nsec = tim->tv_nsec - 1000000000;
       }
   }
   else
   {
      u32ErrorCode = u32ConvertErrorCore(errno);
   }
   return u32ErrorCode;
}

/*****************************************************************************
 *
 * FUNCTION: OSAL_s32MessageQueuePost
 *
 * DESCRIPTION: Sends a message of given length and priority to a MessageQueue.
 *
 * PARAMETER:   hMQ       MessageQueue handle
 *              pcou8Msg  const pointer to the buffer containing the message
 *              u32Length message length in bytes
 *              u32Prio   message priority
 *
 * ERRORS:      OSAL_E_BADFILEDESCRIPTOR  hMQ is not a valid handle
 *              OSAL_E_INVALIDVALUE       prio exceeds the boundary
 *              OSAL_E_MSGTOOLONG         length exceeds the field MaxLength
 *                                        defined in creation
 *              OSAL_E_QUEUEFULL          the MQ is full
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.10.05  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 * 13.12.12  | Fix for OSAL MQ priority issue         |sja3kor 
 * --.--.--  | ----------------                       | -----
 * 16.01.13  |Fix for Wrong timeout for Linux message |sja3kor
 *           | queue post/wait                        |
 *****************************************************************************/
tS32 OSAL_s32MessageQueuePost( OSAL_tMQueueHandle hMQ,
                               tPCU8 pcou8Msg,
                               tU32  u32Length,
                               tU32  u32Prio)
{
	
   CountMessage(u32Length);
#if !defined OSAL_SHM_MQ_FOR_LINUX_MQ || defined DBUS_MQ_SUPPORT
   tU32      u32ErrorCode   = u32CheckMqHandle(hMQ); /* step 1 user handle check */
#endif
#ifdef DBUS_MQ_SUPPORT
   if(pOsalData->u32OsalMqDBus)
   {
     if(u32ErrorCode == OSAL_E_NOERROR)
     {
        if(((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->u16Type == MQ_DBUS )
        {
           return OSAL_s32DBusMessageQueuePost(hMQ,pcou8Msg,u32Length,u32Prio);
        }
     }
   }
#endif
#ifdef OSAL_SHM_MQ_FOR_LINUX_MQ
   return OSAL_s32ShMemMessageQueuePost(hMQ,pcou8Msg,u32Length,u32Prio);
#else 
   tS32 s32ReturnValue = OSAL_OK;
   trMqueueElement *pCurrentEntry = NULL;
   tU32 u32ActualMessage = 0xffffffff;
   struct timespec tim = {0,0};
   tU32 u32Prio_Li = 0;                      //u32Prio_Li variable indicates linux specific priority.
   tU32 msec = 0;
   tU32 u32Temp;
   struct mq_attr mqstat = {0,0,0,0,0};
   char name[TRACE_NAME_SIZE];
   if(u32ErrorCode == OSAL_E_NOERROR) 
   {
      /* check permissions*/
      if(((tMQUserHandle*)hMQ)->enAccess == OSAL_EN_READONLY)
      {
         u32ErrorCode = OSAL_E_NOPERMISSION;
      }
   }
   if(u32ErrorCode == OSAL_E_NOERROR) 
   {
      /* step 2 uparameter check */
      pCurrentEntry = ((tMQUserHandle*)hMQ)->pOrigin; /*lint !e613 *//* pointer already checked*/ 
   }
   if(pCurrentEntry)
   {
/*         if((pCurrentEntry->u32EntryIdx >= pOsalData->MqueueTable.u32UsedEntries)||(pCurrentEntry->u32MqueueID != LINUX_C_U32_MQUEUE_ID))
         {
            CheckandCorrectMqStaticValues();
         }*/

         if( u32Prio < (OSAL_C_U32_MQUEUE_PRIORITY_LOWEST + 1) )
         {  
             /* check if queue isn't deleted */
            if(u32Length > pCurrentEntry->MaxLength)
            {
               u32ErrorCode = OSAL_E_MSGTOOLONG;
            }
         /*      else if(u32Length < pCurrentEntry->MaxLength)
               {
                   TRACE_LINE_INFO;
                   memcpy(cResMsgBuffer,pcou8Msg,u32Length);
                   pcSendMsg = (const tU8*)&cResMsgBuffer[0];
               }*/
            else if((pCurrentEntry->bIsUsed == FALSE)||(prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId == -1))
            {
               u32ErrorCode = OSAL_E_DOESNOTEXIST;
            }
            /*else if(pElement->rStdMsgQueue.bToDelete == TRUE)
            {
                 u32ErrorCode = OSAL_E_NOPERMISSION;
            }*/
            else
            {
            }

            if(u32ErrorCode == OSAL_E_NOERROR) 
            {
               msec = pOsalData->u32MqPostTmo;
               /* step 3 check for full message queue and wait , when queue is greater minimum size*/
               if((pCurrentEntry->ActualMessage == pCurrentEntry->MaxMessages)&&(pCurrentEntry->MaxMessages > 1))
               {
#ifdef BLOCKING_TIMEOUT
                  if(OSAL_ThreadWhoAmI() == pCurrentEntry->RecTsk)
                  {
                     u32Temp = (tU32)OSAL_ThreadWhoAmI();
                     bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,OSAL_ThreadWhoAmI());
                     vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) would block itself MQ: Handle 0x%x Name:%s -> return from OSAL_s32MessageQueuePost\n",
                                        name,(unsigned int)u32Temp, (uintptr_t)pCurrentEntry,pCurrentEntry->szName); /* lint !e613 */ /* u32ErrorCode ensures that hMQ is a valid pointer value */
                     vSetErrorCode(OSAL_C_THREAD_ID_SELF, OSAL_E_QUEUEFULL);
                     return OSAL_ERROR;
                  }
                  if(msec == 0)msec = BLOCKING_TIMEOUT;
                  pCurrentEntry->bBlocked = TRUE;
                  if(pOsalData->u32NoMqBlockEntry & 0x01)
                  {
                     /* no entries required */
                  }
                  else
                  {
                     u32Temp = (tU32)OSAL_ThreadWhoAmI();
                     bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,OSAL_ThreadWhoAmI());
                     vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) block activated at MQ: Handle 0x%x Name:%s \n",
                                        name,(unsigned int)u32Temp, (uintptr_t)pCurrentEntry,pCurrentEntry->szName); /* lint !e613 */ /* u32ErrorCode ensures that hMQ is a valid pointer value */
                  }
                  if(pOsalData->u32NoMqBlockEntry & 0x10)
                  {
                     vReactOnMqOverflow(pCurrentEntry);
                  }
#endif //BLOCKING_TIMEOUT
               }
               if(msec)
               {
                  u32ErrorCode = u32GemTmoStruct(&tim,msec);
               }
            }
            if(u32ErrorCode == OSAL_E_NOERROR) 
            {
               if(pCurrentEntry->bTraceCannel == TRUE)
               {
                   TraceString("OSALMQ TID:%d Start Send Timeout:%d msec MQ:%s",OSAL_ThreadWhoAmI(),msec,pCurrentEntry->szName);
               }
               /* converted the OSAL priority to linux specific priority(for OSAL : 0:higest prio 7:lowest prio
                  for Linux: 0:lowest prio 7:highest prio)*/
               u32Prio_Li = OSAL_C_U32_MQUEUE_PRIORITY_LOWEST - u32Prio ;
               if(pOsalData->bCheckCcaMsg)
               {
                   vTraceCcaMsg((OSAL_trMessage*)pcou8Msg,u32Length,pCurrentEntry,"OSAL_s32MessageQueuePost");/*lint !e826 !e1773*/
               }
#ifndef NO_SIGNALAWARENESS
               while(1)
               {
#endif
                  /* reset error code for retry */
                  u32ErrorCode = OSAL_E_NOERROR;
#ifdef MQ_NOTIFY_VIA_OSAL
                  if(pCurrentEntry->bNotify == TRUE)
                  {
                     if((s32ReturnValue = mq_getattr(prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId,&mqstat)) == 0)
                     {
                         if(mqstat.mq_curmsgs == 0)pCurrentEntry->bTriggerCb = TRUE;
                         pCurrentEntry->ActualMessage = u32ActualMessage = (tU32)mqstat.mq_curmsgs;
                     }
                     else
                     {
                         TraceString("OSALMQ Send mq_getattr errno:%d %s",errno,pCurrentEntry->szName);
                         vWritePrintfErrmem("OSALMQ Send mq_getattr errno:%d %s \n",errno,pCurrentEntry->szName);
                         if(errno == EBADF)
                         {
                            if(((tMQUserHandle*)hMQ)->PrcId != OSAL_ProcessWhoAmI())
                            {
                               TraceString("OSAL_s32MessageQueuePost Process %d uses handle struct from process %d",OSAL_ProcessWhoAmI(),((tMQUserHandle*)hMQ)->PrcId);
                               vWritePrintfErrmem("OSAL_s32MessageQueuePost Process %d uses handle struct from process %d",OSAL_ProcessWhoAmI(),((tMQUserHandle*)hMQ)->PrcId);
                            }
                            vRestoreMq(pCurrentEntry);
                            if((s32ReturnValue = mq_getattr(prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId,&mqstat)) == -1)
                            {
                               TraceString("OSALMQ Send mq_getattr errno:%d %s",errno,pCurrentEntry->szName);
                               vWritePrintfErrmem("OSALMQ Send mq_getattr errno:%d %s \n",errno,pCurrentEntry->szName);
                            }
                         }
                     }
                  }
#endif
                  if(msec)
                  {
                     s32ReturnValue = mq_timedsend((mqd_t)prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId,
                                                   (const char*)pcou8Msg,//pcSendMsg,
                                                   u32Length,//pCurrentEntry->MaxLength, 
                                                   u32Prio_Li,
                                                   &tim);
                  }
                  else
                  {
                     s32ReturnValue = mq_send(prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId,
                                              (const char*)pcou8Msg,//pcSendMsg, 
                                              u32Length,//pCurrentEntry->MaxLength,
                                              u32Prio_Li);
                  }
                  if(s32ReturnValue != -1)
                  {
                      u32ActualMessage = ++pCurrentEntry->ActualMessage;
                      /* check for queue fill level */
                      if(pOsalData->u32NoMqBlockEntry & 0x10000000)
                      {
                          if(pCurrentEntry->u32MaxMsgCount< u32ActualMessage)pCurrentEntry->u32MaxMsgCount = u32ActualMessage;
                      }
#ifndef NO_SIGNALAWARENESS
                      break;
#endif
                  }
                  else if(errno != EINTR)
                  {
                     /* message queue wait is failed not because of signal 
                        check queue overflow */
                     if((EAGAIN == errno)||(ETIMEDOUT == errno))
                     {
                         if(mq_getattr(prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId,&mqstat) == 0)
                         {
                            if(mqstat.mq_curmsgs < mqstat.mq_maxmsg)
                            {
                               TraceString("OSALMQ %s retry Status Cur:%d Max:%d \n",
                                                  pCurrentEntry->szName,mqstat.mq_curmsgs,mqstat.mq_maxmsg);
                               vWritePrintfErrmem("OSALMQ %s retry Status Cur:%d Max:%d \n",
                                                  pCurrentEntry->szName,mqstat.mq_curmsgs,mqstat.mq_maxmsg);
                               return OSAL_s32MessageQueuePost(hMQ,pcou8Msg,u32Length,u32Prio);
                            }
                            pCurrentEntry->ActualMessage = u32ActualMessage = (tU32)mqstat.mq_curmsgs;
                         }
                         else
                         {
                            TraceString("OSALMQ after Send mq_getattr errno:%d %s \n",errno,pCurrentEntry->szName);
                            vWritePrintfErrmem("OSALMQ after Send mq_getattr errno:%d %s \n",errno,pCurrentEntry->szName);
                         }
                         u32ErrorCode = OSAL_E_QUEUEFULL;
                         vReactOnMqOverflow(pCurrentEntry);
                         break;
                     }
                     else
                     {
                        u32ErrorCode = u32ConvertErrorCore(errno);
                        if(errno == EBADF)
                        {
                           if(((tMQUserHandle*)hMQ)->PrcId != OSAL_ProcessWhoAmI())
                           {
                              TraceString("OSAL_s32MessageQueuePost Process %d uses handle struct from process %d",OSAL_ProcessWhoAmI(),((tMQUserHandle*)hMQ)->PrcId);
                              vWritePrintfErrmem("OSAL_s32MessageQueuePost Process %d uses handle struct from process %d",OSAL_ProcessWhoAmI(),((tMQUserHandle*)hMQ)->PrcId);
                           }
                           TraceString("OSALMQ mq_send Task:%d restores Linux File Descriptor Handle:%d",
                                       OSAL_ThreadWhoAmI(),prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId);
                           vWritePrintfErrmem("OSALMQ mq_send Task:%d restores Linux File Descriptor Handle:%d\n",
                                              OSAL_ThreadWhoAmI(),prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId);
                           vRestoreMq(pCurrentEntry);
                        }
                        break;
                     }
                  }
                  /* We are here because of signal. lets try again. */
#ifndef NO_SIGNALAWARENESS
               }// end while
#endif
            } //if(u32ErrorCode == OSAL_E_NOERROR)
      }
      else
      {
         u32ErrorCode = OSAL_E_INVALIDVALUE;
      }
   }//if(pCurrentEntry)
   else
   {
      if(u32ErrorCode == OSAL_E_NOERROR)u32ErrorCode = OSAL_E_DOESNOTEXIST; 
   }

   if((u32ErrorCode == OSAL_E_NOERROR)&&(pCurrentEntry)&&(pCurrentEntry->bTriggerCb == TRUE))
   {
#ifdef MQ_NOTIFY_VIA_OSAL
      if((pCurrentEntry->callbackpididx < (tS32)pOsalData->u32MaxNrProcElements)
      && (pCurrentEntry->pcallback )&&(pCurrentEntry->callbackpididx >= 0))
      {
         pCurrentEntry->bTriggerCb = FALSE;
         if( pCurrentEntry->callbackpid == OSAL_ProcessWhoAmI()) // check if process is the same
         {
            vExecuteCallback(pCurrentEntry);
         }
         else
         {
            vPostToCallbackQueue(pCurrentEntry);
         }
		 
      }
#endif
#ifdef USE_EVENT_AS_CB
      if((pCurrentEntry->mask))
      {
         pCurrentEntry->bTriggerCb = FALSE;
         if(((tMQUserHandle*)hMQ)->hNotifyEvent == 0)
         {
            if(OSAL_s32EventOpen(pCurrentEntry->szEvName,&((tMQUserHandle*)hMQ)->hNotifyEvent) != OSAL_OK)
            {
                 NORMAL_M_ASSERT_ALWAYS();
            }
         }
         OSAL_s32EventPost(((tMQUserHandle*)hMQ)->hNotifyEvent, pCurrentEntry->mask, pCurrentEntry->enFlags);
      }
#endif
   }

   if( u32ErrorCode != OSAL_E_NOERROR )
   {
      s32ReturnValue = OSAL_ERROR;
      if((u32ErrorCode != OSAL_E_TIMEOUT)||
        ((pCurrentEntry)&&(pCurrentEntry->bTraceCannel == TRUE))||(u32OsalSTrace & 0x00000020))
      {
#ifdef SHORT_TRACE_OUTPUT
          char au8Buf[64]={0}; 
          u32Temp = (tU32)OSAL_ThreadWhoAmI();
          OSAL_M_INSERT_T8(&au8Buf[0],(tU8)OSAL_MQ_SEND);
          bGetThreadNameForTID(&au8Buf[1],8,(tS32)u32Temp);
          OSAL_M_INSERT_T32(&au8Buf[9],(tU32)u32Temp);
          OSAL_M_INSERT_T32(&au8Buf[13],(tU32)&hMQ);
          /* u32ActualMessage  */
          OSAL_M_INSERT_T32(&au8Buf[19],u32ErrorCode);
          OSAL_M_INSERT_T32(&au8Buf[23],u32Prio);
          if(pCurrentEntry)
          {
            OSAL_M_INSERT_T16(&au8Buf[17],(tU16)u32ActualMessage);
            u32Temp = strlen(pCurrentEntry->szName);
            memcpy (&au8Buf[27],pCurrentEntry->szName,u32Temp);
          }
          else
          {
            u32Temp = strlen("Unknown");
            memcpy (&au8Buf[27],"Unknown",u32Temp);
          }
          LLD_vTrace(OSAL_C_TR_CLASS_SYS_MQ, TR_LEVEL_FATAL,au8Buf,u32Temp+27);
          if((pOsalData->bLogError)&&(u32ErrorCode != OSAL_E_TIMEOUT)&&(u32OsalSTrace & ~0x00000020))
          {
               vWriteToErrMem(OSAL_C_TR_CLASS_SYS_MQ,au8Buf,u32Temp+27,0);
          }
#else
          u32Temp = (tU32)OSAL_ThreadWhoAmI();
          bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,u32Temp);
          if(pCurrentEntry)
          {
             TraceString("OSALMQ Task:%s(ID:%d) Send      to MQ: User Handle %p MQ Name:%s Msg Count:%d Error:0x%x Prio:%d",
                      name,(unsigned int)u32Temp,pCurrentEntry, pCurrentEntry->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32Prio);
             if((pOsalData->bLogError)&&(u32ErrorCode != OSAL_E_TIMEOUT)&&(u32OsalSTrace & ~0x00000020))vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) Send      to MQ: User Handle %p MQ Name:%s Msg Count:%d Error:0x%x Prio:%d \n",
                      name,(unsigned int)u32Temp,pCurrentEntry,pCurrentEntry->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32Prio);
          }
          else
          {
             TraceString("OSALMQ Task:%s(ID:%d) Send      to MQ: User Handle 0x%x MQ Name:%s Msg Count:%d Error:0x%x Prio:%d",
                      name,(unsigned int)u32Temp,(unsigned int)0, "Unknown",(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32Prio);
             if((pOsalData->bLogError)&&(u32ErrorCode != OSAL_E_TIMEOUT)&&(u32OsalSTrace & ~0x00000020))vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) Send      to MQ: User Handle 0x%x MQ Name:%s Msg Count:%d Error:0x%x Prio:%d \n",
                      name,(unsigned int)u32Temp,(unsigned int)0, "Unknown",(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32Prio);
          }
#endif
          if(bTraceHandle)
          {
             if(pCurrentEntry)
             {
                TraceString("OSALMQ Snd MQ:%s OSAL User Handle:0x%x Linux Handle:%d Offset:0x%x Pid:%d Count:%d",pCurrentEntry->szName,hMQ,
                prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId,
                GetMemPoolOffset(&MqMemPoolHandle, (uintptr_t*)&hMQ),
                prLiMqMap[pCurrentEntry->u32EntryIdx].PrcId,
                prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt);
             }
             else
             {
                TraceString("OSALMQ Snd MQ:? OSAL User Handle:0x%x Linux Handle:? Offset:0x%x Pid:? Count:?",hMQ,GetMemPoolOffset(&MqMemPoolHandle, (uintptr_t*)&hMQ));
             }
          }
      }
      vSetErrorCode(OSAL_C_THREAD_ID_SELF, u32ErrorCode);
   }
   else
   {
       if((pCurrentEntry->bTraceCannel == TRUE)||(u32OsalSTrace & 0x00000020))/*lint !e613 pCurrentEntry already checked */
       {
#ifdef SHORT_TRACE_OUTPUT
           char au8Buf[64];
           u32Temp = (tU32)OSAL_ThreadWhoAmI();
           OSAL_M_INSERT_T8(&au8Buf[0],(tU8)OSAL_MQ_SEND);
           bGetThreadNameForTID(&au8Buf[1],8,(tS32)u32Temp);
           OSAL_M_INSERT_T32(&au8Buf[9],u32Temp);
           OSAL_M_INSERT_T32(&au8Buf[13],(tU32)&hMQ);
           OSAL_M_INSERT_T16(&au8Buf[17],(tU16)u32ActualMessage);
           OSAL_M_INSERT_T32(&au8Buf[19],u32ErrorCode);
           OSAL_M_INSERT_T32(&au8Buf[23],u32Prio);
           u32Temp = (tU32)strlen(pCurrentEntry->szName);/*lint !e613 *//* pointer already checked*/ 
           memcpy (&au8Buf[27],pCurrentEntry->szName,u32Temp);/*lint !e613 *//* pointer already checked*/ 
           LLD_vTrace(OSAL_C_TR_CLASS_SYS_MQ, /*TR_LEVEL_USER_1*/TR_LEVEL_FATAL,au8Buf,u32Temp+27);
#else
           u32Temp = (tU32)OSAL_ThreadWhoAmI();
           bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,u32Temp);
           TraceString("OSALMQ Task:%s(ID:%d) Send      to MQ: User Handle %p MQ Name:%s Msg Count:%d Error:0x%x Prio:%d",
                      name,(unsigned int)u32Temp,pCurrentEntry, pCurrentEntry->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32Prio);
#endif
       }
   }
   return( s32ReturnValue );
#endif
}



void vStartMeasureProcessingTime(trMqueueElement *pCurrentEntry)
{
   if(pCurrentEntry->u32MessagePrcessing != MEASUREMENT_SWITCHOFF)
   {
      pCurrentEntry->u32MessagePrcessing = MSG_PROCESSING_ACTIVE;
      pCurrentEntry->u32IdxMPT++;
      if(pCurrentEntry->u32IdxMPT == OSAL_C_MSG_TIME_ARR_SIZE)
      {   pCurrentEntry->u32IdxMPT = 0;   }
      pCurrentEntry->u32MsgTimStamp[pCurrentEntry->u32IdxMPT] = pOsalData->u32Seconds;
      pCurrentEntry->u32MsgCount[pCurrentEntry->u32IdxMPT] = pCurrentEntry->ActualMessage;
   }
}

void vStopMeasureProcessingTime(trMqueueElement *pCurrentEntry)
{
   if(pCurrentEntry->u32MessagePrcessing != MEASUREMENT_SWITCHOFF)
   {
      char name[16];
      if((pCurrentEntry->u32MessagePrcessing == MSG_PROCESSING_ACTIVE)&&(pOsalData->u32Seconds))
      {
         pCurrentEntry ->u32MsgPrcTim[pCurrentEntry->u32IdxMPT] = pOsalData->u32Seconds - pCurrentEntry->u32MsgTimStamp[pCurrentEntry->u32IdxMPT];
         if((pCurrentEntry ->u32MsgPrcTim[pCurrentEntry->u32IdxMPT] > 2)&&(pCurrentEntry->MaxMessages > 1))/* CSM MQs with size 1 are usually observed with greater intervals*/
         {
            pCurrentEntry->u32LongMsgPrcTim[pCurrentEntry->u32IdxLMPT] = pCurrentEntry->u32MsgPrcTim[pCurrentEntry->u32IdxMPT];
            pCurrentEntry->u32LongMsgTimStamp[pCurrentEntry->u32IdxLMPT] = pCurrentEntry->u32MsgTimStamp[pCurrentEntry->u32IdxMPT];
            bGetThreadNameForTID(&name[0],16,(tS32)pCurrentEntry ->RecTsk);
            TraceString("Task:%s(ID:%u) Message Processing needs %u seconds for MQ:%s",
                      name,pCurrentEntry->RecTsk,pCurrentEntry->u32LongMsgPrcTim[pCurrentEntry->u32IdxLMPT],pCurrentEntry->szName);
            /* store only long runners */
            pCurrentEntry ->u32IdxLMPT++;
            if(pCurrentEntry ->u32IdxLMPT == OSAL_C_LONG_MSG_TIME_ARR_SIZE)
            {   pCurrentEntry ->u32IdxLMPT = 0;   }
         }
      }
      pCurrentEntry->u32MessagePrcessing = MSG_PROCESSING_INACTIVE;
   }
}


/*****************************************************************************
 *
 * FUNCTION:    OSAL_s32MessageQueueWait
 *
 * DESCRIPTION: Query a message from  a MessageQueue. The obtained Message is
 *              the one with the higher priority existing in the MessageQueue
 *              and the actual length of the copied message is defined by the
 *              function param and not by the message length.
 *
 * PARAMETER:   hMQ       MessageQueue handle
 *              pu8Buffer pointer to the output buffer for the message copy
 *              u32Length actual length to e copied in bytes
 *              pu32Prio  pointer to the actual priority of the copied message
 *              msec      timeout
 *
 * ERRORS:      OSAL_E_BADFILEDESCRIPTOR  hMQ is not a valid handle
 *              OSAL_E_MSGTOOLONG         length exceeds the field MaxLength
 *                                        defined in creation
 *              OSAL_E_TIMEOUT            the timeout is reached without
 *                                        any message in the MQueue
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.10.05  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 * 13.12.12  | Fix for OSAL MQ priority issue         |sja3kor 
 * 17.12.12  | Fix for MMS CFAGII-501.Boundary check  |sja3kor
 *           |  condition changed.                    | 
 * --.--.--  | ----------------                       | -------, -----
 * 16.01.13  |Fix for Wrong timeout for               |sja3kor 
 *           |Linux messagequeue post/wait CFAGII-506 | 
 * 21.01.13  | Integrating GMNGA-48007 Fix            | SWM2KOR   
 *****************************************************************************/
tS32 OSAL_s32MessageQueueWait( OSAL_tMQueueHandle hMQ,
                               tPU8 pu8Buffer,
                               tU32 u32Length,
                               tPU32 pu32Prio,
                               OSAL_tMSecond msec)
{
#if !defined OSAL_SHM_MQ_FOR_LINUX_MQ || defined DBUS_MQ_SUPPORT
   tU32 u32ErrorCode = u32CheckMqHandle(hMQ); /* step 1 user handle check */; 
#endif
#ifdef DBUS_MQ_SUPPORT
   if(pOsalData->u32OsalMqDBus)
   {
      if(u32ErrorCode == OSAL_E_NOERROR)
      {
         if(((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->u16Type == MQ_DBUS )
         {
            return OSAL_s32DBusMessageQueueWait(hMQ,pu8Buffer,u32Length,pu32Prio,msec);
         }
      }
   }
#endif
#ifdef OSAL_SHM_MQ_FOR_LINUX_MQ
    return OSAL_s32ShMemMessageQueueWait(hMQ,pu8Buffer,u32Length,pu32Prio,msec);
#else
   
   tU32 u32CopiedBytes = 0;
   tS32 s32Val = 0;
   tU32 u32ActualPrio = 0xffffffff;
   trMqueueElement *pCurrentEntry = NULL;
   tU32 u32ActualMessage = 0xffffffff;
   struct timespec tim ={0,0};
   struct mq_attr mqstat = {0,0,0,0,0};
   tU32 u32Temp = 0;
   tU32 u32Time;
//  tPU8 pRecMsg = pu8Buffer;
//  char cResMsgBuffer[1024];
   char name[TRACE_NAME_SIZE];
   if(u32ErrorCode == OSAL_E_NOERROR) 
   {
      /* check permissions*/
      if(((tMQUserHandle*)hMQ)->enAccess == OSAL_EN_WRITEONLY) 
      {
          u32ErrorCode = OSAL_E_NOPERMISSION;
      }
   }
  if(u32ErrorCode == OSAL_E_NOERROR)
  {
     pCurrentEntry = ((tMQUserHandle*)hMQ)->pOrigin;/*lint !e613 *//* pointer already checked*/ 
  }
  if(pCurrentEntry)
  {
/*        if((pCurrentEntry->u32EntryIdx >= pOsalData->MqueueTable.u32UsedEntries)||(pCurrentEntry->u32MqueueID != LINUX_C_U32_MQUEUE_ID))
        {
           CheckandCorrectMqStaticValues();
        }*/
        if(pCurrentEntry->bToDelete == TRUE)
        {
           msec = OSAL_C_TIMEOUT_NOBLOCKING;
        }
        pCurrentEntry->u16WaitCounter++;

       /*     if(u32Length > pCurrentEntry->MaxLength)
            {
               u32ErrorCode = OSAL_E_MSGTOOLONG;
            }
            else if(u32Length < pCurrentEntry->MaxLength)
            {
                memcpy(cResMsgBuffer,pu8Buffer,u32Length);
                pRecMsg = (tU8*)&cResMsgBuffer[0];
            }*/

        if(pCurrentEntry ->RecTsk == 0)
        {
           pCurrentEntry->RecTsk = OSAL_ThreadWhoAmI();
           pCurrentEntry->RecPrc = OSAL_ProcessWhoAmI();
        }
        /* Stop message processing time measurement */
        vStopMeasureProcessingTime(pCurrentEntry);

        if(msec == OSAL_C_TIMEOUT_NOBLOCKING)
        {
           if(mq_getattr((mqd_t)prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId,&mqstat) == 0)
           {
              if(mqstat.mq_curmsgs == 0)
              {
                 u32ErrorCode = OSAL_E_TIMEOUT;
              }
              else
              {
                 /* message is in the queue we can access with timeout forever */
                 msec = OSAL_C_TIMEOUT_FOREVER;
              }
              pCurrentEntry->ActualMessage = u32ActualMessage = (tU32)mqstat.mq_curmsgs;
           }
           else
           {
               TraceString("OSALMQ wait mq_getattr %s failed errno:%d \n",pCurrentEntry->szName,errno);
               vWritePrintfErrmem("OSALMQ wait mq_getattr %s failed errno:%d \n",pCurrentEntry->szName,errno);
               u32ErrorCode = u32ConvertErrorCore(errno);
           }
        }
        else if(msec != OSAL_C_TIMEOUT_FOREVER)
        {
           if(pCurrentEntry->u32WaitTime != 0xffffffff)
           {
              /* check for MQ supervison for clock monotonic behavior */
              pCurrentEntry->u32WaitTime = OSAL_ClockGetElapsedTime();
              /* store old message content */
              u32Temp = *((tPU32)pu8Buffer);/*lint !e826 PQM_authorized_529 */
           }
        }
        if(u32ErrorCode == OSAL_E_NOERROR)
        {
           while(1)
           {
               /* reset error code for retry */
               u32ErrorCode = OSAL_E_NOERROR;
               if(msec == OSAL_C_TIMEOUT_FOREVER)
               {
                  s32Val = (tS32)mq_receive((mqd_t)prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId,
                                      (char*)pu8Buffer,//pRecMsg,
                                      (size_t)u32Length,//pCurrentEntry->MaxLength, 
                                      (unsigned int*)pu32Prio);
               }
               else
               {
                  u32ErrorCode = u32GemTmoStruct(&tim,msec);
                  s32Val = (tS32)mq_timedreceive((mqd_t)prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId,
                                            (char*)pu8Buffer,//pRecMsg,
                                            u32Length, //pCurrentEntry->MaxLength, 
                                            (unsigned int*)pu32Prio,
                                            &tim);
                  if((pCurrentEntry->u32WaitTime != 0xffffffff)&&(pOsalData->RecalcTime > 0))/* ensure clock monotonic bavior */
                  {
                     u32Time = OSAL_ClockGetElapsedTime();
                     if(u32Time > pOsalData->RecalcTime)
                     {
                        pOsalData->RecalcTime = 0;
                     }
                      if((s32Val ==-1) && (errno == ETIMEDOUT))
                     {
                        if(pOsalData->u32ClkMonotonic & 0x10000000)TraceString("OSALMQ Timeout recalculation ETIMEDOUT Check:%d",pCurrentEntry->bCheckTmo);
                        /* check if timeout is happen to early due setting system time */
                        if((u32Time -  pCurrentEntry->u32WaitTime) < msec)
                        {
                           /* calculate new timeout */
                           msec = msec -(u32Time -  pCurrentEntry->u32WaitTime);
                           /* change error code to stay in the do while loop */
                           errno = EINTR;
                           if(pOsalData->u32ClkMonotonic & 0x10000000)TraceString("OSALMQ %s Timeout recalculated Timeout:%d",pCurrentEntry->szName,msec);
                        }
                        else
                        {
                           if(pOsalData->u32ClkMonotonic & 0x10000000)TraceString("OSALMQ %s valid Timeout:%d  Elapsed:%d WaitTime:%d",pCurrentEntry->szName,msec, u32Time,pCurrentEntry->u32WaitTime);
                        }					
                     }   
                     else if(pCurrentEntry->bCheckTmo)
                     {
                        /* check for time changed trigger */
                        if((s32Val == sizeof(tU32))&&(*((tPU32)pu8Buffer) == 0xA3A3A3A3)) /*lint !e826 PQM_authorized_529 */ /* used buffer is always size of tU32*/
                        {
                           /*mark no vlid message received*/ 
                           s32Val = -1;
                           if(pOsalData->u32ClkMonotonic & 0x10000000)TraceString("OSALMQ received trigger for MQ:%s ", pCurrentEntry->szName);
                           /* change error code to stay in the do while loop */
                           errno = EINTR;
                           /* restore old message buffer content */
                           *((tPU32)pu8Buffer) = u32Temp;/*lint !e826 PQM_authorized_529 */
                           /* check for real timeout */
                           if(u32Time - pCurrentEntry->u32WaitTime >= msec)
                           { 
                              /* check for another message is availbale*/
                              if(mqstat.mq_curmsgs == 0)
                              {
                                 if(pOsalData->u32ClkMonotonic & 0x10000000)TraceString("OSALMQ Timeout via Trigger for %s",pCurrentEntry->szName);
                                 errno = ETIMEDOUT;
                                 break;
                              }
                           }
                           else
                           {
                              /* calculate new timeout */ 
                              msec = msec -(u32Time - pCurrentEntry->u32WaitTime);
                              if(pOsalData->u32ClkMonotonic & 0x10000000)TraceString("OSALMQ %s Timeout recalculated Value:%d",pCurrentEntry->szName,msec);
                           }
                        }
                     }
                     pCurrentEntry->bCheckTmo = 0;
                     /* check if CLOCK_MONOTONIC is needed only once */
                     if(pOsalData->u32ClkMonotonic & 0x01000000)
                     {
                        pCurrentEntry->u32WaitTime = 0xffffffff;
                     }
					 
                  }//if(pOsalData->u32ClkMonotonic & 0x00000001)
               }
               if(s32Val != -1)
               {
                  if(mq_getattr(prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId,&mqstat) == 0)
                  {
                     pCurrentEntry->ActualMessage = u32ActualMessage = (tU32)mqstat.mq_curmsgs;
                  }
                  if(pu32Prio != OSAL_NULL)
                  {
                     /*mq_receive API gives the linux specific prio(returns here pu32Prio )(for OSAL : 0:higest prio 7:lowest prio
                       for Linux: 0:lowest prio 7:highest prio)So,converted prio back to OSAL specific priority */
                     *pu32Prio = u32ActualPrio = OSAL_C_U32_MQUEUE_PRIORITY_LOWEST - *pu32Prio ;
                  }
                  u32CopiedBytes = (tU32)s32Val;
                  break;
               }
               else if(errno != EINTR)
               {
                  /* message queue wait is failed not because of signal */
                  u32ErrorCode = u32ConvertErrorCore(errno);
                  if(errno == EBADF)
                  {
                     if(((tMQUserHandle*)hMQ)->PrcId != OSAL_ProcessWhoAmI())
                     {
                        TraceString("OSAL_s32MessageQueueWait Process %d uses handle struct from process %d",OSAL_ProcessWhoAmI(),((tMQUserHandle*)hMQ)->PrcId);
                        vWritePrintfErrmem("OSAL_s32MessageQueueWait Process %d uses handle struct from process %d",OSAL_ProcessWhoAmI(),((tMQUserHandle*)hMQ)->PrcId);
                     }
                     TraceString("OSALMQ mq_receive Task:%d restores Linux File Descriptor Handle:%d",OSAL_ThreadWhoAmI(),prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId);
                     vWritePrintfErrmem("OSALMQ mq_receive Task:%d restores Linux File Descriptor Handle:%d\n",OSAL_ThreadWhoAmI(),prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId);
                     vRestoreMq(pCurrentEntry);
                  }
                  else
                  {
                     if(OSAL_E_TIMEOUT != u32ErrorCode)
                     {
                        vWritePrintfErrmem("OSALMQ mq_receive EntryIdx:%d errno:%d  %s \n",
                                           pCurrentEntry->u32EntryIdx,errno,pCurrentEntry->szName);
                     }
                     break;
                  }
               }
           /* We are here because of signal. lets try again. */
           }// end while
        }// if(u32ErrorCode == OSAL_E_NOERROR)
#ifdef MQ_NOTIFY_VIA_OSAL
        if(u32ErrorCode == OSAL_E_TIMEOUT)
        {
           pCurrentEntry->bTriggerCb = TRUE;/*lint !e613 when u32ErrorCode == OSAL_E_TIMEOUT valid pCurrentEntry*/
        }
#endif
        if(u32CopiedBytes)
        {
           /* Start measurement of message processing times */
           vStartMeasureProcessingTime(pCurrentEntry);
           
           if(pOsalData->bCheckCcaMsg)
           {
              vTraceCcaMsg((OSAL_trMessage*)pu8Buffer,u32Length,pCurrentEntry,"OSAL_s32MessageQueueWait");/*lint !e826 */
           }
#ifdef BLOCKING_TIMEOUT
           if((pCurrentEntry->bBlocked)&&(pCurrentEntry->ActualMessage < pCurrentEntry->MaxMessages))
           {
              if(pOsalData->u32NoMqBlockEntry & 0x01)
              {
                /* no entries required*/
              }
              else
              {
                 u32Temp = (tU32)OSAL_ThreadWhoAmI();
                 bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,OSAL_ThreadWhoAmI());
                 vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) block released at MQ: Handle 0x%x Name:%s \n",
                                     name,(unsigned int)u32Temp, pCurrentEntry, pCurrentEntry->szName);/* lint !e613 */ /* u32ErrorCode ensures that hMQ is a valid pointer value */
              }
              pCurrentEntry->bBlocked = FALSE;
           }
#endif
        } //if(u32CopiedBytes)
   }//if(pCurrentEntry)
   else
   {
      if(u32ErrorCode == OSAL_E_NOERROR)u32ErrorCode = OSAL_E_DOESNOTEXIST; 
   }



  if( u32ErrorCode != OSAL_E_NOERROR )
  {
     if((u32ErrorCode != OSAL_E_TIMEOUT)||
        ((pCurrentEntry)&&(pCurrentEntry->bTraceCannel == TRUE))||(u32OsalSTrace & 0x00000020))
     {
#ifdef SHORT_TRACE_OUTPUT
         char au8Buf[64];
         u32Temp = (tU32)OSAL_ThreadWhoAmI();
         OSAL_M_INSERT_T8(&au8Buf[0],(tU8)OSAL_MQ_RECEIVE);
         bGetThreadNameForTID(&au8Buf[1],8,(tS32)u32Temp);
         OSAL_M_INSERT_T32(&au8Buf[9],(tU32)u32Temp);
         OSAL_M_INSERT_T32(&au8Buf[13],(tU32)&hMQ);
          /* u32ActualMessage  */
         OSAL_M_INSERT_T32(&au8Buf[19],u32ErrorCode);
         OSAL_M_INSERT_T32(&au8Buf[23],u32ActualPrio);
         if(pCurrentEntry)
         {
            OSAL_M_INSERT_T16(&au8Buf[17],(tU16)u32ActualMessage);
            u32Temp = strlen(pCurrentEntry->szName);
            memcpy (&au8Buf[27],pCurrentEntry->szName,u32Temp);
         }
         else
         {
            u32Temp = strlen("Unknown");
            memcpy (&au8Buf[27],"Unknown",u32Temp);
         }
         LLD_vTrace(OSAL_C_TR_CLASS_SYS_MQ, TR_LEVEL_FATAL,au8Buf,u32Temp+27);
         if((pOsalData->bLogError)&&(u32ErrorCode != OSAL_E_TIMEOUT)&&(u32OsalSTrace & ~0x00000020))
         {
                 vWriteToErrMem(OSAL_C_TR_CLASS_SYS_MQ,au8Buf,u32Temp+27,0);
         }
#else
         u32Temp = (tU32)OSAL_ThreadWhoAmI();
         bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,OSAL_ThreadWhoAmI());
         if(pCurrentEntry)
         {
            TraceString("OSALMQ Task:%s(ID:%d) Receive from MQ: User Handle %p Name:%s(33,32) Msg Count:%d Error:0x%x Prio:%d",
                      name,(unsigned int)u32Temp,pCurrentEntry,pCurrentEntry->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32ActualPrio);
            if((pOsalData->bLogError)&&(u32ErrorCode != OSAL_E_TIMEOUT)&&(u32OsalSTrace & ~0x00000020))vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) Receive from MQ: User Handle %p Name:%s(33,32) Msg Count:%d Error:0x%x Prio:%d \n",
                      name,(unsigned int)u32Temp,pCurrentEntry,pCurrentEntry->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32ActualPrio);
         }
         else
         {
             TraceString("OSALMQ Task:%s(ID:%d) Receive from MQ: User Handle 0x%x Name:%s(33,32) Msg Count:%d Error:0x%x Prio:%d",
                      name,(unsigned int)u32Temp,(unsigned int)u32ErrorCode,"Unknown",(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32ActualPrio);
             if((pOsalData->bLogError)&&(u32ErrorCode != OSAL_E_TIMEOUT)&&(u32OsalSTrace & ~0x00000020))vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) Receive from MQ: User Handle 0x%x Name:%s(33,32) Msg Count:%d Error:0x%x Prio:%d \n",
                      name,(unsigned int)u32Temp,(unsigned int)u32ErrorCode,"Unknown",(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32ActualPrio);
         }
#endif
         if(bTraceHandle)
         {
             if(pCurrentEntry)
             {
                TraceString("OSALMQ Rec MQ:%s OSAL User Handle:0x%x Linux Handle:%d Offset:0x%x Pid:%d Count:%d",pCurrentEntry->szName,hMQ,
                prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId,
                GetMemPoolOffset(&MqMemPoolHandle,(uintptr_t*)&hMQ),
                prLiMqMap[pCurrentEntry->u32EntryIdx].PrcId,
                prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt);
             }
             else
             {
                TraceString("OSALMQ Rec MQ:? OSAL User Handle:0x%x Linux Handle:? Offset:0x%x Pid:? Count:?",hMQ,GetMemPoolOffset(&MqMemPoolHandle, (uintptr_t*)&hMQ));
             }
         }
     }
     vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
  }
  else
  {
     if(((pCurrentEntry) &&(pCurrentEntry->bTraceCannel == TRUE))||(u32OsalSTrace & 0x00000020))
     {
#ifdef SHORT_TRACE_OUTPUT
        char au8Buf[64];
        u32Temp = (tU32)OSAL_ThreadWhoAmI();
        OSAL_M_INSERT_T8(&au8Buf[0],(tU8)OSAL_MQ_RECEIVE);
        bGetThreadNameForTID(&au8Buf[1],8,(tS32)u32Temp);
        OSAL_M_INSERT_T32(&au8Buf[9],(tU32)u32Temp);
        OSAL_M_INSERT_T32(&au8Buf[13],(tU32)&hMQ);
        OSAL_M_INSERT_T16(&au8Buf[17],(tU16)u32ActualMessage);
        OSAL_M_INSERT_T32(&au8Buf[19],u32ErrorCode);
        OSAL_M_INSERT_T32(&au8Buf[23],u32ActualPrio);
        u32Temp = (tU32)strlen(pCurrentEntry->szName);/*lint !e613 *//* pointer already checked*/ 
        memcpy (&au8Buf[27],pCurrentEntry->szName,u32Temp);/*lint !e613 *//* pointer already checked*/ 
        LLD_vTrace(OSAL_C_TR_CLASS_SYS_MQ, /*TR_LEVEL_USER_1*/TR_LEVEL_FATAL,au8Buf,u32Temp+27);
#else
        u32Temp = (tU32)OSAL_ThreadWhoAmI();
        bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,OSAL_ThreadWhoAmI());
        TraceString("OSALMQ Task:%s(ID:%d) Receive from MQ: User Handle %p( %p) Name:%s Msg Count:%d Error:0x%x Prio:%d",
                 name,(unsigned int)u32Temp,(tMQUserHandle*)hMQ,pCurrentEntry,pCurrentEntry->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32ActualPrio);
#endif
     }
  }
  return((tS32)u32CopiedBytes );
#endif
}

tS32 OSAL_s32MessageQueueWaitMonotonic(OSAL_tMQueueHandle hMQ,
                                       tPU8 pu8Buffer,
                                       tU32 u32Length,
                                       tPU32 pu32Prio,
                                       OSAL_tMSecond msec)
{ 
   tS32 s32Ret;
   OSAL_tMSecond rVal[2];
   while(1)
   {
      rVal[0] = OSAL_ClockGetElapsedTime();
      if((s32Ret = OSAL_s32MessageQueueWait(hMQ,pu8Buffer,u32Length,pu32Prio,msec)) > 0 )
      {
         /* valid Message received leave the loop*/
         break;
      }
      if(OSAL_u32ErrorCode() != OSAL_E_TIMEOUT)
      {
         /* No timeout error leave the loop */
         break;
      }
      else
      {
         rVal[1] = OSAL_ClockGetElapsedTime();
         if((rVal[1] - rVal[0]) >= msec)
         {
             /* correct timeout is happened  leave the loop */
             break;
         }
         TraceString("OSALMQ Invalid timeout %d instaead of %d",rVal[1] - rVal[0],msec);
         /* invalid timeout retry calculate new timeout */
         msec -= (rVal[1] - rVal[0]);
      }
   }
   return s32Ret;
}

tS32 OSAL_s32MessageQueuePriorityWait(OSAL_tMQueueHandle hMQ,  
                                      tPU8 pu8Buffer, tU32 u32Length, tPU32 pu32Prio, 
                                      OSAL_tMSecond msec, tU32 u32ThresholdPrio)
{
   ((tVoid)hMQ); // satisfy lint
   ((tVoid)pu8Buffer); // satisfy lint
   ((tVoid)u32Length); // satisfy lint
   ((tVoid)pu32Prio); // satisfy lint
   ((tVoid)msec); // satisfy lint
   ((tVoid)u32ThresholdPrio); // satisfy lint
   return 0;
}

void vTraceMqNotify(trMqueueElement *pCurrentEntry,tU32 u32ErrorCode)
{
   tU32 Temp = (tU32)OSAL_ThreadWhoAmI();
#ifdef SHORT_TRACE_OUTPUT
   char au8Buf[64];
   OSAL_M_INSERT_T8(&au8Buf[0],(tU8)OSAL_MQ_REGISTER);
   bGetThreadNameForTID(&au8Buf[1],8,(tS32)Temp);
   OSAL_M_INSERT_T32(&au8Buf[9],(tU32)Temp);
   OSAL_M_INSERT_T32(&au8Buf[17],u32ErrorCode);
   if(pCurrentEntry)
   {
      OSAL_M_INSERT_T32(&au8Buf[13],(tU32)pCurrentEntry->pcallback);
      Temp = strlen(pCurrentEntry->szName);
      memcpy (&au8Buf[21],pCurrentEntry->szName,Temp);
   }
   else
   {
      Temp =0;
      OSAL_M_INSERT_T32(&au8Buf[13],Temp);
      au8Buf[21] = '\0';
   }
   LLD_vTrace(OSAL_C_TR_CLASS_SYS_MQ, TR_LEVEL_FATAL,au8Buf,Temp+21);
#else
   char name[TRACE_NAME_SIZE];
   bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,(tS32)Temp);
   if(pCurrentEntry)
   {
    TraceString("Task:%s(ID:%d) Register callback %p Error:0x%x for MQ:%s",
             name,(unsigned int)Temp,pCurrentEntry->pcallback,(unsigned int)u32ErrorCode,pCurrentEntry->szName);
   }
   else
   {
     TraceString("Task:%s(ID:%d) Register callback pCurrentEntry = NULL Error:0x%x ",
               name,(unsigned int)Temp,(unsigned int)u32ErrorCode);
   }
#endif
}


/*****************************************************************************
 *
 * FUNCTION:    OSAL_s32MessageQueueNotify
 *
 * DESCRIPTION: this function informs the calling Thread as soon as an empty
 *              message box receives a new message.
 *
 * PARAMETER:   OSAL_tMQueueHandle hMQ: message queue handle
 *              OSAL_tpfCallback pCallback: callback function
 *              tPVoid pvArg: parameter of the callback function
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.10.05  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
tS32 OSAL_s32MessageQueueNotify(OSAL_tMQueueHandle hMQ,
                                OSAL_tpfCallback pCallback,
                                tPVoid pvArg)
{
#ifdef OSAL_SHM_MQ_FOR_LINUX_MQ
   return OSAL_s32ShMemMessageQueueNotify( hMQ,pCallback,pvArg);
#endif
   tS32 s32RetVal = OSAL_OK;
   tU32 u32ErrorCode   = u32CheckMqHandle(hMQ);
   tS32 s32TID;
   trMqueueElement *pCurrentEntry = NULL;

#ifdef DBUS_MQ_SUPPORT
   if((pOsalData->u32OsalMqDBus)&&(u32ErrorCode == OSAL_E_NOERROR))
   {
     if(((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->u16Type == MQ_DBUS )
     {
        TraceString("OSALMQ OSAL_s32MessageQueueNotify not allowed for DBUS MQ %s",((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->szName);
        return OSAL_E_WRONGFUNC;
     }
   }
#endif
   if(u32ErrorCode == OSAL_E_NOERROR)
   {
      /* check if callback is used , then we need callback handler task */
      if(pCallback &&((uintptr_t)pCallback != 0xffffffff))
      {
         /* check if callback handler task for this process already exists */  
         tS32 s32PidEntry = s32FindProcEntry(OSAL_ProcessWhoAmI());
         if(s32PidEntry == OSAL_ERROR)
         {
            NORMAL_M_ASSERT_ALWAYS();
         }
         else
         {
            /* check if callback handler task for this process already exists */  
            s32StartCbHdrTask(s32PidEntry);
         }
      }
      pCurrentEntry = ((tMQUserHandle*)hMQ)->pOrigin;/*lint !e613 *//* pointer already checked*/ 
/*         if((pCurrentEntry->u32EntryIdx >= pOsalData->MqueueTable.u32UsedEntries)||(pCurrentEntry->u32MqueueID != LINUX_C_U32_MQUEUE_ID))
         {
            CheckandCorrectMqStaticValues();
         }*/

         if(LockOsal(&pOsalData->MqueueTable.rLock ) == OSAL_OK)
         {
            /* check if queue isn't deleted */
            if(pCurrentEntry->bIsUsed)
            {
               s32TID = (tS32)OSAL_ThreadWhoAmI();
#ifdef USE_EVENT_AS_CB
               if((pCallback == NULL )&&(pvArg == NULL)&&(pCurrentEntry->pcallback == (OSAL_tpfCallback)0xffffffff))
               {  
                   pCurrentEntry->bNotify = FALSE;
                   (void)memset(pCurrentEntry->szEvName,0,OSAL_C_U32_MAX_NAMELENGTH);
                   pCurrentEntry->mask    = 0;
                   pCurrentEntry->enFlags = (OSAL_tenEventMaskFlag)0;
               }
               else if(((uintptr_t)pCallback == 0xffffffff)&&(pvArg != NULL))
               {
                  trMqEventInf* pEvent = (trMqEventInf*)pvArg;
                  if(strlen(pEvent->coszName) > OSAL_C_U32_MAX_NAMELENGTH)
                  {
                     u32ErrorCode = OSAL_E_NAMETOOLONG;
                     TraceString("MQ Notification event name %s is to long",pEvent->coszName);
                  }
                  else
                  {
                     (void)OSAL_szStringNCopy((tString)pCurrentEntry->szEvName,pEvent->coszName, strlen(pEvent->coszName));
                      pCurrentEntry->mask    = pEvent->mask;
                      pCurrentEntry->enFlags = pEvent->enFlags;
                      pCurrentEntry->bNotify = TRUE;
                      pCurrentEntry->pcallback    = pCallback;
                      /* prepare message processing mesurements 
                      filter out notify queues because their usage is trgggered by OSAL callbacks */
                      pCurrentEntry->u32MessagePrcessing = MEASUREMENT_SWITCHOFF;
                  } 
               }
               else//if((pCallback == NULL )&&(pvArg != NULL))
#endif
               {
                  if( pCallback == NULL )
                  {
                     if( pvArg == NULL )
                     {
#ifdef USE_EVENT_AS_CB
                        if(((tMQUserHandle*)hMQ)->hNotifyEvent != 0)
                        {
                          if(OSAL_s32EventClose(((tMQUserHandle*)hMQ)->hNotifyEvent) != OSAL_OK)
                          {
                             NORMAL_M_ASSERT_ALWAYS();
                          }
                          ((tMQUserHandle*)hMQ)->hNotifyEvent = 0;
                        }
#endif
                        if( pCurrentEntry->pcallback != NULL )
                        {
                           if(pCurrentEntry->callbacktid == s32TID )
                           {						   
#ifdef MQ_NOTIFY_VIA_OSAL
                              pCurrentEntry->bNotify = FALSE;
#else
                              if (mq_notify((mqd_t)rLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId, NULL) != -1)
                              {
#endif
                                 /*force dispatch to disable callback */
                                 OSAL_s32ThreadWait(10);
                                 pCurrentEntry->pcallback = NULL;
                                 pCurrentEntry->callbacktid  = NOTATHREAD;
                                 s32RetVal = OSAL_OK;
#ifdef MQ_NOTIFY_VIA_OSAL
#else
                              }
                              else
                              {
                                 u32ErrorCode = u32ConvertErrorCore(errno);
                              }
#endif
                           }
                           else
                           {
                              s32RetVal=OSAL_ERROR;
                              u32ErrorCode=OSAL_E_BUSY;
                           }
                        }
                        else
                        {
                           s32RetVal=OSAL_ERROR;
                           pCurrentEntry->callbacktid  = NOTATHREAD;
                        }
                     }
                     else
                     {
                        s32RetVal=OSAL_ERROR;
                        u32ErrorCode=OSAL_E_INVALIDVALUE;
                     }
                  }
                  else
                  {
                     if( pCurrentEntry->pcallback == NULL )
                     {
                        pCurrentEntry->callbackpid  = OSAL_ProcessWhoAmI();
                        pCurrentEntry->callbacktid  = s32TID;
                        pCurrentEntry->pcallback    = pCallback;
                        pCurrentEntry->pcallbackarg = pvArg;
#ifdef MQ_NOTIFY_VIA_OSAL
                        pCurrentEntry->callbackpididx = s32GetPrcIdx(pCurrentEntry->callbackpid);
                        if(pCurrentEntry->callbackpididx == -1)
                        {
                          FATAL_M_ASSERT_ALWAYS();
                        }
                        pCurrentEntry->bNotify = TRUE;
#else
                         /* prepare message processing mesurements 
                           filter out notify queues because their usage is trgggered by OSAL callbacks */
                        pCurrentEntry->u32MessagePrcessing = MEASUREMENT_SWITCHOFF;
                        struct sigevent s = { 0 };
                        s.sigev_notify = SIGEV_THREAD;
                        s.sigev_notify_function = (void (*)(sigval_t))vExecuteCallback;
                        s.sigev_notify_attributes = NULL;
                        s.sigev_value.sival_ptr = pCurrentEntry;
                        if (mq_notify((mqd_t)rLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId, &s) != -1)
                        {
                           s32RetVal = OSAL_OK;
                        }
                        else
                        {
                           u32ErrorCode = u32ConvertErrorCore(errno);
                        }
#endif
                     }
                     else
                     {
                        s32RetVal = OSAL_ERROR;
                        u32ErrorCode = OSAL_E_BUSY;
                     }
                  }//if( pCallback == NULL )
               }//if((pCallback == NULL )&&(pvArg != NULL))
            }//if(pCurrentEntry->bIsUsed)
            UnLockOsal(&pOsalData->MqueueTable.rLock);
         }
   }

   if( u32ErrorCode != OSAL_E_NOERROR )
   {
      s32RetVal = OSAL_ERROR;
      vTraceMqNotify(pCurrentEntry,u32ErrorCode);
      vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
   }
   else
   {
      if((pCurrentEntry->bTraceCannel)||(u32OsalSTrace & 0x00000020)) /*lint !e613 pointer already checked */
      {
        vTraceMqNotify(pCurrentEntry,u32ErrorCode);
      }
   }
   return(s32RetVal);
}

/*****************************************************************************
 *
 * FUNCTION: OSAL_s32MessageQueueStatus
 *
 * DESCRIPTION: this function informs the calling Thread as soon as an empty
 *              message box receives a new message.
 *
 * PARAMETER:   OSAL_tMQueueHandle hMQ: message queue handle
 *              tPU32 pu32MaxMessage:   pointer to the number of messages or
 *                                      OSAL_NULL,
 *              tPU32 pu32MaxLength:    pointer to the number to the max
 *                                      length of the message or OSAL_NULL,
 *               tPU32 pu32Message:     pointer to the number of the current
 *                                      existing messages or OSAL_NULL,
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.10.05  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
tS32 OSAL_s32MessageQueueStatus( OSAL_tMQueueHandle hMQ,
                                 tPU32 pu32MaxMessage,
                                 tPU32 pu32MaxLength,
                                 tPU32 pu32Message)
{
#ifdef OSAL_SHM_MQ_FOR_LINUX_MQ
   return OSAL_s32ShMemMessageQueueStatus( hMQ,pu32MaxMessage,pu32MaxLength,pu32Message);
#else
   tS32 s32ReturnValue = OSAL_ERROR;
   tU32 u32ErrorCode   = u32CheckMqHandle(hMQ);
   trMqueueElement *pCurrentEntry = NULL;

#ifdef DBUS_MQ_SUPPORT
  if((pOsalData->u32OsalMqDBus)&&(u32ErrorCode == OSAL_E_NOERROR))
  {
     if(((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->u16Type == MQ_DBUS )
     {
        TraceString("OSALMQ OSAL_s32MessageQueueStatus not allowed for DBUS MQ %s",((trMqueueElement*)((tMQUserHandle*)hMQ)->pOrigin)->szName);
        return OSAL_E_WRONGFUNC;
     }
  }
#endif
   if(u32ErrorCode == OSAL_E_NOERROR)
   {
      /* handle is already checked */
      pCurrentEntry = ((tMQUserHandle*)hMQ)->pOrigin;/*lint !e613 *//* pointer already checked*/ 
   }
   if(pCurrentEntry)
   {
/*         if((pCurrentEntry->u32EntryIdx >= pOsalData->MqueueTable.u32UsedEntries)||(pCurrentEntry->u32MqueueID != LINUX_C_U32_MQUEUE_ID))
         {
            CheckandCorrectMqStaticValues();
         }*/
         /* check if queue isn't deleted */
         if(pCurrentEntry->bIsUsed)
         {
            struct mq_attr mqstat = {0,0,0,0,0};
            if(mq_getattr(prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId,&mqstat) == 0)
            {
               if( pu32MaxMessage != OSAL_NULL )
                         *pu32MaxMessage = pCurrentEntry->MaxMessages = (tU32)mqstat.mq_maxmsg;
               if( pu32MaxLength  != OSAL_NULL )
                         *pu32MaxLength = (tU32)mqstat.mq_msgsize;
               if( pu32Message    != OSAL_NULL )
                         *pu32Message     = pCurrentEntry->ActualMessage = (tU32)mqstat.mq_curmsgs;
                  s32ReturnValue = OSAL_OK;
            }
            else
            {
                u32ErrorCode = u32ConvertErrorCore(errno);
            }
         }
         else
         {
            u32ErrorCode =OSAL_E_DOESNOTEXIST;
         }
   }//if(pCurrentEntry)
   else
   {
	  if(u32ErrorCode == OSAL_E_NOERROR)u32ErrorCode = OSAL_E_DOESNOTEXIST; 
   }

   if( u32ErrorCode != OSAL_E_NOERROR )
   {
      if(pCurrentEntry)
      {
           vTraceMqInfo((tMQUserHandle*)hMQ,OSAL_MQ_STATUS,TR_LEVEL_FATAL,u32ErrorCode,NULL);
      }
      else
      {
           vTraceMqInfo(NULL,OSAL_MQ_STATUS,TR_LEVEL_FATAL,u32ErrorCode,NULL);
      }
      vSetErrorCode(OSAL_C_THREAD_ID_SELF,u32ErrorCode);
   }
   else
   {
      if((pCurrentEntry->bTraceCannel)||(u32OsalSTrace & 0x00000020))/*lint !e613 pointer already checked */
      {
         vTraceMqInfo((tMQUserHandle*)hMQ,OSAL_MQ_STATUS,TR_LEVEL_FATAL,u32ErrorCode,NULL);
      }
   }
   return( s32ReturnValue );
#endif
}


//#define TRACE_STEPS
#define WAIT_FOR_READ    0x001
#define WAIT_FOR_WRITE   0x010
/*****************************************************************************
 *
 * FUNCTION: OSAL_s32CreateRingBuffer
 *
 * DESCRIPTION: this function creates a ring buffer element
 *
 * PARAMETER:   tCString            : ring buffer name
 *              tU32                : ring buffer size
 *              OSAL_tenAccess      : ring buffer access rights
 *              OSAL_tMQueueHandle* : address of ring buffer handle
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.04.16  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
#define  LINUX_C_U32_RING_BUFFER_ID                    ((tU32)0x524E4742)
 tS32 OSAL_s32CreateRingBuffer(tCString coszName, tU32 u32Size,OSAL_tenAccess enAccess,OSAL_tMQueueHandle* pHandle)
{
   OSAL_tShMemHandle hShMem = (OSAL_tShMemHandle)OSAL_ERROR;
   OSAL_tEventHandle hEvent = (OSAL_tEventHandle)OSAL_ERROR;
   OSAL_tSemHandle hSem = (OSAL_tSemHandle)OSAL_ERROR;
   OSAL_tSemHandle hSigSem = (OSAL_tSemHandle)OSAL_ERROR;
   tS32 s32ReturnValue = OSAL_ERROR;
   tU32 u32ErrorCode = OSAL_E_NOERROR;
   tOsalRgBufHandle* pRngBufHandle = (tOsalRgBufHandle*)malloc(sizeof(tOsalRgBufHandle));
   tU32 u32Lenght = 0;
   tOsalRgBufDat* pRngBufDat= NULL;
   char Name[LINUX_C_MQ_MAX_NAMELENGHT];
 
   if(pRngBufHandle)
   {
      if((pHandle)&&(u32Size)&&(coszName)&&(enAccess <= OSAL_EN_READWRITE)&&(strlen(coszName) < LINUX_C_MQ_MAX_NAMELENGHT))
      {
          /* ensure 4 Byte alignment */
          u32Lenght = u32Size % (tU32)sizeof(tU32);
          if(u32Lenght)
          {
             u32Lenght = ((tU32)sizeof(tU32) - u32Lenght) + u32Size + (tU32)sizeof(tOsalRgBufDat);
             TraceString("Align Rng Buffer Memory");
          }			
          else			
          {
            u32Lenght = u32Size + (tU32)sizeof(tOsalRgBufDat);
          }
			  
         snprintf(Name,LINUX_C_MQ_MAX_NAMELENGHT-1,"RG%s",coszName);
         hShMem = OSAL_SharedMemoryCreate(Name,enAccess,u32Lenght);
         if((s32ReturnValue = OSAL_s32EventCreateOpt(Name, &hEvent, CONSUME_EVENT|WAIT_MULTIPLE_TSK)) == OSAL_OK)
         {
            if((s32ReturnValue =  OSAL_s32SemaphoreCreate(coszName,&hSigSem,0)) == OSAL_OK) 
            {
#ifdef SHM_RG_USE_SEM
               s32ReturnValue = OSAL_s32SemaphoreCreate(Name,&hSem,1);
#else
               s32ReturnValue = OSAL_s32MutexCreate(Name,&hSem,0);
#endif
            }
         }
         if((hShMem != (OSAL_tShMemHandle)OSAL_ERROR)&&(s32ReturnValue != OSAL_ERROR))
         {
            pRngBufDat = (tOsalRgBufDat*)OSAL_pvSharedMemoryMap(hShMem,enAccess,u32Lenght,0);
            if(pRngBufDat)
            {
               pRngBufDat->u32Size          = u32Lenght;
               pRngBufDat->u32ReadOffset    = 0;
               pRngBufDat->u32WriteOffset   = 0;;
               pRngBufDat->u32OpnCnt        = 1;
               pRngBufDat->u32MemFree       = u32Size;
               pRngBufDat->u32ActualMessage = 0;               
               pRngBufDat->u32RBlock          = FALSE;               
               pRngBufDat->u32WBlock          = FALSE;               
                
               s32ReturnValue = (tS32)strlen(coszName);
               if(s32ReturnValue >= (tS32)LINUX_C_MQ_MAX_NAMELENGHT)s32ReturnValue = LINUX_C_MQ_MAX_NAMELENGHT-1;
               strncpy(pRngBufDat->szName,coszName,s32ReturnValue);
               pRngBufDat->szName[LINUX_C_MQ_MAX_NAMELENGHT-1] = 0;
               
               pRngBufHandle->StartAdress = (uintptr_t)pRngBufDat;
               pRngBufHandle->MemStart    = pRngBufHandle->StartAdress + sizeof(tOsalRgBufDat);
               pRngBufHandle->MemEnd      = pRngBufHandle->MemStart + u32Size;
               pRngBufHandle->hEvent      = hEvent;
               pRngBufHandle->hSigSem     = hSigSem;
               pRngBufHandle->hLockSem    = hSem;
               pRngBufHandle->hShMem      = hShMem;
               pRngBufHandle->u32Magic    = LINUX_C_U32_RING_BUFFER_ID;
               pRngBufHandle->enAccess    = enAccess;

               s32ReturnValue = OSAL_OK;
               *pHandle = (OSAL_tMQueueHandle)pRngBufHandle;
            }
#ifdef TRACE_STEPS
            TraceString("Create Start:0x%x End:0x%x WriteOffset:%d ReadOffset:%d Free:%d Open:%d",
                        (tU32)pRngBufHandle->MemStart,pRngBufHandle->MemEnd,pRngBufDat->u32WriteOffset,pRngBufDat->u32ReadOffset,pRngBufDat->u32MemFree,pRngBufDat->u32OpnCnt);
#endif                        
         }
      }
      else
      {
         free(pRngBufHandle);
         u32ErrorCode = OSAL_E_INVALIDVALUE;
         vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
      }
   }
   else
   {
      u32ErrorCode = OSAL_E_NOSPACE;
      vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
   }
   
   if(s32ReturnValue == OSAL_ERROR)
   {
      snprintf(Name,LINUX_C_MQ_MAX_NAMELENGHT-1,"RG%s",coszName);
      if(hShMem != (OSAL_tShMemHandle)OSAL_ERROR)
      {
         (void)OSAL_s32SharedMemoryClose(hShMem);
         (void)OSAL_s32SharedMemoryDelete(Name);
      }
      if(hEvent != (OSAL_tEventHandle)OSAL_ERROR)
      {
         (void)OSAL_s32EventClose(hEvent);
         (void)OSAL_s32EventDelete(Name);
      }
      if(hSigSem != (OSAL_tSemHandle)OSAL_ERROR)
      {
         (void)OSAL_s32SemaphoreClose(hSigSem);
         (void)OSAL_s32SemaphoreDelete(coszName);
      }

      if(hSem != (OSAL_tSemHandle)OSAL_ERROR)
	  {
 #ifdef SHM_RG_USE_SEM
         (void)OSAL_s32SemaphoreClose(hSem);
         (void)OSAL_s32SemaphoreDelete(Name);
#else
         (void)OSAL_s32MutexClose(hSem);
         (void)OSAL_s32MutexDelete(Name);
#endif
      }
   }
   if((u32OsalSTrace & 0x00000200)||(s32ReturnValue != OSAL_OK))
   {
      char name[TRACE_NAME_SIZE];
      tU32 u32Temp = (tU32)OSAL_ThreadWhoAmI();
      bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,(tS32)u32Temp);
      if(!coszName)coszName ="Unknown";
      TraceString("OSALRNG_BUF Task:%s(ID:%d) Create RingBuffer Handle %p Access:%d Error:0x%x Name:%s",
                   name,(unsigned int)u32Temp,pHandle,enAccess,u32ErrorCode,coszName);
   }
   return s32ReturnValue;
}

/*****************************************************************************
 *
 * FUNCTION: OSAL_s32OpenRingBuffer
 *
 * DESCRIPTION: this function opens an existing ring buffer element
 *
 * PARAMETER:   tCString            : ring buffer name
 *              OSAL_tenAccess      : ring buffer access rights
 *              OSAL_tMQueueHandle* : address of ring buffer handle
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.04.16  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
tS32 OSAL_s32OpenRingBuffer(tCString coszName,OSAL_tenAccess enAccess,OSAL_tMQueueHandle* pHandle)
{
   OSAL_tShMemHandle hShMem = (OSAL_tShMemHandle)OSAL_ERROR;
   OSAL_tEventHandle hEvent = (OSAL_tEventHandle)OSAL_ERROR;
   OSAL_tSemHandle hSem = (OSAL_tSemHandle)OSAL_ERROR;
   OSAL_tSemHandle hSigSem = (OSAL_tSemHandle)OSAL_ERROR;
   tS32 s32ReturnValue = OSAL_ERROR;
   tU32 u32ErrorCode = OSAL_E_NOERROR;
   tOsalRgBufHandle* pRngBufHandle = (tOsalRgBufHandle*)malloc(sizeof(tOsalRgBufHandle));
   tOsalRgBufDat*   pRngBufDat = NULL;
   void* pRet= NULL;
   char Name[LINUX_C_MQ_MAX_NAMELENGHT];
   
   if(pRngBufHandle)
   {
      if((pHandle)&&(coszName)&&(enAccess <= OSAL_EN_READWRITE))
      {
         snprintf(Name,LINUX_C_MQ_MAX_NAMELENGHT-1,"RG%s",coszName);
         hShMem = OSAL_SharedMemoryOpen(Name,enAccess);
         if((s32ReturnValue = OSAL_s32EventOpen(Name,&hEvent)) == OSAL_OK)
         {
            if((s32ReturnValue = OSAL_s32SemaphoreOpen(coszName,&hSigSem)) == OSAL_OK)
            {
#ifdef SHM_RG_USE_SEM
               s32ReturnValue = OSAL_s32SemaphoreOpen(Name,&hSem);
#else
               s32ReturnValue = OSAL_s32MutexOpen(Name,&hSem);
#endif
            }
         }
         if((hShMem != (OSAL_tShMemHandle)OSAL_ERROR)&&(s32ReturnValue != OSAL_ERROR))
         {
            pRngBufDat = (tOsalRgBufDat*)OSAL_pvSharedMemoryMap(hShMem,enAccess,sizeof(tOsalRgBufDat),0);
            if(pRngBufDat)
            {
               pRet = OSAL_pvSharedMemoryMap(hShMem,enAccess,pRngBufDat->u32Size,0);
               if((pRet)&&(pRngBufDat))
               {
                  pRngBufDat->u32OpnCnt++;
 
                  pRngBufHandle->StartAdress = (uintptr_t)pRet;
                  pRngBufHandle->MemStart    = pRngBufHandle->StartAdress + sizeof(tOsalRgBufDat);
                  pRngBufHandle->MemEnd      = pRngBufHandle->MemStart + pRngBufDat->u32Size - sizeof(tOsalRgBufDat);
                  pRngBufHandle->hEvent      = hEvent;
                  pRngBufHandle->hLockSem    = hSem;
                  pRngBufHandle->hSigSem     = hSigSem;
                  pRngBufHandle->hShMem      = hShMem;
                  pRngBufHandle->u32Magic    = LINUX_C_U32_RING_BUFFER_ID;
                  pRngBufHandle->enAccess    = enAccess;
               
                  s32ReturnValue = OSAL_OK;
                  *pHandle = (OSAL_tMQueueHandle)pRngBufHandle;
              /*    TraceString("Open Start:0x%x WriteOffset:%d ReadOffset:%d Free:%d Open:%d",
                  (tU32)pRngBufHandle->MemStart,pRngBufDat->u32WriteOffset,pRngBufDat->u32ReadOffset,pRngBufDat->u32MemFree,pRngBufDat->u32OpnCnt);*/
               }
            }
         }
      }
      else
      {
         u32ErrorCode = OSAL_E_INVALIDVALUE;
         vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
      }
   }
   else
   {
      free(pRngBufHandle);
      u32ErrorCode = OSAL_E_NOSPACE;
      vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
   }
   
   if((s32ReturnValue == OSAL_ERROR)&&(coszName))
   {
      snprintf(Name,LINUX_C_MQ_MAX_NAMELENGHT-1,"RG%s",coszName);
      if(hShMem != (OSAL_tShMemHandle)OSAL_ERROR)
      {
         (void)OSAL_s32SharedMemoryClose(hShMem);
         (void)OSAL_s32SharedMemoryDelete(Name);
      }
      if(hEvent != (OSAL_tEventHandle)OSAL_ERROR)
      {
         (void)OSAL_s32EventClose(hEvent);
         (void)OSAL_s32EventDelete(Name);
      }
      if(hSigSem != (OSAL_tSemHandle)OSAL_ERROR)
      {
         (void)OSAL_s32SemaphoreClose(hSigSem);
         (void)OSAL_s32SemaphoreDelete(coszName);
      }
      if(hSem != (OSAL_tSemHandle)OSAL_ERROR)
      {
#ifdef SHM_RG_USE_SEM
         (void)OSAL_s32SemaphoreClose(hSem);
         (void)OSAL_s32SemaphoreDelete(Name);
#else
         (void)OSAL_s32MutexClose(hSem);
         (void)OSAL_s32MutexDelete(Name);
#endif
      }
   }
   
   if((u32OsalSTrace & 0x00000200)||(s32ReturnValue != OSAL_OK))
   {
      char name[TRACE_NAME_SIZE];
      tU32 u32Temp = (tU32)OSAL_ThreadWhoAmI();
      bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,(tS32)u32Temp);
      if(!coszName)coszName =(tString)"Unknown";/*lint !e1773 */  /*otherwise linker warning */
      TraceString("OSALRNG_BUF Task:%s(ID:%d) Open RingBuffer Handle %p Access:%d Error:0x%x Name:%s",
                  name,(unsigned int)u32Temp,pHandle,enAccess,u32ErrorCode,coszName);
   }
   return s32ReturnValue;
}


/*****************************************************************************
 *
 * FUNCTION: OSAL_s32CloseRingBuffer
 *
 * DESCRIPTION: this function closes an existing ring buffer element
 *
 * PARAMETER:  OSAL_tMQueueHandle  : ring buffer handle
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.04.16  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
tS32 OSAL_s32CloseRingBuffer(OSAL_tMQueueHandle Handle)
{
   tS32 s32ReturnValue = OSAL_ERROR;
   tU32 u32ErrorCode = OSAL_E_NOERROR;
   char ObjName[LINUX_C_MQ_MAX_NAMELENGHT] = {0};
   tOsalRgBufHandle* pRngBufHandle = (tOsalRgBufHandle*)Handle;
   tOsalRgBufDat* pRngBufDat = NULL;
   
   if(pRngBufHandle && (Handle != (tU32)OSAL_ERROR) && (pRngBufHandle->u32Magic == LINUX_C_U32_RING_BUFFER_ID))
   {
      pRngBufDat = (tOsalRgBufDat*)pRngBufHandle->StartAdress;
    /*     TraceString("Write Start:0x%x WriteOffset:%d ReadOffset:%d Free:%d Open:%d",
                     (tU32)pRngBufHandle->MemStart,pRngBufDat->u32WriteOffset,pRngBufDat->u32ReadOffset,pRngBufDat->u32MemFree,pRngBufDat->u32OpnCnt);*/
      if(pRngBufDat)
      { 
          strncpy(ObjName,pRngBufDat->szName,strlen(pRngBufDat->szName));
          pRngBufDat->u32OpnCnt--;
          if(pRngBufDat->u32OpnCnt == 0)
          { 
           // OSAL_s32DeleteRingBuffer(pRngBufDat->szName);
          }
      }
      if((OSAL_s32SharedMemoryClose(pRngBufHandle->hShMem) == OSAL_OK)
       &&(OSAL_s32EventClose(pRngBufHandle->hEvent) == OSAL_OK)
       &&(OSAL_s32SemaphoreClose(pRngBufHandle->hSigSem) == OSAL_OK)
#ifdef SHM_RG_USE_SEM
       &&(OSAL_s32SemaphoreClose(pRngBufHandle->hLockSem) == OSAL_OK))
#else
       &&(OSAL_s32MutexClose(pRngBufHandle->hLockSem) == OSAL_OK))
#endif
      {
         s32ReturnValue = OSAL_OK;
      }
      free((void*)Handle);
      if(s32ReturnValue != OSAL_OK)u32ErrorCode = OSAL_u32ErrorCode();
   }
   else
   {
      u32ErrorCode = OSAL_E_INVALIDVALUE;
      vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
   }
   
   if((u32OsalSTrace & 0x00000200)||(s32ReturnValue != OSAL_OK))
   {
      char name[TRACE_NAME_SIZE];
      tU32 u32Temp = (tU32)OSAL_ThreadWhoAmI();
      bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,(tS32)u32Temp);
      if(ObjName[0] == 0)strncpy(ObjName,"Unknown",strlen("Unknown"));
      if(s32ReturnValue != OSAL_OK)u32ErrorCode = OSAL_u32ErrorCode();
      TraceString("OSALRNG_BUF Task:%s(ID:%d) Close RingBuffer Handle %p Error:0x%x Name:%s",
                  name,(unsigned int)u32Temp,Handle,u32ErrorCode,name);
   }
   return s32ReturnValue;
}

/*****************************************************************************
 *
 * FUNCTION: OSAL_s32DeleteRingBuffer
 *
 * DESCRIPTION: this function deletes an existing ring buffer element
 *
 * PARAMETER:   tCString            : ring buffer name
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.04.16  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
tS32 OSAL_s32DeleteRingBuffer(tCString coszName)
{
   tS32 s32ReturnValue = OSAL_ERROR;
   tS32 s32ErrorCode = OSAL_E_NOERROR;
   char Name[LINUX_C_MQ_MAX_NAMELENGHT];

   snprintf(Name,LINUX_C_MQ_MAX_NAMELENGHT-1,"RG%s",coszName);
   if((OSAL_s32SharedMemoryDelete(Name) == OSAL_OK)
    &&(OSAL_s32EventDelete(Name) == OSAL_OK)
    &&(OSAL_s32SemaphoreDelete(coszName) == OSAL_OK)
#ifdef SHM_RG_USE_SEM
    &&(OSAL_s32SemaphoreDelete(Name) == OSAL_OK))
#else
    &&(OSAL_s32MutexDelete(Name) == OSAL_OK))
#endif
    {
         s32ReturnValue = OSAL_OK;
    }
    else
    {
      if(OSAL_E_DOESNOTEXIST == OSAL_u32ErrorCode())
      {
         s32ReturnValue = OSAL_OK;
      }
    }

   if((u32OsalSTrace & 0x00000200)||(s32ReturnValue != OSAL_OK))
   {
      char name[TRACE_NAME_SIZE];
      tU32 u32Temp = (tU32)OSAL_ThreadWhoAmI();
      bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,(tS32)u32Temp);
      if(coszName == NULL)coszName =(tString)"Unknown";/*lint !e1773 */  /*otherwise linker warning */
      if(s32ReturnValue == OSAL_ERROR)s32ErrorCode = OSAL_u32ErrorCode();
      TraceString("OSALRNG_BUF Task:%s(ID:%d) Delete RingBuffer Error:%d Name:%s",
                  name,(unsigned int)u32Temp,s32ErrorCode,coszName);
   }

   return s32ReturnValue;
}

void EnterRgBufCriticalSection( tOsalRgBufHandle* pRngBufHandle)
{
#ifdef SHM_RG_USE_SEM
   (void)OSAL_s32SemaphoreWait(pRngBufHandle->hLockSem,OSAL_C_TIMEOUT_FOREVER);
#else
   (void)OSAL_s32MutexLock(pRngBufHandle->hLockSem,OSAL_C_TIMEOUT_FOREVER);
#endif
}

void LeaveRgBufCriticalSection( tOsalRgBufHandle* pRngBufHandle)
{
#ifdef SHM_RG_USE_SEM
  (void)OSAL_s32SemaphorePost(pRngBufHandle->hLockSem);
#else
  (void)OSAL_s32MutexUnLock(pRngBufHandle->hLockSem);
#endif
}

/*****************************************************************************
 *
 * FUNCTION: OSAL_u32WriteToRingBuffer
 *
 * DESCRIPTION: this function writes to an existing ring buffer element
 *
 * PARAMETER:  OSAL_tMQueueHandle  : ring buffer handle
 *             char*               : buffer to copy to ringbuffer    
 *             tU32                : size of the buffer
 *             tU32                : timeout value
 *
 * RETURNVALUE: number of written bytes.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.04.16  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
tU32 OSAL_u32WriteToRingBuffer(OSAL_tMQueueHandle Handle,char* ps8Buffer, tU32 Size,tU32 u32TimeOut)
{
   tS32 s32ReturnValue = OSAL_OK;
   tU32 u32ErrorCode = OSAL_E_NOERROR;
   tOsalRgBufHandle* pRngBufHandle = (tOsalRgBufHandle*)Handle;
   tOsalRgBufDat* pRngBufDat = NULL;
   tBool bTrigger = FALSE;
   tU32 Size1,Size2;
   uintptr_t* pTmp;
   tU32 u32Mask = 0;
   
   if(pRngBufHandle && (Handle != (tU32)OSAL_ERROR) && (pRngBufHandle->u32Magic == LINUX_C_U32_RING_BUFFER_ID))
   {
      if(pRngBufHandle->enAccess == OSAL_EN_READONLY)
      {
         Size = 0;
         u32ErrorCode = OSAL_E_NOPERMISSION;
      }
      else
      {
         pRngBufDat = (tOsalRgBufDat*)pRngBufHandle->StartAdress;
         /* check if enough space available otherwise wait */
         while(1)
         {
            if(pRngBufDat->u32MemFree < (Size+(2*(tU32)sizeof(tU32))))
            {
               EnterRgBufCriticalSection(pRngBufHandle);
               if(pRngBufDat->u32MemFree < (Size+(2*(tU32)sizeof(tU32))))
               {   
                  pRngBufDat->u32WBlock++;
                  LeaveRgBufCriticalSection(pRngBufHandle);
				  
                  s32ReturnValue = OSAL_s32EventWait(pRngBufHandle->hEvent,WAIT_FOR_WRITE,OSAL_EN_EVENTMASK_OR,u32TimeOut,&u32Mask);
                 
                  EnterRgBufCriticalSection(pRngBufHandle);
                  pRngBufDat->u32WBlock--;
                  if(pRngBufDat->u32MemFree > (Size+(2*(tU32)sizeof(tU32))))bTrigger = TRUE;           
                  LeaveRgBufCriticalSection(pRngBufHandle);
                  
                  if(s32ReturnValue == OSAL_ERROR)
                  {
                     u32ErrorCode = OSAL_E_TIMEOUT;
                     u32ErrorCode = OSAL_E_NOSPACE;
                     TraceString("Write RngBuf:%s timeout:%d  %d memory free",pRngBufDat->szName,u32TimeOut,pRngBufDat->u32MemFree);
                     break;
                  }
                  else
                  {
                     if(bTrigger == TRUE)
                     {
                        bTrigger = FALSE;
                        break;
                     }
                     // else repeat
                     bTrigger = FALSE;
                  }
               }
               else
               {
                  LeaveRgBufCriticalSection(pRngBufHandle);
                  break;
               }
            }
            else
            {
               break;
            }
         }//while
      }
   }
   else
   {
      u32ErrorCode = OSAL_E_INVALIDVALUE;
   }
   /* check if any error happened otherwise enter the message */
   if(u32ErrorCode == OSAL_E_NOERROR)
   {
      tU32 u32ReleasedMem = 0;
      EnterRgBufCriticalSection(pRngBufHandle);
#ifdef TRACE_STEPS
      TraceString("Write1 Start:0x%x WriteOffset:%d ReadOffset:%d Free:%d Msg:%d Open:%d ",
                   (tU32)pRngBufHandle->MemStart,pRngBufDat->u32WriteOffset,pRngBufDat->u32ReadOffset,
                   pRngBufDat->u32MemFree,pRngBufDat->u32ActualMessage,pRngBufDat->u32OpnCnt);
#endif
      if(pRngBufDat->u32MemFree >= (Size+(2*(tU32)sizeof(tU32))))
      {
         pRngBufDat->u32ActualMessage++;
         if(pRngBufDat->u32ActualMessage == 1)
         {
            bTrigger = TRUE;
         }
         Size1 = (tS32)(pRngBufHandle->MemEnd - (pRngBufHandle->MemStart + pRngBufDat->u32WriteOffset));     
         /* check for storing size info */
         if(Size1 < (tU32)sizeof(tU32))
         {
            pRngBufDat->u32WriteOffset = 0;
            u32ReleasedMem += Size1;
         }
         pTmp = (uintptr_t*)(pRngBufHandle->MemStart + pRngBufDat->u32WriteOffset);
#ifdef TRACE_STEPS
         TraceString("Write2 Start:0x%x WriteOffset:%d ReadOffset:%d Free:%d Msg:%d Open:%d ",
                     (tU32)pRngBufHandle->MemStart,pRngBufDat->u32WriteOffset,pRngBufDat->u32ReadOffset,
                     pRngBufDat->u32MemFree,pRngBufDat->u32ActualMessage,pRngBufDat->u32OpnCnt);
#endif
         tU32* pu32Tmp = (tU32*)pTmp;
         *pu32Tmp = Size;
         pu32Tmp++;
         u32ReleasedMem += (tU32)sizeof(tU32);
	     pRngBufDat->u32WriteOffset += (tU32)sizeof(tU32);
         if((pRngBufHandle->MemStart + pRngBufDat->u32WriteOffset + Size) <= pRngBufHandle->MemEnd)
         {
            memcpy((char*)(pRngBufHandle->MemStart + pRngBufDat->u32WriteOffset),ps8Buffer,Size);
            pRngBufDat->u32WriteOffset += Size;
            if((pRngBufHandle->MemStart + pRngBufDat->u32WriteOffset) == pRngBufHandle->MemEnd)
            {
               pRngBufDat->u32WriteOffset = 0;
            }
         } 
         else
         {
            Size1 = (tU32)(pRngBufHandle->MemEnd - (pRngBufHandle->MemStart + pRngBufDat->u32WriteOffset));
            memcpy((char*)(pRngBufHandle->MemStart + pRngBufDat->u32WriteOffset),ps8Buffer,Size1);
            pRngBufDat->u32WriteOffset = 0;
            Size2 = Size -Size1;
            memcpy((char*)(pRngBufHandle->MemStart),(char*)(ps8Buffer+Size1),Size2);
            pRngBufDat->u32WriteOffset += Size2;
         }
         u32ReleasedMem +=Size;
       
         /* calculate 4 Byte alignment */       
         Size1 = Size % (tU32)sizeof(tU32);
         if(Size1)
         {
            Size2 = (tU32)sizeof(tU32) - Size1;
            u32ReleasedMem += Size2;
            if((pRngBufHandle->MemStart + pRngBufDat->u32WriteOffset + Size2) <= pRngBufHandle->MemEnd)
            {
               pRngBufDat->u32WriteOffset += Size2;
               if((pRngBufHandle->MemStart + pRngBufDat->u32WriteOffset) == pRngBufHandle->MemEnd)
               {
                  pRngBufDat->u32WriteOffset = 0;
               }
            }
            else
            {
               TraceString("Write Alignment not expected");
               pRngBufDat->u32WriteOffset = 0;
            }
         }
         pRngBufDat->u32MemFree -= u32ReleasedMem;
#ifdef TRACE_STEPS
         TraceString("Write3 Start:0x%x WriteOffset:%d ReadOffset:%d Free:%d Msg:%d Open:%d ",
                     (tU32)pRngBufHandle->MemStart,pRngBufDat->u32WriteOffset,pRngBufDat->u32ReadOffset,
                     pRngBufDat->u32MemFree,pRngBufDat->u32ActualMessage,pRngBufDat->u32OpnCnt);
#endif
         if(pRngBufDat->u32RBlock > 0)
         { 
            s32ReturnValue = OSAL_s32SemaphorePost(pRngBufHandle->hSigSem);
         }
      }
      else
      {
         TraceString("OSALRNG_BUF TasID:%d) Write To RingBuffer failed due race condition",OSAL_ThreadWhoAmI());
         u32ErrorCode = OSAL_E_NOSPACE;
         s32ReturnValue = OSAL_ERROR;
      }
      LeaveRgBufCriticalSection(pRngBufHandle);
   }
   else
   {
      s32ReturnValue = OSAL_ERROR;
   }
   
   if((s32ReturnValue == OSAL_OK)&&(bTrigger == TRUE)&&(pRngBufDat)&&(pRngBufDat->bNotify == TRUE)
    &&(pRngBufDat->callbackpididx < (tS32)pOsalData->u32MaxNrProcElements)&&(pRngBufDat->pcallback )&&(pRngBufDat->callbackpididx >= 0))
   {
        if(pRngBufDat->callbackpid == OSAL_ProcessWhoAmI()) // check if process is the same
        {
             pRngBufDat->pcallback(pRngBufDat->pcallbackarg);
        }
        else
        {
           tOsalMbxMsg Msg;
           Msg.rOsalMsg.Cmd    = MBX_RB_NOTIFY;
           Msg.rOsalMsg.ID     = pRngBufDat->callbackpid;
           Msg.rOsalMsg.pFunc  = pRngBufDat->pcallback;
           Msg.rOsalMsg.pArg   = pRngBufDat->pcallbackarg;

           if(phMQCbPrc[pRngBufDat->callbackpididx] == 0)
           {
              phMQCbPrc[pRngBufDat->callbackpididx] = GetPrcLocalMsgQHandle(pRngBufDat->callbackpid);
           }
           if(OSAL_s32MessageQueuePost(phMQCbPrc[pRngBufDat->callbackpididx],(tPCU8)&Msg,(tU32)sizeof(tOsalMbxMsg),0) == OSAL_ERROR)
           {
             NORMAL_M_ASSERT_ALWAYS();
           }
        }
   }

   if((u32OsalSTrace & 0x00000200)||(s32ReturnValue != OSAL_OK))
   {
      char name[TRACE_NAME_SIZE];
      tString coszName = NULL;
      tU32 u32Temp = (tU32)OSAL_ThreadWhoAmI();
      bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,(tS32)u32Temp);
      tU32 u32Val1 = 0;
      tU32 u32Val2 = 0;
      if(u32ErrorCode != OSAL_E_NOERROR)vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
   
      if(pRngBufDat)
      {
         coszName = pRngBufDat->szName;
         u32Val1 = pRngBufDat->u32MemFree;
         u32Val2 = pRngBufDat->u32ActualMessage;
      }
      if(coszName == NULL)coszName =(tString)"Unknown";/*lint !e1773 */  /*otherwise linker warning */
      TraceString("OSALRNG_BUF Task:%s(ID:%d) Write To RingBuffer Handle %p Bytes:%d Error:0x%x Name:%s Mem:%d Msg:%d",
                  name,(unsigned int)u32Temp,Handle,(tU32)Size,u32ErrorCode,coszName,u32Val1,u32Val2);
  }

  return Size;   
}

/*****************************************************************************
 *
 * FUNCTION: OSAL_u32ReadFromRingBuffer
 *
 * DESCRIPTION: this function reads from an existing ring buffer element
 *
 * PARAMETER:  OSAL_tMQueueHandle  : ring buffer handle
 *             char*               : buffer to copy to ringbuffer    
 *             tU32                : size of the buffer
 *             tU32                : timeout value
 *
 * RETURNVALUE: number of read bytes.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.04.16  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
tU32 OSAL_u32ReadFromRingBuffer(OSAL_tMQueueHandle Handle,char* ps8Buffer, tU32 BufSize,tU32 u32TimeOut)
{
   tS32 s32ReturnValue = OSAL_OK;
   tU32 u32ErrorCode = OSAL_E_NOERROR;
   tU32 Size = 0;
   tOsalRgBufHandle* pRngBufHandle = (tOsalRgBufHandle*)Handle;
   tOsalRgBufDat* pRngBufDat = NULL;
   tBool bTrigger = FALSE;

   if(pRngBufHandle && (Handle != (tU32)OSAL_ERROR) && (pRngBufHandle->u32Magic == LINUX_C_U32_RING_BUFFER_ID))
   {
      if(pRngBufHandle->enAccess == OSAL_EN_WRITEONLY)
      {
         u32ErrorCode = OSAL_E_NOPERMISSION;
      }
      else
      {
         pRngBufDat = (tOsalRgBufDat*)pRngBufHandle->StartAdress;
         /* check for available data*/
         while(1)
         {
            bTrigger = FALSE;
            u32ErrorCode = OSAL_E_NOERROR;
            if(pRngBufDat->u32ActualMessage == 0) 
            {
               if(u32TimeOut == OSAL_C_TIMEOUT_NOBLOCKING)
               {
                  u32ErrorCode = OSAL_E_TIMEOUT;
                  break;
               }
               else
               {
                  EnterRgBufCriticalSection(pRngBufHandle);
                  if(pRngBufDat->u32ActualMessage > 0) 
                  {
                     LeaveRgBufCriticalSection(pRngBufHandle);
                     break;
                  }
                  else
                  {
                     pRngBufDat->u32RBlock++;				 
                     LeaveRgBufCriticalSection(pRngBufHandle);
                  }
                  s32ReturnValue = OSAL_s32SemaphoreWait(pRngBufHandle->hSigSem,u32TimeOut);
              
                  EnterRgBufCriticalSection(pRngBufHandle);                 
                  pRngBufDat->u32RBlock--;
                  if(pRngBufDat->u32ActualMessage > 0)bTrigger = TRUE;
                  LeaveRgBufCriticalSection(pRngBufHandle);
				  
                  if((s32ReturnValue == OSAL_ERROR)/*&&((End - Start) < u32TimeOut)*/)
                  {
                     u32ErrorCode = OSAL_E_TIMEOUT;
                    //  if(u32OsalSTrace & 0x00000200)TraceString("Wait1 ShMem MQ:%s timeout %d -> %d messages in Queue",pCurrentEntry->szName,u32TimeOut,pCurrentEntry->ActualMessage);
                     break;
                  }
                  else
                  {
                     if(bTrigger == TRUE)
                     {
                        bTrigger = 0;
                        break;
                     }
                     else
                     {
                        if(pRngBufDat->u32WBlock > 0)
                        { 
                           s32ReturnValue = OSAL_s32EventPost(pRngBufHandle->hEvent,WAIT_FOR_WRITE,OSAL_EN_EVENTMASK_OR);
						   OSAL_s32ThreadWait(0);//force scheduling of writing task
                        }
                     }
                     // repeat again no message in ringbuffer
                  }
               }
            }
            else
            {
               break;
            }
         }//while
      }
   }
   else
   {
      u32ErrorCode = OSAL_E_INVALIDVALUE;
   }
   
   if(u32ErrorCode == OSAL_E_NOERROR)
   {
      tU32 u32ReleasedMem = 0;

      EnterRgBufCriticalSection(pRngBufHandle);
      if(pRngBufDat->u32ActualMessage == 0) 
      {
         NORMAL_M_ASSERT_ALWAYS();
      }
      tU32 Size1,Size2;
      uintptr_t* pTmp;
      Size1 = (tU32)(pRngBufHandle->MemEnd - (pRngBufHandle->MemStart + pRngBufDat->u32ReadOffset));
      /* check for gap at end of memory */
      if(Size1 < sizeof(tU32))
      {
         u32ReleasedMem += Size1;
         pRngBufDat->u32ReadOffset = 0;
      }
      pTmp = (uintptr_t*)(pRngBufHandle->MemStart + pRngBufDat->u32ReadOffset);
#ifdef TRACE_STEPS
      TraceString("Read1 Start:0x%x WriteOffset:%d ReadOffset:%d Free:%d Msg:%d Open:%d ",
                  (tU32)pRngBufHandle->MemStart,pRngBufDat->u32WriteOffset,pRngBufDat->u32ReadOffset,
                  pRngBufDat->u32MemFree,pRngBufDat->u32ActualMessage,pRngBufDat->u32OpnCnt);
#endif
      /* Get size info , go to data and upddate management data */
      tU32* pu32Tmp = (tU32*)pTmp;
      if(*pu32Tmp > pRngBufDat->u32Size)
      {
         TraceString("Detected entry size %d is bigger than ringbuffer size %d ",*pu32Tmp,pRngBufDat->u32Size);
         OSAL_vProcessExit();
      }
      Size = *pu32Tmp;
      pu32Tmp++;
      u32ReleasedMem += (tU32)sizeof(tU32);
      pRngBufDat->u32ReadOffset += (tU32)sizeof(tU32);
   
      if(BufSize < Size)
      {
         TraceString("Read Start:0x%x WriteOffset:%d ReadOffset:%d Free:%d Msg:%d Open:%d ",
                     (tU32)pRngBufHandle->MemStart,pRngBufDat->u32WriteOffset,pRngBufDat->u32ReadOffset,
                     pRngBufDat->u32MemFree,pRngBufDat->u32ActualMessage,pRngBufDat->u32OpnCnt);
         TraceString("Incomplete msg from ringbuffer due Buffer RecBuf Size:%d < MsgSize:%d",BufSize,Size);  
         OSAL_vProcessExit();
      }
#ifdef TRACE_STEPS
      TraceString("Read2 Start:0x%x WriteOffset:%d ReadOffset:%d Free:%d Msg:%d Open:%d ",
                  (tU32)pRngBufHandle->MemStart,pRngBufDat->u32WriteOffset,pRngBufDat->u32ReadOffset,
                  pRngBufDat->u32MemFree,pRngBufDat->u32ActualMessage,pRngBufDat->u32OpnCnt);
#endif
      if((pRngBufHandle->MemStart + pRngBufDat->u32ReadOffset + Size) <= pRngBufHandle->MemEnd)
      {
         memcpy(ps8Buffer,(char*)(pRngBufHandle->MemStart + pRngBufDat->u32ReadOffset),Size);
         pRngBufDat->u32ReadOffset += Size;
         u32ReleasedMem            += Size;
         if((pRngBufHandle->MemStart + pRngBufDat->u32ReadOffset) == pRngBufHandle->MemEnd)
         {
            pRngBufDat->u32ReadOffset = 0;
         }
      } 
      else
      {
         Size1 = (tU32)(pRngBufHandle->MemEnd - (pRngBufHandle->MemStart + pRngBufDat->u32ReadOffset));
         memcpy(ps8Buffer,(char*)(pRngBufHandle->MemStart + pRngBufDat->u32ReadOffset),Size1);
         pRngBufDat->u32ReadOffset = 0;
         u32ReleasedMem += Size1;
         Size2 = Size -Size1;
         memcpy((char*)(ps8Buffer+Size1),(char*)(pRngBufHandle->MemStart),Size2);
         u32ReleasedMem            += Size2;
         pRngBufDat->u32ReadOffset += Size2;
      }
      /* calculate 4 Byte alignment */
      Size1 = Size % (tU32)sizeof(tU32);
      if(Size1)
      {
         Size2 = (tU32)sizeof(tU32) - Size1;
         u32ReleasedMem += Size2;
         if((pRngBufHandle->MemStart + pRngBufDat->u32ReadOffset + Size2) <= pRngBufHandle->MemEnd)
         {
            pRngBufDat->u32ReadOffset += Size2;
            if((pRngBufHandle->MemStart + pRngBufDat->u32ReadOffset) == pRngBufHandle->MemEnd)
            {
               pRngBufDat->u32ReadOffset = 0;
            }
         }
         else
         {
            TraceString("Read Alignment not expected");
            pRngBufDat->u32ReadOffset = 0;
         }
      }
      pRngBufDat->u32ActualMessage--;
      pRngBufDat->u32MemFree += u32ReleasedMem;
#ifdef TRACE_STEPS
      TraceString("Read3 Start:0x%x WriteOffset:%d ReadOffset:%d Free:%d Msg:%d Open:%d ",
                  (tU32)pRngBufHandle->MemStart,pRngBufDat->u32WriteOffset,pRngBufDat->u32ReadOffset,
                  pRngBufDat->u32MemFree,pRngBufDat->u32ActualMessage,pRngBufDat->u32OpnCnt);
#endif
 //     if(pRngBufDat->u32WBlock > 0)
      { 
         s32ReturnValue = OSAL_s32EventPost(pRngBufHandle->hEvent,WAIT_FOR_WRITE,OSAL_EN_EVENTMASK_OR);
      }
      LeaveRgBufCriticalSection(pRngBufHandle);
   }
   else
   {
	   s32ReturnValue = OSAL_ERROR;
   }
   
   if((u32OsalSTrace & 0x00000200)||(s32ReturnValue != OSAL_OK))
   {
      char name[TRACE_NAME_SIZE];
      tString coszName = NULL;
      tU32 u32Temp = (tU32)OSAL_ThreadWhoAmI();
      tU32 u32Val1 = 0;
	  tU32 u32Val2 = 0;
      bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,(tS32)u32Temp);
      if(u32ErrorCode != OSAL_E_NOERROR)vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
      
      if(pRngBufDat)
      {
         coszName = pRngBufDat->szName;
         u32Val1 = pRngBufDat->u32MemFree;
         u32Val2 = pRngBufDat->u32ActualMessage;
      }
      if(coszName == NULL)coszName =(tString)"Unknown";/*lint !e1773 */  /*otherwise linker warning */
      TraceString("OSALRNG_BUF Task:%s(ID:%d) Read from RingBuffer Handle %p Bytes:%d Error:0x%x Name:%s Mem:%d Msg:%d",
                  name,(unsigned int)u32Temp,Handle,(tU32)Size,u32ErrorCode,coszName,u32Val1,u32Val2);
   }
   return Size;   
}

/*****************************************************************************
 *
 * FUNCTION: OSAL_u32RingBufferNotify
 *
 * DESCRIPTION: this function enter a callback function an existing ring buffer 
 *              element for notify for available message
 *
 * PARAMETER:  OSAL_tMQueueHandle  : ring buffer handle
 *             OSAL_tpfCallback    : callback function for the ringbuffer    
 *             tPVoid              : argument for callback function
 *
  * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
*
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.04.16  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
tS32 OSAL_u32RingBufferNotify(OSAL_tMQueueHandle Handle,OSAL_tpfCallback pCallback,tPVoid pvArg)
{
   tOsalRgBufHandle* pRngBufHandle = (tOsalRgBufHandle*)Handle;
   tS32 s32ReturnValue = OSAL_ERROR;
   tU32 u32ErrorCode = OSAL_E_NOERROR;
   tOsalRgBufDat* pRngBufDat = NULL;
   
   /* check if callback handler task for this process already exists */  
   tS32 s32PidEntry = s32FindProcEntry(OSAL_ProcessWhoAmI());
   if(s32PidEntry == OSAL_ERROR)
   {
       NORMAL_M_ASSERT_ALWAYS();
   }
   else
   {
      /* check if callback handler task for this process already exists */  
      s32StartCbHdrTask(s32PidEntry);
   }
   if(pRngBufHandle && (Handle != (tU32)OSAL_ERROR)  && (pRngBufHandle->u32Magic == LINUX_C_U32_RING_BUFFER_ID))
   {
      pRngBufDat = (tOsalRgBufDat*)pRngBufHandle->StartAdress;
      if(pRngBufDat)
      {
         pRngBufDat->callbackpid    = OSAL_ProcessWhoAmI();
         pRngBufDat->callbackpididx = s32GetPrcIdx(pRngBufDat->callbackpid);
         if(pRngBufDat->callbackpididx == -1)
         {
           FATAL_M_ASSERT_ALWAYS();
         }
         pRngBufDat->pcallback      = pCallback;
         pRngBufDat->pcallbackarg   = pvArg;
         pRngBufDat->bNotify        = TRUE;
         s32ReturnValue             = OSAL_OK;
      }
      else
      {
         u32ErrorCode = OSAL_E_INVALIDVALUE;
      }
   }
   else
   {
      u32ErrorCode = OSAL_E_BADFILEDESCRIPTOR;
   }
   
   if((u32OsalSTrace & 0x00000200)||(s32ReturnValue != OSAL_OK))
   {
      char name[TRACE_NAME_SIZE];
      tString coszName = NULL;
      tU32 u32Temp = (tU32)OSAL_ThreadWhoAmI();
      bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,(tS32)u32Temp);
      if((s32ReturnValue != OSAL_OK))vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
      
      if(pRngBufDat)
      {
         coszName = pRngBufDat->szName;
      }
      if(coszName == NULL)coszName =(tString)"Unknown";/*lint !e1773 */  /*otherwise linker warning */
      TraceString("OSALRNG_BUF Task:%s(ID:%d) RingBuffer Notify Handle %p Error:0x%x Name:%s",
                  name,(unsigned int)u32Temp,Handle,u32ErrorCode,coszName);
   }
   return s32ReturnValue;   
   
}

/*****************************************************************************
 *
 * FUNCTION: OSAL_s32RingBufferStatus
 *
 * DESCRIPTION: this function enter a callback function an existing ring buffer 
 *              element for notify for available message
 *
 * PARAMETER:  OSAL_tMQueueHandle  : ring buffer handle
 *             tPU32               : Adress to store number of elements in the ringbuffer   
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.04.16  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
tS32 OSAL_s32RingBufferStatus( OSAL_tMQueueHandle Handle,tPU32 pu32Message)
{
   tOsalRgBufHandle* pRngBufHandle = (tOsalRgBufHandle*)Handle;
   tS32 s32ReturnValue = OSAL_ERROR;
   tU32 u32ErrorCode = OSAL_E_NOERROR;
   tOsalRgBufDat* pRngBufDat = NULL;

   if(pRngBufHandle && (Handle != (tU32)OSAL_ERROR)  && (pRngBufHandle->u32Magic == LINUX_C_U32_RING_BUFFER_ID))
   {
      pRngBufDat = (tOsalRgBufDat*)pRngBufHandle->StartAdress;
      if(pRngBufDat)
      {
         if( pu32Message    != OSAL_NULL )
            *pu32Message     = pRngBufDat->u32ActualMessage;
         s32ReturnValue = OSAL_OK;
      }
      else
      {
         u32ErrorCode = OSAL_E_INVALIDVALUE;
      }
   }
   else
   {
      u32ErrorCode=OSAL_E_BADFILEDESCRIPTOR;
   } 

   if((u32OsalSTrace & 0x00000200)||(s32ReturnValue != OSAL_OK))
   {
      char name[TRACE_NAME_SIZE];
      tString coszName = NULL;
      tU32 u32Temp = (tU32)OSAL_ThreadWhoAmI();
      tU32 u32ActualMessage = 0xffffffff;
      bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,(tS32)u32Temp);
      if(pRngBufDat)
      {
         coszName = pRngBufDat->szName;
         u32ActualMessage = pRngBufDat->u32ActualMessage;
      }
      if(coszName == NULL)coszName =(tString)"Unknown";/*lint !e1773 */  /*otherwise linker warning */
      if((s32ReturnValue != OSAL_OK))vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
      TraceString("OSALRNG_BUF Task:%s(ID:%d) RingBuffer Status Handle %p Error:0x%x Msg:%d Name:%s",
                  name,(unsigned int)u32Temp,Handle,u32ErrorCode,u32ActualMessage,coszName);
   }
   return( s32ReturnValue );
}


#ifdef OSAL_SHM_MQ

static tS32 s32StoreToMsgBox( trMqueueElement *pMq,
                              tMQUserHandle*  pHandle,
                              tU32            MsgPriority,
                              tU16            MsgSize,
                              tPCVoid         pMsgBuffer )
{
   MsgHdr *pStartHdr, *pEmptyHdr, *pPriorityHdr, *prTemp = NULL;
   tS32 s32ReturnValue = OSAL_ERROR;
   tU32 MsgLoop, MsgOffset;
   tBool bEmptyFound = FALSE;

   /*********************************************/
   /* Search an Emty Box in the MsgBox */
   /************************************ ********/
   MsgLoop = pMq->MaxMessages;
   /* Bytes Offset between two Messages */
   MsgOffset = (tU32)sizeof(struct MsgHdr);
   MsgOffset += (pMq->MaxLength );
   /* from the base of MsgBox untill an empty item is found */
   pStartHdr = (MsgHdr*)pHandle->pAdress;
   if (!pStartHdr) 
   {
       NORMAL_M_ASSERT_ALWAYS();
       return (OSAL_ERROR);
   }
   pEmptyHdr = pStartHdr;
   while( MsgLoop-- )
   {
      /* if the item is empty then break */
      if( pEmptyHdr->msgsize == 0 )
      {
         bEmptyFound = TRUE;
         break;
      }
      /* else go to the next item */
      pEmptyHdr = (MsgHdr *)((uintptr_t)pEmptyHdr + MsgOffset); /*lint !e826 */
   }

   /****************************************************************/
   /* if the item is really empty --> store the incoming message */
   /****************************************************************/
   if( bEmptyFound )
   {
      tPCU8 pRet;
      /* Fill the MsgHdr and copy from the incoming buffer the message */
      pEmptyHdr->msgsize   = MsgSize;
      pEmptyHdr->NextIndex = (uintptr_t)-1;
      if(pMq->bTraceCannel == TRUE)TraceString("OSALMQ SHM s32StoreToMsgBox %s Msg %p  msgsize:%d Prio:%d Idx: %p ",pMq->szName,pEmptyHdr->msgsize,MsgPriority,pEmptyHdr->NextIndex);
      pRet = (tPCU8)OSAL_pvMemoryCopy (&pEmptyHdr->Message[0], pMsgBuffer, MsgSize);
      if( pRet )
      {
         /* update the header list on priority base */
         pPriorityHdr = (MsgHdr *)pMq->PrioMsgListOffset[MsgPriority];
         if(pPriorityHdr == (MsgHdr*)-1)  // emty?
         {  
            pMq->PrioMsgListOffset[MsgPriority] = (uintptr_t)pEmptyHdr - (uintptr_t)pStartHdr;
         }
         else
         {
            // calculate the real memory offset, now
            pPriorityHdr = (MsgHdr *)((uintptr_t)pStartHdr + (uintptr_t)pPriorityHdr);
            /* look for last element in priority list */
            while( pPriorityHdr->NextIndex != (uintptr_t)-1 )
            {
               prTemp = (MsgHdr *)((uintptr_t)pStartHdr + (uintptr_t)(pPriorityHdr->NextIndex));
               pPriorityHdr = prTemp;
            }
            pPriorityHdr->NextIndex = (uintptr_t)pEmptyHdr - (uintptr_t)pStartHdr;
         }
         s32ReturnValue = OSAL_OK;
      }
      else
      {
         TraceString("OSALMQ SHM s32StoreToMsgBox OSAL_pvMemoryCopy failed");
      }
   }
   else
   {
      TraceString("OSALMQ s32StoreToMsgBox %s not entry found, %d Entries occupied",pMq->szName,pMq->ActualMessage);
   }
//   vTraceHexDump("OSALMQ s32StoreToMsgBox",(tPCU8)pStartHdr,pMq->MaxMessages*pMq->MaxLength);
   return( s32ReturnValue );
}

/*****************************************************************************
*
* FUNCTION:    GetMsgFromMsgBox
*
* DESCRIPTION: This function get a message contentof an OSAL message queue based on IOSC
*
* PARAMETER:     trMqueueElement *  pointer to MQ management structure
*                         tU32  search priotrity for message
*                         tU32* pointer to store message size info
*
* RETURNVALUE: void*  pointer to message
* HISTORY:
* Date      |   Modification                         | Authors
* 03.06.10  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
static void* GetMsgFromMsgBox ( trMqueueElement *pMq,
                                tMQUserHandle*  pHandle,
                                tU32 SearchedPrio,
                                tPU32 pActualSize )
{
   MsgHdr *pPriorityHdr;
   tPVoid pMessage = NULL;
   MsgHdr* prTemp;
   *pActualSize = 0;
//   vTraceHexDump("OSALMQ GetMsgFromMsgBox",(tPCU8)pHandle->pAdress,pMq->MaxMessages*pMq->MaxLength);
   /* go in the Priority organised Header List to the searched priority */
   pPriorityHdr = (MsgHdr *)(pMq->PrioMsgListOffset[SearchedPrio]);
   /* if the list is not empty copy ActualSize and address of the message */
   if(pPriorityHdr != (MsgHdr*)-1 )
   {
      /* calculate the real memory pointer, now */
      prTemp = (MsgHdr*)pHandle->pAdress; // get start adr of whole msg mem      
      if (!prTemp) 
      {
          NORMAL_M_ASSERT_ALWAYS();
          return (NULL);
      }
      prTemp = (MsgHdr*)((uintptr_t)prTemp + (uintptr_t)pPriorityHdr);
      if(prTemp->msgsize == 0)
      {
          TraceString("OSALMQ SHM GetMsgFromMsgBox %s with msgsize == 0 Msg %p Idx: %p !!!!!!!!!",pMq->szName,prTemp,prTemp->NextIndex);
      }
      *pActualSize    = prTemp->msgsize;
      if(pMq->bTraceCannel == TRUE)TraceString("OSALMQ SHM GetMsgFromMsgBox %s Msg %p size:%d Idx: %p ",pMq->szName,prTemp,prTemp->msgsize,prTemp->NextIndex);  
      pMessage        = prTemp->Message;
      prTemp->msgsize = 0;
      /* update priority list */
      pMq->PrioMsgListOffset[SearchedPrio] = prTemp->NextIndex;

   }
   return( pMessage );
}

tS32 OSAL_s32ShMemMessageQueueCreate(tCString coszName, tU32 u32MaxMessages, tU32 u32MaxLength,OSAL_tenAccess enAccess, OSAL_tMQueueHandle* pHandle)
{
   OSAL_tSemHandle hSem = (OSAL_tSemHandle)OSAL_ERROR;
   OSAL_tSemHandle hSigSem = (OSAL_tSemHandle)OSAL_ERROR;
   OSAL_tEventHandle hEvent = (OSAL_tEventHandle)OSAL_ERROR;
   tS32 s32ReturnValue = OSAL_ERROR;
   tU32 u32ErrorCode = OSAL_E_NOERROR;
   tU32 u32Loop;
   tU32 u32Lenght = u32MaxMessages *(u32MaxLength + (tU32)sizeof(MsgHdr));
   char Name[LINUX_C_MQ_MAX_NAMELENGHT];
   trMqueueElement* pCurrentEntry = NULL;
   OSAL_tShMemHandle hShMem= (OSAL_tShMemHandle)OSAL_ERROR;
 
   if((pHandle)&&(u32MaxMessages)&&(u32MaxLength)&&(coszName)&&(enAccess <= OSAL_EN_READWRITE))
   {
      /* The Name is not too long */
      if( OSAL_u32StringLength(coszName) < (LINUX_C_MQ_MAX_NAMELENGHT))
      {
         if(LockOsal(&pOsalData->MqueueTable.rLock) == OSAL_OK)
         {
            pCurrentEntry = tMqueueTableSearchEntryByName(coszName);
            if( pCurrentEntry == OSAL_NULL )
            {
               snprintf(Name,LINUX_C_MQ_MAX_NAMELENGHT-1,"MQ%s",coszName);
               if((pCurrentEntry = tMqueueTableGetFreeEntry())!= OSAL_NULL )
               {
                  hShMem = OSAL_SharedMemoryCreate(Name,enAccess,u32Lenght);
                  if((s32ReturnValue = OSAL_s32EventCreateOpt(Name, &hEvent, CONSUME_EVENT|WAIT_MULTIPLE_TSK)) == OSAL_OK)
                  {
                     if((s32ReturnValue = OSAL_s32SemaphoreCreate(Name, &hSigSem,0)) == OSAL_OK)
                     {
#ifdef SHM_MQ_USE_SEM
                         snprintf(Name,LINUX_C_MQ_MAX_NAMELENGHT-1,"Q2%s",coszName);
                         s32ReturnValue = OSAL_s32SemaphoreCreate(Name,&hSem,1);
#else
                         s32ReturnValue = OSAL_s32MutexCreate(Name,&hSem,0);
#endif
                     }
                  }
                  if((hShMem != (OSAL_tShMemHandle)OSAL_ERROR)||(s32ReturnValue != OSAL_ERROR))
                  {
                     if((u32ErrorCode = u32GenerateHandle(enAccess, pCurrentEntry, pHandle)) != OSAL_E_NOERROR)
                     {
                        TraceString("MQ(%s): u32GenerateHandle Error:%d", coszName, u32ErrorCode);
                        FATAL_M_ASSERT_ALWAYS();
                     }
                     else
                     {
                        ((tMQUserHandle*)*pHandle)->pAdress = OSAL_pvSharedMemoryMap(hShMem,enAccess,u32Lenght,0);
                        ((tMQUserHandle*)*pHandle)->hShMem  = hShMem;
                        ((tMQUserHandle*)*pHandle)->hLockSem = hSem;
                        ((tMQUserHandle*)*pHandle)->hSigSem = hSigSem;
                        ((tMQUserHandle*)*pHandle)->hEvent  = hEvent;
                            
                        pCurrentEntry->pcallback = NULL;
                        pCurrentEntry->pcallbackarg = NULL;
                        pCurrentEntry->callbacktid = NOTATHREAD;
                        pCurrentEntry->callbackpid = NOTATHREAD;
                        pCurrentEntry->callbackpididx = NOTATHREAD;
                        pCurrentEntry->bNotify = FALSE;
                        pCurrentEntry->bIsUsed = TRUE;
#ifdef USE_EVENT_AS_CB
                        pCurrentEntry->mask = (OSAL_tEventMask) 0;
                        pCurrentEntry->enFlags = (OSAL_tenEventMaskFlag) 0;
#endif
                        pCurrentEntry->u16Type = MQ_STD;
                        pCurrentEntry->bOverflow = FALSE;
                        pCurrentEntry->PID = MqPid;
                        (void)OSAL_szStringNCopy((tString)pCurrentEntry->szName,
                                                coszName,
                                                strlen(coszName));
                        pCurrentEntry->MaxMessages   = u32MaxMessages;
                        pCurrentEntry->MaxLength     = u32MaxLength;
                        pCurrentEntry->ActualMessage = 0;
                        /* init the other MQ fields */
                        pCurrentEntry->bToDelete = FALSE;
                        pCurrentEntry->u16OpenCounter = 1;
                        pCurrentEntry->RecTsk = 0;
                        pCurrentEntry->RecPrc = 0;
                        for( u32Loop=0; u32Loop < OSAL_C_U32_MQUEUE_PRIORITY_LOWEST+1; u32Loop++ )
                        {
                           pCurrentEntry->PrioMsgListOffset[u32Loop] = -1;
                        }
                     //   pCurrentEntry->PayLoad = ( (u32MaxLength-1)>>2 )<<2;;
                        if(!strcmp("CSMALU,_MQ",coszName))
                        {
                           pCurrentEntry->u32MessagePrcessing = MEASUREMENT_SWITCHOFF;
                        }
                        else
                        {
                           pCurrentEntry->u32MessagePrcessing = MSG_PROCESSING_INACTIVE;
                        }
                        pCurrentEntry->u32IdxMPT    = 0;
                        for( u32Loop=0; u32Loop < OSAL_C_MSG_TIME_ARR_SIZE; u32Loop++ )
                        {
                           pCurrentEntry->u32MsgPrcTim[u32Loop]   = 0;
                           pCurrentEntry->u32MsgTimStamp[u32Loop] = 0;
                        }
                        pCurrentEntry->u32IdxLMPT    = 0;
                        for( u32Loop=0; u32Loop < OSAL_C_LONG_MSG_TIME_ARR_SIZE; u32Loop++ )
                        {
                           pCurrentEntry->u32LongMsgPrcTim[u32Loop]   = 0;
                           pCurrentEntry->u32LongMsgTimStamp[u32Loop] = 0;
                        }
                        if(pOsalData->rMqName[0].szObjName[0] != 0)
                        {
                           for( u32Loop=0; u32Loop < MAX_NR_SUPERVISION_OBJ; u32Loop++ )
                           {		
                             if(pOsalData->rMqName[u32Loop].szObjName[0] == 0)break;
                             if(!strncmp(coszName,&pOsalData->rMqName[u32Loop].szObjName[0],strlen(&pOsalData->rMqName[u32Loop].szObjName[0])))
//                              if(!strncmp(coszName,&pOsalData->rMqName[u32Loop].szObjName[0],strlen(coszName)+1))
                              {
                                TraceString("Enable Trace for %s %d via %s",coszName,strlen(&pOsalData->rMqName[u32Loop].szObjName[0]),&pOsalData->rMqName[u32Loop].szObjName[0]);
                                pCurrentEntry->bTraceCannel = TRUE;
                                break;
                              }
                           }
                        }
                        pCurrentEntry->u16WaitCounter = 0;
                        pCurrentEntry->bBlocked = FALSE;
                        pCurrentEntry->u32Marker = 0;
                        s32ReturnValue = OSAL_OK;
                     }
                  }
                  else                
                  {
                      u32ErrorCode = OSAL_E_UNKNOWN;
                  }
               }
               else
               {
                  NORMAL_M_ASSERT_ALWAYS(); 
                  u32ErrorCode = OSAL_E_NOSPACE;
               }
            }
            else
            {
               u32ErrorCode = OSAL_E_ALREADYEXISTS;
            }
            /* UnLock the table */
            UnLockOsal(&pOsalData->MqueueTable.rLock);
         }
      }
      else
      {
         u32ErrorCode = OSAL_E_NAMETOOLONG;
      }
   }
   else
   {
      u32ErrorCode = OSAL_E_INVALIDVALUE;
   }
   
   if((u32ErrorCode != OSAL_E_NOERROR )||(pHandle == NULL))
   {
      s32ReturnValue = OSAL_ERROR;
      if(u32ErrorCode == OSAL_E_ALREADYEXISTS)
      {
        if((pCurrentEntry&&(pCurrentEntry->bTraceCannel == TRUE))||(u32OsalSTrace & 0x00000020))
        {
           vTraceMqInfo(NULL,OSAL_MQ_CREATE,FALSE,u32ErrorCode,coszName);
        }
      }
      else
      {
           vTraceMqInfo(NULL,OSAL_MQ_CREATE,TRUE,u32ErrorCode,coszName);
      }
      vSetErrorCode(OSAL_C_THREAD_ID_SELF, u32ErrorCode);
   }
   else
   {
      if((pCurrentEntry->bTraceCannel == TRUE)||(u32OsalSTrace & 0x00000020)) /*lint !e613 pCurrentEntry already checked */
      {
         vTraceMqInfo((tMQUserHandle*)*pHandle,OSAL_MQ_CREATE,FALSE,u32ErrorCode,coszName);
      }
   }
   if(s32ReturnValue == OSAL_ERROR)
   {
      snprintf(Name,LINUX_C_MQ_MAX_NAMELENGHT-1,"MQ%s",coszName);
      if(hShMem != (OSAL_tShMemHandle)OSAL_ERROR)
      {
         (void)OSAL_s32SharedMemoryClose(hShMem);
         (void)OSAL_s32SharedMemoryDelete(Name);
      }
      if(hSigSem != (OSAL_tSemHandle)OSAL_ERROR)
      {
         (void)OSAL_s32SemaphoreClose(hSigSem);
         (void)OSAL_s32SemaphoreDelete(Name);
      }
      if(hEvent != (OSAL_tEventHandle)OSAL_ERROR)
      {
         (void)OSAL_s32EventClose(hEvent);
         (void)OSAL_s32EventDelete(Name);
      }
      if(hSem != (OSAL_tSemHandle)OSAL_ERROR)
      {
#ifdef SHM_MQ_USE_SEM
         (void)OSAL_s32SemaphoreClose(hSem);
         snprintf(Name,LINUX_C_MQ_MAX_NAMELENGHT-1,"Q2%s",coszName);
         (void)OSAL_s32SemaphoreDelete(Name);
#else
         (void)OSAL_s32MutexClose(hSem);
         (void)OSAL_s32MutexDelete(Name);
#endif
      }
   }
   return s32ReturnValue;
}

tS32 OSAL_s32ShMemMessageQueueOpen(tCString coszName,OSAL_tenAccess enAccess,OSAL_tMQueueHandle* pHandle)
{
   OSAL_tShMemHandle hShMem = (OSAL_tShMemHandle)OSAL_ERROR;
   OSAL_tSemHandle hSem     = (OSAL_tSemHandle)OSAL_ERROR;
   OSAL_tSemHandle hSigSem = (OSAL_tSemHandle)OSAL_ERROR;
   OSAL_tEventHandle hEvent = (OSAL_tEventHandle)OSAL_ERROR;
   tS32 s32ReturnValue = OSAL_ERROR;
   tU32 u32ErrorCode = OSAL_E_NOERROR;
   char Name[LINUX_C_MQ_MAX_NAMELENGHT];
   trMqueueElement* pCurrentEntry = NULL;
   
   if((pHandle)&&(coszName)&&(enAccess <= OSAL_EN_READWRITE))
   {
      if(LockOsal(&pOsalData->MqueueTable.rLock) == OSAL_OK)
      {
         pCurrentEntry = tMqueueTableSearchEntryByName(coszName);
         if(pCurrentEntry != NULL)
         {
            snprintf(Name,LINUX_C_MQ_MAX_NAMELENGHT-1,"MQ%s",coszName);
            hShMem = OSAL_SharedMemoryOpen(Name,enAccess);
            if((s32ReturnValue = OSAL_s32EventOpen(Name, &hEvent)) == OSAL_OK)
            {
               if((s32ReturnValue = OSAL_s32SemaphoreOpen(Name, &hSigSem)) == OSAL_OK)
               {
#ifdef SHM_MQ_USE_SEM
                  snprintf(Name,LINUX_C_MQ_MAX_NAMELENGHT-1,"Q2%s",coszName);
                  s32ReturnValue = OSAL_s32SemaphoreOpen(Name,&hSem);
#else
                  s32ReturnValue = OSAL_s32MutexOpen(Name,&hSem);
#endif
               }
            }
            if((hShMem != (OSAL_tShMemHandle)OSAL_ERROR)&&(s32ReturnValue != OSAL_ERROR))
            {
               pCurrentEntry->u16OpenCounter++;
            }
            else
            {
               NORMAL_M_ASSERT_ALWAYS();
            }
            if((u32ErrorCode = u32GenerateHandle(enAccess,pCurrentEntry,pHandle)) == OSAL_E_NOERROR)
            {
               ((tMQUserHandle*)*pHandle)->pAdress = OSAL_pvSharedMemoryMap(hShMem,enAccess,pCurrentEntry->MaxMessages * pCurrentEntry->MaxLength,0);
               ((tMQUserHandle*)*pHandle)->hShMem  = hShMem;
               ((tMQUserHandle*)*pHandle)->hSigSem = hSigSem;
               ((tMQUserHandle*)*pHandle)->hEvent  = hEvent;
               ((tMQUserHandle*)*pHandle)->hLockSem= hSem;
               s32ReturnValue = OSAL_OK;
            } 
         }
         else
         {
            u32ErrorCode = OSAL_E_DOESNOTEXIST;
         }//if( pCurrentEntry!= OSAL_NULL )
         UnLockOsal(&pOsalData->MqueueTable.rLock);
      }
   }
   else
   {
      u32ErrorCode = OSAL_E_INVALIDVALUE;
   }
   
   if(s32ReturnValue == OSAL_ERROR)
   {
      if(hShMem != (OSAL_tShMemHandle)OSAL_ERROR)
      {
         (void)OSAL_s32SharedMemoryClose(hShMem);
      }
      if(hSigSem != (OSAL_tSemHandle)OSAL_ERROR)
      {
         (void)OSAL_s32SemaphoreClose(hSigSem);
      }
      if(hEvent != (OSAL_tEventHandle)OSAL_ERROR)
      {
         (void)OSAL_s32EventClose(hEvent);
      }
      if(hSem != (OSAL_tSemHandle)OSAL_ERROR)
      {
#ifdef SHM_MQ_USE_SEM
         (void)OSAL_s32SemaphoreClose(hSem);
#else
         (void)OSAL_s32MutexClose(hSem);
#endif
      }
   }
   
   if((u32ErrorCode != OSAL_E_NOERROR )||(pHandle == NULL))
   {
      if(u32ErrorCode == OSAL_E_DOESNOTEXIST)
      {
        if(u32OsalSTrace & 0x00000020)
        {
           vTraceMqInfo(NULL,OSAL_TSK_CONNECT_TO_MQ,FALSE,u32ErrorCode,coszName);
        }
      }
      else
      {
         vTraceMqInfo(NULL,OSAL_TSK_CONNECT_TO_MQ,TRUE,u32ErrorCode,coszName);
      }
      vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
   }
   else
   {
      if((pCurrentEntry->bTraceCannel == TRUE)||(u32OsalSTrace & 0x00000020))/*lint !e613 pointer already checked */
      {
         vTraceMqInfo((tMQUserHandle*)*pHandle,OSAL_TSK_CONNECT_TO_MQ,FALSE,u32ErrorCode,coszName);
      }
     s32ReturnValue = OSAL_OK;
   }
   return s32ReturnValue;
}

tS32 OSAL_s32ShMemMessageQueueDelete(tCString coszName)
{
   tS32 s32ReturnValue = OSAL_ERROR;
   tU32 u32ErrorCode = OSAL_E_NOERROR;
   char Name[LINUX_C_MQ_MAX_NAMELENGHT];
   trMqueueElement* pCurrentEntry = NULL;

   if(coszName)
   {
      if(LockOsal(&pOsalData->MqueueTable.rLock) == OSAL_OK)
      { 
         pCurrentEntry = tMqueueTableSearchEntryByName(coszName);
         if(pCurrentEntry != NULL)
         {
            pCurrentEntry->bToDelete = TRUE;
        
            if(pCurrentEntry->u16OpenCounter == 0)
            {
               snprintf(Name,LINUX_C_MQ_MAX_NAMELENGHT-1,"MQ%s",coszName);
               (void)OSAL_s32SharedMemoryDelete(Name);
               (void)OSAL_s32SemaphoreDelete(Name);
               (void)OSAL_s32EventDelete(Name);
#ifdef SHM_MQ_USE_SEM
               snprintf(Name,LINUX_C_MQ_MAX_NAMELENGHT-1,"Q2%s",coszName);
               (void)OSAL_s32SemaphoreDelete(Name);
#else
               (void)OSAL_s32MutexDelete(Name);
#endif
               if((pCurrentEntry->bTraceCannel == TRUE)||(u32OsalSTrace & 0x00000020))/*lint !e613 pointer already checked */
               {
                  vTraceMqInfo(NULL,OSAL_TSK_DELETE_MQ,FALSE,u32ErrorCode,coszName);
               }
               vResetEntry(pCurrentEntry->u32EntryIdx);
            }
            s32ReturnValue = OSAL_OK;
         }
         else
         {
            u32ErrorCode = OSAL_E_DOESNOTEXIST;
         }
         UnLockOsal(&pOsalData->MqueueTable.rLock);
      }
   }
   else
   {
      u32ErrorCode = OSAL_E_INVALIDVALUE;
   }

   if(s32ReturnValue != OSAL_OK)
   {
      vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
      vTraceMqInfo(NULL,OSAL_TSK_DELETE_MQ,TRUE,u32ErrorCode,coszName);
   }

   return s32ReturnValue;
}

tS32 OSAL_s32ShMemMessageQueueClose(OSAL_tMQueueHandle Handle)
{
  tS32 s32ReturnValue = OSAL_ERROR;
  trMqueueElement* pCurrentEntry = NULL;
  tU32 u32ErrorCode = u32CheckMqHandle(Handle);
  char Name[LINUX_C_MQ_MAX_NAMELENGHT];

  if(u32ErrorCode == OSAL_E_NOERROR)
  {
     if(LockOsal(&pOsalData->MqueueTable.rLock) == OSAL_OK)
     {
        pCurrentEntry = ((tMQUserHandle*)Handle)->pOrigin;/*lint !e613 *//* pointer already checked*/ 
        if(pCurrentEntry->u16OpenCounter > 0 )
        {
           pCurrentEntry->u16OpenCounter--;
        }
        (void)OSAL_s32SharedMemoryClose(((tMQUserHandle*)Handle)->hShMem);
        (void)OSAL_s32EventClose(((tMQUserHandle*)Handle)->hEvent);
        (void)OSAL_s32SemaphoreClose(((tMQUserHandle*)Handle)->hSigSem);
#ifdef SHM_MQ_USE_SEM
        (void)OSAL_s32SemaphoreClose(((tMQUserHandle*)Handle)->hLockSem);
#else
        (void)OSAL_s32MutexClose(((tMQUserHandle*)Handle)->hLockSem);
#endif
        if((pCurrentEntry->u16OpenCounter == 0)&&(pCurrentEntry->bToDelete))
        {
          if((pCurrentEntry->bTraceCannel == TRUE)||(u32OsalSTrace & 0x00000020))
          {
             vTraceMqInfo((tMQUserHandle*)Handle,OSAL_TSK_DISCONNECT_MQ,FALSE, u32ErrorCode,pCurrentEntry->szName);
          }
          snprintf(Name,LINUX_C_MQ_MAX_NAMELENGHT-1,"MQ%s",pCurrentEntry->szName);
          (void)OSAL_s32SharedMemoryDelete(Name);
          (void)OSAL_s32EventDelete(Name);
          (void)OSAL_s32SemaphoreDelete(Name);
#ifdef SHM_MQ_USE_SEM
          snprintf(Name,LINUX_C_MQ_MAX_NAMELENGHT-1,"Q2%s",pCurrentEntry->szName);
          (void)OSAL_s32SemaphoreDelete(Name);
#else
          (void)OSAL_s32MutexDelete(Name);
#endif
#ifdef USE_EVENT_AS_CB
          if(((tMQUserHandle*)Handle)->hNotifyEvent != 0)
          {
             if(OSAL_s32EventClose(((tMQUserHandle*)Handle)->hNotifyEvent) != OSAL_OK)
             {
                NORMAL_M_ASSERT_ALWAYS();
             }
             ((tMQUserHandle*)Handle)->hNotifyEvent = 0;
          }
#endif
          vResetEntry(pCurrentEntry->u32EntryIdx);
        }
        ((tMQUserHandle*)Handle)->pOrigin = NULL;/*lint !e613 *//* pointer already checked*/ 
        ((tMQUserHandle*)Handle)->PrcId = 0;
        if(OSAL_s32MemPoolFixSizeRelBlockOfPool(&MqMemPoolHandle,(void*)Handle) == OSAL_ERROR)
        {    NORMAL_M_ASSERT_ALWAYS();  }
        s32ReturnValue = OSAL_OK;
        UnLockOsal(&pOsalData->MqueueTable.rLock);
     }
   }
    
   if(s32ReturnValue == OSAL_ERROR)
   {
     vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
     tCString szName = "";
  /*   if(pCurrentEntry)
     {
        szName = pCurrentEntry->szName;
     }*/
     vTraceMqInfo(NULL,OSAL_TSK_DISCONNECT_MQ,TRUE,u32ErrorCode,szName);
   }
   return s32ReturnValue;
}

void EnterMqCriticalSection(OSAL_tMQueueHandle Handle)
{
#ifdef SHM_MQ_USE_SEM
   (void)OSAL_s32SemaphoreWait(((tMQUserHandle*)Handle)->hLockSem,OSAL_C_TIMEOUT_FOREVER);
#else
   (void)OSAL_s32MutexLock(((tMQUserHandle*)Handle)->hLockSem,OSAL_C_TIMEOUT_FOREVER);
#endif
}

void LeaveMqCriticalSection(OSAL_tMQueueHandle Handle)
{
#ifdef SHM_MQ_USE_SEM
   (void)OSAL_s32SemaphorePost(((tMQUserHandle*)Handle)->hLockSem);
#else
   (void)OSAL_s32MutexUnLock(((tMQUserHandle*)Handle)->hLockSem);
#endif
}

tS32 OSAL_s32ShMemMessageQueuePost(OSAL_tMQueueHandle Handle,tPCU8 pcou8Msg, tU32 u32Length,tU32 u32Prio)
{
   tS32 s32ReturnValue = OSAL_OK;
   tU32 u32ErrorCode   = u32CheckMqHandle(Handle); /* step 1 user handle check */
   trMqueueElement *pCurrentEntry = NULL;
   char name[TRACE_NAME_SIZE];
   tU32 u32Temp = (tU32)OSAL_ThreadWhoAmI();
   tU32 u32Mask = 0;
   tU32 u32ActualMessage = -1;
   tBool bTrigger = FALSE;
   
   if(u32ErrorCode == OSAL_E_NOERROR)
   {
      pCurrentEntry = ((tMQUserHandle*)Handle)->pOrigin;
      if(pCurrentEntry)
      {
         u32ActualMessage = pCurrentEntry->ActualMessage;
         if(((tMQUserHandle*)Handle)->enAccess == OSAL_EN_READONLY)
         {
	        u32ErrorCode = OSAL_E_NOPERMISSION;
         }
         if( u32Prio < (OSAL_C_U32_MQUEUE_PRIORITY_LOWEST + 1) )
         {  
           /* check if queue isn't deleted */
           if(u32Length > pCurrentEntry->MaxLength)
           {
              u32ErrorCode = OSAL_E_MSGTOOLONG;
           } 
           else if((pCurrentEntry->bIsUsed == FALSE))
           {
              u32ErrorCode = OSAL_E_DOESNOTEXIST;
           }
        /*else if(pCurrentEntry->bToDelete == TRUE)
        {
           u32ErrorCode = OSAL_E_NOPERMISSION;
        }*/
        }
        else
        {
          u32ErrorCode = OSAL_E_INVALIDVALUE;
        }
     }
      else
      {
         u32ErrorCode = OSAL_E_INVALIDVALUE;
      }
   }
   if((u32ErrorCode == OSAL_E_NOERROR)&&(pCurrentEntry)/*for stupid lint*/)
   {
      /* check if enough space available otherwise wait */
      while(1)
      {
         if(pCurrentEntry->ActualMessage == pCurrentEntry->MaxMessages)
         {
            if(OSAL_ThreadWhoAmI() == pCurrentEntry->RecTsk)
            {
               u32Temp = (tU32)OSAL_ThreadWhoAmI();
               bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,OSAL_ThreadWhoAmI());
               vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) would block itself MQ: Handle 0x%x Name:%s -> return from OSAL_s32MessageQueuePost\n",
                                   name,(unsigned int)u32Temp, (uintptr_t)pCurrentEntry,pCurrentEntry->szName); /* lint !e613 */ /* u32ErrorCode ensures that hMQ is a valid pointer value */
               vSetErrorCode(OSAL_C_THREAD_ID_SELF, OSAL_E_QUEUEFULL);
               return OSAL_ERROR;
            }
            EnterMqCriticalSection(Handle);
            /* check if condition still exist within locked area */
            if(pCurrentEntry->ActualMessage == pCurrentEntry->MaxMessages)
            {
               /* increment marker because this sender is blocked and will wait for an event */
               pCurrentEntry->u32WBlock++;
               /* we have to wait for next post , leave locked section */
               LeaveMqCriticalSection(Handle);

               s32ReturnValue = OSAL_s32EventWait(((tMQUserHandle*)Handle)->hEvent,WAIT_FOR_WRITE,OSAL_EN_EVENTMASK_OR,BLOCKING_TIMEOUT,&u32Mask);

               EnterMqCriticalSection(Handle);
               pCurrentEntry->u32WBlock--;
               if(pCurrentEntry->ActualMessage < pCurrentEntry->MaxMessages)bTrigger = TRUE;
               LeaveMqCriticalSection(Handle);

               if(s32ReturnValue == OSAL_ERROR)
               {
                  u32ErrorCode = OSAL_E_QUEUEFULL;
                  s32ReturnValue = OSAL_ERROR;
                  TraceString("Post1 ShMem MQ:%s timeout:%d  ",pCurrentEntry->szName,OSAL_C_TIMEOUT_FOREVER);
                  vReactOnMqOverflow(pCurrentEntry);
                  break;
               }
               else
               {
                  if(bTrigger)
                  {
                      bTrigger = 0;
                      break;
                  }
                  else
                  {
                     if((pCurrentEntry->ActualMessage < pCurrentEntry->MaxMessages)&&(pCurrentEntry->u32WBlock > 0))
                     { 
                        s32ReturnValue = OSAL_s32EventPost(((tMQUserHandle*)Handle)->hEvent,WAIT_FOR_WRITE,OSAL_EN_EVENTMASK_OR);
//		                 TraceString(")
				        OSAL_s32ThreadWait(0);//force scheduling of writing task
                     }
                  }
               }
            }
            else
            {
                if(pCurrentEntry->ActualMessage < pCurrentEntry->MaxMessages)
                {
                    LeaveMqCriticalSection(Handle);
                    break;
                }
                else
                {
                   FATAL_M_ASSERT_ALWAYS();
                }
             }
         }
         else
         {
            break;
         }
      }
      /* check if memory is now ensured */
      if(u32ErrorCode == OSAL_E_NOERROR)
      {
         EnterMqCriticalSection(Handle);
         if(s32StoreToMsgBox(pCurrentEntry,(tMQUserHandle*)Handle,u32Prio,(tU16)u32Length,pcou8Msg) != OSAL_OK)
         {
            TraceString("OSALMQ s32StoreToMsgBox failed");
         }
         else
         {
            pCurrentEntry->ActualMessage++;
            if(pCurrentEntry->ActualMessage == 1)
            {
                pCurrentEntry->bTriggerCb = TRUE;
            }
            if(pCurrentEntry->u32RBlock > 0)
            { 
               s32ReturnValue = OSAL_s32SemaphorePost(((tMQUserHandle*)Handle)->hSigSem);
            }
         }
         u32ActualMessage = pCurrentEntry->ActualMessage;
         LeaveMqCriticalSection(Handle);
      }
   }
   else
   {
       s32ReturnValue = OSAL_ERROR;
   }
   

   if((u32ErrorCode == OSAL_E_NOERROR)&&(pCurrentEntry)&&(pCurrentEntry->bTriggerCb == TRUE)
    &&(pCurrentEntry->callbackpididx < (tS32)pOsalData->u32MaxNrProcElements)
    && (pCurrentEntry->pcallback )&&(pCurrentEntry->callbackpididx >= 0))
   {
        pCurrentEntry->bTriggerCb = FALSE;
        if( pCurrentEntry->callbackpid == OSAL_ProcessWhoAmI()) // check if process is the same
        {
           vExecuteCallback(pCurrentEntry);
        }
        else
        {
           vPostToCallbackQueue(pCurrentEntry);
        }
   }

 #ifdef USE_EVENT_AS_CB
      if(pCurrentEntry && (pCurrentEntry->mask))
      {
         pCurrentEntry->bTriggerCb = FALSE;
         if(((tMQUserHandle*)Handle)->hNotifyEvent == 0)
         {
            if(OSAL_s32EventOpen(pCurrentEntry->szEvName,&((tMQUserHandle*)Handle)->hNotifyEvent) != OSAL_OK)
            {
                 NORMAL_M_ASSERT_ALWAYS();
            }
         }
         OSAL_s32EventPost(((tMQUserHandle*)Handle)->hNotifyEvent, pCurrentEntry->mask, pCurrentEntry->enFlags);
      }
#endif
    
   if( u32ErrorCode != OSAL_E_NOERROR )
   {
      s32ReturnValue = OSAL_ERROR;
      if((u32ErrorCode != OSAL_E_TIMEOUT)|| /* ((pCurrentEntry)&&(pCurrentEntry->bTraceCannel == TRUE))||*/(u32OsalSTrace & 0x00000020))
      {
          u32Temp = (tU32)OSAL_ThreadWhoAmI();
          bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,u32Temp);
      }
      if(pCurrentEntry)
      {
             TraceString("OSALMQ Task:%s(ID:%d) Send      to MQ: User Handle %p MQ Name:%s Msg Count:%d Error:0x%x Prio:%d",
                      name,(unsigned int)u32Temp,pCurrentEntry, pCurrentEntry->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32Prio);
             if((pOsalData->bLogError)&&(u32ErrorCode != OSAL_E_TIMEOUT)&&(u32OsalSTrace & ~0x00000020))vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) Send      to MQ: User Handle %p MQ Name:%s Msg Count:%d Error:0x%x Prio:%d \n",
                      name,(unsigned int)u32Temp,pCurrentEntry, pCurrentEntry->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32Prio);
      }
      else
      {
             TraceString("OSALMQ Task:%s(ID:%d) Send      to MQ: User Handle 0x%x MQ Name:%s Msg Count:%d Error:0x%x Prio:%d",
                      name,(unsigned int)u32Temp,(unsigned int)0, "Unknown",(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32Prio);
             if((pOsalData->bLogError)&&(u32ErrorCode != OSAL_E_TIMEOUT)&&(u32OsalSTrace & ~0x00000020))vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) Send      to MQ: User Handle 0x%x MQ Name:%s Msg Count:%d Error:0x%x Prio:%d \n",
                      name,(unsigned int)u32Temp,(unsigned int)0, "Unknown",(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32Prio);
      }
      vSetErrorCode(OSAL_C_THREAD_ID_SELF, u32ErrorCode);
   }
   else
   {
       if((pCurrentEntry->bTraceCannel == TRUE)||(u32OsalSTrace & 0x00000020))/*lint !e613 pCurrentEntry already checked */
       {
           u32Temp = (tU32)OSAL_ThreadWhoAmI();
           bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,u32Temp);
           TraceString("OSALMQ Task:%s(ID:%d) Send      to MQ: User Handle %p MQ Name:%s Msg Count:%d Error:0x%x Prio:%d",
                      name,(unsigned int)u32Temp,pCurrentEntry, pCurrentEntry->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32Prio);
       } 
   }

  return s32ReturnValue;   
}

tS32 OSAL_s32ShMemMessageQueueWait(OSAL_tMQueueHandle Handle,tPU8 pu8Buffer, tU32 u32Length, tPU32 pu32Prio,OSAL_tMSecond u32TimeOut)
{
   tU32 u32ErrorCode   = u32CheckMqHandle(Handle); /* step 1 user handle check */
   trMqueueElement *pCurrentEntry = NULL;
   tU32 u32CopiedBytes = 0;
   void* pMsg = NULL;
   tU32 u32ActualPrio = 0;            
   char name[TRACE_NAME_SIZE];
   tS32 s32ReturnValue = OSAL_OK;
   tU32 u32ActualMessage = 0;
   tU32 u32Temp;
   ((void)u32Length);
//   tU32 Start,End;
   
   if(u32ErrorCode == OSAL_E_NOERROR)
   {
      pCurrentEntry = ((tMQUserHandle*)Handle)->pOrigin;
      if(pCurrentEntry)
      {
         if(pCurrentEntry->RecTsk == 0)
         {
            pCurrentEntry->RecTsk = OSAL_ThreadWhoAmI();
            pCurrentEntry->RecPrc = OSAL_ProcessWhoAmI();
         }
         u32ActualMessage = pCurrentEntry->ActualMessage;
         if(((tMQUserHandle*)Handle)->enAccess == OSAL_EN_WRITEONLY)
         {
            u32ErrorCode = OSAL_E_NOPERMISSION;
         }
         else
         {
            /* check for available data*/
            while(1)
            {
               u32ErrorCode = OSAL_E_NOERROR;
               /* check for available data*/
               if(pCurrentEntry->ActualMessage == 0) 
               {
                  if(u32TimeOut == OSAL_C_TIMEOUT_NOBLOCKING)
                  {
                     u32ErrorCode = OSAL_E_TIMEOUT;
                     break;
                  }
                  else
                  {
                     EnterMqCriticalSection(Handle);
                     /* check if situation has changed during getting the lock */
                     if(pCurrentEntry->ActualMessage > 0)
                     {
                        LeaveMqCriticalSection(Handle);
                        break;
                     }
                     else /* situation is still the same */
                     {
                        /* set block marker */
                        pCurrentEntry->u32RBlock++;
                        LeaveMqCriticalSection(Handle);

                        /* wait for post trigger or timeout */
                        s32ReturnValue = OSAL_s32SemaphoreWait(((tMQUserHandle*)Handle)->hSigSem,u32TimeOut);

                        /* remove block marker */
                        EnterMqCriticalSection(Handle);
                        pCurrentEntry->u32RBlock--;
                        LeaveMqCriticalSection(Handle);

                        if((s32ReturnValue == OSAL_ERROR)/*&&((End - Start) < u32TimeOut)*/)
                        {
                           u32ErrorCode = OSAL_E_TIMEOUT;
                           if(u32OsalSTrace & 0x00000020)TraceString("Wait1 ShMem MQ:%s timeout %d -> %d messages in Queue",pCurrentEntry->szName,u32TimeOut,pCurrentEntry->ActualMessage);
                           break;
                        }
                        else
                        {
                           if(pCurrentEntry->ActualMessage > 0)
                           {
                              break;
                           }
                           else
                           {
                              if(pCurrentEntry->u32WBlock > 0)
                              { 
                                 s32ReturnValue = OSAL_s32EventPost(((tMQUserHandle*)Handle)->hEvent,WAIT_FOR_WRITE,OSAL_EN_EVENTMASK_OR);
                                 OSAL_s32ThreadWait(0);//force scheduling of writing task
                              }
                           }
                        }
                     }
                  }
               }
               else
               {
                  break;
               }
            }
         }
      }
      else
      {
         u32ErrorCode = OSAL_E_INVALIDVALUE;
      }
      u32CopiedBytes = 0;
      if(u32ErrorCode == OSAL_E_NOERROR)
      {
         EnterMqCriticalSection(Handle);
         if(pCurrentEntry->ActualMessage == 0) 
         {
            NORMAL_M_ASSERT_ALWAYS();
         }
			
         pCurrentEntry->u32RBlock = 0;
         /* Stop message processing time measurement */
         vStopMeasureProcessingTime(pCurrentEntry);
            
         u32ErrorCode = OSAL_E_INPROGRESS;
         for( u32ActualPrio = 0; u32ActualPrio < (OSAL_C_U32_MQUEUE_PRIORITY_LOWEST + 1); u32ActualPrio ++ )
         {
            pMsg = GetMsgFromMsgBox (pCurrentEntry,((tMQUserHandle*)Handle),u32ActualPrio,&u32CopiedBytes);
            if(pMsg)
            {
/*             if(u32CopiedBytes == 0)
                   {
                      vTraceHexDump("OSALMQ",(tPCU8)((tMQUserHandle*)Handle)->pAdress,pCurrentEntry->MaxMessages*pCurrentEntry->MaxLength);
                      vTraceHexDump("OSALMQ",(tPCU8)pCurrentEntry->PrioMsgListOffset,(OSAL_C_U32_MQUEUE_PRIORITY_LOWEST * 4));
                   }*/
               memcpy(pu8Buffer,(char*)pMsg,u32CopiedBytes);
               if(pCurrentEntry->ActualMessage > 0)pCurrentEntry->ActualMessage--;
               u32ActualMessage = pCurrentEntry->ActualMessage;
               if (pu32Prio != OSAL_NULL)
               {
                  *pu32Prio = u32ActualPrio;
               }
               u32ErrorCode = OSAL_E_NOERROR;
               break;
            }
         }
            
         /* Start measurement of message processing times */
         vStartMeasureProcessingTime(pCurrentEntry);
         if(pCurrentEntry->u32WBlock > 0)
         { 
            s32ReturnValue = OSAL_s32EventPost(((tMQUserHandle*)Handle)->hEvent,WAIT_FOR_WRITE,OSAL_EN_EVENTMASK_OR);
         }
         if(pCurrentEntry->ActualMessage > 0)
         {
            s32ReturnValue = OSAL_s32EventPost(((tMQUserHandle*)Handle)->hEvent,WAIT_FOR_READ,OSAL_EN_EVENTMASK_OR);
         }
         LeaveMqCriticalSection(Handle);
      }
   }
   else
   {
      s32ReturnValue = OSAL_ERROR; 
   }

  if( u32ErrorCode != OSAL_E_NOERROR )
  {
     if((u32ErrorCode != OSAL_E_TIMEOUT)||
        ((pCurrentEntry)&&(pCurrentEntry->bTraceCannel == TRUE))||(u32OsalSTrace & 0x00000020))
     {
         u32Temp = (tU32)OSAL_ThreadWhoAmI();
         bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,u32Temp);
         if(pCurrentEntry)
         {
            TraceString("OSALMQ Task:%s(ID:%d) Receive from MQ: User Handle %p Name:%s(33,32) Msg Count:%d Error:0x%x Prio:%d",
                      name,(unsigned int)u32Temp,pCurrentEntry,pCurrentEntry->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32ActualPrio);
            if((pOsalData->bLogError)&&(u32ErrorCode != OSAL_E_TIMEOUT)&&(u32OsalSTrace & ~0x00000020))vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) Receive from MQ: User Handle %p Name:%s(33,32) Msg Count:%d Error:0x%x Prio:%d \n",
                      name,(unsigned int)u32Temp,pCurrentEntry,pCurrentEntry->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32ActualPrio);
         }
         else
         {
             TraceString("OSALMQ Task:%s(ID:%d) Receive from MQ: User Handle 0x%x Name:%s(33,32) Msg Count:%d Error:0x%x Prio:%d",
                      name,(unsigned int)u32Temp,(unsigned int)u32ErrorCode,"Unknown",(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32ActualPrio);
             if((pOsalData->bLogError)&&(u32ErrorCode != OSAL_E_TIMEOUT)&&(u32OsalSTrace & ~0x00000020))vWritePrintfErrmem("OSALMQ Task:%s(ID:%d) Receive from MQ: User Handle 0x%x Name:%s(33,32) Msg Count:%d Error:0x%x Prio:%d \n",
                      name,(unsigned int)u32Temp,(unsigned int)u32ErrorCode,"Unknown",(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32ActualPrio);
         }
     }
     vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
  }
  else
  {
     if(((pCurrentEntry) &&(pCurrentEntry->bTraceCannel == TRUE))||(u32OsalSTrace & 0x00000020))
     {
        u32Temp = (tU32)OSAL_ThreadWhoAmI();
        bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,u32Temp);
        TraceString("OSALMQ Task:%s(ID:%d) Receive %d Bytes from MQ: User Handle %p( %p) Name:%s Msg Count:%d Error:0x%x Prio:%d",
                 name,(unsigned int)u32Temp,u32CopiedBytes,(tMQUserHandle*)Handle,pCurrentEntry,pCurrentEntry->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32ActualPrio);
     }
  }
  return u32CopiedBytes;   
}


tS32 OSAL_s32ShMemMessageQueueNotify(OSAL_tMQueueHandle hMQ,OSAL_tpfCallback pCallback,tPVoid pvArg)
{
   tS32 s32RetVal = OSAL_OK;
   tU32 u32ErrorCode   = u32CheckMqHandle(hMQ);
   OSAL_tThreadID TID;
   trMqueueElement *pCurrentEntry = NULL;

   if(u32ErrorCode == OSAL_E_NOERROR)
   {
       /* check if callback handler task for this process already exists */  
      tS32 s32PidEntry = s32FindProcEntry(OSAL_ProcessWhoAmI());
      if(s32PidEntry == OSAL_ERROR)
      {
         NORMAL_M_ASSERT_ALWAYS();
      }
      else
      {
         /* check if callback handler task for this process already exists */  
         s32StartCbHdrTask(s32PidEntry);
      }
      pCurrentEntry = ((tMQUserHandle*)hMQ)->pOrigin;/*lint !e613 *//* pointer already checked*/ 
      if(LockOsal(&pOsalData->MqueueTable.rLock ) == OSAL_OK)
      {
         /* check if queue isn't deleted */
         if(pCurrentEntry->bIsUsed)
         {
            TID = OSAL_ThreadWhoAmI();
#ifdef USE_EVENT_AS_CB
            if((pCallback == NULL )&&(pvArg == NULL)&&(pCurrentEntry->pcallback == (OSAL_tpfCallback)0xffffffff))
            {  
               pCurrentEntry->bNotify = FALSE;
               (void)memset(pCurrentEntry->szEvName,0,OSAL_C_U32_MAX_NAMELENGTH);
               pCurrentEntry->mask    = 0;
               pCurrentEntry->enFlags = (OSAL_tenEventMaskFlag)0;
            }
            else if(((uintptr_t)pCallback == 0xffffffff)&&(pvArg != NULL))
            {
               trMqEventInf* pEvent = (trMqEventInf*)pvArg;
               if(strlen(pEvent->coszName) > OSAL_C_U32_MAX_NAMELENGTH)
               {
                  u32ErrorCode = OSAL_E_NAMETOOLONG;
                  TraceString("MQ Notification event name %s is to long",pEvent->coszName);
               }
               else
               {
                  (void)OSAL_szStringNCopy((tString)pCurrentEntry->szEvName,pEvent->coszName, strlen(pEvent->coszName));
                  pCurrentEntry->mask    = pEvent->mask;
                  pCurrentEntry->enFlags = pEvent->enFlags;
                  pCurrentEntry->pcallback    = pCallback;
                  /* prepare message processing mesurements 
                     filter out notify queues because their usage is trgggered by OSAL callbacks */
                  pCurrentEntry->u32MessagePrcessing = MEASUREMENT_SWITCHOFF;
               }
            }
            else//if((pCallback == NULL )&&(pvArg != NULL))
#endif
            {
               if( pCallback == NULL )
               {
                  if( pvArg == NULL )
                  {
#ifdef USE_EVENT_AS_CB
                     if(((tMQUserHandle*)hMQ)->hNotifyEvent != 0)
                     {
                       if(OSAL_s32EventClose(((tMQUserHandle*)hMQ)->hNotifyEvent) != OSAL_OK)
                       {
                          NORMAL_M_ASSERT_ALWAYS();
                       }
                       ((tMQUserHandle*)hMQ)->hNotifyEvent = 0;
                     }
#endif
                     if( pCurrentEntry->pcallback != NULL )
                     {
                        if(pCurrentEntry->callbacktid == TID )
                        {
                           pCurrentEntry->bNotify = FALSE;
                           /*force dispatch to disable callback */
                           OSAL_s32ThreadWait(10);
                           pCurrentEntry->pcallback = NULL;
                           pCurrentEntry->callbacktid  = NOTATHREAD;
                           s32RetVal = OSAL_OK;
                        }
                        else
                        {
                           s32RetVal=OSAL_ERROR;
                           u32ErrorCode=OSAL_E_BUSY;
                        }
                     }
                     else
                     {
                        s32RetVal=OSAL_ERROR;
                        pCurrentEntry->callbacktid  = NOTATHREAD;
                     }
                  }
                  else
                  {
                     s32RetVal=OSAL_ERROR;
                     u32ErrorCode=OSAL_E_INVALIDVALUE;
                  }
               }
               else
               {
                  if( pCurrentEntry->pcallback == NULL )
                  {
                     pCurrentEntry->callbackpid  = OSAL_ProcessWhoAmI();
                     pCurrentEntry->callbacktid  = TID;
                     pCurrentEntry->pcallback    = pCallback;
                     pCurrentEntry->pcallbackarg = pvArg;
                     pCurrentEntry->callbackpididx = s32GetPrcIdx(pCurrentEntry->callbackpid);
                     if(pCurrentEntry->callbackpididx == -1)
                     {
                        FATAL_M_ASSERT_ALWAYS();
                     }
                     pCurrentEntry->bNotify = TRUE;
                  }
                  else
                  {
                     s32RetVal = OSAL_ERROR;
                     u32ErrorCode = OSAL_E_BUSY;
                  }
               }//if( pCallback == NULL )
            }//if((pCallback == NULL )&&(pvArg != NULL))
         }//if(pCurrentEntry->bIsUsed)
         UnLockOsal(&pOsalData->MqueueTable.rLock);
      }
   }
   else
   {
      u32ErrorCode = OSAL_E_INVALIDVALUE;
   }
   if( u32ErrorCode != OSAL_E_NOERROR )
   {
      s32RetVal = OSAL_ERROR;
      vTraceMqNotify(pCurrentEntry,u32ErrorCode);
      vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
   }
   else
   {
      if((pCurrentEntry->bTraceCannel)||(u32OsalSTrace & 0x00000020)) /*lint !e613 pointer already checked */
      {
        vTraceMqNotify(pCurrentEntry,u32ErrorCode);
      }
   }
   return(s32RetVal);
}

tS32 OSAL_s32ShMemMessageQueueStatus( OSAL_tMQueueHandle hMQ,tPU32 pu32MaxMessage,tPU32 pu32MaxLength,tPU32 pu32Message)
{
   tS32 s32ReturnValue = OSAL_ERROR;
   tU32 u32ErrorCode   = u32CheckMqHandle(hMQ);
   trMqueueElement *pCurrentEntry = NULL;

   if(u32ErrorCode == OSAL_E_NOERROR)
   {
      pCurrentEntry = ((tMQUserHandle*)hMQ)->pOrigin;/*lint !e613 *//* pointer already checked*/ 
         /* check if queue isn't deleted */
         if(pCurrentEntry->bIsUsed)
         {
            if( pu32MaxMessage != OSAL_NULL )
                  *pu32MaxMessage = pCurrentEntry->MaxMessages;
            if( pu32MaxLength  != OSAL_NULL )
                  *pu32MaxLength = pCurrentEntry->MaxLength;
            if( pu32Message    != OSAL_NULL )
                  *pu32Message     = pCurrentEntry->ActualMessage;
            s32ReturnValue = OSAL_OK;
         }
         else
         {
            u32ErrorCode =OSAL_E_DOESNOTEXIST;
         }
   }

   if( u32ErrorCode != OSAL_E_NOERROR )
   {
      if(pCurrentEntry)
      {
           vTraceMqInfo((tMQUserHandle*)hMQ,OSAL_MQ_STATUS,TR_LEVEL_FATAL,u32ErrorCode,NULL);
      }
      else
      {
           vTraceMqInfo(NULL,OSAL_MQ_STATUS,TR_LEVEL_FATAL,u32ErrorCode,NULL);
      }
      vSetErrorCode(OSAL_C_THREAD_ID_SELF,u32ErrorCode);
   }
   else
   {
      if((pCurrentEntry->bTraceCannel)||(u32OsalSTrace & 0x00000020))/*lint !e613 pointer already checked */
      {
         vTraceMqInfo((tMQUserHandle*)hMQ,OSAL_MQ_STATUS,TR_LEVEL_FATAL,u32ErrorCode,NULL);
      }
   }
   return( s32ReturnValue );
}

#endif

////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// ´DBUS variant
////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#ifdef DBUS_MQ_SUPPORT
tS32 OSAL_s32DBusMessageQueueCreate(tCString coszName,
                                tU32 u32MaxMessages,
                                tU32 u32MaxLength,
                                OSAL_tenAccess enAccess,
                                OSAL_tMQueueHandle* phMQ)
{
   tS32 s32ReturnValue= OSAL_ERROR;
   tU32 u32ErrorCode=OSAL_E_NOERROR;
   trMqueueElement *pCurrentEntry = NULL;
   tU32 u32Loop = 0;

   
   /* The name is not NULL */
   /* Parameter check */

   if( coszName && phMQ && (enAccess <= OSAL_EN_READWRITE) )
   {
      /* The Name is not too long */
      if( OSAL_u32StringLength(coszName) < LINUX_C_MQ_MAX_NAMELENGHT )
      {
         if(LockOsal(&pOsalData->MqueueTable.rLock) == OSAL_OK)
         {
            pCurrentEntry = tMqueueTableSearchEntryByName(coszName);
            if( pCurrentEntry == OSAL_NULL )
            {
               if((pCurrentEntry = tMqueueTableGetFreeEntry())!= OSAL_NULL )
               {
                    pCurrentEntry->bIsUsed = TRUE;
                    pCurrentEntry->bTraceCannel = pOsalData->bMqTaceAll;
                    pOsalData->u32MqResCount++;
                    if(pOsalData->u32MaxMqResCount < pOsalData->u32MqResCount)
                    {
                         pOsalData->u32MaxMqResCount = pOsalData->u32MqResCount;
                         if(pOsalData->u32MaxMqResCount > (pOsalData->u32MaxNrMqElements*9/10))
                         {
                            pOsalData->u32NrOfChanges++;
                         }
                     }
                     pCurrentEntry->pcallback = NULL;
                     pCurrentEntry->pcallbackarg = NULL;
                     pCurrentEntry->callbacktid = NOTATHREAD;
                     pCurrentEntry->callbackpid = NOTATHREAD;
                     pCurrentEntry->callbackpididx = NOTATHREAD;
                     pCurrentEntry->bNotify = FALSE;
#ifdef USE_EVENT_AS_CB
                     pCurrentEntry->mask = (OSAL_tEventMask) 0;
                     pCurrentEntry->enFlags = (OSAL_tenEventMaskFlag) 0;
#endif
                     pCurrentEntry->u16Type = MQ_DBUS;
                     pCurrentEntry->bOverflow = FALSE;
                     pCurrentEntry->PID = OSAL_ProcessWhoAmI();
                     pCurrentEntry->s32LiHandle = s32ReturnValue;
                     /* export the handle */
                     if((u32ErrorCode = u32GenerateHandle(enAccess, pCurrentEntry, phMQ)) != OSAL_E_NOERROR)
                     {
                        TraceString("MQ(%s): u32GenerateHandle Error:%d", coszName, u32ErrorCode);
                        FATAL_M_ASSERT_ALWAYS();
                     }
                     (void)OSAL_szStringNCopy((tString)pCurrentEntry->szName,
                                               coszName,
                                               LINUX_C_MQ_MAX_NAMELENGHT);
                     pCurrentEntry->MaxMessages   = u32MaxMessages;
                     pCurrentEntry->MaxLength     = u32MaxLength;
                     pCurrentEntry->ActualMessage = 0;
                     /* init the other MQ fields */
                     pCurrentEntry->bToDelete = FALSE;
                     pCurrentEntry->u16OpenCounter = 1;
                     pCurrentEntry->RecTsk = 0;
                     pCurrentEntry->RecPrc = 0;
                     pCurrentEntry->u32IdxMPT    = 0;
                     if(pConnectDBus)
                     {
                        if((s32ReturnValue = pConnectDBus(*phMQ,u32MaxLength,TRUE)) == OSAL_ERROR)
                        {
                           pCurrentEntry->bIsUsed = TRUE;
                        }
                     }
                     pCurrentEntry->u32MessagePrcessing = MEASUREMENT_SWITCHOFF;
                     for( u32Loop=0; u32Loop < OSAL_C_MSG_TIME_ARR_SIZE; u32Loop++ )
                     {
                        pCurrentEntry->u32MsgPrcTim[u32Loop]   = 0;
                        pCurrentEntry->u32MsgTimStamp[u32Loop] = 0;
                     }
                     pCurrentEntry->u32IdxLMPT    = 0;
                     for( u32Loop=0; u32Loop < OSAL_C_LONG_MSG_TIME_ARR_SIZE; u32Loop++ )
                     {
                        pCurrentEntry->u32LongMsgPrcTim[u32Loop]   = 0;
                        pCurrentEntry->u32LongMsgTimStamp[u32Loop] = 0;
                     }
                     pCurrentEntry->u16WaitCounter = 0;
                     pCurrentEntry->bBlocked = FALSE;
                     pCurrentEntry->u32Marker = 0;
               }
               else
               {
                  vWritePrintfErrmem("From %d OSAL Message Queues %d currently in use (Max used:%d) \n",(int)pOsalData->u32MaxNrMqElements,(int)pOsalData->u32MqResCount,(int)pOsalData->u32MaxMqResCount);
                  u32ErrorCode = OSAL_E_NOSPACE;
                  NORMAL_M_ASSERT_ALWAYS();
               }//if((pCurrentEntry = tMqueueTableGetFreeEntry())!= OSAL_NULL )
            }
            else
            {
               u32ErrorCode = OSAL_E_ALREADYEXISTS;
            }//if( pCurrentEntry == OSAL_NULL )
            /* UnLock the table */
            UnLockOsal(&pOsalData->MqueueTable.rLock);
         }
      }
      else
      {
         u32ErrorCode = OSAL_E_NAMETOOLONG;
      }
   }
   else
   {
      u32ErrorCode = OSAL_E_INVALIDVALUE;
   }

   if((u32ErrorCode != OSAL_E_NOERROR )||(phMQ == NULL))
   {
      s32ReturnValue = OSAL_ERROR;
      if(u32ErrorCode == OSAL_E_ALREADYEXISTS)
      {
        if((pCurrentEntry&&(pCurrentEntry->bTraceCannel == TRUE))||(u32OsalSTrace & 0x00000020))
        {
           vTraceMqInfo(NULL,OSAL_MQ_CREATE,FALSE,u32ErrorCode,coszName);
        }
      }
      else
      {
           vTraceMqInfo(NULL,OSAL_MQ_CREATE,TRUE,u32ErrorCode,coszName);
      }
      vSetErrorCode(OSAL_C_THREAD_ID_SELF, u32ErrorCode);
   }
   else
   {
      if((pCurrentEntry->bTraceCannel == TRUE)||(u32OsalSTrace & 0x00000020)) /*lint !e613 pCurrentEntry already checked */
      {
         vTraceMqInfo((tMQUserHandle*)*phMQ,OSAL_MQ_CREATE,FALSE,u32ErrorCode,coszName);
      }
      s32ReturnValue= OSAL_OK;
   }
   return( s32ReturnValue );
}

tS32 OSAL_s32DBusMessageQueueDelete(tCString coszName)
{
  tS32 s32ReturnValue=OSAL_ERROR;
  tU32 u32ErrorCode=OSAL_E_NOERROR;
  trMqueueElement *pCurrentEntry = NULL;
  if( coszName )
  {
    pCurrentEntry = tMqueueTableSearchEntryByName(coszName);
    if(pCurrentEntry)
    {
       if(LockOsal(&pOsalData->MqueueTable.rLock) == OSAL_OK)
       {
           pCurrentEntry->bToDelete = TRUE;
           if(pCurrentEntry->u16OpenCounter == 0)
           {
               pRelIfDBus();
               if((pCurrentEntry->bTraceCannel == TRUE)||(u32OsalSTrace & 0x00000020))/*lint !e613 pointer already checked */
               { 
                  vTraceMqInfo(NULL,OSAL_TSK_DELETE_MQ,FALSE,u32ErrorCode,coszName);
               }
               pOsalData->u32MqResCount--;
               vResetEntry(pCurrentEntry->u32EntryIdx);
           }
           s32ReturnValue = OSAL_OK;
           UnLockOsal(&pOsalData->MqueueTable.rLock);
       }
    }
    else
    {
       u32ErrorCode = OSAL_E_DOESNOTEXIST;
    }
  }
  else
  {
       u32ErrorCode = OSAL_E_INVALIDVALUE;
  }

  if( u32ErrorCode != OSAL_E_NOERROR )
  {
      vTraceMqInfo(NULL,OSAL_TSK_DELETE_MQ,TRUE,u32ErrorCode,coszName);
      vSetErrorCode(OSAL_C_THREAD_ID_SELF, u32ErrorCode);
  }
  return( s32ReturnValue );
}

/*****************************************************************************
 *
 * FUNCTION: OSAL_s32MessageQueueOpen
 *
 * DESCRIPTION: this function opens an existing OSAL event
 *
 * PARAMETER:   coszName (I)
 *                 event name to create.
 *              phEvent (->O)
 *                 pointer to the event handle.
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.10.05  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
tS32 OSAL_s32DBusMessageQueueOpen (tCString coszName,
                               OSAL_tenAccess enAccess,
                               OSAL_tMQueueHandle* phMQ)
{
   tS32 s32ReturnValue=OSAL_ERROR;
   tU32 u32ErrorCode=OSAL_E_NOERROR;
   trMqueueElement *pCurrentEntry = NULL;

   /* Make handle invalid */
   if(phMQ)
   {
        *phMQ=OSAL_C_INVALID_HANDLE;
   }

   if( coszName && phMQ && (enAccess <= OSAL_EN_READWRITE) )
   {
      if(LockOsal(&pOsalData->MqueueTable.rLock) == OSAL_OK)
      {
         pCurrentEntry = tMqueueTableSearchEntryByName(coszName);
         if(pCurrentEntry != NULL)
         {
            if(pCurrentEntry->bToDelete)
            {  
//               u32ErrorCode = OSAL_E_NOPERMISSION;
                vWritePrintfErrmem("TID:%d MQ Open for %s Delete Marked Queue OpenCnt:%d \n",
                                   OSAL_ThreadWhoAmI(),pCurrentEntry->szName,pCurrentEntry->u16OpenCounter);
            }//if( pCurrentEntry!= OSAL_NULL )
          //  else
            {
               u32ErrorCode = u32GenerateHandle(enAccess,pCurrentEntry,phMQ);
               if(pConnectDBus)
               {
                   s32ReturnValue = pConnectDBus(*phMQ, pCurrentEntry->MaxLength,FALSE);
               }
               if(s32ReturnValue == OSAL_OK)
               {
                  pCurrentEntry->u16OpenCounter++;
               }
               else
               {
                   NORMAL_M_ASSERT_ALWAYS();
               }
             }
         }// if(pCurrentEntry != NULL)
         else
         {
            u32ErrorCode = OSAL_E_DOESNOTEXIST;
         }//if( pCurrentEntry!= OSAL_NULL )
      }
      UnLockOsal(&pOsalData->MqueueTable.rLock);
   }
   else
   {
      u32ErrorCode = OSAL_E_INVALIDVALUE;
   }

   if((u32ErrorCode != OSAL_E_NOERROR )||(phMQ == NULL))
   {
      if(u32ErrorCode == OSAL_E_DOESNOTEXIST)
      {
        if((pCurrentEntry&&(pCurrentEntry->bTraceCannel == TRUE))||(u32OsalSTrace & 0x00000020))
        {
           vTraceMqInfo(NULL,OSAL_TSK_CONNECT_TO_MQ,FALSE,u32ErrorCode,coszName);
        }
      }
      else
      {
         vTraceMqInfo(NULL,OSAL_TSK_CONNECT_TO_MQ,TRUE,u32ErrorCode,coszName);
      }
      vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
   }
   else
   {
      if((pCurrentEntry->bTraceCannel == TRUE)||(u32OsalSTrace & 0x00000020))/*lint !e613 pointer already checked */
      {
         vTraceMqInfo((tMQUserHandle*)*phMQ,OSAL_TSK_CONNECT_TO_MQ,FALSE,u32ErrorCode,coszName);
      }
     s32ReturnValue = OSAL_OK;
   }
   return( s32ReturnValue );
}


/*****************************************************************************
 *
 * FUNCTION: OSAL_s32MessageQueueClose
 *
 * DESCRIPTION: this function creates an OSAL event.
 *
 * PARAMETER:   coszName (I)
 *                 event name to create.
 *              phEvent (->O)
 *                 pointer to the event handle.
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.10.05  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
tS32 OSAL_s32DBusMessageQueueClose (OSAL_tMQueueHandle phMQueue)
{
  tS32 s32ReturnValue = OSAL_ERROR;
  tU32 u32ErrorCode   = OSAL_E_NOERROR; // done in OSAL_s32MessageQueueClose u32CheckMqHandle(phMQueue);
  trMqueueElement *pCurrentEntry = NULL;

  if(u32ErrorCode == OSAL_E_NOERROR)
  {
     if(LockOsal(&pOsalData->MqueueTable.rLock) == OSAL_OK)
     {
           pCurrentEntry = ((tMQUserHandle*)phMQueue)->pOrigin;/*lint !e613 *//* pointer already checked*/ 
           if(pCurrentEntry->u16OpenCounter > 0 )
           {
               pCurrentEntry->u16OpenCounter--;
               if(pDisConnectDBus)
               {
                  s32ReturnValue = pDisConnectDBus(phMQueue);
               }
               if( !pCurrentEntry->u16OpenCounter && pCurrentEntry->bToDelete )
               {
                  if(u32ErrorCode == OSAL_E_NOERROR)
                  {
                     if((pCurrentEntry->bTraceCannel == TRUE)||(u32OsalSTrace & 0x00000020))
                     {
                        vTraceMqInfo((tMQUserHandle*)phMQueue,OSAL_TSK_DISCONNECT_MQ,FALSE,u32ErrorCode,pCurrentEntry->szName);
                     }
                     pOsalData->u32MqResCount--;
                     vResetEntry(pCurrentEntry->u32EntryIdx);
                  }//if(u32ErrorCode == OSAL_E_NOERROR)					  
               }//if( !pCurrentEntry->u16OpenCounter && pCurrentEntry->bToDelete )
               /*destroy magic*/
               ((tMQUserHandle*)phMQueue)->pOrigin = NULL;/*lint !e613 *//* pointer already checked*/ 
               ((tMQUserHandle*)phMQueue)->PrcId = 0;
               if(OSAL_s32MemPoolFixSizeRelBlockOfPool(&MqMemPoolHandle,(void*)phMQueue) == OSAL_ERROR)
               {    NORMAL_M_ASSERT_ALWAYS();  }
               phMQueue = 0;
               s32ReturnValue = OSAL_OK;
            }
            else
            {
               TraceString("OSALMQ Unexpected Open counter 0 for MQ %s",pCurrentEntry->szName);
               u32ErrorCode = OSAL_E_UNKNOWN;
            }
        UnLockOsal(&pOsalData->MqueueTable.rLock);
     }
  }

  if( u32ErrorCode != OSAL_E_NOERROR )
  {
     tCString szName = "";
     if(pCurrentEntry)
     {
        szName = pCurrentEntry->szName;
     }
      vTraceMqInfo(NULL,OSAL_TSK_DISCONNECT_MQ,TRUE,u32ErrorCode,szName);
      vSetErrorCode( OSAL_C_THREAD_ID_SELF,u32ErrorCode);
  }
  return( s32ReturnValue );
}


tS32 OSAL_s32DBusMessageQueuePost( OSAL_tMQueueHandle hMQ,
                               tPCU8 pcou8Msg,
                               tU32  u32Length,
                               tU32  u32Prio)
{
   tS32 s32ReturnValue = OSAL_OK;
   tU32 u32ErrorCode   = OSAL_E_NOERROR; // done in OSAL_s32MessageQueuePost u32CheckMqHandle(hMQ); /* step 1 user handle check */
   trMqueueElement *pCurrentEntry = NULL;
   tU32 u32ActualMessage = 0xffffffff;
   tU32 msec = 0;
   tU32 u32Temp;
   char name[TRACE_NAME_SIZE];
 
   if(u32ErrorCode == OSAL_E_NOERROR) 
   {
      /* check permissions*/
      if(((tMQUserHandle*)hMQ)->enAccess == OSAL_EN_READONLY)
      {
         u32ErrorCode = OSAL_E_NOPERMISSION;
      }
   }
   if(u32ErrorCode == OSAL_E_NOERROR) 
   {
      /* step 2 uparameter check */
      pCurrentEntry = ((tMQUserHandle*)hMQ)->pOrigin; /*lint !e613 *//* pointer already checked*/ 
/*         if((pCurrentEntry->u32EntryIdx >= pOsalData->MqueueTable.u32UsedEntries)||(pCurrentEntry->u32MqueueID != LINUX_C_U32_MQUEUE_ID))
         {
            CheckandCorrectMqStaticValues();
         }*/

         if( u32Prio < (OSAL_C_U32_MQUEUE_PRIORITY_LOWEST + 1) )
         {  
             /* check if queue isn't deleted */
            if(u32Length > pCurrentEntry->MaxLength)
            {
               u32ErrorCode = OSAL_E_MSGTOOLONG;
            }
            else
            {
            }

            if(u32ErrorCode == OSAL_E_NOERROR) 
            {
               msec = pOsalData->u32MqPostTmo;
               /* step 3 check for full message queue and wait , when queue is greater minimum size*/
               if((pCurrentEntry->ActualMessage == pCurrentEntry->MaxMessages)&&(pCurrentEntry->MaxMessages > 1))
               {
               }
            }
            if(u32ErrorCode == OSAL_E_NOERROR) 
            {
               if(pCurrentEntry->bTraceCannel == TRUE)
               {
                   TraceString("OSALMQ TID:%d Start Send Timeout:%d msec MQ:%s",OSAL_ThreadWhoAmI(),msec,pCurrentEntry->szName);
               }
               if(pMsgSendDBus)
               {
                   s32ReturnValue = pMsgSendDBus(hMQ,(char*)pcou8Msg,u32Length);
               }
               if(s32ReturnValue != -1)
               {
                   if(u32ActualMessage == 0xffffffff)
                   {
                      u32ActualMessage = pCurrentEntry->ActualMessage++;
                   }
               }
            } //if(u32ErrorCode == OSAL_E_NOERROR)
      }
      else
      {
         u32ErrorCode = OSAL_E_INVALIDVALUE;
      }
   }//if(u32ErrorCode == OSAL_E_NOERROR) 

#ifdef MQ_NOTIFY_VIA_OSAL
   if((u32ErrorCode == OSAL_E_NOERROR)&&(pCurrentEntry)&&(pCurrentEntry->bTriggerCb == TRUE))
   {
      if((pCurrentEntry->callbackpididx < (tS32)pOsalData->u32MaxNrProcElements)
      && (pCurrentEntry->pcallback )&&(pCurrentEntry->callbackpididx >= 0))
      {
         pCurrentEntry->bTriggerCb = FALSE;
         if( pCurrentEntry->callbackpid == OSAL_ProcessWhoAmI()) // check if process is the same
         {
            vExecuteCallback(pCurrentEntry);
         }
         else
         {
            vPostToCallbackQueue(pCurrentEntry);
         }
      }
#endif
#ifdef USE_EVENT_AS_CB
      if((pCurrentEntry->mask))
      {
         pCurrentEntry->bTriggerCb = FALSE;
         if(((tMQUserHandle*)hMQ)->hNotifyEvent == 0)
         {
            if(OSAL_s32EventOpen(pCurrentEntry->szEvName,&((tMQUserHandle*)hMQ)->hNotifyEvent) != OSAL_OK)
            {
               NORMAL_M_ASSERT_ALWAYS();
            }
         }
         OSAL_s32EventPost(((tMQUserHandle*)hMQ)->hNotifyEvent, pCurrentEntry->mask, pCurrentEntry->enFlags);
      }
#endif
   }

   if( u32ErrorCode != OSAL_E_NOERROR )
   {
      s32ReturnValue = OSAL_ERROR;
      if((u32ErrorCode != OSAL_E_TIMEOUT)||
        ((pCurrentEntry)&&(pCurrentEntry->bTraceCannel == TRUE))||(u32OsalSTrace & 0x00000020))
      {
          u32Temp = (tU32)OSAL_ThreadWhoAmI();
          bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,u32Temp);
          if(pCurrentEntry)
          {
             TraceString("OSAL_MQ Task:%s(ID:%d) Send      to MQ: User Handle %p MQ Name:%s Msg Count:%d Error:0x%x Prio:%d",
                      name,(unsigned int)u32Temp,((tMQUserHandle*)hMQ)->pOrigin, (((tMQUserHandle*)hMQ)->pOrigin)->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32Prio);
             if((pOsalData->bLogError)&&(u32ErrorCode != OSAL_E_TIMEOUT)&&(u32OsalSTrace & ~0x00000020))vWritePrintfErrmem("OSAL_MQ Task:%s(ID:%d) Send      to MQ: User Handle %p MQ Name:%s Msg Count:%d Error:0x%x Prio:%d \n",
                      name,(unsigned int)u32Temp,((tMQUserHandle*)hMQ)->pOrigin, (((tMQUserHandle*)hMQ)->pOrigin)->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32Prio);
          }
          else
          {
             TraceString("OSAL_MQ Task:%s(ID:%d) Send      to MQ: User Handle 0x%x MQ Name:%s Msg Count:%d Error:0x%x Prio:%d",
                      name,(unsigned int)u32Temp,(unsigned int)0, "Unknown",(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32Prio);
             if((pOsalData->bLogError)&&(u32ErrorCode != OSAL_E_TIMEOUT)&&(u32OsalSTrace & ~0x00000020))vWritePrintfErrmem("OSAL_MQ Task:%s(ID:%d) Send      to MQ: User Handle 0x%x MQ Name:%s Msg Count:%d Error:0x%x Prio:%d \n",
                      name,(unsigned int)u32Temp,(unsigned int)0, "Unknown",(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32Prio);
          }
      }
      vSetErrorCode(OSAL_C_THREAD_ID_SELF, u32ErrorCode);
   }
   else
   {
       if((pCurrentEntry->bTraceCannel == TRUE)||(u32OsalSTrace & 0x00000020))/*lint !e613 pCurrentEntry already checked */
       {
           u32Temp = (tU32)OSAL_ThreadWhoAmI();
           bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,u32Temp);
           TraceString("OSAL_MQ Task:%s(ID:%d) Send      to MQ: User Handle %p MQ Name:%s Msg Count:%d Error:0x%x Prio:%d",
                      name,(unsigned int)u32Temp,((tMQUserHandle*)hMQ)->pOrigin, (((tMQUserHandle*)hMQ)->pOrigin)->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32Prio);
       }
   }
   return( s32ReturnValue );
}

tS32 OSAL_s32DBusMessageQueueWait( OSAL_tMQueueHandle hMQ,
                               tPU8 pu8Buffer,
                               tU32 u32Length,
                               tPU32 pu32Prio,
                               OSAL_tMSecond msec)
{
   tU32 u32CopiedBytes = 0;
   tU32 u32ErrorCode   = OSAL_E_NOERROR;//u32CheckMqHandle(hMQ); /* done in OSAL_s32MessageQueueWait */
   tU32 u32ActualPrio = 0xffffffff;
   trMqueueElement *pCurrentEntry = NULL;
   tU32 u32ActualMessage = 0xffffffff;
   tU32 u32Temp = 0;
   char name[TRACE_NAME_SIZE];
   ((void)pu32Prio);
   if(u32ErrorCode == OSAL_E_NOERROR) 
   {
      /* check permissions*/
      if(((tMQUserHandle*)hMQ)->enAccess == OSAL_EN_WRITEONLY) 
      {
          u32ErrorCode = OSAL_E_NOPERMISSION;
      }
   }
  if(u32ErrorCode == OSAL_E_NOERROR)
  {
     pCurrentEntry = ((tMQUserHandle*)hMQ)->pOrigin;/*lint !e613 *//* pointer already checked*/ 
        if(pCurrentEntry->bToDelete == TRUE)
        {
           msec = OSAL_C_TIMEOUT_NOBLOCKING;
        }
        pCurrentEntry->u16WaitCounter++;

        /* Stop message processing time measurement */
        vStopMeasureProcessingTime(pCurrentEntry);
        if(pMsgRecvDBus)
        {
           u32CopiedBytes = pMsgRecvDBus(hMQ,(char*)pu8Buffer,u32Length,msec);
		}
        if(pCurrentEntry ->RecTsk == 0)
        {
           pCurrentEntry->RecTsk = OSAL_ThreadWhoAmI();
           pCurrentEntry->RecPrc = OSAL_ProcessWhoAmI();
        }
#ifdef MQ_NOTIFY_VIA_OSAL
        if(u32ErrorCode == OSAL_E_TIMEOUT)
        {
           pCurrentEntry->bTriggerCb = TRUE;/*lint !e613 when u32ErrorCode == OSAL_E_TIMEOUT valid pCurrentEntry*/
        }
#endif
        if(u32CopiedBytes)
        {
           u32ActualMessage = pCurrentEntry->ActualMessage--;
           /* Start measurement of message processing times */
           vStartMeasureProcessingTime(pCurrentEntry);           
        } //if(u32CopiedBytes)
  }//if(u32ErrorCode == OSAL_E_NOERROR)



  if( u32ErrorCode != OSAL_E_NOERROR )
  {
     if((u32ErrorCode != OSAL_E_TIMEOUT)||
        ((pCurrentEntry)&&(pCurrentEntry->bTraceCannel == TRUE))||(u32OsalSTrace & 0x00000020))
     {
         u32Temp = (tU32)OSAL_ThreadWhoAmI();
         bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,OSAL_ThreadWhoAmI());
         if(pCurrentEntry)
         {
            TraceString("OSAL_MQ Task:%s(ID:%d) Receive from MQ: User Handle %p Name:%s Msg Count:%d Error:0x%x Prio:%d",
                      name,(unsigned int)u32Temp,((tMQUserHandle*)hMQ)->pOrigin,(((tMQUserHandle*)hMQ)->pOrigin)->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32ActualPrio);
            if((pOsalData->bLogError)&&(u32ErrorCode != OSAL_E_TIMEOUT)&&(u32OsalSTrace & ~0x00000020))vWritePrintfErrmem("OSAL_MQ Task:%s(ID:%d) Receive from MQ: User Handle %p Name:%s Msg Count:%d Error:0x%x Prio:%d \n",
                      name,(unsigned int)u32Temp,((tMQUserHandle*)hMQ)->pOrigin,(((tMQUserHandle*)hMQ)->pOrigin)->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32ActualPrio);
         }
         else
         {
             TraceString("OSAL_MQ Task:%s(ID:%d) Receive from MQ: User Handle 0x%x Name:%s Msg Count:%d Error:0x%x Prio:%d",
                      name,(unsigned int)u32Temp,(unsigned int)u32ErrorCode,"Unknown",(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32ActualPrio);
             if((pOsalData->bLogError)&&(u32ErrorCode != OSAL_E_TIMEOUT)&&(u32OsalSTrace & ~0x00000020))vWritePrintfErrmem("OSAL_MQ Task:%s(ID:%d) Receive from MQ: User Handle 0x%x Name:%s Msg Count:%d Error:0x%x Prio:%d \n",
                      name,(unsigned int)u32Temp,(unsigned int)u32ErrorCode,"Unknown",(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32ActualPrio);
         }
     }
     vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
  }
  else
  {
     if(((pCurrentEntry) &&(pCurrentEntry->bTraceCannel == TRUE))||(u32OsalSTrace & 0x00000020))
     {
        u32Temp = (tU32)OSAL_ThreadWhoAmI();
        bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,OSAL_ThreadWhoAmI());
        TraceString("OSAL_MQ Task:%s(ID:%d) Receive from MQ: User Handle %p Name:%s Msg Count:%d Error:0x%x Prio:%d",
                 name,(unsigned int)u32Temp,((tMQUserHandle*)hMQ)->pOrigin,(((tMQUserHandle*)hMQ)->pOrigin)->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32ActualPrio);
     }
  }
  return((tS32)u32CopiedBytes );
}
#endif // DBUS_MQ_SUPPORT

#ifdef OSAL_QNX_MQ

void got_pulse(struct _pulse *pulse) 
{
    if(pulse->type == _PULSE_CODE_COIDDEATH) 
	{
        int coid = pulse->value.sival_int;

        if(ConnectServerInfo(0, coid, NULL) != coid) 
		{
            // server's really gone, so clean up the connection state
        } 
		else 
		{
            // stale pulse; probably can ignore it
        }
    }
}

/*****************************************************************************
 *
 * FUNCTION: ONX_MessageQueueCreate
 *
 * DESCRIPTION: Create a MQ with a given max number of messages, each of a
 *              given MaxLenght, with given access right
 *
 * PARAMETER:
 *
 *   u16ContextID                index to distinguish caller context
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 06.12.18  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
int QNX_MessageQueueCreate(tCString coszName, tU32 u32MaxMessages, tU32 u32MaxLength,
    OSAL_tenAccess enAccess, OSAL_tMQueueHandle* phMQ)
{
   tS32 s32ReturnValue= OSAL_ERROR;
   tU32 u32ErrorCode=OSAL_E_NOERROR;
   trMqueueElement *pCurrentEntry = NULL;
   tU32 u32Loop = 0;

   vMqueueLiMqMapInit();
 
   if( coszName && phMQ && (enAccess <= OSAL_EN_READWRITE) )
   {
      /* The Name is not too long */
      if( OSAL_u32StringLength(coszName) < LINUX_C_MQ_MAX_NAMELENGHT )
      {
         if(LockOsal(&pOsalData->MqueueTable.rLock) == OSAL_OK)
         {
            pCurrentEntry = tMqueueTableSearchEntryByName(coszName);
            if( pCurrentEntry == OSAL_NULL )
            {
               if((pCurrentEntry = tMqueueTableGetFreeEntry())!= OSAL_NULL )
               {
                /*  if((s32ReturnValue = ChannelCreate (_NTO_CHF_UNBLOCK|_NTO_SIDE_CHANNEL) == -1))
                  {
                     u32ErrorCode = u32ConvertErrorCore(errno);
                  }
                  else*/
                  {
                     char Name[64];
                     snprintf(Name,64,"osal/%s",coszName);
                     if((prLiMqMap[pCurrentEntry->u32EntryIdx].attach = name_attach(NULL,Name, 0/*NAME_FLAG_ATTACH_GLOBAL*/)) == NULL)
//                    if((s32ReturnValue = ConnectAttach(0,0,s32ReturnValue,100,0)) == -1)
                     {
                       u32ErrorCode = u32ConvertErrorCore(errno);
                    }
                    else
                    {
					   s32ReturnValue = prLiMqMap[pCurrentEntry->u32EntryIdx].attach->chid;
                       prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId = (tU32)s32ReturnValue;
                       prLiMqMap[pCurrentEntry->u32EntryIdx].PrcId = MqPid;
					   prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt = 1;
                       pCurrentEntry->s32LiHandle = s32ReturnValue;
                       pCurrentEntry->bIsUsed = TRUE;
                       pCurrentEntry->bIsUsed = TRUE;
                       pCurrentEntry->bTraceCannel = pOsalData->bMqTaceAll;
                       pOsalData->u32MqResCount++;
                       if(pOsalData->u32MaxMqResCount < pOsalData->u32MqResCount)
                       {
                          pOsalData->u32MaxMqResCount = pOsalData->u32MqResCount;
                          if(pOsalData->u32MaxMqResCount > (pOsalData->u32MaxNrMqElements*9/10))
                          {
                             pOsalData->u32NrOfChanges++;
                          }
                       }
                       pCurrentEntry->pcallback = NULL;
                       pCurrentEntry->pcallbackarg = NULL;
                       pCurrentEntry->callbacktid = NOTATHREAD;
                       pCurrentEntry->callbackpid = NOTATHREAD;
                       pCurrentEntry->callbackpididx = NOTATHREAD;
                       pCurrentEntry->bNotify = FALSE;
#ifdef USE_EVENT_AS_CB
                       pCurrentEntry->mask = (OSAL_tEventMask) 0;
                       pCurrentEntry->enFlags = (OSAL_tenEventMaskFlag) 0;
#endif
                       pCurrentEntry->u16Type = MQ_QNX;
                       pCurrentEntry->bOverflow = FALSE;
                       pCurrentEntry->PID = OSAL_ProcessWhoAmI();
                       /* export the handle */
                       if((u32ErrorCode = u32GenerateHandle(enAccess, pCurrentEntry, phMQ)) != OSAL_E_NOERROR)
                       {
                          TraceString("MQ(%s): u32GenerateHandle Error:%d", coszName, u32ErrorCode);
                          FATAL_M_ASSERT_ALWAYS();
                       }
                       (void)OSAL_szStringNCopy((tString)pCurrentEntry->szName,
                                                 coszName,
                                                 LINUX_C_MQ_MAX_NAMELENGHT);
                       pCurrentEntry->MaxMessages   = u32MaxMessages;
                       /*ensure that pulses can be received */
                       if(u32MaxLength > sizeof(struct _pulse))
                       {  pCurrentEntry->MaxLength = u32MaxLength;   }
                       else
                       {  pCurrentEntry->MaxLength = sizeof(struct _pulse);  }//u32MaxLength; 
                       pCurrentEntry->ActualMessage = 0;
                       /* init the other MQ fields */
                       pCurrentEntry->bToDelete = FALSE;
                       pCurrentEntry->u16OpenCounter = 1;
                       pCurrentEntry->RecTsk = 0;
                       pCurrentEntry->RecPrc = 0;
                       pCurrentEntry->u32IdxMPT    = 0;

                       pCurrentEntry->u32MessagePrcessing = MEASUREMENT_SWITCHOFF;
                       for( u32Loop=0; u32Loop < OSAL_C_MSG_TIME_ARR_SIZE; u32Loop++ )
                       {
                          pCurrentEntry->u32MsgPrcTim[u32Loop]   = 0;
                          pCurrentEntry->u32MsgTimStamp[u32Loop] = 0;
                       }
                       pCurrentEntry->u32IdxLMPT    = 0;
                       for( u32Loop=0; u32Loop < OSAL_C_LONG_MSG_TIME_ARR_SIZE; u32Loop++ )
                       {
                          pCurrentEntry->u32LongMsgPrcTim[u32Loop]   = 0;
                          pCurrentEntry->u32LongMsgTimStamp[u32Loop] = 0;
                       }
                       pCurrentEntry->u16WaitCounter = 0;
                       pCurrentEntry->bBlocked = FALSE;
                       pCurrentEntry->u32Marker = 0;
                     }
                  }
               }
               else
               {
                  vWritePrintfErrmem("From %d OSAL Message Queues %d currently in use (Max used:%d) \n",(int)pOsalData->u32MaxNrMqElements,(int)pOsalData->u32MqResCount,(int)pOsalData->u32MaxMqResCount);
                  u32ErrorCode = OSAL_E_NOSPACE;
                  NORMAL_M_ASSERT_ALWAYS();
               }//if((pCurrentEntry = tMqueueTableGetFreeEntry())!= OSAL_NULL )
            }
            else
            {
               u32ErrorCode = OSAL_E_ALREADYEXISTS;
            }//if( pCurrentEntry == OSAL_NULL )
            /* UnLock the table */
            UnLockOsal(&pOsalData->MqueueTable.rLock);
         }
      }
      else
      {
         u32ErrorCode = OSAL_E_NAMETOOLONG;
      }
   }
   else
   {
      u32ErrorCode = OSAL_E_INVALIDVALUE;
   }

   if((u32ErrorCode != OSAL_E_NOERROR )||(phMQ == NULL))
   {
      s32ReturnValue = OSAL_ERROR;
      if(u32ErrorCode == OSAL_E_ALREADYEXISTS)
      {
        if((pCurrentEntry&&(pCurrentEntry->bTraceCannel == TRUE))||(u32OsalSTrace & 0x00000020))
        {
           vTraceMqInfo(NULL,OSAL_MQ_CREATE,FALSE,u32ErrorCode,coszName);
        }
      }
      else
      {
           vTraceMqInfo(NULL,OSAL_MQ_CREATE,TRUE,u32ErrorCode,coszName);
      }
      vSetErrorCode(OSAL_C_THREAD_ID_SELF, u32ErrorCode);
   }
   else
   {
      if((pCurrentEntry->bTraceCannel == TRUE)||(u32OsalSTrace & 0x00000020)) /*lint !e613 pCurrentEntry already checked */
      {
         vTraceMqInfo((tMQUserHandle*)*phMQ,OSAL_MQ_CREATE,FALSE,u32ErrorCode,coszName);
      }
      s32ReturnValue= OSAL_OK;
   }
   return( s32ReturnValue );
		
}   

/*****************************************************************************
 *
 * FUNCTION: QNX_MessageQueueDelete
 *
 * DESCRIPTION: this function destroys an OSAL event.
 *
 * PARAMETER:   coszName (I)
 *                 event name to create.
 *              phEvent (->O)
 *                 pointer to the event handle.
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 06.12.18  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
int QNX_MessageQueueDelete(tCString coszName)
{
  tS32 s32ReturnValue=OSAL_ERROR;
  tU32 u32ErrorCode=OSAL_E_NOERROR;
  trMqueueElement *pCurrentEntry = NULL;
  if( coszName )
  {
    pCurrentEntry = tMqueueTableSearchEntryByName(coszName);
    if(pCurrentEntry)
    {
       if(LockOsal(&pOsalData->MqueueTable.rLock) == OSAL_OK)
       {
           pCurrentEntry->bToDelete = TRUE;
           if(pCurrentEntry->u16OpenCounter == 0)
           {
			  if(pCurrentEntry->PID == OSAL_ProcessWhoAmI())
			  {
                 if(name_detach(prLiMqMap[pCurrentEntry->u32EntryIdx].attach, 0) == -1)
//                 if(ChannelDestroy(pCurrentEntry->s32LiHandle) == -1)
                 {
                    u32ErrorCode = u32ConvertErrorCore(errno);
                 }
                 else
		         {
                    prLiMqMap[pCurrentEntry->u32EntryIdx].attach    = NULL;
                    prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId   = -1;
                    prLiMqMap[pCurrentEntry->u32EntryIdx].PrcId     = 0;
				    prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt = 0;
			     }
			  }
			  else
			  {
				 u32ErrorCode = OSAL_E_NOPERMISSION;
			  }
              if((pCurrentEntry->bTraceCannel == TRUE)||(u32OsalSTrace & 0x00000020))/*lint !e613 pointer already checked */
              { 
                  vTraceMqInfo(NULL,OSAL_TSK_DELETE_MQ,FALSE,u32ErrorCode,coszName);
              }
              pOsalData->u32MqResCount--;
              prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt--;
              vResetEntry(pCurrentEntry->u32EntryIdx);
           }
           s32ReturnValue = OSAL_OK;
           UnLockOsal(&pOsalData->MqueueTable.rLock);
       }
    }
    else
    {
       u32ErrorCode = OSAL_E_DOESNOTEXIST;
    }
  }
  else
  {
       u32ErrorCode = OSAL_E_INVALIDVALUE;
  }

  if( u32ErrorCode != OSAL_E_NOERROR )
  {
      vTraceMqInfo(NULL,OSAL_TSK_DELETE_MQ,TRUE,u32ErrorCode,coszName);
      vSetErrorCode(OSAL_C_THREAD_ID_SELF, u32ErrorCode);
  }
  return( s32ReturnValue );
}

/*****************************************************************************
 *
 * FUNCTION: QNX_MessageQueueOpen
 *
 * DESCRIPTION: this function opens an existing OSAL event
 *
 * PARAMETER:   coszName (I)
 *                 event name to create.
 *              phEvent (->O)
 *                 pointer to the event handle.
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 06.12.18  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
int QNX_MessageQueueOpen(tCString coszName, OSAL_tenAccess enAccess, OSAL_tMQueueHandle* phMQ)
{
   tS32 s32ReturnValue=OSAL_ERROR;
   tU32 u32ErrorCode=OSAL_E_NOERROR;
   trMqueueElement *pCurrentEntry = NULL;

   vMqueueLiMqMapInit();
   
    /* Make handle invalid */
   if(phMQ)
   {
        *phMQ=OSAL_C_INVALID_HANDLE;
   }

   if( coszName && phMQ && (enAccess <= OSAL_EN_READWRITE) )
   {
      if(LockOsal(&pOsalData->MqueueTable.rLock) == OSAL_OK)
      {
         pCurrentEntry = tMqueueTableSearchEntryByName(coszName);
         if(pCurrentEntry != NULL)
         {
            if(pCurrentEntry->bToDelete)
            {  
//               u32ErrorCode = OSAL_E_NOPERMISSION;
                vWritePrintfErrmem("TID:%d MQ Open for %s Delete Marked Queue OpenCnt:%d \n",
                                   OSAL_ThreadWhoAmI(),pCurrentEntry->szName,pCurrentEntry->u16OpenCounter);
            }//if( pCurrentEntry!= OSAL_NULL )
          //  else
            {
               if(prLiMqMap[pCurrentEntry->u32EntryIdx].PrcId)
               {
                  s32ReturnValue = OSAL_OK;
               }
               else
               {
                  char Name[64];
                  snprintf(Name,64,"osal/%s",pCurrentEntry->szName);
                  if ((s32ReturnValue = name_open(Name, 0)) == -1)
                  {
                      u32ErrorCode = u32ConvertErrorCore(errno);
                  }
	              else
                  {
                     prLiMqMap[pCurrentEntry->u32EntryIdx].attach = NULL;
                     prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId = s32ReturnValue;
                     prLiMqMap[pCurrentEntry->u32EntryIdx].PrcId = OSAL_ProcessWhoAmI();
                     s32ReturnValue = OSAL_OK;
                  }
               }
               if(s32ReturnValue == OSAL_OK)
               {
				  prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt++;
                  u32ErrorCode = u32GenerateHandle(enAccess,pCurrentEntry,phMQ);
                  pCurrentEntry->u16OpenCounter++;
               }
               else
               {
                   NORMAL_M_ASSERT_ALWAYS();
               }
             }
         }// if(pCurrentEntry != NULL)
         else
         {
            u32ErrorCode = OSAL_E_DOESNOTEXIST;
         }//if( pCurrentEntry!= OSAL_NULL )
      }
      UnLockOsal(&pOsalData->MqueueTable.rLock);
   }
   else
   {
      u32ErrorCode = OSAL_E_INVALIDVALUE;
   }

   if((u32ErrorCode != OSAL_E_NOERROR )||(phMQ == NULL))
   {
      if(u32ErrorCode == OSAL_E_DOESNOTEXIST)
      {
        if((pCurrentEntry&&(pCurrentEntry->bTraceCannel == TRUE))||(u32OsalSTrace & 0x00000020))
        {
           vTraceMqInfo(NULL,OSAL_TSK_CONNECT_TO_MQ,FALSE,u32ErrorCode,coszName);
        }
      }
      else
      {
         vTraceMqInfo(NULL,OSAL_TSK_CONNECT_TO_MQ,TRUE,u32ErrorCode,coszName);
      }
      vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
   }
   else
   {
      if((pCurrentEntry->bTraceCannel == TRUE)||(u32OsalSTrace & 0x00000020))/*lint !e613 pointer already checked */
      {
         vTraceMqInfo((tMQUserHandle*)*phMQ,OSAL_TSK_CONNECT_TO_MQ,FALSE,u32ErrorCode,coszName);
      }
     s32ReturnValue = OSAL_OK;
   }
   return( s32ReturnValue );
}


/*****************************************************************************
 *
 * FUNCTION: QNX_MessageQueueClose
 *
 * DESCRIPTION: this function creates an OSAL event.
 *
 * PARAMETER:   coszName (I)
 *                 event name to create.
 *              phEvent (->O)
 *                 pointer to the event handle.
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 06.12.18  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
int QNX_MessageQueueClose(OSAL_tMQueueHandle hMQ)
{
  tS32 s32ReturnValue = OSAL_ERROR;
  tU32 u32ErrorCode   = OSAL_E_NOERROR; // done in OSAL_s32MessageQueueClose u32CheckMqHandle(phMQueue);
  trMqueueElement *pCurrentEntry = NULL;

  if(u32ErrorCode == OSAL_E_NOERROR)
  {
     if(LockOsal(&pOsalData->MqueueTable.rLock) == OSAL_OK)
     {
           pCurrentEntry = ((tMQUserHandle*)hMQ)->pOrigin;/*lint !e613 *//* pointer already checked*/ 
           if(pCurrentEntry->u16OpenCounter > 0 )
           {
               pCurrentEntry->u16OpenCounter--;
               if(!pCurrentEntry->u16OpenCounter)
               {
                  prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt--; 
                  if(prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt == 0)
                  {
                     if(pCurrentEntry->PID != OSAL_ProcessWhoAmI())
                     {
                        if(name_close(prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId) == -1)
//                      if(ConnectDetach(pCurrentEntry->s32LiHandle) == -1)
                        {
                           u32ErrorCode = u32ConvertErrorCore(errno);
                        }
                        else
                        {
                           prLiMqMap[pCurrentEntry->u32EntryIdx].attach   = NULL;
                           prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId  = -1;
                           prLiMqMap[pCurrentEntry->u32EntryIdx].PrcId    = 0;
                           prLiMqMap[pCurrentEntry->u32EntryIdx].u32PrcCnt= 0;
                        }
                     }
                  }
               }
               if( !pCurrentEntry->u16OpenCounter && pCurrentEntry->bToDelete )
               {
                  if(u32ErrorCode == OSAL_E_NOERROR)
                  {
                     if((pCurrentEntry->bTraceCannel == TRUE)||(u32OsalSTrace & 0x00000020))
                     {
                        vTraceMqInfo((tMQUserHandle*)hMQ,OSAL_TSK_DISCONNECT_MQ,FALSE,u32ErrorCode,pCurrentEntry->szName);
                     }
                     pOsalData->u32MqResCount--;
                     vResetEntry(pCurrentEntry->u32EntryIdx);
                  }//if(u32ErrorCode == OSAL_E_NOERROR)					  
               }//if( !pCurrentEntry->u16OpenCounter && pCurrentEntry->bToDelete )
               /*destroy magic*/
               ((tMQUserHandle*)hMQ)->pOrigin = NULL;/*lint !e613 *//* pointer already checked*/ 
               ((tMQUserHandle*)hMQ)->PrcId = 0;
               if(OSAL_s32MemPoolFixSizeRelBlockOfPool(&MqMemPoolHandle,(void*)hMQ) == OSAL_ERROR)
               {    NORMAL_M_ASSERT_ALWAYS();  }
               hMQ = 0;
               s32ReturnValue = OSAL_OK;
            }
            else
            {
               TraceString("OSALMQ Unexpected Open counter 0 for MQ %s",pCurrentEntry->szName);
               u32ErrorCode = OSAL_E_UNKNOWN;
            }
        UnLockOsal(&pOsalData->MqueueTable.rLock);
     }
  }

  if( u32ErrorCode != OSAL_E_NOERROR )
  {
     tCString szName = "";
     if(pCurrentEntry)
     {
        szName = pCurrentEntry->szName;
     }
      vTraceMqInfo(NULL,OSAL_TSK_DISCONNECT_MQ,TRUE,u32ErrorCode,szName);
      vSetErrorCode( OSAL_C_THREAD_ID_SELF,u32ErrorCode);
  }
  return( s32ReturnValue );
}


/*****************************************************************************
 *
 * FUNCTION: QNX_MessageQueuePost
 *
 * DESCRIPTION: Sends a message of given length and priority to a MessageQueue.
 *
 * PARAMETER:   hMQ       MessageQueue handle
 *              pcou8Msg  const pointer to the buffer containing the message
 *              u32Length message length in bytes
 *              u32Prio   message priority
 *
 * ERRORS:      OSAL_E_BADFILEDESCRIPTOR  hMQ is not a valid handle
 *              OSAL_E_INVALIDVALUE       prio exceeds the boundary
 *              OSAL_E_MSGTOOLONG         length exceeds the field MaxLength
 *                                        defined in creation
 *              OSAL_E_QUEUEFULL          the MQ is full
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 06.12.18  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 *
 *****************************************************************************/
int QNX_MessageQueuePost(OSAL_tMQueueHandle hMQ, tPCU8 pcou8Msg, tU32 u32Length, tU32 u32Prio)
{
   tS32 s32ReturnValue = OSAL_OK;
   tU32 u32ErrorCode   = OSAL_E_NOERROR; // done in OSAL_s32MessageQueuePost u32CheckMqHandle(hMQ); /* step 1 user handle check */
   trMqueueElement *pCurrentEntry = NULL;
   tU32 u32ActualMessage = 0xffffffff;
   tU32 msec = 0;
   tU32 u32Temp;
   char name[TRACE_NAME_SIZE];
 
   if(u32ErrorCode == OSAL_E_NOERROR) 
   {
      /* check permissions*/
      if(((tMQUserHandle*)hMQ)->enAccess == OSAL_EN_READONLY)
      {
         u32ErrorCode = OSAL_E_NOPERMISSION;
      }
   }
   if(u32ErrorCode == OSAL_E_NOERROR) 
   {
      /* step 2 uparameter check */
      pCurrentEntry = ((tMQUserHandle*)hMQ)->pOrigin; /*lint !e613 *//* pointer already checked*/ 
      if(pCurrentEntry)
      {
/*         if((pCurrentEntry->u32EntryIdx >= pOsalData->MqueueTable.u32UsedEntries)||(pCurrentEntry->u32MqueueID != LINUX_C_U32_MQUEUE_ID))
         {
            CheckandCorrectMqStaticValues();
         }*/
         if(pCurrentEntry->PID == OSAL_ProcessWhoAmI()) // check for Client otherwise we block ourself
         {
            TraceString("OSALMQ It's not allowed to send Mssage from Server");
            OSAL_vProcessExit();
         }

         if( u32Prio < (OSAL_C_U32_MQUEUE_PRIORITY_LOWEST + 1) )
         {  
             /* check if queue isn't deleted */
            if(u32Length > pCurrentEntry->MaxLength)
            {
               u32ErrorCode = OSAL_E_MSGTOOLONG;
            }
            else
            {
            }

            if(u32ErrorCode == OSAL_E_NOERROR) 
            {
               msec = pOsalData->u32MqPostTmo;
               /* step 3 check for full message queue and wait , when queue is greater minimum size*/
               if((pCurrentEntry->ActualMessage == pCurrentEntry->MaxMessages)&&(pCurrentEntry->MaxMessages > 1))
               {
               }
            }
            if(u32ErrorCode == OSAL_E_NOERROR) 
            {
               if(pCurrentEntry->bTraceCannel == TRUE)
               {
                   TraceString("OSALMQ TID:%d Start Send Timeout:%d msec MQ:%s",OSAL_ThreadWhoAmI(),msec,pCurrentEntry->szName);
               }
/*               s32ReturnValue = MsgSendv(pCurrentEntry->s32LiHandle,
                                         const iov_t* siov,  //An array of buffers that contains the message that you want to send. 
                                         int sparts,           //The number of elements in the siov array. 
                                         const iov_t* //riov,n array of buffers where the reply can be stored
                                         int rparts );//The number of elements in the riov array*/
              char RetBuf[32];
#ifdef BLOCKING_TIMEOUT
              uint64_t timeout = BLOCKING_TIMEOUT*1000000;
              TimerTimeout(CLOCK_MONOTONIC,_NTO_TIMEOUT_SEND | _NTO_TIMEOUT_REPLY,NULL, &timeout,NULL);
#endif
              intptr_t Ret = MsgSend(prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId,(const void*)pcou8Msg,u32Length,(void*)&RetBuf[0],32);
               if(Ret != -1)
               {
				//   TraceString("Reply data %s",RetBuf);
                   if(u32ActualMessage == 0xffffffff)
                   {
                      u32ActualMessage = pCurrentEntry->ActualMessage++;
                   }
               }
               else
               {
                  s32ReturnValue = -1; 
                  u32ErrorCode = u32ConvertErrorCore(errno);
               }
            } //if(u32ErrorCode == OSAL_E_NOERROR)
         }
         else
         {
            u32ErrorCode = OSAL_E_INVALIDVALUE;
         }
      }
   }//if(u32ErrorCode == OSAL_E_NOERROR) 
#ifdef MQ_NOTIFY_VIA_OSAL
   if((u32ErrorCode == OSAL_E_NOERROR)&&(pCurrentEntry)&&(pCurrentEntry->bTriggerCb == TRUE))
   {
      if((pCurrentEntry->callbackpididx < (tS32)pOsalData->u32MaxNrProcElements)
      && (pCurrentEntry->pcallback )&&(pCurrentEntry->callbackpididx >= 0))
      {
         pCurrentEntry->bTriggerCb = FALSE;
         if( pCurrentEntry->callbackpid == OSAL_ProcessWhoAmI()) // check if process is the same
         {
            vExecuteCallback(pCurrentEntry);
         }
         else
         {
            vPostToCallbackQueue(pCurrentEntry);
         }
      }
#endif
#ifdef USE_EVENT_AS_CB
      if((pCurrentEntry->mask))
      {
         pCurrentEntry->bTriggerCb = FALSE;
         if(((tMQUserHandle*)hMQ)->hNotifyEvent == 0)
         {
            if(OSAL_s32EventOpen(pCurrentEntry->szEvName,&((tMQUserHandle*)hMQ)->hNotifyEvent) != OSAL_OK)
            {
               NORMAL_M_ASSERT_ALWAYS();
            }
         }
         OSAL_s32EventPost(((tMQUserHandle*)hMQ)->hNotifyEvent, pCurrentEntry->mask, pCurrentEntry->enFlags);
      }
#endif
   }

   if( u32ErrorCode != OSAL_E_NOERROR )
   {
      s32ReturnValue = OSAL_ERROR;
      if((u32ErrorCode != OSAL_E_TIMEOUT)||
        ((pCurrentEntry)&&(pCurrentEntry->bTraceCannel == TRUE))||(u32OsalSTrace & 0x00000020))
      {
          u32Temp = (tU32)OSAL_ThreadWhoAmI();
          bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,u32Temp);
          if(pCurrentEntry)
          {
             TraceString("OSAL_MQ Task:%s(ID:%d) Send      to MQ: User Handle %p MQ Name:%s Msg Count:%d Error:0x%x Prio:%d",
                      name,(unsigned int)u32Temp,((tMQUserHandle*)hMQ)->pOrigin, (((tMQUserHandle*)hMQ)->pOrigin)->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32Prio);
             if((pOsalData->bLogError)&&(u32ErrorCode != OSAL_E_TIMEOUT)&&(u32OsalSTrace & ~0x00000020))vWritePrintfErrmem("OSAL_MQ Task:%s(ID:%d) Send      to MQ: User Handle %p MQ Name:%s Msg Count:%d Error:0x%x Prio:%d \n",
                      name,(unsigned int)u32Temp,((tMQUserHandle*)hMQ)->pOrigin, (((tMQUserHandle*)hMQ)->pOrigin)->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32Prio);
          }
          else
          {
             TraceString("OSAL_MQ Task:%s(ID:%d) Send      to MQ: User Handle 0x%x MQ Name:%s Msg Count:%d Error:0x%x Prio:%d",
                      name,(unsigned int)u32Temp,(unsigned int)0, "Unknown",(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32Prio);
             if((pOsalData->bLogError)&&(u32ErrorCode != OSAL_E_TIMEOUT)&&(u32OsalSTrace & ~0x00000020))vWritePrintfErrmem("OSAL_MQ Task:%s(ID:%d) Send      to MQ: User Handle 0x%x MQ Name:%s Msg Count:%d Error:0x%x Prio:%d \n",
                      name,(unsigned int)u32Temp,(unsigned int)0, "Unknown",(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32Prio);
          }
      }
      vSetErrorCode(OSAL_C_THREAD_ID_SELF, u32ErrorCode);
   }
   else
   {
       if((pCurrentEntry->bTraceCannel == TRUE)||(u32OsalSTrace & 0x00000020))/*lint !e613 pCurrentEntry already checked */
       {
           u32Temp = (tU32)OSAL_ThreadWhoAmI();
           bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,u32Temp);
           TraceString("OSAL_MQ Task:%s(ID:%d) Send      to MQ: User Handle %p MQ Name:%s Msg Count:%d Error:0x%x Prio:%d",
                      name,(unsigned int)u32Temp,((tMQUserHandle*)hMQ)->pOrigin, (((tMQUserHandle*)hMQ)->pOrigin)->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32Prio);
       }
   }
   return( s32ReturnValue );
}

static char* pMsgMem = NULL;
static int Entry=0;
char* CreateList(int Count,int Lentgh)
{
   return pMsgMem = (char*)malloc(Count * Lentgh);
}

tBool RemoveList(void* pMem)
{
   free(pMem);
}

tS32 AddToList(char* Msg,int Lentgh)
{
   tS32 s32Ret=0;
  // memcpy()pMsgMem
   return s32Ret;
	
}

tU32 GetFromList(char* Msg,int Lentgh)
{
   tU32 u32Ret=0;
  // memcpy()pMsgMem
   return u32Ret;
}


tU32 u32WaitForMessage(trMqueueElement *pCurrentEntry,void* pBuffer,OSAL_tMSecond msec)
{
   tU32 u32CopiedBytes = 0;
   while(1)
   {
     struct _msg_info info;		
     if(msec != OSAL_C_TIMEOUT_FOREVER)
     {
        uint64_t timeout = msec*1000000;
        TimerTimeout(CLOCK_MONOTONIC,_NTO_TIMEOUT_RECEIVE,NULL, &timeout,NULL);
     }
     int ReplyId = MsgReceive(prLiMqMap[pCurrentEntry->u32EntryIdx].u32MqId,pBuffer,pCurrentEntry->MaxLength,&info);
     if(ReplyId == 0) //did we receive a pulse?
     {
        struct _pulse* pPulse = (struct _pulse*)pBuffer;
        switch(pPulse->code)
        {
            case _PULSE_CODE_DISCONNECT: //client disconnected
                 TraceString("OSALMQ Pulse _PULSE_CODE_DISCONNECT (%d) received with %d",pPulse->code,pPulse->value.sival_int);
                 if(ConnectDetach(pPulse->scoid) == -1)// ree scoid
                 {
                    TraceString("OSALMQ freeing scoid %d failed",pPulse->scoid,pPulse->value.sival_int);
                 }
                break;
            case _PULSE_CODE_UNBLOCK: //client wants to unblock
                 TraceString("OSALMQ Pulse _PULSE_CODE_UNBLOCK (%d) received with %d",pPulse->code,pPulse->value.sival_int);
                 MsgError(ReplyId,EINTR);
                break;
            default:
                 TraceString("OSALMQ Unexpected Pulse (%d) received with %d",pPulse->code,pPulse->value.sival_int);
                break;
        }
		// wait again
     }
     else if(ReplyId == -1)
     {
        if(ESRCH == errno)
        {
           TraceString("OSALMQ Client process terminated -> ESRCH");
        }
        else
		{
          // u32ErrorCode = u32ConvertErrorCore(errno);
           break;
        }
     }
     else
     {        
        char Reply[32] = "OK";	
        if(MsgReply(ReplyId,EOK,(const void*)Reply,32) == -1)
        {
           TraceString("MsgReply failed errno:%d",errno);
        }
        u32CopiedBytes = (tU32)info.msglen;
        break;	  
     }
   }
   return u32CopiedBytes;
}

void vMessageHandlerThread(void* pArg)
{
   trMqueueElement *pCurrentEntry = (trMqueueElement*)pArg;
   tU32 u32CopiedBytes = 0;
   char* pBuffer = (char*)malloc(pCurrentEntry->MaxLength+4);
   while(1)
   {
      u32CopiedBytes = u32WaitForMessage(pCurrentEntry,pBuffer,OSAL_C_TIMEOUT_FOREVER);
	  if(u32CopiedBytes > 0)
      if(AddToList(pBuffer,u32CopiedBytes) == OSAL_ERROR)
      {
		vReactOnMqOverflow(pCurrentEntry);  
      }
   }
   free(pBuffer);
}

tS32 s32StartMsgHndler(trMqueueElement *pCurrentEntry)
{
   OSAL_trThreadAttribute threadAttr;
   tS8 ps8ThreadName[16];
   memset(&threadAttr,0,sizeof(OSAL_trThreadAttribute)); 
   snprintf((tString) ps8ThreadName,16,"%s%d", "MQ_HDR_",getpid());
   threadAttr.u32Priority = 0;
   threadAttr.szName = (tString) ps8ThreadName;
   threadAttr.pfEntry = (OSAL_tpfThreadEntry) vMessageHandlerThread;
   threadAttr.pvArg = pCurrentEntry;
   threadAttr.s32StackSize = 0x10000;
   return (tS32)OSAL_ThreadSpawn(&threadAttr);
}


/*****************************************************************************
 *
 * FUNCTION:    QNX_MessageQueueWait
 *
 * DESCRIPTION: Query a message from  a MessageQueue. The obtained Message is
 *              the one with the higher priority existing in the MessageQueue
 *              and the actual length of the copied message is defined by the
 *              function param and not by the message length.
 *
 * PARAMETER:   hMQ       MessageQueue handle
 *              pu8Buffer pointer to the output buffer for the message copy
 *              u32Length actual length to e copied in bytes
 *              pu32Prio  pointer to the actual priority of the copied message
 *              msec      timeout
 *
 * ERRORS:      OSAL_E_BADFILEDESCRIPTOR  hMQ is not a valid handle
 *              OSAL_E_MSGTOOLONG         length exceeds the field MaxLength
 *                                        defined in creation
 *              OSAL_E_TIMEOUT            the timeout is reached without
 *                                        any message in the MQueue
 *
 * RETURNVALUE: s32ReturnValue
 *                 it is the function return value:
 *                 - OSAL_OK if everything goes right;
 *                 - OSAL_ERROR otherwise.
 *
 * HISTORY:
 * Date      |   Modification                         | Authors
 * 03.10.05  | Initial revision                       | MRK2HI
 * --.--.--  | ----------------                       | -----
 * 13.12.12  | Fix for OSAL MQ priority issue         |sja3kor 
 * 17.12.12  | Fix for MMS CFAGII-501.Boundary check  |sja3kor
 *           |  condition changed.                    | 
 * --.--.--  | ----------------                       | -------, -----
 * 16.01.13  |Fix for Wrong timeout for               |sja3kor 
 *           |Linux messagequeue post/wait CFAGII-506 | 
 * 21.01.13  | Integrating GMNGA-48007 Fix            | SWM2KOR   
 *****************************************************************************/
 int QNX_MessageQueueWait( OSAL_tMQueueHandle hMQ,
                               tPU8 pu8Buffer,
                               tU32 u32Length,
                               tPU32 pu32Prio,
                               OSAL_tMSecond msec)
{
   tU32 u32CopiedBytes = 0;
   tU32 u32ErrorCode   = OSAL_E_NOERROR;//u32CheckMqHandle(hMQ); /* done in OSAL_s32MessageQueueWait */
   tU32 u32ActualPrio = 0xffffffff;
   trMqueueElement *pCurrentEntry = NULL;
   tU32 u32ActualMessage = 0xffffffff;
   tU32 u32Temp = 0;
   char name[TRACE_NAME_SIZE];
   ((void)pu32Prio);
   ((void)u32Length);
   if(u32ErrorCode == OSAL_E_NOERROR) 
   {
      /* check permissions*/
      if(((tMQUserHandle*)hMQ)->enAccess == OSAL_EN_WRITEONLY) 
      {
          u32ErrorCode = OSAL_E_NOPERMISSION;
      }
   }
   if(u32ErrorCode == OSAL_E_NOERROR)
   {
        pCurrentEntry = ((tMQUserHandle*)hMQ)->pOrigin;/*lint !e613 *//* pointer already checked*/ 
        if(pCurrentEntry->PID != OSAL_ProcessWhoAmI()) // check for server otherwise we block ourself
        {
           TraceString("OSALMQ It's not allowed to wait for Mssage at client");
           OSAL_vProcessExit();
        }
        if(pCurrentEntry->bToDelete == TRUE)
        {
           msec = OSAL_C_TIMEOUT_NOBLOCKING;
        }
        pCurrentEntry->u16WaitCounter++;

        /* Stop message processing time measurement */
        vStopMeasureProcessingTime(pCurrentEntry);
		
        if(pCurrentEntry ->RecTsk == 0)
        {
       /* additional thread here needed to decouple messaging from the receiver thread of the server 
           to ensure unblocking of the sender if server thread is busy in message processing for longer time*/
           pCurrentEntry->RecTsk = OSAL_ThreadWhoAmI();
           pCurrentEntry->RecPrc = OSAL_ProcessWhoAmI();
#ifdef HANDLER_TASK
           if(s32StartMsgHndler(pCurrentEntry) == OSAL_ERROR)
           {
              FATAL_M_ASSERT_ALWAYS();
           }
#endif
        }
#ifdef HANDLER_TASK
        u32CopiedBytes = GetFromList(pu8Buffer,u32Length);
#else
        u32CopiedBytes = u32WaitForMessage(pCurrentEntry,pu8Buffer,msec);
#endif		
        if(u32CopiedBytes)
        {
           u32ActualMessage = pCurrentEntry->ActualMessage--;
           /* Start measurement of message processing times */
           vStartMeasureProcessingTime(pCurrentEntry);           
        } //if(u32CopiedBytes)
        else
        {
           u32ErrorCode = u32ConvertErrorCore(errno);
        }
#ifdef MQ_NOTIFY_VIA_OSAL
        if(u32ErrorCode == OSAL_E_TIMEOUT)
        {
           pCurrentEntry->bTriggerCb = TRUE;/*lint !e613 when u32ErrorCode == OSAL_E_TIMEOUT valid pCurrentEntry*/
        }
#endif
  }//if(u32ErrorCode == OSAL_E_NOERROR)



  if( u32ErrorCode != OSAL_E_NOERROR )
  {
     if((u32ErrorCode != OSAL_E_TIMEOUT)||
        ((pCurrentEntry)&&(pCurrentEntry->bTraceCannel == TRUE))||(u32OsalSTrace & 0x00000020))
     {
         u32Temp = (tU32)OSAL_ThreadWhoAmI();
         bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,OSAL_ThreadWhoAmI());
         if(pCurrentEntry)
         {
            TraceString("OSAL_MQ Task:%s(ID:%d) Receive from MQ: User Handle %p Name:%s Msg Count:%d Error:0x%x Prio:%d",
                      name,(unsigned int)u32Temp,((tMQUserHandle*)hMQ)->pOrigin,(((tMQUserHandle*)hMQ)->pOrigin)->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32ActualPrio);
            if((pOsalData->bLogError)&&(u32ErrorCode != OSAL_E_TIMEOUT)&&(u32OsalSTrace & ~0x00000020))vWritePrintfErrmem("OSAL_MQ Task:%s(ID:%d) Receive from MQ: User Handle %p Name:%s Msg Count:%d Error:0x%x Prio:%d \n",
                      name,(unsigned int)u32Temp,((tMQUserHandle*)hMQ)->pOrigin,(((tMQUserHandle*)hMQ)->pOrigin)->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32ActualPrio);
         }
         else
         {
             TraceString("OSAL_MQ Task:%s(ID:%d) Receive from MQ: User Handle 0x%x Name:%s Msg Count:%d Error:0x%x Prio:%d",
                      name,(unsigned int)u32Temp,(unsigned int)u32ErrorCode,"Unknown",(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32ActualPrio);
             if((pOsalData->bLogError)&&(u32ErrorCode != OSAL_E_TIMEOUT)&&(u32OsalSTrace & ~0x00000020))vWritePrintfErrmem("OSAL_MQ Task:%s(ID:%d) Receive from MQ: User Handle 0x%x Name:%s Msg Count:%d Error:0x%x Prio:%d \n",
                      name,(unsigned int)u32Temp,(unsigned int)u32ErrorCode,"Unknown",(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32ActualPrio);
         }
     }
     vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
  }
  else
  {
     if(((pCurrentEntry) &&(pCurrentEntry->bTraceCannel == TRUE))||(u32OsalSTrace & 0x00000020))
     {
        u32Temp = (tU32)OSAL_ThreadWhoAmI();
        bGetThreadNameForTID(&name[0],TRACE_NAME_SIZE,OSAL_ThreadWhoAmI());
        TraceString("OSAL_MQ Task:%s(ID:%d) Receive from MQ: User Handle %p Name:%s Msg Count:%d Error:0x%x Prio:%d",
                 name,(unsigned int)u32Temp,((tMQUserHandle*)hMQ)->pOrigin,(((tMQUserHandle*)hMQ)->pOrigin)->szName,(unsigned int)u32ActualMessage,(unsigned int)u32ErrorCode,(unsigned int)u32ActualPrio);
     }
  }
  return((tS32)u32CopiedBytes );
}


   
int QNX_s32MessageQueueNotify(OSAL_tMQueueHandle hMQ, OSAL_tpfCallback pCallback, void* pvArg)
{
   tS32 s32RetVal = OSAL_OK;
   tU32 u32ErrorCode   = u32CheckMqHandle(hMQ);
   tS32 s32TID;
   trMqueueElement *pCurrentEntry = NULL;

   if(u32ErrorCode == OSAL_E_NOERROR)
   {
      /* check if callback is used , then we need callback handler task */
      if(pCallback &&((uintptr_t)pCallback != 0xffffffff))
      {
         /* check if callback handler task for this process already exists */  
         tS32 s32PidEntry = s32FindProcEntry(OSAL_ProcessWhoAmI());
         if(s32PidEntry == OSAL_ERROR)
         {
            NORMAL_M_ASSERT_ALWAYS();
         }
         else
         {
            /* check if callback handler task for this process already exists */  
            s32StartCbHdrTask(s32PidEntry);
         }
      }
      pCurrentEntry = ((tMQUserHandle*)hMQ)->pOrigin;/*lint !e613 *//* pointer already checked*/ 
/*         if((pCurrentEntry->u32EntryIdx >= pOsalData->MqueueTable.u32UsedEntries)||(pCurrentEntry->u32MqueueID != LINUX_C_U32_MQUEUE_ID))
         {
            CheckandCorrectMqStaticValues();
         }*/

         if(LockOsal(&pOsalData->MqueueTable.rLock ) == OSAL_OK)
         {
            /* check if queue isn't deleted */
            if(pCurrentEntry->bIsUsed)
            {
               s32TID = (tS32)OSAL_ThreadWhoAmI();
#ifdef USE_EVENT_AS_CB
               if((uintptr_t)pCallback == 0xffffffff)
               {
                  if(pvArg != NULL)
                  {
                     trMqEventInf* pEvent = (trMqEventInf*)pvArg;
                     if(strlen(pEvent->coszName) > OSAL_C_U32_MAX_NAMELENGTH)
                     {
                        u32ErrorCode = OSAL_E_NAMETOOLONG;
                        TraceString("MQ Notification event name %s is to long",pEvent->coszName);
                     }
                     else
                     {
                        (void)OSAL_szStringNCopy((tString)pCurrentEntry->szEvName,pEvent->coszName, strlen(pEvent->coszName));
                         pCurrentEntry->mask    = pEvent->mask;
                         pCurrentEntry->enFlags = pEvent->enFlags;
                         pCurrentEntry->bNotify = TRUE;
                         /* prepare message processing mesurements 
                           filter out notify queues because their usage is trgggered by OSAL callbacks */
                        pCurrentEntry->u32MessagePrcessing = MEASUREMENT_SWITCHOFF;
                     } 
                  }
                  else
                  {
                     pCurrentEntry->mask = (OSAL_tEventMask) 0;
                     pCurrentEntry->enFlags = (OSAL_tenEventMaskFlag) 0;
                  }
               }
               else//if((pCallback == NULL )&&(pvArg != NULL))
#endif
               {
                  if( pCallback == NULL )
                  {
                     if( pvArg == NULL )
                     {
                        if( pCurrentEntry->pcallback != NULL )
                        {
                           if(pCurrentEntry->callbacktid == s32TID )
                           {
                              pCurrentEntry->bNotify = FALSE;
                                 /*force dispatch to disable callback */
                                 OSAL_s32ThreadWait(10);
                                 pCurrentEntry->pcallback = NULL;
                                 pCurrentEntry->callbacktid  = NOTATHREAD;
                                 s32RetVal = OSAL_OK;
                           }
                           else
                           {
                              s32RetVal=OSAL_ERROR;
                              u32ErrorCode=OSAL_E_BUSY;
                           }
                        }
                        else
                        {
                           s32RetVal=OSAL_ERROR;
                           pCurrentEntry->callbacktid  = NOTATHREAD;
                        }
                     }
                     else
                     {
                        s32RetVal=OSAL_ERROR;
                        u32ErrorCode=OSAL_E_INVALIDVALUE;
                     }
                  }
                  else
                  {
                     if( pCurrentEntry->pcallback == NULL )
                     {
                        pCurrentEntry->callbackpid  = OSAL_ProcessWhoAmI();
                        pCurrentEntry->callbacktid  = s32TID;
                        pCurrentEntry->pcallback    = pCallback;
                        pCurrentEntry->pcallbackarg = pvArg;
                        pCurrentEntry->callbackpididx = s32GetPrcIdx(pCurrentEntry->callbackpid);
                        if(pCurrentEntry->callbackpididx == -1)
                        {
                          FATAL_M_ASSERT_ALWAYS();
                        }
                        pCurrentEntry->bNotify = TRUE;
                     }
                     else
                     {
                        s32RetVal = OSAL_ERROR;
                        u32ErrorCode = OSAL_E_BUSY;
                     }
                  }//if( pCallback == NULL )
               }//if((pCallback == NULL )&&(pvArg != NULL))
            }//if(pCurrentEntry->bIsUsed)
            UnLockOsal(&pOsalData->MqueueTable.rLock);
         }
   }

   if( u32ErrorCode != OSAL_E_NOERROR )
   {
      s32RetVal = OSAL_ERROR;
      vTraceMqNotify(pCurrentEntry,u32ErrorCode);
      vSetErrorCode( OSAL_C_THREAD_ID_SELF, u32ErrorCode );
   }
   else
   {
      if((pCurrentEntry->bTraceCannel)||(u32OsalSTrace & 0x00000020)) /*lint !e613 pointer already checked */
      {
        vTraceMqNotify(pCurrentEntry,u32ErrorCode);
      }
   }
   return(s32RetVal);
}

int QNX_s32MessageQueueStatus(OSAL_tMQueueHandle hMQ, tPU32 pu32MaxMessages, tPU32 pu32MaxLength, tPU32 pu32Messages)
{
   tU32 u32ErrorCode   = OSAL_E_NOERROR;//u32CheckMqHandle(hMQ); /* done in OSAL_s32MessageQueueWait */
   trMqueueElement *pCurrentEntry = NULL;
   tS32 s32ReturnValue = OSAL_ERROR;

   if(u32ErrorCode == OSAL_E_NOERROR) 
   {
      pCurrentEntry = ((tMQUserHandle*)hMQ)->pOrigin;/*lint !e613 *//* pointer already checked*/ 
      if(pCurrentEntry)
      {
         if( pu32MaxMessages != OSAL_NULL )
                *pu32MaxMessages = pCurrentEntry->MaxMessages;
         if( pu32MaxLength  != OSAL_NULL )
                *pu32MaxLength = pCurrentEntry->MaxLength;
         if( pu32Messages    != OSAL_NULL )
                *pu32Messages  = pCurrentEntry->ActualMessage;
         s32ReturnValue = OSAL_OK;
      }
      else
      {
         u32ErrorCode = OSAL_E_DOESNOTEXIST; 
      }
   }
   if( u32ErrorCode != OSAL_E_NOERROR )
   {
      if(pCurrentEntry)
      {
           vTraceMqInfo((tMQUserHandle*)hMQ,OSAL_MQ_STATUS,TR_LEVEL_FATAL,u32ErrorCode,NULL);
      }
      else
      {
           vTraceMqInfo(NULL,OSAL_MQ_STATUS,TR_LEVEL_FATAL,u32ErrorCode,NULL);
      }
      vSetErrorCode(OSAL_C_THREAD_ID_SELF,u32ErrorCode);
   }
   else
   {
      if((pCurrentEntry->bTraceCannel)||(u32OsalSTrace & 0x00000020))/*lint !e613 pointer already checked */
      {
         vTraceMqInfo((tMQUserHandle*)hMQ,OSAL_MQ_STATUS,TR_LEVEL_FATAL,u32ErrorCode,NULL);
      }
   }
   return( s32ReturnValue );
}

#endif

#ifdef __cplusplus
}
#endif

 
/************************************************************************
|end of file
|-----------------------------------------------------------------------*/
