/*****************************************************************************
| FILE:         osalansi.c
| PROJECT:      platform
| SW-COMPONENT: OSAL CORE
|-----------------------------------------------------------------------------
| DESCRIPTION:  This is the implementation file for the OSAL 
|               (Operating System Abstraction Layer) Event-Functions.
|                
|-----------------------------------------------------------------------------
| COPYRIGHT:    (c) 2005 Bosch GmbH
| HISTORY:      
| Date      | Modification               | Author
| 03.10.05  | Initial revision           | MRK2HI
| --.--.--  | ----------------           | -------, -----
|
|*****************************************************************************/

/************************************************************************ 
| includes of component-internal interfaces
| (scope: component-local)
|-----------------------------------------------------------------------*/
#include "OsalConf.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "Linux_osal.h"

#include "ostrace.h"
#include "exception_handler.h"

#ifdef __cplusplus
extern "C" {
#endif

/************************************************************************ 
|defines and macros (scope: module-local) 
|-----------------------------------------------------------------------*/

#define MIN(x,y) ((x)<(y)?(x):(y))

// #define ASSERT_BUFFER_SIZE    MAX_TRACE_SIZE // errmmem buffer is smaller!

#define OUTPUT_BUFFER_SIZE    236
// ERRMEM_MAX_ENTRY_LENGTH (=128) - header for trace
#define ASSERT_BUFFER_SIZE    200

// Line 1
#define ASSERT_BUFFER_MODE_LINE1_SIZE  (0x03)
#define ASSERT_BUFFER_EXPR_LINE1_SIZE  (ASSERT_BUFFER_SIZE - ASSERT_BUFFER_MODE_LINE1_SIZE - 1)

// Line 2
#define ASSERT_PROC_NAME_SIZE   16
#define ASSERT_THREAD_NAME_SIZE 16
#define ASSERT_LINE_NUMBER_SIZE  4
#define ASSERT_BUFFER_EXPR_LINE2_SIZE (ASSERT_BUFFER_SIZE - 1 - ASSERT_PROC_NAME_SIZE - \
                                       ASSERT_THREAD_NAME_SIZE - ASSERT_LINE_NUMBER_SIZE - 1)


#define ASSERT_INITIATOR_OSAL       0x00
// #define ASSERT_INITIATOR_STL        0x01



/************************************************************************ 
|typedefs (scope: module-local) 
|-----------------------------------------------------------------------*/
typedef struct{
   char Output[OUTPUT_BUFFER_SIZE];
}trTraceEntry;

/************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------*/

/************************************************************************ 
| variable definition (scope: global)
|-----------------------------------------------------------------------*/
static char OSAL_vAssert_buf[ OUTPUT_BUFFER_SIZE ];
static trTraceEntry rTraceEntry[MAX_DEEP+1];
static char OSAL_vAssert_buf_Ref[ OUTPUT_BUFFER_SIZE ];


#ifdef LIBUNWIND_USED
extern void vCheckLibUnwind(void);
extern tU32 u32Unwind(void** addr,tU32 u32Size);
extern int s32UnwindInstalled;
#endif

extern tBool bConstructFlag;

/************************************************************************
|function prototype (scope: module-local)
|-----------------------------------------------------------------------*/

void vWriteUtcTime(int fd)
{
   struct tm mytm;
   int i = 0;
   int old_fd = fd;
   char Buffer[100] = {0};
   time_t secSinceEpoch = time(NULL);

   if(fd == -1)
   {
      fd = open("/dev/errmem", O_WRONLY);
   }

   if(fd != -1)
   {
      Buffer[0] = 0;
      if( NULL != gmtime_r(&secSinceEpoch, &mytm))
      {
         i = snprintf(&Buffer[1],99,"OSAL UTC %d.%d.%d  %d:%d:%d \n",mytm.tm_mday,mytm.tm_mon + 1,mytm.tm_year,mytm.tm_hour,mytm.tm_min,mytm.tm_sec);
      }
      else
      {
         i = snprintf(&Buffer[1],99,"OSALElapsed Time %d \n",(int)secSinceEpoch);
      }
      if(write(fd, Buffer, i) == -1){}
      if(old_fd == -1)
      {
         close(fd);
      }
   }
}




/******************************************************FunctionHeaderBegin******
 * FUNCTION    : bGetAssertBehaviour
 * CREATED     : 2009-03-11
 * AUTHOR      : TMS-Rrd
 * DESCRIPTION : get the behaviour to follow after output of assert message
 * SYNTAX      : tVoid teAssertMode bGetAssertBehaviour(tU32 line)
 * ARGUMENTS   : line no
 *                       
 * RETURN VALUE: assert behaviour for current assert 
 *
 * NOTES       :  
 *******************************************************FunctionHeaderEnd******/
static teAssertMode bGetAssertBehaviour(tU32 line)
{
teAssertMode eRes;

    /* the behaviour is controlled by assert mode and SW type (release or debug)

       Release: FATAL_M_ASSERT() is controlled by assert mode; NORMAL_M_ASSERT() triggers no action

       Debug: FATAL_M_ASSERT() and NORMAL_M_ASSERT() have the same behaviour. It is controlled by assert mode.

       By default assert mode is set to ASSERTMODE_RESET.
    */

/* if release SW */
#ifndef __DEBUG_BUILD__
    /* only if it is a FATAL_M_ASSERT() */
    if  (0 == (line & 0x80000000))
#else
    (void)line;
#endif
    {
        eRes = (teAssertMode)pOsalData->u32AssertMode;
    }
#ifndef __DEBUG_BUILD__
    else
    {
        eRes = ASSERTMODE_TASKNOACTION;
    }
#endif

	return (eRes);
}


/******************************************************FunctionHeaderBegin******
 * FUNCTION    : OSAL_generate_callstack 
 * CREATED     : 2006-11-23
 * AUTHOR      : Ulrich Schulz / TMS
 * DESCRIPTION : function traces out the callstack information to TTFis
 * SYNTAX      : static void exc_manager_trace_cs(void)
 * ARGUMENTS   : -
 * RETURN VALUE: -
 *
 * NOTES       : -
 *
 *******************************************************FunctionHeaderEnd******/
#ifdef ANDROID
#include <unwind.h>
//using namespace std;
//#include <typeinfo>
//#include <cxxabi.h>
extern tpOsalBinderBacktrace pOsalBinderBacktrace;


struct BacktraceState
{
    void** current;
    void** end;
};

#ifdef LOAD_LIBUNWIND
typedef int (*pFunc_Unwind_Backtrace)(_Unwind_Trace_Fn Func, void* State);
typedef unsigned int (*pFunc_Unwind_GetIP)(struct _Unwind_Context* context); /*lint !e1773 */

pFunc_Unwind_Backtrace fp_Unwind_Backtrace = NULL;
pFunc_Unwind_GetIP fp_Unwind_GetIP = NULL;

void* pUnwindHandle = NULL;

tBool LoadLibUnwind(void)
{
   OSAL_tProcessID Pid =  OSAL_ProcessWhoAmI();
   dlerror();    /* Clear any existing error */

   pUnwindHandle = dlopen(LIB_UNWIND, RTLD_NOW|RTLD_GLOBAL );
   if(pUnwindHandle == NULL)
   {
       TraceString("Load libunwind.so PID:%d failed %s !!!!",Pid,dlerror());
       return FALSE;
   }
   else
   {
      fp_Unwind_Backtrace  = (pFunc_Unwind_Backtrace)(dlsym(pUnwindHandle, (const char*)"_Unwind_Backtrace"));/*lint !e611 */;
      fp_Unwind_GetIP      = (pFunc_Unwind_GetIP)(dlsym(pUnwindHandle, (const char*)"_Unwind_GetIP"));/*lint !e611 */
         
      if((!fp_Unwind_Backtrace)||(!fp_Unwind_GetIP))
      {
         TraceString("Load libunwind.so PID:%d lookup functions failed !!!!",Pid);
         dlclose(pUnwindHandle);
         return FALSE;
      }
      return TRUE;
   }
}
#endif                                                                      

namespace {

static _Unwind_Reason_Code unwindCallback(struct _Unwind_Context* context, void* arg)
{
    BacktraceState* state = static_cast<BacktraceState*>(arg);
#ifdef LOAD_LIBUNWIND
    uintptr_t pc = fp_Unwind_GetIP(context);
#else
    uintptr_t pc = _Unwind_GetIP(context);
#endif
    if (pc) 
    {
        if (state->current == state->end) 
        {
            return _URC_END_OF_STACK;
        } 
        else 
        {
            *state->current++ = reinterpret_cast<void*>(pc);
        }
    }
    return _URC_NO_REASON;
}

}

size_t captureBacktrace(void** buffer, size_t max)
{
    BacktraceState state = {buffer, buffer + max};
#ifdef LOAD_LIBUNWIND
    fp_Unwind_Backtrace(unwindCallback, (void*)&state);
#else
    _Unwind_Backtrace(unwindCallback, (void*)&state);
#endif

    return state.current - buffer;
}

void dumpBacktrace(void** buffer, size_t count)
{
    for (size_t idx = 4; idx < count; ++idx) 
    {
        const void* addr = buffer[idx];
        const char* symbol = "";
   //     const char* realsymbol = "";
   //     int     status;

        Dl_info info;
        if (dladdr(addr, &info) && info.dli_sname) 
        {
            symbol = info.dli_sname;
 //           realsymbol = abi::__cxa_demangle(symbol, 0, 0, &status);

        }
  //      snprintf(&rTraceEntry[i].Output[0],OUTPUT_BUFFER_SIZE,"callstack= [%s]",OSAL_vAssert_buf+1);
        vWritePrintfErrmem("OSAL callstack= [%s] : 0x%lx \n", symbol,addr);
        TraceString("OSAL callstack= [%s] : 0x%lx", symbol,addr);
    }
}
void AndroidBacktrace(void)
{
    const size_t max = 30;
    void* buffer[max];
 
    if(pOsalBinderBacktrace)
    {
       TraceString("libosalbinder_so.so backtrace");
       pOsalBinderBacktrace();
    }
    else
    {
       TraceString("libosallinux_so.so backtrace");
#ifdef LOAD_LIBUNWIND
       if(pUnwindHandle == NULL)
       {
           LoadLibUnwind();
       }
       if(fp_Unwind_Backtrace && fp_Unwind_GetIP)
       {
#endif		   
          dumpBacktrace(buffer, captureBacktrace(buffer, max));
#ifdef LOAD_LIBUNWIND
       }
       else
       {
         TraceString("No function pointer from libunwind available");
       }
#endif		   
    }
}
#endif //ANDROID

#ifdef QNX
static char MemMapData[4096];
static bt_addr_t addrs[MAX_DEEP];
static bt_addr_t reladdrs[MAX_DEEP];
static bt_addr_t offsets[MAX_DEEP];
static int MapIdx[MAX_DEEP];

static char NameBuffer1[256];
static char NameBuffer2[256];
static char NameBuffer3[256];
static char NameBuffer4[256];
static char NameBuffer5[256];
static char NameBuffer6[256];
static char NameBuffer7[256];
static char NameBuffer8[256];
static char NameBuffer9[256];
static char NameBuffer10[256];
static char NameBuffer11[256];
static char NameBuffer12[256];
static char NameBuffer13[256];
static char NameBuffer14[256];
static char NameBuffer15[256];
static char NameBuffer16[256];
static char NameBuffer17[256];
static char NameBuffer18[256];
static char NameBuffer19[256];
static char NameBuffer20[256];

static  char* cOutPut[MAX_DEEP] = {NameBuffer1,NameBuffer2,NameBuffer3,NameBuffer4,NameBuffer5,
                                   NameBuffer6,NameBuffer7,NameBuffer8,NameBuffer9,NameBuffer10,
	                           NameBuffer11,NameBuffer12,NameBuffer13,NameBuffer14,NameBuffer15,
                                   NameBuffer16,NameBuffer17,NameBuffer18,NameBuffer19,NameBuffer20};


void QnxBacktrace(void)
{
#ifdef LIBBACKTRACE_USED	
   int CallStackDeep = 0;
   int i;
   bt_accessor_t acc;
   strcpy(&rTraceEntry[0].Output[0],"OSAL_BT OSAL ASSERT using QNX backtrace \0");
   if (bt_init_accessor(&acc, BT_SELF) == -1)
   {
      vWritePrintfErrmem("OSAL_BT %s:%i %s (%i)%s\n", __FUNCTION__, __LINE__,"bt_init_accessor", errno, strerror(errno));
      TraceString("OSAL_BT %s:%i %s (%i)%s\n", __FUNCTION__, __LINE__,"bt_init_accessor", errno, strerror(errno));
   }
   else
   {
      bt_memmap_t memmap;
      char* pTemp;
      if(bt_load_memmap(&acc,&memmap ) == 0)
      {
         if(bt_sprn_memmap(&memmap, MemMapData, sizeof(MemMapData)) == -1)
         {
            fprintf( stderr, "%s:%i %s (%i)%s\n", __FUNCTION__, __LINE__,"bt_sprn_memmap", errno, strerror(errno));
         }
         else
         {
            MemMapData[sizeof(MemMapData) - 1] = '\0';
            puts (MemMapData);
            CallStackDeep = bt_get_backtrace(&acc,&addrs[0],MAX_DEEP);
//            TraceString("OSAL_BT bt_get_backtrace finished %d entries for Variant 1",CallStackDeep);			
            if(CallStackDeep > 0)
            {
               for( i = 0; i < CallStackDeep; i++ ) // 1: we don't want to see the function call of OSAL_trace_callstack()
               {
                  bt_translate_addrs(&memmap,&addrs[i],CallStackDeep-i,&reladdrs[i],&offsets[i],&MapIdx[i],&cOutPut[i]);	
                  TraceString("OSAL_BT %d MemAdress[%p] RelAddress:%p ProcMapOffset:%p -> MapIdx:%d %s", i+1,addrs[i],reladdrs[i],offsets[i],MapIdx[i],cOutPut[i]);
                  vWritePrintfErrmem("OSAL_BT %d MemAdress[%p] RelAddress:%p ProcMapOffset:%p -> MapIdx:%d %s", i+1,addrs[i],reladdrs[i],offsets[i],MapIdx[i],cOutPut[i]);
               }
            }				
            else
            {
               TraceString("OSAL_BT No Callstack available %d %s -> check for _BT_LIGHT", errno, strerror(errno));			
               vWritePrintfErrmem("OSAL_BT No Callstack available %d %s", errno, strerror(errno));			
            }

#ifdef CONSOLE_OUTPUT
            /* variant with automatic output to console */
            CallStackDeep = bt_get_backtrace( &acc, addrs, sizeof(addrs) / sizeof(bt_addr_t));
//            TraceString("OSAL_BT bt_get_backtrace finished %d entries for Variant 2",CallStackDeep);			
            if(CallStackDeep <= 0)
            {
                 TraceString("OSAL_BT No Callstack available %d %s", errno, strerror(errno));			
                 vWritePrintfErrmem("OSAL_BT No Callstack available %d %s", errno, strerror(errno));			
            }    
            else
            {		
               int w;		
               for( i = 0; i < CallStackDeep; i++ ) // 1: we don't want to see the function call of OSAL_trace_callstack()
               { 
                   w = bt_sprnf_addrs(&memmap, addrs + i, CallStackDeep-i, "OSAL_BT MemAddress:%a RelAddress:%l ProcMapOffset:%o -> MapIdx:%I %f", OSAL_vAssert_buf, sizeof(OSAL_vAssert_buf), 0);
                   if (w == -1)
                   {
                      TraceString(  "%s:%i %s (%i)%s\n", __FUNCTION__, __LINE__,"bt_sprnf_addrs", errno, strerror(errno));
                      break;
                   }
                   TraceString(OSAL_vAssert_buf);
                   vWritePrintfErrmem(OSAL_vAssert_buf);
                }
            }
#endif
            bt_unload_memmap(&memmap); 
          }
       }
       bt_release_accessor(&acc);
   }
#endif
}
#endif

int OSAL_generate_callstack(void)
{
   int  i=0;
   if(!bConstructFlag)
   {
#ifdef ANDROID
      AndroidBacktrace();
#else // ANDROID
#ifdef QNX
      QnxBacktrace();
#else // QNX
      int CallStackDeep = 0;
      void *buffer[MAX_DEEP];
      char **strings;
#ifdef LIBUNWIND_USED
      if(s32UnwindInstalled == 0)
      {
         vCheckLibUnwind();
      }
      if((pOsalData->u32UseLibunwind)&&(s32UnwindInstalled == 2))
      {
         strcpy(&rTraceEntry[0].Output[0],"OSAL ASSERT using libunwind backtrace \0");
         CallStackDeep = u32Unwind(buffer, MAX_DEEP);
      }
      else
#endif
      {
         strcpy(&rTraceEntry[0].Output[0],"OSAL ASSERT using glibc backtrace \0");
         CallStackDeep = backtrace(buffer, MAX_DEEP);
      }
      vWritePrintfErrmem("%s %s",&rTraceEntry[0].Output[0]," \n");
   
      strings  = backtrace_symbols( buffer, CallStackDeep );
      if( strings == NULL )
      {
         OSAL_vAssert_buf[0] = 0x07;
         sprintf((char*)(OSAL_vAssert_buf+1), "backtrace error");
         strncpy(&rTraceEntry[1].Output[0],OSAL_vAssert_buf, (tU32)strlen(OSAL_vAssert_buf));
         vWriteToErrMem( TR_CLASS_ASSERT, OSAL_vAssert_buf, (tU32)strlen(OSAL_vAssert_buf), 0 );
      }
      else
      {
         for( i = 1; i < CallStackDeep; i++ ) // 1: we don't want to see the function call of OSAL_trace_callstack()
         {
            memset( OSAL_vAssert_buf, 0, OUTPUT_BUFFER_SIZE );
            OSAL_vAssert_buf[0] = 0x07;
            strncpy( OSAL_vAssert_buf+1, strings[i], OUTPUT_BUFFER_SIZE-2 );
            OSAL_vAssert_buf[ OUTPUT_BUFFER_SIZE-1 ] = 0;
#ifdef SHORT_TRACE_OUTPUT 
            strncpy(&rTraceEntry[i].Output[0],OSAL_vAssert_buf, (tU32)strlen(OSAL_vAssert_buf));
            vWriteToErrMem( TR_CLASS_ASSERT, OSAL_vAssert_buf, (tU32)strlen(OSAL_vAssert_buf), 0 );
#else
            snprintf(&rTraceEntry[i].Output[0],OUTPUT_BUFFER_SIZE,"callstack= [%s]",OSAL_vAssert_buf+1);
            vWritePrintfErrmem("callstack= [%s] \n",OSAL_vAssert_buf+1);
#endif
            if(strlen(strings[i]) > OUTPUT_BUFFER_SIZE)
            {
               memset( OSAL_vAssert_buf, 0, OUTPUT_BUFFER_SIZE );
               OSAL_vAssert_buf[0] = 0x07;
               strncpy( OSAL_vAssert_buf+4, strings[i]+OUTPUT_BUFFER_SIZE, OUTPUT_BUFFER_SIZE-5 );
               OSAL_vAssert_buf[ OUTPUT_BUFFER_SIZE-1 ] = 0;
#ifdef SHORT_TRACE_OUTPUT 
               vWriteToErrMem( TR_CLASS_ASSERT, OSAL_vAssert_buf, (tU32)strlen(OSAL_vAssert_buf), 0 );
#else
               vWritePrintfErrmem("callstack= [%s] \n",OSAL_vAssert_buf+1);
#endif 
            }
         }
      }
      free(strings);
#endif
#endif // QNX  
   }
   else
   {
      strcpy(&rTraceEntry[0].Output[0],"OSAL ASSERT from shared library constructor \0");
   }
   return i+1;
}

void OSAL_generate_callstack_secure(void)
{
#if defined __i386__ || defined __x86_64__ || defined __arm__ || defined __arm_64__ || defined __aarch64__
  int lock_return = exception_handler_lock();
  if (lock_return != 0)
  {
     vWritePrintfErrmem("OSAL_generate_callstack_secure -> exception_handler_lock failed for PID:%d \n",getpid());
  }
#else
#error "unknown compiler environment - don't know wheather to use the exception handler or not ..."
#endif
  OSAL_generate_callstack();
#if defined __i386__ || defined __x86_64__ || defined __arm__ || defined __arm_64__ || defined __aarch64__
  if (lock_return == 0)
  {
     exception_handler_unlock();
  }
  else
  {
     vWritePrintfErrmem("OSAL_generate_callstack_secure -> exception_handler_unlock failed for PID:%d \n","exception_handler_lock",getpid());
  }
#else
#error "unknown compiler environment - don't know wheather to use the exception handler or not ..."
#endif
}

void OSAL_trace_callstack(void)
{
   OSAL_generate_callstack_secure();
}

/*****************************************************************************
*
* FUNCTION:    OSAL_vAssert_Trace
*
* DESCRIPTION: This function is for tracing the Assert reasons.                
*
* PARAMETER:   const char *expr, 
*              const char *file, 
*              tU32 line, 
*              int errmem
*
* RETURNVALUE: none.
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.06  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
static void OSAL_vAssert_Trace(const char *expr, const char *file, tU32 line,  char initiator)
{
  int i,Ret=0,AvoidCs =0,AvoidLines = 0;
  unsigned int buflen = 0;
  unsigned int exprlen = (tU32)strlen(expr);


#if defined __i386__ || defined __x86_64__ || defined __arm__ || defined __arm_64__ || defined __aarch64__
  int size;
  int lock_return = exception_handler_lock();
  if (lock_return != 0)
  {
     size = snprintf(OSAL_vAssert_buf,ASSERT_BUFFER_SIZE,"PID:%d ASSERT Print whithout lock \n",getpid());
     vWriteToErrMem((tS32)TR_COMP_OSALCORE,OSAL_vAssert_buf,(tU32)size,OSAL_STRING_OUT);
  }
#else
#error "unknown compiler environment - don't know wheather to use the exception handler or not ..."
#endif
  /* if assert is followed by reset */
  
  /* first line: ASSERT OSAL mode=NORMAL  behave=TASK_NOACTION, expr=[ALWAYS] */
  memset(&rTraceEntry[0].Output[0], 0, ASSERT_BUFFER_SIZE);
  rTraceEntry[0].Output[0] = initiator;
  rTraceEntry[0].Output[1] = (char)((line & 0x80000000) != 0);
  // OSAL_vAssert_buf[2] = (char)pOsalData->eAssertMode;
  rTraceEntry[0].Output[2] = (char)bGetAssertBehaviour(line); // use assert mode coresponding to the line (high bit) coding and debug/release mode
  buflen += ASSERT_BUFFER_MODE_LINE1_SIZE;
  
  exprlen = MIN(ASSERT_BUFFER_EXPR_LINE1_SIZE, exprlen);
  strncpy( &rTraceEntry[0].Output[buflen], expr,exprlen);
  buflen += exprlen;
  
  rTraceEntry[0].Output[ASSERT_BUFFER_SIZE - 1] = 0;

  /* second line: ASSERT Line=302, File=[x:/di_tengine_os/products/TEngine_driver/spm/tengi.... */
  memset(OSAL_vAssert_buf, 0, ASSERT_BUFFER_SIZE);
  OSAL_vAssert_buf[0] = 0x06; // 6 instead of 5 for proc ans task
  buflen = 1;

  // process name  16 Bytes
//   (void) snprintf( &OSAL_vAssert_buf[buflen], ASSERT_PROC_NAME_SIZE, "P_%d", OSAL_ProcessWhoAmI() );

  char* pTmp = strrchr(commandline, '/');
  if(pTmp)
  {
     (void) snprintf( &OSAL_vAssert_buf[buflen], ASSERT_PROC_NAME_SIZE, "%s",pTmp+1);
  }
  else
  {
     (void) snprintf( &OSAL_vAssert_buf[buflen], ASSERT_PROC_NAME_SIZE, "%s",commandline);
  }
  buflen += ASSERT_PROC_NAME_SIZE;

  // task name  16 Bytes
   if( bGetThreadNameForTID(&OSAL_vAssert_buf[buflen], ASSERT_THREAD_NAME_SIZE, OSAL_ThreadWhoAmI() ) == FALSE )
   {
      (void) snprintf(&OSAL_vAssert_buf[buflen], ASSERT_THREAD_NAME_SIZE, "TID=%d", (int)OSAL_ThreadWhoAmI() );
   }   
  buflen += ASSERT_THREAD_NAME_SIZE;

  OSAL_M_INSERT_T32(&OSAL_vAssert_buf[buflen], (line & (~0x80000000)));
  buflen  += ASSERT_LINE_NUMBER_SIZE;


  exprlen = (tU32)strlen(file);
  if( exprlen >= ASSERT_BUFFER_EXPR_LINE2_SIZE )
  {
      file += exprlen - ASSERT_BUFFER_EXPR_LINE2_SIZE + 1;
	  exprlen = ASSERT_BUFFER_EXPR_LINE2_SIZE;
  }
  strncpy( &OSAL_vAssert_buf[buflen], file, MIN(ASSERT_BUFFER_EXPR_LINE2_SIZE - 1, exprlen) );
  buflen += exprlen;
  
 if(pOsalData->u32DisableAssertCS)
 {
    if(!strcmp(OSAL_vAssert_buf,OSAL_vAssert_buf_Ref))
    {  
       AvoidCs = 1;
       if(pOsalData->u32DisableAssertCS >= 2)
       {
           AvoidLines = 1;
       }
       if(pOsalData->u32DisableAssertCS == 3)
       {
          OSAL_s32ThreadWait(100);
       }
    }
    else
    {
       TraceString("Disable Callstack for OSAL Assert is set to 0x%x",pOsalData->u32DisableAssertCS);
       vWritePrintfErrmem("Disable Callstack for OSAL Assert is set to 0x%x",pOsalData->u32DisableAssertCS);
       strcpy(OSAL_vAssert_buf_Ref,OSAL_vAssert_buf);
	   
    }
  }
  
  if(AvoidLines == 0)
  {
#ifdef SHORT_TRACE_OUTPUT 
  LLD_vTrace( (int)TR_CLASS_ASSERT, (int)TR_LEVEL_FATAL, rTraceEntry[0].Output, buflen );
  vWriteToErrMem( TR_CLASS_ASSERT, rTraceEntry[0].Output, buflen, 0 );
  LLD_vTrace((int)TR_CLASS_ASSERT, (int)TR_LEVEL_FATAL, OSAL_vAssert_buf, buflen);
  vWriteToErrMem( TR_CLASS_ASSERT, OSAL_vAssert_buf, buflen, 0);
#else
  tCString Type,Mode = "SYSTEM_RESET";;
  if(rTraceEntry[0].Output[2] == 0x01)
  { Mode = "SYSTEM_RESET";  }
  else if(rTraceEntry[0].Output[2] == 0x02)
  { Mode = "SYSTEM_SUSPEND";  }
  else if(rTraceEntry[0].Output[2] == 0x03)
  { Mode = "TASK_SUSPEND";  }
  else if(rTraceEntry[0].Output[2] == 0x04)
  { Mode = "TASK_NOACTION";  }
  if  (0 == (line & 0x80000000))
  {
     Type = "FATAL";
  }
  else
  {
     Type = "NORMAL";
  }
  TraceString("OSAL type=%s behave=%s, expr=[%s]",Type,Mode,expr);
  vWritePrintfErrmem("OSAL type=%s behave=%s, expr=[%s] \n",Type,Mode,expr);
  char Name[ASSERT_THREAD_NAME_SIZE];
  tU32 u32Val;
  if  (0 == (line & 0x80000000))
  {
      u32Val = line & 0x80000000;
  }
  else
  {
      u32Val = line & (~0x80000000);
  }

  bGetThreadNameForTID(&Name[0], ASSERT_THREAD_NAME_SIZE, OSAL_ThreadWhoAmI() ); 
  TraceString("Proc=[%s] Thread=[%s], Line=%u, File=[%s]",
              gpOsalProcDat->pu8AppName,Name,(unsigned int)u32Val,file);
  vWritePrintfErrmem("Proc=[%s] Thread=[%s], Line=%u, File=[%s] \n",
               gpOsalProcDat->pu8AppName,Name,u32Val,file);
#endif
  } 
  if(AvoidCs == 0)
  {
     /* generate callstack only when lock is available */
     if(lock_return == 0)
     {
        Ret = OSAL_generate_callstack();
     }
  }  
#if defined __i386__ || defined __x86_64__ || defined __arm__ || defined __arm_64__ || defined __aarch64__
  if (lock_return == 0)
  {
     exception_handler_unlock();
  }
  else
  {
     size = snprintf(OSAL_vAssert_buf,ASSERT_BUFFER_SIZE,"PID:%d ASSERT Print whithout lock end \n",getpid());
     vWriteToErrMem((tS32)TR_COMP_OSALCORE,OSAL_vAssert_buf,size,OSAL_STRING_OUT);
  }
#else
#error "unknown compiler environment - don't know wheather to use the exception handler or not ..."
#endif
  if(AvoidCs == 0)
  {
     /* Do trace Output outside of the locked area */
     if(Ret>(MAX_DEEP+1))Ret=(MAX_DEEP+1);
   
     for(i=0;i<Ret;i++)
     {
#ifdef SHORT_TRACE_OUTPUT 
        if(i==0)
        {
           TraceString(&rTraceEntry[i].Output[0]);
        }
        else
        {
          LLD_vTrace( (int)TR_CLASS_ASSERT, (int)TR_LEVEL_FATAL, &rTraceEntry[i].Output[0], strlen(&rTraceEntry[i].Output[0]) );
        }
#else
        TraceString(&rTraceEntry[i].Output[0]);
#endif 
     }  
   }
}



/*****************************************************************************
*
* FUNCTION:    OSAL_vAssertFunction
*
* DESCRIPTION: This function is for Assertion handling
*
* PARAMETER:   const char *expr, 
*              const char *file, 
*              tU32 line, 
*
* RETURNVALUE: none.
* HISTORY:
* Date      |   Modification                         | Authors
* 03.10.06  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
void OSAL_vAssertFunction(const char* expr, const char* file, tU32 line)
{
#ifdef OSAL_ERRMEM_TIMESTAMP
    vWriteUtcTime(-1);
#endif
   // Assemble trace message
   OSAL_vAssert_Trace(expr, file, line, ASSERT_INITIATOR_OSAL);

   // Determine handling. In Debug and Release mode,
   // three modes are configurable :
   //   Reset (default)
   //   System suspend
   //   Task suspend
   switch ((bGetAssertBehaviour(line)))
   {
      case ASSERTMODE_RESET:
         TraceString("OSAL ASSERT will cause a reboot");
#if defined OSAL_GEN3_SIM || defined OSAL_GEN4_SIM
         if((line & 0x80000000) == 0)
         {
            for(int i = 0; i < 4; i++)
            {
               TraceString("<<<< <<<< FATAL ASSERT <<<< <<<<");
               OSAL_s32ThreadWait(100);
            }
            /* only call reboot/exit for FATAL ASSERT */
	        /* set maker for valid eh_reboot call */
            pOsalData->bNoRebootCallstack = TRUE;
            eh_reboot();
         }
         else
         {
            TraceString("<<<< <<<< NORMAL ASSERT <<<< <<<<");
         }
         OSAL_s32ThreadWait(2000);
#else
         OSAL_s32ThreadWait(1000);
         /* set maker for valid eh_reboot call */
         pOsalData->bNoRebootCallstack = TRUE;
         eh_reboot();
#endif
         // Never should come here
         break;

      case ASSERTMODE_TASKSUSPEND:
         {
            OSAL_tThreadID tid= OSAL_ThreadWhoAmI();
            OSAL_s32ThreadSuspend(tid);
         }
         // Never should come here
         break;
      case ASSERTMODE_TASKNOACTION:
         // No action means return to caller
         return;
      case ASSERTMODE_SUSPEND:
      default:
         // Suspend system below
         break;
   }
}

#ifdef __cplusplus
}
#endif
/************************************************************************
|end of file osalansi.c
|-----------------------------------------------------------------------*/

