/*****************************************************************************
| FILE:         prm_dbuc.c
| PROJECT:      Platform
| SW-COMPONENT: OSAL
|-----------------------------------------------------------------------------
| DESCRIPTION:  This is the implementation file for the OSAL
|               (Operating System Abstraction Layer) 
|
|-----------------------------------------------------------------------------
| COPYRIGHT:    (c) 2017 Bosch GmbH
| HISTORY:      
| Date      | Modification               | Author
| 15.03.17  | Initial revision           | MRK2HI
| --.--.--  | ----------------           | -------, -----
|
|*****************************************************************************/
/************************************************************************
| includes of component-internal interfaces
| (scope: component-local)
|-----------------------------------------------------------------------*/
#include "OsalConf.h"

#ifdef DBUS_SUPPORT

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#ifdef __cplusplus
extern "C" {
#endif


#include "Linux_osal.h"
#include "prm_dbus.h"
#include "ostrace.h"
#include <dbus/dbus.h>
#include <pwd.h>

/************************************************************************
|defines and macros (scope: module-local)
|-----------------------------------------------------------------------*/
const char *const INTERFACE_NAME          = "org.bosch.osal.prm";    // interface to call on
const char *const SERVER_BUS_NAME         = "org.bosch.osal.prm.server"; // target for the method call
const char *const SERVER_OBJECT_PATH_NAME = "/org/bosch/osal/prm";   // object to call on


/************************************************************************
|typedefs (scope: module-local)
|-----------------------------------------------------------------------*/

/************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------*/
static DBusConnection* connclient = NULL;

/************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------*/

/************************************************************************
|function prototype (scope: module-local)
|-----------------------------------------------------------------------*/

/************************************************************************
|function prototype (scope: global)
|-----------------------------------------------------------------------*/

/************************************************************************
|function implementation (scope: module-local)
|-----------------------------------------------------------------------*/


/*****************************************************************************
*
* FUNCTION:    ConnectMethodCaller
*
* DESCRIPTION: This function connect the caller of a metod to the dbus subsystem
*
* PARAMETER:     char*  pointer to name of the DBUS caller, 
*
* RETURNVALUE:   DBusConnection* pointer to DBUS connection handle
*
* HISTORY:
* Date      |   Modification                         | Authors
* 15.03.17  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
DBusConnection* ConnectMethodCaller(void)
{
   DBusError err;
//   int ret;
    // initialiset the errors
   dbus_error_init(&err);

   // connect to the system bus and check for errors
   connclient = dbus_bus_get(DBUS_BUS_SYSTEM, &err);
   if (dbus_error_is_set(&err)) 
   { 
      TraceString("PRM_DBUS Connection Error (%s)\n", err.message); 
      dbus_error_free(&err);
   }
   if (NULL == connclient) 
   { 
      return connclient;
   }

   // request our name on the bus
/*  commented out because DBUS policies don't allow this for standard users
   ret = dbus_bus_request_name(connclient, SERVER_BUS_NAME, DBUS_NAME_FLAG_REPLACE_EXISTING , &err);
   if (dbus_error_is_set(&err)) 
   { 
      TraceString("PRM_DBUS Name Error (%s)\n", err.message); 
      TraceString("PRM_DBUS Pid:%d Tid%d \n",OSAL_ProcessWhoAmI(),OSAL_ThreadWhoAmI()); 
      dbus_error_free(&err);
   }
   if (DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER != ret) 
   { 
      return 0;
   }*/
   return connclient;
}

/*****************************************************************************
*
* FUNCTION:    vCloseConnection
*
* DESCRIPTION: This function is the callbackhandler task for initial process
*
* PARAMETER:     DBusConnection* pointer to DBUS connection handle
*
* RETURNVALUE:   none
*
* HISTORY:
* Date      |   Modification                         | Authors
* 15.03.17  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
void vCloseConnection(void)
{
   dbus_connection_unref(connclient);
   connclient = 0;
}

/*****************************************************************************
*
* FUNCTION:    call_a_method
*
* DESCRIPTION: This functioncall a method via DBUS
*
* PARAMETER:     char* pointer to method parameter
*                char* pointer to method name
*
* RETURNVALUE:   none
*
* HISTORY:
* Date      |   Modification                         | Authors
* 15.03.17  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
unsigned int call_a_method(char* param,char* pcMethod,int control) 
{
   DBusMessage* msg;
   DBusMessageIter args;
   DBusPendingCall* pending;
   dbus_uint32_t level = 0;

   TraceStringClassLevel(OSAL_C_TR_CLASS_PRM_DBUS,TR_LEVEL_DATA,"PRM_DBUS Calling remote method ");

    // create a new method call and check for errors
   msg = dbus_message_new_method_call(SERVER_BUS_NAME, // target for the method call
                                      SERVER_OBJECT_PATH_NAME, // object to call on
                                      INTERFACE_NAME, // interface to call on
                                      pcMethod); // method name
   if (NULL == msg) 
   { 
      TraceString("PRM_DBUS Message Null\n");
      return OSAL_E_NOSPACE;
   }

   if(control != OSAL_C_S32_IOCTRL_PRM_SET_TIME)
   {
   // append arguments
   dbus_message_iter_init_append(msg, &args); 
   if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_INT32,&control)) 
   {
         TraceString("PRM_DBUS dbus_message_iter_append_basic Out Of Memory!\n"); 
         return OSAL_E_NOSPACE;
   }
   }   
   switch(control)
   {
      case 0: // stop message to be created
            TraceStringClassLevel(OSAL_C_TR_CLASS_PRM_DBUS,TR_LEVEL_DATA,"PRM_DBUS received %s",param);
            if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_STRING, &param)) 
            {
                TraceString("PRM_DBUS dbus_message_iter_append_basic Out Of Memory!\n"); 
                return OSAL_E_NOSPACE;
            }
          break;
      case OSAL_C_S32_IOCTRL_PRM_REMOUNT:
           {
              OSAL_trRemountData* pData = (OSAL_trRemountData *)param;
              TraceStringClassLevel(OSAL_C_TR_CLASS_PRM_DBUS,TR_LEVEL_DATA,"PRM_DBUS Calling remote method REMOUNT %s %s",pData->szPath,pData->szOption);
              if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_STRING, &pData->szPath)) 
              {
                 TraceString("PRM_DBUS dbus_message_iter_append_basic Out Of Memory!\n"); 
                 return OSAL_E_NOSPACE;
              }
              if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_STRING, &pData->szOption)) 
              {
                 TraceString("PRM_DBUS dbus_message_iter_append_basic Out Of Memory!\n"); 
                 return OSAL_E_NOSPACE;
              }
          }
         break;
#ifdef PRM_LIBUSB_CONNECTION
     case OSAL_C_S32_IOCTRL_PRM_USBPOWER_GET_PORTCOUNT:
         break;
     case OSAL_C_S32_IOCTRL_PRM_USBPOWER_GET_PORTSTATE:
         break;
     case OSAL_C_S32_IOCTRL_PRM_USBPOWER_SET_PORTPOWER:
          {
             usb_port_control* pData = (usb_port_control*)param;
             TraceStringClassLevel(OSAL_C_TR_CLASS_PRM_DBUS,TR_LEVEL_DATA,"PRM_DBUS Calling remote method SET_PORTPOWER %u %u",pData->PortMask,pData->ValueMask);
             if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_UINT32, &pData->PortMask)) 
             {
                TraceString("PRM_DBUS dbus_message_iter_append_basic Out Of Memory!\n"); 
                return OSAL_E_NOSPACE;
             }
             if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_UINT32, &pData->ValueMask)) 
             {
                TraceString("PRM_DBUS Out Of Memory!\n"); 
                return OSAL_E_NOSPACE;
             }
          }
         break;
#endif
     case OSAL_C_S32_IOCTRL_PRM_SYSTEM_CALL:
          TraceStringClassLevel(OSAL_C_TR_CLASS_PRM_DBUS,TR_LEVEL_DATA,"PRM_DBUS Calling remote method SYSTEM_CALL %s",param);
          if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_STRING, &param)) 
          {
             TraceString("PRM_DBUS dbus_message_iter_append_basic Out Of Memory!\n"); 
             return OSAL_E_NOSPACE;
          }
         break;
     case OSAL_C_S32_IOCTRL_PRM_RESET_ADR3:
          {
             level = (dbus_uint32_t)*param;
             TraceStringClassLevel(OSAL_C_TR_CLASS_PRM_DBUS,TR_LEVEL_DATA,"PRM_DBUS Calling remote method RESET_ADR3 with value %u ",level);
             if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_UINT32, &level)) 
             {
                TraceString("PRM_DBUS dbus_message_iter_append_basic Out Of Memory!\n"); 
                return OSAL_E_NOSPACE;
             }
          }
         break;
     case OSAL_C_S32_IOCTRL_PRM_SET_TIME: 
          TraceStringClassLevel(OSAL_C_TR_CLASS_PRM_DBUS,TR_LEVEL_DATA,"PRM_DBUS Calling remote method SETTIME");
          if(dbus_message_append_args(msg,DBUS_TYPE_ARRAY, DBUS_TYPE_BYTE, &param, sizeof(OSAL_trTimeDate), DBUS_TYPE_INVALID) == FALSE)
          {
             TraceString("OSALMQ_DBUS dbus_message_append_args failed");
             return OSAL_E_NOSPACE;
          } 
         break;
     case OSAL_C_S32_IOCTRL_PRM_SET_PRIO: 
         {
             OSAL_trSysCtrlMsg* pData = (OSAL_trSysCtrlMsg *)param;
             TraceStringClassLevel(OSAL_C_TR_CLASS_PRM_DBUS,TR_LEVEL_DATA,"PRM_DBUS Calling remote method SETPRIO");
             if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_UINT32, &pData->u32WParam)) 
             {
                TraceString("OSALMQ_DBUS dbus_message_append_args failed");
                return OSAL_E_NOSPACE;
             } 
             if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_UINT32, &pData->u32LParam)) 
             {
                TraceString("OSALMQ_DBUS dbus_message_append_args failed");
                return OSAL_E_NOSPACE;
            } 
         }
         break;
     default:
          TraceString("PRM_DBUS Message Null\n");
          return OSAL_E_INVALIDVALUE;
         break;
   }
    
   // send message and get a handle for a reply
   if (!dbus_connection_send_with_reply (connclient, msg, &pending, -1)) 
   { // -1 is default timeout
      TraceString("PRM_DBUS Out Of Memory!\n"); 
      return OSAL_E_NOSPACE;
   }
   if (NULL == pending) 
   { 
      TraceString("Pending Call Null\n"); 
      return OSAL_E_CANCELED;
   }
   dbus_connection_flush(connclient);
   
   TraceStringClassLevel(OSAL_C_TR_CLASS_PRM_DBUS,TR_LEVEL_DATA,"PRM_DBUS Request 0x%x Sent",control);
   
   // free message
   dbus_message_unref(msg);

   // block until we recieve a reply
   dbus_pending_call_block(pending);

   // get the reply message
   msg = dbus_pending_call_steal_reply(pending);
   if (NULL == msg) 
   {
      TraceString("PRM_DBUS Reply Null\n"); 
      return OSAL_E_CANCELED;
   }
   // free the pending message handle
   dbus_pending_call_unref(pending);

   // read the parameters
   if (!dbus_message_iter_init(msg, &args))
   {
      TraceString("PRM_DBUS Message has no arguments!\n"); 
      return OSAL_E_CANCELED;
   }
   else 
   {
      int Type = dbus_message_iter_get_arg_type(&args);
      if (DBUS_TYPE_UINT32 == Type) 
      {
         dbus_message_iter_get_basic(&args, &level);
         //TraceString("PRM_DBUS expected Parameter Argument Type %d!",level); 
      }
      else if(DBUS_TYPE_STRING == Type) 
      {
         char* pMessage = "Unknown";
         dbus_message_iter_get_basic(&args, &pMessage);
         TraceString("PRM_DBUS Unexpected Parameter Argument Type %s!",pMessage); 
         level = OSAL_E_UNKNOWN;
         /*Message recipient disconnected from message bus without replying?*/
         if(strstr(pMessage,"disconnected"))
         {
            TraceString("PRM_DBUS new DBus connection setup will happen");
            connclient = NULL;
         }
      }
      else
      {
         TraceString("PRM_DBUS with wrong Parameter Type %d!",Type); 
      }
   }
   TraceStringClassLevel(OSAL_C_TR_CLASS_PRM_DBUS,TR_LEVEL_DATA,"PRM_DBUS Got Reply Error Code: 0x%x\n", level);
   
   // free reply and close connection
   dbus_message_unref(msg);
   if(level != OSAL_E_NOERROR)
   {
      TraceString("PRM_DBUS call_a_method returns 0x%x",level);
   }
   return level;
}

void CheckUser(void)
{
   struct stat rStat;
   char* pBuffer;
   int fd;
   struct passwd *pw;
   uid_t uid;
   uid = geteuid ();
   pw = getpwuid (uid);
   if (pw)
   {
      TraceString("PRM_DBUS current user %s",pw->pw_name);  
      fd = open("/etc/dbus-1/system.d/org.bosch.osal.prm.conf", O_RDONLY);
      if(fd != -1)
      {
         if(fstat(fd,&rStat) == 0)
         {
            pBuffer = (char*)malloc((size_t)rStat.st_size +1);
            if(pBuffer)
            {	
               int len = (int)read(fd,pBuffer,(size_t)rStat.st_size);
               if(len > 0)
               {
                  *(pBuffer + rStat.st_size) = 0;
                  if(!strstr(pBuffer,pw->pw_name))
                  {
                      TraceString("PRM_DBUS User %s  couldn't found in /etc/dbus-1/system.d/org.bosch.osal.prm.conf",pw->pw_name);
                  }
                  else
                  {
                     TraceString("PRM_DBUS found rule for %s in /etc/dbus-1/system.d/org.bosch.osal.prm.conf",pw->pw_name);
                  }
               }
               free(pBuffer);
            }
          }
          close(fd);
      }
      else
      {
         TraceString("PRM_DBUS cannot open /etc/dbus-1/system.d/org.bosch.osal.prm.conf");  
      }
   }
}


/*****************************************************************************
*
* FUNCTION:    CallDBusMethod
*
* DESCRIPTION: This function is called by external user for interact via DBUS
*
* PARAMETER:     char* pointer to method parameter
*                char* pointer to method name
*
* RETURNVALUE:   none
*
* HISTORY:
* Date      |   Modification                         | Authors
* 15.03.17  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
unsigned int CallDBusMethod(char* param,char* name,int control)
{
   unsigned int ret = OSAL_E_NOERROR;
//   TraceString("PRM_DBUS Pid:%d CallDBusMethod after %d",OSAL_ProcessWhoAmI(),OSAL_ClockGetElapsedTime());
   
   /* check for existing connection */
   if(connclient == NULL)
   {
      ConnectMethodCaller();
   }
   if(connclient != NULL)
   {
     if((ret = call_a_method(param,name,control)) != OSAL_E_NOERROR)
     {
        TraceString("PRM_DBUS call_a_method returns: %d\n", ret);
        if(LLD_bIsTraceActive(OSAL_C_TR_CLASS_PRM_DBUS,TR_LEVEL_DATA))
        {
           CheckUser();
        }
     }
   }
   else
   {
      ret = OSAL_E_UNKNOWN;
   }
   return ret;
}

/*****************************************************************************
*
* FUNCTION:    reply_to_method_call
*
* DESCRIPTION: This function is called to replay a method call after processing
*
* PARAMETER:     char* pointer to peplay message 
*                DBusConnection* pointer to DBUS connection handle
*                dbus_uint32_t   error code for reply
*
* RETURNVALUE:   0 for succes or error code
*
* HISTORY:
* Date      |   Modification                         | Authors
* 15.03.17  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
int reply_call(DBusMessage* msg, DBusConnection* conn,dbus_uint32_t func)
{
   DBusMessageIter args;
   DBusMessage* reply;
   dbus_uint32_t serial = 0;
   // create a reply from the message
   reply = dbus_message_new_method_return(msg);

//   tCString Param = "Answer";
   // add the arguments to the reply
   dbus_message_iter_init_append(reply, &args);
   if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_UINT32, &func)) 
//   if (!dbus_message_iter_append_basic(&args, DBUS_TYPE_STRING,&Param)) 
   { 
      TraceString("PRM_DBUS Out Of Memory!\n"); 
      return -3;
   }

   // send the reply && flush the connection
   if (!dbus_connection_send(conn, reply, &serial)) 
   {
      TraceString("PRM_DBUS Out Of Memory!\n"); 
      return -4;
   }
   dbus_connection_flush(conn);
   // free the reply
   dbus_message_unref(reply);
   return 0;
}

/*****************************************************************************
*
* FUNCTION:    reply_to_remount_method_call
*
* DESCRIPTION: This function is processing remount method call
*
* PARAMETER:     char* pointer to peplay message 
*                DBusConnection* pointer to DBUS connection handle
*
* RETURNVALUE:    0 for succes or error code
*
* HISTORY:
* Date      |   Modification                         | Authors
* 15.03.17  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
int reply_to_remount_method_call(DBusMessage* msg, DBusConnection* conn)
{
   DBusMessageIter args;
   dbus_uint32_t func;
   char* Path = "";
   char* Option = "";
   
   // read the arguments
   if (!dbus_message_iter_init(msg, &args))
   { 
      TraceString("PRM_DBUS Message has no arguments!\n"); 
   }
   else if (DBUS_TYPE_STRING != dbus_message_iter_get_arg_type(&args)) 
   {
       // get function ID 
       dbus_message_iter_get_basic(&args, &func);
       TraceString("PRM_DBUS Function Argument is 0x%x!",func); 
       if(dbus_message_iter_next(&args))
       {
          if (DBUS_TYPE_STRING == dbus_message_iter_get_arg_type(&args))
          {
              dbus_message_iter_get_basic(&args, &Path);
          }
       }
       if(dbus_message_iter_next(&args))
       {
          if (DBUS_TYPE_STRING == dbus_message_iter_get_arg_type(&args))
          {
             dbus_message_iter_get_basic(&args, &Option);
          }
       }
       if(bRemount(Path,Option))
       {
          func = OSAL_E_NOERROR;   
       }
       else
       {
          func = OSAL_E_UNKNOWN;   
       } 
       TraceString("PRM_DBUS REMOUNT Method called with %s %s Error:0x%x", Path,Option,func);
       /* reply with OSAL error code */
       reply_call(msg,conn,func);
   }
   else
   {
      dbus_message_iter_get_basic(&args, &Path);
      TraceString("PRM_DBUS reply_to_remount_method_call with Parameter Argument Type %s!",Path); 
      return -2;
   }
   return 0;
}

/*****************************************************************************
*
* FUNCTION:    reply_to_remount_method_call
*
* DESCRIPTION: This function is processing remount method call
*
* PARAMETER:     char* pointer to peplay message 
*                DBusConnection* pointer to DBUS connection handle
*
* RETURNVALUE:    0 for succes or error code
*
* HISTORY:
* Date      |   Modification                         | Authors
* 15.03.17  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
int reply_to_reset_adr3_method_call(DBusMessage* msg, DBusConnection* conn)
{
   OSAL_tIODescriptor hFd = 0;
   intptr_t Val = OSAL_OK;
   DBusMessageIter args;
   dbus_uint32_t func;
   intptr_t param;
   
   // read the arguments
   if (!dbus_message_iter_init(msg, &args))
   { 
      TraceString("PRM_DBUS Message has no arguments!"); 
   }
   else if (DBUS_TYPE_STRING != dbus_message_iter_get_arg_type(&args)) 
   {
       // get function ID 
       dbus_message_iter_get_basic(&args, &func);
       TraceString("PRM_DBUS Function Argument is 0x%x!",func); 
       if(dbus_message_iter_next(&args))
       {
          if (DBUS_TYPE_UINT32 == dbus_message_iter_get_arg_type(&args))
          {
              dbus_message_iter_get_basic(&args, &param);
              TraceString("PRM_DBUS OSAL_C_S32_IOCTRL_ADR3CTRL_INC_ADR3_DOWN with arguments %d",(int)param); 
          }
       }
   
       hFd = OSAL_IOOpen(OSAL_C_STRING_DEVICE_ADR3CTRL, OSAL_EN_READWRITE);
       if(hFd != OSAL_ERROR)
       {
          func = OSAL_E_NOERROR;
          if(OSAL_s32IOControl(hFd,OSAL_C_S32_IOCTRL_ADR3CTRL_INC_ADR3_DOWN,(intptr_t)param) == OSAL_ERROR)
          {
             func = OSAL_u32ErrorCode();
             TraceString("PRM_DBUS OSAL_C_S32_IOCTRL_ADR3CTRL_INC_ADR3_DOWN failed 0x%x!",func); 
             Val = OSAL_ERROR;
          }
          OSAL_s32IOClose(hFd);
       }
       else
       {
         func = OSAL_u32ErrorCode();
       }   
       TraceString("PRM_DBUS RESET ADR Method called Error:0x%x",func);
       /* reply with OSAL error code */
       reply_call(msg,conn,func);
   }
   else if (DBUS_TYPE_STRING == dbus_message_iter_get_arg_type(&args)) 
   {
      char* command = "Unknown";
      dbus_message_iter_get_basic(&args, &command);
      TraceString("PRM_DBUS reply_to_reset_adr3_method_call with unexpected Parameter Argument Type %s!",command); 
      return -2;
   }
   return (int)Val;
}

/*****************************************************************************
*
* FUNCTION:    reply_to_syscall_method_call
*
* DESCRIPTION: This function is processing system call method call
*
* PARAMETER:     char* pointer to peplay message 
*                DBusConnection* pointer to DBUS connection handle
*
* RETURNVALUE:    0 for succes or error code
*
* HISTORY:
* Date      |   Modification                         | Authors
* 15.03.17  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
int reply_to_syscall_method_call(DBusMessage* msg, DBusConnection* conn)
{
   DBusMessageIter args;
   dbus_uint32_t func;
   char* command = "";
     
   // read the arguments
   if (!dbus_message_iter_init(msg, &args))
   { 
      TraceString("PRM_DBUS Message has no arguments!"); 
   }
   else if (DBUS_TYPE_STRING != dbus_message_iter_get_arg_type(&args)) 
   {
       // get function ID 
       dbus_message_iter_get_basic(&args, &func);
       TraceString("PRM_DBUS Function Argument is 0x%x!",func); 
       if(dbus_message_iter_next(&args))
       {
          if (DBUS_TYPE_STRING == dbus_message_iter_get_arg_type(&args))
          {
              dbus_message_iter_get_basic(&args, &command);
          }
       }
 
       func = OSAL_E_NOERROR;
       if(system(command) == -1)
       {
          func = (dbus_uint32_t)u32ConvertErrorCore(errno);
       }
       TraceString("PRM_DBUS SYSCALL Method called with %s error:0x%x",command,func);
       /* reply with OSAL error code */
       reply_call(msg,conn,func);
   }
   else if (DBUS_TYPE_STRING == dbus_message_iter_get_arg_type(&args)) 
   {
      command = "Unknown";
      dbus_message_iter_get_basic(&args, &command);
      TraceString("PRM_DBUS reply_to_syscall_method_call with unexpected Parameter Argument Type %s!",command); 
      return -2;
   }
   return 0;
}

/*****************************************************************************
*
* FUNCTION:    reply_to_stop_method_call
*
* DESCRIPTION: This function is processing system call method call
*
* PARAMETER:     char* pointer to peplay message 
*                DBusConnection* pointer to DBUS connection handle
*
* RETURNVALUE:    0 for succes or error code
*
* HISTORY:
* Date      |   Modification                         | Authors
* 15.03.17  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
int reply_to_stop_method_call(DBusMessage* msg, DBusConnection* conn)
{
   DBusMessageIter args;
   dbus_uint32_t func;
   char* command = "";
     
   // read the arguments
   if (!dbus_message_iter_init(msg, &args))
   { 
      TraceString("PRM_DBUS Message has no arguments!"); 
   }
   else if (DBUS_TYPE_STRING != dbus_message_iter_get_arg_type(&args)) 
   {
       // get function ID 
       dbus_message_iter_get_basic(&args, &func);
       TraceString("PRM_DBUS Function Argument is 0x%x!",func); 
  
       func = OSAL_E_NOERROR;
       TraceString("PRM_DBUS SYSCALL Method called with %s error:0x%x",command,func);
        /* reply with OSAL error code */
       reply_call(msg,conn,func);
   }
   else if (DBUS_TYPE_STRING == dbus_message_iter_get_arg_type(&args)) 
   {
      command = "Unknown";
      dbus_message_iter_get_basic(&args, &command);
      TraceString("PRM_DBUS reply_to_stop_method_call with unexpected Parameter Argument Type %s!",command); 
      return -2;
   }
   return 0;
}



/*****************************************************************************
*
* FUNCTION:    reply_to_usbsetpwr_method_call
*
* DESCRIPTION: This function is processing set USB power method call
*
* PARAMETER:     char* pointer to peplay message 
*                DBusConnection* pointer to DBUS connection handle
*
* RETURNVALUE:    0 for succes or error code
*
* HISTORY:
* Date      |   Modification                         | Authors
* 15.03.17  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
int reply_to_usbsetpwr_method_call(DBusMessage* msg, DBusConnection* conn)
{
   dbus_uint32_t func;

#ifdef PRM_LIBUSB_CONNECTION
   DBusMessageIter args;
   usb_port_control rControl;
   memset(&rControl,0,sizeof(usb_port_control));

   // read the arguments
   if (!dbus_message_iter_init(msg, &args))
   { 
     TraceString("PRM_DBUS Message has no arguments!"); 
   }
   else if (DBUS_TYPE_STRING != dbus_message_iter_get_arg_type(&args)) 
   {
       // get function ID 
       dbus_message_iter_get_basic(&args, &func);
       TraceString("PRM_DBUS Function Argument is 0x%x!",func); 
       if(dbus_message_iter_next(&args))
       {
          if (DBUS_TYPE_UINT32 == dbus_message_iter_get_arg_type(&args))
          {
              dbus_message_iter_get_basic(&args, &rControl.PortMask);
          }
       }
       if(dbus_message_iter_next(&args))
       {
          if (DBUS_TYPE_UINT32 == dbus_message_iter_get_arg_type(&args))
          {
              dbus_message_iter_get_basic(&args, &rControl.ValueMask);
          }
       }
       func = prmUSBPower_control((intptr_t)&rControl);
       TraceString("PRM_DBUS SET_USBPWR Method called Port:%u Value:%u error:0x%x",rControl.PortMask,rControl.ValueMask,func);
       /* reply with OSAL error code */
       reply_call(msg,conn,func);
   }
   else if (DBUS_TYPE_STRING == dbus_message_iter_get_arg_type(&args)) 
   {
      char* command = "Unknown";
      dbus_message_iter_get_basic(&args, &command);
      TraceString("PRM_DBUS reply_to_usbsetpwr_method_call with unexpected Parameter Argument Type %s!",command); 
      return -2;
   }
#else
   func = OSAL_E_NOTSUPPORTED;
   TraceString("PRM_DBUS SET_USBPWR Method not supported error:0x%x",func);
   /* reply with OSAL error code */
   reply_call(msg,conn,func);
#endif
   return 0;
}


/*****************************************************************************
*
* FUNCTION:    reply_to_setprio_method_call
*
* DESCRIPTION: This function is processing set USB power method call
*
* PARAMETER:     char* pointer to peplay message 
*                DBusConnection* pointer to DBUS connection handle
*
* RETURNVALUE:    0 for succes or error code
*
* HISTORY:
* Date      |   Modification                         | Authors
* 15.03.17  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
int reply_to_setprio_method_call(DBusMessage* msg, DBusConnection* conn)
{
   dbus_uint32_t func;

#ifdef PRM_LIBUSB_CONNECTION
//   DBusError err;
   DBusMessageIter args;
   // read the arguments
   if (!dbus_message_iter_init(msg, &args))
   { 
      TraceString("PRM_DBUS Message has no arguments!"); 
   }
   else if (DBUS_TYPE_STRING != dbus_message_iter_get_arg_type(&args)) 
   {
       tU32 u32Tid = 0;
       tU32 u32Prio = 0;
       // get function ID 
       dbus_message_iter_get_basic(&args, &func);
       TraceString("PRM_DBUS Function Argument is 0x%x!",func); 
       if(dbus_message_iter_next(&args))
       {
          if (DBUS_TYPE_UINT32 == dbus_message_iter_get_arg_type(&args))
          {
              dbus_message_iter_get_basic(&args, &u32Tid);
          }
       }
       if(dbus_message_iter_next(&args))
       {
          if (DBUS_TYPE_UINT32 == dbus_message_iter_get_arg_type(&args))
          {
              dbus_message_iter_get_basic(&args, &u32Prio);
          }
       }
       func = OSAL_E_NOERROR;
       char command[128];
       int Pri = (int)u32Prio;
       if((Pri <= 19)&&(Pri >= -20))
       {
         snprintf(command,128,"renice -n %d %d \n",Pri,(int)u32Tid);
       }
       else
       {
          snprintf(command,128,"chrt -p --rr %d %d",Pri,u32Tid);
       }
       if(system(command) == -1)
       {
          func = u32ConvertErrorCore(errno);
          TraceString("PRM_DBUS SET_PRIO %s error:0x%x",command,func);
       }
       /* reply with OSAL error code */
       reply_call(msg,conn,func);

   }
   else if (DBUS_TYPE_STRING == dbus_message_iter_get_arg_type(&args)) 
   {
      char* command = "Unknown";
      dbus_message_iter_get_basic(&args, &command);
      TraceString("PRM_DBUS reply_to_setprio_method_call with unexpected Parameter Argument Type %s!",command); 
      return -2;
   }
#else
   func = OSAL_E_NOTSUPPORTED;
   TraceString("PRM_DBUS SET_TIME Method not supported error:0x%x",func);
   /* reply with OSAL error code */
   reply_call(msg,conn,func);
#endif
   return 0;
}

/*****************************************************************************
*
* FUNCTION:    reply_to_settime_method_call
*
* DESCRIPTION: This function is processing set USB power method call
*
* PARAMETER:     char* pointer to peplay message 
*                DBusConnection* pointer to DBUS connection handle
*
* RETURNVALUE:    0 for succes or error code
*
* HISTORY:
* Date      |   Modification                         | Authors
* 15.03.17  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
int reply_to_settime_method_call(DBusMessage* msg, DBusConnection* conn)
{
   dbus_uint32_t func;

#ifdef PRM_LIBUSB_CONNECTION
   DBusError err;
   DBusMessageIter args;
   int len=0;
 
   // read the arguments
   if (!dbus_message_iter_init(msg, &args))
   { 
      TraceString("PRM_DBUS Message has no arguments!"); 
   }
   else if (DBUS_TYPE_STRING != dbus_message_iter_get_arg_type(&args)) 
   {
       // get function ID 
       dbus_message_iter_get_basic(&args, &func);
       TraceString("PRM_DBUS Function Argument is 0x%x!",func); 
       if(dbus_message_iter_next(&args))
       {
          OSAL_trTimeDate rSetTime;
          if(dbus_message_get_args(msg,&err,DBUS_TYPE_ARRAY, DBUS_TYPE_BYTE,&rSetTime, &len,DBUS_TYPE_INVALID))
          {
             func = OSAL_E_NOERROR;
             if(OSAL_s32ClockSetTime(&rSetTime) == OSAL_ERROR)
             {
                func = OSAL_u32ErrorCode();
             }
          }
          else
          {
             func = OSAL_E_INVALIDVALUE;
          }
          TraceString("PRM_DBUS SET_TIME Method called  error:0x%x",func);
        }
        /* reply with OSAL error code */
        reply_call(msg,conn,func);

   }
   else if (DBUS_TYPE_STRING == dbus_message_iter_get_arg_type(&args)) 
   {
      char* command = "Unknown";
      dbus_message_iter_get_basic(&args, &command);
      TraceString("PRM_DBUS reply_to_settime_method_call with unexpected Parameter Argument Type %s!",command); 
      return -2;
   }
#else
   func = OSAL_E_NOTSUPPORTED;
   TraceString("PRM_DBUS SET_TIME Method not supported error:0x%x",func);
   /* reply with OSAL error code */
   reply_call(msg,conn,func);
#endif
   return 0;
}

/*****************************************************************************
*
* FUNCTION:    reply_to_nonsupported_method_call
*
* DESCRIPTION: This function is processing a not implemented method call
*
* PARAMETER:     char* pointer to peplay message 
*                DBusConnection* pointer to DBUS connection handle
*
* RETURNVALUE:    0 for succes or error code
*
* HISTORY:
* Date      |   Modification                         | Authors
* 15.03.17  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
int reply_to_nonsupported_method_call(DBusMessage* msg, DBusConnection* conn)
{
   DBusMessageIter args;
   dbus_uint32_t func;
   char* param = "";
   
   TraceString("PRM_DBUS unhandled Message received !"); 
 
   // read the arguments
   if (!dbus_message_iter_init(msg, &args))
   { 
      TraceString("PRM_DBUS Message has no arguments!"); 
   }
   else if (DBUS_TYPE_STRING != dbus_message_iter_get_arg_type(&args)) 
   {
       // get function ID 
       dbus_message_iter_get_basic(&args, &func);
       TraceString("PRM_DBUS Function Argument is 0x%x!",func); 
       /* pintout all parameter */
       while(1)
       {
          if(dbus_message_iter_next(&args))
          {
             if (DBUS_TYPE_STRING == dbus_message_iter_get_arg_type(&args))
             {
                dbus_message_iter_get_basic(&args, &param);
                TraceString("PRM_DBUS Function Argument is %s!",param); 
             }
             else
             {
                dbus_message_iter_get_basic(&args, &func);
                TraceString("PRM_DBUS Function Argument is 0x%x!",func); 
             }
          }
          else
          {
             break;
          }
       }
       func = OSAL_E_NOTSUPPORTED;
       /* reply with OSAL error code */
       reply_call(msg,conn,func);
   }
   else
   {
      dbus_message_iter_get_basic(&args, &param);
      TraceString("PRM_DBUS Unknown Message with Parameter Argument is %s!",param); 
   }
   return 0;
}


/*****************************************************************************
*
* FUNCTION:    vDBusReceiver
*
* DESCRIPTION: This function is the DBUS receiver task
*
* PARAMETER:     void* pointer to task argument
*
* RETURNVALUE:   none
*
* HISTORY:
* Date      |   Modification                         | Authors
* 15.03.17  | Initial revision                       | MRK2HI
* --.--.--  | ----------------                       | -----
*
*****************************************************************************/
void vDBusReceiver(void* pArg)
{
   ((void)pArg);
   DBusError err;
   DBusConnection* conn;
   DBusMessage* msg;
   int ret;
//   int MsgCnt = 0;
   
   // initialise the errors
   dbus_error_init(&err);

// connect to the bus
   conn = dbus_bus_get(DBUS_BUS_SYSTEM, &err);
   if (dbus_error_is_set(&err)) 
   { 
      TraceString("PRM_DBUS Connection Error (%s)", err.message); 
      dbus_error_free(&err); 
   }
   if (NULL != conn) 
   { 
      // request a name on the bus
      ret = dbus_bus_request_name(conn, SERVER_BUS_NAME, 
                                  DBUS_NAME_FLAG_REPLACE_EXISTING,
                                  &err);
      if (dbus_error_is_set(&err)) 
      { 
         TraceString("PRM_DBUS Name Error (%s)", err.message); 
         dbus_error_free(&err); 
      }
	  
 //     TraceString("PRM_DBUS Server available after %d msec",OSAL_ClockGetElapsedTime()); 
	  
      if (DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER == ret) 
      { 
          /* wait for incoming requests */
         while(1)
         {
            // Block for msg from client 
            if (!dbus_connection_read_write_dispatch (conn, -1)) 
            { 
               TraceString("PRM_DBUS Not connected now."); 
            }
   
            msg = dbus_connection_pop_message(conn);
            // loop again if we haven't got a message
            if (NULL == msg) 
            { 
               sleep(1); 
               continue; 
            }

            // check this is a method call for the right interface and method
            if(dbus_message_is_method_call(msg, INTERFACE_NAME,METHOD_REMOUNT))
            {
               if((ret = reply_to_remount_method_call(msg, conn)) != 0)
               {
                  TraceString("PRM_DBUS reply_to_remount_method_call Error (%s)", err.message); 
               }
            }
            else if(dbus_message_is_method_call(msg, INTERFACE_NAME,METHOD_RESET_ADR3))
            {   
               if((ret = reply_to_reset_adr3_method_call(msg, conn)) != 0)
               {
                  TraceString("PRM_DBUS reply_to_reset_adr3_method_call Error (%s)", err.message); 
               }
            }
            else if(dbus_message_is_method_call(msg,INTERFACE_NAME, METHOD_SYSTEM_CALL))
            {
               if((ret = reply_to_syscall_method_call(msg, conn)) != 0)
               {
                  TraceString("PRM_DBUS reply_to_remount_method_call Error (%s)", err.message); 
               }
            }
            else if(dbus_message_is_method_call(msg,INTERFACE_NAME,METHOD_USB_SET_PWR))
            {
               if((ret = reply_to_usbsetpwr_method_call(msg, conn)) != 0)
               {
                  TraceString("PRM_DBUS reply_to_usbsetpwr_method_call Error (%s)", err.message); 
               }
            }
            else if(dbus_message_is_method_call(msg,INTERFACE_NAME,METHOD_SET_TIME))
            {
               if((ret = reply_to_settime_method_call(msg, conn)) != 0)
               {
                  TraceString("PRM_DBUS reply_to_set_time_method_call Error (%s)", err.message); 
               }
               break;
            }
            else if(dbus_message_is_method_call(msg,INTERFACE_NAME,METHOD_SET_RT_PRIO))
            {
               if((ret = reply_to_setprio_method_call(msg, conn)) != 0)
               {
                  TraceString("PRM_DBUS reply_to_set_prio_method_call Error (%s)", err.message); 
               }
               break;
            }
            else if(dbus_message_is_method_call(msg,INTERFACE_NAME,METHOD_STOP))
            {
               if((ret = reply_to_stop_method_call(msg, conn)) != 0)
               {
                  TraceString("PRM_DBUS reply_to_stop_method_call Error (%s)", err.message); 
               }
               break;
            }
            else
            {
               if (dbus_error_is_set(&err)) 
               {
                  TraceString("PRM_DBUS unexpected method call Error (%s)", err.message); 
                  dbus_error_free(&err); 
               }
              // if(MsgCnt == 0)
               {
                  if((ret = reply_to_nonsupported_method_call(msg, conn)) != 0)
                  {
                      TraceString("PRM_DBUS reply_to_nonsupported_method_call Error (%s)", err.message); 
                  }
               }
            }
            // free the message
            dbus_message_unref(msg);
         }
         TraceString("PRM_DBUS vDBusReceiver task ends "); 
      }
      else
      {
         TraceString("PRM_DBUS DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER != ret");  
      }	  
      dbus_connection_unref(conn); 
   }
   else
   {
      TraceString("PRM_DBUS No connection established");
   }
}

#ifdef __cplusplus
}
#endif

#endif //DBUS_SUPPORT

/************************************************************************
|end of file prm_dbus.c
|-----------------------------------------------------------------------*/