/*****************************************************************************************
* Copyright (C) RBEI, 2019
* This software is property of Robert Bosch.
* Unauthorized duplication and disclosure to third parties is prohibited.
******************************************************************************************/
/*******************************************************************************
*
* FILE			:	writeprofiler_monitor.c
*
* SW-COMPONENT	:	eMMC Write profiler
*
* PROJECT		:	This will be common for all the projects.
*
* DESCRIPTION	: This contains the implementation to offer the monitoring the
* 				 device stats and gathering the device stats information.
* 				  Upon the conditions like shutdown notification,the recorded
* 				  data shall be stored in to the configured persistent location

* AUTHOR		:	Ravindra Prabha (RBEI/ECF1)
*
* COPYRIGHT:    (c) 2019 Robert Bosch GmbH, Hildesheim
*
* HISTORY      :
*------------------------------------------------------------------------
* Date      |       Version       | Author & comments
*-----------|---------------------|--------------------------------------
*-----------|---------------------|--------------------------------------
*27.Sep.2019| Initial version 1.0 | Ravindra Prabha (RBEI/ECF1)
* -----------------------------------------------------------------------
* -----------------------------------------------------------------------

/******************************************************************************/
/* INCLUDES                                                                   */
/******************************************************************************/
#include "writeprofiler.h"
/******************************************************************************/

/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_UpdatePresistentFile
* PARAMETER 	  : fp			: File pointer for persistent file location
*					ringbuffer	: Date to write on the persistent file location
* RETURNVALUE	  :
* DESCRIPTION	  : This function writes the date, block count and total write count
*					to the persistent file location
**********************************************************************************/

tBool eMMC_ExcessiveWP_UpdatePresistentFile (FILE *fp, node* ringbuffer )
{
	tS32 mon,day,year;

	DEBUG(" <eMMC_ExcessiveWP_UpdatePresistentFile> Date:%02d:%02d:%04d Blk_Count %llu TotalWriteBytes %llu\n",ringbuffer->date.tm_mday,ringbuffer->date.tm_mon,ringbuffer->date.tm_year,ringbuffer->blk_count,ringbuffer->total_write_cycles_count);

	/* The data in RAM maintained in struct tm format but store the data into the file in user understandable date */
	year = ringbuffer->date.tm_year + 1900;
	mon = ringbuffer->date.tm_mon + 1;
	day = ringbuffer->date.tm_mday;

	return fprintf(fp,PERSISTENT_FILE_DATA_FORMAT,day,mon,year,ringbuffer->blk_count,ringbuffer->total_write_cycles_count);
}

/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_ReadLineInPresistentFile
* PARAMETER 	  : fp			: File pointer for persistent file location
*					ringbuffer	: Date to write on the persistent file location
* RETURNVALUE	  :
* DESCRIPTION	  : This function Read the date, block count and total write count
*					from the persistent file location
**********************************************************************************/
tS32 eMMC_ExcessiveWP_ReadLineInPresistentFile (FILE *fp, node* ringbuffer )
{
	tS32 ret;
	tS32 mon,day,year;

	ret=fscanf(fp,PERSISTENT_FILE_DATA_FORMAT,&day,&mon,&year,&ringbuffer->blk_count,&ringbuffer->total_write_cycles_count);

	DEBUG(" in eMMC_ExcessiveWP_ReadLineInPresistentFile,ret=%d,Date:%02d/%02d/%04d Blk_Count %llu TotalWriteBytes %llu\n",ret,ringbuffer->date.tm_mday,ringbuffer->date.tm_mon,ringbuffer->date.tm_year,ringbuffer->blk_count,ringbuffer->total_write_cycles_count);

	/* The data in RAM has to be maintained in struct tm format  */
	ringbuffer->date.tm_year = year - 1900;
	ringbuffer->date.tm_mon = mon - 1;
	ringbuffer->date.tm_mday = day;

	return ret;
}

/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_InsertNode
* PARAMETER 	  : blkioLocationInfo	: Location details which read from cfg file
*					BlkCount			: cumulative block count on single power cycle
*					date				: Date to store on the ring buffer
* RETURNVALUE	  : TRUE : Success ; FALSE : Failure ;
* DESCRIPTION	  : Create the new node and add it at the end of the ring buffer.
*					Also check the total ring buffer(i.e no. of persistent file entries)
*					counts exceeds the maximum entries and remove the last entry.
*Add new ring buffer RB3 (RB-Ring buffer)
*	existing Ring buffer 					: RB1->RB2->RB1
*	Create the New node 					: RB3
*	Inster it on the existing ring buffer	: RB1->RB2->RB3->RB1
* 	Check the total Ring buffer counts with
* 	maximum configured entry count(assume 2)
*	and remove the the first entry			: RB2->RB3->RB2
*
**********************************************************************************/
tBool eMMC_ExcessiveWP_InsertNode (monitor_blkio *blkioLocationInfo, long BlkCount, struct tm date)
{
	node 	*NewNode,*StartNode,*LastNode,*rmNode;
	tS32		NodeCnt = 0;
	if( NULL == blkioLocationInfo ) {
		return FALSE;
	}

	NewNode = eMMC_ExcessiveWP_CreateNode( );
	if( NewNode == NULL ) {
		return FALSE;
	}

	NewNode->blk_count = BlkCount;
	NewNode->date = date;
	NewNode->total_write_cycles_count = 0;

	if ( blkioLocationInfo->ringnode == NULL ) {
		NewNode->next = NewNode;
		NewNode->prev = NewNode;
		blkioLocationInfo->ringnode = NewNode;
		DEBUG("Created First Node <%d/%2d/%4d> Blk<%d>\n",date.tm_mday,date.tm_mon,date.tm_year,BlkCount);
	}
	else {
		DEBUG("Node Inserted\n");

		/* Add the node at the last */
		StartNode 	= blkioLocationInfo->ringnode;
		LastNode	= StartNode->prev;
		NewNode->next= StartNode;
		NewNode->prev= LastNode;
		LastNode->next = NewNode;
		StartNode->prev = NewNode;
		NewNode->total_write_cycles_count = LastNode->blk_count + LastNode->total_write_cycles_count;

		StartNode = blkioLocationInfo->ringnode;
		LastNode = blkioLocationInfo->ringnode;
		do
		{
			NodeCnt++;
			StartNode=StartNode->next;
		}while ( StartNode != LastNode);

		if(NodeCnt > blkioLocationInfo->MaxDuration) {
			DEBUG("Total Entry(%d) Reached Max Entry(%d)\n",NodeCnt,blkioLocationInfo->MaxDuration);

			StartNode   = blkioLocationInfo->ringnode;
			LastNode    = StartNode->prev;

			/*
			The below algorithm removes the first node in the list
			*/
			rmNode = StartNode;

			StartNode = StartNode->next;

			/*
			Isolate the rmnode from the list by disconnecting
			the next of its previous and previous of its next
			*/
			StartNode->prev = LastNode;
			LastNode->next =  StartNode;

			eMMC_ExcessiveWP_DeallocateMemory(rmNode);
			blkioLocationInfo->ringnode=StartNode;
		}
	}

	/* Update the cumulative count with respect to last entry. Whenever a new node has added,
	 * the blk_count and total count is updated based on it's previous node.
	 * So finaly cumulative count is based on the last entry of the list.
	 */
	LastNode = blkioLocationInfo->ringnode->prev;
	blkioLocationInfo->CumulativeBlkCount = LastNode->blk_count + LastNode->total_write_cycles_count;

	return TRUE;
}

/********************************************************************************
* FUNCTION        :  eMMC_ExcessiveWP_GetmmcHostCardPath
* PARAMETER       :  MmcCard - Hold card info,  MmcHost - hold host Info
* RETURNVALUE     :  0 - success , non zero - failure
* DESCRIPTION     :  This will find the mmc card and host path from the sysfs
*------------------------------------------------------------------------
* Date      |       Version       | Author & comments
*-----------|---------------------|--------------------------------------
*07.SEPT.2018| Initial Version 1.0| Selvakumar Kalimuthu (RBEI/ECF22)
*------------------------------------------------------------------------
**********************************************************************************/
tS32 eMMC_ExcessiveWP_GetmmcHostCardPath ( tChar *MmcCard, tChar *MmcHost )
{
    DIR             *sysDirMmcHostPath,
                    *sysDirMmcCardPath;
    struct dirent   *dirMmcHostEntry,
                    *dirMmcCardEntry;
    FILE            *fpMmcDevType;
    tS32            s32retVal = REG_PATH_UNAVAIL;
    /* Since, we know the file format for mmc_card amd host info ,
       so we have limited the length by 32 bytes*/
    tChar             devType [8]                 = {0};
    tChar            	mmcHostCardPath [MAXLENGTH] = {0};
    tBool           bFoundPath = FALSE;

	/* Invalid argument will lead to PATH realization error */
	if ((NULL == MmcCard) || (NULL == MmcHost)) {
		return REG_PATH_UNAVAIL;
	}

    /*MMC_HOST: Open the sysfs directory /sys/class/mmc_host/ */
    sysDirMmcHostPath = opendir(SYS_DIR_MMCHOST_PATH);
    if (NULL == sysDirMmcHostPath) {
        ERROR("!ERROR: Open Dir %s Fails!!! E:%d",SYS_DIR_MMCHOST_PATH,errno);
        s32retVal |= MMC_HOST_INFO_ACCESS_ERROR;
    }
    else {
        /* MMC_HOST: Read the entries in the host direcotry and find the device type as MMC */
        while (((dirMmcHostEntry= readdir(sysDirMmcHostPath)) != NULL) && (bFoundPath == FALSE))
        {
            /* MMC_HOST(mmc0/1/2): find the mmc card */
            if (!strncmp(dirMmcHostEntry->d_name,"mmc",3))
            {
                /* Concordinate the host path with the mmc host path
                    paths: /sys/class/mmc_host/<mmc0/1/2>/ */
                snprintf(mmcHostCardPath,sizeof(mmcHostCardPath)-1,"%s/%s/"
                                                 ,SYS_DIR_MMCHOST_PATH
                                               ,dirMmcHostEntry->d_name);

                /* MMC_CARD: Open the mmc card */
                sysDirMmcCardPath = opendir (mmcHostCardPath);
                if (NULL == sysDirMmcCardPath) {
                    ERROR ("!!!ERROR: Open Dir %s Fails!!! E:%d",mmcHostCardPath,errno);
                    s32retVal |= MMC_CARD_INFO_ACCESS_ERROR;
                    continue;
                }
                else {
                    /* MMC_CARD(mmc0/1/2:001): Read the entries in the mmc card path and find the device type as "MMC"*/
                    while (((dirMmcCardEntry= readdir(sysDirMmcCardPath)) != NULL)  && (bFoundPath == FALSE))
                    {
                        /* MMC_CARD(mmc0/1/2:001): Read the entries in the mmc card path and find the device type */
                        if (!strncmp (dirMmcCardEntry->d_name, dirMmcHostEntry->d_name, strlen (dirMmcHostEntry->d_name)))
                        {
                            /* Concordinate the card path with the mmc host path
                                paths: /sys/class/mmc_host/<mmc0/1/2>/<mmc0/1/2:001>/type */
                            snprintf(mmcHostCardPath,sizeof(mmcHostCardPath)-1,"%s/%s/%s/type"
                                                                            ,SYS_DIR_MMCHOST_PATH
                                                                            ,dirMmcHostEntry->d_name
                                                                            ,dirMmcCardEntry->d_name);

                            /*Read the type of the device*/
                            fpMmcDevType = fopen(mmcHostCardPath,"r");
                            if(fpMmcDevType == NULL ) {
                                ERROR("!!!ERROR: File open fails  <%s>E:%d",mmcHostCardPath,errno);
								s32retVal |= MMC_CARD_INFO_ACCESS_ERROR;
                                break;
                            }

                            (void)fgets(devType, sizeof(devType)-1, fpMmcDevType);
                            if (!strncmp(devType,"MMC",strlen("MMC"))) {
                                strncpy(MmcHost,dirMmcHostEntry->d_name,NODE_LENGTH - 1);
                                strncpy(MmcCard,dirMmcCardEntry->d_name,NODE_LENGTH - 1);
                                s32retVal = 0;
                                bFoundPath = TRUE;
                            }
                            fclose(fpMmcDevType);
                        }
                    }
                    closedir(sysDirMmcCardPath);
                }
            }
        }
        closedir(sysDirMmcHostPath);
    }
	return s32retVal;
}

/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_PopulateNodes
* PARAMETER 	  : fptr				: File pointer for persistent file location
*					blkioLocationInfo	: Location details which read from cfg file
* RETURNVALUE	  : TRUE : Success ; FALSE : Failure ;
* DESCRIPTION	  : This function read the entries from the persistent file and
*					populate the ring buffer.
**********************************************************************************/
tBool eMMC_ExcessiveWP_PopulateNodes(FILE *fptr,monitor_blkio *blkioLocationInfo)
{
	node	nodePersistentdata = {0};
	node 	*ringbuffer;
	node 	*LastNode,*StartNode,*NewNode;
	tS32	NodeCount=0;

	ringbuffer= blkioLocationInfo->ringnode;

	while ( eMMC_ExcessiveWP_ReadLineInPresistentFile(fptr,&nodePersistentdata) > 0 )
	{
		StartNode = ringbuffer;
		NodeCount++;
		DEBUG("(%d)first node creation under insert_node()\n",NodeCount);
		if(ringbuffer==NULL) {
			DEBUG(("Create First Node\n"));
			ringbuffer = eMMC_ExcessiveWP_CreateNode();

			if( ringbuffer == NULL ) {
				return FALSE;
			}
			ringbuffer->date = nodePersistentdata.date;
			ringbuffer->blk_count 					= nodePersistentdata.blk_count;
			ringbuffer->total_write_cycles_count	= nodePersistentdata.total_write_cycles_count;
			ringbuffer->next = ringbuffer;
			ringbuffer->prev = ringbuffer;

			blkioLocationInfo->ringnode = ringbuffer;
		}
		else {
			LastNode=StartNode->prev;

			NewNode=eMMC_ExcessiveWP_CreateNode();
			if( NewNode == NULL ) {
				return FALSE;
			}
			NewNode->date = nodePersistentdata.date;
			NewNode->blk_count 	= nodePersistentdata.blk_count;
			NewNode->total_write_cycles_count	= nodePersistentdata.total_write_cycles_count;
			NewNode->next = StartNode;
			NewNode->prev = LastNode;
			LastNode->next = NewNode;
			StartNode->prev = NewNode;
			NewNode->total_write_cycles_count = LastNode->blk_count + LastNode->total_write_cycles_count;
		}
	}

	if ( NodeCount == 0 ) {
		ERROR("!!ERROR:no entries are present in a file \n");
		return FALSE;
	}

	return TRUE;
}

/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_UpdateTotalBlockCount
* PARAMETER 	  : DevName		: Monitoring device name
*					TotalBlkcnt	: Total block count since beginning day.
* RETURNVALUE	  : TRUE - Success ; FALSE - Failure
* DESCRIPTION	  : Update the cumulative write count
**********************************************************************************/
tBool eMMC_ExcessiveWP_UpdateTotalBlockCount( tChar *DevName, long long TotalBlkcnt)
{
	FILE		*fpTotalClkCnt = NULL;
	tChar		DevNameuf[CFG_NAME_LEN];
	long long	TotalWriteCount = 0;
	tBool		IsFileUpdated = FALSE;

	sem_wait(&RestrictSharedAccessSem);
	fpTotalClkCnt = fopen(TOT_BLK_COUNT_PERSISTENT_FILE,MODE_RW);
	if ( fpTotalClkCnt == NULL ) {
		fpTotalClkCnt = fopen(TOT_BLK_COUNT_PERSISTENT_FILE,MODE_W);
		if ( fpTotalClkCnt == NULL ) {
			ERROR("!!ERROR: File Open Fail <%s>\n",TOT_BLK_COUNT_PERSISTENT_FILE);
		}
	}

	if ( fpTotalClkCnt != NULL ) {
		while ( fscanf ( fpTotalClkCnt,TOTAL_BLK_CNT_READ_PATTERN,DevNameuf,&TotalWriteCount) != -1)
		{
			DEBUG("Dev:%s TotalCount:%016lld\n",DevName,TotalBlkcnt);
			if(!strcmp(DevName,DevNameuf))	{
				/*the value is updated by %016d so added 16 count to move the file offset and
				 * count 4 is for maintain the space and : seperation.*/
				if ( -1 != fseek(fpTotalClkCnt,-(strlen(DevNameuf)+16U+4U),SEEK_CUR)) 	{
					fprintf(fpTotalClkCnt,TOTAL_BLK_CNT_ENTRY_PATTERN,DevName,TotalBlkcnt);
					IsFileUpdated = TRUE;
					break;
				}
			}
		}
		if( IsFileUpdated == FALSE ) {
			DEBUG("End:Dev:%s TotalCount:%016lld\n",DevName,TotalBlkcnt);
			fprintf(fpTotalClkCnt,TOTAL_BLK_CNT_ENTRY_PATTERN,DevName,TotalBlkcnt);
		}
		eMMC_ExcessiveWP_SyncData(fpTotalClkCnt);
		fclose(fpTotalClkCnt);
		fpTotalClkCnt = NULL;
	}
	sem_post(&RestrictSharedAccessSem);
	return TRUE;
}

/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWriteGetContainerPath
* PARAMETER 	  : ContainerName	: Name of the container
*					blkioContainerpath : path in the cgroup to get the io details.
* RETURNVALUE	  :
* DESCRIPTION	  : Get the path from the cgroup for monitor the container IO details.
**********************************************************************************/
tBool eMMC_ExcessiveWriteGetContainerPath(tChar *ContainerName, tChar* blkioContainerpath)
{
	tChar	*blkGetPathCmd;
	FILE	*fpblkGetPathCmd = NULL;
	tBool	ret = TRUE;
	tU32 len;

	if ((NULL == ContainerName) || (NULL == blkioContainerpath)) {
		return FALSE;
	}
	len = strlen(BLK_DEV_THROTTLING_PATH) + strlen(ContainerName) + SHELL_CMD_BYTES;

	blkGetPathCmd = (tChar *)eMMC_ExcessiveWP_AllocateMemory(len);
	if (blkGetPathCmd == NULL)	{
		ERROR( "!!ERROR: Mem fail Size <%d>\n",(tS32)( len));
		ret = FALSE;
		goto end;
	}

	snprintf(blkGetPathCmd,len,BLK_DEV_THROTTLING_PATH,ContainerName);

	fpblkGetPathCmd = popen( blkGetPathCmd, MODE_RO );
	if (fpblkGetPathCmd == NULL) {
		ERROR( "!!ERROR: File open error %s\n", blkGetPathCmd);
		ret = FALSE;
		goto end;
	}

	if (fscanf(fpblkGetPathCmd,"%128s",blkioContainerpath) <= 0)	{
		ERROR( "!!ERROR: File Error %s\n", blkGetPathCmd);
		ret = FALSE;
		goto end;
	}

end:
	if (fpblkGetPathCmd != NULL )
		pclose(fpblkGetPathCmd);
	eMMC_ExcessiveWP_DeallocateMemory(blkGetPathCmd);
	return ret;
}

/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_GetDeviceNode
* PARAMETER 	  : DevMajorNo	: to hold major no of eMMC
*					DevMinorNo	: to hold minor no of eMMC
* RETURNVALUE	  : TRUE : Success ; FALSE : Failure ;
* DESCRIPTION	  : Read the /proc/diskstats file and find the device node of the
*					eMMC device
**********************************************************************************/
tS32 eMMC_ExcessiveWP_GetDeviceNode (tChar *location, tS32 *DevMajorNo, tS32 *DevMinorNo )
{
    tS32    s32retVal ,	s32DevNo;
    tChar    MmcCard [NODE_LENGTH]  = {0},
            MmcHost [NODE_LENGTH]  = {0};
	tChar	cblkGetDevNode[MAXLENGTH]= {0};
	struct stat stats;
	/* Invalid argument will lead to PATH realization error */
	if ((NULL == location) || (NULL == DevMajorNo) || (NULL == DevMinorNo)) {
		return REG_PATH_UNAVAIL;
	}

    s32retVal = eMMC_ExcessiveWP_GetmmcHostCardPath (MmcCard,MmcHost);
    if ( s32retVal != 0 ) {
        ERROR("!!ERROR:Find Card");
        s32retVal = MMC_CARD_INFO_ACCESS_ERROR;
		return s32retVal;
    }

	s32DevNo = MmcHost[strlen(MmcHost)-1]-'0';
	snprintf(cblkGetDevNode,MAXLENGTH-1,ROOT_DEVICE_NODE,s32DevNo);

	DEBUG("<>MMC Card <%s> %c %d<%s>\n",MmcHost,MmcHost[strlen(MmcHost)-1],s32DevNo,cblkGetDevNode);

	memset(&stats,0,sizeof(struct stat));
	if (stat(location, &stats) == 0)
	{
		*DevMajorNo = major(stats.st_rdev);
		*DevMinorNo = minor(stats.st_rdev);
	}
	else
	{
		*DevMajorNo = -1;
		*DevMinorNo = -1;
	}
	if ((*DevMajorNo <= 0) || (*DevMajorNo > 255))	{
		ERROR( "!!ERROR:Device DevMajorNo not found \n");
		s32retVal = REG_PATH_UNAVAIL;
		return s32retVal;
	}
	else {
		s32retVal = 0;
	}
	DEBUG("Root Device no: Major<%d> Minor<%d> Ret<%d>\n",*DevMajorNo,*DevMinorNo,s32retVal);
	DEBUG("##MMC Card <%s> %c %d<%d> Root Device no: Major<%d> Minor<%d>\n",MmcHost,MmcCard[strlen(MmcHost)-1],MmcCard[strlen(MmcHost)-1]-'0',s32retVal,*DevMajorNo,*DevMinorNo);

	return s32retVal;
}

/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_UpdateDB
* PARAMETER 	  : blkioLocationInfo	: 	Location details which read from cfg file
*					PresistentFileName	:	Data base file name
* RETURNVALUE	  : -
* DESCRIPTION	  : This function write the entries to the data base file.
**********************************************************************************/
void eMMC_ExcessiveWP_UpdateDB ( tChar *PresistentFileName, monitor_blkio *blkioLocationInfo)
{
	FILE			*fpPersistentfile;
	node 			*RingBuffer,
					*NodeTmp;

	eMMC_ExcessiveWP_UpdateTotalBlockCount(blkioLocationInfo->location,blkioLocationInfo->CumulativeBlkCount);

	fpPersistentfile = fopen(PresistentFileName,MODE_W);
	if( fpPersistentfile == NULL )
	{
		ERROR("!!ERROR:File opern error : %s\n",PresistentFileName);
	}
	else {
		DEBUG("Flush Presistent File \n");
		NodeTmp = blkioLocationInfo->ringnode;
		RingBuffer= blkioLocationInfo->ringnode;
		do
		{
			eMMC_ExcessiveWP_UpdatePresistentFile(fpPersistentfile,NodeTmp);
			NodeTmp=NodeTmp->next;
		}while ( NodeTmp != RingBuffer);

		#ifdef DUMP_TRACE_INFO
			TRACE_SYNC(fpTraceFile);
		#endif /* DUMP_TRACE_INFO */
		eMMC_ExcessiveWP_SyncData(fpPersistentfile);
		fclose (fpPersistentfile);
	}
	return ;
}

/********************************************************************************
* FUNCTION		  : is_same_date
* PARAMETER 	  : date1	: 	input date1 in struct tm format
*					date2	:	input date2 in struct tm format
* RETURNVALUE	  : TRUE/FALSE
* DESCRIPTION	  : This function returns TRUE if both date1 and date2 are
*    				same date otherwise returns FALSE.
**********************************************************************************/
tBool is_same_date(struct tm date1,struct tm date2)
{
	if (((date1.tm_year)==(date2.tm_year))&&((date1.tm_mon)==(date2.tm_mon))&&((date1.tm_mday)==(date2.tm_mday))) {
		return 1;
	}
	return 0;
}

/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_FlushBlkCount2PersistentLocation
* PARAMETER 	  : PresistentFileName	: Presistent file location name
*					BlkCount			: Cumulative block count for single power cycle
*					blkioLocationInfo	: Location details which read from cfg file
*					WriteDealy			: interval to update the persistent file.
* RETURNVALUE	  : TRUE : Success ; FALSE : Failure ;
* DESCRIPTION	  : Always check the current date with date entry available on
*					last ring node and find the either this call is new power
*					cycle on same day or another day power ON. Also this function
*					will check the days incremented in consecutively and handle the
*					date changes in forward and rewind.
*					persistent file was updates for every PROFILER_NEW_POWERCYCLE interval
**********************************************************************************/
tBool eMMC_ExcessiveWP_FlushBlkCount2PersistentLocation ( tChar *PresistentFileName, unsigned long long BlkCount,monitor_blkio *blkioLocationInfo,int WriteDelay)
{
	node 			*LastEntry;
	struct tm new_date = {0} ;
	struct tm 		PreviousTime = { 0 };
	tS32 			DayDiff;
	tChar			name [CFG_NAME_LEN] = {0};
	long long 		LastTotalWriteCyclesCount;

	/* Parameter check*/
	if ((NULL == PresistentFileName) || (NULL == blkioLocationInfo)) {
		return FALSE;
	}
	if (NULL == blkioLocationInfo->ringnode) {
		return FALSE;
	}
	if (NULL == blkioLocationInfo->ringnode->prev) {
		return FALSE;
	}

	eMMC_ExcessiveWP_ObtainPresentDate(&new_date);

	LastEntry = blkioLocationInfo->ringnode->prev;

	pthread_getname_np(blkioLocationInfo->LocationThread, name, sizeof(name));

	DEBUG("(%u-%s)Flush File Name : %s Current Date <%d:%d:%d> Last Date <%d:%d:%d>\n",WriteDelay,name,PresistentFileName,new_date.tm_mday,new_date.tm_mon,new_date.tm_year,LastEntry->date.tm_mday,LastEntry->date.tm_mon,LastEntry->date.tm_year);

	if(is_same_date(LastEntry->date,new_date))
	{
		/*Handling : During startup the day is one day behind the prevalued as per the Data base. Then the day is shift to current day.
		 *File: blkcount_eMMC.txt prevalue (at 9.07.2017):
		 *	Date:09/072017 Blk_count 900 TotalWriteBytes 0
		 * // after 2 minutes store the first entry
		 * Date:09/07/2017 Blk_count 900 TotalWriteBytes 0
		 * Date:10/07/2017 Blk_count 0 TotalWriteBytes 900    // normal behavoir for next day
		 * Target run for more then 24Hrs and reached the current date(As per DB). Then handle the blk count for latest day in DB .
		 */
		if ( blkioLocationInfo->PowerCycleStats == PROFILER_NEW_POWERCYCLE_DAY_BACKWARD ) {
			blkioLocationInfo->blockcnt = blkioLocationInfo->blockcnt + (LastEntry->prev->blk_count - blkioLocationInfo->PrevBlkcount);
			blkioLocationInfo->PrevBlkcount = LastEntry->blk_count;
		}
		/*Handling: Muliple power cycle on same day. This hanling taken care of blk count calculation for whole day*/
		if( blkioLocationInfo->PowerCycleStats == PROFILER_NEW_STARTUP ) {
			blkioLocationInfo->PrevBlkcount = LastEntry->blk_count;
		}
		DEBUG("<%d><%s>Power Cycle on Same date %2d/%2d/%4d BlkCnt %llu Total Count %llu P{%lld} %llu",
			WriteDelay,
			blkioLocationInfo->location,
			new_date.tm_mday,new_date.tm_mon,new_date.tm_year,
			BlkCount,
			LastEntry->total_write_cycles_count,
			blkioLocationInfo->PrevBlkcount,
			blkioLocationInfo->blockcnt);
		blkioLocationInfo->PowerCycleStats = PROFILER_NEW_POWERCYCLE_IN_SAME_DAY;
		LastEntry->blk_count = BlkCount + blkioLocationInfo->PrevBlkcount - blkioLocationInfo->blockcnt;
		blkioLocationInfo->CumulativeBlkCount = LastEntry->blk_count + LastEntry->total_write_cycles_count;

		DEBUG("CumulativeCnt: %llu\n",blkioLocationInfo->CumulativeBlkCount);
	}
	else
	{
		DEBUG("New Date Current<%2d/%2d/%4d> Previous<%2d/%2d/%4d>\n",new_date.tm_mday,new_date.tm_mon,new_date.tm_year,LastEntry->date.tm_mday,LastEntry->date.tm_mon,LastEntry->date.tm_year);
		DayDiff=  eMMC_ExcessiveWP_CompareDates(LastEntry->date,new_date);

		/* If the day increased by one day. add New entry and calculate the total blk count*/
		if( DayDiff == DAY_DIFF_ONE_DAY_FORWARD )	{
			if( blkioLocationInfo->PowerCycleStats == PROFILER_NEW_POWERCYCLE_IN_SAME_DAY )	{
				blkioLocationInfo->PrevBlkcount = -(LastEntry->blk_count - blkioLocationInfo->PrevBlkcount);
				BlkCount += blkioLocationInfo->PrevBlkcount;
			}
			else {
				blkioLocationInfo->PowerCycleStats = PROFILER_NEW_POWERCYCLE;
			}
			eMMC_ExcessiveWP_InsertNode(blkioLocationInfo,BlkCount,new_date);
		}
/* As per the Flemming Reiner (CM-CI1/EPC3-E) suggestion, we have consider the 1 day back
 * on same power cycle and add new entry for that day*/
#if 0
		/*If the day Go back on same power cycle, no new entry on the db, instead of that
		 * Update blk count to current day
		 */
		else if(( DayDiff < 0  ) &&
				( blkioLocationInfo->PowerCycleStats == PROFILER_NEW_POWERCYCLE_IN_SAME_DAY ))
		{
			ERROR("WARN!!::Day diff same cycle Current<%s> Previous<%s>\n",NewNode.date,LastEntry->date);
			LastEntry->blk_count = BlkCount + blkioLocationInfo->PrevBlkcount;
			blkioLocationInfo->CumulativeBlkCount = LastEntry->blk_count + LastEntry->total_write_cycles_count;
		}
#endif
		/* If day go back by one day*/
		else if ( DayDiff == DAY_DIFF_ONE_DAY_BACKWARD  ) {
			DEBUG("WARN!!::One Day(-)diff <%2d/%2d/%4d> Previous<%2d/%2d/%4d>\n",new_date.tm_mday,new_date.tm_mon,new_date.tm_year,LastEntry->date.tm_mday,LastEntry->date.tm_mon,LastEntry->date.tm_year);
			eMMC_ExcessiveWrite_PopulateCriticalNodes(blkioLocationInfo,BlkCount,new_date);
		}
		/*Day Go back more then one day or the difference is more then the maximum window size have to remove the
		 *DB and recreated it. But we maintiain the total wirte byte count*/
		else if(( DayDiff < DAY_DIFF_ONE_DAY_BACKWARD  ) ||
				( DayDiff > blkioLocationInfo->MaxDuration))
		{
			ERROR("WARN!!::New Date Current<%2d/%2d/%4d> Previous<%2d/%2d/%4d>\n",new_date.tm_mday,new_date.tm_mon,new_date.tm_year,LastEntry->date.tm_mday,LastEntry->date.tm_mon,LastEntry->date.tm_year)
			LastTotalWriteCyclesCount = LastEntry->total_write_cycles_count + LastEntry->blk_count;
			eMMC_ExcessiveWP_DeleteRingBuffer(blkioLocationInfo);
			if ( remove (PresistentFileName) != 0 )
			{
				DEBUG("!!ERROR: File (%s) Remove Fails \n",PresistentFileName);
			}
			if( TRUE == eMMC_ExcessiveWP_InsertNode(blkioLocationInfo,BlkCount,new_date) )
				blkioLocationInfo->ringnode->total_write_cycles_count = LastTotalWriteCyclesCount;
			blkioLocationInfo->PrevBlkcount = blkioLocationInfo->blockcnt = 0;
			blkioLocationInfo->PowerCycleStats = PROFILER_NEW_STARTUP;
			blkioLocationInfo->CumulativeBlkCount = blkioLocationInfo->ringnode->blk_count + \
													blkioLocationInfo->ringnode->total_write_cycles_count;
			WriteDelay = blkioLocationInfo->DataWriteInterval;
		}
		/*If the day go forward more then one day . Added the respective entries to the DB fuukes*/
		else if( DayDiff > 1 )	{
			ERROR("WARN!!::New Date Current<%2d/%2d/%4d> Previous<%2d/%2d/%4d>\n",new_date.tm_mday,new_date.tm_mon,new_date.tm_year,LastEntry->date.tm_mday,LastEntry->date.tm_mon,LastEntry->date.tm_year);
			PreviousTime = LastEntry->date;

			/*Command need to add ??? */
			if ( blkioLocationInfo->PowerCycleStats == PROFILER_NEW_POWERCYCLE_DAY_BACKWARD ) {
				DayDiff = DayDiff + DAY_DIFF_ONE_DAY_BACKWARD;
			}
			while(--DayDiff) {
				PreviousTime.tm_mday++;
			//	(void)mktime(&PreviousTime);
				new_date = PreviousTime;
				DEBUG("previous_date:%2d/%2d/%4d\n",new_date.tm_mday,new_date.tm_mon,new_date.tm_year);
				eMMC_ExcessiveWP_InsertNode(blkioLocationInfo,0,new_date);
			}
			WriteDelay = blkioLocationInfo->DataWriteInterval;
		}
 	}

	if ((blkioLocationInfo->DataWriteInterval) &&
		(!(WriteDelay%(blkioLocationInfo->DataWriteInterval))))
	{
		eMMC_ExcessiveWP_UpdateDB(PresistentFileName,blkioLocationInfo);
	}
	return TRUE;
}

/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_DetectMonitorLocation
* PARAMETER 	  : blkioLocationInfo	: colation info of monitor blkio
* RETURNVALUE	  : valid monitor type if success
					unknown monitor type if error
* DESCRIPTION	  : Get the path from the cgroup for monitor the container IO details.
**********************************************************************************/
MonitorType eMMC_ExcessiveWP_DetectMonitorLocation(monitor_blkio 	*blkioLocationInfo)
{
	MonitorType		LocationType = MONITOR_UNKNOWN;

	if (NULL == blkioLocationInfo) {
		return MONITOR_UNKNOWN;
	}

	DEBUG("\n %s %d database_location=%s LOCATION_BLK_DEV_NAME=%s\n",__FUNCTION__,__LINE__,blkioLocationInfo->location,LOCATION_BLK_DEV_NAME);

	/*Detect the the Monitor location is the container or Native system or eMMC partitions*/
	if( ! strncmp (blkioLocationInfo->location,LOCATION_BLK_DEV_NAME,strlen(LOCATION_BLK_DEV_NAME))) {
		if( access(blkioLocationInfo->location , 0 ) != -1)	{
			if( strlen (LOCATION_PARTITION ) == strlen(blkioLocationInfo->location)) {
				LocationType = MONITOR_BLK_PARTITION;
			}
			else {
				LocationType = MONITOR_BLK_DEVICE;
			}
		}
		else {
			ERROR( "!!ERROR: Invalid Location <%s><%s>\n",blkioLocationInfo->location,blkioLocationInfo->unique_name);
		}
	}
	else {
		LocationType = MONITOR_CONTAINER;
	}
	return LocationType;
}

/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_MonitorThread
* PARAMETER 	  : LocationInfo	: Location details which read from cfg file
* RETURNVALUE	  :
* DESCRIPTION	  : Monitor thread will check the block count for every second and
*					check the possibility for trigger event. Also update the persistent
*					file for every certain interval.
**********************************************************************************/
tPVoid	eMMC_ExcessiveWP_MonitorThread ( tPVoid LocationInfo )
{
	FILE			*fpPersistent = NULL;
	FILE *fpBlkCountFile = NULL;
	tChar			*PersistentFileName = NULL ,*BlkCountFileName = NULL;
	MonitorType		LocationType;
	monitor_blkio 	*blkioLocationInfo = (monitor_blkio *)LocationInfo;
	unsigned long long  PersistentFileWriteDealy = 0;
	tS32				DevMajorNo = 0,
					DevMinorNo = 0,	MultiplsFactor = 1;
	tChar			*ContainerName		= NULL,
					*blkioContainerpath = NULL;
	tS32 ret;
	tU32 len;
	tChar DevNameuf[CFG_NAME_LEN];
	tS32 tmp[12];
	/* remember..timeout must be in ms*/
	tS32 timeout;
	unsigned long long blkcount;
	unsigned long long blkcount_prev = 0;

	if ( blkioLocationInfo == NULL )
	{
		ERROR("!!ERROR: Failed to Start Monitor Thread");
#ifdef DUMP_TRACE_INFO
		TRACE_SYNC(fpTraceFile);
#endif /* DUMP_TRACE_INFO */
		return NULL;
	}

	pthread_setname_np(blkioLocationInfo->LocationThread, blkioLocationInfo->unique_name);

	DEBUG("Profiler{[%s] Thread Started \n",blkioLocationInfo->unique_name);

	/*Created semaphore for event communication between Watch thread and Monitor thread */
	if (createEvent(&blkioLocationInfo->eventObj) == -1) {
		ERROR("!!ERROR:Sem-eventObj Create Failed.. \n");
		goto end;
	}

	/*Created semaphore for ACK_event communication between Watch thread and Monitor thread */
	if (createEvent(&blkioLocationInfo->AckeventObj) == -1) {
		ERROR("!!ERROR:AckeventObj Create Failed.. \n");
		goto end;
	}


	/*Detect the the Monitor location is the container or Native system or eMMC partitions*/
	LocationType = eMMC_ExcessiveWP_DetectMonitorLocation(blkioLocationInfo);
	if ( LocationType == MONITOR_UNKNOWN )
	{
		ERROR("!!ERROR: Failed to eMMC_ExcessiveWP_DetectMonitorLocation");
		goto end;
	}
	else if ((LocationType == MONITOR_BLK_PARTITION) || (LocationType == MONITOR_BLK_DEVICE))	{
		/*eMMC Partition is monitor from the /proc/diskstats and it have the write count
		 * in terms of 512 bytes.
		 */
		MultiplsFactor = 512;
	}

	DEBUG ( "LOCATION Name : %s<%s> <%d>\n",blkioLocationInfo->unique_name,blkioLocationInfo->location,LocationType);

	len = strlen(BLK_COUNT_PRESISTENT_FILE) + strlen(blkioLocationInfo->unique_name) + 1;
	/*Each location we have created the individual db file to store the write byte count details*/
	PersistentFileName = (tChar *)eMMC_ExcessiveWP_AllocateMemory(len );
	if( PersistentFileName == NULL ) {
		ERROR( "!!ERROR: Mem fail Size <%u> <T:%s>\n",len,blkioLocationInfo->unique_name);
		goto end;
	}
	/* Check the Data base file and create the ring buffer for Data handling.
	 * If the file is not available, then recreate the file
	 */
	snprintf(PersistentFileName,len,BLK_COUNT_PRESISTENT_FILE,blkioLocationInfo->unique_name);
	DEBUG("Presistent File: %s\n",PersistentFileName);
	fpPersistent = fopen(PersistentFileName,MODE_RO);
	if ( fpPersistent == NULL )	{
		DEBUG("Create Persistent file <%s>\n",PersistentFileName);
		fpPersistent = fopen(PersistentFileName,MODE_W);
		if ( fpPersistent == NULL )	{
			ERROR( "!!ERROR: open error file <%s><T:%s>\n",PersistentFileName,blkioLocationInfo->unique_name);
			goto end;
		}

		blkioLocationInfo->ringnode = eMMC_ExcessiveWP_CreateNode();
		if( blkioLocationInfo->ringnode == NULL ) {
			goto end;
		}
		blkioLocationInfo->ringnode->prev = blkioLocationInfo->ringnode;
		blkioLocationInfo->ringnode->next = blkioLocationInfo->ringnode;

		eMMC_ExcessiveWP_ObtainPresentDate(&blkioLocationInfo->ringnode->date);
		DEBUG( "Present Date : <%2d/%2d/%4d> \n",blkioLocationInfo->ringnode->date.tm_mday,blkioLocationInfo->ringnode->date.tm_mon,blkioLocationInfo->ringnode->date.tm_year);
		blkioLocationInfo->ringnode->total_write_cycles_count = blkioLocationInfo->CumulativeBlkCount;
		eMMC_ExcessiveWP_UpdatePresistentFile(fpPersistent,blkioLocationInfo->ringnode);
		eMMC_ExcessiveWP_SyncData(fpPersistent);
	}
	else {
		DEBUG("Fopen Success <%s>\n",PersistentFileName);
		if( eMMC_ExcessiveWP_PopulateNodes(fpPersistent,blkioLocationInfo) != TRUE ) {
			ERROR("!!ERROR:Populate Node Failed<T:%s>\n",blkioLocationInfo->unique_name);
			goto end;
		}
	}

	if(fpPersistent != NULL)
	{
		fclose(fpPersistent);
		fpPersistent = NULL;
	}
	/*Read the Major and minor node of the eMMC device and partition.*/
	if ( eMMC_ExcessiveWP_GetDeviceNode(blkioLocationInfo->location,&DevMajorNo,&DevMinorNo) != 0 )
	{
		ERROR( "!!ERROR:Device not found \n");
		goto end;
	}
	/*Based on the device to monitor(either Container or Native or eMMC partition) and bytes count read location will changed*/
	switch ( LocationType )
	{
		case MONITOR_CONTAINER:
		{
			ContainerName		= (tChar *)eMMC_ExcessiveWP_AllocateMemory(strlen(blkioLocationInfo->location)),
			blkioContainerpath 	= (tChar *)eMMC_ExcessiveWP_AllocateMemory(BLKIO_LXC_PATH);

			if(( ContainerName == NULL ) || (blkioContainerpath == NULL) )
			{
				ERROR( "!!ERROR: Mem fail Size <%d,%d>\n",(tS32)strlen(blkioLocationInfo->location),(tS32) BLKIO_LXC_PATH);
				goto end;
			}

			if( TRUE != eMMC_ExcessiveWP_GetContainerName(blkioLocationInfo->location,ContainerName))
			{
				ERROR( "!!WARN:Container name not as expected <%s>.. \n",ContainerName);
			}

			if( TRUE != eMMC_ExcessiveWriteGetContainerPath(ContainerName,blkioContainerpath))
			{
				ERROR( "!!ERROR: Failed to get Contaianer path Con:%s \n",ContainerName);
				goto end;
			}

			len = strlen(BLKIO_PATH_CONTAINER) + strlen(blkioContainerpath) + strlen(blkioLocationInfo->location) + SHELL_CMD_BYTES;

			/*To monitor the container: Read the Bytes counts from the Cgroups */
			BlkCountFileName = (tChar *)eMMC_ExcessiveWP_AllocateMemory(len);
			if( BlkCountFileName == NULL )
			{
				ERROR( "!!ERROR: Mem fail Size <%u>\n",len);
				goto end;
			}
			snprintf(BlkCountFileName,len,"grep '%d:%d Write' "BLKIO_PATH_CONTAINER"|awk '{print $3}'",DevMajorNo,DevMinorNo,blkioContainerpath);
		}
			break;
		case MONITOR_BLK_PARTITION :
		case MONITOR_BLK_DEVICE:
		{
			len = strlen(DISKSTATS_CMD_FILE) + 1 ;
			/*To monitor the eMMC Partitions: Read the Bytes counts from the /proc/diskstats */
			BlkCountFileName = (tChar *)eMMC_ExcessiveWP_AllocateMemory(len);
			if( BlkCountFileName == NULL )
			{
				ERROR( "!!ERROR: Mem fail Size <%u>\n",len);
				goto end;
			}
			snprintf(BlkCountFileName,len,DISKSTATS_CMD_FILE);
			BlkCountFileName[len-1] = 0;
		}
			break;
		default:
			ERROR("!!ERRROR:Unsupported Device{%s} to monitor\n",blkioLocationInfo->location);
			goto end;
	}

	/*5000 ms*/
	timeout = BLKCNT_UPDATE_FREQUENCY_IN_MS;
	/* All the operations in the loop are performed once in every 5000 ms */
	while(1)
	{
		ret = waitEvent(&blkioLocationInfo->eventObj,timeout);
		if (ret < 0) {
			ERROR("!!ERRROR:waitforEvent from Watchthread failed\n");
			goto end;
		}
		else if(ret == 0)	{
			/* Event received for SAVE_Persistency */
			if (blkioLocationInfo->evt_state == EVT_LCM_SAVE_PERSISTENCY)
			{
				#ifdef DUMP_TRACE_INFO
					TRACE_CLOSE(fpTraceFile);
				#endif /* DUMP_TRACE_INFO */

				eMMC_ExcessiveWP_UpdateDB(PersistentFileName,blkioLocationInfo);
				timeout = -1;
			}
			else if (blkioLocationInfo->evt_state == EVT_LCM_LOAD_PERSISTENCY) {
				/*
					Event received for Load_Persistency but nothing much can be done
					except setting the timeout again to 5000ms and reopening the trace device
					Just post the Ack and continue the working
				*/
				timeout = BLKCNT_UPDATE_FREQUENCY_IN_MS;

				#ifdef DUMP_TRACE_INFO
					TRACE_CLOSE(fpTraceFile);
				#endif /* DUMP_TRACE_INFO */
			}

			ret = postEvent(&blkioLocationInfo->AckeventObj);
			if (ret < 0)
			{
				ERROR("!!ERRROR:waitforEvent from Watchthread failed\n");
				goto end;
			}
		}
		else
		{
			/*
				Timeout. This means that no-event received so continue with normal
				operation i.e retrieve the profiling data
			*/
			/* Dont do any operation if monitor thread is still in SAVE_Persistency mode*/
			if (blkioLocationInfo->evt_state != EVT_LCM_SAVE_PERSISTENCY) {

				fpBlkCountFile = fopen( BlkCountFileName, MODE_RO );
				if( fpBlkCountFile == NULL ) {
					ERROR( "!!ERROR: File open error %s\n", BlkCountFileName);
					goto end;
				}

				/*tmp is dummy...just ignore the contents of tmp . Its used inorder to fullfill the pattern based read */

				do
				{
					ret = fscanf(fpBlkCountFile,DISKSTATS_READ_PATTERN,&tmp[0],&tmp[1],DevNameuf,&tmp[2],&tmp[3],&tmp[4],&tmp[5],&tmp[6],&tmp[7],&blkcount,&tmp[8],&tmp[9],&tmp[10],&tmp[11]);
					if (strcmp(DevNameuf,basename(blkioLocationInfo->location)) == 0)
					{
						DEBUG("Block count is %llu\n", blkcount);
						break;
					}
				} while ( NUMBER_OF_FIELDS_IN_DISKSTATS == ret); /*on successful read, return value of fscanf is 14, as 14 paramters are read */

				fclose(fpBlkCountFile);
				DEBUG("Container Blk Count1 : %llu blkcount_prev=%llu MultiplsFactor=%d\n",blkcount,blkcount_prev,MultiplsFactor);
				/* increment the PersistentFileWriteDealy based on timeout */
				PersistentFileWriteDealy = PersistentFileWriteDealy + timeout/COVERT_FROM_MILLISEC_TO_SEC;
				if ((blkcount > blkcount_prev)|| ((blkioLocationInfo->DataWriteInterval) &&
		(!(PersistentFileWriteDealy%(blkioLocationInfo->DataWriteInterval))))){
					DEBUG("Container Blk Count2 : %llu blkcount_prev=%llu MultiplsFactor=%d\n",blkcount,blkcount_prev,MultiplsFactor);
				    blkcount_prev = blkcount;
					blkcount = blkcount * MultiplsFactor;
					if( eMMC_ExcessiveWP_FlushBlkCount2PersistentLocation(PersistentFileName,blkcount ,blkioLocationInfo,PersistentFileWriteDealy) != TRUE )
					{
						DEBUG( "!!ERROR: File Update Fails <%s>\n",PersistentFileName);
					}
					eMMC_ExcessiveWP_CheckWatchWindow(blkioLocationInfo);
				}
			}
		}
	}

end:
	if(fpPersistent != NULL) {
		fclose(fpPersistent);
		fpPersistent = NULL;
	}

	eMMC_ExcessiveWP_DeallocateMemory(ContainerName);
	eMMC_ExcessiveWP_DeallocateMemory(blkioContainerpath);
	eMMC_ExcessiveWP_DeallocateMemory(BlkCountFileName);
	eMMC_ExcessiveWP_DeallocateMemory(PersistentFileName);
	blkioLocationInfo->MonitorThrdState = PROFILER_THREAD_STOPPED;
	DEBUG("{%s,%s} Thread Exited \n",blkioLocationInfo->location,blkioLocationInfo->unique_name);

	pthread_exit(NULL);
}

