/*****************************************************************************************
* Copyright (C) RBEI, 2019
* This software is property of Robert Bosch.
* Unauthorized duplication and disclosure to third parties is prohibited.
******************************************************************************************/
/*******************************************************************************
*
* FILE			:	writeprofiler_dbus.c
*
* SW-COMPONENT	:	eMMC Write profiler
*
* PROJECT		:	This will be common for all the projects.
*
* DESCRIPTION	: This contains the implementation to offer the Dbus
* 					functionalities for write profiler the

* AUTHOR		:	Selvakumar Kalimuthu (RBEI/ECF2)
					Ravindra Prabha (RBEI/ECF1)
*
* COPYRIGHT:    (c) 2019 Robert Bosch GmbH, Hildesheim
*
* HISTORY      :
*------------------------------------------------------------------------
* Date      |       Version       | Author & comments
*-----------|---------------------|--------------------------------------
*-----------|---------------------|--------------------------------------
*27.Sep.2019| Initial version 1.0 | Ravindra Prabha (RBEI/ECF1)
* -----------------------------------------------------------------------
* -----------------------------------------------------------------------

/******************************************************************************/
/* INCLUDES                                                                   */
/******************************************************************************/
#include "writeprofiler.h"
/******************************************************************************/
/********************************************************************************
* FUNCTION        : eMMC_ExcessiveWP_DbusBusAcquiredHandler
* PARAMETER       :	connection 	: D-Bus Handle
*					name		: D-Bus Name
*					user_data	: user date
* RETURNVALUE	  : tVoid
* DESCRIPTION	  : This the callback function and called When the dbus is acquired
*					by server.
**********************************************************************************/
tVoid  eMMC_ExcessiveWP_DbusBusAcquiredHandler (GDBusConnection *connection, const gchar *name, gpointer user_data)
{
	DBusInfo		*trDbusInfo = (DBusInfo *)user_data;

	if ((NULL == trDbusInfo) || (NULL == name)) {
		ERROR("!!Invalid argument in function %s \n",__FUNCTION__);
		return;
	}
	DEBUG("[dbus server] eMMC_ExcessiveWP_DbusBusAcquiredHandler %s %s\n",name,trDbusInfo->DBusName);
	/*Connect the D-bus server*/
	if(NULL == trDbusInfo->g_proxy)
	{
		trDbusInfo->g_proxy = org_monitor_excessive_write_emmc_skeleton_new(); // server side dbus object
		if(NULL == trDbusInfo->g_proxy)
		{
		 	DEBUG("[dbus server error] Error, NULL g_proxy\n");
		}
		else
		{
			if(!g_dbus_interface_skeleton_export(G_DBUS_INTERFACE_SKELETON(trDbusInfo->g_proxy), connection, DBUS_OBJECT_PATH, NULL))
			{
				DEBUG("[dbus server error] server export failed\n");
			}
			else
			{
				DEBUG("[dbus server] BUS ACQUIRED %X\n",trDbusInfo->g_proxy);
				trDbusInfo->IsBusAcquired = TRUE;
			}
		}
	}
	else
	{
		DEBUG("[dbus server] Alread bus acquired %s %s\n",name,trDbusInfo->DBusName);
	}
}
/********************************************************************************
* FUNCTION        : eMMC_ExcessiveWP_DbusNameAcquiredHandler
* PARAMETER       :	connection 	: D-Bus Handle
*					name		: D-Bus Name
*					user_data	: user date
* RETURNVALUE	  : tVoid
* DESCRIPTION	  : Once the dbus cretaed we can match the d-bus name with our
*					d-bus name and ensure the respective dbus the created by us or not?
**********************************************************************************/
tVoid eMMC_ExcessiveWP_DbusNameAcquiredHandler(GDBusConnection *connection, const gchar *name, gpointer user_data)
{
	DBusInfo		*trDbusInfo = (DBusInfo *)user_data;
	DEBUG("[dbus server] eMMC_ExcessiveWP_DbusNameAcquiredHandler %s %s \n",name,trDbusInfo->DBusName);
}
/********************************************************************************
* FUNCTION        : eMMC_ExcessiveWP_DbusNameLostHandler
* PARAMETER       :	connection 	: D-Bus Handle
*					name		: D-Bus Name
*					user_data	: user date
* RETURNVALUE	  : tVoid
* DESCRIPTION	  : This the callback function and called during the bus is disconnected
**********************************************************************************/
tVoid eMMC_ExcessiveWP_DbusNameLostHandler(GDBusConnection *connection, const gchar *name, gpointer user_data)
{
	DBusInfo		*trDbusInfo = (DBusInfo *)user_data;
	DEBUG("[dbus server] eMMC_ExcessiveWP_DbusNameLostHandler %s  %s\n",name,trDbusInfo->DBusName);

	trDbusInfo->IsBusAcquired = FALSE;
}
/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWriteSendDbusNotification
* PARAMETER 	  : TriggerEvent		: Hold the details of excessive write
*					ExcessiveBytes		: Total count which exceeds the cfg threshold
*					blkioLocationInfo	: Location details which read from cfg file
* RETURNVALUE	  : TRUE : Success ; FALSE : Failure ;
* DESCRIPTION	  : Post the excessive write detection informations to D-bus client
**********************************************************************************/
tBool eMMC_ExcessiveWriteSendDbusNotification( write_profiler_params *TriggerEvent, long long ExcessiveBytes,monitor_blkio *blkioLocationInfo )
{
	GVariant 		*trigger = NULL;
	GVariantBuilder *Objbuilder=NULL;

	if( ( NULL == blkioLocationInfo->trDbusInfo->g_proxy ) ||
		( blkioLocationInfo->trDbusInfo->IsBusAcquired != TRUE ) )
	{
		ERROR("[dbus server] !!ERROR: DBus Yet Not Acquired %X{%d}\n",blkioLocationInfo->trDbusInfo->g_proxy,blkioLocationInfo->trDbusInfo->IsBusAcquired);
	}
	else
	{
		Objbuilder = g_variant_builder_new(G_VARIANT_TYPE("a{sv}"));

		if (Objbuilder != NULL)
		{
			DEBUG("DBus- Msg Send \n");
			g_variant_builder_add (Objbuilder, "{sv}", "unique_name", g_variant_new_string(blkioLocationInfo->unique_name));
			g_variant_builder_add (Objbuilder, "{sv}", "location",g_variant_new_string(blkioLocationInfo->location));
			g_variant_builder_add (Objbuilder, "{sv}", "name",g_variant_new_string(TriggerEvent->name));
			g_variant_builder_add (Objbuilder, "{sv}", "duration",g_variant_new_int32(TriggerEvent->duration));
			g_variant_builder_add (Objbuilder, "{sv}", "threshold",g_variant_new_int64((long)TriggerEvent->threshold));
			g_variant_builder_add (Objbuilder, "{sv}", "ExcessiveBytes",g_variant_new_int64((long)ExcessiveBytes));

			trigger = g_variant_builder_end(Objbuilder);
			g_variant_builder_clear(Objbuilder);
			org_monitor_excessive_write_emmc_emit_emmc_excessive_write_info( blkioLocationInfo->trDbusInfo->g_proxy,trigger);
			return TRUE;
		}
		else
		{
		   ERROR("[dbus server]!!ERROR: Falied to Send DBUS Msg \n");
		}
	}
	return FALSE;
}

/********************************************************************************
* FUNCTION        : eMMC_ExcessiveWrite_DbusThread
* PARAMETER       :	Arg - Excessive write info
* RETURNVALUE	  : tPVoid
* DESCRIPTION	  : This thread will create the D-Bus (Act as Server) and own the bus.
**********************************************************************************/
tPVoid	eMMC_ExcessiveWrite_DbusThread ( tPVoid Arg )
{
	DBusInfo		*trDbusInfo = (DBusInfo *)Arg;

	/*Created the D-Bus and wait on loop for client connect*/
	trDbusInfo->main_loop 	= g_main_loop_new(NULL, FALSE);
	trDbusInfo->watcher_id	= g_bus_own_name (	G_BUS_TYPE_SYSTEM,
												trDbusInfo->DBusName,
												G_BUS_NAME_OWNER_FLAGS_NONE,
												eMMC_ExcessiveWP_DbusBusAcquiredHandler,
												eMMC_ExcessiveWP_DbusNameAcquiredHandler,
												eMMC_ExcessiveWP_DbusNameLostHandler,
												(gpointer)trDbusInfo,
												NULL);

	DEBUG("watcher id: %d\n", trDbusInfo->watcher_id);
	g_main_loop_run(trDbusInfo->main_loop);
	DEBUG("after main_loop_run\n");
	return(NULL);

}

