/**********************************************************************************
* Copyright (C) RBEI, 2019
* This software is property of Robert Bosch.
* Unauthorized duplication and disclosure to third parties is prohibited.
**********************************************************************************/
/*********************************************************************************
*
* FILE			:	config.c
*
* SW-COMPONENT	:	eMMC Write profiler
*
* PROJECT		:	This will be common for all the projects.
*
* DESCRIPTION	: This contains the implementation to offer the parsing features
*  of the configuration file for te write profiler

* AUTHOR		:	Selvakumar Kalimuthu (RBEI/ECF2)
					Ravindra Prabha (RBEI/ECF1)
*
* COPYRIGHT:    (c) 2019 Robert Bosch GmbH, Hildesheim
*
* HISTORY      :
*------------------------------------------------------------------------
* Date      |       Version       | Author & comments
*-----------|---------------------|--------------------------------------
*-----------|---------------------|--------------------------------------
*07.Oct.2019| Initial version 1.0 | Ravindra Prabha ,refactored the write profiler
/******************************************************************************/
/* INCLUDES                                                                   */
/******************************************************************************/
#include "writeprofiler.h"
/******************************************************************************/

/********************************************************************************
* FUNCTION		  : eMMC_ExcessiveWP_ParseConfigFile
* PARAMETER 	  : stcfgInfo: to store all the config file info
* RETURNVALUE	  :
* DESCRIPTION	  : This function parse the config file using JSON format and store
*					all the location and trigger points to the configuration structure
**********************************************************************************/
tBool eMMC_ExcessiveWP_ParseConfigFile (monitor_global_info	 *stcfgInfo )
{
	FILE 	*fp;
	tChar 	*string;
	tS32 	filesize,
			itcount,
			buflen;
	struct json_object_iterator itrLocationStart,
								itLocationEnd,
								itTracePointStart,
								itTracePointEnd;
	if( stcfgInfo == NULL )
	{
		ERROR( "!!ERROR:Invalid Aragument in function %s\n",__FUNCTION__);
		return FALSE;
	}

  	fp = fopen( JSON_CONFIG_FILE, MODE_RO );
	if ( fp == NULL )
	{
		ERROR( "!!ERROR:File open Fail in function %s\n",__FUNCTION__);
		return FALSE;
	}

	/*Get the file size*/
	fseek(fp,0,SEEK_END);
	filesize  = (int)ftell(fp);
	if( filesize < 0 )
	{
		fclose(fp);
		DEBUG("!!ERROR:Corrupted Config Fail! <%s> in function %s\n",JSON_CONFIG_FILE,__FUNCTION__);
		return FALSE;
	}
	fseek(fp,0,SEEK_SET);

	string = (tChar *)eMMC_ExcessiveWP_AllocateMemory ( (tU32)filesize );
	if( string == NULL )
	{
		ERROR("Allocation Fails Size <%d>\n in function %s",filesize,__FUNCTION__);
		fclose (fp);
		return FALSE;
	}

	/*Read the configuration details from the config file*/
	itcount = fread(string ,1,filesize,fp);

	if( itcount  != filesize )
	{
		DEBUG("File Read mismatch ReadCnt<%d> Requested<%d> in function %s\n",itcount,filesize,__FUNCTION__);
	}
	fclose(fp);

	/*The configuration file is in JSON format. Parse the JSON details.*/
	struct json_object *jobj = json_tokener_parse(string);

	if(jobj == NULL )
	{
		ERROR("!!! jobj NULL in function %s\n",__FUNCTION__);
		eMMC_ExcessiveWP_DeallocateMemory(string);
		return FALSE;
	}

	/*Find the start and end objects(i.e Location to monitor) and traverse the location one by one.*/
	itrLocationStart = json_object_iter_begin(jobj);
	itLocationEnd = json_object_iter_end(jobj);

	/*Traverse all the objects in the JSON file and form the linked list to store all the available configuration details. */
	while (!json_object_iter_equal(&itrLocationStart, &itLocationEnd))
	{
		DEBUG("Object: %s :\n",json_object_iter_peek_name(&itrLocationStart));

		struct json_object *jobjpeek = json_object_iter_peek_value(&itrLocationStart);

		if(jobjpeek == NULL )
		{
			ERROR("!!! jobjpeek is NULL in function %s\n",__FUNCTION__);
			break;
		}

		if(!strcmp("PersistentDataLocation",json_object_iter_peek_name(&itrLocationStart)))
		{
			buflen = strlen(json_object_get_string(jobjpeek))+1;
			strncpy((tChar*)stcfgInfo->PersistentDataLocation,(const tChar*)json_object_get_string(jobjpeek),buflen );
			DEBUG("PersistentDataLocation: %s\n", stcfgInfo->PersistentDataLocation);
		}
		else if(!strcmp("PersistentDataSaveCycle",json_object_iter_peek_name(&itrLocationStart)))
		{
			stcfgInfo->PersistentDataSaveCycle = atoi(json_object_get_string(jobjpeek));
			DEBUG("PersistentDataSaveCycle: %d\n", stcfgInfo->PersistentDataSaveCycle);
		}
		else if(!strcmp("ErrmemNotification",json_object_iter_peek_name(&itrLocationStart)))
		{
			stcfgInfo->ErrmemNotification = atoi(json_object_get_string(jobjpeek));
			DEBUG("ErrmemNotification: %d\n", stcfgInfo->ErrmemNotification);
		}
		else
		{
			DEBUG("Iter: %s\n", json_object_get_string(jobjpeek));

			itTracePointStart 	= json_object_iter_begin(jobjpeek);
			itTracePointEnd		= json_object_iter_end(jobjpeek);

			monitor_blkio		*blkinfo,*blkTmp;

			blkinfo	= (monitor_blkio *) eMMC_ExcessiveWP_AllocateMemory (sizeof(monitor_blkio));

			if(blkinfo == NULL)
			{
				ERROR("Alloc Fails !!! Size <%d> in function %s\n",(tS32)sizeof(monitor_blkio),__FUNCTION__);
				eMMC_ExcessiveWP_DeallocateMemory(string);
				return FALSE;
			}
			blkinfo->monitor 	= 	NULL;
			blkinfo->next 		= 	NULL;
			blkinfo->ringnode	= 	NULL;
			itcount = 0;
			/*Traverse the nested objects and find the Trace event name, duration and threshold */
			while (!json_object_iter_equal(&itTracePointStart, &itTracePointEnd))
			{
				struct json_object 	*GetObject,
									*jobjTrace = json_object_iter_peek_value(&itTracePointStart);

				if(!strcmp("Name",json_object_iter_peek_name(&itTracePointStart)))
				{
					buflen = strlen(json_object_get_string(jobjTrace)) +1 ;
					strncpy(blkinfo->unique_name,json_object_get_string(jobjTrace),buflen);
					DEBUG( "Name : %s\n",blkinfo->unique_name);
				}
				else if (!strcmp("Location",json_object_iter_peek_name(&itTracePointStart)))
				{
					buflen = strlen(json_object_get_string(jobjTrace)) +1 ;
					strncpy(blkinfo->location,json_object_get_string(jobjTrace),buflen );
					DEBUG("Location : %s\n",blkinfo->location);
				}
				else
				{
					write_profiler_params	*profinfo,*proftmp;

					profinfo	= (write_profiler_params *) eMMC_ExcessiveWP_AllocateMemory (sizeof(write_profiler_params));

					if( profinfo == NULL )
					{
						ERROR("Alloc Fails !!! Size <%d> in function %\n",(tS32)sizeof(write_profiler_params),__FUNCTION__);
						eMMC_ExcessiveWP_DeallocateMemory(string);
						eMMC_ExcessiveWP_DeallocateMemory(blkinfo);
						return FALSE;
					}
					itcount++;
					DEBUG("Nested Object(%d): %s : %s \n",
										itcount,
										json_object_iter_peek_name(&itTracePointStart),
										json_object_get_string(jobjTrace));

					if( (TRUE == json_object_object_get_ex(jobjTrace, "Name", &GetObject) ) &&
						json_object_get_string(GetObject))
					{
						buflen = strlen(json_object_get_string(GetObject)) + 1;
						strncpy ( profinfo->name,json_object_get_string(GetObject),buflen);
						DEBUG("The value of Name is %s\n", profinfo->name);
					}

					if( (TRUE == json_object_object_get_ex(jobjTrace, "Duration", &GetObject) ) &&
							json_object_get_string(GetObject))
					{
						profinfo->duration = atoi(json_object_get_string(GetObject));
						DEBUG("The value of Duration is %d\n", profinfo->duration);
						if ( profinfo->duration > blkinfo->MaxDuration )
						{
							blkinfo->MaxDuration = profinfo->duration;
						}
					}

					if( (TRUE == json_object_object_get_ex(jobjTrace, "Threshold", &GetObject) ) &&
							json_object_get_string(GetObject))
					{
						profinfo->threshold	= atoi(json_object_get_string(GetObject));
						DEBUG("The value of Threshold is %lld\n", profinfo->threshold);
					}

					profinfo->next = NULL;

					blkinfo->TracePointCnt++;

					if( blkinfo->monitor == NULL )
					{
						blkinfo->monitor = profinfo;
					}
					else
					{
						proftmp = blkinfo->monitor;
						while ( proftmp->next != NULL )
						{
							proftmp = proftmp->next;
						}
						proftmp->next = profinfo;
					}
				}
				json_object_iter_next(&itTracePointStart);
			}

			stcfgInfo->MonitorLocationCnt ++;

			if( stcfgInfo->blkio	== NULL )
			{
				stcfgInfo->blkio = blkinfo;
			}
			else
			{
				blkTmp = stcfgInfo->blkio;
				while (blkTmp->next != NULL )
				{
					blkTmp = blkTmp->next;
				}
				blkTmp->next = blkinfo;
			}
		}
		json_object_iter_next(&itrLocationStart);
	}
	eMMC_ExcessiveWP_DeallocateMemory(string);
	return TRUE;
}

