#include "platform_app_manager.h"
#include "types.h"
#include "nvm_refresh.h"

#include <unistd.h>
#include <errno.h>
#include <stdio.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>
/********************************************************************************
 * FUNCTION        : addDevice
 * PARAMETER       : struct nvm_info -> Root node
 * 					 struct nvm_info -> Child to be added
 *
 * RETURNVALUE     : None
 *
 * DESCRIPTION     : This function adds the struture as linked list
 *
 *---------------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|-----------------------------------------------
 *14.August.2019| Initialversion 1.0  | Ragupathi Palanisamy (RBEI/ECF2)
 * --------------------------------------------------------------------------------
 **********************************************************************************/
void addDevice(struct nvm_info **root,struct nvm_info* child)
{
	struct nvm_info *temp = *root;
	if(temp == NULL)
	{
		*root = child;
	}
	else
	{
		while(temp->next != NULL)
		{
			temp = temp->next;
		}
		temp->next = child;
	}
}
/********************************************************************************
 * FUNCTION        : deleteDevice
 * PARAMETER       : struct nvm_info -> Root node *
 *
 * RETURNVALUE     : None
 *
 * DESCRIPTION     : This function deletes the linked list
 *
 *---------------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|-----------------------------------------------
 *14.August.2019| Initialversion 1.0  | Ragupathi Palanisamy (RBEI/ECF2)
 * --------------------------------------------------------------------------------
 **********************************************************************************/
void deleteDevice(struct nvm_info **root)
{
	struct nvm_info *current = *root;
	struct nvm_info *next;
	while(current !=NULL)
	{
		next = current->next;

		if(current->ps8DeviceName)
			free(current->ps8DeviceName);
		if(current->ps8ConfigFileDir)
			free(current->ps8ConfigFileDir);
		if(current->ps8MetaDataFileDir)
			free(current->ps8MetaDataFileDir);
		free(current);
		current = next;
	}
}

/********************************************************************************
 * FUNCTION        : de_alloc_nvm_memory
 * PARAMETER       : struct nvm_info *
 *
 * RETURNVALUE     : None
 *
 * DESCRIPTION     : This function deletes the memory allocated for nvm_info_instance
 *
 *---------------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|-----------------------------------------------
 *14.August.2019| Initialversion 1.0  | Ragupathi Palanisamy (RBEI/ECF2)
 * --------------------------------------------------------------------------------
 **********************************************************************************/

void de_alloc_nvm_memory(struct nvm_info *nvm_info_instance)
{
	if(nvm_info_instance != NULL)
	{
		if(nvm_info_instance->ps8DeviceName != NULL)
			free(nvm_info_instance->ps8DeviceName );
		if(nvm_info_instance->ps8ConfigFileDir != NULL )
			free(nvm_info_instance->ps8ConfigFileDir);
		if(nvm_info_instance->ps8MetaDataFileDir != NULL)
			free(nvm_info_instance->ps8MetaDataFileDir);
		if(nvm_info_instance)
			free(nvm_info_instance);
	}
	
}

/********************************************************************************
 * FUNCTION        : alloc_nvm_memory
 * PARAMETER       : None
 *
 * RETURNVALUE     : struct nvm_info*
 *
 * DESCRIPTION     : This function creates an instance of nvm_info and 
 * returns it
 *
 *---------------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|-----------------------------------------------
 *14.August.2019| Initialversion 1.0  | Ragupathi Palanisamy (RBEI/ECF2)
 * --------------------------------------------------------------------------------
 **********************************************************************************/
struct nvm_info* alloc_nvm_memory()
{
	struct nvm_info *nvm_info_instance = (struct nvm_info*)malloc(sizeof(struct nvm_info));
	if(nvm_info_instance != NULL)
	{
		memset(nvm_info_instance, 0 , sizeof(struct nvm_info));

		nvm_info_instance->ps8DeviceName = (tPS8)malloc(64);
          	nvm_info_instance->ps8ConfigFileDir = (tPS8)malloc(128);
          	nvm_info_instance->ps8MetaDataFileDir = (tPS8)malloc(128);
          	if(nvm_info_instance->ps8DeviceName && nvm_info_instance->ps8ConfigFileDir 
                  && nvm_info_instance->ps8MetaDataFileDir)
                {
                  memset(nvm_info_instance->ps8DeviceName, 0 , 64);
                  memset(nvm_info_instance->ps8ConfigFileDir , 0 , 128);
                  memset(nvm_info_instance->ps8MetaDataFileDir, 0 , 128);
                }
          	else
                {
                  de_alloc_nvm_memory(nvm_info_instance);
                  nvm_info_instance = NULL;
                  fprintf(stderr,"nvm_refresh : Malloc failed in main function lineNo: %d",__LINE__);
                }
	}
	else{
		nvm_info_instance = NULL;
		fprintf(stderr,"nvm_refresh : Malloc failed in main function lineNo: %d",__LINE__);
	}	
	return nvm_info_instance;
}

/********************************************************************************
 * FUNCTION        : main
 * PARAMETER       : argc, argv
 * RETURNVALUE     : None
 *
 * DESCRIPTION     : This function takes the cmd line arguments and populates it
 * 					in the refresh_args structure. Possible arguemnts are
 * 					-d    enable debug output
                    -s    set initial delaytime (seconds)
                    -f    activate refresh even if metadata file indicates no
                    -e    set the deviceName,metadataPath and persistent path of NVM device
 * Usage : //app.exe -e mmcblk1:mmcblk1boot0:mmcblk1boot1 /mnt/tmp2/static/emmc_refresh/ /mnt/tmp2/static/emmc_refresh/ -d -s 2
 *---------------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|-----------------------------------------------
 *14.August.2019| Initialversion 1.0  | Ragupathi Palanisamy (RBEI/ECF2)
 * --------------------------------------------------------------------------------
 **********************************************************************************/
#ifdef UTEST

int nvm_refresh(int argc, char *argv[])

#else

int main(tS32 argc, char *argv[])

#endif
{
	tS32 s32ReturnValue = REFRESH_NOERROR;
	struct nvm_info *nvm_info_root = NULL;

	refresh_args *arguments = (refresh_args*)malloc(sizeof(refresh_args));
	if(arguments != NULL)
	{
		memset(arguments,0,sizeof(refresh_args));
		//setting default wait time
		arguments->uSleepInSec = REFRESH_INITIAL_WAIT_TIME;

		//IF it return negative no need to perform any further operation
		tVSignalHandlerSetup();
		tS32 opt=0;
		optind =1;
		while((opt = getopt(argc,argv,"e:d:fs:")) != -1)
		{
			switch(opt)
			{
				//Parsing the eMMC device names, Config file path and MetaData FilePath
				case 'e':
			{
				optind--;			
				struct nvm_info *nvm_info_instance = alloc_nvm_memory();
				if(nvm_info_instance != NULL)
				{				
					for(tS32 count=0;(optind < argc && strncmp(argv[optind],"-",1));optind++,count++)
					{
						switch(count)
						{
						case 0:
						{
							//Getting eMMC device names
							strncpy(nvm_info_instance->ps8DeviceName,argv[optind],63);
							nvm_info_instance->ps8DeviceName[strlen(nvm_info_instance->ps8DeviceName)] = '\0';
							printf("eMMC %d Device Name : %s\n",count+1,nvm_info_instance->ps8DeviceName);
							break;
						}
						case 1:
							//Getting config file path
						{	strncpy(nvm_info_instance->ps8ConfigFileDir,argv[optind],127);
							nvm_info_instance->ps8ConfigFileDir[strlen(nvm_info_instance->ps8ConfigFileDir)] = '\0';
							printf("eMMC %d Config Name : %s\n",count+1,nvm_info_instance->ps8ConfigFileDir);
							break;
						}
						case 2:
							//Getting metaData file path
							{	strncpy(nvm_info_instance->ps8MetaDataFileDir,argv[optind],127);
							nvm_info_instance->ps8MetaDataFileDir[strlen(nvm_info_instance->ps8MetaDataFileDir)] = '\0';
							printf("eMMC %d MetdaData Name : %s\n",count+1,nvm_info_instance->ps8MetaDataFileDir);
							break;
							}
						default:
							{
							printf("Invalid eMMC device arguments!\n");
							s32ReturnValue = REFRESH_INVALID_ARGUMENT;
							break;
							}
						}
					}
					if(s32ReturnValue == REFRESH_INVALID_ARGUMENT)
					{
						if(arguments != NULL)
							free(arguments);
						de_alloc_nvm_memory(nvm_info_instance);
						return s32ReturnValue;
					}
					if(s32ReturnValue == REFRESH_NOERROR)
					{
						nvm_info_instance->next = NULL;
						addDevice(&nvm_info_root, nvm_info_instance);
						if(nvm_info_root != NULL)
						{
							nvm_info_root->u8DeviceCount++;
						}
					}
					else
					{					
						de_alloc_nvm_memory(nvm_info_instance);
					}				
				}
				else
				{
					fprintf(stderr,"nvm_refresh : Malloc failed in main function lineNo: %d",__LINE__);
				}
				break;
			}
			case 'd':
				//Enable debug logs(debug log file will be created under metadata file path)
				{
					struct stat sb;
					if(stat(optarg,&sb) == 0 && S_ISDIR(sb.st_mode))
					{
						arguments->u8EnableDebug = TRUE;
						InitApplicationLogging(TRUE,optarg);
						printf("eMMC LogFile Folder : %s\n",optarg);
						log_debug("Debugging enabled\n");
					}
					else
						fprintf(stderr,"Unable to get the filePath for storing debug files\n");
					break;
				}
			case 'f':
				//Forcing refresh even when the metadata file says no
				{
				arguments->s8ForceRefresh = TRUE;
				log_debug("Forcefully enabling refresh\n");
				break;
				}
			case 's':
				//initial sleep time of the application
				{
				arguments->uSleepInSec = (tU8)strtol(optarg,NULL,10);
				log_debug("Setting initial sleeping time to %d seconds\n",arguments->uSleepInSec);
				break;
				}
			case '?':
			{
				printf( "Valid options:\n"
						" -d    enable debug output\n"
						" -s    set initial delaytime (seconds)\n"
						" -f    activate refresh even if metadata file indicates no.\n"
						" -e    set the deviceName,metadataPath and persistent path of NVM device \n " \
						"ex: mmcblk0:mmcblk0boot0 <ConfigFilePath> <MetaDataStoragePath>\n"
				);
				s32ReturnValue = REFRESH_ERROR;
				break;
			}
		}
	}
		//If device details were given we are proceeding with the values
		if(nvm_info_root != NULL && s32ReturnValue == REFRESH_NOERROR)
		{
			arguments->nvm_device_details = nvm_info_root;
			sleep(arguments->uSleepInSec);
			if(nvm_info_root->u8DeviceCount > 0)
				s32ReturnValue=s32Perform_eMMCRefresh(arguments);
		}
		if(nvm_info_root != NULL){
			deleteDevice(&nvm_info_root);
		}
		if(arguments != NULL)
			free(arguments);
	}
	else
	{
		fprintf(stderr,"nvm_refresh : Malloc failed in main function lineNo: %d",__LINE__);
		s32ReturnValue = REFRESH_NO_MEMORY;
	}
	application_tearDown();
	return s32ReturnValue;
}
