#include "emmc_platform_access.h"
#include "emmc_refresh.h"
#include "platform_app_manager.h"
#include "nvm_refresh.h"
#include <stdio.h>
#include <errno.h>
#include <dirent.h>
#include <unistd.h>
#include <time.h>
#include <fcntl.h>
#include <sys/types.h>
#include <pthread.h>
#include <stdlib.h>
#include <libgen.h>
static tS32 s32IniteMMCForRefresh(refresh_data *refresh_info);
static tU32 u32checkIfUrgentRefreshNeeded(tU32 u32MMCRevision);
static tS32 s32CheckAndReadConfigFile(tU32 enDevSize, tU32 VendorID, tPC8 cConfigFilePath, refresh_config *read_refresh_config);
static tS32 s32Read_ConfigFile(tPC8 chConfigPath, tBool u8ReadDefaultConfig,tU32 enDevSize, refresh_config *read_refresh_config);
static void vcalculate_checksum( refresh_config *read_refresh_config);
static tS32 s32Write_RefreshMetaData(tPC8 chMetadataPath, refresh_config *refresh_data_config);
static tS32 s32PrepareAndWriteMetaData(tPC8  chMetaDataFilePath, refresh_config *refresh_data_config);
static tS32 s32CheckToStartRefresh(refresh_config *read_refresh_config,struct timespec *readDelay);
static void s32UpdateDueDate(refresh_config *refresh_data_config);
static void vSet_delay(tU32 u32delay_ms , struct timespec *readdelay);
static void s32WaitTillUrgentFlagClears(emmc_dev_data *dev_data);
static tS32 s32PerformRefresh(refresh_data *emmc_refresh_data, tU32 partitionToRefresh);
static tU32 u32UpdateRefreshStatus(emmc_dev_data *dev_data,tU32 *u32sectorIndex,
		tU32 *u32Refresh_ChunkSize, refresh_config *refresh_config_data_local, tBool reachedEndofDisk);
static tS32 s32IsUrgentRefreshNeeded(emmc_dev_data *dev_data);
static void *ptr_align (void *ptr, tU32 alignment);
static tS32 s32StartRefresh(refresh_data *emmc_refresh_data);
//static void vCheckRpmpStatus(tU32 u32EnableRPMPRefresh,emmc_dev_data *dev_data);
static void *SpawnRefreshThread(void *refresh_info);
static void s32Backup_metadataFile(tPC8 chMetadataPath, refresh_config* config);
static tS32 s32MetaDataFileOperations(tPC8 cMetaDataFileDir, refresh_config *refresh_data_config);
static tS32 s32CheckAndReadMetaDataFile(tPC8 cMetaDataFilePath, refresh_config *refresh_data_config);
static tS32 s32Read_RefreshMetaData(tPC8 cMetaDataFilePath,refresh_config *refresh_data_config);
static void vEnableManualBkops(emmc_dev_data *dev_data,tU32 u32mmcRevision);
static tS32 s32checkForValidUsrPartitionFd(emmc_dev_data *dev_data);
const tU32 monthDays[12] = {31, 28, 31, 30, 31, 30,
                           31, 31, 30, 31, 30, 31};
#ifdef METADATA_BACKUP
static void s32Backup_metadataFile(tPC8 chMetadataPath, refresh_config* config)
{
	tPS8 chBackupMetadataFile =(tPS8) concatPath(chMetadataPath,METADATA_FILE_PATH_BK);
	vcalculate_checksum(config);
	(void)s32Write_RefreshMetaData(chBackupMetadataFile,config);
	if(chBackupMetadataFile != NULL)
		free((tPS8)chBackupMetadataFile);
}
#endif
/********************************************************************************
 * FUNCTION        : s32Perform_eMMCRefresh
 * PARAMETER       : refresh_args *arguments -> Structure instance of the
 * 					 parsed arguments
 * 					from main function
 * 					 *
 * RETURNVALUE     : REFRESH_ERROR / REFRESH_NOERROR
 *
 * DESCRIPTION     : This function invokes the necessary functions to initialize
 * 					 eMMC related information.
 *
 *---------------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|-----------------------------------------------
 *14.August.2019| Version 2.0  | Ragupathi Palanisamy (RBEI/ECF2)
 * --------------------------------------------------------------------------------
 **********************************************************************************/
tS32 s32Perform_eMMCRefresh(refresh_args *arguments)
{
	pthread_t *thread = NULL;
	tS32 s32ReturnValue = REFRESH_ERROR;
	if(arguments != NULL)
	{
		if(arguments->nvm_device_details != NULL)
		{
			tU32 noOfDevices = arguments->nvm_device_details->u8DeviceCount;
			emmc_dev_data *find_dev_data = NULL;
			tU8 u8Count = 0;
			struct nvm_info *emmc_info = arguments->nvm_device_details;
			log_debug("eMMC's to refresh : %d\n",noOfDevices);
			//Creating thread object for invoking refresh. n no of threads will be created for n eMMC devices.
			thread = (pthread_t*)malloc((size_t)(sizeof(pthread_t)*noOfDevices));
			memset(thread,0,(size_t)(sizeof(pthread_t)*noOfDevices));
			if(noOfDevices > MAX_REFRESH_DEVICES)
			{
				log_debug("Limiting refresh to %d devices.Given devices count is %d",MAX_REFRESH_DEVICES,noOfDevices);
				noOfDevices = MAX_REFRESH_DEVICES;
			}
			refresh_data *emmc_refresh_data = (refresh_data*)malloc((size_t)(sizeof(refresh_data)*noOfDevices));
			refresh_data *emmc_refresh_data_start = emmc_refresh_data;
			memset(emmc_refresh_data,0,sizeof(refresh_data)*noOfDevices);
			while(emmc_info != NULL)
			{
				find_dev_data = s32CheckAndPopulateDeviceDetails(emmc_info->ps8DeviceName,emmc_refresh_data);
				if(find_dev_data != NULL)
				{
					emmc_refresh_data->dev_data = find_dev_data;
					emmc_refresh_data->u8ForceRefresh = arguments->s8ForceRefresh;
					log_debug("Device found for doing refresh %d %s \n",emmc_refresh_data->u8NumOfPartitions, emmc_refresh_data->dev_data->dev_path_name);
					emmc_refresh_data->ps8ConfigFileDir = (tPS8)malloc(strlen(emmc_info->ps8ConfigFileDir)+1);
					memset(emmc_refresh_data->ps8ConfigFileDir,0,strlen(emmc_info->ps8ConfigFileDir)+1);
					memcpy(emmc_refresh_data->ps8ConfigFileDir,emmc_info->ps8ConfigFileDir,strlen(emmc_info->ps8ConfigFileDir));
					emmc_refresh_data->ps8MetaDataFileDir = (tPS8)malloc(strlen(emmc_info->ps8MetaDataFileDir)+1);
					memset(emmc_refresh_data->ps8MetaDataFileDir,0,strlen(emmc_info->ps8MetaDataFileDir)+1);
					memcpy(emmc_refresh_data->ps8MetaDataFileDir,emmc_info->ps8MetaDataFileDir,strlen(emmc_info->ps8MetaDataFileDir));
					memset(&emmc_refresh_data->refresh_config_data,0,sizeof(refresh_config));
					tS32 ret = pthread_create(&thread[u8Count],NULL,SpawnRefreshThread,emmc_refresh_data);
					if(ret < 0)
						log_error("Unable to create thread for the device %s ErrorCode: %d ErrorString : %s",emmc_refresh_data->dev_data->dev_path_name,errno,strerror(errno));					
					else
						u8Count++;
					//For unit testing
					s32ReturnValue = REFRESH_DEVICE_FOUND;
				}
				else
				{
					log_debug("Unable to get device details for the eMMC %s",emmc_info->ps8DeviceName);
					//For unit testing
					s32ReturnValue = REFRESH_ACCESS_ERROR;
				}
				emmc_info  = emmc_info->next;
				//Increment to store next emmc_refresh_data set
				if(emmc_info != NULL)
					emmc_refresh_data++;
			}
			if(thread)
			{
				for(int i=0;i<u8Count;i++)
				{
					(void)pthread_join(thread[i],NULL);
				}
				free(thread);
			}
			for(tU32 i=0;i<noOfDevices;i++)
			{
				if(emmc_refresh_data_start)
				{
					if(emmc_refresh_data_start[i].dev_data != NULL)
					{
						free(emmc_refresh_data_start[i].dev_data);
					}
					if(emmc_refresh_data_start[i].ps8ConfigFileDir != NULL)
					{
						free(emmc_refresh_data_start[i].ps8ConfigFileDir);
					}
					if(emmc_refresh_data_start[i].ps8MetaDataFileDir != NULL)
					{
						free(emmc_refresh_data_start[i].ps8MetaDataFileDir);
					}
				}
			}
			if(emmc_refresh_data_start)
			{
				free(emmc_refresh_data_start);
			}
		}
	}
	if (s32ReturnValue == REFRESH_DEVICE_FOUND)
	{
		s32ReturnValue = REFRESH_NOERROR;
	}
	return s32ReturnValue;
}
static void *SpawnRefreshThread(void *refresh_info)
{
	s32IniteMMCForRefresh((refresh_data*)refresh_info);
	return NULL;
}
/********************************************************************************
 * FUNCTION        : s32IniteMMCForRefresh
 * PARAMETER       : refresh_data *refresh_info -> Refresh data for eMMC
 * 					from main function
 * 					tBool s8ForceRefresh -> Decides whether to force refresh
 * RETURNVALUE     : REFRESH_ERROR / REFRESH_NOERROR
 *
 * DESCRIPTION     : This function invokes function for reading config files/
 * 					 MetaDataFiles and starts refresh
 *
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 * 14.August.2019| Version 2.0  | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/
static tS32 s32IniteMMCForRefresh(refresh_data *refresh_info)
{
	emmc_dev_data *emmc_info = refresh_info->dev_data;
	tS32    s32ReturnValue  =   REFRESH_ERROR;
	tU32 u32deviceSizeInGb = SIZE_IN_GB(emmc_info->dev_size);
	log_debug("Size of the device is %d\n",u32deviceSizeInGb);
	refresh_config * refresh_data_config = &refresh_info->refresh_config_data;
	if(refresh_data_config != NULL)
	{
		refresh_info->u32UrgentRefreshCheckNeeded = u32checkIfUrgentRefreshNeeded(refresh_info->u32mmc_revision);
		if(refresh_info->u32UrgentRefreshCheckNeeded)
			refresh_data_config->u32RefreshMonthInterval = URGENT_REFRESH_INTERVAL;
		else
			refresh_data_config->u32RefreshMonthInterval = NORMAL_REFRESH_INTERVAL;
		log_debug("Refresh interval chosen %d",refresh_data_config->u32RefreshMonthInterval);
		u32deviceSizeInGb = u32AlignDeviceSize(u32deviceSizeInGb);
		log_debug("Device Size after aligning to power of 2 is %d",u32deviceSizeInGb);
		s32ReturnValue = s32CheckAndReadConfigFile(u32deviceSizeInGb, refresh_info->u32mmc_vendor, refresh_info->ps8ConfigFileDir, refresh_data_config);
		if(s32ReturnValue == REFRESH_NOERROR )
		{
			refresh_data_config->total_partitions = refresh_info->u8NumOfPartitions;
			//vCheckRpmpStatus(refresh_info->refresh_config_data.u32EnableRPMBRefresh,refresh_info->dev_data);
			if(refresh_data_config->u32refresh_type != 1)
			{
				log_debug("Refresh type in Config File"\
						" is %u\n No refresh Needed!\n", refresh_data_config->u32refresh_type);
				s32ReturnValue =  REFRESH_NOT_NEEDED;
			}
			else
			{
				s32ReturnValue = s32MetaDataFileOperations(refresh_info->ps8MetaDataFileDir, refresh_data_config);
			}
			if(s32ReturnValue == REFRESH_NOERROR)
			{
				s32ReturnValue = s32CheckToStartRefresh(refresh_data_config, &refresh_info->readdelay);
				log_debug("Refresh needed status %d ",s32ReturnValue);
				if(s32ReturnValue == REFRESH_NEEDED || (refresh_info->u8ForceRefresh == TRUE))
				{
					ReportApplicationStartup();
					log_debug("Reported Application startup");
					s32ReturnValue = s32StartRefresh(refresh_info);
				}
				else if(s32ReturnValue == REFRESH_NOT_NEEDED)
				{
					log_debug("%s: No eMMC refresh needed\n", __func__);
					log_debug("Present Date:%u-%u-%u\n",refresh_data_config->u32day,refresh_data_config->u32month,refresh_data_config->u32year);
					log_debug("Due date:%u-%u-%u\n",refresh_data_config->u32due_day,refresh_data_config->u32due_month,refresh_data_config->u32due_year);
					log_debug("No refresh needed for the device %s",refresh_info->dev_data->dev_path_name);
				}
			}
			else
			{
				log_debug("Error occurred during MetaDataFile Operations. Refresh Error Code %d",s32ReturnValue);
			}
		}
		else
		{
			log_error("Error occurred while reading the configuration file.Hence, Skipping refresh for the device %s",refresh_info->dev_data->dev_path_name);
		}
	}
	else
	{
		log_warn("refresh_data_config is NULL");
	}
	return s32ReturnValue;
}
//Depreciated we need to check whether Rpmp status has to be checked
/*static void vCheckRpmpStatus(tU32 u32EnableRPMPRefresh,emmc_dev_data *dev_data)
{
    if(0 == u32EnableRPMPRefresh)
    {
        for(int i=0; i<dev_data->numparts; i++)
        {
            if(dev_data[i].bIsRpmb)
            {
                log_debug("dropping Rpmb partition %s from list of partitions found.",dev_data[i].dev_path_name);
                dev_data[i].refresh_needed = 0;
            }
        }
    }
}*/
/********************************************************************************
 * FUNCTION        : s32CountLeapYears
 * PARAMETER       : date d -> date structure
 * RETURNVALUE     : Number of leap years
 *
 * DESCRIPTION     : This function calculates the total number of leap years till the given date
 *
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 * 14.August.2019| Version 2.0  | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/
static tU32 s32CountLeapYears(struct date d)
{
	tU32 years = d.u32year;
	if(d.u32month <= 2)
		years--;
	// An year is a leap year if it is a multiple of 4 and also
    // multiple of 400 and not a multiple of 100.
	return (years / 4) - (years / 100) + (years / 400);
}
/********************************************************************************
 * FUNCTION        : s32getDifferenceInDays
 * PARAMETER       : date startDate -> startdate
 * 					 struct date endDate -> enddate
 * RETURNVALUE     : Diff in dates
 *
 * DESCRIPTION     : This function calculates the date in between the two given dates
 *
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 * 14.August.2019| Version 2.0  | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/
static tS32 s32getDifferenceInDays(struct date startDate, struct date endDate)
{
	tU32 n1 = startDate.u32year*365 + startDate.u32day;
	for(tU32 i=0;i<startDate.u32month-1;i++)
		n1 += monthDays[i];
	n1 += s32CountLeapYears(startDate);
	tU32 n2 = endDate.u32year*365 + endDate.u32day;
	for(tU32 i=0;i<endDate.u32month-1;i++)
		n2 += monthDays[i];
	n2 += s32CountLeapYears(endDate);
	return (tS32)(n2-n1);
}
/********************************************************************************
 * FUNCTION        : s32CheckToStartRefresh
 * PARAMETER       : refresh_config *read_refresh_config -> MetaData configurations
 * 					 struct timespec *readDelay -> Read delay between subsequent reads
 * RETURNVALUE     : REFRESH_NEEDED / REFRESH_NOT_NEEDED
 *
 * DESCRIPTION     : Checks whether refresh needed by comparing due and present date
 *
 * In previous version of refresh due date was calcualted at the end of refresh cycle. From this version
 * due date is calculated during the start of the refresh cycle. Due date will be updated when sectors covered and 
 * partition covered is zero or if there are any time errors.(if due date is some where in the past or much ahead in future)	
 *
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 * 14.August.2019| Version 2.0  | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/
static tS32 s32CheckToStartRefresh(refresh_config *read_refresh_config, struct timespec *readDelay)
{
	tS32 s32ReturnValue = (tS32) REFRESH_NOT_NEEDED;
	if(read_refresh_config && readDelay)
	{
		time_t now = time(NULL);
		struct tm *t = localtime(&now);
		struct date presentDate,dueDate;
		presentDate.u32day		  =	  (tU32)t->tm_mday;
		presentDate.u32month      =   (tU32)t->tm_mon + 1;
		presentDate.u32year       =   (tU32)t->tm_year + 1900;

		//Resetting the due_month to 12 in case of any corruption in the metadata file(Introduced as part of coverity fix)
		if(read_refresh_config->u32due_month > 12)
			read_refresh_config->u32due_month = 12;		
		
		dueDate.u32day             =   read_refresh_config->u32due_day;
		dueDate.u32month           =   read_refresh_config->u32due_month;
		dueDate.u32year            =   read_refresh_config->u32due_year;		
		//Calculate the diff days between current date and due date.
		// 1.If present date is 1/1/2019 , due date is 1/1/2019 then normal refresh
		// 2.If present date is 1/5/2019 , due date is 1/1/2019 then fast refresh
		// 3.If present date is 1/5/2019 , due date is 1/1/1970 then perform refresh // Due to wrong GPS time update
		tS32 diffDays = s32getDifferenceInDays(dueDate,presentDate);		
		vSet_delay(read_refresh_config->u32Refresh_normal_delay_ms,readDelay);
		if(diffDays >= 0)
		{
			// If DiffDays is less than 3 months then perform normal refresh
			if((diffDays/DAYS_PER_MONTH) < 3 )
			{
				read_refresh_config->u32Refresh_multiFactor =1;
				log_debug("%s: Performing Normal refresh \n", __func__ );
				s32ReturnValue = REFRESH_NEEDED;
			}
			//If diff days is greater than 3 months then perform fast refresh
			else if ((diffDays/DAYS_PER_MONTH) >= 3 )
			{
				log_debug("\n%s: High speed refresh advised %u \n", __func__, read_refresh_config->u32Refresh_multiFactor);
				vSet_delay(read_refresh_config->u32Refresh_fastmode_delay_ms,readDelay);
				s32ReturnValue = REFRESH_NEEDED;
			}
			//Updating the due_date when (due_date >= present date) and (number of sectors and partition covered equals zero)
			if(read_refresh_config->u32no_of_sectors_covered ==0 && read_refresh_config->u32partition_covered ==0)			
			{
				s32UpdateDueDate(read_refresh_config);
			}
			
		}
		else
		{
			tU32 diffDaysNeg = (tU32)s32getDifferenceInDays(presentDate,dueDate);
			//If diffDays is not in the range of expected refresh boundry then due date is wrongly updated due to GPS error. So performing normal refresh
			if(diffDaysNeg/(tU32)DAYS_PER_MONTH > read_refresh_config->u32RefreshMonthInterval)
			{
				read_refresh_config->u32Refresh_multiFactor =1;
				//Updating the due_date when due date is much ahead in the future.
				s32UpdateDueDate(read_refresh_config);
				log_debug("%s: refresh advised for correct time update\n", __func__ );
				s32ReturnValue = REFRESH_NEEDED;
			}
		}
		// If metadata has incomplete sector covered then perform refresh
		if((s32ReturnValue == REFRESH_NOT_NEEDED) &&((read_refresh_config->u32no_of_sectors_covered != 0)
				||(read_refresh_config->u32partition_covered != 0)))
		{
			read_refresh_config->u32Refresh_multiFactor = 1;
			log_debug("System Time Error: sector covered =%u partition=%u\n",
					read_refresh_config->u32no_of_sectors_covered,read_refresh_config->u32partition_covered);
			s32ReturnValue=(tS32)REFRESH_NEEDED;
		}		
	}
	else
	{
		s32ReturnValue = REFRESH_NO_MEMORY;
		log_debug("refresh_data_config or timespec is NULL");
	}
	return s32ReturnValue;
}
/********************************************************************************
 * FUNCTION        : vSet_delay
 * PARAMETER       : tU32 u32delay_ms -> Delay time in milliSeconds
 * 					 struct timespec *readdelay -> Readdelay structure
 * RETURNVALUE     : None
 *
 * DESCRIPTION     : Populate read delay structure from the value of u32delay_ms
 *
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *19.Aug.2016| Initialversion 1.0  | Balaji (RBEI/ECF5)
 *-----------|---------------------|--------------------------------------
 *02.Sep.2016| Version 1.1         | Swathi Bolar (RBEI/ECF5)
 *-----------|---------------------|---------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 **********************************************************************************/
static void vSet_delay(tU32 u32delay_ms , struct timespec *readdelay)
{
	/*conversion from millisecond to second*/
	readdelay->tv_sec = (long)(u32delay_ms / MILLISECONDS_PER_SECOND);
	/*conversion from remaining millisecond to nanosecond.*/
	readdelay->tv_nsec = (long)((u32delay_ms % MILLISECONDS_PER_SECOND )* 1000000 );
}
/********************************************************************************
 * FUNCTION        : s32CheckToUpdateDueDate
 * PARAMETER       : refresh_config *emmc_refresh_data -> refresh metadata structure
 *
 * RETURNVALUE     : NONE
 *
 * DESCRIPTION     : Sets the next due date of refresh in the start of current 
 * refresh cycle
 *
 *-------------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|---------------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 **********************************************************************************/
static void s32UpdateDueDate(refresh_config *refresh_data_config)
{	
	time_t now = time(NULL);
	struct tm *t = localtime(&now);

	refresh_data_config->u32day = (tU32) t->tm_mday;
	refresh_data_config->u32month = (tU32) t->tm_mon + 1;
	refresh_data_config->u32year = (tU32) t->tm_year + 1900;

	//Setting the due_year to the current year and setting the due_month to current month+refreshInterval
	refresh_data_config->u32due_year = refresh_data_config->u32year;
	refresh_data_config->u32due_month = refresh_data_config->u32month + refresh_data_config->u32RefreshMonthInterval;

	// If due_month is greater than 12 months then adjust the month and year accordingly.
	if (refresh_data_config->u32due_month > MONTH_PER_YEAR)
	{
		refresh_data_config->u32due_month = refresh_data_config->u32due_month % MONTH_PER_YEAR;
		refresh_data_config->u32due_year = refresh_data_config->u32year + 1;
	}

	if(refresh_data_config->u32TotalRefreshCount == REFRESHCOUNT_INITMAGIC)
		refresh_data_config->u32TotalRefreshCount = 0;
	
}
/********************************************************************************
 * FUNCTION        : s32StartRefresh
 * PARAMETER       : refresh_data *emmc_refresh_data -> Structure holding all the details
 * 					 required for device refresh
 *
 * RETURNVALUE     : NONE
 *
 * DESCRIPTION     : Iterates the number of eMMC partitions and invokes refresh function
 * 					 for each eMMC partition
 *
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|---------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 **********************************************************************************/
static tS32 s32StartRefresh(refresh_data *emmc_refresh_data)
{
	tU32 u32count = 0; tS32 s32ReturnValue = REFRESH_NOERROR;
	if(emmc_refresh_data != NULL)
	{
		//Iterate the number eMMC partitions and invokes s32PerformRefresh function			
		for(u32count = emmc_refresh_data->refresh_config_data.u32partition_covered;
				u32count < emmc_refresh_data->u8NumOfPartitions; (s32ReturnValue == REFRESH_SEEK_ERROR)?u32count=0:u32count++)
		{
			emmc_refresh_data->refresh_config_data.u32partition_covered = u32count;			
			log_debug("Triggering refresh for the partition %s",emmc_refresh_data->dev_data[u32count].dev_path_name);
			s32ReturnValue = s32PerformRefresh(emmc_refresh_data, u32count);
			if(s32ReturnValue == REFRESH_SEEK_ERROR)
			{
				//Added to handle backward compatibility of the metadata.txt file created by the previous refresh application running in the devices.
				emmc_refresh_data->refresh_config_data.u32no_of_sectors_covered = 0;				
				continue;
			}
			else if(s32ReturnValue != REFRESH_NOERROR)
			{
				log_error("Error occurred during refresh of the device %s Refresh ErrorCode : %d ",emmc_refresh_data->dev_data[u32count].dev_path_name,s32ReturnValue);
				break;
			}
		}
	}
	else
	{
		log_warn("Unable to start refresh as respective data structure is NULL");
	}
	return s32ReturnValue;
}
/********************************************************************************
 * FUNCTION        :  vEnableManualBkops
 * PARAMETER       :  emmc_dev_data    data referencing MMC device
 *                    u32mmcRevision   MMC Revision of the device
 * RETURNVALUE     :  void
 * DESCRIPTION     :  Sets EXT_CSD[163], MANUAL_EN to 1 for 5.1 eMMC devices
 *
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *29.OCT.2020| Initialversion 1.0  | Ragupathi Palanisamy(RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/
static void vEnableManualBkops(emmc_dev_data *dev_data,tU32 u32mmcRevision)
{
	if(u32mmcRevision == MMC_SPEC_V5_1)
	{
		tS8 s8EXT_CSD_Reg[512] = {0};
		tS32 S32ReturnValue = s32Read_MMC_EXT_CSD(dev_data,s8EXT_CSD_Reg);
		if(S32ReturnValue == REFRESH_NOERROR)
		{
			//Check whether MANUAL_EN is 1 or 0
			if((s8EXT_CSD_Reg[EXT_CSD_BKOPS_EN] & 1 )== 0)
			{
				log_debug("Setting MANUAL_EN for the device %s",dev_data->dev_path_name);
				s32write_ExtCsd_Value(dev_data,EXT_CSD_BKOPS_EN,1);				
			}
		}
	}
}
/********************************************************************************
 * FUNCTION        :  s32checkForValidUsrPartitionFd
 * PARAMETER       :  emmc_dev_data    data referencing MMC device                     
 * RETURNVALUE     :  REFRESH_ACCESS_ERROR/REFRESH_NOERROR/REFRESH_ERROR
 * DESCRIPTION     :  Checks and open file handle for an eMMC data partition
 *
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *29.OCT.2020| Initialversion 1.0  | Ragupathi Palanisamy(RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/
static tS32 s32checkForValidUsrPartitionFd(emmc_dev_data *dev_data)
{
	tS32 s32Returnvalue = REFRESH_NOERROR;
	if(dev_data != NULL)
	{
		if(dev_data->s32fd < 0 )
		{
			log_debug("File handle not present for the device : %s. Hence trying to open it",dev_data->dev_path_name);
			dev_data->s32fd = open(dev_data->dev_path_name, O_RDWR | O_LARGEFILE |  O_DIRECT);
			if(dev_data->s32fd <0 )
			{
				log_error("Error while opening the user partition device %s ErrorCode : %d ErrorString: %s",dev_data->dev_path_name,errno,strerror(errno));
				s32Returnvalue = REFRESH_ACCESS_ERROR;
			}
		}
	}
	else
	{
		log_debug("NULL Pointer exception");
		s32Returnvalue = REFRESH_ERROR;
	}
	return s32Returnvalue;
}
/********************************************************************************
 * FUNCTION        :  s32PerformRefresh
 * PARAMETER       :  emmc_refresh_data    data referencing MMC device
 *                    partitionToRefresh    partition number to use for this run.
 * RETURNVALUE     :  returns REFRESH_NOERROR on success and Error Values on Failure
 * DESCRIPTION     :  This performs the EMMC refresh algorithm. After reading configures
 * 					  amount of bytes it makes a write to the eMMC(Updating the metadata File)
 *
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *03.DEC.2014| Initialversion 1.0  | SWATHI BOLAR (RBEI/ECF5)
 * -----------------------------------------------------------------------
 *12.DEC.2014| Version 1.1         | SWATHI BOLAR (RBEI/ECF5)
 * -----------------------------------------------------------------------
 *30.JUN.2016| Version 1.5         | CHAKITHA SARASWATHI (RBEI/ECF5)
 *           |                     | CFG3-1958:Implements refresh for the
 *           |                     | Micron eMMC chips.
 *------------------------------------------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 **********************************************************************************/
static tS32 s32PerformRefresh(refresh_data *emmc_refresh_data, tU32 partitionToRefresh)
{
	tU32 u32Total_Sector= 0,u32UrgentRefreshNeeded =0;
	tS32 s32ReturnValue=0,s32ReadCounter=0,
			s32Threshold=0;
	tU32 u32sectorIndex =0 ;
	off64_t offset=0;
	tU32 u32page_size=0;
	tS8 *ibuf = NULL;
	tS32 tS32BytesRead = 0;
	tPS8 ps8Real_buf = NULL;
	
	tPC8 cMetaDataFilePath = concatPath(emmc_refresh_data->ps8MetaDataFileDir,METADATA_FILE_PATH);
	tPC8 cMetaDataBackupFilePath = concatPath(emmc_refresh_data->ps8MetaDataFileDir,METADATA_FILE_PATH_BK);
	emmc_dev_data *dev_data_local = &emmc_refresh_data->dev_data[partitionToRefresh];
	if(dev_data_local != NULL)
	{
		// Considering 0 as always the user parition as we have the qsort operation
		emmc_dev_data *dev_data_user_partition = &emmc_refresh_data->dev_data[0];
		refresh_config *refresh_config_local = &emmc_refresh_data->refresh_config_data;
		tU32 u32Refresh_ChunkSize = refresh_config_local->u32refresh_size;
		u32Total_Sector = (tU32)(dev_data_local->dev_size >> 9);
		log_debug("Total_sector of device %s is %d chunkSize=%d",dev_data_local->dev_path_name,u32Total_Sector,u32Refresh_ChunkSize);
		dev_data_local->s32fd= open(dev_data_local->dev_path_name, O_RDWR | O_LARGEFILE | O_SYNC | O_DIRECT);		
		if(dev_data_local->s32fd >= 0 )
		{
			if(s32checkForValidUsrPartitionFd(dev_data_user_partition) == REFRESH_NOERROR)
			{
     			//Always 0 will be the userdata partition as qsort operation was performed over the emmc_dev_data struct
				vEnableManualBkops(dev_data_user_partition,emmc_refresh_data->u32mmc_revision);					
				refresh_config_local->u32total_numbers_of_sectors = u32Total_Sector;
				if(access(cMetaDataBackupFilePath,F_OK) == -1)
				{
					log_debug("Creating Backup of Metadata file in the dir %s",emmc_refresh_data->ps8MetaDataFileDir);
					s32Backup_metadataFile(emmc_refresh_data->ps8MetaDataFileDir,refresh_config_local);
				}
				log_debug("no of sectors covered is %d",refresh_config_local->u32no_of_sectors_covered);
				offset = lseek64(dev_data_local->s32fd,(off64_t)((tU32)refresh_config_local->u32no_of_sectors_covered *
						(off64_t)SECTOR_SIZE), SEEK_CUR);
				if(offset >= 0 )
				{
					u32page_size = (tU32)getpagesize();
					ps8Real_buf = (tS8 *) malloc( (SECTOR_SIZE * u32Refresh_ChunkSize) * sizeof(tS8) +
							INPUT_BLOCK_SLOP(u32page_size));
					if(ps8Real_buf != NULL)
					{
						ibuf = ps8Real_buf;
						ibuf = (tS8*)(ptr_align(ibuf, u32page_size));
						for(u32sectorIndex = refresh_config_local->u32no_of_sectors_covered;
								((u32sectorIndex < u32Total_Sector) && (tS32BytesRead >= 0));
								(u32sectorIndex += u32Refresh_ChunkSize), s32ReadCounter++ )
						{
							tS32BytesRead = (tS32)read(dev_data_local->s32fd, ibuf, SECTOR_SIZE * u32Refresh_ChunkSize);
							log_debug("Refresh for device : %s SectorIndex : %u - ReadBytes %d\n",dev_data_local->dev_path_name,u32sectorIndex,tS32BytesRead);
							if(tS32BytesRead >= 0 )
							{
								if(tS32BytesRead != (tS32)(SECTOR_SIZE * u32Refresh_ChunkSize))
								{
									tS32BytesRead = (tS32)u32sectorIndex + (tS32BytesRead/SECTOR_SIZE);
									if((tU32)tS32BytesRead >= u32Total_Sector)
									{
										if(emmc_refresh_data->u32UrgentRefreshCheckNeeded)
										{
											u32UrgentRefreshNeeded = u32UpdateRefreshStatus(dev_data_user_partition,&u32sectorIndex,
													&u32Refresh_ChunkSize,refresh_config_local,1);
										}
										else
										{
											refresh_config_local->u32no_of_sectors_covered = 0;
											refresh_config_local->u32partition_covered++;
										}
									}
									else
									{
										if(emmc_refresh_data->u32UrgentRefreshCheckNeeded)
										{
										 log_debug("Only %d bytes read  while reading the device %s SectorIndex: %d ErrorCode : %d ErrorString: %s"
												,tS32BytesRead,dev_data_local->dev_path_name,u32sectorIndex,errno,strerror(errno));
										 //Check whether urgentrefresh is set or not.
											u32UrgentRefreshNeeded = u32UpdateRefreshStatus(dev_data_user_partition,&u32sectorIndex,
													&u32Refresh_ChunkSize,refresh_config_local,2);
										 //If not set skip that chunk and lseek to next chunk for reading
											if(u32UrgentRefreshNeeded == 2)
											{
												//In case of read failure and BKOPS is not set, then skip that particular chunk.
												(void)lseek64(dev_data_local->s32fd,(off64_t)((unsigned int)refresh_config_local->u32no_of_sectors_covered *(off64_t)SECTOR_SIZE), SEEK_SET);
												s32ReadCounter = 0;
											}
										}
									}
								}
								else
								{
									if(emmc_refresh_data->u32UrgentRefreshCheckNeeded)
									{
										u32UrgentRefreshNeeded = u32UpdateRefreshStatus(dev_data_user_partition,&u32sectorIndex,
												&u32Refresh_ChunkSize,refresh_config_local,0);									
									}
									else
									{
										refresh_config_local->u32no_of_sectors_covered += (tU32)u32Refresh_ChunkSize;
									}
									if (refresh_config_local->u32no_of_sectors_covered >= u32Total_Sector)
									{
										refresh_config_local->u32no_of_sectors_covered = 0;
										refresh_config_local->u32partition_covered ++;
									}
								}
								//Set u32UrgentRefreshNeeded only when the return value from u32UpdateRefreshStatus() is 1.
								u32UrgentRefreshNeeded &= 1;
								s32Threshold = (((tS32)refresh_config_local->u32Refresh_threshold) * ((tS32)refresh_config_local->u32Refresh_multiFactor));
								if((s32ReadCounter == s32Threshold) ||
										(refresh_config_local->u32no_of_sectors_covered == 0) || u32UrgentRefreshNeeded )
								{
									log_debug("Writing MetadaFile for the device %s SectorIndex: %u ThresholdValue : %d\n",dev_data_local->dev_path_name,u32sectorIndex,s32Threshold);
									s32ReturnValue = s32PrepareAndWriteMetaData(cMetaDataFilePath,refresh_config_local);
									if(emmc_refresh_data->u32UrgentRefreshCheckNeeded && u32UrgentRefreshNeeded && (s32ReturnValue == REFRESH_NOERROR))
									{
										s32WaitTillUrgentFlagClears(dev_data_user_partition);
										offset = lseek64(dev_data_local->s32fd,(off64_t)((unsigned int)refresh_config_local->u32no_of_sectors_covered *
												(off64_t)SECTOR_SIZE), SEEK_SET);
										if (offset == -1)
										{
											log_error("Error while seeking the device %s ErrorCode : %d ErrorString :%s ",dev_data_local->dev_path_name,errno,strerror(errno));
											s32ReturnValue = (tS32)REFRESH_SEEK_ERROR;
											break;
										}
									}
									s32ReadCounter = 0;
									fflush(stdout);
								}
								if(s32ReturnValue != REFRESH_NOERROR)
								{
									log_debug("Refresh code : %d ", s32ReturnValue);
									break;
								}
								#ifdef METADATA_BACKUP
								else if(refresh_config_local->u32no_of_sectors_covered % (tU32)(BACKUP_THRESHHOLD(u32Total_Sector)) == 0)
								{
									s32Backup_metadataFile(emmc_refresh_data->ps8MetaDataFileDir,refresh_config_local);
								}
								#endif
								nanosleep(&emmc_refresh_data->readdelay,NULL);
							}
							else
							{
								log_debug("Read error while reading the device %s SectorIndex: %d ErrorCode : %d ErrorString: %s",dev_data_local->dev_path_name,u32sectorIndex,errno,strerror(errno));
								fflush(stdout);
								s32ReturnValue = (tS32) REFRESH_READ_ERROR;
								break;
							}
						}
						free(ps8Real_buf);
						ps8Real_buf = NULL;
					}
					else
					{
						log_debug("Malloc Failed for Bytes-error %d ErrorCode : %d",(int)( (SECTOR_SIZE * u32Refresh_ChunkSize) * sizeof(tS8) + INPUT_BLOCK_SLOP(u32page_size) ),
								errno);
						s32ReturnValue = (tS32) REFRESH_NO_MEMORY;
					}
				}
				else
				{
					log_error("Error while seeking the device %s ErrorCode : %d ErrorString :%s ",dev_data_local->dev_path_name,errno,strerror(errno));
					s32ReturnValue = (tS32)REFRESH_SEEK_ERROR;
				}
				if(dev_data_user_partition->s32fd >= 0)
					close(dev_data_user_partition->s32fd);
			}
			if(dev_data_local->s32fd >= 0)
			close(dev_data_local->s32fd);
			
		}
		else
		{
			log_debug("Failed to open the device %s ErrorCode : %d ErrorString : %s",dev_data_local->dev_path_name,errno,strerror(errno));
			s32ReturnValue = (tS32)REFRESH_OPEN_ERROR;
		}
		
	}	
	else
	{
		log_debug("emmc_dev_data is NULL");
		s32ReturnValue = (tS32)REFRESH_ERROR;
	}
	
	if(cMetaDataFilePath != NULL)
		free((tPS8)cMetaDataFilePath);
	if(cMetaDataBackupFilePath != NULL)
		free((tPS8)cMetaDataBackupFilePath);
	return s32ReturnValue;
}
/********************************************************************************
 * FUNCTION        : s32WaitTillUrgentFlagClears
 * PARAMETER       : tPC8 s8MmcPath -> eMMC device Path
 *
 * RETURNVALUE     : URGENT_REFRESH_NEEDED || URGENT_REFRESH_NOT_NEEDED
 *
 * DESCRIPTION     : Waits till  Urgent Bkops operation state is normal.Max retry will be
 * 					 10 times with an interval of 1000us.
 *
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|---------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 **********************************************************************************/
static void s32WaitTillUrgentFlagClears(emmc_dev_data *dev_data)
{
	tS32 u32UrgentRefresh = 1;
	tS32 u32UrgentRefreshRetry = 0;
	while(u32UrgentRefresh && (u32UrgentRefreshRetry < MAX_URGENT_REFRESH_RETRY))
	{
		switch(s32IsUrgentRefreshNeeded(dev_data))
		{
		case URGENT_REFRESH_NEEDED:
			u32UrgentRefresh = 1;
			break;
		case URGENT_REFRESH_NOT_NEEDED:
			u32UrgentRefresh = 0;
			break;
		default:
			u32UrgentRefresh = 0;	
	    	log_debug("\nFailed to get urgent refresh status for the device %s",dev_data->dev_path_name);
		}
		usleep(URGENT_BKOPS_CLEAR_WAIT_US);
		u32UrgentRefreshRetry++;
	}
}
/********************************************************************************
 * FUNCTION        : u32UpdateRefreshStatus
 * PARAMETER       : tPC8 s8MmcPath -> eMMC device Path
 *					 tS32 *u32sectorIndex -> Current read sector index
 *					 tS32 *u32Refresh_ChunkSize -> Chunksize of every read call
 *					 refresh_config *refresh_config_data_local -> refresh metadata information
 *					 tBool reachedEndofDisk -> Whether end of eMMC device is reached
 *
 * RETURNVALUE     : 0 || 1
 *
 * DESCRIPTION     : updates the refresh_config structure with respect to
 * 					 number of partitions covered and sectors covered
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|---------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 **********************************************************************************/
static tU32 u32UpdateRefreshStatus(emmc_dev_data* dev_data,tU32 *u32sectorIndex,
		tU32 *u32Refresh_ChunkSize, refresh_config *refresh_config_data_local, tU8 reachedEndofDisk)
{
	tU32 u32UrgentRefreshNeeded = 0;
	switch(s32IsUrgentRefreshNeeded(dev_data))
	{
	case URGENT_REFRESH_NEEDED:
		//Set urgentRefreshNeeded flag
		u32UrgentRefreshNeeded =1;
		if(*u32sectorIndex > 0)
			*u32sectorIndex -= *u32Refresh_ChunkSize;
		break;
	case URGENT_REFRESH_NOT_NEEDED:
		u32UrgentRefreshNeeded =2;
		if(reachedEndofDisk)
		{
			log_debug("Refresh done for the partition %s",dev_data->dev_path_name);
			refresh_config_data_local->u32no_of_sectors_covered = 0;
			refresh_config_data_local->u32partition_covered++;
		}
		else
		{
			refresh_config_data_local->u32no_of_sectors_covered += (tU32)*u32Refresh_ChunkSize;
		}
		break;
	default:
		/*Going ahead is preferred otherwise we may end up in an infinite loop*/
		log_debug("Failed to get urgent refresh_status for the device %s",dev_data->dev_path_name);
		refresh_config_data_local->u32no_of_sectors_covered = 0;
		refresh_config_data_local->u32partition_covered++;
	}
	return u32UrgentRefreshNeeded;
}
/********************************************************************************
 * FUNCTION        : s32IsBkopsStatusSet
 * PARAMETER       : s8BkopsStatus -> Bkops_status byte
 *
 * RETURNVALUE     : 1 or 0
 *
 * DESCRIPTION     : Returns 1 if BKOPS_STATUS value is 0x3 or 0x2
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|---------------------------------------
 *15.Jun.2020| Version 2.1         | Ragupathi Palanisamy (RBEI/ECF2)
 **********************************************************************************/
static inline tS8 s32IsBkopsStatusSet(tS8 s8BkopsStatus)
{
       return s8BkopsStatus & BKOPS_STATUS_PERFORMANCE_IMPACTED;
}
/********************************************************************************
 * FUNCTION        : s32IsUrgentRefreshNeeded
 * PARAMETER       : tPC8 s8MmcPath -> eMMC device Path
 *
 * RETURNVALUE     : URGENT_REFRESH_NEEDED || URGENT_REFRESH_NOT_NEEDED
 *
 * DESCRIPTION     : Reads the status register and EXT_CSD[246] to check for
 * 					 URGENT_BKOPS_STATUS
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|---------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 **********************************************************************************/
static tS32 s32IsUrgentRefreshNeeded(emmc_dev_data* dev_data)
{
	uint32_t u32StatusReg = 0;
	tS8 s8EXT_CSD_Reg[512] = {0};
	// Get status register value
	tS32 tS32ReturnValue = s32Read_MMC_StatusReg( dev_data , &u32StatusReg );
	// if status is need-backops, OR if status is not available, go check ext-csd.
	// only exit if have status AND do not need backops.
	if( tS32ReturnValue == REFRESH_NOERROR && ( URGENT_BKOPS & u32StatusReg ))
	{
		// Open and Check BKOPS Status in Extended Card Specific Data Register.
		tS32ReturnValue = s32Read_MMC_EXT_CSD(dev_data,s8EXT_CSD_Reg);
		if( tS32ReturnValue == REFRESH_NOERROR )
		{
			/* Read byte 246 from extended csd.
			 * Check If Urgent Bkops operation outstanding state is 0x3 or 0x2.We are setting the
			 * urgent refresh flag when BKOPS_STATUS is 0x2 for all the vendor eMMC's as samsung eMMC
			 * maximum value is 0x2.*/
			if(s32IsBkopsStatusSet(s8EXT_CSD_Reg[EXT_CSD_BKOPS_STATUS]))
			{
				log_debug("Urgent Refresh bit set in the EXT_CSD[246] :%d ",s8EXT_CSD_Reg[EXT_CSD_BKOPS_STATUS]);
				tS32ReturnValue =  URGENT_REFRESH_NEEDED;
			}
			else
			{
				tS32ReturnValue = URGENT_REFRESH_NOT_NEEDED;
			}
		}
		else
		{
			log_error("Error occurred while reading EXT_CSD register.Refresh Error Code : %d ",tS32ReturnValue);
			tS32ReturnValue =  URGENT_REFRESH_NOT_NEEDED;
		}
	}
	else
	{
		log_debug("Urgent Refresh bit not set in the status register Status reg value :%x ",u32StatusReg);
		tS32ReturnValue = URGENT_REFRESH_NOT_NEEDED;
	}
	return tS32ReturnValue;
}
/********************************************************************************
 * FUNCTION        : u32checkIfUrgentRefreshNeeded
 * PARAMETER       : tU32 s32MMCRevision -> MMC Revision
 * 					 tU32 s32VendorId    -> MMC Vendor Id
 *
 * RETURNVALUE     : 0 || 1
 *
 * DESCRIPTION     : For Micron eMMC's revision greater than 5.0 Urgent Bkops bit has to be
 * 					 checked either in status reg or EXT_CSD register. This function
 * 					 returns 1 if EXT_CSD revision is greater than 5.0
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|---------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 **********************************************************************************/
static tU32 u32checkIfUrgentRefreshNeeded(tU32 u32MMCRevision)
{
	tU32 u32urgentRefreshNeeeded = 0 ;
	if(u32MMCRevision >= (tU32)MMC_SPEC_V5_0)
	{
		u32urgentRefreshNeeeded = 1;
	}
	return u32urgentRefreshNeeeded;
}
/********************************************************************************
 * FUNCTION        : s32CheckAndReadConfigFile
 * PARAMETER       : tS32 enDevSize -> eMMC Device size
 * 					 tS32 VendorID  -> Vendor id of eMMC
 * 					 tPC8 cConfigFilePath -> Config file path
 * 					 refresh_config *read_refresh_config -> Metadata file information
 *
 * RETURNVALUE     : REFRESH_NOERROR || ERROR
 *
 * DESCRIPTION     : Gets the VendorName from the VendorList array based on VendorId and tries
 * 					 to read the metaData file.If unable to read the config data for the given vendor and size
 * 					 then read the default config from config_DEFAULT.txt file
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|---------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 **********************************************************************************/
static tS32 s32CheckAndReadConfigFile(tU32 enDevSize, tU32 VendorID, tPC8 cConfigFilePath, refresh_config *read_refresh_config)
{
	tBool u8ReadDefaultConfig = TRUE;
	tS32 s32ReturnValue = REFRESH_NOERROR;
	char chConfigFileName [ MAXLENGTH ] ={ 0 };
	tS8 s8vendorName[16];
	//If the device size is not greater than Zero then we are choosing the
	//config_DEFAULT file
	if(enDevSize > 0)
	{
		switch(VendorID)
		{
		case SAMSUNG_VENDOR_ID:
			log_debug("VendorID = 0x%X, SAMSUNG eMMC detected\n", VendorID);
			strncpy(s8vendorName,"SAMSUNG",sizeof(s8vendorName));
			u8ReadDefaultConfig = FALSE;
			break;
		case MICRON_VENDOR_ID:
		case MICRON_VENDOR_ID_I:
			log_debug("VendorID = 0x%X, MICRON eMMC detected\n", VendorID);
			strncpy(s8vendorName,"MICRON",sizeof(s8vendorName));
			u8ReadDefaultConfig = FALSE;
			break;
		case TOSHIBA_VENDOR_ID:
			log_debug("VendorID = 0x%X, TOSHIBA eMMC detected\n", VendorID);
			strncpy(s8vendorName,"TOSHIBA",sizeof(s8vendorName));
			u8ReadDefaultConfig = FALSE;
			break;
		default:
			log_debug("VendorID = 0x%X, Unknown vendor. Hence, Choosing default configuration",VendorID);
			strncpy(s8vendorName,"DEFAULT",sizeof(s8vendorName));
			VendorID = 0;
		}
		sprintf(chConfigFileName,"%s/config_%s.txt",cConfigFilePath,s8vendorName);
		s32ReturnValue = s32Read_ConfigFile(chConfigFileName,u8ReadDefaultConfig,enDevSize,read_refresh_config);
		//If unable to read the specific size configuration from the vendor file then read default config from config_DEFAULT.txt file
		if((s32ReturnValue == (tS32)REFRESH_OUT_OF_RANGE) && (u8ReadDefaultConfig == FALSE))
		{
			log_debug("Config Details not Found for the size %d Gb in the file %s. Hence proceeding with default configuration",enDevSize, chConfigFileName);
			sprintf(chConfigFileName,"%s/config_DEFAULT.txt",cConfigFilePath);
			s32ReturnValue = s32Read_ConfigFile(chConfigFileName,TRUE,0,read_refresh_config);
		}
	}
	else
	{
		log_debug("Size of the device is zero.Not Proceeding further");
		s32ReturnValue = REFRESH_ERROR;
	}
	return s32ReturnValue;
}
/********************************************************************************
 * FUNCTION        : s32Read_ConfigFile
 * PARAMETER       : tPC8 chConfigPath -> Config file dir
 * 					 tBool u8ReadDefaultConfig -> Whether default config has to be read
 * 					 tS32 s32DevSize -> Size of the eMMC device
 * 					 refresh_config *read_refresh_config -> Structure where the config information
 * 					 will be populated
 *
 * RETURNVALUE     : REFRESH_NOERROR || REFRESH_OUT_OF_RANGE || REFRESH_INVALID_ARGUMENT ||
 *                   REFRESH_CORRUPTED_FILE
 *
 * DESCRIPTION     : This function reads the information from config file and loads it in the
 * 					 refresh_config structure
 * *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *03.DEC.2014| Initialversion 1.0  | SWATHI BOLAR (RBEI/ECF5)
 * -----------------------------------------------------------------------
 *30.JUN.2016| Version 1.5         | CHAKITHA SARASWATHI (RBEI/ECF5)
 *           |                     | CFG3-1958:Implements refresh for the
 *           |                     | Micron eMMC chips.
 *------------------------------------------------------------------------
 *30.JUN.2016| Version 1.9         | SELVAKUMAR KALIMUTHU (RBEI/ECF2)
 *           |                     | Config file optimization
 *------------------------------------------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 *------------------------------------------------------------------------
 **********************************************************************************/
static tS32 s32Read_ConfigFile(tPC8 chConfigPath, tBool u8ReadDefaultConfig,tU32 s32DevSize, refresh_config *read_refresh_config)
{
	FILE                 *fp;
	tS32                  s32ReturnValue = REFRESH_NOERROR;
	tU32                 s32SizeinGB,u32EnDebug = 0;	
	if (NULL != chConfigPath)
	{
		fp = fopen(chConfigPath, "r");
		if (NULL == fp)
		{
			log_error("Failed to open the config file '%s' ErrorCode : %d ErrorString : %s",chConfigPath,errno,strerror(errno));
			s32ReturnValue = (tS32) REFRESH_OPEN_ERROR;
		}
		else
		{
			fseek(fp,0,SEEK_END);
			long size = ftell(fp);
			fseek(fp,0,SEEK_SET);
			//If the config file size is not greater than 100 bytes, then assume that the file is corrupted
			if(size >= 100)
			{
				log_debug("Using config file '%s'\n",chConfigPath);
				while ( fscanf(fp,"Size_GB  : %u\n",&s32SizeinGB) != -1 )
				{
					/* Read all the Data specified in the Configuration File */
					(void)fscanf(fp,"Refresh_type: %u\nRefresh_size: %u\nRefresh_threshold: %u\n",
							&read_refresh_config->u32refresh_type, &read_refresh_config->u32refresh_size,
							&read_refresh_config->u32Refresh_threshold);
					(void)fscanf(fp,"Normal_delay:%u\nFastMode_delay:%u\nMultiFactor:%u\n",
							&read_refresh_config->u32Refresh_normal_delay_ms, &read_refresh_config->u32Refresh_fastmode_delay_ms,
							&read_refresh_config->u32Refresh_multiFactor);
					/*Read RBMP partition refresh details, 0-Ignore RPMB partition refresh, 1- include RPMB partition for refresh*/
					(void)fscanf(fp,"rpmb: %u\n",&read_refresh_config->u32EnableRPMBRefresh);
					/* FUTURE USE- Provision for debugging */
					(void)fscanf(fp,"Enable Logs: %u \n",&u32EnDebug);
					if ( ( s32SizeinGB == (tU32)s32DevSize ) ||
							(TRUE == u8ReadDefaultConfig ) )
					{
						log_debug( "Size_GB: %d\n",s32SizeinGB );
						s32ReturnValue = (tS32) REFRESH_NOERROR;
						break;
					}
					else
					{
						s32ReturnValue = (tS32)REFRESH_OUT_OF_RANGE;
					}
				}
				//protection added to prevent divide by 0 Error
				if (( 0 == read_refresh_config->u32refresh_size ) && ( REFRESH_OUT_OF_RANGE != (ErrorValues)s32ReturnValue ))
				{
					log_error("Configuration File %s seems to be corrupted.",chConfigPath);
					s32ReturnValue = (tS32)REFRESH_CORRUPTED_FILE;
				}
			}
			else
			{
				log_error("Configuration File %s seems to be corrupted.",chConfigPath);
				s32ReturnValue = (tS32)REFRESH_CORRUPTED_FILE;
			}
			fclose(fp);
		}
	}
	else
	{
		log_error("Configuration filePath is NULL");
		s32ReturnValue = (tS32)REFRESH_INVALID_ARGUMENT;
	}
	return s32ReturnValue;
}
/********************************************************************************
 * FUNCTION        : s32Read_ConfigFile
 * PARAMETER       : tPC8 cMetaDataFileDir -> Metadata file dir
 * 					 refresh_config *read_refresh_config -> Information to be written into
 * 					 metadata file
 *
 * RETURNVALUE     : REFRESH_NOERROR || REFRESH_ACCESS_ERROR
 *
 * DESCRIPTION     : This function invokes a function to read refresh_metadata.txt for
 * 					  populating refresh_config strcuture. If unsuccessfull, refresh_config will
 * 					  be initialized with initial values and new refresh_metadata.txt file will be created
 * *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 *------------------------------------------------------------------------
 **********************************************************************************/
static tS32 s32MetaDataFileOperations(tPC8 cMetaDataFileDir, refresh_config *refresh_data_config)
{
	tS32 s32ReturnValue = REFRESH_NOERROR;
	s32ReturnValue = s32CheckAndCreateDir(cMetaDataFileDir);
	if(s32ReturnValue == REFRESH_NOERROR)
	{
		s32ReturnValue = s32CheckAndReadMetaDataFile(cMetaDataFileDir,refresh_data_config);
		if(s32ReturnValue != REFRESH_NOERROR)
		{
			/*Create meta data file - Assuming initial case*/
			refresh_data_config->u32partition_covered = 0;
			refresh_data_config->u32no_of_sectors_covered = 0;
			/*0xDEADDEAD is used as magic so that never this value should be reached*/
			refresh_data_config->u32TotalRefreshCount = REFRESHCOUNT_INITMAGIC;
			tPC8 cMetaDataFilePath = concatPath(cMetaDataFileDir, METADATA_FILE_PATH);
			log_debug("Unable to read data from the Metadata files. So Creating a fresh metaData File : %s",cMetaDataFilePath);
			s32ReturnValue = s32PrepareAndWriteMetaData(cMetaDataFilePath, refresh_data_config);
			if(cMetaDataFilePath != NULL)
				free((tPS8)cMetaDataFilePath);
		}
	}
	else
	{
		log_debug("Unable to find given metadata folder path %s",cMetaDataFileDir);
		s32ReturnValue = (tS32)REFRESH_ACCESS_ERROR;
	}
	return s32ReturnValue;
}
/********************************************************************************
 * FUNCTION        : s32PrepareAndWriteMetaData
 * PARAMETER       : tPC8 chMetaDataFilePath -> Metadata file to be written
 * 					 refresh_config *read_refresh_config -> Information to be written into
 * 					 metadata file
 *
 * RETURNVALUE     : REFRESH_NOERROR || REFRESH_ACCESS_ERROR
 *
 * DESCRIPTION     : This function prepares the current date, due date and other
 * 					  information which will be written to metadata file
 * *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 *------------------------------------------------------------------------
 **********************************************************************************/
static tS32 s32PrepareAndWriteMetaData(tPC8 chMetaDataFilePath, refresh_config *refresh_data_config)
{
	tS32 s32ReturnValue = REFRESH_NOERROR;
	time_t now = time(NULL);
	struct tm *t = localtime(&now);
	refresh_data_config->u32day = (tU32) t->tm_mday;
	refresh_data_config->u32month = (tU32) t->tm_mon + 1;
	refresh_data_config->u32year = (tU32) t->tm_year + 1900;
	if(refresh_data_config->u32due_day == 0)
	{
		refresh_data_config->u32due_day = (tU32) t->tm_mday;
		refresh_data_config->u32due_month = (tU32)  t->tm_mon + 1;
		refresh_data_config->u32due_year = (tU32) t->tm_year + 1900;
	}

	/* 
	* In previous version of refresh due date was calcualted at the end of refresh cycle. From this version
	* due date is calculated during the start of the refresh cycle. Due date will be updated when sectors covered and 
	* partition covered is zero or if there are any time errors.(if due date is some where in the past or much ahead in future).
	*/	

    // If refresh is complete then incrementing totalRefreshCount
	if((refresh_data_config->u32no_of_sectors_covered == 0 ) &&
			(refresh_data_config->u32partition_covered >= refresh_data_config->total_partitions))
	{
		refresh_data_config->u32partition_covered = 0;
		refresh_data_config->u32TotalRefreshCount++;
	}
	vcalculate_checksum(refresh_data_config);
	/* For the backward compatibility we have added this condition. We have added the total refresh count to the end of metadata.txt
	 * in our latest emmc refresh release.
	 */
	if(refresh_data_config->u32TotalRefreshCount == REFRESHCOUNT_INITMAGIC)
	{
		refresh_data_config->u32TotalRefreshCount = 0;
	}
	s32ReturnValue = s32Write_RefreshMetaData(chMetaDataFilePath,refresh_data_config);
	if(s32ReturnValue != REFRESH_NOERROR)
	{
		log_error("Error occurred while writing the metaData file %s ErrorCode : %d ErrorString : %s ",chMetaDataFilePath, errno, strerror(errno));
	}
	return s32ReturnValue;
}
/********************************************************************************
 * FUNCTION        : vcalculate_checksum
 * PARAMETER       : refresh_config *read_refresh_config -> Information to be written into
 * 					 metadata file
 *
 * RETURNVALUE     : Void
 *
 * DESCRIPTION     : Checksum for the data is calculated.By simple method
 *                    We are storing the Hex value to just ensure data integrity
 * *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 *------------------------------------------------------------------------
 *03.DEC.2014| Initialversion 1.0  | SWATHI BOLAR (RBEI/ECF5)
 *------------------------------------------------------------------------
 **********************************************************************************/
static void vcalculate_checksum(refresh_config *refresh_data_config)
{
	refresh_data_config->u32checksum[0] = refresh_data_config->u32due_year;
	refresh_data_config->u32checksum[1] = refresh_data_config->u32due_month;
	refresh_data_config->u32checksum[2] = refresh_data_config->u32due_day;
	refresh_data_config->u32checksum[3] = refresh_data_config->u32year;
	refresh_data_config->u32checksum[4] = refresh_data_config->u32month;
	refresh_data_config->u32checksum[5] = refresh_data_config->u32day;
	refresh_data_config->u32checksum[6] = refresh_data_config->u32partition_covered;
	refresh_data_config->u32checksum[7] = refresh_data_config->u32no_of_sectors_covered;
	return;
}
/********************************************************************************
 * FUNCTION        :  s32Write_RefreshMetaData
 * PARAMETER       :  chMetadataPath -> MetaData FilePath
 * 					  refresh_data_config -> Data to be written to the metadata File
 * RETURNVALUE     :  REFRESH_NOERROR in case of No Error
 * DESCRIPTION     :  Update refresh data to the metadata file in the given chMetadataPath
 * 					 (/chMedataPath/metadata.txt)
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *03.DEC.2014| Initialversion 1.0  | SWATHI BOLAR (RBEI/ECF5)
 * -----------------------------------------------------------------------
 *12.DEC.2014| Version 1.1         | SWATHI BOLAR (RBEI/ECF5)
 * -----------------------------------------------------------------------
 *30.JUN.2016| Version 1.5         | CHAKITHA SARASWATHI (RBEI/ECF5)
 *           |                     | CFG3-1958:Implements refresh for the
 *           |                     | Micron eMMC chips.
 * ----------------------------------------------------------------------
 * 14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 *------------------------------------------------------------------------
 **********************************************************************************/
static tS32 s32Write_RefreshMetaData(tPC8 chMetadataPath ,refresh_config *refresh_data_config)
{
	FILE *fp = NULL;
	tS32 s32ReturnValue = REFRESH_NOERROR;
	//Getting the baseName of the MetaDataFile
	tPS8 chMetaDataFilePathCopy = strdup(chMetadataPath);
	char *baseName = basename(chMetaDataFilePathCopy);
	//Getting the DirName of the MetaDataFile to form MetaData dummy file Path
	char *dirName = dirname(chMetaDataFilePathCopy);
	//Forms the dummy file path
	tPC8 cMetaDataFilePathDummy = concatPath(dirName,METADATA_DUMMY);
	//Forms the actual metadata file path
	tPC8 cMetaDataFilePath = concatPath(dirName,baseName);
	if(cMetaDataFilePathDummy && cMetaDataFilePath)
	{
		fp = fopen(cMetaDataFilePathDummy, "w+");
		if(fp != NULL )
		{
			fprintf(fp,"%u,%u,%u,%u,%u,%u,%u,%u-%x-%x-%x-%x-%x-%x-%x-%x-%x\n", refresh_data_config->u32no_of_sectors_covered,
					refresh_data_config->u32partition_covered,refresh_data_config->u32day, refresh_data_config->u32month, refresh_data_config->u32year,
					refresh_data_config->u32due_day, refresh_data_config->u32due_month, refresh_data_config->u32due_year,
					refresh_data_config->u32checksum[0],refresh_data_config->u32checksum[1],refresh_data_config->u32checksum[2],
					refresh_data_config->u32checksum[3],refresh_data_config->u32checksum[4],refresh_data_config->u32checksum[5],
					refresh_data_config->u32checksum[6],refresh_data_config->u32checksum[7],refresh_data_config->u32TotalRefreshCount);		
			log_debug("%u,%u,%u,%u,%u,%u,%u,%u-%x-%x-%x-%x-%x-%x-%x-%x-%x\n", refresh_data_config->u32no_of_sectors_covered,
					refresh_data_config->u32partition_covered,refresh_data_config->u32day, refresh_data_config->u32month, refresh_data_config->u32year,
					refresh_data_config->u32due_day, refresh_data_config->u32due_month, refresh_data_config->u32due_year,
					refresh_data_config->u32checksum[0],refresh_data_config->u32checksum[1],refresh_data_config->u32checksum[2],
					refresh_data_config->u32checksum[3],refresh_data_config->u32checksum[4],refresh_data_config->u32checksum[5],
					refresh_data_config->u32checksum[6],refresh_data_config->u32checksum[7],refresh_data_config->u32TotalRefreshCount);
			fflush(fp);
			tS32 FilePtr = fileno(fp);
			if (FilePtr == -1)
			{
				log_debug("Unable to get fp for metadata file");
			}
			else
			{
				if( fsync(FilePtr) != 0 )
				{
					log_debug("Fsync failed for the file %s ErrorCode : %d ErrorString %s ",cMetaDataFilePath,errno,strerror(errno));
					//Not returning any error
				}
			}
			fclose(fp);
			s32ReturnValue = rename(cMetaDataFilePathDummy,cMetaDataFilePath);
			if(s32ReturnValue == (tS32) REFRESH_NOERROR)
			{
				fp = fopen(cMetaDataFilePath,"r");
				if(fp != NULL)
				{
					(void)fflush(fp);
					(void)fsync(fileno(fp));
					fclose(fp);
				}
			}
			else
			{
				log_debug("Meta data file %s rename to %s ErrorCode : %d ErrorString %s ", cMetaDataFilePathDummy,cMetaDataFilePath,errno,strerror(errno));
				s32ReturnValue = (tS32) REFRESH_RENAME_ERROR;
			}
		}
		else
		{
			log_debug("Meta data file %s open failed ErrorCode : %d ErrorString : %s ",cMetaDataFilePathDummy, errno, strerror(errno));
			s32ReturnValue = (tS32) REFRESH_OPEN_ERROR;
		}
	}
	if(cMetaDataFilePath)
		free((tPS8)cMetaDataFilePath);
	if(chMetaDataFilePathCopy)
		free(chMetaDataFilePathCopy);
	if(cMetaDataFilePathDummy)
		free((tPS8)cMetaDataFilePathDummy);
	return s32ReturnValue;
}
/********************************************************************************
 * FUNCTION        :  s32CheckAndReadMetaDataFile
 * PARAMETER       :  cMetaDataFilePath -> MetaData FilePath
 * 					  refresh_data_config -> Structure to be populated with metaData
 * 					  which will be read from the file
 * RETURNVALUE     :  REFRESH_NOERROR in case of No Error
 * DESCRIPTION     :  Invokes function to read data from metadata.txt file. If failed
 * 					  again invokes the same function with metadata_backup.txt file path
 * 					  to read the metadata and if successful creates the metadata.txt from
 * 					  the contents read from backup file
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 * 14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 *------------------------------------------------------------------------*/
static tS32 s32CheckAndReadMetaDataFile(tPC8 cMetaDataFilePath, refresh_config *refresh_data_config)
{
	tS32 s32ReturnValue = REFRESH_ACCESS_ERROR;
	//concats cMetaDataFilePath && metadata.txt file
	tPC8 cMetaDataFilePathLocal = concatPath(cMetaDataFilePath,METADATA_FILE_PATH);
	//concats cMetaDataFilePath && metadata_backup.txt file
	tPC8 cMetaDataBackupFilePathLocal = concatPath(cMetaDataFilePath,METADATA_FILE_PATH_BK);
	//Reads the metadata.txt file from the cMetaDataFilePath
	if(cMetaDataFilePathLocal && cMetaDataBackupFilePathLocal)
	{
		s32ReturnValue = s32Read_RefreshMetaData(cMetaDataFilePathLocal,refresh_data_config);
		if(s32ReturnValue != REFRESH_NOERROR)
		{
			//Checks access to backupfile atleast
			if(access(cMetaDataBackupFilePathLocal,F_OK) != -1)
			{
				//Reads data from metadata_backup.txt file
				s32ReturnValue = s32Read_RefreshMetaData(cMetaDataBackupFilePathLocal,refresh_data_config);
				if(s32ReturnValue == REFRESH_NOERROR)
				{
					log_debug("Unable to Read metadata file %s. So Creating it using the data from backup file",cMetaDataFilePathLocal);
					//Creates the metadata.txt file with the contents read from metadata_backup.txt file
					s32ReturnValue = s32PrepareAndWriteMetaData(cMetaDataFilePathLocal, refresh_data_config);
				}
			}
		}
	}
	if(cMetaDataBackupFilePathLocal != NULL)
		free((tPS8)cMetaDataBackupFilePathLocal);
	if(cMetaDataFilePathLocal != NULL)
		free((tPS8)cMetaDataFilePathLocal);
	return s32ReturnValue;
}
/********************************************************************************
 * FUNCTION        :  s32calculate_checksum
 * PARAMETER       :  peMMCRefreshdata - Refresh metadata content
 * RETURNVALUE     :  CRC Success/Failure
 * DESCRIPTION     :  Checksum for the data is Validated.By simple method
 *                    We are validating the stored Hex value to just ensure data integrity
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *19.SEP.2016| Initialversion 1.0  | SELVAKUMAR KALIMUTHU (RBEI/ECF2)
 *------------------------------------------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 * -----------------------------------------------------------------------
 **********************************************************************************/
static tS32 s32ValidateChecksum( const  refresh_config *peMMCRefreshdata)
{
	tS32 s32ReturnValue = (tS32)REFRESH_NOERROR;
	if( ( peMMCRefreshdata->u32checksum[0] != peMMCRefreshdata->u32due_year) ||
			( peMMCRefreshdata->u32checksum[1] != peMMCRefreshdata->u32due_month ) ||
			( peMMCRefreshdata->u32checksum[2] != peMMCRefreshdata->u32due_day ) ||
			( peMMCRefreshdata->u32checksum[3] != peMMCRefreshdata->u32year ) ||
			( peMMCRefreshdata->u32checksum[4] != peMMCRefreshdata->u32month ) ||
			( peMMCRefreshdata->u32checksum[5] != peMMCRefreshdata->u32day ) ||
			( peMMCRefreshdata->u32checksum[6] != peMMCRefreshdata->u32partition_covered ) ||
			( peMMCRefreshdata->u32checksum[7] != peMMCRefreshdata->u32no_of_sectors_covered ) )
	{
		s32ReturnValue = (tS32)REFRESH_CORRUPTED_FILE;
	}
	return s32ReturnValue;
}
/********************************************************************************
 * FUNCTION        :  s32Read_RefreshMetaData
 * PARAMETER       :  chMetadataPath - Metadata Path
 * 					  refresh_data_config - Metadata contents
 * RETURNVALUE     :  REFRESH_NOERROR in case of no errors
 * DESCRIPTION     :  Read Refresh meta data file and populates the
 *					  refresh_data_config structure
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *03.DEC.2014| Initialversion 1.0  | SWATHI BOLAR (RBEI/ECF5)
 * -----------------------------------------------------------------------
 *12.DEC.2014| Version 1.1         | SWATHI BOLAR (RBEI/ECF5)
 * -----------------------------------------------------------------------
 *30.JUN.2016| Version 1.5         | CHAKITHA SARASWATHI (RBEI/ECF5)
 *           |                     | CFG3-1958:Implements refresh for the
 *           |                     | Micron eMMC chips.
 * -----------------------------------------------------------------------
 *20.SEP.2016| Version 1.9         | SELVAKUMAR KALIMUTHU (RBEI/ECF2)
 *           |                     | Validate Checksum of medatadaa's
 *------------------------------------------------------------------------
 *14.Aug.2019| Version 2.0         | Ragupathi Palanisamy (RBEI/ECF2)
 **********************************************************************************/
static tS32 s32Read_RefreshMetaData(tPC8 cMetaDataFilePath,refresh_config *refresh_data_config)
{
	FILE *fp;tS64 l_size;
	tS32 s32ReturnValue = REFRESH_NOERROR;
	tS32 s32Readcount = 0;
	tS8  s8Token = 0;
	fp = fopen(cMetaDataFilePath,"r");
	if(fp != NULL)
	{
		if(fseek(fp,0,SEEK_END) !=-1)
		{
			l_size = ftell(fp);
			if((l_size >= 30 ) && (l_size <= 130 ))
			{
				fseek(fp,0,SEEK_SET);
				//reads the due date and other information from metadata file
				(void)fscanf(fp,"%u,%u,%u,%u,%u,%u,%u,%u\n", &refresh_data_config->u32no_of_sectors_covered,
						&refresh_data_config->u32partition_covered,&refresh_data_config->u32day, &refresh_data_config->u32month,
						&refresh_data_config->u32year, &refresh_data_config->u32due_day, &refresh_data_config->u32due_month,
						&refresh_data_config->u32due_year);
				/* Reading Checksum value */
				for(s32Readcount=0; s32Readcount < CHECKSUM_COUNT; s32Readcount++)
				{
					(void)fscanf(fp,"-%x",&refresh_data_config->u32checksum[s32Readcount]);
				}
				(void)fscanf(fp,"%c",&s8Token);
				if(s8Token != '-')
				{
					refresh_data_config->u32TotalRefreshCount = REFRESHCOUNT_INITMAGIC;
				}
				else
				{
					(void)fscanf(fp,"%x",&refresh_data_config->u32TotalRefreshCount);
				}
				if(REFRESH_NOERROR != s32ValidateChecksum(refresh_data_config))
				{
					log_error("Checksum Error Metadata file seems to be corrupted %s ",cMetaDataFilePath);
					//Closing the file before removing it.
					fclose(fp);
					vRemoveFile(cMetaDataFilePath);
					s32ReturnValue = (tS32)REFRESH_CORRUPTED_FILE;
				}
				else
				{
					//Closing the file at no error scenario
					fclose(fp);
				}
			}
			else
			{
				log_debug("Corrupted MetaData File %s FileSize : %lld",cMetaDataFilePath,l_size);
				fclose(fp);
				vRemoveFile(cMetaDataFilePath);
				s32ReturnValue = (tS32) REFRESH_INVALID_FILESIZE;
			}
		}
		else
		{
			//logging to debug because even when we do not have this file, we will create this.
			s32ReturnValue = (tS32) REFRESH_SEEK_ERROR;
			fclose(fp);
			log_debug("fseek failed for the file : %s ErrorCode : %d ErrorString : %s ",cMetaDataFilePath,errno, strerror(errno));
		}
	}
	else
	{
		log_warn("File Open failed : %s ErrorCode : %d ErrorString : %s ",cMetaDataFilePath,errno, strerror(errno));
		s32ReturnValue = (tS32) REFRESH_ACCESS_ERROR;
	}
	return s32ReturnValue;
}
/********************************************************************************
 * FUNCTION        : ptr_align
 * PARAMETER       : ptr            -     buffer
 *                   alignment      -     page size
 * RETURNVALUE     : pointer after alignment
 * DESCRIPTION     : This function is needed for the page alignment.
 *------------------------------------------------------------------------
 * Date      |       Version       | Author & comments
 *-----------|---------------------|--------------------------------------
 *03.DEC.2014| Initialversion 1.0  | SWATHI BOLAR (RBEI/ECF5)
 * -----------------------------------------------------------------------
 **********************************************************************************/
static void *ptr_align (void *ptr, tU32 alignment)
{
	uint8_t *p0 = (uint8_t*)ptr;
	uint32_t p1 = (uint32_t)(uintptr_t)p0;
	p1 = p1 % alignment;
	if(p1 != 0 )
	{
		p0 = (p0 + (alignment - p1));
	}
	return p0;
}
