/******************************************************************************
*FILE         : dev_acc_TestFunctions.cpp
*
*SW-COMPONENT : Google Testing framework
*
*DESCRIPTION  : This file implements the Google@target test cases for the ACC
*               device.
*
*AUTHOR       : Abhimanyu Pandit (RBEI/ECF12)
*
*COPYRIGHT    : (C) COPYRIGHT RBEI - All Rights Reserved
******************************************************************************/
#include "dev_acc_TestFunctions.h"

using namespace testing;

/****************************************************************
Function prototypes
*****************************************************************/
static void ACCOedt_vPlotdata(const OSAL_trIOCtrlAccData*);
static tU32 u32AccTimestampValidationCallback(tU32);

class AccTestFunctions : public ::testing::Test {
   protected :
   OSAL_tIODescriptor hDevAcc;
   OSAL_tIODescriptor hDevGyro;
   OSAL_tIODescriptor hDevOdo;

   public:
   virtual void SetUp()
   {
	  hDevGyro = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GYRO, OSAL_EN_READONLY );
	  hDevOdo = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ODOMETER, OSAL_EN_READONLY );
   }

   virtual void TearDown()
   {
      OSAL_s32IOClose( hDevGyro );
      OSAL_s32IOClose( hDevOdo );

      if(hDevAcc != OSAL_ERROR )
      {
         OSAL_s32IOClose ( hDevAcc );
      }
   }

   tU32 u32AccCheckACCValuesSeq(char);

};

/***************************************************************************
||FUNCTION    :  ACCOedt_vPlotdata()
||
||DESCRIPTION :  Plots the acc R-S-T data graphically on terminal
||
||PARAMETER   :  prIOCtrlAccData
||
||RETURNVALUE :  tU32 error code
||
***************************************************************************/
void ACCOedt_vPlotdata(const OSAL_trIOCtrlAccData *prIOCtrlAccData)
{
   tU8 au8PlotData[ACC_OEDT_TOTAL_PLOT_WIDTH + 1];
   tU16 u16X_Index, u16Y_Index, u16Z_Index;

   if (OSAL_NULL != prIOCtrlAccData)
   {
      OSAL_pvMemorySet( au8PlotData, '_', ACC_OEDT_TOTAL_PLOT_WIDTH);
      u16X_Index = (tU16)(prIOCtrlAccData->u16Acc_x / ACC_OEDT_DATA_SCALE_FACTOR);
      u16Y_Index = (tU16)(prIOCtrlAccData->u16Acc_y / ACC_OEDT_DATA_SCALE_FACTOR);
      u16Z_Index = (tU16)(prIOCtrlAccData->u16Acc_z / ACC_OEDT_DATA_SCALE_FACTOR);

      au8PlotData[u16X_Index + ACC_OEDT_X_PLOT_OFFSET] = 'R';
      au8PlotData[u16Y_Index + ACC_OEDT_Y_PLOT_OFFSET] = 'S';
      au8PlotData[u16Z_Index + ACC_OEDT_Z_PLOT_OFFSET] = 'T';

      au8PlotData[ACC_OEDT_TOTAL_PLOT_WIDTH] = '\0';
      std::cout<< " "<< au8PlotData<<"\n";
   }
}

/***************************************************************************
||FUNCTION    :  u32AccCheckACCValuesSeq()
||
||DESCRIPTION :  Reads 100 data records from the driver,
||               Checks the variation in axis data is within 0.5%
||               of full scale when the target is still(not moving).
||
||PARAMETER   :  cAxis
||
||RETURNVALUE :  tU32 error code
||
***************************************************************************/
tU32 AccTestFunctions::u32AccCheckACCValuesSeq(char cAxis)
{
   OSAL_trIOCtrlAccData xData = {0,0,0,0,0};
   tU32 u32NoOfRecsForRead = 100;
   tU8  u8Index = 0;
   tU16 au16AccReadVal_xyz[100] = {0};
   tU32 u32LowVal = 0;
   tU32 u32HighVal = 0;
   tU32 u32Sum = 0;
   tU32 u32AvgVal = 0;
   tU8  u8PercentageOfRange = 5;
   tU16 u16Acc_Mid_z_Value_High = 0;
   tU16 u16Acc_Mid_z_Value_Low = 0;
   tU16 u16BaseAvgVal = 0;
   tU32 u32RetVal = ACC_ERROR_NONE;

   for( u8Index= 0; ( u8Index < u32NoOfRecsForRead ) && ( u32RetVal != ACC_ERROR_IOREAD ); u8Index++)
   {
      if( OSAL_ERROR != OSAL_s32IORead( hDevAcc, (tPS8 )&xData,sizeof(xData)) )
      {
         if(cAxis == 'x')
            au16AccReadVal_xyz[u8Index] = xData.u16Acc_x;
         else if(cAxis == 'y')
            au16AccReadVal_xyz[u8Index] = xData.u16Acc_y;
         else if(cAxis == 'z')
            au16AccReadVal_xyz[u8Index] = xData.u16Acc_z;
      }

      else
      {
         u32RetVal = ACC_ERROR_IOREAD;
      }
   }

   if( u32RetVal == ACC_ERROR_NONE )
   {
      for(u8Index= 0; u8Index < u32NoOfRecsForRead; u8Index++)
      {
         u32Sum  +=  au16AccReadVal_xyz[u8Index];
      }

      //Calculate the 5 % of average value
      u32AvgVal = (u32Sum / u32NoOfRecsForRead);
      u16BaseAvgVal = ( tU16 )( (u32AvgVal * 5)/100);
      u16Acc_Mid_z_Value_High = ( tU16 )( u32AvgVal + u16BaseAvgVal );
      u16Acc_Mid_z_Value_Low = ( tU16 )( u32AvgVal - u16BaseAvgVal );
      u32LowVal = u32AvgVal;

      for( u8Index= 0; u8Index < u32NoOfRecsForRead; u8Index++)
      {
         if(au16AccReadVal_xyz[u8Index] > u32HighVal)
            u32HighVal = au16AccReadVal_xyz[u8Index];
         if(au16AccReadVal_xyz[u8Index] <  u32LowVal)
            u32LowVal = au16AccReadVal_xyz[u8Index];
      }

      if( (u32HighVal - u32LowVal) > ((u8PercentageOfRange*u32AvgVal)/100))
      {
         u32RetVal=ACC_ERROR_VALUE_RANGE;
      }

   }

   if( u32RetVal == ACC_ERROR_NONE )
   {
      if( (u32AvgVal < u16Acc_Mid_z_Value_Low) || (u32AvgVal > u16Acc_Mid_z_Value_High) )
      {
         u32RetVal=ACC_ERROR_MID_VALUE;
      }
   }

   return u32RetVal;
}

/***************************************************************************
||FUNCTION    :  u32AccTimestampValidationCallback()
||
||DESCRIPTION :  Validates if the acc data timestamp difference is 50ms
||
||PARAMETER   :  u32SamplesLogged
||
||RETURNVALUE :  tU32 error code
||
***************************************************************************/
tU32 u32AccTimestampValidationCallback(tU32 u32SamplesLogged)
{
   tU32 u32PreviousTimeStamp;
   tU32 u32CurrentTimeStamp;
   tU32 u32ReturnValue = ACC_ERROR_NONE;
   tS32 s32TimeDiff;
   FILE *fDescriptor1;
   tU32 u32ReadBuffer;
   tU32 u32NumOfTimestampErrors = 0;

   if((fDescriptor1 = fopen(ACC_TIMESTAMP_RECORD, "r")) != NULL)    //open the file in read mode
   {
	  if ( 1 == fread(&u32ReadBuffer,sizeof(u32ReadBuffer),1,fDescriptor1))    //read from the file
      {
         u32CurrentTimeStamp =  u32ReadBuffer;
         u32SamplesLogged--;

         while(u32SamplesLogged > 0)
         {
        	 if ( 1 == fread(&u32ReadBuffer,sizeof(u32ReadBuffer),1,fDescriptor1))
            {
               u32PreviousTimeStamp = u32CurrentTimeStamp;
               u32CurrentTimeStamp = u32ReadBuffer;
               s32TimeDiff = u32CurrentTimeStamp - u32PreviousTimeStamp;
               u32SamplesLogged--;

               if(s32TimeDiff >= ACC_TIMESTAMP_DIFFERENCE_LOWER_LIMIT  && s32TimeDiff <= ACC_TIMESTAMP_DIFFERENCE_UPPER_LIMIT)
               {
                  u32ReturnValue |= ACC_ERROR_NONE;
               }

               else
               {
                  u32NumOfTimestampErrors++;
                  u32ReturnValue |= ACC_ERROR_TIMESTAMP;
               }
            }
         }
      }
	  else
	  {
		 u32ReturnValue = ACC_ERROR_TIMESTAMP_RECORD_READ;
	  }

	  fclose(fDescriptor1);
   }
   else
   {
      u32ReturnValue = ACC_ERROR_TIMESTAMP_RECORD_OPEN;
   }
   return u32ReturnValue;
 }

/*****************************************************************
* TEST CASE   : Acc_IOOpenClose_Test
* DESCRIPTION : Tests device open and close.
******************************************************************/
TEST_F(AccTestFunctions, Acc_IOOpenClose_Test)
{
   ASSERT_NE( OSAL_ERROR, hDevAcc = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READWRITE ) )
         << "AccOpen Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOClose( hDevAcc ) )
         << "AccClose Fail, OSAL Error code :"<< OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Acc_IOReOpen_Test
* DESCRIPTION : Tests device re-open. Driver should not allow re-open
******************************************************************/
TEST_F(AccTestFunctions, Acc_IOReOpen_Test)
{
   OSAL_tIODescriptor hDevAcc2;

   ASSERT_NE( OSAL_ERROR, hDevAcc = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READWRITE ) )
         << "AccOpen Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_EQ( OSAL_ERROR, hDevAcc2 = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READWRITE ) )
         << "AccReOpen Test Fail, OSAL Error code :" << OSAL_u32ErrorCode(),OSAL_s32IOClose( hDevAcc2 );

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOClose( hDevAcc ) )
         << "AccClose Fail, OSAL Error code :"<< OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Acc_IOCloseAlreadyClosed_Test
* DESCRIPTION : Tests device re-close. Driver should not allow re-close
******************************************************************/
TEST_F(AccTestFunctions, Acc_IOCloseAlreadyClosed_Test)
{
   ASSERT_NE( OSAL_ERROR, hDevAcc = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READWRITE ) )
          << "AccOpen Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOClose( hDevAcc ) )
          << "AccClose Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_EQ( OSAL_ERROR, OSAL_s32IOClose( hDevAcc ) )
          << "AccIOReclose Test Failed, OSAL Error code :"<< OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Acc_IOOpenCloseRepeatedly_Test
* DESCRIPTION : Tests driver response for repeated open and close calls.
                Full project autosar must be flashed
******************************************************************/
TEST_F(AccTestFunctions, Acc_IOOpenCloseRepeatedly_Test)
{
   tU8 u8Index = 0;

   while(( u8Index < 20 ) && (hDevAcc != OSAL_ERROR))
   {
      ASSERT_NE( OSAL_ERROR, hDevAcc = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READWRITE ) )
             << "AccOpen Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

      ASSERT_NE( OSAL_ERROR, OSAL_s32IOClose( hDevAcc ) )
             << "AccClose Fail, OSAL Error code :"<<OSAL_u32ErrorCode();

      u8Index++;
   }
}

/*****************************************************************
* TEST CASE   : Acc_Self_Test
* DESCRIPTION : Tests Acc self test functionality.
******************************************************************/
TEST_F(AccTestFunctions, Acc_Self_Test)
{
   tS32 s32SelfTestResult = -1;

   ASSERT_NE( OSAL_ERROR, hDevAcc = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READWRITE ) )
          << "AccOpen Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOControl (hDevAcc,OSAL_C_S32_IOCTRL_ACC_SELF_TEST,(tLong) &s32SelfTestResult) )
          << "AccSelfTest Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_EQ(1,s32SelfTestResult) << "GyroSelfTest Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOClose( hDevAcc ) )
          << "AccClose Fail, OSAL Error code :" << OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Acc_GetCount_Test
* DESCRIPTION : Tests Acc get count functionality.
******************************************************************/
TEST_F(AccTestFunctions, Acc_GetCount_Test)
{
   tU32 u32NumOfSamplesInBuff = 0;

   ASSERT_NE( OSAL_ERROR, hDevAcc = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READWRITE ) )
          << "AccOpen Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   OSAL_s32ThreadWait(500);

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOControl( hDevAcc, OSAL_C_S32_IOCTRL_ACC_GETCNT,(tLong)&u32NumOfSamplesInBuff ) )
          << "AccGetCount Test Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevAcc ) )
          << "AccClose Fail, OSAL Error code :"<< OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Acc_GetCycleTime_Test
* DESCRIPTION : Tests Acc get cycle time functionality.
******************************************************************/
TEST_F(AccTestFunctions, Acc_GetCycleTime_Test)
{
   tU32 u32AccCycleTimeInNS;

   ASSERT_NE( OSAL_ERROR, hDevAcc = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READWRITE ) )
          << "AccOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOControl( hDevAcc, OSAL_C_S32_IOCTRL_ACC_GETCYCLETIME,(tLong)&u32AccCycleTimeInNS ) )
          << "AccGetCycleTime Test Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_EQ( u32AccCycleTimeInNS, ACC_DEFAULT_TIME_INTERVAL )
          << "ACC_CYCLE_TIME_VALUE_MISMATCH_ERROR"; //lint !e774

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevAcc ) )
          << "AccClose Fail, OSAL Error code :"<< OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Acc_BasicRead_Test
* DESCRIPTION : Checks the number of records available in the buffer
                and reads them.
******************************************************************/
TEST_F(AccTestFunctions, Acc_BasicRead_Test)
{
   tU32 u32NumOfSamplesInBuff = 0;
   tU32 u32RecSize = sizeof(OSAL_trIOCtrlAccData);
   OSAL_trIOCtrlAccData *DataBufferPointer = NULL;
   OSAL_trIOCtrlAccData *NullPointer = NULL;
   tS32 s32AccDataRead;

   ASSERT_NE( OSAL_ERROR, hDevAcc = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READWRITE ) )
          << "AccOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   OSAL_s32ThreadWait(500);

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOControl( hDevAcc, OSAL_C_S32_IOCTRL_ACC_GETCNT,(tLong)&u32NumOfSamplesInBuff ) )
          << "AccGetCount Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   ASSERT_NE( NullPointer , DataBufferPointer = (OSAL_trIOCtrlAccData*)OSAL_pvMemoryAllocate( u32NumOfSamplesInBuff * u32RecSize ) )
          << "AccMemAlloc Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, s32AccDataRead = OSAL_s32IORead( hDevAcc, (tPS8)DataBufferPointer, u32NumOfSamplesInBuff * u32RecSize ) )
          << "AccRead Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_EQ( u32NumOfSamplesInBuff*u32RecSize, s32AccDataRead )
          << "Read lesser records than available" ;

   OSAL_vMemoryFree(DataBufferPointer);

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevAcc ) )
          << "AccClose Fail, OSAL Error code :"<< OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Acc_ReadPassingNullBuffer_Test
* DESCRIPTION : Tests Acc driver response for the read functionality
                by passing null buffer.
******************************************************************/
TEST_F(AccTestFunctions, Acc_ReadPassingNullBuffer_Test)
{
   ASSERT_NE( OSAL_ERROR, hDevAcc = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READWRITE ) )
          << "AccOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_EQ( OSAL_ERROR, OSAL_s32IORead( hDevAcc, OSAL_NULL, 1 ) )
          << "AccReadPassingNullBuffer Fail, OSAL Error code :"<<OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevAcc ) )
          << "AccClose Fail, OSAL Error code :"<< OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Acc_IOCRTLPassingNullBuffer_Test
* DESCRIPTION : Tests Acc driver response for IO control functionality
                by passing null buffer.
******************************************************************/
TEST_F(AccTestFunctions, Acc_IOCRTLPassingNullBuffer_Test)
{
   ASSERT_NE( OSAL_ERROR, hDevAcc = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READWRITE ) )
          << "AccOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_EQ( OSAL_ERROR, OSAL_s32IOControl( hDevAcc, OSAL_C_S32_IOCTRL_VERSION,(tLong)NULL ) )
          << "AccDriverVersionTest, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_EQ( OSAL_ERROR, OSAL_s32IOControl( hDevAcc, OSAL_C_S32_IOCTRL_ACC_GETCNT,(tLong)NULL ) )
          << "AccGetCountTest Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_EQ( OSAL_ERROR, OSAL_s32IOControl( hDevAcc, OSAL_C_S32_IOCTRL_ACC_GETCYCLETIME,(tLong)NULL ) )
          << "AccGetCycleTimeTest  Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevAcc ) )
          << "AccClose Fail, OSAL Error code :" << OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Acc_InterfaceCheckSeq_Test
* DESCRIPTION : Tests all the IO Control interface provided by the
                Acc Driver.
******************************************************************/
TEST_F(AccTestFunctions, Acc_InterfaceCheckSeq_Test)
{
   OSAL_trIOCtrlAccData rData;
   tU32 u32Version, u32CycleTimeInNs, u32NumOfSamplesInBuff;

   ASSERT_NE( OSAL_ERROR, hDevAcc = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READWRITE ) )
          << "AccOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOControl( hDevAcc, OSAL_C_S32_IOCTRL_VERSION,(tLong)&u32Version ) )
          << "AccDriverVersionTest, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOControl( hDevAcc, OSAL_C_S32_IOCTRL_ACC_GETCYCLETIME,(tLong)&u32CycleTimeInNs ) )
          << "AccGetCycleTime Fail, OSAL Error code :"<<OSAL_u32ErrorCode();

   OSAL_s32ThreadWait(500);

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOControl( hDevAcc, OSAL_C_S32_IOCTRL_ACC_GETCNT,(tLong)&u32NumOfSamplesInBuff ) )
          << "AccGetCount Fail, OSAL Error code :"<<OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, OSAL_s32IORead( hDevAcc, (tPS8) &rData, sizeof(rData) ) )
          << "AccRead Fail-1, OSAL Error code :"<< OSAL_u32ErrorCode();//lint !e774

   EXPECT_NE( OSAL_ERROR, OSAL_s32IORead( hDevAcc, (tPS8) &rData, sizeof(rData) ) )
          << "AccRead Fail-2, OSAL Error code :" << OSAL_u32ErrorCode();//lint !e774

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevAcc ) )
          << "AccClose Fail, OSAL Error code :" << OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Acc_GetHwInfo_Test
* DESCRIPTION : Tests Acc get hardware info functionality.
******************************************************************/
TEST_F(AccTestFunctions, Acc_GetHwInfo_Test)
{
   OSAL_trIOCtrlHwInfo sAccHwInfo;

   ASSERT_NE( OSAL_ERROR, hDevAcc = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READWRITE ) )
          << "AccOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, OSAL_s32IOControl( hDevAcc, OSAL_C_S32_IOCTRL_ACC_GET_HW_INFO,(tLong)&sAccHwInfo ) )
          << "AccGetHwInfoTest Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevAcc ) )
          << "AccClose Fail, OSAL Error code :" << OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : AccValuesSeq_x_Test
* DESCRIPTION : Reads 100 data records from the driver,
                Checks the variation in X-axis data is within 0.5%
                of full scale when the target is still(not moving).
******************************************************************/
TEST_F(AccTestFunctions, AccValuesSeq_x_Test)
{
   ASSERT_NE( OSAL_ERROR, hDevAcc = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READWRITE ) )
          << "AccOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_EQ( ACC_ERROR_NONE, u32AccCheckACCValuesSeq('x') )
          << "AccValuesSeq_x Fail,OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevAcc ) )
          << "AccClose Fail, OSAL Error code :" << OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : AccValuesSeq_y_Test
* DESCRIPTION : Reads 100 data records from the driver,
                Checks the variation in Y-axis data is within 0.5%
                of full scale when the target is still(not moving).
******************************************************************/
TEST_F(AccTestFunctions, AccValuesSeq_y_Test)
{
   ASSERT_NE( OSAL_ERROR, hDevAcc = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READWRITE ) )
          << "AccOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_EQ( ACC_ERROR_NONE,u32AccCheckACCValuesSeq('y') )
          << "AccValuesSeq_y Fail,OSAL Error code :"<< OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevAcc ) )
          << "AccClose Fail, OSAL Error code :" << OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : AccValuesSeq_z_Test
* DESCRIPTION : Reads 100 data records from the driver,
                Checks the variation in Z-axis data is within 0.5%
                of full scale when the target is still(not moving).
******************************************************************/
TEST_F(AccTestFunctions, AccValuesSeq_z_Test)
{
   ASSERT_NE( OSAL_ERROR, hDevAcc = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READWRITE ) )
          << "AccOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_EQ( ACC_ERROR_NONE,u32AccCheckACCValuesSeq('z') )
          << "AccValuesSeq_z Fail,OSAL Error code :" << OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevAcc ) )
          << "AccClose Fail, OSAL Error code :" << OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : ACCOedt_u32ValuesPlot_Test
* DESCRIPTION : Plot acc data on console . This test is disabled.
******************************************************************/
TEST_F(AccTestFunctions, ACCOedt_u32ValuesPlot)
{
   OSAL_trIOCtrlAccData rIOCtrlAccData;
   tBool bTestEndFlag = FALSE;
   tU32 u32AccPrevPlotTS = 0;
   tU32 u32TestEndTime = OSAL_ClockGetElapsedTime() + ACC_OEDT_TEST_DURATION_MS;

   ASSERT_NE( OSAL_ERROR, hDevAcc = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READWRITE ) )
          << "AccOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   while ( TRUE != bTestEndFlag )
   {
      if( OSAL_ERROR != OSAL_s32IORead( hDevAcc,(tPS8) &rIOCtrlAccData,sizeof(rIOCtrlAccData)) )
      {
         if ( (rIOCtrlAccData.u32TimeStamp - u32AccPrevPlotTS) > ACC_OEDT_PLOT_DATA_INTERVALL )
         {
            ACCOedt_vPlotdata(&rIOCtrlAccData);
            u32AccPrevPlotTS = rIOCtrlAccData.u32TimeStamp;
         }
      }

      if ( OSAL_ClockGetElapsedTime() > u32TestEndTime )
      {
         bTestEndFlag = TRUE;
      }
   }

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevAcc ) )
          << "AccClose Fail, OSAL Error code :" << OSAL_u32ErrorCode();
}

/*****************************************************************
* TEST CASE   : Acc_TimestampValidation_Test
* DESCRIPTION : Checks whether the timestamp differences between consecutive
                Acc data is in between ACC_TIMESTAMP_DIFFERENCE_LOWER_LIMIT
                and ACC_TIMESTAMP_DIFFERENCE_UPPER_LIMIT, for 5 minuntes.
******************************************************************/
TEST_F(AccTestFunctions, Acc_TimestampValidation_Test)
{
   tU32 u32NumOfSamplesLogged = 0,u32NumOfSamplesInBuff;
   tBool bTimeoutFlag = FALSE;
   OSAL_trIOCtrlAccData DataArray[MAX_SAMPLES_PER_READ];
   tU32 u32Index;
   FILE *fDescriptor1 = NULL ;
   FILE *u32NullPtr = NULL;
   tU32 u32TestEndTime = OSAL_ClockGetElapsedTime() + ACC_SAMPLE_READ_TIME_IN_MS;

   ASSERT_NE( OSAL_ERROR, hDevAcc = OSAL_IOOpen( OSAL_C_STRING_DEVICE_ACC, OSAL_EN_READWRITE ) )
          << "AccOpen Fail, OSAL Error code :" << OSAL_u32ErrorCode();

   ASSERT_NE( u32NullPtr,( fDescriptor1 = fopen(ACC_TIMESTAMP_RECORD, "w+")))
          << "ACC_TIMESTAMP_RECORD Open Fail, OSAL Error code :" << OSAL_u32ErrorCode();        //open the file in write/append mode

   std::cout<<"this test runs for 5mins\n";

   /* discarding initial chunk of data to avoid large gap in timestamps at the beginning of test*/
   OSAL_s32ThreadWait(2000);
   EXPECT_NE( OSAL_ERROR, OSAL_s32IOControl( hDevAcc, OSAL_C_S32_IOCTRL_ACC_GETCNT,(tLong)&u32NumOfSamplesInBuff ) )
          << "AccGetCount Fail, OSAL Error code :"<<OSAL_u32ErrorCode();

   EXPECT_NE( OSAL_ERROR, OSAL_s32IORead( hDevAcc, (tPS8)DataArray, (tU32)(u32NumOfSamplesInBuff * sizeof(OSAL_trIOCtrl3dGyroData))))
          << "AccRead Fail, OSAL Error code :"<< OSAL_u32ErrorCode();

   /* begin reading the data for timestamp validation*/
   OSAL_s32ThreadWait(500);
   while(!bTimeoutFlag)
   {
      if( OSAL_ERROR != OSAL_s32IOControl( hDevAcc, OSAL_C_S32_IOCTRL_ACC_GETCNT,(tLong)&u32NumOfSamplesInBuff ) )
      {
         if( u32NumOfSamplesInBuff > 0 )
         {
            if(OSAL_ERROR != OSAL_s32IORead( hDevAcc, (tPS8)DataArray, (tU32)(u32NumOfSamplesInBuff * sizeof(OSAL_trIOCtrl3dGyroData)) ) )
            {
               for(u32Index = 0; u32Index < u32NumOfSamplesInBuff;  u32Index++)
               {
                  if(1 == fwrite(&DataArray[u32Index].u32TimeStamp, sizeof(tU32),1,fDescriptor1) )
                     u32NumOfSamplesLogged++;
               }
            }
         }
      }
      else
      {
         OSAL_s32ThreadWait(250);
      }

      if ( OSAL_ClockGetElapsedTime() > u32TestEndTime )
      {
         bTimeoutFlag = TRUE;
      }

      u32NumOfSamplesInBuff = 0;
   }

   EXPECT_NE( OSAL_ERROR,  OSAL_s32IOClose( hDevAcc ) )
          << "AccClose Fail, OSAL Error code :" << OSAL_u32ErrorCode();
   EXPECT_EQ(0,fclose(fDescriptor1))
          << "AccLog file close Fail, OSAL Error code :"<<OSAL_u32ErrorCode();  //close the file

   EXPECT_EQ( ACC_ERROR_NONE, u32AccTimestampValidationCallback(u32NumOfSamplesLogged))
          << "Acc timestamp validationCallback Fail, OSAL Error code :"<<OSAL_u32ErrorCode();
}

//EOF
