/*******************************************************************************
*
* FILE:         dev_volt_sharedlibentry.h
*
* SW-COMPONENT: Device Voltage
*
* PROJECT:      ADIT Gen3 Platform
*
* DESCRIPTION:  Process attach and detach functions to let the /dev/volt
*               operate as a shared library.
*
* AUTHOR:       CM-AI/ECO3-Kalms
*
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
*
*******************************************************************************/

/******************************************************************************/
/*                                                                            */
/* INCLUDES                                                                   */
/*                                                                            */
/******************************************************************************/

#include "OsalConf.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "Linux_osal.h"

#define DEV_VOLT_IMPORT_INTERFACE_OSAL
#include "dev_volt_if.h"

/******************************************************************************/
/*                                                                            */
/* DEFINES                                                                    */
/*                                                                            */
/******************************************************************************/

#define DEV_VOLT_C_U8_TRACE_TYPE_STRING                                     0x01

#define DEV_VOLT_C_U8_TRACE_SEND_BUFFER_LENGTH                               128

/******************************************************************************/
/*                                                                            */
/* GLOBAL VARIABLES                                                           */
/*                                                                            */
/******************************************************************************/

static sem_t* g_hSemLock = SEM_FAILED;
static tS32   g_hShMem = -1;
static tS32*  g_ps32CreatorProcessId = MAP_FAILED;

/******************************************************************************/
/*                                                                            */
/* LOCAL FUNCTIONS                                                            */
/*                                                                            */
/******************************************************************************/

/*******************************************************************************
*
* Shared library attach function which is called by OSAL for each process at
* their first call of an OSAL_IOOpen() for /dev/volt.
*
* Call init function of ...
*
*   - DEV_VOLT_OsalIO_u32Init()
*
* ... if this is the FIRST process which tries to open this device.
*
* Remember the ID of this so called creator-process in the shared memory to
* be used when this library is detached again.
*
*******************************************************************************/
void __attribute__ ((constructor)) volt_process_attach(void)
{
	tU8 au8TraceSendBuffer[DEV_VOLT_C_U8_TRACE_SEND_BUFFER_LENGTH];

	tBool bIsCreatorProcess = FALSE;

	OSAL_trProcessControlBlock rProcessControlBlock;

	rProcessControlBlock.szName = NULL;
	rProcessControlBlock.id     = 0;

	OSAL_s32ProcessControlBlock(OSAL_ProcessWhoAmI(), &rProcessControlBlock);

	g_hSemLock = sem_open("VOLT_INIT",O_EXCL | O_CREAT, 0660, 0);

	if (g_hSemLock != SEM_FAILED) {
		if(s32OsalGroupId) {
			if(chown("/dev/shm/sem.VOLT_INIT",(uid_t)-1,(gid_t)s32OsalGroupId) == -1)
				vWritePrintfErrmem("volt_process_attach  -> chown error %d \n",errno);

			if(chmod("/dev/shm/sem.VOLT_INIT", OSAL_ACCESS_RIGTHS) == -1)
				vWritePrintfErrmem("volt_process_attach chmod -> fchmod error %d \n",errno);
		}
		
		g_hShMem = shm_open("VOLT_INIT", O_EXCL|O_RDWR|O_CREAT|O_TRUNC, OSAL_ACCESS_RIGTHS);

		if (g_hShMem == -1) {
			FATAL_M_ASSERT_ALWAYS();
			goto error_out; /*lint !e801, authorized LINT-deactivation #<71> */
		}

		if (s32OsalGroupId) {
			if (fchown(g_hShMem, (uid_t)-1, (gid_t)s32OsalGroupId) == -1)
				vWritePrintfErrmem("Attach shm_open -> fchown error %d",errno);
			// umask (022) is overwriting the permissions on creation, so we have to set it again
			if (fchmod(g_hShMem, OSAL_ACCESS_RIGTHS) == -1)
				vWritePrintfErrmem("Attach shm_open -> fchmod error %d",errno);
		}

		if (ftruncate(g_hShMem, sizeof(tU32)) == -1) {
			FATAL_M_ASSERT_ALWAYS();
			goto error_out; /*lint !e801, authorized LINT-deactivation #<71> */
		}

		g_ps32CreatorProcessId = (tS32*) mmap(
			NULL,
			sizeof(tU32),
			PROT_READ | PROT_WRITE,
			MAP_SHARED,
			g_hShMem,
			0);

		if (g_ps32CreatorProcessId == MAP_FAILED) {
			FATAL_M_ASSERT_ALWAYS();
			goto error_out; /*lint !e801, authorized LINT-deactivation #<71> */
		}

		bIsCreatorProcess = TRUE;
	} else {
		g_hSemLock = sem_open("VOLT_INIT", 0);

		if (g_hSemLock == SEM_FAILED) {
			FATAL_M_ASSERT_ALWAYS();
			goto error_out; /*lint !e801, authorized LINT-deactivation #<71> */
		}

		sem_wait(g_hSemLock);

		g_hShMem = shm_open("VOLT_INIT", O_RDWR ,0);

		if (g_hShMem  == -1) {
			FATAL_M_ASSERT_ALWAYS();
			goto error_out; /*lint !e801, authorized LINT-deactivation #<71> */
		}

		g_ps32CreatorProcessId = (tS32*) mmap(
			NULL,
			sizeof(tU32),
			PROT_READ | PROT_WRITE,
			MAP_SHARED,
			g_hShMem,
			0);

		if (g_ps32CreatorProcessId == MAP_FAILED) {
			FATAL_M_ASSERT_ALWAYS();
			goto error_out; /*lint !e801, authorized LINT-deactivation #<71> */
		}
	}

	au8TraceSendBuffer[0] = DEV_VOLT_C_U8_TRACE_TYPE_STRING;

	(tVoid)OSALUTIL_s32SaveNPrintFormat(
		(char*)(&(au8TraceSendBuffer[1])),
		sizeof(au8TraceSendBuffer)-1,
		"volt_process_attach() => Library for DEV_VOLT attached to process '%.15s' (#%d).%s",
		(rProcessControlBlock.szName != NULL) ? rProcessControlBlock.szName : "UNKNOWN",
		rProcessControlBlock.id,
		bIsCreatorProcess ? " (INIT)" : "");

	LLD_vTrace(
		TR_CLASS_DEV_VOLT,
		TR_LEVEL_USER_1,
		au8TraceSendBuffer,
		(tU32)OSAL_u32StringLength(au8TraceSendBuffer));

	if (bIsCreatorProcess == TRUE) {
		*g_ps32CreatorProcessId = rProcessControlBlock.id;
		(tVoid)DEV_VOLT_OsalIO_u32Init();
	}

error_out:

	if (g_hSemLock != SEM_FAILED)
		sem_post(g_hSemLock);
}

/*******************************************************************************
*
* Shared library detach function which is called by OSAL at the time the last
* registered client of a process has called the OSAL_IOClose() function for
* the /dev/volt.
*
* Only call the de-init function ...
*
*   - DEV_VOLT_OsalIO_u32Deinit()
*
* ... when the shared library is detached from the creator-process.
*
*******************************************************************************/
void __attribute__((destructor)) volt_process_detach(void)
{
	tU8 au8TraceSendBuffer[DEV_VOLT_C_U8_TRACE_SEND_BUFFER_LENGTH];

	OSAL_trProcessControlBlock rProcessControlBlock;

	if ((g_ps32CreatorProcessId == MAP_FAILED) ||
	    (g_hShMem  == -1)                      ||
	    (g_hSemLock == SEM_FAILED)               ) {
		FATAL_M_ASSERT_ALWAYS();
		return;
	}

	rProcessControlBlock.szName = NULL;
	rProcessControlBlock.id     = 0;

	OSAL_s32ProcessControlBlock(OSAL_ProcessWhoAmI(), &rProcessControlBlock);

	au8TraceSendBuffer[0] = DEV_VOLT_C_U8_TRACE_TYPE_STRING;

	(tVoid)OSALUTIL_s32SaveNPrintFormat(
		(char*)(&(au8TraceSendBuffer[1])),
		sizeof(au8TraceSendBuffer)-1,
		"volt_process_detach() => Library for DEV_VOLT detached from process '%.15s' (#%d).%s",
		(rProcessControlBlock.szName != NULL) ? rProcessControlBlock.szName : "UNKNOWN",
		rProcessControlBlock.id,
		((*g_ps32CreatorProcessId) == rProcessControlBlock.id) ? " (DE-INIT)" : "");

	LLD_vTrace(
		TR_CLASS_DEV_VOLT,
		TR_LEVEL_USER_1,
		au8TraceSendBuffer,
		(tU32)OSAL_u32StringLength(au8TraceSendBuffer));

	if ((*g_ps32CreatorProcessId) == rProcessControlBlock.id) {

		(tVoid)DEV_VOLT_OsalIO_u32Deinit();

		munmap(g_ps32CreatorProcessId, sizeof(tU32));
		shm_unlink("VOLT_INIT");

		sem_close(g_hSemLock);
		sem_unlink("VOLT_INIT");
	}
}

/******************************************************************************/
