/******************************************************************************
 *FILE         : oedt_GNSS_TestFuncs.c
 *
 *SW-COMPONENT : Google Testing Framework
 *
 *DESCRIPTION  : This Is The Source File Of GNSS Google@tgt Test Functions
 *
 * 
 *AUTHOR       : Ananth Deshpande(RBEI/ECF1)
 *
 *COPYRIGHT    : (C) COPYRIGHT RBEI - All Rights Reserved
********************************************************************************/
#include "osal_dev_Gnss_Testfuncs.h"
#define GNSS_SUCCESS_RETVAL 0

/* These bit fields are used keep track of received NMEA messages */
#define GNSS_OEDT_GPGGA_ID         (0x00000001)
#define GNSS_OEDT_GPGSA_ID         (0x00000002)
#define GNSS_OEDT_GPGSV_ID         (0x00000004)
#define GNSS_OEDT_GPRMC_ID         (0x00000008)
#define GNSS_OEDT_PSTMPV_ID        (0x00000040)
#define GNSS_OEDT_PSTMPVQ_ID       (0x00000080)
#define GNSS_OEDT_CHECKSUM_ERROR   (0x80000000)
#define GTEST_COUT std::cerr << "[          ] [ INFO ]"

#define GNSS_OEDT_NMEA_MSG_LIST_THREAD_NAME ("GnssNmeaLstThread")
#define GNSS_OEDT_NMEA_LIST_THREAD_STACK_SIZE (2048)

#define GNSS_OEDT_NMEA_LIST_EVENT_NAME ("GnssNmeaOedtList")
#define GNSS_NMEA_LIST_WAIT_TIME_MS (15000)

#define GNSS_OEDT_NMEA_LIST_PASSED  (0x000001)
#define GNSS_OEDT_NMEA_LIST_FAILED  (0x000002)

/* After firmware update is released, again the expected list has to be configured.
 * If configuration of Teseo changes, this has to be updated.  */
#define GNSS_PROXY_OEDT_EXPEC_NMEA_LIST ( (GNSS_OEDT_GPGGA_ID)| \
                                          (GNSS_OEDT_GPGSA_ID)| \
                                          (GNSS_OEDT_GPGSV_ID)| \
                                          (GNSS_OEDT_GPRMC_ID)| \
                                          (GNSS_OEDT_PSTMPV_ID)| \
                                          (GNSS_OEDT_PSTMPVQ_ID) )

/* Location of teseo binaries for firmware update */
#define X_LOADER_BIN_PATH "/home/root/teseo/flasher.bin"
#define TESEO_FW_BIN_PATH "/home/root/teseo/teseofw.bin"

/*location of Gnss Timestamp record File*/
#define GNSSRECORD "/var/opt/bosch/dynamic/ffs/gnssrecord.txt"

#define FIRMWARE_UPDATE_BUFF_SIZE (1024 * 16) /* 16KB  */
#define GNSS_OEDT_MAX_IOOPN_CLS_CNT 20
#define TIMESTAMP_DIFF_FAILURE_COUNT_LIMIT 5  /*This count can be set according to GNSSOedt_u32TimestampValidation oedt run time, here it is set 5 for 5 minutes timer*/
#define GNSS_TIMESTAMP_MONITORINGINTERVAL 300000 /*This is 5mins in milliseconds(5min = 300000 ms)*/

/* Constants for months and number of days in non-leap and leap years */
#define NO_OF_MONTHS_IN_A_YEAR                        (12)

#define MONTH_JAN                (1)
#define MONTH_FEB                (2)

#define OEDT_GNSS_GPS_INDEX                        0
#define OEDT_GNSS_GLONASS_INDEX                    1
#define OEDT_GNSS_SET_SAT_SYS_LIMIT                2
#define OEDT_GNSS_SET_SAT_SYS_LIMIT_CFG_BLK_227    1
#define OEDT_GNSS_SET_SAT_SYS_LMT_CFG_BLK_227      2
#define OEDT_GNSS_SET_SAT_SYS_LIMIT_CFG_BLK_200    4
#define OEDT_GNSS_SET_SAT_SYS_LIMIT_CFG_BLK        14
#define OEDT_GNSS_SET_SAT_SYS_LMT_CFG_BLK          21

#define GNSS_NO_ERROR                                      (0) //00000000000
#define GNSS_IOOPEN_ERROR                                  (1) //00000000001
#define GNSS_IOCLOSE_ERROR                                 (2) //00000000010
#define GNSS_IOCTRL_SET_EPOCH_ERROR                        (4) //00000000100 soon
#define GNSS_IOCTRL_GET_EPOCH_ERROR                        (8)
#define GNSS_GET_EPOCH_VALUE_MISMATCH                      (16)
#define GNSS_GET_EPOCH_NEW_DATE_GEN_ERROR                  (32)
#define GNSS_LAST_SAT_SYS_USED_FILE_DOESNOT_EXIST_ERROR    (64) 
#define FILE_OPEN_ERROR                                    (128)
#define TIMER_CREATE_ERROR                                 (256)
#define TIMER_SET_ERROR                                    (512)
#define GNSS_IOCTRL_FLUSH_SENSOR_DATA_ERROR                (1024)
#define TIMER_DELETE_ERROR                                 (2048)
#define FILE_CLOSE_ERROR                                   (4096)
#define TIMESTAMP_DIFF_FAILURE_COUNT_LIMIT_SURPASSED_ERROR (8192)
#define THREADWAIT_ERROR                                   (16384)
#define GNSS_READ_ERROR                                    (32768)
#define GNSS_IOCTRL_SET_SAT_SYS_ERROR                      (65536)
#define GNSS_IOCTRL_GET_SAT_SYS_ERROR                      (131072)
#define GNSS_IOREOPEN_ERROR                                (262144)
#define GNSS_IOCTRL_GNSS_SET_EPOCH_ERROR                   (524288)
#define OSAL_C_S32_IOCTL_GNSS_GET_GNSS_CHIP_CRC_ERROR      (1048576)

/*Local Function Declarations*/
static tVoid  GnssOedt_vNmeaListThread(tPVoid pVDummyArg );
static tU32 u32SetSatSys( tU32 u32Arg );
static tU32 u32GetSatSys( tVoid );
static tU32 GnssOedt_u32GetTeseoFlasherCrc32( FILE *fp, tU32 *pu32FinalCrc, tU32 u32ImageLength );

static tBool bCheckDateLiesInWeek( const OSAL_trGnssTimeUTC * rStartDateOfWeek, const OSAL_trGnssTimeUTC * rDateToCheck );
static tBool bCheckLeapYear( tU16 u16Year );
static tS32 s32AddDaysToDate( OSAL_trGnssTimeUTC * rUTCDate, tS32 s32AddDays );
static tVoid TimerExpiryFlagSetter(tPVoid);/*Timer expiry call back function*/
static tU32 u32ComputeDiffbwTimeStamp(tPU32 pu32TimeStampDifFailCount);

/* Global variables */

tBool bTimerExpiryFlag = 0;

OSAL_tEventHandle hGnssOedtNmeaListEvent;
static tS32  s32SendBinToTeseo(FILE *fp);
static tU32  u32_crc32(tU32 crc32val, const char *buf, tU32 len);
static tU32 GnssOedt_u32GetTeseoFwCrc32( FILE *fp, tU32 *pu32FinalCrc, tU32 u32ImageLength );

OSAL_tIODescriptor hDeviceHandle = 0;
OSAL_tIODescriptor hTeseoDeviceHandle = 0;    /*-----------changed handle name to remove lint-----------*/
static tU8 u8TesFwBuff[FIRMWARE_UPDATE_BUFF_SIZE];
tU32 u32NmeaListRetVal;

static tU32 crc32_tab[] = {
   0x00000000, 0x77073096, 0xee0e612c, 0x990951ba, 0x076dc419, 0x706af48f,
   0xe963a535, 0x9e6495a3, 0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988,
   0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91, 0x1db71064, 0x6ab020f2,
   0xf3b97148, 0x84be41de, 0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7,
   0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec, 0x14015c4f, 0x63066cd9,
   0xfa0f3d63, 0x8d080df5, 0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172,
   0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b, 0x35b5a8fa, 0x42b2986c,
   0xdbbbc9d6, 0xacbcf940, 0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59,
   0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116, 0x21b4f4b5, 0x56b3c423,
   0xcfba9599, 0xb8bda50f, 0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924,
   0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d, 0x76dc4190, 0x01db7106,
   0x98d220bc, 0xefd5102a, 0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433,
   0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818, 0x7f6a0dbb, 0x086d3d2d,
   0x91646c97, 0xe6635c01, 0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e,
   0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457, 0x65b0d9c6, 0x12b7e950,
   0x8bbeb8ea, 0xfcb9887c, 0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65,
   0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2, 0x4adfa541, 0x3dd895d7,
   0xa4d1c46d, 0xd3d6f4fb, 0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0,
   0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9, 0x5005713c, 0x270241aa,
   0xbe0b1010, 0xc90c2086, 0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
   0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4, 0x59b33d17, 0x2eb40d81,
   0xb7bd5c3b, 0xc0ba6cad, 0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a,
   0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683, 0xe3630b12, 0x94643b84,
   0x0d6d6a3e, 0x7a6a5aa8, 0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1,
   0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe, 0xf762575d, 0x806567cb,
   0x196c3671, 0x6e6b06e7, 0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc,
   0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5, 0xd6d6a3e8, 0xa1d1937e,
   0x38d8c2c4, 0x4fdff252, 0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b,
   0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60, 0xdf60efc3, 0xa867df55,
   0x316e8eef, 0x4669be79, 0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236,
   0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f, 0xc5ba3bbe, 0xb2bd0b28,
   0x2bb45a92, 0x5cb36a04, 0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d,
   0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a, 0x9c0906a9, 0xeb0e363f,
   0x72076785, 0x05005713, 0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38,
   0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21, 0x86d3d2d4, 0xf1d4e242,
   0x68ddb3f8, 0x1fda836e, 0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777,
   0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c, 0x8f659eff, 0xf862ae69,
   0x616bffd3, 0x166ccf45, 0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2,
   0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db, 0xaed16a4a, 0xd9d65adc,
   0x40df0b66, 0x37d83bf0, 0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
   0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6, 0xbad03605, 0xcdd70693,
   0x54de5729, 0x23d967bf, 0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94,
   0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d
};


using namespace testing;

class GnssTestFunctions : public ::testing::Test {
   
   protected:
   OSAL_tIODescriptor hDevHandle;
   OSAL_trGnssFullRecord rGnssFullRecord;
   OSAL_trGnssConfigData rGnssConfigData;
   OSAL_trThreadAttribute rThreadAttr;
   
   public:
   tU32 GNSSOedt_u32SatInterChk( tVoid );
   tU32 GNSSOedt_u32SatCfgBlk200( tVoid );
   tU32 GNSSOedt_u32ChkSatCfgBlk227( tVoid );
   tU32 GNSSOedt_u32ChkSatCfgBlk200227( tVoid );
   tU32 u32OedtGNSSProxyGetEpoch(tVoid);
   tU32 GNSSOedt_u32TimestampValidation(tVoid);
   tU32 GnssOedt_u32Teseo3FwUpdate(tVoid);
   tU32 GnssOedt_u32Teseo2FwUpdate(tVoid);
   tU32 GnssOedt_u32GetNmeaMsgList(tVoid);
   tU32 Gnss_u32VerifyGnssQualityon3DFix(tVoid);
   void SetUp(void);
   void TearDown(void);
};

void GnssTestFunctions::SetUp(void)
{
 
}

void GnssTestFunctions::TearDown(void)
{
   
}

/***************************************************************************
||FUNCTION    :  tU32 u32GetSatSys( tVoid )
||
||DESCRIPTION :  Getting the current satellite system value
||
||PARAMETER   :  Nil
||
||RETURNVALUE :  tU32
||               Sat Sys Value          - success
||               Error code             - failure
||HISTORY     :
||
***************************************************************************/
static tU32 u32GetSatSys( tVoid )
{
   tU32 u32ReturnValue = 0;

   /*---GET THE CURRENT SAT SYS--*/
   if( OSAL_ERROR == OSAL_s32IOControl( hDeviceHandle, OSAL_C_S32_IOCTRL_GNSS_GET_SAT_SYS, ( tLong )&u32ReturnValue ) )
   {
      u32ReturnValue = (tU32)OSAL_ERROR;
   }

   return ( ( tU32 )u32ReturnValue );
}

/***************************************************************************
||FUNCTION    :  tU32 u32SetSatSys( tVoid )
||
||DESCRIPTION :  Setting the satellite system value
||
||PARAMETER   :  tU32
||               u32Arg SAT_SYS value
||
||RETURNVALUE :  tU32
||               Sat Sys Value that was set         - success
||               Error code                         - failure
||HISTORY     :
||
***************************************************************************/
static tU32 u32SetSatSys( tU32 u32Arg )
{
   tU32 u32Argument = u32Arg;
   tU32 u32ReturnValue;
   /*---SET THE SAT SYS--*/
   if( OSAL_ERROR == OSAL_s32IOControl( hDeviceHandle, OSAL_C_S32_IOCTRL_GNSS_SET_SAT_SYS, ( tLong )&u32Argument ) )
   {
      u32ReturnValue = (tU32)OSAL_ERROR;
      
   }
   else
   {
      u32ReturnValue = u32Argument;
   }
   return ( u32ReturnValue );
}

/******************************************************************************
||FUNCTION    :  tU32 GNSSOedt_u32SatInterChk( tVoid )
||
||DESCRIPTION :  This function is implemented to check the satellite interface 
||               system and also to set the satellite system to desired value
||               and check for the same if it is set to desired 
||               value. Finally we are setting to the initial value 
||
||PARAMETER   :  Nil
||
||RETURNVALUE :  tU32
||               0          - success 
||               Error code - failure
||HISTORY     :
||
******************************************************************************/
tU32 GnssTestFunctions::GNSSOedt_u32SatInterChk( tVoid )
{
   tU32 u32Arg = OSAL_C_U8_GNSS_SATSYS_GPS|OSAL_C_U8_GNSS_SATSYS_GLONASS;
   tU32 u32SetFunctionReturn = 0;
   tU32 u32GetFunctionReturn = 0;
   tU32 u32InitialSatValue = 0;
   tBool bErrorOccurred = FALSE;
   tU32 u32ErrorValue = 0;
   
   /*---Open the GNSS driver---*/
   hDeviceHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_READONLY );
   if( hDeviceHandle == OSAL_ERROR )
   {
      u32ErrorValue += 1;
      bErrorOccurred = TRUE;
   }
   else
   { 
     /*---Flush Sensor Data---*/
      if(OSAL_ERROR == OSAL_s32IOControl( hDeviceHandle, OSAL_C_S32_IOCTL_GNSS_FLUSH_SENSOR_DATA, 0) )
      {
         u32ErrorValue += 2;
         bErrorOccurred = TRUE;
      }
      
      if( bErrorOccurred == FALSE )
      {
         /*---Get the current Satellite System--- */
         u32InitialSatValue = u32GetSatSys();
         if( OSAL_ERROR == ( tS32 ) u32InitialSatValue )
         {
            u32ErrorValue += 4;
            bErrorOccurred = TRUE;
         }
      }
      
      if( bErrorOccurred == FALSE )
      {
         /*---Set the Satellite System to the value given---*/
         u32SetFunctionReturn = u32SetSatSys( u32Arg );
         if( OSAL_ERROR == ( tS32 ) u32SetFunctionReturn )
         {
            u32ErrorValue += 8;
            bErrorOccurred = TRUE;
         }
      }

      if( bErrorOccurred == FALSE )
      {
         /*---Get the Satellite System which was set and check if it is same as the one set---*/
         u32GetFunctionReturn = u32GetSatSys();
         if( OSAL_ERROR == ( tS32 )u32GetFunctionReturn )
         {
            u32ErrorValue += 16;
            bErrorOccurred = TRUE;
         }
         else
         {
            if( u32GetFunctionReturn != u32SetFunctionReturn )
            {
               u32ErrorValue += 32;
               bErrorOccurred = TRUE;
            }
         }
      }
      
      if( bErrorOccurred == FALSE )
      {
         /*---Set the Initial Satellite System---*/
         u32SetFunctionReturn = u32SetSatSys( u32InitialSatValue );
         if( OSAL_ERROR == ( tS32 )u32SetFunctionReturn )
         {
            u32ErrorValue += 64;
            bErrorOccurred = TRUE;
         }
      }
      
      if( bErrorOccurred == FALSE )
      {
         /*---Get the current Satellite System---*/
         u32GetFunctionReturn = u32GetSatSys();
         if( OSAL_ERROR == ( tS32 )u32GetFunctionReturn )
         {
            u32ErrorValue += 128;
            bErrorOccurred = TRUE;
         }
         else
         {
            if( u32GetFunctionReturn != u32InitialSatValue )
            {
               u32ErrorValue += 256;
               bErrorOccurred = TRUE;
            }
         }
      }
   }
   
   if( hDeviceHandle !=  OSAL_ERROR && OSAL_s32IOClose( hDeviceHandle ) == OSAL_ERROR )
   {
      u32ErrorValue += 512;
   }
   return ( u32ErrorValue ); 
}

/******************************************************************************
||FUNCTION    :  tU32 GNSSOedt_u32SatCfgBlk200( tVoid )
||
||DESCRIPTION :  This function is implemented to check if all
||                       the satelllites in Cfg block 200 are set as reqd.
||
||PARAMETER   :  Nil
||
||RETURNVALUE :  tU32
||               0          - success 
||               Error code - failure
||HISTORY     :
||
******************************************************************************/

tU32 GnssTestFunctions::GNSSOedt_u32SatCfgBlk200( tVoid )
{
   tU32 u32SetFuncRet = 0;
   tU32 u32GetFuncRet = 0;
   tU32 u32ErrorValue = 0;
   tU8 u8SatSystobeSet[OEDT_GNSS_SET_SAT_SYS_LIMIT_CFG_BLK_200] = {  OSAL_C_U8_GNSS_SATSYS_GPS,
                                                                     OSAL_C_U8_GNSS_SATSYS_GLONASS,
                                                                     OSAL_C_U8_GNSS_SATSYS_SBAS,
                                                                     OSAL_C_U8_GNSS_SATSYS_QZSS };
   tU16 u16Loop = 0;

   
   /*---Open the GNSS driver---*/
   hDeviceHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_READONLY );
   if( hDeviceHandle == OSAL_ERROR )
   {
      u32ErrorValue += 1;
   }
   else
   {
      /*---Flush Sensor Data---*/
      if(OSAL_ERROR == OSAL_s32IOControl( hDeviceHandle, OSAL_C_S32_IOCTL_GNSS_FLUSH_SENSOR_DATA, 0) )
      {
         u32ErrorValue += 2;
      }

      if( 0 == u32ErrorValue )
      {
         /*---Set the Satellite System to the value given---*/
         for( u16Loop = 0; u16Loop < OEDT_GNSS_SET_SAT_SYS_LIMIT_CFG_BLK_200; u16Loop ++)
         { 
            u32SetFuncRet = u32SetSatSys( u8SatSystobeSet[u16Loop] );
            if( OSAL_ERROR == ( tS32 ) u32SetFuncRet )
            {
               u32ErrorValue += 4;
            }

            if( 0 == u32ErrorValue )
            {
               u32GetFuncRet = u32GetSatSys();
               if( OSAL_ERROR == ( tS32 ) u32GetFuncRet )
               {
                  u32ErrorValue += 8;
               }
            }

            if( 0 == u32ErrorValue )
            {
               if( u8SatSystobeSet[u16Loop] != u32GetFuncRet )
               {
                  u32ErrorValue += 16;
               }
            }
         }
         
      }
      
      if( OSAL_s32IOClose( hDeviceHandle ) == OSAL_ERROR )
      {
         u32ErrorValue += 16;
      }
   }
   
   return ( u32ErrorValue );
   
}

/******************************************************************************
||FUNCTION    :  tU32 GNSSOedt_u32ChkSatCfgBlk227( tVoid )
||
||DESCRIPTION :  This function is implemented to check if all
||                       the satelllites in Cfg block 227 are set as reqd.
||
||PARAMETER   :  Nil
||
||RETURNVALUE :  tU32
||               0          - success 
||               Error code - failure
||HISTORY     :
||
******************************************************************************/
tU32 GnssTestFunctions::GNSSOedt_u32ChkSatCfgBlk227( tVoid )
{
   tU32 u32SetFuncRet = 0;
   tU32 u32GetFuncRet = 0;
   tU32 u32ErrorValue = 0;
   tU8 u8SatSystobeSet[OEDT_GNSS_SET_SAT_SYS_LMT_CFG_BLK_227] = {OSAL_C_U8_GNSS_SATSYS_GALILEO,
                                                                 OSAL_C_U8_GNSS_SATSYS_COMPASS};
   tU16 u16Loop = 0;

   
   /*---Open the GNSS driver---*/
   hDeviceHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_READONLY );
   if( hDeviceHandle == OSAL_ERROR )
   {
      u32ErrorValue += 1;
   }
   else
   {
      
      /*---Flush Sensor Data---*/
      if(OSAL_ERROR == OSAL_s32IOControl( hDeviceHandle, OSAL_C_S32_IOCTL_GNSS_FLUSH_SENSOR_DATA, 0) )
      {
         u32ErrorValue += 2;
      }

      if( 0 == u32ErrorValue )
      {
         /*---Set the Satellite System to the value given---*/
         for( u16Loop = 0; u16Loop < OEDT_GNSS_SET_SAT_SYS_LIMIT_CFG_BLK_227; u16Loop ++)
         { 
            u32SetFuncRet = u32SetSatSys( u8SatSystobeSet[u16Loop] );
            if( OSAL_ERROR == ( tS32 ) u32SetFuncRet )
            {
               u32ErrorValue += 4;
            }

            if( 0 == u32ErrorValue )
            {
               u32GetFuncRet = u32GetSatSys();
               if( OSAL_ERROR == ( tS32 ) u32GetFuncRet )
               {
                  u32ErrorValue += 8;
               }
            }

            if( 0 == u32ErrorValue )
            {
               if( u8SatSystobeSet[u16Loop] != u32GetFuncRet )
               {
                  u32ErrorValue += 16;
               }
            }
         }
         
      }
      
      if( OSAL_s32IOClose( hDeviceHandle ) == OSAL_ERROR )
      {
         u32ErrorValue += 16;
      }
   }
   
   return ( u32ErrorValue );
   
}

/******************************************************************************
||FUNCTION    :  tU32 GNSSOedt_u32ChkSatCfgBlk200227( tVoid )
||
||DESCRIPTION :  This function is implemented to check if combination of
||                       the satelllites in Cfg block 200 and Cfg Block 227
||                       are set as reqd.
||
||PARAMETER   :  Nil
||
||RETURNVALUE :  tU32
||               0          - success 
||               Error code - failure
||HISTORY     :
||
******************************************************************************/
tU32 GnssTestFunctions::GNSSOedt_u32ChkSatCfgBlk200227( tVoid )
{
   tU32 u32SetFuncRet = 0;
   tU32 u32GetFuncRet = 0;
   tU32 u32ErrorValue = 0;
   tU8 u8SatSystobeSet[OEDT_GNSS_SET_SAT_SYS_LMT_CFG_BLK] = {OSAL_C_U8_GNSS_SATSYS_ALL,
                                                             OSAL_C_U8_GNSS_SATSYS_GPS|OSAL_C_U8_GNSS_SATSYS_GLONASS,
                                                             OSAL_C_U8_GNSS_SATSYS_GPS|OSAL_C_U8_GNSS_SATSYS_GALILEO,
                                                             OSAL_C_U8_GNSS_SATSYS_GPS|OSAL_C_U8_GNSS_SATSYS_COMPASS,
                                                             OSAL_C_U8_GNSS_SATSYS_GPS|OSAL_C_U8_GNSS_SATSYS_SBAS,
                                                             OSAL_C_U8_GNSS_SATSYS_GPS|OSAL_C_U8_GNSS_SATSYS_QZSS,
                                                             OSAL_C_U8_GNSS_SATSYS_GPS|OSAL_C_U8_GNSS_SATSYS_SBAS|\
                                                             OSAL_C_U8_GNSS_SATSYS_QZSS,
                                                             OSAL_C_U8_GNSS_SATSYS_GPS|OSAL_C_U8_GNSS_SATSYS_SBAS|\
                                                             OSAL_C_U8_GNSS_SATSYS_GLONASS,
                                                             OSAL_C_U8_GNSS_SATSYS_GPS|OSAL_C_U8_GNSS_SATSYS_GLONASS|\
                                                             OSAL_C_U8_GNSS_SATSYS_QZSS,
                                                             OSAL_C_U8_GNSS_SATSYS_GLONASS|OSAL_C_U8_GNSS_SATSYS_GALILEO,
                                                             OSAL_C_U8_GNSS_SATSYS_GLONASS|OSAL_C_U8_GNSS_SATSYS_COMPASS,
                                                             OSAL_C_U8_GNSS_SATSYS_GLONASS|OSAL_C_U8_GNSS_SATSYS_SBAS,
                                                             OSAL_C_U8_GNSS_SATSYS_GLONASS|OSAL_C_U8_GNSS_SATSYS_QZSS,
                                                             OSAL_C_U8_GNSS_SATSYS_GALILEO|OSAL_C_U8_GNSS_SATSYS_COMPASS,
                                                             OSAL_C_U8_GNSS_SATSYS_GALILEO|OSAL_C_U8_GNSS_SATSYS_SBAS,
                                                             OSAL_C_U8_GNSS_SATSYS_GALILEO|OSAL_C_U8_GNSS_SATSYS_QZSS,
                                                             OSAL_C_U8_GNSS_SATSYS_COMPASS|OSAL_C_U8_GNSS_SATSYS_SBAS,
                                                             OSAL_C_U8_GNSS_SATSYS_COMPASS|OSAL_C_U8_GNSS_SATSYS_QZSS,
                                                             OSAL_C_U8_GNSS_SATSYS_COMPASS|OSAL_C_U8_GNSS_SATSYS_SBAS|
                                                             OSAL_C_U8_GNSS_SATSYS_QZSS,
                                                             OSAL_C_U8_GNSS_SATSYS_SBAS|OSAL_C_U8_GNSS_SATSYS_QZSS,
                                                             OSAL_C_U8_GNSS_SATSYS_GPS|OSAL_C_U8_GNSS_SATSYS_GALILEO|\
                                                             OSAL_C_U8_GNSS_SATSYS_SBAS|OSAL_C_U8_GNSS_SATSYS_GLONASS};
   tU16 u16Loop = 0;

   /*---Open the GNSS driver---*/
   hDeviceHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_READONLY );
   if( hDeviceHandle == OSAL_ERROR )
   {
      u32ErrorValue += 1;
   }
   else
   {  
      /*---Flush Sensor Data---*/
      if(OSAL_ERROR == OSAL_s32IOControl( hDeviceHandle, OSAL_C_S32_IOCTL_GNSS_FLUSH_SENSOR_DATA, 0) )
      {
         u32ErrorValue += 2;
      }

      if( 0 == u32ErrorValue )
      {
         /*---Set the Satellite System to the value given---*/
         for( u16Loop = 0; u16Loop < OEDT_GNSS_SET_SAT_SYS_LIMIT_CFG_BLK; u16Loop ++)
         { 
            u32SetFuncRet = u32SetSatSys( u8SatSystobeSet[u16Loop] );
            if( OSAL_ERROR == ( tS32 ) u32SetFuncRet )
            {
               u32ErrorValue += 4;
            }

            if( 0 == u32ErrorValue )
            {
               u32GetFuncRet = u32GetSatSys();
               if( OSAL_ERROR == ( tS32 ) u32GetFuncRet )
               {
                  u32ErrorValue += 8;
               }
            }
            
            if( 0 == u32ErrorValue )
            {
               if( u8SatSystobeSet[u16Loop] != u32GetFuncRet )
               {
                  u32ErrorValue += 16;
               }
            }
         }
         
      }
      
      if( OSAL_s32IOClose( hDeviceHandle ) == OSAL_ERROR )
      {
         u32ErrorValue += 16;
      }
   }
   
   return ( u32ErrorValue ); 
}

/************************************************************************
||FUNCTION       :  static tS32 s32AddDaysToDate( OSAL_trGnssTimeUTC * rUTCDate, tS32 s32AddDays )
||
||DESCRIPTION  :  Adds specified number of days to a given date
||                        
||                        
||
||PARAMETER     :   OSAL_trGnssTimeUTC * rUTCDate
                             tS32 s32AddDays
||
||RETURNVALUE :  tS32 
||                        OSAL_E_NOERROR -            if the days are added to the given date without any errors
||                        OSAL_E_INVALIDVALUE -     if the date parameter is NULL
||HISTORY        :
||
************************************************************************/
static tS32 s32AddDaysToDate( OSAL_trGnssTimeUTC * rUTCDate, tS32 s32AddDays )
{
   tS32 s32ReturnValue      = (tS32)OSAL_E_NOERROR;
   tS32 s32Days             = 0;
   tU8  u8Month             = 0;
   tU16 u16Year             = 0;
   tU8  u8MonthTable[12]    = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31};
   tS32 s32DaysTemp         = 0;

   if( rUTCDate == OSAL_NULL )
   {
       s32ReturnValue = (tS32)OSAL_E_INVALIDVALUE;
   }
   else
   {
       s32Days = (tS32)(rUTCDate->u8Day);
       u8Month = rUTCDate->u8Month;
       u16Year = rUTCDate->u16Year;

       while(s32DaysTemp < s32AddDays )
       {
         s32Days += 1;

         if(s32Days > u8MonthTable[u8Month - 1])
         {
            if((u8Month == MONTH_FEB) && (TRUE == bCheckLeapYear(u16Year)) && (s32Days == 29))
            {
               s32Days = 29;
            }
            else
            {
               u8Month += 1;
               s32Days = 1;
            }
         }
         if(u8Month > NO_OF_MONTHS_IN_A_YEAR)
         {
            s32Days = 1;
            u8Month = MONTH_JAN;
            u16Year += 1;
         }
         s32DaysTemp ++;
       }

       rUTCDate->u16Year = u16Year;
       rUTCDate->u8Month = u8Month;
       rUTCDate->u8Day   = (tU8)s32Days;
   }

   return s32ReturnValue;
}

/************************************************************************
||FUNCTION       :  static tBool  bCheckDateLiesInWeek( const OSAL_trGnssTimeUTC * rStartDateOfWeek, const OSAL_trGnssTimeUTC * rDateToCheck )
||
||DESCRIPTION  :  Checks if the given date lies within the same week as the other date
||                        
||                        
||
||PARAMETER     :   OSAL_trGnssTimeUTC * rStartDateOfWeek
                             OSAL_trGnssTimeUTC * rDateToCheck
||
||RETURNVALUE :  tU8 
||                        FALSE - If the date does not lie in the same week as the other date 
||                        TRUE  - If the date lies in the same week as the other date
||HISTORY        :
||
************************************************************************/
static tBool bCheckDateLiesInWeek( const OSAL_trGnssTimeUTC * rStartDateOfWeek, const OSAL_trGnssTimeUTC * rDateToCheck )
{
   tBool bReturnValue = FALSE;
   tU8 u8WeekDayAdd;
   OSAL_trGnssTimeUTC rDateInWeek;

   rDateInWeek.u8Day   = rStartDateOfWeek->u8Day;
   rDateInWeek.u8Month = rStartDateOfWeek->u8Month;
   rDateInWeek.u16Year = rStartDateOfWeek->u16Year;

   if((rDateToCheck->u8Day == rStartDateOfWeek->u8Day) && (rDateToCheck->u8Month == rStartDateOfWeek->u8Month) && (rDateToCheck->u16Year == rStartDateOfWeek->u16Year))
   {
      bReturnValue = TRUE;
   }
   else
   {
      for( u8WeekDayAdd = 1; ((u8WeekDayAdd < 7) && (bReturnValue != TRUE)); u8WeekDayAdd++ )
      {
         if((tS32)OSAL_E_NOERROR == s32AddDaysToDate(&rDateInWeek, 1))
         {
            if((rDateToCheck->u8Day == rDateInWeek.u8Day) && (rDateToCheck->u8Month == rDateInWeek.u8Month) && (rDateToCheck->u16Year == rDateInWeek.u16Year))
            {
               bReturnValue = TRUE;
            }
         }
      }
   }
   return bReturnValue;
}

/*************************************************************************************
* FUNCTION     : bCheckLeapYear
*
* PARAMETER    : tU16 u16Year - Year to check
*
* RETURNVALUE  : FALSE - If the year is common year
*                         TRUE - If the year is leap year
*
* DESCRIPTION  : Checks whether the input year is leap or not.
*
* HISTORY      :
***************************************************************************************/
static tBool bCheckLeapYear( tU16 u16Year )
{

   tBool bRetVal = 0;

   if( u16Year % 4 != 0 )
   {
      bRetVal = FALSE;   //Common year
   }
   else if ( u16Year % 100 != 0 )
   {
      bRetVal = TRUE;   //Leap year
   }
   else if ( u16Year % 400 != 0 )
   {
      bRetVal = FALSE;   //Common year
   }
   else
   {
      bRetVal = TRUE;   //Leap year
   }

   return bRetVal;
}

/************************************************************************
||FUNCTION       :  tU32  u32OedtGNSSProxyGetEpoch(tVoid)
||
||DESCRIPTION  :  
||                        Sets the value of epoch to 5 different values and reads back the set value each time.
||                        
||
||PARAMETER     :  Nil
||
||RETURNVALUE :  tU32 
||                        GNSS_NO_NOERROR     -                                   success 
||                        GNSS_IOCTRL_SET_EPOCH_ERROR                -    FAILURE in setting Epoch
||                        GNSS_IOCTRL_GET_EPOCH_ERROR                -    FAILURE in getting Epoch
||                        GNSS_GET_EPOCH_NEW_DATE_GEN_ERROR    -    FAILURE in generating new date
||                        GNSS_GET_EPOCH_VALUE_MISMATCH             -    FAILURE due to mismatch in set and get values
||                        GNSS_IOOPEN_ERROR                                 -     FAILURE in opening GNSS device
||                        GNSS_IOCLOSE_ERROR                -                     FAILURE in closing GNSS device
||HISTORY        :
||
************************************************************************/
tU32 GnssTestFunctions::u32OedtGNSSProxyGetEpoch(tVoid)
{
   tU32 u32ReturnValue = GNSS_NO_ERROR;
   OSAL_trGnssTimeUTC rEpochSetTime = {0};
   OSAL_trGnssTimeUTC rEpochSetTimeNextWeek = {0};
   OSAL_tIODescriptor hDevHandle;
   OSAL_trGnssTimeUTC rGPSTimeUTC[] = { {2015,12,9,0,0,0,0},
                                        {2060,2,29,0,0,0,0},
                                        {2060,3,1,0,0,0,0},
                                        {2000,1,1,0,0,0,0},
                                        {1999,12,31,0,0,0,0},
                                        {2115,12,3,0,0,0,0},
                                        {2088,2,29,0,0,0,0},
                                        {2010,12,31,0,0,0,0},
                                        {2011,1,1,0,0,0,0},
                                        {2099,12,31,0,0,0,0}};

   hDevHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_READONLY );
   if ( hDevHandle == OSAL_ERROR)
   {
      u32ReturnValue = GNSS_IOOPEN_ERROR;
   }
   else if ( OSAL_ERROR == OSAL_s32IOControl(hDevHandle, OSAL_C_S32_IOCTL_GNSS_FLUSH_SENSOR_DATA, 0))
   {
      u32ReturnValue = GNSS_IOCTRL_FLUSH_SENSOR_DATA_ERROR;
   }
   else
   {
      tU8 i = 0;
      do
      {
         OSAL_s32ThreadWait(10);

         if( OSAL_ERROR == OSAL_s32IOControl( hDevHandle, OSAL_C_S32_IOCTL_GNSS_SET_EPOCH, (tLong)&rGPSTimeUTC[i]) )
         {
            u32ReturnValue = GNSS_IOCTRL_SET_EPOCH_ERROR;
         }
         else
         {
          
            if( (tS32)OSAL_ERROR == OSAL_s32IOControl( hDevHandle, OSAL_C_S32_IOCTL_GNSS_GET_EPOCH, (tLong)&rEpochSetTime))
            {
               u32ReturnValue = GNSS_IOCTRL_GET_EPOCH_ERROR;
            }
            else
            {
               /* To get the date of the end of week, 6 days are added to rEpochSetTime */
               rEpochSetTimeNextWeek.u8Day   = rEpochSetTime.u8Day;
               rEpochSetTimeNextWeek.u8Month = rEpochSetTime.u8Month; 
               rEpochSetTimeNextWeek.u16Year = rEpochSetTime.u16Year;

               if((tS32)OSAL_E_NOERROR != s32AddDaysToDate(&rEpochSetTimeNextWeek, 6))
               {
                  u32ReturnValue = GNSS_GET_EPOCH_NEW_DATE_GEN_ERROR;
               }
               else
               {
                  if( TRUE == bCheckDateLiesInWeek( &rEpochSetTime, &rGPSTimeUTC[i] ))
                  {
                 
                  }
                  else
                  {
                     u32ReturnValue = GNSS_GET_EPOCH_VALUE_MISMATCH;
                  }
               }
               
            }
         }
         i++;
      }while((u32ReturnValue == GNSS_NO_ERROR) && (i < 10));
   }

   if (hDevHandle != OSAL_ERROR && OSAL_s32IOClose(hDevHandle) == OSAL_ERROR)
   {
      u32ReturnValue = (tS32)GNSS_IOCLOSE_ERROR;
   }
   return u32ReturnValue;
}

/***************************************************************************
||FUNCTION    :  tU32  GNSSOedt_u32TimestampValidation(tVoid)
||
||DESCRIPTION : 
||               Checks Whether difference between consecutive Gnss Timestamp is within the 
||               limit of 990ms to 1010ms.In case the difference is beyond this limit, the number of such 
||              occurences must be within TIMESTAMP_DIFF_FAILURE_COUNT_LIMIT.
||  
||PARAMETER   :  Nil
||
||RETURNVALUE :  tU32
||               0          - success 
||               Error code - failure
||HISTORY     :
||
*************************************************************************/
tU32 GnssTestFunctions::GNSSOedt_u32TimestampValidation(tVoid)
{
   OSAL_tTimerHandle hTimer;
   OSAL_tIODescriptor hDevHandle;
   FILE *fd = NULL;
   tU32 u32TimeStampbuffer = 0;
   tU32 u32TimeStampcnt = 0;
   tU32 u32RetVal = GNSS_NO_ERROR;
   OSAL_trGnssFullRecord rGnssFullRecord;
   tU32 u32TimeStampDeviationCount = 0;
   bTimerExpiryFlag = 0; //Global variable 

   // open a text file to store Gnss TimeStamp values.
   if( OSAL_NULL == (fd = fopen(GNSSRECORD , "w+")) )
   {
      u32RetVal = FILE_OPEN_ERROR;
   }
   
   // Create a timer, set the time duration for which TimeStamps have to be stored
   else if(OSAL_ERROR ==(OSAL_s32TimerCreate((OSAL_tpfCallback)TimerExpiryFlagSetter, (tPVoid)NULL, &hTimer)))
   {
      u32RetVal = TIMER_CREATE_ERROR;
   }

   // Start the timer
   else if(OSAL_ERROR ==(OSAL_s32TimerSetTime(hTimer, GNSS_TIMESTAMP_MONITORINGINTERVAL, GNSS_TIMESTAMP_MONITORINGINTERVAL)))
   {
      u32RetVal = TIMER_SET_ERROR;
   }
   
   else
   {
      // open the Gnss device for reading Gnss timestamp
      hDevHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_READONLY );
         
      if ( OSAL_ERROR == hDevHandle)
      {
         u32RetVal = GNSS_IOOPEN_ERROR;
      }
      
      //Flush the buffer on V850 in order to get Gnss sensor data
      else if(OSAL_ERROR == OSAL_s32IOControl( hDevHandle, OSAL_C_S32_IOCTL_GNSS_FLUSH_SENSOR_DATA, 0) )
      {
         u32RetVal = GNSS_IOCTRL_FLUSH_SENSOR_DATA_ERROR;                           
      }   
         
      else
      {
         OSAL_s32ThreadWait(2000);
         // While loop executes till the Timer expires
         
         while(bTimerExpiryFlag == 0)
         {
            
            if ( OSAL_ERROR == OSAL_s32IORead(hDevHandle,(tPS8)&rGnssFullRecord,sizeof(OSAL_trGnssFullRecord)))
            {
               continue;
            }
            
            u32TimeStampbuffer = rGnssFullRecord.u32TimeStamp; // Storing the Gnss timestamp into u32TimeStampbuffer,which is inturn written into GNSSRECORD file
            
            fwrite(&u32TimeStampbuffer,sizeof(u32TimeStampbuffer),1,fd);
            
            u32TimeStampcnt = u32TimeStampcnt + 1;
            u32TimeStampbuffer = 0;
         }
         
         u32TimeStampbuffer = 0;         
      }
   
      //Closing the Gnss device file
      
      if(!(GNSS_IOOPEN_ERROR & u32RetVal))
      {   
         if (OSAL_ERROR == OSAL_s32IOClose( hDevHandle ))
         {
            u32RetVal = GNSS_IOCLOSE_ERROR;
         }
      }
   }
   
   //Closing GNSSRECORD file
   
   if(!(FILE_OPEN_ERROR & u32RetVal))
   {   
      if(EOF == (fclose(fd)))
      {
         u32RetVal = FILE_CLOSE_ERROR;
      }   
      
      fd = NULL;
      // Deleting the Timer
   
      if(!(TIMER_CREATE_ERROR & u32RetVal))
      {
         if(OSAL_ERROR == OSAL_s32TimerDelete(hTimer))
         {
            u32RetVal = TIMER_DELETE_ERROR;
         }   
      }
   }
   
   // Calling the function u32ComputeDiffbwTimeStamp    
   u32RetVal |= u32ComputeDiffbwTimeStamp(&u32TimeStampDeviationCount);

   if(u32TimeStampDeviationCount > TIMESTAMP_DIFF_FAILURE_COUNT_LIMIT)
   {
      u32RetVal = TIMESTAMP_DIFF_FAILURE_COUNT_LIMIT_SURPASSED_ERROR;
   }
   return u32RetVal;
}

//callback function after timer expiry.
static tVoid TimerExpiryFlagSetter(tPVoid pArg)
{
   bTimerExpiryFlag = 1;
}
 
//computes the time difference between 2 consecutive timestamps, that were recorded for time set in timer. 
static tU32 u32ComputeDiffbwTimeStamp(tPU32 pu32TimeStampDifFailCount)
{
   tU32 u32PrevTimeStamp = 0;
   tU32 u32NextTimeStamp = 0;
   tU32 u32DiffbwTimeStamp = 0;
   int numberofbytes = 0;
   FILE *fd = NULL;
   tS32 u32RetVal = GNSS_NO_ERROR;
   *pu32TimeStampDifFailCount = 0;
   
   if(OSAL_NULL == (fd = fopen(GNSSRECORD,"r")))
   {
      u32RetVal = FILE_OPEN_ERROR;
   }   

   else
   {
      fseek(fd,0,SEEK_SET);
      numberofbytes= fread(&u32PrevTimeStamp,sizeof(u32PrevTimeStamp),1,fd);
      
      while(numberofbytes > 0)
      {
        numberofbytes = fread(&u32NextTimeStamp,sizeof(u32NextTimeStamp),1,fd);
            
         if( u32NextTimeStamp != u32PrevTimeStamp)
         {
            u32DiffbwTimeStamp = u32NextTimeStamp - u32PrevTimeStamp;
            
            if( (u32DiffbwTimeStamp < 990) || (u32DiffbwTimeStamp > 1010))               
            {
               *pu32TimeStampDifFailCount = *pu32TimeStampDifFailCount + 1 ;
            }

            u32PrevTimeStamp = u32NextTimeStamp;
         }
      }
   }   
   
   if(!(FILE_OPEN_ERROR & u32RetVal))
   {   
      if(EOF == (fclose(fd)))
      {
         u32RetVal = FILE_CLOSE_ERROR;
      }
      fd = NULL;
   }
   
   return u32RetVal;
} 

/***************************************************************************
||FUNCTION    :  tU32  GnssOedt_u32Teseo3FwUpdate(tVoid)
||
||DESCRIPTION : 
||               Do Teseo-3 firmware update
||PARAMETER   :  Nil
||
||RETURNVALUE :  tU32
||               0          - success
||               Error code - failure
||HISTORY     :
||
***************************************************************************/
tU32 GnssTestFunctions::GnssOedt_u32Teseo3FwUpdate(tVoid)
{

   FILE *Xldrfp  ;
   FILE *TesFwfp = OSAL_NULL;
   tS32 s32BtLdrSize = 0;
   tS32 s32FwImgSize = 0;
   tS32 s32RetVal = 0;
   trImageOptions rImageOptions;

/* Open teseo boot loader */
   Xldrfp = fopen( X_LOADER_BIN_PATH, "r" );
   if ( OSAL_NULL == Xldrfp )
   {
      s32RetVal |= 1;
   }
   
   /* open Teseo firmware image */
   if ( 0 == s32RetVal)
   {
      TesFwfp = fopen( TESEO_FW_BIN_PATH, "r" );
      if ( OSAL_NULL == TesFwfp )
      {
          s32RetVal |= 2; 
      }
   }
   /* Get the size of boot loader */
   if ( ( Xldrfp != OSAL_NULL )&& (0 == s32RetVal))      /*-----------changes made to remove lint-----------*/
   {
      if( 0 != fseek( Xldrfp,0L, SEEK_END ) )
      {
         s32RetVal |= 4;
      }
   }

   if ( ( Xldrfp != OSAL_NULL ) && (0 == s32RetVal))     /*-----------changes made to remove lint-----------*/
   {
      s32BtLdrSize = ftell (Xldrfp);
      if ( 0 == s32BtLdrSize )
      {
         s32RetVal |= 8;
      }
      /* file position indicator to the Beginning */
      rewind(Xldrfp); 
   }

   /* Get the size of firmware */
   if ( ( TesFwfp != OSAL_NULL) && (0 == s32RetVal))     /*-----------changes made to remove lint-----------*/
   {
      if( 0 != fseek( TesFwfp,0L, SEEK_END ) )
      {
         s32RetVal |= 16;
      }
   }
   if ( ( TesFwfp != OSAL_NULL) && (0 == s32RetVal))     /*-----------changes made to remove lint-----------*/
   {
      s32FwImgSize = ftell (TesFwfp);
      if ( 0 == s32FwImgSize )
      {
         s32RetVal |= 32;
      }
      /*file position indicator to the Beginning */
      rewind(TesFwfp);
   }
   
   /* Open GNSS device for flashing */
   if ( 0 == s32RetVal)
   {
      hTeseoDeviceHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_WRITEONLY );
      if ( hTeseoDeviceHandle == OSAL_ERROR)
      {
         s32RetVal |= 64;
      }
   }
   
   if(OSAL_ERROR == OSAL_s32IOControl( hTeseoDeviceHandle, OSAL_C_S32_IOCTL_GNSS_SET_CHIP_TYPE, (tS32)GNSS_CHIP_TYPE_TESEO_3 ))
   {
      s32RetVal |= 128;
   }

   if ( 0 == s32RetVal)
   {
      /* Get the CRC for Teseo boot loader */
      if ( OSAL_OK != GnssOedt_u32GetTeseoFlasherCrc32( Xldrfp ,&rImageOptions.u32ChkSum, (tU32)s32BtLdrSize ) )
      {
                 s32RetVal |= 256;
      }
   }

   /* Configuration to send Teseo Boot loader */
   if ( 0 == s32RetVal)
   {
      rImageOptions.enType = IMAGE_TESEO_FLASHER_BIN;
      rImageOptions.u32Size = (tU32)s32BtLdrSize;

      if(OSAL_ERROR == OSAL_s32IOControl( hTeseoDeviceHandle, OSAL_C_S32_IOCTL_GNSS_FLASH_IMAGE, (tLong)&rImageOptions) )
      {
         s32RetVal |= 512;
      }
   }
   /* Send Teseo Boot loader */
   if ( 0 == s32RetVal)
   {
      s32RetVal =  s32SendBinToTeseo(Xldrfp);
      if ( OSAL_OK != s32RetVal)
      {
         s32RetVal |= 1024;
      }
   }
   if ( 0 == s32RetVal)
   {
      /* Get the CRC for Teseo Firmware */
      if ( OSAL_OK != GnssOedt_u32GetTeseoFwCrc32( TesFwfp ,&rImageOptions.u32ChkSum, (tU32)s32FwImgSize ) )
      {
                 s32RetVal |= 2048;
      }
   }

   if ( 0 == s32RetVal)
   {
      /* Configuration to send Teseo Firmware */
      rImageOptions.u32Size = (tU32)s32FwImgSize;
      rImageOptions.enType = IMAGE_TESEO_FIRMWARE;
      if(OSAL_ERROR == OSAL_s32IOControl( hTeseoDeviceHandle, OSAL_C_S32_IOCTL_GNSS_FLASH_IMAGE, (tLong)&rImageOptions) )
      {
         s32RetVal |= 4096;
      }
   }


   /* Send Teseo Firmware */
   if ( 0 == s32RetVal)
   {
      s32RetVal = s32SendBinToTeseo(TesFwfp);
      if ( OSAL_OK != s32RetVal)
      {
         s32RetVal |= 8192;
      }
	   
   }
  
   /*Close GNSS Proxy*/
   if ( OSAL_s32IOClose( hTeseoDeviceHandle ) == OSAL_ERROR )
   {
      s32RetVal |= 16384;
   }
   if( TesFwfp != OSAL_NULL)
   {
	   fclose(TesFwfp);
   }
   if( Xldrfp != OSAL_NULL)
   {
	   fclose(Xldrfp);
   }
   return (tU32)s32RetVal;

}

/***************************************************************************
||FUNCTION    :  tU32  GnssOedt_u32GetTeseoFlasherCrc32(tVoid)
||
||DESCRIPTION :  Calculates CRC32 for fp for teseo boot loader
||PARAMETER   :  FILE * fp
||
||RETURNVALUE :  tU32
||               0          - success
||               Error code - failure
||HISTORY     :
||
***************************************************************************/
static tU32 GnssOedt_u32GetTeseoFlasherCrc32( FILE *fp, tU32 *pu32FinalCrc, tU32 u32ImageLength )
{
   tS32 s32RetVal =  OSAL_OK;
   tU32 u32Crc32 = 0;
   tU32 u32EntryAdr = 0;
   
   if ((OSAL_NULL == pu32FinalCrc) || (OSAL_NULL == fp) )
   {
      s32RetVal = OSAL_ERROR;
   }
   else
   {
      /* calculate CRC on Image size field first*/
      u32Crc32 = u32_crc32( u32Crc32, (const char *)&u32ImageLength, (tU32)sizeof(u32ImageLength) );
      /* calculate CRC on Flasher entry address field */
      u32Crc32 = u32_crc32( u32Crc32, (const char *)&u32EntryAdr, (tU32)sizeof(u32EntryAdr) );
      rewind (fp);
      /* calculate crc for full binary */
      do
      {
         s32RetVal = (tS32)(fread( u8TesFwBuff, 1, FIRMWARE_UPDATE_BUFF_SIZE,fp ));
         if( 0 >= s32RetVal )
         {
            if (0 != feof(fp))
               s32RetVal = OSAL_ERROR;
            break;
         }
         else
         {
            u32Crc32 = u32_crc32( u32Crc32, (const char *)u8TesFwBuff, (tU32)s32RetVal );
            s32RetVal = OSAL_OK;
         }

      }while (0 == feof(fp));

      if (OSAL_OK == s32RetVal)
      {
         *pu32FinalCrc = u32Crc32;
      }
      rewind (fp);
   }
   return (tU32)s32RetVal;
}

/***************************************************************************
||FUNCTION    :  tS32  s32SendBinToTeseo(tVoid)
||
||DESCRIPTION :  This uses write calls to transfer binary image
||PARAMETER   :  handle to binary file
||
||RETURNVALUE :  tU32
||               0          - success
||               Error code - failure
||HISTORY     :
||
***************************************************************************/
static tS32  s32SendBinToTeseo(FILE *fp)
{

   tS32 s32RetVal =  OSAL_OK;
   if( fp != OSAL_NULL  )
   {
      do
      {
         s32RetVal = (tS32)(fread( u8TesFwBuff, 1, FIRMWARE_UPDATE_BUFF_SIZE,fp ));
         if( 0 >= s32RetVal )
         {
            if (0 != feof(fp))
               s32RetVal = OSAL_ERROR;
            break;
         }

         if (s32RetVal != OSAL_s32IOWrite( hTeseoDeviceHandle, (tPCS8)u8TesFwBuff, (tU32)s32RetVal))
         {
            s32RetVal = OSAL_ERROR;
            break;
         }
          else
         {
            s32RetVal =  OSAL_OK;
         }
      }while ( 0 == feof(fp) ) ;
   }
   else
   {
      s32RetVal = OSAL_ERROR;
   }

   return s32RetVal;
}

/***************************************************************************
||FUNCTION    :  tU32  GnssOedt_u32GetTeseoFwCrc32(tVoid)
||
||DESCRIPTION :  Calculates CRC32 for fp of teseo-firmware
||PARAMETER   :  FILE * fp
||
||RETURNVALUE :  tU32
||               0          - success
||               Error code - failure
||HISTORY     :
||
***************************************************************************/
static tU32 GnssOedt_u32GetTeseoFwCrc32( FILE *fp, tU32 *pu32FinalCrc, tU32 u32ImageLength )
{
   tS32 s32RetVal =  OSAL_OK;
   tU32 u32Crc32 = 0;

   if ((OSAL_NULL == pu32FinalCrc) || (OSAL_NULL == fp) )
   {
      s32RetVal = OSAL_ERROR;
   }
   else
   {
      /* calculate CRC on Image size field first*/
      u32Crc32 = u32_crc32( u32Crc32, (const char *)&u32ImageLength, (tU32)sizeof(tU32) );
      rewind (fp);
      do
      {
         s32RetVal = (tS32)(fread( u8TesFwBuff, 1, FIRMWARE_UPDATE_BUFF_SIZE,fp ));
         if( 0 >= s32RetVal )
         {
            if (0 != feof(fp))
               s32RetVal = OSAL_ERROR;
            break;
         }
         else
         {
            u32Crc32 = u32_crc32( u32Crc32, (const char *)u8TesFwBuff, (tU32)s32RetVal );
            s32RetVal = OSAL_OK;
         }

      }while (0 == feof(fp));

      if (OSAL_OK == s32RetVal)
      {
         *pu32FinalCrc = u32Crc32;
      }
      rewind (fp);
   }
   return (tU32)s32RetVal;
}

/***************************************************************************
||FUNCTION    :  u32_crc32
||
||DESCRIPTION :  Calculates CRC32
||PARAMETER   :  CRC, buffer, size,
||
||RETURNVALUE :  tU32
||               0          - success
||               Error code - failure
||HISTORY     :
||
***************************************************************************/
static tU32  u32_crc32(tU32 crc32val, const char *buf, tU32 len)
{
   tU32 i;
   
   if (buf == 0) return 0L;
   
   crc32val = crc32val ^ 0xffffffff;
   for (i = 0;  i < len;  i++) 
   {
      crc32val = crc32_tab[(crc32val ^ buf[i]) & 0xff] ^ (crc32val >> 8);
   }
  return crc32val ^ 0xffffffff;
}

tU32 GnssTestFunctions::GnssOedt_u32Teseo2FwUpdate(tVoid)
{
   FILE *Xldrfp ;
   FILE *TesFwfp = OSAL_NULL;
   tS32 s32BtLdrSize = 0;
   tS32 s32FwImgSize = 0;
   tS32 s32RetVal = 0;
   trImageOptions rImageOptions;

/* Open teseo boot loader */
   Xldrfp = fopen( X_LOADER_BIN_PATH, "r" );
   if ( OSAL_NULL == Xldrfp )
   {
      s32RetVal = 1;
   }

   /* open Teseo firmware image */
   if ( 0 == s32RetVal)
   {
      TesFwfp = fopen( TESEO_FW_BIN_PATH, "r" );
      if ( OSAL_NULL == TesFwfp )
      {
          s32RetVal = 2;
      }
   }
   /* Get the size of boot loader */
   if ( ( Xldrfp != OSAL_NULL )&& (0 == s32RetVal))      /*-----------changes made to remove lint-----------*/
   {
      if( 0 != fseek( Xldrfp,0L, SEEK_END ) )
      {
         s32RetVal = 4;
      }
   }

   if ( ( Xldrfp != OSAL_NULL ) && (0 == s32RetVal))     /*-----------changes made to remove lint-----------*/
   {
      s32BtLdrSize = ftell (Xldrfp);
      if ( 0 == s32BtLdrSize )
      {
         s32RetVal = 8;
      }
      /* file position indicator to the Beginning */
      rewind(Xldrfp);
   }

   /* Get the size of firmware */
   if ( ( TesFwfp != OSAL_NULL) && (0 == s32RetVal))     /*-----------changes made to remove lint-----------*/
   {
      if( 0 != fseek( TesFwfp,0L, SEEK_END ) )
      {
         s32RetVal = 16;
      }
   }
   if ( ( TesFwfp != OSAL_NULL) && (0 == s32RetVal))     /*-----------changes made to remove lint-----------*/
   {
      s32FwImgSize = ftell (TesFwfp);
      if ( 0 == s32FwImgSize )
      {
         s32RetVal = 32;
      }

      /*file position indicator to the Beginning */
      rewind(TesFwfp);
   }

#if 1
   tU32 u32crc = 0;

   /* Open GNSS device for flashing */
   if ( 0 == s32RetVal)
   {
      hTeseoDeviceHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_WRITEONLY );
      if ( hTeseoDeviceHandle == OSAL_ERROR)
      {
         s32RetVal = 64;
      }
   }
   if(OSAL_ERROR == OSAL_s32IOControl( hTeseoDeviceHandle, OSAL_C_S32_IOCTL_GNSS_GET_GNSS_CHIP_CRC, (tLong)&u32crc) )
   {
 
   }

   /* Configuration to send Teseo Boot loader */
   if ( 0 == s32RetVal)
   {
      rImageOptions.enType = IMAGE_TESEO_FLASHER_BIN;
      rImageOptions.u32Size = (tU32)s32BtLdrSize;

      if(OSAL_ERROR == OSAL_s32IOControl( hTeseoDeviceHandle, OSAL_C_S32_IOCTL_GNSS_FLASH_IMAGE, (tLong)&rImageOptions) )
      {
         s32RetVal = 128;
      }
   }
   /* Send Teseo Boot loader */
   if ( 0 == s32RetVal)
   {
      s32RetVal =  s32SendBinToTeseo(Xldrfp);
      if ( OSAL_OK != s32RetVal)
      {
         s32RetVal = 256;
      }
   }
   if ( 0 == s32RetVal)
   {
      /* Get the CRC for Teseo Firmware */
      if ( OSAL_OK != GnssOedt_u32GetTeseoFwCrc32( TesFwfp ,&rImageOptions.u32ChkSum, (tU32)s32FwImgSize ) )
      {
                 s32RetVal = 512;
      }
   }

   if ( 0 == s32RetVal)
   {
      /* Configuration to send Teseo Firmware */
      rImageOptions.u32Size = (tU32)s32FwImgSize;
      rImageOptions.enType = IMAGE_TESEO_FIRMWARE;
      if(OSAL_ERROR == OSAL_s32IOControl( hTeseoDeviceHandle, OSAL_C_S32_IOCTL_GNSS_FLASH_IMAGE, (tLong)&rImageOptions) )
      {
         s32RetVal = 1024;
      }
   }


   /* Send Teseo Firmware */
   if ( 0 == s32RetVal)
   {
      s32RetVal = s32SendBinToTeseo(TesFwfp);
      if ( OSAL_OK != s32RetVal)
      {
         s32RetVal = 2048;
      }
   }


   /*Close GNSS Proxy*/
   if ( hTeseoDeviceHandle !=OSAL_ERROR && OSAL_s32IOClose( hTeseoDeviceHandle ) == OSAL_ERROR )
   {
      s32RetVal = 4096;
   }
#endif
   if( TesFwfp != OSAL_NULL)
   {
	   fclose(TesFwfp);
   }
   if( Xldrfp != OSAL_NULL)
   {
	   fclose(Xldrfp);
   }
   return (tU32)s32RetVal;
}

/***************************************************************************
||FUNCTION    :  tU32  GnssOedt_u32GetNmeaMsgList(tVoid)
||
||DESCRIPTION :  Gets the List of NMEA messages used in current config
||               Compares it with the expected list of NMEA messages.
||PARAMETER   :  Nil
||
||RETURNVALUE :  tU32
||               0          - success if (received list == expected list)
||               Error code - failure
||HISTORY     :
||
***************************************************************************/
tU32 GnssTestFunctions::GnssOedt_u32GetNmeaMsgList(tVoid)
{
   OSAL_trThreadAttribute rThreadAttr;
   tU32 u32ResultMask;
   u32NmeaListRetVal = 0;
   /*Update thread attributes*/
   rThreadAttr.szName       = GNSS_OEDT_NMEA_MSG_LIST_THREAD_NAME;
   rThreadAttr.u32Priority  = OSAL_C_U32_THREAD_PRIORITY_NORMAL;
   rThreadAttr.s32StackSize = GNSS_OEDT_NMEA_LIST_THREAD_STACK_SIZE;
   rThreadAttr.pfEntry      = GnssOedt_vNmeaListThread;
   rThreadAttr.pvArg        = OSAL_NULL;

   /* Wait for previous test to complete. GNSS driver thread may take some time to shut down */
   OSAL_s32ThreadWait(2000);
   
   if( OSAL_OK != OSAL_s32EventCreate( GNSS_OEDT_NMEA_LIST_EVENT_NAME,
                                       &hGnssOedtNmeaListEvent ))
   {
      u32NmeaListRetVal += 2;
   }
   else
   {
      if(OSAL_ERROR == OSAL_ThreadSpawn(&rThreadAttr) )
      {
         u32NmeaListRetVal += 4;
      }
      else if ( OSAL_ERROR == OSAL_s32EventWait( hGnssOedtNmeaListEvent,
                                                 (GNSS_OEDT_NMEA_LIST_PASSED | GNSS_OEDT_NMEA_LIST_FAILED),
                                                 OSAL_EN_EVENTMASK_OR,
                                                 GNSS_NMEA_LIST_WAIT_TIME_MS,
                                                 &u32ResultMask ))
      {
         if ( OSAL_E_TIMEOUT == OSAL_u32ErrorCode() )
         {
            u32NmeaListRetVal += 8;
         }
         else
         {
            u32NmeaListRetVal += 16;
         }
      }
      /* Event wait returned with a success, Clear the event */
      else if(OSAL_ERROR == OSAL_s32EventPost( hGnssOedtNmeaListEvent,
                                               (OSAL_tEventMask) ~u32ResultMask,
                                               OSAL_EN_EVENTMASK_AND))
      {
         u32NmeaListRetVal += 32;
      }

      if( OSAL_OK != OSAL_s32EventClose( hGnssOedtNmeaListEvent ))
      {
         u32NmeaListRetVal += 8192;
      }
      else if ( OSAL_OK != OSAL_s32EventDelete( GNSS_OEDT_NMEA_LIST_EVENT_NAME ))
      {
         u32NmeaListRetVal += 16384;
      }
   }

   return u32NmeaListRetVal;
}

/***************************************************************************
||FUNCTION    :  tU32  GnssOedt_vNmeaListThread(tPVoid)
||
||DESCRIPTION :  Gets the List of NMEA messages used in current config
||               Compares it with the expected list of NMEA messages.
||PARAMETER   :  Nil
||
||RETURNVALUE :  None
||
||HISTORY     :
||
***************************************************************************/
static tVoid  GnssOedt_vNmeaListThread(tPVoid pVDummyArg )
{
   OSAL_tIODescriptor hDevHandle;
   tU32 u32NmeaList =0;
   u32NmeaListRetVal = 0;
   (tVoid)pVDummyArg;   /*-------for lint fix -------*/
   
   /*------------------------------------------Open the device--------------------------------------------------*/
   hDevHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_READONLY );
   if ( hDevHandle == OSAL_ERROR)
   {
      u32NmeaListRetVal += 64;
   }
   else
   {
      /*-----------------------------------------Get the NMEA LIST---------------------------------------------------*/
      /* Wait for some time so that messages are received */
      OSAL_s32ThreadWait(5000);
      // Flush GNSS buffer
      if(OSAL_ERROR == OSAL_s32IOControl( hDevHandle, OSAL_C_S32_IOCTL_GNSS_FLUSH_SENSOR_DATA, 0) )
      {
         u32NmeaListRetVal += 32768;
      }
      /* Get NMEA message list */
      if ( 0 == u32NmeaListRetVal )
      {
         if(OSAL_ERROR == OSAL_s32IOControl( hDevHandle,
                                             OSAL_C_S32_IOCTL_GNSS_GET_NMEA_RECVD_LIST,
                                             (tLong)&u32NmeaList) )
         {
            /* Some times UART communication between Teseo and V850 may be delayed
            * So wait for some time and retry */
            OSAL_s32ThreadWait(2000);
            if(OSAL_ERROR == OSAL_s32IOControl( hDevHandle, OSAL_C_S32_IOCTL_GNSS_GET_NMEA_RECVD_LIST, (tLong)&u32NmeaList) )
            {
               u32NmeaListRetVal += 128;
            }
         }
      }
      /*-----------------------------------Retry in case of checksum error--------------------------------*/
      if ( 0 == u32NmeaListRetVal )
      {
         if ( u32NmeaList & GNSS_OEDT_CHECKSUM_ERROR ) //lint !e845, !e774
         {
            u32NmeaList = 0;
            /* Try getting the list for next fix */
            OSAL_s32ThreadWait(2000);
            if(OSAL_ERROR == OSAL_s32IOControl( hDevHandle,
                                                OSAL_C_S32_IOCTL_GNSS_GET_NMEA_RECVD_LIST,
                                                (tLong)&u32NmeaList) )
            {
               u32NmeaListRetVal += 256;
            }
         }
      }
      /*-----------------------------------Compare Received NMEA List with expected list--------------------------------*/
      if ( 0 == u32NmeaListRetVal )
      {
         if ( (u32NmeaList & GNSS_PROXY_OEDT_EXPEC_NMEA_LIST) != GNSS_PROXY_OEDT_EXPEC_NMEA_LIST ) //lint !e845, !e774 PQM_authorized_551
         {
            u32NmeaListRetVal +=512;
         }
      }
      /*------------------------------------------CLOSE the device--------------------------------------------------*/
      if ( OSAL_ERROR == OSAL_s32IOClose(hDevHandle) )
      {
         u32NmeaListRetVal +=1024;
      }
   }
   /*---------------------------------------------Post success or failure event---------------------------------------*/
   if ( 0 == u32NmeaListRetVal )
   {
      if(OSAL_ERROR == OSAL_s32EventPost( hGnssOedtNmeaListEvent,
                                          GNSS_OEDT_NMEA_LIST_PASSED,
                                          OSAL_EN_EVENTMASK_OR ))
      {
         u32NmeaListRetVal +=2048;
      }
   }
   else if(OSAL_ERROR == OSAL_s32EventPost( hGnssOedtNmeaListEvent,
                                            GNSS_OEDT_NMEA_LIST_FAILED,
                                            OSAL_EN_EVENTMASK_OR ))
   {
      u32NmeaListRetVal +=4096;
   }
}

/***************************************************************************
||FUNCTION    :  tU32  Gnss_u32VerifyGnssQualityon3DFix(tPVoid)
||
||DESCRIPTION :  To verify Gnss Quality when 3D Fix is available
||               
||PARAMETER   :  Nil
||
||RETURNVALUE :  tU32
||               0          - success if (received list == expected list)
||               Error code - failure
||
||HISTORY     :
||
***************************************************************************/
tU32 GnssTestFunctions :: Gnss_u32VerifyGnssQualityon3DFix(tVoid)
{
   OSAL_tIODescriptor hDevHandle;
   OSAL_trGnssFullRecord rGnssFullRecord;
   tU32 u32RetVal = GNSS_SUCCESS_RETVAL;
   
   OSAL_pvMemorySet(&rGnssFullRecord,0,sizeof(OSAL_trGnssFullRecord));
   
   /*open Gnss Proxy*/
   hDevHandle = OSAL_IOOpen(OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_READONLY);
   if(hDevHandle == OSAL_ERROR)
   {
      u32RetVal = 1;
   }
   else
   {
      if(OSAL_ERROR == OSAL_s32IOControl(hDevHandle,OSAL_C_S32_IOCTL_GNSS_FLUSH_SENSOR_DATA,0))
      {
         u32RetVal += 2;
      }
      else if(OSAL_ERROR == OSAL_s32IORead(hDevHandle,(tPS8)&rGnssFullRecord,sizeof(OSAL_trGnssFullRecord)))
      {
         u32RetVal += 4;
      }
      else
      {
         if( GNSS_FIX_TYPE_3D == rGnssFullRecord.rPVTData.rFixStatus.enMode)
         {
            if((GNSSQUALITY_AUTONOMOUS != rGnssFullRecord.rPVTData.rFixStatus.enQuality) && 
                   (GNSSQUALITY_DIFFERENTIAL != rGnssFullRecord.rPVTData.rFixStatus.enQuality))
            {
               u32RetVal += 8;
            }
         }
      }
      /*Close Gnss Proxy*/
      if(OSAL_ERROR == OSAL_s32IOClose(hDevHandle))
      {
         u32RetVal += 16;
      }
   }
   
   return u32RetVal;
}
      
/************************************************************************
||FUNCTION    :  GNSSProxyBasiOpenClose_Test
||
||DESCRIPTION :  Test Case-1
||               Checks Basic Open And Close Interfaces Of GNSS Proxy.
||
||PARAMETER   :  Nil
************************************************************************/
TEST_F(GnssTestFunctions , GNSSProxyBasicOpenClose_Test)
{
   ASSERT_NE(OSAL_ERROR,(hDevHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_READONLY )))<<"Failed to open device file"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_NE(OSAL_ERROR,OSAL_s32ThreadWait(10))<<"Failed to execute threadwait"<< " Error code="<< OSAL_u32ErrorCode();
   ASSERT_NE(OSAL_ERROR,OSAL_s32IOClose( hDevHandle ))<<"Failed to close device file"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
}

/************************************************************************
||FUNCTION    :  GNSSProxyMultipleIOOpenIOClose_Test
||
||DESCRIPTION :  Test Case-2
||               Performs multiple IO Open and Close back to back .
||
||PARAMETER   :  Nil
************************************************************************/

TEST_F(GnssTestFunctions , GNSSProxyMultipleIOOpenIOClose_Test)
{
   tU32 u32OpenClsCount;
   for(u32OpenClsCount=0;(u32OpenClsCount<GNSS_OEDT_MAX_IOOPN_CLS_CNT);u32OpenClsCount++)
   {
      ASSERT_NE(OSAL_ERROR,(hDevHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_READONLY )))<<"Failed to open device file"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
      EXPECT_NE(OSAL_ERROR,OSAL_s32ThreadWait(10))<<"Failed to execute threadwait"<< " Error code="<< OSAL_u32ErrorCode();
      ASSERT_NE(OSAL_ERROR,OSAL_s32IOClose( hDevHandle ))<<"Failed to close device file"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
   }
}

/***************************************************************************
||FUNCTION    :  GNSSProxyReOpen_Test
||
||DESCRIPTION :  Test Case-3
||               Checks Whether GNSS Proxy Allowing Multiple Opens.If Allows
||               Then It will Be Consider As Error. 
||PARAMETER   :  Nil
***************************************************************************/
TEST_F(GnssTestFunctions , GNSSProxyReOpen_Test)
{
   OSAL_tIODescriptor hDevHandle_reopen;
   ASSERT_NE(OSAL_ERROR,(hDevHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_READONLY )))<<"Failed to open device file"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_NE(OSAL_ERROR,OSAL_s32ThreadWait(10))<<"Failed to execute threadwait"<< " Error code="<< OSAL_u32ErrorCode();
   EXPECT_EQ(OSAL_ERROR,(hDevHandle_reopen = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_READONLY )))<<"device reopen successful"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
   ASSERT_NE(OSAL_ERROR,OSAL_s32IOClose( hDevHandle ))<<"Failed to close device file"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
}

/***************************************************************************
||FUNCTION    :  GNSSProxyReClose_Test
||
||DESCRIPTION :  Test Case-4
||               Checks Whether GNSS Proxy Allowing Multiple Closes.If Allows
||               Then It will Be Consider As Error. 
||PARAMETER   :  Nil
***************************************************************************/
TEST_F(GnssTestFunctions , GNSSProxyReClose_Test)
{
   ASSERT_NE(OSAL_ERROR,(hDevHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_READONLY )))<<"Failed to open device file"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_NE(OSAL_ERROR,OSAL_s32ThreadWait(10))<<"Failed to execute threadwait"<< " Error code="<< OSAL_u32ErrorCode();
   ASSERT_NE(OSAL_ERROR,OSAL_s32IOClose( hDevHandle ))<<"Failed to close device file"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_NE(OSAL_ERROR,OSAL_s32ThreadWait(10))<<"Failed to execute threadwait"<< " Error code="<< OSAL_u32ErrorCode();
   ASSERT_EQ(OSAL_ERROR,OSAL_s32IOClose( hDevHandle ))<<"Device reclose successful"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
}

/***************************************************************************
||FUNCTION    :  GNSSProxyBasicRead_Test
||
||DESCRIPTION :  Test Case-5
||               Checks Read Interface Of GNSS Proxy.
||
||PARAMETER   :  Nil
***************************************************************************/
TEST_F(GnssTestFunctions , GNSSProxyBasicRead_Test)
{   
   OSAL_pvMemorySet(&rGnssFullRecord,0,sizeof(OSAL_trGnssFullRecord));
   ASSERT_NE(OSAL_ERROR,(hDevHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_READONLY )))<<"Failed to open device file"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_NE(OSAL_ERROR,OSAL_s32IOControl( hDevHandle, OSAL_C_S32_IOCTL_GNSS_FLUSH_SENSOR_DATA, 0))<<"Failed to Flush POS sensor data"<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_NE(OSAL_ERROR,OSAL_s32IORead(hDevHandle,(tPS8)&rGnssFullRecord,sizeof(OSAL_trGnssFullRecord)))<<"Read failed"<<" Error code="<< OSAL_u32ErrorCode();
   ASSERT_NE(OSAL_ERROR,OSAL_s32IOClose( hDevHandle ))<<"Failed to close device file"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
}

/***************************************************************************
||FUNCTION    :  GNSSProxyReadByPassingNullBuffer_Test
||
||DESCRIPTION :  Test Case-6
||               Checks Read Interface Of GNSS Proxy.
||
||PARAMETER   :  Nil
***************************************************************************/
TEST_F(GnssTestFunctions , GNSSProxyReadByPassingNullBuffer_Test)
{   
   ASSERT_NE(OSAL_ERROR,(hDevHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_READONLY )))<<"Failed to open device file"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_NE(OSAL_ERROR,OSAL_s32IOControl( hDevHandle, OSAL_C_S32_IOCTL_GNSS_FLUSH_SENSOR_DATA, 0))<<"Failed to Flush POS sensor data"<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_EQ(OSAL_ERROR,OSAL_s32IORead (hDevHandle,(tS8 *)OSAL_NULL,0))<<"Read failed, as expected"<<" Error code="<< OSAL_u32ErrorCode();;
   ASSERT_NE(OSAL_ERROR,OSAL_s32IOClose( hDevHandle ))<<"Failed to close device file"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
}

/***************************************************************************
||FUNCTION    :  GNSSProxyIOCntInterfaceCheck_Test
||
||DESCRIPTION :  Test Case-7
||               Checks Read Interface Of GNSS Proxy.
||
||PARAMETER   :  Nil
***************************************************************************/
TEST_F(GnssTestFunctions , GNSSProxyIOCntInterfaceCheck_Test)
{
   tU32 u32Arg = OSAL_C_U8_GNSS_SATSYS_GPS|OSAL_C_U8_GNSS_SATSYS_GALILEO|OSAL_C_U8_GNSS_SATSYS_GLONASS;
   OSAL_pvMemorySet(&rGnssConfigData, 0, sizeof(rGnssConfigData));
   ASSERT_NE(OSAL_ERROR,(hDevHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_READONLY )))<<"Failed to open device file"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_NE(OSAL_ERROR,OSAL_s32IOControl( hDevHandle, OSAL_C_S32_IOCTL_GNSS_FLUSH_SENSOR_DATA, 0))<<"Failed to Flush POS sensor data"<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_NE(OSAL_ERROR,OSAL_s32IOControl( hDevHandle, OSAL_C_S32_IOCTRL_GNSS_SET_SAT_SYS, ( tLong )&u32Arg))<<"OSAL_C_S32_IOCTRL_GNSS_SET_SAT_SYS Failed"<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_NE(OSAL_ERROR,OSAL_s32IOControl( hDevHandle, OSAL_C_S32_IOCTRL_GNSS_GET_SAT_SYS, ( tLong )&u32Arg))<<"OSAL_C_S32_IOCTRL_GNSS_GET_SAT_SYS Failed"<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_NE(OSAL_ERROR,OSAL_s32IOControl( hDevHandle, OSAL_C_S32_IOCTRL_GNSS_GET_CONFIG_DATA, ( tLong )&rGnssConfigData))<<"OSAL_C_S32_IOCTRL_GNSS_GET_CONFIG_DATA Failed"<<" Error code="<< OSAL_u32ErrorCode();
   ASSERT_NE(OSAL_ERROR,OSAL_s32IOClose( hDevHandle ))<<"Failed to close device file"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
}

/***************************************************************************
||FUNCTION    :  GNSSProxyIOCntChkInvPar_Test
||
||DESCRIPTION :  Test Case-8
||               Checks Read Interface Of GPS Proxy.
||
||PARAMETER   :  Nil
***************************************************************************/
TEST_F(GnssTestFunctions , GNSSProxyIOCntChkInvPar_Test)
{
   tU32 u32Arg = 0;
   // open device
   ASSERT_NE(OSAL_ERROR,(hDevHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_READONLY )))<<"Failed to open device file"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_NE(OSAL_ERROR,OSAL_s32IOControl( hDevHandle, OSAL_C_S32_IOCTL_GNSS_FLUSH_SENSOR_DATA, 0))<<"Failed to Flush POS sensor data"<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_EQ(OSAL_ERROR,OSAL_s32IOControl( hDevHandle, OSAL_C_S32_IOCTRL_GNSS_SET_SAT_SYS, ( tLong )&u32Arg))<<"OSAL_C_S32_IOCTRL_GNSS_SET_SAT_SYS Failed"<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_EQ(OSAL_ERROR,OSAL_s32IOControl( hDevHandle, OSAL_C_S32_IOCTRL_GNSS_SET_SAT_SYS, OSAL_NULL))<<"OSAL_C_S32_IOCTRL_GNSS_SET_SAT_SYS Failed"<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_EQ(OSAL_ERROR,OSAL_s32IOControl( hDevHandle, OSAL_C_S32_IOCTRL_GNSS_GET_SAT_SYS, OSAL_NULL))<<"OSAL_C_S32_IOCTRL_GNSS_GET_SAT_SYS Failed"<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_EQ(OSAL_ERROR,OSAL_s32IOControl( hDevHandle, OSAL_C_S32_IOCTRL_GNSS_GET_CONFIG_DATA, OSAL_NULL))<<"OSAL_C_S32_IOCTRL_GNSS_GET_CONFIG_DATA Failed"<<" Error code="<< OSAL_u32ErrorCode();
   ASSERT_NE(OSAL_ERROR,OSAL_s32IOClose( hDevHandle ))<<"Failed to close device file"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
}

/***************************************************************************
||FUNCTION    :  GNSSProxySetEpoch_Test
||
||DESCRIPTION : Test Case-9 
            Sets GNSS epoch
||
||PARAMETER   :  Nil
***************************************************************************/
TEST_F(GnssTestFunctions , GNSSProxySetEpoch_Test)
{
   tU8 u8epochValue = 0;
   OSAL_trGnssTimeUTC  rGPSTimeUTC[] = { {1999,8,21,0,0,0,0},
                                         {2014,12,31,0,0,0,0},
                                         {2015,1,1,0,0,0,0},
                                         {2030,8,1,0,0,0,0},
                                         {2014,8,21,0,0,0,0} };
   ASSERT_NE(OSAL_ERROR,(hDevHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_READONLY )))<<"Failed to open device file"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_NE(OSAL_ERROR,OSAL_s32IOControl( hDevHandle, OSAL_C_S32_IOCTL_GNSS_FLUSH_SENSOR_DATA, 0))<<"Failed to Flush POS sensor data"<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_NE(OSAL_ERROR,OSAL_s32ThreadWait(10))<<"Failed to execute threadwait"<< " Error code="<< OSAL_u32ErrorCode();
    do
      {
         EXPECT_NE(OSAL_ERROR,OSAL_s32IOControl( hDevHandle, OSAL_C_S32_IOCTL_GNSS_SET_EPOCH, (tLong)&rGPSTimeUTC[u8epochValue]))<<"OSAL_C_S32_IOCTL_GNSS_SET_EPOCH Failed"<<" Error code="<< OSAL_u32ErrorCode();
         u8epochValue++;
      }while( u8epochValue < 5 );
   ASSERT_NE(OSAL_ERROR,OSAL_s32IOClose( hDevHandle ))<<"Failed to close device file"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
}

/******************************************************************************
||FUNCTION    :  GNSSFwOpenCloseContinuously_Test
||
||DESCRIPTION : Test Case-10
            OPEN AND CLOSE GNSS FIRMWARE UPDATE DRIVER CONTINIOUSLY
||
||PARAMETER   :  Nil
******************************************************************************/

TEST_F(GnssTestFunctions , GNSSFwOpenCloseContinuously_Test)
{
   OSAL_tIODescriptor hTeseoDevHandle;
   tU32 u32LootCnt;
   for ( u32LootCnt =0;u32LootCnt < 10; u32LootCnt++ )
   {
      ASSERT_NE(OSAL_ERROR,(hTeseoDevHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_WRITEONLY )))<<"Failed to open device file "<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
      ASSERT_NE(OSAL_ERROR, OSAL_s32IOClose( hTeseoDevHandle ))<<"Failed to close device file "<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();   
   }   
}

/******************************************************************************
||FUNCTION    :  GNSSProxyInfiniteRead_Test
||
||DESCRIPTION :  Test Case-11
            Device infinite read, Never Ends and should not be mved to regression data set
||
||PARAMETER   :  Nil
******************************************************************************/
TEST_F(GnssTestFunctions , DISABLED_GNSSProxyInfiniteRead_Test)
{
   tBool bEntry = TRUE;
   char value ;
   std::cout<<"press q to exit! "<<std::endl;
   
   OSAL_pvMemorySet(&rGnssFullRecord,0,sizeof(OSAL_trGnssFullRecord));
   ASSERT_NE(OSAL_ERROR,(hDevHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_READONLY )))<<"Failed to open device file"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_NE(OSAL_ERROR,OSAL_s32IOControl( hDevHandle, OSAL_C_S32_IOCTL_GNSS_FLUSH_SENSOR_DATA, 0))<<"Failed to Flush POS sensor data"<<" Error code="<< OSAL_u32ErrorCode();
   while(bEntry)
   {
    EXPECT_NE(OSAL_ERROR,OSAL_s32IORead(hDevHandle,(tPS8)&rGnssFullRecord,sizeof(OSAL_trGnssFullRecord)))<<"Read failed"<<"Error code="<< OSAL_u32ErrorCode();
    std::cin >> value;
    if (value=='q')
    bEntry = false;
   }
   ASSERT_NE(OSAL_ERROR,OSAL_s32IOClose( hDevHandle ))<<"Failed to close device file"<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
}

/***************************************************************************
||FUNCTION    :  GnssTesGetCrc_Test
||
||DESCRIPTION : Test Case-12 
            Get the CRC of the Teseo firmware flashed in the device
||               
||PARAMETER   :  Nil
***************************************************************************/
TEST_F(GnssTestFunctions , GnssTesGetCrc_Test)
{
   OSAL_tIODescriptor hTeseoDevHandle;
   tU32 u32crc = 0;
   ASSERT_NE(OSAL_ERROR,(hTeseoDevHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GNSS, OSAL_EN_WRITEONLY )))<<"Failed to open device file "<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
   EXPECT_NE(OSAL_ERROR,OSAL_s32IOControl( hTeseoDevHandle, OSAL_C_S32_IOCTL_GNSS_GET_GNSS_CHIP_CRC, (tLong)&u32crc))<<"OSAL_C_S32_IOCTL_GNSS_GET_GNSS_CHIP_CRC Failed"<< OSAL_u32ErrorCode();
   ASSERT_NE(OSAL_ERROR, OSAL_s32IOClose( hTeseoDevHandle ))<<"Failed to close device file "<<OSAL_C_STRING_DEVICE_GNSS<<" Error code="<< OSAL_u32ErrorCode();
}


// Modular approach is complicated for below Test cases, hence not implemented

/******************************************************************************
||FUNCTION    :  GNSSSatInterChk_Test
||
||DESCRIPTION :  Test Case-13
             The OEDT is implemented to check the satellite interface 
||               system and also to set the satellite system to desired value
||               and check for the same if it is set to desired 
||               value. Finally we are setting to the initial value 
||
||PARAMETER   :  Nil
******************************************************************************/
TEST_F(GnssTestFunctions , GNSSSatInterChk_Test)
{
   ASSERT_EQ(GNSS_SUCCESS_RETVAL ,GNSSOedt_u32SatInterChk())<<"Google test case to check satellite interface Failed";
}

/******************************************************************************
||FUNCTION    :  GNSSSatCfgBlk200_Test
||
||DESCRIPTION :  Test Case-15
               This OEDT is implemented to check if all
||                       the satelllites in Cfg block 200 are set as reqd.
||
||PARAMETER   :  Nil
******************************************************************************/
TEST_F(GnssTestFunctions , GNSSSatCfgBlk200_Test)
{
   ASSERT_EQ(GNSS_SUCCESS_RETVAL ,GNSSOedt_u32SatCfgBlk200())<<"Google test case to check all satellites in Cfg block 200 Failed";
}

/******************************************************************************
||FUNCTION    :  GNSSChkSatCfgBlk227_Test
||
||DESCRIPTION :  Test Case-16
                  This OEDT is implemented to check if all
||                       the satelllites in Cfg block 227 are set as reqd.
||
||PARAMETER   :  Nil
******************************************************************************/
TEST_F(GnssTestFunctions , GNSSChkSatCfgBlk227_Test)
{
   ASSERT_EQ(GNSS_SUCCESS_RETVAL ,GNSSOedt_u32ChkSatCfgBlk227())<<"Google test case to check all satellites in Cfg block 227 Failed";
}

/******************************************************************************
||FUNCTION    :  GNSSChkSatCfgBlk200227_Test
||
||DESCRIPTION :  Test Case-17
               This OEDT is implemented to check if combination of
||                       the satelllites in Cfg block 200 and Cfg Block 227
||                       are set as reqd.
||
||PARAMETER   :  Nil
******************************************************************************/
TEST_F(GnssTestFunctions , GNSSChkSatCfgBlk200227_Test)
{
   ASSERT_EQ(GNSS_SUCCESS_RETVAL ,GNSSOedt_u32ChkSatCfgBlk200227())<<"Google test case to check all satellites in Cfg block 200 and Cfg block 227 Failed";
}

/************************************************************************
||FUNCTION       :  GNSSProxyGetEpoch_Test
||
||DESCRIPTION  :  Test Case-18
||                        Sets the value of epoch to 5 different values and reads back the set value each time.
||                        
||
||PARAMETER     :  Nil
************************************************************************/
TEST_F(GnssTestFunctions , GNSSProxyGetEpoch_Test)
{
   ASSERT_EQ(GNSS_SUCCESS_RETVAL ,u32OedtGNSSProxyGetEpoch())<<"Google Test case to Set the value of epoch to 5 different values and read back the set value each time Failed";
}

/***************************************************************************
||FUNCTION    :  GNSSTimestampValidation_Test
||
||DESCRIPTION :  Test Case-19
||               Checks Whether difference between consecutive Gnss Timestamp is within the 
||               limit of 990ms to 1010ms.In case the difference is beyond this limit, the number of such 
||              occurences must be within TIMESTAMP_DIFF_FAILURE_COUNT_LIMIT.
||  
||PARAMETER   :  Nil 
*************************************************************************/
TEST_F(GnssTestFunctions , DISABLED_GNSSTimestampValidation_Test)
{
   ASSERT_EQ(GNSS_SUCCESS_RETVAL ,GNSSOedt_u32TimestampValidation())<<"Google Test case to validate GNSS Timestamp failed";
}

/***************************************************************************
||FUNCTION    :  GnssTeseo3FwUpdate_Test
||
||DESCRIPTION :  Test Case-20
||               Do Teseo-3 firmware update
||PARAMETER   :  Nil
***************************************************************************/
TEST_F(GnssTestFunctions , DISABLED_GnssTeseo3FwUpdate_Test)
{
   ASSERT_EQ(GNSS_SUCCESS_RETVAL ,GnssOedt_u32Teseo3FwUpdate())<<"Google Test case to update Teseo3 firmware failed";
}

/***************************************************************************
||FUNCTION    :  GnssTeseo2FwUpdate_Test
||
||DESCRIPTION :  Test Case-21
               This can be used for GNSS firmware update
||PARAMETER   :  Nil
***************************************************************************/
TEST_F(GnssTestFunctions , DISABLED_GnssTeseo2FwUpdate_Test)
{
   ASSERT_EQ(GNSS_SUCCESS_RETVAL ,GnssOedt_u32Teseo2FwUpdate())<<"Google Test case to update Teseo2 firmware failed";
}

/***************************************************************************
||FUNCTION    : GnssGetNmeaMsgList_Test
||
||DESCRIPTION :  Test Case-22
            Gets the List of NMEA messages used in current config
||               Compares it with the expected list of NMEA messages.
||PARAMETER   :  Nil
***************************************************************************/
TEST_F(GnssTestFunctions , DISABLED_GnssGetNmeaMsgList_Test)
{
   ASSERT_EQ(GNSS_SUCCESS_RETVAL ,GnssOedt_u32GetNmeaMsgList())<<"Google Test case get Nmea Msglist failed";
}

/***************************************************************************
||FUNCTION    : GnssVerifyGnssQualityon3DFix_Test
||
||DESCRIPTION :  Test Case-23
                  To verify Gnss Quality when 3D Fix is available
            
||               
||PARAMETER   :  Nil
***************************************************************************/
TEST_F(GnssTestFunctions, GnssVerifyGnssQualityon3DFix_Test)
{
   ASSERT_EQ(GNSS_SUCCESS_RETVAL,Gnss_u32VerifyGnssQualityon3DFix())<<"Google Test case to verify Gnss Quality when 3Dfix is available failed";
}
