/******************************************************************************
 * FILE        :  OSAL_RPCMessageParser.cpp
 * SW-COMPONENT:  OSAL-RPC-SERVER
 *-----------------------------------------------------------------------------
 * DESCRIPTION :  Implementation of OSAL-RPC Message Parser class
 *-----------------------------------------------------------------------------
 * COPYRIGHT   :  (c) 2009-2010 Robert Bosch Engg. & Business Solns. Ltd.
 * HISTORY     :
 * Date       |   Modification            | Author
 *-----------------------------------------------------------------------------
 * 01/02/2009 |   Initial Version         | Soj Thomas, RBEI/ECF
 *****************************************************************************/
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "osal_rpcmessageparser.h"

#ifdef VARIANT_S_FTR_ENABLE_CPPUNIT_TEST
#include "messageparsertest.h"
#endif

tclOSAL_RPCMessageParser * pvGettclOSAL_RPCMessageParser()
{
   return new tclOSAL_RPCMessageParser();
}

/******************************************************************************
 Engineering and development support functions 
 *****************************************************************************/
#if OSAL_OS == OSAL_WINNT && !defined VARIANT_S_FTR_ENABLE_TSIM
    extern OSAL_DECL tS32 WOS_s32IOEngineeringFct( const char *device, tS32 fun, tS32 arg );
    typedef bool (*Osal_Engine_bEngFakeExclAccess)  (const char *szDevice);
    typedef bool (*Osal_Engine_bEngRemoveExclAccess)(const char *szDevice);
    extern Osal_Engine_bEngFakeExclAccess   pure_osal_io_bEngFakeExclAccess;
    extern Osal_Engine_bEngRemoveExclAccess pure_osal_io_bEngRemoveExclAccess;
    extern "C" void vTerminateCcaApplications();
#elif OSAL_OS == OSAL_LINUX || OSAL_OS==OSAL_DARWIN
    extern "C" void vTerminateCcaApplications();
    extern OSAL_DECL tS32 WOS_s32IOEngineeringFct( const char *device, tS32 fun, tS32 arg );
#endif

/******************************************************************************
 FUNCTION:   | tclOSAL_RPCMessageParser::tclOSAL_RPCMessageParser()
-------------------------------------------------------------------------------
 DESCRIPTION:| Constructor
-------------------------------------------------------------------------------
 PARAMETERS: | none
-------------------------------------------------------------------------------
 RETURN TYPE:| none
-------------------------------------------------------------------------------
 HISTORY:    | Date      | Author            | Modification
-------------------------------------------------------------------------------
             | 01.02.09  | Susmith, RBEI/ECF | Initial version
******************************************************************************/
tclOSAL_RPCMessageParser::tclOSAL_RPCMessageParser()
:m_poMessageSender(OSAL_NULL),
m_u32Fid(0),
m_u32FidPointer(0),
m_bFidComplete(FALSE),
m_pau8CopyBuffer(OSAL_NULL),
m_u32CopyBufferIndex(0),
m_u32CopyBufferSize(0),
m_enInternalState(EN_WAITING_FOR_FID),
m_localMsgCounter(0),
m_localThreadNo(0)
{
}
/******************************************************************************
 FUNCTION:   | tclOSAL_RPCMessageParser::~tclOSAL_RPCMessageParser()
-------------------------------------------------------------------------------
 DESCRIPTION:| Destructor
-------------------------------------------------------------------------------
 PARAMETERS: | none
-------------------------------------------------------------------------------
 RETURN TYPE:| none
-------------------------------------------------------------------------------
 HISTORY:    | Date      | Author            | Modification
-------------------------------------------------------------------------------
             | 01.02.09  | Susmith, RBEI/ECF | Initial version
******************************************************************************/
tclOSAL_RPCMessageParser::~tclOSAL_RPCMessageParser()
{
   m_pau8CopyBuffer = OSAL_NULL;
   m_poMessageSender = OSAL_NULL;
}

/******************************************************************************
 FUNCTION:   | tclOSAL_RPCMessageParser::vSetMessageSender()
-------------------------------------------------------------------------------
 DESCRIPTION:| Links the MessagePareser to a MessageSender
-------------------------------------------------------------------------------
 PARAMETERS: | tclIMessageSender *poMessageSender : [->I] MessageSender 
-------------------------------------------------------------------------------
 RETURN TYPE:| tVoid
-------------------------------------------------------------------------------
 HISTORY:    | Date      | Author            | Modification
-------------------------------------------------------------------------------
             | 01.02.09  | Susmith, RBEI/ECF | Initial version
******************************************************************************/
tVoid tclOSAL_RPCMessageParser::vSetMessageSender( tclIMessageSender *poMessageSender )
{
   if (poMessageSender != OSAL_NULL)
      this->m_poMessageSender = poMessageSender;
}

/******************************************************************************
 FUNCTION:   | tclOSAL_RPCMessageParser::bParseMessage()
-------------------------------------------------------------------------------
 DESCRIPTION:| The main function that parses the RPC message and completes
             | the OSAL Remote Procedure Call.
             | Calls bExecuteAndSendResult().
             | Called from MessageRouter library.
-------------------------------------------------------------------------------
 PARAMETERS: | tChar* pau8Buffer  : [->I] RPC Message Buffer
             | tU32 u32BufferSize : [I] Length of RPC Message Buffer
-------------------------------------------------------------------------------
 RETURN TYPE:| tVoid : not used
-------------------------------------------------------------------------------
 HISTORY:    | Date      | Author            | Modification
-------------------------------------------------------------------------------
             | 01.02.09  | Susmith, RBEI/ECF | Initial version from el_gateway
             | 01.02.09  | Soj, RBEI/ECF     | datat type modifications
******************************************************************************/
tBool tclOSAL_RPCMessageParser::bParseMessage( tPU8 pau8Buffer, tU32 u32BufferSize )
{
   tBool bMessageComplete = FALSE;

   if ( m_u32FidPointer < sizeof(m_u32Fid) ) // FID partially or not received
   {
      m_enInternalState = EN_WAITING_FOR_FID;

      if ( (u32BufferSize + m_u32FidPointer) >= sizeof(m_u32Fid) ) // Complete FID present in message
      {
         tPU8 pau8Fid = (tPU8)&m_u32Fid;

         OSAL_pvMemoryCopy( &pau8Fid[m_u32FidPointer], pau8Buffer, (sizeof(m_u32Fid) - m_u32FidPointer) );

         u32BufferSize -= (tU32)( sizeof(m_u32Fid) - m_u32FidPointer );
         pau8Buffer += ( sizeof(m_u32Fid) - m_u32FidPointer );

         m_u32FidPointer = sizeof(m_u32Fid);
         m_enInternalState = EN_FID_COMPLETE;
      }
      else //Partial FID present in message
      {
         tPU8 pau8Fid = (tPU8)&m_u32Fid;
         OSAL_pvMemoryCopy( &pau8Fid[m_u32FidPointer], pau8Buffer, u32BufferSize );
         m_u32FidPointer += u32BufferSize;
         u32BufferSize = 0;
      }
   }

   if ( m_u32FidPointer >= sizeof(m_u32Fid) ) // Complete FID Received
   {
      switch ( m_u32Fid )
      {
      case  OSAL_S32IOCREATE:
      case  OSAL_S32IOOPEN:
         {
            if ( FALSE == bAdjustCopyBuffer( IOCREATE_REQUEST_LENGTH ) )
               break;

            if ( bConsumeMessage( IOCREATE_ACCESS_REQUEST_LENGTH, pau8Buffer, u32BufferSize ) ) // Null terminated name buffer partially or not received
            {
               tU16 u16Index;
               tBool bEndOfString = FALSE;

               for ( u16Index = 0; u16Index < u32BufferSize; u16Index++ )
               {
                  if ( pau8Buffer[u16Index] == 0 )
                  {
                     bEndOfString = TRUE;
                     break;
                  }
               }

               if ( bEndOfString ) // Complete null terminated string present
               {
                  OSAL_pvMemoryCopy( &m_pau8CopyBuffer[m_u32CopyBufferIndex], pau8Buffer, (u16Index + 1) );
                  m_u32CopyBufferIndex += ( u16Index + 1 );
                  m_u32CopyBufferSize = m_u32CopyBufferIndex + 4;
                  u32BufferSize -= ( u16Index + 1 );
                  pau8Buffer    += ( u16Index + 1 );

                  bMessageComplete = TRUE;
               }
               else // Partial string present
               {
                  OSAL_pvMemoryCopy( &m_pau8CopyBuffer[m_u32CopyBufferIndex], pau8Buffer, u32BufferSize );
                  m_u32CopyBufferIndex += u32BufferSize;
                  u32BufferSize = 0;
               }
            }
         }
         break;

      case  OSAL_S32MESSAGEQUEUECREATE:
         {
            const size_t ctuiFixWidthParametersLength = sizeof( tU32 ) + sizeof( tU32 ) + sizeof( tU32/*OSAL_tenAccess*/ );
            if ( FALSE == bAdjustCopyBuffer( MESSAGEQUEUECREATE_REQUEST_LENGTH ) )
               break;

            tU16 u16Index;
            tBool bEndOfString = FALSE;
            for ( u16Index = 0; u16Index < m_u32CopyBufferIndex; u16Index++ )
            {
               if ( m_pau8CopyBuffer[u16Index] == 0 )
               {
                  bEndOfString = TRUE;
                  break;
               }
            }
            if ( bEndOfString )
            {
               m_u32CopyBufferSize = u16Index + 1 + ctuiFixWidthParametersLength;
            }
            else
            {
               size_t uiStringLength = 0;
               for ( u16Index = 0; u16Index < u32BufferSize; u16Index ++ )
               {
                  if ( pau8Buffer[u16Index] == 0 )
                  {
                     bEndOfString = TRUE;
                     break;
                  }
               }
               if ( bEndOfString )
               {
                  uiStringLength = (u16Index + 1);
                  OSAL_pvMemoryCopy( &m_pau8CopyBuffer[m_u32CopyBufferIndex], pau8Buffer, uiStringLength );
                  m_u32CopyBufferIndex += uiStringLength;
                  m_u32CopyBufferSize  = m_u32CopyBufferIndex + ctuiFixWidthParametersLength;
                  u32BufferSize        -= uiStringLength;
                  pau8Buffer           += uiStringLength;
               }
               else
               {
                  OSAL_pvMemoryCopy( &m_pau8CopyBuffer[m_u32CopyBufferIndex], pau8Buffer, u32BufferSize );
                  m_u32CopyBufferIndex += u32BufferSize;
                  u32BufferSize = 0;
               }
            }
            if ( u32BufferSize > 0 )
            {
               bMessageComplete = bConsumeMessage( m_u32CopyBufferSize, pau8Buffer, u32BufferSize );
            }
         }
         break;

      case  OSAL_S32MESSAGEQUEUEOPEN:
      case  OSAL_SHAREDMEMORYOPEN:
         {
            if ( FALSE == bAdjustCopyBuffer( MESSAGEQUEUESHMOPEN_REQUEST_LENGTH ) )
               break;

            tU16 u16Index;
            tBool bEndOfString = FALSE;
            for ( u16Index = 0; u16Index < m_u32CopyBufferIndex; u16Index++ )
            {
               if ( m_pau8CopyBuffer[u16Index] == 0 )
               {
                  bEndOfString = TRUE;
                  break;
               }
            }
            if ( bEndOfString )
            {
               m_u32CopyBufferSize = u16Index + 1 + sizeof( tU32/*OSAL_tenAccess*/ );
            }
            else
            {
               for ( u16Index = 0; u16Index < u32BufferSize; u16Index++ )
               {
                  if ( pau8Buffer[u16Index] == 0 )
                  {
                     bEndOfString = TRUE;
                     break;
                  }
               }
               if ( bEndOfString )
               {
                  OSAL_pvMemoryCopy( &m_pau8CopyBuffer[m_u32CopyBufferIndex], pau8Buffer, (u16Index + 1) );
                  m_u32CopyBufferIndex += ( u16Index + 1 );
                  m_u32CopyBufferSize = m_u32CopyBufferIndex + sizeof( tU32/*OSAL_tenAccess*/ );
                  u32BufferSize -= ( u16Index + 1 );
                  pau8Buffer    += ( u16Index + 1 );
               }
               else 
               {
                  OSAL_pvMemoryCopy( &m_pau8CopyBuffer[m_u32CopyBufferIndex], pau8Buffer, u32BufferSize );
                  m_u32CopyBufferIndex += u32BufferSize;
                  u32BufferSize = 0;
               }
            }
            if ( u32BufferSize > 0 )
            {
               bMessageComplete = bConsumeMessage( m_u32CopyBufferSize, pau8Buffer, u32BufferSize );
            }
         }
         break;

      case  OSAL_S32MESSAGEQUEUEDELETE:
      case  OSAL_S32SHAREDMEMORYDELETE:
      case  OSAL_S32IOREMOVE:
      case  ENG_FAKE_EXCL_ACCESS:
      case  ENG_REMOVE_EXCL_ACCESS:
         {
            if ( FALSE == bAdjustCopyBuffer( OSALRPC_REQUEST_MAXNAME_LENGTH ) )
               break;

            tU16 u16Index;
            tBool bEndOfString = FALSE;

            for ( u16Index = 0; u16Index < u32BufferSize; u16Index++ )
            {
               if ( pau8Buffer[u16Index] == 0 )
               {
                  bEndOfString = TRUE;
                  break;
               }
            }

            if ( bEndOfString )
            {
               OSAL_pvMemoryCopy( &m_pau8CopyBuffer[m_u32CopyBufferIndex], pau8Buffer, (u16Index + 1) );
               m_u32CopyBufferIndex += ( u16Index + 1 );
               m_u32CopyBufferSize = m_u32CopyBufferIndex;
               u32BufferSize -= ( u16Index + 1 );
               pau8Buffer    += ( u16Index + 1 );

               bMessageComplete = TRUE;
            }
            else 
            {
               OSAL_pvMemoryCopy( &m_pau8CopyBuffer[m_u32CopyBufferIndex], pau8Buffer, u32BufferSize );
               m_u32CopyBufferIndex += u32BufferSize;
               u32BufferSize = 0;
            }
         }
         break;

      case  OSAL_S32IOWRITE:
         {
            if ( FALSE == bAdjustCopyBuffer( IOWRITE_WRITEBUFFER_OFFSET ) )
               break;

            if ( m_u32CopyBufferIndex <= IOWRITE_WRITEBUFFER_OFFSET )
            {
               if ( bConsumeMessage( IOWRITE_WRITEBUFFER_OFFSET, pau8Buffer, u32BufferSize ) )
               {
                  ahl_tclStreamer oInputStream;
                  oInputStream.vSetStreamBuffer( (tPC8)m_pau8CopyBuffer, IOWRITE_WRITEBUFFER_OFFSET, IOWRITE_WRITEBUFFER_OFFSET );

                  OSAL_tIODescriptor rFileHandle;
                  tU32 u32MessageSize;

                  oInputStream >> rFileHandle;
                  oInputStream >> u32MessageSize;

                  delete[] m_pau8CopyBuffer;

                  m_pau8CopyBuffer = new tU8[IOWRITE_WRITEBUFFER_OFFSET + u32MessageSize];
                  if (m_pau8CopyBuffer == NULL ) break;
                  m_u32CopyBufferSize = IOWRITE_WRITEBUFFER_OFFSET + u32MessageSize;

                  ahl_tclStreamer oTotalInputStream;
                  oTotalInputStream.vSetStreamBuffer( (tPC8)m_pau8CopyBuffer, ( IOWRITE_WRITEBUFFER_OFFSET + u32MessageSize), 0 );

                  oTotalInputStream << rFileHandle;
                  oTotalInputStream << u32MessageSize;
               }
            }

            if ( u32BufferSize > 0 )
            {
               if ( u32BufferSize >= (m_u32CopyBufferSize - m_u32CopyBufferIndex) )
               {
                  OSAL_pvMemoryCopy( &m_pau8CopyBuffer[m_u32CopyBufferIndex],
                                     pau8Buffer,
                                     (m_u32CopyBufferSize - m_u32CopyBufferIndex) ); //lint !e662 !e669
                  u32BufferSize -= ( m_u32CopyBufferSize - m_u32CopyBufferIndex );
                  pau8Buffer    += ( m_u32CopyBufferSize - m_u32CopyBufferIndex );
                  m_u32CopyBufferIndex = m_u32CopyBufferSize;
                  bMessageComplete = TRUE;
               }
               else
               {
                  OSAL_pvMemoryCopy( &m_pau8CopyBuffer[m_u32CopyBufferIndex], pau8Buffer, u32BufferSize );
                  m_u32CopyBufferIndex += u32BufferSize;
                  u32BufferSize = 0;
                  bMessageComplete = FALSE;
               }
            }
         }
         break;

      case  OSAL_S32MESSAGEQUEUEPOST:
         {
            if ( FALSE == bAdjustCopyBuffer( MESSAGEQUEUEPOST_MESSAGE_OFFSET ) )
               break;

            if ( m_u32CopyBufferIndex <=  MESSAGEQUEUEPOST_MESSAGE_OFFSET )
            {
               if ( bConsumeMessage( MESSAGEQUEUEPOST_MESSAGE_OFFSET, pau8Buffer, u32BufferSize ) )
               {
                  ahl_tclStreamer oInputStream;
                  oInputStream.vSetStreamBuffer( (tPC8)m_pau8CopyBuffer, MESSAGEQUEUEPOST_MESSAGE_OFFSET, MESSAGEQUEUEPOST_MESSAGE_OFFSET );

                  OSAL_tMQueueHandle rMessageQueueHandle;
                  tU32 u32MessageSize;

                  oInputStream >> rMessageQueueHandle;
                  oInputStream >> u32MessageSize;

                  delete[] m_pau8CopyBuffer;

                  m_pau8CopyBuffer = new tU8[MESSAGEQUEUEPOST_MESSAGE_OFFSET + u32MessageSize + 4];
                  if (m_pau8CopyBuffer == NULL ) break;
                  m_u32CopyBufferSize = MESSAGEQUEUEPOST_MESSAGE_OFFSET + u32MessageSize + 4;

                  ahl_tclStreamer oTotalInputStream;
                  oTotalInputStream.vSetStreamBuffer( (tPC8)m_pau8CopyBuffer, (MESSAGEQUEUEPOST_MESSAGE_OFFSET + u32MessageSize + 4), 0 );

                  oTotalInputStream << rMessageQueueHandle;
                  oTotalInputStream << u32MessageSize;
               }
            }
            if ( u32BufferSize > 0 )
            {
               if ( u32BufferSize >= (m_u32CopyBufferSize - m_u32CopyBufferIndex) )
               {
                  OSAL_pvMemoryCopy( &m_pau8CopyBuffer[m_u32CopyBufferIndex],
                                     pau8Buffer,
                                     (m_u32CopyBufferSize - m_u32CopyBufferIndex) ); //lint !e826
                  u32BufferSize -= ( m_u32CopyBufferSize - m_u32CopyBufferIndex );
                  pau8Buffer    += ( m_u32CopyBufferSize - m_u32CopyBufferIndex );
                  m_u32CopyBufferIndex = m_u32CopyBufferSize;
                  bMessageComplete = TRUE;
               }
               else
               {
                  OSAL_pvMemoryCopy( &m_pau8CopyBuffer[m_u32CopyBufferIndex], pau8Buffer, u32BufferSize );
                  m_u32CopyBufferIndex += u32BufferSize;
                  u32BufferSize = 0;
                  bMessageComplete = FALSE;
               }
            }
         }
         break;

      case  OSAL_S32MESSAGEQUEUEWAIT:
         {
            if ( FALSE == bAdjustCopyBuffer( MESSAGEQUEUEWAIT_REQUEST_LENGTH ) )
               break;

            bMessageComplete = bConsumeMessage( m_u32CopyBufferSize, pau8Buffer, u32BufferSize );
         }
         break;

      case  OSALPIPE_S32IOCTRL:
         {
            if ( FALSE == bAdjustCopyBuffer( PIPEIOCTRL_ARGUMENT_OFFSET ) )
               break;

            if ( m_u32CopyBufferIndex <= PIPEIOCTRL_ARGUMENT_OFFSET )
            {
               if ( bConsumeMessage( PIPEIOCTRL_ARGUMENT_OFFSET, pau8Buffer, u32BufferSize ) )
               {
                  ahl_tclStreamer oInputStream;
                  oInputStream.vSetStreamBuffer( (tPC8)m_pau8CopyBuffer, PIPEIOCTRL_ARGUMENT_OFFSET, PIPEIOCTRL_ARGUMENT_OFFSET );

                  OSAL_tIODescriptor rFileHandle;
                  tU32 u32FunctionId;
                  tU32 u32ArgumentSize;

                  oInputStream >> rFileHandle;
                  oInputStream >> u32FunctionId;
                  oInputStream >> u32ArgumentSize;

                  delete[] m_pau8CopyBuffer;

                  m_pau8CopyBuffer = new tU8[PIPEIOCTRL_ARGUMENT_OFFSET + u32ArgumentSize];
                  if (m_pau8CopyBuffer == NULL ) break;
                  m_u32CopyBufferSize = PIPEIOCTRL_ARGUMENT_OFFSET + u32ArgumentSize;

                  ahl_tclStreamer oTotalInputStream;
                  oTotalInputStream.vSetStreamBuffer( (tPC8)m_pau8CopyBuffer, (PIPEIOCTRL_ARGUMENT_OFFSET + u32ArgumentSize), 0 );

                  oTotalInputStream << rFileHandle;
                  oTotalInputStream << u32FunctionId;
                  oTotalInputStream << u32ArgumentSize;
               }
            }

            if ( u32BufferSize > 0 )
            {
               if ( u32BufferSize >= (m_u32CopyBufferSize - m_u32CopyBufferIndex) )
               {
                  OSAL_pvMemoryCopy( &m_pau8CopyBuffer[m_u32CopyBufferIndex], 
                                      pau8Buffer, 
                                      (m_u32CopyBufferSize - m_u32CopyBufferIndex) ); //lint !e662 !e669
                  u32BufferSize -= ( m_u32CopyBufferSize - m_u32CopyBufferIndex );
                  pau8Buffer    += ( m_u32CopyBufferSize - m_u32CopyBufferIndex );
                  m_u32CopyBufferIndex = m_u32CopyBufferSize;
                  bMessageComplete = TRUE;
               }
               else
               {
                  OSAL_pvMemoryCopy( &m_pau8CopyBuffer[m_u32CopyBufferIndex], pau8Buffer, u32BufferSize );//lint !e669
                  m_u32CopyBufferIndex += u32BufferSize;
                  u32BufferSize = 0;
                  bMessageComplete = FALSE;
               }
            }
         }
         break;

      case  OSAL_S32SHAREDMEMORYUNMAP:
      case  OSAL_S32IOREAD:
         {
            if ( FALSE == bAdjustCopyBuffer( IOREADSHMUNMAP_REQUEST_LENGTH ) )
               break;

            bMessageComplete = bConsumeMessage( IOREADSHMUNMAP_REQUEST_LENGTH, pau8Buffer, u32BufferSize );
         }
         break;

      case  OSAL_S32MESSAGEQUEUECLOSE:
		  {
			 if ( FALSE == bAdjustCopyBuffer( MESSAGEQUEUECLOSE_REQUEST_LENGTH ) )
				break;

			 bMessageComplete = bConsumeMessage( MESSAGEQUEUECLOSE_REQUEST_LENGTH, pau8Buffer, u32BufferSize );
		  }
		  break;
      case  OSAL_S32SHAREDMEMORYCLOSE:
		  {
			 if ( FALSE == bAdjustCopyBuffer( SHAREDMEMORYCLOSE_REQUEST_LENGTH ) )
				break;

			 bMessageComplete = bConsumeMessage( SHAREDMEMORYCLOSE_REQUEST_LENGTH, pau8Buffer, u32BufferSize );
		  }
		  break;
      case  OSAL_S32IOCLOSE:
		  {
			 if ( FALSE == bAdjustCopyBuffer( IOCLOSE_REQUEST_LENGTH ) )
				break;

			 bMessageComplete = bConsumeMessage( IOCLOSE_REQUEST_LENGTH, pau8Buffer, u32BufferSize );
		  }
		  break;
      case  OSAL_S32MESSAGEPOOLCREATE:
         {
            if ( FALSE == bAdjustCopyBuffer( MESSAGEPOOLCREATE_REQUEST_LENGTH ) )
               break;

            bMessageComplete = bConsumeMessage( MESSAGEPOOLCREATE_REQUEST_LENGTH, pau8Buffer, u32BufferSize );
         }
         break;

      case  OSAL_S32MESSAGEPOOLOPEN:
         {
            m_u32CopyBufferIndex = 0;
            bMessageComplete = TRUE;
         }
         break;

      case  OSAL_S32MESSAGEPOOLCLOSE:
         {
            m_u32CopyBufferIndex = 0;
            bMessageComplete = TRUE;
         }
         break;

      case  OSAL_S32MESSAGEPOOLDELETE:
         {
            m_u32CopyBufferIndex = 0;
            bMessageComplete = TRUE;
         }
         break;

      case  OSALPIPE_S32MESSAGEQUEUEPOST:
         {
            if ( FALSE == bAdjustCopyBuffer( MESSAGEQUEUEPOST_MESSAGE_OFFSET ) )
               break;

            if ( m_u32CopyBufferIndex <= MESSAGEQUEUEPOST_MESSAGE_OFFSET )
            {
               if ( bConsumeMessage( MESSAGEQUEUEPOST_MESSAGE_OFFSET, pau8Buffer, u32BufferSize ) )
               {
                  ahl_tclStreamer oInputStream;
                  oInputStream.vSetStreamBuffer( (tPC8)m_pau8CopyBuffer, MESSAGEQUEUEPOST_MESSAGE_OFFSET, MESSAGEQUEUEPOST_MESSAGE_OFFSET );

                  OSAL_tMQueueHandle rMessageQueueHandle;
                  tU32 u32MessageSize;

                  oInputStream >> rMessageQueueHandle;
                  oInputStream >> u32MessageSize;

                  delete[] m_pau8CopyBuffer;
                  const size_t ctuiNewBufferSize = MESSAGEQUEUEPOST_MESSAGE_OFFSET + u32MessageSize + 4;
                  
                  m_pau8CopyBuffer = new tU8[ctuiNewBufferSize];

                  if (m_pau8CopyBuffer == NULL ) break;

                  m_u32CopyBufferSize = ctuiNewBufferSize;

                  ahl_tclStreamer oTotalInputStream;
                  oTotalInputStream.vSetStreamBuffer( (tPC8)m_pau8CopyBuffer, ctuiNewBufferSize, 0 );

                  oTotalInputStream << rMessageQueueHandle;
                  oTotalInputStream << u32MessageSize;
               }
            }
            if ( u32BufferSize > 0 )
            {
               bMessageComplete = bConsumeMessage( m_u32CopyBufferSize, pau8Buffer, u32BufferSize );
            }
         }
         break;

      case  OSALPIPE_PING:
         {
            if ( FALSE == bAdjustCopyBuffer( PING_MESSAGE_OFFSET ) )
               break;

            if ( m_u32CopyBufferIndex <= PING_MESSAGE_OFFSET )
            {
               if ( bConsumeMessage( PING_MESSAGE_OFFSET, pau8Buffer, u32BufferSize ) )
               {
                  ahl_tclStreamer oInputStream;
                  oInputStream.vSetStreamBuffer( (tPC8)m_pau8CopyBuffer, PING_MESSAGE_OFFSET, PING_MESSAGE_OFFSET );

                  tU32 u32MessageCount;
                  tU32 u32SendBack;
                  tU32 u32MessageSize;

                  oInputStream >> u32MessageCount;
                  oInputStream >> u32SendBack;
                  oInputStream >> u32MessageSize;

                  delete[] m_pau8CopyBuffer;

                  m_pau8CopyBuffer = new tU8[PING_MESSAGE_OFFSET + u32MessageSize + 4];
                  if (m_pau8CopyBuffer == NULL ) break;
                  m_u32CopyBufferSize = PING_MESSAGE_OFFSET + (tU16)u32MessageSize + 4;

                  ahl_tclStreamer oTotalInputStream;
                  oTotalInputStream.vSetStreamBuffer( (tPC8)m_pau8CopyBuffer, (PING_MESSAGE_OFFSET + u32MessageSize + 4), 0 );

                  oTotalInputStream << u32MessageCount;
                  oTotalInputStream << u32SendBack;
                  oTotalInputStream << u32MessageSize;
               }
            }
            if ( u32BufferSize > 0 )
            {
               bMessageComplete = bConsumeMessage( m_u32CopyBufferSize, pau8Buffer, u32BufferSize );
            }
         }
         break;

      case  OSALPIPE_S32MESSAGEQUEUEWAIT:
      case  OSAL_S32IOCTRL:
         {
            if ( FALSE == bAdjustCopyBuffer( MESSAGEQUEUEWAIT_REQUEST_LENGTH ) )
               break;

            bMessageComplete = bConsumeMessage( MESSAGEQUEUEWAIT_REQUEST_LENGTH, pau8Buffer, u32BufferSize );
         }
         break;

      case OSAL_SHAREDMEMORYCREATE:
         {
            if ( FALSE == bAdjustCopyBuffer( SHAREDMEMORYCREATE_REQUEST_LENGTH ) )
               break;

            // Shared memory name length is max 256 bytes.
            // Receive until null terminated SHM name is received completely.
            tU16 u16Index;
            tBool bEndOfString = FALSE;
            for ( u16Index = 0; u16Index < m_u32CopyBufferIndex; u16Index++ )
            {
               if ( m_pau8CopyBuffer[u16Index] == 0 )
               {
                  bEndOfString = TRUE;
                  break;
               }
            }
            if ( bEndOfString )
            {
               m_u32CopyBufferSize = u16Index + 1 + 8;
            }
            else
            {
               for ( u16Index = 0; u16Index < u32BufferSize; u16Index ++ )
               {
                  if ( pau8Buffer[u16Index] == 0 )
                  {
                     bEndOfString = TRUE;
                     break;
                  }
               }
               if ( bEndOfString )
               {
                  OSAL_pvMemoryCopy( &m_pau8CopyBuffer[m_u32CopyBufferIndex], pau8Buffer, (u16Index + 1) );
                  m_u32CopyBufferIndex += ( u16Index + 1 );
                  m_u32CopyBufferSize = m_u32CopyBufferIndex + 8;
                  u32BufferSize -= ( u16Index + 1 );
                  pau8Buffer    += ( u16Index + 1 );
               }
               else
               {
                  OSAL_pvMemoryCopy( &m_pau8CopyBuffer[m_u32CopyBufferIndex], pau8Buffer, u32BufferSize );
                  m_u32CopyBufferIndex += u32BufferSize;
                  u32BufferSize = 0;
               }
            }
            // Null terminated shm name received completely, receive rest of the RPC arguments
            if ( u32BufferSize > 0 )
            {
               bMessageComplete = bConsumeMessage( m_u32CopyBufferSize, pau8Buffer, u32BufferSize );
            }
         }
         break;

      case OSAL_PVSHAREDMEMORYMAP:
         {
            if ( FALSE == bAdjustCopyBuffer( SHAREDMEMORYMAP_REQUEST_LENGTH ) )
               break;

            bMessageComplete = bConsumeMessage( SHAREDMEMORYMAP_REQUEST_LENGTH, pau8Buffer, u32BufferSize );
         }
         break;

      case  WOS_S32IO_ENGINEERING_FCT:
         {
            if ( FALSE == bAdjustCopyBuffer( OSALRPC_REQUEST_MAXNAME_LENGTH + IOENGINEERINGFCT_ARGUMENT_OFFSET ) )
               break;

            // Step 1. Copy until ArgumentSize parameter
            tU16 u16Index;
            tBool bEndOfString = FALSE;
            for ( u16Index = 0; u16Index < m_u32CopyBufferIndex; u16Index++ )
            {
               if ( m_pau8CopyBuffer[u16Index] == 0 )
               {
                  bEndOfString = TRUE;
                  break;
               }
            }
            if ( bEndOfString )
            {
               if ( m_enInternalState != EN_WAITING_FOR_ARGUMENTS )
                  m_u32CopyBufferSize = u16Index + 1 + IOENGINEERINGFCT_ARGUMENT_OFFSET;
            }
            else
            {
               for ( u16Index = 0; u16Index < u32BufferSize; u16Index ++ )
               {
                  if ( pau8Buffer[u16Index] == 0 )
                  {
                     bEndOfString = TRUE;
                     break;
                  }
               }
               if ( bEndOfString )  // Null-terminated device-name completely received
               {
                  OSAL_pvMemoryCopy( &m_pau8CopyBuffer[m_u32CopyBufferIndex], pau8Buffer, (u16Index + 1) );
                  m_u32CopyBufferIndex += ( u16Index + 1 );
                  m_u32CopyBufferSize = m_u32CopyBufferIndex + IOENGINEERINGFCT_ARGUMENT_OFFSET;
                  u32BufferSize -= ( u16Index + 1 );
                  pau8Buffer    += ( u16Index + 1 );
               }
               else  // Null-terminated device-name partly received
               {
                  OSAL_pvMemoryCopy( &m_pau8CopyBuffer[m_u32CopyBufferIndex], pau8Buffer, u32BufferSize );
                  m_u32CopyBufferIndex += u32BufferSize;
                  u32BufferSize = 0;
               }
            }

            // More bytes in the stream to copy beyond device name
            if ( u32BufferSize > 0 )
            {
               if ( bConsumeMessage( m_u32CopyBufferSize, pau8Buffer, u32BufferSize ) )
               {
                  if ( m_enInternalState != EN_WAITING_FOR_ARGUMENTS )
                  {
                     // Message bytes up to device-name, ioctl-fid and argument-size received.
                     // Keep aside for combining later into the total message stream.
                     ahl_tclStreamer oInputStream;
                     oInputStream.vSetStreamBuffer( (tPC8)m_pau8CopyBuffer,
                        OSALRPC_REQUEST_MAXNAME_LENGTH + IOENGINEERINGFCT_ARGUMENT_OFFSET,
                        OSALRPC_REQUEST_MAXNAME_LENGTH + IOENGINEERINGFCT_ARGUMENT_OFFSET );

                     tU32 u32FunctionId;
                     tU32 u32ArgumentSize;
                     tU32 u32NameLength = (tU32)OSAL_u32StringLength( (tPC8)m_pau8CopyBuffer ) + 1;
                     tPC8 pcDeviceName = new tC8[u32NameLength];
                     if (pcDeviceName == NULL ) break;
                     oInputStream.vReadString( pcDeviceName, (tU16)u32NameLength );
                     oInputStream >> u32FunctionId;
                     oInputStream >> u32ArgumentSize;

                     delete[] m_pau8CopyBuffer;

                     // Re-allocate copy buffer with the new found name length and argument size
                     m_pau8CopyBuffer = new tU8[ u32NameLength + IOENGINEERINGFCT_ARGUMENT_OFFSET + u32ArgumentSize];
                     if (m_pau8CopyBuffer == NULL ) break;
                     m_u32CopyBufferSize = u32NameLength + IOENGINEERINGFCT_ARGUMENT_OFFSET + u32ArgumentSize;

                     ahl_tclStreamer oTotalInputStream;
                     oTotalInputStream.vSetStreamBuffer( (tPC8)m_pau8CopyBuffer, m_u32CopyBufferSize, 0 );

                     ahl_tclStreamer oDeviceName;
                     oDeviceName.vSetStreamBuffer( (tPC8)pcDeviceName, u32NameLength, u32NameLength );

                     oTotalInputStream.rfWriteStream(oDeviceName);
                     oTotalInputStream << u32FunctionId;
                     oTotalInputStream << u32ArgumentSize;

                     m_u32CopyBufferIndex = u32NameLength + IOENGINEERINGFCT_ARGUMENT_OFFSET;

                     // If there is still more bytes in the message (must be arguments) copy them too
                     bMessageComplete = bConsumeMessage( m_u32CopyBufferSize, pau8Buffer, u32BufferSize );
                     if ( FALSE == bMessageComplete )
                     {
                        m_enInternalState = EN_WAITING_FOR_ARGUMENTS;
                     }

                     delete[] pcDeviceName;
                  }
                  else // EN_WAITING_FOR_ARGUMENTS 
                  {
                     // If there is still more bytes in the message (must be arguments) copy them too
                     bMessageComplete = bConsumeMessage( m_u32CopyBufferSize, pau8Buffer, u32BufferSize );
                     if ( FALSE == bMessageComplete )
                     {
                        m_enInternalState = EN_WAITING_FOR_ARGUMENTS;
                     }
                  }
               }
            }
         }
         break;

      case  MINISPM_SHUTDOWN:
         {
            m_u32CopyBufferIndex = 0;
            bMessageComplete = TRUE;
         }
         break;

      default:
         break;
      }
   }

   if ( bMessageComplete )
   {
#ifdef VARIANT_S_FTR_ENABLE_CPPUNIT_TEST
      // Check correctness of RPC message
      CPPUNIT_ASSERT( bValidateProtocol( m_pau8CopyBuffer, m_u32CopyBufferIndex ) );
#endif

      bExecuteAndSendResult( m_u32Fid, m_pau8CopyBuffer, m_u32CopyBufferIndex );

      if ( m_pau8CopyBuffer )
      {
         delete[] m_pau8CopyBuffer;
         m_pau8CopyBuffer = NULL;
      }
      m_u32CopyBufferSize = 0;
      m_u32CopyBufferIndex = 0;
      m_u32FidPointer = 0;
   }

   return bMessageComplete;
}

/******************************************************************************
 FUNCTION:   | tclOSAL_RPCMessageParser::bExecuteAndSendResult()
-------------------------------------------------------------------------------
 DESCRIPTION:| Complete the OSAL Remote Procedure Call.and SendResult 
             | back to the caller.
-------------------------------------------------------------------------------
 PARAMETERS: | tU32 u32Fid          : [I] RPC Command code
             | tPU8 pau8Message     : [->I] Message bytes
             | tU32 u32BufferLength : [I] Length of message
-------------------------------------------------------------------------------
 RETURN TYPE:| tBool 
-------------------------------------------------------------------------------
 HISTORY:    | Date      | Author            | Modification
-------------------------------------------------------------------------------
             | 01.02.09  | Susmith, RBEI/ECF | Initial version
             | 01.05.09  | Soj, RBEI/ECF     | Datatype modifications, 
             |           |                   | Removed unneeded typecasts 
******************************************************************************/
tBool tclOSAL_RPCMessageParser::bExecuteAndSendResult( tU32 u32Fid, tPCU8 pau8Message, tU32 u32BufferLength )
{
   tBool bMessageSend = FALSE;

   if ( m_poMessageSender != OSAL_NULL )
   {
      switch ( u32Fid )
      {
      case  OSALPIPE_PING:
         {
            ahl_tclStreamer oSendMessageStream;
            oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

            tS32 s32MessageCount;
            tS32 s32ThreadNo;
            tS32 s32SendMsgBack;

            tU32 u32MessageSize;

            oSendMessageStream >> s32MessageCount;
            oSendMessageStream >> s32SendMsgBack;
            oSendMessageStream >> u32MessageSize;

            ahl_tclStreamer pcMessage( u32MessageSize );

            oSendMessageStream.rfReadStream(pcMessage);

            oSendMessageStream >> s32ThreadNo;

            tU32 u32AnswerSize =  PIPEMESSAGE_PING_ANSWER_LENGTH_BASE;
            if ( s32SendMsgBack == 1 )
            {
               u32AnswerSize += u32MessageSize;
            }

            ahl_tclStreamer oOutputStream( u32AnswerSize );

            oOutputStream << s32MessageCount;

            oOutputStream << s32ThreadNo;

            oOutputStream << s32SendMsgBack;

            if ( s32SendMsgBack == 1 )
            {
               oOutputStream << u32MessageSize;
               oOutputStream.rfWriteStream(pcMessage);
            }
            else
            {
               tU32 tmp = 0;
               oOutputStream << tmp;
            }

            oOutputStream << m_localMsgCounter;
            oOutputStream << m_localThreadNo;

            ++m_localMsgCounter;

            if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), u32AnswerSize ) != OSAL_ERROR )
                bMessageSend = TRUE;
         }
         break;

      case  OSALPIPE_S32MESSAGEQUEUEPOST:
         {
            ahl_tclStreamer oSendMessageStream;
            oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

            tU32 u32MessageSize;
            tU32 u32Prior;
            tS32 s32ReturnValue;
            tU32 u32LastError;

            OSAL_trMessage rMessage;
            OSAL_tMQueueHandle rMessageQueueHandle;

            oSendMessageStream >> rMessageQueueHandle;
            oSendMessageStream >> u32MessageSize;

            ahl_tclStreamer pcMessage( u32MessageSize );

            oSendMessageStream.rfReadStream(pcMessage);

            oSendMessageStream >> u32Prior;

            s32ReturnValue = OSAL_s32MessageCreate( &rMessage, u32MessageSize , OSAL_EN_MEMORY_SHARED );
            if ( s32ReturnValue == OSAL_OK )
            {
               tPU8 pU8Pointer = OSAL_pu8MessageContentGet( rMessage, OSAL_EN_READWRITE );
               OSAL_pvMemoryCopy( pU8Pointer, pcMessage.pGetStream(), u32MessageSize );
               s32ReturnValue = OSAL_s32MessageQueuePost( rMessageQueueHandle, (unsigned char*)&rMessage, sizeof(OSAL_trMessage), u32Prior );
            }
            u32LastError = OSAL_u32ErrorCode();

            ahl_tclStreamer oOutputStream( PIPEMESSAGEQUEUEPOST_ANSWER_LENGTH );
            oOutputStream << u32LastError;
            oOutputStream << s32ReturnValue;

            if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), PIPEMESSAGEQUEUEPOST_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;
         }
         break;

      case  OSAL_S32MESSAGEQUEUECREATE:
         {
            if( pau8Message != NULL )
            {
               tU32 u32MaxMessages;
               tU32 u32MaxLength;
               tU32 u32Access;
               tS32 s32ReturnValue;
               tU32 u32LastError;

               ahl_tclStreamer oSendMessageStream;    // Input Stream - RPC Request
               oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

               OSAL_tMQueueHandle oMessageQueueHandle;

               tU32 u32NameLength = (tU32)OSAL_u32StringLength( (tPC8)pau8Message ) + 1;

               tPC8 pcQueueName = new tC8[u32NameLength];
               oSendMessageStream.vReadString( pcQueueName, (tU16)u32NameLength );

               oSendMessageStream >> u32MaxMessages;
               oSendMessageStream >> u32MaxLength;
               oSendMessageStream >> u32Access;

               ahl_tclStreamer oOutputStream( MESSAGEQUEUECREATE_ANSWER_LENGTH ); // Output Stream - RPC Reply
               s32ReturnValue = OSAL_s32MessageQueueCreate( pcQueueName,
                                                            u32MaxMessages,
                                                            u32MaxLength,
                                                            (OSAL_tenAccess)u32Access,
                                                            &oMessageQueueHandle );
               u32LastError = OSAL_u32ErrorCode();

               oOutputStream << u32LastError;
               oOutputStream << s32ReturnValue;
               oOutputStream << oMessageQueueHandle;

               if ( m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), MESSAGEQUEUECREATE_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;

               delete[] pcQueueName;
            }
         }
         break;

      case  OSAL_S32MESSAGEQUEUEOPEN:
         {
            if( pau8Message!=NULL )
            {
               tU32 u32Access;
               tS32 s32ReturnValue;
               tU32 u32LastError;

               ahl_tclStreamer oSendMessageStream;
               oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

               OSAL_tMQueueHandle oMessageQueueHandle;

               tU32 u32NameLength = (tU32)OSAL_u32StringLength( (tPC8)pau8Message ) + 1;

               tPC8 pcQueueName = new tC8[u32NameLength];
               oSendMessageStream.vReadString( pcQueueName, (tU16)u32NameLength );

               oSendMessageStream >> u32Access;

               ahl_tclStreamer oOutputStream( MESSAGEQUEUEOPEN_ANSWER_LENGTH );
               s32ReturnValue = OSAL_s32MessageQueueOpen( pcQueueName,
                                                          (OSAL_tenAccess)u32Access,
                                                          &oMessageQueueHandle );
               u32LastError = OSAL_u32ErrorCode();

               oOutputStream << u32LastError;
               oOutputStream << s32ReturnValue;
               oOutputStream << oMessageQueueHandle;

               if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), MESSAGEQUEUEOPEN_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;

               delete[] pcQueueName;
            }
         }
         break;

      case  OSAL_S32MESSAGEQUEUECLOSE:
         {
            tS32 s32ReturnValue;
            tU32 u32LastError;

            ahl_tclStreamer oSendMessageStream;
            oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

            OSAL_tMQueueHandle oMessageQueueHandle;
            oSendMessageStream >> oMessageQueueHandle;

            ahl_tclStreamer oOutputStream( MESSAGEQUEUECLOSE_ANSWER_LENGTH );

            s32ReturnValue = OSAL_s32MessageQueueClose( oMessageQueueHandle );
            u32LastError = OSAL_u32ErrorCode();

            oOutputStream << u32LastError;
            oOutputStream << s32ReturnValue;

            if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), MESSAGEQUEUECLOSE_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;
         }
         break;

      case  OSAL_S32MESSAGEQUEUEDELETE:
         {
            if(pau8Message!=NULL)
            {
               tS32 s32ReturnValue;
               tU32 u32LastError;

               ahl_tclStreamer oSendMessageStream;
               oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

               tU32 u32NameLength = (tU32)OSAL_u32StringLength( (tPC8)pau8Message ) + 1;

               tPC8 pcQueueName = new tC8[u32NameLength];
               oSendMessageStream.vReadString( pcQueueName, (tU16)u32NameLength );

               ahl_tclStreamer oOutputStream( MESSAGEQUEUEDELETE_ANSWER_LENGTH );
               s32ReturnValue = OSAL_s32MessageQueueDelete( pcQueueName );
               u32LastError = OSAL_u32ErrorCode();

               oOutputStream << u32LastError;
               oOutputStream << s32ReturnValue;

               if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), MESSAGEQUEUEDELETE_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;

               delete[] pcQueueName;
            }
         }
         break;

      case  OSAL_S32MESSAGEQUEUEPOST:
         {
            tU32 u32MessageSize;
            tU32 u32Prior;
            tS32 s32ReturnValue;
            tU32 u32LastError;

            ahl_tclStreamer oSendMessageStream;
            oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

            OSAL_tMQueueHandle rMessageQueueHandle;

            oSendMessageStream >> rMessageQueueHandle;
            oSendMessageStream >> u32MessageSize;

            ahl_tclStreamer pcMessage( u32MessageSize );

            oSendMessageStream.rfReadStream(pcMessage);

            oSendMessageStream >> u32Prior;

            s32ReturnValue = OSAL_s32MessageQueuePost( rMessageQueueHandle,
                                                       (tPU8)pcMessage.pGetStream(),
                                                       u32MessageSize,u32Prior );
            u32LastError = OSAL_u32ErrorCode();

            ahl_tclStreamer oOutputStream( MESSAGEQUEUEPOST_ANSWER_LENGTH );
            oOutputStream << u32LastError;
            oOutputStream << s32ReturnValue;

            if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), MESSAGEQUEUEPOST_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;
         }
         break;

      case  OSAL_S32MESSAGEQUEUEWAIT:
         {
            tU32 u32BufferSize;
            tU32 u32Timeout;
            tS32 s32ReturnValue;
            tU32 u32LastError;

            tU32 u32Prio = 0;
            ahl_tclStreamer oSendMessageStream;
            oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

            OSAL_tMQueueHandle oMessageQueueHandle;

            oSendMessageStream >> oMessageQueueHandle;
            oSendMessageStream >> u32BufferSize;
            oSendMessageStream >> u32Timeout;

            tPU8 pcMessage = new tU8[u32BufferSize]; 

            s32ReturnValue = OSAL_s32MessageQueueWait( oMessageQueueHandle,
                                                       pcMessage,
                                                       u32BufferSize,
                                                       &u32Prio,
                                                       (OSAL_tMSecond)u32Timeout );
            u32LastError = OSAL_u32ErrorCode();

            ahl_tclStreamer oMessageStream;

            oMessageStream.vSetStreamBuffer( (tPC8)pcMessage, s32ReturnValue, s32ReturnValue );

            ahl_tclStreamer oOutputStream( MESSAGEQUEUEWAIT_ANSWER_LENGTH + s32ReturnValue );
            oOutputStream << u32LastError;
            oOutputStream << s32ReturnValue;
            oOutputStream.rfWriteStream(oMessageStream);
            oOutputStream << u32Prio;

            if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), (MESSAGEQUEUEWAIT_ANSWER_LENGTH + s32ReturnValue) ) != OSAL_ERROR )
                bMessageSend = TRUE;

            delete[] pcMessage;
         }
         break;

      case  OSAL_S32MESSAGEPOOLCREATE:
         {
            tU32 u32PoolSize;
            tS32 s32ReturnValue;
            tU32 u32LastError;

            ahl_tclStreamer oSendMessageStream;
            oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );
            oSendMessageStream >> u32PoolSize;

            s32ReturnValue = OSAL_s32MessagePoolCreate( u32PoolSize );
            u32LastError = OSAL_u32ErrorCode();

            ahl_tclStreamer oOutputStream( MESSAGEPOOLCREATE_ANSWER_LENGTH );
            oOutputStream << u32LastError;
            oOutputStream << s32ReturnValue;

            if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), MESSAGEPOOLCREATE_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;
         }
         break;

      case  OSAL_S32MESSAGEPOOLOPEN:
         {
            tS32 s32ReturnValue;
            tU32 u32LastError;

            s32ReturnValue = OSAL_s32MessagePoolOpen();
            u32LastError = 0;

            ahl_tclStreamer oOutputStream( MESSAGEPOOLOPEN_ANSWER_LENGTH );
            oOutputStream << u32LastError;
            oOutputStream << s32ReturnValue;

            if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), MESSAGEPOOLOPEN_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;
         }
         break;

      case  OSAL_S32MESSAGEPOOLCLOSE:
         {
            tS32 s32ReturnValue;
            tU32 u32LastError;

            s32ReturnValue = OSAL_s32MessagePoolClose();
            u32LastError = 0;

            ahl_tclStreamer oOutputStream( MESSAGEPOOLCLOSE_ANSWER_LENGTH );
            oOutputStream << u32LastError;
            oOutputStream << s32ReturnValue;

            if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), MESSAGEPOOLCLOSE_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;
         }
         break;

      case  OSAL_S32MESSAGEPOOLDELETE:
         {
            tS32 s32ReturnValue;
            tU32 u32LastError;

            s32ReturnValue = OSAL_s32MessagePoolDelete();
            u32LastError = OSAL_u32ErrorCode();

            ahl_tclStreamer oOutputStream( MESSAGEPOOLDELETE_ANSWER_LENGTH );
            oOutputStream << u32LastError;
            oOutputStream << s32ReturnValue;

            if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), MESSAGEPOOLDELETE_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;
         }
         break;

      case  OSALPIPE_S32MESSAGEQUEUEWAIT:
         {
            tU32 u32BufferSize;
            tU32 u32Timeout;
            tS32 s32ReturnValue;
            tU32 u32LastError;

            tU32 u32Prio=0;

            ahl_tclStreamer oSendMessageStream;
            oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

            OSAL_tMQueueHandle oMessageQueueHandle;

            oSendMessageStream >> oMessageQueueHandle;
            oSendMessageStream >> u32BufferSize;
            oSendMessageStream >> u32Timeout;

            OSAL_trMessage rMessage;

            s32ReturnValue = OSAL_s32MessageQueueWait( oMessageQueueHandle,
                                                       (tPU8)&rMessage,
                                                       sizeof(OSAL_trMessage),
                                                       &u32Prio,
                                                       (OSAL_tMSecond)u32Timeout );
            u32LastError = OSAL_u32ErrorCode();

            if ( s32ReturnValue > 0 )
            {
               tU32 u32Size = OSAL_u32GetMessageSize( rMessage );

               ahl_tclStreamer oOutputStream( PIPEMESSAGEQUEUEWAIT_ANSWER_LENGTH + u32Size );
               ahl_tclStreamer oMessageStream;

               tPU8 pu8MessageContent = OSAL_pu8MessageContentGet( rMessage, OSAL_EN_READWRITE );
               //fix for MMS 249653 ( removed tU16 typecast for u32Size argument )
               oMessageStream.vSetStreamBuffer( (tPC8)pu8MessageContent, u32Size, u32Size );
               oOutputStream << u32LastError;
               oOutputStream << u32Size;
               oOutputStream.rfWriteStream(oMessageStream);
               oOutputStream << u32Prio;

               OSAL_s32MessageDelete( rMessage );
               if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), (PIPEMESSAGEQUEUEWAIT_ANSWER_LENGTH + u32Size) ) != OSAL_ERROR )
                bMessageSend = TRUE;
            }
            else
            {
               ahl_tclStreamer oOutputStream( PIPEMESSAGEQUEUEWAIT_ANSWER_LENGTH );

               tU32 u32Size = 0;
               oOutputStream << u32LastError;
               oOutputStream << u32Size;
               oOutputStream << u32Prio;
               if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), PIPEMESSAGEQUEUEWAIT_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;
            }
         }
         break;

      case  OSAL_S32IOCREATE:
         {
            if( pau8Message!=NULL )
            {
               tU32 u32Access;
               OSAL_tIODescriptor rHandle;
               tU32 u32LastError;

               ahl_tclStreamer oSendMessageStream;
               oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

               oSendMessageStream >> u32Access;

               tU32 u32NameLength = (tU32)OSAL_u32StringLength((tPC8)(pau8Message + 4)) + 1;

               tPC8 pcDevName = new tC8[u32NameLength];
               oSendMessageStream.vReadString( pcDevName, (tU16)u32NameLength );

               ahl_tclStreamer oOutputStream( IOCREATE_ANSWER_LENGTH );

               rHandle = OSAL_IOCreate( pcDevName, (OSAL_tenAccess)u32Access );
               u32LastError = OSAL_u32ErrorCode();

               oOutputStream << u32LastError;
               oOutputStream << rHandle;

               if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), IOCREATE_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;

               delete[] pcDevName;
            }
         }
         break;

      case  OSAL_S32IOREMOVE:
         {
            if(pau8Message!=NULL)  
            {
               tU32 u32LastError;
               tS32 s32ReturnValue;

               ahl_tclStreamer oSendMessageStream;
               oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

               tU32 u32NameLength = (tU32)OSAL_u32StringLength((tPC8)(pau8Message)) + 1;

               tPC8 pcDevName = new tC8[u32NameLength];
               oSendMessageStream.vReadString( pcDevName, (tU16)u32NameLength );

               ahl_tclStreamer oOutputStream( IOREMOVE_ANSWER_LENGTH );

               s32ReturnValue = OSAL_s32IORemove( pcDevName );
               u32LastError = OSAL_u32ErrorCode();

               oOutputStream << u32LastError;
               oOutputStream << s32ReturnValue;

               if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), IOREMOVE_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;

               delete[] pcDevName;
            }
         }
         break;

      case  OSAL_S32IOOPEN:
         {
            if( pau8Message!=NULL )
            {
               tU32 u32Access;
               OSAL_tIODescriptor rHandle;
               tU32 u32LastError;

               ahl_tclStreamer oSendMessageStream;
               oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

               oSendMessageStream >> u32Access;

               tU32 u32NameLength = (tU32)OSAL_u32StringLength((tPC8)(pau8Message+4)) + 1;

               tPC8 pcDevName = new tC8[u32NameLength];
               oSendMessageStream.vReadString( pcDevName, (tU16)u32NameLength );

               ahl_tclStreamer oOutputStream( IOOPEN_ANSWER_LENGTH );

               rHandle = OSAL_IOOpen( pcDevName, (OSAL_tenAccess)u32Access );
               u32LastError = OSAL_u32ErrorCode();

               oOutputStream << u32LastError;
               oOutputStream << rHandle;

               if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), IOOPEN_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;

               delete[] pcDevName;
            }
         }
         break;

      case  OSAL_S32IOCLOSE:
         {
            OSAL_tIODescriptor rHandle;
            tU32 u32LastError;
            tS32 s32ReturnValue;

            ahl_tclStreamer oSendMessageStream;
            oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

            oSendMessageStream >> rHandle;

            ahl_tclStreamer oOutputStream( IOCLOSE_ANSWER_LENGTH );

            s32ReturnValue = OSAL_s32IOClose( rHandle );
            u32LastError = OSAL_u32ErrorCode();

            oOutputStream << u32LastError;
            oOutputStream << s32ReturnValue;

            if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), IOCLOSE_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;
         }
         break;

      case  OSAL_S32IOCTRL:
         {
            OSAL_tIODescriptor rHandle;
            tU32 u32FunctionId;
            intptr_t iArgument;
            tU32 u32LastError;
            tS32 s32ReturnValue;

            ahl_tclStreamer oSendMessageStream;
            oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

            oSendMessageStream >> rHandle;
            oSendMessageStream >> u32FunctionId;
            oSendMessageStream >> iArgument;

            ahl_tclStreamer oOutputStream( IOCTRL_ANSWER_LENGTH );

            s32ReturnValue = OSAL_s32IOControl( rHandle, u32FunctionId, iArgument );
            u32LastError = OSAL_u32ErrorCode();

            oOutputStream << u32LastError;
            oOutputStream << s32ReturnValue;

            if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), IOCTRL_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;
         }
         break;

      case  OSAL_S32IOWRITE:
         {
            OSAL_tIODescriptor rHandle;
            tU32 u32DataSize;
            tU32 u32LastError;
            tS32 s32ReturnValue;

            ahl_tclStreamer oSendMessageStream;
            oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

            oSendMessageStream >> rHandle;
            oSendMessageStream >> u32DataSize;

            ahl_tclStreamer pcData( (tU16)u32DataSize );

            oSendMessageStream.rfReadStream(pcData);

            ahl_tclStreamer oOutputStream( IOWRITE_ANSWER_LENGTH );

            s32ReturnValue = OSAL_s32IOWrite( rHandle, (tPCS8)pcData.pGetStream(), u32DataSize );
            u32LastError = OSAL_u32ErrorCode();

            oOutputStream << u32LastError;
            oOutputStream << s32ReturnValue;

            if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), IOWRITE_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;
         }
         break;

      case  OSAL_S32IOREAD:
         {
            OSAL_tIODescriptor rHandle;
            tU32 u32DataSize;
            tU32 u32LastError;
            tS32 s32ReturnValue;

            ahl_tclStreamer oSendMessageStream;
            oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

            oSendMessageStream >> rHandle;
            oSendMessageStream >> u32DataSize;

            tPS8 pas8Data = new tS8[u32DataSize];

            s32ReturnValue = OSAL_s32IORead( rHandle, pas8Data, u32DataSize );
            u32LastError = OSAL_u32ErrorCode();

            ahl_tclStreamer oMessageStream;

            oMessageStream.vSetStreamBuffer( (tPC8)pas8Data, s32ReturnValue, s32ReturnValue );

            ahl_tclStreamer oOutputStream( (IOREAD_ANSWER_LENGTH + s32ReturnValue) );

            oOutputStream << u32LastError;
            oOutputStream << s32ReturnValue;
            oOutputStream.rfWriteStream(oMessageStream);

            if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), (IOREAD_ANSWER_LENGTH + s32ReturnValue) ) != OSAL_ERROR )
                bMessageSend = TRUE;

            delete[] pas8Data;
         }
         break;

      case  OSALPIPE_S32IOCTRL:
         {
            OSAL_tIODescriptor rHandle;
            tU32 u32FunctionId;
            tU32 u32ArgumentSize;

            tU32 u32LastError;
            tS32 s32ReturnValue;

            ahl_tclStreamer oSendMessageStream;
            oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

            oSendMessageStream >> rHandle;
            oSendMessageStream >> u32FunctionId;
            oSendMessageStream >> u32ArgumentSize;

            intptr_t iArgumentPointer = 0;
            ahl_tclStreamer pcArgument( u32ArgumentSize );
            if ( u32ArgumentSize > 0 )
            {
               // Fix for MMS 248844 : Only if there is an argument...
               pcArgument.rfWriteStream(oSendMessageStream);
               iArgumentPointer = (intptr_t)pcArgument.pGetStream();
            }

            s32ReturnValue = OSAL_s32IOControl( rHandle, u32FunctionId, iArgumentPointer );

            u32LastError = OSAL_u32ErrorCode();

            ahl_tclStreamer oOutputStream( (PIPEIOCTRL_ANSWER_LENGTH + u32ArgumentSize) );

            oOutputStream<<u32LastError;
            oOutputStream<<s32ReturnValue;
            oOutputStream<<u32ArgumentSize;

            if ( u32ArgumentSize > 0 )
            {
               // Fix for MMS 248844 : Only if there is an argument...
               oOutputStream.rfWriteStream(pcArgument);
            }

            if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), (PIPEIOCTRL_ANSWER_LENGTH + u32ArgumentSize) ) != OSAL_ERROR )
                bMessageSend = TRUE;
         }
         break;

      case OSAL_SHAREDMEMORYCREATE:
         {
            if( pau8Message != NULL )
            {
               tU32 u32Access;
               tU32 u32Size;
               tS32 s32ReturnValue = SHAREDMEMORYCREATE_ANSWER_MAGIC;
               tU32 u32LastError;

               ahl_tclStreamer oSendMessageStream;    // Input Stream - RPC Request
               oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

               OSAL_tShMemHandle oShmemHandle;

               tU32 u32NameLength = (tU32)OSAL_u32StringLength( (tPC8)pau8Message ) + 1;

               tPC8 pcShmName = new tC8[u32NameLength];
               oSendMessageStream.vReadString( pcShmName, (tU16)u32NameLength );

               oSendMessageStream >> u32Access;
               oSendMessageStream >> u32Size;

               ahl_tclStreamer oOutputStream( SHAREDMEMORYCREATE_ANSWER_LENGTH ); // Output Stream - RPC Reply
               oShmemHandle = OSAL_SharedMemoryCreate( pcShmName, 
                                                      (OSAL_tenAccess)u32Access,
                                                       u32Size );
               u32LastError = OSAL_u32ErrorCode();

               oOutputStream << s32ReturnValue;
               oOutputStream << oShmemHandle;
               oOutputStream << u32LastError;

               if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), SHAREDMEMORYCREATE_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;

               delete[] pcShmName;
            }
         }
         break;

      case OSAL_SHAREDMEMORYOPEN:
         {
            if( pau8Message!=NULL )
            {
               tU32 u32Access;
               tS32 s32ReturnValue = SHAREDMEMORYOPEN_ANSWER_MAGIC;
               tU32 u32LastError;

               ahl_tclStreamer oSendMessageStream; // Input Stream - RPC Request
               oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

               OSAL_tShMemHandle oShmHandle;

               tU32 u32NameLength = (tU32)OSAL_u32StringLength( (tPC8)pau8Message ) + 1;

               tPC8 pcShmName = new tC8[u32NameLength];
               oSendMessageStream.vReadString( pcShmName, (tU16)u32NameLength );

               oSendMessageStream >> u32Access;

               ahl_tclStreamer oOutputStream( SHAREDMEMORYOPEN_ANSWER_LENGTH );  // Output Stream - RPC Reply
               oShmHandle = OSAL_SharedMemoryOpen( pcShmName, (OSAL_tenAccess)u32Access );
               u32LastError = OSAL_u32ErrorCode();

               oOutputStream << s32ReturnValue;
               oOutputStream << oShmHandle;
               oOutputStream << u32LastError;

               if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), SHAREDMEMORYOPEN_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;
               
               delete[] pcShmName;
            }
         }
         break;

      case OSAL_S32SHAREDMEMORYCLOSE:
         {
            tU32 u32Magic  = SHAREDMEMORYCLOSE_ANSWER_MAGIC;
            tS32 s32ReturnValue;
            tU32 u32LastError;

            ahl_tclStreamer oSendMessageStream; // Input Stream - RPC Request
            oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

            OSAL_tShMemHandle oShmHandle;
            oSendMessageStream >> oShmHandle;

            ahl_tclStreamer oOutputStream( SHAREDMEMORYCLOSE_ANSWER_LENGTH ); // Output Stream - RPC Reply

            s32ReturnValue = OSAL_s32SharedMemoryClose( oShmHandle );
            u32LastError = OSAL_u32ErrorCode();

            oOutputStream << u32Magic;
            oOutputStream << s32ReturnValue;
            oOutputStream << u32LastError;

            if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), SHAREDMEMORYCLOSE_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;
         }
         break;

      case OSAL_S32SHAREDMEMORYDELETE:
         {
            if( pau8Message != NULL )
            {
               tU32 u32Magic  = SHAREDMEMORYDELETE_ANSWER_MAGIC;
               tS32 s32ReturnValue;
               tU32 u32LastError;

               ahl_tclStreamer oSendMessageStream; // Input Stream - RPC Request
               oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

               tU32 u32NameLength = (tU32)OSAL_u32StringLength( (tPC8)pau8Message ) + 1;

               tPC8 pcShmName = new tC8[u32NameLength];
               oSendMessageStream.vReadString( pcShmName, (tU16)u32NameLength );

               ahl_tclStreamer oOutputStream( SHAREDMEMORYDELETE_ANSWER_LENGTH ); // Output Stream - RPC Reply
               s32ReturnValue = OSAL_s32SharedMemoryDelete( pcShmName );
               u32LastError = OSAL_u32ErrorCode();

               oOutputStream << u32Magic;
               oOutputStream << s32ReturnValue;
               oOutputStream << u32LastError;

               if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), SHAREDMEMORYDELETE_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;

               delete[] pcShmName;
            }
         }
         break;

      case OSAL_PVSHAREDMEMORYMAP:
         {
            tU32 u32Magic = SHAREDMEMORYMAP_ANSWER_MAGIC;
            OSAL_tShMemHandle oShmHandle;
            tU32 enAccess;
            tU32 u32Length;
            tU32 u32Offset;
            tU32 u32LastError;
#ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT            
            uintptr_t pvMemory;
#else
            tU32 pvMemory;
#endif

            ahl_tclStreamer oSendMessageStream; // Input Stream - RPC Request
            oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

            oSendMessageStream >> oShmHandle;
            oSendMessageStream >> enAccess;
            oSendMessageStream >> u32Length;
            oSendMessageStream >> u32Offset;

            tPS8 pas8Data = new tS8[u32Length];
            if ( pas8Data != NULL )
            OSAL_pvMemorySet( pas8Data, 0, u32Length );

            ahl_tclStreamer oOutputStream( SHAREDMEMORYMAP_ANSWER_LENGTH + u32Length ); // Output Stream - RPC Reply

            pvMemory = (uintptr_t)OSAL_pvSharedMemoryMap( oShmHandle, (OSAL_tenAccess)enAccess, u32Length, u32Offset );
            u32LastError = OSAL_u32ErrorCode();

            tU32 u32MappedBytes = 0;
            if (( pvMemory != OSAL_NULL ) && ( pas8Data != NULL ) )
            {
               u32MappedBytes = u32Length;
               OSAL_pvMemoryCopy( pas8Data, (tPVoid)pvMemory, u32MappedBytes );
            }

            ahl_tclStreamer oMemoryStream;  // Intermediate stream to hold the mapped bytes
            oMemoryStream.vSetStreamBuffer( (tPC8)pas8Data, u32MappedBytes, u32MappedBytes );

            oOutputStream << u32Magic;
            oOutputStream << pvMemory;
            oOutputStream << u32MappedBytes;
            oOutputStream.rfWriteStream(oMemoryStream);
            oOutputStream << u32LastError;

            if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(),
                                               SHAREDMEMORYMAP_ANSWER_LENGTH + u32MappedBytes ) != OSAL_ERROR )
                bMessageSend = TRUE;

            delete[] pas8Data;
         }
         break;

      case OSAL_S32SHAREDMEMORYUNMAP:
         {
            tU32 u32Magic = SHAREDMEMORYUNMAP_ANSWER_MAGIC;
#ifdef VARIANT_S_FTR_ENABLE_64_BIT_SUPPORT            
            uintptr_t pvSharedMemory;
#else
            tU32 pvSharedMemory;
#endif
            tU32 u32Size;
            tU32 u32LastError;
            tS32 s32ReturnValue;

            ahl_tclStreamer oSendMessageStream; // Input Stream - RPC Request
            oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

            oSendMessageStream >> pvSharedMemory;
            oSendMessageStream >> u32Size;

            s32ReturnValue = OSAL_s32SharedMemoryUnmap( (tPVoid)pvSharedMemory, u32Size );
            u32LastError = OSAL_u32ErrorCode();

            ahl_tclStreamer oOutputStream( SHAREDMEMORYUNMAP_ANSWER_LENGTH );   // Output Stream - RPC Reply

            oOutputStream << u32Magic;
            oOutputStream << s32ReturnValue;
            oOutputStream << u32LastError;

            if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), SHAREDMEMORYUNMAP_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;
         }
         break;
#if (OSAL_OS == OSAL_WINNT && !defined VARIANT_S_FTR_ENABLE_TSIM) || (OSAL_OS == OSAL_LINUX || OSAL_OS==OSAL_DARWIN)
      case  WOS_S32IO_ENGINEERING_FCT:
         {
            if( pau8Message != NULL )
            {
               tU32 u32Magic = WOS_S32IO_ENGINEERING_FCT_MAGIC;
               tS32 s32ReturnValue;
               tU32 u32LastError;
               tS32 s32IoctlFID;
               tU32 u32ArgSize;

               ahl_tclStreamer oSendMessageStream; // Input Stream - RPC Request
               oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

               tU32 u32NameLength = (tU32)OSAL_u32StringLength( (tPC8)pau8Message ) + 1;

               tPC8 pcDeviceName = new tC8[u32NameLength];
               oSendMessageStream.vReadString( pcDeviceName, (tU16)u32NameLength );
               oSendMessageStream >> s32IoctlFID;
               oSendMessageStream >> u32ArgSize;

               ahl_tclStreamer pcArgument( u32ArgSize );
               pcArgument.rfWriteStream(oSendMessageStream);
               intptr_t s32ArgumentPointer = (intptr_t)pcArgument.pGetStream();

               ahl_tclStreamer oOutputStream( IOENGINEERINGFCT_ANSWER_LENGTH + u32ArgSize ); // Output Stream - RPC Reply

               s32ReturnValue = WOS_s32IOEngineeringFct( pcDeviceName, s32IoctlFID, (tS32)s32ArgumentPointer );
               u32LastError = OSAL_u32ErrorCode();

               oOutputStream << u32Magic;
               oOutputStream << u32LastError;
               oOutputStream << s32ReturnValue;
               oOutputStream << u32ArgSize;
               oOutputStream.rfWriteStream(pcArgument);

               if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), (IOENGINEERINGFCT_ANSWER_LENGTH + u32ArgSize) ) != OSAL_ERROR )
                bMessageSend = TRUE;

               delete[] pcDeviceName;
            }
         }
         break;
#endif
#if OSAL_OS == OSAL_WINNT && !defined VARIANT_S_FTR_ENABLE_TSIM
      case  ENG_FAKE_EXCL_ACCESS:
         {
            // Call through osalengine function pointer in osal_pure-NT
            if ( (pau8Message != NULL) && (pure_osal_io_bEngFakeExclAccess != NULL) )
            {
               tU32 u32Magic = ENG_FAKE_EXCL_ACCESS_MAGIC;
               tS32 s32ReturnValue;
               tU32 u32LastError;

               ahl_tclStreamer oSendMessageStream; // Input Stream - RPC Request
               oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

               tU32 u32NameLength = (tU32)OSAL_u32StringLength( (tPC8)pau8Message ) + 1;

               tPC8 pcDeviceName = new tC8[u32NameLength];
               oSendMessageStream.vReadString( pcDeviceName, (tU16)u32NameLength );

               ahl_tclStreamer oOutputStream( ENG_FAKE_EXCL_ANSWER_LENGTH ); // Output Stream - RPC Reply

               s32ReturnValue = ( pure_osal_io_bEngFakeExclAccess(pcDeviceName) == true ) ? OSAL_OK : OSAL_ERROR;
               u32LastError = OSAL_u32ErrorCode();

               oOutputStream << u32Magic;
               oOutputStream << u32LastError;
               oOutputStream << s32ReturnValue;

               if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), ENG_FAKE_EXCL_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;

               delete[] pcDeviceName;
            }
         }
         break;

      case  ENG_REMOVE_EXCL_ACCESS:
         {
            // Call through osalengine function pointer in osal_pure-NT
            if( (pau8Message != NULL) && (pure_osal_io_bEngRemoveExclAccess != NULL) )
            {
               tU32 u32Magic = ENG_REMOVE_EXCL_ACCESS_MAGIC;
               tS32 s32ReturnValue;
               tU32 u32LastError;

               ahl_tclStreamer oSendMessageStream; // Input Stream - RPC Request
               oSendMessageStream.vSetStreamBuffer( (tPC8)pau8Message, u32BufferLength, u32BufferLength );

               tU32 u32NameLength = (tU32)OSAL_u32StringLength( (tPC8)pau8Message ) + 1;

               tPC8 pcDeviceName = new tC8[u32NameLength];
               oSendMessageStream.vReadString( pcDeviceName, (tU16)u32NameLength );

               ahl_tclStreamer oOutputStream( ENG_REMOVE_EXCL_ACCESS_MAGIC ); // Output Stream - RPC Reply

               s32ReturnValue = ( pure_osal_io_bEngRemoveExclAccess(pcDeviceName) == true ) ? OSAL_OK : OSAL_ERROR;
               u32LastError = OSAL_u32ErrorCode();

               oOutputStream << u32Magic;
               oOutputStream << u32LastError;
               oOutputStream << s32ReturnValue;

               if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), ENG_REMOVE_EXCL_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;

               delete[] pcDeviceName;
            }
         }
         break;
#endif

#if (OSAL_OS == OSAL_WINNT && !defined VARIANT_S_FTR_ENABLE_TSIM) || (OSAL_OS == OSAL_LINUX) || (OSAL_OS == OSAL_DARWIN)
      case  MINISPM_SHUTDOWN:
         {
            tU32 u32Magic = MINISPM_SHUTDOWN_MAGIC;
            vTerminateCcaApplications();

            ahl_tclStreamer oOutputStream( MINISPM_SHUTDOWN_ANSWER_LENGTH );   // Output Stream - RPC Reply

            tS32 s32ReturnValue = OSAL_OK;
            oOutputStream << u32Magic;
            oOutputStream << s32ReturnValue;

            if (m_poMessageSender->s32SendMessage( oOutputStream.pGetStream(), MINISPM_SHUTDOWN_ANSWER_LENGTH ) != OSAL_ERROR )
                bMessageSend = TRUE;
         }
         break;
#endif

      default:
         break;
      }
   }
   return bMessageSend;
}


/******************************************************************************
 FUNCTION:   | tclOSAL_RPCMessageParser::bAdjustCopyBuffer()
-------------------------------------------------------------------------------
 DESCRIPTION:| Allocates required memory for copy buffer and initializes 
             | expected size (m_u32CopyBufferSize) and message buffer 
             | index (m_u32CopyBufferIndex).
-------------------------------------------------------------------------------
 PARAMETERS: | tU32 u32RequestedSize: [I] Number of bytes required for message
-------------------------------------------------------------------------------
 RETURN TYPE:| tBool : TRUE if adjusting was successful else FALSE
-------------------------------------------------------------------------------
 HISTORY:    | Date      | Author            | Modification
-------------------------------------------------------------------------------
             | 2017-04-10| Dke               | Minor refactoring
******************************************************************************/
tBool tclOSAL_RPCMessageParser::bAdjustCopyBuffer( tU32 u32RequestedSize )
{
   tBool bResult = TRUE;
   if ( m_pau8CopyBuffer == NULL )
   {
      m_pau8CopyBuffer = new tU8[u32RequestedSize];
      if ( m_pau8CopyBuffer != NULL )
      {
         m_u32CopyBufferSize = u32RequestedSize;
         m_u32CopyBufferIndex = 0;
         bResult = TRUE;
      }
      else
      {
         bResult = FALSE;
      }
   }

   return bResult;
}


/******************************************************************************
 FUNCTION:   | tclOSAL_RPCMessageParser::bConsumeMessage()
-------------------------------------------------------------------------------
 DESCRIPTION:| Consumes bytes from the given pau8Buffer until the internal
             | buffer contains all data required for the expected message
             | as given in u32RequiredMsgSize.
             | This method will adjust pau8Buffer and u32BufferSize to reflect
             | the amount of consumed bytes.
-------------------------------------------------------------------------------
 PARAMETERS: | tU32 u32RequiredMsgSize: [I] Final size of expected message
             | tPU8 pau8Buffer        : [IO] Pointer to new data
             | tU32 u32BufferSize     : [IO] Number of bytes in pau8Buffer
-------------------------------------------------------------------------------
 RETURN TYPE:| tBool : TRUE if message was completed else FALSE
-------------------------------------------------------------------------------
 HISTORY:    | Date      | Author            | Modification
-------------------------------------------------------------------------------
             | 2017-04-10| Dke               | Minor refactoring
******************************************************************************/
tBool tclOSAL_RPCMessageParser::bConsumeMessage( tU32 u32RequiredMsgSize, tPU8& pau8Buffer, tU32& u32BufferSize )
{
   tBool bIsMessageComplete = FALSE;

   if ( m_u32CopyBufferIndex < u32RequiredMsgSize )
   {
      const tS32  s32BytesToCompletion = (u32RequiredMsgSize - m_u32CopyBufferIndex);
      size_t      uiBytesToRead        = u32BufferSize;

      if ( s32BytesToCompletion > 0 )
      {
         if ( s32BytesToCompletion < uiBytesToRead )
            uiBytesToRead = s32BytesToCompletion;
      }
      else
      {
         uiBytesToRead = 0;
      }

      // Complete access param present in message
      OSAL_pvMemoryCopy( &m_pau8CopyBuffer[m_u32CopyBufferIndex],
                        pau8Buffer,
                        uiBytesToRead );

      u32BufferSize        -= uiBytesToRead;
      pau8Buffer           += uiBytesToRead;
      m_u32CopyBufferIndex += uiBytesToRead;

      if ( m_u32CopyBufferIndex == u32RequiredMsgSize )
      {
         bIsMessageComplete = TRUE;
      }
   }
   else
   {
      bIsMessageComplete = TRUE;
   }

   return bIsMessageComplete;
}

/******************************************************************************
 FUNCTION:   | tclOSAL_RPCMessageParser::bValidateProtocol()
-------------------------------------------------------------------------------
 DESCRIPTION:| Compare the bytes after parsing with that sent by the client 
             | to verify whether the message is complying to RPC protocol.
-------------------------------------------------------------------------------
 PARAMETERS: | tPU8 pau8Message     : [->I] Message bytes (sans FID)
             | tU32 u32BufferLength : [I] Length of message (sans FID)
-------------------------------------------------------------------------------
 RETURN TYPE:| tBool : TRUE if OK else FALSE
-------------------------------------------------------------------------------
 HISTORY:    | Date      | Author            | Modification
-------------------------------------------------------------------------------
             | 13.02.09  | Soj Thomas, RBEI  | Initial version
******************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_CPPUNIT_TEST
tBool tclOSAL_RPCMessageParser::bValidateProtocol( tPU8 pau8Buffer, tU32 u32BufferSize )
{
   tBool bRet = FALSE;

   // Do not validate unless called from rpctestclient
   // When called from osal_pure_unittest through the socket interface, 
   // the message protocol cannot be validated since we do not habe the original messsge bytes
   if ( (tclRPCTestClient::m_pcSendBuffer != OSAL_NULL) && (tclRPCTestClient::m_u32SendBufferSize != 0) )
   {
      // If length is not OK nothing else is OK
      if ( u32BufferSize == (tclRPCTestClient::m_u32SendBufferSize - sizeof(tU32)) )
      {
         // pau8Buffer is excluding FID and length of FID. Reduce this from sent message
         if ( 0 == OSAL_s32MemoryCompare( pau8Buffer,
                                          ( tclRPCTestClient::m_pcSendBuffer + sizeof(tU32) ),
                                          ( tclRPCTestClient::m_u32SendBufferSize - sizeof(tU32) ) ) ) bRet = TRUE;
      }
   }
   else
   {
      bRet = TRUE;
   }

   return bRet;
}
#endif
