/************************************************************************
 *
 * FILE:         osutilio.c
 * PROJECT:      VASCO
 * SW-COMPONENT: OSAL
 *----------------------------------------------------------------------
 * DESCRIPTION:  This source file contains osal utility io functions
 *               
 *----------------------------------------------------------------------
 * COPYRIGHT:    (c) 2000 Robert Bosch GmbH, Hildesheim
 * HISTORY:      
 * Date      |   Modification               | Author
 * 09.08.00  |   Initial revision           | Scherber, EFT43
 * 05.09.00  |   OSALUTIL_OpenDir           | Scherber,EFT43
 * 15.09.00  |   NT -> common OSAL Folder   | Uelschen, EFT42
 * 21.05.03  |   OSALUTIL_OpenDir           | CM-CR/EES4-kos2hi
 *************************************************************************/

/**************************************************************************/
/* includes                                                               */
/**************************************************************************/
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

/**************************************************************************/
/* typedefs                                                               */
/**************************************************************************/

/**************************************************************************/
/* defines                                                               */
/**************************************************************************/

/************************************************************************
 *
 * FUNCTION:     OSALUTIL_s32NPrintFormat
 *----------------------------------------------------------------------
 * DESCRIPTION:  writes a formatted string to buffer (aka vsnprintf)
 *----------------------------------------------------------------------
 * PARAMETER:    (I) ps8Buffer: buffer
 * PARAMETER:    (I) u32BufferLength: buffer length
 * PARAMETER:    (I) coszFormat: format string like sprintf
 * PARAMETER:    (I) ...: arguments
 * RETURNVALUE:  number of written bytes or OSAL_ERROR if max buffer size is reached
 * HISTORY:      
 * Date      |   Modification                         | Author
 * 12.07.01  |   Initial revision                     | Uelschen, ENT1
 *
 *************************************************************************/
tS32 OSALUTIL_s32NPrintFormat(tPS8 ps8Buffer, tU32 u32BufferLength, tCString coszFormat, ...) { 
	tS32 s32Size;
    OSAL_tVarArgList argList;

    // argList is a built-in array, deLint warning
	OSAL_VarArgStart(argList,coszFormat); //lint !e530
	s32Size=OSAL_s32VarNPrintFormat((tString)ps8Buffer,u32BufferLength,coszFormat,argList); //lint !e530 
                                                                                           // args has been initialized within OSAL_VarArgStart
    // ArgEnd was forgotten, fix it. TMS-Mrh 28.05.05
    OSAL_VarArgEnd(argList); //lint !e530 
                             // args has been initialized within OSAL_VarArgStart
	return s32Size<0 ? OSAL_ERROR : s32Size;
}

/************************************************************************
 *
 * FUNCTION:     OSALUTIL_s32FPrintf
 *----------------------------------------------------------------------
 * DESCRIPTION:  writes a string into a file
 *----------------------------------------------------------------------
 * PARAMETER:    (I) fd: file descriptor
 * PARAMETER:    (I) szFormat: format string
 * PARAMETER:    (I) ...: arguments
 * RETURNVALUE:  number of written bytes or OSAL_ERROR if max buffer size is reached
 * HISTORY:      
 * Date      |   Modification                         | Author
 * 09.08.00  |   Initial revision                     | Scherber,EFT43
 *
 *************************************************************************/
tS32 OSALUTIL_s32FPrintf(OSAL_tIODescriptor fd, tCString szFormat, ...)
{
   tChar szBuffer[OSALUTIL_C_U32_MAX_BUFFERSIZE];
   tS32 s32Size;
   va_list argList;

   // argList is a built-in array, deLint warning
   OSAL_VarArgStart(argList, szFormat); //lint !e530
   s32Size = OSAL_s32VarNPrintFormat(szBuffer, OSALUTIL_C_U32_MAX_BUFFERSIZE, szFormat, argList); //lint !e530 
                                                                                                  // args has been initialized within OSAL_VarArgStart
   OSAL_VarArgEnd(argList); //lint !e530 
                            // args has been initialized within OSAL_VarArgStart
   if (s32Size<0)
      return OSAL_ERROR;
   else
      return OSAL_s32IOWrite(fd, (tPCS8) szBuffer, (tU32)s32Size);
}


/************************************************************************
 *
 * FUNCTION:     OSALUTIL_s32FSeek
 *----------------------------------------------------------------------
 * DESCRIPTION:  set file pointer
 *----------------------------------------------------------------------
 * PARAMETER:    (I) fd: file descriptor
 * PARAMETER:    (I) offset: 
 * PARAMETER:    (I) origin: SEEK_SET, SEEK_CUR or SEEK_END
 * RETURNVALUE:  OSAL_OK or OSAL_ERROR
 * HISTORY:      
 * Date      |   Modification                         | Author
 * 11.08.00  |   Initial revision                     | Scherber,EFT43
 *
 *************************************************************************/
tS32 OSALUTIL_s32FSeek(OSAL_tIODescriptor fd, tS32 offset, tS32 origin)
{
   intptr_t curPos;
   intptr_t endPos;
   if (origin == OSAL_C_S32_SEEK_SET)
   {
      return OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_FIOSEEK, (intptr_t)offset);
   }
   else if (origin == OSAL_C_S32_SEEK_CUR)
   {
       if (OSAL_ERROR == OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_FIOWHERE, (intptr_t) &curPos))
          return OSAL_ERROR;
       offset += (tS32)curPos;
       return OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_FIOSEEK, (intptr_t)offset);
   }
   else if (origin == OSAL_C_S32_SEEK_END)
   {
      if (OSAL_ERROR == OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_FIONREAD, (intptr_t) &endPos))
         return OSAL_ERROR;
      if (OSAL_ERROR == OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_FIOWHERE, (intptr_t) &curPos))
         return OSAL_ERROR;
      offset = (tS32)(curPos + endPos + offset);
      return OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_FIOSEEK, (intptr_t)offset);
   }
   else
      return OSAL_ERROR;
}

/************************************************************************
 *
 * FUNCTION:     OSALUTIL_s32FTell
 *----------------------------------------------------------------------
 * DESCRIPTION:  returns current file position
 *----------------------------------------------------------------------
 * PARAMETER:    (I) fd: file descriptor
 * PARAMETER:    (I) offset: 
 * PARAMETER:    (I) origin: SEEK_SET, SEEK_CUR or SEEK_END
 * RETURNVALUE:  OSAL_ERROR or position
 * HISTORY:      
 * Date      |   Modification                         | Author
 * 11.08.00  |   Initial revision                     | Scherber,EFT43
 *
 *************************************************************************/
tS32 OSALUTIL_s32FTell(OSAL_tIODescriptor fd)
{
   intptr_t curPos;
   if (OSAL_ERROR == OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_FIOWHERE, (intptr_t) &curPos))
      return OSAL_ERROR;
   else
      return (tS32)curPos;
}


/************************************************************************
 *
 * FUNCTION:     OSALUTIL_s32FGetpos
 *----------------------------------------------------------------------
 * DESCRIPTION:  returns current file position
 *----------------------------------------------------------------------
 * PARAMETER:    (I) fd: file descriptor
 * PARAMETER:    (->0) ptr: file position
 * RETURNVALUE:  OSAL_ERROR or OSAL_OK
 * HISTORY:      
 * Date      |   Modification                         | Author
 * 11.08.00  |   Initial revision                     | Scherber,EFT43
 *
 *************************************************************************/
tS32 OSALUTIL_s32FGetpos(OSAL_tIODescriptor fd, intptr_t* ptr)
{
   return OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_FIOWHERE, *ptr);
}
   
/************************************************************************
 *
 * FUNCTION:     OSALUTIL_s32FSetpos
 *----------------------------------------------------------------------
 * DESCRIPTION:  sets current file position
 *----------------------------------------------------------------------
 * PARAMETER:    (I) fd: file descriptor
 * PARAMETER:    (->I) ptr: file position
 * RETURNVALUE:  OSAL_ERROR or OSAL_OK
 * HISTORY:      
 * Date      |   Modification                         | Author
 * 11.08.00  |   Initial revision                     | Scherber,EFT43
 *
 *************************************************************************/
tS32 OSALUTIL_s32FSetpos(OSAL_tIODescriptor fd, const intptr_t* ptr)
{
   return OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_FIOSEEK, *ptr);
}
   
/************************************************************************
 *
 * FUNCTION:     OSALUTIL_s32FGetSize
 *----------------------------------------------------------------------
 * DESCRIPTION:  returns file size
 *----------------------------------------------------------------------
 * PARAMETER:    (I) fd: file descriptor
 * RETURNVALUE:  OSAL_ERROR or file size
 * HISTORY:      
 * Date      |   Modification                         | Author
 * 11.08.00  |   Initial revision                     | Scherber,EFT43
 *
 *************************************************************************/
tS32 OSALUTIL_s32FGetSize(OSAL_tIODescriptor fd)
{
   intptr_t front;
   intptr_t back;
   if (OSAL_ERROR == OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_FIOWHERE, (intptr_t) &front))
      return OSAL_ERROR;
   if (OSAL_ERROR == OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_FIONREAD, (intptr_t) &back))
      return OSAL_ERROR;
   return (tS32)(front+back);
}

/************************************************************************
 *
 * FUNCTION:     OSALUTIL_s32CreateDir
 *----------------------------------------------------------------------
 * DESCRIPTION: creates a directory
 *----------------------------------------------------------------------
 * PARAMETER:    (I) fd: file descriptor
 * PARAMETER:    (I) szDirectory: directory name
 * RETURNVALUE:  OSAL_ERROR or OSAL_OK
 * HISTORY:      
 * Date      |   Modification                         | Author
 * 11.08.00  |   Initial revision                     | Scherber,EFT43
 *
 *************************************************************************/
tS32 OSALUTIL_s32CreateDir(OSAL_tIODescriptor fd, tCString szDirectory)
{
   return OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_FIOMKDIR, (intptr_t) szDirectory);
}

/************************************************************************
 *
 * FUNCTION:     OSALUTIL_s32RemoveDir
 *----------------------------------------------------------------------
 * DESCRIPTION: removes a directory
 *----------------------------------------------------------------------
 * PARAMETER:    (I) fd: file descriptor
 * PARAMETER:    (I) szDirectory: directory name
 * RETURNVALUE:  OSAL_ERROR or OSAL_OK
 * HISTORY:      
 * Date      |   Modification                         | Author
 * 11.08.00  |   Initial revision                     | Scherber,EFT43
 *
 *************************************************************************/
tS32 OSALUTIL_s32RemoveDir(OSAL_tIODescriptor fd, tCString szDirectory)
{
   return OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_FIORMDIR, (intptr_t) szDirectory);
}

/************************************************************************
 *
 * FUNCTION:     OSALUTIL_OpenDir
 *----------------------------------------------------------------------
 * DESCRIPTION: opens a directory for reading
 *----------------------------------------------------------------------
 * PARAMETER:    (I) szDirectory: directory name
 * RETURNVALUE:  OSAL_tIODir or OSAL_NULL
 * HISTORY:      
 * Date      |   Modification                         | Author
 * 11.08.00  |   Initial revision                     | Scherber,EFT43
 * 05.09.00  |   Parameter change                     | Scherber,EFT43
 * 21.05.03  |   Change access right from readonly to | CM-CR/EES4-kos2hi
 *           |   readwrite                            |
 *
 *************************************************************************/
OSAL_trIOCtrlDir *OSALUTIL_prOpenDir(tCString szDirectory)
{
   OSAL_trIOCtrlDir *pDir;

   /* allocate memory */
   pDir = (OSAL_trIOCtrlDir*) OSAL_pvMemoryAllocate(sizeof(OSAL_trIOCtrlDir));
   if (pDir == NULL)
      return OSAL_NULL;

   /* open dir */
//#ifdef VASCO_OS_TENGINE  // -bn: 110221: line replaced by next one, because we observed the following issue in TSIM: 
                           //      An OSAL_IOOpen(.) with access type OSAL_EN_READWRITE would result in a success even 
                           //      for a file. This is wrong here, as we want to open a directory by calling 
                           //      OSALUTIL_prOpenDir(.)
#if defined(VASCO_OS_TENGINE) || defined(TSIM_OSAL)
   pDir->fd = OSAL_IOOpen(szDirectory,(OSAL_tenAccess)OSAL_EN_ACCESS_DIR);
#else
   pDir->fd = OSAL_IOOpen(szDirectory, OSAL_EN_READWRITE);
#endif
   if (pDir->fd == OSAL_ERROR)
   {
      OSAL_vMemoryFree(pDir);
      return OSAL_NULL;
   }
   pDir->s32Cookie = 0;
   return pDir;
}

/************************************************************************
 *
 * FUNCTION:     OSALUTIL_ReadDir
 *----------------------------------------------------------------------
 * DESCRIPTION: opens a directory for reading
 *----------------------------------------------------------------------
 * PARAMETER:    (I) dir: Directory
 * RETURNVALUE:  OSAL_tIODir or OSAL_NULL
 * HISTORY:      
 * Date      |   Modification                         | Author
 * 11.08.00  |   Initial revision                     | Scherber,EFT43
 * 05.09.00  |   Bug fix                              | Scherber,EFT43
 *
 *************************************************************************/
OSAL_trIOCtrlDirent *OSALUTIL_prReadDir(OSAL_trIOCtrlDir* pDir)
{
   if (pDir == OSAL_NULL)
   {
      OSAL_vSetErrorCode( OSAL_E_INVALIDVALUE );      /* Set error code */
      return OSAL_NULL;
   }

   if (OSAL_ERROR == OSAL_s32IOControl(pDir->fd, OSAL_C_S32_IOCTRL_FIOREADDIR, (intptr_t)pDir))
   {
      return OSAL_NULL;
   }
   else
   {
      return &pDir->dirent;
   }
}

/************************************************************************
 *
 * FUNCTION:     OSALUTIL_CloseDir
 *----------------------------------------------------------------------
 * DESCRIPTION: closes a directory 
 *----------------------------------------------------------------------
 * PARAMETER:    (I) dir: Directory
 * RETURNVALUE:  OSAL_OK or OSAL_ERROR
 * HISTORY:      
 * Date      |   Modification                         | Author
 * 11.08.00  |   Initial revision                     | Scherber,EFT43
 *
 *************************************************************************/
tS32 OSALUTIL_s32CloseDir(OSAL_trIOCtrlDir* pDir)
{
   if (pDir == OSAL_NULL)
   {
      OSAL_vSetErrorCode( OSAL_E_INVALIDVALUE );      /* Set error code */
      return OSAL_ERROR;
   }

   OSAL_s32IOClose(pDir->fd);
   OSAL_vMemoryFree(pDir);
   return OSAL_OK;
}

/************************************************************************
 *
 * FUNCTION:     OSALUTIL_s32TraceWrite
 *----------------------------------------------------------------------
 * DESCRIPTION: writes a buffer to the trace device
 *----------------------------------------------------------------------
 * PARAMETER:    (I) io: descriptor of io device
 * PARAMETER:    (I) u32Level: trace or debug level
 * PARAMETER:    (I) u32Class: trace or debug class
 * PARAMETER:    (I) pcos8Buffer: pointer to char buffer
 * PARAMETER:    (I) u32Length: size of buffer
 * RETURNVALUE:  Number of written bytes or OSAL_ERROR
 * HISTORY:      
 * Date      |   Modification                         | Author
 * 05.03.01  |   Initial revision                     | Uelschen, ENT1
 *
 *************************************************************************/
tS32 OSALUTIL_s32TraceWrite(OSAL_tIODescriptor io, tU32 u32Level, tU32 u32Class,
							tPCS8 pcos8Buffer, tU32 u32Length) {
	/* --Simply create an appropriate struct and exec a write to the trace device. */
	OSAL_trIOWriteTrace rWriteTrace;
	rWriteTrace.u32Length=u32Length;
	rWriteTrace.u32Level=u32Level;
	rWriteTrace.pcos8Buffer=pcos8Buffer;
	rWriteTrace.u32Class=u32Class;
	return OSAL_s32IOWrite(io,(tPCS8)&rWriteTrace,sizeof(rWriteTrace));
}

/************************************************************************
 *
 * FUNCTION:     OSALUTIL_s32TracePrintf
 *----------------------------------------------------------------------
 * DESCRIPTION:  writes a string into the trace device
 *----------------------------------------------------------------------
 * PARAMETER:    (I) fd: file descriptor
 * PARAMETER:    (I) u32Level: trace or debug level
 * PARAMETER:    (I) u32Class: trace or debug class
 * PARAMETER:    (I) szFormat: format string
 * PARAMETER:    (I) ...: arguments
 * RETURNVALUE:  number of written bytes or OSAL_ERROR if max buffer size is reached
 * HISTORY:
 * Date      |   Modification                         | Author
 * 27.03.01  |   Initial revision                     | Scherber,ENT1
 *
 *************************************************************************/
tS32 OSALUTIL_s32TracePrintf(OSAL_tIODescriptor fd, tU32 u32Level, tU32 u32Class, tCString coszFormat, ...)
{
   tChar szBuffer[OSALUTIL_C_U32_MAX_BUFFERSIZE];
   tS32 s32Size;
   va_list argList;

   // argList is a built-in array, deLint warning
   OSAL_VarArgStart(argList, coszFormat); //lint !e530
   s32Size = OSAL_s32VarNPrintFormat(szBuffer, OSALUTIL_C_U32_MAX_BUFFERSIZE, coszFormat, argList); //lint !e530 
                                                                                                    // args has been initialized within OSAL_VarArgStart
   OSAL_VarArgEnd(argList); //lint !e530 
                            // args has been initialized within OSAL_VarArgStart
   if (s32Size < 0)
      return OSAL_ERROR;
   else      
      return OSALUTIL_s32TraceWrite(fd, u32Level, u32Class, (tPCS8)szBuffer, (tU32)s32Size);

}

/************************************************************************
 *
 * FUNCTION:     OSALUTIL_u32GetBaseAPIVersion
 *----------------------------------------------------------------------
 * DESCRIPTION:  returns the OSAL API version supported by the base SW 
 *----------------------------------------------------------------------
 * RETURNVALUE:  OSAL API version supported by base SW
 * HISTORY:
 * Date      |   Modification                         | Author
 * 27.08.07  |   Initial revision                     | Ramscheid, TMS
 *
 *************************************************************************/
tU32 OSALUTIL_u32GetBaseAPIVersion(tVoid)
{
tU32 u32Ret=OSAL_API_V0100;
  
  #if (OSAL_OS==OSAL_NUCLEUS) 
    u32Ret = OSAL_VERSION;
  #endif // OSAL_NUCLEUS

  return u32Ret;
}



