/************************************************************* +MHDR * .MA *

 .FILE             [ ossema.cpp  ]

 .SW-COMPONENT      osal

 .DESCRIPTION
  This file contains the implementation of the class OSAL_tclSemaphore.

 .COPYRIGHT            [ (c) 2001 Robert Bosch GmbH, Hildesheim           ]
 .AUTHOR               [ U. Walkling                                      ]

 .HISTORY

  Date     | Name            Modification
 ----------|---------------------------------------------------------------
  18.01.02 | CM-DI/EST-Wg    New
  07.02.02 | EST-Wg          Bugfixes: some checks for null pointer as name 
           |                 parameter
           |                 add OSAL_tclBase class
  01.03.02 | CM-DI/EST-Wg    destructor: name deleted if exists;
           |                 NULL -> OSAL_NULL
           |                 s32Close: check first the handle before calling
           |                 the OSAL-close function.
************************************************************** -MHDR * .ME */
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
/* Constructor of this class. */
OSAL_tclSemaphore::OSAL_tclSemaphore()
{
   /*
      Initialisation of the member variables of the class.
   */
   szName = OSAL_NULL;
   hSemaphoreHandle = OSAL_C_INVALID_HANDLE;
}
/* Destructor of this class. */
OSAL_tclSemaphore::~OSAL_tclSemaphore()
{
   this->s32Close();
   if (OSAL_NULL != this->szName)
   {
      /*
         Delete the name string.
      */
      OSAL_vMemoryFree( (tPVoid)this->szName);
      this->szName = OSAL_NULL;
   }
}
/*
   This methode creates the semaphore. 
   If the creation of the semaphore has finished without
   an error the s32Open methode is called afterwards.
   (For more detail see OSAL Refernzhandbuch Kapitel Semaphore.)

   Additional Errors or meaning:
   OSAL_E_UNKNOWN: A handle already stored in this object.
   OSAL_E_NOSPACE: There is no space to save the name.
*/
tS32 OSAL_tclSemaphore::s32Create( tCString coszName    /* I */,
                                   tU32 u32InitialValue /* I */)
{
   tS32 s32ErrorValue = OSAL_OK;
   /*
      This member stores the temporary handle of the semaphore.
   */
   OSAL_tSemHandle hTempSemaphoreHandle = OSAL_C_INVALID_HANDLE;
   do
   {
      if (OSAL_NULL == coszName)
      {
         /*
            There is a NULL pointer as name specified.
         */
         s32ErrorValue = OSAL_ERROR;
         OSAL_vSetErrorCode( OSAL_E_INVALIDVALUE );
         break;
      }
      if (OSAL_C_INVALID_HANDLE != this->hSemaphoreHandle)
      {
         /*
            There is a handle already stored in this object.
         */
         s32ErrorValue = OSAL_ERROR;
         OSAL_vSetErrorCode( OSAL_E_UNKNOWN );
         break;
      }
      if (OSAL_NULL != this->szName)
      {
         /*
            Delete the name string.
         */
         OSAL_vMemoryFree( (tPVoid)this->szName);
         this->szName = OSAL_NULL;
      }
      s32ErrorValue = this->s32SaveName( coszName, (tCString&)this->szName );
      if (OSAL_OK == s32ErrorValue)
      {
         /*
            Call the OSAL function to create the semaphore.
         */
         s32ErrorValue = OSAL_s32SemaphoreCreate( coszName, 
                                                  &hTempSemaphoreHandle,
                                                  u32InitialValue);
      }
      else
      {
         break;
      }
      if (OSAL_ERROR == s32ErrorValue)
      {
         /*
            Delete the name string.
         */
         OSAL_vMemoryFree( (tPVoid)this->szName);
         this->szName = OSAL_NULL;
      }
      /*
         Save the object internal Values.
      */
      this->hSemaphoreHandle = hTempSemaphoreHandle;
   }
   while (FALSE);
   return s32ErrorValue;
}
/*
   This methode opens a semaphore.
   (For more detail see OSAL Refernzhandbuch Kapitel Semaphore.)

   Additional Errors or meaning:
   OSAL_E_ALREADYEXISTS: There exists already an opened semaphore.
   OSAL_E_NOSPACE: There is no space to save the name.
*/
tS32 OSAL_tclSemaphore::s32Open( tCString coszName /* (I) */ )
{
   tS32 s32ErrorValue = OSAL_OK;
   /*
      This member stores the temporary handle of the semaphore.
   */
   OSAL_tSemHandle hTempSemaphoreHandle = OSAL_C_INVALID_HANDLE;
   /*
      This member stores temporary the name of the semaphore.
   */
   tString szTempName = this->szName;
   do
   {
      if (OSAL_NULL == coszName)
      {
         /*
            There is a NULL pointer as name specified.
         */
         s32ErrorValue = OSAL_ERROR;
         OSAL_vSetErrorCode( OSAL_E_INVALIDVALUE );
         break;
      }
      /*
         Check whether there is already a handle saved in the object.
      */
      if (OSAL_C_INVALID_HANDLE != this->hSemaphoreHandle)
      {
         /*
            Set the error to tell the user there exists already an 
            opened semaphore.
         */
         s32ErrorValue = OSAL_ERROR;
         OSAL_vSetErrorCode( OSAL_E_ALREADYEXISTS );
         break;
      }
      s32ErrorValue = this->s32SaveName( coszName, (tCString&)this->szName );
      /*
         Check whether there is already a name saved in the object.
      */
      if (OSAL_OK == s32ErrorValue)
      {
         s32ErrorValue = OSAL_s32SemaphoreOpen( coszName, 
                                                &hTempSemaphoreHandle );
         if (OSAL_OK == s32ErrorValue)
         {
            /*
               Save the object internal Values.
            */
            this->hSemaphoreHandle = hTempSemaphoreHandle;
            /*
               Delete the new name string.
            */
            OSAL_vMemoryFree( (tPVoid)szTempName );
         }
         else
         {
            /*
               Delete the new name string.
            */
            OSAL_vMemoryFree( (tPVoid)this->szName );
            /*
               Restore the last name.
            */
            this->szName = szTempName;
         }
      }
      else
      {
         /*
            Restore the last name.
         */
         this->szName = szTempName;
      }
   }
   while (FALSE);
   return s32ErrorValue;
}
/*
   This methode opens a semaphore with the name stored
   in this object.
   (This is an additional method!)

   Additional Errors or meaning:
   OSAL_E_ALREADYEXISTS: There exists already an opened semaphore.
   OSAL_E_UNKNOWN: There is no name saved in the object. 
                   (Use s32Open( tCString coszName ) )
*/
tS32 OSAL_tclSemaphore::s32Open()
{
   tS32 s32ErrorValue = OSAL_OK;
   do
   {
      /*
         Check whether there is already a handle saved in the object.
      */
      if (OSAL_C_INVALID_HANDLE != this->hSemaphoreHandle)
      {
         /*
            Set the error to tell the user there exists already an 
            opened semaphore.
         */
         s32ErrorValue = OSAL_ERROR;
         OSAL_vSetErrorCode( OSAL_E_ALREADYEXISTS );
         break;
      }
      if (OSAL_NULL == this->szName)
      {
         /*
            Set the error to tell the user there is no name specified.
         */
         s32ErrorValue = OSAL_ERROR;
         OSAL_vSetErrorCode( OSAL_E_UNKNOWN );
         break;
      }
      s32ErrorValue = OSAL_s32SemaphoreOpen( this->szName, 
                                             &this->hSemaphoreHandle );
      if (OSAL_OK == s32ErrorValue)
      {
         /*
            Save the object internal Values.
         */
         this->hSemaphoreHandle = OSAL_C_INVALID_HANDLE;
      }
   }
   while (FALSE);
   return s32ErrorValue;
}
/*
   This methode closes a semaphore.
   (For more detail see OSAL Refernzhandbuch Kapitel Semaphore.)
*/
tS32 OSAL_tclSemaphore::s32Close( )
{
   tS32 s32ErrorValue = OSAL_OK;
   if (OSAL_C_INVALID_HANDLE != this->hSemaphoreHandle)
   {
      s32ErrorValue = OSAL_s32SemaphoreClose( this->hSemaphoreHandle );
      if (OSAL_OK == s32ErrorValue)
      {
         this->hSemaphoreHandle = OSAL_C_INVALID_HANDLE;
      }
   }
   return s32ErrorValue;
}
/*
   This methode deletes the semaphore. 
   Before deleting the semaphore the methode s32Close will be
   performed to close the semaphore. After performing this 
   methode this semaphore object provides no semaphore 
   functionality but it is possible to create a new semaphore 
   in the same object.
   (For more detail see OSAL Refernzhandbuch Kapitel Semaphore.)
*/
tS32 OSAL_tclSemaphore::s32Delete( )
{
   tS32 s32ErrorValue = OSAL_OK;
   tU32 u32ErrorCode = OSAL_E_NOERROR;
   do
   {
      if (OSAL_C_INVALID_HANDLE != this->hSemaphoreHandle)
      {
         s32ErrorValue = this->s32Close();
      }
      if (OSAL_OK == s32ErrorValue)
      {
         /*
            Reset the value of the handel.
         */
         this->hSemaphoreHandle = OSAL_C_INVALID_HANDLE;
      }
      else
      {
         break;
      }
      s32ErrorValue = OSAL_s32SemaphoreDelete( this->szName );
      u32ErrorCode = OSAL_u32ErrorCode();
      if (( OSAL_OK == s32ErrorValue )
          ||
          (
             ( OSAL_ERROR == s32ErrorValue )
             &&
             ( OSAL_E_INVALIDVALUE == u32ErrorCode)
          ))
      {
         /*
            Delete the name string.
         */
         OSAL_vMemoryFree( (tPVoid)this->szName);
         this->szName = OSAL_NULL;
      }
   }
   while (FALSE);
   return s32ErrorValue;
}
/*
   This methode is to obtain the value of the semaphore.
   (For more detail see OSAL Refernzhandbuch Kapitel Semaphore.)
*/
tS32 OSAL_tclSemaphore::s32GetValue( tPS32 ps32Value /* (->O) */ )
{
   return OSAL_s32SemaphoreGetValue( this->hSemaphoreHandle,
                                     ps32Value );
}
/*
   This methode increments the value of the semaphore.
   (For more detail see OSAL Refernzhandbuch Kapitel Semaphore.)
*/
tS32 OSAL_tclSemaphore::s32Post()
{
   return OSAL_s32SemaphorePost( this->hSemaphoreHandle );   
}
/*
   This methode decrements the value of the semaphore and the thread
   will be blocked in case of a semaphore value 0.
   (For more detail see OSAL Refernzhandbuch Kapitel Semaphore.)
*/
tS32 OSAL_tclSemaphore::s32Wait( OSAL_tMSecond msec )
{
   return OSAL_s32SemaphoreWait( this->hSemaphoreHandle,
                                 msec);
}
/* 
   The following methods are new and/or with modification 
   to the OSAL specification. 
*/
/*
   To get the name of the created semaphore this methode can be used.
   After deletion the semaphore there is no name available furthermore.
*/
tCString OSAL_tclSemaphore::pszGetName()
{
   return this->szName;
}
