/*!
*******************************************************************************
* \file               JSON_MessageHandler.h
*******************************************************************************
\verbatim
PROJECT:        Seamless pairing
SW-COMPONENT:   Seamless pairing server
DESCRIPTION:    JSON_MessageHandler
COPYRIGHT:      &copy; RBEI
HISTORY:
Date       | Author                   | Modifications
01.08.2018 | svs7kor                  | Initial Version
\endverbatim
******************************************************************************/

#include "JSON_MessageHandler.h"
#include <sys/types.h>
#include <signal.h>
#include <errno.h>
#include <ctype.h>

#define HASH_LEN 64
#define PASSWORD_FILE "/var/opt/bosch/dynamic/ipcm/pwfile.txt"
#define ACL_FILE "/var/opt/bosch/dynamic/ipcm/acl"
#define MOSQUITTO_FILE "/usr/sbin/mosquitto"
DLT_DECLARE_CONTEXT(JSON);

X509 *server_cert = NULL;
BIO  *certbio = NULL;
char* Common_name =NULL;
unsigned char finalPass[10]; 
static unsigned char *g_strCertHash = NULL;

FILE *fptr = NULL;
FILE *g_pTmpFilePtr = NULL;

#define SALT_LEN 12
#define PID_LEN 10
#define MAX_BUFFER_LEN 1024

gboolean bCalculateMessageDigest(unsigned char *arg_sMessage,
                                 int arg_iMessageLen,
                                 unsigned char **arg_sDigest,
                                 unsigned int *arg_iDigestLen)
{
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("IN "),DLT_STRING(__FUNCTION__));
    EVP_MD_CTX *l_pMdctx;
    gboolean l_bResult = TRUE;

    if((l_pMdctx = EVP_MD_CTX_create()) == NULL)
    {
        l_bResult = FALSE;
    }

    if(1 != EVP_DigestInit_ex(l_pMdctx, EVP_sha256(), NULL))
    {
        l_bResult = FALSE;
    }

    if(1 != EVP_DigestUpdate(l_pMdctx, arg_sMessage, arg_iMessageLen))
    {
        l_bResult = FALSE;
    }

    if((*arg_sDigest = (unsigned char *)OPENSSL_malloc(EVP_MD_size(EVP_sha256()))) == NULL)
    {
        l_bResult = FALSE;
    }

    if(1 != EVP_DigestFinal_ex(l_pMdctx, *arg_sDigest, arg_iDigestLen))
    {
        l_bResult = FALSE;
    }

    EVP_MD_CTX_destroy(l_pMdctx);
    return l_bResult;
}

/******************************************************************************
** FUNCTION   : base64_encode
*******************************************************************************
* \fn     base64_encode
* \brief  Function to use base64 encoding
* \param  unsigned char *in, unsigned int in_len, char **encoded
* \retval int.
******************************************************************************/
int iEncode_tobase64(unsigned char *in, unsigned int in_len, char **encoded)
{
    BIO *bmem, *b64;
    BUF_MEM *bptr;

    b64 = BIO_new(BIO_f_base64());
    BIO_set_flags(b64, BIO_FLAGS_BASE64_NO_NL);
    bmem = BIO_new(BIO_s_mem());
    b64 = BIO_push(b64, bmem);
    BIO_write(b64, in, in_len);
    if(BIO_flush(b64) != 1){
        BIO_free_all(b64);
        return 1;
    }
    BIO_get_mem_ptr(b64, &bptr);
    *encoded = calloc(bptr->length+1, 1);
    if(!(*encoded)){
        BIO_free_all(b64);
        return 1;
    }
    memcpy(*encoded, bptr->data, bptr->length);
    (*encoded)[bptr->length] = '\0';
    BIO_free_all(b64);

    return 0;
}

/******************************************************************************
** FUNCTION   : iOutput_new_password
*******************************************************************************
* \fn     iOutput_new_password
* \brief  Function to write sha512 hash to a file
* \param  FILE *fptr, const char *username, const char *password
* \retval int.
******************************************************************************/
gboolean iOutput_new_password(FILE *fptr, const char *username, const char *password)
{
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("IN "),DLT_STRING(__FUNCTION__));
    int rc;
    unsigned char salt[SALT_LEN];
    char *salt64 = NULL, *hash64 = NULL;
    unsigned char hash[EVP_MAX_MD_SIZE];
    unsigned int hash_len;
    const EVP_MD *digest = NULL;
	gboolean bReturn = FALSE;
    EVP_MD_CTX context;
    OpenSSL_add_all_digests();

    rc = RAND_bytes(salt, SALT_LEN);
    if(!rc){
                DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),DLT_STRING("Error: Insufficient entropy available \
                                                                       to perform password generation"));
                return bReturn;
    }

    rc = iEncode_tobase64(salt, SALT_LEN, &salt64);
    if(rc){
        if(salt64) free(salt64);
        DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),DLT_STRING("Error: Unable to encode salt"));
        return bReturn;
    }

    digest = EVP_get_digestbyname("sha512");
    if(!digest){
        DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),DLT_STRING("Error: Unable to create openssl digest"));
        if(salt64) free(salt64);
        return bReturn;
    }

    EVP_MD_CTX_init(&context);
    EVP_DigestInit_ex(&context, digest, NULL);
    EVP_DigestUpdate(&context, password, strlen(password));
    EVP_DigestUpdate(&context, salt, SALT_LEN);
    EVP_DigestFinal_ex(&context, hash, &hash_len);
    EVP_MD_CTX_cleanup(&context);

    rc = iEncode_tobase64(hash, hash_len, &hash64);
    if(rc){
        if(salt64) free(salt64);
        if(hash64) free(hash64);
        DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),DLT_STRING("Error: Unable to encode hash"));
        return bReturn;
    }

    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("salt64: "),DLT_STRING(salt64));
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("hash64: "),DLT_STRING(hash64));

	if(!fptr)
	{
            if(salt64) free(salt64);
            if(hash64) free(hash64);
		return bReturn;
	}
	fprintf(fptr, "%s:$6$%s$%s\n", username, salt64, hash64);
	fclose(fptr);
        if(salt64) free(salt64);
        if(hash64) free(hash64);

    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("OUT"),DLT_STRING(__FUNCTION__));
	bReturn = TRUE;
    return bReturn;
}

/******************************************************************************
** FUNCTION   : bReadCertificate
*******************************************************************************
* \fn     bReadCertificate
* \brief  Function to to read .pem file to x509 cert
* \param  void
* \retval gboolean.
******************************************************************************/
gboolean bReadCertificate(gchar *arg_strCertPath)
{
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("IN "),DLT_STRING(__FUNCTION__));
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Cert path: "),DLT_STRING(arg_strCertPath));
    int ret = 0; gboolean l_bReturn = FALSE;

    // These function calls initialize openssl for correct work.
    OpenSSL_add_all_algorithms();
    ERR_load_BIO_strings();
    ERR_load_crypto_strings();

    //Create the Input/Output BIO's.
    certbio = BIO_new(BIO_s_file());
    if(certbio)
        ret = BIO_read_filename(certbio, arg_strCertPath); //return 1 for success and 0 for failure
    else
        return l_bReturn;

    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("cert read status : "),DLT_INT(ret));

    if(ret)
    {
        DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Load x509 cert from buffer"));
        if (! (server_cert = PEM_read_bio_X509(certbio, NULL, 0, NULL))) {
            DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Error loading cert into memory\n"));
        }
        else
        {
            DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Load x509 cert from buffer is success"));
            l_bReturn = TRUE;
        }
    }
    else{
        DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("BIO_read_filename failure"));
    }

    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("OUT "),DLT_STRING(__FUNCTION__));
    return l_bReturn;
}

/******************************************************************************
** FUNCTION   : iGet_CN_FromCert
*******************************************************************************
* \fn     iGet_CN_FromCert
* \brief  Function to to read .pem file to x509 cert
* \param  void
* \retval int.
******************************************************************************/
int iGet_CN_FromCert()
{
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("IN "),DLT_STRING(__FUNCTION__));
    int Error = 0;
    int common_name_loc = -1;
    X509_NAME_ENTRY *common_name_entry = NULL;
    ASN1_STRING *common_name_asn1 = NULL;

    if(!server_cert)
    {
        DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("cert is null"));
        return Error;
    }

    // Find the position of the CN field in the Subject field of the certificate
    common_name_loc = X509_NAME_get_index_by_NID(X509_get_subject_name((X509 *) server_cert),
                                                 NID_commonName, -1);
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("common_name_loc : "),DLT_INT(common_name_loc));
    if (common_name_loc < 0) {
        return Error;
    }

    // Extract the CN field
    common_name_entry = X509_NAME_get_entry(X509_get_subject_name((X509 *) server_cert),
                                            common_name_loc);
    if (common_name_entry == NULL) {
        return Error;
    }

    // Convert the CN field to a C string
    common_name_asn1 = X509_NAME_ENTRY_get_data(common_name_entry);
    if (common_name_asn1 == NULL) {
        return Error;
    }
    Common_name = (char *) ASN1_STRING_data(common_name_asn1);

    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("common_name_str : "),DLT_STRING(Common_name));

    // Make sure there isn't an embedded NUL character in the CN
    if (ASN1_STRING_length(common_name_asn1) != strlen(Common_name)) {
        return Error;
    }

    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("OUT "),DLT_STRING(__FUNCTION__));
    return 1;

}

/******************************************************************************
** FUNCTION   : vGet_CN_FromCert
*******************************************************************************
* \fn     vGet_CN_FromCert
* \brief  Function to to read .pem file to x509 cert
* \param  void
* \retval void.
******************************************************************************/
void vCertHash_cal(gchar *arg_strCertPath)
{
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("IN "),DLT_STRING(__FUNCTION__));
    const EVP_MD *fprint_type = NULL;
    int j=0,fprint_size;
    int l_iHashSize;
    char *l_strCert = NULL;
	unsigned char finalPassInCaps[10];
    FILE *f = fopen(arg_strCertPath, "rb");

    if(NULL == f)
        return;

    fseek(f, 0, SEEK_END);

    long fsize = ftell(f);
    if(fsize == -1)
    {
        fclose(f);
        return;
    }

    fseek(f, 0, SEEK_SET);  //same as rewind(f);

    l_strCert = malloc(fsize + 1);

    if(NULL == l_strCert)
    {
        fclose(f);
        return;
    }

    l_strCert[fsize] = '\0'; //To fix Coverity warning 184180

    if(!fread(l_strCert, fsize, 1, f))
    {
        fclose(f);
        free(l_strCert);
        return;
    }

    fclose(f);

    gboolean l_bResult = bCalculateMessageDigest(l_strCert,
                                                 strlen(l_strCert),
                                                 &g_strCertHash,
                                                 &l_iHashSize);

    if(!l_bResult)
    {
        free(l_strCert);
        return;
    }

    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Cert hash: "),
            DLT_RAW(g_strCertHash,l_iHashSize));

    if(l_strCert != NULL)
        free(l_strCert);

    //Convert the hash to Hex representation
    size_t i = 0;
    unsigned char certificateHash[HASH_LEN+1];
    char tmpStr[3] = { '\0' };
    for(i = 0 ; i < l_iHashSize ; i++)
    {
        sprintf(tmpStr,"%02X",g_strCertHash[i]);
        certificateHash[j++] = tmpStr[0];
        certificateHash[j++] = tmpStr[1];
    }
    certificateHash[HASH_LEN] = '\0';
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Hash in Hex: "),DLT_STRING(certificateHash));

    int k=0;
    l_iHashSize = l_iHashSize * 2;
    for(i=(l_iHashSize-10);k<10;i++)
    {
        finalPassInCaps[k] = certificateHash[i];
        k++;
    }

    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Password: "),DLT_STRING(finalPassInCaps));
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Password length: "),DLT_INT(strlen(finalPassInCaps)));

    for(int i=0;i<strlen(finalPassInCaps);i++)
    {
        finalPass[i]= tolower(finalPassInCaps[i]);
    }

    if(g_strCertHash != NULL)
        free(g_strCertHash);

    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Password in lower case: "),DLT_STRING(finalPass));
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("OUT "),DLT_STRING(__FUNCTION__));
}
/******************************************************************************
** FUNCTION   : cleanup
*******************************************************************************
* \fn     cleanup
* \brief  Function to cleanup unused memory
* \param  void
* \retval void.
******************************************************************************/
void vCleanup()
{
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("IN "),DLT_STRING(__FUNCTION__));

    //free server_cert
    if(server_cert)
        X509_free(server_cert);

    if(certbio)
        BIO_free_all(certbio);

    /* if(fptr)
        fclose(fptr);*/

    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("OUT"),DLT_STRING(__FUNCTION__));
}

/******************************************************************************
** FUNCTION   : iSignal_mosquito
*******************************************************************************
* \fn     iSignal_mosquito
* \brief  Function to send SIGHUP to mosquitto to reload the conf
* \param  void
* \retval int.
******************************************************************************/
gboolean iSignal_mosquito()
{
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("IN "),DLT_STRING(__FUNCTION__));
    int l_iResult = -1;

    char l_strPid[PID_LEN];
    fflush(stdout); 
    FILE *l_pPidCommand = popen("pidof /usr/sbin/mosquitto 1>&1", "r");
    if(l_pPidCommand == NULL)
    {
        DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Fopen failed for pid"));
        return FALSE;
    }
    fgets(l_strPid, PID_LEN, l_pPidCommand);
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("PID in String format: "),DLT_STRING(l_strPid));
    pid_t l_iMqttPid = strtoul(l_strPid, NULL, 10);
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("PID of mosquitto is: "),DLT_INT(l_iMqttPid));
    pclose(l_pPidCommand);

    if(l_iMqttPid == -1)
        return FALSE;

    l_iResult = kill(l_iMqttPid,SIGHUP);
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Ret: "),DLT_INT(l_iResult));

    if(l_iResult != 0)
    {
        DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Error: "),DLT_STRING(strerror(errno)));
        return FALSE;
    }

    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("OUT:"),DLT_STRING(__FUNCTION__));
    return TRUE;
}

/******************************************************************************
** FUNCTION   : bUpdateACL
*******************************************************************************
* \fn     bUpdateACL
* \brief  Function to update ACL user & topic authorizations of smart phone
* \param  void
* \retval gboolean.
******************************************************************************/
gboolean bUpdateACL()
{
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("IN "),DLT_STRING(__FUNCTION__));

    FILE *fptr;
    fptr = fopen(ACL_FILE, "a");
    if(!fptr)
    {
        DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Fopen failed for ACL"));
        return FALSE;
    }
    //ACL topics rules
    fprintf(fptr, "%s %s %s\n", "#",Common_name,"Smart phone acl details");
    fprintf(fptr, "%s %s\n","user",Common_name);

    fprintf(fptr, "%s %s %s\n", "pattern","readwrite","ipcm/sp/%c/rpc/request");
    fprintf(fptr, "%s %s %s\n", "pattern","readwrite","ipcm/sp/%c/rpc/response");
    fprintf(fptr, "%s %s %s\n", "pattern","readwrite","ipcm/sp/%c/state");
    fprintf(fptr, "%s %s %s\n", "pattern","readwrite","sp/ipcm/%c/rpc/request");
    fprintf(fptr, "%s %s %s\n", "pattern","readwrite","sp/ipcm/%c/rpc/response");
    fclose(fptr);
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("OUT"),DLT_STRING(__FUNCTION__));
    return TRUE;
}

int iCopyFileContents(FILE *arg_pSource, FILE *arg_pDest)
{
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("IN iCopyFileContents"));
    char l_sBuffer[MAX_BUFFER_LEN];
    int l_iLength;

        if(!arg_pSource || !arg_pDest)
            return 1;

        rewind(arg_pSource);
        rewind(arg_pDest);

        if(ftruncate(fileno(arg_pDest), 0))
            return 1;

        while(!feof(arg_pSource)){
                l_iLength = fread(l_sBuffer, 1, MAX_BUFFER_LEN, arg_pSource);
                if(l_iLength > 0){
                        if(fwrite(l_sBuffer, 1, l_iLength, arg_pDest) != l_iLength){
                                return 1;
                        }
                }else{
                        return !feof(arg_pSource);
                }
        }

        DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("OUT iCopyFileContents"));
        return 0;
}

int iDeletePasswordUserEntry(FILE *arg_pTmpFile, const char *arg_pUsername)
{
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("IN "),DLT_STRING(__FUNCTION__));
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Common name is : "),DLT_STRING(arg_pUsername));
    char l_sBuf[MAX_BUFFER_LEN];
    char l_sLocalBuf[MAX_BUFFER_LEN], *l_pToken;
    int l_iFound = 0;

    if(NULL == arg_pTmpFile || NULL == arg_pUsername)
    {
        DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Error: temp/username empty"));
        return l_iFound;
    }

    while(!feof(fptr) && fgets(l_sBuf, MAX_BUFFER_LEN, fptr)){
        memcpy(l_sLocalBuf, l_sBuf, MAX_BUFFER_LEN);
        l_pToken = strtok(l_sLocalBuf, ":");
        if(strcmp(arg_pUsername, l_pToken)){
            fprintf(arg_pTmpFile, "%s", l_sBuf);
        }else{
            l_iFound = 1;
        }
    }
    if(!l_iFound){
        DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Warning: User not found in password file: "),DLT_STRING(arg_pUsername));
    }

    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Common name is : "),DLT_STRING(arg_pUsername));
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("OUT "),DLT_STRING(__FUNCTION__));
    return l_iFound;
}

int iDeletePasswordEntry()
{
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("IN "),DLT_STRING(__FUNCTION__));
    int l_iUsernameFound = 0;

    do
    {
        if(NULL != Common_name)
        {
            DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Common name is : "),DLT_STRING(Common_name));

            //Delete the password entry
            fptr = fopen(PASSWORD_FILE, "r+t");
            g_pTmpFilePtr = tmpfile();

            if(!fptr){
                DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Fopen failed for pwfile"));
                DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Error in fopen (pwfile): "),
                        DLT_STRING(strerror(errno)));
                break;
            }

            l_iUsernameFound = iDeletePasswordUserEntry(g_pTmpFilePtr, Common_name);

            if(l_iUsernameFound)
            {
                if(iCopyFileContents(g_pTmpFilePtr, fptr)){
                    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Error occurred updating password file"));
                }
            }
            else
            {
                DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Error occurred in iDeletePasswordUserEntry"));
            }
        }
    }while(0);

    if(fptr)
    {
        fclose(fptr);
        fptr = NULL;
    }

    if(g_pTmpFilePtr)
    {
        fclose(g_pTmpFilePtr);
        g_pTmpFilePtr = NULL;
    }

    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Common name is : "),DLT_STRING(Common_name));
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("OUT "),DLT_STRING(__FUNCTION__));

    return l_iUsernameFound;
}

/******************************************************************************
** FUNCTION   : vSet_SphoneMQTT_credentials
*******************************************************************************
* \fn     vSet_SphoneMQTT_credentials
* \brief  Function to create/set MQTT credentials and ACL auths
* \param  void
* \retval void.
******************************************************************************/
void vSet_SphoneMQTT_credentials(gchar *arg_strCertPath)
{
    DLT_REGISTER_CONTEXT(JSON,"JSON","JSON message Handler");
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("IN "),DLT_STRING(__FUNCTION__));
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Cert path: "),DLT_STRING(arg_strCertPath));

    gboolean l_bUserNameFound = FALSE;

    gboolean bReadResult =  bReadCertificate(arg_strCertPath);

    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("bReadCertificate result: "),DLT_INT(bReadResult));
    if(!bReadResult)
    {
        DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("bReadCertificate failure"));
        return;
    }

    int iReturn = iGet_CN_FromCert();
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("vGet_CN_FromCert ::iReturn : "),DLT_INT(iReturn));
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Common name is : "),DLT_STRING(Common_name));

    if(!iReturn)
        return;

    //Delete the current CN entry first
    l_bUserNameFound = iDeletePasswordEntry();
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Common name is :: after delete"),DLT_STRING(Common_name));

    vCertHash_cal(arg_strCertPath);
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Final password is: "),DLT_STRING(finalPass));

    fptr = fopen(PASSWORD_FILE, "a+");
    if(!fptr){
        DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Fopen failed for pwfile"));
        DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Error in fopen (pwfile): "),
                DLT_STRING(strerror(errno)));
        return ;
    }

    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("Common name is before writing to file: "),DLT_STRING(Common_name));
    gboolean bResultPassword = iOutput_new_password(fptr, Common_name, finalPass);
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("return value of iOutput_new_password: "),DLT_INT(bResultPassword));

    if(!l_bUserNameFound){
        gboolean bAclupdate = bUpdateACL();
        DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("bAclupdate: "),DLT_INT(bAclupdate));
    }

    //Send SIGHUP to mosquito to reload mosquitto.conf
    gboolean killStatus = iSignal_mosquito();
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("killStatus : "),DLT_INT(killStatus));

    vCleanup();
    DLT_LOG(JSON,DLT_LOG_INFO,DLT_STRING("OUT:"),DLT_STRING(__FUNCTION__));
}

