#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <malloc.h>
#include <glib.h>
#include <gio/gio.h>
#include "qrcode.h"
#include "dlt/dlt.h"
#include "seamless_settings.h"


#define BLACK 0x00000000
#define WHITE 0xFFFFFFFF
#define BITS_PER_PIXEL 32
#define NUM_OF_PLANES 1
#define COMPRESSION_VAL 0
#define PIXELS_PER_METER 0x130B 
#pragma pack(push,1)

DLT_IMPORT_CONTEXT(SPM_SPL);

typedef struct{
    uint32_t bmpheadersize;
    uint32_t width;
    uint32_t height;
    uint16_t planes;
    uint16_t bitsperpixel;
    uint32_t compression;
    uint32_t imagesize;
    uint32_t ypixelpermeter;
    uint32_t xpixelpermeter;
    uint32_t numcolorspallette;
    uint32_t mostimpcolor;
} bmpHeader;

typedef struct{
    uint8_t signature[2];
    uint32_t filesize;
    uint32_t reserved;
    uint32_t fileoffset_to_pixelarray;
} fileHeader;

typedef struct {
    fileHeader fheader;
    bmpHeader  bmp_header;
} bmpFileData;

#pragma pack(pop)

/**********************************************************************
* Function:    prepareQrBmp
* Description: Creates the BMP image data for the QRCode
* Parameters:  QRCode, fileName
* Returns:     TRUE,  if QRCode generation is success
               FALSE, if QRCode generation fails
************************************************************************/  
gboolean  prepareQrBmp(QRCode qrcode, char*fileName)
{
    DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    unsigned int* imageBuffer = NULL;        
    unsigned int* bit_value = NULL;
    unsigned int  image_size = 0;
    unsigned char* imageHeader = NULL; 
    GVariant*      poImageData = NULL;
    GVariantBuilder builder;
    int scalingFactor = 3;
    int repeatImagePortion = 1;

    if(!bGetSPMIntegerSetting("bmpScalingFactor" , &scalingFactor))
    {
        DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
             DLT_STRING("(): Failed to fetch scaliongFactor"));
    }
    image_size = ( sizeof(unsigned int) * (qrcode.size*scalingFactor) 
	                                        * (qrcode.size*scalingFactor));
    bit_value = (unsigned int *)malloc(image_size);
        
	if (bit_value == NULL)
	{
        DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                      DLT_STRING("malloc failed for imageBuffer")); 
		return FALSE;
	}
	memset(bit_value, 0, image_size);
	imageBuffer = bit_value;

	for ( int y = (qrcode.size - 1); y >= 0; y--)
	{
	    for ( int x = 0; x < qrcode.size ; x++)
	    {
                int repeatIndex = 0;
	        if (qrcode_getModule(&qrcode, y, x))
	        {
                     for(repeatIndex = 0; repeatIndex < scalingFactor; 
                                                           repeatIndex++)
                     {
        	          *bit_value = BLACK;
	   	           bit_value += 1;
                     }  
	        }
	        else
	        {
                    for(repeatIndex = 0; repeatIndex < scalingFactor;
                                                          repeatIndex++)
                    {
                          *bit_value = WHITE;
                           bit_value += 1;
                    }
                    
	        }
	    }
            if(repeatImagePortion == scalingFactor)
            {  
                repeatImagePortion = 1;
            }
            else
            {
               repeatImagePortion++;
               y++;
            } 
	}
       
	unsigned int pixelbytesize = image_size *BITS_PER_PIXEL/8;
	unsigned int filesize = pixelbytesize + sizeof(bmpFileData);
	
        bmpFileData *bmpData  = (bmpFileData*)calloc(1,sizeof(bmpFileData));
//        strncpy(bmpData->fheader.signature,"BM", 2);
    char bmpPrefix[] = "BM";
    memcpy(bmpData->fheader.signature, bmpPrefix, 2);
    bmpData->fheader.filesize = filesize;
    bmpData->fheader.fileoffset_to_pixelarray = sizeof(bmpFileData);
    bmpData->bmp_header.bmpheadersize =sizeof(bmpHeader);
    bmpData->bmp_header.width = qrcode.size*scalingFactor;
    bmpData->bmp_header.height = qrcode.size*scalingFactor;
    bmpData->bmp_header.planes = NUM_OF_PLANES;
    bmpData->bmp_header.bitsperpixel = BITS_PER_PIXEL;
    bmpData->bmp_header.compression = COMPRESSION_VAL;
    bmpData->bmp_header.imagesize = (image_size *BITS_PER_PIXEL/8);
    bmpData->bmp_header.ypixelpermeter = PIXELS_PER_METER ;
    bmpData->bmp_header.xpixelpermeter = PIXELS_PER_METER ;
    bmpData->bmp_header.numcolorspallette = 0;
     

    imageHeader = (unsigned  char*) bmpData;

    FILE *fp = fopen(fileName,"wb");
    if(!fp)
    {
        DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                 DLT_STRING("Failed to create /tmp/qrcode"));
        free(bmpData);
        free(imageBuffer);
        return FALSE;
    }
    else
    {
        fwrite (bmpData, 1, sizeof(bmpFileData),fp);
        fwrite(imageBuffer, 1, image_size , fp);
        fclose(fp);

        DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                   DLT_STRING("qrcode image created")); 
    }

    free(bmpData);
    free(imageBuffer);

    DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("-"),
                          DLT_STRING(__FUNCTION__));
    return true;
}

