/**************************************************************************
* FILE:          seamless_aes_sessionID.c
* PROJECT:       Seamless Pairing
* DESCRIPTION:   Seamless pairing session control logic
* AUTHOR:        IPD5KOR
* COPYRIGHT:     (c) 2018 Robert Bosch GmbH, Hildesheim
****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <openssl/aes.h>
#include <openssl/rand.h>
#include "seamless_aes_sessionID.h"
#include "seamless_data.h"
#include "dlt/dlt.h"

DLT_DECLARE_CONTEXT(SPM_SPL);

#define AES_ENCYPTION_TYPE 256
#define AES_INPUT_LENGTH   30

static    unsigned char  g_aes_input[AES_INPUT_LENGTH + 1] = { '\0' }; 
static    unsigned char* g_aes_key = "awisojdninhgsdy62sdmvidfj9delkoi";
static    unsigned char *iv_enc = NULL;

/**************************************************************
* Function:    bByteToHexStrConvert
* Description: This function converts a given string 
               to hex format
* Parameters:  input, length, output
* Returns:     gboolean
***************************************************************/
gboolean bByteToHexStrConvert( unsigned char* byteInput, 
                               size_t         len,
                               char*          hexOutput )
{
    DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("+"), 
                           DLT_STRING(__FUNCTION__));

    if((NULL == byteInput) || (NULL == hexOutput))
    {
        DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): Null input(s)"));
        return FALSE;
    }

    size_t i = 0, j =0;
    char tempStr[3] = { '\0' };
    for(i = 0 ; i < len ; i++)
    {
        sprintf(tempStr,"%02X",byteInput[i]);
        hexOutput[j++] = tempStr[0];
        hexOutput[j++] = tempStr[1];
    }

    DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("Output string"),
                           DLT_STRING(hexOutput));

    DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("-"),
                           DLT_STRING(__FUNCTION__));
    return TRUE;
}

/**************************************************************
* Function:    bHexToByteStrConvert
* Description: This function converts a hex string to byte array
* Parameters:  input, output
* Returns:     gboolean
****************************************************************/
gboolean bHexToByteStrConvert( char*          input,
                               unsigned char* output)
{
    DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("+"),
                           DLT_STRING(__FUNCTION__));

    if((NULL == input) || (NULL == output))
    {
        DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): Null input(s)"));
        return FALSE;
    }
    char *l_strIndex = input;
    size_t i = 0;
    for(i = 0; i < (strlen(input)/2); i++)
    {
       sscanf(l_strIndex, "%2hhx", &output[i]);
       l_strIndex = l_strIndex+2;
    }
    output[SESSION_ID_LENGTH/2] = '\0';

    DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("-"),
                           DLT_STRING(__FUNCTION__));

    return TRUE;
}
                              
/**************************************************************
* Function:    bGenerateSessionID
* Description: This function generates AES256 coded sessionID
* Parameters:  hexSessionID
* Returns:     gboolean
****************************************************************/   
gboolean bGenerateSessionID(char* hexSessionID)
{
    DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("+"),
                           DLT_STRING(__FUNCTION__));

    memset(g_aes_input, '\0', AES_INPUT_LENGTH);
    if(!RAND_bytes(g_aes_input, AES_INPUT_LENGTH))
    {
        DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("(): RAND_bytes failed"));
        return FALSE; 
    }  

    if(iv_enc)
    {
       free(iv_enc); 
       iv_enc = NULL;
    } 
    iv_enc = (unsigned char*)malloc(sizeof(unsigned char) *AES_BLOCK_SIZE);
    if(!iv_enc)
    {
        DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("(): iv_enc NULL"));
        return FALSE;
    }   
    RAND_bytes(iv_enc, AES_BLOCK_SIZE);

     // buffers for encryption and decryption
    const size_t encslength = ((AES_INPUT_LENGTH + AES_BLOCK_SIZE) 
                                   / AES_BLOCK_SIZE) * AES_BLOCK_SIZE;
    if(encslength != 32)
    {
        DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("(): Encryption length error"));
        //encryption length error
        return FALSE;
    }
    unsigned char *enc_out = (unsigned char*)malloc(
                               sizeof(unsigned char) *encslength);
    if(!enc_out)
    {
        DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("(): NULL encryption output"));
        free(iv_enc);
        iv_enc = NULL; 
        return FALSE;
    }

    memset(enc_out, 0, encslength);

    AES_KEY enc_key;
    AES_set_encrypt_key(g_aes_key, AES_ENCYPTION_TYPE, &enc_key);
    AES_cbc_encrypt(g_aes_input, enc_out, AES_INPUT_LENGTH, 
                               &enc_key, iv_enc, AES_ENCRYPT);

    DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("-"),
                           DLT_STRING(__FUNCTION__));

    gboolean bConversionResult= bByteToHexStrConvert(enc_out, encslength, hexSessionID);
    free(enc_out);
    return bConversionResult;
}

/**************************************************************
* Function:    bIsValidSessionID
* Description: This function validates the given sessionID
* Parameters:  sessionID
* Returns:     gboolean
***************************************************************/
gboolean bIsValidSessionID(char* sessionID)
{
    DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("+"),
                           DLT_STRING(__FUNCTION__));

    size_t encslength = 0;
    gboolean result = FALSE;

    if(!iv_enc)
    {
        DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("(): NULL iv_enc"));
        return FALSE;
    }

    unsigned char *dec_out = (unsigned char*)malloc(
	                            sizeof(unsigned char) * AES_INPUT_LENGTH);
    if(!dec_out)
    {
        DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("(): malloc failed"));
        free(iv_enc);
        iv_enc = NULL;
        return FALSE;
    }  
    memset(dec_out, '\0', AES_INPUT_LENGTH);

    unsigned char dec_input[SESSION_ID_LENGTH+1] = { '\0' };
    memset(dec_input, '\0', SESSION_ID_LENGTH+1);

    if(!bHexToByteStrConvert(sessionID, dec_input))
    {
        DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("(): Hex to Bytes conversion failed"));
        free(dec_out);
        free(iv_enc);
        iv_enc = NULL;
        return FALSE;
    }

     // so i can do with this aes-cbc-128 aes-cbc-192 aes-cbc-256
    AES_KEY  dec_key;

    AES_set_decrypt_key( g_aes_key, AES_ENCYPTION_TYPE, &dec_key );
    AES_cbc_encrypt(dec_input, dec_out, encslength,
                          	&dec_key, iv_enc, AES_DECRYPT );
    if(strcmp(g_aes_input, dec_out) == 0)
    {
       DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
           DLT_STRING("(): Session ID match success"));
       result = TRUE;
    }
    else 
    {
       DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("(): Session ID mismatch"));
       result = FALSE;
    }
    free(dec_out);
    free(iv_enc);
    iv_enc = NULL;

    memset(g_aes_input, '\0', AES_INPUT_LENGTH);

    DLT_LOG(SPM_SPL, DLT_LOG_INFO, DLT_STRING("-"),
                          DLT_STRING(__FUNCTION__));

    return result;
}

 
