/****************************************************************************
* FILE          : TrustedDeviceList_Handler.c
* PROJECT       : Seamless Pairing For IPCM
* DESCRIPTION   : TrustedDeviceList logic
* AUTHOR        : RVH5KOR
* COPYRIGHT     : (c) 2018 Robert Bosch GmbH, Hildesheim
****************************************************************************/
#include "TrustedDeviceList_Handler.h"
#include "SeamlessMidw_HMI_DBusHandler.h"
#include "dlt/dlt.h"
#include "mdns_manager.h"
#include <errno.h>
//FOR TESTING PURPOSE
#if(ENABLE_STATIC_STORAGE == 1)
#define DEVICEINFO_FILE "/var/opt/bosch/dynamic/seamlesspairing/TrustedDeviceList"
#else
#define DEVICEINFO_FILE "/tmp/TrustedDeviceList"
#endif

#define MAXIMUM_TRUSTED_DEVICES 20
#define SSID_MAXIMUM_LENGTH     32
#define BSSID_MAXIMUM_LENGTH    18

#define MQTT_PORT           8883
#define MQTT_SERVICE_NAME   "IVI MQTT"
#define MQTT_PROTOCOL       "_mqtt._tcp"


DLT_IMPORT_CONTEXT(SPM_SPL);

//THIS HASH CONTAINS ALL INFO ABOUT TRUSTED DEVICES
static GKeyFile *m_pTrustedDeviceInfoFile = NULL;


/*************************************************************************
 * FUNCTION     : vCreateNewHashTable
 * DESCRIPTION  : This function creates the new GKeyFile object
 * PARAMETER    : void
 * RETURNVALUE  : void
*************************************************************************/
void vCreateNewHashTable()
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("+"),DLT_STRING( __FUNCTION__));
    m_pTrustedDeviceInfoFile = g_key_file_new ();
    DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING( __FUNCTION__));
}

/*************************************************************************
 * FUNCTION     : bBuildDeviceInfoPropertyUpdate
 * DESCRIPTION  : This function builds GBuilders with all devices' info
 *                to pass onto HMI
 * PARAMETER    : void
 * RETURNVALUE  : void
*************************************************************************/
void vBuildDeviceInfoPropertyUpdate()
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING( __FUNCTION__));
    GVariant *l_pDeviceInfoProperty = NULL;
    gchar **l_pGroups = g_key_file_get_groups(m_pTrustedDeviceInfoFile,NULL);
    guint l_iTotalDevices = g_strv_length(l_pGroups);
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("l_iTotalDevices: "),
            DLT_UINT(l_iTotalDevices));

    if(l_iTotalDevices == 0)
    {
        g_strfreev(l_pGroups);
        vUpdateTrustedDeviceList(l_pDeviceInfoProperty);
        return;
    }

    //Check if atleast one device exists
    if(l_iTotalDevices <= MAXIMUM_TRUSTED_DEVICES)
    {
        guint iIndex;
        GVariantBuilder l_gInnerBuilder;
        GVariantBuilder l_gOuterbuilder;

        gchar *l_strSsid;
        gchar *l_strPsk;
        gchar *l_strBssid;
        gchar *l_strName;
        gchar *l_strTimeStamp;
        gchar *l_strSerialNumber;
        g_variant_builder_init (&l_gOuterbuilder, G_VARIANT_TYPE ("aa{sv}"));
        
        //Loop until all devices are added into the builder
        for(iIndex=0;iIndex<l_iTotalDevices;iIndex++)
        {
            g_variant_builder_init (&l_gInnerBuilder, G_VARIANT_TYPE ("a{sv}"));


            l_strSsid = g_key_file_get_string(m_pTrustedDeviceInfoFile,l_pGroups[iIndex],"sp_wifi_ssid",NULL);
            l_strPsk = g_key_file_get_string(m_pTrustedDeviceInfoFile,l_pGroups[iIndex],"sp_wifi_psk",NULL);
            l_strBssid = g_key_file_get_string(m_pTrustedDeviceInfoFile,l_pGroups[iIndex],"sp_wifi_bssid",NULL);
            l_strName = g_key_file_get_string(m_pTrustedDeviceInfoFile,l_pGroups[iIndex],"sp_name",NULL);
            l_strTimeStamp = g_key_file_get_string(m_pTrustedDeviceInfoFile,l_pGroups[iIndex],"sp_pair_time",NULL);
            l_strSerialNumber = g_key_file_get_string(m_pTrustedDeviceInfoFile,l_pGroups[iIndex],"sp_serial_number",NULL);
        
            g_variant_builder_add(&l_gInnerBuilder, "{sv}", "sp_wifi_ssid", g_variant_new_string(l_strSsid));
            g_variant_builder_add(&l_gInnerBuilder, "{sv}", "sp_wifi_psk", g_variant_new_string(l_strPsk));
            g_variant_builder_add(&l_gInnerBuilder, "{sv}", "sp_wifi_bssid",g_variant_new_string(l_strBssid));
            g_variant_builder_add(&l_gInnerBuilder, "{sv}", "sp_name", g_variant_new_string(l_strName));
            g_variant_builder_add(&l_gInnerBuilder, "{sv}", "sp_pair_time", g_variant_new_string(l_strTimeStamp));
            g_variant_builder_add(&l_gInnerBuilder, "{sv}", "sp_serial_number", g_variant_new_string(l_strSerialNumber));

            g_variant_builder_add(&l_gOuterbuilder, "a{sv}", &l_gInnerBuilder );
            g_variant_builder_end(&l_gInnerBuilder);             
        }
        l_pDeviceInfoProperty = g_variant_builder_end(&l_gOuterbuilder);

        //Call to update the device info to HMI
        vUpdateTrustedDeviceList(l_pDeviceInfoProperty);
    }
    else
    {
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING(__FUNCTION__),
             DLT_STRING("(): Device list exceeded maximum"));
        //NO DEVICES IN THE LIST
    }
    g_strfreev(l_pGroups);
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("-"), DLT_STRING( __FUNCTION__));
}

void vFetchSsid(char *arg_strSsid)
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING( __FUNCTION__));
    int l_iNetworkSocketFd = socket(AF_INET, SOCK_DGRAM, 0);
    if(l_iNetworkSocketFd < 0 )
    {
       DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): socket call failed" ));
       return;
    }              
    char l_sBuffer[SSID_MAXIMUM_LENGTH];
    struct iwreq l_sNetworkParamRequest;

    memset(l_sBuffer, 0, SSID_MAXIMUM_LENGTH);
    memset(&l_sNetworkParamRequest, 0, sizeof(l_sNetworkParamRequest));
    strncpy(l_sNetworkParamRequest.ifr_name, "wlan0", sizeof(l_sNetworkParamRequest.ifr_name));
    l_sNetworkParamRequest.u.essid.pointer = l_sBuffer;
    l_sNetworkParamRequest.u.essid.length = SSID_MAXIMUM_LENGTH;

    //this will gather the SSID of the connected network
    if(ioctl(l_iNetworkSocketFd, SIOCGIWESSID, &l_sNetworkParamRequest) == -1){
        //die with error, invalid interface
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("Error in IOCTL"));
        close(l_iNetworkSocketFd);
        return;
    }
    else{
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("SSID: "),
                DLT_STRING(l_sNetworkParamRequest.u.essid.pointer));
        strncpy(arg_strSsid, l_sNetworkParamRequest.u.essid.pointer, l_sNetworkParamRequest.u.essid.length);
        arg_strSsid[l_sNetworkParamRequest.u.essid.length] = '\0';
    }
    close(l_iNetworkSocketFd);
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("SSID after assignment: "),
            DLT_STRING(arg_strSsid));
}

void vCheckTDLForBSSID(char *arg_strBssid)
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING( __FUNCTION__));

    gboolean l_bMDNSRegisterResult = FALSE;
    if(g_key_file_has_group(m_pTrustedDeviceInfoFile, arg_strBssid))
    {
        //BSSID exists in TDL
        /***********Call MDNSRegister HERE************/
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("bIsMQTTServiceRegistered "),
                DLT_INT(bIsMQTTServiceRegistered));
        if(!bIsMQTTServiceRegistered)
        {
            l_bMDNSRegisterResult = MDNSRegister(MQTT_SERVICE_NAME,MQTT_PROTOCOL,MQTT_PORT);
            DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("l_bMDNSRegisterResult "),
                    DLT_INT(l_bMDNSRegisterResult));
            if(l_bMDNSRegisterResult)
                bIsMQTTServiceRegistered = TRUE;
        }

        char *l_strSsid = malloc(SSID_MAXIMUM_LENGTH * sizeof(char));

        if(l_strSsid == NULL)
            return;

        vFetchSsid(l_strSsid);

        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("SSID after return: "),
                DLT_STRING(l_strSsid));

        if(l_strSsid != NULL)
        {
            if(!strcmp(l_strSsid,g_key_file_get_value(m_pTrustedDeviceInfoFile,
                                                      arg_strBssid,
                                                      "sp_wifi_ssid",
                                                      NULL)))
            {
                //SSID already exists in the TDL
                DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("+"),
                        DLT_STRING("SSID already exists in TDL"));
            }
            else
            {
                //NEW SSID. Update TDL and property update
                //SSID already exists in the TDL
                DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("+"),
                        DLT_STRING("SSID is different in TDL"));
                gboolean l_bResult;
                g_key_file_set_value(m_pTrustedDeviceInfoFile,
                                     arg_strBssid,
                                     "sp_wifi_ssid",
                                     l_strSsid);
                free(l_strSsid);
                l_bResult = g_key_file_save_to_file(m_pTrustedDeviceInfoFile,
                                                    DEVICEINFO_FILE,
                                                    NULL);
                vBuildDeviceInfoPropertyUpdate();

            }
        }
        else
        {
            DLT_LOG(SPM_SPL,DLT_LOG_ERROR, DLT_STRING("+"),
                    DLT_STRING("SSID is NULL"));
        }
    }
    else
    {
        //BSSID does not exist in TDL
        DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("+"),
                DLT_STRING("BSSID does not exist in TDL"));
    }
    DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING( __FUNCTION__));
}

void vFetchBssid()
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING( __FUNCTION__));
    int l_iNetworkSocketFd = socket(AF_INET, SOCK_DGRAM, 0);
    if(l_iNetworkSocketFd < 0)
    {
        DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
              DLT_STRING("(): socket call failed" ));
        return;
    }
    struct iwreq l_sNetworkParamRequest;
    strcpy(l_sNetworkParamRequest.ifr_name, "wlan0");

    char l_strBssidBuffer[BSSID_MAXIMUM_LENGTH] = {0};
    //this will get the bitrate of the link
    if(ioctl(l_iNetworkSocketFd, SIOCGIWAP, &l_sNetworkParamRequest) == -1)
    {
        DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING(
                    "IOCTL failed"));
        close(l_iNetworkSocketFd);
        return;
    }

    //Convert Hex values to a proper string
    size_t i = 0, j =0;
    char tmpStr[3] = { '\0' };
    for(i = 0 ; i < 6 ; i++)
    {
        sprintf(tmpStr,"%02X",(unsigned char)l_sNetworkParamRequest.u.ap_addr.sa_data[i]);
        l_strBssidBuffer[j++] = tmpStr[0];
        l_strBssidBuffer[j++] = tmpStr[1];
        l_strBssidBuffer[j++] = ':';
    }
    l_strBssidBuffer[--j]='\0';

    vCheckTDLForBSSID(l_strBssidBuffer);

    close(l_iNetworkSocketFd);
    DLT_LOG(SPM_SPL,DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING( __FUNCTION__));
}

/*************************************************************************
 * FUNCTION     : vDestroyHashTable
 * DESCRIPTION  : This function destroys the existing GKeyFile object
 * PARAMETER    : void
 * RETURNVALUE  : void
*************************************************************************/
void vDestroyHashTable()
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));

    //Destroys the GKeyFile object and associated pointer
    g_key_file_free(m_pTrustedDeviceInfoFile);
	m_pTrustedDeviceInfoFile = NULL;
}

/*************************************************************************
 * FUNCTION     : bLoadTrustedDeviceInfo
 * DESCRIPTION  : This function loads all the existing device info from
 *                the file in persistent storage into the GKeyFile object
 * PARAMETER    : void
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bLoadTrustedDeviceInfo()
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));

    bIsMQTTServiceRegistered = FALSE;

    if(m_pTrustedDeviceInfoFile == NULL)
    {
        vCreateNewHashTable();
    }

    //THIS CODE SNIPPET IS TO MAKE SURE THAT THE FILE EXISTS AT THE STARTUP
    FILE * l_pTrustedDeviceInfoFile = NULL;
    DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING("TDL Filename: "),
          DLT_STRING(DEVICEINFO_FILE));
    l_pTrustedDeviceInfoFile = fopen (DEVICEINFO_FILE,"a");
    if(l_pTrustedDeviceInfoFile)
    {
       fclose(l_pTrustedDeviceInfoFile);
	   l_pTrustedDeviceInfoFile = NULL; 
           DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING( __FUNCTION__),
                 DLT_STRING("No error in TDL creation"));
    }
    else
    {
        DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING( __FUNCTION__),
              DLT_STRING("(): Unable to access trusted device info file"));
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("Error in fopen (TDL): "),
                DLT_STRING(strerror(errno)));
        return FALSE;    
    }   
    //THIS CODE SNIPPET IS TO MAKE SURE THAT THE FILE EXISTS AT THE STARTUP

    gboolean l_bResult = g_key_file_load_from_file(m_pTrustedDeviceInfoFile,DEVICEINFO_FILE,G_KEY_FILE_NONE,NULL);
    if(!l_bResult)
    {
        //FAILURE IN READING DEVICE INFO FROM THE FILE
	DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING("bLoadTrustedDeviceInfo: Error in loading device info"));
        return l_bResult; 
    }
    else
    {
        //SUCCESS IN READING DEVICE INFO FROM THE FILE
		DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING("bLoadTrustedDeviceInfo: Loading device info success"));
    }

    //BUILD GVARIANT OBJECT TO UPDATE PROPERTY
    vBuildDeviceInfoPropertyUpdate();

    return l_bResult;
}

/*************************************************************************
 * FUNCTION     : bInsertDeviceInfo
 * DESCRIPTION  : This function inserts the new device info into the hash
 *                and the file in persistent storage
 * PARAMETER    : arg_DeviceInfo
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bInsertDeviceInfo(TrustedDeviceInfo *arg_DeviceInfo)
{
    GError *error = NULL;
    if(m_pTrustedDeviceInfoFile == NULL)
    {
        vCreateNewHashTable();
    }
    else
    {
        DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("Key file NOT empty"));
    }
	
	gboolean l_bResult = FALSE;
	gchar **l_pGroups = g_key_file_get_groups(m_pTrustedDeviceInfoFile,NULL);
    guint l_iTotalDevices = g_strv_length(l_pGroups);
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("l_iTotalDevices: "),
            DLT_UINT(l_iTotalDevices));

    //Check if atleast one device exists
    if(l_iTotalDevices < MAXIMUM_TRUSTED_DEVICES)
	{
		//These calls add all the info about a new device into GKeyFile object
		g_key_file_set_value(m_pTrustedDeviceInfoFile, arg_DeviceInfo->sBssid,"sp_wifi_ssid",arg_DeviceInfo->sSsid);
		g_key_file_set_value(m_pTrustedDeviceInfoFile, arg_DeviceInfo->sBssid,"sp_wifi_psk",arg_DeviceInfo->sPsk);
		g_key_file_set_value(m_pTrustedDeviceInfoFile, arg_DeviceInfo->sBssid,"sp_wifi_bssid",arg_DeviceInfo->sBssid);
                g_key_file_set_value(m_pTrustedDeviceInfoFile, arg_DeviceInfo->sBssid,"sp_name",arg_DeviceInfo->sName);
		g_key_file_set_value(m_pTrustedDeviceInfoFile, arg_DeviceInfo->sBssid,"sp_pair_time",arg_DeviceInfo->sPairTime);
                g_key_file_set_value(m_pTrustedDeviceInfoFile, arg_DeviceInfo->sBssid,"sp_serial_number",arg_DeviceInfo->sSerialNumber);
                
		DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
			 DLT_STRING("New values set"));

		//This call is to write the device info from GKeyFile object
		//to the file after addition of new device
                DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING("TDL Filename: "),
                      DLT_STRING(DEVICEINFO_FILE));
                l_bResult = g_key_file_save_to_file(m_pTrustedDeviceInfoFile, DEVICEINFO_FILE, &error);
                DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING("TDL Filename: "),
                      DLT_STRING(DEVICEINFO_FILE));
		if(!l_bResult)
		{
			//ERROR IN SAVING INFO TO FILE
			DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
				 DLT_STRING("Error in saving to the file"));
                        DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING("Error :")
                                ,DLT_STRING(error->message));
                        DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING("TDL Filename: "),
                              DLT_STRING(DEVICEINFO_FILE));
		}
		else
		{
			//SUCCESS IN SAVING INFO TO FILE
			DLT_LOG(SPM_SPL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
				 DLT_STRING("NO Error in saving to the file"));
                        DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING("TDL Filename: "),
                              DLT_STRING(DEVICEINFO_FILE));
		}

		//UPDATE GVARIANT OBJECT TO UPDATE PROPERTY
		vBuildDeviceInfoPropertyUpdate();
	}
	else
	{
		DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING("bInsertDeviceInfo: ERROR since list has reached maximum limit"));
	}

	g_strfreev(l_pGroups);
    return l_bResult;
}

/*************************************************************************
 * FUNCTION     : bRemoveDeviceInfo
 * DESCRIPTION  : This function removes the requested device info from
 *                the hash and the file in persistent storage
 * PARAMETER    : arg_DeviceInfo
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bRemoveDeviceInfo(TrustedDeviceInfo *arg_DeviceInfo)
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));

	if(m_pTrustedDeviceInfoFile == NULL)
    {
		DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING("bRemoveAllDeviceInfo: Key file is empty. Nothing to remove"));
		return FALSE;
    }
	
    //This call removes all the info about one device (Identified by its BSSID)
    gboolean l_bResult = g_key_file_remove_group(m_pTrustedDeviceInfoFile,arg_DeviceInfo->sBssid,NULL);

    //This call is to write the device info to the file after device removal from the list
    if(l_bResult)
	{
		l_bResult = g_key_file_save_to_file(m_pTrustedDeviceInfoFile, DEVICEINFO_FILE, NULL);
		//UPDATE GVARIANT OBJECT TO UPDATE PROPERTY
		vBuildDeviceInfoPropertyUpdate();
	}
	else
	{
		DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING("bRemoveAllDeviceInfo: Device does not exist in the list. Hence NOT removed"));
	}
    
	return l_bResult;
}

/*************************************************************************
 * FUNCTION     : bRemoveAllDeviceInfo
 * DESCRIPTION  : This function removes device info of all devices from
 *                the hash and the file in persistent storage
 * PARAMETER    : None
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bRemoveAllDeviceInfo()
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));

	gboolean l_bResult = FALSE;
	if(m_pTrustedDeviceInfoFile == NULL)
    {
		DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING("bRemoveAllDeviceInfo: Key file is empty. Nothing to remove"));
		return l_bResult;
    }
	
	vDestroyHashTable();
	
	if(m_pTrustedDeviceInfoFile == NULL)
	{
		vCreateNewHashTable();
		
		//THIS CODE SNIPPET IS TO MAKE SURE THAT THE FILE CONTENT IS DESTROYED
		FILE * l_pTrustedDeviceInfoFile = NULL;
		l_pTrustedDeviceInfoFile = fopen (DEVICEINFO_FILE,"w");
		if(l_pTrustedDeviceInfoFile)
		{	 
		   fclose(l_pTrustedDeviceInfoFile);
		   l_pTrustedDeviceInfoFile = NULL;
        }
        else      
        {
		   DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING( __FUNCTION__),
		                 DLT_STRING("(): fopen failed"));
           return FALSE;
		} 		   
		//THIS CODE SNIPPET IS TO MAKE SURE THAT THE FILE CONTENT IS DESTROYED
		
		l_bResult = g_key_file_load_from_file(m_pTrustedDeviceInfoFile,DEVICEINFO_FILE,G_KEY_FILE_NONE,NULL);
		if(!l_bResult)
		{
			//FAILURE IN READING DEVICE INFO FROM THE FILE
			DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING("bRemoveAllDeviceInfo: Error in loading device info"));
                        return l_bResult;
		}
		else
		{
			//SUCCESS IN READING DEVICE INFO FROM THE FILE
			DLT_LOG(SPM_SPL,DLT_LOG_ERROR,DLT_STRING("bRemoveAllDeviceInfo: Loading device info success"));
			//UPDATE GVARIANT OBJECT TO UPDATE PROPERTY
			vBuildDeviceInfoPropertyUpdate();
		}
	}

    return l_bResult;
}

/*************************************************************************
 * FUNCTION     : bUpdateTrustedDeviceListInfo
 * DESCRIPTION  : This function updates Trusted devices' info
 * PARAMETER    : arg_TrustedDeviceInfo, arg_eAction
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bUpdateTrustedDeviceListInfo(GVariant *arg_TrustedDeviceInfo, SeamlessPairingAction arg_eAction)
{
    DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING( __FUNCTION__));

    //Parse the GVariant and extract the device info
    TrustedDeviceInfo l_deviceInfo = {NULL,NULL,NULL,NULL};
    GVariantIter l_poIter;
    gboolean l_bResult = FALSE;

    if(arg_eAction == REMOVE_ALL)
    {
       return bRemoveAllDeviceInfo();
    }

    const gchar* chVal = g_variant_get_type_string(arg_TrustedDeviceInfo);

    if (!(g_strcmp0(chVal, "a{sv}")))
    {
        GVariant *value;
        gchar *key;

        g_variant_iter_init (&l_poIter, arg_TrustedDeviceInfo);

        while (g_variant_iter_next (&l_poIter, "{sv}", &key, &value))
        {
            gsize l_len = 0;
            
            if(!g_strcmp0 (key, "sp_wifi_ssid"))
            {
                l_deviceInfo.sSsid = g_variant_get_string(value, &l_len);
                DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("SSID: "),
                        DLT_STRING(l_deviceInfo.sSsid));
            }
            else if(!g_strcmp0 (key, "sp_wifi_psk"))
            {
                l_deviceInfo.sPsk = g_variant_get_string(value, &l_len);
                DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("PSK: "),
                        DLT_STRING(l_deviceInfo.sPsk));
            }
            else if(!g_strcmp0 (key, "sp_wifi_bssid"))
            {
                l_deviceInfo.sBssid =  g_variant_get_string(value, &l_len);
                DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("BSSID: "),
                        DLT_STRING(l_deviceInfo.sBssid));
            }
            else if(!g_strcmp0 (key, "sp_name"))
            {
                l_deviceInfo.sName = g_variant_get_string(value, &l_len);
                DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("NAME: "),
                        DLT_STRING(l_deviceInfo.sName));
            }
            else if(!g_strcmp0 (key, "sp_pair_time"))
            {
                l_deviceInfo.sPairTime = g_variant_get_string(value, &l_len);
                DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("PAIR TIME: "),
                        DLT_STRING(l_deviceInfo.sPairTime));
            }
            else if(!g_strcmp0 (key, "sp_serial_number"))
            {
                l_deviceInfo.sSerialNumber = g_variant_get_string(value, &l_len);
                DLT_LOG(SPM_SPL,DLT_LOG_INFO,DLT_STRING("sSerialNumber "),
                        DLT_STRING(l_deviceInfo.sSerialNumber));
            }
            else
            {
                //INVALID INPUT
            }
        }
    }

    switch(arg_eAction)
    {
    case ADD_DEVICE:{
        //Call method to Insert the new device info to the Hash Table
        l_bResult = bInsertDeviceInfo(&l_deviceInfo);
		break;
    }
    case REMOVE_DEVICE:{
        //Call method to remove the device from Hash table
        l_bResult = bRemoveDeviceInfo(&l_deviceInfo);
	break;
    }
    case UPDATE_DEVICE:{
        //Call insert method to update the device info list
        l_bResult = bInsertDeviceInfo(&l_deviceInfo);
		break;
    }
    case REMOVE_ALL:{
        //Call insert method to update the device info list
        l_bResult = bRemoveAllDeviceInfo(&l_deviceInfo);
                break;
    }
    default:{
        //INVALID INPUT
    }
    }

    return l_bResult;
}

