/*!
*******************************************************************************
* @file             : seamless_controller.h
*******************************************************************************
*  - PROJECT:       : SEAMLESS PAIRING
*  - SW-COMPONENT   : FC_SEAMLESSPAIRING_IPCM
*  - DESCRIPTION    : Header for seamless controller
*  - COPYRIGHT      : &copy; 2017 Robert Bosch Engineering & Business Solutions
*  - Documents      :
*  - HISTORY
*
*  Date     | Name          |  Version | Modification
* ----------|---------------|--------------------------|-------------------------
* 24.01.2017  | IPD5KOR (RBEI/ECO2) | 2.1.0 | Seamless controller methods
******************************************************************************/

    #ifndef _SEAMLESS_PAIRING_CONTROLLER_H_
    #define _SEAMLESS_PAIRING_CONTROLLER_H_

    #include <glib.h>
    #include <gio/gio.h>
    #include "SeamlessMidw_HMI_enums.h"
    #include "Seamless_WBL_DBusHandler.h"
    #include <openssl/evp.h>
    #include <openssl/bio.h>
    #include <openssl/pem.h>
    #include <openssl/x509.h>
	

    /****************************************************************************
    * Function:    vStartSeamlessPairing
    * Description: This method intitiates the seamless pairing mechanism
    * Parameters:  void
    * Return:      void
    ***************************************************************************/
    void vStartSeamlessPairing();

    /************************************************************************
    * Function:    bIsPairingInProgress
    * Description: This method tells whether pairing
                   is in progress or idle
    * Parameters:  void
    * Return:      gboolean
    *************************************************************************/
    gboolean bIsPairingInProgress();

    /****************************************************************************
    * Function:    bGenerateWPAKey
    * Description: This method generates the wpa key for wifi AP
    * Parameters:  void
    * Return:      void
    ***************************************************************************/
    gboolean bGenerateWPAKey();

   /***************************************************************************
    * Function:    bGenerateSSID
    * Description: This method generates SSID of length 32
    * Parameters:  void
    * Return:      gboolean
    ***************************************************************************/
    gboolean bGenerateSSID();

    /*************************************************************************
    * Function:    bValidateTempWiFiSetup
    * Description: This function validates the temporary
                   wifi AP setup
    * Parameters:  ssid, wpa_key
    * Returns:     gboolean
    *************************************************************************/
    gboolean bValidateTempWiFiSetup( char* ssid,
                                    char* wpa_key );

    /*************************************************************************
    * Function:    bGenerateQRCode
    * Description: This function generates the QR code with
                   the current seamless session data
    * Parameters:  void
    * Returns:     gboolean
    *************************************************************************/
    gboolean bGenerateQRCode();

    /***********************************************************************
    * Function:    vHandleActiveWiFiUpdate
    * Description: This method generates SSID of length 32
    * Parameters:  ssid, wpa_key, WiFiAPState
    * Return:      None
    *************************************************************************/
    void vHandleActiveWiFiUpdate( gchar* ssid,
                                  gchar* wpa_key,
                                  WiFiAPState wifiAPMode );

   /***************************************************************************
   * Function:    vResetData
   * Description: This method initializes the seamless pairing
                  data items
   * Parameters:  void
   * Returns:     void
   ****************************************************************************/
   void vResetData();

   /**************************************************************************
    * Function:    vCloseSeamlessSession
    * Description: This function terminates the current
                   seamless pairing session
    * Parameters:  SPDeviceUpdateStatus, SPDeviceUpdateErrCode
    * Returns:     void
    *************************************************************************/
    void vCloseSeamlessSession( SPDeviceUpdateStatus  status,
                                 SPDeviceUpdateErrCode errCode );
      
    /*************************************************************************
    * Function:    vStartSeamlessTimer
    * Description: This function starts the seamless pairing timer
    * Parameters:  void
    * Returns:     void
    *************************************************************************/
    void vStartSeamlessTimer();

    /*************************************************************************
    * Function:    bSeamlessTimerExpiredCB
    * Description: Terminates the seamless pairing sesison
    * Parameters:  gpointer
    * Return:      gboolean
    **************************************************************************/
    gboolean bSeamlessTimerExpiredCB(gpointer data);

    /**********************************************************************
    * Function:    vUpdatePairTime
    * Description: This function updates the device pair time in
                   format 'yyyy:mm:dd hh:mm:ss'
    * Parameters:  void
    * Returns:     void
    ***********************************************************************/
    void vUpdatePairTime();

    /**********************************************************************
    * Function:    vUpdateCertificate_Serialnumber
    * Description: This function updates smart phone's certificate serialNumber
    * Parameters:  void
    * Returns:     void
    ***********************************************************************/
    void vUpdateCertificateSerialnumber();
    /***********************************************************************
    * Function:    pGetCurrentDeviceInfo
    * Description: This function provides the info of
                   current device (smart phone)
    * Parameters:  void
    * Returns:     GVariant*
    ************************************************************************/
    GVariant* pGetCurrentDeviceInfo();

    /**************************************************************************
    * Function:    pGetServerConnInfo
    * Description: This function provides the info
                    for server connection
    * Parameters:  void
    * Returns:     GVariant*
    *************************************************************************/
    GVariant*  pGetServerConnInfo();

    /*************************************************************************
    * Function:    vProcessAppMsg
    * Description: This function proccesses the Msg data sent by SP App
    * Parameters:  key, value
    * Returns:     void
    *************************************************************************/
    void vProcessAppMsg(gchar* key, GVariant *value);

    /************************************************************************
    * Function:    vUpdateCurrentDeviceInfo
    * Description: This function updates the current device info
    * Parameters:  key, value
    * Returns:     void
    *************************************************************************/
      void vUpdateCurrentDeviceInfo( gchar*  key,
                                     gchar* value);

    /************************************************************************
    * Function:    bUpdateTrustedDevice
    * Description: This function requests for trusted device list update
    * Parameters:  void
    * Returns:     gboolean
    *************************************************************************/
    gboolean bUpdateTrustedDevice();

    /************************************************************************
    * Function:    bStoreTrustedDeviceCert
    * Description: This method stores the Trusted device cert
                   on the root file system
    * Parameters:  void
    * Returns:     void
    *************************************************************************/
    gboolean bStoreTrustedDeviceCert();

    /************************************************************************
    * Function:    bRehashTrustedCert
    * Description: This method rehashes the Trusted device cert
                   on the stunnel cert verification path
    * Parameters:  certFileName, bssid
    * Returns:     void
    *************************************************************************/
    gboolean bRehashTrustedCert( gchar* certFileName,
                                 gchar* bssid );

    /***********************************************************************
    * Function:    bGetTrustedCertFileName
    * Description: This method gives the file name for the
                   trusted smartphone certificate
    * Parameters:  bssid, certFileName
    * Returns:     gboolean
    ************************************************************************/
    gboolean bGetTrustedCertFileName( gchar* bssid,
                                      gchar* certFileName );

    /************************************************************************
    * Function:    bGetRehashCertFileName
    * Description: This method gives the file name for the
                   rehash certificate
    * Parameters:  Original cert file name, hash file name
    * Returns:     gboolean
    *************************************************************************/
    gboolean bGetRehashCertFileName( gchar* certFileName, 
                                     gchar* hashFileName );
 
    /*********************************************************************
    * Function:    bCheckTrustedCertExistence
    * Description: This method checks if current cert is
                   already available in the trusted list
    * Parameters:  void
    * Returns:     gboolean
                   TRUE, if cert already exists or some failure
                   FALSE, if cert doesn`t exist
    ***********************************************************************/
    gboolean bCheckTrustedCertExistence();  

    /***********************************************************************
    * Function:    bRemoveTrustedDevice
    * Description: This method removes a device from trusted list
    * Parameters:  deviceInfo
    * Returns:     gboolean
    ************************************************************************/
    gboolean bRemoveTrustedDevice(GVariant* deviceInfo);

    /************************************************************************
    * Function:    bRemoveAllTrustedDevices
    * Description: This method deletes all the
                   devices in the trusted list
    * Parameters:  TrustedDeviceList
    * Returns:     gboolean
    *************************************************************************/
    gboolean bRemoveAllTrustedDevices(GVariant* trustedList);

    /************************************************************************
    * Function:    bRemoveTrustedCert
    * Description: This method deletes the cert of
                   a given trusted device (bssid)
    * Parameters:  bssid
    * Returns:     gboolean
    *************************************************************************/
    gboolean bRemoveTrustedCert(const char* bssid);

    /***************************************************************************
    * Function:    bFetchCertInfo
    * Description: This method fetches the certificate details
                   for QR code generation
    * Parameters:  void
    * Return:      gboolean
    ****************************************************************************/
    gboolean bFetchCertInfo();

    /**********************************************************************
    * Function:    vSendBTInfoAndAckToApp
    * Description: This function updates sends the success acknowledgement
                   and the Bluetooth pairing info to the companion App
    * Parameters:  void
    * Returns:     void
    ***********************************************************************/
    void vSendBTInfoAndAckToApp();
    
   /**********************************************************************
   * Function:    vUpdateSPConnectedStatus
   * Description: This function updates the SP Connected status
   * Parameters:  status
   * Returns:     void
   ***********************************************************************/
   void vUpdateSPConnectedStatus(gboolean status);
   
   /*************************************************************************
   * Function:    vFinalizePairingSession
   * Description: This function finalize the seamless pairing
                  session status and triggers HMI update
                  seamless pairing session
   * Parameters:  void
   * Returns:     void
   **************************************************************************/
   void vFinalizePairingSession();
 
   /**********************************************************************
   * Function:    bProcessSignedSessionID
   * Description: This function is for processing Session ID
   * Parameters:  unsigned char *arg_signedSessionID, unsigned char *arg_pSPCert
   * Returns:     gboolean
   ***********************************************************************/
   gboolean bProcessSignedSessionID( char *arg_signedSessionID, 
                                     unsigned char *arg_pSPCert,
  				                     size_t  certLength );

#endif
