/****************************************************************************
* FILE          : certificateHandler.h
* PROJECT       : Seamless Pairing For IPCM
* DESCRIPTION   : Certificate Data handling logic
* AUTHOR        : RVH5KOR
* COPYRIGHT     : (c) 2018 Robert Bosch GmbH, Hildesheim
****************************************************************************/

#ifndef _CERTIFICATEHANDLER_H_
#define _CERTIFICATEHANDLER_H_

#include <openssl/bio.h>
#include <openssl/err.h>
#include <openssl/pem.h>
#include <openssl/x509.h>
#include <stdio.h>
#include <string.h>
#include <glib.h>

#if !defined (_MSC_VER)
#include "adit-components/sdc_session.h"
#include "adit-components/sdc_keystore_keys.h"
#endif

#define FAILURE                     0
#define CERT_BEGIN_TRUNCATION_INDEX 26
#define HASH_SIZE                   32
#define KEY_LENGTH                  451
#define CERT_END_TRUNCATION_INDEX   392

/*************************************************************************
 * FUNCTION     : bExtractCertInfo
 * DESCRIPTION  : This function extracts certificates' info
 * PARAMETER    : void
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bExtractCertInfo();

/*************************************************************************
 * FUNCTION     : bComputeHash
 * DESCRIPTION  : This function computes the hashes
 * PARAMETER    : unsigned char *, unsigned char *
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bComputeHash(unsigned char *arg_sPubKey,
                      unsigned char *arg_sIssuerName);


/*************************************************************************
 * FUNCTION     : bUnwrapCert
 * DESCRIPTION  : This function unwraps the wrapped certificate
 * PARAMETER    : void
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bUnwrapCert();

/*************************************************************************
 * FUNCTION     : iReadFromWrappedFile
 * DESCRIPTION  : This function reads the wrapped certificate
 * PARAMETER    : FILE *, unsigned char **
 * RETURNVALUE  : int
*************************************************************************/
int iReadFromWrappedFile(FILE *arg_pDeviceCertFilePointer,
                         unsigned char **arg_pWrappedCert);

/*************************************************************************
 * FUNCTION     : bCheckPEM
 * DESCRIPTION  : This function checks the unwrapped cert for PEM format
 * PARAMETER    : unsigned char **
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bCheckPEM(unsigned char **arg_pCertificate);

/*************************************************************************
 * FUNCTION     : bFreeUpCertMemory
 * DESCRIPTION  : This function frees up memory used by Cert data
 * PARAMETER    : void
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bFreeUpCertMemory();

/*************************************************************************
 * FUNCTION     : bGetCertificateHash
 * DESCRIPTION  : This is a Get function for Cert hash
 * PARAMETER    : unsigned char *pCertHash
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bGetCertificateHash(unsigned char *arg_pCertHash);

/*************************************************************************
 * FUNCTION     : bGetKeyHash
 * DESCRIPTION  : This is a Get function for Key hash
 * PARAMETER    : unsigned char *pKeyHash
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bGetKeyHash(unsigned char *arg_pKeyHash);

/*************************************************************************
 * FUNCTION     : bGetIssuerNameHash
 * DESCRIPTION  : This is a Get function for Cert hash
 * PARAMETER    : unsigned char *pIssuerNameHash
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bGetIssuerNameHash(unsigned char *arg_pIssuerNameHash);

/*************************************************************************
 * FUNCTION     : bGetSerialNumber
 * DESCRIPTION  : This is a Get function for Cert hash
 * PARAMETER    : unsigned char *pSerialNumber, int *serialNumberLength
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bGetSerialNumber(unsigned char *arg_pSerialNumber,
                          int *arg_serialNumberLength);

/*************************************************************************
 * FUNCTION     : bCalculateMessageDigest
 * DESCRIPTION  : This method calculates the hash of input message
 * PARAMETER    : unsigned char *arg_pCertHash
 * RETURNVALUE  : gboolean
*************************************************************************/
gboolean bCalculateMessageDigest(unsigned char *arg_sMessage,
                                 int arg_iMessageLen,
                                 unsigned char **arg_sDigest,
                                 unsigned int *arg_iDigestLen);

/*************************************************************************
 * FUNCTION     : vCorrectPublicKeyString
 * DESCRIPTION  : This method corrects the public key string
 * PARAMETER    : unsigned char *arg_sPublicKey
 * RETURNVALUE  : void
*************************************************************************/
void vCorrectPublicKeyString(unsigned char *arg_sPublicKey);

#endif
