#include <glib.h>
#include <gio/gio.h>
#include <string.h>
#include "dlt/dlt.h"
#include "Seamless_WBL_DBusHandler.h"
#include "seamless_controller.h"
#include "seamless_data.h"

DLT_DECLARE_CONTEXT(SPM_WBL);
 
#define WBL_CONFLICT_MGMT_OBJ_PATH     "/org/bosch/wbl/conflictmanagement"
#define WBL_CONFLICT_MGMT_IFACE        "org.bosch.wbl.conflictmanagement"
#define WBL_CONFLICT_MGMT_DBUS_NAME    "org.bosch.Wbl" 
#define WBL_DEFAULT_SEAMLESS_FREQUENCY "2.4 GHz"

static GDBusProxy*  pWBLClientProxyObj    = NULL;
static guint        seamlessWBLWatcherdID = 0;

/*************************************************************************
* Function:    bWatchWBLService
* Description: This methods intiates the watch for WBL interface
* Parameters:  void
* Return:      gboolean
***************************************************************************/
gboolean bWatchWBLService()
{
    DLT_REGISTER_CONTEXT(SPM_WBL,"SMWB","Gateway WBL"
                         "context for DLT Logging");

    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));

    seamlessWBLWatcherdID = g_bus_watch_name(
                                     G_BUS_TYPE_SYSTEM,
                                     WBL_CONFLICT_MGMT_DBUS_NAME,
                                     G_BUS_NAME_WATCHER_FLAGS_NONE,
                                     vWBLServiceAppearedCB,
                                     vWBLServiceVanishedCB,
                                     NULL,
                                     NULL );

    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
    return ( seamlessWBLWatcherdID ? TRUE : FALSE );
}

/**************************************************************************
* Function:    vUnwacthWBLService
* Description: This methods stops watcing for WBL interface
* Parameters:  void
* Return:      void
***************************************************************************/
void  vUnwacthWBLService()
{
    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));

    if(seamlessWBLWatcherdID)
    {
       g_bus_unwatch_name(seamlessWBLWatcherdID);
       seamlessWBLWatcherdID = 0;
    }
    if(pWBLClientProxyObj)
    {
       g_clear_object(&pWBLClientProxyObj);
    }
     
    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}

/***************************************************************************
* Function:    vWBLServiceAppearedCB
* Description: service appeared call back for WBL service
* Parameters:  connection , service_name , owner_name, userdata
* Return:      void
****************************************************************************/
void vWBLServiceAppearedCB( GDBusConnection *connection,
                            const gchar *name,
                            const gchar *name_owner,
                            gpointer user_data )
{
    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("+"),
                            DLT_STRING(__FUNCTION__));
    if(connection)
    { 
         g_dbus_proxy_new ( connection,
                            G_DBUS_PROXY_FLAGS_NONE,
                            NULL,
                            WBL_CONFLICT_MGMT_DBUS_NAME,
                            WBL_CONFLICT_MGMT_OBJ_PATH,
                            WBL_CONFLICT_MGMT_IFACE,
                            NULL,
                            vWBLClientProxyCB,
                            NULL );
    }
    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("-"),
                         DLT_STRING(__FUNCTION__));
} 

/**************************************************************************
* Function:    vWBLServiceVanishedCB
* Description: Name disappeared call back for wbl service
* Parameters:  connection, service_name, user_data
* Return:      void
***************************************************************************/
void  vWBLServiceVanishedCB( GDBusConnection *connection,
                             const gchar     *service_name,
                             gpointer        user_data )
{
    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    if(pWBLClientProxyObj)
    {
        g_clear_object(&pWBLClientProxyObj);
    }
    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}

/******************************************************************
* Function:     vWBLClientProxyCB
* Description:  This is call back function for WBL 
                proxy object creation
* Parameters:   source_object, res, user_data
* Return:       void
 ******************************************************************/
void vWBLClientProxyCB( GObject *source_object,
                        GAsyncResult *res,
                        gpointer user_data )
{
    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    GError *error = NULL;
    pWBLClientProxyObj = g_dbus_proxy_new_finish (res, &error);

    if(pWBLClientProxyObj)
    {
        DLT_LOG(SPM_WBL, DLT_LOG_INFO,DLT_STRING(__FUNCTION__), 
                     DLT_STRING("():WBL Proxy object created"));
        g_signal_connect( pWBLClientProxyObj,
                          "g-properties-changed",
                          G_CALLBACK (vHandleWifiSetupUpdate),
			  NULL );  
    }
    else
    {
       if(error != NULL)
       {
           DLT_LOG(SPM_WBL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
		   DLT_STRING("():error = "), 
		   DLT_STRING(error->message?error->message: ""));
           g_clear_error (&error);
       }
       DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__), DLT_STRING(
                                             "WBL proxy creation failed"));
                     
    }
    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}

/************************************************************************
* Function:    bWBLAvailable()
* Description: This function gives the WBL proxy availability status
* Parameters:  void
* Return:      void
*************************************************************************/
gboolean bWBLAvailable()
{
    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    if(pWBLClientProxyObj)
       return TRUE;
    else
       return FALSE;
    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}


/***********************************************************************
* Function:    bCreateTempWiFiSetup
* Descriptiom: Method to create temporary wifi setup
* Parameters:  ssid, wpa_key
* Return:      gboolean
************************************************************************/
gboolean  bCreateTempWiFiSetup(gchar* ssid, gchar* wpa_key)
{
    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    GVariantBuilder builder;
    GVariant *dict,*apconfig, *dict_ssid, *bytes_ssid;

    g_variant_builder_init(&builder,G_VARIANT_TYPE("ay"));
    size_t i = 0;
    for(i = 0; i  < strlen(ssid); i++)
    {
        g_variant_builder_add(&builder, "y", ssid[i]);
    }
    bytes_ssid = g_variant_builder_end(&builder);

    g_variant_builder_init(&builder,G_VARIANT_TYPE("a{sv}"));

       
    g_variant_builder_add(&builder, "{sv}", "ssid", bytes_ssid);
    g_variant_builder_add(&builder, "{sv}", "utf8",
                               	   g_variant_new_string("false"));

    dict_ssid = g_variant_builder_end(&builder);

    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                      DLT_STRING("(): ssid is "), DLT_STRING(ssid));

    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                       DLT_STRING("(): wpa key is "), DLT_STRING(wpa_key));     

    g_variant_builder_init(&builder,G_VARIANT_TYPE("a{sv}"));

    g_variant_builder_add(&builder, "{sv}", "Type", 
                                     g_variant_new_string("Restricted"));

    g_variant_builder_add(&builder, "{sv}", "StationsToBeReserved", 
                                              g_variant_new_uint16(1));

    g_variant_builder_add(&builder, "{sv}", "Passphrase",
                                     g_variant_new_string(wpa_key));

    g_variant_builder_add(&builder, "{sv}", "Frequency",
                    g_variant_new_string(WBL_DEFAULT_SEAMLESS_FREQUENCY));

    g_variant_builder_add(&builder, "{sv}", "SSID",
                                        dict_ssid);



    apconfig = g_variant_builder_end(&builder);


    g_variant_builder_init(&builder,G_VARIANT_TYPE("a{sv}"));
    g_variant_builder_add(&builder, "{sv}", "Mode", 
                                       g_variant_new_string("AP"));
    g_variant_builder_add(&builder, "{sv}", "APConfig", apconfig);
    dict = g_variant_builder_end(&builder);

    if(pWBLClientProxyObj)
    {
         g_dbus_proxy_call( pWBLClientProxyObj, 
                            "PrepareSetup",
                              g_variant_new ("(@a{sv})",
                                    dict ),
                              G_DBUS_CALL_FLAGS_NONE, 
                              -1, 
                              NULL,
                              NULL,   //vCreatetempWiFiResultCB, 
                              NULL  );       
    }
    else
    {
        DLT_LOG(SPM_WBL, DLT_LOG_ERROR,DLT_STRING(__FUNCTION__), DLT_STRING(
                            "():No connection established with WBL"));
        return FALSE;
    }
            
    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__)); 
    return TRUE;      
}

/***********************************************************************
* Function:    vDeActivateTempAP
* Descriptiom: Sends request to WBL to de-activate temp AP
* Parameters:  void
* Return:      void
************************************************************************/
void vDeActivateTempAP()
{
    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
    if(pWBLClientProxyObj)
    {
        g_dbus_proxy_call( pWBLClientProxyObj,
                           "DeActivateSetup",
                           g_variant_new ("(s)",
                                           "AP" ),
                           G_DBUS_CALL_FLAGS_NONE,
                           -1,
                           NULL,
                           NULL,   //vCreatetempWiFiResultCB,
                           NULL  );
    }
    else
    {
       DLT_LOG(SPM_WBL, DLT_LOG_ERROR,DLT_STRING(__FUNCTION__), DLT_STRING(
                                      "(): NULL WBL Proxy"));
    }
    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
}

/***********************************************************************
* Function:    vHandleWifiSetupUpdate 
* Descriptiom: Handles the WBL ActiveWifiSetups property update
* Parameters:  proxy, changed_properties, invalidated_properties
* Return:      void 
************************************************************************/  
void  vHandleWifiSetupUpdate( GDBusProxy          *proxy,
                              GVariant            *changed_properties,
                              const gchar* const  *invalidated_properties,
	                      gpointer            user_data )   
{
    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__)); 
    if (g_variant_n_children (changed_properties) > 0)
    {
        GVariantIter *iter;
        const gchar *key;
        GVariant *value;
        g_variant_get ( changed_properties,
 		       "a{sv}",
		       &iter );
        while (g_variant_iter_loop (iter, "{&sv}", &key, &value))
        {
           g_autofree gchar *value_str;
	   value_str = g_variant_print (value, TRUE);
	   DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("Key - Value: "),
           DLT_STRING(key),DLT_STRING("->"),DLT_STRING(value_str));
           if(strcmp(key, "ActiveWiFiSetups") == 0)
           {
               vFetchActiveWiFiSetup();
           }   
        }
        g_variant_iter_free (iter);
    }

    if (g_strv_length ((GStrv) invalidated_properties) > 0)
    {
       guint ucount;
       DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("Properties Invalidated:"));
       for (ucount = 0; invalidated_properties[ucount] != NULL; ucount++)
       {
	  const gchar *key = invalidated_properties[ucount];
	  DLT_LOG(SPM_WBL, DLT_LOG_INFO,DLT_STRING("  "),DLT_STRING( key));
       }
    }      
    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
} 

/***********************************************************************
* Function:    vFetchActiveWiFiSetup
* Descriptiom: Fetches the current active Wifi setups
* Parameters:  void
* Return:      void
************************************************************************/
void vFetchActiveWiFiSetup() 
{
    DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));
    GVariant* wifiInfo =  g_dbus_proxy_get_cached_property(pWBLClientProxyObj, 
                                                     "ActiveWiFiSetups" );
    if(wifiInfo == NULL)
    {
       DLT_LOG(SPM_WBL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
           DLT_STRING("(): Recieved Null property update: ActiveWifiSetups"));
       return;
    }   
    char* wifiStr = NULL;
    wifiStr = 	g_variant_print (wifiInfo, TRUE);
    if(wifiStr)
    {
        DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
            DLT_STRING(wifiStr));
    }		   
	
    char  ssid[SSID_LENGTH+1] = { '\0' };
    char  wpa_key[WPA_KEY_LENGTH+1] = { '\0' };
    gboolean apModeDetected = FALSE;
    GVariantIter* iter1 = NULL;
    g_variant_get (wifiInfo, "a(oa{sv})", &iter1);
    if(iter1 == NULL)
    {
       DLT_LOG(SPM_WBL, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
           DLT_STRING("(): iterator parsed to NULL"));
	   return;   
    } 
	
    const gchar *objPath;
    GVariantIter* iter2 = NULL;
    WiFiAPState wifiAPMode = WIFI_AP_INACTIVE;

    while (g_variant_iter_loop (iter1, "(&oa{sv})", &objPath, &iter2))
    {
         gchar *key_outer;
         GVariant* value_outer = NULL;

         DLT_LOG(SPM_WBL, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                            DLT_STRING("(): Objpath = "), DLT_STRING(objPath));
         if(strstr(objPath, "/AP") == NULL)
             continue;
         
         while (g_variant_iter_loop (iter2, "{&sv}", &key_outer, &value_outer))
         {
             gchar *key_inner;
             GVariant* value_inner = NULL;


             if( strcmp(key_outer, "APConfig") == 0 )
             {
                 apModeDetected = TRUE;    
                 gsize configSize = g_variant_n_children(value_outer);
                 GVariantIter iter3 ;
                 g_variant_iter_init (&iter3, value_outer);
                 while (g_variant_iter_next(&iter3, "{sv}", &key_inner, &value_inner))
                 {
                     if(strcmp(key_inner, "SSID") == 0)
                     {
                          gsize size;
                          gconstpointer cvByteArray = g_variant_get_fixed_array( value_inner, 
						                                 &size,
 							 		  sizeof(gchar)
									       );

                         if ( (NULL != cvByteArray) && (size > 0 ))
                         {
                              strncpy(ssid, (char*)cvByteArray, size);                
                              ssid[size] = '\0';
                         }
                     }
                     else if(strcmp(key_inner, "Passphrase") == 0)
                     {
                         char* passphrase = NULL;
                         gsize len = 0;
                         passphrase = g_variant_get_string(value_inner, &len);
                         if((NULL != passphrase) && (len <= WPA_KEY_LENGTH))
                         {
                             strncpy(wpa_key, passphrase, len);
                             wpa_key[len] = '\0';
                         } 
                     }
                     g_variant_unref(value_inner);
                     g_free(key_inner);
                 }    
              }
              if(!strcmp(key_outer, "PowerState"))
              {
                 gsize len = 0;
                 char* state = NULL;
                 state = g_variant_get_string(value_outer, &len); 
                 if(!strcmp(state, "WIFI_STATE_POWERED_OFF" ))
                 {  
                     wifiAPMode = WIFI_AP_INACTIVE;  
                 }
                 else if(!strcmp(state, "WIFI_STATE_POWERED_ON"))
                 {
                     wifiAPMode = WIFI_AP_ACTIVE;
                 } 
                 else
                 {
                    wifiAPMode = WIFI_MODE_TRANSITION;
                 }
              }           
        }
    } 
    if(apModeDetected)
        vHandleActiveWiFiUpdate(ssid, wpa_key, wifiAPMode); 
}

