#include <string.h>
#include "dlt/dlt.h"
#include "SP_SystemSettings1_DBusHandler.h"
#include "seamless_controller.h"
#include "seamlessMidw_DBusHandler.h"

DLT_DECLARE_CONTEXT(SPM_FactoryReset);

#define SP_SYSTEMSETTINGS_OBJ_PATH "/org/bosch/cm/diagnosis/dbus/Diagnosis1/SystemSettings1/SeamlessPairing"
#define SP_SYSTEM_SETTING_ID "DIA_SYSSET_ID_SeamlessPairing"
#define SUCCESS 0
#define FAILURE 1

static Diagnosis1SystemSettings1* pSystemSettingsProxyObj = NULL;

/***************************************************************************
* Function:     bExportSPFactoryReset
* Description:  This method exposes the seamless pairing 
                Factory Reset interface
* Parameters:   void
* Return:       gboolean
****************************************************************************/
gboolean bExportSPFactoryReset()
{ 
    DLT_REGISTER_CONTEXT(SPM_FactoryReset,"SMFR", "Seamless Midw Factory Reset log");
    DLT_LOG(SPM_FactoryReset, DLT_LOG_INFO, DLT_STRING("+"),DLT_STRING(__FUNCTION__));

    GError *error = NULL;
    GDBusConnection* connection = NULL;

    if((connection = poGetSPMDBusConn()) == NULL)
    {
        DLT_LOG(SPM_FactoryReset, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
            DLT_STRING("(): poGetSPMDBusConn returned NULL"));
        return FALSE;
    }

    if(pSystemSettingsProxyObj)
    {
        if(g_dbus_interface_skeleton_has_connection (
               G_DBUS_INTERFACE_SKELETON(pSystemSettingsProxyObj), connection))
        {
            vUnExportSPFactoryReset();
        }
        else
          g_clear_object(&pSystemSettingsProxyObj);
    }

    pSystemSettingsProxyObj = diagnosis1_system_settings1_skeleton_new();

    if(!pSystemSettingsProxyObj)
    {
        DLT_LOG(SPM_FactoryReset, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                                        DLT_STRING("(): NULL Proxy"));
        return FALSE;
    }

    if (!g_dbus_interface_skeleton_export(
             G_DBUS_INTERFACE_SKELETON( pSystemSettingsProxyObj),
                                                      connection,
                                      SP_SYSTEMSETTINGS_OBJ_PATH,
                                                       &error) )
    {
        DLT_LOG(SPM_FactoryReset, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__), 
	                         DLT_STRING("():Interface export error"),
 		  	         DLT_STRING(error ? error->message : ""));
        if(error) g_clear_error(&error);
        return FALSE;
    }
     
    g_signal_connect(pSystemSettingsProxyObj, "handle-prepare-system-setting",
                     G_CALLBACK (handle_prepare_system_setting), NULL);

    g_signal_connect(pSystemSettingsProxyObj, "handle-execute-system-setting",
                     G_CALLBACK (handle_execute_system_setting), NULL);

    g_signal_connect(pSystemSettingsProxyObj, "handle-finalize-system-setting",
                     G_CALLBACK (handle_finalize_system_setting), NULL);

     DLT_LOG(SPM_FactoryReset, DLT_LOG_INFO, DLT_STRING("-"),DLT_STRING(__FUNCTION__));
     return TRUE;
}

/*************************************************************************
* Function:    vUnExportSPFactoryReset
* Description: This method unexports the Factory reset interface
* Parameters:  void
* Return:      void
**************************************************************************/
void vUnExportSPFactoryReset()
{
    DLT_LOG(SPM_FactoryReset, DLT_LOG_INFO, DLT_STRING("+"),
                 DLT_STRING(__FUNCTION__));
    if(pSystemSettingsProxyObj != NULL)
    {
         g_dbus_interface_skeleton_unexport(
		      G_DBUS_INTERFACE_SKELETON(pSystemSettingsProxyObj));
         g_clear_object(&pSystemSettingsProxyObj);
    }

    DLT_LOG(SPM_FactoryReset,DLT_LOG_INFO, DLT_STRING("-"),
                      DLT_STRING(__FUNCTION__));
    DLT_UNREGISTER_CONTEXT(SPM_FactoryReset);
}  
  
/**************************************************************************
* Function:    handle_prepare_system_setting
* Description: This method handles the PrepareSystemSetting
               request from the Factory Reset
* Parameters:  proxy object, invocation pointer, data, user data
* Return:      gboolean
****************************************************************************/
gboolean  handle_prepare_system_setting( Diagnosis1SystemSettings1 *object,
                                         GDBusMethodInvocation *invocation,
                                         const gchar *sysSetID,
                                         const gchar *sysSetType,
                                         GVariant *extendedData,
                                         guint cookie
                                       )  
{
    DLT_LOG(SPM_FactoryReset, DLT_LOG_INFO, DLT_STRING("+"),
                         DLT_STRING(__FUNCTION__));
    guint status = SUCCESS;
    if((!sysSetID) || (strcmp(SP_SYSTEM_SETTING_ID, sysSetID)))
    { 
        DLT_LOG(SPM_FactoryReset, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
           DLT_STRING("(): Invalid System setting ID"));
        status = FAILURE; 
    }
    else if( bIsPairingInProgress())
    {
        vCloseSeamlessSession(SP_CANCELLED, SP_ERR_NO_ERROR);
    }
     
    diagnosis1_system_settings1_complete_prepare_system_setting( object,
                                                                 invocation,
                                                                 sysSetID,
                                                                 sysSetType,
                                                                 extendedData,
                                                                 status,
                                                                 cookie );

    DLT_LOG(SPM_FactoryReset, DLT_LOG_INFO, DLT_STRING("-"),
                         DLT_STRING(__FUNCTION__));
    return TRUE;
}
            
/**************************************************************************
* Function:    handle_execute_system_setting
* Description: This method handles the ExecuteSystemSetting
               request from the Factory Reset
* Parameters:  proxy object, invocation pointer, data, user data
* Return:      gboolean
****************************************************************************/
gboolean  handle_execute_system_setting( Diagnosis1SystemSettings1 *object,
                                         GDBusMethodInvocation *invocation,
                                         const gchar *sysSetID,
                                         const gchar *sysSetType,
                                         GVariant *extendedData,
                                         guint cookie
                                       )
{
    DLT_LOG(SPM_FactoryReset, DLT_LOG_INFO, DLT_STRING("+"),
                         DLT_STRING(__FUNCTION__));

    guint status = SUCCESS;

    if((!sysSetID) || (strcmp(SP_SYSTEM_SETTING_ID, sysSetID)))
    {
        DLT_LOG(SPM_FactoryReset, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
           DLT_STRING("(): Invalid System setting ID"));
        status = FAILURE;
    }
    else
    {
       if( bIsPairingInProgress())
       {
           vCloseSeamlessSession(SP_CANCELLED, SP_ERR_NO_ERROR);
       }
    
       if(!bRemoveAllTrustedDevices(NULL))
       {
           DLT_LOG(SPM_FactoryReset, DLT_LOG_ERROR, DLT_STRING(__FUNCTION__),
                    DLT_STRING("(): bRemoveAllTrustedDevices failed"));
           status = FAILURE; 
       }
       else
       {
           DLT_LOG(SPM_FactoryReset, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
                      DLT_STRING("(): Factory Reset success"));
           status = SUCCESS;
       }
    }
    diagnosis1_system_settings1_complete_execute_system_setting( object,
                                                                 invocation,
                                                                 sysSetID,
                                                                 sysSetType,
                                                                 extendedData,
                                                                 status,
                                                                 cookie );
    DLT_LOG(SPM_FactoryReset, DLT_LOG_INFO, DLT_STRING("-"),
                         DLT_STRING(__FUNCTION__));
    return TRUE;
}
    
/**************************************************************************
* Function:    handle_finalize_system_setting
* Description: This method handles the FinalizeSystemSetting
               request from the Factory Reset
* Parameters:  proxy object, invocation pointer, data, user data
* Return:      gboolean
****************************************************************************/
gboolean  handle_finalize_system_setting( Diagnosis1SystemSettings1 *object,
                                          GDBusMethodInvocation *invocation,
                                          const gchar *sysSetID,
                                          const gchar *sysSetType,
                                          GVariant *extendedData,
                                          guint cookie
                                        )
{
    DLT_LOG(SPM_FactoryReset, DLT_LOG_INFO, DLT_STRING("+"),
                         DLT_STRING(__FUNCTION__));
    guint status = SUCCESS;

    if((!sysSetID) || (strcmp(SP_SYSTEM_SETTING_ID, sysSetID)))
    {
        DLT_LOG(SPM_FactoryReset, DLT_LOG_INFO, DLT_STRING(__FUNCTION__),
           DLT_STRING("(): Invalid System setting ID"));
        status = FAILURE;
    }

    diagnosis1_system_settings1_complete_finalize_system_setting( object,
                                                                  invocation,
                                                                  sysSetID,
                                                                  sysSetType,
                                                                  extendedData,
                                                                  SUCCESS,
                                                                  cookie );

    DLT_LOG(SPM_FactoryReset, DLT_LOG_INFO, DLT_STRING("-"),
                         DLT_STRING(__FUNCTION__));
    return TRUE;
}

