/*****************************************************************************
* FILE:         onlineFacade.cpp
* PROJECT:      A-IVI project
* SW-COMPONENT: profilemanager
*----------------------------------------------------------------------------
* DESCRIPTION:  core logic for Online Profile Manager
*----------------------------------------------------------------------------
* COPYRIGHT:    (c) 2017 Robert Bosch GmbH, Hildesheim
*****************************************************************************/
#include "online/onlineFacade.h"
#include "online/serviceprovider/IServiceProvider.h"
#include "online/serviceprovider/ConfigMgmtSvcAdapter.h"
#include "online/serviceprovider/ProfileSvcAdapter.h"
#include "online/onlineStatus.h"

#include "SDCHandler.h"

#include <semaphore.h>
#include <pthread.h>

#include <systemd/sd-daemon.h>
#include <yajl/yajl_gen.h>
#include <unistd.h>

#include <dlt/dlt.h>
#include "online/jsonParser.h"
#include "online/syncObject.h"

DLT_IMPORT_CONTEXT(PROFILEDATA_COMPONENT);

using namespace std;
//using namespace ::app::core;
using namespace ::profileMngr;
namespace profileMngr {

onlineFacade *onlineFacade::m_pInstance=NULL;

//*************************** onlineFacade() ***********************************

onlineFacade::onlineFacade(IServiceProvider& serviceProvider,
		ILinkProfileReply& linkProfileReply,
		IUnlinkProfileReply& unlinkProfileReply,
		IAccountNameUpdate& accountNameReply,
		IKIDUpdate& kIDReply,
		IRemoteVehicleSettingsActivationUpdate& vehcleSetngActReply)
        : m_serviceProvider(serviceProvider)
        , m_linkProfileReply(linkProfileReply)
        , m_unlinkProfileReply(unlinkProfileReply)
        , m_accountNameReply(accountNameReply)
        , m_kIDReply(kIDReply)
        , m_remtVehcleSettngsActReply(vehcleSetngActReply)
{
	DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("onlineFacade constructor called"));
	m_pInstance=this;
}
//*************************** ~onlineFacade ***************************************

onlineFacade::~onlineFacade()
{

}

void onlineFacade::onAllServicesAvailable()
{
	m_serviceProvider.getProfileSvcAdapter()->registerForProperties();
	m_serviceProvider.getConfigMgmtSvcAdapter()->sendRegisterReqForUAM();
	// Get the Number of Profiles from ProfileBase which is required for onlineData init
	if(!m_getProfileTypesCmd.isBusy())
	{
		m_getProfileTypesCmd.init(m_serviceProvider.getProfileSvcAdapter(),*this);
		m_getProfileTypesCmd.execute();
	}
}

void onlineFacade::startLinkProfile(const uintptr_t act, const std::string& accountName, const std::string& accountPwd)
{
	DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("onlineFacade::startLinkProfile"),DLT_INT(m_linkProfileCmd.isBusy()));

	if(m_linkProfileCmd.isBusy())
	{
		m_linkProfileReply.onLinkProfileError(act,rsFailed);
	    return;
	}

	m_linkProfileCmd.init(act,accountName,accountPwd,m_serviceProvider.getDummSvcAdapter(),m_serviceProvider.getUsbTcuSvcAdapter(),m_serviceProvider.getProfileSvcAdapter(),m_Data,*this);
	m_linkProfileCmd.execute();

}

void onlineFacade::startUnlinkProfile(const uintptr_t act, const uint8_t userID)
{
	if(m_unlinkProfileCmd.isBusy())
	{
		m_unlinkProfileReply.onUnlinkProfileError(act,rsFailed);
	    return;
	}

	m_unlinkProfileCmd.init(act,userID,m_serviceProvider.getDummSvcAdapter(),m_serviceProvider.getProfileSvcAdapter(),m_Data,*this);
	m_unlinkProfileCmd.execute();
}

void onlineFacade::linkProfileCompletionSuccess(uintptr_t act, resultState result)
{
	sendLinkProfileState(true);
	sendAccountNameUpdate();
	sendKIDUpdate();
	m_linkProfileReply.onLinkProfileSuccess(act, result);
}

void onlineFacade::linkProfileCompletionError(uintptr_t act, resultState result)
{
	m_linkProfileReply.onLinkProfileError(act, result);
}

void onlineFacade::unlinkProfileCompletionSuccess(uintptr_t act, resultState result)
{
	sendLinkProfileState(false);
	sendAccountNameUpdate();
	sendKIDUpdate();
	m_unlinkProfileReply.onUnlinkProfileSuccess(act, result);
}

void onlineFacade::unlinkProfileCompletionError(uintptr_t act, resultState result)
{
	m_unlinkProfileReply.onUnlinkProfileError(act, result);
}

void onlineFacade::getProfileTypesCompleted(bool bSuccess)
{
	// Trigger the Init of onlineData object
	if(bSuccess && (!m_Data.isInit()))
	{
		//TODO: Get the Variant Information from the respective variant library once it is setup.
		//For now hardcoded: 0 for Renault
		VariantInfo info;
		info.variantID = 0;
		info.versionID = 2;
		m_Data.Init(m_getProfileTypesCmd.getProfileTypes().size(),info);
		m_remtVehcleSettngsActReply.onRemVehcleSettngsActivationUpdate(m_Data.getRemoteVehicleSettitngActivation());
	}
}

void onlineFacade::setPINAuthenticationDelay(uint32_t delay)
{
	::boost::shared_ptr<ProfileSvcAdapter> profileProxyAdapter = m_serviceProvider.getProfileSvcAdapter();
	if(profileProxyAdapter->isValid())
	{
		profileProxyAdapter->setPINAuthenticationDelay(delay);
	}
}
void onlineFacade::setAuthenticationValidTime(uint32_t validTime)
{
	::boost::shared_ptr<ProfileSvcAdapter> profileProxyAdapter = m_serviceProvider.getProfileSvcAdapter();
	if(profileProxyAdapter->isValid())
	{
		profileProxyAdapter->setAuthenticationValidTime(validTime);
	}
}
void onlineFacade::setMaxPINAttemptsTotal(uint32_t value)
{
	::boost::shared_ptr<ProfileSvcAdapter> profileProxyAdapter = m_serviceProvider.getProfileSvcAdapter();
	if(profileProxyAdapter->isValid())
	{
		profileProxyAdapter->setMaxPINAttemptsTotal(value);
	}
}
void onlineFacade::setMaxPINAttemptsBeforeDelay(uint32_t value)
{
	::boost::shared_ptr<ProfileSvcAdapter> profileProxyAdapter = m_serviceProvider.getProfileSvcAdapter();
	if(profileProxyAdapter->isValid())
	{
		profileProxyAdapter->setMaxPINAttemptsBeforeDelay(value);
	}
}
void onlineFacade::setMaxSecuredProfiles(uint32_t value)
{
	m_Data.setMaxSecuredProfiles(value);
}
void onlineFacade::setLocalLinkOrderTimeout(uint32_t value)
{
	m_Data.setLocalLinkOrderTimeout(value);
}

void onlineFacade::setRmtVhclSttngsAct(uint32_t value)
{
	m_Data.setRemoteVehicleSettitngActivation(value);
	m_remtVehcleSettngsActReply.onRemVehcleSettngsActivationUpdate(m_Data.getRemoteVehicleSettitngActivation());
}

void onlineFacade::setConnectionStatus(uint32_t value)
{
	if((value == 1) || (value == 0))
		m_Data.setConnectionStatus(value);
}

uint32_t onlineFacade::getPINAuthenticationDelay()
{
	uint32_t delay = 0;
	::boost::shared_ptr<ProfileSvcAdapter> profileProxyAdapter = m_serviceProvider.getProfileSvcAdapter();
	if(profileProxyAdapter->isValid())
	{
		delay = profileProxyAdapter->getPINAuthenticationDelay();
	}
	return delay;
}
uint32_t onlineFacade::getAuthenticationValidTime()
{
	uint32_t time = 0;
	::boost::shared_ptr<ProfileSvcAdapter> profileProxyAdapter = m_serviceProvider.getProfileSvcAdapter();
	if(profileProxyAdapter->isValid())
	{
		time = profileProxyAdapter->getAuthenticationValidTime();
	}
	return time;
}
uint32_t onlineFacade::getMaxPINAttemptsTotal()
{
	uint8_t total = 0;
	::boost::shared_ptr<ProfileSvcAdapter> profileProxyAdapter = m_serviceProvider.getProfileSvcAdapter();
	if(profileProxyAdapter->isValid())
	{
		total = profileProxyAdapter->getMaxPINAttemptsTotal();
	}
	return total;
}
uint32_t onlineFacade::getMaxPINAttemptsBeforeDelay()
{
	uint8_t total = 0;
	::boost::shared_ptr<ProfileSvcAdapter> profileProxyAdapter = m_serviceProvider.getProfileSvcAdapter();
	if(profileProxyAdapter->isValid())
	{
		total = profileProxyAdapter->getMaxPINAttemptsBeforeDelay();
	}
	return total;
}
uint32_t onlineFacade::getMaxSecuredProfiles()
{
	return m_Data.getMaxSecuredProfiles();
}
uint32_t onlineFacade::getRmtVhclSttngsAct()
{
    return m_Data.getRemoteVehicleSettitngActivation();
}
uint32_t onlineFacade::getLocalLinkOrderTimeout()
{
	return m_Data.getLocalLinkOrderTimeout();
}

void onlineFacade::sendAccountNameUpdate()
{
	::boost::shared_ptr<ProfileSvcAdapter> prfSrvcAdap = m_serviceProvider.getProfileSvcAdapter();
	if(prfSrvcAdap->isValid())
	{
		uint8_t userID = prfSrvcAdap->getActiveProfile();
		::std::string accName = m_Data.getName(userID);
		m_accountNameReply.onAccountNameUpdate(accName);
	}
}

void onlineFacade::sendKIDUpdate()
{
	::boost::shared_ptr<ProfileSvcAdapter> prfSrvcAdap = m_serviceProvider.getProfileSvcAdapter();
	if(prfSrvcAdap->isValid())
	{
		uint8_t userID = prfSrvcAdap->getActiveProfile();
		::std::string kID = m_Data.getkID(userID);
		m_kIDReply.onKIDUpdate(kID);
	}
}

void onlineFacade::sendLinkProfileState(bool isLinked)
{
	::boost::shared_ptr<ProfileSvcAdapter> prfSrvcAdap = m_serviceProvider.getProfileSvcAdapter();
	if(prfSrvcAdap->isValid())
		prfSrvcAdap->setLinkProfileStatus(isLinked);
}

}

