/*****************************************************************************
* FILE:         onlineComponentImpl.cpp
* PROJECT:      A-IVI project
* SW-COMPONENT: profilemanager
*----------------------------------------------------------------------------
* DESCRIPTION:  core logic for Online Profile Manager
*----------------------------------------------------------------------------
* COPYRIGHT:    (c) 2017 Robert Bosch GmbH, Hildesheim
*****************************************************************************/
#include "online/onlineComponentImpl.h"
#include "online/asfComp/dbus/ProfileMngrOnlineDBUSServiceImpl.h"
#include "online/asfComp/dbus/ProfileMngrConfigDataServiceImpl.h"

#include <semaphore.h>
#include <pthread.h>

#include <systemd/sd-daemon.h>
#include <yajl/yajl_gen.h>
#include <unistd.h>
#include <list>

#include <dlt/dlt.h>
#include "online/jsonParser.h"
#include "online/syncObject.h"

DLT_IMPORT_CONTEXT(PROFILEDATA_COMPONENT);

using namespace std;
using namespace bosch::cm::profileConnected;

namespace profileMngr {

onlineComponentImpl *onlineComponentImpl::m_pInstance=NULL;


using namespace ::app::core;
using namespace ::profileMngr;

//*************************** onlineComponentImpl() ***********************************
const std::list<AdapterType> initProfileServiceProvider()
{
   std::list<AdapterType> requiredAdapter;
   requiredAdapter.push_back(CONFIGURATIONMGMT_SVC);
   requiredAdapter.push_back(PROFILE_SVC);
   requiredAdapter.push_back(DUMM_SVC);
   requiredAdapter.push_back(USBTCU_SVC);
   return requiredAdapter;
}
onlineComponentImpl::onlineComponentImpl()
        : m_profileServiceImpl(0),
		  m_configDataServiceImpl(0),
		  m_serviceProvider(*this, initProfileServiceProvider()),
		  m_initialized(false),
		  m_onlineFacade(m_serviceProvider, *this, *this, *this, *this, *this)
          //m_Parser(* new jsonParser(RootBlockName,m_SyncList)),
{

	DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("onlineComponentImpl constructor called"));
	m_initialized = true; //TODO: When to set m_initialized to true
	m_pInstance=this;
	//ask the service provider if all service are just up and running
	m_serviceProvider.allServicesAvailable();

}
//*************************** ~onlineComponentImpl ***************************************

onlineComponentImpl::~onlineComponentImpl()
{
   m_serviceProvider.clear(); //clear all used proxy resources before destroying facade and all its dependencies
   destroyProfileServiceStub();
   destroyConfigDataServiceStub();
}

void onlineComponentImpl::onAvailable()
{
	DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("onlineComponentImpl::onAvailable"));
	if(m_initialized)
	{
		createProfileServiceStub();
		createConfigDataServiceStub();
		m_onlineFacade.onAllServicesAvailable();
	}
}

void onlineComponentImpl::onUnavailable()
{
	DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("onlineComponentImpl::onUnavailable"));
	destroyProfileServiceStub();
	destroyConfigDataServiceStub();
}
void onlineComponentImpl::destroyProfileServiceStub()
{
	if ( m_profileServiceImpl != 0 )
	{
		delete m_profileServiceImpl;
		m_profileServiceImpl = 0;
	}
	else
	{
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("ProfileService Stub can't be deleted if it is not created"));
	}
}
void onlineComponentImpl::destroyConfigDataServiceStub()
{
	if ( m_configDataServiceImpl != 0 )
	{
		delete m_configDataServiceImpl;
		m_configDataServiceImpl = 0;
	}
	else
	{
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("ConfigDataService Stub can't be deleted if it is not created"));
	}
}

void onlineComponentImpl::createProfileServiceStub()
{
   if ( m_profileServiceImpl != 0 )
   {
	   DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("Service Stub can't be created if it is still available"));
   }
   else
   {
      if(m_initialized)
      {
         m_profileServiceImpl = new (std::nothrow) ProfileMngrOnlineDBUSServiceImpl(m_onlineFacade);
         if(!m_profileServiceImpl)
            DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Service Stub can't be created because lack of memory."));
      }
   }
}

void onlineComponentImpl::createConfigDataServiceStub()
{
   if ( m_configDataServiceImpl != 0 )
   {
	   DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("Service Stub can't be created if it is still available"));
   }
   else
   {
      if(m_initialized)
      {
       	 m_configDataServiceImpl = new (std::nothrow) ProfileMngrConfigDataServiceImpl(m_onlineFacade);
         if(!m_configDataServiceImpl)
        	 DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Service Stub can't be created because lack of memory."));
      }
   }
}

void onlineComponentImpl::onLinkProfileSuccess(uintptr_t act, resultState result)
{
   if ( m_profileServiceImpl )
   {
	   m_profileServiceImpl->sendLinkProfileResponse(result, act);
   }
   else
   {
	   DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_DEBUG, DLT_STRING("Service unavailable - onLinkProfileSuccess skipped "));
   }
}

void onlineComponentImpl::onLinkProfileError(uintptr_t act, resultState result)
{
   if ( m_profileServiceImpl )
   {
	   m_profileServiceImpl->sendLinkProfileResponse(result, act);
   }
   else
   {
	   DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_DEBUG, DLT_STRING("Service unavailable - onLinkProfileError skipped "));
   }
}

void onlineComponentImpl::onUnlinkProfileSuccess(uintptr_t act, resultState result)
{
   if ( m_profileServiceImpl )
   {
	   m_profileServiceImpl->sendUnlinkProfileResponse(result, act);
   }
   else
   {
	   DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_DEBUG, DLT_STRING("Service unavailable - onLinkProfileSuccess skipped "));
   }
}

void onlineComponentImpl::onUnlinkProfileError(uintptr_t act, resultState result)
{
   if ( m_profileServiceImpl )
   {
	   m_profileServiceImpl->sendUnlinkProfileResponse(result, act);
   }
   else
   {
	   DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_DEBUG, DLT_STRING("Service unavailable - onLinkProfileError skipped "));
   }
}

void onlineComponentImpl::onAccountNameUpdate(const ::std::string& accountName)
{
	if ( m_profileServiceImpl )
	{
		m_profileServiceImpl->setAccountName(accountName);
	}
	else
	{
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_DEBUG, DLT_STRING("Service unavailable - onAccountNameUpdate skipped "));
	}
}

// IKIDUpdate
void onlineComponentImpl::onKIDUpdate(const ::std::string& kID)
{
	if ( m_profileServiceImpl )
	{
		m_profileServiceImpl->setKID(kID);
	}
	else
	{
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_DEBUG, DLT_STRING("Service unavailable - onKIDUpdate skipped "));
	}
}

// IRemoteVehicleSettingsActivationUpdate
void onlineComponentImpl::onRemVehcleSettngsActivationUpdate(bool status)
{
	if ( m_profileServiceImpl )
	{
		m_profileServiceImpl->setRemoteVehicleSettingsActivation(status);
	}
	else
	{
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_DEBUG, DLT_STRING("Service unavailable - onRemVehcleSettngsActivationUpdate skipped "));
	}
}

} //::profileMngr




