/*****************************************************************************
* FILE:         ProfileMngrConfigDataServiceImpl.cpp
* PROJECT:      A-IVI project
* SW-COMPONENT: profilemanager
*----------------------------------------------------------------------------
* DESCRIPTION:
*----------------------------------------------------------------------------
* COPYRIGHT:    (c) 2017 Robert Bosch GmbH, Hildesheim
*****************************************************************************/

#include <dlt/dlt.h>
#include "ProfileMngrConfigDataServiceImpl.h"
#include "online/onlineFacade.h"
#include "online/jsonUtils.h"

DLT_IMPORT_CONTEXT(PROFILEDATA_COMPONENT);

namespace app { namespace core {

using namespace::asf::core;
using namespace profileMngr;
using namespace::de::bosch::cm::ConfigManager::ConfigData;

struct ConfigMgrItemRangeCheck
{
	const char* Group;
	const char* Item;
	const char* Key;
	bool isStringData;
	uint32_t Min;
	uint32_t Max;
	uint32_t Factor;
	void (onlineFacade::*SetItemfnPtr)(uint32_t);
	uint32_t (onlineFacade::*GetItemfnPtr)();
};

static const ConfigMgrItemRangeCheck privateItemRangeCheck[] =
{
	/**************************************************************************************************************************/
	/*  1. GROUP           2. ITEM                     3. KEY        4.MIN    5. MAX  6. Factor    7.Function to be invoked   */
	/**************************************************************************************************************************/
	{jsonUtils::GROUP_UAM,jsonUtils::PINAUTHENTICATIONDELAY,jsonUtils::DELAY,0,0,254,1000,&onlineFacade::setPINAuthenticationDelay,&onlineFacade::getPINAuthenticationDelay},
	{jsonUtils::GROUP_UAM,jsonUtils::AUTHENTICATIONVALIDTIME,jsonUtils::VALIDTIME,0,0,31,(60*1000),&onlineFacade::setAuthenticationValidTime,&onlineFacade::getAuthenticationValidTime},
	{jsonUtils::GROUP_UAM,jsonUtils::MAXPINATTEMPTSTOTAL,jsonUtils::TOTAL,0,0,15,1,&onlineFacade::setMaxPINAttemptsTotal,&onlineFacade::getMaxPINAttemptsTotal},
	{jsonUtils::GROUP_UAM,jsonUtils::MAXPINATTEMPTSBEFOREDELAY,jsonUtils::DELAY,0,0,7,1,&onlineFacade::setMaxPINAttemptsBeforeDelay,&onlineFacade::getMaxPINAttemptsBeforeDelay},
	{jsonUtils::GROUP_UPM,jsonUtils::LOCALLINKORDERTIMEOUT,jsonUtils::TIMEOUT,0,0,254,1000,&onlineFacade::setLocalLinkOrderTimeout,&onlineFacade::getLocalLinkOrderTimeout},
	{jsonUtils::GROUP_UPM,jsonUtils::MAXSECUREDPROFILES,jsonUtils::MAX,0,0,15,1,&onlineFacade::setMaxSecuredProfiles,&onlineFacade::getMaxSecuredProfiles},
	{jsonUtils::GROUP_RVS,jsonUtils::REMOTE_VEHICLE_SETTINGS_ACTIVATION,jsonUtils::STATUS,1,0,1,1,&onlineFacade::setRmtVhclSttngsAct,&onlineFacade::getRmtVhclSttngsAct}
};

static const ConfigMgrItemRangeCheck publicItemRangeCheck[] =
{
	/**************************************************************************************************************************/
	/*  1. GROUP           2. ITEM                     3. KEY        4.MIN    5. MAX  6. Factor    7.Function to be invoked   */
	/**************************************************************************************************************************/
	{jsonUtils::GROUP_CSA,jsonUtils::CONNECTIVITY_AND_ACTIVATION_STATUS,jsonUtils::STATUS,0,0,1,1,&onlineFacade::setConnectionStatus,&onlineFacade::getConnectionStatus}
};

DEFINE_CLASS_LOGGER_AND_LEVEL("app/core/ProfileMngrConfigDataServiceImpl", ProfileMngrConfigDataServiceImpl, Info);

#define CFGMGR_SEPARATOR "."


ProfileMngrConfigDataServiceImpl::ProfileMngrConfigDataServiceImpl(onlineFacade& onlineFacade)
: ConfigDataStub("configDataPort")
  , m_facade(onlineFacade),m_CurrentAct(0)
{
	DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("ProfileMngrConfigDataServiceImpl constructor called"));
}

ProfileMngrConfigDataServiceImpl::~ProfileMngrConfigDataServiceImpl(){
   LOG_INFO("~ProfileMngrConfigDataServiceImpl called");

}

void ProfileMngrConfigDataServiceImpl::onSetPrivateItemRequest (const ::boost::shared_ptr< SetPrivateItemRequest >& request)
{
	/*
	 * The Data Handling is currently adapted to the following format of CfgMgr:
	 * CFGMGR_SET_PRIVATE_DATA "UAM.MaxPINAttemptsTotal;Total:2\0"
     * CFGMGR_GET_PRIVATE_DATA "UAM.MaxPINAttemptsTotal\0"
     */
	std::string prItemName = request->getPrivateItemName();
	EnItemsStatus enStatus = EnItemsStatus__INVALID_SIGNATURE;
	std::size_t pos = prItemName.find(CFGMGR_SEPARATOR);
	if(pos != std::string::npos)
	{
		std::vector< ConfigInfo > prItem = request->getPrivateItem();
		std::string strGroup,strMethod,strKey,strValue;
		strGroup = prItemName.substr(0,pos);
		strMethod = prItemName.substr(pos+1);
		if(prItem.size() == 1)
		{
			ConfigInfo info = prItem.at(0);
			strKey = info.getKey();
			strValue = info.getValue();
			enStatus = updatePrvCfgItemSetting(strGroup,strMethod,strKey,strValue);
		}
		else
		{
			//TODO: The current data packing mechanism does not allow a Single Set Request to include data for for more than One setting Item.
			//It has to be checked why "getPrivateItem" returns a Vector in that case ?
			enStatus = EnItemsStatus__INVALID_SIGNATURE;
		}
	}
	sendSetPrivateItemResponse(enStatus);
}

void ProfileMngrConfigDataServiceImpl::onGetPrivateItemRequest (const ::boost::shared_ptr< GetPrivateItemRequest >& request)
{
	EnItemsStatus enStatus = EnItemsStatus__INVALID_SIGNATURE;
	std::vector< ConfigInfo > vConfigInfo;

	std::string prItemName = request->getPrivateItemName();
	std::size_t pos = prItemName.find(CFGMGR_SEPARATOR);
	if(pos != std::string::npos)
	{
		std::string strGroup,strMethod;
		strGroup = prItemName.substr(0,pos);
		strMethod = prItemName.substr(pos+1);
		enStatus = getPrvCfgItemSetting(strGroup,strMethod,vConfigInfo);
	}
	//send data
	sendGetPrivateItemResponse(vConfigInfo, enStatus);
}

void ProfileMngrConfigDataServiceImpl::onExchangeDESCMORequest (const ::boost::shared_ptr< ExchangeDESCMORequest >& request)
{
}

void ProfileMngrConfigDataServiceImpl::onUpdatePublicConfigItemRequest(const ::boost::shared_ptr< UpdatePublicConfigItemRequest >& request)
{
	std::string publicItem = request->getPublicItemName();
	DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("onUpdatePublicConfigItemRequest:"),DLT_STRING("publicItem:"),DLT_STRING(publicItem.c_str()));
	std::size_t pos = publicItem.find(CFGMGR_SEPARATOR);
	if(pos != std::string::npos)
	{
		::std::vector< ConfigInfo > configList = request->getPublicItem();
		std::string strGroup,strMethod;
		strGroup = publicItem.substr(0,pos);
		strMethod = publicItem.substr(pos+1);
		if(configList.size() == 1)
		{
			ConfigInfo info = configList.at(0);
			std::string strKey = info.getKey();
			int iValue = atoi(info.getValue().c_str());
			updatePubCfgItemSetting(strGroup,strMethod,strKey,iValue);
	    }
	}
}

EnItemsStatus ProfileMngrConfigDataServiceImpl::updatePrvCfgItemSetting(const std::string& group, const std::string& item,const std::string& key,const std::string& value)
{
	EnItemsStatus enStatus = EnItemsStatus__UNKNOWN_ITEM;
	const int mapSize = sizeof(privateItemRangeCheck)/sizeof(ConfigMgrItemRangeCheck);
	for(int i = 0;i < mapSize; i++)
	{
	   if((group == privateItemRangeCheck[i].Group) &&
		  (item == privateItemRangeCheck[i].Item) &&
		  (key == privateItemRangeCheck[i].Key))
	   {
		   enStatus = EnItemsStatus__INAVLID_VALUE;
		   if(privateItemRangeCheck[i].isStringData)
		   {
               uint32_t setting = 0;
               enStatus = EnItemsStatus__SUCCESS;
               void (onlineFacade::*fnPtr)(uint32_t) = privateItemRangeCheck[i].SetItemfnPtr;
			   if(value == "ON")
			   {
				   setting = 1;
				   (m_facade.*fnPtr)(setting * privateItemRangeCheck[i].Factor);
			   }
			   else if(value == "OFF")
			   {
				   (m_facade.*fnPtr)(setting * privateItemRangeCheck[i].Factor);
			   }
			   else if(value == "NC")
			   {
				   // Nothing to be done for this property. Don't send Negative acknowledgement
			   }
			   else
			   {
				   enStatus = EnItemsStatus__INAVLID_VALUE;
			   }
 		   }
		   else
		   {
			   // Look for any non-integer values
			   size_t pos = value.find_first_not_of("0123456789");
			   if(pos == string::npos)
			   {
				   int iValue = atoi(value.c_str());
				   if(iValue >= privateItemRangeCheck[i].Min && iValue <= privateItemRangeCheck[i].Max)
				   {
					   enStatus = EnItemsStatus__SUCCESS;
					   void (onlineFacade::*fnPtr)(uint32_t) = privateItemRangeCheck[i].SetItemfnPtr;
					   (m_facade.*fnPtr)(iValue * privateItemRangeCheck[i].Factor);
				   }
			   }
		   }
	   }
	}
	return enStatus;
}

EnItemsStatus ProfileMngrConfigDataServiceImpl::getPrvCfgItemSetting(const std::string& group, const std::string& item,std::vector<ConfigInfo>& vConfigInfo)
{
	EnItemsStatus enStatus = EnItemsStatus__UNKNOWN_ITEM;
	const int mapSize = sizeof(privateItemRangeCheck)/sizeof(ConfigMgrItemRangeCheck);
	for(int i = 0;i < mapSize; i++)
	{
		if((group == privateItemRangeCheck[i].Group) &&
				(item == privateItemRangeCheck[i].Item))
		{
			enStatus = EnItemsStatus__SUCCESS;
			std::string strKey = privateItemRangeCheck[i].Key;
			ConfigInfo info;
			info.setKey(strKey);
			uint32_t (onlineFacade::*fnPtr)() = privateItemRangeCheck[i].GetItemfnPtr;
			uint32_t Value = (m_facade.*fnPtr)();
			if(privateItemRangeCheck[i].isStringData)
			{
				std::string strData;
				if(Value == 1)
					{strData.assign("ON");}
				else if(Value == 0)
					{strData.assign("OFF");}
				else
					{}//TODO: Handle Invalid Value
				info.setValue(strData);
			}
			else
			{

				//Perform the Conversion if required
				Value /= privateItemRangeCheck[i].Factor;
				char value[10];
				sprintf ( value, "%d", Value );
				info.setValue(std::string(value));
			}
			vConfigInfo.push_back(info);
		}
	}
	return enStatus;
}

EnItemsStatus ProfileMngrConfigDataServiceImpl::updatePubCfgItemSetting(const std::string& group, const std::string& item,const std::string& key,int value)
{
	EnItemsStatus enStatus = EnItemsStatus__UNKNOWN_ITEM;
	DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("updatePubCfgItemSetting:"),DLT_STRING("Group:"),DLT_STRING(group.c_str()),
			DLT_STRING("Item:"),DLT_STRING(item.c_str()),DLT_STRING("Key:"),DLT_STRING(key.c_str()),DLT_STRING("Value:"),DLT_INT(value));
	const int mapSize = sizeof(publicItemRangeCheck)/sizeof(ConfigMgrItemRangeCheck);
	for(int i = 0;i < mapSize; i++)
	{
	   if((group == publicItemRangeCheck[i].Group) &&
		  (item == publicItemRangeCheck[i].Item) &&
		  (key == publicItemRangeCheck[i].Key))
	   {
		   enStatus = EnItemsStatus__INAVLID_VALUE;
		   if(value >= publicItemRangeCheck[i].Min && value <= publicItemRangeCheck[i].Max)
		   {
			   enStatus = EnItemsStatus__SUCCESS;
			   void (onlineFacade::*fnPtr)(uint32_t) = publicItemRangeCheck[i].SetItemfnPtr;
			   (m_facade.*fnPtr)(value * publicItemRangeCheck[i].Factor);
		   }
	   }
	}
	return enStatus;
}

}} // namespace app { namespace core {

