/*****************************************************************************
* FILE:         SDCHandler.cpp
* PROJECT:      A-IVI project
* SW-COMPONENT: profilemanager
*----------------------------------------------------------------------------
* DESCRIPTION:  core logic for Online Profile Manager
*----------------------------------------------------------------------------
* COPYRIGHT:    (c) 2017 Robert Bosch GmbH, Hildesheim
*****************************************************************************/

#include "SDCHandler.h"

#include <stdio.h>
#include <string.h>
#include <dlt/dlt.h>
#include <stdlib.h>
DLT_IMPORT_CONTEXT(PROFILEDATA_COMPONENT);
using namespace std;
namespace profileMngr{

SDCHandler::SDCHandler()
{
	sdc_error = SDC_UNKNOWN_ERROR;
	sdc_kid = 12006;
	sdc_key_len = 32;
	sdc_kid_opt = SDC_CREATE_KEY_FIXED_ID;
	sdc_key_stor_opt = SDC_PERSISTENT_STORAGE_KEY;
	permissions = NULL;
	session = NULL;
	type = NULL;
	encrypted_data = NULL;
	encrypted_data_len = 0;
	data_to_decrypt = NULL;
	data_to_decrypt_len = 0;

}

SDCHandler::~SDCHandler()
{

}

bool SDCHandler::bGenerateStorageKey(uint32_t kid, size_t key_length)
{
	DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("SDCHandler::bGenerateStorageKey"));
	sdc_kid = kid;
	sdc_key_len = key_length;

	sdc_error = sdc_permissions_alloc(&permissions);
	if (sdc_error != SDC_OK){
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Failed to allocate permissions structure"),DLT_INT(sdc_error),DLT_STRING(sdc_get_error_string(sdc_error)));
		return false;
	}

	sdc_error = sdc_set_default_permissions_current_gid(permissions);
	if (sdc_error != SDC_OK){
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Failed to allocate permissions structure"),DLT_INT(sdc_error),DLT_STRING(sdc_get_error_string(sdc_error)));
		return false;
	}

	sdc_error = sdc_generate_random_storage_key(&sdc_kid, sdc_kid_opt, sdc_key_len, sdc_key_stor_opt, permissions);
	if ((sdc_error != SDC_OK) && (sdc_error != SDC_KID_EXISTS)){
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Failed to generate the key"),DLT_INT(sdc_error),DLT_STRING(sdc_get_error_string(sdc_error)));
		return false;
	}
	return true;
}

bool SDCHandler::bEncryptData(uint32_t kid, const char *in_data_to_wrap, size_t in_data_to_wrap_len, unsigned char*& outData, uint32_t& length)
{
	DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("SDCHandler::bEncryptData"));
	length = 0;
	sdc_kid = kid;

	sdc_error = sdc_open_session(&session);
	if (sdc_error != SDC_OK){
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Could not open the Session"),DLT_INT(sdc_error),DLT_STRING(sdc_get_error_string(sdc_error)));
		return false;
	}

	sdc_error = sdc_session_load_storage_key(session, kid);
	if (sdc_error != SDC_OK){
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Could not assign the key to the session"),DLT_INT(sdc_error),DLT_STRING(sdc_get_error_string(sdc_error)));
		return false;
	}

	sdc_error = sdc_wrap_unwrap_type_alloc(&type);
	if (sdc_error != SDC_OK){
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Could not allocate the default algorithm and block mode"),DLT_INT(sdc_error),DLT_STRING(sdc_get_error_string(sdc_error)));
		return false;
	}

	sdc_error = sdc_wrap_formatted(session, type, (uint8_t *)in_data_to_wrap, in_data_to_wrap_len, &encrypted_data, &encrypted_data_len);
	if (sdc_error != SDC_OK){
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Failed to encrypt data"),DLT_INT(sdc_error),DLT_STRING(sdc_get_error_string(sdc_error)));
		return false;
	}
	else {
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("Encryption successful"));
		if(encrypted_data==NULL){
			DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Encrypted buf is NULL"));
			return false;
		}
		else
		{
			DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("Encrypted data Length:"),DLT_INT(encrypted_data_len));
			outData = new unsigned char[encrypted_data_len];
			memcpy(outData,encrypted_data,encrypted_data_len);
			length = encrypted_data_len;
		}
	}

	sdc_error = sdc_wrap_unwrap_type_free(type);
	if (sdc_error != SDC_OK){
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Could not free wrap_unwrap type"),DLT_INT(sdc_error),DLT_STRING(sdc_get_error_string(sdc_error)));
	}

	sdc_error = sdc_cipher_buffer_free(encrypted_data, encrypted_data_len);
	if (sdc_error != SDC_OK){
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Could not free the ciphered data"),DLT_INT(sdc_error),DLT_STRING(sdc_get_error_string(sdc_error)));
	}

	sdc_error = sdc_close_session(session);
		if (sdc_error != SDC_OK){
			DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Session could not be closed"));
	}

	return true;
}

bool SDCHandler::bDecryptData(uint32_t kid, uint8_t **out_unwrapped_data, size_t *out_unwrapped_data_len,  uint8_t * inData, uint32_t len)
{
	DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_INFO, DLT_STRING("SDCHandler::bDecryptData"));
	sdc_kid = kid;

	data_to_decrypt_len = len;//inData.size();
	data_to_decrypt = new char[data_to_decrypt_len];
	memcpy(data_to_decrypt,inData,data_to_decrypt_len);

	if(data_to_decrypt == NULL){
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Could not read the file to get data_to_decrypt"));
		return false;
	}

	sdc_error = sdc_open_session(&session);
	if (sdc_error != SDC_OK){
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Could not open the session "),DLT_INT(sdc_error),DLT_STRING(sdc_get_error_string(sdc_error)));
		return false;
	}

	sdc_error = sdc_session_load_storage_key(session, sdc_kid);
	if (sdc_error != SDC_OK){
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Could not assign the key to the session"),DLT_INT(sdc_error),DLT_STRING(sdc_get_error_string(sdc_error)));
		return false;
	}

	sdc_error = sdc_wrap_unwrap_type_alloc(&type);
	if (sdc_error != SDC_OK){
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Could not allocate the default algorithm and block mode"),DLT_INT(sdc_error),DLT_STRING(sdc_get_error_string(sdc_error)));
		return false;
	}

	sdc_error = sdc_unwrap_formatted(session, type, (uint8_t *)data_to_decrypt, data_to_decrypt_len, out_unwrapped_data, out_unwrapped_data_len);
	if (sdc_error != SDC_OK){
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Failed to decrypt data"),DLT_INT(sdc_error),DLT_STRING(sdc_get_error_string(sdc_error)));
		return false;
	}
	else{
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Could not free the ciphered data"),DLT_INT(sdc_error),DLT_STRING(sdc_get_error_string(sdc_error)));
	}

	sdc_error = sdc_wrap_unwrap_type_free(type);
	if (sdc_error != SDC_OK){
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Could not free wrap_unwrap type"));
	}

	sdc_error = sdc_close_session(session);
	if (sdc_error != SDC_OK){
		DLT_LOG(PROFILEDATA_COMPONENT, DLT_LOG_ERROR, DLT_STRING("Session could not be closed"),DLT_INT(sdc_error),DLT_STRING(sdc_get_error_string(sdc_error)));
	}

	return true;
}



}
