/*****************************************************************************
* FILE:         profileApp.h
* PROJECT:      A-IVI project
* SW-COMPONENT: profilemanager
*----------------------------------------------------------------------------
* DESCRIPTION:  core logic for Profile Manager
*----------------------------------------------------------------------------
* COPYRIGHT:    (c) 2016 Robert Bosch GmbH, Hildesheim
*****************************************************************************/

#ifndef PROFILE_APP_H
#define PROFILE_APP_H

#include "core/profileStatus.h"
#include "core/jobs/job.h"
#include "core/profileData.h"
#include <string>
#include <semaphore.h>
#include <stdint.h>
#include "core/clientList.h"

using namespace std;

class dp_tclSrvIf;
uint32_t getTickCount();

namespace profileMngr {
class configIF;

#define num_interfaces 2

class ProfileAppIF
{
public:
	virtual void startFailTimer(Job * pJob)=0;
	virtual profileData & getData()=0;
	virtual const std::string & getClientName(uint8_t Index)=0;

	virtual void loadUser(uint8_t newUser)=0;

	virtual void sendDataUpdate(dataChangeStatus)=0;
	virtual void sendProfileID(uint8_t id)=0;
	virtual void sendProfileStatus(profileState state)=0;
	virtual void sendCopyProfile(uint8_t src,uint8_t dest)=0;
	virtual void sendCreateProfile(uint8_t id)=0;
	virtual void sendDeleteProfile(uint8_t id)=0;
	virtual void sendPinRequest(unsigned char bRequest)=0;
	virtual void sendActiveProfileType()=0;

	virtual bool setDPUser(uint8_t userID)=0;
	virtual bool copyDPUser(uint8_t src,uint8_t dest)=0;
	virtual bool clearDPUser(uint8_t userID)=0;
	virtual bool lockDP()=0;
	virtual bool unlockDP()=0;

protected:
	virtual ~ProfileAppIF(){};
};

class loginUser;

class ProfileApp:public ProfileAppIF
{
public:
	enum enDefSetMode
	{
		dsPrepare,
		dsExecute,
		dsFinalize
	};

	ProfileApp(configIF & App);
	~ProfileApp();
	static ProfileApp & getInst(){return *m_pInstance;}

	//state driven requests
	actionState doSwitchToProfile(commInterface * requester,uintptr_t act, unsigned char ID);
	actionState doCreateProfile(commInterface & requester, uintptr_t act, const std::string& profileName, unsigned char imgID);
	actionState doLoadImage(commInterface & requester, uintptr_t act, const std::string & FileName,unsigned short width,unsigned short height,unsigned short slot,bool bAssign );
	actionState doCopyProfile(commInterface & requester,uintptr_t act, unsigned char srcID,unsigned char destID);
	actionState doDeleteProfile(commInterface & requester,uintptr_t act, unsigned char userId);

	//non state driven requests
	actionState doDeleteUserImage(unsigned short slot);
	actionState setProfileImage( unsigned char ImgID,unsigned char userID);
	actionState setProfileName(const std::string& profileName,unsigned char userID);
	void setPrivacyMode(uint32_t Mode);
	actionState setProfilePin(uint8_t Index,std::string Pin);
	actionState checkPin(uint8_t Index,std::string Pin);
	actionState setRememberMe(uint8_t Index);
	actionState unprotectProfile(uint8_t Index);
	actionState setlinkProfileStatus(bool isLinked);
	actionState setValetMode(commInterface & requester,uintptr_t act,bool enable);

	void storeLogoffTime();

	//getter / setter
	const std::vector<unsigned char>& getProfileImages(){return  m_Data.getProfileImages();}
	const std::vector<std::string>& getProfileNames(){return m_Data.getProfileNames();}
	const std::vector<profileMngr::enUserStatus>& getProfileTypes(){return m_Data.getProfileStatus();}
	uint8_t getCurrentUser(){return m_Data.m_CurrentProfile;}
	const std::vector<Client*> getClients(){return m_Clients;}
	bool isUserUnlocked(uint8_t Index);

	//initialization
	void AddInterface(commInterface * pPs);
	void updateUser(uint8_t newUser);

	void updateConfiguredProcessList(std::vector<std::string>& oProcesList);

	bool doDefSet(enDefSetMode Mode);

	// functions for client status updates
	void onClientStatus(uint32_t ClientID,ClientStatusUpdate status);
	uint32_t onNewClient(std::string & Name);

	void TraceRegisteredClientList();

	void handleSysPwrModeTrigger();

public:
	bool onTraceCmd(uint16_t Cmd,const unsigned char* pData);
	virtual bool setEarlyDPUser(uint8_t newUser);// for startup use case needs to be called by main()

	virtual void JobDone();

protected: //profielAppIF functions

	virtual const std::string & getClientName(uint8_t Index);
	virtual profileData & getData(){return m_Data;}
	virtual void startFailTimer(Job * pJob);
	virtual void loadUser(uint8_t newUser);

	virtual void sendPinRequest(unsigned char  bRequest);
	virtual void sendDataUpdate(dataChangeStatus);
	virtual void sendProfileID(uint8_t id);
	virtual void sendProfileStatus(profileState state);
	virtual void sendCopyProfile(uint8_t src,uint8_t dest);
	virtual void sendCreateProfile(uint8_t id);
	virtual void sendDeleteProfile(uint8_t id);
	virtual void sendActiveProfileType();


	virtual bool lockDP();
	virtual bool unlockDP();
	virtual bool clearDPUser(uint8_t userID);
	virtual bool setDPUser(uint8_t userID);
	virtual bool copyDPUser(uint8_t src,uint8_t dest);

protected:
	bool updateCurrentUser(uint8_t newUser);
	bool AddJob(Job* pJob);
	void lock(){sem_wait(&m_LockSem);}
	void unlock(){sem_post(&m_LockSem);}

private:
	static void  *timerproc(void * pParam);
	void updateVariantType();
	void trimStr(std::string& s);
	void cancelLogin(uint8_t Index);

	void operator = (ProfileApp & ref){};// by intention private
	ProfileApp(ProfileApp & ref):m_bDPLocked(false),m_dpSrvIf(ref.m_dpSrvIf),m_NumInterfaces(0),m_pInterfaces(0),m_Config(ref.m_Config),m_pCurrentJob(0),
			m_pLoginJob(0),m_VariantType(0),m_NewTimerID(0),m_pOngoingProfileJob(0),m_ClientConfiguration(*this){};// by intention private
private:
	std::vector<uint32_t> m_PinTimeStamp;
	bool m_bDPLocked;
	profileData m_Data;
	const dp_tclSrvIf & m_dpSrvIf;
	int m_NumInterfaces;
	commInterface ** m_pInterfaces;
	static ProfileApp *m_pInstance;
	configIF & m_Config;
	sem_t m_LockSem;
	uint32_t m_NewTimerID;
	uint8_t m_VariantType;
	std::vector<Client*> m_Clients;
	std::vector<std::string> m_IgnoredClients;//TODO:To be retained only during development
	Job * m_pCurrentJob;
	loginUser *m_pLoginJob;
	Job* m_pOngoingProfileJob;
	clientList m_ClientConfiguration;
	bool m_bStartupLoginChecked;

};





} // ::profileMngr

#endif
