/*!
*******************************************************************************
* \file               watchdog.c
*******************************************************************************
\verbatim
PROJECT:        IPCM
SW-COMPONENT:   IPCM
DESCRIPTION:    Implementation for watchdog
COPYRIGHT:      &copy; RBEI
HISTORY:
Date       | Author                   | Modifications
30.11.2017 | rhk6kor                  | Initial Version
\endverbatim
******************************************************************************/
#include "watchdog.h"

static guint hWatchDogTimerHandle = 0;
DLT_DECLARE_CONTEXT(IPCM_WATCHDOG);

gint uGetTimeOutInterval();
gboolean bWatchDogTimerCallback(gpointer data);

/******************************************************************************
** FUNCTION   : bWatchDogTimerCallback
*******************************************************************************
* \fn     bWatchDokgTimerCallback
* \brief  callback function triggered on watchdog timer expiry
* \param  None.
* \retval seconds.
******************************************************************************/
gboolean bWatchDogTimerCallback(gpointer data)
{
  DLT_LOG(IPCM_WATCHDOG,DLT_LOG_INFO,DLT_STRING("WD- bWatchDogTimerCallback"));
  gint RetVal = sd_notify(0, "WATCHDOG=1");
  if(RetVal < 0)
  {
     DLT_LOG(IPCM_WATCHDOG,DLT_LOG_ERROR, 
     DLT_STRING("WD- sd_notify Error: "),DLT_INT(RetVal));
  }
  else
  {
     DLT_LOG(IPCM_WATCHDOG, DLT_LOG_INFO,
     DLT_STRING("WD- sd_notify(WATCHDOG=1) Success"),DLT_INT(RetVal));
  }
  return TRUE;
}



/******************************************************************************
** FUNCTION   : uGetTimeOutInterval
*******************************************************************************
* \fn     uGetTimeOutValue
* \brief  Function to retrive the current watchdog interval time
* \param  None.
* \retval seconds.
******************************************************************************/
gint uGetTimeOutInterval()
{
  DLT_LOG(IPCM_WATCHDOG,DLT_LOG_INFO,DLT_STRING("WD- uGetTimeOutInterval"));
  gint uIntervalSec = 0;
  
  char* cpoSockPath = (char*)g_getenv("ISOINIT_SOCK_PATH");
  if(!cpoSockPath)
  {
	return uIntervalSec;
  }
  
  DLT_LOG(IPCM_WATCHDOG ,DLT_LOG_INFO,DLT_STRING("WD- ISOINIT_SOCK_PATH: "),DLT_STRING(cpoSockPath));

  int RetVal = setenv("NOTIFY_SOCKET",cpoSockPath,0);
  DLT_LOG(IPCM_WATCHDOG ,DLT_LOG_INFO,DLT_STRING("WD- Set NOTIFY_SOCKET: "),DLT_INT(RetVal));

  char* cpoWatchdogInterval = getenv("WATCHDOG_USEC");
  if(cpoWatchdogInterval)
  {
     uIntervalSec = atoi(cpoWatchdogInterval);
     DLT_LOG(IPCM_WATCHDOG ,DLT_LOG_INFO,DLT_STRING("WD- Watchdog interval: "),DLT_INT(uIntervalSec));
  }
  else
  {
	 DLT_LOG(IPCM_WATCHDOG ,DLT_LOG_ERROR,DLT_STRING("WD- Cannot retrieve Watchdog interval"));
  }
  return uIntervalSec;
}



/******************************************************************************
** FUNCTION   : bInitWatchDog
*******************************************************************************
* \fn     bInitWatchDog
* \brief  Function to initialize the watchdog
* \param  None.
* \retval gboolean.
******************************************************************************/
gboolean bInitWatchDog()
{
  DLT_REGISTER_CONTEXT(IPCM_WATCHDOG,"WTDG","IPCM WATCHDOG");
  DLT_LOG(IPCM_WATCHDOG,DLT_LOG_INFO,DLT_STRING("WD- bInitWatchDog"));
  
  if(hWatchDogTimerHandle)
    return FALSE;

  gint uIntervalSec = uGetTimeOutInterval();	//Getting the timeout interval
  if(!uIntervalSec)
    return FALSE;
 
  gint uWatchDogInterval = (gint)uIntervalSec/5000;
  if(!uWatchDogInterval)
    return FALSE;
  
  hWatchDogTimerHandle = g_timeout_add(uWatchDogInterval,
								bWatchDogTimerCallback,
                                NULL); 
  if(!hWatchDogTimerHandle)
    return FALSE;
  
  DLT_LOG(IPCM_WATCHDOG,DLT_LOG_INFO,
  DLT_STRING("WD- WATCHDOG started"),DLT_UINT(hWatchDogTimerHandle));

  int RetVal = sd_notify(0, "READY=1");
  if(RetVal < 0)
  {
     DLT_LOG(IPCM_WATCHDOG, DLT_LOG_ERROR, 
     DLT_STRING("WD- sd_notify Error: "),DLT_INT(RetVal));
     return FALSE;
  }
  else
  {
    DLT_LOG(IPCM_WATCHDOG,DLT_LOG_INFO,
    DLT_STRING("WD- sd_notify(READY=1) Success"),DLT_INT(RetVal));
  }
  return TRUE;
}



/******************************************************************************
** FUNCTION   : vStopWatchDog
*******************************************************************************
* \fn     vStopWatchDog
* \brief  Function to stop the watchdog
* \param  None.
* \retval gboolean.
******************************************************************************/
void vStopWatchDog()
{
  DLT_LOG(IPCM_WATCHDOG,DLT_LOG_INFO,DLT_STRING("WD- vStopWatchDog"));
  if(!hWatchDogTimerHandle)
    return;
  g_source_remove(hWatchDogTimerHandle);
  DLT_UNREGISTER_CONTEXT(IPCM_WATCHDOG);
}
