/*!
*******************************************************************************
* @file             : MessageDispatcher.c
*******************************************************************************
*  - PROJECT:       : IPCM
*  - SW-COMPONENT   : Message dispatcher for smart proxy
*  - DESCRIPTION    : Provides abstraction over MQTT transport protocol
*  - COPYRIGHT      : &copy; 2017 Robert Bosch Engineering & Business Solutions
*  - Documents      : Give link of relevant documents
*  - HISTORY
*
*  Date     | Name          | Version  | Modification
* ----------|---------------|----------|---------------------------------------
*05.10.2017 | Deepa Jose(RBEI/ECO2)    | 0.0.1    | Initial version
*10.01.2018 | Ashsih Kumar (RBEI/ECO2) | 0.0.2    | Scheduler modification
******************************************************************************/
//HEADERS
#include <stdbool.h>
#include <glib.h>
#include <stdlib.h>
#include <string.h>

// User defined headers
#include "SmartProxy.h"
#include "MessageDispatcher.h"
#include "agentplugin.h"

//DLT CONTEXT DECLARATION
DLT_DECLARE_CONTEXT(SMART_PROXY);

//THREAD VARIABLES
static pthread_t DispatcherThread;
static gboolean bRunDispatcherThread = FALSE;
static pthread_mutex_t mutexHash = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t mutexQueue = PTHREAD_MUTEX_INITIALIZER;
static pthread_cond_t  cond  = PTHREAD_COND_INITIALIZER;

//PRIORITY QUEUE
static GQueue *hPriorityQueue = NULL;
//PENDING HASHTABLE
static GHashTable* MessagePendingTable = NULL;


/**
*******************************************************************************
** FUNCTION   : sendMethodResponse
*******************************************************************************
* \fn     sendMethodResponse
* \brief  forwards the response received to DBus Communication manager
* \param  Message*
* \retval void.
******************************************************************************/
void display()
{
	if((NULL == hPriorityQueue) ||
			(g_queue_is_empty(hPriorityQueue)))
	{
		DLT_LOG(SMART_PROXY, DLT_LOG_INFO
		, DLT_STRING ("smartproxy: message_queue is empty"));
	}
	else
	{

		//        void prt(gpointer msg) {
		//         printf("%s   ", ((Task*)msg)->name);
		//        }

		//         g_queue_foreach(q, (GFunc)prt, NULL);
	}

}



/**
*******************************************************************************
** FUNCTION   : destroy_pendingmessage_hash_table
*******************************************************************************
* \fn     destroy_pendingmessage_hash_table
* \brief  forwards the response received to DBus Communication manager
* \param  Message*
* \retval void.
******************************************************************************/
void destroy_pendingmessage_hash_table()
{
	GHashTableIter iter;
	gchar* key;
	Message* value;
	g_hash_table_iter_init (&iter, MessagePendingTable);
	//    while (g_hash_table_iter_next (&iter, (gpointer)&key, (gpointer)&value))
	//    {
	//        if(NULL != value->header->sender)
	//            g_free((const char*)value->header->sender);

	//        if(NULL != value->header->intent)
	//                g_free(value->header->intent);

	//        if(NULL != value->header->name)
	//                g_free((const char*)value->header->name);

	//        g_object_unref (value->payload->params);
	//        free(value);
	//    }
	//    g_hash_table_destroy(MessagePendingTable);
	//    MessagePendingTable = NULL;
}

/**
*******************************************************************************
** FUNCTION   : sendMethodResponse
*******************************************************************************
* \fn     sendMethodResponse
* \brief  forwards the response received to DBus Communication manager
* \param  Message*
* \retval void.
******************************************************************************/
gboolean search(guint64 requestHandle)
{
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- search"));
	gboolean bMatchFound = FALSE;
	
	pthread_mutex_lock(&mutexHash);
	
	GHashTableIter iter;
	gchar* key;
	Message* value;

	g_hash_table_iter_init (&iter, MessagePendingTable);
	while (g_hash_table_iter_next (&iter, (gpointer)&key, (gpointer)&value))
	{
		DLT_LOG(SMART_PROXY, DLT_LOG_INFO
		, DLT_STRING("Message - req ID: ")
		, DLT_UINT32(value->header->requestHandle));

		if(requestHandle == value->header->requestHandle)
		{
			DLT_LOG(SMART_PROXY, DLT_LOG_INFO, DLT_STRING("match found"));
			bMatchFound = TRUE;
		}
	}
	pthread_mutex_unlock(&mutexHash);
	
	return bMatchFound;
}

/******************************************************************************
* Function:       vMessagePrint
* Description:    vMessagePrint
* Parameters:     (Message*)
* Return:         void
*****************************************************************************/
void vMessagePrint(Message* msg)
{
	if(NULL != msg)	
	{
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- vMessagePrint"));
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- MESSAGE HEADER REQUESTHANDLE "),DLT_UINT64(msg->header->requestHandle));
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- MESSAGE HEADER REQUESTTYPE "),DLT_UINT32(msg->header->messageType));
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- MESSAGE HEADER NAME"),DLT_STRING(msg->header->name));
		//DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- MESSAGE HEADER SENDER"),DLT_STRING(msg->header->sender));	//error CRASH NOT PRINTABLE
		char* Spayload = g_variant_print(msg->payload,TRUE);
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- PAYLOAD PARAMS"),DLT_STRING(Spayload)); 
	}
	else
	{
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- MESSAGE IS NULL"));
	}
}


/******************************************************************************
* Function:       Dequeue
* Description:    Dequeue
* Parameters:     (void)
* Return:         Message*
*****************************************************************************/
Message* Dequeue()
{
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- Dequeue"));
	Message* deletedNode = NULL;

	if((NULL == hPriorityQueue)||(g_queue_is_empty(hPriorityQueue)))
	{
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING ("SP- message_queue is empty"));
	}
	else
	{
		/*guint64 QueueLen = g_queue_get_length(hPriorityQueue);
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- QUEUE LEN BEFORE POP = "),DLT_UINT64(QueueLen));*/    
		
		deletedNode = g_queue_pop_head(hPriorityQueue);

		/*QueueLen = g_queue_get_length(hPriorityQueue);
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- QUEUE LEN AFTER POP = "),DLT_UINT64(QueueLen));*/
		
		DLT_LOG(SMART_PROXY, DLT_LOG_INFO,
		DLT_STRING ("SP- dequeued message id:"),
		DLT_UINT32(deletedNode->header->requestHandle));
	}
	return deletedNode;
}



/******************************************************************************
* Function:       Enqueue
* Description:    Enqueue
* Parameters:     (Message*)
* Return:         gboolean
*****************************************************************************/
gboolean Enqueue(Message* pMsgData)
{
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- Enqueue"));

	if(NULL != pMsgData && NULL != hPriorityQueue)
	{
		vMessagePrint(pMsgData);
		
		/*guint64 QueueLen = g_queue_get_length(hPriorityQueue);
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- QUEUE LEN BEFORE PUSH = "),DLT_UINT64(QueueLen));*/
		
		g_queue_push_tail(hPriorityQueue,pMsgData);
		
		/*QueueLen = g_queue_get_length(hPriorityQueue);
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- QUEUE LEN AFTER PUSH = "),DLT_UINT64(QueueLen));*/
		
		gint64 ui_msd_pos = g_queue_index(hPriorityQueue,pMsgData);                   
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),
		DLT_STRING("SP- message is inserted in the pos: "),
		DLT_INT64(ui_msd_pos));
		return TRUE;
	}
	else
	{
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- Message is NULL"));
		return FALSE;
	}
}

/******************************************************************************
* Function:       bHandleMessageRequest
* Description:    bHandleMessageRequest
* Parameters:     (Message*)
* Return:         gboolean
*****************************************************************************/
void bHandleMessageRequest(Message *pMsgData)
{
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- bHandleMessageRequest"));
	
	if(NULL != pMsgData)
	{
		pthread_mutex_lock(&mutexQueue);
		Enqueue(pMsgData);           	//Critical Section
		pthread_cond_signal(&cond);
		pthread_mutex_unlock(&mutexQueue);
	}
	else
	{
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- Message is NULL"));
	}
	
}

/******************************************************************************
* Function:       vpDispatcherRoutine
* Description:    vpDispatcherRoutine
* Parameters:     (void *)
* Return:         void
*****************************************************************************/
void *vpDispatcherRoutine(void *l_poThreadParams)
{
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- vpDispatcherRoutine"));
	bRunDispatcherThread = TRUE;
	
	while(bRunDispatcherThread)
	{
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- Waiting for messages"));
		
		pthread_mutex_lock(&mutexQueue);
		pthread_cond_wait(&cond,&mutexQueue);
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- Waiting Over"));
		Message* msg = Dequeue();
		vMessagePrint(msg);
		pthread_mutex_unlock(&mutexQueue);
		
		if(MT_REQUEST == msg->header->messageType)
		{
			DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_STRING("SP- Message Type is MT_REQUEST"));

			pthread_mutex_lock(&mutexHash);
			guint64 ReqHandelKey = msg->header->requestHandle;
			g_hash_table_insert(MessagePendingTable,&ReqHandelKey,msg);
			DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- PendingTable size "),DLT_UINT32(g_hash_table_size(MessagePendingTable)));
			pthread_mutex_unlock(&mutexHash);
			
			//Forward the method req to RPC Client
			HandleRequestFromSmartProxy(msg);
		}
		else if (MT_SIGNAL == msg->header->messageType)
		{
			DLT_LOG(SMART_PROXY, DLT_LOG_INFO,DLT_STRING("SP- Message Type is MT_SIGNAL"));
			vMessagePrint(msg);
			//Forward the signal to RPC Client
			HandleRequestFromSmartProxy(msg);
		}
		else
		{
			DLT_LOG(SMART_PROXY, DLT_LOG_INFO,DLT_STRING("SP- Message Type is INVALID"));
		}
	}
	
	//Destroying thread variable 
	pthread_cond_destroy(&cond);
	pthread_mutex_destroy(&mutexQueue);
	pthread_exit(NULL);
}



/******************************************************************************
* Function:       bInitMessageDispatcher
* Description:    Function to init Message Dispatcher
* Parameters:     void
* Return:         gboolean
*****************************************************************************/
gboolean bInitMessageDispatcher()
{
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- bInitMessageDispatcher"));
	
	//Queue Initialisation
	hPriorityQueue = g_queue_new();
	
	//Message PendingTable Initialisation
	MessagePendingTable = g_hash_table_new_full(g_int_hash, g_int_equal, g_free, g_free);	
	
	
	//Dispatcher Thread Initialisation
	int iDispatcherThreadRet = pthread_create(&DispatcherThread,
	NULL,
	*vpDispatcherRoutine,
	NULL);
	if(iDispatcherThreadRet)
	{
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- pthread creation failed "),
		DLT_INT(iDispatcherThreadRet));
		return FALSE;
	}
	else
	{
		DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- pthread created "),
		DLT_INT(iDispatcherThreadRet));
		return TRUE;
	}
}



/******************************************************************************
* Function:       vStopMessageDispatcher
* Description:    vStopMessageDispatcher
* Parameters:     (void)
* Return:         void
*****************************************************************************/
void vStopMessageDispatcher()
{
	DLT_LOG(SMART_PROXY,DLT_LOG_INFO,DLT_UINT32((unsigned int)pthread_self()),DLT_STRING("SP- vStopMessageDispatcher"));

	destroy_pendingmessage_hash_table();

	bRunDispatcherThread = FALSE;
}


