/*!
*******************************************************************************
* \file               rpcserver.c
*******************************************************************************
\verbatim
PROJECT:        IPCM
SW-COMPONENT:   IPCM
DESCRIPTION:    Implementation for RPC server
COPYRIGHT:      &copy; RBEI
HISTORY:
Date       | Author                   | Modifications
28.09.2017 | Naveen D R (RBEI/ECO2) | 0.0.1    | Initial version
\endverbatim
******************************************************************************/
#include "../pluginmanager/pluginmanager.h"
#include "atl_if.h"
#include "HelloService.h"

#include "rpcserver.h"
#include "jsoncutils.h"

#include "JsonRPC2_0Error.h"
#include "Jsonvalidationcutils.h"


DLT_DECLARE_CONTEXT(RPCSERVER);

static GHashTable* hash_servicelist = NULL;
static GHashTable* hash_requestmessagelist = NULL;

#define TIMEOUT 3
#define ID                     (tString)   "id"
#define METHOD             (tString)   "method"
#define PARAMS             (tString)   "params"


static guint64 counter = 0;

typedef void (*startTimerFun)(void *);
typedef struct
{
    gchar* appId;
    gchar* methodName;
    gint64 uRequestId;
    guint uTimerId;
    startTimerFun vTimerFun;

}requestMessage; 


/******************************************************************************
** FUNCTION   : start_Timer
*******************************************************************************
* \fn     start_Timer
* \brief  Function to start timer
* \param  None.
* \retval userdata.
******************************************************************************/
void start_Timer(void* fcounter);

/******************************************************************************
** FUNCTION   : Timeout_CallBack
*******************************************************************************
* \fn     Timeout_CallBack
* \brief  Function to handle timer callback
* \param  None.
* \retval userdata.
******************************************************************************/
gboolean Timeout_CallBack(gpointer user_data);
/******************************************************************************
** FUNCTION   : HandleResponseFromPlugin
*******************************************************************************
* \fn     HandleResponseFromPlugin
* \brief  Function to handle response from plugins
* \param  GVariant,GError,gint64.
* \retval None.
******************************************************************************/
tVoid HandleResponseFromPlugin(GVariant*, GError**,gint64, eTopicType);

/******************************************************************************
** FUNCTION   : bValidatePayload
*******************************************************************************
* \fn     bValidatePayload
* \brief  Function to validate method request from ATL with plugin. 
* \param  None.
* \retval gboolean.
******************************************************************************/

gboolean bValidatePayload(tCString arg_key);

/******************************************************************************
** FUNCTION   : vfree_requestmessage_hashdata
*******************************************************************************
* \fn     vfree_requestmessage_hashdata
* \brief  Function to destroy request message
* \param  gpointer.
* \retval None.
******************************************************************************/
void vfree_requestmessage_hashdata(gpointer data)
{
	DLT_LOG(RPCSERVER, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	requestMessage * l_value_messageItem = (requestMessage*)data;
	g_free(l_value_messageItem->appId);
	g_free(l_value_messageItem->methodName);
	g_free(l_value_messageItem);
}

/******************************************************************************
** FUNCTION   : vfree_servicelist_hashdata
*******************************************************************************
* \fn     vfree_servicelist_hashdata
* \brief  Function to destroy service list
* \param  gpointer.
* \retval None.
******************************************************************************/
void vfree_servicelist_hashdata (gpointer data)
{
	DLT_LOG(RPCSERVER, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	g_free(data);
}


/******************************************************************************
** FUNCTION   : bInitRPCServer
*******************************************************************************
* \fn     bInitRPCServer
* \brief  Function to init RPC server
* \param  None.
* \retval gboolean.
******************************************************************************/
gboolean bInitRPCServer()
{
    DLT_REGISTER_CONTEXT(RPCSERVER,"RPCS","RPCC context for logging");
	DLT_LOG(RPCSERVER, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	
	//ServiceList Hash
    hash_servicelist = g_hash_table_new_full(g_str_hash,g_str_equal,NULL
                                             ,vfree_servicelist_hashdata);
	//RequestList Hash										 
    hash_requestmessagelist = g_hash_table_new_full(g_direct_hash,g_direct_equal
                                                    ,NULL,vfree_requestmessage_hashdata);
    //Register to ATL
	iATL_Register(RPC_SERVER,&iHandleATLResponse);
    return TRUE;
}


/******************************************************************************
** FUNCTION   : vCleanUpRPCServer
*******************************************************************************
* \fn     vCleanUpRPCServer
* \brief  Function to cleanup rpcserver
* \param  None.
* \retval void.
******************************************************************************/
tVoid vCleanUpRPCServer()
{
    DLT_LOG(RPCSERVER, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    g_hash_table_destroy (hash_servicelist);
    g_hash_table_destroy (hash_requestmessagelist);
}

/******************************************************************************
** FUNCTION   : vRegister_RPCServer
*******************************************************************************
* \fn     vRegister_RPCServer
* \brief  Function to send registration request to ATL
* \param  None.
* \retval service.
******************************************************************************/
tVoid vRegister_RPCServer  (service ** services_list_arg,tInt services_len)
{
    DLT_LOG(RPCSERVER, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    DLT_LOG(RPCSERVER ,DLT_LOG_INFO,DLT_STRING("service length :")
            ,DLT_INT(services_len));
    for(tInt n=0;n < services_len;n++)
    {
        DLT_LOG(RPCSERVER ,DLT_LOG_INFO,DLT_STRING("service name : \n")
                ,DLT_STRING( (*services_list_arg[n]).service));
        if(!g_hash_table_lookup(hash_servicelist, (*services_list_arg[n]).service))
        {
            DLT_LOG(RPCSERVER ,DLT_LOG_INFO,
                    DLT_STRING("IN- IPCM- Entry not available in hash table"));
            g_hash_table_insert (hash_servicelist,(*services_list_arg[n]).service
                                 ,services_list_arg[n]);

        }
        else{
            DLT_LOG(RPCSERVER ,DLT_LOG_INFO
                    ,DLT_STRING("IN- IPCM- Entry already available in hash table"));

        }

    }


}

/******************************************************************************
** FUNCTION   : vGetServiceList
*******************************************************************************
* \fn     vGetServiceList
* \brief  Function to get service list
* \param  None.
* \retval service struct.
******************************************************************************/
void* vGetServiceList(tUInt size) //for hello service
{
    DLT_LOG(RPCSERVER, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    GHashTableIter iter; guint64 *key;
    service* value;
    tInt loop_itr =0;
    service *l_servicelist = g_new0(service, 1);
    g_hash_table_iter_init(&iter, hash_servicelist);

    while(g_hash_table_iter_next (&iter, (gpointer)&key,(gpointer)&value))
    {
        l_servicelist = g_renew(service, l_servicelist, (loop_itr+1));

        l_servicelist[loop_itr].service	= value->service;
        l_servicelist[loop_itr].major_version = value->major_version;
        l_servicelist[loop_itr].minor_version = value->minor_version;
        l_servicelist[loop_itr].handleE2EMessageCBAsync = NULL;

        DLT_LOG(RPCSERVER ,DLT_LOG_INFO,DLT_STRING("service name : \n")
                ,DLT_STRING( l_servicelist[loop_itr].service));
        DLT_LOG(RPCSERVER ,DLT_LOG_INFO,DLT_STRING("major_version : \n")
                ,DLT_INT(l_servicelist[loop_itr].major_version));

        loop_itr++;
    }
    size = loop_itr;

    return l_servicelist;
}

/******************************************************************************
** FUNCTION   : iHandleATLResponse
*******************************************************************************
* \fn     iHandleATLResponse
* \brief  Function to handle senddestination request from atl
* \param  None.
* \retval none.
******************************************************************************/
tVoid iHandleATLResponse(tCString arg_appid,tCString arg_payload)
{
    DLT_LOG(RPCSERVER, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    DLT_LOG(RPCSERVER, DLT_LOG_INFO,DLT_STRING("RPCSERVER- JSON Messsage= "),DLT_STRING(arg_payload));
	
	
	JsonRPC2_0Err l_eJsonRPC2_0Err;
    l_eJsonRPC2_0Err  = validateJson(arg_payload);
	
	if(l_eJsonRPC2_0Err != NOERROR)	
	{
		DLT_LOG(RPCSERVER,DLT_LOG_INFO,DLT_STRING("RPCSERVER- JSON Validation failed"));
		const char* l_ErrorResponse = sConstruct_ErrorResponse(l_eJsonRPC2_0Err,NULL);

		//Sending Error Response to ATL
		iATL_SendMessage(RPC_SERVER,arg_appid,l_ErrorResponse, IPCM_DEFAULT_TOPIC);
	}
	else
	{		
		tCString l_methodname = sParseJsonStr(arg_payload,METHOD);
		DLT_LOG(RPCSERVER,DLT_LOG_INFO,DLT_STRING("IPCM- parsed string =")
				,DLT_STRING(l_methodname));
		DLT_LOG(RPCSERVER,DLT_LOG_INFO,DLT_STRING("IPCM- arg_appid =")
				,DLT_STRING(arg_appid));
		DLT_LOG(RPCSERVER ,DLT_LOG_INFO,DLT_STRING("length of parsed string")
				,DLT_INT(strlen(l_methodname)));
		if(!bValidatePayload(l_methodname))
		{
			DLT_LOG(RPCSERVER ,DLT_LOG_INFO,DLT_STRING("method name is invalid"));
			return;
		}
		else{
			DLT_LOG(RPCSERVER ,DLT_LOG_INFO,DLT_STRING("valid method"));
		}
		tString convertstr;
		tString sRequestId = strdup(sParseJsonStr(arg_payload,ID));
		convertstr = strtok(sRequestId, "\"");
		DLT_LOG(RPCSERVER,DLT_LOG_INFO,DLT_STRING("IPCM- sId=")
				,DLT_STRING(convertstr));
		
		gint64 l_RequestId = g_ascii_strtoll (convertstr, NULL, 10);//validation
		
		DLT_LOG(RPCSERVER,DLT_LOG_INFO,DLT_STRING("IPCM- RequestId=")
				,DLT_INT(l_RequestId));
		
		if(!g_hash_table_lookup(hash_requestmessagelist, GINT_TO_POINTER (l_RequestId)))
		{
			DLT_LOG(RPCSERVER ,DLT_LOG_INFO
					,DLT_STRING("IN- Adding New request to hash_requestmessagelist table"));
			requestMessage* value_messageItem  =
					(requestMessage*)malloc(sizeof(requestMessage));
			memset((void*)value_messageItem, 0 , sizeof(requestMessage));
			
			value_messageItem->appId =   g_strdup(arg_appid);
			value_messageItem->methodName =   g_strdup(l_methodname);
			value_messageItem->uTimerId = 0;
			value_messageItem->uRequestId =l_RequestId;
			value_messageItem->vTimerFun = &start_Timer;
			value_messageItem->vTimerFun(value_messageItem) ;
			DLT_LOG(RPCSERVER,DLT_LOG_INFO,DLT_STRING("uRequestId =")
					,DLT_UINT64( value_messageItem->uRequestId));
		
			g_hash_table_insert( hash_requestmessagelist
						,GINT_TO_POINTER (l_RequestId),value_messageItem);		
			
				
			tCString l_sParsedString =  sParseJsonStr(arg_payload,PARAMS);
			DLT_LOG(RPCSERVER,DLT_LOG_INFO,DLT_STRING("RPCSERVER- JSON Messsage parsed string =")
					,DLT_STRING(l_sParsedString));
			GVariant* g_variant = sJsontoGvariant(l_sParsedString);
			char* Spayload = g_variant_print(g_variant,TRUE);
			DLT_LOG(RPCSERVER,DLT_LOG_INFO,DLT_STRING("RPCSERVER- JSON2Gvariant ="),DLT_STRING(Spayload)); //Payload Print 		
			
			if(g_hash_table_lookup(hash_servicelist,l_methodname))
			{
				DLT_LOG(RPCSERVER ,DLT_LOG_INFO
						,DLT_STRING("IN- IPCM- Entry available in hash table"));
				service * l_poservice_list  =g_new0(service, 1);
				l_poservice_list	= g_hash_table_lookup(hash_servicelist,l_methodname);
				if(l_poservice_list)
				{
					DLT_LOG(RPCSERVER,DLT_LOG_INFO,DLT_STRING("request to plugin"));
					HandleE2EMessageAsync f_HandleE2EMessageAsync
							= l_poservice_list->handleE2EMessageCBAsync;
					GError *l_poError = NULL;
					DLT_LOG(RPCSERVER,DLT_LOG_INFO,DLT_STRING("RPC SERVER PLUGIN- REQUEST ID METHOD CALL 0="),DLT_INT(l_RequestId));
					f_HandleE2EMessageAsync(g_variant,NULL,&l_poError
											,&HandleResponseFromPlugin,l_RequestId);
					//g_free(l_poservice_list);
				}

			}
			else
			{
				DLT_LOG(RPCSERVER ,DLT_LOG_INFO
						,DLT_STRING("IN- IPCM- Entry not available in hash table"));

			}
		}
		else
		{
			DLT_LOG(RPCSERVER ,DLT_LOG_INFO
					,DLT_STRING("Entry already available in hash_requestmessagelist"));
		}
		//Free Respo ID
		free(sRequestId);
	}
}



/******************************************************************************
** FUNCTION   : HandleResponseFromPlugin
*******************************************************************************
* \fn     HandleResponseFromPlugin
* \brief  Function to handle response from plugins
* \param  None.
* \retval GVariant,GError,gint64.
******************************************************************************/
tVoid HandleResponseFromPlugin( GVariant*  var, 
                                GError**   err,
                                gint64     request_id,
                                eTopicType topicType )
{
    DLT_LOG(RPCSERVER, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    requestMessage *  l_porequestMessage =
            g_hash_table_lookup(hash_requestmessagelist, GINT_TO_POINTER (request_id));
    if(l_porequestMessage)
    {
        DLT_LOG(RPCSERVER ,DLT_LOG_INFO
                ,DLT_STRING("Entry available in hash_requestmessagelist table"));
        DLT_LOG(RPCSERVER,DLT_LOG_INFO,DLT_STRING("Received Appid =")
                ,DLT_STRING(l_porequestMessage->appId));
				
        const char* converted_JSON = sGvarianttoJson(var);
		DLT_LOG(RPCSERVER,DLT_LOG_INFO,DLT_STRING("IPCM- converted json =")
                ,DLT_STRING(converted_JSON));
		const char* JSON_Response  = sConstruct_Response(converted_JSON,l_porequestMessage->uRequestId);
		
        DLT_LOG(RPCSERVER,DLT_LOG_INFO,DLT_STRING("IPCM- json response =")
                ,DLT_STRING(JSON_Response));

        iATL_SendMessage(RPC_SERVER,l_porequestMessage->appId,JSON_Response, topicType);
        g_source_remove(l_porequestMessage->uTimerId);
        if(g_hash_table_remove(hash_requestmessagelist
                               ,GINT_TO_POINTER (request_id)))
        {
            DLT_LOG(RPCSERVER ,DLT_LOG_INFO,DLT_STRING("Entry removed"));

        }
    }
    else
    {
        DLT_LOG(RPCSERVER ,DLT_LOG_INFO,DLT_STRING("IN-pointer is NULL"));

    }

}
/******************************************************************************
** FUNCTION   : start_Timer
*******************************************************************************
* \fn     start_Timer
* \brief  Function to start timer
* \param  None.
* \retval userdata.
******************************************************************************/
void start_Timer(void* value_messageItem)
{
    DLT_LOG(RPCSERVER, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    if((requestMessage *) value_messageItem)
    {
        DLT_LOG(RPCSERVER,DLT_LOG_INFO,DLT_STRING("method name =")
                ,DLT_STRING(((requestMessage *) value_messageItem)->methodName));
        ((requestMessage *) value_messageItem)->uTimerId
                = g_timeout_add_seconds (TIMEOUT
                                         , Timeout_CallBack
                                         , GINT_TO_POINTER(value_messageItem));
    }
}

/******************************************************************************
** FUNCTION   : Timeout_CallBack
*******************************************************************************
* \fn     Timeout_CallBack
* \brief  Function to handle timer callback
* \param  None.
* \retval userdata.
******************************************************************************/
gboolean Timeout_CallBack(gpointer user_data)
{
   DLT_LOG(RPCSERVER, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    if((requestMessage *) user_data)
    {
        DLT_LOG(RPCSERVER,DLT_LOG_INFO,DLT_STRING("uRequestId =")
                ,DLT_UINT64(((requestMessage *) user_data)->uRequestId));
        gint64 l_uRequestId = ((requestMessage *) user_data)->uRequestId;
        if(g_hash_table_lookup(hash_requestmessagelist
                               , GINT_TO_POINTER (l_uRequestId)))
        {
            DLT_LOG(RPCSERVER ,DLT_LOG_INFO
                    ,DLT_STRING("Entry available in hash_requestmessagelist table"));
            if(g_hash_table_remove(hash_requestmessagelist
                                   ,GINT_TO_POINTER (l_uRequestId)))
            {
                DLT_LOG(RPCSERVER ,DLT_LOG_INFO,DLT_STRING("Entry removed"));

            }
        }
    }
    else{
        DLT_LOG(RPCSERVER,DLT_LOG_INFO,DLT_STRING("user_data is NULL"));
    }
    return FALSE;// to make timer singleshot
}

/******************************************************************************
** FUNCTION   : bValidatePayload
*******************************************************************************
* \fn     bValidatePayload
* \brief  Function to validate method request from ATL with plugin. 
* \param  None.
* \retval gboolean.
******************************************************************************/
gboolean bValidatePayload(tCString arg_key)
{	
    DLT_LOG(RPCSERVER, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
    DLT_LOG(RPCSERVER ,DLT_LOG_INFO,DLT_STRING(arg_key));
    if(g_hash_table_lookup(hash_servicelist, arg_key))
    {
        DLT_LOG(RPCSERVER ,DLT_LOG_INFO,DLT_STRING("IN- Entry is available"));
        return TRUE;
    }
    else{
        DLT_LOG(RPCSERVER ,DLT_LOG_INFO,DLT_STRING("IN- Entry is not available"));
        return FALSE;
    }

}

