/*!
*******************************************************************************
* @file             : pluginmanager.c
*******************************************************************************
*  - PROJECT:       : IPCM
*  - SW-COMPONENT   : Plugin Manager
*  - DESCRIPTION    : Plugin Manager loads and validates the plugins 
*  - COPYRIGHT      : &copy; 2017 Robert Bosch Engineering & Business Solutions
*  - Documents      : Give link of relevant documents
*  - HISTORY
*
*  Date     | Name          | Version  | Modification
* ----------|---------------|----------|---------------------------------------
* 8.11.2017 | Ashsih Kumar (RBEI/ECO2) | 0.0.1    | Initial version
******************************************************************************/
//HEADERS
#include "pluginmanager.h"
#include "ipcmversion.h"

//DLT CONTEXT DECLARATION
DLT_DECLARE_CONTEXT(PLGM);

//Variables
static GArray* poPluginList = NULL;
static gboolean bInitDone = FALSE;
func_Init bInit = NULL;
func_Deinit bDeinit = NULL;
char* pluginsDir = NULL;

/******************************************************************************
* Function:       poLoadDynamicLibrary
* Description:    poLoadDynamicLibrary
* Parameters:     (const gchar*)
* Return:         GModule*
*****************************************************************************/
GModule* poLoadDynamicLibrary(const gchar* filename)
{   
	DLT_LOG(PLGM,DLT_LOG_INFO,DLT_STRING("PluginManager- poLoadDynamicLibrary"));
	DLT_LOG(PLGM,DLT_LOG_INFO,DLT_STRING("PluginManager- File name "),DLT_STRING(filename));	
	DLT_LOG(PLGM,DLT_LOG_INFO,DLT_STRING("PluginManager- File path "),DLT_STRING(pluginsDir));
	
	GModule* module = NULL;
	filename = g_strjoin(NULL,pluginsDir,filename,NULL);
	DLT_LOG(PLGM,DLT_LOG_INFO,DLT_STRING("PluginManager- File name with path "),DLT_STRING(filename));	
		
	module = g_module_open(filename,G_MODULE_BIND_LAZY);
	if (!module)
	{
		DLT_LOG(PLGM,DLT_LOG_INFO,
		DLT_STRING("PluginManager- Plugin loading failed"),
		DLT_STRING(g_module_error()));
		return NULL;
	}
	//g_free(filename);
	return module;
}


/******************************************************************************
* Function:       bLoadPlugins
* Description:    Load plugins from the Directory path
* Parameters:     (void)
* Return:         gboolean
*****************************************************************************/
gboolean bLoadPlugins()
{
	DLT_LOG(PLGM,DLT_LOG_INFO,DLT_STRING("PluginManager- bLoadPlugins"));

	if(bInitDone)
	{
		DLT_LOG(PLGM,DLT_LOG_INFO,
		DLT_STRING("PluginManager-  Plugins already loaded"));
		return FALSE;  
	}

	//ENVIRONMENT VARIABLE
	pluginsDir = (char*)getenv("IPCM_PLUGINS_DIR");
	if((pluginsDir == NULL)||(!g_strcmp0 (pluginsDir,"")))
	{
		DLT_LOG(PLGM,DLT_LOG_INFO,DLT_STRING("PluginManager- Plugins Directory is NULL"));
		return FALSE; 
	}
	DLT_LOG(PLGM,DLT_LOG_INFO,DLT_STRING("Environment Variable value"),DLT_STRING(pluginsDir));

	GDir   *directory = NULL;
	GError *error = NULL;
	directory = g_dir_open(pluginsDir,0,&error);
	if(!directory)
	{
		DLT_LOG(PLGM,DLT_LOG_INFO,
		DLT_STRING("PluginManager- Failed to open plugin directory"),
		DLT_STRING(error->message));
		g_clear_error(&error);
		return FALSE;
	}
	else
	{
		const gchar *filename = NULL;
		func_Init bInit = NULL;
		poPluginList = g_array_new(FALSE,FALSE,(guint)sizeof(GModule*));
		
		while ((filename = g_dir_read_name(directory)))
		{
			DLT_LOG(PLGM,DLT_LOG_INFO,DLT_STRING("PluginManager- Filename = "),
			DLT_STRING(filename));
			GModule* poModule = poLoadDynamicLibrary(filename);
			if(poModule)
			{
				if(g_module_symbol(poModule,IPCM_PLUGIN_FUNC_INIT,(gpointer *)&bInit))
				{
					DLT_LOG(PLGM,DLT_LOG_INFO,
					DLT_STRING("PluginManager- Got function reference bInit"),
					DLT_STRING(g_module_name(poModule)));
					if(bInit(IPCM_VERSION))
					{
						DLT_LOG(PLGM,DLT_LOG_INFO,
						DLT_STRING("PluginManager- bInit sucess"),
						DLT_STRING(g_module_name(poModule)));
						
						//Adding module to plugin list
						g_array_append_val(poPluginList,poModule);
					}
					else
					{
						DLT_LOG(PLGM,DLT_LOG_INFO,
						DLT_STRING("PluginManager- bInit failed"),
						DLT_STRING(g_module_name(poModule)));
					}
				}
				else
				{
					DLT_LOG(PLGM,DLT_LOG_INFO,
					DLT_STRING("PluginManager- bInit reference not found"),
					DLT_STRING(g_module_name(poModule)),
					DLT_STRING(g_module_error()));
				}
			}
			else
			{
				// DLT_LOG(PLGM,DLT_LOG_INFO,
				// DLT_STRING("PluginManager- Plugin loading failed"),
				// DLT_STRING(g_module_name(poModule)));
				return FALSE;
			}
		}
		bInit = NULL;
		bInitDone = TRUE;
		g_dir_close(directory);
		return TRUE;
	}
}


/******************************************************************************
* Function:       bInitPluginmanager
* Description:    Plugin Manager initialization
* Parameters:     (void)
* Return:         gboolean
*****************************************************************************/
gboolean bInitPluginmanager()
{
	DLT_REGISTER_CONTEXT(PLGM,"PLGM","IPCM PLUGIN MANAGER");
	DLT_LOG(PLGM,DLT_LOG_INFO,DLT_STRING("PluginManager- bInitPluginmanager"));

	//Checks GModule can be loaded or not
	if(!g_module_supported())
	{
		DLT_LOG(PLGM,DLT_LOG_INFO,DLT_STRING("PluginManager- GModule not supported"));
		return FALSE;
	}
	else
	{
		//Loading of Hello Service Plugin
		if(bInit_HelloService())
		{
			DLT_LOG(PLGM,DLT_LOG_INFO,
			DLT_STRING("PluginManager- HelloService plugin loaded"));
		}
		else
		{
			DLT_LOG(PLGM,DLT_LOG_INFO,
			DLT_STRING("PluginManager- HelloService plugin failed to loaded"));
		}
		
		//Loading of plugins 
		if(bLoadPlugins())
		{
			DLT_LOG(PLGM,DLT_LOG_INFO,
			DLT_STRING("PluginManager- Plugins loaded"));
		}
		else
		{
			DLT_LOG(PLGM,DLT_LOG_INFO,
			DLT_STRING("PluginManager- Plugins loading failed"));
		}
		return TRUE;
	}
	return TRUE;
}


/******************************************************************************
* Function:       vStopPlugins
* Description:    Stop & unload the Plugins 
* Parameters:     (void)
* Return:         void
*****************************************************************************/
void vStopPlugins()
{     
	DLT_LOG(PLGM,DLT_LOG_INFO,DLT_STRING("PluginManager- vStopPlugins"));
	guint16 i =0;
	GModule* poModule = NULL;

	for(i=0;i<poPluginList->len;i++)
	{
		poModule = g_array_index(poPluginList,GModule*,i);
		if(poModule)
		{
			if(g_module_symbol(poModule,IPCM_PLUGIN_FUNC_DEINIT,(gpointer *)&bDeinit))
			{
				DLT_LOG(PLGM,DLT_LOG_INFO,
				DLT_STRING("PluginManager- Got function reference bDeinit"),
				DLT_STRING(g_module_name(poModule)));
				if(bDeinit())
				{
					DLT_LOG(PLGM,DLT_LOG_INFO,
					DLT_STRING("PluginManager- bDeinit sucess"),
					DLT_STRING(g_module_name(poModule)));
				}
				else
				{
					DLT_LOG(PLGM,DLT_LOG_INFO,
					DLT_STRING("PluginManager- bDeinit failed"),
					DLT_STRING(g_module_name(poModule)));
				}
			}
			else
			{
				DLT_LOG(PLGM,DLT_LOG_INFO,
				DLT_STRING("PluginManager- bDeinit reference not found"),
				DLT_STRING(g_module_name(poModule)),
				DLT_STRING(g_module_error()));
			}
		}
		g_module_close(poModule);
		poModule = NULL;
		bDeinit = NULL;
	}
	g_array_free(poPluginList,TRUE);
	poPluginList = NULL;
	bInitDone=FALSE;	 
}


/******************************************************************************
* Function:       vDeinitPluginManager
* Description:    Deinitialise the PluginManager
* Parameters:     (void)
* Return:         void
*****************************************************************************/
void vDeinitPluginManager()
{
	DLT_LOG(PLGM,DLT_LOG_INFO,DLT_STRING("PluginManager- vDeinitPluginManager"));

	//Exiting HelloService
	vExit();

	//Exiting up the plugins
	if(!bInitDone)
	{
		DLT_LOG(PLGM,DLT_LOG_INFO,
		DLT_STRING("PluginManager- Plugins not loaded yet"));
	}
	else
	{
		vStopPlugins();
	}
}

