/*!
*******************************************************************************
* \file               dbuscommmanager.h
*******************************************************************************
\verbatim
PROJECT:        IPCM
SW-COMPONENT:   IPCM
DESCRIPTION:    Header for DBUS Communication manager
COPYRIGHT:      &copy; RBEI
HISTORY:
Date       | Author                   | Modifications
28.09.2017 | Naveen D R (RBEI/ECO2) | 0.0.1    | Initial version
\endverbatim
******************************************************************************/
#include "dbuscommmanager.h"

DLT_DECLARE_CONTEXT(DBUS_MR);


static GList* g_list_interfaceagents = NULL;

//struct for maintaining dbus interface agents
struct _interface_agents 
{
	tString bus_name;
	tString obj_path;
	GDBusInterfaceInfo *interface_info;
	GHashTable* hash_subscribed_signals;
	GenericMethodCall f_genericmethod;
	RegisterClient_cb f_registerclient_cb;
	guint registration_id;
	GDBusProxy *proxy;
	GDBusConnection *poconnection_object;
	guint owner_id;
	guint  watcher_id;
	gboolean binterface_has_signal;
};

//struct for maintaining subscription_ID for signal registration
typedef struct 
{
	dbus_signal_cb DbusresponseCB;
	guint subscription_ID;
} signal_info;

/******************************************************************************
** FUNCTION   : handle_method_call
*******************************************************************************
* \fn     handle_method_call
* \brief  Function to handle server method callback.
* \param  None.
* \retval void.
******************************************************************************/ 
static void handle_method_call (GDBusConnection       *connection,
const gchar           *sender,
const gchar           *object_path,
const gchar           *interface_name,
const gchar           *method_name,
GVariant              *parameters,
GDBusMethodInvocation *invocation,
gpointer               user_data)
{
	DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));

	DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING(method_name));

	//GVariantBuilder *builder;
	//GVariant *info_toSend =NULL;
	//builder = g_variant_builder_new(G_VARIANT_TYPE("a{sv}"));
	// gchar *service_name = "",*params = "";
	// g_variant_get(parameters, "(ss)", &service_name, &params);
	// DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("IPCM- service_name"),DLT_STRING(service_name));
	// DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("IPCM- params"),DLT_STRING(params));
	// g_variant_builder_add (builder, "{sv}", service_name,g_variant_new("s",service_name));
	// g_variant_builder_add (builder, "{sv}", params,g_variant_new("s",params));
	// info_toSend = g_variant_builder_end (builder);
	// g_variant_builder_unref (builder);

	if(!strcmp(method_name,"invokeSmartService"))
	{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("IPCM- invokeSmartService"));
		interface_agents* l_interface_agents = (interface_agents*) user_data;
		if(l_interface_agents != NULL)
		{
			l_interface_agents->f_genericmethod(sender,method_name,
			parameters
			,(void*)invocation);
		}
	}

}

static const GDBusInterfaceVTable interface_vtable =
{
	handle_method_call
};

/******************************************************************************
** FUNCTION   : vCreate_interface_agents
*******************************************************************************
* \fn     vCreate_interface_agents
* \brief  function to initilise interface agents struct
* \param  None.
* \retval interface agents.
******************************************************************************/

interface_agents* vCreate_interface_agents();

/******************************************************************************
** FUNCTION   : bUpdateInterfaceAgents
*******************************************************************************
* \fn     bUpdateInterfaceAgents
* \brief  function to get all interface agents
* \param  interface agents.
* \retval interface agents.
******************************************************************************/
gboolean bUpdateInterfaceAgents(tString arg_xml
, interface_agents * l_interface_agents);

/******************************************************************************
** FUNCTION   : vHandle_dbus_signal
*******************************************************************************
* \fn     vHandle_dbus_signal
* \brief  handler function for signal emit from AGW
* \param  interface agents.
* \retval None.
******************************************************************************/
static tVoid  vHandle_dbus_signal (GDBusConnection *connection,
const gchar *sender_name,
const gchar *object_path,
const gchar *interface_name,
const gchar *signal_name,
GVariant *parameters,
gpointer user_data);

/******************************************************************************
** FUNCTION   : vMethodAsyncCallback
*******************************************************************************
* \fn     vMethodAsyncCallback
* \brief  Function to get call back for method request to AGW
* \param  None.
* \retval None.
******************************************************************************/
tVoid vMethodAsyncCallback(GObject *source_object, GAsyncResult *res
, gpointer user_data);

/******************************************************************************
** FUNCTION   : vSendAsyncMethodRequest
*******************************************************************************
* \fn     vSendAsyncMethodRequest
* \brief  Function to send method request to AGW
* \param  None.
* \retval None.
******************************************************************************/

tVoid vSendAsyncMethodRequest( interface_agents* arg_agents
,tString methodname
,GVariant* g_methodRequestparams
,GAsyncReadyCallback vMethodAsyncCallback
,method_request_info* request_info);


/******************************************************************************
** FUNCTION   : vlookupinterfaces
*******************************************************************************
* \fn     vlookupinterfaces
* \brief  Function to fetch methods,signals and properties from the interfaces
* \param  None.
* \retval char*.
******************************************************************************/	
tVoid vlookupinterfaces(tCString xml);


/******************************************************************************
** FUNCTION   : bInitDBusCommMgr
*******************************************************************************
* \fn     bInitDBusCommMgr
* \brief  Function to initialize the DBUS communication manager
* \param  None.
* \retval gboolean.
******************************************************************************/
gboolean bInitDBusCommMgr()
{
	DLT_REGISTER_CONTEXT(DBUS_MR,"DCMR","IPCM DBUS COMMUNICATION MANAGER");
	DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	return TRUE;


}

/******************************************************************************
** FUNCTION   : vfree_hash_data
*******************************************************************************
* \fn     vfree_hash_data
* \brief  Function to clear hash data
* \param  gpointer.
* \retval None.
******************************************************************************/
void vfree_hash_data (gpointer data)
{
	DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	g_free (data);
}


/******************************************************************************
** FUNCTION   : vStopDBusCommMgr
*******************************************************************************
* \fn     vStopDBusCommMgr
* \brief  Function to cleanup DBus communication manager.
* \param  None.
* \retval void.
******************************************************************************/
tVoid vStopDBusCommMgr()
{
	DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	g_list_free_full (g_list_interfaceagents, g_object_unref);
}

/******************************************************************************
** FUNCTION   : GenericMethodResponse
*******************************************************************************
* \fn     GenericMethodResponse
* \brief  Function to handle Method response.
* \param  None.
* \retval void.
******************************************************************************/ 
tVoid GenericMethodResponse(const void* request_handle,GVariant *parameters,GError** err)
{
	DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));

	if(err != NULL)
	{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING(
		"Error in creating proxy"));
	}
	else if((parameters != NULL)&&(request_handle != NULL))
	{
		char* sPayload = g_variant_print(parameters,TRUE);
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("DBCM- RESPONSE PAYLOAD ")
		,DLT_STRING(sPayload));

		g_dbus_method_invocation_return_value((GDBusMethodInvocation*)request_handle,parameters);
		//g_variant_unref (sPayload);
	}
}

/******************************************************************************
** FUNCTION   : vCreate_interface_agents
*******************************************************************************
* \fn     vCreate_interface_agents
* \brief  function to initilise interface agents struct
* \param  None.
* \retval interface agents.
******************************************************************************/

interface_agents* vCreate_interface_agents()
{
	DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	interface_agents * l_interface_agents =NULL;
	l_interface_agents =g_new0(interface_agents, 1);
	l_interface_agents->bus_name = NULL;
	l_interface_agents->obj_path = NULL;
	l_interface_agents->proxy = NULL;
	l_interface_agents->interface_info = NULL;
	l_interface_agents->hash_subscribed_signals = NULL;
	l_interface_agents->f_genericmethod =NULL;
	l_interface_agents->registration_id =0;
	l_interface_agents->poconnection_object = NULL;
	l_interface_agents->f_registerclient_cb =NULL;
	l_interface_agents->owner_id =0;
	l_interface_agents->watcher_id =0;
	l_interface_agents->binterface_has_signal = FALSE;
	return l_interface_agents;

}


/******************************************************************************
** FUNCTION   : vProxyCallback
*******************************************************************************/
tVoid vProxyCallback(GObject *source_object, GAsyncResult *res,
gpointer user_data)
{
	DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	
	interface_agents* l_interface_agents = (interface_agents*) user_data;
	
	GError *l_poerror = NULL;
	
	GDBusProxy *l_proxy = g_dbus_proxy_new_finish (res,&l_poerror);
	
	if(l_proxy == NULL)
	{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,
		DLT_STRING("DbusCommMgr- Error in proxy call back"),
		DLT_STRING(l_poerror->message));
		g_clear_error(&l_poerror);
	}
	else{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING(
		"sent response for client registration"));
		
		l_interface_agents->proxy = l_proxy;
		l_interface_agents->f_registerclient_cb(l_interface_agents);
	}					 					 
}


/******************************************************************************
** FUNCTION   : on_client_bus_name_appeared
*******************************************************************************
* \fn     on_client_bus_name_appeared
* \brief  function for watch bus
* \param  None.
* \retval None
******************************************************************************/

static void
on_client_bus_name_appeared (GDBusConnection *pConnection, const gchar *pName,
const gchar *pName_owner,gpointer pUserData)
{
	DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING(
	"GatewayDBusServer : - on_client_bus_name_appeared - "),DLT_STRING( pName));
	
	GError *l_poerror = NULL;

	if(pConnection)
	{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO, DLT_STRING("IPCM- create proxy "));
		interface_agents* l_interface_agents = (interface_agents*) pUserData;
		
		l_interface_agents->poconnection_object = g_object_ref(pConnection);
		//GDBusProxy *l_proxy  = 
		g_dbus_proxy_new(l_interface_agents->poconnection_object,
		G_DBUS_PROXY_FLAGS_NONE,
		l_interface_agents->interface_info,
		l_interface_agents->bus_name, /* name */
		l_interface_agents->obj_path,/*objpath */
		l_interface_agents->interface_info->name,
		NULL,
		vProxyCallback,
		pUserData);
		
		
		
		/*if((l_poerror != NULL)|| (l_proxy == NULL))
		{
			DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING(
						"Error in creating proxy"));
		}
		else{
			DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING(
						"sent response for client registration"));
			l_interface_agents->proxy = l_proxy;
			l_interface_agents->f_registerclient_cb(l_interface_agents);
		}*/
	}
	else{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING(
		"poconnection_object is NULL"));
	}


}



/******************************************************************************
** FUNCTION   : on_client_bus_name_lost
*******************************************************************************
* \fn     on_client_bus_name_lost
* \brief  function on bus name lost
* \param  None.
* \retval None
******************************************************************************/
static void
on_client_bus_name_lost (GDBusConnection *pConnection, const gchar *pName,
gpointer pUserData)
{
	DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING(
	"GatewayDBusServer : -client  name lost"),DLT_STRING(pName));
	interface_agents* l_interface_agents = (interface_agents*) pUserData;
	if(l_interface_agents)
	{
		g_object_unref(pConnection);
		l_interface_agents->poconnection_object =NULL;
		l_interface_agents->proxy = NULL;
	}
}

/******************************************************************************
** FUNCTION   : on_server_bus_acquired
*******************************************************************************
* \fn     on_server_bus_acquired
* \brief  function on bus acquired
* \param  None.
* \retval None
******************************************************************************/
static void
on_server_bus_acquired (GDBusConnection *pConnection, const gchar *pName,
gpointer pUserData)
{
	DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING(
	"GatewayDBusServer : - on_server_bus_acquired ")
	,DLT_STRING( pName));
	GError *l_poerror = NULL;
	
	if(pConnection)
	{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("pConnection"));
		interface_agents* l_interface_agents = (interface_agents*) pUserData;
		l_interface_agents->poconnection_object = g_object_ref(pConnection);

		if(l_interface_agents->poconnection_object)
		{
			l_interface_agents->registration_id =
			g_dbus_connection_register_object (
			l_interface_agents->poconnection_object,
			l_interface_agents->obj_path,
			l_interface_agents->interface_info,
			&interface_vtable,
			pUserData,  /* user_data */
			NULL,  /* user_data_free_func */
			&l_poerror); /* GError** */
			if((l_poerror != NULL)|| (l_interface_agents->registration_id == 0))
			{
				DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING(
				"Error in registering object"));
			}
		}
		else{
			DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING(
			"poconnection_object is NULL"));
		}
	}

}

/******************************************************************************
** FUNCTION   : on_server_name_acquired
*******************************************************************************
* \fn     on_server_name_acquired
* \brief  function on bus name acquired
* \param  None.
* \retval None
******************************************************************************/
static void
on_server_name_acquired (GDBusConnection *pConnection, const gchar *pName,
gpointer pUserData)
{
	DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING(
	"GatewayDBusServer : - server name acquired")
	,DLT_STRING(pName));
	if(pConnection)
	{
		interface_agents* l_interface_agents = (interface_agents*) pUserData;
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING(
		"obj path"),DLT_STRING(l_interface_agents->obj_path));
	}


}

/******************************************************************************
** FUNCTION   : on_server_name_lost
*******************************************************************************
* \fn     on_server_name_lost
* \brief  function on server name lost
* \param  None.
* \retval None
******************************************************************************/
static void
on_server_name_lost (GDBusConnection *pConnection,const gchar *pName,gpointer pUserData)
{
	DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING(
	"GatewayDBusServer : -  server name lost"),DLT_STRING(pName));
	interface_agents* l_interface_agents = (interface_agents*) pUserData;
	if(l_interface_agents)
	{
		g_dbus_connection_unregister_object
				(pConnection,l_interface_agents->registration_id);
		g_object_unref(pConnection);
		l_interface_agents->poconnection_object =NULL;
	}
}

/******************************************************************************
** FUNCTION   : bInterfaceHasSignal
*******************************************************************************
* \fn     bInterfaceHasSignal
* \brief  Function to check given interface has signal
* \param  None.
* \retval None.
******************************************************************************/
gboolean bInterfaceHasSignal(tCString arg_xml)
{
	DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	GError *l_poerror = NULL;
	GDBusNodeInfo *nodeinfo;
	if(arg_xml != NULL)
	{
		nodeinfo = g_dbus_node_info_new_for_xml (arg_xml, &l_poerror);
		for (tInt n = 0; nodeinfo->interfaces != NULL &&
		nodeinfo->interfaces[n] != NULL; n++)
		{
			tInt m = 0;
			if(nodeinfo->interfaces[n]->signals[m] != NULL)
			{
				return TRUE;
			}
			m++;
		}
	}
	else{
		DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING("interface is null"));
	}
	return FALSE;
}

/******************************************************************************
** FUNCTION   : RegisterClient
*******************************************************************************
* \fn     RegisterClient
* \brief  Function to handle client request
* \param  None.
* \retval None.
******************************************************************************/

interface_agents* RegisterClient(tString arg_xml,tString arg_bus_name
, tString arg_obj_path,GError **error
,RegisterClient_cb f_registerclient_cb)
{
	DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	GError *l_poerror = NULL;
	interface_agents * l_interface_agents =NULL;
	if(arg_xml != NULL)
	{
		tString l_xml = g_strdup(arg_xml);
		l_interface_agents = vCreate_interface_agents();
		l_interface_agents->bus_name = g_strdup(arg_bus_name);
		l_interface_agents->obj_path = g_strdup(arg_obj_path);
		l_interface_agents->f_registerclient_cb = f_registerclient_cb;
		if(bUpdateInterfaceAgents(l_xml,l_interface_agents))
		{
			DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("interface agents updated"));
		}
		// if(bInterfaceHasSignal(l_xml))
		// {
		l_interface_agents->binterface_has_signal = TRUE;
		l_interface_agents->hash_subscribed_signals
		= g_hash_table_new_full(g_str_hash,g_str_equal,NULL
		,vfree_hash_data);
		// }
		// else{
		// DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("interface not having signals"));
		// }
		g_free(l_xml);
		l_interface_agents->watcher_id
		= g_bus_watch_name ( G_BUS_TYPE_SESSION
		, l_interface_agents->bus_name
		, G_BUS_NAME_WATCHER_FLAGS_NONE
		, on_client_bus_name_appeared
		, on_client_bus_name_lost
		, l_interface_agents
		, NULL);
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("bus_name")
		, DLT_STRING(l_interface_agents->bus_name));
		g_list_interfaceagents = g_list_append (g_list_interfaceagents
		, l_interface_agents);

		return l_interface_agents;
		

	}
	else
	{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("IPCM- NULL value returned"));
		return NULL;
	}
	return NULL;
}

/******************************************************************************
** FUNCTION   : vUnRegisterServer
*******************************************************************************
* \fn     vUnRegisterServer
* \brief  Function to handle client request
* \param  None.
* \retval None.
******************************************************************************/
tVoid vUnRegisterServer(interface_agents* arg_agents,GError **l_poError)
{
	DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	if(arg_agents)
	{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("bus_name")
		, DLT_STRING(arg_agents->bus_name));
		g_bus_unown_name(arg_agents->owner_id);
		arg_agents->owner_id =  0;
		g_dbus_connection_unregister_object (
		arg_agents->poconnection_object,
		arg_agents->registration_id);
		arg_agents->registration_id =0;
		g_list_interfaceagents
		= g_list_remove(g_list_interfaceagents,arg_agents);
		g_free(arg_agents->bus_name);
		g_free(arg_agents->obj_path);
		arg_agents->interface_info = NULL;
		g_free(arg_agents);
		arg_agents=NULL;

	}
}

/******************************************************************************
** FUNCTION   : vUnRegisterClient
*******************************************************************************
* \fn     vUnRegisterClient
* \brief  Function to handle client request
* \param  None.
* \retval None.
******************************************************************************/
tVoid vUnRegisterClient(interface_agents* arg_agents,GError **l_poError)
{
	DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	if(arg_agents)
	{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("bus_name")
		, DLT_STRING(arg_agents->bus_name));
		g_bus_unwatch_name(arg_agents->watcher_id);
		g_list_interfaceagents
		= g_list_remove(g_list_interfaceagents,arg_agents);
		g_free(arg_agents->bus_name);
		g_free(arg_agents->obj_path);


		GHashTableIter iter; guint64 *key;
		signal_info* value;
		g_hash_table_iter_init(&iter, arg_agents->hash_subscribed_signals);

		while(g_hash_table_iter_next (&iter, (gpointer)&key,(gpointer)&value))
		{
			DLT_LOG(DBUS_MR ,DLT_LOG_INFO,DLT_STRING("subscription_ID : \n")
			,DLT_INT(value->subscription_ID));
			g_dbus_connection_signal_unsubscribe (arg_agents->poconnection_object
			, value->subscription_ID);
		}
		g_hash_table_destroy (arg_agents->hash_subscribed_signals);
		arg_agents->hash_subscribed_signals = NULL;
		arg_agents->proxy = NULL;
		arg_agents->interface_info = NULL;
		g_free(arg_agents);
		arg_agents=NULL;
	}

}

/******************************************************************************
** FUNCTION   : RegisterServer
*******************************************************************************
* \fn     RegisterServer
* \brief  Function to handle server request
* \param  None.
* \retval None.
******************************************************************************/

interface_agents* RegisterServer(tString arg_xml,tString arg_bus_name
, tString arg_obj_path, GError** err
, GenericMethodCall fnptr)
{
	DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	GError *l_poerror = NULL;
	interface_agents * l_interface_agents =NULL;
	if(arg_xml != NULL)
	{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("arg_xml is not NULL"));
		tString l_xml = g_strdup(arg_xml);
		l_interface_agents = vCreate_interface_agents();
		l_interface_agents->bus_name = g_strdup(arg_bus_name);
		l_interface_agents->obj_path = g_strdup(arg_obj_path);
		DLT_LOG(DBUS_MR,DLT_LOG_INFO
		, DLT_STRING(l_interface_agents->bus_name));
		if(bUpdateInterfaceAgents(l_xml,l_interface_agents))
		{
			DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("interface agents updated"));
		}
		l_interface_agents->f_genericmethod = fnptr;
		l_interface_agents->owner_id = g_bus_own_name( G_BUS_TYPE_SESSION,
		l_interface_agents->bus_name,
		G_BUS_NAME_OWNER_FLAGS_NONE,
		on_server_bus_acquired,
		on_server_name_acquired,
		on_server_name_lost,
		l_interface_agents,
		NULL);
		g_free(l_xml);
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("bus_name")
		, DLT_STRING(l_interface_agents->bus_name));
		g_list_interfaceagents = g_list_append (g_list_interfaceagents
		, l_interface_agents);

		return l_interface_agents;

	}
	else
	{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("IPCM- NULL value returned"));
		return NULL;
	}
	return NULL;
}

/******************************************************************************
** FUNCTION   : bUpdateInterfaceAgents
*******************************************************************************
* \fn     bUpdateInterfaceAgents
* \brief  function to get all interface agents
* \param  interface agents.
* \retval interface agents.
******************************************************************************/
gboolean bUpdateInterfaceAgents(char * arg_xml
, interface_agents * l_interface_agents)
{
	DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	GError *l_poerror = NULL;
	int n;
	if((arg_xml)&&(l_interface_agents))
	{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("IPCM- extracting node info"));
		GDBusNodeInfo *nodeinfo = g_dbus_node_info_new_for_xml (arg_xml
		, &l_poerror);
		if(nodeinfo)
		{
			DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("nodeinfo is not NULL"));
			for ( n = 0; nodeinfo->interfaces != NULL &&
			nodeinfo->interfaces[n] != NULL; n++)
			{
				l_interface_agents->interface_info
				= nodeinfo->interfaces[n];
			}


			DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("IPCM- interface_info value =")
			,DLT_STRING(l_interface_agents->interface_info->name));

			return TRUE;
		}
		else{
			DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("IPCM- nodeinfo is NULL"));
			return FALSE;
		}
	}
	else{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("IPCM- NULL value returned"));
		return FALSE;
	}
}

/******************************************************************************
** FUNCTION   : bSubscribesignal
*******************************************************************************
* \fn     bSubscribesignal
* \brief  function to subsribe connected signal
* \param  interface agents.
* \retval gboolean.
******************************************************************************/
gboolean bSubscribesignal(interface_agents* arg_interfaceagents
, tString arg_signalname
,dbus_signal_cb DbusresponseCB)
{
	DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	if(arg_interfaceagents)
	{
		if(g_list_find(g_list_interfaceagents,arg_interfaceagents))
		{
			DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("IPCM- valid list item"));
			DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("signalname")
			, DLT_STRING(arg_signalname));

			DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("busname")
			,DLT_STRING(arg_interfaceagents->bus_name));
			DLT_LOG(DBUS_MR,DLT_LOG_INFO
			, DLT_STRING("IPCM- arg_interfaceagents not null"));
			if(!arg_interfaceagents->binterface_has_signal)
			{
				DLT_LOG(DBUS_MR,DLT_LOG_INFO
				, DLT_STRING("IPCM- interface not having signals"));
				return FALSE;
			}
			if(g_hash_table_lookup((GHashTable*)arg_interfaceagents->hash_subscribed_signals
						, arg_signalname))
			{
				DLT_LOG(DBUS_MR,DLT_LOG_INFO
				, DLT_STRING("IPCM- signal is already registered"));
				return FALSE;
			}
			else{

				DLT_LOG(DBUS_MR,DLT_LOG_INFO
				, DLT_STRING("IPCM- new signal will be registered"));
				
				if(arg_interfaceagents->poconnection_object)
				{
					DLT_LOG(DBUS_MR,DLT_LOG_INFO
					, DLT_STRING("IPCM-poconnection_object is not null"));
					signal_info * l_posignal_info =g_new0(signal_info, 1);

					l_posignal_info->subscription_ID
					= g_dbus_connection_signal_subscribe(
					arg_interfaceagents->poconnection_object
					, arg_interfaceagents->bus_name
					, arg_interfaceagents->interface_info->name
					, arg_signalname
					, arg_interfaceagents->obj_path
					, NULL
					, G_DBUS_SIGNAL_FLAGS_NONE
					, vHandle_dbus_signal
					, arg_interfaceagents
					, NULL);

					DLT_LOG(DBUS_MR ,DLT_LOG_INFO,DLT_STRING("ID Received : \n")
					,DLT_INT(l_posignal_info->subscription_ID));

					l_posignal_info->DbusresponseCB =DbusresponseCB;
					if(g_hash_table_insert (arg_interfaceagents->hash_subscribed_signals
								, arg_signalname,l_posignal_info))
					{
						DLT_LOG(DBUS_MR,DLT_LOG_INFO
						, DLT_STRING("Hash insertion successfull"));
					}
				}
				else{
					DLT_LOG(DBUS_MR,DLT_LOG_INFO
					, DLT_STRING("IPCM-poconnection_object is null"));
					return FALSE;
				}
				return TRUE;

			}

		}
		else
		{
			DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("IPCM- invalid list item"));
			return FALSE;

		}
	}
	else{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("arg_interfaceagents is NULL"));
		return FALSE;
	}

	return FALSE;

}

/******************************************************************************
** FUNCTION   : bUnSubscribesignal
*******************************************************************************
* \fn     bUnSubscribesignal
* \brief  function to unsubsribe connected signal
* \param  interface agents.
* \retval gboolean.
******************************************************************************/
gboolean bUnSubscribesignal(interface_agents* arg_interfaceagents
, tString arg_signalname)
{
	DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));

	if(arg_interfaceagents)
	{
		if(g_list_find(g_list_interfaceagents,arg_interfaceagents))
		{
			DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("IPCM- found list item"));

			signal_info *l_posignal_info =
			(signal_info *)g_hash_table_lookup (arg_interfaceagents->hash_subscribed_signals,
			arg_signalname);
			DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("arg_signalname :")
			, DLT_STRING(arg_signalname));
			if(l_posignal_info)
			{
				DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("IPCM- list having signal"));
				DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("signalname :")
				, DLT_STRING(arg_signalname));
				g_dbus_connection_signal_unsubscribe (
				arg_interfaceagents->poconnection_object
				, l_posignal_info->subscription_ID);


				gboolean l_removed
				= g_hash_table_remove((GHashTable*)arg_interfaceagents->hash_subscribed_signals
				, arg_signalname );
				if(l_removed)
				{
					DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("IPCM- item removed"));
					return TRUE;
				}
				else{
					DLT_LOG(DBUS_MR,DLT_LOG_INFO
					, DLT_STRING("IPCM- item remove failed"));
					return FALSE;
				}
			}
			else{
				DLT_LOG(DBUS_MR,DLT_LOG_INFO
				, DLT_STRING("IPCM- signal is already unsubscribed"));
				return FALSE;
			}

		}
		else
		{
			DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("IPCM- invalid list item"));
			return FALSE;

		}
	}
	else{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("IPCM-arg_interfaceagents is NULL"));
		return FALSE;
	}
	return FALSE;
}

/******************************************************************************
** FUNCTION   : vHandle_dbus_signal
*******************************************************************************
* \fn     vHandle_dbus_signal
* \brief  handler function for signal emit from AGW
* \param  interface agents.
* \retval None.
******************************************************************************/
static tVoid  vHandle_dbus_signal (GDBusConnection *connection,
const gchar *sender_name,
const gchar *object_path,
const gchar *interface_name,
const gchar *signal_name,
GVariant *parameters,
gpointer user_data)
{
	DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	DLT_LOG(DBUS_MR,DLT_LOG_INFO
	,DLT_STRING("signal_name:"),DLT_STRING(signal_name));
	DLT_LOG(DBUS_MR,DLT_LOG_INFO
	,DLT_STRING("object_path:"),DLT_STRING(object_path));
	if(user_data)
	{
		interface_agents* l_interface_agents
		= (interface_agents*) user_data;
		signal_info * l_posignal_info =
		g_hash_table_lookup(l_interface_agents->hash_subscribed_signals
		, signal_name);
		if(l_posignal_info != NULL)
		{
			dbus_signal_cb f_DbusresponseCB =  l_posignal_info->DbusresponseCB;
			char* sPayload = g_variant_print(parameters,TRUE);
			DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("DBUSSIGNAL- parameters"),DLT_STRING(sPayload));
			g_free (sPayload);
			f_DbusresponseCB(signal_name,parameters);
		}
		else{
			DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("Error:signal is not registered"));
		}

	}
	else
	{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("user_data is NULL"));
	}
}


/******************************************************************************
** FUNCTION   : iTriggerMethodAsync
*******************************************************************************
* \fn     iTriggerMethodAsync
* \brief  Function to handle method request to AGW
* \param  None.
* \retval None.
******************************************************************************/

tVoid  iTriggerMethodAsync(interface_agents* arg_agents, tString name
, GVariant* g_methodRequestparams
, method_request_info* request_info)
{
	DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	if(arg_agents)
	{
		vSendAsyncMethodRequest(arg_agents, name,g_methodRequestparams
		, vMethodAsyncCallback,request_info );
	}
	else{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO
		,DLT_STRING("IPCM- interface_agents is NULL"));
	}
}

/******************************************************************************
** FUNCTION   : vSendAsyncMethodRequest
*******************************************************************************
* \fn     vSendAsyncMethodRequest
* \brief  Function to send method request to AGW
* \param  None.
* \retval None.
******************************************************************************/

tVoid vSendAsyncMethodRequest( interface_agents* arg_agents
,tString methodname
,GVariant* g_methodRequestparams
,GAsyncReadyCallback vMethodAsyncCallback
,method_request_info* request_info)
{
	DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("IPCM- arg_agents->bus_name")
	,DLT_STRING(arg_agents->bus_name));
	DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("IPCM- arg_agents->obj_path")
	,DLT_STRING(arg_agents->obj_path));
	DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("arg_agents->interface_info->name")
	,DLT_STRING(arg_agents->interface_info->name));
	if(arg_agents->poconnection_object)
	{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("IPCM- call method"));
		g_dbus_connection_call (arg_agents->poconnection_object,
		arg_agents->bus_name,
		arg_agents->obj_path,
		arg_agents->interface_info->name,
		methodname,
		g_methodRequestparams,
		NULL,
		G_DBUS_CALL_FLAGS_NONE,
		-1,
		NULL,
		(GAsyncReadyCallback) vMethodAsyncCallback,
		request_info);

	}

}
/******************************************************************************
** FUNCTION   : vMethodAsyncCallback
*******************************************************************************
* \fn     vMethodAsyncCallback
* \brief  Function to get call back for method request to AGW
* \param  None.
* \retval None.
******************************************************************************/
tVoid vMethodAsyncCallback(GObject *source_object, GAsyncResult *res,
gpointer user_data)
{
	DLT_LOG(DBUS_MR, DLT_LOG_INFO, DLT_STRING(__FUNCTION__));
	GError *error = NULL;
	GVariant *puser_info
	= g_dbus_connection_call_finish((GDBusConnection*)source_object
	, res, &error);
	if(error)
	{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("ERROR in vMethodAsyncCallback")
		, DLT_STRING(error->message));

		// HandleMethodResponse l_fMethodAsyncResponse =
		// ((method_request_info *) user_data)->handleMethodResponseCBAsync;
		// l_fMethodAsyncResponse(NULL,&error,user_data);
		// g_error_free(error);
	}
	else if ((puser_info)&&(user_data))
	{
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("Sending response"));
		gint64 *l_request_id = ((method_request_info *) user_data)->request_id;
		
		gint32 t_requestId = (gint32)*l_request_id;   
		DLT_LOG(DBUS_MR,DLT_LOG_INFO,DLT_STRING("DBUS- REQUESTID METHOD RESPONSE= "),DLT_INT(t_requestId));

		HandleMethodResponse l_fMethodAsyncResponse =
		((method_request_info *) user_data)->handleMethodResponseCBAsync;
		l_fMethodAsyncResponse(puser_info,&error,user_data);
	}
}




