/************************************************************************
* FILE:        	fc_auxJackDetection.cpp
* PROJECT:     	Renault / Nissan A-IVI
* SW-COMPONENT:	fc_aux
*----------------------------------------------------------------------
*
* DESCRIPTION: class for jack detection
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2015 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 18.05.2015| CM-AI/EPB2 Schurig | initial version
*
*************************************************************************/
#ifndef OSAL_S_IMPORT_INTERFACE_GENERIC
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#endif //#ifndef OSAL_S_IMPORT_INTERFACE_GENERIC

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h" 

#ifndef DP_S_IMPORT_INTERFACE_FI
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_fc_aux_if.h"
#endif

#include "I_fc_auxJackDetection.h"
#include "fc_auxJackDetection.h"
#include "../audio_routing/I_fc_auxAudioRouting.h"
#include "I_fc_aux_ClientDiagLogIf.h"

#include <errno.h>
#include <pthread.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <dirent.h>
#include <pthread.h>
#include <poll.h>
#include <sys/epoll.h>

#define VD_DIAGLOG_S_IMPORT_INTERFACE_MSG
#include "vd_diaglog_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_AUX_APP_JACK_DETECT
#include "trcGenProj/Header/fc_auxJackDetection.cpp.trc.h"
#endif


pthread_t 	m_tDetectionLoopThread;
uint8_t		u8_runDetection = 1;
bool        b_JackConnected = false;
bool        bIsTTFisCommand = false;
bool        bIsDiagnosisError = false;

int POLLING_TIME               =  100;
int SATISFYING_NUMBER_OF_POLLS =  3;

fc_auxJackDetection* fc_auxJackDetection::_poJackDetectionInstance = NULL;

/*************************************************************************
 * Constructors / Destructor
 ************************************************************************/
fc_auxJackDetection::fc_auxJackDetection(const fc_aux_tclAppMain* poMainAppl)
: I_fc_auxJackDetection(poMainAppl)
{
   _poJackDetectionInstance             = this;
   _bVoltageIsOK                        = false;
   m_s32MyThreadID                      = OSAL_ThreadWhoAmI();

	// start detection
   vStartDetectionLoop();
}

fc_auxJackDetection::~fc_auxJackDetection()
{
   ETG_TRACE_USR4(("fc_auxJackDetection::~fc_auxJackDetection()"));
   vEndDetectionLoop();
   sleep(1); // need to wait at least 200 ms for the thread to close
}

//fc_auxJackDetection&
/*************************************************************************
 *
 * FUNCTION:    bSetJackConnected
 *
 * DESCRIPTION: sets the protected static tBool _bJackConnected
 *
 * PARAMETER:   [in] tBool jackConnected
 *
 * RETURNVALUE: none
 *
 ************************************************************************/
void fc_auxJackDetection::bSetJackConnected(bool jackConnected)
{
	ETG_TRACE_USR4(("fc_auxJackDetection::bSetJackConnected() entered with %d",jackConnected));
	b_JackConnected = jackConnected;
}

/*************************************************************************
 *
 * FUNCTION:    bGetJackConnectionState
 *
 * DESCRIPTION: returns the state of the connection state flag
 *
 * PARAMETER:   none
 *
 * RETURNVALUE: tBool
 *
 ************************************************************************/
bool fc_auxJackDetection::bGetJackConnectionState()
{
	ETG_TRACE_USR4(("fc_auxJackDetection::bGetJackConnectionState() entered. Return %d",b_JackConnected));
	return b_JackConnected;
}

/*************************************************************************
 *
 * FUNCTION:    vStartDetectionLoop
 *
 * DESCRIPTION: This method is called by constructor at startup and creates
 *              a single thread which is starting the detection method.
 *
 * PARAMETER:   None
 *
 * RETURNVALUE: None
 *
 ************************************************************************/
void fc_auxJackDetection::vStartDetectionLoop(void)
{
   ETG_TRACE_USR4(("fc_auxJackDetection::vStartDetectionLoop() entered"));

   // if this is true, we are a RENAULT target
   if((_cpoMain->getU8BrandOfTarget() == 0x08) && (_cpoMain->getbAuxConfig() == true) && (_cpoMain->getbStartDetection() == true))
   {
	   ETG_TRACE_USR4(("RENAULT TARGET DETECTED, STARTING JACK DETECTION THREAD"));
       int returnvalue = pthread_create(&m_tDetectionLoopThread,NULL,&fc_auxJackDetection::vDetectionLoop,0);

	   if (returnvalue)
	   {
	      ETG_TRACE_USR4(("ERROR; return code from pthread_create() is %d\n", returnvalue));
	      exit(-1);
	   }
   }
   else if((_cpoMain->getbAuxConfig() == false) && (_cpoMain->getbStartDetection() == false))
   {
	   bSetJackConnected(FALSE);
	   vSetNewConnectionStatus();
   }
   else // we are a NISSAN target
   {
      bSetJackConnected(TRUE);
      vSetNewConnectionStatus();
   }
}

/*************************************************************************
 *
 * FUNCTION:    vDetectionLoop
 *
 * DESCRIPTION: The detection loop ...
 *
 * PARAMETER:   path to Linux file system for GPIO (sys/devices/...)
 *
 * RETURNVALUE:	None
 *l
 ************************************************************************/
void * fc_auxJackDetection::vDetectionLoop(void *device_path)
{
	ETG_TRACE_USR4(("fc_auxJackDetection::vDetectionLoop entered"));

	bIsTTFisCommand = false;

   // Open detect_plug_1
   int epoll_ctl_result_detect_plug;
   int epfd_detect_plug = epoll_create(1);
   int fd_detect_plug_1 = open("/sys/devices/virtual/boardcfg/boardcfg/detect-plug-1/value", O_RDONLY | O_NONBLOCK);

   struct epoll_event plug1_event;
   struct epoll_event plug1_events;

   if(fd_detect_plug_1 > 0)
   {
	   //char buf = 0;

	   plug1_event.events = EPOLLPRI | EPOLLERR | EPOLLET;
	   plug1_event.data.fd = fd_detect_plug_1;

	   epoll_ctl_result_detect_plug = epoll_ctl(epfd_detect_plug, EPOLL_CTL_ADD, fd_detect_plug_1, &plug1_event);
	   ETG_TRACE_USR4(("GPIO detect_plug_1 successful opened!"));
   }
   else
   {
	   ETG_TRACE_FATAL(("COULD NOT OPEN GPIO /sys/devices/virtual/boardcfg/boardcfg/detect-plug-1/value"));
	   fd_detect_plug_1 = 0;
   }

   // Open detect_plug_1_diag
   int epoll_ctl_result_detect_plug_diag;
   int epfd_detect_plug_diag = epoll_create(1);
   int fd_detect_plug_1_diag = open("/sys/devices/virtual/boardcfg/boardcfg/detect-plug-1-diag/value", O_RDONLY | O_NONBLOCK);

   struct epoll_event plug1diag_event;
   struct epoll_event plug1diag_events;

   if(fd_detect_plug_1_diag > 0)
   {
	   //char buf_diag = 0;

	   plug1diag_event.events = EPOLLPRI | EPOLLERR | EPOLLET;
	   plug1diag_event.data.fd = fd_detect_plug_1_diag;

	   epoll_ctl_result_detect_plug_diag = epoll_ctl(epfd_detect_plug_diag, EPOLL_CTL_ADD, fd_detect_plug_1_diag, &plug1diag_event);
	   ETG_TRACE_USR4(("GPIO detect_plug_1_diag successful opened!"));
   }
   else
   {
	   ETG_TRACE_FATAL(("COULD NOT OPEN GPIO /sys/devices/virtual/boardcfg/boardcfg/detect-plug-1-diag/value"));
	   fd_detect_plug_1_diag = 0;
   }

   // for debouncing
   static int pollCounter = 0;

   tU8 _u8detect_plug_1_new_value      = 49;
   tU8 _u8detect_plug_1_old_value      = 0;
   tU8 _u8detect_plug_1_diag_old_value = 0;
   tU8 _u8detect_plug_1_diag_new_value = 48;

   // HERE STARTS OUR JACK DETECTION LOOP
   while(u8_runDetection != 0)
   {

	   epoll_ctl_result_detect_plug = epoll_wait(epfd_detect_plug, &plug1_events, 1, POLLING_TIME);

	   if(epoll_ctl_result_detect_plug == 0)
	   {
		   char buf = 0;
		   epoll_ctl_result_detect_plug = lseek(fd_detect_plug_1, 0, SEEK_SET);
		   epoll_ctl_result_detect_plug = read(fd_detect_plug_1, &buf, 1);
		   _u8detect_plug_1_new_value = buf;
	   }

	   epoll_ctl_result_detect_plug_diag = epoll_wait(epfd_detect_plug_diag, &plug1diag_events, 1, POLLING_TIME);

	   if(epoll_ctl_result_detect_plug_diag == 0)
	   {
		   char buf_diag = 0;
		   epoll_ctl_result_detect_plug_diag = lseek(fd_detect_plug_1_diag, 0, SEEK_SET);
		   epoll_ctl_result_detect_plug_diag = read(fd_detect_plug_1_diag, &buf_diag, 1);
		   _u8detect_plug_1_diag_new_value = buf_diag;
	   }

	   if(((_u8detect_plug_1_old_value != _u8detect_plug_1_new_value)          ||
		 (_u8detect_plug_1_diag_old_value != _u8detect_plug_1_diag_new_value)) &&
	     (_poJackDetectionInstance->bGetVoltageIsOK() == true))
	   {
		   pollCounter++;

		   if(pollCounter == SATISFYING_NUMBER_OF_POLLS)
		   {
			   if(_u8detect_plug_1_new_value == 49) // detect_plug_1 is set to 1
			   {
				   if(_u8detect_plug_1_diag_new_value == 49) // detect_plug_1_DIAG is set to 1
				   {
					   ETG_TRACE_USR4(("+===============+====================+==========================+=============+"));
					   ETG_TRACE_USR4(("| DETECT PLUG 1 | DETECT PLUG 1 DIAG | Meaning                  | DIAGNOSIS   |"));
					   ETG_TRACE_USR4(("+---------------+--------------------+--------------------------+-------------+"));
					   ETG_TRACE_USR4(("|      HIGH     |        HIGH        | Plug NOT attached        |     OPEN    |"));
					   ETG_TRACE_USR4(("+===============+====================+==========================+=============+"));
					   _u8detect_plug_1_diag_old_value = _u8detect_plug_1_diag_new_value;
					   bIsDiagnosisError = true;

					   // send diagnosis error to diaglog
					   _poJackDetectionInstance->vCallDiagLog();

					   b_JackConnected = false;
					   pollCounter = 0;
				   }
				   else if(_u8detect_plug_1_diag_new_value == 48) // detect_plug_1_DIAG is set to 0, PLUG REMOVED
				   {
					   ETG_TRACE_USR4(("+===============+====================+==========================+=============+"));
					   ETG_TRACE_USR4(("| DETECT PLUG 1 | DETECT PLUG 1 DIAG | Meaning                  | DIAGNOSIS   |"));
					   ETG_TRACE_USR4(("+---------------+--------------------+--------------------------+-------------+"));
					   ETG_TRACE_USR4(("|      HIGH     |         LOW        | Plug NOT attached        |      OK     |"));
					   ETG_TRACE_USR4(("+===============+====================+==========================+=============+"));
					   _u8detect_plug_1_diag_old_value = _u8detect_plug_1_diag_new_value;
					   b_JackConnected = false;
					   bIsDiagnosisError = false;
					   _poJackDetectionInstance->vSetNewConnectionStatus(true);
					   dp_tclAuxiliaryDPfcAuxBoolIsNewDevice AuxIsNewDevice;
					   AuxIsNewDevice.vSetData(TRUE);                           // Now we have a "NEW DEVICE"
					   pollCounter = 0;
				   }
				   _u8detect_plug_1_old_value = _u8detect_plug_1_new_value;
			   }
			   else if(_u8detect_plug_1_new_value == 48) // detect_plug_1 is set to 0
			   {
				   if(_u8detect_plug_1_diag_new_value == 49) // detect_plug_1_DIAG is set to 1, SHOULD NEVER COME
				   {
					   ETG_TRACE_USR4(("+===============+====================+==========================+=============+"));
					   ETG_TRACE_USR4(("| DETECT PLUG 1 | DETECT PLUG 1 DIAG | Meaning                  | DIAGNOSIS   |"));
					   ETG_TRACE_USR4(("+---------------+--------------------+--------------------------+-------------+"));
					   ETG_TRACE_USR4(("|      LOW      |        HIGH        | Invalid                  |   Invalid   |"));
					   ETG_TRACE_USR4(("+===============+====================+==========================+=============+"));
					   _u8detect_plug_1_diag_old_value = _u8detect_plug_1_diag_new_value;
					   b_JackConnected = false;
					   bIsDiagnosisError = false;
					   pollCounter = 0;
				   }
				   else if(_u8detect_plug_1_diag_new_value == 48) // detect_plug_1_DIAG is set to 0, PLUG ATTACHED
				   {
					   ETG_TRACE_USR4(("+===============+====================+==========================+=============+"));
					   ETG_TRACE_USR4(("| DETECT PLUG 1 | DETECT PLUG 1 DIAG | Meaning                  | DIAGNOSIS   |"));
					   ETG_TRACE_USR4(("+---------------+--------------------+--------------------------+-------------+"));
					   ETG_TRACE_USR4(("|      LOW      |         LOW        | Plug attached            |      OK     |"));
					   ETG_TRACE_USR4(("+===============+====================+==========================+=============+"));
					   _u8detect_plug_1_diag_old_value = _u8detect_plug_1_diag_new_value;
					   b_JackConnected = true;
					   bIsDiagnosisError = false;

					   // send diagnosis error to diaglog
					   _poJackDetectionInstance->vCallDiagLog();

					   dp_tclAuxiliaryDPfcAuxBoolIsNewDevice AuxIsNewDevice;
					   AuxIsNewDevice.vSetData(FALSE);				           // Now we have a "SAME DEVICE"
					   pollCounter = 0;
				   }
				   _u8detect_plug_1_old_value = _u8detect_plug_1_new_value;
			   }
			   _poJackDetectionInstance->vSetNewConnectionStatus();
		   }
	   }
   }

   close(fd_detect_plug_1);
   close(fd_detect_plug_1_diag);

   pthread_exit(NULL); // Stop thread

   ETG_TRACE_USR4(("EXIT DETECTION"));

   return NULL;
}

/*************************************************************************
 *
 * FUNCTION:    vSetNewConnectionStatus
 *
 * DESCRIPTION: set the latest connection status when this has been
 *              changed
 *
 * PARAMETER:   none
 *
 * RETURNVALUE: none
 *l
 ************************************************************************/
void fc_auxJackDetection::vSetNewConnectionStatus(bool bIsNewDevice)
{
	ETG_TRACE_USR4(("AuxJackDetection::vSetNewConnectionStatus entered"));
	I_fc_auxAudioRouting* _audio = dynamic_cast<I_fc_auxAudioRouting*>(_cpoMain->getHandler("I_fc_auxAudioRouting"));
	FC_AUX_NULL_POINTER_CHECK(_audio);
	_audio->vSetJackConnectionState(b_JackConnected);
	if(bIsNewDevice == true) _audio->bSetIsNewDevice(true);
}

/*************************************************************************
 *
 * FUNCTION:    vEndDetectionLoop
 *
 * DESCRIPTION: set "runDetection" to 0 to stop detection thread
 *
 * PARAMETER:   none
 *
 * RETURNVALUE: none
 *
 ************************************************************************/
void fc_auxJackDetection::vEndDetectionLoop(void)
{
	u8_runDetection = 0;
}

/*************************************************************************
 *
 * FUNCTION:    bGetVoltageIsOK
 *
 * DESCRIPTION: return value of protected "bVoltageIsOK"
 *
 * PARAMETER:   none
 *
 * RETURNVALUE: bool _bVoltageIsOK
 *
 ************************************************************************/
bool fc_auxJackDetection::bGetVoltageIsOK()
{
   ETG_TRACE_USR4(("AuxJackDetection::bGetVoltageIsOK() entered"));
   return _bVoltageIsOK;
}
/*************************************************************************
 *
 * FUNCTION:    vCallDiagLog
 *
 * DESCRIPTION: calling diaglog to snd the error
 *
 * PARAMETER:   n/a
 *
 * RETURNVALUE: none
 *
 ************************************************************************/
void fc_auxJackDetection::vCallDiagLog()
{
	ETG_TRACE_USR4(("fc_auxJackDetection::vCallDiagLog() entered"));

	midw_fi_tcl_TestResultList _oList;
	midw_fi_tcl_TestResult _oResult;

	_oResult.TroubleCode    = ITC_AUX_INPUT_1_OPEN;
	if(bIsDiagnosisError == true)  _oResult.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_FAILED;
	if(bIsDiagnosisError == false) _oResult.Result.enType = midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
    _oList.TestResultList.push_back(_oResult);

    Ifc_aux_tclClientDiagLogIf* my_diaglog = dynamic_cast<Ifc_aux_tclClientDiagLogIf*>(_cpoMain->getHandler("Ifc_aux_tclClientDiagLogIf"));
    FC_AUX_NULL_POINTER_CHECK(my_diaglog);

    my_diaglog->sendSaveTestResultMStart(_oList);
}

/*************************************************************************
 *
 * FUNCTION:    bSetVoltageIsOK
 *
 * DESCRIPTION: set protected "_bVoltageIsOK"
 *
 * PARAMETER:   bool voltageIsOK
 *
 * RETURNVALUE: none
 *
 ************************************************************************/
void fc_auxJackDetection::bSetVoltageIsOK(bool voltageIsOK)
{
   ETG_TRACE_USR4(("AuxJackDetection::bSetVoltageIsOK() entered. _bVoltageIsOK = %d",voltageIsOK));
   _bVoltageIsOK = voltageIsOK;
}

// needs to be implemented
void fc_auxJackDetection::vHandleMessage(fc_aux_tclBaseIf::TMsg* pMsg) {}

void fc_auxJackDetection::vHandleTraceMessage(const tUChar* puchData)
{
	FC_AUX_NULL_POINTER_CHECK(puchData);

	tU32 u32MsgCode = ((puchData[1]<<8) | puchData[2]);
	ETG_TRACE_USR4(("fc_auxJackDetection::vHandleTraceMessage(): trace command %d", u32MsgCode ));

	if(u32MsgCode == 7)	vCallDiagLog();
}

void fc_auxJackDetection::vGetReferences(){}
void fc_auxJackDetection::vStartCommunication() {}
void fc_auxJackDetection::vTraceInfo() {}
