/*
 * Copyright (C) 2008-2013 J.Rios <anonbeat@gmail.com>
 * Copyright (C) 2013 Jörn Magens
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This Program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file LICENSE.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth 
 * Floor, Boston, MA  02110-1301  USA
 * https://www.gnu.org/licenses/lgpl-2.1.txt
 *
 * Author:
 * 	Jörn Magens <shuerhaaken@googlemail.com>
 * 	Matias De lellis <mati86dl@gmail.com>
 * 	Pavel Vasin <rat4vier@gmail.com>
 */


#include "taginfo.h"
#include "taginfo_internal.h"


using namespace TagInfo;



WavPackInfo::WavPackInfo(const String &filename) : Info(filename) {
    taglib_apetag = NULL;
    if(!file_name.isEmpty() && !create_file_ref()) {
        printf("Error creating file ref! %s\n", filename.toCString(false));
    }
    if(taglib_file) {
        taglib_apetag = ((TagLib::WavPack::File *) taglib_file)->APETag(true);
        if(!taglib_apetag || taglib_apetag->isEmpty()) {
            taglib_tag = ((TagLib::WavPack::File *) taglib_file)->tag();
            if(!taglib_tag) {
                printf("Cant get tag object from '%s'\n", file_name.toCString(false));
                valid = false;
            }
        }
    }
    else {
        printf("Cant get id3 tag from '%s'\n", file_name.toCString(false));
        taglib_apetag = NULL;
        valid = false;
    }
}


WavPackInfo::~WavPackInfo() {
}


bool WavPackInfo::create_file_ref() {
    if(file_name.isEmpty())
        return false;
    taglib_file = new TagLib::WavPack::File(file_name.toCString(false), true, TagLib::AudioProperties::Fast);
    if(taglib_file) {
        return true;
    }
    else {
        printf("TagLib::File creation failed for '%s'\n", file_name.toCString(false));
        return false;
    }
}


bool WavPackInfo::read(void) {
    if(Info::read()) {
        if(taglib_apetag) {
            if(!taglib_tag)
                read_virtual_tags((TagLib::Tag *)taglib_apetag);
            if(taglib_apetag->itemListMap().contains("COMPOSER")) {
                composer = taglib_apetag->itemListMap()["COMPOSER"].toStringList().front();
            }
            if(taglib_apetag->itemListMap().contains("DISCNUMBER")) {
                disk_number = 
                    atol(taglib_apetag->itemListMap()["DISCNUMBER"].toStringList().front().toCString(false));
                if(disk_number < 1)
                    disk_number = 1;
            }
            if(taglib_apetag->itemListMap().contains("COMPILATION")) {
                is_compilation = taglib_apetag->itemListMap()["COMPILATION"].toStringList().front() == String("1");
            }
            if(taglib_apetag->itemListMap().contains("ALBUM ARTIST")) {
                album_artist = taglib_apetag->itemListMap()["ALBUM ARTIST"].toStringList().front();
            }
            else if(taglib_apetag->itemListMap().contains("ALBUMARTIST")) {
                album_artist = taglib_apetag->itemListMap()["ALBUMARTIST"].toStringList().front();
            }
            // Rating
            if(taglib_apetag->itemListMap().contains("RATING")) {
                long rat = 0;
                rat = atol(taglib_apetag->itemListMap()["RATING"].toStringList().front().toCString(true));
                if(rat) {
                    if(rat > 5) {
                        rating = popularity_to_rating(rat);
                    }
                    else {
                        rating = rat;
                    }
                }
            }
            if(taglib_apetag->itemListMap().contains("PLAY_COUNTER")) {
                long PlayCount = 0;
                PlayCount = atol(taglib_apetag->itemListMap()["PLAY_COUNTER"].toStringList().front().toCString(true));
                playcount = PlayCount;
            }
            // Labels
            if(track_labels.size() == 0) {
                if(taglib_apetag->itemListMap().contains("TRACK_LABELS")) {
                    track_labels_string = taglib_apetag->itemListMap()["TRACK_LABELS"].toStringList().front();
                    track_labels = split(track_labels_string, "|");
                }
            }
            if(artist_labels.size() == 0) {
                if(taglib_apetag->itemListMap().contains("ARTIST_LABELS")) {
                    artist_labels_string = taglib_apetag->itemListMap()["ARTIST_LABELS"].toStringList().front();
                    artist_labels = split(artist_labels_string, "|");
                }
            }
            if(album_labels.size() == 0) {
                if(taglib_apetag->itemListMap().contains("ALBUM_LABELS")) {
                    album_labels_string = taglib_apetag->itemListMap()["ALBUM_LABELS"].toStringList().front();
                    album_labels = split(album_labels_string, "|");
                }
            }
            return true; 
        }
    }
    return false;
}


bool WavPackInfo::write(void) {
    if(taglib_apetag) {
        if(changedflag) {
            if(changedflag & CHANGED_COMPOSER_TAG)
                taglib_apetag->addValue("COMPOSER", composer);
            if(changedflag & CHANGED_DATA_DISK_NUM)
                taglib_apetag->addValue("DISCNUMBER", format("%u", disk_number));
            
            if(changedflag & CHANGED_IS_COMPILATION_TAG) {
                if(is_compilation) {
                    taglib_apetag->addValue("COMPILATION", "1");
                }
                else {
                    taglib_apetag->addValue("COMPILATION", "0");
                }
            }
            
            if(changedflag & CHANGED_DATA_ALBUMARTIST)
                taglib_apetag->addValue("ALBUM ARTIST", album_artist);
            if(changedflag & CHANGED_DATA_RATING) {
                taglib_apetag->addValue("RATING", format("%u", rating_to_popularity(rating)));
            }
            if(changedflag & CHANGED_DATA_PLAYCOUNT) {
                taglib_apetag->addValue("PLAY_COUNTER", format("%u", playcount));
            }
            // The Labels
            if(changedflag & CHANGED_ARTIST_LABELS)
                check_ape_label_frame(taglib_apetag, "ARTIST_LABELS", artist_labels_string);
            if(changedflag & CHANGED_ALBUM_LABELS)
                check_ape_label_frame(taglib_apetag, "ALBUM_LABELS", album_labels_string);
            if(changedflag & CHANGED_TRACK_LABELS)
                check_ape_label_frame(taglib_apetag, "TRACK_LABELS", track_labels_string);
            
            write_virtual_tags((TagLib::Tag *)taglib_apetag);
        }
    }
    return Info::write();
}

bool WavPackInfo::can_handle_images(void) const {
    return false;
}

bool WavPackInfo::get_image(char*& data, int &data_length, ImageType &image_type) const {
    return get_ape_image(taglib_apetag, data, data_length, image_type);
}

bool WavPackInfo::set_image(const char* data, int data_length, ImageType image_type) {
    return set_ape_image(taglib_apetag, data, data_length, image_type);
}


bool WavPackInfo::can_handle_lyrics(void) const {
    return true;
}

String WavPackInfo::get_lyrics(void) const {
    return get_ape_lyrics(taglib_apetag);
}

bool WavPackInfo::set_lyrics(const String &lyrics) {
    return set_ape_lyrics(taglib_apetag, lyrics);
}

