/************************************************************************
* FILE:         vdmmgr_servicediag.cpp
* PROJECT:      MIB2_ENTRY
* SW-COMPONENT: Virtual Device Media Manager
*----------------------------------------------------------------------
*
* DESCRIPTION: VD MediaManager
*              
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2008 Robert Bosch GmbH, Hildesheim
* HISTORY:      
* Date       | Author             | Modification
* 18.02.2008 | CM-DI/PJ36 Fiebing | initial version
*
*************************************************************************/

// shorthand for our trace class
#define _CLASS VDMMGR_TR_SERVICE_DIAG

//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------
#include "Config.h"
#include <map>
#include <vector>

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_MEDIAMANAGER_SERVICE_DIAG
#include "trcGenProj/Header/vdmmgr_servicediag.cpp.trc.h"
#endif

#define AIL_S_IMPORT_INTERFACE_GENERIC
#include "ail_if.h"         // use AIL template with MessageMaps
#define AHL_S_IMPORT_INTERFACE_NOTIFICTABLE
#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"

//#include "osal_if.h" with OSAL_S_IMPORT_INTERFACE_GENERIC comes with "etrace_if.h"

#define VDMMGR_S_IMPORT_INTERFACE_MSG
#include "vdmmgr_if.h"                       // For VD MMgr interface

/* ************************************************************************** */
/* include the public interface                                               */
/* ************************************************************************** */
#include "vdmmgr_main.h"
#include "vdmmgr_timer.h"                 // 'automatic reinsert CD' timer
#include "vdmmgr_service.h"
#include "vdmmgr_ITC_Container.h"
#include "vdmmgr_ITC_Map.h"
#include "vdmmgr_clienthandlerdiaglog.h"  // get ITC status changes
#include "vdmmgr_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
   #include "vdmmgr_cdctrlif.h"
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE


/******************************************************************************
*
* FUNCTION:    vSendAnswerDiagFirmwareVersion( const amt_tclServiceData* poMessage,
*                    const T_DiagFirmwareVersion* pDiagFirmwareVersion )
* 
* DESCRIPTION: sends an answer message
*
* PARAMETER:   message to be answered, data, opcode of the answer
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclservice::vSendAnswerDiagFirmwareVersion( const amt_tclServiceData* poMessage,
                                           const T_DiagFirmwareVersion* pDiagFirmwareVersion )
{
   ETG_TRACE_USR1(( "vSendAnswerDiagFirmwareVersion( ): Start" ));

   // Prepare method result data
   mplay_mmgrfi_tclMsgDiagFirmwareVersionStatus oResultData;
   // Copy data
   oResultData.FirmwareVersion.u8FirmwareVersion00 = pDiagFirmwareVersion->au8FirmwareVersion[0];
   oResultData.FirmwareVersion.u8FirmwareVersion01 = pDiagFirmwareVersion->au8FirmwareVersion[1];
   oResultData.FirmwareVersion.u8FirmwareVersion02 = pDiagFirmwareVersion->au8FirmwareVersion[2];
   oResultData.FirmwareVersion.u8FirmwareVersion03 = pDiagFirmwareVersion->au8FirmwareVersion[3];
   oResultData.FirmwareVersion.u8FirmwareVersion04 = pDiagFirmwareVersion->au8FirmwareVersion[4];
   oResultData.FirmwareVersion.u8FirmwareVersion05 = pDiagFirmwareVersion->au8FirmwareVersion[5];
   oResultData.FirmwareVersion.u8FirmwareVersion06 = pDiagFirmwareVersion->au8FirmwareVersion[6];
   oResultData.FirmwareVersion.u8FirmwareVersion07 = pDiagFirmwareVersion->au8FirmwareVersion[7];

   // construct result message
   fi_tclVisitorMessage oResultMsg( oResultData );

   // Send it to the client
   vSendStatusAnswerVisitorMessage( poMessage, &oResultMsg );
}


/******************************************************************************
*
* FUNCTION:    vSendAnswerDiagDriveVersion ( const amt_tclServiceData* poMessage,
*                   const T_DiagDriveVersion * pDiagDriveVersion )
* 
* DESCRIPTION: sends an answer message
*
* PARAMETER:   message to be answered, data, opcode of the answer
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclservice::vSendAnswerDiagDriveVersion ( const amt_tclServiceData* poMessage,
                                               const T_DiagDriveVersion* pDiagDriveVersion )
{
   // Prepare method result data
   mplay_mmgrfi_tclMsgDiagDriveVersionStatus oResultData;
   // Copy data
   oResultData.DriveVersion.u8DriveVersion00 = pDiagDriveVersion->au8DriveVersion[0];
   oResultData.DriveVersion.u8DriveVersion01 = pDiagDriveVersion->au8DriveVersion[1];
   oResultData.DriveVersion.u8DriveVersion02 = pDiagDriveVersion->au8DriveVersion[2];
   oResultData.DriveVersion.u8DriveVersion03 = pDiagDriveVersion->au8DriveVersion[3];
   oResultData.DriveVersion.u8DriveVersion04 = pDiagDriveVersion->au8DriveVersion[4];
   oResultData.DriveVersion.u8DriveVersion05 = pDiagDriveVersion->au8DriveVersion[5];
   oResultData.DriveVersion.u8DriveVersion06 = pDiagDriveVersion->au8DriveVersion[6];
   oResultData.DriveVersion.u8DriveVersion07 = pDiagDriveVersion->au8DriveVersion[7];
   oResultData.DriveVersion.u8DriveVersion08 = pDiagDriveVersion->au8DriveVersion[8];
   oResultData.DriveVersion.u8DriveVersion09 = pDiagDriveVersion->au8DriveVersion[9];
   oResultData.DriveVersion.u8DriveVersion10 = pDiagDriveVersion->au8DriveVersion[10];
   oResultData.DriveVersion.u8DriveVersion11 = pDiagDriveVersion->au8DriveVersion[11];
   oResultData.DriveVersion.u8DriveVersion12 = pDiagDriveVersion->au8DriveVersion[12];
   oResultData.DriveVersion.u8DriveVersion13 = pDiagDriveVersion->au8DriveVersion[13];
   oResultData.DriveVersion.u8DriveVersion14 = pDiagDriveVersion->au8DriveVersion[14];
   oResultData.DriveVersion.u8DriveVersion15 = pDiagDriveVersion->au8DriveVersion[15];
   oResultData.DriveVersion.u8DriveVersion16 = pDiagDriveVersion->au8DriveVersion[16];
   oResultData.DriveVersion.u8DriveVersion17 = pDiagDriveVersion->au8DriveVersion[17];
   oResultData.DriveVersion.u8DriveVersion18 = pDiagDriveVersion->au8DriveVersion[18];
   oResultData.DriveVersion.u8DriveVersion19 = pDiagDriveVersion->au8DriveVersion[19];
   oResultData.DriveVersion.u8DriveVersion20 = pDiagDriveVersion->au8DriveVersion[20];
   oResultData.DriveVersion.u8DriveVersion21 = pDiagDriveVersion->au8DriveVersion[21];
   oResultData.DriveVersion.u8DriveVersion22 = pDiagDriveVersion->au8DriveVersion[22];
   oResultData.DriveVersion.u8DriveVersion23 = pDiagDriveVersion->au8DriveVersion[23];
   oResultData.DriveVersion.u8DriveVersion24 = pDiagDriveVersion->au8DriveVersion[24];
   oResultData.DriveVersion.u8DriveVersion25 = pDiagDriveVersion->au8DriveVersion[25];
   oResultData.DriveVersion.u8DriveVersion26 = pDiagDriveVersion->au8DriveVersion[26];
   oResultData.DriveVersion.u8DriveVersion27 = pDiagDriveVersion->au8DriveVersion[27];
   oResultData.DriveVersion.u8DriveVersion28 = pDiagDriveVersion->au8DriveVersion[28];
   oResultData.DriveVersion.u8DriveVersion29 = pDiagDriveVersion->au8DriveVersion[29];
   oResultData.DriveVersion.u8DriveVersion30 = pDiagDriveVersion->au8DriveVersion[30];
   oResultData.DriveVersion.u8DriveVersion31 = pDiagDriveVersion->au8DriveVersion[31];
   oResultData.DriveVersion.u8DriveVersion32 = pDiagDriveVersion->au8DriveVersion[32];
   oResultData.DriveVersion.u8DriveVersion33 = pDiagDriveVersion->au8DriveVersion[33];
   oResultData.DriveVersion.u8DriveVersion34 = pDiagDriveVersion->au8DriveVersion[34];
   oResultData.DriveVersion.u8DriveVersion35 = pDiagDriveVersion->au8DriveVersion[35];
   oResultData.DriveVersion.u8DriveVersion36 = pDiagDriveVersion->au8DriveVersion[36];
   oResultData.DriveVersion.u8DriveVersion37 = pDiagDriveVersion->au8DriveVersion[37];
   oResultData.DriveVersion.u8DriveVersion38 = pDiagDriveVersion->au8DriveVersion[38];
   oResultData.DriveVersion.u8DriveVersion39 = pDiagDriveVersion->au8DriveVersion[39];

   // construct result message
   fi_tclVisitorMessage oResultMsg( oResultData );

   // Send it to the client
   vSendStatusAnswerVisitorMessage( poMessage, &oResultMsg );
}


/******************************************************************************
*
* FUNCTION:    vSendAnswerDiagLoadStatus( const amt_tclServiceData* poMessage,
*                                         T_e8_DiagLoadStatus e8LoadStatus )
* 
* DESCRIPTION: sends an answer message
*
* PARAMETER:   message to be answered, data, opcode of the answer
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclservice::vSendAnswerDiagLoadStatus( const amt_tclServiceData* poMessage,
                                                    T_e8_DiagLoadStatus e8LoadStatus )
{
   // Prepare method result data
   mplay_mmgrfi_tclMsgDiagLoadStatusStatus oResultData;
   // Copy data
   oResultData.LoadStatus.enType = (mplay_fi_tcl_e8_DiagLoadStatus::tenType)e8LoadStatus;

   // construct result message
   fi_tclVisitorMessage oResultMsg( oResultData );

   // Send it to the client
   vSendStatusAnswerVisitorMessage( poMessage, &oResultMsg );
}


/******************************************************************************
*
* FUNCTION:    vSendAnswerDiagErrorStatus ( const amt_tclServiceData* poMessage,
*                                           CP_TSL_V poTroubleCodeStatusList )
* 
* DESCRIPTION: sends an answer message
*
* PARAMETER:   message to be answered, data, opcode of the answer
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclservice::vSendAnswerDiagErrorStatus( const amt_tclServiceData* poMessage,
                                                      CP_TSL_V poTroubleCodeStatusList )
{
   // Prepare method result data
   mplay_mmgrfi_tclMsgDiagErrorStatusStatus   oResultData;
   // Copy data
   oResultData.TroubleCodeStatusList.TroubleCodeStatusList = *poTroubleCodeStatusList;

   // construct result message
   fi_tclVisitorMessage oResultMsg( oResultData );

   // Send it to the client
   vSendStatusAnswerVisitorMessage( poMessage, &oResultMsg );
}


/******************************************************************************
*
* FUNCTION:    vSendAnswerDiagDriveTest( const amt_tclServiceData* poMessage,
*                                        const T_DiagMethodControlResultDriveTest* pDiagResultDriveTest )
* 
* DESCRIPTION: sends an answer message
*
* PARAMETER:   message to be answered, data, opcode of the answer
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclservice::vSendAnswerDiagDriveTest( const trMessageDataForMethodResult* poMsgDataForResult,
                                                   const T_DiagMethodControlResultDriveTest* pDiagResultDriveTest )
{
   // Prepare method result data
   mplay_mmgrfi_tclMsgDiagControlDriveTestMethodResult oResultData;
   // Copy data
   oResultData.e8Duration.enType                          = (mplay_fi_tcl_e8_DiagDuration::tenType)pDiagResultDriveTest->e8Duration;
   oResultData.ResultDriveTest.e8RetValueDriveTest.enType = (mplay_fi_tcl_e8_DiagRetValueDriveTest::tenType)pDiagResultDriveTest->ResultDriveTest.e8RetValueDriveTest;
   oResultData.ResultDriveTest.e16DiagDrvTesttype.enType  = (mplay_fi_tcl_e16_DiagDrvTestParam::tenType)pDiagResultDriveTest->ResultDriveTest.e16DiagDrvTesttype;
   oResultData.ResultDriveTest.e16Drive.enType            = (mplay_fi_tcl_e16_DiagDrv::tenType)pDiagResultDriveTest->ResultDriveTest.e16Drive;
   oResultData.ResultDriveTest.u32SectorsRead             = pDiagResultDriveTest->ResultDriveTest.u32SectorsRead;
   oResultData.ResultDriveTest.u16ReadAttemps             = pDiagResultDriveTest->ResultDriveTest.u16ReadAttempts;
   oResultData.ResultDriveTest.u16CntErrors               = pDiagResultDriveTest->ResultDriveTest.u16CntErrors;
   oResultData.ResultDriveTest.u16AverageReadtimeSector   = pDiagResultDriveTest->ResultDriveTest.u16AverageReadtimeSector;
   oResultData.ResultDriveTest.u32TotalReadTime           = pDiagResultDriveTest->ResultDriveTest.u32TotalReadTime;
   oResultData.ResultDriveTest.u16MaxReadTimePerSector    = pDiagResultDriveTest->ResultDriveTest.u16MaxReadTimePerSector;
   oResultData.ResultDriveTest.u32MaxLBA                  = pDiagResultDriveTest->ResultDriveTest.u32MaxLBA;
   oResultData.ResultDriveTest.u8CntDVDLayer              = pDiagResultDriveTest->ResultDriveTest.u8CntDVDLayer;

   // construct result message
   fi_tclVisitorMessage oResultMsg( oResultData );

   ETG_TRACE_USR1(( "vSendAnswerDiagDriveTest: (part1) duration: %d, result: %d, test type: %d, drive: %d", ETG_ENUM(DURATION, pDiagResultDriveTest->e8Duration),
                    ETG_ENUM(DRV_TEST_RESULT, pDiagResultDriveTest->ResultDriveTest.e8RetValueDriveTest),
                    ETG_ENUM(TEST_TYPE, (tU8)pDiagResultDriveTest->ResultDriveTest.e16DiagDrvTesttype),
                    ETG_ENUM(DRIVE_TYPE, (tU8)pDiagResultDriveTest->ResultDriveTest.e16Drive) ));
   ETG_TRACE_USR1(( "vSendAnswerDiagDriveTest: (part2) Sectors read: %d, Read attempts: %d, Cnt errors: %d, Average read time per sector: %d, Total read time: %d, Max read time per sector: %d, Max LBA: %d",
                    pDiagResultDriveTest->ResultDriveTest.u32SectorsRead,   pDiagResultDriveTest->ResultDriveTest.u16ReadAttempts,
                    pDiagResultDriveTest->ResultDriveTest.u16CntErrors,     pDiagResultDriveTest->ResultDriveTest.u16AverageReadtimeSector,
                    pDiagResultDriveTest->ResultDriveTest.u32TotalReadTime, pDiagResultDriveTest->ResultDriveTest.u16MaxReadTimePerSector,
                    pDiagResultDriveTest->ResultDriveTest.u32MaxLBA  ));
   // initialise result message
   oResultMsg.vInitServiceData(  CCA_C_U16_APP_MMGR,                    // source
                                 poMsgDataForResult->u16SourceAppID,    // Target
                                 AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,     // StreamType
                                 0,                                     // StreamCounter
                                 poMsgDataForResult->u16RegisterID,     // RegisterID
                                 poMsgDataForResult->u16CmdCounter,     // nCmdCounter, 
                                 CCA_C_U16_SRV_MMGR,                    // nServiceID, 
                                 poMsgDataForResult->u16FunctionID,     // function ID
                                 AMT_C_U8_CCAMSG_OPCODE_METHODRESULT
                              );
   // send it to one client
   vSendVisitorMessage( &oResultMsg, poMsgDataForResult->u16SourceAppID );
}


/******************************************************************************
*
* FUNCTION:    vSendAnswerDiagDriveTestFirstMessageResult( const amt_tclServiceData* poMessage,
*                                                          tU8 u8Opcode )
* 
* DESCRIPTION: sends an answer message
*
* PARAMETER:   message to be answered, data, opcode of the answer
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclservice::vSendAnswerDiagDriveTestFirstMessageResult( const trMessageDataForMethodResult* poMsgDataForResult,
                                                                     tU8 u8Opcode )
{
   // +++   create a message, 
   //use the data of given message to get the address information    +++
   gm_tclEmptyMessage oFirstResultMsg = gm_tclEmptyMessage( CCA_C_U16_APP_MMGR,                    // source
                                                            poMsgDataForResult->u16SourceAppID,    // Target
                                                            poMsgDataForResult->u16RegisterID,     // RegisterID
                                                            poMsgDataForResult->u16CmdCounter,     // nCmdCounter,
                                                            CCA_C_U16_SRV_MMGR,                    // nServiceID,
                                                            poMsgDataForResult->u16FunctionID,     // function ID
                                                            u8Opcode
                                                          );
   // --- send it to the client ---
   if( poMain )
   {
      TRACE_CCA_OutMessage(&oFirstResultMsg, _CLASS, __LINE__);
      if( poMain->enPostMessage( &oFirstResultMsg ) != AIL_EN_N_NO_ERROR )
      {
         // +++         can't send :  enter error handling here ( trace or assert ) +++
         ETG_TRACE_ERR(( "vSendAnswerDiagDriveTestFirstMessageResult( ): Error: Cannot send CCA message to queue of APP ID: 0x%x", ETG_ENUM(ail_u16AppId, poMsgDataForResult->u16SourceAppID) ));
         if( !oFirstResultMsg.bDelete() )
         {
            ETG_TRACE_ERR(( "vSendAnswerDiagDriveTestFirstMessageResult( ): Error: CCA message which couldn't be posted couldn't be deleted also" ));
         }
      }
   }
}


/******************************************************************************
*
* FUNCTION:    vSendAnswerDiagEject( const trMessageDataForMethodResult* poMsgDataForResult,
*                              T_DiagMethodControlResultEject* pDiagResultEject, 
*                              tU8 u8Opcode )
* 
* DESCRIPTION: sends an answer message
*
* PARAMETER:   message to be answered, data, opcode of the answer
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclservice::vSendAnswerDiagEject( const trMessageDataForMethodResult* poMsgDataForResult,
                                  const T_DiagMethodControlResultEject* pDiagResultEject )
{
   // Prepare method result data
   mplay_mmgrfi_tclMsgDiagControlEjectMethodResult oResultData;
   // Copy data
   oResultData.e8Duration.enType            = (mplay_fi_tcl_e8_DiagDuration::tenType)pDiagResultEject->e8Duration;
   oResultData.ResultEject.e16Result.enType = (mplay_fi_tcl_e16_DiagResultEject::tenType)pDiagResultEject->ResultEject.e16Result;

   // construct result message
   fi_tclVisitorMessage oResultMsg( oResultData );

   // initialise result message
   oResultMsg.vInitServiceData(  CCA_C_U16_APP_MMGR,                    // source
                                 poMsgDataForResult->u16SourceAppID,    // Target
                                 AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,     // StreamType
                                 0,                                     // StreamCounter
                                 poMsgDataForResult->u16RegisterID,     // RegisterID
                                 poMsgDataForResult->u16CmdCounter,     // nCmdCounter,
                                 CCA_C_U16_SRV_MMGR,                    // nServiceID,
                                 poMsgDataForResult->u16FunctionID,     // function ID
                                 AMT_C_U8_CCAMSG_OPCODE_METHODRESULT
                              );
   // send it to one client
   vSendVisitorMessage( &oResultMsg, poMsgDataForResult->u16SourceAppID );
}


/******************************************************************************
*
* FUNCTION:    vSendAnswerDiagPerformTest( const amt_tclServiceData* poMessage,
*                                mplay_fi_tcl_DiagTestRoutineList* pDiagResultPerformTest )
* 
* DESCRIPTION: sends an answer message
*
* PARAMETER:   message to be answered, data, opcode of the answer
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclservice::vSendAnswerDiagPerformTest( const amt_tclServiceData* poMessage,
                                  const mplay_fi_tcl_DiagTestRoutineList* pDiagResultPerformTest )
{
   // Prepare method result data
   mplay_mmgrfi_tclMsgDiagPerformTestMethodResult oResultData;
   // Copy data
   oResultData.TestRoutineParameterList  = *pDiagResultPerformTest;

   // construct result message
   fi_tclVisitorMessage oResultMsg( oResultData );

   // Send it to the client
   vSendMethodResultVisitorMessage( poMessage, &oResultMsg );
}


/******************************************************************************
*
* FUNCTION:    vSendMethodResultDiagDrvTest( T_e8_DiagRetValueDriveTest e8TestResult,
*                                            T_e8_DiagDuration e8Duration,
*                                            T_DiagMethodResultDrvTest DiagMethodResultDrvTest )
* 
* DESCRIPTION: Preparation for vSendAnswerDiagDriveTest
*
* PARAMETER:   T_e8_DiagRetValueDriveTest    e8TestResult
*                 possible values: MMGR_RESULT_FALSE, MMGR_RESULT_OK
*                                  MMGR_RESULT_NO_MEDIA, MMGR_RESULT_INVALID_MEDIA
*              T_e8_DiagDuration             e8Duration
*                 possible values: MMGR_CONTINUOUS, MMGR_SINGLE_RUN
*              T_DiagMethodResultDrvTest     DiagMethodResultDrvTest
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclservice::vSendMethodResultDiagDrvTest( T_e8_DiagRetValueDriveTest e8TestResult,
                                                       T_e8_DiagDuration e8Duration,
                                                       const T_DiagMethodResultDrvTest* pDiagMethodResultDrvTest )
{
   T_DiagMethodControlResultDriveTest     DiagResultDriveTest;

   // Check if the test wasn't started via TTFis
   if( ! _bSimulatedCdDriveTest )
   {
      ETG_TRACE_USR1(( "vSendMethodResultDiagDrvTest( ): Enter: with drive test result: %d", ETG_ENUM(DRV_TEST_RESULT, e8TestResult) ));

      // Fill result struct
      DiagResultDriveTest.e8Duration                                 = e8Duration;
      DiagResultDriveTest.ResultDriveTest.e8RetValueDriveTest        = e8TestResult;
      DiagResultDriveTest.ResultDriveTest.e16DiagDrvTesttype         = pDiagMethodResultDrvTest->e16DiagDrvTesttype;
      DiagResultDriveTest.ResultDriveTest.e16Drive                   = pDiagMethodResultDrvTest->e16Drive;
      DiagResultDriveTest.ResultDriveTest.u32SectorsRead             = pDiagMethodResultDrvTest->u32SectorsRead;
      DiagResultDriveTest.ResultDriveTest.u16ReadAttempts            = pDiagMethodResultDrvTest->u16ReadAttempts;
      DiagResultDriveTest.ResultDriveTest.u16CntErrors               = pDiagMethodResultDrvTest->u16CntErrors;
      DiagResultDriveTest.ResultDriveTest.u16AverageReadtimeSector   = pDiagMethodResultDrvTest->u16AverageReadtimeSector;
      DiagResultDriveTest.ResultDriveTest.u32TotalReadTime           = pDiagMethodResultDrvTest->u32TotalReadTime;
      DiagResultDriveTest.ResultDriveTest.u16MaxReadTimePerSector    = pDiagMethodResultDrvTest->u16MaxReadTimePerSector;
      DiagResultDriveTest.ResultDriveTest.u32MaxLBA                  = pDiagMethodResultDrvTest->u32MaxLBA;
      DiagResultDriveTest.ResultDriveTest.u8CntDVDLayer              = 1; 

      // call send function
      vSendAnswerDiagDriveTest( &_MessageDataForMethodResultDiagDrvTest, &DiagResultDriveTest );
   }
}


/******************************************************************************
*
* FUNCTION:    vSendMethodResultDiagSelectTrackType( T_e8_TrackType e8ReqTrType, T_e8_TrackType e8ActTrType, T_e16_TrackTypeResult e8Result)
* 
* DESCRIPTION: sends an answer message
*
* PARAMETER:   message to be answered, data, opcode of the answer
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclservice::vSendMethodResultDiagSelectTrackType( T_e8_TrackType e8ActTrType, T_e8_TrackType e8ReqTrType, T_e16_TrackTypeResult e8Result )
{
   // Check if the test wasn't started via TTFis
   if( ! _bSimulatedSelectTrackType )
   {
      // Prepare method result data
      mplay_mmgrfi_tclMsgDiagCdSelectTrackTypeMethodResult   oResultData;

      // Copy data
      oResultData.ActTrackType.enType  = (mplay_fi_tcl_e8_TrackType::tenType)e8ActTrType;
      oResultData.ReqTrackType.enType  = (mplay_fi_tcl_e8_TrackType::tenType)e8ReqTrType;
      oResultData.Result.enType        = (mplay_fi_tcl_e8_TrackTypeResult::tenType)e8Result;

      // construct result message
      fi_tclVisitorMessage oResultMsg( oResultData );
      ETG_TRACE_USR1(( "vSendMethodResultDiagSelectTrackType( ): Act Type=%d; Req. Type=%d; Result=%d ", ETG_ENUM( TRACK_TYPE, e8ActTrType), ETG_ENUM( TRACK_TYPE, e8ReqTrType), ETG_ENUM( TRACK_TYPE_RESULT, e8Result) ));

      // initialise result message
      oResultMsg.vInitServiceData(  CCA_C_U16_APP_MMGR,                             // source
                                    _MsgDataMethResDiagSelectTrType.u16SourceAppID, // Target
                                    AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,              // StreamType
                                    0,                                              // StreamCounter
                                    _MsgDataMethResDiagSelectTrType.u16RegisterID,  // RegisterID
                                    _MsgDataMethResDiagSelectTrType.u16CmdCounter,  // nCmdCounter, 
                                    CCA_C_U16_SRV_MMGR,                             // nServiceID, 
                                    _MsgDataMethResDiagSelectTrType.u16FunctionID,  // function ID
                                    AMT_C_U8_CCAMSG_OPCODE_METHODRESULT
                                 );
      // send it to one client
      vSendVisitorMessage( &oResultMsg, _MsgDataMethResDiagSelectTrType.u16SourceAppID );
   }
}


/******************************************************************************
*
* FUNCTION:    vSendAnswerReadOnlyDevInterfaceTest
* 
* DESCRIPTION: sends an answer message to CD interface test
*
* PARAMETER:   message to be answered, Test initiator, test result 
*
* RETURNVALUE: void
*
*******************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
tVoid vdmmgr_tclservice::vSendAnswerReadOnlyDevInterfaceTest( const amt_tclServiceData* poMessage,
                                   mplay_fi_tcl_e8_DiagnosisTestType::tenType e8TestInitiator,
                                   mplay_fi_tcl_DiagTestResult::tenType e8TestResult )
{
   // Prepare method result data
   mplay_mmgrfi_tclMsgDiagReadOnlyDevInterfaceTestStatus oResultData;
   // Copy data
   oResultData.DiagnosisTestType.enType = e8TestInitiator;
   oResultData.DiagTestResult.enType = e8TestResult;

   // construct result message
   fi_tclVisitorMessage oResultMsg( oResultData );

   // Send it to the client
   vSendStatusAnswerVisitorMessage( poMessage, &oResultMsg );
}
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE


/******************************************************************************
*
* FUNCTION:    vSendMethodResultFirstDiagDrvTest( tVoid )
* 
* DESCRIPTION: Preparation for vSendAnswerDiagDriveTestFirstMessageResult()
*
* PARAMETER:   none
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclservice::vSendMethodResultFirstDiagDrvTest( tVoid )
{
   // Check if the test wasn't started via TTFis
   if( ! _bSimulatedCdDriveTest )
   {
      // call send function
      vSendAnswerDiagDriveTestFirstMessageResult( &_MessageDataForMethodResultDiagDrvTest,
                                                  AMT_C_U8_CCAMSG_OPCODE_METHODRESULTFIRST );
   }
}


/******************************************************************************
*  FUNCTION:    vHandleDiagFirmwareVersion( amt_tclServiceData* poMessage )
*
*  DESCRIPTION: react on messages with FID = MMGR_C_U16_FKTID_DIAGVERSIONFIRMWARE
*
*  PARAMETER:   message to analyse
*
*  RETURNVALUE: NONE
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tVoid vdmmgr_tclservice::vHandleDiagFirmwareVersion( amt_tclServiceData* poMessage )
{
   tU8                     u8OpCode;
   tU8                     u8Cnt;
   T_e16_DiagDrv           e16Drive;
   T_DiagFirmwareVersion   DiagFirmwareVersion;

   // Init result array
   for( u8Cnt = 0; u8Cnt< MMGR_MAX_FIRMWAREVERSIONLENGHT; u8Cnt++)
   {
      DiagFirmwareVersion.au8FirmwareVersion[u8Cnt] = VDMMGR_DIAG_INIT;
   }

   u8OpCode = poMessage->u8GetOpCode();

   TRACE_CCA_InMessage(poMessage, _CLASS);

   // ---   the opcode tells what we should do   ---
   switch( u8OpCode )
   {
      case AMT_C_U8_CCAMSG_OPCODE_GET: // --- the client asks for some data ---
      {
         // Get requested drive
         fi_tclVisitorMessage                      oMsg( poMessage );
         mplay_mmgrfi_tclMsgDiagFirmwareVersionGet  oData;
         (tVoid)oMsg.s32GetData( oData );
         e16Drive = (T_e16_DiagDrv)oData.Drive.enType;  // FI type is mplay_fi_tcl_e16_DiagDrv
         /*lint -save -e788 Info 788;enum constant 'x' not used within defaulted switch */
         switch( e16Drive )
         {
         #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
            case MMGR_CD:
               if( poMain && poMain->_vdmmgr_poCdctrlIf )
               {
                  poMain->_vdmmgr_poCdctrlIf->vDiagGetFirmwareVersion( &DiagFirmwareVersion );
               }
               break;
         #endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
            default:
               break;
         }
         /*lint -restore*/
         // +++ send the requested data to the client  +++
         vSendAnswerDiagFirmwareVersion( poMessage, &DiagFirmwareVersion );
         break;
      }
      default:
      {
         // +++  unknown opcode: enter error handling here ( trace or assert ) +++
         TRACE_Warning("unhandled OpCode", _CLASS);
         vSendError(poMessage, CCA_C_U16_ERROR_INVALID_OPCODE, _CLASS, __LINE__);
         break;
      }
   }
   (tVoid)poMessage->bDelete();
}


tVoid vdmmgr_tclservice::vHandleReadOnlyDevInterfaceTest( amt_tclServiceData* poMessage )
{
   tU8                     u8OpCode;
   T_e16_DiagDrv           e16Drive;

   u8OpCode = poMessage->u8GetOpCode();

   TRACE_CCA_InMessage(poMessage, _CLASS);

   switch( u8OpCode )
   {
      case AMT_C_U8_CCAMSG_OPCODE_GET: // --- the client asks for some data ---
      {
         // Get requested drive
         fi_tclVisitorMessage                      oMsg( poMessage );
         mplay_mmgrfi_tclMsgDiagReadOnlyDevInterfaceTestGet oData;
         mplay_fi_tcl_e8_DiagnosisTestType::tenType e8TestInitiator;
         mplay_fi_tcl_e16_DiagTestType::tenType e8TestType;
         mplay_fi_tcl_DiagTestResult::tenType e8TestResult = mplay_fi_tcl_DiagTestResult::FI_EN_TEST_PASSED;

         (tVoid)oMsg.s32GetData( oData );
         e16Drive = (T_e16_DiagDrv)oData.DiagnosisTestParam.e16Drive.enType;
         e8TestInitiator = oData.DiagnosisTestParam.DiagnosisTestInitiator.enType;
         e8TestType = oData.DiagnosisTestParam.TestType.enType;

         switch( e16Drive )
         {
         #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
            case MMGR_CD:
               if( poMain && poMain->_vdmmgr_poCdctrlIf )
               {
                  e8TestResult = poMain->_vdmmgr_poCdctrlIf->e8DiagCDInterfaceTest( e8TestInitiator, e8TestType );
               }
               break;
         #endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
            default:
               break;
         }
         vSendAnswerReadOnlyDevInterfaceTest( poMessage, e8TestInitiator, e8TestResult );
         break;
      }
      default:
      {
         break;
      }
   }
   (tVoid)poMessage->bDelete();
}


/******************************************************************************
*  FUNCTION:    vHandleDiagCdSelectTrackType( amt_tclServiceData* poMessage )
*
*  DESCRIPTION: react on messages with FID = MPLAY_MMGRFI_C_U16_DIAGCDSELECTTRACKTYPE
*
*  PARAMETER:   message to analyse
*
*  RETURNVALUE: NONE
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tVoid vdmmgr_tclservice::vHandleDiagCdSelectTrackType( amt_tclServiceData* poMessage )
{
   tU8                     u8OpCode;
   T_e8_TrackType          e8TrackType;

   u8OpCode = poMessage->u8GetOpCode( );

   TRACE_CCA_InMessage( poMessage, _CLASS );

   // --- the opcode tells what we should do ---
   switch( u8OpCode )
   {
      case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:     // --- the client wants the server to do something ---
      {
         _bSimulatedSelectTrackType = FALSE;       // No simulation via TTFis
         // +++ every METHODSTART message must be confirmed with a METHODRESULT message.
         // so send the actual state of the server ( concerning the given FID ) to the client +++ */
         // Get method start data into fi type
         fi_tclVisitorMessage                               oMsg( poMessage );
         mplay_mmgrfi_tclMsgDiagCdSelectTrackTypeMethodStart oData;
         (tVoid)oMsg.s32GetData( oData );
         // Copy data
         e8TrackType = (T_e8_TrackType)oData.TrackType.enType;
         // Store necessary message information for a later send of the methold result
         vCopyMsgDataToStoreForAnswer( poMessage, &_MsgDataMethResDiagSelectTrType );
         if( poMain && poMain->_vdmmgr_poCdctrlIf )
         {
            poMain->_vdmmgr_poCdctrlIf->vDiagSelectTrackType( e8TrackType );
         }
         break;
      }
      default:
      {
         // +++ unknown opcode: enter error handling here ( trace or assert ) +++
         TRACE_Warning("unhandled OpCode", _CLASS);
         vSendError(poMessage, CCA_C_U16_ERROR_INVALID_OPCODE, _CLASS, __LINE__);
         break;
      }
   }
   (tVoid)poMessage->bDelete();
}


/******************************************************************************
*  FUNCTION:    vHandleDiagDriveVersion( amt_tclServiceData* poMessage )
*
*  DESCRIPTION: react on messages with FID = MMGR_C_U16_FKTID_DIAGVERSIONDRIVE
*
*  PARAMETER:   message to analyse
*
*  RETURNVALUE: NONE
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tVoid vdmmgr_tclservice::vHandleDiagDriveVersion( amt_tclServiceData* poMessage )
{
   tU8                     u8OpCode;
   tU8                     u8Cnt;
   T_e16_DiagDrv           e16Drive;
   T_DiagDriveVersion      DiagDriveVersion;

   // Init result array
   for( u8Cnt = 0; u8Cnt< MMGR_MAX_FIRMWAREVERSIONLENGHT; u8Cnt++)
   {
      DiagDriveVersion.au8DriveVersion[u8Cnt] = VDMMGR_DIAG_INIT;
   }

   u8OpCode = poMessage->u8GetOpCode();

   TRACE_CCA_InMessage(poMessage, _CLASS);

   // ---   the opcode tells what we should do   ---
   switch( u8OpCode )
   {
      case AMT_C_U8_CCAMSG_OPCODE_GET: // --- the client asks for some data ---
      {
         // Get requested drive
         fi_tclVisitorMessage                   oMsg( poMessage );
         mplay_mmgrfi_tclMsgDiagDriveVersionGet  oData;
         (tVoid)oMsg.s32GetData( oData );
         e16Drive = (T_e16_DiagDrv)oData.Drive.enType;  // FI type is mplay_fi_tcl_e16_DiagDrv
         switch( e16Drive )
         {
         #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
            case MMGR_CD:
               if( poMain && poMain->_vdmmgr_poCdctrlIf )
               {
                  poMain->_vdmmgr_poCdctrlIf->vDiagGetDriveVersion( &DiagDriveVersion );
               }
               break;
         #endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
            default:
               break;
         }
         // +++ send the requested data to the client +++
         vSendAnswerDiagDriveVersion ( poMessage, &DiagDriveVersion );
         break;
      }
      default:
      {
         // +++ unknown opcode: enter error handling here ( trace or assert ) +++
         TRACE_Warning("unhandled OpCode", _CLASS);
         vSendError(poMessage, CCA_C_U16_ERROR_INVALID_OPCODE, _CLASS, __LINE__);
         break;
      }
   }
   (tVoid)poMessage->bDelete();
}


/******************************************************************************
*  FUNCTION:    vHandleDiagLoadStatus( amt_tclServiceData* poMessage )
*
*  DESCRIPTION: react on messages with FID = MMGR_C_U16_FKTID_DIAGVERSIONDRIVE
*
*  PARAMETER:   message to analyse
*
*  RETURNVALUE: NONE
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tVoid vdmmgr_tclservice::vHandleDiagLoadStatus( amt_tclServiceData* poMessage )
{
   tU8                     u8OpCode;
   T_e8_DiagLoadStatus     e8LoadStatus;

   // Init loader status -> lint
   e8LoadStatus = MMGR_NO_MEDIA_IN_DRIVE;

   u8OpCode = poMessage->u8GetOpCode();

   TRACE_CCA_InMessage(poMessage, _CLASS);

   // --- the opcode tells what we should do ---
   switch( u8OpCode )
   {
      case AMT_C_U8_CCAMSG_OPCODE_GET:       // --- the client asks for some data ---
      {
         // Get requested drive
         fi_tclVisitorMessage                oMsg( poMessage );
         mplay_mmgrfi_tclMsgDiagLoadStatusGet oData;
         (tVoid)oMsg.s32GetData( oData );
         switch( oData.Drive.enType )
         {
         #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
            case mplay_fi_tcl_e16_DiagDrv::FI_EN_MMGR_CD:
               if( poMain && poMain->_vdmmgr_poCdctrlIf)
               {
                  e8LoadStatus = poMain->_vdmmgr_poCdctrlIf->e8GetLoaderState( );
               }
               break;
         #endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
            default:
               break;
         }
         // +++ send the requested data to the client +++
         vSendAnswerDiagLoadStatus( poMessage, e8LoadStatus );
         break;
      }
      default:
      {
         // +++ unknown opcode: +++
         TRACE_Warning("unhandled OpCode", _CLASS);
         vSendError(poMessage, CCA_C_U16_ERROR_INVALID_OPCODE, _CLASS, __LINE__);
         break;
      }
   }
   (tVoid)poMessage->bDelete();
}


/******************************************************************************
*  FUNCTION:    vHandleDiagErrorStatus( amt_tclServiceData* poMessage )
*
*  DESCRIPTION: react on messages with FID = MMGR_C_U16_FKTID_DIAGERRORSTATUS
*
*  PARAMETER:   message to analyse
*
*  RETURNVALUE: NONE
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tVoid vdmmgr_tclservice::vHandleDiagErrorStatus( amt_tclServiceData* poMessage )
{
   tU8                                 u8OpCode;
   mplay_fi_tcl_e16_DiagTroubleCode     oDiagTroubleCode;
   mplay_fi_tcl_DiagTroubleStatus       oDiagTroubleStatus;

   //bpstl::vector<mplay_fi_tcl_DiagTroubleStatus,bpstl::allocator<mplay_fi_tcl_DiagTroubleStatus> > oTroubleStatusList;
   std::vector<mplay_fi_tcl_DiagTroubleStatus,std::allocator<mplay_fi_tcl_DiagTroubleStatus> > oTroubleStatusList;

   u8OpCode = poMessage->u8GetOpCode();

   TRACE_CCA_InMessage(poMessage, _CLASS);

   // ---   the opcode tells what we should do   ---
   switch( u8OpCode )
   {
      case AMT_C_U8_CCAMSG_OPCODE_GET: // --- the client asks for some data ---
      {
         // Get data
         fi_tclVisitorMessage                   oMsg( poMessage );
         mplay_mmgrfi_tclMsgDiagErrorStatusGet   oData;

         (tVoid)oMsg.s32GetData( oData );
         // Get data
         // vector for request list; init iterator -> .begin()
         //bpstl::vector< mplay_fi_tcl_e16_DiagTroubleCode,bpstl::allocator<mplay_fi_tcl_e16_DiagTroubleCode> >::iterator it = oData.TroubleCodeList.e16TroubleCodeList.begin();
		 std::vector< mplay_fi_tcl_e16_DiagTroubleCode,std::allocator<mplay_fi_tcl_e16_DiagTroubleCode> >::iterator it = oData.TroubleCodeList.e16TroubleCodeList.begin();
         // Go through ITC list
         for( ;it != oData.TroubleCodeList.e16TroubleCodeList.end(); ++it )
         {
            oDiagTroubleCode = *it;
            ETG_TRACE_USR1(( "vHandleDiagErrorStatus( ): Receive Parameter ITC: %d", ETG_ENUM(ITC, oDiagTroubleCode.enType) ));
            // Set ITC
            oDiagTroubleStatus.e16TroubleCode.enType = oDiagTroubleCode.enType;
            if( poMain && poMain->_vdmmgr_poCCAClienthandlerdiaglog )
            {
               // Get error status for ITC
               oDiagTroubleStatus.e16ErrorStatus.enType = (mplay_fi_tcl_e16_DiagErrorStatus::tenType)poMain->_vdmmgr_poCCAClienthandlerdiaglog->_ItcMap.bGetErrorStatusForITC( (tU16)oDiagTroubleCode.enType );
               // Get routine status for ITC
               oDiagTroubleStatus.e16RoutineStatus.enType = (mplay_fi_tcl_e16_DiagRoutineStatus::tenType)poMain->_vdmmgr_poCCAClienthandlerdiaglog->_ItcMap.bGetRoutineStatusForITC( (tU16)oDiagTroubleCode.enType );
            }
            // Add the trouble code status for actual ITC into trouble code status list
            oTroubleStatusList.push_back( oDiagTroubleStatus );
         }
         // Send Status
         vSendAnswerDiagErrorStatus( poMessage, &oTroubleStatusList );
         break;
      }
      default:
      {
         // +++ unknown opcode:   enter error handling here ( trace or assert ) +++
         TRACE_Warning("unhandled OpCode", _CLASS);
         vSendError(poMessage, CCA_C_U16_ERROR_INVALID_OPCODE, _CLASS, __LINE__);
         break;
      }
   }
   (tVoid)poMessage->bDelete();
}


/******************************************************************************
*  FUNCTION:    vHandleDiagDriveTest( amt_tclServiceData* poMessage )
*
*  DESCRIPTION: react on messages with FID = MMGR_C_U16_FKTID_DIAGDRIVETEST
*
*  PARAMETER:   message to analyse
*
*  RETURNVALUE: NONE
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tVoid vdmmgr_tclservice::vHandleDiagDriveTest( amt_tclServiceData* poMessage )
{
   tU8                                 u8OpCode;
   T_DiagMethodControlStartDriveTest   DiagStartDriveTest;

   u8OpCode = poMessage->u8GetOpCode( );

   TRACE_CCA_InMessage( poMessage, _CLASS );

   // --- the opcode tells what we should do ---
   switch( u8OpCode )
   {
      case AMT_C_U8_CCAMSG_OPCODE_METHODSTART:     // --- the client wants the server to do something ---
      {
         _bSimulatedCdDriveTest = FALSE;           // No simulation via TTFis
         // +++ every METHODSTART message must be confirmed with a METHODRESULT message.
         // so send the actual state of the server ( concerning the given FID ) to the client +++ */
         // Get method start data into fi type
         fi_tclVisitorMessage                               oMsg( poMessage );
         mplay_mmgrfi_tclMsgDiagControlDriveTestMethodStart  oData;
         (tVoid)oMsg.s32GetData( oData );
         // Copy data
         DiagStartDriveTest.e8Duration                            = (T_e8_DiagDuration)oData.e8Duration.enType;
         DiagStartDriveTest.StartDriveTestParam.e16Drive          = (T_e16_DiagDrv)oData.StartDriveTestParam.e16Drive.enType;
         DiagStartDriveTest.StartDriveTestParam.e16DriveTestCmd   = (T_e16_DiagDrvTestCmd)oData.StartDriveTestParam.e16DriveTestCmd.enType;
         DiagStartDriveTest.StartDriveTestParam.e16DriveTestParam = (T_e16_DiagDrvTestParam)oData.StartDriveTestParam.e16DriveTestParam.enType;
         // Store necessary message information for a later send of the method result
         vCopyMsgDataToStoreForAnswer( poMessage, &_MessageDataForMethodResultDiagDrvTest );
         // switch requested drive and call drive specific drive test function(s)
         switch( DiagStartDriveTest.StartDriveTestParam.e16Drive )
         {
         #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
            case MMGR_CD:
               if( poMain && poMain->_vdmmgr_poCdctrlIf )
               {
                  poMain->_vdmmgr_poCdctrlIf->vDiagDriveTest( &DiagStartDriveTest );
               }
               break;
         #endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
            default:
               break;
         }
         break;
      }
      default:
      {
         // +++ unknown opcode: enter error handling here ( trace or assert ) +++
         TRACE_Warning("unhandled OpCode", _CLASS);
         vSendError(poMessage, CCA_C_U16_ERROR_INVALID_OPCODE, _CLASS, __LINE__);
         break;
      }
   }
   (tVoid)poMessage->bDelete();
}


/******************************************************************************
*  FUNCTION:    vHandleDiagEject( amt_tclServiceData* poMessage )
*
*  DESCRIPTION: react on messages with FID = MMGR_C_U16_FKTID_DIAGEJECT
*
*  PARAMETER:   message to analyse
*
*  RETURNVALUE: NONE
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tVoid vdmmgr_tclservice::vHandleDiagEject( amt_tclServiceData* poMessage )
{
   tU8               u8OpCode;
   T_e16_DiagDrv     e16Drive;
   T_e8_DiagDuration e8DiagDuration;

   u8OpCode = poMessage->u8GetOpCode();

   TRACE_CCA_InMessage(poMessage, _CLASS);

   // ---   the opcode tells what we should do   ---
   switch( u8OpCode )
   {
      case AMT_C_U8_CCAMSG_OPCODE_METHODSTART: // --- the client wants the server to do something ---
      {
         // +++   every METHODSTART message must be confirmed with a METHODRESULT message.
         // so send the actual state of the server ( concerning the given FID ) to the client
         // you can change the function bSendAnswer to fulfill your needs          +++ 

         // Get method start data into FI type
         mplay_mmgrfi_tclMsgDiagControlEjectMethodStart   oData;
         (tVoid)fi_tclVisitorMessage( poMessage ).s32GetData( oData );
         // Copy data
         e8DiagDuration = (T_e8_DiagDuration)oData.e8Duration.enType;
         e16Drive       = (T_e16_DiagDrv)oData.StartEjectParam.e16Drive.enType;
         // Store necessary message information for a later send of the method result
         _MessageDataForMethodResultDiagEject.u16SourceAppID  = poMessage->u16GetSourceAppID();
         _MessageDataForMethodResultDiagEject.u16RegisterID   = poMessage->u16GetRegisterID();
         _MessageDataForMethodResultDiagEject.u16CmdCounter   = poMessage->u16GetCmdCounter();
         _MessageDataForMethodResultDiagEject.u16FunctionID   = poMessage->u16GetFunctionID();
         _MessageDataForMethodResultDiagEject.u16SourceSubID  = poMessage->u16GetSourceSubID( );
         // evalute method start data
         if( e8DiagDuration!= MMGR_SINGLE_RUN )
         {
            ETG_TRACE_ERR(( "vHandleDiagEject( ): Only SingelRun expected for DiagEject: 0x%x", ETG_ENUM(DURATION, e8DiagDuration) ));
         }
         switch( e16Drive )
         {
         #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
            case MMGR_CD:
               if( poMain && poMain->_vdmmgr_poCdctrlIf )
               {
                  // Set marker that CD eject was started and a method result 
                  // must be send if eject was successful (or not).
                  _MessageDataForMethodResultDiagEject.u32InternalData |= VDMMGR_DIAG_EJECT_CD_STARTED;
                  if( !poMain->_vdmmgr_poCdctrlIf->bIsEjectActive() )
                  {
                     // Store CD type before eject -> fast CD recognition in case of 'auto insert'.
                     _e8CDTypeBeforeEject = _e8ActCDType;
                     vNewCdType ( MMGR_EJECTING,        FALSE );  // Set CD type to 'Eject'.
                     vNewCdState( MMGR_MEDIA_NOT_READY, TRUE );   // Set CD state to 'media not ready'.
                  }
                  // Start eject.
                  (tVoid)poMain->_vdmmgr_poCdctrlIf->bEjectInThread( );
               }
               break;
         #endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
            default:
               break;
         }
         break;
      }
      default:
      {
         // +++ unknown opcode:   enter error handling here ( trace or assert ) +++
         TRACE_Warning( "unhandled OpCode", _CLASS );
         vSendError(poMessage, CCA_C_U16_ERROR_INVALID_OPCODE, _CLASS, __LINE__);
         break;
      }
   }
   (tVoid)poMessage->bDelete();
}


/******************************************************************************
*  FUNCTION:    vHandleDiagPerformTest( amt_tclServiceData* poMessage )
*
*  DESCRIPTION: react on messages with FID = MMGR_C_U16_FKTID_DIAGPERFOMRTEST
*
*  PARAMETER:   message to analyse
*
*  RETURNVALUE: NONE
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tVoid vdmmgr_tclservice::vHandleDiagPerformTest( amt_tclServiceData* poMessage )
{
   tU8                                    u8OpCode;
   mplay_fi_tcl_DiagTestRoutineParameter   oDiagTestRoutineParameter;

   u8OpCode = poMessage->u8GetOpCode();

   TRACE_CCA_InMessage(poMessage, _CLASS);

   // ---   the opcode tells what we should do   ---
   switch( u8OpCode )
   {
      case AMT_C_U8_CCAMSG_OPCODE_METHODSTART: // --- the client wants the server to do something ---
      {
         // +++   every METHODSTART message must be confirmed with a METHODRESULT message.
         // so send the actual state of the server ( concerning the given FID ) to the client
         // you can change the function bSendAnswer to fulfill your needs          +++
         // Get method start data into fi type
         fi_tclVisitorMessage                         oMsg( poMessage );
         mplay_mmgrfi_tclMsgDiagPerformTestMethodStart oData;
         (tVoid)oMsg.s32GetData( oData );
         // Get data
         //bpstl::vector<mplay_fi_tcl_DiagTestRoutineParameter, bpstl::allocator<mplay_fi_tcl_DiagTestRoutineParameter> >::iterator it = oData.TestRoutineParameterList.TestRoutineParameterList.begin();
		 std::vector<mplay_fi_tcl_DiagTestRoutineParameter, std::allocator<mplay_fi_tcl_DiagTestRoutineParameter> >::iterator it = oData.TestRoutineParameterList.TestRoutineParameterList.begin();
         for( ;it != oData.TestRoutineParameterList.TestRoutineParameterList.end(); ++it )
         {
            oDiagTestRoutineParameter = *it;

            ETG_TRACE_USR1(( "vHandleDiagPerformTest( ): Receive Parameter ITC: %d, Write to RM: %d, MonitoringMode: %d; Supervision Control %d",
                             ETG_ENUM(ITC, oDiagTestRoutineParameter.e16TroubleCode.enType),               ETG_ENUM(BOOL, oDiagTestRoutineParameter.e16TroubleCode.enType),
                             ETG_ENUM(MONITORING_MODE, oDiagTestRoutineParameter.e8MonitoringMode.enType), ETG_ENUM(SUPERVISION_CONTROL, oDiagTestRoutineParameter.e8SupervisionControl.enType) ));
         }
         // Send MethodResult
         vSendAnswerDiagPerformTest( poMessage, &oData.TestRoutineParameterList );
         break;
      }
      default:
      {
         // +++ unknown opcode:   enter error handling here ( trace or assert ) +++
         TRACE_Warning("unhandled OpCode", _CLASS);
         vSendError(poMessage, CCA_C_U16_ERROR_INVALID_OPCODE, _CLASS, __LINE__);
         break;
      }
   }
   (tVoid)poMessage->bDelete();
}

