/************************************************************************
* FILE:         vdmmgr_service.cpp
* PROJECT:      MIB2_ENTRY
* SW-COMPONENT: Virtual Device Media Manager
*----------------------------------------------------------------------
*
* DESCRIPTION: VD MediaManager
*              
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2005 Robert Bosch GmbH, Hildesheim
* HISTORY:      
* Date      | Author             | Modification
* 08.07.05  | CM-DI/ESA2 Fiebing | initial version
*
*************************************************************************/


#define _CLASS VDMMGR_TR_SERVICE

//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------
#include "Config.h"
#include <vector>

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_MEDIAMANAGER_SERVICE
#include "trcGenProj/Header/vdmmgr_service.cpp.trc.h"
#endif

#define AIL_S_IMPORT_INTERFACE_GENERIC
#include "ail_if.h"         // use AIL template with MessageMaps
#define AHL_S_IMPORT_INTERFACE_NOTIFICTABLE
#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"

#define VDMMGR_S_IMPORT_INTERFACE_MSG
#include "vdmmgr_if.h"                       // For VD MMgr interface

/* ************************************************************************** */
/* include the public interface                                               */
/* ************************************************************************** */
#include "vdmmgr_main.h"
#include "vdmmgr_timer.h"        // 'automatic reinsert CD' timer
#include "vdmmgr_service.h"
#include "vdmmgr_trace.h"
#include "vdmmgr_cdctrlif.h"

// +++ MESSAGE MAP: enter the function IDs (FID) and the corresponding functions here.
//        the function will be called when a message with the corresponding FID arrives +++ */
BEGIN_MSG_MAP( vdmmgr_tclservice, ahl_tclBaseWork )
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
   ON_MESSAGE( MPLAY_MMGRFI_C_U16_CDINFO,                          vHandleCDInfo )
   ON_MESSAGE( MPLAY_MMGRFI_C_U16_CDEJECT,                         vHandleEject )
   ON_MESSAGE( MPLAY_MMGRFI_C_U16_CDTEMP,                          vHandleCDTemp )
   ON_MESSAGE( MPLAY_MMGRFI_C_U16_CDINSERT,                        vHandleInsert )
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
   ON_MESSAGE( MPLAY_MMGRFI_C_U16_DRIVEERRORINFORMATION,           vHandleDriveErrorInformation )
   ON_MESSAGE( MPLAY_MMGRFI_C_U16_DIAGFIRMWAREVERSION,             vHandleDiagFirmwareVersion )
   ON_MESSAGE( MPLAY_MMGRFI_C_U16_DIAGDRIVEVERSION,                vHandleDiagDriveVersion )
   ON_MESSAGE( MPLAY_MMGRFI_C_U16_DIAGLOADSTATUS,                  vHandleDiagLoadStatus )
   ON_MESSAGE( MPLAY_MMGRFI_C_U16_DIAGERRORSTATUS,                 vHandleDiagErrorStatus )
   ON_MESSAGE( MPLAY_MMGRFI_C_U16_DIAGCONTROLDRIVETEST,            vHandleDiagDriveTest )
   ON_MESSAGE( MPLAY_MMGRFI_C_U16_DIAGCONTROLEJECT,                vHandleDiagEject )
   ON_MESSAGE( MPLAY_MMGRFI_C_U16_DIAGPERFORMTEST,                 vHandleDiagPerformTest )
   ON_MESSAGE( MPLAY_MMGRFI_C_U16_DIAGREADONLYDEVINTERFACETEST,    vHandleReadOnlyDevInterfaceTest )
   ON_MESSAGE( MPLAY_MMGRFI_C_U16_DIAGCDSELECTTRACKTYPE,           vHandleDiagCdSelectTrackType )
END_MSG_MAP( )


/******************************************************************************
*
* FUNCTION:    vdmmgr_tclservice( vdmmgr_tclApp* poVdMMgrMainApp )
* 
* DESCRIPTION: constructor, creates object vdmmgr_tclservice - object
*
* PARAMETER:   vdmmgr_tclApp* poVdMMgrMainApp: main - object of this application 
*
* RETURNVALUE: none
*
*******************************************************************************/
vdmmgr_tclservice::vdmmgr_tclservice( vdmmgr_tclApp* poVdMMgrMainApp )
{
   // This constructor calls ( implicit ) the constructor of the upper class 
   // ( framework ) so it registers the combinations of FID and message handler 
   // with the framework
   poMain                                 = poVdMMgrMainApp;
   poNotTable                             = poVdMMgrMainApp->_poNotTable;
   hNotTableSem                           = poVdMMgrMainApp->_hNotTableSem;//lint !e1705 Info 1705: static class members may be accessed by the scoping operator
   _bSendStatusOK                         = FALSE;
                                            
   _e8ActCDType                           = MMGR_NO_MEDIA;
   _bEjectStartet                         = FALSE;
   _e8CDTypeBeforeEject                   = MMGR_NO_MEDIA;
   _e8ActCDState                          = MMGR_MEDIA_NOT_READY;
   _e8ActCDInsertState                    = MMGR_INSERT_BEFOREON;
   _e8ActCDDeviceState                    = MMGR_DEVICE_NOT_READY;
   _e8CdDrivevMounted                     = TRUE;
   _clActDriveError.e16Drive.enType       = mplay_fi_tcl_e16_DiagDrv::FI_EN_MMGR_CD;
   _clActDriveError.e16ErrorValue.enType  = mplay_fi_tcl_e16_DriveErrorValue::FI_EN_DEFECT_NO_ERROR;
   _e8CheckDiskInitiator                  = mplay_fi_tcl_e8_DiagnosisTestType::FI_EN_MMGR_TEST_TYPE_OTHER;
   _bAppStateChangeWaitForInsert          = FALSE;
   _MessageDataForMethodResultDiagEject.u32InternalData = 0;
   // Create timer for 'automatic reinsert of CD'
   (tVoid)_oAutoReInsertTimer.Create( VDMMGR_TIMER_TIMEOUT_FIRST_AUTOMATI_CDREINSERT, VDMMGR_TIMER_TIMEOUT_INTERVALL,
                                      this, VDMMGR_TIMERTYPE_AUTOMATIC_REINSERT );
   // Create timer for 'ensure CD NO_MEDIA' (if MASCA CD drive don't give insert clamps information)
   (tVoid)_oEnsureCdNoMediaTimer.Create( VDMMGR_TIMER_TIMEOUT_FIRST_ENSURE_CD_NO_MEDIA, VDMMGR_TIMER_TIMEOUT_INTERVALL,
                                         this, VDMMGR_TIMERTYPE_ENSURE_CD_NO_MEDIA );
   // For simulation via TTFis
   _bSimulatedSelectTrackType             = FALSE;
   _bSimulatedCdDriveTest                 = FALSE;
   for(tU8 u8Cnt = 0; u8Cnt< MMGR_MAX_DRIVEVERSIONLENGTH; u8Cnt++)
   {
      _diagDriveVersion.au8DriveVersion[u8Cnt] = VDMMGR_DIAG_INIT;
   }
}


vdmmgr_tclservice::vdmmgr_tclservice( )
{
}//lint !e1744 member 'x possibly not initialized by private constructor


tVoid vdmmgr_tclservice::vSetIsSendStatusOK( tBool bSendStatusOK )
{
   // Copy new status into private member
   _bSendStatusOK = bSendStatusOK;
}


/******************************************************************************
*
* FUNCTION:    vOnUnknownMessage( amt_tclBaseMessage* poMessage )
* 
* DESCRIPTION: handle unknown message
*              the frame work call this function when it receives a message 
*              with a unknown FID. You can use it for error handling.
*
* PARAMETER:   unknown message
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclservice::vOnUnknownMessage(amt_tclBaseMessage* poMessage )
{
   amt_tclServiceData* poServiceMsg;

   if( poMessage )
   {
      poServiceMsg = dynamic_cast<amt_tclServiceData*>(poMessage);
      if( poServiceMsg )
      {
         ETG_TRACE_ERR(( "vOnUnknownMessage: Function ID:%d; From App:%d",
                         ETG_ENUM(MPLAY_MMGRFI, poServiceMsg->u16GetFunctionID()),
                         ETG_ENUM(ail_u16AppId, poMessage->u16GetSourceAppID()) ));
      }
      (tVoid)poMessage->bDelete();
   }
}


/******************************************************************************
*
* FUNCTION:    vOnNewAppState( tU32 u32OldAppState, tU32 u32AppState )
* 
* DESCRIPTION: handles state - change messages from the spm
*
* PARAMETER:   old state, new state
*
* RETURNVALUE: time in ms to delay the system shutdown due to CD insert
*
*******************************************************************************/
tU32 vdmmgr_tclservice::u32OnNewAppState( tU32 u32OldAppState, tU32 u32AppState )
{
   OSAL_tMSecond  tTimeInMilliSecondsToStateChange = 0;

   TRACE_SPM_Transition( u32OldAppState, u32AppState, _CLASS );

   if( u32OldAppState != u32AppState )
   {
      switch( u32AppState )
      {
         case AMT_C_U32_STATE_NORMAL:
         case AMT_C_U32_STATE_DIAGNOSIS:
         {
         }
         break;

         case AMT_C_U32_STATE_PAUSE:
         {
         #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
            tTimeInMilliSecondsToStateChange = u32SendCDInsertEvDueToNewAppState( );
         #endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
         }
         break;

         case AMT_C_U32_STATE_OFF:
         {
         #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
            tTimeInMilliSecondsToStateChange = u32SendCDInsertEvDueToNewAppState( );
            // Store last mode data in flash
            vSetLastModeData( );
         #endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
         }
         break;

         default:
         {
            TRACE_Warning("unhandled power state", _CLASS);
            break;
         }
      }
   }
   // else: switching to the same state -> no action required

   return tTimeInMilliSecondsToStateChange;
}


tU32 vdmmgr_tclservice::u32SendCDInsertEvDueToNewAppState( )
{
   //T_e8_DiagLoadStatus  e8LoadStatus;
   OSAL_tMSecond        tTimeInMilliSecondsToStateChange = 0;
   if( poMain && poMain->_vdmmgr_poCdctrlIf )
   {
      // First get loader state
      //e8LoadStatus = poMain->_vdmmgr_poCdctrlIf->e8GetLoaderState( );
      //ETG_TRACE_USR4(( "u32SendCDInsertEvDueToNewAppState(): LoaderState=%d", ETG_ENUM(LOADER_STATE, e8LoadStatus) ));
      // Check if CD is in slot (static or ejecting)
      /*if(   (   ( e8LoadStatus == MMGR_EJECT_IN_PROGRESS )
             && ( _bEjectStartet )
            )
         || ( e8LoadStatus == MMGR_MEDIA_IN_SLOT )
        )*/ 
	    //comments above lines for avoid OSAL hung in GetLoaderState() when Power off
		ETG_TRACE_USR4(( "u32SendCDInsertEvDueToNewAppState(): bEjectOngoing=%d", poMain->_vdmmgr_poCdctrlIf->bEjectOngoing ));
        if(poMain->_vdmmgr_poCdctrlIf->bEjectOngoing)
      {
         ETG_TRACE_USR1(( "u32SendCDInsertEvDueToNewAppState(): Delay system shutdown up to 20s due to CD insert. Already delayed before: %d", ETG_ENUM(BOOL, _bAppStateChangeWaitForInsert) ));
         tTimeInMilliSecondsToStateChange = VDMMGR_TIME_DELAY_NEW_APPSTATE_FOR_CDINSERT;     // delay 20s
      }
      // Check if the event was already send once.
      if( ! _bAppStateChangeWaitForInsert )
      {
         poMain->_vdmmgr_poCdctrlIf->vSendEventToCdCtrlThread( VDMMGR_EVENT_MASK_CMD_AUTOREINSERTCD );
         _bAppStateChangeWaitForInsert = TRUE;                                               // set marker to call vAppStateChanged() once the insert funktion returned
      }
   }
   return tTimeInMilliSecondsToStateChange;
}


//-----------------------------------------------------------------------------
// handle functions
//-----------------------------------------------------------------------------

/*************************************************************************
*
* FUNCTION:    tVoid vSendVisitorMessage( fi_tclVisitorMessage*  poResultMsg, tU16 u16AppId )
* 
* DESCRIPTION: send messages function for message made with 
*              tcl_VistorMessages
*
* PARAMETER:   Function - Visitor message to send
*
* RETURNVALUE: void
*
*************************************************************************/
tVoid vdmmgr_tclservice::vSendVisitorMessage( fi_tclVisitorMessage*  poResultMsg, tU16 u16AppId )
{
   // ---   send it to one client ---
   if( poMain )
   {
      TRACE_CCA_OutMessage(((amt_tclServiceData*)poResultMsg), _CLASS, __LINE__);

      if( poMain->enPostMessage( poResultMsg ) != AIL_EN_N_NO_ERROR )
      {
         // +++ can't send status:  enter error handling here ( trace or assert ) +++
         ETG_TRACE_ERR(( "vSendVisitorMessage( ): Error: Cannot send CCA message to queue of APP ID: 0x%x", ETG_ENUM(ail_u16AppId, u16AppId) ));
         if( !poResultMsg->bDelete() )
         {
            ETG_TRACE_ERR(( "vSendVisitorMessage( ): Error: CCA message wich couldn't be posted couldn't be deleted also" ));
         }
      }
   }
}


/*************************************************************************
*
* FUNCTION:    tVoid vSendStatusVisitorMessage( ahl_tNotification* pNot, 
*                          tU16 u16FID, fi_tclVisitorMessage* poResultMsg )
* 
* DESCRIPTION: send messages function for message made with 
*              tcl_VistorMessages
*
* PARAMETER:   Function - Visitor message to send
*
* RETURNVALUE: void
*
*************************************************************************/
tVoid vdmmgr_tclservice::vSendStatusVisitorMessage( const ahl_tNotification* pNot, tU16 u16FID, fi_tclVisitorMessage* poResultMsg )
{
   // initialise result message
   poResultMsg->vInitServiceData(   CCA_C_U16_APP_MMGR,                 // source
                                    pNot->u16AppID,                     // Target
                                    AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,  // StreamType
                                    0,                                  // StreamCounter
                                    pNot->u16RegisterID,                // RegisterID
                                    pNot->u16CmdCounter,                // nCmdCounter,
                                    CCA_C_U16_SRV_MMGR,                 // nServiceID,
                                    u16FID,                             // function ID
                                    AMT_C_U8_CCAMSG_OPCODE_STATUS       // OpCode
                                );
   // send it to one client
   vSendVisitorMessage( poResultMsg, pNot->u16AppID );
}


/*************************************************************************
*
* FUNCTION: tVoid vSendStatusAnswerVisitorMessage( const amt_tclServiceData* poMessage,
*                                                  fi_tclVisitorMessage* poResultMsg,
*                                                   const trMessageDataForMethodResult* poMsgData=NULL)
* 
* DESCRIPTION: send messages function for message made with 
*              tcl_VistorMessages
*
* PARAMETER:   Function - Visitor message to send
*
* RETURNVALUE: void
*
*************************************************************************/
tVoid vdmmgr_tclservice::vSendStatusAnswerVisitorMessage( const amt_tclServiceData* poMessage, fi_tclVisitorMessage* poResultMsg,
                                                          const trMessageDataForMethodResult* poMsgData/*=NULL*/ )
{
   if( poResultMsg )
   {
      if( poMessage )
      {
         // initialise result message
         poResultMsg->vInitServiceData( CCA_C_U16_APP_MMGR,                // source
                                        poMessage->u16GetSourceAppID(),    // Target
                                        AMT_C_U8_CCAMSG_STREAMTYPE_NODATA, // StreamType
                                        0,                                 // StreamCounter
                                        poMessage->u16GetRegisterID(),     // RegisterID
                                        poMessage->u16GetCmdCounter(),     // nCmdCounter, 
                                        CCA_C_U16_SRV_MMGR,                // nServiceID, 
                                        poMessage->u16GetFunctionID(),     // function ID
                                        AMT_C_U8_CCAMSG_OPCODE_STATUS
                                      );
         // send it to one client
         vSendVisitorMessage( poResultMsg, poMessage->u16GetSourceAppID() );
      }
      else if( poMsgData )
      {
         // initialise result message
         poResultMsg->vInitServiceData( CCA_C_U16_APP_MMGR,                // source
                                        poMsgData->u16SourceAppID,         // Target
                                        AMT_C_U8_CCAMSG_STREAMTYPE_NODATA, // StreamType
                                        0,                                 // StreamCounter
                                        poMsgData->u16RegisterID,          // RegisterID
                                        poMsgData->u16CmdCounter,          // nCmdCounter, 
                                        CCA_C_U16_SRV_MMGR,                // nServiceID, 
                                        poMsgData->u16FunctionID,          // function ID
                                        AMT_C_U8_CCAMSG_OPCODE_STATUS
                                    );
         // send it to one client
         vSendVisitorMessage( poResultMsg, poMsgData->u16SourceAppID );
      }
      else
      {
         TRACE_Error( "vSendStatusAnswerVisitorMessage: Error: Missing pointer to message data. Neither a pointer to a message nor a pointer to message data struct is given!", _CLASS );
      }
   }
}



/*************************************************************************
*
* FUNCTION:    tVoid vSendMethodResultVisitorMessage( const amt_tclServiceData* poMessage,
*                                                     fi_tclVisitorMessage* poResultMsg,
*                                                     tU8 u8Opcode=AMT_C_U8_CCAMSG_OPCODE_METHODRESULT )
* 
* DESCRIPTION: send method-result(-result_first, ...) messages for message made with
*              tcl_VistorMessages
*
* PARAMETER:   Function - Visitor message to send
*
* RETURNVALUE: void
*
*************************************************************************/
tVoid vdmmgr_tclservice::vSendMethodResultVisitorMessage( const amt_tclServiceData* poMessage, fi_tclVisitorMessage* poResultMsg,
                                                          tU8 u8Opcode/*=AMT_C_U8_CCAMSG_OPCODE_METHODRESULT*/ )
{
   // initialise result message
   poResultMsg->vInitServiceData(  CCA_C_U16_APP_MMGR,                 // source
                                   poMessage->u16GetSourceAppID(),     // Target
                                   AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,  // StreamType
                                   0,                                  // StreamCounter
                                   poMessage->u16GetRegisterID(),      // RegisterID
                                   poMessage->u16GetCmdCounter(),      // nCmdCounter,
                                   CCA_C_U16_SRV_MMGR,                 // nServiceID,
                                   poMessage->u16GetFunctionID(),      // function ID
                                   u8Opcode
                                );
   // send it to one client
   vSendVisitorMessage( poResultMsg, poMessage->u16GetSourceAppID() );
}


/******************************************************************************
 *
 * FUNCTION: vCopyMsgDataToStoreForAnswer( amt_tclServiceData* poMessage, trMessageDataForMethodResult* pStoreStruct )
 *
 * 
 * DESCRIPTION:  Copy AMT Message Data to struct trMessageDataForMethodResult
 *
 *  
 * PARAMETER:  poMessage   :     copy from pointer
 *
 *             pStoreStruct:     copy to pointer
 *  
 * RETURNVALUE: tVoid
 * 
 *
*******************************************************************************/
tVoid vdmmgr_tclservice::vCopyMsgDataToStoreForAnswer( const amt_tclServiceData* poMessage, trMessageDataForMethodResult* pStoreStruct ) const
{
   if( poMessage )
   {
      pStoreStruct->u16SourceAppID = poMessage->u16GetSourceAppID( );
      pStoreStruct->u16RegisterID  = poMessage->u16GetRegisterID( );
      pStoreStruct->u16CmdCounter  = poMessage->u16GetCmdCounter( );
      pStoreStruct->u16FunctionID  = poMessage->u16GetFunctionID( );
      pStoreStruct->u16SourceSubID = poMessage->u16GetSourceSubID( );
   }
   else     // For Simulation from TTFis -> Don't send message if Application ID if VD MMgr Application ID
   {
      pStoreStruct->u16SourceAppID = CCA_C_U16_APP_MMGR;
      pStoreStruct->u16RegisterID  = 0;
      pStoreStruct->u16CmdCounter  = 0;
      pStoreStruct->u16FunctionID  = 0;
      pStoreStruct->u16SourceSubID = 0;
   }
}


/******************************************************************************
*
* FUNCTION:    tVoid dl_tclApp::vSendError
* 
* DESCRIPTION: sends an error - message
*
* PARAMETER:   data for addressing the message
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclservice::vSendError(  amt_tclServiceData* poInMessage, tU16 u16ErrorCode, tU16 u16Class, tU32 u32Line ) const
{
   TRACE_On_Invalid_Pointer(poInMessage, "poInMessage", u16Class);
   TRACE_On_Invalid_Pointer(poMain, "poMain", u16Class);

   if ((poInMessage != NULL) && (poMain != NULL))
   {
      amt_tclServiceDataError oErrorMsg(poInMessage, u16ErrorCode);
      TRACE_CCA_OutMessage(&oErrorMsg, u16Class, u32Line);

      if( poMain->enPostMessage(&oErrorMsg)!= AIL_EN_N_NO_ERROR )
      {
         TRACE_Error("Message posting failed", u16Class);
         if ( ! oErrorMsg.bDelete())
         {
            TRACE_Error("Message deletion failed", u16Class);
         };
      }
   }
  
}

/******************************************************************************
*
* FUNCTION:    tBool vdmmgr_tclservice::bUpreg(...)
* 
* DESCRIPTION: insert the sender of the message to the NotifcationTable
*              (i.e. register this client)
*
* PARAMETER:   amt_tclServiceData* poMessage: upreg request message
*
* RETURNVALUE: void
*
*******************************************************************************/
tBool vdmmgr_tclservice::bUpreg( const amt_tclServiceData* poMessage )
{
   tBool bRegOpStatus;

   (tVoid)ahl_bEnterCritical(hNotTableSem);

   bRegOpStatus =  poNotTable->bAddNotification(   poMessage->u16GetFunctionID(),
                                                   poMessage->u16GetSourceAppID(),
                                                   poMessage->u16GetRegisterID(),
                                                   1,
                                                   poMessage->u16GetCmdCounter(),
                                                   poMessage->u16GetSourceSubID()
                                               );
   (tVoid)ahl_bReleaseCritical(hNotTableSem); 

   return bRegOpStatus;
}

/******************************************************************************
*
* FUNCTION:    tBool vdmmgr_tclservice::bRelUpreg(...)
* 
* DESCRIPTION: remove the sender of the message from the NotifcationTable
*              (i.e. unregister this client)
*
* PARAMETER:   amt_tclServiceData* poMessage: relupreg request message
*
* RETURNVALUE: void
*
*******************************************************************************/
tBool vdmmgr_tclservice::bRelUpreg( const amt_tclServiceData* poMessage  )
{
   tBool bRegOpStatus;

   (tVoid)ahl_bEnterCritical( hNotTableSem );

   bRegOpStatus =  poNotTable->bRemoveNotification(  poMessage->u16GetFunctionID(),
                                                     poMessage->u16GetSourceAppID(),
                                                     poMessage->u16GetRegisterID(),
                                                     1,
                                                     poMessage->u16GetCmdCounter(),
                                                     poMessage->u16GetSourceSubID()
                                                  );
   (tVoid)ahl_bReleaseCritical(hNotTableSem); 

   return bRegOpStatus;
}


/******************************************************************************
*  FUNCTION:    vHandleDriveErrorInformation( amt_tclServiceData* poMessage )
*
*  DESCRIPTION: react on messages with FID = MMGR_C_U16_FKTID_DRIVE_ERROR_INFORMATION
*
*  PARAMETER:   message to analyse
*
*  RETURNVALUE: NONE
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tVoid vdmmgr_tclservice::vHandleDriveErrorInformation( amt_tclServiceData* poMessage )
{
   TRACE_CCA_InMessage(poMessage, _CLASS);

   // ---   the opcode tells what we should do   ---
   tU8 u8OpCode = poMessage->u8GetOpCode();
   switch( u8OpCode )
   {
      case AMT_C_U8_CCAMSG_OPCODE_UPREG:   // ---   upreg: register client  ---
      {
         if( !bUpreg( poMessage ) )
         {  // --- relupreg failed: send an error message ---
            vSendError( poMessage, CCA_C_U16_ERROR_UPREG_FAILURE, _CLASS, __LINE__ );
         }
         else
         {  // --- upreg was successful: every upreg message must be confirmed with a status message.
            //so send the actual state of the server ( concerning the given FID ) to the client  ---
            vSendAnswerDriveErrorInformation( poMessage, &_clActDriveError );
         }
         break;
      }
      case AMT_C_U8_CCAMSG_OPCODE_RELUPREG: // ---   upreg: unregister client  ---
      {
         if( !bRelUpreg(poMessage) )
         {  // --- relupreg failed: send an error message ---
            vSendError( poMessage, CCA_C_U16_ERROR_RELUPREG_FAILURE, _CLASS, __LINE__ );
         }
         // no else: --- the relupreg was successful so you can't talk to this client anymore.
         //   therefore there are normally no actions to be done in this case  --- */
         break;
      }
      case AMT_C_U8_CCAMSG_OPCODE_GET: 
      {
         vSendAnswerDriveErrorInformation( poMessage, &_clActDriveError );
         break;
      }
      default:
      {  // unknown opcode
         TRACE_Warning("unhandled OpCode", _CLASS);
         vSendError(poMessage, CCA_C_U16_ERROR_INVALID_OPCODE, _CLASS, __LINE__);
         break;
      }
   }
   (tVoid)poMessage->bDelete();
}


/******************************************************************************
*
* FUNCTION:    vSendStatusDriveErrorInformation ( mplay_fi_tcl_DriveErrorInfo* pclDriveErrorInfo )
* 
* DESCRIPTION: sends an status - message for FID MMGR_C_U16_FKTID_DRIVE_ERROR_INFORMATION
*
* PARAMETER:   Function - ID and data to send
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclservice::vSendStatusDriveErrorInformation( const mplay_fi_tcl_DriveErrorInfo* pclDriveErrorInfo )
{
   (tVoid)ahl_bEnterCritical( hNotTableSem );

   if( _bSendStatusOK )    // Test if service is available
   {
      // Run through the table to find all registered clients
      for( ahl_tNotification* pNot= poNotTable->poGetNotificationList( MMGR_C_U16_FKTID_DRIVE_ERROR_INFORMATION );
           pNot != OSAL_NULL; pNot=pNot->pNext )
      {
         // Prepare method result data
         mplay_mmgrfi_tclMsgDriveErrorInformationStatus   oResultData;
         // Copy data
         oResultData.DriveErrorInfo.e16Drive.enType      = pclDriveErrorInfo->e16Drive.enType;
         oResultData.DriveErrorInfo.e16ErrorValue.enType = pclDriveErrorInfo->e16ErrorValue.enType;

         // construct result message
         fi_tclVisitorMessage oResultMsg( oResultData );

         vSendStatusVisitorMessage( pNot, MMGR_C_U16_FKTID_DRIVE_ERROR_INFORMATION, &oResultMsg );
      }
   }
   (tVoid)ahl_bReleaseCritical( hNotTableSem );
}


/******************************************************************************
*
* FUNCTION:    vSendAnswerDriveErrorInformation ( const amt_tclServiceData* poMessage,
*                                 const mplay_fi_tcl_DriveErrorInfo* pclDriveErrorInfo )
* 
* DESCRIPTION: sends an answer message
*
* PARAMETER:   message to be answered, data, opcode of the answer
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclservice::vSendAnswerDriveErrorInformation ( const amt_tclServiceData* poMessage, 
                                                            const mplay_fi_tcl_DriveErrorInfo* pclDriveErrorInfo )
{
   // Prepare method result data
   mplay_mmgrfi_tclMsgDriveErrorInformationStatus   oResultData;
   // Copy data
   oResultData.DriveErrorInfo.e16Drive.enType      = pclDriveErrorInfo->e16Drive.enType;
   oResultData.DriveErrorInfo.e16ErrorValue.enType = pclDriveErrorInfo->e16ErrorValue.enType;

   // construct result message
   fi_tclVisitorMessage oResultMsg( oResultData );

   // send it to client
   vSendStatusAnswerVisitorMessage( poMessage, &oResultMsg );
}

