/******************************************************************************
* FILE:         vdmmgr_main.cpp
* PROJECT:      MIB2_ENTRY
* SW-COMPONENT: Virtual Device Media Manager
*------------------------------------------------------------------------------
*
* DESCRIPTION: VD MediaManager
*
*------------------------------------------------------------------------------
* COPYRIGHT:    (c) 2005 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 27.05.05  | CM-DI/ESA2 Fiebing | initial version
*
*******************************************************************************/
#define _CLASS  VDMMGR_TR_MAIN

//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------
#include "Config.h"
#include <map>
#include <vector>

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_MEDIAMANAGER_MAIN
#include "trcGenProj/Header/vdmmgr_main.cpp.trc.h"
#endif

#define AIL_S_IMPORT_INTERFACE_GENERIC
#include "ail_if.h"                                   // use AIL template with MessageMaps
//#define AHL_S_IMPORT_INTERFACE_NOTIFICTABLE
//#define AHL_S_IMPORT_INTERFACE_GENERIC
//#include "ahl_if.h"

#define VDMMGR_S_IMPORT_INTERFACE_MSG
#include "vdmmgr_if.h"                                // For VD MMgr interface

#include "vdmmgr_main.h"
#include "vdmmgr_timer.h"                             // 'automatic reinsert CD' timer
#include "vdmmgr_service.h"
#include "vdmmgr_errorif.h"
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
   #include "vdmmgr_cdctrlif.h"
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
#define CFC_FI_S_IMPORT_INTERFACE_CFC_SPMFI_TYPES     // or #define CFC_FI_S_IMPORT_INTERFACE_CFC_SPMFI_STDVISITORS
#include "cfc_fi_if.h"
#include "vdmmgr_clienthandlerspm.h"
#include "vdmmgr_ITC_Container.h"
#include "vdmmgr_ITC_Map.h"
#include "vdmmgr_clienthandlerdiaglog.h"
#include "vdmmgr_trace.h"

OSAL_tSemHandle vdmmgr_tclApp::_hNotTableSem  = OSAL_C_INVALID_HANDLE;
tBool           vdmmgr_tclApp::_bAppClosed    = FALSE;

vdmmgr_tclApp*   vdmmgr_tclApp::_vdmmgr_poExemplar = NULL;
vdmmgr_tclApp*   vdmmgr_tclApp::_theServer         = NULL;


/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     vdmmgr_tclApp::theServer
 *
 *DESCRIPTION:  create new server object
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  vdmmgr_tclApp* _theServer
 *
 *HISTORY:
 *27.01.07 kol2hi
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
vdmmgr_tclApp* vdmmgr_tclApp::theServer()
{
   if (_theServer == NULL)
   {
      _theServer = new vdmmgr_tclApp();
   }
   return _theServer;
}


/******************************************************************************
*
* FUNCTION:    vdmmgr_tclApp::vdmmgr_tclApp()
*
* DESCRIPTION:
*
* Constructor: - Initialize AMT, create semaphore, create notification table
*
*
* PARAMETER:   void
*
* RETURNVALUE: none
*
*******************************************************************************/
vdmmgr_tclApp::vdmmgr_tclApp()
 :
   _vdmmgr_poCCAService(OSAL_NULL),
   _vdmmgr_poCCAClienthandlerspm(OSAL_NULL),
   _vdmmgr_poCCAClienthandlerdiaglog(OSAL_NULL),
   _vdmmgr_poErrorIf(OSAL_NULL),
   _vdmmgr_poCdctrlIf(OSAL_NULL),
   _vdmmgr_poTraceHandler(OSAL_NULL)
{
   // --- initialize the AMT/CCA framework for this component  ---
   (tVoid)amt_bInit( );
   // --- create semaphore for protecting NotificationTable ---
   if( OSAL_s32SemaphoreCreate( VDMMGR_NOTTABLE_SEM_NAME, &_hNotTableSem, 1 ) == OSAL_ERROR )
   {
      TRACE_OSAL_ERROR( VDMMGR_TR_MAIN );
   }
   // --- NotificationTable:    Here we store information about all registered clients ---
   _poNotTable = new ahl_tclNotificationTable(VDMMGR_FUNCARRAY_SIZE);
   // Init AppState
   _u32AppState = AMT_C_U32_STATE_INVALID;
   // --- create the CCA service handler: It analyses the received CCA messages and calls the corresponding handler function   ---
   _vdmmgr_poCCAService = new vdmmgr_tclservice(this);
   // ~~~ --- create the CCA message handler: It analyses the received CCA messages and calls the corresponding handler function  --- ~~~
   _vdmmgr_poCCAClienthandlerspm       = new vdmmgr_tclclienthandlerspm(this);
   // Give information to error/report memory
   _vdmmgr_poCCAClienthandlerdiaglog   = new vdmmgr_tclclienthandlerdiaglog( this );
   // Gets information from VD Dimming
   // Trace stuff
   _vdmmgr_poExemplar                  = this;
   _vdmmgr_poTraceHandler              = new vdmmgr_tclTrace( this );

   _fdTrace = OSAL_ERROR;
   vRegisterTraceService();
}


/******************************************************************************
*
* FUNCTION:    ~vdmmgr_tclApp()
*
* DESCRIPTION: destructor, frees all resources
*             ( if not done by vOnApplicationClose before )
*
* PARAMETER:   void
*
* RETURNVALUE: none
*
*******************************************************************************/
vdmmgr_tclApp::~vdmmgr_tclApp()
{
   // +++ free all allocated resources +++
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
   if( _vdmmgr_poCdctrlIf != NULL )
   {
      vDeleteNoThrow( _vdmmgr_poCdctrlIf );
   }
#else
   _vdmmgr_poCdctrlIf = NULL;    // to satisfy lint
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
   if( _vdmmgr_poErrorIf != NULL )
   {
      vDeleteNoThrow( _vdmmgr_poErrorIf );
   }
   if( _vdmmgr_poTraceHandler != NULL )
   {
      vDeleteNoThrow( _vdmmgr_poTraceHandler );
   }
   if( _vdmmgr_poCCAClienthandlerdiaglog != NULL )
   {
      vDeleteNoThrow( _vdmmgr_poCCAClienthandlerdiaglog );
   }
   if( _vdmmgr_poCCAClienthandlerspm != NULL )
   {
      vDeleteNoThrow( _vdmmgr_poCCAClienthandlerspm );
   }
   if( _vdmmgr_poCCAService != NULL )
   {
      vDeleteNoThrow( _vdmmgr_poCCAService );
   }

   if( _hNotTableSem != OSAL_C_INVALID_HANDLE )
   {
      if( OSAL_s32SemaphoreClose( _hNotTableSem ) == OSAL_ERROR )
      {
         TRACE_OSAL_ERROR( VDMMGR_TR_MAIN );
      }
      if( OSAL_s32SemaphoreDelete( VDMMGR_NOTTABLE_SEM_NAME ) == OSAL_ERROR )
      {
         TRACE_OSAL_ERROR( VDMMGR_TR_MAIN );
      }
      _hNotTableSem = OSAL_C_INVALID_HANDLE;
   }
   if( _poNotTable != NULL )
   {
      vDeleteNoThrow( _poNotTable);
   }
}


/******************************************************************+FUNCHEADER**
 *
 * FUNCTION:     vdmmgr_tclApp::vRegisterTraceService
 *
 * DESCRIPTION:  Registers trace channel to send commands from TTFis
 *
 * PARAMETER:    None
 *
 * RETURNVALUE:  None
 *
 ******************************************************************-FUNCHEADER*/
tVoid vdmmgr_tclApp::vRegisterTraceService( )
{
   tS32                        s32Error;
   OSAL_trIOCtrlLaunchChannel  oTraceChannel;

   if( _fdTrace == OSAL_ERROR )
   {
      _fdTrace = OSAL_IOOpen( OSAL_C_STRING_DEVICE_TRACE, OSAL_EN_READWRITE );

      if( _fdTrace != OSAL_ERROR )
      {
         ETG_TRACE_USR1(( "vRegisterTraceService( ): Start" ));
         oTraceChannel.enTraceChannel = TR_TTFIS_MEDIAMANAGER;
         oTraceChannel.pCallback = (OSAL_tpfCallback)vChannelMgr;

         s32Error = OSAL_s32IOControl( _fdTrace, OSAL_C_S32_IOCTRL_CALLBACK_REG, (intptr_t)&oTraceChannel );
         if( s32Error != OSAL_OK )
         {
            ETG_TRACE_ERR(( "vRegisterTraceService( ): Error: Fail to register trace service" ));
         }
         else
         {
            ETG_TRACE_USR1(( "vRegisterTraceService( ): Trace service registered" ));
         }
      }
      else
      {
         ETG_TRACE_ERR(( "vRegisterTraceService( ): Error: failed to open OSAL_C_STRING_DEVICE_TRACE with parameter: OSAL_EN_READWRITE" ));
      }
   }
}


/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     vdmmgr_tclApp::vUnregisterService
 *
 *DESCRIPTION:  De-registers trace channel used to send commands from TTFis
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 ******************************************************************-FUNCHEADER*/
tVoid vdmmgr_tclApp::vUnregisterTraceService( )
{
   tS32 s32Error;
   OSAL_trIOCtrlLaunchChannel oTraceChannel;

   if( _fdTrace != OSAL_ERROR )
   {
      oTraceChannel.enTraceChannel = TR_TTFIS_MEDIAMANAGER;
      oTraceChannel.pCallback      = (OSAL_tpfCallback)vChannelMgr;
      ETG_TRACE_USR1(( "vUnregisterTraceService( ): Start" ));

      s32Error = OSAL_s32IOControl( _fdTrace, OSAL_C_S32_IOCTRL_CALLBACK_UNREG, (intptr_t)&oTraceChannel );
      if( s32Error != OSAL_OK )
      {
         ETG_TRACE_ERR(( "vUnregisterTraceService( ): Error: Fail to unregister trace service" ));
      }
      else
      {
         ETG_TRACE_USR1(( "vUnregisterTraceService( ): Trace service is unregistered" ));
      }
      if( OSAL_s32IOClose( _fdTrace ) == OSAL_ERROR )
      {
         TRACE_OSAL_ERROR( VDMMGR_TR_MAIN );
      }
   }
}


/******************************************************************+FUNCHEADER**
 *
 *FUNCTION:     vdkbd_tclApp::vChannelMgr
 *
 *DESCRIPTION:  React on TTFis commands
 *
 *PARAMETER:    None
 *
 *RETURNVALUE:  None
 *
 *HISTORY:
 *02.08.05 sat2hi
 *         Initial version
 *
 ******************************************************************-FUNCHEADER*/
tVoid vdmmgr_tclApp::vChannelMgr( tPUChar puchData )
{
   _vdmmgr_poExemplar->_vdmmgr_poTraceHandler->vdmmgr_Trace_vRx( puchData );
}//lint !e818 Info 818 -> compiler doesn't like a const pointer


/******************************************************************************
*
* FUNCTION:    bOnInit ()
*
* DESCRIPTION: is called after the initialisation of the framework is finished
*
* PARAMETER:   void
*
* RETURNVALUE: TRUE
*
*******************************************************************************/
tBool vdmmgr_tclApp::bOnInit ()
{
   // +++  HERE YOU CAN INITIALIZE YOUR APPLICATION
   //      - start your threads
   //      - register your call backs
   //      - create anything you need                 +++

#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
   if( _vdmmgr_poCCAService != OSAL_NULL )
   {
      // Read KDS (costumer data) values to check if a CD drive is mounted in this device
      _vdmmgr_poCCAService->vReadKdsCd( );
      if( _vdmmgr_poCCAService->bIsCDDriveMounted() )    // Next steps are not necessary without mounted drive
      {
         // Read last mode data from flash
         _vdmmgr_poCCAService->vGetLastModeData( );
         // Set insert state
         _vdmmgr_poCCAService->vSetInsertStateAfterLastModeData( );
      }
   }
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE

   _vdmmgr_poErrorIf = vdmmgr_tclerrorif::poGetInstance( this );
   if( _vdmmgr_poErrorIf != NULL )
   {
      _vdmmgr_poErrorIf->vStartThread();
   }

#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
   if( _vdmmgr_poCCAService && _vdmmgr_poCCAService->bIsCDDriveMounted() )    // Not necessary if no CD drive is mounted
   {
      _vdmmgr_poCdctrlIf = vdmmgr_tclcdctrlif::poGetInstance( this );
      if( _vdmmgr_poCdctrlIf != NULL )
      {
         _vdmmgr_poCdctrlIf->vStartThread();
      }
   }
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE

   return  TRUE;
}


/******************************************************************************
*
* FUNCTION:    vOnServiceState(...)
*
* DESCRIPTION: the framework uses this method to inform the application about
*              a service state change of another server
*              this can only happen if the application has a client part and
*              is registered with this server
*
* PARAMETER:
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclApp::vOnServiceState( tU16 u16ServiceId, tU16 u16ServerId,
                                      tU16 u16RegisterId,
                                      tU8  u8ServiceState, tU16 u16SubId
                                    )
{
   ETG_TRACE_USR1(( "vOnServiceState(): Service Id:%d, New Service State:%d", ETG_ENUM(ail_u16ServiceId, u16ServiceId), ETG_ENUM(AMT_SERVICE_STATE, u8ServiceState) ));
   // ~~~ ---  check which service has changed  --- ~~~
   switch( u16ServiceId )
   {
      case CCA_C_U16_SRV_SPM:                               // -> insert clamps
      {
         // ~~~ --- analyze new service state with the corresponding Clienthandler --- ~~~
         if( _vdmmgr_poCCAClienthandlerspm != OSAL_NULL )
         {
            _vdmmgr_poCCAClienthandlerspm->vOnServiceState( u16ServiceId, u16ServerId, u16RegisterId, u8ServiceState, u16SubId );
         }
         break;
      }
      case CCA_C_U16_SRV_DIAGLOG:                           // -> store errors
      {
         // ~~~ --- analyze new service state with the corresponding Clienthandler --- ~~~
         if( _vdmmgr_poCCAClienthandlerdiaglog != OSAL_NULL )
         {
            _vdmmgr_poCCAClienthandlerdiaglog->vOnServiceState( u16ServiceId, u16ServerId, u16RegisterId, u8ServiceState, u16SubId );
         }
         break;
      }
      default:
      {
         break;
      }
   }

   // ---  call the corresponding method of the super class to give AIL the chance to do the  rest  ---
   ail_tclAppInterface::vOnServiceState( u16ServiceId, u16ServerId, u16RegisterId, u8ServiceState, u16SubId );
}


tVoid vdmmgr_tclApp::vOnAsyncRegisterConf( tU16 u16RegisterId, tU16 u16ServerAppId, tU16 u16ServiceId, tU16 u16TargetSubId )
{
   ETG_TRACE_USR4(( "vOnAsyncRegisterConf(): u16RegisterId:%d, u16ServerAppId:%d, u16ServiceId:%d", u16RegisterId, ETG_ENUM(ail_u16AppId, u16ServerAppId), ETG_ENUM(ail_u16ServiceId, u16ServiceId) ));
   switch( u16ServiceId )
   {
      case CCA_C_U16_SRV_SPM:
      {
         if( _vdmmgr_poCCAClienthandlerspm && u16ServerAppId == CCA_C_U16_APP_SPM )
         {
            _vdmmgr_poCCAClienthandlerspm->vOnAsyncRegisterConf( u16RegisterId, u16ServerAppId, u16ServiceId, u16TargetSubId );
         }
         else
         {   // Trace this corrupted reference
             NORMAL_M_ASSERT (NULL != _vdmmgr_poCCAClienthandlerspm);
         }
         break;
      }
      case CCA_C_U16_SRV_DIAGLOG:
      {
         if( _vdmmgr_poCCAClienthandlerdiaglog && u16ServerAppId == CCA_C_U16_APP_DIAGLOG )
         {
            _vdmmgr_poCCAClienthandlerdiaglog->vOnAsyncRegisterConf( u16RegisterId, u16ServerAppId, u16ServiceId, u16TargetSubId );
         }
         else
         {   // Trace this corrupted reference
             NORMAL_M_ASSERT (NULL != _vdmmgr_poCCAClienthandlerdiaglog);
         }
         break;
      }
      default:
      {
         break;
      }
   }
}


/******************************************************************************
*
* FUNCTION:    vOnNewAppState( tU32 u32OldAppState, tU32 u32AppState )
*
* DESCRIPTION: * is call from SPM to inform this application about new
*                power states
*              * changes the state of the application
*              * sends ServiceAvailability state to all notified clients
*              * changes state of MediaManager service(-object)
*
* PARAMETER:   tU32 u32OldAppState (actual state),
*              tU32 u32AppState (state to change to)
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclApp::vOnNewAppState( tU32 u32OldAppState, tU32 u32AppState )
{
   OSAL_tMSecond tTimeInMilliSecondsToStateChange = 0;      // default: 0; no delay for system shut down

   TRACE_SPM_Transition( u32OldAppState, u32AppState, _CLASS );
   // ---   analyse the state we have to go to   ---
   switch(u32AppState)
   {
      case AMT_C_U32_STATE_NORMAL:
      {
         // --- the server has already entered this state, so it has nothing to do ---
         if( u32OldAppState == AMT_C_U32_STATE_NORMAL )
         {
            break;      //switching to same state
         }
         // +++  do everything you have to do to enter the state NORMAL
         //      the state NORMAL is entered on system start and when the system leaves
         //      the states PAUSE and DIAG so if you want to do something only once,
         //      don't do it here!!!!   +++

         // --- inform the framework that we are ready to receive and send messages ---
         vServiceAvailabilityChanged( CCA_C_U16_SRV_MMGR, AMT_C_U8_SVCSTATE_AVAILABLE );
         if( _vdmmgr_poCCAService != OSAL_NULL )
         {
            _vdmmgr_poCCAService->vSetIsSendStatusOK( TRUE );
         }
         break;
      }
      case AMT_C_U32_STATE_DIAGNOSIS:
      {
         //  --- the server has already entered this state, so it has nothing to do ---
         if (u32OldAppState == AMT_C_U32_STATE_DIAGNOSIS)
         {
            break; //switching to same state
         }
         // +++ do everything you have to do to enter the state DIAGNOSIS +++

         // +++ inform the framework if you are ready to receive and send messages in this state   +++
         vServiceAvailabilityChanged( CCA_C_U16_SRV_MMGR, AMT_C_U8_SVCSTATE_AVAILABLE );
         if( _vdmmgr_poCCAService != OSAL_NULL )
         {
            _vdmmgr_poCCAService->vSetIsSendStatusOK( TRUE );
         }
         break;
      }
      case AMT_C_U32_STATE_PAUSE:
      {
         // --- the server has already entered this state, so it has nothing to do ---
         if( u32OldAppState == AMT_C_U32_STATE_PAUSE )
         {
            break; //switching to same state
         }
         // +++ do everything you have to do to enter the state PAUSE +++
         // +++  inform the framework if you are ready to receive and send messages in this state  +++
         vServiceAvailabilityChanged( CCA_C_U16_SRV_MMGR, AMT_C_U8_SVCSTATE_NOT_AVAILABLE );
         if( _vdmmgr_poCCAService != OSAL_NULL )
         {
            _vdmmgr_poCCAService->vSetIsSendStatusOK( FALSE );
         }
         break;
      }
      case AMT_C_U32_STATE_OFF:
      {
        // --- the server has already entered this state, so it has nothing to do ---
         if( u32OldAppState == AMT_C_U32_STATE_OFF )
         {
            break; //switching to same state
         }
         // +++ do everything you have to do to enter the state OFF +++
         // --- inform the framework that we are not ready to received or send messages ---
         vServiceAvailabilityChanged( CCA_C_U16_SRV_MMGR, AMT_C_U8_SVCSTATE_NOT_AVAILABLE );
         if( _vdmmgr_poCCAService != OSAL_NULL )
         {
            _vdmmgr_poCCAService->vSetIsSendStatusOK( FALSE );
         }
         break;
      }
      default:
      {
         // +++ this should never happen, you can call a trace or assert here  +++
         break;
      }
   }

   // --- inform the message handler about the new state ---
   if( _vdmmgr_poCCAService != OSAL_NULL )
   {
      tTimeInMilliSecondsToStateChange = _vdmmgr_poCCAService->u32OnNewAppState( u32OldAppState, u32AppState );
   }
   else
   {
      // +++ this should never happen, you can call a trace or assert here +++
   }

   if( _vdmmgr_poCCAClienthandlerspm != OSAL_NULL )
   {
      // Check if a CD insert (change to OFF or PAUSE delays the system shutdown)
      _vdmmgr_poCCAClienthandlerspm->vOnNewAppState( u32OldAppState, u32AppState );
   }
   else
   {
      // +++ this should never happen, you can call a trace or assert here +++
   }

   if( _vdmmgr_poCCAClienthandlerdiaglog != OSAL_NULL )
   {
      _vdmmgr_poCCAClienthandlerdiaglog->vOnNewAppState( u32OldAppState, u32AppState );
   }
   else
   {
      // +++ this should never happen, you can call a trace or assert here +++
   }

   // Set AppState
   _u32AppState = u32AppState;

   // --- create and send an acknowledge message so the SPM knows you have reached the new state ---

   ETG_TRACE_USR1(( "vOnServiceState(): Switched to Applikation state: %d; apply for shutdown delay = %d[ms]", ETG_ENUM(SPM_APPSTATE_DEF, u32AppState), tTimeInMilliSecondsToStateChange));
   vAppStateChanged( u32AppState, tTimeInMilliSecondsToStateChange );
}


/******************************************************************************
*
* FUNCTION:    vOnNewMessage (amt_tclBaseMessage* poMessage)
*
* DESCRIPTION: * casts Message to amt_tclServiceData (if legal)
*              * gives amt_tclServiceData - Message to service for analyzing
*              * ignores all unexpected messages
*
* PARAMETER:   amt_tclBaseMessage* poMessage ( Message to analyse)
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclApp::vOnNewMessage( amt_tclBaseMessage* poMessage )
{
   // --- check if the received message is a AMT Service data message ---
   if( poMessage->u8GetType( ) == AMT_C_U8_CCAMSGTYPE_SVCDATA )
   {
      // --- cast the received message to a AMT Service data message so we can
      //access the standard members ---
      amt_tclServiceData oServiceData(poMessage);

      tU16 u16ServiceID = oServiceData.u16GetServiceID();

      // ---  one server can offer more than one service and it can act as a client. So analyse
      //      the ServiceID to know which service handler we have to call.    ---
      switch( u16ServiceID )
      {
         // +++  enter your Service ID here   +++
         // --- For every clienthandler: give the message to the cca service handler for analysing.vDispatchMessage is a
         //     framework function. it will compare the function ID with the message map an call the corresponding function  ---
         case CCA_C_U16_SRV_MMGR:
         {
            if( _vdmmgr_poCCAService != OSAL_NULL )
            {
               _vdmmgr_poCCAService->vDispatchMessage( &oServiceData );
            }
            break;
         }
         case CCA_C_U16_SRV_SPM:                            // -> insert clamps
         {
            if( _vdmmgr_poCCAClienthandlerspm != OSAL_NULL )
            {
               _vdmmgr_poCCAClienthandlerspm->vDispatchMessage( &oServiceData );
            }
            break;
         }
         case CCA_C_U16_SRV_DIAGLOG:                        // -> insert clamps
         {
            if( _vdmmgr_poCCAClienthandlerdiaglog != OSAL_NULL )
            {
               _vdmmgr_poCCAClienthandlerdiaglog->vDispatchMessage( &oServiceData );
            }
            break;
         }
         default:
         {
            // +++ we have received a message with an unknown service ID. this is an error so make a trace or assert here  +++
            break;
         }
      }
   }
   else
   {
      // +++ we have received a message with an unknown format. +++
      // --- we can't do anything with this message so free the resources ---
      (tVoid)poMessage->bDelete();
   }
}


/******************************************************************************
*
* FUNCTION:    bGetServiceVersion (tU16 u16ServiceID, tU16& rfu16MajorVersion,
                               tU16& rfu16MinorVersion, tU16& rfu16PatchVersion)
*
* DESCRIPTION: stores data about the version of this service (magic)
*
* PARAMETER:   * u16ServiceID: the service we should describe
*              * rfu16MajorVersion, rfu16MinorVersion,
                 rfu16PatchVersion: targets for storing the data
*
* RETURNVALUE: success: true / false
*
*******************************************************************************/
tBool vdmmgr_tclApp::bGetServiceVersion( tU16 u16ServiceID, tU16& rfu16MajorVersion, tU16& rfu16MinorVersion, tU16& rfu16PatchVersion )
{
   if( u16ServiceID == CCA_C_U16_SRV_MMGR )
   {
      // +++  Enter the version of this service you offer  +++
      rfu16MajorVersion = 1;
      rfu16MinorVersion = 0;
      rfu16PatchVersion = 0;
      return TRUE;
   }
   else
   {
      // +++ we have received a request with an unknown service ID. this is an error so make a trace or assert here +++
      return FALSE;
   }
}


/******************************************************************************
*
* FUNCTION:    vSendError(...)
*
* DESCRIPTION: sends an error - message
*
* PARAMETER:   data for addressing the message
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclApp::vSendError( tU16 u16SourceID, tU16 u16TargetID, tU16 u16RegisterID, tU16 u16CmdCounter, tU16 u16Sid, tU16 u16Fid )
{
   gm_tclU8Message oErrorMsg( u16SourceID, u16TargetID, u16RegisterID, u16CmdCounter, u16Sid, u16Fid, AMT_C_U8_CCAMSG_OPCODE_ERROR );

   if( enPostMessage( &oErrorMsg ) != AIL_EN_N_NO_ERROR )
   {
      // +++ can't send message: enter error handling here ( trace or assert ) +++
   }
}


/******************************************************************************
*
* FUNCTION:    bOnWatchdog( tVoid )
*
* DESCRIPTION: ????
*
* PARAMETER:   void
*
* RETURNVALUE: true
*
*******************************************************************************/
tBool vdmmgr_tclApp::bOnWatchdog( tVoid )
{
   return TRUE;
}


/******************************************************************************
*
* FUNCTION:    vOnApplicationClose( tVoid )
*
* DESCRIPTION: releases all resources and sends close - message
*
* PARAMETER:   void
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclApp::vOnApplicationClose( tVoid )
{
   vUnregisterTraceService();
   _bAppClosed = TRUE;

   // +++ free all allocated resources +++
   // Close threads started in bOnInit()
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
   if( _vdmmgr_poCdctrlIf != NULL )
   {
      delete _vdmmgr_poCdctrlIf;
      _vdmmgr_poCdctrlIf = NULL;
   }
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
   if( _vdmmgr_poErrorIf != NULL )
   {
      delete _vdmmgr_poErrorIf;
      _vdmmgr_poErrorIf = NULL;
   }
   // Close clients started in constructor
   if( _vdmmgr_poTraceHandler != NULL )
   {
      delete _vdmmgr_poTraceHandler;
      _vdmmgr_poTraceHandler = NULL;
   }
   if( _vdmmgr_poCCAClienthandlerdiaglog != NULL )
   {
      delete _vdmmgr_poCCAClienthandlerdiaglog;
      _vdmmgr_poCCAClienthandlerdiaglog = NULL;
   }
   if( _vdmmgr_poCCAClienthandlerspm != NULL )
   {
      delete _vdmmgr_poCCAClienthandlerspm;
      _vdmmgr_poCCAClienthandlerspm = NULL;
   }
   if( _vdmmgr_poCCAService != NULL )
   {
      delete _vdmmgr_poCCAService;
      _vdmmgr_poCCAService = NULL;
   }

   if( _hNotTableSem != OSAL_C_INVALID_HANDLE )
   {
      if( OSAL_s32SemaphoreClose( _hNotTableSem ) == OSAL_ERROR )
      {
         TRACE_OSAL_ERROR( VDMMGR_TR_MAIN );
      }
      if( OSAL_s32SemaphoreDelete( VDMMGR_NOTTABLE_SEM_NAME ) == OSAL_ERROR )
      {
         TRACE_OSAL_ERROR( VDMMGR_TR_MAIN );
      }
      _hNotTableSem = OSAL_C_INVALID_HANDLE;
   }
   if( _poNotTable != NULL )
   {
      delete _poNotTable;
      _poNotTable = NULL;
   }


   // --- inform SPM about successful shut down ---
   vApplicationCloseAcknowledge( AIL_C_U8_APP_END_SUCCESSFUL );
}


/* *****************************************************************************

   FUNCTION:      vDeleteNoThrow

   DESCRIPTION:   Deletes memory without throwing an exception.

   PARAMETERS:    rfpArg: reference of pointer to memory, to be deleted

   RETURN TYPE:   void

   HINT:          Why this function? If delete is called inside the destructor,
                  Lint complains with warning 1551 that delete might throw an
                  exception, which is not caught.

                  But try / catch must not be used, and delete does not throw
                  an exception, anyway. To prevent a Lint warning         ,
                  this function guarantees that no exception is thrown.
                  Furthermore, it sets the given pointer to 0, and Lint is
                  happy.

                  It is a template, because two different memory types have to
                  be deleted inside this class.

----------------------------------------------------------------------------- */
template< class T > void vdmmgr_tclApp::vDeleteNoThrow( T*& rfpArg ) throw()
{
   OSAL_DELETE rfpArg;
   rfpArg = 0;
}


/*-----------------------------------------------------------------------------*
 * VD_MediaManager * GetInstance()
 *-----------------------------------------------------------------------------*/
vdmmgr_tclApp *vdmmgr_tclApp::GetInstance()
{
    ETG_TRACE_USR4(("Run  :GetInstance()"));
    return theServer();
}

/*-----------------------------------------------------------------------------*
 * void DestroyInstance()                                                      *
 *-----------------------------------------------------------------------------*/
void vdmmgr_tclApp::DestroyInstance()
{
    ETG_TRACE_USR4(("Run  : DestroyInstance() - tbd."));
}


