/******************************************************************************
* FILE:         vdmmgr_errorif.cpp
* PROJECT:      MIB2_ENTRY
* SW-COMPONENT: Virtual Device Media Manager
*------------------------------------------------------------------------------
*
* DESCRIPTION: VD MediaManager
*              
*------------------------------------------------------------------------------
* COPYRIGHT:    (c) 2008 Robert Bosch GmbH, Hildesheim
* HISTORY:      
* Date      | Author                | Modification
* 04.02.08  | CM-DI/PJ-VW36 Fiebing | initial version
*
*******************************************************************************/
#include "Config.h"
#include <map>
#include <vector>

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h" 

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_MEDIAMANAGER_ERRORIF
#include "trcGenProj/Header/vdmmgr_errorif.cpp.trc.h"
#endif

#define AIL_S_IMPORT_INTERFACE_GENERIC
#include "ail_if.h"         // use AIL template with MessageMaps
#define AHL_S_IMPORT_INTERFACE_NOTIFICTABLE
#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"

#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h"

#define VDMMGR_S_IMPORT_INTERFACE_MSG
#include "vdmmgr_if.h"                       // For VD MMgr interface

#define VD_DIAGLOG_S_IMPORT_INTERFACE_MSG
#include "vd_diaglog_if.h"

/* ************************************************************************** */
/* include the public interface                                               */
/* ************************************************************************** */
#include "vdmmgr_main.h"                  // For the pointer to main application
#include "vdmmgr_errorif.h"               // class vdmmgr_tclerrorif + defines
#include "vdmmgr_timer.h"                 // 'automatic reinsert CD' timer
#include "vdmmgr_service.h"               // (CCA) send functions
#include "vdmmgr_ITC_Container.h"
#include "vdmmgr_ITC_Map.h"
#include "vdmmgr_clienthandlerdiaglog.h"  // store ITC status changes
#include "vdmmgr_trace.h"                 // trace stuff

/* ************************************************************************** */
/* INITIALIZE STATIC CLASS VARIABLES                                          */
/* ************************************************************************** */

tU16                 vdmmgr_tclerrorif::_u16CdDefect      = 0;

vdmmgr_tclerrorif*  vdmmgr_tclerrorif::_poInstance = OSAL_NULL;

/******************************************************************************/
/*
* FUNCTION:    poGetInstance( vdmmgr_tclApp* poVdmmgrMain )
* 
* DESCRIPTION: creates the static  vdmmgr_tclerrorif - object and returns a 
*              pointer or returns a pointer to the already existing object
*  
* PARAMETER:   vdmmgr_tclApp: Pointer to vdmmgr main application
*
* RETURNVALUE: pointer to the vdmmgr_tclerrorif - object
*/
/******************************************************************************/
vdmmgr_tclerrorif* vdmmgr_tclerrorif::poGetInstance( vdmmgr_tclApp* poVdmmgrMain )
{
   if(_poInstance == OSAL_NULL)
   {
      _poInstance = new vdmmgr_tclerrorif( poVdmmgrMain );
   }

   if( _poInstance == OSAL_NULL )
   {
      ETG_TRACE_ERR(( "poGetInstance( ): Error: Not able to create a new ERROR control object in poGetInstance( App* poMain)" ));
   }

   return _poInstance;
}

/******************************************************************************/
/*
* FUNCTION:    poGetInstance( tVoid )
* 
* DESCRIPTION: returns the value of a member - variable 
*
* PARAMETER:   void
*
* RETURNVALUE: pointer to a vdmmgr_tclerrorif - object
*/
/******************************************************************************/
vdmmgr_tclerrorif* vdmmgr_tclerrorif::poGetInstance( tVoid )
{
   if( _poInstance == OSAL_NULL )
   {
      ETG_TRACE_ERR(( "poGetInstance( ): Error: Call of poGetInstance( tVoid ) and no pointer to instance available" ));
   }
   return _poInstance;
}


/*************************************************************************/
/*
* FUNCTION:    vdmmgr_tclerrorif( vdmmgr_tclApp* poVdmmgrMain )
* 
* DESCRIPTION: constructor
*
* PARAMETER:   poVdmmgrMain: 
*
* RETURNVALUE: none
*/ 
/*************************************************************************/
vdmmgr_tclerrorif::vdmmgr_tclerrorif( vdmmgr_tclApp* poVdmmgrMain )
:
_bThreadStarted      ( FALSE ),
_hDevice             ( OSAL_ERROR ),
_bTerminate          ( FALSE ),
_tThreadID           ( OSAL_ERROR ),
_hEvent              ( OSAL_C_INVALID_HANDLE )
{
   tBool bRetVal;

   // Check pointer to service handler
   bRetVal = ( poVdmmgrMain != OSAL_NULL );
   if( !bRetVal )
   {
      ETG_TRACE_ERR(( "vdmmgr_tclerrorif::vdmmgr_tclerrorif: Error: given pointer poVdmmgrMain is NULL" ));
   }
   // Get pointer to main application
   _poMainApp  = poVdmmgrMain;

   // Create Event and check result
   bRetVal = OSAL_s32EventCreate( VDMMGR_ERRORIF_EVENTNAME, &_hEvent) == OSAL_OK;
   if( !bRetVal )
   {
      ETG_TRACE_ERR(( "vdmmgr_tclerrorif::vdmmgr_tclerrorif: Error: Not able to create event VDMMGR_ERRORIF_EVENTNAME" ));
   }
}


/*************************************************************************/
/*
* FUNCTION:    vdmmgr_tclerrorif::~vdmmgr_tclerrorif( )
* 
* DESCRIPTION: destructor: deletes the thread
*
* PARAMETER:   void
*
* RETURNVALUE: none
*/
/*************************************************************************/
vdmmgr_tclerrorif::~vdmmgr_tclerrorif( )
{
   tS32 s32Success;

   // Deactivate Thread
   vDeActivateSrc();//lint !e1551 Warning 1551;Function may throw exception

   if( _tThreadID != OSAL_ERROR)
   {
      // Delete thread
      s32Success = OSAL_s32ThreadDelete( _tThreadID );
      if( s32Success == OSAL_ERROR )
      {
         ETG_TRACE_ERR(( "vdmmgr_tclerrorif::~vdmmgr_tclerrorif: Error: Unable to delete thread in destructor" ));
      }
   }

   // Reset pointer to main application
   _poMainApp = OSAL_NULL;
}


/*************************************************************************/
/*
* FUNCTION: tVoid vdmmgr_tclerrorif::vStartThread( tVoid )
* 
* DESCRIPTION: Create and start error interface Thread. To be called only in
*              GetInstance().
*
* PARAMETER: void
*
* RETURNVALUE: void
*/
/*************************************************************************/
tVoid vdmmgr_tclerrorif::vStartThread( tVoid )
{
   tBool bRetVal;

   if( !_bThreadStarted )
   {
      // Create thread and check result
      bRetVal = bCreateThread();

      // Activate thread if create's are OK.
      if( bRetVal )
      {
         vActivateSrc();
      }
      else
      {
         ETG_TRACE_ERR(( "vStartThread( ): Error: Call of bCreateThread() returns with FALSE" ));
      }
      _bThreadStarted = TRUE;
   }
}


/*************************************************************************/
/*
* FUNCTION: tVoid vdmmgr_tclerrorif::vActivateSrc()
* 
* DESCRIPTION: * starts this MediaManager Source as a thread, 
*              * registers it at PRM (Physical Resource Manager) if not done this before
*
* PARAMETER: void
*
* RETURNVALUE: void
*/
/*************************************************************************/
tVoid vdmmgr_tclerrorif::vActivateSrc()
{
   tS32 s32Success = OSAL_s32ThreadActivate( _tThreadID );
   if( s32Success == OSAL_ERROR )
   {
      ETG_TRACE_ERR(( "vActivateSrc( ): Error: Unable to activate thread" ));
   }
}


/*************************************************************************/
/*
* FUNCTION: tVoid vdmmgr_tclerrorif::vDeActivateSrc()
* 
* DESCRIPTION: unregister at the PRM
*
* PARAMETER: void
*
* RETURNVALUE: void
*/
/*************************************************************************/
tVoid vdmmgr_tclerrorif::vDeActivateSrc() const
{
   // Nothing to do until now...
}


/*************************************************************************/
/*
* FUNCTION: tBool vdmmgr_tclerrorif::bCreateThread()
* 
* DESCRIPTION: starts the member - function "dwThreadProc" as thread
*
* PARAMETER: void 
*
* RETURNVALUE: void
*/
/*************************************************************************/
tBool vdmmgr_tclerrorif::bCreateThread()
{
   OSAL_trThreadAttribute  rThAttr;
   tBool                   bRetVal        = TRUE;
   tU32                    u32ThreadPrio  = 0;     // Local variable to store data read from registry
   tU32                    u32StackSize   = 0;     // Local variable to store data read from registry

   ETG_TRACE_USR1(( "bCreateThread( ): Create Error interface thread" ));

   // Read the values from registry. In case of a failure, use default values
   // Read thread priority from registry
   if( scd_bGetAppConfigurationValue( CCA_C_U16_APP_MMGR,                        // CCA Application ID
                                      VDMMGR_ERRORIF_REGPATH_THREAD_DATA,        // Key name
                                      VDMMGR_ERRORIF_REGVALUE_THREAD_PRIO_NAME,  // Key value
                                      &u32ThreadPrio )                   == FALSE )
   {
      // Set thread priority to default priority
      u32ThreadPrio = VDMMGR_ERRORIF_DEFAULT_PRIO;
      // Add trace on warning level here! The thread runs on default values!
      ETG_TRACE_SYS(( "bCreateThread( ): Warning: Error IF Thread runs with default priority" ));
   }
   // read thread stack size  from registry
   if( scd_bGetAppConfigurationValue( CCA_C_U16_APP_MMGR,                        // CCA Application ID
                                      VDMMGR_ERRORIF_REGPATH_THREAD_DATA,        // Key name
                                      VDMMGR_ERRORIF_REGVALUE_STACK_SIZE_NAME,   // Key value
                                      &u32StackSize  )                   == FALSE )
   {
      u32StackSize  = VDMMGR_ERRORIF_DEFAULT_STACKSIZE;
      // Indicate that the thread is running on default priority
      ETG_TRACE_SYS(( "bCreateThread( ): Warning: Error IF Thread runs with default stack size" ));
   }

   // setup thread
   _bTerminate = FALSE;

   rThAttr.szName       = const_cast<tString>( VDMMGR_ERRORIF_THREADNAME );
   rThAttr.u32Priority  = u32ThreadPrio;
   rThAttr.s32StackSize = (tS32)u32StackSize;
   rThAttr.pfEntry      = (OSAL_tpfThreadEntry)dwThreadProc;
   rThAttr.pvArg        = (tPVoid)this;
   
   // create thread suspended
   _tThreadID = OSAL_ThreadCreate( &rThAttr );
   if(_tThreadID == OSAL_ERROR)
   {
      //can't create Cd Thread
      ETG_TRACE_ERR(( "bCreateThread( ): OSAL_ThreadCreate() returns with OSAL_ERROR" ));
      bRetVal = FALSE;
   }
   return bRetVal;
}



/*************************************************************************/
/*
* FUNCTION:    dwThreadProc (tPVoid pvArg)
* 
* DESCRIPTION: * this function is started as thread by "vdmmgr_tclerrorif::bCreateThread()"
*              * as argument the this - pointer is passed, 
*                so in fact the function vdmmgr_tclerrorif::vExecute() is started as thread
*
* PARAMETER:   void - pointer ( in this implementation: this - pointer of a vdmmgr_tclerrorif - object)
*
* RETURNVALUE:
*/
/*************************************************************************/
OSAL_tpfThreadEntry vdmmgr_tclerrorif::dwThreadProc( tPVoid pvArg )
{
   vdmmgr_tclerrorif*  poCdCtrlIf;

   poCdCtrlIf = (vdmmgr_tclerrorif*)pvArg;
   poCdCtrlIf->vExecute();
   return 0;
}


/*************************************************************************/
/*
* FUNCTION:    vExecute()
* 
* DESCRIPTION: * this is the working - thread of this MediaManager Source
*              * after some init stuff it listens to MediaManager
*                source - events and calls the event - handler 
*
* PARAMETER:   void
*
* RETURNVALUE: void
*/
/*************************************************************************/
tVoid vdmmgr_tclerrorif::vExecute()
{
   OSAL_tEventMask   oMask;
   OSAL_tEventMask   oResultMask = 0;
   tS32              s32Success;

   ETG_TRACE_USR1(( "vExecute(): Start waiting for events" ));

   oMask = VDMMGR_ERRORIF_EVENT_MASK_DEFECT_CD;

   if( OSAL_s32EventOpen( VDMMGR_ERRORIF_EVENTNAME, &_hEvent ) == OSAL_OK )
   {
      for( ;/*ever*/; )
      {
         s32Success = OSAL_s32EventWait( _hEvent, oMask, OSAL_EN_EVENTMASK_OR,
                                         OSAL_C_TIMEOUT_FOREVER, &oResultMask );
         if( s32Success == OSAL_ERROR )
         {
            ETG_TRACE_ERR(( "vExecute(): Error: OSAL_s32EventWait() returns with error" ));
         }
         //clear event
         if( OSAL_OK != OSAL_s32EventPost( _hEvent,~oResultMask,OSAL_EN_EVENTMASK_AND ) )
         {
            ETG_TRACE_ERR(( "vExecute(): Error: Unable to clear event -> OSAL_s32EventPost( )" ));
         }
         //handle event
         vHandleEvent( oResultMask );
      }//for ever
   }
   else
   {
      ETG_TRACE_ERR(( "vExecute(): Error: OSAL_s32EventOpen returns with error" ));
      // Get error code if not successful.
      TRACE_OSAL_ERROR( VDMMGR_TR_ERRORIF );
   }
}


/*************************************************************************/
/*
* FUNCTION:    vHandleEvent (OSAL_tEventMask rEventMask)
* 
* DESCRIPTION: handler for MediaManager Source - events 
*
* PARAMETER:   eventMask
*
* RETURNVALUE: void
*/
/*************************************************************************/
tVoid vdmmgr_tclerrorif::vHandleEvent( OSAL_tEventMask rEventMask ) const
{
   ETG_TRACE_USR1(( "vHandleEvent( ): Start: -> Switch Event Mask" ));

   // Check for device failure
   if( rEventMask & VDMMGR_ERRORIF_EVENT_MASK_DEFECT_CD )
   {
      vHandleDefectCd( );
   }
}


/* **************************************************FunctionHeaderBegin** *//**
 *
 *  Function: vHandleDefectCd( tVoid ) const
 *
 *  Evaluate the notification data for 'defect'.
 *
 *  Evaluate the notification data for 'defect'. 
 *  No parameter. No return value.
 *
 *      Possible values for _u16Defect in osioctrl.h.
 *
 * @date    2008-02-04
 *
 * @note
 *      
 *
 *//* ***********************************************FunctionHeaderEnd******* */

tVoid vdmmgr_tclerrorif::vHandleDefectCd( tVoid ) const
{
   mplay_fi_tcl_e16_DriveErrorValue::tenType  e16ErrorValue = mplay_fi_tcl_e16_DriveErrorValue::FI_EN_DEFECT_NO_ERROR;

   ETG_TRACE_USR1(( "vHandleDefectCd( ): Start: %d with Message %d", ETG_ENUM(ERRORIF_FCT, VDMMGR_ERRORIF_CDFCT), ETG_ENUM(OSAL_NOTIFI_DEFECT, _u16CdDefect) ));
   if( _poMainApp && _poMainApp->_vdmmgr_poCCAClienthandlerdiaglog )
   {

      switch( _u16CdDefect )
      {
#ifdef USE_REPLACE_E8_TESTRESULT
         case OSAL_C_U16_DEFECT_LOAD_EJECT:
            _poMainApp->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_LOAD_ERROR, (tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED );
            e16ErrorValue = mplay_fi_tcl_e16_DriveErrorValue::FI_EN_DEFECT_LOAD_EJECT;
            break;
         case OSAL_C_U16_DEFECT_LOAD_INSERT:
            _poMainApp->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_LOAD_ERROR, (tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED );
            e16ErrorValue = mplay_fi_tcl_e16_DriveErrorValue::FI_EN_DEFECT_LOAD_INSERT ;
            break;
         case OSAL_C_U16_DEFECT_COM:
            _poMainApp->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_COMUNICATION_ERROR, (tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED );
            e16ErrorValue = mplay_fi_tcl_e16_DriveErrorValue::FI_EN_DEFECT_COM;
            break;
         case OSAL_C_U16_DEFECT_COM_OK:
            _poMainApp->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_COMUNICATION_ERROR, (tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED );
            e16ErrorValue = mplay_fi_tcl_e16_DriveErrorValue::FI_EN_DEFECT_COM_OK;
            break;
         case OSAL_C_U16_DEFECT_CMD_TIMEOUT:
            _poMainApp->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_COMUNICATION_TIMEOUT, (tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED );
            e16ErrorValue = mplay_fi_tcl_e16_DriveErrorValue::FI_EN_DEFECT_CMD_TIMEOUT;
            break;
         case OSAL_C_U16_DEFECT_CMD_TIMEOUT_OK:
            _poMainApp->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_COMUNICATION_TIMEOUT, (tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED );
            e16ErrorValue = mplay_fi_tcl_e16_DriveErrorValue::FI_EN_DEFECT_CMD_TIMEOUT_OK;
            break;
#else
         case OSAL_C_U16_DEFECT_LOAD_EJECT:
            _poMainApp->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_LOAD_ERROR, midw_fi_tcl_TestResult::FI_EN_FAILED );
            e16ErrorValue = mplay_fi_tcl_e16_DriveErrorValue::FI_EN_DEFECT_LOAD_EJECT;
            break;
         case OSAL_C_U16_DEFECT_LOAD_INSERT:
            _poMainApp->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_LOAD_ERROR, midw_fi_tcl_TestResult::FI_EN_FAILED );
            e16ErrorValue = mplay_fi_tcl_e16_DriveErrorValue::FI_EN_DEFECT_LOAD_INSERT ;
            break;
         case OSAL_C_U16_DEFECT_COM:
            _poMainApp->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_COMUNICATION_ERROR, midw_fi_tcl_TestResult::FI_EN_FAILED );
            e16ErrorValue = mplay_fi_tcl_e16_DriveErrorValue::FI_EN_DEFECT_COM;
            break;
         case OSAL_C_U16_DEFECT_COM_OK:
            _poMainApp->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_COMUNICATION_ERROR, midw_fi_tcl_TestResult::FI_EN_PASSED );
            e16ErrorValue = mplay_fi_tcl_e16_DriveErrorValue::FI_EN_DEFECT_COM_OK;
            break;
         case OSAL_C_U16_DEFECT_CMD_TIMEOUT:
            _poMainApp->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_COMUNICATION_TIMEOUT, midw_fi_tcl_TestResult::FI_EN_FAILED );
            e16ErrorValue = mplay_fi_tcl_e16_DriveErrorValue::FI_EN_DEFECT_CMD_TIMEOUT;
            break;
         case OSAL_C_U16_DEFECT_CMD_TIMEOUT_OK:
            _poMainApp->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_COMUNICATION_TIMEOUT, midw_fi_tcl_TestResult::FI_EN_PASSED );
            e16ErrorValue = mplay_fi_tcl_e16_DriveErrorValue::FI_EN_DEFECT_CMD_TIMEOUT_OK;
            break;
#endif
         default:                                           // Unknown defect value
            ETG_TRACE_SYS(( "vHandleDefectCd( ): Warning: %d; Unknown defect %d", ETG_ENUM(ERRORIF_FCT, VDMMGR_ERRORIF_CDFCT), _u16CdDefect ));
            break;
      }
      if(   e16ErrorValue != mplay_fi_tcl_e16_DriveErrorValue::FI_EN_DEFECT_NO_ERROR 
         && _poMainApp->_vdmmgr_poCCAService
         )
      {
         // Send defect to registered clients
         _poMainApp->_vdmmgr_poCCAService->vSendCdDriveErrorInformation( e16ErrorValue );
      }
   }
}


tVoid vdmmgr_tclerrorif::vSendDefectEvent( OSAL_tEventMask u32DefectEvent, tU16 u16DefectData ) const
{
   tS32              s32Success;
   OSAL_tEventHandle hCbkEvent;

   switch( u32DefectEvent )
   {
      case VDMMGR_ERRORIF_EVENT_MASK_DEFECT_CD:
         _u16CdDefect = u16DefectData;
         break;
      default:
         ETG_TRACE_ERR(( "vSendDefectEvent( ): Error: Unknown Event type in vSendDefectEvent(): %d", u32DefectEvent ));
         break;
   } 

   // Open, send and close event
   if(    ( u32DefectEvent != 0)
       && ( OSAL_s32EventOpen( VDMMGR_ERRORIF_EVENTNAME, &hCbkEvent ) == OSAL_OK )
     )
   {
      s32Success = OSAL_s32EventPost( hCbkEvent, u32DefectEvent, OSAL_EN_EVENTMASK_OR );
      if( s32Success == OSAL_ERROR )
      {
         ETG_TRACE_ERR(( "vSendDefectEvent( ): Error: OSAL_s32EventPost( ) returns with error" ));
      }

      s32Success = OSAL_s32EventClose( hCbkEvent );
      if(s32Success == OSAL_ERROR)
      { 
         ETG_TRACE_ERR(( "vSendDefectEvent( ): Error: OSAL_s32EventClose() returns with error" ));
      }
   }
   else if( u32DefectEvent != 0)  // -> OSAL_s32EventOpen(...) returned with error
   {
      ETG_TRACE_ERR(( "vSendDefectEvent( ): Error: OSAL_s32EventOpen( ) return with error" ));
   }
   ETG_TRACE_USR1(( "vSendDefectEvent( ): End" ));
}


tVoid vdmmgr_tclerrorif::vSetDefectCdForSimulation( tU16 u16Defect ) const
{
   _u16CdDefect = u16Defect;
}

