/************************************************************************
* FILE:         vdmmgr_clienthandlerspm.cpp
* PROJECT:      MIB2_ENTRY
* SW-COMPONENT: Virtual Device Media Manager
*----------------------------------------------------------------------
*
* DESCRIPTION: VD MediaManager
*              
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2005 Robert Bosch GmbH, Hildesheim
* HISTORY:      
* Date      | Author             | Modification
* 27.05.05  | CM-DI/ESA2 Fiebing | initial version
*
*************************************************************************/

#define _CLASS  VDMMGR_TR_CLIENTSPM

//-----------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------
#include "Config.h"
#include <vector>

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_MEDIAMANAGER_CLIENTSPM
#include "trcGenProj/Header/vdmmgr_clienthandlerspm.cpp.trc.h"
#endif

#define AIL_S_IMPORT_INTERFACE_GENERIC
#include "ail_if.h"                                   // use AIL template with MessageMaps
#define AHL_S_IMPORT_INTERFACE_NOTIFICTABLE
#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"

#define CFC_FI_S_IMPORT_INTERFACE_CFC_SPMFI_FUNCTIONIDS
#define CFC_FI_S_IMPORT_INTERFACE_CFC_SPMFI_TYPES     // or #define CFC_FI_S_IMPORT_INTERFACE_CFC_SPMFI_STDVISITORS
#include "cfc_fi_if.h"

#define VDMMGR_S_IMPORT_INTERFACE_MSG
#include "vdmmgr_if.h"                                // For VD MMgr interface

#include "vdmmgr_main.h"
#include "vdmmgr_clienthandlerspm.h"
#include "vdmmgr_timer.h"                             // 'automatic reinsert CD' timer
#include "vdmmgr_service.h"
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
   #include "vdmmgr_cdctrlif.h"
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
#include "vdmmgr_trace.h"

/* ************************************************************************** */
/* INITIALIZE STATIC CLASS VARIABLES                                          */
/* ************************************************************************** */
OSAL_tEventHandle vdmmgr_tclclienthandlerspm::_hEvent = OSAL_C_INVALID_HANDLE;

// +++ MESSAGE MAP: enter the function IDs (FID) and the corresponding functions here.
//      the function will be called when a message with the corresponding FID arrives +++
BEGIN_MSG_MAP(vdmmgr_tclclienthandlerspm, ahl_tclBaseWork)
   ON_MESSAGE( CFC_SPMFI_C_U16_SUBSTATES,       vHandleSubStates)
   ON_MESSAGE( CFC_SPMFI_C_U16_SYSTEMSTATE,     vHandleSystemState)
   ON_MESSAGE( CFC_SPMFI_C_U16_WAKEUPREASON,    vHandleWakeUpReason)
   ON_MESSAGE( CFC_SPMFI_C_U16_STARTUPREASON,   vHandleStartUpReason)
END_MSG_MAP()


/*************************************************************************
*
* FUNCTION: vdmmgr_tclclienthandlerspm::vdmmgr_tclclienthandlerspm(vdmmgr_tclApp* poMainApp)
* 
* DESCRIPTION: constructor, creates object vdmmgr_tclclienthandlerspm - object
*
* PARAMETER: vdmmgr_tclApp* poMainApp: main - object of this application 
*
* RETURNVALUE: none
*
*************************************************************************/
vdmmgr_tclclienthandlerspm::vdmmgr_tclclienthandlerspm( vdmmgr_tclApp* poVdMMgrMainApp )
{
   tBool  bRetVal;
   // ---  this constructor calls ( implicit ) the constructor of the upper class ( framework )
   //      so it registers the combinations of FID and message handler with the framework --- 
   poMain               = poVdMMgrMainApp;
   _u16RegID            = AMT_C_U16_REGID_INVALID;
   _bFidReg             = FALSE;
   _bRegAsyncStarted    = FALSE;
   _bMediaInSlot        = FALSE;
   _bInsertClampsActive = FALSE;
   _ThreadID            = 0;
   _hEvent              = OSAL_C_INVALID_HANDLE;
   #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
   _bFirstSystemStatusAfterRegForFID       = FALSE;
   _bBackGrdActive                         = FALSE;
   #endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE

   // Create Event and check result
   bRetVal = OSAL_s32EventCreate( VDMMGR_CLIENTSPM_EVENTNAME, &_hEvent) == OSAL_OK;
   if( !bRetVal )
   {
      ETG_TRACE_ERR(( "vdmmgr_tclclienthandlerspm::vdmmgr_tclclienthandlerspm: Error: Not able to create event VDMMGR_ERRORIF_EVENTNAME" ));
   }
   OSAL_trThreadAttribute  rThAttr = { const_cast<tString>(VDMMGR_CLIENTSPM_THREADNAME), VDMMGR_CLIENTSPM_DEFAULT_PRIO, VDMMGR_CLIENTSPM_DEFAULT_STACKSIZE,
                                       (OSAL_tpfThreadEntry)vGPIOThread /*entry function*/, (tVoid *)this /*arg to entry function*/ };
   _ThreadID = OSAL_ThreadSpawn(&rThAttr);

   // GPOIO
   bInstallGpioCallback( (OSAL_tGPIODevID)OSAL_EN_SPM_GPIO_WAKEUP_CD, (OSAL_tpfGPIOCallback)vCallbackGPIOHdl, poMain );
}


vdmmgr_tclclienthandlerspm::vdmmgr_tclclienthandlerspm( )
{
}//lint !e1744 member 'x possibly not initialized by private constructor


/*************************************************************************/
/*
* FUNCTION:    vdmmgr_tclclienthandlerspm::~vdmmgr_tclclienthandlerspm( )
* 
* DESCRIPTION: destructor: deletes the thread
*
* PARAMETER:   void
*
* RETURNVALUE: none
*/
/*************************************************************************/
vdmmgr_tclclienthandlerspm::~vdmmgr_tclclienthandlerspm( )
{
   tS32 s32Success;

   if( _ThreadID != OSAL_ERROR)
   {
      // Delete thread
      s32Success = OSAL_s32ThreadDelete( _ThreadID );
      if( s32Success == OSAL_ERROR )
      {
         ETG_TRACE_ERR(( "vdmmgr_tclclienthandlerspm::~vdmmgr_tclclienthandlerspm: Error: Unable to delete thread in destructor" ));
      }
   }

   // Reset pointer to main application
   poMain = OSAL_NULL;
}


/******************************************************************************
*
* FUNCTION:tVoid vdmmgr_tclclienthandlerspm::vOnUnknownMessage(amt_tclBaseMessage* poMessage)
* 
* DESCRIPTION: handle unknown message
*
* PARAMETER:  unknown message
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclclienthandlerspm::vOnUnknownMessage(amt_tclBaseMessage* poMessage)
{
  // +++ the frame work calls this function when it receives a message with a unknown 
  //     FID. You can use it for error handling.  +++
   TRACE_On_Invalid_Pointer(poMessage, "poMessage", _CLASS);
   if (poMessage == NULL)
   {
      return;
   }

   TRACE_Warning("Unknown Message", _CLASS);

   tU8 u8Type = poMessage->u8GetType();
   if (u8Type != CCA_C_U8_TYPE_SVCDATA)
   {
      // incompatible to amt_tclServiceData; should not happen
      TRACE_Value(4, u8Type, "incompatible Message Type", _CLASS); 
      return;
   }

   amt_tclServiceData* poMsg = dynamic_cast<amt_tclServiceData*>( poMessage );
   if( poMsg )
   {
      TRACE_CCA_InMessage(poMsg, _CLASS);

      TRACE_On_Invalid_Pointer( poMain, "poMain", _CLASS);
      if( poMain && poMain->_vdmmgr_poCCAService )
      {
         poMain->_vdmmgr_poCCAService->vSendError(poMsg, CCA_C_U16_ERROR_UNKNOWN_FCT_ID, _CLASS, __LINE__);
      }
   }

   (tVoid)poMessage->bDelete();
}


/******************************************************************************
*
* FUNCTION: tVoid vdmmgr_tclclienthandlerspm::vOnNewAppState(tU32 u32OldAppState, tU32 u32AppState)
* 
* DESCRIPTION: handle state change messages form the SPM
*
* PARAMETER: old state, new state
*
* RETURNVALUE: void
*
********************************************************************************/
tVoid vdmmgr_tclclienthandlerspm::vOnNewAppState(tU32 u32OldAppState, tU32 u32AppState)
{
   TRACE_SPM_Transition(u32OldAppState, u32AppState, _CLASS);

   if( u32OldAppState == u32AppState ) 
   {
      return; //switching to the same state; no action required
   }

   switch( u32AppState )
   {
      case AMT_C_U32_STATE_NORMAL:
      case AMT_C_U32_STATE_DIAGNOSIS:
      {
         // register for service
         (tVoid)bRegisterForService();
         break;
      }
      case AMT_C_U32_STATE_PAUSE:
      case AMT_C_U32_STATE_OFF:
         break;
      default:
      {
         TRACE_Warning("unhandled power state", _CLASS);
         break;
      }
   }
}


//-----------------------------------------------------------------------------
// handle functions
//-----------------------------------------------------------------------------


/*************************************************************************
*
* FUNCTION: tBool bIsMediaInSlot( tVoid) const
* 
* DESCRIPTION:
*
* PARAMETER: 
*
* RETURNVALUE: void
*
* HISTORY:     InitialVersion
* 
*************************************************************************/
tBool vdmmgr_tclclienthandlerspm::bIsMediaInSlot( tVoid) const
{
   return _bMediaInSlot;
}


/*************************************************************************
*
* FUNCTION: tBool bInsertClampsActive( tVoid) const
* 
* DESCRIPTION:
*
* PARAMETER: 
*
* RETURNVALUE: void
*
* HISTORY:     InitialVersion
* 
*************************************************************************/
tBool vdmmgr_tclclienthandlerspm::bInsertClampsActive( tVoid) const
{
   return _bInsertClampsActive;
}


/*************************************************************************
*
* FUNCTION: vHandleSubStates
* 
* DESCRIPTION: handle a sub state message
*
* PARAMETER: 
*
* RETURNVALUE: void
*
* HISTORY:     InitialVersion
* 
*************************************************************************/
tVoid vdmmgr_tclclienthandlerspm::vHandleSubStates( amt_tclServiceData* poMessage )
{
   TRACE_CCA_InMessage(poMessage, _CLASS);

   switch( poMessage->u8GetOpCode( ) )
   {
      case AMT_C_U8_CCAMSG_OPCODE_STATUS:
      {
         // no need to handle a new status
         break;
      }
      default:
      {
         TRACE_Warning("unhandled OpCode", _CLASS);
         break;
      }
   }
   (tVoid)poMessage->bDelete();
} //lint !e1762  DevStudios says:'not const, please'


/*************************************************************************
*
* FUNCTION: vHandleSystemState
* 
* DESCRIPTION: handle a MediaManager Status
*
* PARAMETER: 
*
* RETURNVALUE: void
*
* HISTORY:     InitialVersion
* 
*************************************************************************/
tVoid vdmmgr_tclclienthandlerspm::vHandleSystemState( amt_tclServiceData* poMessage )
{
   TRACE_CCA_InMessage(poMessage, _CLASS);

   switch( poMessage->u8GetOpCode( ) )
   {
      case AMT_C_U8_CCAMSG_OPCODE_STATUS:
      {
         // Get data
         fi_tclVisitorMessage                oMsg( poMessage );
         cfc_spmfi_tclMsgSystemStateStatus   oData;
         (tVoid)oMsg.s32GetData( oData );
         ETG_TRACE_USR1(( "vHandleSystemState( ): Get new System Status from SPM: %d", ETG_ENUM(SPM_SYSTEM_STATES, (tU8)oData.SystemState.enType) ));
      #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
         if(   oData.SystemState.enType != cfc_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_MMI_ON
            && oData.SystemState.enType != cfc_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_MMI_ON_DIAG
            && oData.SystemState.enType != cfc_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_MMI_ON_TEL
            && oData.SystemState.enType != cfc_fi_tcl_SPM_e32_SYSTEM_STATES::FI_EN_SPM_SYSTEM_MMI_ON_SWDL
           )
         {
            // insert CD (only if not the first status)
            if( poMain  && !_bFirstSystemStatusAfterRegForFID )
            {
               vSendEventToCdCtrlThread( VDMMGR_EVENT_MASK_CMD_AUTOREINSERTCD );
            }
            // else ignore the first status
            // Store act. value
            _bBackGrdActive = TRUE;
         }
         else
         {
            _bBackGrdActive = FALSE;
         }
         // Reset marker for first MMgr status after registration for function ID
         _bFirstSystemStatusAfterRegForFID = FALSE;
      #endif
         break;
      }
      default:
      {
         TRACE_Warning("unhandled OpCode", _CLASS);
         break;
      }
   }
   (tVoid)poMessage->bDelete();
}


/*************************************************************************
*
* FUNCTION: bIsBackgrdActive
* 
* DESCRIPTION: handle a MediaManager Status
*
* PARAMETER: 
*
* RETURNVALUE: void
*
* HISTORY:     InitialVersion
* 
*************************************************************************/
#ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
tBool vdmmgr_tclclienthandlerspm::bIsBackgrdActive( tVoid ) const
{
   ETG_TRACE_USR1(( "bIsBackgrdActive(): Request BackGrdActice = %d", ETG_ENUM(BOOL, _bBackGrdActive) ));
   return _bBackGrdActive;
}
#endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE


/******************************************************************************
*
* FUNCTION:    bSetNewSPMSubState( tBool bSubStateData, 
*                    cfc_fi_tcl_SPM_e32_SubStateType::tenType e32SubStateType )
*
* DESCRIPTION: handle MMgr state from SPM
*              
*
* PARAMETER:   tBool bSubStateData
*
* RETURNVALUE: tBool bRet
*
*******************************************************************************/
tBool vdmmgr_tclclienthandlerspm::bSetNewSPMSubState( tBool bSubStateData, cfc_fi_tcl_SPM_e32_SubStateType::tenType e32SubStateType /*=cfc_fi_tcl_SPM_e32_SubStateType::FI_EN_SPM_U32_SUBSTATE_SET_SD_CARD_ACCESS*/ )
{
   tBool    bRet;

   if( _u16RegID != AMT_C_U16_REGID_INVALID )
   {
      bRet = TRUE;

      // Prepare method result data
      cfc_spmfi_tclMsgSubStatesSet  oResultData;
      // Copy data
      oResultData.SubStateType.enType  = e32SubStateType;
      oResultData.SubStateData         = bSubStateData;
      // +++   create a message, use the data of given message to get the address information    +++
      // construct result message
      fi_tclVisitorMessage oResultMsg( oResultData );
      // initialise result message
      oResultMsg.vInitServiceData(  CCA_C_U16_APP_MMGR,                 // AppID of this application
                                    CCA_C_U16_APP_SPM,                  // AppID of the Server
                                    AMT_C_U8_CCAMSG_STREAMTYPE_NODATA,  // StreamType
                                    0,                                  // StreamCounter
                                    _u16RegID,                          // RegisterID
                                    0,                                  // nCmdCounter
                                    CCA_C_U16_SRV_SPM,                  // SID of the service
                                    CFC_SPMFI_C_U16_SUBSTATES,          // function ID
                                    CCA_C_U8_OPCODE_SET                 // opcode
                                 );
      // ---   send it to the client    ---
      TRACE_CCA_OutMessage(&oResultMsg, _CLASS, __LINE__);
      if( poMain )
      {
         if( poMain->enPostMessage( &oResultMsg ) != AIL_EN_N_NO_ERROR )
         {
            // +++         can't send status:  enter error handling here ( trace or assert ) +++ 
            ETG_TRACE_ERR(( "bSetNewSPMSubState( ): Error: Cannot send CCA message to queue of APP ID: CCA_C_U16_APP_SPM" ));
            if( !oResultMsg.bDelete() )
            {
               ETG_TRACE_ERR(( "bSetNewSPMSubState( ): Error: CCA message which couldn't be posted couldn't be deleted also" ));
            }
            bRet = FALSE;
         }
      }
   }
   else
   {
      ETG_TRACE_ERR(( "bSetNewSPMSubState( ): Invalid RegId -> _u16RegID == AMT_C_U16_REGID_INVALID" ));
      bRet = FALSE;
   }
   return bRet;
}


/*static*/ tVoid vdmmgr_tclclienthandlerspm::vSendEventToCdCtrlThread( tU32 u32Event )
{
   tS32              s32Success;
   OSAL_tEventHandle hCbkEvent;

   ETG_TRACE_USR1(( "vSendEventToCdCtrlThread( ): Start: Event: %d", ETG_ENUM(MMGR_CD_EVENT, u32Event) ));
   if( OSAL_s32EventOpen( VDMMGR_CDCTRLIF_EVENTNAME, &hCbkEvent ) == OSAL_OK )
   {
      s32Success = OSAL_s32EventPost( hCbkEvent, u32Event, OSAL_EN_EVENTMASK_OR );
      if( s32Success == OSAL_ERROR )
      {
         ETG_TRACE_ERR(( "vSendEventToCdCtrlThread( ): Error sending event %d", ETG_ENUM(MMGR_CD_EVENT, u32Event) ));
      }
      
      s32Success = OSAL_s32EventClose( hCbkEvent );
      if(s32Success == OSAL_ERROR)
      { 
         ETG_TRACE_ERR(( "vSendEventToCdCtrlThread( ): Error closing event handle" ));
      }
   }
   else
   {
      ETG_TRACE_ERR(( "vSendEventToCdCtrlThread( ): Error opening event handle" ));
      TRACE_OSAL_ERROR( VDMMGR_TR_CLIENTSPM );
   }
}


/*************************************************************************
*
* FUNCTION:    tVoid vHandleWakeUpReason(...)
* 
* DESCRIPTION: react on messages with FID = CFC_SPMFI_C_U16_WAKEUPREASON
*
* PARAMETER: 
*
* RETURNVALUE: void
*
* HISTORY:     InitialVersion
* 
*************************************************************************/
tVoid vdmmgr_tclclienthandlerspm::vHandleWakeUpReason( amt_tclServiceData* poMessage )
{
   TRACE_CCA_InMessage(poMessage, _CLASS);

   // ---   the opcode tells what we should do   ---
   tU8  u8OpCode = poMessage->u8GetOpCode();
   switch( u8OpCode )
   {
      case AMT_C_U8_CCAMSG_OPCODE_STATUS:
      {
         fi_tclVisitorMessage                oMsg( poMessage );
         cfc_spmfi_tclMsgWakeupReasonStatus  oData;
         (tVoid)oMsg.s32GetData( oData );
         ETG_TRACE_USR1(( "vHandleWakeUpReason(): New WakeUpReason: %d", ETG_ENUM(WAKEUP_REASON, oData.WakeupReason.enType) ));
         // +++         handle data message form server         +++
         /*lint -save -e788 Info 788;enum constant 'x' not used within defaulted switch */
         switch( oData.WakeupReason.enType )
         {
            case cfc_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_EJECT:
               break;
            case cfc_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_POWER_ON:   // case, no break
            case cfc_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_RESTART:    // case, no break
            case cfc_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_RESET:      // case, no break
               #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
               // If one of these WakeUpReasons was set check if it's better to reset the stored CD type
               if( poMain && poMain->_vdmmgr_poCCAService && poMain->_vdmmgr_poCdctrlIf )
               {                         
				  #ifndef VARIANT_S_FTR_ENABLE_MMGR_BUGFIX_RECOGNIZE_CD_INSERT_AFTERON
                  ETG_TRACE_USR1(( "vHandleWakeUpReason( ): Reset CD Type to NO_MEDIA due to WakeUp Reason" ));	
				  // Last wake-up reason was a reset and there wasn't a CD type from OSAL -> reset CD Type
                  poMain->_vdmmgr_poCCAService->vNewCdType( MMGR_NO_MEDIA );
				  #endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_BUGFIX_RECOGNIZE_CD_INSERT_AFTERON
				  
                  // Start CD recognition with last OSAL value if there was already one.
                  if( poMain->_vdmmgr_poCdctrlIf->vGetFirstCdTypeValueArrived( ) )
                  {
                     poMain->_vdmmgr_poCdctrlIf->vRestartHandleMediaChange( );
                  }
               }
               #endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE
               break;
            #ifdef VARIANT_S_FTR_ENABLE_MMGR_PLAY_CD_AFTER_INSERT_FROM_DISPLAY_OFF_STATE
            // If a CD should be played after if the WakeUpReason CD_INSERT -> set the insert state
            case cfc_fi_tcl_SPM_e32_WAKEUP_REASON::FI_EN_SPM_U32_WAKEUP_INSERT:
               if( poMain && poMain->_vdmmgr_poCCAService )
               {
                  poMain->_vdmmgr_poCCAService->vNewStateInsert( MMGR_INSERT_AFTERON );
               }
               break;
            #endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_PLAY_CD_AFTER_INSERT_FROM_DISPLAY_OFF_STATE
            default:  // FI_EN_SPM_U32_WAKEUP_INVALID, FI_EN_SPM_U32_WAKEUP_CAN, FI_EN_SPM_U32_WAKEUP_EJECT, FI_EN_SPM_U32_WAKEUP_INSERT, FI_EN_SPM_U32_WAKEUP_IGNITION, FI_EN_SPM_U32_WAKEUP_RTC, FI_EN_SPM_U32_WAKEUP_ON_TIPPER, FI_EN_SPM_U32_WAKEUP_COPRO_REQ, FI_EN_SPM_U32_WAKEUP_MAUS_BUS, FI_EN_SPM_U32_WAKEUP_DOOR_OPEN, FI_EN_SPM_U32_SDCARD_INSERT, FI_EN_SPM_U32_WAKEUP_PHONE_MUTE
               break;
         }
         /*lint -restore*/
         break;
      }
      default:
      {
         TRACE_Warning("unhandled OpCode", _CLASS);
         break;
      }
   }
   (tVoid)poMessage->bDelete();
} //lint !e1762  DevStudios says:'not const, please'


/*************************************************************************
*
* FUNCTION:    tVoid vHandleStartUpReason(...)
* 
* DESCRIPTION: react on messages with FID = CFC_SPMFI_C_U16_STARTUPREASON
*
* PARAMETER: 
*
* RETURNVALUE: void
*
* HISTORY:     InitialVersion
* 
*************************************************************************/
tVoid vdmmgr_tclclienthandlerspm::vHandleStartUpReason( amt_tclServiceData* poMessage )
{
   TRACE_CCA_InMessage(poMessage, _CLASS);

   // ---   the opcode tells what we should do   ---
   tU8 u8OpCode = poMessage->u8GetOpCode();
   switch( u8OpCode )
   {
      case AMT_C_U8_CCAMSG_OPCODE_STATUS:
      {
         fi_tclVisitorMessage                   oMsg( poMessage );
         cfc_spmfi_tclMsgStartupReasonStatus    oData;
         (tVoid)oMsg.s32GetData( oData );
         ET_TRACE_INFO_BIN( VDMMGR_TR_CLIENTSPM, ET_EN_T16 _ VDMMGR_TR_CLIENTSPM_24 _ ET_EN_T8 _ (tU8)oData.StartupReason.enType _ ET_EN_DONE );
         // +++         handle data message form server         +++
         // Check if the bit for startup reason CD INSERT is set
         if( poMain && poMain->_vdmmgr_poCCAService )
         {
            #ifdef VARIANT_S_FTR_ENABLE_MMGR_PLAY_CD_AFTER_INSERT_FROM_DISPLAY_OFF_STATE
            if( (tU32)oData.StartupReason.enType  &  (tU32)(cfc_fi_tcl_SPM_e32_STARTUP_REASON::FI_EN_SPM_U32_STARTUP_ON_INSERT) )
            {
               // If a CD should be played if the WakeUpReason is CD_INSERT -> set the insert state
               poMain->_vdmmgr_poCCAService->vNewStateInsert( MMGR_INSERT_AFTERON );
            }
            #endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_PLAY_CD_AFTER_INSERT_FROM_DISPLAY_OFF_STATE
         }  // Other possible bits are FI_EN_SPM_U32_STARTUP_ON_INSERT, 
         break;
      }
      default:
      {
         TRACE_Warning("unhandled OpCode", _CLASS);
         break;
      }
   }
   (tVoid)poMessage->bDelete();
}  //lint !e1762  DevStudios says:'not const, please'


//////////////////////////// GPIO handling (Insert clamps)  ///////////////////

/******************************************************************************
*  FUNCTION:    bInstallGpioCallback ( OSAL_tGPIODevID Gpio, OSAL_tpfGPIOCallback pvCallback,
*                                      tVoid* pvArg, tU16 u16TriggerLevel ) const
*
*  DESCRIPTION: This method facilitates installation of a callback function for a designated  gpio pin
*
*  PARAMETER:   u32Gpio: pin id.
*               pvCallback: callback function.
*
*  RETURNVALUE: TRUE:   NO_ERROR
*               FALSE:  ERROR
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tBool vdmmgr_tclclienthandlerspm::bInstallGpioCallback( OSAL_tGPIODevID Gpio, OSAL_tpfGPIOCallback pvCallback, tVoid* pvArg/*=NULL*/, tU16 u16TriggerLevel/*=OSAL_GPIO_EDGE_BOTH*/ ) const
{
   tBool                bReturn = TRUE;
   OSAL_trGPIOData      tGpioData;
   OSAL_tIODescriptor   tDevGpioHandle;

   ETG_TRACE_USR2(( "bInstallGpioCallback( ): Start" ));
   // Update the dev id
   tGpioData.tId = Gpio;
   // Open gpio device in read/write mode
   tDevGpioHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GPIO, OSAL_EN_READWRITE );
   if( OSAL_ERROR != tDevGpioHandle )
   {
      // param to pass to the callback function
      OSAL_trGPIOCallbackData    tGpioCallbackData;
      tGpioCallbackData.rData.unData.pfCallback = pvCallback;
      tGpioCallbackData.pvArg                   = pvArg;
      tGpioCallbackData.rData.tId               = Gpio;
      //param for setting the trigger edge
      tGpioData.unData.u16Edge                  = u16TriggerLevel;
      // register callback
      if( OSAL_s32IOControl( tDevGpioHandle, OSAL_C_32_IOCTRL_GPIO_SET_CALLBACK, (intptr_t)(&tGpioCallbackData) ) == OSAL_ERROR )
      {
         ETG_TRACE_ERR(( "bInstallGpioCallback: Error: Failed to set callback function for GPIO" ));
         bReturn = FALSE;
      }
      // Set trigger edge BOTH
      else if( OSAL_s32IOControl( tDevGpioHandle, OSAL_C_32_IOCTRL_GPIO_SET_TRIGGER, (intptr_t)(&tGpioData) ) == OSAL_ERROR )
      {
         ETG_TRACE_ERR(( "bInstallGpioCallback: Error: Failed to set trigger for GPIO" ));
         bReturn = FALSE;
      }
      else
      {
         //param for setting enabling interrupt
         tGpioData.unData.bState = TRUE;
         // enable interrupts
         if( OSAL_s32IOControl( tDevGpioHandle, OSAL_C_32_IOCTRL_GPIO_ENABLE_INT, (intptr_t)(&tGpioData)) == OSAL_ERROR )
         {
            ETG_TRACE_ERR(( "bInstallGpioCallback: Error: Failed to enable interrupt for GPIO" ));
            bReturn = FALSE;
            tGpioCallbackData.rData.unData.pfCallback = NULL;
            OSAL_s32IOControl( tDevGpioHandle, OSAL_C_32_IOCTRL_GPIO_SET_CALLBACK, (intptr_t)&tGpioCallbackData );
         }
      }
      // Close GPIO device
      if( OSAL_s32IOClose(tDevGpioHandle) != OSAL_OK )
      {
         ETG_TRACE_ERR(( "bInstallGpioCallback: Error: Failed to close the OSAL device gpio" ));
      }
   }
   else  // if (OSAL_ERROR != tDevGpioHandle)
   {
      ETG_TRACE_ERR(( "bInstallGpioCallback: Error: Failed to open OSAL deivice gpio" ));
      bReturn = FALSE;
   }
   ETG_TRACE_USR2(( "bInstallGpioCallback( ): End" ));

   return bReturn;
}


/*static*/tVoid vdmmgr_tclclienthandlerspm::vCallbackGPIOHdl( tVoid* /*pvArg*/ )
{
   ETG_TRACE_USR4(("vCallbackGPIOHdl()" ));

   tS32              s32Success;
   OSAL_tEventHandle hCbkEvent;

   // send event and leave callback context
   if( OSAL_s32EventOpen( VDMMGR_CLIENTSPM_EVENTNAME, &hCbkEvent ) == OSAL_OK )
   {
      s32Success = OSAL_s32EventPost( hCbkEvent, VDMMGR_CLIENTSPM_EVENT_MASK_CD_DRIVE_INSERT_CLAMPS, OSAL_EN_EVENTMASK_OR );
      if( s32Success == OSAL_ERROR )
      {
         ETG_TRACE_ERR(( "vCallbackGPIOHdl( ): Error sending event VDMMGR_CLIENTSPM_EVENT_MASK_CD_DRIVE_INSERT_CLAMPS" ));
      }

      s32Success = OSAL_s32EventClose( hCbkEvent );
      if(s32Success == OSAL_ERROR)
      { 
         ETG_TRACE_ERR(( "vCallbackGPIOHdl( ): Error closing event handle" ));
      }
   }
   else
   {
      ETG_TRACE_ERR(( "vCallbackGPIOHdl( ): Error opening event handle" ));
      TRACE_OSAL_ERROR( VDMMGR_TR_CLIENTSPM );
   }
}


tVoid  vdmmgr_tclclienthandlerspm::vGPIOThread( tVoid* pvArg )
{
   OSAL_tEventMask   oMask;
   OSAL_tEventMask   oResultMask = 0;
   tS32              s32Success;

   ETG_TRACE_USR1(( "vGPIOThread(): Start waiting for events" ));

   vdmmgr_tclclienthandlerspm* pClientSpm = (vdmmgr_tclclienthandlerspm*)pvArg;

   oMask = VDMMGR_CLIENTSPM_EVENT_MASK_CD_DRIVE_INSERT_CLAMPS;

   if( OSAL_s32EventOpen( VDMMGR_CLIENTSPM_EVENTNAME, &_hEvent ) == OSAL_OK )
   {
      for( ;/*ever*/; )
      {
         s32Success = OSAL_s32EventWait( _hEvent, oMask, OSAL_EN_EVENTMASK_OR,
                                         OSAL_C_TIMEOUT_FOREVER, &oResultMask );
         if( s32Success == OSAL_ERROR )
         {
            ETG_TRACE_ERR(( "vGPIOThread(): Error: OSAL_s32EventWait() returns with error" ));
         }
         //clear event
         if( OSAL_OK != OSAL_s32EventPost( _hEvent,~oResultMask,OSAL_EN_EVENTMASK_AND ) )
         {
            ETG_TRACE_ERR(( "vGPIOThread(): Error: Unable to clear event -> OSAL_s32EventPost( )" ));
         }
         //handle event
         pClientSpm->vHandleEvent( oResultMask );
      }//for ever
   }
   else
   {
      ETG_TRACE_ERR(( "vGPIOThread(): Error: OSAL_s32EventOpen returns with error" ));
      // Get error code if not successful.
      TRACE_OSAL_ERROR( VDMMGR_TR_ERRORIF );
   }
}


/*************************************************************************/
/*
* FUNCTION:    vHandleEvent (OSAL_tEventMask rEventMask)
* 
* DESCRIPTION: handler for MediaManager Source - events 
*
* PARAMETER:   eventMask
*
* RETURNVALUE: void
*/
/*************************************************************************/
tVoid vdmmgr_tclclienthandlerspm::vHandleEvent( OSAL_tEventMask rEventMask )
{
   ETG_TRACE_USR1(( "vHandleEvent( ): Start: -> Switch Event Mask" ));

   // Check for device failure
   if( rEventMask & VDMMGR_CLIENTSPM_EVENT_MASK_CD_DRIVE_INSERT_CLAMPS )
   {
      vHandleInsertClampsGPIO( );
   }
}



/******************************************************************************
*  FUNCTION:    vHandleInsertClampsGPIO( )
*
*  DESCRIPTION: This method helps to read the gpio pin state as either input/output
*
*  PARAMETER:   none
*
*  RETURNVALUE: none
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tVoid vdmmgr_tclclienthandlerspm::vHandleInsertClampsGPIO( )
{
   OSAL_trGPIOData      tGpioData;
   OSAL_tIODescriptor   tDevGpioHandle;

   _bMediaInSlot = FALSE;        // Reset marker for indication: media in slot

   ETG_TRACE_USR2(( "vHandleInsertClampsGPIO( ): Start" ));
   // Update the dev id
   tGpioData.tId = (OSAL_tGPIODevID)OSAL_EN_SPM_GPIO_WAKEUP_CD;
   // Open gpio device in read/write mode
   tDevGpioHandle = OSAL_IOOpen( OSAL_C_STRING_DEVICE_GPIO, OSAL_EN_READWRITE );
   if( tDevGpioHandle != OSAL_ERROR )
   {
      // ask for insert clamps state
      if( OSAL_s32IOControl( tDevGpioHandle, OSAL_C_32_IOCTRL_GPIO_IS_STATE_ACTIVE, (intptr_t)(&tGpioData) ) == OSAL_ERROR )
      {
         ETG_TRACE_ERR(( "vHandleInsertClampsGPIO: Error: Failed to request GPIO active state" ));
      }
      else
      {
         ETG_TRACE_USR1(( "vHandleInsertClampsGPIO( ): Get InsertStateStatus (Insert Clamps) via GPIO: New: %d", ETG_ENUM(BOOL, (tU8)tGpioData.unData.bState) ));
         // Stop 'ensure CD NO_MEDIA' timer for MEDIA_EJECTED
         if( poMain && poMain->_vdmmgr_poCCAService )
         {
            ETG_TRACE_USR3(( "vHandleInsertClampsGPIO( ): Stop timer for automatic reinsert" ));
            (tVoid)poMain->_vdmmgr_poCCAService->_oEnsureCdNoMediaTimer.bStop();
         }
         // check state of InsertClamps pin
         if( tGpioData.unData.bState ) // bState == TRUE (Fix(Swap call functions) for NCG3D-7014)
         {
             vInsertClampsActive();     // ... if the clamps are active (CD in slot)
         }
         else                          // bState == FALSE
         {
             vInsertClampsInactive();   // ... if the clamps are inactive
         }
      }
      if( OSAL_s32IOClose( tDevGpioHandle ) != OSAL_OK )
      {
         ETG_TRACE_ERR(( "vHandleInsertClampsGPIO: Error: Failed to close the OSAL device gpio" ));
      }
   }
   else  // if( tDevGpioHandle != OSAL_ERROR )
   {
      ETG_TRACE_ERR(( "vHandleInsertClampsGPIO: Error: Failed to open OSAL deivice gpio" ));
   }
   ETG_TRACE_USR2(( "vHandleInsertClampsGPIO( ): End" ));
}


tVoid vdmmgr_tclclienthandlerspm::vInsertClampsActive( )
{
   T_e8_InsertState     e8CdInsertState = MMGR_INSERT_AFTERON;       // Init with default for FI_EN_SPM_U32_CD_INSERT_ACTIVE

   ETG_TRACE_USR2(( "vInsertClampsActive( ): Start" ));
   if( poMain && poMain->_vdmmgr_poCCAService )
   {
      // Store state
      _bInsertClampsActive = TRUE;
      // Start 10s timer for automatic reinsert of CD
      (tVoid)poMain->_vdmmgr_poCCAService->_oAutoReInsertTimer.bStart( );
      ETG_TRACE_USR1(( "vInsertClampsActive( ): Insert Clamps active. -> CD in Slot -> Start timer for automatic reinsert" ));
      if( poMain->_vdmmgr_poCCAService->_bEjectStartet == TRUE )
      {
         ETG_TRACE_USR1(( "vInsertClampsActive( ): Insert Clamps active and eject started." ));
         // Set marker for indication: media in slot
         _bMediaInSlot = TRUE;
         // Set CD type TO MEDIA_IN_SLOT
         poMain->_vdmmgr_poCCAService->vNewCdType( MMGR_IN_SLOT );
      }
      else if( (poMain->_vdmmgr_poCCAService->enGetActCdType( ) ) == MMGR_NO_MEDIA )
      {
      #ifndef VARIANT_S_FTR_ENABLE_MMGR_PLAY_CD_AFTER_INSERT_FROM_DISPLAY_OFF_STATE
         // Check if background mode is active -> Display off. (!Not for VW LLTFT!)
         if( bIsBackgrdActive() )
         {
            // Set insert state to a value that this CD isn't played automatically
            e8CdInsertState = MMGR_INSERT_BEFOREON;
         }
      #endif // #ifndef VARIANT_S_FTR_ENABLE_MMGR_PLAY_CD_AFTER_INSERT_FROM_DISPLAY_OFF_STATE
         // Set insert state
         poMain->_vdmmgr_poCCAService->vNewStateInsert( e8CdInsertState, FALSE );
         // Set CD type
         poMain->_vdmmgr_poCCAService->vNewCdType( MMGR_INSERTION, TRUE );
         // To trigger SPM to change from system state STANDBY to ON
         (tVoid)bSetNewSPMSubState( FALSE, cfc_fi_tcl_SPM_e32_SubStateType::FI_EN_SPM_U32_SUBSTATE_CD_INSERT );
         ETG_TRACE_USR1(( "vInsertClampsActive( ): Insert Clamps active and no eject started. CD state 'insert after sytem ON' and CD type'insertion'" ));
      }
      // else: could happen if ejected was directly started in OSAL (e.g. via TTFis)
   }
   ETG_TRACE_USR2(( "vInsertClampsActive( ): End" ));
}


tVoid vdmmgr_tclclienthandlerspm::vInsertClampsInactive( )
{
   T_e8_DiagLoadStatus  e8LoaderState;
   T_e8_CdType          e8CdType;

   ETG_TRACE_USR2(( "vInsertClampsInactive( ): Start" ));
   if( poMain && poMain->_vdmmgr_poCCAService && poMain->_vdmmgr_poCdctrlIf )
   {
      // Store state
      _bInsertClampsActive = FALSE;
      // Stop poll timer. CD is no longer inside or CD is being inserted.
      (tVoid)poMain->_vdmmgr_poCCAService->_oAutoReInsertTimer.bStop( );
      ETG_TRACE_USR1(( "vInsertClampsInActive( ): Stop poll timer. The CD is no longer inside the slot or the CD is being inserted." ));
      // Reset 'eject started' flag
      poMain->_vdmmgr_poCCAService->_bEjectStartet = FALSE;
      // Delay the loader state request for TN 2007 drive because there is a mechanical difference between BP7(/VA) and TN2007
      if( OSAL_s32ThreadWait( VDMMGR_CLIENTSPM_WAITTIME_GETLOADERSTATE ) == OSAL_ERROR )
      {
         TRACE_OSAL_ERROR( VDMMGR_TR_CLIENTSPM );
      }
      // Get loader and CD type
      e8LoaderState  = poMain->_vdmmgr_poCdctrlIf->e8GetLoaderState( );
      e8CdType       = poMain->_vdmmgr_poCCAService->enGetActCdType( );
      // Check if a CD was only a short time in slot before it was taken out again.
      if(   ( e8CdType == MMGR_INSERTION )
         && ( e8LoaderState == MMGR_NO_MEDIA_IN_DRIVE )
        )
      {
         ETG_TRACE_USR1(( "vInsertClampsInactive( ): Insert Clamps not active and NoMediaInDrive -> CD was inserted at removed before full insert" ));
         // Set CD type
         poMain->_vdmmgr_poCCAService->vNewCdType( MMGR_NO_MEDIA );
      }
      // or if the CD was forced to insert again (before a full eject)
      else if(    (   ( e8CdType == MMGR_NO_MEDIA )
                   || ( e8CdType == MMGR_EJECTING )
                   || ( e8CdType == MMGR_IN_SLOT )
                  )
               && (   ( e8LoaderState == MMGR_EJECT_IN_PROGRESS )
                   || ( e8LoaderState == MMGR_MEDIA_INSIDE )
                  )
               && !poMain->_vdmmgr_poCdctrlIf->bGetCdReinsertActive()
             )
      {
         ETG_TRACE_USR1(( "vInsertClampsInactive( ): Insert Clamps not active and EjectInProgress -> CD was in slot (not taken out) and forced to insert again" ));
         // Set insert state
         poMain->_vdmmgr_poCCAService->vNewStateInsert( MMGR_INSERT_AFTERON );
      #ifndef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE_PIONEER
         // Only if the CD drive is not PIONEER because: If the CD is already in slot (after eject) and the eject command is given again 
         // the drive insert the CD until the insert clamps are inactive. This leads to a CD Type insertion and a PopUp 'CD Loading'
         // Set CD type
         poMain->_vdmmgr_poCCAService->vNewCdType( MMGR_INSERTION );
      #endif // #ifdef VARIANT_S_FTR_ENABLE_MMGR_CD_DRIVE_PIONEER
         // To trigger SPM to change from system state STANDBY to ON
         (tVoid)bSetNewSPMSubState( FALSE, cfc_fi_tcl_SPM_e32_SubStateType::FI_EN_SPM_U32_SUBSTATE_CD_INSERT );
      }
   }
   ETG_TRACE_USR2(( "vInsertClampsInactive( ): End" ));
}


//-----------------------------------------------------------------------------
// help functions
//-----------------------------------------------------------------------------


/******************************************************************************
*  FUNCTION:    vOnAsyncRegisterConf ( tU16 u16RegisterId, tU16 u16ServerAppId,
*                                      tU16 u16ServiceId, tU16 u16TargetSubId )
*
*  DESCRIPTION: 
*
*  PARAMETER:   u16RegisterId
*               u16ServiceId
*
*  RETURNVALUE: TRUE:   NO_ERROR
*               FALSE:  ERROR
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tVoid vdmmgr_tclclienthandlerspm::vOnAsyncRegisterConf( tU16 u16RegisterId, tU16 u16ServerAppId, tU16 u16ServiceId, tU16 u16TargetSubId )
{
   FOR_FUTURE_USE( u16ServerAppId );
   FOR_FUTURE_USE( u16TargetSubId );

   if( u16ServiceId == CCA_C_U16_SRV_SPM )
   {
      if( AMT_C_U16_REGID_INVALID != u16RegisterId )
      {
         _u16RegID = u16RegisterId;
      }
      else
      {
         // +++ registration failed. Enter error handling here ( trace or assert ) +++
         ETG_TRACE_USR1(( "bRegisterForService(): registration failed" ));
         // Assert if registration is failed => assert to show that something is wrong
         NORMAL_M_ASSERT_ALWAYS();
      }
   }
}


/*************************************************************************
*  FUNCTION:    tBool vdmmgr_tclclienthandlerspm::bRegisterForService();
*
*  DESCRIPTION: register for a service
*
*  PARAMETER:   void
*
*  RETURNVALUE: TRUE == NO_ERROR, FALSE == ERROR
*
*  History:
*  InitialVersion
* 
*************************************************************************/
tBool vdmmgr_tclclienthandlerspm::bRegisterForService()
{
   tBool bRetVal = FALSE;

   // ---    registration invalid?   ---
   if( _u16RegID == AMT_C_U16_REGID_INVALID && _bRegAsyncStarted == FALSE )
   {
      // --- we register for the service so the server can't have any FID - registration ---
      _bFidReg = FALSE;
      // --- try to register ---
      if( poMain )
      {
         bRetVal = poMain->bRegisterAsync( CCA_C_U16_SRV_SPM, VDMMGR_C_U16_SERVICE_SPM_MAJOR_VERSION, 
                                           VDMMGR_C_U16_SERVICE_SPM_MINOR_VERSION, 0,CCA_C_U16_APP_SPM );
         _bRegAsyncStarted = TRUE;
         ETG_TRACE_USR1(( "bRegisterForService( ): called u16RegisterService()" ));
      }
   }
   else
   {
     // +++  it could be an error to call this function, if we are already registered
     //      decide if you need error handling here     +++
     ETG_TRACE_USR1(( "bRegisterForService(): -> _u16RegID == AMT_C_U16_REGID_INVALID" ));
   }

   return bRetVal;
}


/******************************************************************************
*  FUNCTION:    tBool vdmmgr_tclclienthandlerspm::bUnregisterForService();
*
*  DESCRIPTION: register for a service
*
*  PARAMETER:   void
*
*  RETURNVALUE: TRUE == NO_ERROR, FALSE == ERROR
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tBool vdmmgr_tclclienthandlerspm::bUnregisterForService( )
{
   tBool bRetVal;

   bRetVal = TRUE;

   // +++   check if your application is ready to unregister   +++
   // ---    registration valid?   ---
   if ( _u16RegID != AMT_C_U16_REGID_INVALID )
   {
      if( poMain )
      {
         // ---      unregister for Service      ---
         poMain->vUnregisterService( (tU16) CCA_C_U16_SRV_SPM );
         // Make register-ID invalid:
         _u16RegID = AMT_C_U16_REGID_INVALID;
         _bRegAsyncStarted = FALSE;
         ETG_TRACE_USR1(( "bUnregisterForService( ): unregistered DiagLog service" ));
      }
   }
   else
   {
      // +++     it could be an error to call this function, if we are already unregistered
      //         decide if you need error handling here     +++
      ETG_TRACE_USR1(( "bUnregisterForService( ): unregistered DiagLog service already done" ));
   }
   return bRetVal;
}


/******************************************************************************
*  FUNCTION:    tBool vdmmgr_tclclienthandlerspm::bRegisterForFID( tU16 u16FID )
*
*  DESCRIPTION: register for a function of a service
*               !!! before this, you must register for the service !!!  
*
*  PARAMETER:   u16FID: FID of the function 
*
*  RETURNVALUE: TRUE == NO_ERROR, FALSE == ERROR
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tBool vdmmgr_tclclienthandlerspm::bRegisterForFID(tU16 u16FID)
{
   tBool bRetVal = TRUE;

   // ---    registration invalid?   ---
   if ( _u16RegID == AMT_C_U16_REGID_INVALID )
   {
      bRetVal= FALSE;
      // +++      registration for the service failed or not done  +++
      ETG_TRACE_USR1(( "bRegisterForFID( ): service is not registered" ));
   }
   else
   {
      // +++   create message with UpReg - request:   +++
      gm_tclEmptyMessage oUpRegMessage( CCA_C_U16_APP_MMGR, CCA_C_U16_APP_SPM, _u16RegID, 0, CCA_C_U16_SRV_SPM, u16FID, AMT_C_U8_CCAMSG_OPCODE_UPREG );
      TRACE_On_Invalid_Pointer( poMain, "poMain", _CLASS );
      if( poMain )
      {
         TRACE_CCA_OutMessage(&oUpRegMessage, _CLASS, __LINE__);
         if( poMain->enPostMessage( &oUpRegMessage ) == AIL_EN_N_NO_ERROR )
         {
            // +++      message send success      make a trace if you want  +++
         }
         else  // +++      message send error  +++
         {
            bRetVal= FALSE;
            ETG_TRACE_ERR(( "bRegisterForFID( ):Error: bRegisterForFID(): Failed to register for function ID: %d", ETG_ENUM(MMGR_FUNCTIONID_SPM, u16FID) ));
            if ( !oUpRegMessage.bDelete() )
            {
               TRACE_Error("message deletion failed", _CLASS);
            }
         }
      }
   }
   return bRetVal;
}


/******************************************************************************
*  FUNCTION:    tBool vdmmgr_tclclienthandlerspm::bSendGetForFID( tU16 u16FID )
*
*  DESCRIPTION: Send a AMT_C_U8_CCAMSG_OPCODE_GET to service to request actual
*               status
*               !!! before this, you must register for the service !!!  
*
*  PARAMETER:   u16FID: FID of the function 
*
*  RETURNVALUE: TRUE == NO_ERROR, FALSE == ERROR
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tBool vdmmgr_tclclienthandlerspm::bSendGetForFID( tU16 u16FID )
{
   tBool bRetVal = TRUE;

   // ---    registration invalid?   ---
   if ( _u16RegID == AMT_C_U16_REGID_INVALID )
   {
      bRetVal= FALSE;
      // +++      registration for the service failed or not done  +++
      ETG_TRACE_USR1(( "bSendGetForFID(): service is not registered" ));
   }
   else
   {
      // +++   create message with UpReg - request:   +++
      gm_tclEmptyMessage oGetMessage( CCA_C_U16_APP_MMGR, CCA_C_U16_APP_SPM, _u16RegID, 0, CCA_C_U16_SRV_SPM, u16FID, AMT_C_U8_CCAMSG_OPCODE_GET );
      TRACE_On_Invalid_Pointer( poMain, "poMain", _CLASS );
      if( poMain )
      {
         TRACE_CCA_OutMessage(&oGetMessage, _CLASS, __LINE__);
         if( poMain->enPostMessage( &oGetMessage ) == AIL_EN_N_NO_ERROR )
         {
            // +++  message send success      make a trace if you want  +++
         }
         else
         {
            bRetVal= FALSE;
            // +++      message send error  +++
            ETG_TRACE_ERR(( "bSendGetForFID( ): Error: Failed to send OPCODE_GET for function ID: %d", ETG_ENUM(MMGR_FUNCTIONID_SPM, u16FID) ));
            if (!oGetMessage.bDelete())
            {
               TRACE_Error("message deletion failed", _CLASS);
            }
         }
      }
   }
   return bRetVal;
}


/******************************************************************************
*  FUNCTION:    bUnregisterForFID(tU16 u16FID)
*
*  DESCRIPTION: unregister for a function of a service
*
*  PARAMETER:   u16FID: FID of the function
*
*  RETURNVALUE: TRUE == NO_ERROR, FALSE == ERROR
*
*  History:
*  InitialVersion
* 
*******************************************************************************/
tBool vdmmgr_tclclienthandlerspm::bUnregisterForFID(tU16 u16FID)
{
   tBool bRetVal= TRUE;

   // ---    registration invalid?   ---
   if( _u16RegID == AMT_C_U16_REGID_INVALID )
   {
      bRetVal= FALSE;
      // +++      registration for the service failed or not done or already unregistered!  +++
      ETG_TRACE_USR1(( "bUnregisterForFID( ): Registration failed or deregistration already done" ));
   }
   // +++   create message with RelUpReg - request:   +++
   gm_tclEmptyMessage oRelupRegMessage( CCA_C_U16_APP_MMGR, CCA_C_U16_APP_SPM, _u16RegID, 0, CCA_C_U16_SRV_SPM, u16FID, AMT_C_U8_CCAMSG_OPCODE_RELUPREG );

   TRACE_On_Invalid_Pointer( poMain, "poMain", _CLASS );
   if( poMain )
   {
      TRACE_CCA_OutMessage(&oRelupRegMessage, _CLASS, __LINE__);
      if( poMain->enPostMessage( &oRelupRegMessage ) == AIL_EN_N_NO_ERROR )
      {
         // +++      message send success   +++
      }
      else
      {
         bRetVal= FALSE;
         // +++      message send error  +++
         ETG_TRACE_USR1(( "bUnregisterForFID(): Error: Failed to unregister for function ID: %d", ETG_ENUM(MMGR_FUNCTIONID_SPM, u16FID) ));
         if ( !oRelupRegMessage.bDelete())
         {
            TRACE_Error("message deletion failed", _CLASS);
         }
      }
   }
   return bRetVal;
}


/******************************************************************************
*
* FUNCTION:    vOnServiceState
* 
* DESCRIPTION: handle a state change of the service we use
*
* PARAMETER: 
*
* RETURNVALUE: void
*
*******************************************************************************/
tVoid vdmmgr_tclclienthandlerspm::vOnServiceState( tU16 u16ServiceId, tU16 u16ServerId, tU16 u16RegisterId, tU8  u8ServiceState, tU16 u16SubId )
{
   tBool bRetVal;

   FOR_FUTURE_USE( u16RegisterId );
   FOR_FUTURE_USE( u16ServerId );
   FOR_FUTURE_USE( u16SubId );

   ETG_TRACE_USR1(( "vOnServiceState(): Get service state: %d", ETG_ENUM(AMT_SERVICE_STATE, u8ServiceState) ));

   if( u16ServiceId != CCA_C_U16_SRV_SPM )
   {
      // +++      this is not the service we use    +++
      ETG_TRACE_USR1(( "vOnServiceState(): u16ServiceId != CCA_C_U16_SRV_SPM -> ID: %u(8,2,M)", ETG_ENUM(ail_u16ServiceId, u16ServiceId) ));
   }
   // ---   analyze the state of the server   ---
   switch( u8ServiceState )
   {
      case AMT_C_U8_SVCSTATE_AVAILABLE:
      {
         // +++  the service is now available.
         // ---  register only once  ---
         if( !_bFidReg )
         {
            // +++  register for all FIDs you need +++ 
            bRetVal = bRegisterForFID( CFC_SPMFI_C_U16_SUBSTATES );
            bRetVal = bRegisterForFID( CFC_SPMFI_C_U16_SYSTEMSTATE )    && bRetVal;
            if( bRetVal  )
            {
               _bFirstSystemStatusAfterRegForFID = TRUE;
            }
            (tVoid)bSendGetForFID( CFC_SPMFI_C_U16_WAKEUPREASON );
            (tVoid)bSendGetForFID( CFC_SPMFI_C_U16_STARTUPREASON );

            if( bRetVal )
            {
               _bFidReg = TRUE;
            }
         }
         break;
      }
      case AMT_C_U8_SVCSTATE_NOT_AVAILABLE:
      {
         if( _bFidReg )
         {
            (tVoid)bUnregisterForFID( CFC_SPMFI_C_U16_SUBSTATES );
            (tVoid)bUnregisterForFID( CFC_SPMFI_C_U16_SYSTEMSTATE );
            (tVoid)bUnregisterForFID( CFC_SPMFI_C_U16_WAKEUPREASON );
            (tVoid)bUnregisterForFID( CFC_SPMFI_C_U16_STARTUPREASON );
            _bFidReg = FALSE;
         }
         // +++ the service is not available anymore. +++
         break;
      }
      case AMT_C_U8_SVCSTATE_REG_INVALID:
      {
         //  --- the server has lost our registration. --- 
         //  --- register again. ---
         (tVoid)bRegisterForService( );
         _bFidReg = FALSE;
         break;
      }
      default:
      {
         // +++ unknown service state. Enter error handling ( trace or assert ) +++
         break;
      }
   }//switch
}
