/******************************************************************************
* FILE:         vdmmgr_cdctrl.cpp
* PROJECT:      MIB2_ENTRY
* SW-COMPONENT: Virtual Device Media Manager
*------------------------------------------------------------------------------
*
* DESCRIPTION: VD MediaManager
*              
*------------------------------------------------------------------------------
* COPYRIGHT:    (c) 2005 Robert Bosch GmbH, Hildesheim
* HISTORY:      
* Date      | Author             | Modification
* 27.05.05  | CM-DI/ESA2 Fiebing | initial version
*
*******************************************************************************/
#include "Config.h"
#include <map>
#include <vector>

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h" 

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_MEDIAMANAGER_CDCTRLIF
#include "trcGenProj/Header/vdmmgr_cdctrlif.cpp.trc.h"
#endif

#define AIL_S_IMPORT_INTERFACE_GENERIC
#include "ail_if.h"         // use AIL template with MessageMaps
#define AHL_S_IMPORT_INTERFACE_NOTIFICTABLE
#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"

#define SCD_S_IMPORT_INTERFACE_GENERIC
#include "scd_if.h"

#define VDMMGR_S_IMPORT_INTERFACE_MSG
#include "vdmmgr_if.h"                    // For VD MMgr interface

#define VD_DIAGLOG_S_IMPORT_INTERFACE_MSG
#include "vd_diaglog_if.h"

#define CFC_FI_S_IMPORT_INTERFACE_CFC_SPMFI_TYPES        // or #define CFC_FI_S_IMPORT_INTERFACE_CFC_SPMFI_STDVISITORS
#include "cfc_fi_if.h"

/* ************************************************************************** */
/* include the public interface                                               */
/* ************************************************************************** */
#include "vdmmgr_main.h"                  // For the pointer to main application
#include "vdmmgr_timer.h"                 // 'automatic reinsert CD' timer
#include "vdmmgr_service.h"               // (CCA) send functions
#include "vdmmgr_cdctrlif.h"              // class vdmmgr_tclcdctrlif + defines
#include "vdmmgr_clienthandlerspm.h"      // class vdmmgr_tclcdctrlif + defines
#include "vdmmgr_ITC_Container.h"
#include "vdmmgr_ITC_Map.h"
#include "vdmmgr_clienthandlerdiaglog.h"  // store ITC status changes
#include "vdmmgr_errorif.h"               // Error handling interface
#include "vdmmgr_trace.h"                 // trace stuff


/* ************************************************************************** */
/* INITIALIZE STATIC CLASS VARIABLES                                          */
/* ************************************************************************** */

tU16                 vdmmgr_tclcdctrlif::_u16ModeChange   = 0;
tU16                 vdmmgr_tclcdctrlif::_u16MediaState   = OSAL_C_U16_MEDIA_NOT_READY;
tU16                 vdmmgr_tclcdctrlif::_u16MediaChange  = OSAL_C_U16_UNKNOWN_MEDIA;
tU16                 vdmmgr_tclcdctrlif::_u16TotalFailure = 0;
tU16                 vdmmgr_tclcdctrlif::_u16Defect       = 0;
tU16                 vdmmgr_tclcdctrlif::_u16DeviceState  = OSAL_C_U16_DEVICE_NOT_READY;

vdmmgr_tclcdctrlif*  vdmmgr_tclcdctrlif::_poInstance      = OSAL_NULL;

tBool                vdmmgr_tclcdctrlif::_bLowVoltageDuringEject = FALSE;
tU8                  vdmmgr_tclcdctrlif::_u8CDEjectLowVoltageRetryCnt = 0;


/******************************************************************************/
/*
* FUNCTION:    poGetInstance( vdmmgr_tclApp* poVdmmgrMain )
* 
* DESCRIPTION: creates the static  vdmmgr_tclcdctrlif - object and returns a 
*              pointer or returns a pointer to the already existing object
*  
* PARAMETER:   vdmmgr_tclApp: Pointer to vdmmgr main application.
*
* RETURNVALUE: pointer to the vdmmgr_tclcdctrlif - object
*/
/******************************************************************************/
vdmmgr_tclcdctrlif* vdmmgr_tclcdctrlif::poGetInstance( vdmmgr_tclApp* poVdmmgrMain )
{
    if(_poInstance == OSAL_NULL)
    {
        _poInstance = new vdmmgr_tclcdctrlif( poVdmmgrMain );
    }

    if( _poInstance == OSAL_NULL )
    {
        ETG_TRACE_ERR(( "poGetInstance( ): Error: Not able to create a new CD control object in poGetInstance( vdmmgr_tclApp* poMain)" ));
    }

    return _poInstance;
}

/******************************************************************************/
/*
* FUNCTION:    poGetInstance( tVoid )
* 
* DESCRIPTION: returns the value of a member - variable 
*
* PARAMETER:   void
*
* RETURNVALUE: pointer to a vdmmgr_tclcdctrlif - object
*/
/******************************************************************************/
vdmmgr_tclcdctrlif* vdmmgr_tclcdctrlif::poGetInstance( tVoid )
{
    if( _poInstance == OSAL_NULL )
    {
        ETG_TRACE_ERR(( "poGetInstance( ): Error: Call of poGetInstance( tVoid ) and no pointer to instance available" ));
    }
    return _poInstance;
}


/*************************************************************************/
/*
* FUNCTION:    vdmmgr_tclcdctrlif( vdmmgr_tclApp* poVdmmgrMain )
* 
* DESCRIPTION: constructor
*
* PARAMETER:   poVdmmgrMain: 
*
* RETURNVALUE: none
*/ 
/*************************************************************************/
vdmmgr_tclcdctrlif::vdmmgr_tclcdctrlif( vdmmgr_tclApp* poVdmmgrMain )
    :
      _bThreadStarted( FALSE ),
      _hDevice       ( OSAL_ERROR ),
      _bTerminate    ( FALSE ),
      _tThreadID     ( OSAL_ERROR ),
      _hEvent        ( OSAL_C_INVALID_HANDLE ),
      _ps8DiagReadMem                  ( NULL ),
      _u32DiagMaxLBA                   ( VDMMGR_DIAG_CD_INIT_MAX_LBA ),
      _u16DiagMaxBlocksPerReadAttempt  ( VDMMGR_DIAG_CD_INIT ),
      _u32DiagDriveTestBitCmdField     ( VDMMGR_DIAG_CD_INIT ),
      _u32DiagDriveTestBitStatusField  ( VDMMGR_DIAG_CD_INIT )
{
    tBool bRetVal;

    // Check pointer to service handler
    bRetVal = ( poVdmmgrMain != OSAL_NULL );
    if( !bRetVal )
    {
        ETG_TRACE_ERR(( "vdmmgr_tclcdctrlif::vdmmgr_tclcdctrlif: Error: Call of constructor without pointer to main application" ));
    }
    _bEjectActive              = FALSE;                         // Init marker for active CD eject
    _bCdReinsertActive         = FALSE;                         // Init marker for active CD reinsert
    _poMain                    = poVdmmgrMain;                  // Get pointer to main application
    _u32DriveTestStartLBA      = VDMMGR_DIAG_CD_INIT;           // Init start LBA for drive test
    _u8DriveTestCntReadData    = VDMMGR_DIAG_CD_INIT_PASS_CNT;  // Init counter for first read
    _bFirstCdTypeValueArrived  = FALSE;                         // Init marker for indication that the first media type from OSAL driver arrived
    _e8ReqTrackType            = INIT;                          // Init track type for switching between Audio and Data on MixMode CD during TEF
    // Create Event and check result
    bRetVal = OSAL_s32EventCreate( VDMMGR_CDCTRLIF_EVENTNAME, &_hEvent) == OSAL_OK;
    if( !bRetVal )
    {
        ETG_TRACE_ERR(( "vdmmgr_tclcdctrlif::vdmmgr_tclcdctrlif: Error: Not able to create event VDMMGR_CDCTRLIF_EVENTNAME" ));
    }
    // Init diagnosis values to satisfy lint
    _DiagMethodResultDrvTest.u32SectorsRead            = VDMMGR_DIAG_CD_RESULT_INIT;
    _DiagMethodResultDrvTest.u16ReadAttempts           = VDMMGR_DIAG_CD_RESULT_INIT;
    _DiagMethodResultDrvTest.u32TotalReadTime          = VDMMGR_DIAG_CD_RESULT_INIT;
    _DiagMethodResultDrvTest.u16AverageReadtimeSector  = VDMMGR_DIAG_CD_RESULT_INIT;
    _DiagMethodResultDrvTest.u16MaxReadTimePerSector   = VDMMGR_DIAG_CD_RESULT_INIT;
    _DiagMethodResultDrvTest.u16CntErrors              = VDMMGR_DIAG_CD_RESULT_INIT;

    _au32DiagLBA_IMO[VDMMGR_DIAG_CD_INNER_RIM]         = VDMMGR_DIAG_CD_INIT;
    _au32DiagLBA_IMO[VDMMGR_DIAG_CD_MIDDLE]            = VDMMGR_DIAG_CD_INIT;
    _au32DiagLBA_IMO[VDMMGR_DIAG_CD_OUTER_RIM]         = VDMMGR_DIAG_CD_INIT;

    _au32DiagLBA_IMO_TEF_CD[VDMMGR_DIAG_CD_INNER_RIM]  = VDMMGR_DIAG_CD_INIT;
    _au32DiagLBA_IMO_TEF_CD[VDMMGR_DIAG_CD_MIDDLE]     = VDMMGR_DIAG_CD_INIT;
    _au32DiagLBA_IMO_TEF_CD[VDMMGR_DIAG_CD_OUTER_RIM]  = VDMMGR_DIAG_CD_INIT;

    _au8DiagOrderShortTest[VDMMGR_DIAG_CD_IMO_POS_0]   = VDMMGR_DIAG_CD_INIT;
    _au8DiagOrderShortTest[VDMMGR_DIAG_CD_IMO_POS_1]   = VDMMGR_DIAG_CD_INIT;
    _au8DiagOrderShortTest[VDMMGR_DIAG_CD_IMO_POS_2]   = VDMMGR_DIAG_CD_INIT;
    _au8DiagOrderShortTest[VDMMGR_DIAG_CD_IMO_POS_3]   = VDMMGR_DIAG_CD_INIT;
    _au8DiagOrderShortTest[VDMMGR_DIAG_CD_IMO_POS_4]   = VDMMGR_DIAG_CD_INIT;
    _au8DiagOrderShortTest[VDMMGR_DIAG_CD_IMO_POS_5]   = VDMMGR_DIAG_CD_INIT;
    _au8DiagOrderShortTest[VDMMGR_DIAG_CD_IMO_POS_6]   = VDMMGR_DIAG_CD_INIT;
    bEjectOngoing = false;
    // lock the mutex to stop the thread after starting
    pthread_mutex_init(&mLock, NULL);
    _bEjectTimerStart = FALSE;
}


/*************************************************************************/
/*
* FUNCTION:    vdmmgr_tclcdctrlif::~vdmmgr_tclcdctrlif( )
* 
* DESCRIPTION: destructor: deletes the thread
*
* PARAMETER:   void
*
* RETURNVALUE: none
*/
/*************************************************************************/
vdmmgr_tclcdctrlif::~vdmmgr_tclcdctrlif( )
{
    // Delete (stop and delete) the timer
    _oPollCdDriveTempTimer.vDelete( );         //lint !e1551 Warning 1551;Function may throw exception

    // Deactivate Thread
    vDeActivateSrc();//lint !e1551 Warning 1551;Function may throw exception
    tS32 s32Success;

    if( _tThreadID != OSAL_ERROR)
    {
        // Delete thread
        s32Success = OSAL_s32ThreadDelete( _tThreadID );
        if( s32Success == OSAL_ERROR )
        {
            ETG_TRACE_ERR(( "vdmmgr_tclcdctrlif::~vdmmgr_tclcdctrlif: Error: Unable to delete thread in destructor" ));
        }
    }

    if( _ps8DiagReadMem != NULL )
    {
        delete [] _ps8DiagReadMem;
        // Set pointer to deleted memory to NULL.
        _ps8DiagReadMem = NULL;
    }

    // Reset pointer to main app
    _poMain = OSAL_NULL;
}


/*************************************************************************/
/*
* FUNCTION: tVoid vdmmgr_tclcdctrlif::vStartThread( tVoid )
* 
* DESCRIPTION: Create and start CD control Thread. To be called only in
*              GetInstance().
*
* PARAMETER: void
*
* RETURNVALUE: void
*/
/*************************************************************************/
tVoid vdmmgr_tclcdctrlif::vStartThread( tVoid )
{
    tBool bRetVal;

    if( !_bThreadStarted )
    {
        // Create thread and check result
        bRetVal = bCreateThread();

        // Activate thread if create's are OK.
        if( bRetVal )
        {
            vActivateSrc();
        }
        else
        {
            ETG_TRACE_ERR(( "vStartThread( ): Error: Call of bCreateThread() returns with FALSE" ));
        }
        _bThreadStarted = TRUE;
    }
}


/*************************************************************************/
/*
* FUNCTION: tVoid vdmmgr_tclcdctrlif::vActivateSrc()
* 
* DESCRIPTION: * starts this CD control as a thread,
*              * registers it at PRM (Physical Resource Manager) if not done this before
*
* PARAMETER: void
*
* RETURNVALUE: void
*/
/*************************************************************************/
tVoid vdmmgr_tclcdctrlif::vActivateSrc()
{
    tS32 s32Success = OSAL_s32ThreadActivate( _tThreadID );
    if( s32Success == OSAL_ERROR )
    {
        ETG_TRACE_ERR(( "vActivateSrc( ): Error: Unable to activate thread" ));
    }

    (tVoid)bOpen( );
}

/*************************************************************************/
/*
* FUNCTION: tVoid vdmmgr_tclcdctrlif::vDeActivateSrc()
* 
* DESCRIPTION: unregister at the PRM
*
* PARAMETER: void
*
* RETURNVALUE: void
*/
/*************************************************************************/
tVoid vdmmgr_tclcdctrlif::vDeActivateSrc()
{
    // unregister OSAL notifications
    vUnRegNotification( );

    // Close OSAL device "dev/cdctrl1"
    (tVoid)bClose( );
}

/*************************************************************************/
/*
* FUNCTION: tBool vdmmgr_tclcdctrlif::bCreateThread()
* 
* DESCRIPTION: starts the member - function "dwThreadProc" as thread
*
* PARAMETER: void 
*
* RETURNVALUE: void
*/
/*************************************************************************/
tBool vdmmgr_tclcdctrlif::bCreateThread()
{
    OSAL_trThreadAttribute  rThAttr;
    tBool                   bRetVal        = TRUE;
    tU32                    u32ThreadPrio  = 0;     // Local variable to store data read from registry
    tU32                    u32StackSize   = 0;     // Local variable to store data read from registry

    ETG_TRACE_USR1(( "bCreateThread( ): Create CD thread" ));

    // Read the values from registry. In case of a failure, use default values
    // Read thread priority from registry
    if( scd_bGetAppConfigurationValue( CCA_C_U16_APP_MMGR,                        // CCA Application ID
                                       VDMMGR_CDCTRLIF_REGPATH_THREAD_DATA,       // Key name
                                       VDMMGR_CDCTRLIF_REGVALUE_THREAD_PRIO_NAME, // Key value
                                       &u32ThreadPrio )                   == FALSE )
    {
        // Set thread priority to default priority
        u32ThreadPrio = VDMMGR_CDCTRLIF_DEFAULT_PRIO;
        // Add trace on warning level here! The thread runs on default values!
        ETG_TRACE_SYS(( "bCreateThread( ): Warning: CD IF Thread runs with default priority" ));
    }
    // read thread stack size  from registry
    if( scd_bGetAppConfigurationValue( CCA_C_U16_APP_MMGR,                        // CCA Application ID
                                       VDMMGR_CDCTRLIF_REGPATH_THREAD_DATA,       // Key name
                                       VDMMGR_CDCTRLIF_REGVALUE_STACK_SIZE_NAME,  // Key value
                                       &u32StackSize  )                   == FALSE )
    {
        u32StackSize  = VDMMGR_CDCTRLIF_DEFAULT_STACKSIZE;
        // Indicate that the thread is running on default priority
        ETG_TRACE_SYS(( "bCreateThread( ): Warning: CD IF Thread runs with default stacksize" ));
    }

    // setup thread
    _bTerminate = FALSE;

    rThAttr.szName       = const_cast<tString>( VDMMGR_CDCTRLIF_THREADNAME );
    rThAttr.u32Priority  = u32ThreadPrio;
    rThAttr.s32StackSize = (tS32)u32StackSize;
    rThAttr.pfEntry      = (OSAL_tpfThreadEntry)dwThreadProc;
    rThAttr.pvArg        = (tPVoid)this;

    // create thread suspended
    _tThreadID = OSAL_ThreadCreate( &rThAttr );
    if( _tThreadID == OSAL_ERROR )
    {
        //can't create CD control thread
        ETG_TRACE_ERR(( "bCreateThread( ): Error: OSAL_ThreadCreate( &rThAttr ) fails to create thread" ));
        bRetVal = FALSE;
    }
    return bRetVal;
}



/*************************************************************************/
/*
* FUNCTION:    dwThreadProc (tPVoid pvArg)
* 
* DESCRIPTION: * this function is started as thread by "vdmmgr_tclcdctrlif::bCreateThread()"
*              * as argument the this - pointer is passed, 
*                so in fact the function vdmmgr_tclcdctrlif::vExecute() is started as thread
*
* PARAMETER:   void - pointer ( in this implementation: this - pointer of a vdmmgr_tclcdctrlif - object)
*
* RETURNVALUE:
*/
/*************************************************************************/
OSAL_tpfThreadEntry vdmmgr_tclcdctrlif::dwThreadProc( tPVoid pvArg )
{
    vdmmgr_tclcdctrlif*  poCdCtrlIf;

    poCdCtrlIf = (vdmmgr_tclcdctrlif*)pvArg;
    poCdCtrlIf->vExecute();
    return 0;
}


/*************************************************************************/
/*
* FUNCTION:    vExecute()
* 
* DESCRIPTION: * this is the working - thread of this MediaManager Source
*              * after some init stuff it listens to OSAL CD events and calls the event - handler
*
* PARAMETER:   void
*
* RETURNVALUE: void
*/
/*************************************************************************/
tVoid vdmmgr_tclcdctrlif::vExecute()
{
    OSAL_tEventMask   oMask;
    OSAL_tEventMask   oResultMask = 0;
    tS32              s32Success;

    ETG_TRACE_USR1(( "vExecute( ): Start waiting for events" ));

    oMask =  VDMMGR_EVENT_MASK_TOTAL_FAILURE
            | VDMMGR_EVENT_MASK_MEDIA_CHANGE      | VDMMGR_EVENT_MASK_MEDIA_STATE
            | VDMMGR_EVENT_MASK_CD_DEVICE_STATE   | VDMMGR_EVENT_MASK_CMD_EJECTCD
            | VDMMGR_EVENT_MASK_CMD_AUTOREINSERTCD
            | VDMMGR_EVENT_MASK_CMD_DIAG_START_CDDRIVE_TEST
            | VDMMGR_EVENT_MASK_CMD_DIAG_CONTINUE_CD_DRV_SHORT_TEST
            | VDMMGR_EVENT_MASK_CMD_DIAG_CONTINUE_CD_DRV_INT_TEST
            | VDMMGR_EVENT_MASK_CMD_DIAG_SET_REQUESTED_TRACKTYPE
            | VDMMGR_EVENT_MASK_CD_TEMP_TIMEOUT   | VDMMGR_EVENT_MASK_CD_GET_FIRST_TEMP;

    if( OSAL_s32EventOpen( VDMMGR_CDCTRLIF_EVENTNAME, &_hEvent ) == OSAL_OK )
    {
        for( ;/*ever*/; )
        {
            s32Success = OSAL_s32EventWait( _hEvent, oMask, OSAL_EN_EVENTMASK_OR, OSAL_C_TIMEOUT_FOREVER, &oResultMask );
            if( s32Success == OSAL_ERROR )
            {
                ETG_TRACE_ERR(( "vExecute( ): Error: OSAL_s32EventWait() returns with error" ));
            }
            //clear event
            if( OSAL_OK != OSAL_s32EventPost( _hEvent,~oResultMask,OSAL_EN_EVENTMASK_AND ) )
            {
                ETG_TRACE_ERR(( "vExecute( ): Error: Unable to clear event -> OSAL_s32EventPost( )" ));
            }
            vHandleEvent( oResultMask );     //handle event
        }//for ever
    }
    else
    {
        ETG_TRACE_ERR(( "vExecute( ): Error: OSAL_s32EventOpen returns with error" ));
        // Get error code if not successful.
        TRACE_OSAL_ERROR( VDMMGR_TR_CDCTRLIF );
    }
}


/*************************************************************************/
/*
* FUNCTION:    vHandleEvent (OSAL_tEventMask rEventMask)
* 
* DESCRIPTION: handler for MediaManager Source - events 
*
* PARAMETER:   eventMask
*
* RETURNVALUE: void
*/
/*************************************************************************/
tVoid vdmmgr_tclcdctrlif::vHandleEvent( OSAL_tEventMask rEventMask )
{
    ETG_TRACE_USR2(( "vHandleEvent -> Switch Event Mask: %d", ETG_ENUM(MMGR_CD_EVENT, rEventMask) ));

    if( rEventMask & VDMMGR_EVENT_MASK_TOTAL_FAILURE )       // Check for device failure
    {
        vHandleTotalFailure( );
    }
    if( rEventMask & VDMMGR_EVENT_MASK_MEDIA_CHANGE )        // Check for media type changes
    {
        vHandleMediaChange( );
    }
    if( rEventMask & VDMMGR_EVENT_MASK_MEDIA_STATE )         // Check for media state change
    {
        vHandleMediaState( );
    }
    if( rEventMask & VDMMGR_EVENT_MASK_CD_DEVICE_STATE )     // Check for CD device state
    {
        vHandleDeviceState( );
    }
    if( rEventMask & VDMMGR_EVENT_MASK_CMD_EJECTCD )         // Check for eject command.
    {
        vHandleCommandEject( );
    }
    if( rEventMask & VDMMGR_EVENT_MASK_CMD_AUTOREINSERTCD )  // Check for insert command.
    {
        vHandleCommandAutoReinsert( );
    }
    if( rEventMask & VDMMGR_EVENT_MASK_CD_TEMP_TIMEOUT )     // Check for event from timer callback function -> get temperature and send it
    {
        vHandleTimeOutToGetTemp( );
    }
    if( rEventMask & VDMMGR_EVENT_MASK_CD_GET_FIRST_TEMP )   // Check for event to get the first temperature and send it
    {
        vHandleFirstGetTemp( );
    }
    if( rEventMask & VDMMGR_EVENT_MASK_CMD_DIAG_START_CDDRIVE_TEST )        // Check for diagnosis start CD drive test command
    {
        vHandleDiagCommandStart( );
    }
    if( rEventMask & VDMMGR_EVENT_MASK_CMD_DIAG_CONTINUE_CD_DRV_SHORT_TEST )// Check for diagnosis start CD drive test command
    {
        vHandleDiagDriveShortTestContinue( );
    }
    if( rEventMask & VDMMGR_EVENT_MASK_CMD_DIAG_CONTINUE_CD_DRV_INT_TEST )  // Check for diagnosis start CD drive test command
    {
        vHandleDiagDriveIntensiveTestContinue( );
    }
    if( rEventMask & VDMMGR_EVENT_MASK_CMD_DIAG_SET_REQUESTED_TRACKTYPE )   // Check for diagnosis start CD drive test command
    {
        vHandleDiagSelectTrackType( );
    }
}


/* **************************************************FunctionHeaderBegin** *//**
 *
 *  Function: vHandleTotalFailure( tVoid ) const
 *
 *  Handle the notification data for total failure.
 *
 *  Handle the notification data for change failure.
 *  No return value.
 *
 * _u16TotalFailure
 *      Parameter to evaluate device state. Possible values are:
 *      - OSAL_C_U16_DEVICE_OK:   Device is OK
 *      - OSAL_C_U16_DEVICE_FAIL: Device isn't OK
 *
 * @date    2005-07-04
 *
 * @note
 *
 *
 *//* ***********************************************FunctionHeaderEnd******* */
tVoid vdmmgr_tclcdctrlif::vHandleTotalFailure( tVoid ) const
{
    ETG_TRACE_USR1(( "vHandleTotalFailure( ): New Value Total Failure CD: %d", ETG_ENUM(OSAL_DEVICE_CODE, _u16TotalFailure) ));

    // Evaluate total failure value
    if( _poMain && _poMain->_vdmmgr_poCCAClienthandlerdiaglog && _poMain->_vdmmgr_poCCAService )
    {
        switch( _u16TotalFailure )
        {
        case OSAL_C_U16_DEVICE_OK:          // Device is OK
            // Set ITC in VD DiagLog
#ifdef USE_REPLACE_E8_TESTRESULT
            _poMain->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_DEVICE_ERROR, (tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED );
#else
            _poMain->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_DEVICE_ERROR, midw_fi_tcl_TestResult::FI_EN_PASSED );
#endif

            _poMain->_vdmmgr_poCCAService->vSendCdDriveErrorInformation( mplay_fi_tcl_e16_DriveErrorValue::FI_EN_DEFECT_CMD_TIMEOUT_OK );
            break;
        case OSAL_C_U16_DEVICE_FAIL:        // Device isn't OK
#ifdef USE_REPLACE_E8_TESTRESULT
            _poMain->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_DEVICE_ERROR, (tU8)midw_fi_tcl_e8_TestResult::FI_EN_FAILED );
#else
            _poMain->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_DEVICE_ERROR, midw_fi_tcl_TestResult::FI_EN_FAILED );
#endif
            _poMain->_vdmmgr_poCCAService->vSendCdDriveErrorInformation( mplay_fi_tcl_e16_DriveErrorValue::FI_EN_DEFECT_CMD_TIMEOUT );
            break;
        default:                            // There shouldn't be a default
            ETG_TRACE_ERR(( "vHandleTotalFailure( ): vUnknown value for TotalFailure %d", _u16TotalFailure ));
            break;
        }
    }
}


/* **************************************************FunctionHeaderBegin** *//**
 *
 *  Function: vHandleMediaChange( tVoid)
 *
 *  Evaluate the notification data for media change.
 *
 *  Evaluate the notification data for media change and inform the service
 *  handler
 *  No return value.
 *
 *  _u16MediaChange
 *      Parameter to evaluate media change. Possible values are:
 *      - OSAL_C_U16_MEDIA_EJECTED:   CD is ejected
 *      - OSAL_C_U16_INCORRECT_MEDIA: CD is incorrect
 *      - OSAL_C_U16_DATA_MEDIA:      CD is a CD ROM
 *      - OSAL_C_U16_AUDIO_MEDIA      CD is a Audio CD
 *      - OSAL_C_U16_UNKNOWN_MEDIA:   CD is inserted but not recognized
 *
 * @date    2005-07-04
 *
 * @note
 *
 *
 *//* ***********************************************FunctionHeaderEnd******* */
tVoid vdmmgr_tclcdctrlif::vHandleMediaChange( tVoid)
{
    T_e8_CdType                      e8CdType;

    ETG_TRACE_USR1(( "vHandleMediaChange( ): New Value Media Change: %d", ETG_ENUM(OSAL_MEDIA_TYPE_CODE, _u16MediaChange) ));

    if( _poMain && _poMain->_vdmmgr_poCCAClienthandlerdiaglog && _poMain->_vdmmgr_poCCAService )
    {


        // fetch the unique device ID
        if (    (OSAL_C_U16_AUDIO_MEDIA == _u16MediaChange )
                || (OSAL_C_U16_DATA_MEDIA == _u16MediaChange )) {
            tS32  s32ReturnValue  = OSAL_ERROR;
            OSAL_trMediaInfo mediaInfo;

            s32ReturnValue = OSAL_s32IOControl( _hDevice,OSAL_C_S32_IOCTRL_CDCTRL_GETMEDIAINFO, (intptr_t)&mediaInfo);
            if(s32ReturnValue == OSAL_OK) {
                _poMain->_vdmmgr_poCCAService->vSetUniqueID(mediaInfo.szcMediaID);
            }
        }
        else if (OSAL_C_U16_MEDIA_EJECTED == _u16MediaChange) {
            _poMain->_vdmmgr_poCCAService->vSetUniqueID(NULL);
        }



        /* Evaluate the new media type. */
        switch( _u16MediaChange )
        {
        case OSAL_C_U16_MEDIA_EJECTED:          // Media ejected
            // Check if the insert clamps have already reported 'MediaInSlot'
            e8CdType = _poMain->_vdmmgr_poCCAService->enGetActCdType( );
            if(   ( e8CdType != MMGR_IN_SLOT )
                  && ( e8CdType != MMGR_NO_MEDIA )
                  && ( e8CdType != MMGR_INSERTION )
                  )
            {
                // There is no media in drive but there was no 'InSlot' status before
                // -> set InSlot before NoMedia to avoid PopUp 'Ejecting CD' for ever
                _poMain->_vdmmgr_poCCAService->vNewCdType( MMGR_IN_SLOT );
            }
            _poMain->_vdmmgr_poCCAService->vNewCdType( MMGR_NO_MEDIA );
            // Set ITC for LOAD error to 'no error'
#ifdef USE_REPLACE_E8_TESTRESULT
            _poMain->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_LOAD_ERROR, (tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED );
#else
            _poMain->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_LOAD_ERROR, midw_fi_tcl_TestResult::FI_EN_PASSED );
#endif
            // Stop 'ensure CD NO_MEDIA' timer for MEDIA_EJECTED
            (tVoid)_poMain->_vdmmgr_poCCAService->_oEnsureCdNoMediaTimer.bStop();
            bEjectSet(FALSE);
            break;
        case OSAL_C_U16_INCORRECT_MEDIA:        // Media is incorrect, e.g. upside down
            _poMain->_vdmmgr_poCCAService->vNewCdType( MMGR_INCORRECT );
            break;
        case OSAL_C_U16_DATA_MEDIA:             // CD ROM inserted
            // Set CD type CD ROM
            _poMain->_vdmmgr_poCCAService->vNewCdType( MMGR_DATA );
            // Set ITC for LOAD error to 'no error'
#ifdef USE_REPLACE_E8_TESTRESULT
            _poMain->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_LOAD_ERROR, (tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED );
#else
            _poMain->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_LOAD_ERROR, midw_fi_tcl_TestResult::FI_EN_PASSED );
#endif
            break;
        case OSAL_C_U16_AUDIO_MEDIA:            // Audio CD inserted
            // Check if OSAL already notified the media state as MediaReady
            if( _u16MediaState == OSAL_C_U16_MEDIA_NOT_READY )
            {
                ETG_TRACE_USR4(( "vHandleMediaChange( ): New Type is AUDIO but the state is still NOT_READY -> wait for 200ms if becomes READY." ));
                OSAL_s32ThreadWait( 200 );
            }
            if( _u16MediaState == OSAL_C_U16_MEDIA_READY )
            {
                _poMain->_vdmmgr_poCCAService->vNewCdType ( MMGR_AUDIO,        FALSE );  // Set CD type to 'Audio'.
                _poMain->_vdmmgr_poCCAService->vNewCdState( MMGR_MEDIA_READY,  TRUE );   // Set CD state to 'media ready'.
            }
            else  // media isn't ready (maybe due to media type from LMM)) -> only set the new media type
            {
                _poMain->_vdmmgr_poCCAService->vNewCdType( MMGR_AUDIO );
            }
            // Set ITC for LOAD INSERT error to 'no error'
#ifdef USE_REPLACE_E8_TESTRESULT
            _poMain->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_LOAD_ERROR, (tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED );
#else
            _poMain->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_LOAD_ERROR, midw_fi_tcl_TestResult::FI_EN_PASSED );
#endif
            break;
        case OSAL_C_U16_UNKNOWN_MEDIA:          // CD inside but not recognized
            // Set ITC for LOAD error to 'no error'
#ifdef USE_REPLACE_E8_TESTRESULT
            _poMain->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_LOAD_ERROR, (tU8)midw_fi_tcl_e8_TestResult::FI_EN_PASSED );
#else
            _poMain->_vdmmgr_poCCAClienthandlerdiaglog->vSetITC( ITC_OPTICALDISC_LOAD_ERROR, midw_fi_tcl_TestResult::FI_EN_PASSED );
#endif
            break;
        default:                                // There shouldn't be a default
            ETG_TRACE_ERR(( "vHandleMediaChange( ): vUnknown value for MediaChange %d", _u16MediaChange ));
            break;
        }
    }

    // Check if a 'real' CD type has arrived
    if( _u16MediaChange != OSAL_C_U16_UNKNOWN_MEDIA )
    {
        // Set marker that the first OSAL CD type has arrived
        vSetFirstCdTypeValueArrived( TRUE );
    }

}

/* **************************************************FunctionHeaderBegin** *//**
 *
 *  Function: vHandleMediaState  ( tVoid) const
 *
 *  Evaluate the notification data (_u16MediaState) for changes in media state.
 *
 *  Evaluate the notification data (_u16MediaState) for changes in media state
 *  and get the media information if the media is ready else the media isn't
 *  ready and we set the depending server variables.
 *  Eject:  Changes in media state will be send before changes is media change.
 *  Insert: Changes in media state will be send after changes is media change.
 *  No return value.
 *
 *  _u16MediaState
 *      Parameter to evaluate media change. Possible values are:
 *      - OSAL_C_U16_MEDIA_READY:     CD is ready -> Get media info
 *      - OSAL_C_U16_MEDIA_NOT_READY: CD isn't ready -> set server variables
 *
 * @date    2003-01-21
 *
 * @note
 *
 *
 *//* ***********************************************FunctionHeaderEnd******* */
tVoid vdmmgr_tclcdctrlif::vHandleMediaState( tVoid) 
{
    ETG_TRACE_USR1(( "vHandleMediaState( ): New Value Media State: %d", ETG_ENUM(OSAL_MEDIA_STATE_CODE, _u16MediaState) ));

    if( _poMain && _poMain->_vdmmgr_poCCAService )
    {
        // evaluate media state
        switch( _u16MediaState )
        {
        case OSAL_C_U16_MEDIA_READY:
            _poMain->_vdmmgr_poCCAService->vNewCdState( MMGR_MEDIA_READY );
            if(_poMain && _poMain->_vdmmgr_poCCAService)
            {
                tBool bEjectStatus = false;
                bEjectGet(bEjectStatus);
                if(bEjectStatus)
                {
                    ETG_TRACE_USR1(("CD is ejected from slot , and inserted again before 5 sec"));
                    // Stop 'ensure CD NO_MEDIA' timer for MEDIA_EJECTED
                    (tVoid)_poMain->_vdmmgr_poCCAService->_oEnsureCdNoMediaTimer.bStop();
                    bEjectSet(FALSE);
                }
            }
            break;
        case OSAL_C_U16_MEDIA_NOT_READY:          // Media isn't ready
            _poMain->_vdmmgr_poCCAService->vNewCdState( MMGR_MEDIA_NOT_READY );
            break;
        default:
            ETG_TRACE_ERR(( "vHandleMediaState( ): vUnknown value for MediaState %d", _u16MediaState ));
            break;
        }
    }
}


tVoid vdmmgr_tclcdctrlif::vHandleDeviceState( tVoid ) const
{
    ETG_TRACE_USR1(( "vHandleDeviceState( ): CD Device State %d", ETG_ENUM(OSAL_NOTIFI_DEVICE_STATE, _u16DeviceState) ));

    if( _poMain && _poMain->_vdmmgr_poCCAService )
    {
        // evaluate device state
        switch( _u16DeviceState )
        {
        case OSAL_C_U16_DEVICE_READY:             // Device is ready
            _poMain->_vdmmgr_poCCAService->vNewCdDeviceState( MMGR_DEVICE_READY );
            break;
        case OSAL_C_U16_DEVICE_NOT_READY:         // Device isn't ready
            _poMain->_vdmmgr_poCCAService->vNewCdDeviceState( MMGR_DEVICE_NOT_READY );
            break;
        default:
            break;
        }
    }
}


tVoid vdmmgr_tclcdctrlif::vHandleCommandEject( tVoid )
{
    // Call eject function
    ETG_TRACE_USR1(( "vHandleCommandEject( ): Receive event to eject CD -> call bEject()" ));
    (tVoid)bEject( );
}


tVoid vdmmgr_tclcdctrlif::vHandleCommandAutoReinsert( ) const
{
    ETG_TRACE_USR1(( "vHandleCommandAutoReinsert( ): Receive event to reinsert automatically" ));

    if( _poMain && _poMain->_vdmmgr_poCCAService )
    {
        // Reinsert CD automatically after 10s
        _poMain->_vdmmgr_poCCAService->vCloseDoorAutoInsert( );
    }
}


tVoid vdmmgr_tclcdctrlif::vHandleTimeOutToGetTemp( )
{
    tS16                       s16CDTemp;
    T_e8DriveTempSensorStatus  DriveTempSensorStatus;

    // Get temperature
    s16CDTemp = s16GetTemperature( &DriveTempSensorStatus );
    // Give actual drive temperature to clients.
    if( _poMain && _poMain->_vdmmgr_poCCAService )
    {
        _poMain->_vdmmgr_poCCAService->vSendStatusCDTemp( s16CDTemp, DriveTempSensorStatus );
    }
}


tVoid vdmmgr_tclcdctrlif::vHandleFirstGetTemp( )
{
    tS16                       s16CDTemp;
    T_e8DriveTempSensorStatus  DriveTempSensorStatus;

    // Get temperature
    s16CDTemp = s16GetTemperature( &DriveTempSensorStatus );
    // Give actual drive temperature to clients.
    if( _poMain && _poMain->_vdmmgr_poCCAService )
    {
        // send the requested data to the client
        _poMain->_vdmmgr_poCCAService->vSendAnswerCDTemp( NULL, s16CDTemp, DriveTempSensorStatus, &(_poMain->_vdmmgr_poCCAService->_MessageDataForGetFirstCdTemp) );
    }
    // Start timeout in CD ctrl
    vStartTemperatureTimeout();
}


/*************************************************************************/
/*
* FUNCTION:    vPRMCallbackFunction (tU32* pu32Data)
* 
* DESCRIPTION: handles messages from the PRM 
*              * analyse type of data 
*              * set corresponding member - variable
*              * set corresponding event 
*  
* PARAMETER: message 
*
* RETURNVALUE: void
*/
/*************************************************************************/
tVoid vdmmgr_tclcdctrlif::vCallbackFct( tPU32 pu32Data )
{
    tU16              u16NotificationType; // For type of notification
    tU16              u16NotificationData; // For data of a notification
    tS32              s32Success;
    OSAL_tEventHandle hCbkEvent;
    OSAL_tEventMask   oMask = 0;

    // Get type and data from OSAL notification
    u16NotificationType = (tU16)(*pu32Data >> 16); //Extract first 16 Bit (31 - 16) for Notification Type
    u16NotificationData = (tU16) *pu32Data;

    ET_TRACE_INFO_BIN( VDMMGR_TR_CDCTRLIF, ET_EN_T16 _ VDMMGR_TR_CDCTRLIF_2D _ ET_EN_T16 _ u16NotificationType _ ET_EN_T16 _ u16NotificationData _ ET_EN_DONE );

    switch( u16NotificationType )
    {
    //the lower 16 bytes contain the value
    case OSAL_C_U16_NOTI_TOTAL_FAILURE:
        _u16TotalFailure  = u16NotificationData;
        oMask             = VDMMGR_EVENT_MASK_TOTAL_FAILURE;
        break;
    case OSAL_C_U16_NOTI_MEDIA_CHANGE:
        // Filter the media change due to a CD driver bug.
        // TNCD could send DATA -> INCORRECT (wrong) -> DATA (wrong), (without changing MediaState to NotReady)
        if(  ( u16NotificationData != OSAL_C_U16_INCORRECT_MEDIA )     // Only if new value is not INCORRECT
             ||(   ( _u16MediaChange != OSAL_C_U16_DATA_MEDIA )          // or old value isn't DATA
                   && ( _u16MediaChange != OSAL_C_U16_AUDIO_MEDIA )         //    and isn't AUDIO
                   )
             )
        {
            _u16MediaChange   = u16NotificationData;
            oMask             = VDMMGR_EVENT_MASK_MEDIA_CHANGE;
        }
        break;
    case OSAL_C_U16_NOTI_MEDIA_STATE:
        _u16MediaState    = u16NotificationData;
        oMask             = VDMMGR_EVENT_MASK_MEDIA_STATE;
        break;
    case OSAL_C_U16_NOTI_DEFECT:
        if( _poInstance && _poInstance->_poMain && _poInstance->_poMain ->_vdmmgr_poErrorIf )
        {
            _poInstance->_poMain->_vdmmgr_poErrorIf->vSendDefectEvent( VDMMGR_ERRORIF_EVENT_MASK_DEFECT_CD, u16NotificationData );
        }
        break;
    case OSAL_C_U16_NOTI_DEVICE_READY:
        _u16DeviceState   = u16NotificationData;
        oMask             = VDMMGR_EVENT_MASK_CD_DEVICE_STATE;
        // Set marker used for low voltage during eject
        if( _u16DeviceState == OSAL_C_U16_DEVICE_NOT_READY )
        {
            _bLowVoltageDuringEject = TRUE;
        }
        else
        {
            _bLowVoltageDuringEject = FALSE;
            _u8CDEjectLowVoltageRetryCnt = 0;
        }
        break;
    default:
        ETG_TRACE_ERR(( "vCallbackFct( ): Error: Unknown/unhandled Notification type in callback: %d", ETG_ENUM(OSAL_NOTI_TYPE, u16NotificationType) ));
        break;
    }

    // Open, send and close event
    if(    ( oMask != 0)
           && ( OSAL_s32EventOpen( VDMMGR_CDCTRLIF_EVENTNAME, &hCbkEvent ) == OSAL_OK )
           )
    {
        s32Success = OSAL_s32EventPost( hCbkEvent, oMask, OSAL_EN_EVENTMASK_OR );
        if( s32Success == OSAL_ERROR )
        {
            ETG_TRACE_ERR(( "vCallbackFct( ): Error: OSAL_s32EventPost( ) returns with error" ));
        }

        s32Success = OSAL_s32EventClose( hCbkEvent );
        if(s32Success == OSAL_ERROR)
        {
            ETG_TRACE_ERR(( "vCallbackFct( ): Error: OSAL_s32EventClose() returns with error" ));
        }
    }
    else if(    (oMask != 0)
                && (u16NotificationType != OSAL_C_U16_NOTI_DEFECT)
                )                   // -> OSAL_s32EventOpen(...) returned with error
    {
        ETG_TRACE_ERR(( "vCallbackFct( ): Error: OSAL_s32EventOpen( ) return with error" ));
    }
    ETG_TRACE_USR1(( "vCallbackFct( ): End: New callback: info from /dev/cdtrl1" ));
}//lint !e818 Info 818 -> compiler doesn't like a const pointer



/* ************************************************************************** */
/* Local helper functions for OSAL functions call                             */
/* ************************************************************************** */



/* **************************************************FunctionHeaderBegin** *//**
 *
 *  Function: bOpen( tVOID )
 *
 *  Open the OSAL device CD Control.
 *
 *  Open the OSAL device CD Control ("/dev/cdctrl").
 *  No parameters.
 *
 * @return
 *      bRet: TRUE: successfully opened; FALSE: something went wrong
 *
 * @date    2005-07-01
 *
 * @note
 *
 *
 *//* ***********************************************FunctionHeaderEnd******* */
tBool vdmmgr_tclcdctrlif::bOpen( tVoid )
{
    tBool    bRet = TRUE;

    // Open only if device wasn't already opened
    if( _hDevice == OSAL_ERROR )
    {
        ETG_TRACE_USR1(( "bOpen( ): Open OSAL device /dev/cdctrl1" ));
        // Opening OSAL-device cdctrl; get OSAL IO handle
        _hDevice = OSAL_IOOpen( OSAL_C_STRING_RES_CDCTRL, OSAL_EN_READONLY );

        if( _hDevice  == OSAL_ERROR )
        {
            ETG_TRACE_ERR(( "bOpen( ): Error: Open OSAL device /dev/cdctrl1 returns with error" ));
            // Get error code if not successful.
            TRACE_OSAL_ERROR( VDMMGR_TR_CDCTRLIF );
            bRet = FALSE;
        }
        else
        {
            (tVoid)bRegNotification();
        }
    }
    return bRet;
}

/* **************************************************FunctionHeaderBegin** *//**
 *
 *  Function: bClose( tVoid )
 *
 *  Close the OSAL device CD Control
 *
 *  Close the OSAL device CD Control ("/dev/cdctrl").
 *  No parameters.
 *
 * @return
 *      bRet: TRUE: successfully closed; FALSE: something went wrong
 *
 * * @date    2005-07-01
 *
 * @note
 *
 *
 *//* ***********************************************FunctionHeaderEnd******* */
tBool vdmmgr_tclcdctrlif::bClose( tVoid )
{
    tS32  s32RetValue;
    tBool bRet = TRUE;

    ETG_TRACE_USR1(( "bClose( ): Close OSAL device /dev/cdctrl1" ));
    // Close only if there is an IO handle
    if( _hDevice != OSAL_ERROR )
    {
        // Close OSAL-device /dev/cdctrl1
        s32RetValue = OSAL_s32IOClose( _hDevice );

        if( s32RetValue  != OSAL_OK )
        {
            ETG_TRACE_ERR(( "bClose( ): Error: Close OSAL device /dev/cdctrl1 returns with error" ));
            bRet = FALSE;
        }
        // Set OSAL handle to an invalid value if device could be closed
        _hDevice = OSAL_ERROR;
    }
    return bRet;
}


/* **************************************************FunctionHeaderBegin** *//**
 *
 *  Function: bRegNotification( tVoid )
 *
 *  Register Callback function in OSAL.
 *
 *  Register the Callback function and the types of interest for notification.
 *  Also we get back an actual CD device status and evaluate it. If a media
 *  is already inside, try to get the media information. Set the corresponding
 *  server variables.
 *  No parameters.
 *
 * @date    2003-01-08
 *
 * @note
 *
 *
 *//* ***********************************************FunctionHeaderEnd******* */
tBool vdmmgr_tclcdctrlif::bRegNotification( tVoid )
{
    tS32                s32ReturnValue  = OSAL_ERROR;  // Return value from OSAL function
    OSAL_trNotifyData   rNotifyData;                   // OSAL struct for notify data
    tBool               bRet            = TRUE;

    if( _hDevice != OSAL_ERROR )
    {
        // Fill the NotifyData struct
        rNotifyData.u16AppID            = OSAL_C_U16_CDMANAGER_APPID;
        rNotifyData.ResourceName        = OSAL_C_STRING_RES_CDCTRL;
        rNotifyData.u16NotificationType =   OSAL_C_U16_NOTI_MEDIA_CHANGE | OSAL_C_U16_NOTI_TOTAL_FAILURE
                | OSAL_C_U16_NOTI_MEDIA_STATE  | OSAL_C_U16_NOTI_DEFECT
                | OSAL_C_U16_NOTI_DEVICE_READY;
        rNotifyData.pCallback           = /*(OSALCALLBACKFUNC)*/ vCallbackFct;

        ETG_TRACE_USR1(( "bRegNotification( ): Register notification for CD control" ));
        // Register the callback function for notification
        s32ReturnValue = OSAL_s32IOControl( _hDevice, OSAL_C_S32_IOCTRL_REG_NOTIFICATION, (intptr_t)&rNotifyData );
    }
    else  // if( _hDevice != OSAL_ERROR )
    {
        ETG_TRACE_ERR(( "bRegNotification( ): Error: No valid handle for /dev/cdctrl" ));
    }

    if( s32ReturnValue != OSAL_OK )
    {
        // Should never happen.
        ETG_TRACE_ERR(( "bRegNotification( ): Error: Failed to register notification for CD control" ));
        bRet = FALSE;
    }

    return bRet;
}


/* **************************************************FunctionHeaderBegin** *//**
 *
 *  Function: vUnRegNotification( tVoid )
 *
 *  Unregister Callback function in OSAL.
 *
 *  No parameters.
 *
 * @date    2008-12-11
 *
 * @note
 *
 *
 *//* ***********************************************FunctionHeaderEnd******* */
tVoid vdmmgr_tclcdctrlif::vUnRegNotification( tVoid )
{
    tS32                    s32ReturnValue = OSAL_ERROR;  // Return value from OSAL function
    OSAL_trRelNotifyData    rRelNotifyData;               // OSAL struct for notify data

    if( _hDevice != OSAL_ERROR )
    {
        // Fill the RelNotifyData struct
        rRelNotifyData.u16AppID             = OSAL_C_U16_CDMANAGER_APPID;
        rRelNotifyData.ResourceName         = OSAL_C_STRING_RES_CDCTRL;
        rRelNotifyData.u16NotificationType  =   OSAL_C_U16_NOTI_MEDIA_CHANGE | OSAL_C_U16_NOTI_TOTAL_FAILURE
                | OSAL_C_U16_NOTI_MEDIA_STATE  | OSAL_C_U16_NOTI_DEFECT
                | OSAL_C_U16_NOTI_DEVICE_READY;

        ETG_TRACE_USR1(( "vUnRegNotification( ): UnRegister notification for CD control" ));
        // UnRegister the callback function implicit by unregister all notifications
        s32ReturnValue = OSAL_s32IOControl( _hDevice, OSAL_C_S32_IOCTRL_UNREG_NOTIFICATION, (intptr_t)&rRelNotifyData );
    }
    else  // if( _hDevice != OSAL_ERROR )
    {
        ETG_TRACE_ERR(( "vUnRegNotification( ): Error: No valid handle for /dev/cdctrl" ));
    }

    if( s32ReturnValue != OSAL_OK )
    {
        // Should never happen.
        TRACE_OSAL_ERROR( VDMMGR_TR_CDCTRLIF );
    }
}


/* **************************************************FunctionHeaderBegin** *//**
 *
 *  Function:  bEject( tVoid )
 *
 *  Call the OSAL Eject Media function.
 *
 *  Call the OSAL Eject Media function and trigger the loader check during
 *  event OS_EV_TIMEOUT.
 *  No parameters.
 *
 * @return
 *      bRet: TRUE: successfully ejected; FALSE: something went wrong
 *
 * @date    2005-07-01
 *
 * @note
 *
 *
 *//* ***********************************************FunctionHeaderEnd******* */
tBool vdmmgr_tclcdctrlif::bEject( tVoid )
{
    tS32                             s32ReturnValue        = OSAL_ERROR; // Return value from OSAL function
    tS32                             s32Dummy;                           // Dummy variable for OSAL function
    tBool                            bRet                  = TRUE;
    tBool                            bRetryDueToLowVoltage = FALSE;
    T_e8_DiagLoadStatus              e8LoaderState;
    T_DiagMethodControlResultEject   DiagResultEject;

    _u8CDEjectLowVoltageRetryCnt = 0;
    s32Dummy = 0;                          // initialize to avoid compiler warning 'used without'.

    ETG_TRACE_USR1(( "bEject( ): Start" ));

    if( _poMain && _poMain->_vdmmgr_poCCAService && _poMain->_vdmmgr_poCCAClienthandlerspm )
    {
        if( _hDevice != OSAL_ERROR )
        {
            // set flag 'eject active'. This is necessary because it's possible to
            // press eject (no CD inside; lld_bpcd waits/needs approx. 5s). Press
            // a second time and insert a CD. Eject function comes back from OSAL,
            // get a new 'Eject trigger' and the inserted CD will be ejected.
            _bEjectActive = TRUE;

            do
            {
                // Reset retry marker
                bRetryDueToLowVoltage   = FALSE;
                // Eject CD
                s32ReturnValue = OSAL_s32IOControl( _hDevice, OSAL_C_S32_IOCTRL_CDCTRL_EJECTMEDIA, s32Dummy );
                ETG_TRACE_ERR(( "bEject( ): CDEject: _bLowVoltageDuringEject: %d, _u8CDEjectLowVoltageRetryCnt=%d", ETG_ENUM(BOOL, _bLowVoltageDuringEject), _u8CDEjectLowVoltageRetryCnt ));
                if(   ( s32ReturnValue != OSAL_OK )
                      && ( _bLowVoltageDuringEject )   // Check for low voltage -> device not is/was ready
                      )                                 // Set in callback function (OSAL runtime)
                {
                    _u8CDEjectLowVoltageRetryCnt++;  // Increment Retry counter
                    bRetryDueToLowVoltage = TRUE;    // Set marker to retry CD eject
                    // Wait some second before next try
                    if( OSAL_s32ThreadWait( VDMMGR_CDCDTRLIF_WAITTIME_EJECT_RETRY ) == OSAL_ERROR )
                    {
                        TRACE_OSAL_ERROR( VDMMGR_TR_CDCTRLIF );
                    }
                }
            }
            while(   ( bRetryDueToLowVoltage )                                               // Low voltage happened
                     && ( _u8CDEjectLowVoltageRetryCnt <= VDMMGR_CDCDTRLIF_MAXRETRY_EJECT )     // Max. retries not exceeded
                     && ( _poMain->_vdmmgr_poCCAService->enGetActCdType() != MMGR_INSERTION )   // No CD inserted in meantime
                     );
            // Helper flag to detect active eject process for automatic reinsert. Reseted to
            // FALSE if the CD was taken out of the slot.
            // But first check if Media isn't already in slot (or no media inside)
            // -> so media is inside or eject in progress
            e8LoaderState = e8GetLoaderState( );
            if(   ( e8LoaderState == MMGR_MEDIA_INSIDE )
                  || ( e8LoaderState == MMGR_EJECT_IN_PROGRESS )
                  || (     e8LoaderState == MMGR_MEDIA_IN_SLOT                            // If the CD is fixed in the slot
                           && ! _poMain->_vdmmgr_poCCAClienthandlerspm->bInsertClampsActive()  // short be before complete insert
                           )
                  )
            {
                _poMain->_vdmmgr_poCCAService->_bEjectStartet = TRUE;
            }

            _bEjectActive = FALSE;                 // reset 'eject active'
        }
        else  // if( _hDevice != OSAL_ERROR )
        {
            ETG_TRACE_ERR(( "bEject( ): Error: No valid handle for /dev/cdctrl" ));
        }
        if( s32ReturnValue != OSAL_OK )
        {
            // Get error code if not successful.
            TRACE_OSAL_ERROR( VDMMGR_TR_CDCTRLIF );
            ETG_TRACE_ERR(( "bEject( ): Call of CD eject function in OSAL IO devive cdctrl1 returns with error" ));
            bRet = FALSE;
        }
        else
        {
            ETG_TRACE_USR1(( "bEject( ): Call of CD eject function succesfull. Set flag Eject-started" ));
            // Start 5s timer to ensure CD NO_MEDIA due to missing insert clamps info from MASCA CD drive
            (tVoid)_poMain->_vdmmgr_poCCAService->_oEnsureCdNoMediaTimer.bStart( );
            bEjectSet(TRUE);
        }

        // CD eject started by diagnosis?
        if( _poMain->_vdmmgr_poCCAService->_MessageDataForMethodResultDiagEject.u32InternalData & VDMMGR_DIAG_EJECT_CD_STARTED )
        {
            DiagResultEject.e8Duration = MMGR_SINGLE_RUN;
            DiagResultEject.ResultEject.e16Result = ( s32ReturnValue == OSAL_OK ) ?  MMGR_EJECT_OK : MMGR_EJECT_FAILED;
            _poMain->_vdmmgr_poCCAService->vSendAnswerDiagEject( &(_poMain->_vdmmgr_poCCAService->_MessageDataForMethodResultDiagEject),
                                                                 &DiagResultEject );
            // Clear flag
            _poMain->_vdmmgr_poCCAService->_MessageDataForMethodResultDiagEject.u32InternalData &= ~VDMMGR_DIAG_EJECT_CD_STARTED;
        }
    }

    // Clear diagnosis INIT DONE flag
    _u32DiagDriveTestBitStatusField  &= ~VDMMGR_DIAG_CD_STATUS_INIT_DONE;
    // Delete
    if( _ps8DiagReadMem != NULL )
    {
        delete [] _ps8DiagReadMem;
        // Set pointer to deleted memory to NULL.
        _ps8DiagReadMem = NULL;
    }
    return bRet;
}
void vdmmgr_tclcdctrlif::bEjectSet(tBool status)
{
    pthread_mutex_lock(&mLock);

  
   _bEjectTimerStart = status;

    pthread_mutex_unlock(&mLock);    
}

void vdmmgr_tclcdctrlif::bEjectGet( tBool &status)
{
    pthread_mutex_lock(&mLock);

    status = _bEjectTimerStart;

    pthread_mutex_unlock(&mLock);   
}

/* **************************************************FunctionHeaderBegin** *//**
 *
 *  Function:  bSetMotorOn( tVoid )
 *
 *  Call the OSAL Set Motor On function.
 *
 *  Call the OSAL Set Motor On function and inform diagnosis.
 *  No parameters.
 *
 * @return
 *      bRet: TRUE: OSAL fct. returns successfully; FALSE: something went wrong
 *
 * @date    2005-07-01
 *
 * @note
 *
 *
 *//* ***********************************************FunctionHeaderEnd******* */
tBool vdmmgr_tclcdctrlif::bSetMotorOn( tVoid )
{
    ETG_TRACE_USR1(( "bSetMotorOn( ): Start" ));

    tBool    bRet;
    tS8      s8DiagReadMem[VDMMGR_DIAG_CD_LBA_SIZE];

    bRet = bReadRawDataLBA( 0, 1, s8DiagReadMem);
    return bRet;
}


/* **************************************************FunctionHeaderBegin** *//**
 *
 *  Function: tBool CDMGRS_iCloseDoor( tVOID )
 *
 *  Call the OSAL Close Door function.
 *
 *  Call the OSAL Close Door function to insert the CD.
 *  No parameters.
 *
 * @return
 *      bRet: TRUE: Insert was OK; FALSE: something went wrong
 *
 * @date    2003-01-21
 *
 * @note
 *
 *
 *//* ***********************************************FunctionHeaderEnd******* */
tBool vdmmgr_tclcdctrlif::bCloseDoor( tVoid )
{
    tS32    s32ReturnValue = OSAL_ERROR;   // Return value from OSAL function
    tS32    s32Dummy;                      // Dummy variable for OSAL function
    tBool   bRet = TRUE;

    ETG_TRACE_USR1(( "bCloseDoor( ): Start: Automatic reinsert of CD media" ));

    if( _hDevice != OSAL_ERROR )
    {
        vSetCdReinsertActive( TRUE );
        // Drawing in the CD and close door
        s32Dummy = 0;      // initialize to avoid compiler warning 'used without...'
        s32ReturnValue = OSAL_s32IOControl( _hDevice, OSAL_C_S32_IOCTRL_CDCTRL_CLOSEDOOR, s32Dummy );
        vSetCdReinsertActive( FALSE );
    }
    else  // if( _hDevice != OSAL_ERROR )
    {
        ETG_TRACE_ERR(( "bCloseDoor( ): Error: No valid handle for /dev/cdctrl" ));
    }

    // Get error code if not successful.
    if( s32ReturnValue != OSAL_OK )
    {
        ETG_TRACE_ERR(( "bCloseDoor( ): Error: Failed to reinsert CD media automatically" ));
        // Get error code if not successful.
        TRACE_OSAL_ERROR( VDMMGR_TR_CDCTRLIF );
        bRet = FALSE;
    }
    ETG_TRACE_USR3(( "bCloseDoor( ): End" ));
    return bRet;
}

/* **************************************************FunctionHeaderBegin** *//**
 *
 *  Function:  e8GetLoaderState( tVoid )
 *
 *  Call the OSAL Get Loader State function.
 *
 *  Call the Get Loader State function and evaluate the loader state. Return
 *  the loader state.
 *  No parameters.
 *
 * @return
 *     u8Ret: Return the loader state. Possible states are:
 *            - MMGR_MEDIA_IN_SLOT:      Media is ejected but still in slot
 *            - MMGR_NO_MEDIA_IN_DRIVE:  No media in drive
 *            - MMGR_MEDIA_INSIDE:       Media Completely inserted
 *            - MMGR_EJECT_IN_PROGRESS:  Ejecting (insertion) in progress
 *
 * @date    2006-06-23
 *
 * @note
 *
 *
 *//* ***********************************************FunctionHeaderEnd******* */
T_e8_DiagLoadStatus vdmmgr_tclcdctrlif::e8GetLoaderState( tVoid )
{
    T_e8_DiagLoadStatus  e8Ret;                        // Return value
    tS32                 s32ReturnValue = OSAL_ERROR;  // Return value from OSAL function
    OSAL_trLoaderInfo    rLoaderState;                 // OSAL struct for loader state

    e8Ret                         = MMGR_NO_MEDIA_IN_DRIVE;  // Init e8Ret -> lint
    rLoaderState.u8LoaderInfoByte = 0;                       // Init -> lint

    bEjectOngoing = false;

    if( _hDevice != OSAL_ERROR )
    {
        // Get loader information
        s32ReturnValue = OSAL_s32IOControl( _hDevice, OSAL_C_S32_IOCTRL_CDCTRL_GETLOADERINFO, (intptr_t)&rLoaderState );
    }
    else  // if( _hDevice != OSAL_ERROR )
    {
        ETG_TRACE_ERR(( "e8GetLoaderState( ): Error: No valid handle for /dev/cdctrl" ));
    }

    if( s32ReturnValue == OSAL_OK )
    {
        switch( rLoaderState.u8LoaderInfoByte )   // evaluate return value
        {
        case OSAL_C_U8_MEDIA_IN_SLOT:          // Media is ejected but still in slot
            e8Ret = MMGR_MEDIA_IN_SLOT;
            bEjectOngoing = true;
            break;
        case OSAL_C_U8_NO_MEDIA:               // no media in drive
            e8Ret = MMGR_NO_MEDIA_IN_DRIVE;
            bEjectOngoing = false;
            break;
        case OSAL_C_U8_MEDIA_INSIDE:           // Media Completely inserted
            e8Ret = MMGR_MEDIA_INSIDE;
            bEjectOngoing = false;
            break;
        case OSAL_C_U8_EJECT_IN_PROGRESS:      // Ejecting in progress
            e8Ret = MMGR_EJECT_IN_PROGRESS;
            bEjectOngoing = true;
            break;
        default:
            break;
        }
    }
    else  // if( s32ReturnValue == OSAL_OK )   ->  s32ReturnValue == OSAL_ERROR
    {
        // Get error code if not successful.
        TRACE_OSAL_ERROR( VDMMGR_TR_CDCTRLIF );
        ETG_TRACE_ERR(( "e8GetLoaderState( ): Error: e8GetLoaderState() (OSAL_C_S32_IOCTRL_CDCTRL_GETLOADERINFO) returns with error" ));
    }
    ETG_TRACE_USR1(( "e8GetLoaderState: End: Returns with LoaderState: %d", ETG_ENUM(LOADER_STATE, e8Ret) ));

    return e8Ret;
}


/* **************************************************FunctionHeaderBegin** *//**
 *
 *  Function: s16GetTemperature( T_e8DriveTempSensorStatus* pDriveTempSensorStatus )
 *
 *  Call the OSAL Get Temperature function.
 *
 *  Call the Get Temperature function and return the CD drive temperature.
 *
 * @param   pDriveTempSensorStatus
 *      Pointer to CD drive sensor status to be filled and returned
 *
 * @return
 *   s16Ret: Return the CD drive temperature. Possible values in degree Centigrade in steps of 1K
 *      VDMMGR_CDCTRLIF_TEMP_NOT_SUPPORTED:   Temperature is not supported from
 *                                            this drive
 *      VDMMGR_CDCTRLIF_TEMP_TMP_NOT_AVAIL:   Temperature is temporarily not
 *                                            available
 *
 * @date    2003-01-21
 *
 * @note: Not supported from BP7 drives. Only supported from Kenwood/Pioneer MASCA drive
 *
 *
 *//* ***********************************************FunctionHeaderEnd******* */
tS16 vdmmgr_tclcdctrlif::s16GetTemperature( T_e8DriveTempSensorStatus* pDriveTempSensorStatus )
{
    tS16                    s16Ret;                       // Returns the temperature
    tS32                    s32ReturnValue = OSAL_ERROR;  // Return value from OSAL function
    OSAL_trDriveTemperature rTemperature;                 // OSAL struct for temperature

    s16Ret = VDMMGR_CDCTRLIF_TEMP_TMP_NOT_AVAIL;          // Init the return value
    *pDriveTempSensorStatus = DRIVE_MOUNTED_OK;           // Set default value for sensor status

    if( _hDevice != OSAL_ERROR )
    {
        // Get temperature
        s32ReturnValue = OSAL_s32IOControl( _hDevice,  OSAL_C_S32_IOCTRL_CDCTRL_GETTEMP, (intptr_t)&rTemperature );
        // Copy temperature value (if functions returns with OSAL_OK)
        if( s32ReturnValue == OSAL_OK )
        {
            s16Ret = rTemperature.u8Temperature -100;      // copy return value; Offset -100 is due to old tU8 OSAL IF (->negative temperatures)
        }
        else  // if( s32ReturnValue == OSAL_OK )  ->   s32ReturnValue == OSAL_ERROR
        {
            tU32 u32ErrorCode = OSAL_u32ErrorCode();
            // Evaluate the OSAL error code
            /*lint -save -e788 Info 788;enum constant 'x' not used within defaulted switch */
            switch( u32ErrorCode )
            {
            case OSAL_E_NOTSUPPORTED:
                s16Ret = VDMMGR_CDCTRLIF_TEMP_NOT_SUPPORTED;   // return: temperature from drive is not supported
                *pDriveTempSensorStatus = DRIVE_MOUNTED_NO_SENSOR;
                break;
            case OSAL_E_TEMP_NOT_AVAILABLE:                    // case falls into default
            default:
                s16Ret = VDMMGR_CDCTRLIF_TEMP_TMP_NOT_AVAIL;    // return: temperature is temporarily not available
                *pDriveTempSensorStatus = DRIVE_MOUNTED_ERROR_READ_TEMP;
                break;
            }
            /*lint -restore*/
            ET_TRACE_ERROR_BIN( VDMMGR_TR_CDCTRLIF, ET_EN_T16 _ VDMMGR_TR_99 _ ET_EN_T16 _ __LINE__ _ ET_EN_T32 _ u32ErrorCode _
                                ET_EN_STRING _ OSAL_coszErrorText( u32ErrorCode ) _  ET_EN_DONE );
            ETG_TRACE_ERR(( "s16GetTemperature( ): Error: u8GetTemperature() OSAL_C_S32_IOCTRL_CDCTRL_GETTEMP returns with error" ));
        }
    }
    else  // if( _hDevice != OSAL_ERROR )
    {
        s16Ret = VDMMGR_CDCTRLIF_TEMP_NOT_SUPPORTED;
        *pDriveTempSensorStatus = DRIVE_MOUNTED_ERROR_READ_TEMP;
        ETG_TRACE_ERR(( "s16GetTemperature( ): Error: No valid handle for /dev/cdctrl" ));
    }
    ETG_TRACE_USR1(( "s16GetTemperature( ): CD drive temperature is %d degree Centigrade, Sensor status: %d", s16Ret, ETG_ENUM(DRIVE_SENSOR_STATE, *pDriveTempSensorStatus) ));
    return s16Ret;
}

/* **************************************************FunctionHeaderBegin** *//**
 *
 *  Function:  bGetDriveVersion( OSAL_trCDDriveVersion* prDriveVersion )
 *
 *  Call the OSAL Get drive version function.
 *
 *  Call the OSAL Get drive version function and return the CD drive HW version
 *  in high byte and the SW version in low byte of return value.
 *  No parameter.
 *
 * @return
 *      bRet:  TRUE:    Version information could be read.
 *             FALSE:   Version information couln'd be read.
 *
 * @date    2006-03-30
 *
 * @note
 *
 *
 *//* ***********************************************FunctionHeaderEnd******* */
tBool vdmmgr_tclcdctrlif::bGetDriveVersion( OSAL_trDriveVersion* prDriveVersion )
{
    tBool                   bRet           = TRUE;        // Get version (or not)
    tS32                    s32ReturnValue = OSAL_ERROR;  // Return value from OSAL function

    if( _hDevice != OSAL_ERROR )
    {
        // Get drive version
        s32ReturnValue = OSAL_s32IOControl( _hDevice, OSAL_C_S32_IOCTRL_CDCTRL_GETDRIVEVERSION, (intptr_t)prDriveVersion );
    }
    else  // if( _hDevice != OSAL_ERROR )
    {
        ETG_TRACE_ERR(( "bGetDriveVersion( ): Error: No valid handle for /dev/cdctrl" ));
    }

    if( s32ReturnValue == OSAL_OK )
    {
        ETG_TRACE_USR1(( "bGetDriveVersion( ): Get drive versions HW: %s", (tPChar)prDriveVersion->au8ModelNumber ));
        ETG_TRACE_USR1(( "bGetDriveVersion( ): Get drive versions SW: %s", (tPChar)prDriveVersion->au8FirmwareRevision ));
    }
    else  // if( s32ReturnValue == OSAL_OK )  ->   s32ReturnValue == OSAL_ERROR
    {
        // Get error code if not successful.
        TRACE_OSAL_ERROR( VDMMGR_TR_CDCTRLIF );
        ETG_TRACE_ERR(( "bGetDriveVersion( ): Error: (OSAL_C_S32_IOCTRL_CDCTRL_GET_DRIVE_VERSION) returns with error" ));
        // Set return value to FALSE
        bRet = FALSE;
    }
    return bRet;
}


/* **************************************************FunctionHeaderBegin** *//**
 *
 *  Function: bReadRawDataLBA( tU32 u32LBAAdress, tU32 u32CntLBA,
 *                             tS8* ps8Read )
 *
 *  Read raw data.
 *
 *  Read raw data at given LBA. The OSAL retries up to 100 times to read if a
 *  DMA error occur (-> shouldn't be a problem during diagnosis!) and it
 *  retries up to 25 times if an CRC error occur. If the CD block couldn't be
 *  read within this retries the read function returns an error (OSAL_E_IOERROR)
 *
 * @param   u32LBAAdress
 *      LBA: Address of sector to read data.
 *
 * @param   u32CntLBA
 *      Amount of LBAs to read
 *
 * @param   ps8Read
 *      Pointer to byte array
 *
 * @return
 *      bRet
 *        TRUE:   -> success
 *        FALSE:  -> error
 *
 * @date    2006-03-09
 *
 * @note
 *
 *
 *//* ***********************************************FunctionHeaderEnd******* */
tBool vdmmgr_tclcdctrlif::bReadRawDataLBA( tU32 u32LBAAdress, tU32 u32CntLBA, tS8* ps8ReadBuf )
{
    tS32                 s32ReturnValue = OSAL_ERROR;  // Return value from OSAL function
    tBool                bRet;                         // Function return value
    OSAL_trReadRawInfo   rReadRawInfo;                 // OSAL struct for raw data info

    bRet = TRUE;
    if( ps8ReadBuf )
    {
        // Init struct OSAL_trReadRawInfo
        rReadRawInfo.ps8Buffer     = ps8ReadBuf;        // Pointer to read buffer
        rReadRawInfo.u32LBAAddress = u32LBAAdress;      // Requested LBA
        rReadRawInfo.u32NumBlocks  = u32CntLBA;
        if( _hDevice != OSAL_ERROR )
        {
            // Read raw data (uncached)
            s32ReturnValue = OSAL_s32IOControl( _hDevice, OSAL_C_S32_IOCTRL_CDCTRL_READRAWDATAUNCACHED, (intptr_t)&rReadRawInfo );
        }
        else  // if( _hDevice != OSAL_ERROR )
        {
            ETG_TRACE_ERR(( "bReadRawDataLBA( ): Error: No valid handle for /dev/cdctrl" ));
        }
        // evaluate return value
        if( s32ReturnValue == OSAL_OK )
        {
            ETG_TRACE_USR1(( "bReadRawDataLBA( ): Could read (uncached) raw data at %d )", u32LBAAdress ));
        }
        else  // if( s32ReturnValue == OSAL_OK )  ->   s32ReturnValue == OSAL_ERROR
        {
            // Get error code if not successful.
            TRACE_OSAL_ERROR( VDMMGR_TR_CDCTRLIF );
            ETG_TRACE_SYS(( "bReadRawDataLBA( ): Warning: Couldn't read (uncached) raw data at %d", u32LBAAdress ));
            bRet = FALSE;
        }
    }
    else
    {
        ETG_TRACE_ERR(( "bReadRawDataLBA( ): Enter bReadRawDataLBA() without valid pointer for data to read" ));
    }

    return bRet;
}


tBool vdmmgr_tclcdctrlif::bEjectInThread( tVoid ) const
{
    tS32              s32Success;
    OSAL_tEventHandle hCbkEvent;
    tBool             bRet = TRUE;

    ETG_TRACE_USR1(( "bEjectInThread( ): Start" ));

    if( OSAL_s32EventOpen( VDMMGR_CDCTRLIF_EVENTNAME, &hCbkEvent ) == OSAL_OK )
    {
        s32Success = OSAL_s32EventPost( hCbkEvent, VDMMGR_EVENT_MASK_CMD_EJECTCD,
                                        OSAL_EN_EVENTMASK_OR );
        if( s32Success == OSAL_ERROR )
        {
            ETG_TRACE_ERR(( "bEjectInThread( ): Error: Eject in Thread; OSAL_s32EventPost( )returns with error" ));
            // Get error code if not successful.
            TRACE_OSAL_ERROR( VDMMGR_TR_CDCTRLIF );
            bRet = FALSE;
        }

        s32Success = OSAL_s32EventClose( hCbkEvent );
        if(s32Success == OSAL_ERROR)
        {
            ETG_TRACE_ERR(( "bEjectInThread( ): Error: Eject in Thread; OSAL_s32EventClose( )returns with error" ));
            // Get error code if not successful.
            TRACE_OSAL_ERROR( VDMMGR_TR_CDCTRLIF );
            bRet = FALSE;
        }
    }
    else
    {
        ETG_TRACE_ERR(( "bEjectInThread( ): Error: Eject in Thread; OSAL_s32EventOpen( )returns with error" ));
        // Get error code if not successful.
        TRACE_OSAL_ERROR( VDMMGR_TR_CDCTRLIF );
        bRet = FALSE;
    }
    return bRet;
}


tBool vdmmgr_tclcdctrlif::bIsEjectActive( tVoid ) const
{
    return _bEjectActive;
}


tVoid vdmmgr_tclcdctrlif::vStartTemperatureTimeout( tVoid )
{
    if(_poMain && _poMain->_vdmmgr_poCCAService )
    {
        // Create and start PollCdDriveTempTimer
        (tVoid)_oPollCdDriveTempTimer.Create( VDMMGR_CDCTRL_TIMEOUT_CD_DRIVE_TEMP, VDMMGR_CDCTRL_TIMEOUT_CD_DRIVE_TEMP,
                                              _poMain->_vdmmgr_poCCAService, VDMMGR_TIMERTYPE_CD_GET_TEMPERATURE );
        (tVoid)_oPollCdDriveTempTimer.bStart( );
    }
}


tVoid vdmmgr_tclcdctrlif::vStopTemperatureTimeout( tVoid )
{
    _oPollCdDriveTempTimer.vDelete();
}


tVoid vdmmgr_tclcdctrlif::vSetCdReinsertActive( tBool bCdReinsertActive )
{
    _bCdReinsertActive = bCdReinsertActive;
}


tBool vdmmgr_tclcdctrlif::bGetCdReinsertActive( tVoid ) const
{
    return _bCdReinsertActive;
}


/* **************************************************FunctionHeaderBegin** *//**
 *
 *  Function: vDiagDriveTest( const T_DiagMethodControlStartDriveTest* pDiagStartDriveTest )
 *
 *  Evaluates drive test commands.
 *
 *  Evaluates drive test commands.
 *
 *
 *
 *
 * @param   pDiagStartDriveTest
 *      Pointer to test parameter. T_DiagMethodControlStartDriveTest has following
 *      elements and possible values_
 *        e8Duration:                              MMGR_CONTINUOUS, MMGR_SINGLE_RUN, MMGR_SHORT_TEF_CD
 *        StartDriveTestParam.e16Drive:            MMGR_CD
 *        StartDriveTestParam.e16DriveTestCmd:     MMGR_START, MMGR_STOP, MMGR_CANCEL
 *        StartDriveTestParam.e16DriveTestParam:   MMGR_SHORT, MMGR_INTENSIVE
 *
 * @return
 *      no return value
 *
 * @date    2006-03-27
 *
 * @note
 *
 *
 *//* ***********************************************FunctionHeaderEnd******* */
tVoid vdmmgr_tclcdctrlif::vDiagDriveTest( const T_DiagMethodControlStartDriveTest* pDiagStartDriveTest )
{
    ETG_TRACE_USR1(( "vDiagDriveTest( ): Enter: with command: %d", ETG_ENUM(CMD_DRIVE_TEST, (tU16)pDiagStartDriveTest->StartDriveTestParam.e16DriveTestCmd) ));
    switch( pDiagStartDriveTest->StartDriveTestParam.e16DriveTestCmd )
    {
    case MMGR_START:
        if( ( _u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_CANCEL_IN_PROGRESS ) )
        {
            //todo: Release after Lei's changes
            /*
           _poMain->_vdmmgr_poCCAService->vSendMethodResultDiagDrvTest( MMGR_RESULT_BUSY,
               ( (_u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_DUR_CONTINUOUS ) ? MMGR_CONTINUOUS : MMGR_SINGLE_RUN),
               &_DiagMethodResultDrvTest );
             */
        }
        else
        {
            // Set command bit(s)
            vDiagSetCmdDurationBit( pDiagStartDriveTest->e8Duration );
            vDiagSetCmdTestTypeBit( pDiagStartDriveTest->StartDriveTestParam.e16DriveTestParam );
            // Check if no test is running
            if(  ! (_u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_TESTRUNING) )
            {
                // No test running -> Set bit for 'test running'
                _u32DiagDriveTestBitStatusField |= VDMMGR_DIAG_CD_STATUS_TESTRUNING;
                // Set motor on and wait 2 s before starting the CD read test. So
                // we are sure that the motor is running when we start the read test.
                (tVoid)bSetMotorOn( );
                if( OSAL_s32ThreadWait( VDMMGR_DIAG_CD_WAIT_FOR_MOTOR_ON ) == OSAL_ERROR )
                {
                    TRACE_OSAL_ERROR( VDMMGR_TR_CDCTRLIF );
                }
                // Start test -> do init stuff and send event to start test in thread
                vDiagStartDriveTest();
            }
        }
        // no else necessary because a test is already running. So it' only necessary
        // to set the command bits (done before if(...) )
        break;
    case MMGR_STOP:
        // Is a drive test running? And runs the test with CONTINUOUS?
        if(   ( _u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_TESTRUNING )
              && ( _u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_DUR_CONTINUOUS )
              )
        {
            // Set control bit to stop
            _u32DiagDriveTestBitCmdField |= VDMMGR_DIAG_CD_CMD_STOP_TEST;
        }  // else: nothing to do
        break;
    case MMGR_CANCEL:
        // Is a drive test running? And runs the test with CONTINUOUS?
        if( _u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_TESTRUNING )
        {
            // Set control bit to cancel
            _u32DiagDriveTestBitCmdField |= VDMMGR_DIAG_CD_CMD_CANCEL_TEST;
            // If cancel is not in progress set in progress and send test cancelled to client, this is needed because diagnosis expects a
            // a immediate method result
            if( !( _u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_CANCEL_IN_PROGRESS ))
            {
                vDiagCalcNewActualResultForMethodCancel();
                _u32DiagDriveTestBitStatusField |= VDMMGR_DIAG_CD_STATUS_CANCEL_IN_PROGRESS ;
                if( _poMain && _poMain->_vdmmgr_poCCAService )
                {
                    _poMain->_vdmmgr_poCCAService->vSendMethodResultDiagDrvTest( MMGR_RESULT_CANCELED,
                                                                                 ( (_u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_DUR_CONTINUOUS ) ? MMGR_CONTINUOUS : MMGR_SINGLE_RUN),
                                                                                 &_DiagMethodResultDrvTest );
                }
            }
        }  // else: nothing to do
        break;
    default:
        break;
    }
}


tVoid vdmmgr_tclcdctrlif::vDiagStartDriveTest( tVoid )
{
    tS32              s32Success;
    OSAL_tEventHandle hCbkEvent;
    tBool             bInitPossible;
    tU8               u8MediaCheck;

    ETG_TRACE_USR1(( "vDiagStartDriveTest( ): Start" ));

    bInitPossible = TRUE;

    // Init drive type in result struct
    _DiagMethodResultDrvTest.e16Drive = MMGR_CD;

    u8MediaCheck = u8DiagIsMediaTypeOkForDriveTest();

    if( _poMain && _poMain->_vdmmgr_poCCAService )
    {
        // Check Media-Type
        if( u8MediaCheck == VDMMGR_DIAG_CD_MEDIA_OK )
        {
            // Init stuff done?
            if( ! (_u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_INIT_DONE) )
            {
                bInitPossible = bDiagInitAllDriveTestParameters( );
            }
            // Init possible (if necessary) ?
            if( bInitPossible )
            {
                // Send 'MessageResultFirst' to inform diagnosis that drive could be started
                _poMain->_vdmmgr_poCCAService->vSendMethodResultFirstDiagDrvTest( );
                // Send an event to thread.
                ETG_TRACE_USR1(( "vDiagStartDriveTest( ): Start drive test in Thread" ));
                if( OSAL_s32EventOpen( VDMMGR_CDCTRLIF_EVENTNAME, &hCbkEvent ) == OSAL_OK )
                {
                    s32Success = OSAL_s32EventPost( hCbkEvent, VDMMGR_EVENT_MASK_CMD_DIAG_START_CDDRIVE_TEST,
                                                    OSAL_EN_EVENTMASK_OR );
                    if( s32Success == OSAL_ERROR )
                    {
                        // Get error code if not successful.
                        TRACE_OSAL_ERROR( VDMMGR_TR_CDCTRLIF );
                        ETG_TRACE_ERR(( "vDiagStartDriveTest( ): Error: OSAL_s32EventPost( )returns with error" ));
                    }
                    s32Success = OSAL_s32EventClose( hCbkEvent );
                    if(s32Success == OSAL_ERROR)
                    {
                        ETG_TRACE_ERR(( "vDiagStartDriveTest( ): Error: OSAL_s32EventClose( )returns with error" ));
                        // Get error code if not successful.
                        TRACE_OSAL_ERROR( VDMMGR_TR_CDCTRLIF );
                    }
                }
                else
                {
                    ETG_TRACE_ERR(( "vDiagStartDriveTest( ): Error: OSAL_s32EventOpen( )returns with error" ));
                    // Get error code if not successful.
                    TRACE_OSAL_ERROR( VDMMGR_TR_CDCTRLIF );
                }
            }
            else  // if( bInitPossible )
            {
                ETG_TRACE_USR1(( "vDiagStartDriveTest( ): Init not possible -> return MMGR_RESULT_FALSE" ));
                // Init result struct because of the failed start condition
                vDiagInitDriveTestResultStructOnError( );
                // Send result 'MMGR_RESULT_FALSE' to diagnostics
                _poMain->_vdmmgr_poCCAService->vSendMethodResultDiagDrvTest( MMGR_RESULT_FALSE,
                                                                             ( (_u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_DUR_CONTINUOUS ) ? MMGR_CONTINUOUS : MMGR_SINGLE_RUN),
                                                                             &_DiagMethodResultDrvTest );
                // Reset command and status bits
                vDiagResetDriveCmdStatusBitsOnError( );
            }
        }
        else  // if( u8MediaCheck != VDMMGR_DIAG_CD_MEDIA_OK )  -> VDMMGR_DIAG_CD_MEDIA_NO == MMGR_RESULT_NO_MEDIA; VDMMGR_DIAG_CD_MEDIA_INVALID == MMGR_RESULT_INVALID_MEDIA
        {
            ETG_TRACE_SYS(( "vDiagStartDriveTest( ): Media Type <%d> is not suitable for drive test", ETG_ENUM(CD_TYPE, _poMain->_vdmmgr_poCCAService->enGetActCdType()) ));
            // Init result struct because of the failed start condition
            vDiagInitDriveTestResultStructOnError( );
            // Send result 'MMGR_RESULT_NO_MEDIA' or 'MMGR_RESULT_INVALID_MEDIA' to diagnostics
            _poMain->_vdmmgr_poCCAService->vSendMethodResultDiagDrvTest( (T_e8_DiagRetValueDriveTest)u8MediaCheck,
                                                                         ( (_u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_DUR_CONTINUOUS ) ? MMGR_CONTINUOUS : MMGR_SINGLE_RUN),
                                                                         &_DiagMethodResultDrvTest );
            // Reset command and status bits
            vDiagResetDriveCmdStatusBitsOnError( );
        }
    }
}


tU8 vdmmgr_tclcdctrlif::u8DiagIsMediaTypeOkForDriveTest( tVoid )
{
    tU8                     u8Return;
    tS32                    s32ReturnValue;
    OSAL_trCDROMTrackInfo   rCDRomTrackInfo; /* OSAL struct for track information */

    // Init u8Return -> lint
    u8Return = (tU8)VDMMGR_DIAG_CD_MEDIA_NO;

    ETG_TRACE_USR1(( "u8DiagIsMediaTypeOkForDriveTest( ): Start" ));

    if( _poMain && _poMain->_vdmmgr_poCCAService )
    {
        /*lint -save -e788 Info 788; enum constant 'x' not used within defaulted switch */
        switch( _poMain->_vdmmgr_poCCAService->enGetActCdType() )
        {
        case MMGR_DATA:                        // case, no break
            u8Return = VDMMGR_DIAG_CD_MEDIA_OK;
            break;
        case MMGR_NO_MEDIA:                    // case, no break
        case MMGR_EJECTING:                    // case, no break
        case MMGR_IN_SLOT:                     // case, no break
        case MMGR_INSERTION:
            u8Return = (tU8)VDMMGR_DIAG_CD_MEDIA_NO;
            break;
        case MMGR_AUDIO:
            // short TEF CD read test wanted (-> ActCdType == AUDIO ) is possible
            if( _u32DiagDriveTestBitCmdField & VDMMGR_DIAG_CD_CMD_START_SHORT_TEF_CD )
            {
                // Production test. Mix-Mode CD with data and audio tracks
                // - Get information about wanted track. Set first track -> data
                rCDRomTrackInfo.u32TrackNumber = VDMMGR_DIAG_CD_TEF_CD_DATA_TRACK_NO;
                // Set pick up to track 1. The cdctrl function 'get track info' position
                // the pick up at the track 1 to get information. If the pick up is
                // positioned above a data track the CD type change to DATA'.
                s32ReturnValue = OSAL_s32IOControl( _hDevice, OSAL_C_S32_IOCTRL_CDCTRL_GETTRACKINFO, (intptr_t)&rCDRomTrackInfo );
                u8Return = ( s32ReturnValue == OSAL_OK ) ? (tU8)VDMMGR_DIAG_CD_MEDIA_OK : (tU8)VDMMGR_DIAG_CD_MEDIA_INVALID;
            }
            else  // CD DA is invalid (/useless) for read test
            {
                u8Return = (tU8)VDMMGR_DIAG_CD_MEDIA_INVALID;
            }
            break;
        default:    // MMGR_INCORRECT
            u8Return = (tU8)VDMMGR_DIAG_CD_MEDIA_INVALID;
            break;
        }
    }
    /*lint -restore*/
    return u8Return;
}


tBool vdmmgr_tclcdctrlif::bDiagInitAllDriveTestParameters( tVoid )
{
    tU32  u32WantedMemSize;         // Wanted memory size in byte
    tBool bRet;

    bRet = TRUE;

    ETG_TRACE_USR1(( "bDiagInitAllDriveTestParameters( ): Start" ));

    // Set LBA for short TEF CD test
    _au32DiagLBA_IMO_TEF_CD[VDMMGR_DIAG_CD_INNER_RIM] = VDMMGR_DIAG_CD_TEF_CD_MIN_LBA;
    _au32DiagLBA_IMO_TEF_CD[VDMMGR_DIAG_CD_MIDDLE]    = (VDMMGR_DIAG_CD_TEF_CD_MAX_LBA - VDMMGR_DIAG_CD_TEF_CD_MIN_LBA ) / 2;
    _au32DiagLBA_IMO_TEF_CD[VDMMGR_DIAG_CD_OUTER_RIM] = VDMMGR_DIAG_CD_TEF_CD_MAX_LBA;

    // Get max LBA
    _u32DiagMaxLBA = u32DiagGetMaxLBA();
    // Set max/last LBA in result struct
    _DiagMethodResultDrvTest.u32MaxLBA = _u32DiagMaxLBA;

    // Check max LBA
    if( _u32DiagMaxLBA != VDMMGR_DIAG_CD_INIT_MAX_LBA )
    {
        // Set LBA for short test
        _au32DiagLBA_IMO[VDMMGR_DIAG_CD_INNER_RIM] = VDMMGR_DIAG_CD_MIN_LBA;
        _au32DiagLBA_IMO[VDMMGR_DIAG_CD_MIDDLE]    = _u32DiagMaxLBA / 2;
        _au32DiagLBA_IMO[VDMMGR_DIAG_CD_OUTER_RIM] = _u32DiagMaxLBA;
        // Set read order for short read test
        _au8DiagOrderShortTest[VDMMGR_DIAG_CD_IMO_POS_0] = VDMMGR_DIAG_CD_INNER_RIM;
        _au8DiagOrderShortTest[VDMMGR_DIAG_CD_IMO_POS_1] = VDMMGR_DIAG_CD_OUTER_RIM;
        _au8DiagOrderShortTest[VDMMGR_DIAG_CD_IMO_POS_2] = VDMMGR_DIAG_CD_MIDDLE;
        _au8DiagOrderShortTest[VDMMGR_DIAG_CD_IMO_POS_3] = VDMMGR_DIAG_CD_INNER_RIM;
        _au8DiagOrderShortTest[VDMMGR_DIAG_CD_IMO_POS_4] = VDMMGR_DIAG_CD_MIDDLE;
        _au8DiagOrderShortTest[VDMMGR_DIAG_CD_IMO_POS_5] = VDMMGR_DIAG_CD_OUTER_RIM;
        _au8DiagOrderShortTest[VDMMGR_DIAG_CD_IMO_POS_6] = VDMMGR_DIAG_CD_INNER_RIM;

        // allocated memory for read test
        // Init with max value
        u32WantedMemSize = VDMMGR_DIAG_CD_WANTED_MEM_SIZE_MAX;
        // Try until memory is available or memory size to to small
        while(    ( _ps8DiagReadMem == NULL )
                  && ( u32WantedMemSize > VDMMGR_DIAG_CD_WANTED_MEM_SIZE_MIN )
                  )
        {
            _ps8DiagReadMem = new tS8 [u32WantedMemSize];
            // Check if memory was available
            if( _ps8DiagReadMem == NULL )
            {
                // Try to get the half among
                u32WantedMemSize = u32WantedMemSize / 2;
            }
        }

        // Check if memory was available
        if( _ps8DiagReadMem != NULL )
        {
            // Calculate max possible count of blocks to read at once
            _u16DiagMaxBlocksPerReadAttempt = (tU16)(u32WantedMemSize / VDMMGR_DIAG_CD_LBA_SIZE);
            ETG_TRACE_USR1(( "bDiagInitAllDriveTestParameters( ): Alloc %d bytes to read %d blocks at once.", u32WantedMemSize, _u16DiagMaxBlocksPerReadAttempt ));
            // Set bit marker for 'init done'
            _u32DiagDriveTestBitStatusField |= VDMMGR_DIAG_CD_STATUS_INIT_DONE;
        }
        else
        {
            bRet = FALSE;
            ETG_TRACE_ERR(( "bDiagInitAllDriveTestParameters( ): Error: Unable to alloc %u(8,4,M) for read test", VDMMGR_DIAG_CD_WANTED_MEM_SIZE_MIN ));
        }
    }
    else
    {
        bRet = FALSE;
    }

    return bRet;
}


tVoid vdmmgr_tclcdctrlif::vDiagInitDriveTestResultStruct( tVoid )
{
    ETG_TRACE_USR1(( "vDiagInitDriveTestResultStruct( ): Start" ));
    _DiagMethodResultDrvTest.u32SectorsRead            = VDMMGR_DIAG_CD_RESULT_INIT;
    _DiagMethodResultDrvTest.u16ReadAttempts           = VDMMGR_DIAG_CD_RESULT_INIT;
    _DiagMethodResultDrvTest.u32TotalReadTime          = VDMMGR_DIAG_CD_RESULT_INIT;
    _DiagMethodResultDrvTest.u16AverageReadtimeSector  = VDMMGR_DIAG_CD_RESULT_INIT;
    _DiagMethodResultDrvTest.u16MaxReadTimePerSector   = VDMMGR_DIAG_CD_RESULT_INIT;
    _DiagMethodResultDrvTest.u16CntErrors              = VDMMGR_DIAG_CD_RESULT_INIT;
}


tVoid vdmmgr_tclcdctrlif::vDiagInitDriveTestResultStructOnError( tVoid )
{
    // First do the 'normal' init stuff
    vDiagInitDriveTestResultStruct( );
    // and now init the test type
    // Short CD test
    if( _u32DiagDriveTestBitCmdField & VDMMGR_DIAG_CD_CMD_START_SHORT )
    {
        _DiagMethodResultDrvTest.e16DiagDrvTesttype = MMGR_SHORT;
    }
    // Intensive test
    else if( _u32DiagDriveTestBitCmdField & VDMMGR_DIAG_CD_CMD_START_INTENSIVE )
    {
        _DiagMethodResultDrvTest.e16DiagDrvTesttype = MMGR_INTENSIVE;
    }
    // Short TEF CD test
    else if( _u32DiagDriveTestBitCmdField & VDMMGR_DIAG_CD_CMD_START_SHORT_TEF_CD )
    {
        _DiagMethodResultDrvTest.e16DiagDrvTesttype = MMGR_SHORT_TEF_CD;
    }
    // At last reset the maximum Logical Block Address
    _DiagMethodResultDrvTest.u32MaxLBA = VDMMGR_DIAG_CD_RESULT_INIT;
}


tVoid vdmmgr_tclcdctrlif::vDiagResetDriveCmdStatusBitsOnError( tVoid )
{
    // Reset bits for command and status if a 'start read test' fails
    _u32DiagDriveTestBitCmdField    = VDMMGR_DIAG_CD_INIT;
    _u32DiagDriveTestBitStatusField = VDMMGR_DIAG_CD_INIT;
}


/* **************************************************FunctionHeaderBegin** *//**
 *
 *  Function: tU32 u32DiagGetMaxLBA( tVOID )
 *
 *  Get (nearly) max/last LBA.
 *
 *  Get (nearly) max/last LBA of CD. Use dev/cdctrl function
 *  OSAL_C_S32_IOCTRL_CDCTRL_GETCDINFO to get TOC data.
 *  This OSAL function gives originally only the min/max track
 *  but in GEN3 we use the dev/cdaudio struct OSAL_trCDAudioInfo
 *
 *  No parameter.
 *
 * @return
 *      A value > 0:  Value of (nearly) last LBA
 *      FALSE:        Error occurred.
 *
 * @date    2003-04-04
 *
 * @note
 *
 *
 *//* ***********************************************FunctionHeaderEnd******* */
tU32 vdmmgr_tclcdctrlif::u32DiagGetMaxLBA( tVoid ) const
{
    tU32                  u32RetMaxLBA;     // Return value. Max LBA or failure
    tS32                  s32ReturnValue;   // Return value from OSAL function
    OSAL_trCDROMTrackInfo rCDTrackInfo;     // OSAL struct for track information

    ETG_TRACE_USR1(( "u32DiagGetMaxLBA( ): Start" ));

    u32RetMaxLBA                = VDMMGR_DIAG_CD_INIT_MAX_LBA;        // Init return value with init/error value
    rCDTrackInfo.u32TrackNumber = (tU32)VDMMGR_DIAG_CD_MAGIC_NUMBER;  // magic track number to get max LBA of CD
    // Check OSAL handle to dev/cdctrl
    if( _hDevice != OSAL_ERROR )
    {
        // Read CD TOC information
        s32ReturnValue = OSAL_s32IOControl( _hDevice, OSAL_C_S32_IOCTRL_CDCTRL_GETTRACKINFO, (intptr_t)&rCDTrackInfo );
        // Check if read TOC data function was successful
        if( s32ReturnValue == OSAL_OK )
        {
            // Calculate max LBA. Get max LBA value via track info for magic track number
            // Take an offset into consideration.
            // LBA 0 is corresponding to the MSF (Minute Second Frame) 020.
            u32RetMaxLBA = (  rCDTrackInfo.u32LBAAddress
                              - ( VDMMGR_DIAG_CD_LBA_MFS_OFFSET * VDMMGR_DIAG_CD_FRAMES_PER_SEC)
                              -  VDMMGR_DIAG_CD_SECURITY_OFFSET_END              //lint !e834  Info 834: Operator '-' followed by operator '-' is confusing.  Use parentheses.
                              );
            ETG_TRACE_USR1(( "u32DiagGetMaxLBA( ): Max/last LBA: %d", u32RetMaxLBA ));
        }
        else
        {
            TRACE_OSAL_ERROR( VDMMGR_TR_CDCTRLIF );
        }
    }
    else
    {
        ETG_TRACE_ERR(( "u32DiagGetMaxLBA( ): Error: No valid handle for /dev/cdctrl" ));
    }
    // Return value of last LBA or failure value
    return u32RetMaxLBA;
}


tVoid vdmmgr_tclcdctrlif::vDiagSetCmdDurationBit( T_e8_DiagDuration e8Duration )
{
    ETG_TRACE_USR1(( "vDiagSetCmdDurationBit( ): Enter: duration: %d", ETG_ENUM(DURATION, e8Duration) ));
    if( e8Duration == MMGR_CONTINUOUS )
    {
        // Set continuous flag and clear single flag
        _u32DiagDriveTestBitCmdField |=  VDMMGR_DIAG_CD_CMD_DUR_CONTINUOUS;
        _u32DiagDriveTestBitCmdField &= ~VDMMGR_DIAG_CD_CMD_DUR_SINGLE;

    }
    else if( e8Duration == MMGR_SINGLE_RUN )
    {
        // Set single flag and clear continuous flag
        _u32DiagDriveTestBitCmdField |=  VDMMGR_DIAG_CD_CMD_DUR_SINGLE;
        _u32DiagDriveTestBitCmdField &= ~VDMMGR_DIAG_CD_CMD_DUR_CONTINUOUS;
    }
    else
    {
        ETG_TRACE_ERR(( "vDiagSetCmdDurationBit( ): vUnknown value for e8Duration %d", e8Duration ));
    }
}


tVoid vdmmgr_tclcdctrlif::vDiagSetCmdTestTypeBit( T_e16_DiagDrvTestParam e16DriveTestParam )
{
    ETG_TRACE_USR1(( "vDiagSetCmdTestTypeBit( ): Enter: test type: %d", ETG_ENUM(TEST_TYPE, e16DriveTestParam) ));
    if( e16DriveTestParam == MMGR_SHORT )
    {
        // Set 'start short' command flag and clear 'start intensive' +
        // 'start short TEF CD' flag
        _u32DiagDriveTestBitCmdField |=  VDMMGR_DIAG_CD_CMD_START_SHORT;
        _u32DiagDriveTestBitCmdField &= ~VDMMGR_DIAG_CD_CMD_START_INTENSIVE;
        _u32DiagDriveTestBitCmdField &= ~VDMMGR_DIAG_CD_CMD_START_SHORT_TEF_CD;
    }
    else if( e16DriveTestParam == MMGR_INTENSIVE )
    {
        // Set 'start intensive' command flag and clear 'start short' +
        // 'start short TEF CD' flag
        _u32DiagDriveTestBitCmdField |=  VDMMGR_DIAG_CD_CMD_START_INTENSIVE;
        _u32DiagDriveTestBitCmdField &= ~VDMMGR_DIAG_CD_CMD_START_SHORT;
        _u32DiagDriveTestBitCmdField &= ~VDMMGR_DIAG_CD_CMD_START_SHORT_TEF_CD;
    }
    else if( e16DriveTestParam == MMGR_SHORT_TEF_CD )
    {
        // Set 'start short TEF CD' command flag and clear 'start intensive' +
        // 'start short' flag
        _u32DiagDriveTestBitCmdField |=  VDMMGR_DIAG_CD_CMD_START_SHORT_TEF_CD;
        _u32DiagDriveTestBitCmdField &= ~VDMMGR_DIAG_CD_CMD_START_SHORT;
        _u32DiagDriveTestBitCmdField &= ~VDMMGR_DIAG_CD_CMD_START_INTENSIVE;
    }
    else
    {
        ETG_TRACE_ERR(( "vDiagSetCmdTestTypeBit( ): vUnknown value for e16DriveTestParam %d", e16DriveTestParam ));

    }
}


tVoid vdmmgr_tclcdctrlif::vDiagSelectTrackType( T_e8_TrackType e8ReqTrackType )
{
    OSAL_tEventHandle       hCbkEvent;
    tBool                   bSend       = TRUE;  // Send answer direct after local check; no further action necessary/possible
    T_e16_TrackTypeResult   e8Result    = OK;
    T_e8_TrackType          e8ActTrType = INIT;

    ETG_TRACE_USR1(( "vDiagSelectTrackType( ): Start" ));

    _e8ReqTrackType = INIT;                         // Init member track type

    if( _poMain && _poMain->_vdmmgr_poCCAService )  // Check if the service is there
    {
        // Check the requested track type
        if( (e8ReqTrackType != AUDIO) && (e8ReqTrackType != DATA) )
        {
            e8Result = INVALID_REQUESTED_TRACKTYPE;
            ETG_TRACE_USR1(( "vDiagSelectTrackType( ): Requested track type is invalid (%d)", ETG_ENUM( TRACK_TYPE, e8ReqTrackType) ));
        }
        // Get the actual track type
        e8ActTrType = e8DiagGetActualTrackType( );
        ETG_TRACE_USR3(( "vDiagSelectTrackType( ): Actual track type = (%d)", ETG_ENUM( TRACK_TYPE, e8ActTrType) ));
        // Check if a actual track type is maybe invalid (check only if everything was OK until here...)
        if(   (e8Result == OK)                                   // OK until here...
              && (e8ActTrType != AUDIO) && (e8ActTrType != DATA)    // invalid track type
              )
        {
            e8Result = ( e8ActTrType == NONE ) ? NO_CD : INCORRECT_MEDIA;
            ETG_TRACE_USR1(( "vDiagSelectTrackType( ): Wrong actual track type (%d)", ETG_ENUM( TRACK_TYPE, e8ActTrType ) ));
        }
        // Check if a different track type was requested (if everything is OK until here...)
        if(   (e8Result == OK)                 // OK until here...
              && (e8ActTrType != e8ReqTrackType)  // different track type
              )
        {
            bSend = FALSE;                      // don't send a MethodResult -> further action is necessary
        }

        if( bSend ) // Check if the answer could be send here OR if further action is necessary (-> Switch track type)
        {
            _poMain->_vdmmgr_poCCAService->vSendMethodResultDiagSelectTrackType( e8ActTrType, e8ReqTrackType, e8Result );
        }
        else        // Further action necessary -> Send an event to thread.
        {
            if( OSAL_s32EventOpen( VDMMGR_CDCTRLIF_EVENTNAME, &hCbkEvent ) == OSAL_OK )
            {
                _e8ReqTrackType = e8ReqTrackType;   // Set member variable requested track type
                ETG_TRACE_USR1(( "vDiagSelectTrackType( ): Different track type requested. Send event to start further action in CD control thread; Act Type=%d; Req. Type=%d ", ETG_ENUM( TRACK_TYPE, e8ActTrType), ETG_ENUM( TRACK_TYPE, e8ReqTrackType) ));
                (tVoid)OSAL_s32EventPost( hCbkEvent, VDMMGR_EVENT_MASK_CMD_DIAG_SET_REQUESTED_TRACKTYPE, OSAL_EN_EVENTMASK_OR );
                (tVoid)OSAL_s32EventClose( hCbkEvent );
            }
        }
    }
    ETG_TRACE_USR1(( "vDiagSelectTrackType( ): End" ));
}


T_e8_TrackType vdmmgr_tclcdctrlif::e8DiagGetActualTrackType( tVoid ) const
{
    T_e8_TrackType    e8ActTrType = INIT;

    switch( _u16MediaChange )
    {
    case OSAL_C_U16_MEDIA_EJECTED:
        e8ActTrType = NONE;
        break;
    case OSAL_C_U16_INCORRECT_MEDIA:
        e8ActTrType = INVALID;
        break;
    case OSAL_C_U16_DATA_MEDIA:
        e8ActTrType = DATA;
        break;
    case OSAL_C_U16_AUDIO_MEDIA:
        e8ActTrType = AUDIO;
        break;
    case OSAL_C_U16_UNKNOWN_MEDIA:   // fall through
    default:                         // do nothing; INIT is OK
        break;
    }
    return e8ActTrType;
}


tVoid vdmmgr_tclcdctrlif::vHandleDiagCommandStart( tVoid )
{
    // Handle diagnosis drive test
    ETG_TRACE_USR1(( "vHandleDiagCommandStart( ): Start: Receive event to start diagnosis CD drive test" ));

    // switch for short or intensive drive test
    while(   ( _u32DiagDriveTestBitCmdField & VDMMGR_DIAG_CD_CMD_START_SHORT )
             || ( _u32DiagDriveTestBitCmdField & VDMMGR_DIAG_CD_CMD_START_INTENSIVE )
             || ( _u32DiagDriveTestBitCmdField & VDMMGR_DIAG_CD_CMD_START_SHORT_TEF_CD )
             )
    {
        if( _u32DiagDriveTestBitCmdField & VDMMGR_DIAG_CD_CMD_START_SHORT )
        {
            ETG_TRACE_USR1(( "vHandleDiagCommandStart( ): Set flag and start short read test" ));
            // Clear command flag
            _u32DiagDriveTestBitCmdField &= ~VDMMGR_DIAG_CD_CMD_START_SHORT;
            // Set status flag
            _u32DiagDriveTestBitStatusField |= VDMMGR_DIAG_CD_STATUS_TYPE_SHORT;
            // Start short test
            vDiagInitDriveTestResultStruct();
            vDiagShortDriveTest( TRUE );
        }
        else if( _u32DiagDriveTestBitCmdField & VDMMGR_DIAG_CD_CMD_START_INTENSIVE )
        {
            ETG_TRACE_USR1(( "vHandleDiagCommandStart( ): Set flag and start short read test" ));
            // Clear command flag
            _u32DiagDriveTestBitCmdField &= ~VDMMGR_DIAG_CD_CMD_START_INTENSIVE;
            // Set status flag
            _u32DiagDriveTestBitStatusField |= VDMMGR_DIAG_CD_STATUS_TYPE_INTENSIVE;
            // Start short test
            vDiagInitDriveTestResultStruct();
            vDiagIntensiveDriveTest( TRUE );
        }
        else if( _u32DiagDriveTestBitCmdField & VDMMGR_DIAG_CD_CMD_START_SHORT_TEF_CD )
        {
            ETG_TRACE_USR1(( "vHandleDiagCommandStart( ): Set flag and start short TEF CD read test" ));
            // Clear command flag
            _u32DiagDriveTestBitCmdField &= ~VDMMGR_DIAG_CD_CMD_START_SHORT_TEF_CD;
            // Set status flag
            _u32DiagDriveTestBitStatusField |= VDMMGR_DIAG_CD_STATUS_TYPE_SHORT_TEF_CD;
            // Start short test
            vDiagInitDriveTestResultStruct();
            vDiagShortDriveTest( TRUE );
        }
    };
    ETG_TRACE_USR1(( "vHandleDiagCommandStart( ): End" ));
}


tVoid vdmmgr_tclcdctrlif::vHandleDiagDriveShortTestContinue( tVoid )
{
    vDiagShortDriveTest( FALSE );
}


tVoid vdmmgr_tclcdctrlif::vHandleDiagDriveIntensiveTestContinue( tVoid )
{
    vDiagIntensiveDriveTest( FALSE );
}


tVoid vdmmgr_tclcdctrlif::vHandleDiagSelectTrackType( tVoid )
{
    tS32                    s32ReturnValue;
    OSAL_trCDROMTrackInfo   rCDRomTrackInfo;  // OSAL struct for track information
    T_e8_TrackType          e8ActTrType = e8DiagGetActualTrackType( );
    T_e16_TrackTypeResult   e8Result    = FAILED_TO_SELECT;

    ETG_TRACE_USR1(( "vHandleDiagSelectTrackType( ): Start" ));

    if( _poMain && _poMain->_vdmmgr_poCCAService )
    {
        // Production test. Mix-Mode CD with data and audio tracks - Get information about wanted track. Set first track -> data; second track -> audio
        rCDRomTrackInfo.u32TrackNumber = (_e8ReqTrackType == AUDIO) ? VDMMGR_DIAG_CD_TEF_CD_AUDIO_TRACK_NO: VDMMGR_DIAG_CD_TEF_CD_DATA_TRACK_NO ;
        // Set pick up to track 1 (-> Data) or track 2 (-> Audio). The cdctrl function 'get track info' positions the pick up at
        // the track 1/2 to get information. If the pick up is positioned above a data track the CD type change to DATA / AUDIO'.
        s32ReturnValue = OSAL_s32IOControl( _hDevice, OSAL_C_S32_IOCTRL_CDCTRL_GETTRACKINFO, (intptr_t)&rCDRomTrackInfo );
        if( s32ReturnValue == OSAL_OK )
        {
            tInt nCnt = 0;
            do
            {
                ETG_TRACE_USR4(( "vHandleDiagSelectTrackType( ): Waiting for new track type" ));
                (tVoid)OSAL_s32ThreadWait( VDMMGR_CDCDTRLIF_WAITTIME_TRACKTYPE );
                e8ActTrType = e8DiagGetActualTrackType( );
                nCnt++;
            }
            while(   (_e8ReqTrackType != e8ActTrType)             // check
                     && (nCnt < VDMMGR_DIAG_CD_WAIT_CNT_TRACKTYPE)   // wait for 4 * 500ms
                     );
        }
        // calculate the result
        e8Result = ( _e8ReqTrackType == e8ActTrType ) ? OK : FAILED_TO_SELECT;
        // Send the result to client
        _poMain->_vdmmgr_poCCAService->vSendMethodResultDiagSelectTrackType( e8ActTrType, _e8ReqTrackType, e8Result );
        // reset the requested track type
        _e8ReqTrackType = INIT;
    }
    ETG_TRACE_USR1(( "vHandleDiagSelectTrackType( ): End" ));
}


tVoid vdmmgr_tclcdctrlif::vDiagShortDriveTest( tBool bFirstPass )
{
    tU32              u32LBA;
    OSAL_tMSecond     MSecondTime1;
    OSAL_tMSecond     MSecondTime2;
    OSAL_tMSecond     MSecondDiff;
    tBool             bReadSuccess;
    tS32              s32Success;
    OSAL_tEventHandle hCbkEvent;

    ETG_TRACE_USR1(( "vDiagShortDriveTest( ): Start" ));

    if( bFirstPass )
    {
        // Init counter for first read
        _u8DriveTestCntReadData = VDMMGR_DIAG_CD_INIT_PASS_CNT;
    }

    // Get system time before read
    MSecondTime1 = OSAL_ClockGetElapsedTime( );
    // Get LBA dependent from test type ('Short' or 'Short TEF'
    // First check the index -> to satisfy lint
    if( _u8DriveTestCntReadData >= VDMMGR_DIAG_CD_CNT_LBA_SHORT_TEST )
    {
        // Set index to max. value
        _u8DriveTestCntReadData = VDMMGR_DIAG_CD_CNT_LBA_SHORT_TEST - 1;
    }
    if( _u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_TYPE_SHORT )
    {
        u32LBA = _au32DiagLBA_IMO[ _au8DiagOrderShortTest[_u8DriveTestCntReadData] ];
    }
    else // -> _u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_TYPE_SHORT_TEF_CD
    {
        u32LBA = _au32DiagLBA_IMO_TEF_CD[ _au8DiagOrderShortTest[_u8DriveTestCntReadData] ];
    }
    // Read (uncached) raw data
    bReadSuccess = bReadRawDataLBA( u32LBA, VDMMGR_DIAG_CD_SHORT_TEST_CNT_LBA, _ps8DiagReadMem );
    // Get system time after read
    MSecondTime2 = OSAL_ClockGetElapsedTime( );
    // Get time difference
    MSecondDiff = MSecondTime2 - MSecondTime1;
    // Calculate and set method result values with latest values
    vDiagCalcNewActualResult( VDMMGR_DIAG_CD_SHORT_TEST_CNT_LBA, 1, bReadSuccess, MSecondDiff );
    // Increase counter
    ++_u8DriveTestCntReadData;
    // check if all reads are done
    if( _u8DriveTestCntReadData >= VDMMGR_DIAG_CD_CNT_LBA_SHORT_TEST )
    {
        // Send result to diagnostics only if test is not already cancelled
        if(   !( _u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_CANCEL_IN_PROGRESS )
              && _poMain && _poMain->_vdmmgr_poCCAService && _poMain->_vdmmgr_poCCAClienthandlerdiaglog
              )
        {
            _poMain->_vdmmgr_poCCAService->vSendMethodResultDiagDrvTest( MMGR_RESULT_OK,
                                                                         ( (_u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_DUR_CONTINUOUS ) ? MMGR_CONTINUOUS : MMGR_SINGLE_RUN),
                                                                         &_DiagMethodResultDrvTest );
        }
        // single run or continuous?
        if( _u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_DUR_CONTINUOUS )
        {
            ETG_TRACE_USR1(( "vDiagShortDriveTest( ): all blocks read; start next run" ));
            // continuous run -> reset counter and go on
            _u8DriveTestCntReadData = VDMMGR_DIAG_CD_INIT_PASS_CNT;
            // 'Re'init result struct
            vDiagInitDriveTestResultStruct( );
        }
        else // if ( _u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_DUR_CONTINUOUS )
        {
            ETG_TRACE_USR1(( "vDiagShortDriveTest( ): all blocks read; stop read test" ));
            // only single started -> Set stop maker
            _u32DiagDriveTestBitCmdField |= VDMMGR_DIAG_CD_CMD_STOP_TEST;
        }
    }

    // Check for End of test
    if(   ( _u32DiagDriveTestBitCmdField & VDMMGR_DIAG_CD_CMD_STOP_TEST )   // stop/cancel marker was set?
          || ( _u32DiagDriveTestBitCmdField & VDMMGR_DIAG_CD_CMD_CANCEL_TEST )
          )
    {
        // If the drive test was canceled by diagnosis inform diagnosis that diagnosis has canceled the test
        if( _u32DiagDriveTestBitCmdField & VDMMGR_DIAG_CD_CMD_CANCEL_TEST )
        {
            // 'Re'init result struct
            vDiagInitDriveTestResultStruct( );
        }
        // Clear bit flag short test and stop/cancel
        _u32DiagDriveTestBitStatusField  &= ~VDMMGR_DIAG_CD_STATUS_TYPE_SHORT;
        _u32DiagDriveTestBitStatusField  &= ~VDMMGR_DIAG_CD_STATUS_TYPE_SHORT_TEF_CD;
        _u32DiagDriveTestBitCmdField     &= ~VDMMGR_DIAG_CD_CMD_STOP_TEST;
        _u32DiagDriveTestBitCmdField     &= ~VDMMGR_DIAG_CD_CMD_CANCEL_TEST;
        // Clear 'test running' flag
        _u32DiagDriveTestBitStatusField  &= ~VDMMGR_DIAG_CD_STATUS_TESTRUNING;
        _u32DiagDriveTestBitStatusField  &= ~VDMMGR_DIAG_CD_STATUS_CANCEL_IN_PROGRESS;
    }
    else
    {
        // Open, send and close event to start next read data
        if( OSAL_s32EventOpen( VDMMGR_CDCTRLIF_EVENTNAME, &hCbkEvent ) == OSAL_OK )
        {
            s32Success = OSAL_s32EventPost( hCbkEvent, VDMMGR_EVENT_MASK_CMD_DIAG_CONTINUE_CD_DRV_SHORT_TEST, OSAL_EN_EVENTMASK_OR );
            if( s32Success == OSAL_ERROR )
            {
                ETG_TRACE_ERR(( "vDiagShortDriveTest( ): Error: OSAL_s32EventPost( ) returns with error" ));
            }

            s32Success = OSAL_s32EventClose( hCbkEvent );
            if(s32Success == OSAL_ERROR)
            {
                ETG_TRACE_ERR(( "vDiagShortDriveTest( ): Error: OSAL_s32EventClose() returns with error" ));
            }
        }
    }
    ETG_TRACE_USR1(( "vDiagShortDriveTest( ): End" ));
}


tVoid vdmmgr_tclcdctrlif::vDiagIntensiveDriveTest( tBool bFirstPass )
{
    tU16              u16BlocksPerRead;
    OSAL_tMSecond     MSecondTime1;
    OSAL_tMSecond     MSecondTime2;
    OSAL_tMSecond     MSecondDiff;
    tBool             bReadSuccess;
    tBool             bLastReadAttempt;
    tS32              s32Success;
    OSAL_tEventHandle hCbkEvent;

    ETG_TRACE_USR1(( "vDiagIntensiveDriveTest( ): Start" ));

    // Init stop marker;
    bLastReadAttempt = FALSE;
    // Init count LBA per read attempt
    u16BlocksPerRead = _u16DiagMaxBlocksPerReadAttempt;
    // Init only before first pass
    if( bFirstPass )
    {
        // Init start LBA
        _u32DriveTestStartLBA = VDMMGR_DIAG_CD_INIT;
    }

    // Check count LBA per read attempt -> 'should be read' >= 'could be read'
    if( u16BlocksPerRead >= (_u32DiagMaxLBA - _u32DriveTestStartLBA) )
    {
        // Set new count LBA per read attempt
        u16BlocksPerRead = (tU16)(_u32DiagMaxLBA - _u32DriveTestStartLBA);
        // Set flag for last read attempt
        bLastReadAttempt = TRUE;
        ETG_TRACE_USR1(( "vDiagIntensiveDriveTest( ): set new count of blocks to read %d before the last read attempt", u16BlocksPerRead ));
    }
    // Get system time before read
    MSecondTime1 = OSAL_ClockGetElapsedTime( );
    // Read (not cached) raw data
    bReadSuccess = bReadRawDataLBA( _u32DriveTestStartLBA, u16BlocksPerRead, _ps8DiagReadMem );
    // Get system time after read
    MSecondTime2 = OSAL_ClockGetElapsedTime( );
    // Get time difference
    MSecondDiff = MSecondTime2 - MSecondTime1;
    // Calculate and set method result values with latest values
    vDiagCalcNewActualResult( u16BlocksPerRead, 1, bReadSuccess, MSecondDiff );

    // Check if all possible blocks are read
    if( bLastReadAttempt )
    {
        // Send result to diagnostics
        // This should not come if test is cancelled
        if(   !( _u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_CANCEL_IN_PROGRESS )
              && _poMain &&_poMain->_vdmmgr_poCCAService && _poMain->_vdmmgr_poCCAClienthandlerdiaglog
              )
        {
            _poMain->_vdmmgr_poCCAService->vSendMethodResultDiagDrvTest( MMGR_RESULT_OK,
                                                                         ( (_u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_DUR_CONTINUOUS ) ? MMGR_CONTINUOUS : MMGR_SINGLE_RUN),
                                                                         &_DiagMethodResultDrvTest );
        }
        // single run or continuous?
        if( _u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_DUR_CONTINUOUS )
        {
            ETG_TRACE_USR1(( "vDiagIntensiveDriveTest( ): whole media read; start next run" ));
            // continuous run -> reset start LBA, 'blocks to read' and go on
            _u32DriveTestStartLBA = VDMMGR_DIAG_CD_INIT;
            // 'Re'init result struct
            vDiagInitDriveTestResultStruct( );
        }
        else // if ( _u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_DUR_CONTINUOUS )
        {
            ETG_TRACE_USR1(( "vDiagIntensiveDriveTest( ): whole media read; stop read test" ));
            // only single started -> Set stop maker
            _u32DiagDriveTestBitCmdField |= VDMMGR_DIAG_CD_CMD_STOP_TEST;
        }
    }
    // Increase start LBA for next read attempt
    _u32DriveTestStartLBA += u16BlocksPerRead;

    // Check if the drive test is over
    if(      ( _u32DiagDriveTestBitCmdField & VDMMGR_DIAG_CD_CMD_STOP_TEST )   // until stop/cancel marker wasn't set
             || ( _u32DiagDriveTestBitCmdField & VDMMGR_DIAG_CD_CMD_CANCEL_TEST )
             )
    {
        // Drive test is over
        // If the drive test was canceled by diagnosis inform diagnosis that diagnosis has canceled the test
        if( _u32DiagDriveTestBitCmdField & VDMMGR_DIAG_CD_CMD_CANCEL_TEST )
        {
            // 'Re'init result struct
            vDiagInitDriveTestResultStruct( );
        }
        // Clear bit flag short test and stop/cancel
        _u32DiagDriveTestBitStatusField  &= ~VDMMGR_DIAG_CD_STATUS_TYPE_INTENSIVE;
        _u32DiagDriveTestBitCmdField     &= ~VDMMGR_DIAG_CD_CMD_STOP_TEST;
        _u32DiagDriveTestBitCmdField     &= ~VDMMGR_DIAG_CD_CMD_CANCEL_TEST;
        // Clear 'test running' flag
        _u32DiagDriveTestBitStatusField  &= ~VDMMGR_DIAG_CD_STATUS_TESTRUNING;
        _u32DiagDriveTestBitStatusField  &= ~VDMMGR_DIAG_CD_STATUS_CANCEL_IN_PROGRESS;
    }
    else
    {
        // Open, send and close event to start next read data
        if( OSAL_s32EventOpen( VDMMGR_CDCTRLIF_EVENTNAME, &hCbkEvent ) == OSAL_OK )
        {
            s32Success = OSAL_s32EventPost( hCbkEvent, VDMMGR_EVENT_MASK_CMD_DIAG_CONTINUE_CD_DRV_INT_TEST, OSAL_EN_EVENTMASK_OR );
            if( s32Success == OSAL_ERROR )
            {
                ETG_TRACE_ERR(( "vDiagIntensiveDriveTest( ): Error: OSAL_s32EventPost( ) returns with error" ));
            }

            s32Success = OSAL_s32EventClose( hCbkEvent );
            if(s32Success == OSAL_ERROR)
            {
                ETG_TRACE_ERR(( "vDiagIntensiveDriveTest( ): Error: OSAL_s32EventClose() returns with error" ));
            }
        }
    }
    ETG_TRACE_USR1(( "vDiagIntensiveDriveTest( ): End" ));
}


tVoid vdmmgr_tclcdctrlif::vDiagCalcNewActualResult( tU32 u32NewReadLBA, tU8 u8NewReadAttempts, 
                                                    tBool bReadSuccess, OSAL_tMSecond MSecondReadTime )
{
    ETG_TRACE_USR1(( "vDiagCalcNewActualResult( ): Start: read blocks: %d, new read attempts: %d, read succes: %d, read time: %d [ms]",
                     u32NewReadLBA, u8NewReadAttempts, ETG_ENUM(BOOL, bReadSuccess), MSecondReadTime ));
    // Evaluate test type
    if( _u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_TYPE_SHORT )
    {
        _DiagMethodResultDrvTest.e16DiagDrvTesttype = MMGR_SHORT;
    }
    else if( _u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_TYPE_INTENSIVE )
    {
        _DiagMethodResultDrvTest.e16DiagDrvTesttype = MMGR_INTENSIVE;
    }
    else if( _u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_TYPE_SHORT_TEF_CD )
    {
        _DiagMethodResultDrvTest.e16DiagDrvTesttype = MMGR_SHORT_TEF_CD;
    }
    else
    {
        ETG_TRACE_ERR(( "vDiagCalcNewActualResult( ): vUnknown value (or no value) set for test type  %d", _u32DiagDriveTestBitStatusField ));
    }
    _DiagMethodResultDrvTest.u32SectorsRead            += u32NewReadLBA;
    _DiagMethodResultDrvTest.u16ReadAttempts           += u8NewReadAttempts;
    _DiagMethodResultDrvTest.u32TotalReadTime          += MSecondReadTime;
    if( _DiagMethodResultDrvTest.u32SectorsRead )
    {
        _DiagMethodResultDrvTest.u16AverageReadtimeSector  = (tU16)(_DiagMethodResultDrvTest.u32TotalReadTime /
                                                                    _DiagMethodResultDrvTest.u32SectorsRead);
    }
    // Check for max. readtime
    if( u32NewReadLBA )
    {
        if( (MSecondReadTime / u32NewReadLBA) > _DiagMethodResultDrvTest.u16MaxReadTimePerSector )
        {
            _DiagMethodResultDrvTest.u16MaxReadTimePerSector = (tU16)(MSecondReadTime / u32NewReadLBA);
        }
    }
    // Check for errors
    if( bReadSuccess == FALSE )
    {
        _DiagMethodResultDrvTest.u16CntErrors ++;
    }
    ETG_TRACE_USR1(( "vDiagCalcNewActualResult( ): End: new result; Sectors read: %d, read attempts: %d, total read time: %d [ms], Average read/sector: %d [ms], max. read time/sector: %d [ms], error count: %d",
                     _DiagMethodResultDrvTest.u32SectorsRead, _DiagMethodResultDrvTest.u16ReadAttempts, _DiagMethodResultDrvTest.u32TotalReadTime,
                     _DiagMethodResultDrvTest.u16AverageReadtimeSector, _DiagMethodResultDrvTest.u16MaxReadTimePerSector, _DiagMethodResultDrvTest.u16CntErrors ));
}


tVoid vdmmgr_tclcdctrlif::vDiagCalcNewActualResultForMethodCancel( )
{
    tBool    bAllreadyCalled = _DiagMethodResultDrvTest.u32TotalReadTime != VDMMGR_DIAG_CD_RESULT_INIT;      // Lint requests this
    ETG_TRACE_USR1(( "vDiagCalcNewActualResultForMethodCancel( ): vDiagCalcNewActualResult already called: %d", ETG_ENUM(BOOL, bAllreadyCalled) ));

    if( _u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_TYPE_SHORT )
    {
        _DiagMethodResultDrvTest.e16DiagDrvTesttype = MMGR_SHORT;
    }
    else if( _u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_TYPE_INTENSIVE )
    {
        _DiagMethodResultDrvTest.e16DiagDrvTesttype = MMGR_INTENSIVE;
    }
    else if( _u32DiagDriveTestBitStatusField & VDMMGR_DIAG_CD_STATUS_TYPE_SHORT_TEF_CD )
    {
        _DiagMethodResultDrvTest.e16DiagDrvTesttype = MMGR_SHORT_TEF_CD;
    }
}


tVoid vdmmgr_tclcdctrlif::vDiagGetFirmwareVersion( T_DiagFirmwareVersion* pDiagFirmwareVersion )
{
    OSAL_trDriveVersion  rDriveVersion;
    tU8                  u8Cnt;

    ETG_TRACE_USR1(( "vDiagGetFirmwareVersion( ): Start" ));

    // Init result array
    for( u8Cnt = 0; u8Cnt< MMGR_MAX_FIRMWAREVERSIONLENGHT; u8Cnt++ )
    {
        pDiagFirmwareVersion->au8FirmwareVersion[u8Cnt] = VDMMGR_DIAG_CD_INIT;
    }
    // Try to get version
    if( bGetDriveVersion( &rDriveVersion ) )
    {
        // Copy drive version
        for( u8Cnt = 0; u8Cnt< MMGR_MAX_FIRMWAREVERSIONLENGHT; u8Cnt++ )
        {
            pDiagFirmwareVersion->au8FirmwareVersion[u8Cnt] = rDriveVersion.au8FirmwareRevision[u8Cnt];
        }
    }
}


mplay_fi_tcl_DiagTestResult::tenType vdmmgr_tclcdctrlif::e8DiagCDInterfaceTest(
        mplay_fi_tcl_e8_DiagnosisTestType::tenType e8TestInitiator,
        mplay_fi_tcl_e16_DiagTestType::tenType e16TestType
        )
{
    OSAL_trDriveVersion  rDriveVersion;
#ifdef USE_REPLACE_E8_TESTRESULT
    midw_fi_tcl_e8_TestResult::tenType e8DiagITCTestResult;
#else
    midw_fi_tcl_TestResult::tenType e8DiagITCTestResult;
#endif
    mplay_fi_tcl_DiagTestResult::tenType e8DiagTestResult;

    ETG_TRACE_USR1(( "e8DiagCDInterfaceTest( ): e8TestInitiator=%d, e16TestType=%d", e8TestInitiator, e16TestType ));

    if( bGetDriveVersion( &rDriveVersion ) == TRUE )
    {
        if( e16TestType == mplay_fi_tcl_e16_DiagTestType::FI_EN_MMGR_DIAGNOSIS_CONTINUOUS )
        {
#ifdef USE_REPLACE_E8_TESTRESULT
            e8DiagITCTestResult = midw_fi_tcl_e8_TestResult::FI_EN_PASSEDDIAGNOSTIC;
#else
            e8DiagITCTestResult = midw_fi_tcl_TestResult::FI_EN_PASSEDDIAGNOSTIC;
#endif
        }
        else
        {
#ifdef USE_REPLACE_E8_TESTRESULT
            e8DiagITCTestResult = midw_fi_tcl_e8_TestResult::FI_EN_PASSED;
#else
            e8DiagITCTestResult = midw_fi_tcl_TestResult::FI_EN_PASSED;
#endif
        }

        e8DiagTestResult = mplay_fi_tcl_DiagTestResult::FI_EN_TEST_PASSED;
    }
    else
    {
        if( e16TestType == mplay_fi_tcl_e16_DiagTestType::FI_EN_MMGR_DIAGNOSIS_CONTINUOUS )
        {
#ifdef USE_REPLACE_E8_TESTRESULT
            e8DiagITCTestResult = midw_fi_tcl_e8_TestResult::FI_EN_FAILEDDIAGNOSTIC;
#else
            e8DiagITCTestResult = midw_fi_tcl_TestResult::FI_EN_FAILEDDIAGNOSTIC;
#endif
        }
        else
        {
#ifdef USE_REPLACE_E8_TESTRESULT
            e8DiagITCTestResult = midw_fi_tcl_e8_TestResult::FI_EN_FAILED;
#else
            e8DiagITCTestResult = midw_fi_tcl_TestResult::FI_EN_FAILED;
#endif
        }
        e8DiagTestResult = mplay_fi_tcl_DiagTestResult::FI_EN_TEST_FAILED;
    }

    ETG_TRACE_USR1(( "e8DiagCDInterfaceTest( ): End. e8DiagTestResult=%d, e8DiagITCTestResult=%d", e8DiagTestResult, ETG_ENUM(DIAGLOG_STATUS, e8DiagITCTestResult) ));

    return e8DiagTestResult;
}


tVoid vdmmgr_tclcdctrlif::vDiagGetDriveVersion( T_DiagDriveVersion* pDiagDriveVersion )
{
    OSAL_trDriveVersion  rDriveVersion;
    tU8                  u8Cnt;

    ETG_TRACE_USR4(( "vDiagGetDriveVersion( ): Start" ));

    // Init result array
    for( u8Cnt = 0; u8Cnt< MMGR_MAX_DRIVEVERSIONLENGTH; u8Cnt++)
    {
        pDiagDriveVersion->au8DriveVersion[u8Cnt] = VDMMGR_DIAG_CD_INIT;
    }
    // Try to get version
    if( bGetDriveVersion( &rDriveVersion ) )
    {
        // Copy drive version
        for( u8Cnt = 0; u8Cnt< MMGR_MAX_DRIVEVERSIONLENGTH; u8Cnt++ )
        {
            pDiagDriveVersion->au8DriveVersion[u8Cnt] = rDriveVersion.au8ModelNumber[u8Cnt];
        }
    }
}


tU16 vdmmgr_tclcdctrlif::u16GetTotalFailure( tVoid ) const
{
    return _u16TotalFailure;
}


tVoid vdmmgr_tclcdctrlif::vSetTotalFailureForSimulation( tU16 u16TotalFailure ) const
{
    _u16TotalFailure = u16TotalFailure;
}


tVoid vdmmgr_tclcdctrlif::vSetMediaChangeForSimulation( tU16 u16MediaChange ) const
{
    _u16MediaChange = u16MediaChange;
}


tVoid vdmmgr_tclcdctrlif::vSetMediaStateForSimulation( tU16 u16MediaState ) const
{
    _u16MediaState = u16MediaState;
}


tVoid vdmmgr_tclcdctrlif::vSetDefectForSimulation( tU16 u16Defect ) const
{
    _u16Defect = u16Defect;
}


tVoid vdmmgr_tclcdctrlif::vSetDeviceStateForSimulation( tU16 u16DeviceState ) const
{
    _u16DeviceState = u16DeviceState;
}


tVoid vdmmgr_tclcdctrlif::vSetFirstCdTypeValueArrived( tBool  bArrived )
{
    _bFirstCdTypeValueArrived = bArrived;
}


tBool vdmmgr_tclcdctrlif::vGetFirstCdTypeValueArrived( tVoid ) const
{
    return _bFirstCdTypeValueArrived;
}


tVoid vdmmgr_tclcdctrlif::vRestartHandleMediaChange( tVoid ) const
{
    OSAL_tEventHandle hCbkEvent;

    if( OSAL_s32EventOpen( VDMMGR_CDCTRLIF_EVENTNAME, &hCbkEvent ) == OSAL_OK )
    {
        // Send event for mask MediaChange
        if( OSAL_s32EventPost( hCbkEvent, VDMMGR_EVENT_MASK_MEDIA_CHANGE, OSAL_EN_EVENTMASK_OR ) != OSAL_ERROR )
        {
            if( OSAL_s32EventClose( hCbkEvent ) == OSAL_ERROR )
            {
                TRACE_OSAL_ERROR( VDMMGR_TR_CDCTRLIF );   // Get error code if not successful.
            }
        }
        else
        {
            TRACE_OSAL_ERROR( VDMMGR_TR_CDCTRLIF );   // Get error code if not successful.
        }
    }
    else
    {
        TRACE_OSAL_ERROR( VDMMGR_TR_CDCTRLIF );   // Get error code if not successful.
    }
}


tVoid vdmmgr_tclcdctrlif::vPrintCDState( )
{
    if( _poMain && _poMain->_vdmmgr_poCCAService )
    {
        T_e8DriveTempSensorStatus  DriveTempSensorStatus;
        ETG_TRACE_USR1(( "vPrintCDState( ): +++ CD Properties +++\nCDType:        %d\nMedia state:   %d\nInsert state:  %d\nCDLoaderState: %d\nCD Drive temp: %d\nCD Drive temp sensor state: %d",
                         ETG_ENUM(CD_TYPE, _poMain->_vdmmgr_poCCAService->enGetActCdType()),                ETG_ENUM(MEDIA_STATE_CODE, _poMain->_vdmmgr_poCCAService->enGetActCDState()),
                         ETG_ENUM(CD_INSERT_STATE, _poMain->_vdmmgr_poCCAService->enGetActCDInsertState()), ETG_ENUM(LOADER_STATE, e8GetLoaderState()),
                         s16GetTemperature( &DriveTempSensorStatus ), ETG_ENUM(DRIVE_SENSOR_STATE, DriveTempSensorStatus)  ));
    }
}


tVoid vdmmgr_tclcdctrlif::vSendEventToCdCtrlThread( tU32 u32Event ) const
{
    tS32              s32Success;
    OSAL_tEventHandle hCbkEvent;

    ETG_TRACE_USR1(( "vSendEventToCdCtrlThread( ): Start: Event: %d", ETG_ENUM(MMGR_CD_EVENT, u32Event) ));
    if( OSAL_s32EventOpen( VDMMGR_CDCTRLIF_EVENTNAME, &hCbkEvent ) == OSAL_OK )
    {
        s32Success = OSAL_s32EventPost( hCbkEvent, u32Event, OSAL_EN_EVENTMASK_OR );
        if( s32Success == OSAL_ERROR )
        {
            ETG_TRACE_ERR(( "vSendEventToCdCtrlThread( ): Error sending event %d", ETG_ENUM(MMGR_CD_EVENT, u32Event) ));
        }

        s32Success = OSAL_s32EventClose( hCbkEvent );
        if(s32Success == OSAL_ERROR)
        {
            ETG_TRACE_ERR(( "vSendEventToCdCtrlThread( ): Error closing event handle" ));
        }
    }
    else
    {
        ETG_TRACE_ERR(( "vSendEventToCdCtrlThread( ): Error opening event handle" ));
        TRACE_OSAL_ERROR( VDMMGR_TR_CLIENTSPM );
    }
}

