/*-----------------------------------------------------------------------------*
 * VD_DeviceManager_service_DeviceManager.h                                    *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*! 
 * \file VD_DeviceManager_service_DeviceManager.h
 *
 * \brief  This file holds the Services for DeviceManager
 *
 * \version Initial Version
 * \version 05.08.2011, Koechling, Christian (Bosch), Interface VD_DeviceManager changed
 * \version 01.11.2011, Koechling, Christian (Bosch),
 *          -# added function to enable statetable to inform clients
 *          -# support for trace commands to see last Msg send to clients
 * \version 17.11.2011, Koechling, Christian (Bosch), cleanup: 1st step to shift
 *          includes to Common.h
 * \version 21.11.2011, Koechling, Christian (Bosch), cleanup includes and
 *          intermediate defines which marked bugfixes
 * \version 06.12.2011, Koechling, Christian (Bosch), cleanup
 * \version 23.02.2012, Koechling, Christian (Bosch), extend status property to
 *          send with index>0 all connected devices
 * \version 10.03.2012, Koechling, Christian (Bosch), cleanup function to trace
 *          history of messages send to clients
 * \version 22.03.2012, Negi, Sunder (Montavista), Remove LOGs
 * \version 02.06.2012, Koechling, Christian (Bosch),
 *          -# added mechanism to use central device list manager,
 *          -# improved critical voltage behaviour and
 *          -# state machine to send combined messages for output of the first
 *          state machine
 * \version 19.07.2012, Koechling, Christian (Bosch), add mechanism to log
 *          histories for analysis to USB-stick
 * \version 27.07.2012, Negi, Sunder (MontaVista), Cleaned up file hierarchy
 * \version 02.08.2012, Negi, Sunder (MontaVista),
 *          -# Refactored debug information logging
 *          -# Added History Manager, refer define HISTORY_MANAGER
 * \version 05.08.2012, Negi, Sunder (MontaVista), Modified and fixed doxygen comments
 * \version 12.08.2012, Negi, Sunder (MontaVista), cleanup HISTORY_MANAGER
 * \version 13.08.2012, Negi, Sunder (MontaVista), cleanup logging to usb (WRITEHISTORYTOUSBSTICK)
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH 2010-2016
 */

#ifndef _VD_DEVICEMANAGER_SERVICE_DEVICEMANAGER_H_
#define _VD_DEVICEMANAGER_SERVICE_DEVICEMANAGER_H_

#include "Config.h"

#define VD_DEVICEMANAGER_S_IMPORT_INTERFACE_MSG
#include "VD_DeviceManager_interface.h"

#include "Enums.h"


// Loop back message defines to be processed by vOnLoopbackService().

/* Definition of internal Function IDs (iFIDs) and message        */
/* content IDs (MSGID) which are not public and therefore are     */
/* not in the public FI of our service. Such internal function    */
/* and message content IDs are necessary for 'loopback' messages, */
/* i.e. messages the application sends itself. These internal     */
/* function IDs must be different to the ones from the public FI. */
/* The developer has to take care about that. To have a maximum   */
/* separation of the public and the internal FIDs we start the    */
/* definition for the internal ones nearly at the end of the      */
/* possible range and then count downwards.*/

// Internal function ID defines.
#define DEVICEMANAGER_C_U16_IFID_EXAMPLE_1               0xFFF0
#define DEVICEMANAGER_C_U16_IFID_INTERNAL_CYCLICDIAG_MSG 0xFFF1


// Message content ID defines.
#define DEVICEMANAGER_C_U32_MSGID_EXAMPLE_1               0x00000001
#define DEVICEMANAGER_C_U32_MSGID_INTERNAL_CYCLICDIAG_MSG 0x00000002


/*-----------------------------------------------------------------------------*
 * Namespaces                                                                  *
 *-----------------------------------------------------------------------------*/
using namespace statetbl;
using namespace servicedvm;

/*-----------------------------------------------------------------------------*
 * Forward decelerations                                                       *
 *-----------------------------------------------------------------------------*/
class CDevice;
class StateTable;
class ahl_tclBaseOneThreadService;

/*-----------------------------------------------------------------------------*
 * Class declaration                                                           *
 *-----------------------------------------------------------------------------*/
/*!
 * \class vd_devicemanager_tclService_DeviceManager
 *
 * \brief VD_DeviceManager service
 *
 * Provides service to all the other F-Blocks by providing required information
 *
 * \see ahl_tclBaseOneThreadService
 */
class vd_devicemanager_tclService_DeviceManager :public ahl_tclBaseOneThreadService
{
public:

    tBool m_bConnectClients;

    /*!
         * \brief constructor
         *
         * Create an object of the base class ahl_tclBaseOneThreadService with
         * a pointer to this application, the service identifier, the application
         * identifier and the service version as parameters.
         *
         * \see ahl_tclBaseOneThreadService::ahl_tclBaseOneThreadService()
         */
    vd_devicemanager_tclService_DeviceManager();

    /*!
         * \brief Parameterized constructor
         *
         * \param [in] poMainAppl the main application
         */
    vd_devicemanager_tclService_DeviceManager(
            vd_devicemanager_tclApp* poMainAppl);

    /*!
         * \brief Destructor.
         */
    virtual ~vd_devicemanager_tclService_DeviceManager();

    /*!
         * \brief the loopback function
         *
         * This function is called from the vOnLoopback() function of this CCA
         * application if a message from one of this applications services is
         * received (so called self or loopback messages).
         *
         * \param [in] poMessage    message object.
         */
    tVoid vOnLoopbackService(amt_tclServiceData* poMessage) const;

    /*!
         * \brief timer function
         *
         * This function is called from the vOnTimer() function of this CCA
         * application on the expiration of a previously via function
         * bStartTimer() started timer.
         *
         * \param [in] u16TimerId Identifier of the expired timer.
         *
         * \see bStarttimer()
         */
    tVoid vProcessTimer(tU16 u16TimerId);

    //#ifdef REFACTOR_STATETABLE_OUTPUT
    /*!
          * \brief Notify the client about connection of new device, set undervoltage flag true/false, Send list with HW signals of each port
          *
          */
    OUT tVoid RequestDeviceManagerNotifyConnectionStatus();
    //#endif



    /*!
         * \brief Notify the client about connection of new device
         *
         * \param [in] f_pNotifyClients the device object
         * \param [in] pStateTable      pointer to StateTable
         */
    OUT tVoid RequestDeviceManagerNotifyConnectionStatus(
            const structNotifyClients *f_pNotifyClients,
            const StateTable *pStateTable); //used via StateTable //adjust for Roadmap 13001_ExtendedUserInformation

    /*!
         * \brief Notify the client about changes for SD Cards
         */
    OUT tVoid RequestDeviceManagerNotifyStatusCard( ); //used via StateTable

    /*!
         * \brief Create Property DevManager object of any device
         *
         * \param [in] f_pCurPropertyDevice the device object
         *
         * \return [FIDEVMGR_DEVICEPROPERTY] the framework
         *         specific device object
         *
         */
    FIDEVMGR_DEVICEPROPERTY CreateDevicePropertyObject(
            const CDevice* f_pCurPropertyDevice) const;

    /*!
        * \brief Create DevManager_PortStates from input parameter
        *
        * \param [in] f_rPortState
        *
        * \return [FIDEVMGR_PORTSTATES] the framework
        *         specific device object
        *
        */
    FIDEVMGR_PORTSTATES  CreatePortListObject
    (const trPortStates& f_rPortState) const;


    tVoid CreateDeviceSDCardPropertyObject( FIDEVMGR_SDCARDINFO* f_pSDCard , const CDeviceCard* f_pSDCardDevice ) const;

    /*!
         * \brief enable/disable connection to clients
         *
         * \param [in] bConnect TRUE means connect, FALSE means disconnect
         */
    tVoid connectClients(tBool bConnect);

    /*!
         * \brief function to set service availability
         *
         * \param [in] bAvailable TRUE means available, FALSE means unavailable
         */
    tVoid vSetServiceAvailable(tBool bAvailable);

    /*!
        * \brief Handler function for CCA message
        *        DEVMGR_MOUNTCDROMINTERNAL
        *
        *  used by DEviceManagerInterface to return result to FC_Diagnosis
        *
        * \param [in] poMessage Message object
        */
    OUT tVoid vHandle_DeviceManagerMountCDRomInternal_MethodReturn(tBool bSuccess);

    /*!
        * \brief Handler function for CCA message
        *        DEVMGR_EJECTOPTICALDISC
        *
        *  used by DEviceManagerInterface to send the  request to client handler media manager
        *
        * \param [in] poMessage Message object
        */

    IN tVoid vHandle_DeviceManagerOpticalDiscEject_MethodStart(amt_tclServiceData* poMessage); //Roadmap 15002 CD Audio device Detection

    /*!
        * \brief to send the cdda device slot status to interface
        *
        *\param [IN]tDMOpticalDiscSlotState Optical Disc Slot state
        *
        */

    IN tVoid RequestDeviceManagerNotifyOpticalDiscSloteState(IN const tDMOpticalDiscSlotState  eOpticalDiscSlotState);


    /*!
        * \brief to send the cdda device CD Info to interface
        *
        *\param [IN]trOpticalDiskCDInfo Optical Disc Slot state
        *
        */

    IN tVoid RequestDeviceManagerNotifyOpticalDiscCDInfo(IN const trOpticalDiskCDInfo  f_OpticalDiskCDInfo);

    /*!
        * \brief Handler function for CCA message
        *        DEVMGR_EJECTOPTICALDISC
        *
        *
        *  used by DEviceManagerInterface to send the  method result of client handler media manager
        *
        * \param [in] e8CDEjectResult contains the info whether the Mediamanager accepted, reject or not ready to accept the cd eject
        */
    IN tVoid vSendOpticalDiscEjectMethodResult(IN tOpticalDiscEjectMethodResult e8OpticalDiscEjectResult);//Roadmap 15002 CD Audio device Detection

protected:

    /*---------------------------------------------------------------------*
         * Overriding ahl_tclBaseOneThreadService methods                      *
         *---------------------------------------------------------------------*/

    /*!
         * \brief This function is called by the CCA framework when the service
         *        which is offered by this server has become available.
         *
         * \see ahl_tclBaseOneThreadService::vOnServiceAvailable();
         */
    virtual tVoid vOnServiceAvailable();

    /*!
         * \brief This function is called by the CCA framework when the service
         *        which is offered by this server has become unavailable.
         *
         * \see ahl_tclBaseOneThreadService::vOnServiceAvailable();
         */
    virtual tVoid vOnServiceUnavailable();

    /*!
         * \brief Provides information about properties
         *
         * This function is called by the CCA framework to request ANY property
         * which is offered by this service. For each property accessed via
         * parameter 'u16FunctionId' the user has to prepare the corresponding
         * FI data object and to copy it to the referenced service data object
         * 'roOutMsg'.
         *
         * \param [in]  u16FunctionId Function ID of the requested property.
         * \param [out] roOutMsg      the service data object to which the
         *              content of the prepared FI data object should be copied to.
         * \param [in]  poInMsg       Selector message which is used to select
         *              dedicated content to be copied to 'roOutMsg' instead of
         *              updating the entire FI data object.
         *
         * \return -# TRUE, For the requested property the FI data object was
         *         successfully created and copied to the referenced service
         *         data object 'roOutMsg'. The CCA framework sends a 'STATUS'
         *         message of the updated property to each registered clients.
         *         -# FALSE, Failed to create the FI data object for the requested
         *         property and to update the referenced service data object
         *         'roOutMsg' with the new property content. The CCA framework
         *         sends an error message to the requesting client.
         */
    virtual tBool bStatusMessageFactory(tU16 u16FunctionId,
                                        amt_tclServiceData& roOutMsg, amt_tclServiceData* poInMsg);

    /*!
         * \brief Set a property
         *
         * This function is called by the CCA framework when it has received a
         * message for a property with Opcode 'SET' or 'PURESET' and there is no
         * dedicated handler function defined in the message map for this pair
         * of FID and opcode. The user has to set the application specific
         * property to the requested value and the CCA framework then cares
         * about informing the requesting client as well as other registered
         * clients.
         *
         * \param [in]  poMessage        Property to be set.
         * \param [out] bPropertyChanged Property changed flag to be set to TRUE
         *              if property has changed. Otherwise to be set to FALSE
         *              (default).
         * \param [out] u16ErrorCode     Error code to be set if a CCA error
         *              occurs, otherwise don't touch.
         *
         * \return -# TRUE, Property successfully set to the passed value. The
         *         CCA framework sends a 'STATUS' message to the requesting
         *         client if the opcode was 'SET'. Othwerwise if opcode was
         *         'PURESET' there is no 'STATUS' message sent to the requesting
         *         client. The CCA framework will send a 'STATUS' message to
         *         other registered clients as well if the [OUT] parameter
         *         'bPropertyChanged' is set to TRUE.
         *         -# FALSE, Failed to set the property to the passed value. The
         *         CCA framework sends an error message to the requesting client.
         */
    virtual tBool bProcessSet(amt_tclServiceData* poMessage,
                              tBool& bPropertyChanged, tU16& u16ErrorCode);

    /*---------------------------------------------------------------------*
         * Handler function declarations used by message map                   *
         *---------------------------------------------------------------------*/

    /*!
         * \brief Handler function for CCA message
         *        MIDW_DEVICEMANAGERFI_C_U16_DEVMANAGERGETDEVPROPERTIES
         *
         * Provides information about the device whose information is requested
         * by the client
         *
         * \param [in] poMessage Message object
         */
    IN  tVoid vHandle_DeviceManagerGetDeviceProperties_MethodStart(amt_tclServiceData* poMessage);
    OUT tVoid vHandle_DeviceManagerGetDeviceProperties_MethodReturn(const trReceiverInfo &f_oReceiverInfo, CDevice *f_pCDeviceProperty);

    /*!
         * \brief Handler function for CCA message
         *        DEVMGR_GETALLCONNECTEDDEVICES
         *
         * Provides information about all the connected devices when requested
         * by a client
         *
         * \param [in] poMessage Message object
         */
    IN  tVoid vHandle_DeviceManagerGetAllConnectedDevices_MethodStart(amt_tclServiceData* poMessage); //Result method is public see above!

    OUT tVoid vHandle_DeviceManagerGetAllConnectedDevices_MethodReturn(const trReceiverInfo &f_oReceiverInfo,
                                                                       vector<structNotifyClients> &f_vClientsAllConnected,
                                                                       const StateTable *f_pStateTableManagerApp);

    /*!
            * \brief Handler function for CCA message
            *        DEVMGR_MOUNTCDROMINTERNAL
            *
            *  used by FC_Diagnosis to mount a mixed mode cd
            *
            * \param [in] poMessage Message object
            */
    IN tVoid vHandle_DeviceManagerMountCDRomInternal_MethodStart( amt_tclServiceData* poMessage); //Roadmap 14002 MountCDROM MixMode Diag


    /*!
             * \brief Handler function for CCA message
             *        DEVMGR_ADDTELEMATICSBOX
             *
             *  used by projects to add a new telematics unit to the  VD_Devicemanager
             *
             * \param [in] poMessage Message object
             */
     IN tVoid vHandle_DeviceManagerAddTelematicsBox_MethodStart( amt_tclServiceData* poMessage);

     OUT tVoid vHandle_DeviceManagerAddTelematicsBox_MethodReturn( const trReceiverInfo &f_oReceiverInfo, tBool f_bSuccess);


     /*!
              * \brief Handler function for CCA message
              *        DEVMGR_GETCONFIGURATIONVALUE
              *
              *  used by projects to get a Configuration value to the  VD_Devicemanager
              *
              * \param [in] poMessage Message object
              */
      IN tVoid vHandle_DeviceManagerGetConfigurationValue_MethodStart( amt_tclServiceData* poMessage);
      OUT tVoid vHandle_DeviceManagerGetConfigurationValue_MethodReturn( const trReceiverInfo &f_oReceiverInfo, tU16 f_u16Value,tBool f_bSuccess);


      /*!
               * \brief Handler function for CCA message
               *        DEVMGR_SETCONFIGURATIONVALUE
               *
               *  used by projects to set a Configuration value to the  VD_Devicemanager
               *
               * \param [in] poMessage Message object
               */
       IN tVoid vHandle_DeviceManagerSetConfigurationValue_MethodStart( amt_tclServiceData* poMessage);
       OUT tVoid vHandle_DeviceManagerSetConfigurationValue_MethodReturn(const trReceiverInfo &f_oReceiverInfo, tBool f_bSuccess );

      /*!
               * \brief Handler function for CCA message
               *        DEVMGR_ADDEXTERNALHUB
               *
               *  used by projects to set External Hub(vendorID-ProductID-Portnumber) value to the  VD_Devicemanager
               *
               * \param [in] poMessage Message object
               */
       IN tVoid vHandle_DeviceManagerAddExternalHub_MethodStart( amt_tclServiceData* poMessage);
       OUT tVoid vHandle_DeviceManagerAddExternalHub_MethodReturn(const trReceiverInfo &f_oReceiverInfo, tBool f_bSuccess);



    ///@todo add comment
    tVoid ConvertString(OUT FIDEVMGR_DEVICEPROPERTY &f_objConnectNotify,IN GENSTRING f_oString, IN tenElemOfProperty eElemOfProp)const;
    tVoid ConvertDeviceType(OUT FIDEVMGR_DEVICEPROPERTY &f_objConnectNotify, IN CGlobalEnumerations::DEVICE_TYPE_Type f_eDeviceType) const;
    tVoid ConvertConnectStatus(OUT FIDEVMGR_DEVICEPROPERTY &f_objConnectNotify, IN DEVICE_CONNECTSTATUS_Type f_eConnectStatus) const;
    tVoid ConvertFileSystemType(OUT FIDEVMGR_DEVICEPROPERTY &f_objConnectNotify,CGlobalEnumerations::FILESYSTEM_TYPE_Type eFSType) const;
    tVoid ConvertDeviceSpeedUSB(OUT FIDEVMGR_DEVICEPROPERTY &f_objConnectNotify,IN  tenDeviceSpeedUSB f_DeviceSpeedUSB) const;
    tVoid ConvertDeviceUnsupportedReason(OUT FIDEVMGR_DEVICEPROPERTY &f_objConnectNotify,IN  tenDevice_UnsupportedReason f_DeviceUnsupportedReason) const;

    /*!
     * \brief function for convert the slot state to cca object type
     *
     *
     *  used by DEVMGR_NOTIFYSLOTSTATEOPTICALDISC
     *
     * \param [out] f_OpticalDiscSlotStateFI the  FIDEVMGR_NOTIFYSLOTSTATEOPTICALDISC object that wraps the slot state
     * \param[in] vd_devicemanager internal set slot state 
     */

    tVoid ConvertOpticalDiscSlotState(OUT FISHARED_E8_NOTIFYSLOTSTATEOPTICALDISC &f_OpticalDiscSlotStateFI,IN const tDMOpticalDiscSlotState f_OpticalDiscSlotStateInternal) const;

    /*!
     * \brief function for convert the slot state to cca object type
     *
     *
     *  used by DEVMGR_NOTIFYOPTICALDEVICECDINFO
     *
     * \param [out] f_OpticalDiscDeviceStateFI the  FISHARED_E8_NOTIFYCDINFO object that wraps the slot state
     * \param[in] vd_devicemanager internal set Device State
     */

    tVoid ConvertOpticalDiscDeviceState(OUT mplay_fi_tcl_e8DeviceState &f_OpticalDiscDeviceStateFI,IN  tenOpticalDeviceState f_OpticalDiscDeviceStateInternal) const;

    /*---------------------------------------------------------------------*
         * Message map definition macro                                        *
         *---------------------------------------------------------------------*/
    DECLARE_MSG_MAP(vd_devicemanager_tclService_DeviceManager)


    protected:
        /*!
            * \brief function to trace device property
            *
            *  added in course of maintenance to get code readable in a better way
            *
            * \param [in] poMessage Message object
            */
    tVoid vShowDeviceProperty(const vector<structNotifyClients> &f_vClientsAllConnected) const;
    tVoid vShowDeviceProperty(const CDevice &f_objCurNotifiedDevice) const ;
    tVoid vShowDeviceProperty(const CDevice *f_pCDeviceProperty) const  ;

    tVoid vClearReceiverInfo(trReceiverInfo &f_oReceiverInfo) const;


    /*!
           * \brief function to fill undervoltage state of target to f_oFiDataObject
           *
           * \param [OUT] f_oFiDataObject will be filled by function
           */
    tVoid vFillUndervoltage( FIDEVMGR_NOTIFYCONNECTIONSTATUS &f_oFiDataObject) const;

    /*!
           * \brief function to fill portlistvector to FI-object
           *
           * \param [OUT] f_oFiDataObject will be filled by function
           */
    tVoid vFillUSBPortListVector( FIDEVMGR_NOTIFYCONNECTIONSTATUS &f_oFiDataObject) const;

    /*!
           * \brief function to fill device type property to FI-object
           *
           * \param [OUT] f_oFiDataObject will be filled by function
           */
    tVoid vFillDeviceTypeVector(FIDEVMGR_NOTIFYCONNECTIONSTATUS &f_oFiDataObject);

    /*!
            * \brief function to fill sdcard (internal sdcard e.g. used in Suzuki or VAGMIBE) info to FI-object
            *
            * \param [OUT] f_oFiDataObject will be filled by function
            */
    tVoid vFillSDCardProperty(FIDEVMGR_NOTIFYSDCARDSTATUS &f_oFiDataObject) const;




private:

    // object of CDevice Class to hold information of current Notified Device
    // (connected & disconnected ). ifm_objCurNotifiedDevice.sendNo = -1 then it is invalid
    CDevice                  m_objCurNotifiedDevice;
    trReceiverInfo           m_oReceiverInfo;
    DVMLOCK                  m_table;
    tBool                    m_bServAvailable; ///< service availability
    vd_devicemanager_tclApp *m_poMainApp; //Raodmap 13035 Overtemperature because singleton StateTable and therefore DeviceManagerInterface need this - check if StateTable can live without it in future //@todo check for clean

    tDMOpticalDiscSlotState  m_eOpticalDiscSlotState; // to store the optical disc slot state;
    trOpticalDiskCDInfo      m_rOpticalDiskCDInfo; //Store CD Info


};

#endif // _VD_DEVICEMANAGER_SERVICE_DEVICEMANAGER_H_

////////////////////////////////////////////////////////////////////////////////
// <EOF>
