/*-----------------------------------------------------------------------------*
 * PortListManager.cpp                                                       *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen2                                                   *
 * COPYRIGHT   : (c) 2014-2020 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file PortListManager.cpp
 *
 * \brief This file holds implementation for class PortListManager - see also description of headerfile
 *
 * \version 25.08.2011,Koechling, Christian (Bosch), Initial Version

 *-----------------------------------------------------------------
 *									development for Gen3:
 *-----------------------------------------------------------------
 *\version 26.03.2014, Christian Koechling (Bosch) 
 *
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2016
 */

/*-----------------------------------------------------------------------------*
 * Includes                                                                    *
 *-----------------------------------------------------------------------------*/
#include "Config.h"

#define INCLUDE_VD_DVM_OSAL
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"

#include "Enums.h"
#include "ports/PortListManager.h"

/*-----------------------------------------------------------------------------*
* ETG Tracing                                                                 *
*-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"
     
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_UDEVMANAGER
#include "trcGenProj/Header/PortListManager.cpp.trc.h"
#endif
#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

/*-----------------------------------------------------------------------------*
 * unsing namespace                                                               *
 *-----------------------------------------------------------------------------*/
using namespace statetbl; 

/*-----------------------------------------------------------------------------*
 * static variables                                                                    *
 *-----------------------------------------------------------------------------*/
PortListManager *PortListManager::m_pPortListManager = NULL; // static pointer used to ensure a single instance of PrmManager
DVMLOCK          PortListManager::m_singelton;

/*-----------------------------------------------------------------------------*
 * Constructor                                                                  *
 *-----------------------------------------------------------------------------*/
PortListManager::PortListManager()
{
    ETG_TRACE_USR4(("Begin: PortListManager"));
    vClearArray();
    vInit();
    ETG_TRACE_USR4(("End  : PortListManager"));
}

/*-----------------------------------------------------------------------------*
 * Destructor                                                                  *
 *-----------------------------------------------------------------------------*/
PortListManager::~PortListManager()
{
    ETG_TRACE_USR4(("Begin: ~PortListManager"));


    ETG_TRACE_USR4(("End  : ~PortListManager"));
}

/*-----------------------------------------------------------------------------*
 * PortListManager *PortListManager::GetInstance()
 *-----------------------------------------------------------------------------*/
PortListManager *PortListManager::GetInstance(tVoid)
{
    ETG_TRACE_USR4(("Begin: GetInstance"));
    PortListManager::m_singelton.lock();

    if(NULL == m_pPortListManager)
    {
        m_pPortListManager = new PortListManager();
    }

    PortListManager::m_singelton.unlock();

    ETG_TRACE_USR4(("End  : GetInstance"));

    return m_pPortListManager;
}

/*-----------------------------------------------------------------------------*
 * tVoid PortListManager::DestroyInstance()
 *-----------------------------------------------------------------------------*/
tVoid PortListManager::DestroyInstance()
{
    ETG_TRACE_USR4(("Begin: DestroyInstance"));

    PortListManager::m_singelton.lock();
    delete m_pPortListManager;
    PortListManager::m_singelton.unlock();

    ETG_TRACE_USR4(("End  : DestroyInstance"));
}

/*-----------------------------------------------------------------------------*
 * tVoid PortListManager::vInit() 
 *-----------------------------------------------------------------------------*/
tVoid PortListManager::vInit()
{
    ETG_TRACE_USR4(("Begin: vInit"));
    tU8 i;
    for(i=(tU8)eUSB1; i<ARRAYSIZEFORUSB ;i++)
    {
        m_rArrayPortStates[i].u8USBPortNo   = (tU8)i;
    }
    ETG_TRACE_USR2(("vInit: m_rArrayPortStates[%d...%d] valid",(tU8)eUSB1,i));
    ETG_TRACE_USR4(("End  : vInit"));
}

/*-----------------------------------------------------------------------------*
 * tVoid PortListManager::vClear() 
 *-----------------------------------------------------------------------------*/
tVoid PortListManager::vClearArray()
{
    ETG_TRACE_USR4(("Begin: vClear"));
    tUInt i;
    for(i=0; i<ARRAYSIZEFORUSB ;i++)
    {
        m_rArrayPortStates[i].u8USBPortNo             = (tU8)eUSBUndef; //marks m_rArrayPortStates[i] invalid, undefined,cleared  i..e not usable yet
        m_rArrayPortStates[i].bOverCurrent            = FALSE;
        m_rArrayPortStates[i].bUndervoltage           = FALSE;
        m_rArrayPortStates[i].bHubConnected           = FALSE;
        m_rArrayPortStates[i].bOpenCircuit            = FALSE;
        m_rArrayPortStates[i].bElectricalFailure      = FALSE;
        m_rArrayPortStates[i].bUSBPortConfiguredUsed  = FALSE;
        m_rArrayPortStates[i].bUSBPowerON             = TRUE;//By default we are considering the port is powered ON

    }
    ETG_TRACE_USR2(("vClear: m_rArrayPortStates[0...%d] cleared",i));
    ETG_TRACE_USR4(("End  : vClear"));
}


/*-----------------------------------------------------------------------------*
 * tVoid PortListManager::vClear(...) 
 *-----------------------------------------------------------------------------*/
tVoid PortListManager::vClear(IN trPortStates &f_rPortState) const
{
    f_rPortState.u8USBPortNo             = (tU8)eUSBUndef; //marks f_rPortState invalid, undefined,cleared  i..e not usable yet
    f_rPortState.bOverCurrent            = FALSE;
    f_rPortState.bUndervoltage           = FALSE;
    f_rPortState.bHubConnected           = FALSE;
    f_rPortState.bOpenCircuit            = FALSE;
    f_rPortState.bElectricalFailure      = FALSE;
    f_rPortState.bUSBPortConfiguredUsed  = FALSE;
    f_rPortState.bUSBPowerON             = TRUE;//By default we are considering the port is powered ON

}

/*-----------------------------------------------------------------------------*
 * tVoid PortListManager::vSetHardwareSignals (...)
 *-----------------------------------------------------------------------------*/
tVoid PortListManager::vSetHardwareSignals(IN const trPortStates (&f_rArrayHWSignals)[ARRAYSIZEFORUSB])
{
    ETG_TRACE_USR4(("Begin: vSetHardwareSignals"));
    tUInt i;
    //vStoreHWSignals:
    for(i = (tUInt)eUSB1; i<ARRAYSIZEFORUSB ;i++)
    {
        m_rArrayPortStates[i].u8USBPortNo             = f_rArrayHWSignals[i].u8USBPortNo;
        m_rArrayPortStates[i].bOverCurrent            = f_rArrayHWSignals[i].bOverCurrent;
        m_rArrayPortStates[i].bUndervoltage           = f_rArrayHWSignals[i].bUndervoltage;
        m_rArrayPortStates[i].bHubConnected           = f_rArrayHWSignals[i].bHubConnected;
        m_rArrayPortStates[i].bOpenCircuit            = f_rArrayHWSignals[i].bOpenCircuit;
        //note that the PowerState of port will be filled by a different function vSetPortPowerState
        m_rArrayPortStates[i].bElectricalFailure      = f_rArrayHWSignals[i].bElectricalFailure;
        m_rArrayPortStates[i].bUSBPortConfiguredUsed  = f_rArrayHWSignals[i].bUSBPortConfiguredUsed;


    }
    ETG_TRACE_USR2(("vSetHardwareSignals: m_rArrayPortStates[1...%d] copied",i));

    ETG_TRACE_USR4(("End  : vSetHardwareSignals"));
}


tVoid PortListManager::vSetPortPowerState(IN const trPortStates &f_rPortStates)
{
    ETG_TRACE_USR4(("Begin: vSetPortPowerState"));
    if(f_rPortStates.u8USBPortNo >(tU8)eUSBUndef && f_rPortStates.u8USBPortNo <ARRAYSIZEFORUSB )
    {
        m_rArrayPortStates[f_rPortStates.u8USBPortNo].u8USBPortNo            = f_rPortStates.u8USBPortNo;
        m_rArrayPortStates[f_rPortStates.u8USBPortNo].bUSBPowerON            = f_rPortStates.bUSBPowerON;
        m_rArrayPortStates[f_rPortStates.u8USBPortNo].bUSBPortConfiguredUsed = f_rPortStates.bUSBPortConfiguredUsed;
        if(FALSE == f_rPortStates.bUSBPowerON)
        {
            m_rArrayPortStates[f_rPortStates.u8USBPortNo].bOverCurrent           = FALSE;
            m_rArrayPortStates[f_rPortStates.u8USBPortNo].bUndervoltage          = FALSE;
            m_rArrayPortStates[f_rPortStates.u8USBPortNo].bElectricalFailure     = FALSE;
            m_rArrayPortStates[f_rPortStates.u8USBPortNo].bHubConnected          = FALSE;
            m_rArrayPortStates[f_rPortStates.u8USBPortNo].bOpenCircuit           = FALSE;
        }


    }
    ETG_TRACE_USR4(("End: vSetPortPowerState"));


}

/*-----------------------------------------------------------------------------*
 * tVoid PortListManager::vGetHardwareSignals (...)
 *-----------------------------------------------------------------------------*/
tVoid PortListManager::vGetHardwareSignals(OUT trPortStates (&f_rArrayHWSignals)[ARRAYSIZEFORUSB])
{
    ETG_TRACE_USR4(("Begin: vGetHardwareSignals"));
    tUInt i;
    //vStoreHWSignals:
    for(i = (tUInt)eUSB1; i<ARRAYSIZEFORUSB ;i++)
    {
        f_rArrayHWSignals[i].u8USBPortNo             = m_rArrayPortStates[i].u8USBPortNo;
        f_rArrayHWSignals[i].bOverCurrent            = m_rArrayPortStates[i].bOverCurrent;
        f_rArrayHWSignals[i].bUndervoltage           = m_rArrayPortStates[i].bUndervoltage;
        f_rArrayHWSignals[i].bHubConnected           = m_rArrayPortStates[i].bHubConnected;
        f_rArrayHWSignals[i].bOpenCircuit            = m_rArrayPortStates[i].bOpenCircuit;
        f_rArrayHWSignals[i].bUSBPowerON             = m_rArrayPortStates[i].bUSBPowerON;
        f_rArrayHWSignals[i].bElectricalFailure      = m_rArrayPortStates[i].bElectricalFailure;
        f_rArrayHWSignals[i].bUSBPortConfiguredUsed  = m_rArrayPortStates[i].bUSBPortConfiguredUsed;

    }
    ETG_TRACE_USR2(("vGetHardwareSignals: m_rArrayPortStates[1...%d] copied",i));

    ETG_TRACE_USR4(("End  : vGetHardwareSignals"));
}

/*-----------------------------------------------------------------------------*
 * tBool PortListManager::vCompareHardwareSignals (...)
 *-----------------------------------------------------------------------------*/
tBool PortListManager::GetIsDifferentHardwareSignals(IN const trPortStates (&f_rArrayHWSignals)[ARRAYSIZEFORUSB])
{
    ETG_TRACE_USR4(("Begin: bCompareHardwareSignals"));
    tBool bRes = FALSE;
    tUInt i;
    tUInt uiDifferences = 0;
    //vStoreHWSignals:
    for(i = (tUInt)eUSB1; i<ARRAYSIZEFORUSB ;i++)
    {
        if(f_rArrayHWSignals[i].u8USBPortNo             != m_rArrayPortStates[i].u8USBPortNo)          {uiDifferences++;}
        if(f_rArrayHWSignals[i].bOverCurrent            != m_rArrayPortStates[i].bOverCurrent) {uiDifferences++;}
        if(f_rArrayHWSignals[i].bUndervoltage           != m_rArrayPortStates[i].bUndervoltage){uiDifferences++;}
        if(f_rArrayHWSignals[i].bHubConnected           != m_rArrayPortStates[i].bHubConnected){uiDifferences++;}
        if(f_rArrayHWSignals[i].bOpenCircuit            != m_rArrayPortStates[i].bOpenCircuit) {uiDifferences++;}
        //Note we are not comparing whether the USB port is powered on or not
        if(f_rArrayHWSignals[i].bElectricalFailure      != m_rArrayPortStates[i].bElectricalFailure) {uiDifferences++;}
        if(f_rArrayHWSignals[i].bUSBPortConfiguredUsed  != m_rArrayPortStates[i].bUSBPortConfiguredUsed) {uiDifferences++;}


    }
    ETG_TRACE_USR2(("bCompareHardwareSignals: uiDifferences: %d",uiDifferences));

    if(uiDifferences > 0)
    {
        bRes = TRUE; //different
    }

    ETG_TRACE_USR4(("End  : bCompareHardwareSignals"));
    
    return bRes;
}


/*-----------------------------------------------------------------------------*
 * tVoid PortListManager::vShowHardwareSignals(...)
 *-----------------------------------------------------------------------------*/
tVoid PortListManager::vShowHardwareSignals() const
{
    ETG_TRACE_USR4(("Begin: ---------------/vShowHardwareSignals-----------------------------"));

    for(tInt i = (tInt)eUSB1; i<ARRAYSIZEFORUSB ;i++)
    {
        ETG_TRACE_USR4(("m_rArrayPortStates[%d].u8USBPortNo     %d            ",i,m_rArrayPortStates[i].u8USBPortNo));
        ETG_TRACE_USR4(("m_rArrayPortStates[%d].bOverCurrent :0x%x            ",i,m_rArrayPortStates[i].bOverCurrent));
        ETG_TRACE_USR4(("m_rArrayPortStates[%d].bUndervoltage:0x%x            ",i,m_rArrayPortStates[i].bUndervoltage));
        ETG_TRACE_USR4(("m_rArrayPortStates[%d].bHubConnected:0x%x            ",i,m_rArrayPortStates[i].bHubConnected));
        ETG_TRACE_USR4(("m_rArrayPortStates[%d].bOpenCircuit :0x%x            ",i,m_rArrayPortStates[i].bOpenCircuit));
        ETG_TRACE_USR4(("m_rArrayPortStates[%d].bUSBPowerON  :0x%x            ",i,m_rArrayPortStates[i].bUSBPowerON));
        ETG_TRACE_USR4(("m_rArrayPortStates[%d].bElectricalFailure:0x%x       ",i,m_rArrayPortStates[i].bElectricalFailure));
        ETG_TRACE_USR4(("m_rArrayPortStates[%d].bUSBPortConfiguredUsed  :0x%x ",i,m_rArrayPortStates[i].bUSBPortConfiguredUsed));
/* somtime used for unittests
        printf("-------------------------------------------------------------------------------\n");
        printf("m_rArrayPortStates[%d].u8USBPortNo     %d \n",i,m_rArrayPortStates[i].u8USBPortNo);
        printf("m_rArrayPortStates[%d].bOverCurrent :0x%x \n",i,m_rArrayPortStates[i].bOverCurrent);
        printf("m_rArrayPortStates[%d].bUndervoltage:0x%x \n",i,m_rArrayPortStates[i].bUndervoltage);
        printf("m_rArrayPortStates[%d].bHubConnected:0x%x \n",i,m_rArrayPortStates[i].bHubConnected);
        printf("m_rArrayPortStates[%d].bOpenCircuit :0x%x \n",i,m_rArrayPortStates[i].bOpenCircuit);
        printf("-------------------------------------------------------------------------------\n\n");
*/
    }
    ETG_TRACE_USR4(("End  : ---------------vShowHardwareSignals/-----------------------------"));
}

/*-----------------------------------------------------------------------------*
 * tInt PortListManager::vGetSizeOfArrayHardwareSignals (...)
 *-----------------------------------------------------------------------------*/
tUInt PortListManager::uiGetSizeOfArrayHardwareSignals(tVoid) const
{
    tUInt uiRetVal = NUMOFUSBCONNECTORS;

    ETG_TRACE_USR1(("uiGetSizeOfArrayHardwareSignals: %d", uiRetVal));
    return uiRetVal;
}


/*-----------------------------------------------------------------------------*
 * tBool PortListManager::bGetElement (...)
 *-----------------------------------------------------------------------------*/
tBool PortListManager::bGetElement (IN tUInt f_uiIndex, OUT trPortStates &f_rPortState) const
{
    tBool bRetVal = FALSE;

    ETG_TRACE_USR4(("bGetElement: f_uiIndex:%d",f_uiIndex));

    if( (f_uiIndex >= (tUInt)eUSB1) && (f_uiIndex < (tUInt)eUSBAll))
    {
        f_rPortState.u8USBPortNo             = m_rArrayPortStates[f_uiIndex].u8USBPortNo;
        f_rPortState.bOverCurrent            = m_rArrayPortStates[f_uiIndex].bOverCurrent;
        f_rPortState.bUndervoltage           = m_rArrayPortStates[f_uiIndex].bUndervoltage;
        f_rPortState.bHubConnected           = m_rArrayPortStates[f_uiIndex].bHubConnected;
        f_rPortState.bOpenCircuit            = m_rArrayPortStates[f_uiIndex].bOpenCircuit;
        f_rPortState.bUSBPowerON             = m_rArrayPortStates[f_uiIndex].bUSBPowerON;
        f_rPortState.bElectricalFailure      = m_rArrayPortStates[f_uiIndex].bElectricalFailure;
        f_rPortState.bUSBPortConfiguredUsed  = m_rArrayPortStates[f_uiIndex].bUSBPortConfiguredUsed;

        if(f_rPortState.u8USBPortNo == (tU8)f_uiIndex)
        {
            //-------------
            //return success
            //-------------
            bRetVal = TRUE;
        }
        else
        {
            if(f_rPortState.u8USBPortNo != (tU8)eUSBUndef)
            {
                ETG_TRACE_FATAL(("[WARNING] bGetElement: unexpected value f_uiIndex:%d wheras f_rPortState.u8USBPortNo:%d ",f_uiIndex,f_rPortState.u8USBPortNo));
            }
        }
    }
    else
    {
        ETG_TRACE_FATAL(("bGetElement: f_uiIndex: %d invalid (NUMOFUSBCONNECTORS: %d)",f_uiIndex,NUMOFUSBCONNECTORS));
    }

    return bRetVal;
}

tVoid PortListManager::GetCopyArrayPortStates(OUT trPortStates (&f_rArrayPortStates)[ARRAYSIZEFORUSB])
{
    for(int i = (tUInt)eUSB1; i<ARRAYSIZEFORUSB ;i++)
    {
        f_rArrayPortStates[i].u8USBPortNo             = m_rArrayPortStates[i].u8USBPortNo;
        f_rArrayPortStates[i].bOverCurrent            = m_rArrayPortStates[i].bOverCurrent;
        f_rArrayPortStates[i].bUndervoltage           = m_rArrayPortStates[i].bUndervoltage;
        f_rArrayPortStates[i].bHubConnected           = m_rArrayPortStates[i].bHubConnected;
        f_rArrayPortStates[i].bOpenCircuit            = m_rArrayPortStates[i].bOpenCircuit;
        f_rArrayPortStates[i].bUSBPowerON             = m_rArrayPortStates[i].bUSBPowerON;
        f_rArrayPortStates[i].bElectricalFailure      = m_rArrayPortStates[i].bElectricalFailure;
        f_rArrayPortStates[i].bUSBPortConfiguredUsed  = m_rArrayPortStates[i].bUSBPortConfiguredUsed;
    }
}


