//@todo stop thread is missing

/*-----------------------------------------------------------------------------*
 *VoltageManager.cpp                                                              *
 *-----------------------------------------------------------------------------*
 *                                                                             *
 * SW-COMPONENT: VD_DeviceManager                                              *
 * PROJECT     : GM NextGen3                                                   *
 * COPYRIGHT   : (c) 2012 Robert Bosch GmbH, Hildesheim                        *
 *                                                                             *
 *-----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------*
 * doxygen style header                                                        *
 *-----------------------------------------------------------------------------*/
/*!
 * \file voltageManager.cpp
 *
 * \brief This file holds the implementation of the Voltage manager. In dependency with ConfigurationManager it
              informs StateTable about critical voltage
 
 *-----------------------------------------------------------------
 *                                    development for Gen3:
 *-----------------------------------------------------------------
 *\version 11.06.2014, Christian Koechling (Bosch) 
 *          -# add drv_volt to react on low or critical voltage -some projects switch off voltage supply for 
*                USB at critical voltage and not at 5Volt furthermore cd drive has to b taken into account
 *
.*
 * \copyright Copyright (c) Robert Bosch Car Multimedia GmbH  2010-2016
 */


#include <stdio.h>
#include "Config.h"

#define INCLUDE_VD_DVM_OSAL  //statetable uses ahl thread otherwhise this 
#define INCLUDE_VD_DVM_BASICS
#include "Common.h"
#include "Enums.h"
#include "DeviceManager_ErrorCodes.h"
#include "Device.h"
#include "DeviceCard.h"
#include "StateTable.h"

#include "LockDVM.h"
#include "VoltageManager.h"
#include "debug/HistoryManager.h"
#include "config/ConfigurationManager.h"

/*-----------------------------------------------------------------------------*
 * namespace                                                                     *
 *-----------------------------------------------------------------------------*/
using namespace statetbl;

/*-----------------------------------------------------------------------------*
 * ETG Tracing                                                                 *
 *-----------------------------------------------------------------------------*/
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_dvm.h"

#ifndef VARIANT_S_FTR_ENABLE_UNITTEST
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_DEVICEMANAGER_VOLTAGEMANAGER
#include "trcGenProj/Header/VoltageManager.cpp.trc.h"
#endif

#include "ETGTrace.h"
#endif //VARIANT_S_FTR_ENABLE_UNITTEST

/*-----------------------------------------------------------------------------*
 * defines                                                                 *
 *-----------------------------------------------------------------------------*/
#ifndef DEV_VOLT_C_U32_EVENT_MASK_VOLTAGE_CHANGED_NOTIFY
#define DEV_VOLT_C_U32_EVENT_MASK_VOLTAGE_CHANGED_NOTIFY 1
#warning "missing OSAL-definition: DEV_VOLT_C_U32_EVENT_MASK_VOLTAGE_CHANGED_NOTIFY"
#endif

#define MODULEXYZ_C_U32_EVENT_MASK_STOP_THREAD   0x01000000
 
#define MODULEXYZ_C_U32_EVENT_MASK_ALL \
        (DEV_VOLT_C_U32_EVENT_MASK_VOLTAGE_CHANGED_NOTIFY | \
         DEV_VOLT_C_U32_EVENT_MASK_PERMANENT_HIGH_VOLTAGE | \
         DEV_VOLT_C_U32_EVENT_MASK_PERMANENT_CRITICAL_HIGH_VOLTAGE | \
         MODULEXYZ_C_U32_EVENT_MASK_STOP_THREAD)



VoltageManager::VoltageManager(IStateTableHWMalfunction& oIStateTableHWMalfunction):m_oIStateTableHWMalfunction(oIStateTableHWMalfunction)
{
    m_hIOVoltDescriptor = OSAL_ERROR;
    m_eCriticalVoltage  = SIG_UNDEF;
    m_bStopThread = FALSE;
}

VoltageManager::~VoltageManager()
{
    if(m_hIOVoltDescriptor != OSAL_ERROR)
    {
        if (OSAL_s32IOClose(m_hIOVoltDescriptor) == OSAL_OK)
        {
           ETG_TRACE_USR4(("DestroyInstance: ok"));
        }
        else
        {
           ETG_TRACE_USR4(("[ERROR] DestroyInstance: Close failed "));
        }
    }
}

/*-----------------------------------------------------------------------------*
 * int Initialize()                                                            *
 *-----------------------------------------------------------------------------*/
int VoltageManager::Initialize()
{
    tS32 s32RetVal = OSAL_OK;
    ETG_TRACE_USR4(("Begin: Initialize                                         "));
    m_hIOVoltDescriptor = OSAL_IOOpen(OSAL_C_STRING_DEVICE_VOLT, OSAL_EN_READWRITE);

    if(m_hIOVoltDescriptor == OSAL_ERROR)
    {
        ETG_TRACE_FATAL(("[ERROR] Initialize: OSAL_IOOpen(OSAL_C_STRING_DEVICE_VOLT, OSAL_EN_READWRITE)== OSAL_ERROR %d",OSAL_u32ErrorCode()));
        ETG_TRACE_ERRMEM(("[ERROR] Initialize: OSAL_IOOpen(OSAL_C_STRING_DEVICE_VOLT, OSAL_EN_READWRITE)== OSAL_ERROR %d",OSAL_u32ErrorCode()));
        s32RetVal = OSAL_ERROR;
    }
    else
    {
#ifndef VARIANT_S_FTR_ENABLE_UNITTEST  //intermediate
         HWMALFUNCTIONHISTORY.AddToHistory("[INFO] VoltageManager::Initialize() ",NULL);
#endif
         ETG_TRACE_USR4(("Initialize: OSAL_IOOpen(OSAL_C_STRING_DEVICE_VOLT, OSAL_EN_READWRITE) == OSAL_OK"));
         tU32             l_u32BoardVoltage_milliVolt;
         tenSystemVoltage l_eSystemVoltage;

         vGetBoardVoltage(OUT  l_u32BoardVoltage_milliVolt); 
         vGetSystemVoltage(OUT l_eSystemVoltage);

         vMapVoltage(OUT m_eCriticalVoltage,IN l_eSystemVoltage); //m_eCriticalVoltage is set now

         tInt iThreadIndex = ThreadFactoryDVM::GetThreadFactory()->Do(IN this, (int)VoltageManager::eThread_SystemVolt, NULL); //starts thread eThread_SystemVolt - see Do function above

         ETG_TRACE_COMP(("VoltageManager: iThreadIndex:%d",iThreadIndex));
    }

    ETG_TRACE_USR4(("End  : Initialize                                        "));

    return (int)s32RetVal;
}

/*-----------------------------------------------------------------------------*
 * void VoltageManager:IsCriticalVoltage(...)
 *-----------------------------------------------------------------------------*/
bool VoltageManager::IsCriticalVoltage()
{
    bool bRet = false;

    tenSystemVoltage l_eSystemVoltage;

    vGetSystemVoltage(OUT l_eSystemVoltage);

    if(VoltageManager::eVolt_CriticalLow == l_eSystemVoltage)
    {
        bRet = true;
    }

    return bRet;
}

/*-----------------------------------------------------------------------------*
 * void VoltageManager:vGetSystemVoltage(...)
 *-----------------------------------------------------------------------------*/
void VoltageManager::vGetSystemVoltage(OUT tenSystemVoltage &f_eSystemVoltage)
{

    tS32         s32Success = 0;
    tU32 l_u32SystemVoltage = 0;

    s32Success = OSAL_s32IOControl( IN m_hIOVoltDescriptor,
                                    IN OSAL_C_S32_IOCTRL_VOLT_GET_SYSTEM_VOLTAGE_STATE,
                                   OUT (intptr_t) &l_u32SystemVoltage);
    if (OSAL_OK == s32Success)
    {

        vMapVoltage(OUT f_eSystemVoltage, IN l_u32SystemVoltage);
    }
    else
    {
        ETG_TRACE_FATAL(("[ERROR] Initialize OSAL_s32IOControl(m_hIOVoltDescriptor,OSAL_C_S32_IOCTRL_VOLT_GET_SYSTEM_VOLTAGE_STATE...) is NOK"));
    }
}

/*-----------------------------------------------------------------------------*
 * void VoltageManager:vGetBoardVoltage(...)
 *-----------------------------------------------------------------------------*/
void VoltageManager::vGetBoardVoltage(OUT tU32 &f_u32BoardVoltage_milliVolt)
{
    tS32 s32Success;
    if(m_hIOVoltDescriptor != OSAL_ERROR)
    {
        s32Success = OSAL_s32IOControl(IN m_hIOVoltDescriptor,IN OSAL_C_S32_IOCTRL_VOLT_GET_BOARD_VOLTAGE,(intptr_t) &f_u32BoardVoltage_milliVolt);
        if (OSAL_OK == s32Success)
        {
            ETG_TRACE_USR2(("f_u32BoardVoltage_milliVolt: %d",f_u32BoardVoltage_milliVolt));
        }
        else
        {
            ETG_TRACE_FATAL(("[ERROR] vGetBoardVoltage"));
        }
    }
}

/*-----------------------------------------------------------------------------*
 *void VoltageManager:vMapVoltage(...)
 *-----------------------------------------------------------------------------*/
void VoltageManager::vMapVoltage(OUT VoltageManager::tenSystemVoltage &eVolt, IN tU32 f_u32SystemVoltage) const
{
    switch(f_u32SystemVoltage)
    {
        case DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_CRITICAL_LOW:
            eVolt = VoltageManager::eVolt_CriticalLow;
            break;
        case DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_LOW:
            eVolt = VoltageManager::eVolt_Low;
            break;
        case DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_OPERATING:
            eVolt = VoltageManager::eVolt_Operating;
            break;
        case DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_HIGH:
            eVolt = VoltageManager::eVolt_High;
            break;
        case DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_CRITICAL_HIGH:
            eVolt = VoltageManager::eVolt_CriticalHigh;
            break;
        default:
            ETG_TRACE_USR4(("vGetSystemVoltage: %d",f_u32SystemVoltage));
            eVolt = VoltageManager::eVolt_Undefined;
            break;
    }


    ETG_TRACE_USR2(("vMapVoltage:%d",ETG_CENUM(VoltageManager::tenSystemVoltage,eVolt)));
}

/*-----------------------------------------------------------------------------*
 *void VoltageManager:vMapVoltage(...)
 *-----------------------------------------------------------------------------*/
void VoltageManager::vMapVoltage(OUT tenSignalValue &eCriticalVoltage,IN VoltageManager::tenSystemVoltage eVolt)
{
    switch(eVolt)
    {
       case eVolt_CriticalLow:
           eCriticalVoltage = SIG_TRUE;
           break;
       case eVolt_Operating:
           eCriticalVoltage = SIG_FALSE;
            break;
       case eVolt_Low:
       case eVolt_High:
       case eVolt_CriticalHigh:
       case eVolt_Undefined:
           eCriticalVoltage = SIG_UNDEF;
           break;
       default:
           eCriticalVoltage = SIG_UNDEF;
           break;
    }
}

/*-----------------------------------------------------------------------------*
 *void VoltageManager:vShowCriticalVoltage(...)
 *-----------------------------------------------------------------------------*/
void VoltageManager::vShowCriticalVoltage(tenSignalValue &eCriticalVoltage)
{
    switch(eCriticalVoltage)
    {
       case SIG_TRUE:
           ETG_TRACE_SYS(("vHandleCVMEvent_internal: >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
           ETG_TRACE_SYS(("vHandleCVMEvent_internal:     CRITICAL_LOW_VOLTAGE_START"));
           ETG_TRACE_SYS(("vHandleCVMEvent_internal: >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
           break;
       case SIG_FALSE:
           ETG_TRACE_SYS(("vHandleCVMEvent_internal: >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>")); 
           ETG_TRACE_SYS(("vHandleCVMEvent_internal:      CRITICAL_LOW_VOLTAGE_END"));
           ETG_TRACE_SYS(("vHandleCVMEvent_internal: >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>"));
            break;
       case SIG_UNDEF:
           break;
       default:
           break;
    }
}
/*-----------------------------------------------------------------------------*
 *void VoltageManager::vReceiveSystemVoltageNotification(...)
 *-----------------------------------------------------------------------------*/
void VoltageManager::vReceiveSystemVoltageNotification()
{
    ETG_TRACE_USR4(("Begin: vReceiveSystemVoltageNotification"));

    tS32 s32Success;
    OSAL_tEventHandle hEventHandle                    = OSAL_C_INVALID_HANDLE;
    tU32              u32MySystemVoltageState         = DEV_VOLT_C_U32_SYSTEM_VOLTAGE_STATE_OPERATING; // Per default everything is fine
    tBool             bIsPermanentHighVoltage         = FALSE;
    tBool             bIsPermanentCriticalHighVoltage = FALSE;

    DEV_VOLT_trClientRegistration        rClientRegistration;
    DEV_VOLT_trSystemVoltageRegistration rSystemVoltageRegistration;


    //register client
    s32Success = OSAL_s32IOControl(m_hIOVoltDescriptor,OSAL_C_S32_IOCTRL_VOLT_REGISTER_CLIENT,(intptr_t) &rClientRegistration);
    if (OSAL_OK ==s32Success )
    {
        ETG_TRACE_USR4(("vReceiveSystemVoltageNotification: OSAL_C_S32_IOCTRL_VOLT_REGISTER_CLIENT"));

        s32Success = OSAL_s32EventOpen(rClientRegistration.szNotificationEventName, &hEventHandle);
        if (OSAL_OK == s32Success)
        {
            ETG_TRACE_USR4(("vReceiveSystemVoltageNotification: Event opened"));

            rSystemVoltageRegistration.u32ClientId              = rClientRegistration.u32ClientId;
            rSystemVoltageRegistration.u32VoltageIndicationMask = (DEV_VOLT_C_U32_BIT_MASK_INDICATE_LOW_VOLTAGE           |
                                                                   DEV_VOLT_C_U32_BIT_MASK_INDICATE_CRITICAL_LOW_VOLTAGE  |
                                                                   DEV_VOLT_C_U32_BIT_MASK_INDICATE_HIGH_VOLTAGE          |
                                                                   DEV_VOLT_C_U32_BIT_MASK_INDICATE_CRITICAL_HIGH_VOLTAGE  );

            s32Success = OSAL_s32IOControl(m_hIOVoltDescriptor, OSAL_C_S32_IOCTRL_VOLT_REGISTER_SYSTEM_VOLTAGE_NOTIFICATION,
                                            (intptr_t) &rSystemVoltageRegistration);
            if (OSAL_OK == s32Success)
            {
                   ETG_TRACE_USR4(("vReceiveSystemVoltageNotification: OSAL_C_S32_IOCTRL_VOLT_REGISTER_SYSTEM_VOLTAGE__NOTIFICATION"));
                  // Get system voltage state pending at the time of registration
                  u32MySystemVoltageState = rSystemVoltageRegistration.u32CurrentSystemVoltageState;
                  // Check for already occurred permanent high and permanent critical-high voltages at the time of registration
                  bIsPermanentHighVoltage         = (rSystemVoltageRegistration.u8PermHighVoltageCounter         != 0) ? TRUE : FALSE;
                  bIsPermanentCriticalHighVoltage = (rSystemVoltageRegistration.u8PermCriticalHighVoltageCounter != 0) ? TRUE : FALSE;

                  ETG_TRACE_USR4(("vReceiveSystemVoltageNotification:u32MySystemVoltageState        =%d"   ,u32MySystemVoltageState));
                  ETG_TRACE_USR4(("vReceiveSystemVoltageNotification:bIsPermanentHighVoltage        = 0x%x",bIsPermanentHighVoltage));
                  ETG_TRACE_USR4(("vReceiveSystemVoltageNotification:bIsPermanentCriticalHighVoltage= 0x%x",bIsPermanentCriticalHighVoltage));
                    
                  //===============================================================
                  vWaitLoopForSystemVoltageChanged(hEventHandle, rClientRegistration.u32ClientId);
                  //===============================================================

                  s32Success = OSAL_s32IOControl(m_hIOVoltDescriptor,OSAL_C_S32_IOCTRL_VOLT_UNREGISTER_SYSTEM_VOLTAGE_NOTIFICATION,
                                        (intptr_t)rSystemVoltageRegistration.u32ClientId);
                  if (OSAL_ERROR == s32Success)
                  {
                    // ERROR : Your code ...
                  }
            }
        }
        else
        {
          // ERROR : Your code ...
        }

        s32Success = OSAL_s32IOControl(m_hIOVoltDescriptor,OSAL_C_S32_IOCTRL_VOLT_UNREGISTER_CLIENT,(intptr_t)rClientRegistration.u32ClientId);
        if (OSAL_ERROR == s32Success)
        {
          // ERROR : Your code ...
        }
    }
    else
    {
    // ERROR : Your code ...
    }
    ETG_TRACE_USR4(("End  : vReceiveSystemVoltageNotification"));
}

/*-----------------------------------------------------------------------------*
 *void VoltageManager::vWaitLoopForSystemVoltageChange(...)
 *-----------------------------------------------------------------------------*/
void VoltageManager::vWaitLoopForSystemVoltageChanged(const OSAL_tEventHandle &f_hEventHandle, tU32 f_u32ClientID)
{
    ETG_TRACE_USR4(("Begin: vWaitLoopForSystemVoltageChanged"));

    OSAL_tEventMask   rEventMaskResult                = 0;
    DEV_VOLT_trSystemVoltageHistory      rSystemVoltageHistory;
    tenSystemVoltage eVolt = VoltageManager::eVolt_Undefined;
    tenSignalValue eU60_drop = SIG_UNDEF;

    if(f_hEventHandle != OSAL_C_INVALID_HANDLE)
    {
        m_bStopThread = FALSE;
    }
    else
    {
        m_bStopThread = TRUE;
    }

    //in WorkThread receiver for Criticalvolt messages is created before VoltageManager - hnece it makes sense to send such condition if available to inform mediaplayer asap at startup
    // about critical voltage
    tenSystemVoltage l_eSystemVoltage;
    vGetSystemVoltage(OUT l_eSystemVoltage);
    vMapVoltage(OUT m_eCriticalVoltage,IN l_eSystemVoltage); //m_eCriticalVoltage is set now
    if(m_eCriticalVoltage == SIG_TRUE)
    {
        ETG_TRACE_FATAL(("vWaitLoopForSystemVoltageChanged: VD_DVM started during CritVoltage"));
        VOLTCLIENTHISTORY.AddToHistory(eU60_drop); //saves all notifications of drv_volt those - note: not all will be send to stateTable see below vSendVoltageWarning(...)
        vShowCriticalVoltage(IN m_eCriticalVoltage);
        vSendVoltageWarning(m_eCriticalVoltage);
    }

    while (FALSE == m_bStopThread)
    {
      //------------------------------------------------------------------------
      //                      waiting for change of system voltage here
      //------------------------------------------------------------------------
      if (OSAL_s32EventWait(f_hEventHandle,
                              MODULEXYZ_C_U32_EVENT_MASK_ALL, 
                              OSAL_EN_EVENTMASK_OR, 
                              OSAL_C_TIMEOUT_FOREVER,
                              &rEventMaskResult) == OSAL_OK)
      {
          ETG_TRACE_USR2(("vWaitLoopForSystemVoltageChanged: received Event rEventMaskResult: 0x%x",rEventMaskResult));
          if (OSAL_s32EventPost(f_hEventHandle, ~rEventMaskResult, OSAL_EN_EVENTMASK_AND) == OSAL_OK) // Clear evaluated event bits
          {
              if (rEventMaskResult & DEV_VOLT_C_U32_EVENT_MASK_VOLTAGE_CHANGED_NOTIFY) //check here is in error
              {
                  rSystemVoltageHistory.u32ClientId = f_u32ClientID;

                  if (OSAL_s32IOControl(m_hIOVoltDescriptor,OSAL_C_S32_IOCTRL_VOLT_GET_SYSTEM_VOLTAGE_HISTORY, 
                                          (intptr_t) &rSystemVoltageHistory) == OSAL_OK)
                  {
                      //----------------------------------
                      // OK : Your code .
                      //----------------------------------
                      /*
                                          typedef struct
                                          {
                                            tU32 u32CriticalLowVoltageCounter;
                                            tU32 u32LowVoltageCounter;
                                            tU32 u32CriticalHighVoltageCounter;
                                            tU32 u32HighVoltageCounter;
                                            tU32 u32CurrentSystemVoltageState;
                                          } DEV_VOLT_trSystemVoltage;

                                         typedef struct
                                         {
                                           tU32                     u32ClientId;
                                           DEV_VOLT_trSystemVoltage rSystemVoltage;
                                         } DEV_VOLT_trSystemVoltageHistory
                                         */
                      vMapVoltage(OUT eVolt, IN rSystemVoltageHistory.rSystemVoltage.u32CurrentSystemVoltageState);
                      ETG_TRACE_COMP(("vWaitLoopForSystemVoltageChanged: eVolt:%d ",ETG_CENUM(VoltageManager::tenSystemVoltage,eVolt)));
                     
                      vMapVoltage(OUT eU60_drop,IN eVolt);
 

                      
                      if((eU60_drop == SIG_TRUE) || (eU60_drop == SIG_FALSE))
                      {
                          if(m_eCriticalVoltage != eU60_drop)
                          {
                              m_eCriticalVoltage = eU60_drop;

                              VOLTCLIENTHISTORY.AddToHistory(eU60_drop); //saves all notifications of drv_volt those - note: not all will be send to stateTable see below vSendVoltageWarning(...)
                              vShowCriticalVoltage(IN eU60_drop);

                              vSendVoltageWarning(eU60_drop);
                          }
                      }

                  }
                  else
                  {
                    // ERROR : Your code ...
                  }
              }
              else if (rEventMaskResult & DEV_VOLT_C_U32_EVENT_MASK_PERMANENT_HIGH_VOLTAGE)
              {
                  // OK : Your code ...
              }
              else if (rEventMaskResult & DEV_VOLT_C_U32_EVENT_MASK_PERMANENT_CRITICAL_HIGH_VOLTAGE)
              {
              // OK : Your code ...
              }
              else if (rEventMaskResult & MODULEXYZ_C_U32_EVENT_MASK_STOP_THREAD)
              {
                  m_bStopThread = TRUE;
              }
              else
              {
              // ERROR : Your code ...
              }
          }
          else
          {
            // ERROR : Your code ...
          }
    }
    else
    {
      // ERROR : Your code ...
    }

    ETG_TRACE_USR4(("End  : vWaitForSystemVoltageChange"));
    }
}


/*-----------------------------------------------------------------------------*
 *tVoid VoltageManager::vSendVoltageWarning(...)
 *-----------------------------------------------------------------------------*/
tVoid VoltageManager::vSendVoltageWarning(tenSignalValue f_eU60_drop) const
{
    ETG_TRACE_USR4(("vSendVoltageWarning: f_eU60_drop:%d",ETG_ENUM(TENSIGNALVALUE,f_eU60_drop)));

    if((f_eU60_drop == SIG_TRUE)||(f_eU60_drop == SIG_FALSE) )
    {
       //one message for all USB connectors
         ETG_TRACE_USR2(("vWaitLoopForSystemVoltageChanged: eUSBAll : eU60_drop:%d",ETG_ENUM(TENSIGNALVALUE,f_eU60_drop)));

#ifdef USE_HIGHRISK4USB_HWMALFUNCTION
         tBool bIsMalfunction = FALSE;
         if(SIG_TRUE == f_eU60_drop)
         {
            bIsMalfunction = TRUE;
         }
         //if voltage drops to critical then usb connectors might get malfunction. Risk is higher but it might also happen that usb continues to work
         for(int i=(int)eUSB1; i< ARRAYSIZEFORUSB; i++)
         {
             m_oIStateTableHWMalfunction.SetMalfunctionUSBConnector(IN bIsMalfunction, IN(tenDevManagerUSBHost)i,eIsHighRiskForMalfunction);
         }
#else 
         StateTable::GetInstance()->setVoltage((tenDevManagerUSBHost)eUSBAll,f_eU60_drop);
#endif


    }
    else
    {
        ETG_TRACE_FATAL(("[WARNING] vSendVoltageWarning: f_eU60_drop:%d (value ignored)",ETG_ENUM(TENSIGNALVALUE,f_eU60_drop)));
    }
    
}


 /*! 
     * \brief function to stop loop of this thread
     *
     * \param [IN] eCriticalVoltage
     */
tBool VoltageManager::bStopThread()
{
    ETG_TRACE_USR4(("Begin:bStopThread"));
    m_bStopThread = TRUE;
    ETG_TRACE_USR4(("End  :bStopThread"));
    return m_bStopThread;
}

/*-----------------------------------------------------------------------------*
 *void VoltageManager::Do(...) THREAD FUNCTION
 *-----------------------------------------------------------------------------*/
void VoltageManager::Do(int functionID, void *ptr)
{
    ETG_TRACE_USR4(("Begin:Do"));

    (void)ptr;

    tenThreadFunction eFunctionID = (tenThreadFunction)functionID;
    switch(eFunctionID)
    {
        case VoltageManager::eThread_SystemVolt: //execution tree for thread eThread_SystemVolt 
            ThreadFactoryDVM::GetThreadFactory()->SetName("VD_DVM:eThread_SystemVolt");
            vReceiveSystemVoltageNotification();
            break;
        default:
            break;
    }
    ETG_TRACE_USR4(("End  :Do"));
}


////////////////////////////////////////////////////////////////////////////////
// <EOF>

